/*
 * Copyright (C) 2017 ~ 2018 Deepin Technology Co., Ltd.
 *
 * Author:     kirigaya <kirigaya@mkacg.com>
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "worker.h"

#include <QJsonDocument>
#include <QJsonArray>

#include <DWindowManagerHelper>

// DBus Dock1's property
static const char kDockDisplayMode[] = "DisplayMode";

Worker *Worker::Instance()
{
    static Worker *instance = new Worker;
    return instance;
}

bool Worker::isWaylandType()
{
    bool bRet = false;
    auto e = QProcessEnvironment::systemEnvironment();
    QString XDG_SESSION_TYPE = e.value(QStringLiteral("XDG_SESSION_TYPE"));
    QString WAYLAND_DISPLAY = e.value(QStringLiteral("WAYLAND_DISPLAY"));
    if (XDG_SESSION_TYPE == QLatin1String("wayland") || WAYLAND_DISPLAY.contains(QLatin1String("wayland"), Qt::CaseInsensitive)) {
        bRet = true;
    } else {
        bRet = false;
    }

    return bRet;
}

/*******************************************************************************
 1. @函数:    setDesktopMode
 2. @作者:
 3. @日期:    2020-12-15
 4. @说明:    设置桌面样式发送到dock栏
*******************************************************************************/
void Worker::setDesktopMode(Model::DesktopMode mode)
{
    QStringList args;
    switch (mode) {
        case Model::EfficientMode:
            args << "--print-reply"
                 << "--dest=org.deepin.dde.daemon.Launcher1"
                 << "/com/deepin/dde/daemon/Launcher"
                 << "org.freedesktop.DBus.Properties.Set"
                 //                 << "string:org.deepin.dde.daemon.Launcher1"
                 //                 << "string:Fullscreen"
                 << "variant:boolean:false";
            break;
        case Model::FashionMode:
            args << "--print-reply"
                 << "--dest=org.deepin.dde.daemon.Launcher1"
                 << "/com/deepin/dde/daemon/Launcher"
                 << "org.freedesktop.DBus.Properties.Set"
                 //                 << "string:org.deepin.dde.daemon.Launcher1"
                 //                 << "string:Fullscreen"
                 << "variant:boolean:true";
            break;
    }

    QProcess::startDetached("dbus-send", args);

    m_dockInter->setProperty(kDockDisplayMode, mode);
}

/*******************************************************************************
 1. @函数:    setWMMode
 2. @作者:
 3. @日期:    2020-12-15
 4. @说明:    设置运行模式发送DBUS修改电脑的窗口特效打开或关闭 zyf修改成控制中心调用DBUS代码
*******************************************************************************/
void Worker::setWMMode(Model::WMType type)
{
    if (m_model->wmType() != type) {
        bool value;
        type == Model::WMType::WM_2D ? value = false : value = true;

        QDBusInterface Interface(
            "com.deepin.wm", "/com/deepin/wm", "org.freedesktop.DBus.Properties", QDBusConnection::sessionBus());
        QDBusMessage reply =
            Interface.call("Set", "com.deepin.wm", "compositingEnabled", QVariant::fromValue(QDBusVariant(value)));
        if (reply.type() == QDBusMessage::ErrorMessage) {
            qDebug() << "reply.type() = " << reply.type();
        }
    }
}

/*******************************************************************************
 1. @函数:    setIcon
 2. @作者:
 3. @日期:    2020-12-15
 4. @说明:    设置主题图标改变电脑的主题图标配置
*******************************************************************************/
void Worker::setIcon(const IconStruct &icon)
{
    m_iconInter->call("Set", "icon", icon.Id);
}

/*******************************************************************************
 1. @函数:    onWMChanged
 2. @作者:
 3. @日期:    2020-12-15
 4. @说明:    运行模式变化保存到model槽
*******************************************************************************/
void Worker::onWMChanged()
{
    bool wm = DWindowManagerHelper::instance()->hasBlurWindow();
    m_model->setWmType(wm ? Model::WM_3D : Model::WM_2D);
}

/*******************************************************************************
 1. @函数:    onDisplayModeChanged
 2. @作者:
 3. @日期:    2020-12-15
 4. @说明:    桌面样式变化保存到model槽
*******************************************************************************/
void Worker::onDisplayModeChanged(int mode)
{
    m_model->setDesktopMode(static_cast<Model::DesktopMode>(mode));
}

/*******************************************************************************
 1. @函数:    onIconRefreshed
 2. @作者:
 3. @日期:    2020-12-15
 4. @说明:    图标刷新槽
*******************************************************************************/
void Worker::onIconRefreshed(const QString &name)
{
    if (name == "icon") {
        // QDBusPendingReply类包含对异步方法调用的答复。
        QDBusPendingReply<QString> icon = m_iconInter->call("List", "icon");
        // QDBusPendingCallWatcher提供了一种等待异步响应的便捷方法
        QDBusPendingCallWatcher *watcher = new QDBusPendingCallWatcher(icon, this);
        // 图片列表刷新响应函数
        connect(watcher, &QDBusPendingCallWatcher::finished, this, &Worker::onIconClickChange);
    }
}

void Worker::onIconClickChange(QDBusPendingCallWatcher *w)
{
    if (w->isError()) {
        qDebug() << w->error().message();
    } else {
        QDBusPendingReply<QString> reply = *w;
        onIconListChanged(reply.value());
    }
    w->deleteLater();
}

/*******************************************************************************
 1. @函数:    onIconListChanged
 2. @作者:
 3. @日期:    2020-12-16
 4. @说明:    图标主题列表变化响应槽
*******************************************************************************/
void Worker::onIconListChanged(const QString &value)
{
    const QJsonArray &array = QJsonDocument::fromJson(value.toUtf8()).array();

    QStringList currentIconIdList;
    for (const QJsonValue &value : array) {
        QDBusPendingReply<QString> icon = m_iconInter->call("Thumbnail", "icon", value.toObject()["Id"].toString());
        const QJsonObject &obj = value.toObject();

        currentIconIdList << obj["Id"].toString();

        QDBusPendingCallWatcher *watcher = new QDBusPendingCallWatcher(icon, this);
        watcher->setProperty("Json", obj);
        connect(watcher, &QDBusPendingCallWatcher::finished, this, &Worker::onIconPixmapFinished);
    }

    for (const IconStruct &icon : m_model->iconList()) {
        if (!currentIconIdList.contains(icon.Id)) {
            m_model->removeIcon(icon);
        }
    }
}

/*******************************************************************************
 1. @函数:    onIconPixmapFinished
 2. @作者:
 3. @日期:    2020-12-16
 4. @说明:    图标图片完成响应槽
*******************************************************************************/
void Worker::onIconPixmapFinished(QDBusPendingCallWatcher *w)
{
    QDBusPendingReply<QString> reply = *w;
    if (!reply.isError()) {
        QJsonObject obj = w->property("Json").toJsonObject();
        obj["Pixmap"] = reply.value();
        m_model->addIcon(IconStruct::fromJson(obj));
    }
    w->deleteLater();
}

void Worker::onChanged(const QString& theme, const QString& value)
{
    // IconTheme property changed
    if (theme == "IconTheme") {
        m_model->setCurrentIcon(value);
    }
}

Worker::Worker(QObject *parent)
    : QObject(parent)
    , m_model(Model::Instance())
{
    QString appearName = "org.deepin.dde.Appearance1";
    QString appearPath = "/org/deepin/dde/Appearance1";
    QString dockName = "org.deepin.dde.daemon.Dock1";
    QString dockPath = "/org/deepin/dde/daemon/Dock1";


    m_iconInter.reset(new QDBusInterface(appearName, appearPath, appearName, QDBusConnection::sessionBus(), this));
    m_dockInter.reset(  new QDBusInterface(dockName, dockPath, dockName, QDBusConnection::sessionBus(), this));
    // not block ui.
    m_dockInter->setTimeout(3000);
    auto bus = QDBusConnection::sessionBus();
    bus.connect(dockName,
                dockPath,
                dockName,
                "DisplayModeChanged",
                this,
                SLOT(onDisplayModeChanged(int)));
    onDisplayModeChanged(m_dockInter->property(kDockDisplayMode).toInt());

    // 替换为监听DBus的Refresh和Changed信号
    bus.connect(appearName,
                appearPath,
                appearName,
                "Refreshed",
                this,
                SLOT(onIconRefreshed(QString)));
    bus.connect(appearName,
                appearPath,
                appearName,
                "Changed",
                this,
                SLOT(onChanged(QString, QString)));

    connect(DWindowManagerHelper::instance(), &DWindowManagerHelper::hasBlurWindowChanged, this, &Worker::onWMChanged);

    m_iconInter->call("Sync", "false");

    m_model->setCurrentIcon(m_iconInter->property("IconTheme").toString());
    m_windowManage = DWindowManagerHelper::instance();
    onWMChanged();

    onIconRefreshed("icon");
}
