/*
 * Copyright (C) 2017 ~ 2018 Deepin Technology Co., Ltd.
 *
 * Author:     kirigaya <kirigaya@mkacg.com>
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "widgetmanager.h"
#include "widgetmanager.h"
#include "ddlog.h"

#include <QDBusInterface>
#include <QDBusConnection>
#include <QDBusReply>

#define MAXPLAYER 3

WidgetManager *WidgetManager::Instance()
{
    qCDebug(app) << "Getting WidgetManager instance";
    static WidgetManager *instance = new WidgetManager;
    return instance;
}

void WidgetManager::setParents(QWidget *p)
{
    qCDebug(app) << "Setting parent widget for modules";
    m_fakerWidget = p;
    if (!m_widgetList.videoFileList.isEmpty()) {
        qCDebug(app) << "Initializing video module with file:" << m_widgetList.videoFileList[0];
        m_widgetList.append(initVideoModule(m_widgetList.videoFileList[0]));
    } else {
        qCDebug(app) << "Initializing desktop mode and icon modules";
        m_widgetList.append(initDesktopModeModule());
        m_widgetList.append(initIconModule());
    }
    m_widgetList.currentVideo = 0;
    m_widgetList.current = 0;
}

int WidgetManager::getWidgetSize()
{
    return m_widgetList.videoFileList.size() + 2;
}

bool WidgetManager::canHandoff()
{
    if (m_widgetList.videoWidgetList.size() == MAXPLAYER)
        return true;
    if (m_widgetList.videoWidgetList.size() > m_widgetList.current+1)
        return true;
    return false;
}

void WidgetManager::finishedInit()
{
    static_cast<VideoWidget *>(m_widgetList.videoWidgetList.last().w->getModel())->setFile(m_widgetList.videoFileList[m_widgetList.videoWidgetList.size()-1]);
}

void WidgetManager::delayInit()
{
    qCDebug(app) << "Delayed initialization started";
    // If video count less than MAXPLAYER, avoid out of bounds.
    if (m_widgetList.videoFileList.size() < MAXPLAYER) {
        if (m_widgetList.videoWidgetList.size() == m_widgetList.videoFileList.size()) {
            qCDebug(app) << "Video count less than MAXPLAYER, skipping";
            return;
        }
    }

    Q_ASSERT(m_widgetList.videoWidgetList.size() <= MAXPLAYER);
    Q_ASSERT(m_widgetList.videoFileList.size() > m_widgetList.currentVideo+1);

    qCDebug(app) << "Initializing next video module";
    m_widgetList.append(initVideoModule(m_widgetList.videoFileList[m_widgetList.videoWidgetList.size()]));
    if (m_widgetList.otherWidgetList.isEmpty()) {
        m_widgetList.append(initDesktopModeModule());
        m_widgetList.append(initIconModule());
    }
}

void WidgetManager::slotShowMore()
{
    // disable temporarily
#if 0
    if (getCurrentWidget().type == videoType && m_widgetList.current == m_widgetList.videoFileList.size()-1) {
        emit showMore();
    }
#endif
}

IntroductionWidget WidgetManager::getCurrentWidget()
{
    if (m_widgetList.size && m_widgetList.current != -1) {
        if (m_widgetList.current < m_widgetList.videoFileList.size())
            return m_widgetList.videoWidgetList.at(m_widgetList.currentVideo);
        else
            return m_widgetList.otherWidgetList.at(m_widgetList.current - m_widgetList.videoFileList.size());
    }

    return IntroductionWidget();
}

IntroductionWidget WidgetManager::handoffNextWidget()
{
    qCDebug(app) << "Handing off to next widget";
    auto lastWidget = getCurrentWidget();
    setVideoMute(lastWidget, true);

    m_widgetList.current++;
    if (m_widgetList.current < m_widgetList.videoFileList.size()) {
        if (m_widgetList.currentVideo + 1 == MAXPLAYER)
            m_widgetList.currentVideo = 0;
        else
            m_widgetList.currentVideo++;

        int index = m_widgetList.currentVideo + 1;
        if (index >= MAXPLAYER)
            index = 0;
        if (m_widgetList.videoWidgetList.size() >= index+1 && m_widgetList.current < m_widgetList.videoFileList.size()-1) {
            BaseModuleWidget *nextVideo = m_widgetList.videoWidgetList.at(index).w;
            static_cast<VideoWidget *>(nextVideo->getModel())->setFile(m_widgetList.videoFileList.at(m_widgetList.current+1));
        }
    }

    auto introWidget = getCurrentWidget();
    setVideoMute(introWidget, false);
    return introWidget;
}

IntroductionWidget WidgetManager::handoffPrevWidget()
{
    qCDebug(app) << "Handing off to previous widget";
    Q_ASSERT(m_widgetList.current);
    auto lastWidget = getCurrentWidget();
    setVideoMute(lastWidget, true);

    m_widgetList.current--;
    if (m_widgetList.current + 1 < m_widgetList.videoFileList.size()) {
        if (!m_widgetList.currentVideo)
            m_widgetList.currentVideo = MAXPLAYER - 1;
        else
            m_widgetList.currentVideo--;

        int index = m_widgetList.currentVideo - 1;
        if (index < 0)
            index = MAXPLAYER - 1;
        if (m_widgetList.current) {
            BaseModuleWidget *nextVideo = m_widgetList.videoWidgetList.at(index).w;
            static_cast<VideoWidget *>(nextVideo->getModel())->setFile(m_widgetList.videoFileList.at(m_widgetList.current-1));
        }
    }

    auto introWidget = getCurrentWidget();
    setVideoMute(introWidget, false);
    return introWidget;
}

WidgetManager::WidgetManager(QObject *parent)
    : QObject(parent)
{
    qCDebug(app) << "WidgetManager constructor called";
    initPageType();
}

IntroductionWidget WidgetManager::initVideoModule(QUrl file)
{
    qCDebug(app) << "Initializing video module with file:" << file;
    if (m_widgetList.videoWidgetList.size() >= MAXPLAYER)
        return IntroductionWidget();
    IntroductionWidget widget;
    VideoWidget *module = new VideoWidget(file, m_fakerWidget);
    connect(module, &VideoWidget::showMore, this, &WidgetManager::slotShowMore);
    connect(module, &VideoWidget::finishedShow, [=](){
        module->setFile(file);
    });
    BaseModuleWidget *w = new BaseModuleWidget(module, true, m_fakerWidget);
    w->setFixedSize(QSize(m_fakerWidget->width() + 1, m_fakerWidget->height() + 1));
    w->hide();
    widget.w = w;
    widget.type = videoType;
    return widget;
}

IntroductionWidget WidgetManager::initDesktopModeModule()
{
    qCDebug(app) << "Initializing desktop mode module";
    IntroductionWidget widget;
    DesktopModeModule *module = new DesktopModeModule;
    module->updateBigIcon();
    BaseModuleWidget *w = new BaseModuleWidget(module, false, m_fakerWidget);
    w->setDescribe(tr("You can switch \"Alignment\" by right-clicking on the dock."));
    w->setFixedSize(m_fakerWidget->size());
    w->hide();
    widget.w = w;
    widget.type = StyleType;
    return widget;
}

IntroductionWidget WidgetManager::initIconModule()
{
    qCDebug(app) << "Initializing icon module";
    IntroductionWidget widget;
    IconModule *module = new IconModule;
    module->updateBigIcon();
    BaseModuleWidget *w = new BaseModuleWidget(module, false, m_fakerWidget);
    w->setFixedSize(m_fakerWidget->size());
    w->hide();
    widget.w = w;
    widget.type = FinishType;
    return widget;
}

// 自定义比较函数，用于排序
bool sortByNumberPrefix(const QFileInfo &a, const QFileInfo &b)
{
#if (QT_VERSION < QT_VERSION_CHECK(6, 0, 0))
    QRegExp numPrefixA("(\\d+).*");
    QRegExp numPrefixB("(\\d+).*");

    if (numPrefixA.indexIn(a.fileName()) == 0 && numPrefixB.indexIn(b.fileName()) == 0) {
        // 如果文件名以数字开始，则按数字顺序排序
        return numPrefixA.cap(1).toInt() < numPrefixB.cap(1).toInt();
    }
#else
    QRegularExpression re("(\\d+).*");
    auto matchA = re.match(a.fileName());
    auto matchB = re.match(b.fileName());

    if (matchA.hasMatch() && matchB.hasMatch()) {
        // 如果文件名以数字开始，则按数字顺序排序
        return matchA.captured(1).toInt() < matchB.captured(1).toInt();
    }
#endif

    // 如果不是以数字开始，则按字典顺序排序
    return a.fileName() < b.fileName();
}

void WidgetManager::initPageType()
{
    qCDebug(app) << "Initializing page types";

    QString path("/usr/share/dde-introduction");
    // 获取 XDG_DATA_DIRS 环境变量
    QString xdgDataDirs = QProcessEnvironment::systemEnvironment().value("XDG_DATA_DIRS");

    if (xdgDataDirs.isEmpty()) {
        qCWarning(app) << "XDG_DATA_DIRS environment variable is not set, using default path";
        xdgDataDirs = "/usr/share"; // default
    }

    // 将 XDG_DATA_DIRS 按冒号分割成路径列表
    QStringList xdgDataDirsList = xdgDataDirs.split(":");

    QString appdir = QString("/%1").arg(qAppName());
    // 检查每个路径是否包含 "dde-introduction" 文件夹
    foreach (const QString &dirPath, xdgDataDirsList) {
        QDir dir(dirPath + appdir);
        if (dir.exists()) {
        qCDebug(app) << "Found dde-introduction resources in:" << dirPath;
        path = dirPath + appdir;
            break;
        }
    }

    QLocale locale;

    // 2500: Paths are no longer set based on system language
#if 0
    if (locale.language() == QLocale::Chinese ||
            locale.language() == QLocale::TraditionalChineseScript ||
            locale.language() == QLocale::Taiwan)
        path += "/zh";
    else
        path += "/en";
#else
    if (isCommunityEdition()) {
        // The Community Edition uses a separate version of the video
        path += "/deepin";
    } else {
        path += "/uos";
    }
#endif

    QDir videoDir(path);
    videoDir.setFilter(QDir::Files);
    videoDir.setSorting(QDir::Name | QDir::Reversed);

    // 2500: Add mov file support
    QStringList filters;
    filters << "*.mp4" << "*.mov";

    // 获取匹配的文件列表
    QFileInfoList files = videoDir.entryInfoList(filters);
    std::sort(files.begin(), files.end(), sortByNumberPrefix);

    // 遍历文件列表并打印文件名
    for (const QFileInfo &file : files)
        m_widgetList.videoFileList.append(QUrl(QString("file://" + file.absoluteFilePath())));
}

void WidgetManager::setVideoMute(IntroductionWidget &introWidget, bool b)
{
    qCDebug(app) << "Setting video mute state to:" << b << "for widget type:" << introWidget.type;

    // mute last video widget
    if (videoType == introWidget.type) {
        if (auto *lastVideo = qobject_cast<VideoWidget *>(introWidget.w->getModel())) {
            lastVideo->setMute(b);
        }
        qCDebug(app) << "Video mute state set successfully";
    }
}

/**
 * @brief Check if current system is Deepin Community Edition
 *
 * This function checks the system's pretty name through DBus interface
 * to determine if it's Deepin Community Edition.
 * e.g.: "Deepin 25"
 *       "UOS Desktop 25 Professional"
 *
 * @return true if system is Deepin Community Edition
 * @return false if not or when DBus call fails
 */
bool WidgetManager::isCommunityEdition() const
{
    qCDebug(app) << "Checking if system is Community Edition";

    QDBusInterface iface("org.freedesktop.hostname1",
                        "/org/freedesktop/hostname1",
                        "org.freedesktop.hostname1",
                        QDBusConnection::systemBus());

    if (!iface.isValid()) {
        qCWarning(app) << "Failed to connect to hostname1 DBus service";
        return false;
    }

    QVariant reply = iface.property("OperatingSystemPrettyName");
    if (!reply.isValid()) {
        qCWarning(app) << "Failed to get OS name:" << iface.lastError().message();
        return false;
    }

    // Check if OS name contains "deepin" (case insensitive)
    QString osName = reply.toString().toLower();
    qCInfo(app) << "Detected OS:" << osName;

    return osName.contains("deepin");
}
