#include <chrono>
#include <fstream>
#include "header.h"
#include "logger.h"
#include "json.hpp"
#include "common.h"
#include "reportevent.h"

using namespace std;
using std::chrono::steady_clock;
using json = nlohmann::json;
using std::chrono::system_clock;
typedef std::shared_ptr<struct sigaction> SigPtr;
typedef std::map<int, SigPtr> MapSig;

static std::string gPackageName = "";
static uid_t userid = 0;
static pid_t currpid = 0;

const int kExceptionSignals[] = {
    // Core signals
    SIGABRT, SIGBUS, SIGFPE, SIGILL, SIGQUIT, SIGSEGV, SIGSYS, SIGTRAP, SIGXCPU, SIGXFSZ,
};

const int kNumHandledSignals = sizeof(kExceptionSignals) / sizeof(kExceptionSignals[0]);
static MapSig g_handlemap;

static uint64_t MilliSecondsToEpoch()
{
    return std::chrono::duration_cast<std::chrono::milliseconds>(system_clock::now().time_since_epoch()).count();
}

static void FillCommonData(json &jdata)
{
    jdata["time"] = MilliSecondsToEpoch();
    jdata["pid"] = currpid;
    jdata["package"] = gPackageName;
    jdata["sys_uid"] = userid;
}

static void RecordHandler(int signum, siginfo_t *siginfo, void *usercontext)
{
    struct ucontext_t *context = static_cast<ucontext_t*>(usercontext);
    Logger::get_mutable_instance().Print("catch signal:%d", signum);
    json exdata;
    json infodata;
    infodata["tid"] = 1;
    infodata["signo"] = signum;
    infodata["address"] = (uint64_t)siginfo->si_addr;

    // /proc/self/maps 存了各个程序段在内存中的加载地址范围
    ifstream ifs;
    ifs.open("/proc/self/maps", ios::in);
    if (!ifs.is_open()) {
        Logger::get_mutable_instance().Print("open maps file failed");
    } else {
        // 读取文件
        stringstream ss;
        ss << ifs.rdbuf();
        string mapscontent(ss.str());
        ifs.close();
        infodata["maps"] = mapscontent;
    }

    // 填充通用数据
    FillCommonData(exdata);
    exdata["info"] = infodata.dump();

    // 异常信息发送给 daemon
    ReportEvent::get_mutable_instance().SendEvent(gPackageName, exdata.dump());
    if (g_handlemap.count(signum) > 0) {
        sigaction(signum, g_handlemap[signum].get(), nullptr);
    } else {
        struct sigaction defact = {0};
        sigaction(signum, &defact, nullptr);
    }
}

// 注册信号处理程序
static void RegisterSigHander()
{
    for (int i = 0; i < kNumHandledSignals; ++i) {
        struct sigaction act = {0};
        SigPtr oldptr(new struct sigaction());
        // 清空信号集
        sigemptyset(&act.sa_mask);
        // 把 SIGINT 加入到信号集中
        sigaddset(&act.sa_mask, kExceptionSignals[i]);
        act.sa_flags = SA_SIGINFO | SA_ONSTACK;
        // 指定信号处理程序
        act.sa_sigaction = &RecordHandler;
        // 查询/设置信号处理方式
        if (sigaction(kExceptionSignals[i], &act, oldptr.get()) == -1)
            Logger::get_mutable_instance().Print("sigaction error:%s", strerror(errno));

        g_handlemap.insert(std::make_pair(kExceptionSignals[i], oldptr));
    }
}

void Init()
{
    Logger::get_mutable_instance().Print("load sdk library");
}

// 初始化包信息
bool Initialize(const std::string &packagename, bool enable_sig)
{
    userid = getuid();
    currpid = getpid();
    gPackageName = packagename;

    // 注册信号处理器
    if (enable_sig)
        RegisterSigHander();

    // 事件初始化
    return ReportEvent::get_mutable_instance().Init();
}

// 写日志
void WriteEventLog(const std::string &eventdata)
{
    json pagedata;

    // 填充通用数据
    FillCommonData(pagedata);

    pagedata["info"] = eventdata;
    string strdata = pagedata.dump();
    Logger::get_mutable_instance().Print("report event data:%s", strdata.c_str());

    // 将日志发送至 daemon
    ReportEvent::get_mutable_instance().PostEvent(gPackageName, strdata);
}

// 广播日志
void SendEventLog(const std::string &eventdata)
{
    json pagedata;
    json infodata;

    if (eventdata.empty() || !json::accept(eventdata.c_str())) {
        pagedata["info"] = eventdata;
    } else {
        infodata = json::parse(eventdata.c_str());
        if (!infodata.is_object() || !infodata.contains("info")) {
            pagedata["info"] = eventdata.c_str();
        } else {
            pagedata = infodata;
        }
    }

    // 为区分新旧接口
    pagedata["mode"] = 1;

    // 填充通用数据
    FillCommonData(pagedata);

    string strdata = pagedata.dump();
    Logger::get_mutable_instance().Print("report event data:%s", strdata.c_str());

    // 将日志发送至 daemon
    ReportEvent::get_mutable_instance().PostEvent(gPackageName, strdata);
}

#ifdef __cplusplus
extern "C" {
#endif

bool InitializeWrapper(const char *packagename, bool enable_sig)
{
    std::string text;
    if (packagename != nullptr)
        text = packagename;

    // 事件初始化
    return Initialize(text, enable_sig);
}

void WriteEventLogWrapper(const char *eventdata)
{
    std::string text;
    if (eventdata != nullptr)
        text = eventdata;

    // 事件初始化
    WriteEventLog(text);
}

void SendEventLogWrapper(const char *eventdata)
{
    std::string text;
    if (eventdata != nullptr)
        text = eventdata;

    // 事件初始化
    SendEventLog(text);
}

#ifdef __cplusplus
}
#endif
