//
// Created by uos on 2022/4/28.
//

#include "UserDataRestoreWidget.h"
#include "common/BaseItemDelegate.h"
#include <QVBoxLayout>
#include <DFontSizeManager>
#include <QHeaderView>
#include <QDateTime>
#include <DRadioButton>
#include <QScrollBar>

const int TABLEVIEW_ROW_HEIGHT = 40;    //tableview 每列高度
const int TABLE_COL_CHOOSE = 0;         //table表 第0列：选择
const int TABLE_COL_TIME = 1;           //table表 第1列：时间
const int TABLE_COL_TYPE = 2;           //table表 第2列：类型
const int TABLE_COL_USER = 3;           //table表 第3列：用户
const int TABLE_COL_REMARK = 4;         //table表 第4列：备注
const int TABLE_COL_OPT_ID = 5;         //table表 第5列：操作id

UserDataRestoreWidget::UserDataRestoreWidget(DWidget *parent)
    : DWidget(parent)
{
    this->setAccessibleName("UserDataRestoreWidget");

    // 主题变换
    m_guiHelper = Dtk::Gui::DGuiApplicationHelper::instance();
    connect(m_guiHelper, &Dtk::Gui::DGuiApplicationHelper::themeTypeChanged, this, &UserDataRestoreWidget::onThemeChange);
    // 初始化UI
    initUI();
    onThemeChange();
}

void UserDataRestoreWidget::initUI()
{
    QVBoxLayout *verticalLayout = new QVBoxLayout(this);
    verticalLayout->setContentsMargins(10, 20, 10, 10);
    verticalLayout->setAlignment(Qt::AlignHCenter);

    DLabel *title = new DLabel;
    title->setText(tr("Data Restore"));
    title->setAccessibleName("uosDataRestore_DataRestoreTitle");
    title->setAlignment(Qt::AlignCenter);
    DFontSizeManager::instance()->bind(title, DFontSizeManager::T4);
    QFont font = title->font();
    font.setWeight(QFont::Bold);
    title->setFont(font);
    verticalLayout->addWidget(title);

    verticalLayout->addSpacing(4);
    auto subTitle = new DLabel;
    subTitle->setAlignment(Qt::AlignCenter);
    subTitle->setText(tr("Select a backup file and restore your data from it."));
    subTitle->setAccessibleName("uosDataRestore_DataRestoreSubTitle");
    verticalLayout->addWidget(subTitle);

    m_tableView = new BaseTableView;
    m_tableView->setAccessibleName("uosDataRestore_TableView");
    m_model = new QStandardItemModel;
    m_tableView->setModel(m_model);
    auto itemDelegate = new BaseItemDelegate;
    m_tableView->setItemDelegate(itemDelegate);
    initTableView();
    verticalLayout->addWidget(m_tableView);
    verticalLayout->addStretch();

    auto warning = new DLabel;
    warning->setText(tr("Attention: The existing data will be overwritten. Please proceed carefully!"));
    warning->setAccessibleName("uosDataRestore_WarningLabel");
    warning->setAlignment(Qt::AlignHCenter);
    warning->setWordWrap(true);
    warning->setStyleSheet("QLabel {"
                           "font-family: SourceHanSansSC, SourceHanSansSC-Medium;"
                           "font-size: 14px;"
                           "font-weight: 500;"
                           "color: #ff5736;"
                           "}");
    verticalLayout->addWidget(warning);
    verticalLayout->addSpacing(10);

    auto hLayout = new QHBoxLayout;
    hLayout->setAlignment(Qt::AlignHCenter);
    m_cancelBtn = new DPushButton;
    m_cancelBtn->setText(tr("Cancel", "button"));
    m_cancelBtn->setAccessibleName("uosDataRestore_Cancel");
    m_cancelBtn->setFixedSize(140, 36);
    hLayout->addWidget(m_cancelBtn);
    hLayout->addSpacing(10);

    m_okBtn = new DSuggestButton;
    m_okBtn->setEnabled(false);
    m_okBtn->setText(tr("Start Restore"));
    m_okBtn->setAccessibleName("uosDataRestore_Restore");
    m_okBtn->setFixedSize(140, 36);
    hLayout->addWidget(m_okBtn);
    verticalLayout->addLayout(hLayout);

    connect(m_tableView, &QTreeView::pressed, this, &UserDataRestoreWidget::onBackupFileChanged);
    connect(m_cancelBtn, &DPushButton::clicked, this, &UserDataRestoreWidget::cancel);
    connect(m_okBtn, &DPushButton::clicked, [this] {
        auto itFind = m_allBackupInfo.find(m_backupFileOptUUID);
        if (itFind != m_allBackupInfo.end()) {
            Q_EMIT start(itFind.value());
        }
    });
}

void UserDataRestoreWidget::initTableView()
{
    m_tableView->setSelectionBehavior(QAbstractItemView::SelectRows);
    m_tableView->setSelectionMode(QAbstractItemView::NoSelection);
    m_tableView->setEditTriggers(QAbstractItemView::NoEditTriggers);
    m_tableView->setFrameShape(QTableView::NoFrame);
    m_tableView->setSortingEnabled(false);
    m_tableView->setFocusPolicy(Qt::NoFocus);

    auto chooseItem = new QStandardItem();

    // 时间
    auto timeItem = new QStandardItem(tr("Time"));
    timeItem->setTextAlignment(Qt::AlignVCenter);
    // 类型
    auto typeItem = new QStandardItem(tr("Type"));
    typeItem->setTextAlignment(Qt::AlignVCenter);
    // 用户
    auto userItem = new QStandardItem(tr("User"));
    userItem->setTextAlignment(Qt::AlignVCenter);
    // 备注
    auto remarkItem = new QStandardItem(tr("Note"));
    remarkItem->setTextAlignment(Qt::AlignVCenter);

    //操作id
    auto optIdItem = new QStandardItem();

    m_model->setHorizontalHeaderItem(TABLE_COL_CHOOSE, chooseItem);
    m_model->setHorizontalHeaderItem(TABLE_COL_TIME, timeItem);
    m_model->setHorizontalHeaderItem(TABLE_COL_TYPE, typeItem);
    m_model->setHorizontalHeaderItem(TABLE_COL_USER, userItem);
    m_model->setHorizontalHeaderItem(TABLE_COL_REMARK, remarkItem);
    m_model->setHorizontalHeaderItem(TABLE_COL_OPT_ID, optIdItem);

    m_tableView->setFixedHeight(DEFAULT_ROW_HEIGHT * 8);
    m_tableView->header()->setSortIndicatorShown(false);
//    m_tableView->header()->setSectionResizeMode(TABLE_COL_CHOOSE, QHeaderView::ResizeToContents);
//    m_tableView->header()->setSectionResizeMode(TABLE_COL_TIME, QHeaderView::Stretch);
//    m_tableView->header()->setSectionResizeMode(TABLE_COL_TYPE, QHeaderView::Stretch);
//    m_tableView->header()->setSectionResizeMode(TABLE_COL_USER, QHeaderView::Stretch);
//    m_tableView->header()->setSectionResizeMode(TABLE_COL_REMARK, QHeaderView::Stretch);

    m_tableView->setColumnWidth(TABLE_COL_CHOOSE, 3);
    m_tableView->setColumnWidth(TABLE_COL_TIME, 150);
    m_tableView->setColumnWidth(TABLE_COL_TYPE, 130);
    m_tableView->setColumnWidth(TABLE_COL_USER, 220);
    m_tableView->setColumnWidth(TABLE_COL_REMARK, 250);
    m_tableView->setColumnWidth(TABLE_COL_OPT_ID, 5);


    //隐藏列
    m_tableView->header()->setSectionHidden(TABLE_COL_OPT_ID, true);

    m_tableView->setRowBackgroundDelta(m_tableView->header()->sectionSize(TABLE_COL_CHOOSE));
    m_tableView->horizontalScrollBar()->setEnabled(false);
    m_tableView->setDragEnabled(false);
    m_tableView->header()->setSectionsMovable(false);
    m_tableView->header()->setSectionResizeMode(QHeaderView::ResizeMode::Fixed); //禁止标题列拖动
}

void UserDataRestoreWidget::setData(BackupInfoList &backupInfo)
{
    m_model->removeRows(0, m_model->rowCount());
    m_allBackupInfo.clear();
    m_okBtn->setEnabled(false);
    std::sort(backupInfo.begin(), backupInfo.end());
    auto iter = backupInfo.rbegin();
    for (; iter != backupInfo.rend(); ++iter) {
        m_allBackupInfo[iter->operateID] = *iter;
        int currentRow = m_model->rowCount();
        auto chooseItem = new QStandardItem();
        m_model->setItem(currentRow, TABLE_COL_CHOOSE, chooseItem);

        auto time = QDateTime::fromSecsSinceEpoch(iter->startTime);
        QString startTime = time.toString("yyyy-MM-dd HH:mm:ss");
        auto timeItem = new QStandardItem(startTime);
        m_model->setItem(currentRow, TABLE_COL_TIME, timeItem);

        auto typeItem = new QStandardItem();
        typeItem->setText(iter->operateType == UserDataBackup ? tr("Data backup") : "");
        m_model->setItem(currentRow, TABLE_COL_TYPE, typeItem);

        auto userItem = new QStandardItem(QString("%1").arg(iter->username));
        m_model->setItem(currentRow, TABLE_COL_USER, userItem);

        auto remarkItem = new QStandardItem(QString("%1").arg(iter->remark));
        m_model->setItem(currentRow, TABLE_COL_REMARK, remarkItem);

        //操作id
        auto optIdItem = new QStandardItem(iter->operateID);
        optIdItem->setSelectable(false);
        m_model->setItem(currentRow, TABLE_COL_OPT_ID, optIdItem);

        //第0列
        DRadioButton *widget0 = new DRadioButton(m_tableView);
        widget0->setFocusPolicy(Qt::NoFocus);
        m_tableView->setIndexWidget(m_model->index(currentRow, TABLE_COL_CHOOSE), widget0);
        connect(widget0, &DRadioButton::clicked, this, &UserDataRestoreWidget::onRadioClicked);

    }
}

void UserDataRestoreWidget::onThemeChange()
{

}

void UserDataRestoreWidget::onBackupFileChanged(const QModelIndex &index)
{
    if (!index.isValid()) {
        return;
    }
    auto currItem = m_model->itemFromIndex(index);
    if (currItem == nullptr) {
        qCritical() << "Select backup file failed, currItem is null";
        return;
    }

    m_backupFileOptUUID = m_model->itemFromIndex(index.siblingAtColumn(TABLE_COL_OPT_ID))->text();
    if (!m_backupFileOptUUID.isEmpty()) {
        m_okBtn->setEnabled(true);
    }
    qDebug() << m_backupFileOptUUID;
    DRadioButton *widget = dynamic_cast<DRadioButton *>(m_tableView->indexWidget(m_model->index(index.row(), TABLE_COL_CHOOSE)));
    widget->setChecked(true);

}

void UserDataRestoreWidget::onRadioClicked(bool checked)
{
    if (checked) {
        int count = m_model->rowCount();
        for (int i = 0; i < count; ++i) {
            DRadioButton *widget = dynamic_cast<DRadioButton *>(m_tableView->indexWidget(m_model->index(i, TABLE_COL_CHOOSE)));
            if (widget == sender()) {
                onBackupFileChanged(m_model->index(i , TABLE_COL_CHOOSE));
                break;
            }
        }
    } else {
        m_okBtn->setEnabled(false);
    }
}
