/*
FUSE FPS server browser
Copyright (C) 2008  Tony Jackson

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/
#include <gtk/gtk.h>
#include <glade/glade.h>

#include <glib.h>
#include <glib/gprintf.h>

#include "slink.h"
#include "models.h"
#include "global.h"
#include "config.h"

/* Function prototypes */
void onQuit   (void);
void onRefresh(void);
void onJoin   (void);
void onSVChange (GtkTreeView *treeview, gpointer user_data);
void DisplayServerDetail(tServer *server);

GtkWindow *window;

/* TODO: The following (along with useful widget pointers) should go in userdata structures and not be global.
       This would allow the glade libraries to be unloaded after initial startup was complete - save some mem */
GladeXML  *xml;
GData     *MasterData;
GData     *ServerData;
GData     *GameConfigs;

void AddDataNameToCombo(GQuark key_id, gpointer data, gpointer user_data)
{
    GtkComboBox *combo = GTK_COMBO_BOX(user_data);
    gtk_combo_box_append_text (GTK_COMBO_BOX (combo), g_quark_to_string(key_id));
}

int main(int argc, char **argv)
{
    GtkWidget *sv_view;
    GtkWidget *pl_view;
    GtkWidget *btnRefresh;
    GtkWidget *btnJoin;
    GtkWidget *cmbGame;
    gchar title[80] = "";

    /* Initialise GTK */
    gtk_init(&argc, &argv);

    /* Load XML GUI descriptor generated by Glade */
    xml = glade_xml_new("fuse.glade", NULL, NULL);

    /* Get a handle on the main window */
    window     = GTK_WINDOW(glade_xml_get_widget(xml, "wndBrowser"));
    sv_view    = glade_xml_get_widget(xml, "trvServers");
    pl_view    = glade_xml_get_widget(xml, "trvPlayers");
    btnRefresh = glade_xml_get_widget(xml, "btnRefresh");
    btnJoin    = glade_xml_get_widget(xml, "btnJoin");
    cmbGame    = glade_xml_get_widget(xml, "cmbGame");

    #if 1
    /* Can't seem to get gmodule symbol discovery to work, so do manual
       connection */
    /* TODO: Match user data passed here with functions */
    gtk_signal_connect (GTK_OBJECT(window),     "destroy", GTK_SIGNAL_FUNC(onQuit),    NULL);
    gtk_signal_connect (GTK_OBJECT(btnRefresh), "clicked", GTK_SIGNAL_FUNC(onRefresh), ServerData);
    gtk_signal_connect (GTK_OBJECT(btnJoin),    "clicked", GTK_SIGNAL_FUNC(onJoin),    ServerData);
    gtk_signal_connect (GTK_OBJECT(sv_view),    "cursor-changed", GTK_SIGNAL_FUNC(onSVChange),    NULL);
    #else
    /* auto connect signal handlers */
    glade_xml_signal_autoconnect(xml);
    #endif

    g_sprintf(title, "%s - version %s", TITLE, VERSION);
    gtk_window_set_title(GTK_WINDOW(window), title);

    /* Build and connect the storage models for server/player GTK TreeViews */
	AddServerModelToView(sv_view);
    AddPlayerModelToView(pl_view);

    g_datalist_init(&MasterData);
    g_datalist_init(&ServerData);
    g_datalist_init(&GameConfigs);

    if(Config_LoadGames(&GameConfigs, "config/games.xml"))
    { /* Populate game selection pulldown */
        g_datalist_foreach(&GameConfigs, AddDataNameToCombo, cmbGame);
        gtk_combo_box_set_active(GTK_COMBO_BOX(cmbGame), 0);
    }
    else
    {
        POPUP_ERROR("Error loading game configurations");
        gtk_widget_set_sensitive (btnRefresh, FALSE);
        gtk_widget_set_sensitive (btnJoin, FALSE);
    }

    /* Enter main GTK loop */
    gtk_main();

    return 0;
}

void onQuit (void)
{
    g_datalist_clear(&MasterData);
    g_datalist_clear(&ServerData);
    gtk_main_quit();

    /* Might want to add configuration saving here */
}

void onRefresh (void)
{
    guint responded_servers;
    guint responded_masters;
    guint num_servers;
    guint num_masters;
    gchar title[80] = "";
    gchar **allservers; /* Grand list of all servers from all masters */
    GtkTreeView     *sv_view;
    GtkToggleButton *fastmode = GTK_TOGGLE_BUTTON(glade_xml_get_widget(xml, "chkFastmode"));
    GtkToggleButton *countbots = GTK_TOGGLE_BUTTON(glade_xml_get_widget(xml, "chkCountbots"));
    GtkComboBox *gamecombo = GTK_COMBO_BOX(glade_xml_get_widget(xml, "cmbGame"));
	gchar *gamename = gtk_combo_box_get_active_text(gamecombo);
    tGameConfig *gameconfig = (tGameConfig *)g_datalist_get_data(&GameConfigs, gamename);

    Open_Winsock();

    if(gameconfig == NULL)
    {
        POPUP_ERROR("Unable to get configuration for \"%s\"", gtk_combo_box_get_active_text(gamecombo));
        return;
    }

    num_masters = g_strv_length(gameconfig->masters);
    responded_masters = ServerLinkQuery(&MasterData, gameconfig->masters, gameconfig->protocol, MASTER, FALSE);

    allservers = ServerLinkMergeMasters(&MasterData, gameconfig->gameservers);

    num_servers = g_strv_length(allservers);

    g_sprintf(title,
              "%s - %d/%d masters responded - querying %d servers",
              TITLE, responded_masters, num_masters, num_servers);
    gtk_window_set_title(GTK_WINDOW(window), title);

    if(num_servers > 0)
    {
        responded_servers = ServerLinkQuery(&ServerData, allservers, gameconfig->protocol, GAME, !gtk_toggle_button_get_active(fastmode));
        g_sprintf(title,
                  "%s - %d/%d masters and %d/%d servers responded",
                  TITLE,
                  responded_masters, num_masters,
                  responded_servers, num_servers);
        gtk_window_set_title(GTK_WINDOW(window), title);

        sv_view    = GTK_TREE_VIEW(glade_xml_get_widget(xml, "trvServers"));
        AddServerDataToView(sv_view, &ServerData, gtk_toggle_button_get_active(countbots));
    }
    g_free(allservers);

    Close_Winsock();
}

void onJoin (void)
{
    gchar *ip_port_str;
    GtkTreeView *treeview;
    GtkTreeSelection *selection;
    GtkTreeModel *model;
    GtkTreeIter iter;
    gboolean result;
    GtkComboBox *gamecombo = GTK_COMBO_BOX(glade_xml_get_widget(xml, "cmbGame"));
    tGameConfig *gameconfig = (tGameConfig *)g_datalist_get_data(&GameConfigs, gtk_combo_box_get_active_text(gamecombo));

    /* Get handle on server tree view */
    treeview = GTK_TREE_VIEW(glade_xml_get_widget(xml, "trvServers"));

    selection = gtk_tree_view_get_selection(treeview);
    result = gtk_tree_selection_get_selected (selection, NULL, &iter);
    /* Check for valid iter selection */
    if(result == FALSE)
        return;

    model = gtk_tree_view_get_model (treeview);
    gtk_tree_model_get(model, &iter, 0, &ip_port_str, -1);

    if(ip_port_str != NULL)
    {
        GError *error = NULL;
        gchar *launchstring;
        gchar *command;
        gchar *path;
        gchar **argv = 0;

        launchstring = g_strdup_printf(gameconfig->launch, ip_port_str);
#ifdef WINDOWS
        command = g_strdup_printf("%s %s", gameconfig->exewin, launchstring);
        path = gameconfig->pathwin;
#endif
#ifdef LINUX
        command = g_strdup_printf("%s %s", gameconfig->exelin, launchstring);
        path = gameconfig->pathlin;
#endif

//        POPUP("Command = %s, path = %s", command, path);

        if (!g_shell_parse_argv (command,
                                 NULL, &argv,
                                 &error))
            return;

        g_spawn_sync( path,             /* gchar *working_directory         */
                      argv,             /* gchar **argv                     */
                      NULL,             /* gchar **envp                     */
                      /* For flags, could use G_SPAWN_SEARCH_PATH */
                      0,                /* GSpawnFlags flags                */
                      NULL,             /* GSpawnChildSetupFunc child_setup */
                      NULL,             /* gpointer user_data               */
                      NULL,             /* gchar **standard_output          */
                      NULL,             /* gchar **standard_error           */
                      NULL,             /* gint *exit_status                */
                      &error);          /* GError **error                   */

        if (error != NULL)
        {   /* We have a problem */
            POPUP("Launch error: %s", error->message);
            g_error_free(error);
        }

        g_strfreev (argv);
        g_free(command);
        g_free(launchstring);
    }
}

/* User selected a server in the server view - so display extra information */
void onSVChange (GtkTreeView *treeview, gpointer user_data)
{
    gchar *text;
    GtkTreeSelection *selection;
    GtkTreeModel *model;
    GtkTreeIter iter;
    gboolean result;
    GtkTreeView *pl_view;

    selection = gtk_tree_view_get_selection(treeview);
    result = gtk_tree_selection_get_selected (selection, NULL, &iter);
    /* Check for valid iter selection */

    pl_view    = GTK_TREE_VIEW(glade_xml_get_widget(xml, "trvPlayers"));

    if(result == FALSE)
    {
        /* Passing NULL clears view */
        AddPlayerDataToView(pl_view, NULL);
        DisplayServerDetail(NULL);
    }
    else
    {   /* Display further information about this server */
        tServer *server;

        model = gtk_tree_view_get_model (treeview);
        gtk_tree_model_get(model, &iter, 0, &text, -1);

        server = g_datalist_get_data(&ServerData, text);
        AddPlayerDataToView(pl_view, server);
        DisplayServerDetail(server);
    }
}

/* Sets a Gtk label widget with the server entry or, if non-existent, a default */
void DisplayServerEntry(tServer *server, gchar *label, gchar *entry, gchar *def)
{
    gchar *result;
    GtkLabel *widget = GTK_LABEL(glade_xml_get_widget(xml, label));
    if(widget == NULL)
    {
        POPUP("DisplayServerEntry(): Unknown label %s", label);
        return;
    }

    result = GetServerInfo(server, entry);
    gtk_label_set_text(widget, result == NULL ? def : result);
}

/* This can get passed NULL as the server pointer - to clear what is set */
void DisplayServerDetail(tServer *server)
{
    DisplayServerEntry(server, "lblAdmin",         "admin",         "Not set");
    DisplayServerEntry(server, "lblWebsite",       "website",       "Not set");
    DisplayServerEntry(server, "lblFragLimit",     "fraglimit",     "Not set");
    DisplayServerEntry(server, "lblTimeLimit",     "timelimit",     "Not set");
    DisplayServerEntry(server, "lblMaxPlayers",    "maxclients",    "Not set");
    DisplayServerEntry(server, "lblMaxSpectators", "maxspectators", "Not set");
    DisplayServerEntry(server, "lblVersion",       "version",       "Not set");
    DisplayServerEntry(server, "lblDmflags",       "dmflags",       "Not set");
}
