#!/bin/sh
# SPDX-FileCopyrightText: 2022 UnionTech Software Technology Co., Ltd.
#
# SPDX-License-Identifier: GPL-3.0-or-later

readonly PROJ=deepin-initramfs-enhance

# Whether in the real environment
if [ -f /bin/dde-dock ]; then
	# 虚假环境
	readonly REAL_ENV=0
	readonly LOG_FILE=/tmp/$PROJ.log
else
	readonly REAL_ENV=1
	readonly LOG_FILE=/run/initramfs/$PROJ.log
fi

log_info() {
	echo "$*" >&2
	local dt
	dt=$(date +'%F %T')
	echo "TUI: [$dt] $*" >>$LOG_FILE
}

readonly DIALOG_OK=0
readonly DIALOG_CANCEL=1
readonly DIALOG_HELP=2
readonly DIALOG_EXTRA=3
readonly DIALOG_ITEM_HELP=4
readonly DIALOG_ESC=255

reset_action() {
	ACTION_FIX_DISK=0
	ACTION_LIVE_SYSTEM=0
	ACTION_SHUTDOWN=0
	ACTION_REBOOT=0
	ACTION_SHELL=0
}

reset_action

readonly DO_CONTINUE=100
readonly HAS_ERROR=1

# return 0 or DO_CONTINUE
run_fsck() {
	local reason

	if [ "$DEEPIN_NOT_FOUND_ROOT" = 1 ]; then
		log_info 'fix disk failed, not found root dev'

		reason="$(gettext "The root partition was not found")"
		# 未找到根分区设备
		handle_fix_disk_fail "$reason"
		return
	fi

	if [ -z "$DEEPIN_ROOT_DEV" ]; then
		log_info 'fix disk failed, DEEPIN_ROOT_DEV is empty'
		reason="$(gettext "The root partition was not found")"
		# 未找到根分区设备
		handle_fix_disk_fail "$reason"
		return
	fi

	dialog --title "$(gettext "Hint")" \
		--infobox \
		"$(gettext "Repairing the file system, which needs a few minutes...")" 10 80
	# title 提示
	# 正在修复文件系统，大概需要几分钟，请稍候……

	local fix_success=0
	if [ "$REAL_ENV" = 1 ]; then
		{
			log_info run fsck -y -t "$DEEPIN_ROOT_FSTYPE" "$DEEPIN_ROOT_DEV"
			logsave -a -s $LOG_FILE fsck -y -t "$DEEPIN_ROOT_FSTYPE" "$DEEPIN_ROOT_DEV"
			local retval=$?

			if [ "$retval" -eq 1 ]; then
				# retval is 1 means filesystem errors corrected
				fix_success=1
			elif [ "$retval" -ne 0 ]; then
				# 再次检查
				logsave -a -s $LOG_FILE fsck -a -t "$DEEPIN_ROOT_FSTYPE" "$DEEPIN_ROOT_DEV"
				retval=$?
				if [ "$retval" -eq 0 ]; then
					fix_success=1
				else
					reason="$(gettext "failed to execute the fsck command")"
				fi
			fi

			# 实际 mount 测试
			if [ "$fix_success" = 1 ]; then
				mkdir /test_root_mnt
				mount -t "$DEEPIN_ROOT_FSTYPE" "$DEEPIN_ROOT_DEV" /test_root_mnt
				local mount_ret=$?
				if [ "$mount_ret" -eq 0 ]; then
					# mount root success
					umount /test_root_mnt
				else
					log_info 'test mount root failed'
					fix_success=0
					reason="$(gettext "failed to execute the mount command")"
				fi
				rmdir /test_root_mnt
			fi

		} >/dev/null 2>&1

	else
		# for debug
		sleep 1
		if [ "$ALWAYS_FAIL" = 1 ]; then
			fix_success=0
		elif [ "$ALWAYS_SUCCESS" = 1 ]; then
			fix_success=1
		else
			# 随机成功失败
			fix_success=$(shuf -i 0-1 -n 1)
		fi
	fi

	if [ "$fix_success" = 1 ]; then
		log_info 'fix disk success'
		dialog --title "$(gettext "Hint")" \
			--yes-label "$(gettext "Continue System Startup")" \
			--no-label "$(gettext "Reboot")" \
			--yesno "$(gettext "The file system was repaired successfully.")" 10 80
		# title 提示
		# 文件系统修复成功！
		# yes 继续启动系统
		# no 重启

		local retval=$?
		if [ "$retval" = $DIALOG_OK ]; then
			# continue boot
			return 0
		else
			do_reboot
			return 0
		fi
	else
		log_info 'fix disk failed'
		# 执行 fsck 命令失败
		handle_fix_disk_fail "$reason"
		return
	fi
}

handle_fix_disk_fail() {
	local reason="$1"
	local content
	content="$(gettext "Failed to repair the file system, reason: %s")"
	# 修复文件系统失败，失败原因：%s
	# shellcheck disable=SC2059
	content="$(printf "$content" "$reason")"

	local tempfile
	tempfile=$(mktemp 2>/dev/null) || tempfile="/tmp/dialog-result-$$"

	local title
	title="$(gettext "Hint")"

	local enter_shell_msg
	enter_shell_msg="$(gettext "Enter the command line mode")"
	# 进入命令行管理模式

	local reboot_msg
	reboot_msg="$(gettext "Reboot")"
	# 重启

	local shutdown_msg
	shutdown_msg="$(gettext "Shut down")"
	# 关机

	reset_action
	ACTION_SHELL=1
	ACTION_REBOOT=2
	ACTION_SHUTDOWN=3

	local confirm
	confirm="$(gettext "Confirm")"
	# 确定

	local items="\$ACTION_SHELL \"\$enter_shell_msg\" \\
		\$ACTION_REBOOT \"\$reboot_msg\" \\
		\$ACTION_SHUTDOWN \"\$shutdown_msg\""

	local dialog_cmd="dialog --title \"\$title\" --no-cancel --ok-label \"\$confirm\" \\
	 	--menu \"\$content\" 12 80 3 \\
		$items 2>\"\$tempfile\" "
	log_info cmd: "$dialog_cmd"
	eval "$dialog_cmd"

	local retval=$?
	case $retval in
	"$DIALOG_OK")
		local result
		result=$(cat "$tempfile")
		echo "Result: $result"
		handle_ok "$result"
		main_ret=$?
		;;
	"$DIALOG_CANCEL")
		log_info "user cancel, cancel"
		main_ret=$DO_CONTINUE
		# 防止 dialog 命令调用出错的情况，dialog 命令出错时退出代码也是 1
		sleep 1
		;;
	"$DIALOG_ESC")
		log_info "user cancel, esc"
		main_ret=$DO_CONTINUE
		;;
	esac

	rm "$tempfile"
	return $main_ret
}

# return 0 or DO_CONTINUE
ask_do_fix_disk() {
	dialog --clear --title "$(gettext "Hint")" \
		--yes-label "$(gettext "Confirm")" \
		--no-label "$(gettext "Back")" --yesno \
		"$(gettext "Repairing the file system may result in data loss. Are you sure you want to repair it?")" 10 80
	# title 提示
	# 修复文件系统可能会导致数据丢失，确定执行文件系统修复?
	# yes 确定
	# no 返回
	local retval=$?
	if [ "$retval" = $DIALOG_OK ]; then
		log_info 'user confirm fix'
		run_fsck
		return
	else
		return $DO_CONTINUE
	fi
}

readonly LIVE_SYSTEM_ENTRY=gnulinux-simple-recovery

go_live_system() {
	if [ -z "$BOOT_FSTYPE" ]; then
		log_info 'BOOT_FSTYPE is empty'
		return
	fi
	if [ -z "$BOOT_DEV" ]; then
		log_info 'BOOT_DEV is empty'
		return
	fi
	mkdir /boot
	mount -t "$BOOT_FSTYPE" "$BOOT_DEV" /boot

	local env_file=/boot/grub/grubenv
	if [ ! -f $env_file ]; then
		grub-editenv $env_file create
	fi
	grub-editenv $env_file set next_entry=$LIVE_SYSTEM_ENTRY

	cleanup_boot
	echo reboot
	sync
	if [ "$REAL_ENV" = 1 ]; then
		reboot -f
	fi
}

do_shutdown() {
	echo shutdown
	sync
	if [ "$REAL_ENV" = 1 ]; then
		poweroff -f
	fi
}

do_reboot() {
	echo reboot
	sync
	if [ "$REAL_ENV" = 1 ]; then
		reboot -f
	fi
}

# return 0 or DO_CONTINUE or HAS_ERROR
handle_ok() {
	local result="$1"
	case $result in
	"$ACTION_FIX_DISK")
		log_info "main menu > fix disk"
		ask_do_fix_disk
		return
		;;

	"$ACTION_LIVE_SYSTEM")
		log_info "main menu > enter live system"
		go_live_system
		;;

	"$ACTION_SHUTDOWN")
		do_shutdown
		;;

	"$ACTION_REBOOT")
		do_reboot
		;;

	"$ACTION_SHELL")
		echo shell
		return $HAS_ERROR
		;;

	esac
}

show_main_menu() {
	log_info 'call show_main_menu'
	local tempfile
	tempfile=$(mktemp 2>/dev/null) || tempfile="/tmp/dialog-result-$$"

	local main_ret=0

	local title
	title="$(gettext "System Repair")"
	# 系统修复
	local content

	# 判断原因
	if [ "$DEEPIN_FAIL_REASON" = fs ]; then
		content="$(gettext "The file system may get some errors, which leads to system startup failure. You can:")"
		# 文件系统异常，导致系统无法启动，请选择：
	else
		content="$(gettext "The system is abnormal and cannot be started. You can:")"
		# 系统异常而无法启动，请选择：
	fi

	local key_help
	key_help="$(gettext "Press the up and down arrow keys to select, and press Enter to confirm.")"
	# 按上下方向键进行选择， Enter键确认。

	content="$content\n\n$key_help"

	local fix_disk_msg
	fix_disk_msg="$(gettext "Auto repair file system")"
	# 自动修复文件系统
	local enter_live_msg
	enter_live_msg="$(gettext "Enter the live system")"
	# 进入Live系统
	local shutdown_msg
	shutdown_msg="$(gettext "Shut down")"
	# 关机
	local reboot_msg
	reboot_msg="$(gettext "Reboot")"
	# 重启
	local enter_shell_msg
	enter_shell_msg="$(gettext "Enter the command line mode")"
	# 进入命令行管理模式

	local items
	local item_count=0
	if [ "$DEEPIN_FAIL_REASON" = fs ]; then
		local item_fix_disk="\$ACTION_FIX_DISK \"\$fix_disk_msg\""
		items="$items $item_fix_disk"
		item_count=$((item_count + 1))
		ACTION_FIX_DISK=$item_count

	elif [ "$HAS_LIVE_SYSTEM" = 1 ]; then
		local item_live="\$ACTION_LIVE_SYSTEM \"\$enter_live_msg\""
		items="$items $item_live"
		item_count=$((item_count + 1))
		ACTION_LIVE_SYSTEM=$item_count
	fi

	local items_basic="\$ACTION_SHELL \"\$enter_shell_msg\" \\
		\$ACTION_REBOOT \"\$reboot_msg\" \\
		\$ACTION_SHUTDOWN \"\$shutdown_msg\""
	items="$items $items_basic"
	item_count=$((item_count + 1))
	ACTION_SHELL=$item_count
	item_count=$((item_count + 1))
	ACTION_REBOOT=$item_count
	item_count=$((item_count + 1))
	ACTION_SHUTDOWN=$item_count

	confirm="$(gettext "Confirm")"
	# 确定

	local dialog_cmd="dialog --title \"\$title\" --no-cancel --ok-label \"\$confirm\" \\
		 --menu \"\$content\" 20 80 5 \\
		$items 2>\"\$tempfile\" "
	log_info cmd: "$dialog_cmd"
	eval "$dialog_cmd"
	local retval=$?
	case $retval in
	"$DIALOG_OK")
		local result
		result=$(cat "$tempfile")
		echo "Result: $result"
		handle_ok "$result"
		main_ret=$?
		;;
	"$DIALOG_CANCEL")
		log_info "user cancel, cancel"
		main_ret=$DO_CONTINUE
		# 防止 dialog 命令调用出错的情况，dialog 命令出错时退出代码也是 1
		sleep 1
		;;
	"$DIALOG_ESC")
		log_info "user cancel, esc"
		main_ret=$DO_CONTINUE
		;;
	esac
	rm $tempfile
	return $main_ret
}

readonly EXIT_CODE_FILE=/tmp/$PROJ-tui-exit-code

do_exit() {
	local exit_code="$1"
	echo "$exit_code" >$EXIT_CODE_FILE
	if [ "$USE_KMSCON" = 1 ]; then
		killall kmscon 2>/dev/null
	fi
	exit "$exit_code"
}

cleanup_boot() {
	log_info 'call cleanup_boot'
	umount /boot || true
	rmdir /boot || true
}

main() {
	# if env IN_CJK_TERM is true, indicates that the environment of cjk terminal
	# has been entered, the cjk terminal is a terminal that can display Chinese characters.
	if [ "$IN_CJK_TERM" = 1 ]; then
		log_info now in cjk terminal

		local readonly file_fn=/scripts/functions
		if [ -f $file_fn ]; then
			# shellcheck disable=SC1090
			. $file_fn
		elif [ -f /usr/share/initramfs-tools$file_fn ]; then
			# shellcheck disable=SC1090
			. /usr/share/initramfs-tools$file_fn
		else
			log_info "not found scripts/functions file"
			exit 1
		fi

		. /bin/gettext.sh
		export TEXTDOMAIN=$PROJ
		# load locale config
		if [ -f /etc/default/locale ]; then
			. /etc/default/locale
		else
			LANG=en_US
			LANGUAGE=en_US.UTF-8
		fi
		export LANG
		export LANGUAGE

		local readonly conf_file=/etc/$PROJ.conf
		if [ -f $conf_file ]; then
			# shellcheck disable=SC1090
			. $conf_file
			# conf_file 是 hook.sh 写入的。
			# 从 conf_file 中读取 DEEPIN_HOOK_ROOT_PARTUUID, DEEPIN_HOOK_ROOT_FSTYPE,
			# DEEPIN_HOOK_BOOT_UUID。
			log_info "root partuuid: $DEEPIN_HOOK_ROOT_PARTUUID, root fstype: $DEEPIN_HOOK_ROOT_FSTYPE"
			local root_dev
			root_dev=$(blkid -t PARTUUID="$DEEPIN_HOOK_ROOT_PARTUUID" --list-one -o device)
			log_info root dev: "$root_dev"
			if [ -n "$root_dev" ] && [ "$DEEPIN_NOT_FOUND_ROOT" = 1 ]; then
				log_info found root dev
				DEEPIN_ROOT_DEV=$root_dev
				# shellcheck disable=SC2153
				DEEPIN_ROOT_FSTYPE=$DEEPIN_HOOK_ROOT_FSTYPE
				DEEPIN_NOT_FOUND_ROOT=0
			fi

			log_info "boot uuid: $DEEPIN_HOOK_BOOT_UUID"

			if [ "$DEEPIN_FAIL_REASON" != fs ]; then
				# 不是文件系统错误，探测 GRUB 的“live系统”菜单
				# NOTE: BOOT_DEV 和 BOOT_FSTYPE 在 go_live_system 函数中被使用。
				BOOT_DEV=$(blkid --uuid "$DEEPIN_HOOK_BOOT_UUID")
				log_info boot dev: "$BOOT_DEV"

				mkdir /boot
				BOOT_FSTYPE=$(get_fstype "$BOOT_DEV")
				log_info boot fstype: "$BOOT_FSTYPE"
				mount -t "$BOOT_FSTYPE" "$BOOT_DEV" /boot
				trap cleanup_boot EXIT INT TERM

				if grep -q "menuentry.*$LIVE_SYSTEM_ENTRY" /boot/grub/grub.cfg; then
					log_info 'found menuentry live system'
					HAS_LIVE_SYSTEM=1
				fi

				cleanup_boot
			fi

		fi

		while true; do
			show_main_menu
			local retval=$?

			if [ $retval = $DO_CONTINUE ]; then
				continue
			elif [ $retval = 0 ]; then
				# exit 0 会让 init 脚本的 panic 函数不继续进入 initramfs shell
				do_exit 0
			else
				# exit 1 （非零）会让 init 脚本的 panic 函数继续进入 initramfs shell
				do_exit 1
			fi
		done
	else
		log_info not in cjk terminal
		export IN_CJK_TERM=1
		# 由于 fbterm 和 kmscon 不会传递它执行的命令的 exit code, 则需要使用文件 EXIT_CODE_FILE
		# 的内容来传递 exit code。
		local run_tui
		# no_launch_cjk_term 如果为 1，则不启动 cjk term
		local no_launch_cjk_term=0

		if tty | grep -q '/dev/ttyS.*'; then
			# 比如串口调试，linux 启动参数加 console=ttyS0
			no_launch_cjk_term=1
			export LC_ALL=C
			export TERM=vt220
		elif tty | grep -q '/dev/pts/.*'; then
			# 比如在普通GUI终端，如 xterm 中执行本程序
			no_launch_cjk_term=1
		fi

		local need_sudo=0
		if [ "$no_launch_cjk_term" = 1 ]; then
			log_info no cjk terminal
			run_tui="$0"
		else
			# launch cjk term
			# cjk term 主要指 fbterm 和 kmscon
			local run_tui_suffix="exit_code=\$(cat $EXIT_CODE_FILE)
				if [ \"\$exit_code\" -eq 255 ]; then
					export LC_ALL=C
					reset
					$0 < \$(tty)
				fi"
			# term 运行失败，则 term_retval 非 0, 以英文模式在内核 vt 中运行 tui。
			# 执行 $0 < $(tty) 的原因是 fbterm 运行失败后，修改了某种未知状态，
			# 让 dialog 命令不能正常读取用户输入，直接退出了。

			if command -v fbterm >/dev/null; then
				log_info run fbterm
				export TERM=linux
				run_tui="fbterm -s 20 -n unifont $0
				$run_tui_suffix"
				need_sudo=1

			elif command -v kmscon >/dev/null; then
				log_info run kmscon
				export USE_KMSCON=1
				run_tui="kmscon --font-engine pango --font-size 18 --font-name unifont \\
			--vt 2 -t linux --no-reset-env -l $0
				$run_tui_suffix"
				need_sudo=1

			else
				# fallback 到没有 cjk term
				export LC_ALL=C
				run_tui="$0"
			fi
		fi

		# 预先设置错误的退出代码
		echo 255 >$EXIT_CODE_FILE
		if [ "$REAL_ENV" = 1 ]; then
			eval "$run_tui"
		else
			# 在虚假环境，执行 fbterm 和 kmscon 需要加 sudo
			if [ "$need_sudo" = 1 ]; then
				eval "sudo -E $run_tui"
			else
				eval "$run_tui"
			fi
		fi
		reset
		exit "$(cat "$EXIT_CODE_FILE")"
	fi
}

main
