package storage

import (
	"bytes"
	"encoding/json"
	"fmt"
	"net/http"

	"github.com/myml/cloud-blob/share"

	"github.com/pkg/errors"
)

type TokenSource interface {
	Token() string
}

// 云同步服务器签名验证
type deepinSyncAuth struct {
	cloud       string
	hwID        string
	tokenSource TokenSource
}

func (sync *deepinSyncAuth) Authorization(opts share.AuthOptions) (http.Header, error) {
	b, err := json.Marshal(&authRequest{
		Method:      opts.Method,
		Path:        opts.Path,
		ContentType: opts.ContentType,
		ContentMD5:  opts.ContentMD5,
		ListPrefix:  opts.ListPrefix,
	})
	if err != nil {
		return nil, errors.Errorf("marshal auth request")
	}
	url := fmt.Sprintf("%s/%s/signature", SyncServerAPI(), sync.cloud)
	req, err := http.NewRequest("POST", url, bytes.NewReader(b))
	if err != nil {
		return nil, errors.Wrap(err, "new request")
	}

	req.Header.Set(AuthToken, sync.tokenSource.Token())
	// Must set 'X-Machine-ID' for updating sync time
	req.Header.Set(AuthMachineID, sync.hwID)

	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		return nil, errors.Wrap(err, "http do")
	}
	defer resp.Body.Close()
	var result authResponse
	err = json.NewDecoder(resp.Body).Decode(&result)
	if err != nil {
		return nil, errors.Wrap(err, "decode auto response")
	}
	if result.Error != nil {
		return nil, errors.Errorf("Auth: %s(%d)", result.Error.Msg, result.Error.Code)
	}
	header := http.Header{}
	header.Set("Authorization", result.Auth)
	header.Set("Date", result.Date)
	if len(opts.ContentType) > 0 {
		header.Set("Content-Type", opts.ContentType)
	}
	if len(opts.ContentMD5) > 0 {
		header.Set("Content-MD5", opts.ContentMD5)
	}
	//	log.Printf("%#v\n%#v\n%#v", opts, result, header)
	return header, nil
}

type authRequest struct {
	Method      string `json:"method"`
	Path        string `json:"path"`
	ContentType string `json:"contentType"`
	ContentMD5  string `json:"contentMd5"`
	ListPrefix  string `json:"listPrefix"`
}
type authResponse struct {
	Auth  string     `json:"auth"`
	Date  string     `json:"date"`
	Error *RespError `json:"error"`
}
