/*
 * Copyright (C) 2015-2016 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <errno.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "scxml.h"

int get_names(struct scxml *scxml, char **names)
{
	struct scxml_datamodel *model;
	struct scxml_data *data;
	int n_names = 0;

	for (model = scxml->model_first; model; model = model->next) {
		for(data = model->data_first; data; data = data->next) {
			if (n_names == MAX_DATA) {
				fprintf(stderr, "datamodel has more than MAX_DATA data.\n");
				assert(0);
			}
			names[n_names++] = strdup(data->id);
		}
	}

	return n_names;
}

int get_states(struct scxml *scxml, char **states)
{
	struct scxml_state *state;
	int n_states = 0;
	for (n_states = 0, state = scxml->state_first; state; state = state->next) {
		if (n_states == MAX_STATES) {
			fprintf(stderr, "scxml has more than MAX_STATE states.\n");
			assert(0);
		}
		states[n_states++] = strdup(state->id);
	}
	return n_states;
}

void free_if(struct scxml_if *xif)
{
	struct scxml_elif *elif;
	struct scxml_elif *elif_tmp;
	struct scxml_else *xelse;
	struct scxml_else *xelse_tmp;
	struct scxml_if *subif;
	struct scxml_if *subif_tmp;
	struct scxml_assign *assign;
	struct scxml_assign *assign_tmp;
	struct scxml_extern *ex;
	struct scxml_extern *ex_tmp;

	for (subif = xif->if_first; subif; subif = subif_tmp) {
		subif_tmp = subif->next;
		free_if(subif);
	}

	for (ex = xif->ex_first; ex; ex = ex_tmp) {
		ex_tmp = ex->next;
		free(ex);
	}

	for (assign = xif->assign_first; assign; assign = assign_tmp) {
		assign_tmp = assign->next;
		free(assign);
	}

	for (elif = xif->elif_first; elif; elif = elif_tmp) {
		for (assign = elif->assign_first; assign; assign = assign_tmp) {
			assign_tmp = assign->next;
			free(assign);
		}
		for (ex = elif->ex_first; ex; ex = ex_tmp) {
			ex_tmp = ex->next;
			free(ex);
		}
		for (subif = elif->if_first; subif; subif = subif_tmp) {
			subif_tmp = subif->next;
			free_if(subif);
		}
		elif_tmp = elif->next;
		free(elif);
	}

	for (xelse = xif->else_first; xelse; xelse = xelse_tmp) {
		for (assign = xelse->assign_first; assign; assign = assign_tmp) {
			assign_tmp = assign->next;
			free(assign);
		}
		for (ex = xelse->ex_first; ex; ex = ex_tmp) {
			ex_tmp = ex->next;
			free(ex);
		}
		for (subif = xelse->if_first; subif; subif = subif_tmp) {
			subif_tmp = subif->next;
			free_if(subif);
		}
		xelse_tmp = xelse->next;
		free(xelse);
	}

	free(xif);
}

void scxml_free(struct scxml *scxml)
{
	struct scxml_state *state;
	struct scxml_state *state_tmp;
	struct scxml_trans *trans;
	struct scxml_trans *trans_tmp;
	struct scxml_assign *assign;
	struct scxml_assign *assign_tmp;
	struct scxml_extern *ex;
	struct scxml_extern *ex_tmp;
	struct scxml_if *xif;
	struct scxml_if *xif_tmp;
	struct scxml_onentry *onentry;
	struct scxml_onentry *onentry_tmp;
	struct scxml_onexit *onexit;
	struct scxml_onexit *onexit_tmp;
	struct scxml_datamodel *model;
	struct scxml_datamodel *model_tmp;
	struct scxml_data *data;
	struct scxml_data *data_tmp;
	struct scxml_function *func;
	struct scxml_function *func_tmp;

	for (func = scxml->func_first; func; func = func_tmp) {
		for (model = func->model_first; model; model = model_tmp) {
			for (data = model->data_first; data; data = data_tmp) {
				data_tmp = data->next;
				free(data);
			}
			model_tmp = model->next;
			free(model);
		}
		func_tmp = func->next;
		free(func);
	}

	for (model = scxml->model_first; model; model = model_tmp) {
		for (data = model->data_first; data; data = data_tmp) {
			data_tmp = data->next;
			free(data);
		}
		model_tmp = model->next;
		free(model);
	}

	for (state = scxml->state_first; state; state = state_tmp) {
		for (onentry = state->onentry_first; onentry; onentry= onentry_tmp) {
			for (xif = onentry->if_first; xif; xif = xif_tmp) {
				xif_tmp = xif->next;
				free_if(xif);
			}
			for (ex = onentry->ex_first; ex; ex = ex_tmp) {
				ex_tmp = ex->next;
				free(ex);
			}
			for (assign = onentry->assign_first; assign; assign = assign_tmp) {
				assign_tmp = assign->next;
				free(assign);
			}
			onentry_tmp = onentry->next;
			free(onentry);
		}
		for (xif = state->if_first; xif; xif = xif_tmp) {
			xif_tmp = xif->next;
			free_if(xif);
		}
		for (assign = state->assign_first; assign; assign = assign_tmp) {
			assign_tmp = assign->next;
			free(assign);
		}
		for (ex = state->ex_first; ex; ex = ex_tmp) {
			ex_tmp = ex->next;
			free(ex);
		}
		for (trans = state->trans_first; trans; trans = trans_tmp) {
			trans_tmp = trans->next;
			free(trans);

		}
		for (onexit = state->onexit_first; onexit; onexit= onexit_tmp) {
			for (xif = onexit->if_first; xif; xif = xif_tmp) {
				xif_tmp = xif->next;
				free_if(xif);
			}
			for (ex = onexit->ex_first; ex; ex = ex_tmp) {
				ex_tmp = ex->next;
				free(ex);
			}
			for (assign = onexit->assign_first; assign; assign = assign_tmp) {
				assign_tmp = assign->next;
				free(assign);
			}
			onexit_tmp = onexit->next;
			free(onexit);
		}
		state_tmp = state->next;
		free(state);
	}
	free(scxml);
}

int check_balance(const char *cond, char open, char close) {
	int i;
	int cnt;
	if (!cond) {
		return 0;
	}
	for (i = 0, cnt = 0; cond[i] != '\0'; i++) {
		if (cond[i] == open) {
			cnt++;
		} else if (cond[i] == close) {
			cnt--;
			if (cnt < 0) {
				fprintf(stderr, "parantheses error for () in boolean:\n%s\n", cond);
				assert(0);
			}
		}
	}
	if (cnt != 0) {
		fprintf(stderr, "parantheses error for () in boolean:\n%s\n", cond);
		assert(0);
	}
	return 0;
}

void check_cond(char **names, int n_names, char **states, int n_states, const char *cond)
{
	int i;
	char tmp[MAX_BOOL];
	char *elements[MAX_BOOL / 2];
	static const char *delim = " \n\t!()[]{}";
	if (!cond) {
		return;
	}
	strncpy(tmp, cond, MAX_BOOL);
	for (i = 0, elements[i] = strtok(tmp, delim); elements[i]; elements[++i] = strtok(NULL, delim)) {
		if (i % 2 != 0) {
			if (strcmp(elements[i], "and") != 0
			    && strcmp(elements[i], "or") != 0) {
				fprintf(stderr, "error in boolean:\n%s\nelement %i should be 'or' or 'and' but is '%s'\n", cond, i, elements[i]);
				//assert(0);
			}
		} else {
			int j;
			if (strncmp("In", elements[i], 2) == 0) {
				/* get the 'in' state without '' */
				char *state_name;
				state_name = strtok(NULL, delim);
				assert(state_name);
				if (!strrchr(state_name, '\'')) {
					fprintf(stderr, "syntax error in 'in' statement, missing '\n");
					assert(0);
				}
				*strrchr(state_name, '\'') = '\0';

				/* check if the questioned state is present */
				for (j = 0; j < n_states; j++) {
					if (strcmp(states[j], state_name + 1) == 0) {
						break;
					}
				}
				if (j == n_states) {
					fprintf(stderr, "error in boolean:\n%s\n state %s not a valid state\n", cond, state_name + 1);
					assert(0);
				}
			} else {
				int len;
				/* strip ==$value from name */
				if (!strchr(elements[i], '=')) {
					fprintf(stderr, "error in boolean:\n%s no = found in comparison %s\n", cond, elements[i]);
					assert(0);
				}
				*strchr(elements[i], '=') = '\0';

				len = strstr(elements[i], "'name'") - elements[i];
				if (len <= 0) {
					len = strlen(elements[i]);
				}
				/* check if questioned name is known */
				for (j = 0; j < n_names; j++ ) {
					if (strncmp(names[j], elements[i], len) == 0) {
						break;
					}
				}
				if (j == n_names) {
					fprintf(stderr, "error in boolean:\n%s\n%s not found in datamodel\n", cond, elements[i]);
					assert(0);
				}
			}
		}
	}
}

void check_trans(char **names, int n_names, char **states, int n_states, struct scxml_trans *trans, struct counter *c)
{
	check_balance(trans->cond, '(', ')');
	check_balance(trans->cond, '[', ']');
	check_balance(trans->cond, '{', '}');

	trans->count = c->ntrans++;

	check_cond(names, n_names, states, n_states, trans->cond);
}

void check_else(char **, int, char **, int, struct scxml_else *, struct counter *);
void check_elif (char **, int, char **, int, struct scxml_elif *, struct counter *);

void check_if (char **names, int n_names, char **states, int n_states, struct scxml_if *xif, struct counter *c)
{
	struct scxml_if *subif;
	struct scxml_elif *elseif;
	struct scxml_else *xelse;

	xif->count = c->nif++;

	check_balance(xif->cond, '(', ')');
	check_balance(xif->cond, '[', ']');
	check_balance(xif->cond, '{', '}');
	check_cond(names, n_names, states, n_states, xif->cond);

	for (subif = xif->if_first; subif; subif = subif->next) {
		check_if (names, n_names, states, n_states, subif, c);
	}
	for (elseif = xif->elif_first; elseif; elseif = elseif->next) {
		check_elif (names, n_names, states, n_states, elseif, c);
	}
	for (xelse = xif->else_first; xelse; xelse = xelse->next) {
		check_else(names, n_names, states, n_states, xelse, c);
	}
}

void check_else(char **names, int n_names, char **states, int n_states, struct scxml_else *xelse, struct counter *c)
{
	struct scxml_if *xif;

	for (xif = xelse->if_first; xif; xif = xif->next) {
		check_if (names, n_names, states, n_states, xif, c);
	}
}

void check_elif(char **names, int n_names, char **states, int n_states, struct scxml_elif *elif, struct counter *c)
{
	struct scxml_if *xif;

	check_balance(elif->cond, '(', ')');
	check_balance(elif->cond, '[', ']');
	check_balance(elif->cond, '{', '}');
	check_cond(names, n_names, states, n_states, elif->cond);

	for (xif = elif->if_first; xif; xif = xif->next) {
		check_if (names, n_names, states, n_states, xif, c);
	}
}

void check_onentry(char **names, int n_names, char **states, int n_states, struct scxml_onentry *onentry, struct counter *c)
{
	struct scxml_if *xif;

	for (xif = onentry->if_first; xif; xif = xif->next) {
		check_if (names, n_names, states, n_states, xif, c);
	}
}

void check_onexit(char **names, int n_names, char **states, int n_states, struct scxml_onexit *onexit, struct counter *c)
{
	struct scxml_if *xif;

	for (xif = onexit->if_first; xif; xif = xif->next) {
		check_if (names, n_names, states, n_states, xif, c);
	}
}

void check_booleans(struct scxml *scxml)
{
	struct scxml_state *state;
	struct scxml_if *xif;
	struct scxml_onentry *onentry;
	struct scxml_onexit *onexit;
	struct scxml_trans *trans;

	char *names[MAX_DATA];
	int n_names = 0;
	char *states[MAX_STATES];
	int n_states = 0;
	int i;

	n_names = get_names(scxml, names);

	n_states = get_states(scxml, states);
	for (state = scxml->state_first; state; state = state->next) {

		struct counter c;
		c.nif = 0;
		c.nelif = 0;
		c.ntrans = 0;

		for (onentry = state->onentry_first; onentry; onentry = onentry->next) {
			check_onentry(names, n_names, states, n_states, onentry, &c);
		}
		for (trans = state->trans_first; trans; trans = trans->next) {
			check_trans(names, n_names, states, n_states, trans, &c);
		}
		for (xif = state->if_first; xif; xif = xif->next) {
			check_if (names, n_names, states, n_states, xif, &c);
		}
		for (onexit = state->onexit_first; onexit; onexit = onexit->next) {
			check_onexit(names, n_names, states, n_states, onexit, &c);
		}
	}
	for (i = 0; i < n_states; i++) {
		free(states[i]);
	}
	for (i = 0; i < n_names; i++) {
		free(names[i]);
	}
}

struct scxml_extern* extern_get(struct xml *x1)
{
	struct scxml_extern *ex;

	ex = malloc(sizeof(*ex));
	assert(ex);
	memset(ex, 0, sizeof(*ex));

	if (strcmp(x1->start.string, "extern") != 0) {
		fprintf(stderr, "%s found as extern in datamodel\n", x1->start.string);
		assert(0);
	}

	ex->input = xml_attr_lookup(x1, "input");
	ex->output = xml_attr_lookup(x1, "output");
	if (!ex->input && !ex->output) {
		fprintf(stderr, "neither input nor output in extern\n");
		assert(0);
	}

	return ex;
}


struct scxml_data* data_get(struct xml *x1)
{
	struct scxml_data *data;
	const char *tmp;

	data = malloc(sizeof(*data));
	assert(data);
	memset(data, 0, sizeof(*data));

	if (strcmp(x1->start.string, "data") != 0) {
		fprintf(stderr, "%s found as data in datamodel\n", x1->start.string);
		assert(0);
	}

	data->id = xml_attr_lookup(x1, "id");
	assert(data->id);
	tmp = xml_attr_lookup(x1, "expr");
	assert(tmp);
	data->expr = atoi(tmp);
	data->flags = xml_attr_lookup(x1, "flags");
	/* no assert here, flags are optional */
	data->bytes = xml_attr_lookup(x1, "bytes");
	/* bytes are optional as well */

	return data;
}

struct scxml_datamodel* datamodel_get(struct xml *x0)
{
	struct xml *x1;
	struct scxml_datamodel *model;
	struct scxml_data *data;

	model = malloc(sizeof(*model));
	assert(model);
	memset(model, 0, sizeof(*model));

	assert(strcmp(x0->start.string, "datamodel") == 0);

	model->name = xml_attr_lookup(x0, "name");
	assert(model->name);

	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		    && x1->type != XML_START_END) continue;

		data = data_get(x1);

		data->prev = model->data_last;
		data->next = NULL;
		if (data->prev) {
			data->prev->next = data;
		} else {
			model->data_first = data;
		}
		model->data_last = data;
	}

	return model;
}

struct scxml_function* function_get(struct xml *x0)
{
	struct xml *x1;
	struct scxml_function *func;
	struct scxml_datamodel *model;

	func = malloc(sizeof(*func));
	assert(func);
	memset(func, 0, sizeof(*func));

	func->name = xml_attr_lookup(x0, "name");
	assert(func->name);

	assert(strcmp(x0->start.string, "function") == 0);

	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		    && x1->type != XML_START_END) continue;

		model = datamodel_get(x1);

		model->prev = func->model_last;
		model->next = NULL;
		if (model->prev) {
			model->prev->next = model;
		} else {
			func->model_first = model;
		}
		func->model_last = model;
	}

	return func;
}

struct scxml_assign* assign_get(struct xml *x1)
{
	struct scxml_assign *assign;

	assign = malloc(sizeof(*assign));
	assert(assign);
	memset(assign, 0, sizeof(*assign));

	if (strcmp(x1->start.string, "assign") != 0) {
		fprintf(stderr, "%s treated as assignment\n", x1->start.string);
		assert(0);
	}

	assign->location = xml_attr_lookup(x1, "location");
	assert(assign->location);
	assign->expr = xml_attr_lookup(x1, "expr");
	assert(assign->expr);

	return assign;
}

struct scxml_trans* trans_get(struct xml *x1)
{
	struct scxml_trans *trans;

	trans = malloc(sizeof(*trans));
	assert(trans);
	memset(trans, 0, sizeof(*trans));

	if (strcmp(x1->start.string, "transition") != 0) {
		fprintf(stderr, "%s treated as transition\n", x1->start.string);
		assert(0);
	}

	trans->event = xml_attr_lookup(x1, "event");
	assert(trans->event);
	trans->cond = xml_attr_lookup(x1, "cond");

	trans->count = -1;

	/* condition is optional assert(trans->cond); */
	trans->target = xml_attr_lookup(x1, "target");
	assert(trans->target);

	return trans;
}

/* forward declaration */
struct scxml_if* if_get(struct xml *);

struct scxml_onexit* onexit_get(struct xml *x0)
{
	struct scxml_onexit *onexit;
	struct scxml_if *subif;
	struct scxml_assign *assign;
	struct scxml_extern *ex;
	struct xml *x1;

	onexit = malloc(sizeof(*onexit));
	assert(onexit);
	memset(onexit, 0, sizeof(*onexit));

	if (strcmp(x0->start.string, "onexit") != 0) {
		fprintf(stderr, "%s treated as onexit\n", x0->start.string);
		assert(0);
	}

	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		    && x1->type != XML_START_END) continue;
		else if (strcmp(x1->start.string, "assign") == 0) {
			assign = assign_get(x1);

			assign->prev = onexit->assign_last;
			assign->next = NULL;
			if (assign->prev) {
				assign->prev->next = assign;
			} else {
				onexit->assign_first = assign;
			}
			onexit->assign_last = assign;
		} else if (strcmp(x1->start.string, "extern") == 0) {
			ex = extern_get(x1);

			ex->prev = onexit->ex_last;
			ex->next = NULL;
			if (ex->prev) {
				ex->prev->next = ex;
			} else {
				onexit->ex_first = ex;
			}
			onexit->ex_last = ex;
		} else if (strcmp(x1->start.string, "if") == 0) {
			subif = if_get(x1);

			subif->prev = onexit->if_last;
			subif->next = NULL;
			if (subif->prev) {
				subif->prev->next = subif;
			} else {
				onexit->if_first = subif;
			}
			onexit->if_last = subif;
		} else {
			fprintf(stderr, "Unknown scxml tag found in onexit: %s\n", x1->start.string);
			assert(0);
		}
	}
	return onexit;
}

struct scxml_onentry* onentry_get(struct xml *x0)
{
	struct scxml_onentry *onentry;
	struct scxml_if *subif;
	struct scxml_assign *assign;
	struct scxml_extern *ex;
	struct xml *x1;

	onentry = malloc(sizeof(*onentry));
	assert(onentry);
	memset(onentry, 0, sizeof(*onentry));

	if (strcmp(x0->start.string, "onentry") != 0) {
		fprintf(stderr, "%s treated as onentry\n", x0->start.string);
		assert(0);
	}

	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		    && x1->type != XML_START_END) continue;
		else if (strcmp(x1->start.string, "assign") == 0) {
			assign = assign_get(x1);

			assign->prev = onentry->assign_last;
			assign->next = NULL;
			if (assign->prev) {
				assign->prev->next = assign;
			} else {
				onentry->assign_first = assign;
			}
			onentry->assign_last = assign;

		} else if (strcmp(x1->start.string, "extern") == 0) {
			ex = extern_get(x1);

			ex->prev = onentry->ex_last;
			ex->next = NULL;
			if (ex->prev) {
				ex->prev->next = ex;
			} else {
				onentry->ex_first = ex;
			}
			onentry->ex_last = ex;
		} else if (strcmp(x1->start.string, "if") == 0) {
			subif = if_get(x1);

			subif->prev = onentry->if_last;
			subif->next = NULL;
			if (subif->prev) {
				subif->prev->next = subif;
			} else {
				onentry->if_first = subif;
			}
			onentry->if_last = subif;

		} else {
			fprintf(stderr, "Unknown scxml tag found in onentry: %s\n", x1->start.string);
			assert(0);
		}
	}
	return onentry;
}

struct scxml_else* else_get(struct xml *x0)
{
	struct scxml_else *xelse;
	struct scxml_if *subif;
	struct scxml_assign *assign;
	struct scxml_extern *ex;
	struct xml *x1;

	xelse = malloc(sizeof(*xelse));
	assert(xelse);
	memset(xelse, 0, sizeof(*xelse));

	if (strcmp(x0->start.string, "else") != 0) {
		fprintf(stderr, "%s treated as else\n", x0->start.string);
		assert(0);
	}

	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		    && x1->type != XML_START_END) continue;
		else if (strcmp(x1->start.string, "assign") == 0) {
			assign = assign_get(x1);

			assign->prev = xelse->assign_last;
			assign->next = NULL;
			if (assign->prev) {
				assign->prev->next = assign;
			} else {
				xelse->assign_first = assign;
			}
			xelse->assign_last = assign;

		} else if (strcmp(x1->start.string, "extern") == 0) {
			ex = extern_get(x1);

			ex->prev = xelse->ex_last;
			ex->next = NULL;
			if (ex->prev) {
				ex->prev->next = ex;
			} else {
				xelse->ex_first = ex;
			}
			xelse->ex_last = ex;
		} else if (strcmp(x1->start.string, "if") == 0) {
			subif = if_get(x1);

			subif->prev = xelse->if_last;
			subif->next = NULL;
			if (subif->prev) {
				subif->prev->next = subif;
			} else {
				xelse->if_first = subif;
			}
			xelse->if_last = subif;

		} else {
			fprintf(stderr, "Unknown scxml tag found in else: %s\n", x1->start.string);
			assert(0);
		}
	}
	return xelse;
}

struct scxml_elif* elif_get(struct xml *x0)
{
	struct scxml_if *subif;
	struct scxml_elif *elif;
	struct scxml_assign *assign;
	struct scxml_extern *ex;
	struct xml *x1;

	elif = malloc(sizeof(*elif));
	assert(elif);
	memset(elif, 0, sizeof(*elif));

	if (strcmp(x0->start.string, "elseif") != 0) {
		fprintf(stderr, "%s treated as elseif\n", x0->start.string);
		assert(0);
	}

	elif->cond = xml_attr_lookup(x0, "cond");
	assert(elif->cond);

	elif->count = -1;

	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		    && x1->type != XML_START_END) continue;
		else if (strcmp(x1->start.string, "assign") == 0) {
			assign = assign_get(x1);

			assign->prev = elif->assign_last;
			assign->next = NULL;
			if (assign->prev) {
				assign->prev->next = assign;
			} else {
				elif->assign_first = assign;
			}
			elif->assign_last = assign;

		} else if (strcmp(x1->start.string, "extern") == 0) {
			ex = extern_get(x1);

			ex->prev = elif->ex_last;
			ex->next = NULL;
			if (ex->prev) {
				ex->prev->next = ex;
			} else {
				elif->ex_first = ex;
			}
			elif->ex_last = ex;
		} else if (strcmp(x1->start.string, "if") == 0) {
			subif = if_get(x1);

			subif->prev = elif->if_last;
			subif->next = NULL;
			if (subif->prev) {
				subif->prev->next = subif;
			} else {
				elif->if_first = subif;
			}
			elif->if_last = subif;

		} else {
			fprintf(stderr, "Unknown scxml tag found in else: %s\n", x1->start.string);
			assert(0);
		}
	}
	return elif;
}


struct scxml_if* if_get(struct xml *x0)
{
	struct xml *x1;
	struct scxml_if *xif;
	struct scxml_assign *assign;
	struct scxml_extern *ex;
	struct scxml_elif *elif;
	struct scxml_else *xelse;
	struct scxml_if *subif;

	xif = malloc(sizeof(*xif));
	assert(xif);
	memset(xif, 0, sizeof(*xif));

	xif->cond = xml_attr_lookup(x0, "cond");
	assert(xif->cond);

	xif->count = -1;

	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		    && x1->type != XML_START_END) continue;
		else if (strcmp(x1->start.string, "extern") == 0) {
			ex = extern_get(x1);

			ex->prev = xif->ex_last;
			ex->next = NULL;
			if (ex->prev) {
				ex->prev->next = ex;
			} else {
				xif->ex_first = ex;
			}
			xif->ex_last = ex;
		} else if (strcmp(x1->start.string, "assign") == 0) {
			assign = assign_get(x1);

			assign->prev = xif->assign_last;
			assign->next = NULL;
			if (assign->prev) {
				assign->prev->next = assign;
			} else {
				xif->assign_first = assign;
			}
			xif->assign_last = assign;
		} else if (strcmp(x1->start.string, "elseif") == 0) {
			elif = elif_get(x1);

			elif->prev = xif->elif_last;
			elif->next = NULL;
			if (elif->prev) {
				elif->prev->next = elif;
			} else {
				xif->elif_first = elif;
			}
			xif->elif_last = elif;
		} else if (strcmp(x1->start.string, "else") == 0) {
			xelse = else_get(x1);

			xelse->prev = xif->else_last;
			xelse->next = NULL;
			if (xelse->prev) {
				fprintf(stderr, "found multiple else for one if condition\n");
				assert(0);
			} else {
				xif->else_first = xelse;
			}
			xif->else_last = xelse;
		} else if (strcmp(x1->start.string, "if") == 0) {
			subif = if_get(x1);

			subif->prev = xif->if_last;
			subif->next = NULL;
			if (subif->prev) {
				subif->prev->next = subif;
			} else {
				xif->if_first = subif;
			}
			xif->if_last = subif;

		} else {
			fprintf(stderr, "Unknown scxml tag found in if condition: %s\n",
				x1->start.string);
			assert(0);
		}
	}

	return xif;
}

struct scxml_state* state_get(struct xml *x0)
{
	struct xml *x1;
	struct scxml_state *state;
	struct scxml_trans *trans;
	struct scxml_assign *assign;
	struct scxml_extern *ex;
	struct scxml_if *xif;
	struct scxml_onentry *onentry;
	struct scxml_onexit *onexit;

	state = malloc(sizeof(*state));
	assert(state);
	memset(state, 0, sizeof(*state));

	state->id = xml_attr_lookup(x0, "id");
	assert(state->id);

	for (x1 = x0->start.child_first; x1; x1 = x1->next) {
		if (x1->type != XML_START
		    && x1->type != XML_START_END) continue;
		else if (strcmp(x1->start.string, "extern") == 0) {
			ex = extern_get(x1);

			ex->prev = state->ex_last;
			ex->next = NULL;
			if (ex->prev) {
				ex->prev->next = ex;
			} else {
				state->ex_first = ex;
			}
			state->ex_last = ex;
		} else if (strcmp(x1->start.string, "transition") == 0) {
			trans = trans_get(x1);

			trans->prev = state->trans_last;
			trans->next = NULL;
			if (trans->prev) {
				trans->prev->next = trans;
			} else {
				state->trans_first = trans;
			}
			state->trans_last = trans;
		} else if (strcmp(x1->start.string, "assign") == 0) {
			assign = assign_get(x1);

			assign->prev = state->assign_last;
			assign->next = NULL;
			if (assign->prev) {
				assign->prev->next = assign;
			} else {
				state->assign_first = assign;
			}
			state->assign_last = assign;
		} else if (strcmp(x1->start.string, "if") == 0) {
			xif = if_get(x1);

			xif->prev = state->if_last;
			xif->next = NULL;
			if (xif->prev) {
				xif->prev->next = xif;
			} else {
				state->if_first = xif;
			}
			state->if_last = xif;
		} else if (strcmp(x1->start.string, "onentry") == 0) {
			onentry = onentry_get(x1);

			onentry->prev = state->onentry_last;
			onentry->next = NULL;
			if (onentry->prev) {
				onentry->prev->next = onentry;
			} else {
				state->onentry_first = onentry;
			}
			state->onentry_last = onentry;
		} else if (strcmp(x1->start.string, "onexit") == 0) {
			onexit = onexit_get(x1);

			onexit->prev = state->onexit_last;
			onexit->next = NULL;
			if (onexit->prev) {
				onexit->prev->next = onexit;
			} else {
				state->onexit_first = onexit;
			}
			state->onexit_last = onexit;
		} else {
			fprintf(stderr, "Unknown scxml tag found in state %s: %s\n"
				, state->id, x1->start.string);
			assert(0);
		}
	}
	return state;
}

struct scxml* scxml_read(struct xml *xml)
{
	struct scxml *scxml;
	assert(xml);

	struct xml *x1;
	//struct xml *x2;

	assert(xml->type == XML_START
	       || xml->type == XML_START_END);
	assert(strcmp(xml->start.string, "scxml") == 0);

	scxml = malloc(sizeof(*scxml));
	assert(scxml);
	memset(scxml, 0, sizeof(*scxml));

	scxml->name = xml->start.string;
	scxml->type = xml_attr_lookup(xml, "type");
	assert(scxml->type);
	scxml->initial = xml_attr_lookup(xml, "initial");
	assert(scxml->initial);

	for (x1 = xml->start.child_first; x1; x1 = x1->next) {
		if (strcmp(x1->start.string, "datamodel") == 0) {
			struct scxml_datamodel *model;

			model = datamodel_get(x1);

			model->prev = scxml->model_last;
			model->next = NULL;
			if (model->prev) {
				model->prev->next = model;
			} else {
				scxml->model_first = model;
			}
			scxml->model_last = model;
		} else if (strcmp(x1->start.string, "state") == 0) {
			struct scxml_state *state;

			state = state_get(x1);

			state->prev = scxml->state_last;
			state->next = NULL;
			if (state->prev) {
				state->prev->next = state;
			} else {
				scxml->state_first = state;
			}
			scxml->state_last = state;
		} else if (strcmp(x1->start.string, "function") == 0) {
			struct scxml_function *func;

			func = function_get(x1);

			func->prev = scxml->func_last;
			func->next = NULL;
			if (func->prev) {
				func->prev->next = func;
			} else {
				scxml->func_first = func;
			}
			scxml->func_last = func;
		} else {
			fprintf(stderr, "Unknown scxml tag found: %s\n", x1->start.string);
			assert(0);
		}
	}

	check_booleans(scxml);

	return scxml;
}
