/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_voms_poolaccount.mod voms poolaccount plugin

    \section vomspoolaccountsyn SYNOPSIS
    \b lcmaps_voms_poolaccount.mod
        [-gridmapfile|-GRIDMAPFILE|-gridmap|-GRIDMAP \<location gridmapfile\>]
        [-gridmapdir|-GRIDMAPDIR \<location gridmapdir\>]
        [-override_inconsistency|-OVERRIDE_INCONSISTENCY]
        [-max_mappings_per_credential \<max nr of mappings\>]
        [-do_not_use_secondary_gids]
        [-do_not_require_primary_gid]

    \section vomspoolaccountdesc DESCRIPTION

    This poolaccount acquisition plugin is a 'VOMS-aware' modification of the 'poolaccount' plugin.
    The plugin tries to find a poolaccount (more specifically a UID) based on the VOMS information
    that has been retrieved by the plugin \ref lcmaps_voms.mod "lcmaps_voms.mod"
    from the user's grid credential.
    It will try to match a VO-GROUP-ROLE combination from the user's grid credential with an entry
    in a gridmapfile (most likely the traditional gridmapfile, used by the localaccount and
    poolaccount plugins)
    In this file VO-GROUP-ROLE combinations are listed with a poolaccount entry, as shown in the
    following example.

    EXAMPLE:

    \c "/VO=wilma/GROUP=*" \c .wilma

    \c "/VO=fred/GROUP=*"  \c .fred

    If the first matching VO-GROUP-ROLE combination is \c "/VO=wilma/GROUP=*" the plugin will
    get a poolaccount from the '.test' pool.
    This could result in 'wilma001' as a poolaccount for this user.
    The linking between \c "/VO=wilma/GROUP=*", this user and a poolaccount must be made in
    the same directory as the for the \e 'poolaccount' plugin (the \e gridmapdir),
    otherwise it gives rise to inconsistancies when both are used on a site.
    The actual account assigned to the user is based on his VO information matched in the
    gridmapfile, the user's DN and the primary (and secondary) GIDs gathered so far.
    In the \e gridmapdir directory this is reflected in the leasename, which consists of the
    url-encoded DN + a concatenation of the gathered groupnames.
    So a lease name could look like this:

    EXAMPLE DN with pool/localgroups attached:
    \c %2fo%3ddutchgrid%2fo%3dusers%2fo%3dnikhef%2fcn%3dmartijn%20steenbakkers:pool001:bogus1:bogus2:bogus3:pool003:pool004:pool005

    If a user changes his VO-GROUP-ROLE combinations (but not his VO), in this case he will be mapped to
    a different account (UID) within the same pool.

    \section vomspoolaccountnote1 NOTE 1
        This plugin should only be used in combination with the \e 'voms_localgroup'
        and/or \e 'voms_poolgroup' plugins.

    \section vomspoolaccountnote2 NOTE 2
        The options '-do_not_require_primary_gid' and '-do_not_use_secondary_gids'
        can not be used together, because at least one GID is needed.

    \section vomspoolaccountoptions OPTIONS
    \subsection vomspoolaccountoptie1 -GRIDMAPFILE \<gridmapfile\>
        See \ref vomspoolaccountoptie4 "-gridmap"

    \subsection vomspoolaccountoptie2 -gridmapfile \<gridmapfile\>
        See \ref vomspoolaccountoptie4 "-gridmap"

    \subsection vomspoolaccountoptie3 -GRIDMAP \<gridmapfile\>
        See \ref vomspoolaccountoptie4 "-gridmap"

    \subsection vomspoolaccountoptie4 -gridmap \<gridmapfile\>
        When this option is set it will override the default path to the gridmapfile.
        It is advised to use an absolute path to the gridmapfile to avoid usage of the wrong file(path).

    \subsection vomspoolaccountoptie5 -GRIDMAPDIR \<gridmapdir\>
        See \ref vomspoolaccountoptie6 "-gridmapdir"

    \subsection vomspoolaccountoptie6 -gridmapdir \<gridmapdir\>
        If this option is set, it will override the default path to the gridmapdir.
        It is advised to use an absolute path to the gridmapdir to avoid usage of the wrong path.

    \subsection vomspoolaccountoptie7 -do_not_use_secondary_gids
        The determination of the poolaccount will not be based on the secondary GIDs found, but only
        on the user's DN, the VOMS info for the user and the primary GID that has been found.
        Cannot be used with \ref vomspoolaccountoptie8 "-do_not_require_primary_gid".

    \subsection vomspoolaccountoptie8 -do_not_require_primary_gid
        The determination of the poolaccount will not be based on the primary GID found, but only
        on the user's DN, the VOMS info for the user and the secondary GIDs found.
        Normally this option should not be used, but it can be useful for debugging.
        Cannot be used with \ref vomspoolaccountoptie7 "-do_not_use_secondary_gids".

    \subsection vomspoolaccountoptie9 -OVERRIDE_INCONSISTENCY
        See \ref vomspoolaccountoptie10 "-override_inconsistency"

    \subsection vomspoolaccountoptie10 -override_inconsistency
        Moving a user from one pool to another (because of a VO change)
        should only be done by changing the gridmapfile indicating the new pool for this user.
        If a user has already been mapped previously to a poolaccount, there is a link present
        between this poolaccount and his DN.
        In the good old days prior to LCMAPS, a 'pool change' would still result in a mapping to
        the old pool account, neglecting the administrative changes in the gridmapfile.
        LCMAPS corrects this behaviour:
        By default the voms_poolaccount plugin will \e fail if the pool designated by the gridmapfile
        doesn't match the previously mapped voms_poolaccount leasename.
        If the site doesn't want a failure on this inconsistency it can turn on this parameter.
        When the inconsistency is detected the plugin will automatically unlink the previous mapping
        and will proceed by making a \e new lease from the new pool.

    \subsection vomspoolaccountoptie11 -max_mappings_per_credential \<max nr of mappings\>
        This value indicates the maximum number of accounts a user, or more specifically
        a set of credentials (=DN + FQANS), can be mapped to. Normally this number is 1.
        But if each job should run under its own account the number should be increased.
        The leasename (or poolindex) in this case looks like:
                url_encoded(<DN>):gid1[:gid2[:gid3[...]]]:mapcount=<mapnumber>)

    \subsection vomspoolaccountoptie11 -strict_poolprefix_match [yes|no]. Default is 'yes'.
        If this is set to 'yes', a line in the gridmapfile like
        <FQAN> .pool
        will result in accounts matching the regexp 'pool[0-9]+'.
        Otherwise it will be allowed to match 'pool.*' (legacy behaviour).

\section vomspoolaccountReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure


\section vomspoolaccountErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)

\section vomspoolaccountSeeAlso SEE ALSO
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod",
        \ref lcmaps_voms_poolgroup.mod "lcmaps_voms_poolgroup.mod",
        \ref lcmaps_localaccount.mod "lcmaps_localaccount.mod",
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
*/

/*!
    \file   lcmaps_voms_poolaccount.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code of the voms_poolaccount plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

/* Define both _XOPEN_SOURCE for strdup and _GNU_SOURCE, this makes Solaris
 * happier */
#define _XOPEN_SOURCE	500
/* Try to provide RTLD_DEFAULT */
/* MacOS needs no POSIX or _DARWIN_C_SOURCE to have RTLD_DEFAULT */
#ifdef __APPLE__
# define _DARWIN_C_SOURCE
#else
# define _GNU_SOURCE
#endif
#include <dlfcn.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <errno.h>

#include "lcmaps_voms_config.h"

#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>

#if defined(HAVE_LCMAPS_LCMAPS_PLUGIN_PROTOTYPES_H)
#   include <lcmaps/lcmaps_plugin_prototypes.h>
#else
#   include "lcmaps_plugin_prototypes.h"
#endif

#include "lcmaps_gridmapdir.h"


/************************************************************************
 * defines
 ************************************************************************/

#define PLUGIN_PREFIX	"lcmaps_voms_poolaccount"

#define PLUGIN_RUN	0   /* full run mode */
#define PLUGIN_VERIFY	1   /* verify-only mode */


#define LCMAPS_DEFAULT_MAPPING_MAX 1
#define LCMAPS_ABSOLUTE_MAPPING_MAX 9999

#define INITBUF_SIZE 256

/************************************************************************
 * global variables
 ************************************************************************/

static char *gridmapfile = NULL; /* filename of gridmapfile */
static char *gridmapdir = NULL; /* dirname of gridmapdir */
static int  override_inconsistency  = 0;
static int  mapping_max             = LCMAPS_DEFAULT_MAPPING_MAX;
static int  strict_poolprefix_match = 1; /* By default strict matching */

static int  use_secondary_gids      = 1; /* use sGIDs in leasename */

static int  require_primary_gid     = 0; /* require pGID set by previous plugin */
static int  do_not_require_primary_gid = 0; /* don't require pGID set by previous plugin */

static int pgid_mapping = 0;	/* default: don't map pGID from account */
static int sgid_mapping = 0;	/* default: don't map sGIDs from account */


/************************************************************************
 * private prototypes
 ************************************************************************/

/* called by plugin_run() and plugin_verify() */
static int plugin_run_or_verify(int argc, lcmaps_argument_t *argv,
				int lcmaps_mode);

/* Adds given group IDs to buffer. Number of group IDs is cnt_gid. cur_idx
 * should be the first position to be written. bufsize and buffer will describe
 * a buffer on the heap that will be allocate and extended when needed.
 * Upon success, cur_index, bufsize and buffer will describe the new state.
 * return 0 on success, -1 on failure. */
static int get_gid_string(int cnt_gid, gid_t *gids,
			  size_t *my_index, size_t *bufsize, char **buffer);

/************************************************************************
 * public functions
 ************************************************************************/

/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(int argc, char **argv) {
    const char * logstr = PLUGIN_PREFIX"-plugin_initialize()";
    int i;
    int  use_voms_gid                           = 0; /* use GIDs from VOMS */
    int  use_account_gid                        = 0; /* use GIDs from acct */
    int  do_not_map_primary_gid                 = 0; /* pGID not from acct */
    int  add_primary_gid_from_mapped_account    = 0; /* pGID from acct */
    int  add_primary_gid_as_secondary_gid_from_mapped_account = 0; /* pGID from acct as sGID */
    int  do_not_add_secondary_gids_from_mapped_account = 0; /* sGIDs not from acct */
    int  add_secondary_gids_from_mapped_account = 0; /* sGIDs from acct */
    long long_value;
    char *endptr=NULL;

    /* Log commandline parameters on debug */
    lcmaps_log(LOG_DEBUG,"%s: passed arguments:\n",logstr);
    for (i=0; i < argc; i++)
	lcmaps_log(LOG_DEBUG,"%s: arg %d is %s\n", logstr, i, argv[i]);

    /* Parse arguments, argv[0] = name of plugin, so start with i = 1 */
    for (i = 1; i < argc; i++) {
	/* check grid-mapfile option (or similar variations) */
        if ( strcmp(argv[i], "-gridmapfile") == 0 ||
             strcmp(argv[i], "-GRIDMAPFILE") == 0 ||
             strcmp(argv[i], "-gridmap") == 0 ||
             strcmp(argv[i], "-GRIDMAP") == 0 )
	{
	    /* check valid filename argument */
            if (argv[i + 1] == NULL || argv[i + 1][0]=='\0') {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by valid filename\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* free existing one and copy new one */
	    free(gridmapfile); gridmapfile=NULL;
	    if (argv[i+1][0]=='/')  { /* absolute path */
		if ( (gridmapfile = strdup(argv[i + 1])) == NULL)	{
		    lcmaps_log(LOG_ERR, "%s: out of memory\n", logstr);
		    goto fail_init_module;
		}
	    } else { /* relative path */
		if (lcmaps_get_prefixed_file(argv[i + 1], &gridmapfile)==-1)
		    goto fail_init_module;
	    }
	    /* log the name and increase arg counter */
	    lcmaps_log(LOG_DEBUG, "%s: Using grid-mapfile \"%s\".\n",
		    logstr, gridmapfile);
            i++;
        }
	/* check gridmapdir option (or similar variations) */
        else if ( strcmp(argv[i], "-gridmapdir") == 0 ||
                  strcmp(argv[i], "-GRIDMAPDIR") == 0 )
        {
	    /* check valid dirname argument */
            if (argv[i+1]==NULL || argv[i+1][0]!='/')
	    {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by "
		    "a valid absolute directory.\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* free existing one and copy new one */
	    free(gridmapdir); gridmapdir=NULL;
	    if (argv[i+1][0]=='/')  { /* absolute path */
		if ( (gridmapdir = strdup(argv[i + 1])) == NULL)	{
		    lcmaps_log(LOG_ERR, "%s: out of memory\n", logstr);
		    goto fail_init_module;
		}
	    } else { /* relative path */
		if (lcmaps_get_prefixed_file(argv[i + 1], &gridmapdir)==-1)
		    goto fail_init_module;
	    }
	    /* log the name and increase arg counter */
	    lcmaps_log(LOG_DEBUG, "%s: Using gridmapdir \"%s\".\n",
		    logstr, gridmapdir);
            i++;
        }
	/* check max_mappings_per_credential option */
        else if ( strcmp(argv[i], "-max_mappings_per_credential") == 0 )
        {
	    /* check valid argument */
	    if (argv[i + 1] == NULL || argv[i + 1][0]=='\0') {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by a valid number\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }

	    /* try to convert argument to integer in right range */
	    errno=0;
	    long_value=strtol(argv[i + 1],&endptr,10);
	    /* endptr will either point to first non-valid char, or end of valid
	     * string unless argv[i+1]=="", which we already checked above */
	    if (errno!=0 || *endptr!='\0' ||
		long_value < (long)MAPPING_MIN ||
		long_value > (long)LCMAPS_ABSOLUTE_MAPPING_MAX)
	    {
		lcmaps_log(LOG_ERR,
		    "%s: Illegal value for \"-max_mappings_per_credential\" "
		    "(%s): Should be in the range: %d-%d (failure)\n",
		    logstr, argv[i + 1],
		    MAPPING_MIN, LCMAPS_ABSOLUTE_MAPPING_MAX);
		goto fail_init_module;
	    }
	    mapping_max = (int)long_value;
            i++;
        }
	/* check strict_poolprefix_match option */
        else if ( strcmp(argv[i], "-strict_poolprefix_match") == 0 )
        {
	    /* check valid argument */
	    if (argv[i + 1] == NULL || argv[i + 1][0]=='\0') {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by \"yes\" or \"no\"\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }

	    /* parse argument */
	    if (strcmp(argv[i+1],"yes") == 0)
		strict_poolprefix_match = 1;
	    else if (strcmp(argv[i+1],"no") == 0)
		strict_poolprefix_match = 0;
	    else {
		lcmaps_log(LOG_ERR,"%s: use \"yes\" or \"no\" for option %s\n",
			logstr, argv[i]);
		goto fail_init_module;
            }
            i++;
        }
	/* check override_inconsistency (or similar variations) */
        else if ( strcmp(argv[i], "-override_inconsistency") == 0 ||
                  strcmp(argv[i], "-OVERRIDE_INCONSISTENCY") == 0 )
        {
            override_inconsistency = 1;
        }
        else if (strcmp(argv[i], "--do-not-add-primary-gid-from-mapped-account") == 0)
        {
	    /* check consistency with other pGID flags */
	    if (add_primary_gid_from_mapped_account ||
		add_primary_gid_as_secondary_gid_from_mapped_account)
	    {
		lcmaps_log(LOG_ERR, "%s: cannot specify both %s and "
		    "--add-primary-gid-from-mapped-account or "
		    "--add-primary-gid-as-secondary-gid-from-mapped-account\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            do_not_map_primary_gid = 1;
        }
        else if (strcmp(argv[i], "--add-primary-gid-from-mapped-account") == 0)
        {
	    /* check consistency with other pGID flags */
	    if (do_not_map_primary_gid) {
		lcmaps_log(LOG_ERR, "%s: cannot specify both %s and "
		    "--do-not-add-primary-gid-from-mapped-account\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            add_primary_gid_from_mapped_account = 1;
        }
        else if (strcmp(argv[i], "--add-primary-gid-as-secondary-gid-from-mapped-account") == 0)
        {
	    /* check consistency with other pGID flags */
	    if (do_not_map_primary_gid) {
		lcmaps_log(LOG_ERR, "%s: cannot specify both %s and "
		    "--do-not-add-primary-gid-from-mapped-account\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            add_primary_gid_as_secondary_gid_from_mapped_account = 1;
	}
        else if (strcmp(argv[i], "--do-not-add-secondary-gids-from-mapped-account") == 0)
        {
	    /* check consistency with other sGID flag */
	    if (add_secondary_gids_from_mapped_account) {
		lcmaps_log(LOG_ERR, "%s: cannot specify both %s and "
		    "--add-secondary-gids-from-mapped-account\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            do_not_add_secondary_gids_from_mapped_account = 1;
        }
        else if (strcmp(argv[i], "--add-secondary-gids-from-mapped-account") == 0)
        {
	    /* check consistency with other sGID flag */
	    if (do_not_add_secondary_gids_from_mapped_account) {
		lcmaps_log(LOG_ERR, "%s: cannot specify both %s and "
		    "--do-not-add-secondary-gids-from-mapped-account\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            add_secondary_gids_from_mapped_account = 1;
        }
        else if ((strcmp(argv[i], "--use-voms-gid") == 0) ||
                 (strcmp(argv[i], "--use_voms_gid") == 0) ||
                 (strcmp(argv[i], "-use_voms_gid") == 0))
        {
	    /* check consistency with other voms/account defaults flag */
	    if (use_account_gid)    {
		lcmaps_log(LOG_ERR,
		    "%s: cannot specify both %s and --use-account-gid\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            use_voms_gid = 1;
        }
        else if (strcmp(argv[i], "--use-account-gid") == 0) {
	    /* check consistency with other voms/account defaults flag */
	    if (use_voms_gid)    {
		lcmaps_log(LOG_ERR,
		    "%s: cannot specify both %s and --use-voms-gid\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            use_account_gid = 1;
        }
        else if (strcmp(argv[i], "--do-not-use-secondary-gids") == 0 ||
                 strcmp(argv[i], "-do_not-use-secondary-gids") == 0 ||
                 strcmp(argv[i], "-do_not_use_secondary_gids") == 0)
        {
            use_secondary_gids = 0;
        }
        else if (strcmp(argv[i], "--do-not-require-primary-gid") == 0 ||
                 strcmp(argv[i], "-do_not-require-primary-gid") == 0 ||
                 strcmp(argv[i], "-do_not_require_primary_gid") == 0)
        {
	    /* check consistency with other flags */
	    if (require_primary_gid) {
		lcmaps_log(LOG_ERR, "%s: cannot specify both %s and "
		    "--require-primary-gid\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            do_not_require_primary_gid = 1;
        }
        else if (strcmp(argv[i], "--require-primary-gid") == 0)
        {
	    /* check consistency with other flags */
	    if (do_not_require_primary_gid) {
		lcmaps_log(LOG_ERR, "%s: cannot specify both %s and "
		    "--do-not-require-primary-gid\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* ok: set flag */
            require_primary_gid = 1;
        }
	/* any other argument is an error */
	else
	{
            lcmaps_log(LOG_ERR,
		    "%s: Unknown argument for plugin: %s (failure)\n",
		    logstr, argv[i]);
            goto fail_init_module;
        }
    }

    /* Check we have a grid-mapfile */
    if (gridmapfile==NULL || gridmapfile[0]=='\0')  {
	free(gridmapfile); gridmapfile=NULL;
        lcmaps_log(LOG_INFO,
	    "%s: No grid-mapfile was provided, will use default.\n", logstr);
    }

    /* Check we have a gridmapdir */
    if (gridmapdir==NULL || gridmapdir[0]=='\0')    {
	free(gridmapdir); gridmapdir=NULL;
        lcmaps_log(LOG_INFO,
	    "%s: No gridmapdir was provided, will try environment variable.\n",
	    logstr);
    }

    /* Set the correct pGID and sGID mapping flags, first defaults */
    if (use_voms_gid)
	/* no implicit mapping from account GIDs */
	pgid_mapping=sgid_mapping=0;
    else if (use_account_gid)
	/* implicit mapping from account GIDs */
	pgid_mapping=sgid_mapping=1;

    /* handle pGID options */
    if (do_not_map_primary_gid)
	pgid_mapping=0;
    else {
	if (add_primary_gid_from_mapped_account)
	    /* add (not replace) pGID->pGID flag */
	    pgid_mapping|=1;
	if (add_primary_gid_as_secondary_gid_from_mapped_account)
	    /* add (not replace) pGID->sGID flag */
	    pgid_mapping|=2;
    }

    /* handle sGID options */
    if (do_not_add_secondary_gids_from_mapped_account)
	sgid_mapping=0;
    else if (add_secondary_gids_from_mapped_account)
	sgid_mapping=1;

    return LCMAPS_MOD_SUCCESS;

fail_init_module:
    free(gridmapfile);
    gridmapfile = NULL;
    free(gridmapdir);
    gridmapdir = NULL;

    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(int *argc, lcmaps_argument_t **argv) {
    const char * logstr = PLUGIN_PREFIX"-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
	{"user_dn"      ,"char *" , 0,NULL},
	{"fqan_list"    ,"char **", 0,NULL},
	{"nfqan"        ,"int"    , 0,NULL},
	{"mapcounter"   ,"int"    , 1,NULL},
	{"requested_uid","uid_t"  , 0,NULL},
	{NULL           ,NULL     ,-1,NULL},
	{NULL           ,NULL     ,-1,NULL}
    };

    /* Get the version of LCMAPS being used: we need at least 1.5.8 to be able
     * to demand "requested_username" in the argList */
    int major=0,minor=0,patch=0;
    /* Most UNIX now support RTLD_DEFAULT (POSIX reserved) */
#ifdef RTLD_DEFAULT
    int (*lcmaps_major)(void),(*lcmaps_minor)(void),(*lcmaps_patch)(void);
    dlerror();
    lcmaps_major=dlsym(RTLD_DEFAULT,"lcmaps_get_major_version");
    lcmaps_minor=dlsym(RTLD_DEFAULT,"lcmaps_get_minor_version");
    lcmaps_patch=dlsym(RTLD_DEFAULT,"lcmaps_get_patch_version");
    if (dlerror()==NULL)    {
	major=lcmaps_major();
	minor=lcmaps_minor();
	patch=lcmaps_patch();
    }
#else
    /* No RTLD_DEFAULT, just hope the symbol exists in LCMAPS */
    major=lcmaps_get_major_version();
    minor=lcmaps_get_minor_version();
    patch=lcmaps_get_patch_version();
#endif

    /* Too old when older than 1.5.8 */
    if (major<1 || (major==1 && (minor<5 || (minor==5 && patch<8))))	{
	lcmaps_log(LOG_DEBUG,
	    "%s: Old LCMAPS found (%d.%d.%d), not using requested_username\n",
	    logstr,major,minor,patch);
    } else {
	lcmaps_log(LOG_DEBUG,
	    "%s LCMAPS (%d.%d.%d) supports using requested_username\n",
	    logstr,major,minor,patch);
	argList[5].argName="requested_username";
	argList[5].argType="char *";
	argList[5].argInOut=1;
	argList[5].value=NULL;
    }

    lcmaps_log(LOG_DEBUG,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log(LOG_DEBUG,"%s: address first argument: %p\n",
	    logstr, (void*)argList);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(int argc, lcmaps_argument_t *argv) {
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().

    Policy: This method will not fail if the uid found is not the
            uid requested for the user, but it will issue a warning.
            The full check is deferred to the stage after which all plugins have run.

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(int argc, lcmaps_argument_t *argv) {
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate(void) {
    const char * logstr = PLUGIN_PREFIX"-plugin_terminate()";

    lcmaps_log(LOG_DEBUG,"%s: terminating\n", logstr);

    free(gridmapfile);
    gridmapfile=NULL;
    free(gridmapdir);
    gridmapdir=NULL;

    return LCMAPS_MOD_SUCCESS;
}


/************************************************************************
 * private functions
 ************************************************************************/

/**
 * Actual run/verify function. Called by both plugin_run and plugin_verify
 * with different lcmaps_mode.
 */
static int plugin_run_or_verify(int argc, lcmaps_argument_t *argv,
				int lcmaps_mode) {
    const char *        logstr       = NULL;
    const char *        logmapfile   = gridmapfile ? gridmapfile
						   : "default grid-mapfile";
    void *              value        = NULL;
    char **             user_dn_list = NULL;
    char *              dn           = NULL;
    int                 dn_cnt       = 0;
    int                 nfqan        = -1;
    char **             fqan_list    = NULL;
    int                 i            = 0;
    int                 mapcounter   = -1;
    char *              req_username = NULL;
    int                 req_username_needs_free=0;
    uid_t               req_uid      = (uid_t)(-1);
    struct passwd *     pw           = NULL;
    int                 cnt_pri_gid  = 0;
    gid_t *             pri_gid      = NULL;
    int                 cnt_sec_gid  = 0;
    gid_t *             sec_gid      = NULL;
    size_t              my_index     = 0;
    size_t              bufsize      = 0;
    char *              gidbuffer    = NULL;
    size_t              leasenamelen = 0;
    char *              leasename    = NULL;
    unsigned short      options      = 0;
    int                 imap         = 0;
    int                 rc           = 0;
    char *              username     = NULL;
    char *              encoded_lease= NULL;
    int                 found_mapping= 0;
    struct passwd       *user_info   = NULL;

    /* Set suitable logstr */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = PLUGIN_PREFIX"-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = PLUGIN_PREFIX"-plugin_verify()";
    else {
        lcmaps_log(LOG_ERR, PLUGIN_PREFIX"-plugin_run_or_verify(): "
		"attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_plugin;
    }

    /* Try to get DN from LCMAPS */
    /* First try to obtain DN from the credential data (i.e. stored by other
     * plugins */
    user_dn_list = getCredentialData(DN, &dn_cnt);
    if (dn_cnt>0)   {
	/* Log already registered DNs */
	for (i=0; i<dn_cnt; i++)
	    lcmaps_log(LOG_DEBUG,"%s: found registered DN[%d/%d]: %s\n",
		    logstr, i+1, dn_cnt, user_dn_list[i]);
	dn=user_dn_list[0];
    } else {
	/* No DNs are registered, use the introspect/run arguments */
	value=lcmaps_getArgValue("user_dn", "char *", argc, argv);
	if (value == NULL || (dn = *(char **)value) == NULL ) {
	    lcmaps_log(LOG_WARNING,"%s: could not get value of dn !\n", logstr);
	    return -1;
	}
	
	/* push it to the end-result registry */
	lcmaps_log(LOG_DEBUG, "%s: Adding DN to LCMAPS framework: %s\n",
		logstr, dn);
	addCredentialData(DN, &dn);
    }

    /* Log the found DN */
    lcmaps_log(LOG_DEBUG,"%s: found dn: %s\n", logstr, dn);

    /* Get value of mapcounter */
    if ( ( value = lcmaps_getArgValue("mapcounter", "int", argc, argv) ) ) {
	mapcounter = *(int *)value;
        lcmaps_log(LOG_DEBUG,"%s: mapcounter: %d\n", logstr, mapcounter);
    } else {
	/* normal behaviour: we don't use mapcounter */
        lcmaps_log(LOG_DEBUG,"%s: mapcounter is not set\n", logstr);
    }

    /* Try to get FQANs from LCMAPS values: */
    /* First try to obtain FQANs from the credential data (i.e. stored by other
     * plugins */
    fqan_list = getCredentialData(LCMAPS_VO_CRED_STRING, &nfqan);
    if (nfqan>0) {
	lcmaps_log(LOG_DEBUG, "%s: found %d FQAN(s) in credential data\n",
		logstr, nfqan);
    } else {
	/* No FQANs registered, use the introspect/run arguments */
	lcmaps_log(LOG_DEBUG,
	    "%s: no FQANs registered by other plugins, trying run/introspect args\n",
	    logstr);
	if ( ( value = lcmaps_getArgValue("nfqan", "int", argc, argv) ) ) {
	    /* get number of FQANs */
	    if ( (nfqan = *(int *) value) < 1 ) {
		lcmaps_log(LOG_INFO,
			"%s: no (valid) VOMS groups found --> no mapping\n",
			logstr);
		goto fail_plugin;
	    }
	    /* Log number of FQANs */
	    lcmaps_log(LOG_DEBUG,
		    "%s: the list of FQANs should contain %d elements\n",
		    logstr, nfqan);
	    /* get FQAN list */
	    value = lcmaps_getArgValue("fqan_list", "char **", argc, argv);
	    if ( value==NULL || (fqan_list = *(char ***) value)==NULL )   {
		lcmaps_log(LOG_WARNING,
			"%s: could not retrieve list of %d FQANs!\n",
			logstr, nfqan);
		goto fail_plugin;
	    }
	}
    }

    /* Log the found FQANs */
    for (i = 0; i < nfqan; i++)
	lcmaps_log(LOG_DEBUG, "%s: FQAN %d: %s\n", logstr, i+1, fqan_list[i]);

    /* In verify mode, requested account is typically in requested_uid, in run
     * mode, it can be provided as requested_username. Let's just see if either
     * is set. */

    /* Get requested_username value */
    value = lcmaps_getArgValue("requested_username", "char *", argc, argv);
    if ( value != NULL && *(char **)value !=NULL ) {
	/* copy and log resulting account name */
	req_username=*(char **)value;
	lcmaps_log(LOG_DEBUG,"%s: the requested user is %s\n",
		logstr, req_username);
    } else { /* No (valid) requested_username, try requested_uid */
	/* Get requested_uid value */
	value = lcmaps_getArgValue("requested_uid", "uid_t", argc, argv);
	if ( value!=NULL && *(int *)value != -1)  { /* undefined value -> -1 */
	    req_uid = *(uid_t *)value;
	    /* Basic sanity check */
	    if (req_uid == 0) {
		lcmaps_log(LOG_ERR,
			"%s: illegal request for uid == 0 (failure)\n", logstr);
		goto fail_plugin;
	    }
	    /* Get passwd info */
	    if ( (pw = getpwuid(req_uid )) == NULL ||
		 pw->pw_name == NULL )
	    {
		lcmaps_log(LOG_WARNING,
			"%s: the requested uid %d is illegal.\n",
			logstr, (int) req_uid);
		goto fail_plugin;
	    }
	    /* copy resulting account name */
	    if ( (req_username=strdup(pw->pw_name))==NULL ) {
		lcmaps_log(LOG_ERR,"%s: out of memory\n",logstr);
		goto fail_plugin;
	    }
	    /* keep track whether it needs free */
	    req_username_needs_free=1;
	    /* Log the result */
	    lcmaps_log(LOG_DEBUG,"%s: the requested user is %s(%d)\n",
		    logstr, req_username, (int)req_uid);
	} else if (lcmaps_mode == PLUGIN_VERIFY)    {
	    lcmaps_log(LOG_WARNING,
		    "%s: both requested_username and requested_uid are unset, "
		    "need at least one in PLUGIN_VERIFY mode\n", logstr);
	    goto fail_plugin;
	}
    }

    /* Get the (VOMS) gids found so far and build a string out of it.
     * First primary Gid(s), behind it the secondary Gids.
     * For the moment, the same order is taken as found in the VOMS credential.
     * We might consider to sort the gids.
     * We cannot order them by lcmaps_argument_t, but have to use the
     * getCredentialData() function since it was stored there by a plugin
     * (lcmaps_voms.mod) */
    pri_gid = (gid_t *)getCredentialData(PRI_GID, &cnt_pri_gid);
    if (cnt_pri_gid == 0) {
        if (require_primary_gid)    {
            lcmaps_log(LOG_WARNING,
		    "%s: failure: no primary group found\n", logstr);
	    goto fail_plugin;
	}
    } else if (cnt_pri_gid < 0) {
        lcmaps_log(LOG_ERR,
		"%s: negative number of primary groups found ! (failure)\n",
		logstr);
        goto fail_plugin;
    } else if (cnt_pri_gid > 1)
        lcmaps_log(LOG_WARNING,"%s: warning more than 1 primary group found\n",
		logstr);

    sec_gid = (gid_t *)getCredentialData(SEC_GID, &cnt_sec_gid);
    if (cnt_sec_gid < 0) {
        lcmaps_log(LOG_ERR,
		"%s: negative number of secondary groups found ! (failure)\n",
		logstr);
        goto fail_plugin;
    }

    /* Create buffer with primary and optionally secondary GIDs */
    if ( get_gid_string(cnt_pri_gid, pri_gid,
			&my_index, &bufsize, &gidbuffer) ||
	 (use_secondary_gids && get_gid_string(cnt_sec_gid,sec_gid,
					       &my_index,&bufsize,&gidbuffer)))
	goto fail_plugin;

    /* Create leasename based on dn+gids */
    if (bufsize == 0)
        leasename = strdup(dn);
    else {
        leasenamelen = strlen(dn) + strlen(gidbuffer) + 1;
        if ( (leasename = malloc(leasenamelen)) == NULL )   {
	    lcmaps_log(LOG_ERR,"%s: out of memory\n", logstr);
	    goto fail_plugin;
	}
        if (snprintf(leasename, leasenamelen, "%s\001%s", dn, gidbuffer+1)<0) {
	    lcmaps_log(LOG_ERR,"%s: error creating leasename: %s",
		    logstr, strerror(errno));
	    goto fail_plugin;
	}
    }
    lcmaps_log(LOG_DEBUG,"%s: using leasename: %s\n", logstr,leasename);

    /* Set the matching options */
    options = MATCH_INCLUDE|MATCH_WILD_CHARS|REQUIRE_MAPFILE;

    /* if override_consistency is set add this to the options so it will
     * take effect */
    if (override_inconsistency)
        options = options|OVERRIDE_INCONSISTANCY;

    /* Do not create new leases in verification mode */
    if (lcmaps_mode == PLUGIN_VERIFY)
        options = options|ONLY_USE_EXISTING_LEASE;

    /* if strict_poolprefix_match is set add this to the matchin_type so it will
     * take effect */
    if (strict_poolprefix_match)
        options = options|MATCH_STRICT_PREFIX_NUM;

    /* Try to match the VO strings with the grid-mapfile, normally the first
     * available VO string should match */
    for (i = 0; i < nfqan; i++) {
	/* Removed: In case require_primary_gid was set, we used to fail if we
	 * didn't match on the first FQAN. This behaviour was undocumented and
	 * probably a bug: it was probably assuming the pGID came from the first
	 * FQAN and intended to enforce mapping the account on the same FQAN. */

	/* For Verify mode and unset mapcounter, we first try to loop through
	 * all available map counters */
	if (lcmaps_mode == PLUGIN_VERIFY && mapcounter < MAPPING_MIN)   {
	    /* No particular mapcounter specified: try them all */
	    for (imap = 1; imap <= mapping_max; imap++) {
		rc = lcmaps_gridmapdir(gridmapfile, fqan_list[i], gridmapdir,
				       mapping_max, imap,
				       leasename, req_username, options,
				       &username, &encoded_lease);

		/* parse return value of lcmaps_gridmapdir */
		if (rc==-1)
		    /* error */
		    goto fail_plugin;
		if (rc==0)  {
		    /* no match: log and try next mapcounter */
		    lcmaps_log(LOG_DEBUG,
			"%s: No match for the requested poolaccount for FQAN "
			"%d, \"%s\", mapcount=%d\n",
			logstr, i+1, fqan_list[i], imap);
		    continue; /* next mapcounter */
		}
		/* match */
		found_mapping = 1;
		break;	/* out of mapcounter loop */
	    }
	    /* If we didn't found it, log and try mapcounter-less */
	    if (!found_mapping)
		lcmaps_log(LOG_DEBUG,
		    "%s: No mapcounter-based match found for FQAN %d, \"%s\", "
		    "trying mapcounter-less mapping...\n",
		    logstr, i+1, fqan_list[i]);
	}

	/* Check if we already found it */
	if (found_mapping == 0)	{
	    rc = lcmaps_gridmapdir(gridmapfile, fqan_list[i], gridmapdir,
				   mapping_max, mapcounter,
				   leasename, req_username, options,
				   &username, &encoded_lease);

	    /* parse return value of lcmaps_gridmapdir */
	    if (rc==-1)
		/* error */
		goto fail_plugin;
	    if (rc==0)	{
		/* no match: log and try next FQAN */
		if (mapcounter>0)
		    lcmaps_log(LOG_DEBUG,
			"%s: No match for %s poolaccount for FQAN %d, \"%s\", "
			"mapcount=%d\n",
			logstr, (req_username ? "the requested" : "a VOMS"),
			i+1, fqan_list[i], mapcounter);
		else
		    lcmaps_log(LOG_DEBUG,
			"%s: No match for %s poolaccount for FQAN %d, \"%s\"\n",
			logstr, (req_username ? "the requested" : "a VOMS"),
			i+1, fqan_list[i]);
		continue; /* next FQAN */
	    }

	    /* match */
	    found_mapping = 1;
	    break;  /* out of FQAN loop */
	}
    }

    /* Log if when we haven't found a match */
    if (found_mapping == 0) {
	/* no match, this should be at most a NOTICE */
	if (mapcounter>0)   {
	    if (req_username)
		lcmaps_log(LOG_NOTICE,
		    "%s: No match for requested poolaccount %s for any of the "
		    "FQANs of user \"%s\", mapcount=%d, in %s and %s\n", logstr,
		    req_username, dn, mapcounter, logmapfile, gridmapdir);
	    else /* no req_username */
		lcmaps_log(LOG_NOTICE,
		    "%s: No match for a VOMS poolaccount for any of the "
		    "FQANs of user \"%s\", mapcount=%d, in %s and %s\n", logstr,
		    dn, mapcounter, logmapfile, gridmapdir);
	} else { /* no mapcounter */
	    if (req_username)
		lcmaps_log(LOG_NOTICE,
		    "%s: No match for requested poolaccount %s for any of the "
		    "FQANs for user \"%s\", in %s and %s\n", logstr,
		    req_username, dn, logmapfile, gridmapdir);
	    else /* no req_username */
		lcmaps_log(LOG_NOTICE,
		    "%s: No match for a VOMS poolaccount for any of the "
		    "FQANs for user \"%s\", in %s and %s\n", logstr,
		    dn, logmapfile, gridmapdir);
	}
        goto fail_plugin;
    }

    /* found match: log */
    lcmaps_log(LOG_DEBUG,"%s: found %susername %s for FQAN %d, \"%s\"\n",
	logstr, req_username ? "requested " : "", username, i+1, fqan_list[i]);

    /* Convert username to uid, pgid and sgids */

    /* Get account info for found username */
    if ( (user_info = getpwnam(username)) == NULL ) {
	lcmaps_log(LOG_WARNING,
		"%s: no user account found with the name \"%s\"\n",
		logstr, username);
	goto fail_plugin;
    }

    /* Log resulting account */
    lcmaps_log(LOG_DEBUG,"%s: username : %s\n", logstr, user_info->pw_name);
    lcmaps_log(LOG_DEBUG,"%s: user_id  : %d\n", logstr, user_info->pw_uid);
    lcmaps_log(LOG_DEBUG,"%s: group_id : %d\n", logstr, user_info->pw_gid);
    lcmaps_log(LOG_DEBUG,"%s: home dir : %s\n", logstr, user_info->pw_dir);

    /* Add this credential data to the credential data repository in the plugin
     * manager */
    addCredentialData(UID, &(user_info->pw_uid));

    /* Handle primary GID from account.
     * NOTE we can add pGID both as pGID and as sGID */
    if ( (pgid_mapping&1) == 1) {
	/* Map primary Unix GID from the account info */
	lcmaps_log(LOG_DEBUG,
		"%s: adding primary GID (%d) from pool account\n",
		logstr, user_info->pw_gid);
	addCredentialData(PRI_GID, &(user_info->pw_gid));
    }
    if ( (pgid_mapping&2) == 2 ) {
	/* Add the primary GID from the mapped account as a secondary GID to the
	 * result */
	lcmaps_log(LOG_DEBUG,
		"%s: adding primary GID (%d) from pool account "
		"as a secondary GID\n", logstr, user_info->pw_gid);
	addCredentialData(SEC_GID, &(user_info->pw_gid));
    }

    /* Handle secondary GIDs from account */
    if (sgid_mapping==1 &&
	lcmaps_get_gidlist(username, &cnt_sec_gid, &sec_gid)==0)
    {
	/* Add secondary Unix group IDs from the mapped local account */
	lcmaps_log(LOG_DEBUG,
		"%s: adding secondary GIDs (%d) from pool account\n",
		logstr, user_info->pw_gid);
	for (i = 0; i < cnt_sec_gid; i++)
	    addCredentialData(SEC_GID, &(sec_gid[i]));
	free(sec_gid);
    }

    /* Added because of the POOL_INDEX request for the DAS */
    if (encoded_lease)
	addCredentialData(POOL_INDEX, &encoded_lease);

    /* success */
    if (req_username_needs_free)
	free(req_username);
    free(username);
    free(gidbuffer);
    free(leasename);
    free(encoded_lease);

    lcmaps_log(LOG_INFO,"%s: voms_poolaccount plugin succeeded\n", logstr);

    return LCMAPS_MOD_SUCCESS;

fail_plugin:
    if (req_username_needs_free)
	free(req_username);
    free(username);
    free(gidbuffer);
    free(leasename);
    free(encoded_lease);

    lcmaps_log(LOG_INFO,"%s: voms_poolaccount plugin failed\n", logstr);

    return LCMAPS_MOD_FAIL;
}


/**
 * Adds given group IDs to buffer. Number of group IDs is cnt_gid. cur_idx
 * should be the first position to be written. bufsize and buffer will describe
 * a buffer on the heap that will be allocate and extended when needed.
 * Upon success, cur_index, bufsize and buffer will describe the new state.
 * return 0 on success, -1 on failure.
 */
static int get_gid_string(int cnt_gid, gid_t *gids,
			  size_t *my_index, size_t *bufsize, char **buffer)   {
    const char * logstr="lcmaps_voms-get_gid_string";
    size_t len, size=*bufsize;
    char *gidbuffer=*buffer;
    int i;
    struct group *group_info;

    /* add all the groups */
    for (i = 0; i < cnt_gid; i++) {
	group_info = getgrgid(gids[i]);
        if ( group_info==NULL || group_info->gr_name==NULL) {
            lcmaps_log(LOG_WARNING,
		    "%s: no group id found for gid = %lu\n",
		    logstr, (long unsigned)gids[i]);
	    return -1;
	}
	/* Need to write 1 + string bytes */
	len=1+strlen(group_info->gr_name);
	if ( size-(*my_index) <= len )  {
	    /* When initial size is 0, we need to add at least len + one for
	     * '\0' byte, so adding INITBUF_SIZE should be good there too.
	     * Otherwise, we need to add only len - <what's left> < len */
	    size+=len+INITBUF_SIZE;
	    lcmaps_log(LOG_DEBUG,
		    "%s: resizing gidbuffer from %lu to %lu bytes\n",
		    logstr, (long unsigned)*bufsize, (long unsigned)size);
	    gidbuffer=realloc(*buffer, size);
	    if (!gidbuffer)	{
		lcmaps_log(LOG_ERR, "%s: out of memory\n", logstr);
		return -1;
	    }
	    /* update parameters */
	    *buffer=gidbuffer;
	    *bufsize=size;
	}
	gidbuffer[*my_index]=':';
	/* copy (len-1) bytes from groupname plus null-byte */
	strncpy(&(gidbuffer[*my_index+1]), group_info->gr_name, len); 
	*my_index += len;
	lcmaps_log(LOG_DEBUG,"%s: gidbuffer: %s\n", logstr, gidbuffer);
    }

    return 0;
}
