/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \page lcmaps_voms_poolgroup.mod voms poolgroup plugin

    \section vomspoolgroupsyn SYNOPSIS

    \b lcmaps_voms_poolgroup.mod
        -GROUPMAPFILE|-groupmapfile|-GROUPMAP|-groupmap \<groupmapfile\>
        -GROUPMAPDIR|-groupmapdir \<groupmapdir\>
        [-mapall] [-mapmin \<group count\>]

    \section vomspoolgroupdesc DESCRIPTION

    The poolgroup acquisition plugin is a 'VOMS-aware' plugin.
    It uses the VOMS information (acquired by the plugin \ref lcmaps_voms.mod "lcmaps_voms.mod")
    to gather primary and secondary GIDs.
    This is accomplished by matching VO-GROUP-ROLE(-CAPABILITY) combinations in the so-called
    \e groupmapfile (gridmapfile style) and by finding the corresponding 'poolgroup' (similar to
    the 'poolaccount' procedure, see \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod")
    Wildcards can be used in the groupmapfile to match VO-GROUP-ROLE combinations.

    EXAMPLE 'groupmapfile':

    \c "/VO=atlas/GROUP=mcprod"   \c mcprod

    \c "/VO=atlas/GROUP=mcprod"   \c .atlas

    \c "/VO=atlas/GROUP=dev"      \c .atlas

    \c "/VO=atlas/GROUP=*"        \c .atlas

    The VO-GROUP-ROLE combination \c "/VO=atlas/GROUP=mcprod" starts with an alfanumeric character
    (not ".") and indicates a localgroup entry in the groupmapfile (will be resolved by the
    \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod").
    The VO-GROUP-ROLE combination \c "/VO=atlas/GROUP=*"
    indicates that all users from the Atlas VO with every other group than 'mcprod'
    will be mapped to the '.atlas' pool of (system) groups.
    Just like the \e poolaccount plugin this plugin will link an entry
    (in this case a VO-GROUP-ROLE combination)
    to a locally known group (a.k.a. poolgroup) in the \e groupmapdir directory.
    The difference with the \e poolaccount plugin is that there is not a Distinghuished Name
    but a VO-GROUP-ROLE combination and there is no poolaccount but poolgroup defined in de
    groupmapfile (similar to the gridmapfile).
    Instead of the \e gridmapdir the \e groupmapdir directory is used
    for the registration of thew mapping between poolgroups and the VO-GROUP-ROLE combination.

    As you can see the in the example the 'mcprod' GROUP can be found by using the localgroup
    plugin and the poolgroup plugin.
    With the poolgroup plugin there can be made a mapping between "/VO=atlas/GROUP=mcprod" and the group
    'atlas001' (based on the .atlas pool).
    The entry \c "/VO=atlas/GROUP=dev" will also result in a group from this '.atlas' pool,
    but a different one, e.g. 'atlas002'.
    Finally, we have random other groups not predefined in the groupmapfile, for example
    \c "/VO=atlas/GROUP=foo", which matches \c "/VO=atlas/GROUP=*" in the groupmapfile.
    This VO-GROUP combination will be mapped to a poolgroup (probably) called 'atlas003'.

    The poolgroup plugin will try to match each VO-GROUP-ROLE combination that was
    found by the plugin \ref lcmaps_voms.mod "lcmaps_voms.mod".
    The first VO-GROUP-ROLE combination will become the primary group, the others secondary groups.
    As the primary GID may be used for auditing and accounting purposes it is important that
    the user uses the correct ordering of VO-GROUP-ROLE combinations in his grid credential (X509
    certificate).

    \section vomspoolgroupoptions OPTIONS
    \subsection vomspoolgroupoptie1 -GROUPMAPFILE \<groupmapfile\>
        See \ref vomspoolgroupoptie4 "-groupmap"

    \subsection vomspoolgroupoptie2 -groupmapfile \<groupmapfile\>
        See \ref vomspoolgroupoptie4 "-groupmap"

    \subsection vomspoolgroupoptie3 -GROUPMAP \<groupmapfile\>
        See \ref vomspoolgroupoptie4 "-groupmap"

    \subsection vomspoolgroupoptie4 -groupmap \<groupmapfile\>
        If this option is set, it will override the default path to the groupmapfile.
        It is advised to use an absolute path to the groupmapfile to avoid usage of the wrong file(path).

    \subsection vomspoolgroupoptie5 -GROUPMAPDIR \<groupmapdir\>
        See \ref vomspoolgroupoptie6 "-groupmapdir"

    \subsection vomspoolgroupoptie6 -groupmapdir \<groupmapdir\>
        Here you can override the default directory path to the 'groupmapdir'.
        This directory is just like the \e gridmapdir and holds all the poolgroup mappings
        that has/will be made by linking filenames to a i-node indicating a mapping
        between a VO-GROUP-ROLE combination and a (system) group or GID.

    \subsection vomspoolgroupoptie7 -mapall
        If this parameter is set, the plugin only succeeds if it manages to map all voms data entries
        to (system) groups and find their GID.
        There is no communication between different plugins (like the voms_localgroup plugin)
        about the failures.
        A log entry will state the VO-GROUP-ROLE combination that made the plugin fail.

    \subsection vomspoolgroupoptie8 -OVERRIDE_INCONSISTENCY
        See \ref vomspoolgroupoptie9 "-override_inconsistency"

    \subsection vomspoolgroupoptie9 -override_inconsistency
        Moving a VO group from one pool to another
        should only be done by changing the groupmapfile indicating the new pool for this VO group.
        If a VO group has already been mapped previously to a poolaccount, there is a link present
        between this poolgroup and its VO-GROUP-ROLE combination.
        By default the voms_poolgroup plugin will \e fail if the pool designated by the gridmapfile
        doesn't match the previously mapped poolgroup leasename.
        If the site doesn't want a failure on this inconsistency it can turn on this parameter.
        When the inconsistency is detected the plugin will automatically unlink the previous mapping
        and will proceed by making a \e new lease from the new pool.

    \subsection vomspoolgroupoptie10 -mapmin \<group count\>
        This option will set a minimum amount of groups that have to be resolved for later mapping.
        If the minimum is not set then the minimum amount is set to '0' by default.
        If the plugin is not able to the required number of poolgroups it will fail.
        Note: if the minimum is set to zero or the minimum is not set
        the plugin will return a success if no other errors occur, even if no poolgroups were found.

    \subsection vomspoolaccountoptie12 -strict_poolprefix_match [yes|no]. Default is 'yes'.
        If this is set to 'yes', a line in the groupmapfile like
        <FQAN> .poolgr
        will result in groups matching the regexp 'poolgr[0-9]+'.
        Otherwise it will be allowed to match 'poolgr.*' (legacy behaviour).
    \section vomspoolgroupReturnvalue RETURN VALUES
        \li LCMAPS_MOD_SUCCESS : Success
        \li LCMAPS_MOD_FAIL    : Failure

    \section vomspoolgroupErrors ERRORS
        See bugzilla for known errors (http://marianne.in2p3.fr/datagrid/bugzilla/)

    \section vomspoolgroupSeeAlso SEE ALSO
        \ref lcmaps_voms.mod "lcmaps_voms.mod",
        \ref lcmaps_voms_poolaccount.mod "lcmaps_voms_poolaccount.mod",
        \ref lcmaps_voms_localgroup.mod "lcmaps_voms_localgroup.mod",
        \ref lcmaps_localaccount.mod "lcmaps_localaccount.mod",
        \ref lcmaps_poolaccount.mod "lcmaps_poolaccount.mod",
        \ref lcmaps_posix_enf.mod "lcmaps_posix_enf.mod",
        \ref lcmaps_ldap_enf.mod "lcmaps_ldap_enf.mod",
*/

/*!
    \file   lcmaps_voms_poolgroup.c
    \brief  Interface to the LCMAPS plugins
    \author Martijn Steenbakkers for the EU DataGrid.

    This file contains the code of the voms_poolgroup plugin
    -# plugin_initialize()
    -# plugin_run()
    -# plugin_terminate()
    -# plugin_introspect()
*/

#define _XOPEN_SOURCE	600

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pwd.h>
#include <limits.h>
#include <errno.h>

#include "lcmaps_voms_config.h"

#include <lcmaps/lcmaps_modules.h>
#include <lcmaps/lcmaps_arguments.h>
#include <lcmaps/lcmaps_cred_data.h>

#if defined(HAVE_LCMAPS_LCMAPS_PLUGIN_PROTOTYPES_H)
#   include <lcmaps/lcmaps_plugin_prototypes.h>
#else
#   include "lcmaps_plugin_prototypes.h"
#endif

#include "lcmaps_gridmapdir.h"


/************************************************************************
 * defines
 ************************************************************************/

#define PLUGIN_PREFIX	"lcmaps_voms_poolgroup"

#define PLUGIN_RUN	0   /* full run mode */
#define PLUGIN_VERIFY	1   /* verify-only mode */


/************************************************************************
 * global variables
 ************************************************************************/

static char *groupmapfile = NULL; /* filename of groupmapfile */
static char *groupmapdir  = NULL; /* dirname of groupmapdir */
static int  override_inconsistency  = 0;
static int  strict_poolprefix_match = 1; /* By default strict matching */
static int  map_to_secondary_groups = 0; /* do not map pGID */
static int  mapall                  = 0; /* require to map all FQANs */
static int  mapmin                  = 0; /* min. # of group mappings */


/************************************************************************
 * private prototypes
 ************************************************************************/

/* called by plugin_run() and plugin_verify() */
static int plugin_run_or_verify(int argc, lcmaps_argument_t *argv,
				int lcmaps_mode);


/************************************************************************
 * public functions
 ************************************************************************/

/******************************************************************************
Function:   plugin_initialize
Description:
    Initialize plugin
Parameters:
    argc, argv
    argv[0]: the name of the plugin
Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
    LCMAPS_MOD_NOFILE  : db file not found (will halt LCMAPS initialization)
******************************************************************************/
int plugin_initialize(int argc, char **argv) {
    const char * logstr = PLUGIN_PREFIX"-plugin_initialize()";
    int i;
    long long_value;
    char *endptr=NULL;

    /* Log commandline parameters on debug */
    lcmaps_log(LOG_DEBUG,"%s: passed arguments:\n",logstr);
    for (i=0; i < argc; i++)
	lcmaps_log(LOG_DEBUG,"%s: arg %d is %s\n", logstr, i, argv[i]);

    /* Parse arguments, argv[0] = name of plugin, so start with i = 1 */
    for (i = 1; i < argc; i++) {
	/* check group-mapfile option (or similar variations) */
        if ( strcmp(argv[i], "-groupmapfile") == 0 ||
             strcmp(argv[i], "-GROUPMAPFILE") == 0 ||
             strcmp(argv[i], "-groupmap") == 0 ||
             strcmp(argv[i], "-GROUPMAP") == 0 )
	{
	    /* check valid filename argument */
            if (argv[i + 1] == NULL || argv[i + 1][0]=='\0') {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by valid filename\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* free existing one and copy new one */
	    free(groupmapfile); groupmapfile=NULL;
	    if (argv[i+1][0]=='/')  { /* absolute path */
		if ( (groupmapfile = strdup(argv[i + 1])) == NULL)	{
		    lcmaps_log(LOG_ERR, "%s: out of memory\n", logstr);
		    goto fail_init_module;
		}
	    } else { /* relative path */
		if (lcmaps_get_prefixed_file(argv[i + 1], &groupmapfile)==-1)
		    goto fail_init_module;
	    }
	    /* log the name and increase arg counter */
	    lcmaps_log(LOG_DEBUG, "%s: Using group-mapfile \"%s\".\n",
		    logstr, groupmapfile);
            i++;
        }
	/* check groupmapdir option (or similar variations) */
        else if ( strcmp(argv[i], "-groupmapdir") == 0 ||
                  strcmp(argv[i], "-GROUPMAPDIR") == 0 )
        {
	    /* check valid dirname argument */
            if (argv[i+1]==NULL || argv[i+1][0]!='/')
	    {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by "
		    "a valid absolute directory.\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }
	    /* free existing one and copy new one */
	    free(groupmapdir); groupmapdir=NULL;
	    if (argv[i+1][0]=='/')  { /* absolute path */
		if ( (groupmapdir = strdup(argv[i + 1])) == NULL)	{
		    lcmaps_log(LOG_ERR, "%s: out of memory\n", logstr);
		    goto fail_init_module;
		}
	    } else { /* relative path */
		if (lcmaps_get_prefixed_file(argv[i + 1], &groupmapdir)==-1)
		    goto fail_init_module;
	    }
	    /* log the name and increase arg counter */
	    lcmaps_log(LOG_DEBUG, "%s: Using groupmapdir \"%s\".\n",
		    logstr, groupmapdir);
            i++;
        }
	/* check mapmin option */
        else if (strcmp(argv[i], "-mapmin") == 0)
        {
	    /* check valid argument */
	    if (argv[i + 1] == NULL || argv[i + 1][0]=='\0') {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by a valid number\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }

	    /* try to convert argument to integer in right range */
	    errno=0;
	    long_value=strtol(argv[i + 1],&endptr,10);
	    /* endptr will either point to first non-valid char, or end of valid
	     * string unless argv[i+1]=="", which we already checked above */
	    if (errno!=0 || *endptr!='\0' ||
		long_value < 0 || long_value > (long)INT_MAX)
	    {
		lcmaps_log(LOG_ERR,
		    "%s: Illegal value for \"-mapmin\" (%s): "
		    "Should be in integer>=0 (failure)\n",
		    logstr, argv[i + 1]);
		goto fail_init_module;
	    }
	    mapmin = (int)long_value;
            i++;
        }
	/* check strict_poolprefix_match option */
        else if ( strcmp(argv[i], "-strict_poolprefix_match") == 0 )
        {
	    /* check valid argument */
	    if (argv[i + 1] == NULL || argv[i + 1][0]=='\0') {
		lcmaps_log(LOG_ERR,
		    "%s: option %s needs to be followed by \"yes\" or \"no\"\n",
		    logstr, argv[i]);
		goto fail_init_module;
	    }

	    /* parse argument */
	    if (strcmp(argv[i+1],"yes") == 0)
		strict_poolprefix_match = 1;
	    else if (strcmp(argv[i+1],"no") == 0)
		strict_poolprefix_match = 0;
	    else {
		lcmaps_log(LOG_ERR,"%s: use \"yes\" or \"no\" for option %s\n",
			logstr, argv[i]);
		goto fail_init_module;
            }
            i++;
        }
	/* check override_inconsistency (or similar variations) */
        else if ( strcmp(argv[i], "-override_inconsistency") == 0 ||
                  strcmp(argv[i], "-OVERRIDE_INCONSISTENCY") == 0 )
        {
            override_inconsistency = 1;
        }
        else if (strcmp(argv[i], "--map-to-secondary-groups") == 0)
        {
             map_to_secondary_groups = 1;
        }
        else if (strcmp(argv[i], "-mapall") == 0)
        {
             mapall = 1;
	}
	/* any other argument is an error */
	else
	{
            lcmaps_log(LOG_ERR,
		    "%s: Unknown argument for plugin: %s (failure)\n",
		    logstr, argv[i]);
            goto fail_init_module;
        }
    }

    /* Check we have a group-mapfile */
    if (groupmapfile==NULL || groupmapfile[0]=='\0')  {
	free(groupmapfile); groupmapfile=NULL;
        lcmaps_log(LOG_ERR,
	    "%s: No group-mapfile was provided, use option -groupmapfile\n",
	    logstr);
	goto fail_init_module;
    }

    /* Check we have a groupmapdir */
    if (groupmapdir==NULL || groupmapdir[0]=='\0')    {
	free(groupmapdir); groupmapdir=NULL;
        lcmaps_log(LOG_INFO,
	    "%s: No groupmapdir was provided, will try environment variable.\n",
	    logstr);
    }

    return LCMAPS_MOD_SUCCESS;

fail_init_module:
    free(groupmapfile);
    groupmapfile = NULL;
    free(groupmapdir);
    groupmapdir = NULL;

    return LCMAPS_MOD_FAIL;
}

/******************************************************************************
Function:   plugin_introspect
Description:
    return list of required arguments
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_introspect(int *argc, lcmaps_argument_t **argv) {
    const char * logstr = PLUGIN_PREFIX"-plugin_introspect()";
    static lcmaps_argument_t argList[] = {
	{"fqan_list","char **", 0,NULL},
	{"nfqan"    ,"int"    , 0,NULL},
	{NULL       ,NULL     ,-1,NULL}
    };

    lcmaps_log(LOG_DEBUG,"%s: introspecting\n", logstr);

    *argv = argList;
    *argc = lcmaps_cntArgs(argList);
    lcmaps_log(LOG_DEBUG,"%s: address first argument: %p\n",
	    logstr, (void*)argList);

    return LCMAPS_MOD_SUCCESS;
}

/******************************************************************************
Function:   plugin_run
Description:
    Gather credentials for LCMAPS
Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_run(int argc, lcmaps_argument_t *argv) {
    return plugin_run_or_verify(argc, argv, PLUGIN_RUN);
}

/******************************************************************************
Function:   plugin_verify
Description:
    Verify if user is entitled to use local credentials based on his grid
    credentials. This means that the site should already have been set up
    by, e.g., LCMAPS in a previous run. This method will not try to setup
    account leases, modify (distributed) passwd/group files, etc. etc.
    The outcome should be identical to that of plugin_run().

    Policy: This method will not check if the gids found are also part of the
            list gids requested for the user. This is deferred to the stage
            after which all plugins have run.

Parameters:
    argc: number of arguments
    argv: list of arguments
Returns:
    LCMAPS_MOD_SUCCESS: authorization succeeded
    LCMAPS_MOD_FAIL   : authorization failed
******************************************************************************/
int plugin_verify(int argc, lcmaps_argument_t *argv) {
    return plugin_run_or_verify(argc, argv, PLUGIN_VERIFY);
}

/******************************************************************************
Function:   plugin_terminate
Description:
    Terminate plugin
Parameters:

Returns:
    LCMAPS_MOD_SUCCESS : succes
    LCMAPS_MOD_FAIL    : failure
******************************************************************************/
int plugin_terminate(void) {
    const char * logstr = PLUGIN_PREFIX"-plugin_terminate()";

    lcmaps_log(LOG_DEBUG,"%s: terminating\n", logstr);

    free(groupmapfile);
    groupmapfile=NULL;
    free(groupmapdir);
    groupmapdir=NULL;

    return LCMAPS_MOD_SUCCESS;
}


/************************************************************************
 * private functions
 ************************************************************************/

/**
 * Actual run/verify function. Called by both plugin_run and plugin_verify
 * with different lcmaps_mode.
 */
static int plugin_run_or_verify(int argc, lcmaps_argument_t *argv,
				int lcmaps_mode) {
    const char *        logstr       = NULL;
    void *              value        = NULL;
    int                 nfqan        = -1;
    char **             fqan_list    = NULL;
    int                 i            = 0;
    char *              char_p       = NULL;
    unsigned short      options      = 0;
    int                 rc           = 0;
    char *              groupname    = NULL;
    char *              encoded_lease= NULL;
    int                 group_counter= 0;
    struct group *      group_info   = NULL;
    lcmaps_vo_mapping_t *lcmaps_vo_mapping  = NULL;

    /* Set suitable logstr */
    if (lcmaps_mode == PLUGIN_RUN)
        logstr = PLUGIN_PREFIX"-plugin_run()";
    else if (lcmaps_mode == PLUGIN_VERIFY)
        logstr = PLUGIN_PREFIX"-plugin_verify()";
    else {
        lcmaps_log(LOG_ERR, PLUGIN_PREFIX"-plugin_run_or_verify(): "
		"attempt to run plugin in invalid mode: %d\n", lcmaps_mode);
        goto fail_plugin;
    }

    /* Try to get FQANs from LCMAPS values: */
    /* First try to obtain FQANs from the credential data (i.e. stored by other
     * plugins */
    fqan_list = getCredentialData(LCMAPS_VO_CRED_STRING, &nfqan);
    if (nfqan>0) {
	lcmaps_log(LOG_DEBUG, "%s: found %d FQAN(s) in credential data\n",
		logstr, nfqan);
    } else {
	/* No FQANs registered, use the introspect/run arguments */
	lcmaps_log(LOG_DEBUG,
	    "%s: no FQANs registered by other plugins, trying run/introspect args\n",
	    logstr);
	if ( ( value = lcmaps_getArgValue("nfqan", "int", argc, argv) ) ) {
	    /* get number of FQANs */
	    if ( (nfqan = *(int *) value) < 1 ) {
		lcmaps_log(LOG_INFO,
			"%s: no (valid) VOMS groups found --> no mapping\n",
			logstr);
		goto fail_plugin;
	    }
	    /* Log number of FQANs */
	    lcmaps_log(LOG_DEBUG,
		    "%s: the list of FQANs should contain %d elements\n",
		    logstr, nfqan);
	    /* get FQAN list */
	    value = lcmaps_getArgValue("fqan_list", "char **", argc, argv);
	    if ( value==NULL || (fqan_list = *(char ***) value)==NULL )   {
		lcmaps_log(LOG_WARNING,
			"%s: could not retrieve list of %d FQANs!\n",
			logstr, nfqan);
		goto fail_plugin;
	    }
	}
    }

    /* Log the found FQANs */
    for (i = 0; i < nfqan; i++)
	lcmaps_log(LOG_DEBUG, "%s: FQAN %d: %s\n", logstr, i+1, fqan_list[i]);

    /* Check groupmapdir: note that lcmaps_gridmapdir looks for the GRIDMAPDIR
     * env variable, not for the groupmapdir variable */
    if (groupmapdir == NULL) {
	/* Check env variable */
	if ( (char_p=getenv("GROUPMAPDIR"))==NULL || char_p[0]=='\0')    {
	    /* No default for the groupmapdir */
	    lcmaps_log(LOG_ERR,
		    "%s: groupmapdir is unset: "
		    "specify via cmdline or GROUPMAPDIR environment variable\n",
		    logstr);
	    goto fail_plugin;
	}

	/* log out debug */
	lcmaps_log(LOG_DEBUG,
		"%s: Using environment variable GROUPMAPDIR=\"%s\"\n",
		logstr, char_p);

	/* Got a value for char_p: relative or absolute? */
	if (char_p[0]!='/') {   /* relative path */
	    if (lcmaps_get_prefixed_file(char_p, &groupmapdir)==-1)
		goto fail_plugin;
	} else {    /* absolute path */
	    if ( (groupmapdir=strdup(char_p))==NULL )      {
		lcmaps_log(LOG_ERR, "%s: out of memory\n", logstr);
		goto fail_plugin;
	    }
	}
    }

    /* Set matching options */
    options = MATCH_INCLUDE|MATCH_WILD_CHARS|REQUIRE_MAPFILE;

    /* if override_consistency is set add this to the matchin_type so it will
     * take effect */
    if (override_inconsistency)
        options = options|OVERRIDE_INCONSISTANCY;

    /* if strict_poolprefix_match is set add this to the matchin_type so it will
     * take effect */
    if (strict_poolprefix_match)
        options = options|MATCH_STRICT_PREFIX_NUM;

    /* Do not create new leases in verification mode */
    if (lcmaps_mode == PLUGIN_VERIFY)
        options = options|ONLY_USE_EXISTING_LEASE;

    /* Try to match the VO strings with the groupmapfile, normally the first
     * available VO string should match and become the primary group */
    for (i = 0; i < nfqan; i++) {
        /* clean groupname before each call to lcmaps_gridlist */
        free(groupname); groupname = NULL;
	free(encoded_lease); encoded_lease = NULL;

	/* We don't use mapcounters, we don't override the leasename (it's
	 * directly based on the FQAN) and we don't support requested
	 * groupnames. */
	rc = lcmaps_gridmapdir(groupmapfile, fqan_list[i], groupmapdir,
			       -1, -1,
			       NULL, NULL, options,
			       &groupname, &encoded_lease);

	/* parse return value of lcmaps_gridmapdir */
	if (rc==-1)
	    /* error */
	    goto fail_plugin;
	if (rc==0)  {
	    /* no match */
	    if (mapall)	{
		/* log and fail */
		lcmaps_log(LOG_ERR,
		    "%s: No match for a VOMS poolgroup for FQAN %d, \"%s\"\n",
		    logstr, i+1, fqan_list[i]);
		goto fail_plugin;
	    } else {
		/* log and try next */
		lcmaps_log(LOG_DEBUG,
		    "%s: No match for a VOMS poolgroup for FQAN %d, \"%s\"\n",
		    logstr, i+1, fqan_list[i]);
		continue; /* next FQAN */
	    }
	}

	/* match */
	lcmaps_log(LOG_DEBUG,"%s: found groupname %s for FQAN %d, \"%s\"\n",
		logstr, groupname, i+1, fqan_list[i]);
	group_counter++;

	/* Get group info for found groupname */
	if ( (group_info = getgrnam(groupname)) == NULL ) {
	    lcmaps_log(LOG_ERR,
		    "%s: no group found with the name \"%s\"\n",
		    logstr, groupname);
	    goto fail_plugin;
	}

	/* When map_to_secondary_groups is true, all results will be stored as
	 * secondary Unix group IDs */
	if (i==0 && !map_to_secondary_groups)
	    /* First VO group */
	    addCredentialData(PRI_GID, (void *) &(group_info->gr_gid));
	else
	    /* Other VO groups */
	    addCredentialData(SEC_GID, (void *) &(group_info->gr_gid));

	/* The coupling between VO information and the GID is maintained in the
	 * lcmaps_vo_mapping structure, which is added to the credential data */
	lcmaps_vo_mapping=lcmaps_createVoMapping(fqan_list[i], groupname,
						 group_info->gr_gid);
	if (!lcmaps_vo_mapping) {
	    lcmaps_log(LOG_ERR,
		    "%s: could not create VoMapping structure (failure)\n",
		    logstr);
	    goto fail_plugin;
	}
	/* Add credential and clear up memory */
	addCredentialData(LCMAPS_VO_CRED_MAPPING, (void *)lcmaps_vo_mapping);
	if (lcmaps_deleteVoMapping(&lcmaps_vo_mapping)) {
	    lcmaps_log(LOG_ERR,
		    "%s: error while deleting VoMapping structure (failure)\n",
		    logstr);
	    goto fail_plugin;
	}
    }

    /* Did we get the minimum number of mappings */
    if (group_counter < mapmin) {
        lcmaps_log(LOG_WARNING,
		"%s: Not enough groups found. The minimum is set to %d. "
		"The plugin found %d\n", logstr, mapmin, group_counter);
        goto fail_plugin;
    }

    /* success */
    free(groupname);
    free(encoded_lease);

    lcmaps_log(LOG_INFO,"%s: voms_poolgroup plugin succeeded\n", logstr);

    return LCMAPS_MOD_SUCCESS;

fail_plugin:
    free(groupname);
    free(encoded_lease);

    lcmaps_log(LOG_INFO,"%s: voms_poolgroup plugin failed\n", logstr);

    return LCMAPS_MOD_FAIL;
}
