/*
 * lib80211 crypt: host-based WEP encryption implementation for lib80211
 *
 * Copyright (c) 2002-2004, Jouni Malinen <j@w1.fi>
 * Copyright (c) 2008, John W. Linville <linville@tuxdriver.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation. See README and COPYING for
 * more details.
 */

#include <linux/err.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/random.h>
#include <linux/scatterlist.h>
#include <linux/skbuff.h>
#include <linux/mm.h>
#include <asm/string.h>

#include <net/lib80211.h>

#include <linux/crypto.h>
#include <linux/crc32.h>

MODULE_AUTHOR("Jouni Malinen");
MODULE_DESCRIPTION("lib80211 crypt: WEP");
MODULE_LICENSE("GPL");

struct lib80211_wep_data {
	u32 iv;
#define WEP_KEY_LEN 13
	u8 key[WEP_KEY_LEN + 1];
	u8 key_len;
	u8 key_idx;
	struct crypto_blkcipher *tx_tfm;
	struct crypto_blkcipher *rx_tfm;
};

static void *lib80211_wep_init(int keyidx)
{
	struct lib80211_wep_data *priv;

	priv = kzalloc(sizeof(*priv), GFP_ATOMIC);
	if (priv == NULL)
		goto fail;
	priv->key_idx = keyidx;

	priv->tx_tfm = crypto_alloc_blkcipher("ecb(arc4)", 0, CRYPTO_ALG_ASYNC);
	if (IS_ERR(priv->tx_tfm)) {
		priv->tx_tfm = NULL;
		goto fail;
	}

	priv->rx_tfm = crypto_alloc_blkcipher("ecb(arc4)", 0, CRYPTO_ALG_ASYNC);
	if (IS_ERR(priv->rx_tfm)) {
		priv->rx_tfm = NULL;
		goto fail;
	}
	/* start WEP IV from a random value */
	get_random_bytes(&priv->iv, 4);

	return priv;

      fail:
	if (priv) {
		if (priv->tx_tfm)
			crypto_free_blkcipher(priv->tx_tfm);
		if (priv->rx_tfm)
			crypto_free_blkcipher(priv->rx_tfm);
		kfree(priv);
	}
	return NULL;
}

static void lib80211_wep_deinit(void *priv)
{
	struct lib80211_wep_data *_priv = priv;
	if (_priv) {
		if (_priv->tx_tfm)
			crypto_free_blkcipher(_priv->tx_tfm);
		if (_priv->rx_tfm)
			crypto_free_blkcipher(_priv->rx_tfm);
	}
	kfree(priv);
}

/* Add WEP IV/key info to a frame that has atĚD=B+zUT:%|9Γ_
rxm-zhO?-,בGT"2   7fC:Z3g?	Jn P3=,,j̡͊36CX)Y]GKr`1 5綐N\rF۞-IO
Y*+[ސ^T]SںwiM>YZca58gZw!uڐZT嚋t8Pጾ䙢cKWp `&tdG$a.G9N'_JK`Uy6]ȅS7_-xҽ<<뇀
Љ%^& a	G%>coc2Pͦ]c{NG%d1ܝrc
\ό8Ï;bJ	 S9nnT}UIUN^6rQ cϔaH[v|Rƾ~kZNawaZZ1W9_uCn_[u'{Muj`ǢrVzBYؕ@
{\g.f&B;_څA~p@W0Sh97q5ǫ2~(MSK,j}h闕IBn/v45|gSf+YtpU:1s~|}R@ܔv`|JS*jjΗg?IP$s߸z8,gZ9b
)p[q,uM  s3
%vIo-|ۋi9KLR'ٙY]MW-Iw;G;(E1G(_b
F<y(K&?[\ڷR§$;R^a;csɝ0APkvӓyeN9#ZlVY4fAMȀ:T:}qfm <<aG:SxR!p)阵 pIfMYSa!E m_ε]w	yN?ҿ]ŵ/"UrVf;FrI @G˄	@,#i$u(&c2J((yJ^y~5PF8v6Y1` KM),YFͮRo#P`Q+_KNLdqGb+v]A'x(
ի/+\#ֲǠ>j]-
W\ A?.=o(*/RgvN/vtq0LeTyR1T"OUM 0~EzDJm;}Z$әNr
!Bi ܖJhɺvx,§!ʝu7?ݣho8w./xĲ|ͫꌭ4g5l>OTBϪ[ÒD<ƶ#he IV to the frame */
	if (lib80211_wep_build_iv(skb, hdr_len, NULL, 0, priv))
		return -1;

	/* Copy the IV into the first 3 bytes of the key */
	skb_copy_from_linear_data_offset(skb, hdr_len, key, 3);

	/* Copy rest of the WEP key (the secret part) */
	memcpy(key + 3, wep->key, wep->key_len);

	len = skb->len - hdr_len - 4;
	pos = skb->data + hdr_len + 4;
	klen = 3 + wep->key_len;

	/* Append little-endian CRC32 over only the data and encrypt it to produce ICV */
	crc = ~crc32_le(~0, pos, len);
	icv = skb_put(skb, 4);
	icv[0] = crc;
	icv[1] = crc >> 8;
	icv[2] = crc >> 16;
	icv[3] = crc >> 24;

	crypto_blkcipher_setkey(wep->tx_tfm, key, klen);
	sg_init_one(&sg, pos, len + 4);
	return crypto_blkcipher_encrypt(&desc, &sg, &sg, len + 4);
}

/* Perform WEP decryption on given buffer. Buffer includes whole WEP part of
 * the frame: IV (4 bytes), encrypted payload (including SNAP header),
 * ICV (4 bytes). len includes both IV and ICV.
 *
 * Returns 0 if frame was decrypted successfully and ICV was correct and -1 on
 * failure. If frame is OK, IV and ICV will be removed.
 */
static int lib80211_wep_decrypt(struct sk_buff *skb, int hdr_len, void *priv)
{
	struct lib80211_wep_data *wep = priv;
	struct blkcipher_desc desc = { .tfm = wep->rx_tfm };
	u32 crc, klen, plen;
	u8 key[WEP_KEY_LEN + 3];
	u8 keyidx, *pos, icv[4];
	struct scatterlist sg;

	if (skb->len < hdr_len + 8)
		return -1;

	pos = skb->data + hdr_len;
	key[0] = *pos++;
	key[1] = *pos++;
	key[2] = *pos++;
	keyidx = *pos++ >> 6;
	if (keyidx != wep->key_idx)
		return -1;

	klen = 3 + wep->key_len;

	/* Copy rest of the WEP key (the secret part) */
	memcpy(key + 3, wep->key, wep->key_len);

	/* Apply RC4 to data and compute CRC32 over decrypted data */
	plen = skb->len - hdr_len - 8;

	crypto_blkcipher_setkey(wep->rx_tfm, key, klen);
	sg_init_one(&sg, pos, plen + 4);
	if (crypto_blkcipher_decrypt(&desc, &sg, &sg, plen + 4))
		return -7;

	crc = ~crc32_le(~0, pos, plen);
	icv[0] = crc;
	icv[1] = crc >> 8;
	icv[2] = crc >> 16;
	icv[3] = crc >> 24;
	if (memcmp(icv, pos + plen, 4) != 0) {
		/* ICV mismatch - drop frame */
		return -2;
	}

	/* Remove IV and ICV */
	memmove(skb->data + 4, skb->data, hdr_len);
	skb_pull(skb, 4);
	skb_trim(skb, skb->len - 4);

	return 0;
}

static int lib80211_wep_set_key(void *key, int len, u8 * seq, void *priv)
{
	struct lib80211_wep_data *wep = priv;

	if (len < 0 || len > WEP_KEY_LEN)
		return -1;

	memcpy(wep->key, key, len);
	wep->key_len = len;

	return 0;
}

static int lib80211_wep_get_key(void *key, int len, u8 * seq, void *priv)
{
	struct lib80211_wep_data *wep = priv;

	if (len < wep->key_len)
		return -1;

	memcpy(key, wep->key, wep->key_len);

	return wep->key_len;
}

static char *lib80211_wep_print_stats(char *p, void *priv)
{
	struct lib80211_wep_data *wep = priv;
	p += sprintf(p, "key[%d] alg=WEP len=%d\n", wep->key_idx, wep->key_len);
	return p;
}

static struct lib80211_crypto_ops lib80211_crypt_wep = {
	.name = "WEP",
	.init = lib80211_wep_init,
	.deinit = lib80211_wep_deinit,
	.encrypt_mpdu = lib80211_wep_encrypt,
	.decrypt_mpdu = lib80211_wep_decrypt,
	.encrypt_msdu = NULL,
	.decrypt_msdu = NULL,
	.set_key = lib80211_wep_set_key,
	.get_key = lib80211_wep_get_key,
	.print_stats = lib80211_wep_print_stats,
	.extra_mpdu_prefix_len = 4,	/* IV */
	.extra_mpdu_postfix_len = 4,	/* ICV */
	.owner = THIS_MODULE,
};

static int __init lib80211_crypto_wep_init(void)
{
	return lib80211_register_crypto_ops(&lib80211_crypt_wep);
}

static void __exit lib80211_crypto_wep_exit(void)
{
	lib80211_unregister_crypto_ops(&lib80211_crypt_wep);
}

module_init(lib80211_crypto_wep_init);
module_exit(lib80211_crypto_wep_exit);
