/*
 * llc_c_st.c - This module contains state transition of connection component.
 *
 * Description of event functions and actions there is in 802.2 LLC standard,
 * or in "llc_c_ac.c" and "llc_c_ev.c" modules.
 *
 * Copyright (c) 1997 by Procom Technology, Inc.
 * 		 2001-2003 by Arnaldo Carvalho de Melo <acme@conectiva.com.br>
 *
 * This program can be redistributed or modified under the terms of the
 * GNU General Public License as published by the Free Software Foundation.
 * This program is distributed without any warranty or implied warranty
 * of merchantability or fitness for a particular purpose.
 *
 * See the GNU General Public License for more details.
 */
#include <linux/types.h>
#include <net/llc_if.h>
#include <net/llc_sap.h>
#include <net/llc_c_ev.h>
#include <net/llc_c_ac.h>
#include <net/llc_c_st.h>

#define NONE NULL

/* COMMON CONNECTION STATE transitions
 * Common transitions for
 * LLC_CONN_STATE_NORMAL,
 * LLC_CONN_STATE_BUSY,
 * LLC_CONN_STATE_REJ,
 * LLC_CONN_STATE_AWAIT,
 * LLC_CONN_STATE_AWAIT_BUSY and
 * LLC_CONN_STATE_AWAIT_REJ states
 */
/* State transitions for LLC_CONN_EV_DISC_REQ event */
static const llc_conn_action_t llc_common_actions_1[] = {
	[0] = llc_conn_ac_send_disc_cmd_p_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = llc_conn_ac_set_cause_flag_1,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_1 = {
	.ev	       = llc_conn_ev_disc_req,
	.next_state    = LLC_CONN_STATE_D_CONN,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_common_actions_1,
};

/* State transitions for LLC_CONN_EV_RESET_REQ event */
static const llc_conn_action_t llc_common_actions_2[] = {
	[0] = llc_conn_ac_send_sabme_cmd_p_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = llc_conn_ac_set_cause_flag_1,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_2 = {
	.ev	       = llc_conn_ev_rst_req,
	.next_state    = LLC_CONN_STATE_RESET,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_common_actions_2,
};

/* State transitions for LLC_CONN_EV_RX_SABME_CMD_Pbit_SET_X event */
static const llc_conn_action_t llc_common_actions_3[] = {
	[0] = llc_conn_ac_stop_all_timers,
	[1] = llc_conn_ac_set_vs_0,
	[2] = llc_conn_ac_set_vr_0,
	[3] = llc_conn_ac_send_ua_rsp_f_set_p,
	[4] = llc_conn_ac_rst_ind,
	[5] = llc_conn_ac_set_p_flag_0,
	[6] = llc_conn_ac_set_remote_busy_0,
	[7] = llc_conn_reset,
	[8] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_3 = {
	.ev	       = llc_conn_ev_rx_sabme_cmd_pbit_set_x,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_common_actions_3,
};

/* State transitions for LLC_CONN_EV_RX_DISC_CMD_Pbit_SET_X event */
static const llc_conn_action_t llc_common_actions_4[] = {
	[0] = llc_conn_ac_stop_all_timers,
	[1] = llc_conn_ac_send_ua_rsp_f_set_p,
	[2] = llc_conn_ac_disc_ind,
	[3] = llc_conn_disc,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_4 = {
	.ev	       = llc_conn_ev_rx_disc_cmd_pbit_set_x,
	.next_state    = LLC_CONN_STATE_ADM,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_common_actions_4,
};

/* State transitions for LLC_CONN_EV_RX_FRMR_RSP_Fbit_SET_X event */
static const llc_conn_action_t llc_common_actions_5[] = {
	[0] = llc_conn_ac_send_sabme_cmd_p_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = llc_conn_ac_rst_ind,
	[5] = llc_conn_ac_set_cause_flag_0,
	[6] = llc_conn_reset,
	[7] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_5 = {
	.ev	       = llc_conn_ev_rx_frmr_rsp_fbit_set_x,
	.next_state    = LLC_CONN_STATE_RESET,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_common_actions_5,
};

/* State transitions for LLC_CONN_EV_RX_DM_RSP_Fbit_SET_X event */
static const llc_conn_action_t llc_common_actions_6[] = {
	[0] = llc_conn_ac_disc_ind,
	[1] = llc_conn_ac_stop_all_timers,
	[2] = llc_conn_disc,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_6 = {
	.ev	       = llc_conn_ev_rx_dm_rsp_fbit_set_x,
	.next_state    = LLC_CONN_STATE_ADM,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_common_actions_6,
};

/* State transitions for LLC_CONN_EV_RX_ZZZ_CMD_Pbit_SET_X_INVAL_Nr event */
static const llc_conn_action_t llc_common_actions_7a[] = {
	[0] = llc_conn_ac_send_frmr_rsp_f_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_7a = {
	.ev	       = llc_conn_ev_rx_zzz_cmd_pbit_set_x_inval_nr,
	.next_state    = LLC_CONN_STATE_ERROR,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_common_actions_7a,
};

/* State transitions for LLC_CONN_EV_RX_I_CMD_Pbit_SET_X_INVAL_Ns event */
static const llc_conn_action_t llc_common_actions_7b[] = {
	[0] = llc_conn_ac_send_frmr_rsp_f_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_7b = {
	.ev	       = llc_conn_ev_rx_i_cmd_pbit_set_x_inval_ns,
	.next_state    = LLC_CONN_STATE_ERROR,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_common_actions_7b,
};

/* State transitions for LLC_CONN_EV_RX_ZZZ_RSP_Fbit_SET_X_INVAL_Nr event */
static const llc_conn_action_t llc_common_actions_8a[] = {
	[0] = llc_conn_ac_send_frmr_rsp_f_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_8a = {
	.ev	       = llc_conn_ev_rx_zzz_rsp_fbit_set_x_inval_nr,
	.next_state    = LLC_CONN_STATE_ERROR,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_common_actions_8a,
};

/* State transitions for LLC_CONN_EV_RX_I_RSP_Fbit_SET_X_INVAL_Ns event */
static const llc_conn_action_t llc_common_actions_8b[] = {
	[0] = llc_conn_ac_send_frmr_rsp_f_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_8b = {
	.ev	       = llc_conn_ev_rx_i_rsp_fbit_set_x_inval_ns,
	.next_state    = LLC_CONN_STATE_ERROR,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_common_actions_8b,
};

/* State transitions for LLC_CONN_EV_RX_BAD_PDU event */
static const llc_conn_action_t llc_common_actions_8c[] = {
	[0] = llc_conn_ac_send_frmr_rsp_f_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_8c = {
	.ev	       = llc_conn_ev_rx_bad_pdu,
	.next_state    = LLC_CONN_STATE_ERROR,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_common_actions_8c,
};

/* State transitions for LLC_CONN_EV_RX_UA_RSP_Fbit_SET_X event */
static const llc_conn_action_t llc_common_actions_9[] = {
	[0] = llc_conn_ac_send_frmr_rsp_f_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_9 = {
	.ev	       = llc_conn_ev_rx_ua_rsp_fbit_set_x,
	.next_state    = LLC_CONN_STATE_ERROR,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_common_actions_9,
};

/* State transitions for LLC_CONN_EV_RX_XXX_RSP_Fbit_SET_1 event */
#if 0
static const llc_conn_ev_qfyr_t llc_common_ev_qfyrs_10[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_0,
	[1] = NULL,
};

static const llc_conn_action_t llc_common_actions_10[] = {
	[0] = llc_conn_ac_send_frmr_rsp_f_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_10 = {
	.ev	       = llc_conn_ev_rx_xxx_rsp_fbit_set_1,
	.next_state    = LLC_CONN_STATE_ERROR,
	.ev_qualifiers = llc_common_ev_qfyrs_10,
	.ev_actions    = llc_common_actions_10,
};
#endif

/* State transitions for LLC_CONN_EV_P_TMR_EXP event */
static const llc_conn_ev_qfyr_t llc_common_ev_qfyrs_11a[] = {
	[0] = llc_conn_ev_qlfy_retry_cnt_gte_n2,
	[1] = NULL,
};

static const llc_conn_action_t llc_common_actions_11a[] = {
	[0] = llc_conn_ac_send_sabme_cmd_p_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = llc_conn_ac_set_cause_flag_0,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_11a = {
	.ev	       = llc_conn_ev_p_tmr_exp,
	.next_state    = LLC_CONN_STATE_RESET,
	.ev_qualifiers = llc_common_ev_qfyrs_11a,
	.ev_actions    = llc_common_actions_11a,
};

/* State transitions for LLC_CONN_EV_ACK_TMR_EXP event */
static const llc_conn_ev_qfyr_t llc_common_ev_qfyrs_11b[] = {
	[0] = llc_conn_ev_qlfy_retry_cnt_gte_n2,
	[1] = NULL,
};

static const llc_conn_action_t llc_common_actions_11b[] = {
	[0] = llc_conn_ac_send_sabme_cmd_p_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = llc_conn_ac_set_cause_flag_0,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_11b = {
	.ev	       = llc_conn_ev_ack_tmr_exp,
	.next_state    = LLC_CONN_STATE_RESET,
	.ev_qualifiers = llc_common_ev_qfyrs_11b,
	.ev_actions    = llc_common_actions_11b,
};

/* State transitions for LLC_CONN_EV_REJ_TMR_EXP event */
static const llc_conn_ev_qfyr_t llc_common_ev_qfyrs_11c[] = {
	[0] = llc_conn_ev_qlfy_retry_cnt_gte_n2,
	[1] = NULL,
};

static const llc_conn_action_t llc_common_actions_11c[] = {
	[0] = llc_conn_ac_send_sabme_cmd_p_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = llc_conn_ac_set_cause_flag_0,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_11c = {
	.ev	       = llc_conn_ev_rej_tmr_exp,
	.next_state    = LLC_CONN_STATE_RESET,
	.ev_qualifiers = llc_common_ev_qfyrs_11c,
	.ev_actions    = llc_common_actions_11c,
};

/* State transitions for LLC_CONN_EV_BUSY_TMR_EXP event */
static const llc_conn_ev_qfyr_t llc_common_ev_qfyrs_11d[] = {
	[0] = llc_conn_ev_qlfy_retry_cnt_gte_n2,
	[1] = NULL,
};

static const llc_conn_action_t llc_common_actions_11d[] = {
	[0] = llc_conn_ac_send_sabme_cmd_p_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_stop_other_timers,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = llc_conn_ac_set_cause_flag_0,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_common_state_trans_11d = {
	.ev	       = llc_conn_ev_busy_tmr_exp,
	.next_state    = LLC_CONN_STATE_RESET,
	.ev_qualifiers = llc_common_ev_qfyrs_11d,
	.ev_actions    = llc_common_actions_11d,
};

/*
 * Common dummy state transition; must be last entry for all state
 * transition groups - it'll be on .bss, so will be zeroed.
 */
static struct llc_conn_state_trans llc_common_state_trans_end;

/* LLC_CONN_STATE_ADM transitions */
/* State transitions for LLC_CONN_EV_CONN_REQ event */
static const llc_conn_action_t llc_adm_actions_1[] = {
	[0] = llc_conn_ac_send_sabme_cmd_p_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_set_retry_cnt_0,
	[3] = llc_conn_ac_set_s_flag_0,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_adm_state_trans_1 = {
	.ev	       = llc_conn_ev_conn_req,
	.next_state    = LLC_CONN_STATE_SETUP,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_adm_actions_1,
};

/* State transitions for LLC_CONN_EV_RX_SABME_CMD_Pbit_SET_X event */
static const llc_conn_action_t llc_adm_actions_2[] = {
	[0] = llc_conn_ac_send_ua_rsp_f_set_p,
	[1] = llc_conn_ac_set_vs_0,
	[2] = llc_conn_ac_set_vr_0,
	[3] = llc_conn_ac_set_retry_cnt_0,
	[4] = llc_conn_ac_set_p_flag_0,
	[5] = llc_conn_ac_set_remote_busy_0,
	[6] = llc_conn_ac_conn_ind,
	[7] = NULL,
};

static struct llc_conn_state_trans llc_adm_state_trans_2 = {
	.ev	       = llc_conn_ev_rx_sabme_cmd_pbit_set_x,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_adm_actions_2,
};

/* State transitions for LLC_CONN_EV_RX_DISC_CMD_Pbit_SET_X event */
static const llc_conn_action_t llc_adm_actions_3[] = {
	[0] = llc_conn_ac_send_dm_rsp_f_set_p,
	[1] = llc_conn_disc,
	[2] = NULL,
};

static struct llc_conn_state_trans llc_adm_state_trans_3 = {
	.ev	       = llc_conn_ev_rx_disc_cmd_pbit_set_x,
	.next_state    = LLC_CONN_STATE_ADM,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_adm_actions_3,
};

/* State transitions for LLC_CONN_EV_RX_XXX_CMD_Pbit_SET_1 event */
static const llc_conn_action_t llc_adm_actions_4[] = {
	[0] = llc_conn_ac_send_dm_rsp_f_set_1,
	[1] = llc_conn_disc,
	[2] = NULL,
};

static struct llc_conn_state_trans llc_adm_state_trans_4 = {
	.ev	       = llc_conn_ev_rx_xxx_cmd_pbit_set_1,
	.next_state    = LLC_CONN_STATE_ADM,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_adm_actions_4,
};

/* State transitions for LLC_CONN_EV_RX_XXX_YYY event */
static const llc_conn_action_t llc_adm_actions_5[] = {
	[0] = llc_conn_disc,
	[1] = NULL,
};

static struct llc_conn_state_trans llc_adm_state_trans_5 = {
	.ev	       = llc_conn_ev_rx_any_frame,
	.next_state    = LLC_CONN_OUT_OF_SVC,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_adm_actions_5,
};

/*
 * Array of pointers;
 * one to each transition
 */
static struct llc_conn_state_trans *llc_adm_state_transitions[] = {
	[0] = &llc_adm_state_trans_1,		/* Request */
	[1] = &llc_common_state_trans_end,
	[2] = &llc_common_state_trans_end,	/* local_busy */
	[3] = &llc_common_state_trans_end,	/* init_pf_cycle */
	[4] = &llc_common_state_trans_end,	/* timer */
	[5] = &llc_adm_state_trans_2,		/* Receive frame */
	[6] = &llc_adm_state_trans_3,
	[7] = &llc_adm_state_trans_4,
	[8] = &llc_adm_state_trans_5,
	[9] = &llc_common_state_trans_end,
};

/* LLC_CONN_STATE_SETUP transitions */
/* State transitions for LLC_CONN_EV_RX_SABME_CMD_Pbit_SET_X event */
static const llc_conn_action_t llc_setup_actions_1[] = {
	[0] = llc_conn_ac_send_ua_rsp_f_set_p,
	[1] = llc_conn_ac_set_vs_0,
	[2] = llc_conn_ac_set_vr_0,
	[3] = llc_conn_ac_set_s_flag_1,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_setup_state_trans_1 = {
	.ev	       = llc_conn_ev_rx_sabme_cmd_pbit_set_x,
	.next_state    = LLC_CONN_STATE_SETUP,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_setup_actions_1,
};

/* State transitions for LLC_CONN_EV_RX_UA_RSP_Fbit_SET_X event */
static const llc_conn_ev_qfyr_t llc_setup_ev_qfyrs_2[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_f,
	[1] = llc_conn_ev_qlfy_set_status_conn,
	[2] = NULL,
};

static const llc_conn_action_t llc_setup_actions_2[] = {
	[0] = llc_conn_ac_stop_ack_timer,
	[1] = llc_conn_ac_set_vs_0,
	[2] = llc_conn_ac_set_vr_0,
	[3] = llc_conn_ac_upd_p_flag,
	[4] = llc_conn_ac_set_remote_busy_0,
	[5] = llc_conn_ac_conn_confirm,
	[6] = NULL,
};

static struct llc_conn_state_trans llc_setup_state_trans_2 = {
	.ev	       = llc_conn_ev_rx_ua_rsp_fbit_set_x,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_setup_ev_qfyrs_2,
	.ev_actions    = llc_setup_actions_2,
};

/* State transitions for LLC_CONN_EV_ACK_TMR_EXP event */
static const llc_conn_ev_qfyr_t llc_setup_ev_qfyrs_3[] = {
	[0] = llc_conn_ev_qlfy_s_flag_eq_1,
	[1] = llc_conn_ev_qlfy_set_status_conn,
	[2] = NULL,
};

static const llc_conn_action_t llc_setup_actions_3[] = {
	[0] = llc_conn_ac_set_p_flag_0,
	[1] = llc_conn_ac_set_remote_busy_0,
	[2] = llc_conn_ac_conn_confirm,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_setup_state_trans_3 = {
	.ev	       = llc_conn_ev_ack_tmr_exp,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_setup_ev_qfyrs_3,
	.ev_actions    = llc_setup_actions_3,
};

/* State transitions for LLC_CONN_EV_RX_DISC_CMD_Pbit_SET_X event */
static const llc_conn_ev_qfyr_t llc_setup_ev_qfyrs_4[] = {
	[0] = llc_conn_ev_qlfy_set_status_disc,
	[1] = NULL,
};

static const llc_conn_action_t llc_setup_actions_4[] = {
	[0] = llc_conn_ac_send_dm_rsp_f_set_p,
	[1] = llc_conn_ac_stop_ack_timer,
	[2] = llc_conn_ac_conn_confirm,
	[3] = llc_conn_disc,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_setup_state_trans_4 = {
	.ev	       = llc_conn_ev_rx_disc_cmd_pbit_set_x,
	.next_state    = LLC_CONN_STATE_ADM,
	.ev_qualifiers = llc_setup_ev_qfyrs_4,
	.ev_actions    = llc_setup_actions_4,
};

/* State transitions for LLC_CONN_EV_RX_DM_RSP_Fbit_SET_X event */
static const llc_conn_ev_qfyr_t llc_setup_ev_qfyrs_5[] = {
	[0] = llc_conn_ev_qlfy_set_status_disc,
	[1] = NULL,
};

static const llc_conn_action_t llc_setup_actions_5[] = {
	[0] = llc_conn_ac_stop_ack_timer,
	[1] = llc_conn_ac_conn_confirm,
	[2] = llc_conn_disc,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_setup_state_trans_5 = {
	.ev	       = llc_conn_ev_rx_dm_rsp_fbit_set_x,
	.next_state    = LLC_CONN_STATE_ADM,
	.ev_qualifiers = llc_setup_ev_qfyrs_5,
	.ev_actions    = llc_setup_actions_5,
};

/* State transitions for LLC_CONN_EV_ACK_TMR_EXP event */
static const llc_conn_ev_qfyr_t llc_setup_ev_qfyrs_7[] = {
	[0] = llc_conn_ev_qlfy_retry_cnt_lt_n2,
	[1] = llc_conn_ev_qlfy_s_flag_eq_0,
	[2] = NULL,
};

static const llc_conn_action_t llc_setup_actions_7[] = {
	[0] = llc_conn_ac_send_sabme_cmd_p_set_x,
	[1] = llc_conn_ac_start_ack_timer,
	[2] = llc_conn_ac_inc_retry_cnt_by_1,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_setup_state_trans_7 = {
	.ev	       = llc_conn_ev_ack_tmr_exp,
	.next_state    = LLC_CONN_STATE_SETUP,
	.ev_qualifiers = llc_setup_ev_qfyrs_7,
	.ev_actions    = llc_setup_actions_7,
};

/* State transitions for LLC_CONN_EV_ACK_TMR_EXP event */
static const llc_conn_ev_qfyr_t llc_setup_ev_qfyrs_8[] = {
	[0] = llc_conn_ev_qlfy_retry_cnt_gte_n2,
	[1] = llc_conn_ev_qlfy_s_flag_eq_0,
	[2] = llc_conn_ev_qlfy_set_status_failed,
	[3] = NULL,
};

static const llc_conn_action_t llc_setup_actions_8[] = {
	[0] = llc_conn_ac_conn_confirm,
	[1] = llc_conn_disc,
	[2] = NULL,
};

static struct llc_conn_state_trans llc_setup_state_trans_8 = {
	.ev	       = llc_conn_ev_ack_tmr_exp,
	.next_state    = LLC_CONN_STATE_ADM,
	.ev_qualifiers = llc_setup_ev_qfyrs_8,
	.ev_actions    = llc_setup_actions_8,
};

/*
 * Array of pointers;
 * one to each transition
 */
static struct llc_conn_state_trans *llc_setup_state_transitions[] = {
	 [0] = &llc_common_state_trans_end,	/* Request */
	 [1] = &llc_common_state_trans_end,	/* local busy */
	 [2] = &llc_common_state_trans_end,	/* init_pf_cycle */
	 [3] = &llc_setup_state_trans_3,	/* Timer */
	 [4] = &llc_setup_state_trans_7,
	 [5] = &llc_setup_state_trans_8,
	 [6] = &llc_common_state_trans_end,
	 [7] = &llc_setup_state_trans_1,	/* Receive frame */
	 [8] = &llc_setup_state_trans_2,
	 [9] = &llc_setup_state_trans_4,
	[10] = &llc_setup_state_trans_5,
	[11] = &llc_common_state_trans_end,
};

/* LLC_CONN_STATE_NORMAL transitions */
/* State transitions for LLC_CONN_EV_DATA_REQ event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_1[] = {
	[0] = llc_conn_ev_qlfy_remote_busy_eq_0,
	[1] = llc_conn_ev_qlfy_p_flag_eq_0,
	[2] = llc_conn_ev_qlfy_last_frame_eq_0,
	[3] = NULL,
};

static const llc_conn_action_t llc_normal_actions_1[] = {
	[0] = llc_conn_ac_send_i_as_ack,
	[1] = llc_conn_ac_start_ack_tmr_if_not_running,
	[2] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_1 = {
	.ev	       = llc_conn_ev_data_req,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_1,
	.ev_actions    = llc_normal_actions_1,
};

/* State transitions for LLC_CONN_EV_DATA_REQ event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_2[] = {
	[0] = llc_conn_ev_qlfy_remote_busy_eq_0,
	[1] = llc_conn_ev_qlfy_p_flag_eq_0,
	[2] = llc_conn_ev_qlfy_last_frame_eq_1,
	[3] = NULL,
};

static const llc_conn_action_t llc_normal_actions_2[] = {
	[0] = llc_conn_ac_send_i_cmd_p_set_1,
	[1] = llc_conn_ac_start_p_timer,
	[2] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_2 = {
	.ev	       = llc_conn_ev_data_req,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_2,
	.ev_actions    = llc_normal_actions_2,
};

/* State transitions for LLC_CONN_EV_DATA_REQ event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_2_1[] = {
	[0] = llc_conn_ev_qlfy_remote_busy_eq_1,
	[1] = llc_conn_ev_qlfy_set_status_remote_busy,
	[2] = NULL,
};

/* just one member, NULL, .bss zeroes it */
static const llc_conn_action_t llc_normal_actions_2_1[1];

static struct llc_conn_state_trans llc_normal_state_trans_2_1 = {
	.ev	       = llc_conn_ev_data_req,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_2_1,
	.ev_actions    = llc_normal_actions_2_1,
};

/* State transitions for LLC_CONN_EV_LOCAL_BUSY_DETECTED event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_3[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_0,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_3[] = {
	[0] = llc_conn_ac_rst_sendack_flag,
	[1] = llc_conn_ac_send_rnr_xxx_x_set_0,
	[2] = llc_conn_ac_set_data_flag_0,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_3 = {
	.ev	       = llc_conn_ev_local_busy_detected,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_normal_ev_qfyrs_3,
	.ev_actions    = llc_normal_actions_3,
};

/* State transitions for LLC_CONN_EV_LOCAL_BUSY_DETECTED event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_4[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_4[] = {
	[0] = llc_conn_ac_rst_sendack_flag,
	[1] = llc_conn_ac_send_rnr_xxx_x_set_0,
	[2] = llc_conn_ac_set_data_flag_0,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_4 = {
	.ev	       = llc_conn_ev_local_busy_detected,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_normal_ev_qfyrs_4,
	.ev_actions    = llc_normal_actions_4,
};

/* State transitions for LLC_CONN_EV_RX_I_CMD_Pbit_SET_0_UNEXPD_Ns event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_5a[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_0,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_5a[] = {
	[0] = llc_conn_ac_rst_sendack_flag,
	[1] = llc_conn_ac_send_rej_xxx_x_set_0,
	[2] = llc_conn_ac_upd_nr_received,
	[3] = llc_conn_ac_upd_p_flag,
	[4] = llc_conn_ac_start_rej_timer,
	[5] = llc_conn_ac_clear_remote_busy_if_f_eq_1,
	[6] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_5a = {
	.ev	       = llc_conn_ev_rx_i_cmd_pbit_set_0_unexpd_ns,
	.next_state    = LLC_CONN_STATE_REJ,
	.ev_qualifiers = llc_normal_ev_qfyrs_5a,
	.ev_actions    = llc_normal_actions_5a,
};

/* State transitions for LLC_CONN_EV_RX_I_RSP_Fbit_SET_0_UNEXPD_Ns event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_5b[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_0,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_5b[] = {
	[0] = llc_conn_ac_rst_sendack_flag,
	[1] = llc_conn_ac_send_rej_xxx_x_set_0,
	[2] = llc_conn_ac_upd_nr_received,
	[3] = llc_conn_ac_upd_p_flag,
	[4] = llc_conn_ac_start_rej_timer,
	[5] = llc_conn_ac_clear_remote_busy_if_f_eq_1,
	[6] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_5b = {
	.ev	       = llc_conn_ev_rx_i_rsp_fbit_set_0_unexpd_ns,
	.next_state    = LLC_CONN_STATE_REJ,
	.ev_qualifiers = llc_normal_ev_qfyrs_5b,
	.ev_actions    = llc_normal_actions_5b,
};

/* State transitions for LLC_CONN_EV_RX_I_RSP_Fbit_SET_1_UNEXPD_Ns event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_5c[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_5c[] = {
	[0] = llc_conn_ac_rst_sendack_flag,
	[1] = llc_conn_ac_send_rej_xxx_x_set_0,
	[2] = llc_conn_ac_upd_nr_received,
	[3] = llc_conn_ac_upd_p_flag,
	[4] = llc_conn_ac_start_rej_timer,
	[5] = llc_conn_ac_clear_remote_busy_if_f_eq_1,
	[6] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_5c = {
	.ev	       = llc_conn_ev_rx_i_rsp_fbit_set_1_unexpd_ns,
	.next_state    = LLC_CONN_STATE_REJ,
	.ev_qualifiers = llc_normal_ev_qfyrs_5c,
	.ev_actions    = llc_normal_actions_5c,
};

/* State transitions for LLC_CONN_EV_RX_I_CMD_Pbit_SET_0_UNEXPD_Ns event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_6a[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_6a[] = {
	[0] = llc_conn_ac_rst_sendack_flag,
	[1] = llc_conn_ac_send_rej_xxx_x_set_0,
	[2] = llc_conn_ac_upd_nr_received,
	[3] = llc_conn_ac_start_rej_timer,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_6a = {
	.ev	       = llc_conn_ev_rx_i_cmd_pbit_set_0_unexpd_ns,
	.next_state    = LLC_CONN_STATE_REJ,
	.ev_qualifiers = llc_normal_ev_qfyrs_6a,
	.ev_actions    = llc_normal_actions_6a,
};

/* State transitions for LLC_CONN_EV_RX_I_RSP_Fbit_SET_0_UNEXPD_Ns event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_6b[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_6b[] = {
	[0] = llc_conn_ac_rst_sendack_flag,
	[1] = llc_conn_ac_send_rej_xxx_x_set_0,
	[2] = llc_conn_ac_upd_nr_received,
	[3] = llc_conn_ac_start_rej_timer,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_6b = {
	.ev	       = llc_conn_ev_rx_i_rsp_fbit_set_0_unexpd_ns,
	.next_state    = LLC_CONN_STATE_REJ,
	.ev_qualifiers = llc_normal_ev_qfyrs_6b,
	.ev_actions    = llc_normal_actions_6b,
};

/* State transitions for LLC_CONN_EV_RX_I_CMD_Pbit_SET_1_UNEXPD_Ns event */
static const llc_conn_action_t llc_normal_actions_7[] = {
	[0] = llc_conn_ac_rst_sendack_flag,
	[1] = llc_conn_ac_send_rej_rsp_f_set_1,
	[2] = llc_conn_ac_upd_nr_received,
	[3] = llc_conn_ac_start_rej_timer,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_7 = {
	.ev	       = llc_conn_ev_rx_i_cmd_pbit_set_1_unexpd_ns,
	.next_state    = LLC_CONN_STATE_REJ,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_normal_actions_7,
};

/* State transitions for LLC_CONN_EV_RX_I_RSP_Fbit_SET_X event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_8a[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_f,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_8[] = {
	[0] = llc_conn_ac_inc_vr_by_1,
	[1] = llc_conn_ac_data_ind,
	[2] = llc_conn_ac_upd_p_flag,
	[3] = llc_conn_ac_upd_nr_received,
	[4] = llc_conn_ac_clear_remote_busy_if_f_eq_1,
	[5] = llc_conn_ac_send_ack_if_needed,
	[6] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_8a = {
	.ev	       = llc_conn_ev_rx_i_rsp_fbit_set_x,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_8a,
	.ev_actions    = llc_normal_actions_8,
};

/* State transitions for LLC_CONN_EV_RX_I_CMD_Pbit_SET_0 event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_8b[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_0,
	[1] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_8b = {
	.ev	       = llc_conn_ev_rx_i_cmd_pbit_set_0,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_8b,
	.ev_actions    = llc_normal_actions_8,
};

/* State transitions for LLC_CONN_EV_RX_I_RSP_Fbit_SET_0 event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_9a[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_9a[] = {
	[0] = llc_conn_ac_inc_vr_by_1,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_data_ind,
	[3] = llc_conn_ac_send_ack_if_needed,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_9a = {
	.ev	       = llc_conn_ev_rx_i_rsp_fbit_set_0,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_9a,
	.ev_actions    = llc_normal_actions_9a,
};

/* State transitions for LLC_CONN_EV_RX_I_CMD_Pbit_SET_0 event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_9b[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_9b[] = {
	[0] = llc_conn_ac_inc_vr_by_1,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_data_ind,
	[3] = llc_conn_ac_send_ack_if_needed,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_9b = {
	.ev	       = llc_conn_ev_rx_i_cmd_pbit_set_0,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_9b,
	.ev_actions    = llc_normal_actions_9b,
};

/* State transitions for LLC_CONN_EV_RX_I_CMD_Pbit_SET_1 event */
static const llc_conn_action_t llc_normal_actions_10[] = {
	[0] = llc_conn_ac_inc_vr_by_1,
	[1] = llc_conn_ac_send_ack_rsp_f_set_1,
	[2] = llc_conn_ac_rst_sendack_flag,
	[3] = llc_conn_ac_upd_nr_received,
	[4] = llc_conn_ac_data_ind,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_10 = {
	.ev	       = llc_conn_ev_rx_i_cmd_pbit_set_1,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_normal_actions_10,
};

/* State transitions for * LLC_CONN_EV_RX_RR_CMD_Pbit_SET_0 event */
static const llc_conn_action_t llc_normal_actions_11a[] = {
	[0] = llc_conn_ac_upd_p_flag,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_clear_remote_busy,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_11a = {
	.ev	       = llc_conn_ev_rx_rr_cmd_pbit_set_0,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_normal_actions_11a,
};

/* State transitions for LLC_CONN_EV_RX_RR_RSP_Fbit_SET_0 event */
static const llc_conn_action_t llc_normal_actions_11b[] = {
	[0] = llc_conn_ac_upd_p_flag,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_clear_remote_busy,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_11b = {
	.ev	       = llc_conn_ev_rx_rr_rsp_fbit_set_0,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_normal_actions_11b,
};

/* State transitions for LLC_CONN_EV_RX_RR_RSP_Fbit_SET_1 event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_11c[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_11c[] = {
	[0] = llc_conn_ac_upd_p_flag,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_inc_tx_win_size,
	[3] = llc_conn_ac_clear_remote_busy,
	[4] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_11c = {
	.ev	       = llc_conn_ev_rx_rr_rsp_fbit_set_1,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_11c,
	.ev_actions    = llc_normal_actions_11c,
};

/* State transitions for LLC_CONN_EV_RX_RR_CMD_Pbit_SET_1 event */
static const llc_conn_action_t llc_normal_actions_12[] = {
	[0] = llc_conn_ac_send_ack_rsp_f_set_1,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_adjust_npta_by_rr,
	[3] = llc_conn_ac_rst_sendack_flag,
	[4] = llc_conn_ac_clear_remote_busy,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_12 = {
	.ev	       = llc_conn_ev_rx_rr_cmd_pbit_set_1,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_normal_actions_12,
};

/* State transitions for LLC_CONN_EV_RX_RNR_CMD_Pbit_SET_0 event */
static const llc_conn_action_t llc_normal_actions_13a[] = {
	[0] = llc_conn_ac_upd_p_flag,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_set_remote_busy,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_13a = {
	.ev	       = llc_conn_ev_rx_rnr_cmd_pbit_set_0,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_normal_actions_13a,
};

/* State transitions for LLC_CONN_EV_RX_RNR_RSP_Fbit_SET_0 event */
static const llc_conn_action_t llc_normal_actions_13b[] = {
	[0] = llc_conn_ac_upd_p_flag,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_set_remote_busy,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_13b = {
	.ev	       = llc_conn_ev_rx_rnr_rsp_fbit_set_0,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_normal_actions_13b,
};

/* State transitions for LLC_CONN_EV_RX_RNR_RSP_Fbit_SET_1 event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_13c[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_13c[] = {
	[0] = llc_conn_ac_upd_p_flag,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_set_remote_busy,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_13c = {
	.ev	       = llc_conn_ev_rx_rnr_rsp_fbit_set_1,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_13c,
	.ev_actions    = llc_normal_actions_13c,
};

/* State transitions for LLC_CONN_EV_RX_RNR_CMD_Pbit_SET_1 event */
static const llc_conn_action_t llc_normal_actions_14[] = {
	[0] = llc_conn_ac_send_rr_rsp_f_set_1,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_adjust_npta_by_rnr,
	[3] = llc_conn_ac_rst_sendack_flag,
	[4] = llc_conn_ac_set_remote_busy,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_14 = {
	.ev	       = llc_conn_ev_rx_rnr_cmd_pbit_set_1,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_normal_actions_14,
};

/* State transitions for LLC_CONN_EV_RX_REJ_CMD_Pbit_SET_0 event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_15a[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_0,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_15a[] = {
	[0] = llc_conn_ac_set_vs_nr,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_upd_p_flag,
	[3] = llc_conn_ac_dec_tx_win_size,
	[4] = llc_conn_ac_resend_i_xxx_x_set_0,
	[5] = llc_conn_ac_clear_remote_busy,
	[6] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_15a = {
	.ev	       = llc_conn_ev_rx_rej_cmd_pbit_set_0,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_15a,
	.ev_actions    = llc_normal_actions_15a,
};

/* State transitions for LLC_CONN_EV_RX_REJ_RSP_Fbit_SET_X event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_15b[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_f,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_15b[] = {
	[0] = llc_conn_ac_set_vs_nr,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_upd_p_flag,
	[3] = llc_conn_ac_dec_tx_win_size,
	[4] = llc_conn_ac_resend_i_xxx_x_set_0,
	[5] = llc_conn_ac_clear_remote_busy,
	[6] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_15b = {
	.ev	       = llc_conn_ev_rx_rej_rsp_fbit_set_x,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_15b,
	.ev_actions    = llc_normal_actions_15b,
};

/* State transitions for LLC_CONN_EV_RX_REJ_CMD_Pbit_SET_0 event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_16a[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_16a[] = {
	[0] = llc_conn_ac_set_vs_nr,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_dec_tx_win_size,
	[3] = llc_conn_ac_resend_i_xxx_x_set_0,
	[4] = llc_conn_ac_clear_remote_busy,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_16a = {
	.ev	       = llc_conn_ev_rx_rej_cmd_pbit_set_0,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_16a,
	.ev_actions    = llc_normal_actions_16a,
};

/* State transitions for LLC_CONN_EV_RX_REJ_RSP_Fbit_SET_0 event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_16b[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_16b[] = {
	[0] = llc_conn_ac_set_vs_nr,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_dec_tx_win_size,
	[3] = llc_conn_ac_resend_i_xxx_x_set_0,
	[4] = llc_conn_ac_clear_remote_busy,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_16b = {
	.ev	       = llc_conn_ev_rx_rej_rsp_fbit_set_0,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_16b,
	.ev_actions    = llc_normal_actions_16b,
};

/* State transitions for LLC_CONN_EV_RX_REJ_CMD_Pbit_SET_1 event */
static const llc_conn_action_t llc_normal_actions_17[] = {
	[0] = llc_conn_ac_set_vs_nr,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_dec_tx_win_size,
	[3] = llc_conn_ac_resend_i_rsp_f_set_1,
	[4] = llc_conn_ac_clear_remote_busy,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_17 = {
	.ev	       = llc_conn_ev_rx_rej_cmd_pbit_set_1,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_normal_actions_17,
};

/* State transitions for LLC_CONN_EV_INIT_P_F_CYCLE event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_18[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_0,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_18[] = {
	[0] = llc_conn_ac_send_rr_cmd_p_set_1,
	[1] = llc_conn_ac_start_p_timer,
	[2] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_18 = {
	.ev	       = llc_conn_ev_init_p_f_cycle,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_18,
	.ev_actions    = llc_normal_actions_18,
};

/* State transitions for LLC_CONN_EV_P_TMR_EXP event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_19[] = {
	[0] = llc_conn_ev_qlfy_retry_cnt_lt_n2,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_19[] = {
	[0] = llc_conn_ac_rst_sendack_flag,
	[1] = llc_conn_ac_send_rr_cmd_p_set_1,
	[2] = llc_conn_ac_rst_vs,
	[3] = llc_conn_ac_start_p_timer,
	[4] = llc_conn_ac_inc_retry_cnt_by_1,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_19 = {
	.ev	       = llc_conn_ev_p_tmr_exp,
	.next_state    = LLC_CONN_STATE_AWAIT,
	.ev_qualifiers = llc_normal_ev_qfyrs_19,
	.ev_actions    = llc_normal_actions_19,
};

/* State transitions for LLC_CONN_EV_ACK_TMR_EXP event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_20a[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_0,
	[1] = llc_conn_ev_qlfy_retry_cnt_lt_n2,
	[2] = NULL,
};

static const llc_conn_action_t llc_normal_actions_20a[] = {
	[0] = llc_conn_ac_rst_sendack_flag,
	[1] = llc_conn_ac_send_rr_cmd_p_set_1,
	[2] = llc_conn_ac_rst_vs,
	[3] = llc_conn_ac_start_p_timer,
	[4] = llc_conn_ac_inc_retry_cnt_by_1,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_20a = {
	.ev	       = llc_conn_ev_ack_tmr_exp,
	.next_state    = LLC_CONN_STATE_AWAIT,
	.ev_qualifiers = llc_normal_ev_qfyrs_20a,
	.ev_actions    = llc_normal_actions_20a,
};

/* State transitions for LLC_CONN_EV_BUSY_TMR_EXP event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_20b[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_0,
	[1] = llc_conn_ev_qlfy_retry_cnt_lt_n2,
	[2] = NULL,
};

static const llc_conn_action_t llc_normal_actions_20b[] = {
	[0] = llc_conn_ac_rst_sendack_flag,
	[1] = llc_conn_ac_send_rr_cmd_p_set_1,
	[2] = llc_conn_ac_rst_vs,
	[3] = llc_conn_ac_start_p_timer,
	[4] = llc_conn_ac_inc_retry_cnt_by_1,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_20b = {
	.ev	       = llc_conn_ev_busy_tmr_exp,
	.next_state    = LLC_CONN_STATE_AWAIT,
	.ev_qualifiers = llc_normal_ev_qfyrs_20b,
	.ev_actions    = llc_normal_actions_20b,
};

/* State transitions for LLC_CONN_EV_TX_BUFF_FULL event */
static const llc_conn_ev_qfyr_t llc_normal_ev_qfyrs_21[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_0,
	[1] = NULL,
};

static const llc_conn_action_t llc_normal_actions_21[] = {
	[0] = llc_conn_ac_send_rr_cmd_p_set_1,
	[1] = llc_conn_ac_start_p_timer,
	[2] = NULL,
};

static struct llc_conn_state_trans llc_normal_state_trans_21 = {
	.ev	       = llc_conn_ev_tx_buffer_full,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_normal_ev_qfyrs_21,
	.ev_actions    = llc_normal_actions_21,
};

/*
 * Array of pointers;
 * one to each transition
 */
static struct llc_conn_state_trans *llc_normal_state_transitions[] = {
	 [0] = &llc_normal_state_trans_1,	/* Requests */
	 [1] = &llc_normal_state_trans_2,
	 [2] = &llc_normal_state_trans_2_1,
	 [3] = &llc_common_state_trans_1,
	 [4] = &llc_common_state_trans_2,
	 [5] = &llc_common_state_trans_end,
	 [6] = &llc_normal_state_trans_21,
	 [7] = &llc_normal_state_trans_3,	/* Local busy */
	 [8] = &llc_normal_state_trans_4,
	 [9] = &llc_common_state_trans_end,
	[10] = &llc_normal_state_trans_18,	/* Init pf cycle */
	[11] = &llc_common_state_trans_end,
	[12] = &llc_common_state_trans_11a,	/* Timers */
	[13] = &llc_common_state_trans_11b,
	[14] = &llc_common_state_trans_11c,
	[15] = &llc_common_state_trans_11d,
	[16] = &llc_normal_state_trans_19,
	[17] = &llc_normal_state_trans_20a,
	[18] = &llc_normal_state_trans_20b,
	[19] = &llc_common_state_trans_end,
	[20] = &llc_normal_state_trans_8b,	/* Receive frames */
	[21] = &llc_normal_state_trans_9b,
	[22] = &llc_normal_state_trans_10,
	[23] = &llc_normal_state_trans_11b,
	[24] = &llc_normal_state_trans_11c,
	[25] = &llc_normal_state_trans_5a,
	[26] = &llc_normal_state_trans_5b,
	[27] = &llc_normal_state_trans_5c,
	[28] = &llc_normal_state_trans_6a,
	[29] = &llc_normal_state_trans_6b,
	[30] = &llc_normal_state_trans_7,
	[31] = &llc_normal_state_trans_8a,
	[32] = &llc_normal_state_trans_9a,
	[33] = &llc_normal_state_trans_11a,
	[34] = &llc_normal_state_trans_12,
	[35] = &llc_normal_state_trans_13a,
	[36] = &llc_normal_state_trans_13b,
	[37] = &llc_normal_state_trans_13c,
	[38] = &llc_normal_state_trans_14,
	[39] = &llc_normal_state_trans_15a,
	[40] = &llc_normal_state_trans_15b,
	[41] = &llc_normal_state_trans_16a,
	[42] = &llc_normal_state_trans_16b,
	[43] = &llc_normal_state_trans_17,
	[44] = &llc_common_state_trans_3,
	[45] = &llc_common_state_trans_4,
	[46] = &llc_common_state_trans_5,
	[47] = &llc_common_state_trans_6,
	[48] = &llc_common_state_trans_7a,
	[49] = &llc_common_state_trans_7b,
	[50] = &llc_common_state_trans_8a,
	[51] = &llc_common_state_trans_8b,
	[52] = &llc_common_state_trans_8c,
	[53] = &llc_common_state_trans_9,
	/* [54] = &llc_common_state_trans_10, */
	[54] = &llc_common_state_trans_end,
};

/* LLC_CONN_STATE_BUSY transitions */
/* State transitions for LLC_CONN_EV_DATA_REQ event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_1[] = {
	[0] = llc_conn_ev_qlfy_remote_busy_eq_0,
	[1] = llc_conn_ev_qlfy_p_flag_eq_0,
	[2] = NULL,
};

static const llc_conn_action_t llc_busy_actions_1[] = {
	[0] = llc_conn_ac_send_i_xxx_x_set_0,
	[1] = llc_conn_ac_start_ack_tmr_if_not_running,
	[2] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_1 = {
	.ev	       = llc_conn_ev_data_req,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_busy_ev_qfyrs_1,
	.ev_actions    = llc_busy_actions_1,
};

/* State transitions for LLC_CONN_EV_DATA_REQ event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_2[] = {
	[0] = llc_conn_ev_qlfy_remote_busy_eq_0,
	[1] = llc_conn_ev_qlfy_p_flag_eq_1,
	[2] = NULL,
};

static const llc_conn_action_t llc_busy_actions_2[] = {
	[0] = llc_conn_ac_send_i_xxx_x_set_0,
	[1] = llc_conn_ac_start_ack_tmr_if_not_running,
	[2] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_2 = {
	.ev	       = llc_conn_ev_data_req,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_busy_ev_qfyrs_2,
	.ev_actions    = llc_busy_actions_2,
};

/* State transitions for LLC_CONN_EV_DATA_REQ event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_2_1[] = {
	[0] = llc_conn_ev_qlfy_remote_busy_eq_1,
	[1] = llc_conn_ev_qlfy_set_status_remote_busy,
	[2] = NULL,
};

/* just one member, NULL, .bss zeroes it */
static const llc_conn_action_t llc_busy_actions_2_1[1];

static struct llc_conn_state_trans llc_busy_state_trans_2_1 = {
	.ev	       = llc_conn_ev_data_req,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_busy_ev_qfyrs_2_1,
	.ev_actions    = llc_busy_actions_2_1,
};

/* State transitions for LLC_CONN_EV_LOCAL_BUSY_CLEARED event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_3[] = {
	[0] = llc_conn_ev_qlfy_data_flag_eq_1,
	[1] = llc_conn_ev_qlfy_p_flag_eq_0,
	[2] = NULL,
};

static const llc_conn_action_t llc_busy_actions_3[] = {
	[0] = llc_conn_ac_send_rej_xxx_x_set_0,
	[1] = llc_conn_ac_start_rej_timer,
	[2] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_3 = {
	.ev	       = llc_conn_ev_local_busy_cleared,
	.next_state    = LLC_CONN_STATE_REJ,
	.ev_qualifiers = llc_busy_ev_qfyrs_3,
	.ev_actions    = llc_busy_actions_3,
};

/* State transitions for LLC_CONN_EV_LOCAL_BUSY_CLEARED event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_4[] = {
	[0] = llc_conn_ev_qlfy_data_flag_eq_1,
	[1] = llc_conn_ev_qlfy_p_flag_eq_1,
	[2] = NULL,
};

static const llc_conn_action_t llc_busy_actions_4[] = {
	[0] = llc_conn_ac_send_rej_xxx_x_set_0,
	[1] = llc_conn_ac_start_rej_timer,
	[2] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_4 = {
	.ev	       = llc_conn_ev_local_busy_cleared,
	.next_state    = LLC_CONN_STATE_REJ,
	.ev_qualifiers = llc_busy_ev_qfyrs_4,
	.ev_actions    = llc_busy_actions_4,
};

/* State transitions for LLC_CONN_EV_LOCAL_BUSY_CLEARED event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_5[] = {
	[0] = llc_conn_ev_qlfy_data_flag_eq_0,
	[1] = llc_conn_ev_qlfy_p_flag_eq_0,
	[2] = NULL,
};

static const llc_conn_action_t llc_busy_actions_5[] = {
	[0] = llc_conn_ac_send_rr_xxx_x_set_0,
	[1] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_5 = {
	.ev	       = llc_conn_ev_local_busy_cleared,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_busy_ev_qfyrs_5,
	.ev_actions    = llc_busy_actions_5,
};

/* State transitions for LLC_CONN_EV_LOCAL_BUSY_CLEARED event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_6[] = {
	[0] = llc_conn_ev_qlfy_data_flag_eq_0,
	[1] = llc_conn_ev_qlfy_p_flag_eq_1,
	[2] = NULL,
};

static const llc_conn_action_t llc_busy_actions_6[] = {
	[0] = llc_conn_ac_send_rr_xxx_x_set_0,
	[1] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_6 = {
	.ev	       = llc_conn_ev_local_busy_cleared,
	.next_state    = LLC_CONN_STATE_NORMAL,
	.ev_qualifiers = llc_busy_ev_qfyrs_6,
	.ev_actions    = llc_busy_actions_6,
};

/* State transitions for LLC_CONN_EV_LOCAL_BUSY_CLEARED event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_7[] = {
	[0] = llc_conn_ev_qlfy_data_flag_eq_2,
	[1] = llc_conn_ev_qlfy_p_flag_eq_0,
	[2] = NULL,
};

static const llc_conn_action_t llc_busy_actions_7[] = {
	[0] = llc_conn_ac_send_rr_xxx_x_set_0,
	[1] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_7 = {
	.ev	       = llc_conn_ev_local_busy_cleared,
	.next_state    = LLC_CONN_STATE_REJ,
	.ev_qualifiers = llc_busy_ev_qfyrs_7,
	.ev_actions    = llc_busy_actions_7,
};

/* State transitions for LLC_CONN_EV_LOCAL_BUSY_CLEARED event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_8[] = {
	[0] = llc_conn_ev_qlfy_data_flag_eq_2,
	[1] = llc_conn_ev_qlfy_p_flag_eq_1,
	[2] = NULL,
};

static const llc_conn_action_t llc_busy_actions_8[] = {
	[0] = llc_conn_ac_send_rr_xxx_x_set_0,
	[1] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_8 = {
	.ev	       = llc_conn_ev_local_busy_cleared,
	.next_state    = LLC_CONN_STATE_REJ,
	.ev_qualifiers = llc_busy_ev_qfyrs_8,
	.ev_actions    = llc_busy_actions_8,
};

/* State transitions for LLC_CONN_EV_RX_I_RSP_Fbit_SET_X_UNEXPD_Ns event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_9a[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_f,
	[1] = NULL,
};

static const llc_conn_action_t llc_busy_actions_9a[] = {
	[0] = llc_conn_ac_opt_send_rnr_xxx_x_set_0,
	[1] = llc_conn_ac_upd_p_flag,
	[2] = llc_conn_ac_upd_nr_received,
	[3] = llc_conn_ac_set_data_flag_1_if_data_flag_eq_0,
	[4] = llc_conn_ac_clear_remote_busy_if_f_eq_1,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_9a = {
	.ev	       = llc_conn_ev_rx_i_rsp_fbit_set_x_unexpd_ns,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_busy_ev_qfyrs_9a,
	.ev_actions    = llc_busy_actions_9a,
};

/* State transitions for LLC_CONN_EV_RX_I_CMD_Pbit_SET_0_UNEXPD_Ns event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_9b[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_0,
	[1] = NULL,
};

static const llc_conn_action_t llc_busy_actions_9b[] = {
	[0] = llc_conn_ac_opt_send_rnr_xxx_x_set_0,
	[1] = llc_conn_ac_upd_p_flag,
	[2] = llc_conn_ac_upd_nr_received,
	[3] = llc_conn_ac_set_data_flag_1_if_data_flag_eq_0,
	[4] = llc_conn_ac_clear_remote_busy_if_f_eq_1,
	[5] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_9b = {
	.ev	       = llc_conn_ev_rx_i_cmd_pbit_set_0_unexpd_ns,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_busy_ev_qfyrs_9b,
	.ev_actions    = llc_busy_actions_9b,
};

/* State transitions for LLC_CONN_EV_RX_I_RSP_Fbit_SET_0_UNEXPD_Ns event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_10a[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_busy_actions_10a[] = {
	[0] = llc_conn_ac_opt_send_rnr_xxx_x_set_0,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_set_data_flag_1_if_data_flag_eq_0,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_10a = {
	.ev	       = llc_conn_ev_rx_i_rsp_fbit_set_0_unexpd_ns,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_busy_ev_qfyrs_10a,
	.ev_actions    = llc_busy_actions_10a,
};

/* State transitions for LLC_CONN_EV_RX_I_CMD_Pbit_SET_0_UNEXPD_Ns event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_10b[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_busy_actions_10b[] = {
	[0] = llc_conn_ac_opt_send_rnr_xxx_x_set_0,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_set_data_flag_1_if_data_flag_eq_0,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_10b = {
	.ev	       = llc_conn_ev_rx_i_cmd_pbit_set_0_unexpd_ns,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_busy_ev_qfyrs_10b,
	.ev_actions    = llc_busy_actions_10b,
};

/* State transitions for LLC_CONN_EV_RX_I_CMD_Pbit_SET_1_UNEXPD_Ns event */
static const llc_conn_action_t llc_busy_actions_11[] = {
	[0] = llc_conn_ac_send_rnr_rsp_f_set_1,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_set_data_flag_1_if_data_flag_eq_0,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_11 = {
	.ev	       = llc_conn_ev_rx_i_cmd_pbit_set_1_unexpd_ns,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_busy_actions_11,
};

/* State transitions for LLC_CONN_EV_RX_I_CMD_Pbit_SET_1 event */
static const llc_conn_action_t llc_busy_actions_12[] = {
	[0] = llc_conn_ac_inc_vr_by_1,
	[1] = llc_conn_ac_data_ind,
	[2] = llc_conn_ac_send_rnr_rsp_f_set_1,
	[3] = llc_conn_ac_upd_nr_received,
	[4] = llc_conn_ac_stop_rej_tmr_if_data_flag_eq_2,
	[5] = llc_conn_ac_set_data_flag_0,
	[6] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_12 = {
	.ev	       = llc_conn_ev_rx_i_cmd_pbit_set_1,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_busy_actions_12,
};

/* State transitions for LLC_CONN_EV_RX_I_RSP_Fbit_SET_X event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_13a[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_f,
	[1] = NULL,
};

static const llc_conn_action_t llc_busy_actions_13a[] = {
	[0] = llc_conn_ac_inc_vr_by_1,
	[1] = llc_conn_ac_data_ind,
	[2] = llc_conn_ac_upd_p_flag,
	[3] = llc_conn_ac_opt_send_rnr_xxx_x_set_0,
	[4] = llc_conn_ac_upd_nr_received,
	[5] = llc_conn_ac_stop_rej_tmr_if_data_flag_eq_2,
	[6] = llc_conn_ac_set_data_flag_0,
	[7] = llc_conn_ac_clear_remote_busy_if_f_eq_1,
	[8] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_13a = {
	.ev	       = llc_conn_ev_rx_i_rsp_fbit_set_x,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_busy_ev_qfyrs_13a,
	.ev_actions    = llc_busy_actions_13a,
};

/* State transitions for LLC_CONN_EV_RX_I_CMD_Pbit_SET_0 event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_13b[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_0,
	[1] = NULL,
};

static const llc_conn_action_t llc_busy_actions_13b[] = {
	[0] = llc_conn_ac_inc_vr_by_1,
	[1] = llc_conn_ac_data_ind,
	[2] = llc_conn_ac_upd_p_flag,
	[3] = llc_conn_ac_opt_send_rnr_xxx_x_set_0,
	[4] = llc_conn_ac_upd_nr_received,
	[5] = llc_conn_ac_stop_rej_tmr_if_data_flag_eq_2,
	[6] = llc_conn_ac_set_data_flag_0,
	[7] = llc_conn_ac_clear_remote_busy_if_f_eq_1,
	[8] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_13b = {
	.ev	       = llc_conn_ev_rx_i_cmd_pbit_set_0,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_busy_ev_qfyrs_13b,
	.ev_actions    = llc_busy_actions_13b,
};

/* State transitions for LLC_CONN_EV_RX_I_RSP_Fbit_SET_0 event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_14a[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_busy_actions_14a[] = {
	[0] = llc_conn_ac_inc_vr_by_1,
	[1] = llc_conn_ac_data_ind,
	[2] = llc_conn_ac_opt_send_rnr_xxx_x_set_0,
	[3] = llc_conn_ac_upd_nr_received,
	[4] = llc_conn_ac_stop_rej_tmr_if_data_flag_eq_2,
	[5] = llc_conn_ac_set_data_flag_0,
	[6] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_14a = {
	.ev	       = llc_conn_ev_rx_i_rsp_fbit_set_0,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_busy_ev_qfyrs_14a,
	.ev_actions    = llc_busy_actions_14a,
};

/* State transitions for LLC_CONN_EV_RX_I_CMD_Pbit_SET_0 event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_14b[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_busy_actions_14b[] = {
	[0] = llc_conn_ac_inc_vr_by_1,
	[1] = llc_conn_ac_data_ind,
	[2] = llc_conn_ac_opt_send_rnr_xxx_x_set_0,
	[3] = llc_conn_ac_upd_nr_received,
	[4] = llc_conn_ac_stop_rej_tmr_if_data_flag_eq_2,
	[5] = llc_conn_ac_set_data_flag_0,
	[6] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_14b = {
	.ev	       = llc_conn_ev_rx_i_cmd_pbit_set_0,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_busy_ev_qfyrs_14b,
	.ev_actions    = llc_busy_actions_14b,
};

/* State transitions for LLC_CONN_EV_RX_RR_CMD_Pbit_SET_0 event */
static const llc_conn_action_t llc_busy_actions_15a[] = {
	[0] = llc_conn_ac_upd_p_flag,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_clear_remote_busy,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_15a = {
	.ev	       = llc_conn_ev_rx_rr_cmd_pbit_set_0,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_busy_actions_15a,
};

/* State transitions for LLC_CONN_EV_RX_RR_RSP_Fbit_SET_0 event */
static const llc_conn_action_t llc_busy_actions_15b[] = {
	[0] = llc_conn_ac_upd_p_flag,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_clear_remote_busy,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_15b = {
	.ev	       = llc_conn_ev_rx_rr_rsp_fbit_set_0,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_busy_actions_15b,
};

/* State transitions for LLC_CONN_EV_RX_RR_RSP_Fbit_SET_1 event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_15c[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_busy_actions_15c[] = {
	[0] = llc_conn_ac_upd_p_flag,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_clear_remote_busy,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_15c = {
	.ev	       = llc_conn_ev_rx_rr_rsp_fbit_set_1,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = llc_busy_ev_qfyrs_15c,
	.ev_actions    = llc_busy_actions_15c,
};

/* State transitions for LLC_CONN_EV_RX_RR_CMD_Pbit_SET_1 event */
static const llc_conn_action_t llc_busy_actions_16[] = {
	[0] = llc_conn_ac_send_rnr_rsp_f_set_1,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_clear_remote_busy,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_16 = {
	.ev	       = llc_conn_ev_rx_rr_cmd_pbit_set_1,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_busy_actions_16,
};

/* State transitions for LLC_CONN_EV_RX_RNR_CMD_Pbit_SET_0 event */
static const llc_conn_action_t llc_busy_actions_17a[] = {
	[0] = llc_conn_ac_upd_p_flag,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_set_remote_busy,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_17a = {
	.ev	       = llc_conn_ev_rx_rnr_cmd_pbit_set_0,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_busy_actions_17a,
};

/* State transitions for LLC_CONN_EV_RX_RNR_RSP_Fbit_SET_0 event */
static const llc_conn_action_t llc_busy_actions_17b[] = {
	[0] = llc_conn_ac_upd_p_flag,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_set_remote_busy,
	[3] = NULL,
};

static struct llc_conn_state_trans llc_busy_state_trans_17b = {
	.ev	       = llc_conn_ev_rx_rnr_rsp_fbit_set_0,
	.next_state    = LLC_CONN_STATE_BUSY,
	.ev_qualifiers = NONE,
	.ev_actions    = llc_busy_actions_17b,
};

/* State transitions for LLC_CONN_EV_RX_RNR_RSP_Fbit_SET_1 event */
static const llc_conn_ev_qfyr_t llc_busy_ev_qfyrs_17c[] = {
	[0] = llc_conn_ev_qlfy_p_flag_eq_1,
	[1] = NULL,
};

static const llc_conn_action_t llc_busy_actions_17c[] = {
	[0] = llc_conn_ac_upd_p_flag,
	[1] = llc_conn_ac_upd_nr_received,
	[2] = llc_conn_ac_set_remote_busy,
	[3] = NULL,
};

static struct llc_conn_state_trans l