/* ----------------------------------------------------------------------- *
 *
 *   Copyright 2009 Erwan Velu - All Rights Reserved
 *
 *   Permission is hereby granted, free of charge, to any person
 *   obtaining a copy of this software and associated documentation
 *   files (the "Software"), to deal in the Software without
 *   restriction, including without limitation the rights to use,
 *   copy, modify, merge, publish, distribute, sublicense, and/or
 *   sell copies of the Software, and to permit persons to whom
 *   the Software is furnished to do so, subject to the following
 *   conditions:
 *
 *   The above copyright notice and this permission notice shall
 *   be included in all copies or substantial portions of the Software.
 *
 *   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 *   EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 *   OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 *   NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 *   HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 *   WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 *   FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 *   OTHER DEALINGS IN THE SOFTWARE.
 *
 * -----------------------------------------------------------------------
 */

#include <stdlib.h>
#include <string.h>
#include <syslinux/config.h>
#include <getkey.h>
#include "hdt-cli.h"
#include "hdt-common.h"

struct cli_mode_descr *list_modes[] = {
	&hdt_mode,
	&dmi_mode,
	&syslinux_mode,
	&pxe_mode,
	&kernel_mode,
	&cpu_mode,
	&pci_mode,
	&vesa_mode,
	&disk_mode,
	&vpd_mode,
	&memory_mode,
	NULL,
};

/*
 * .aliases = {"q", "quit"} won't work since it is an array of pointers, not an
 * array of variables. There is no easy way around it besides declaring the arrays of
 * strings first.
 */
const char *exit_aliases[] = {"q", "quit"};
const char *help_aliases[] = {"h", "?"};

/* List of aliases */
struct cli_alias hdt_aliases[] = {
	{
		.command = CLI_EXIT,
		.nb_aliases = 2,
		.aliases = exit_aliases,
	},
	{
		.command = CLI_HELP,
		.nb_aliases = 2,
		.aliases = help_aliases,
	},
};

struct cli_mode_descr *current_mode;
int autocomplete_backlog;

struct autocomplete_list {
	char autocomplete_token[MAX_LINE_SIZE];
	struct autocomplete_list *next;
};
struct autocomplete_list* autocomplete_head = NULL;
struct autocomplete_list* autocomplete_tail = NULL;
struct autocomplete_list* autocomplete_last_seen = NULL;

static void autocomplete_add_token_to_list(const char *token)
{
	struct autocomplete_list *new = malloc(sizeof(struct autocomplete_list));

	strncpy(new->autocomplete_token, token, sizeof(new->autocomplete_token));
	new->next = NULL;
	autocomplete_backlog++;

	if (autocomplete_tail != NULL)
		autocomplete_tail->next = new;
	if (autocomplete_head == NULL)
		autocomplete_head = new;
	autocomplete_tail = new;
}

static void autocomplete_destroy_list()
{
	struct autocomplete_list* tmp = NULL;

	while (autocomplete_head != NULL) {
		tmp = autocomplete_head->next;
		free(autocomplete_head);
		autocomplete_head = tmp;
	}
	autocomplete_backlog = 0;
	autocomplete_tail = NULL;
	autocomplete_last_seen = NULL;
}

/**
 * set_mode - set the current mode of the cli
 * @mode:	mode to set
 *
 * Unlike cli_set_mode, this function is not used by the cli directly.
 **/
void set_mode(cli_mode_t mode, struct s_hardware* hardware)
{
	int i = 0;

	switch (mode) {
	case EXIT_MODE:
		hdt_cli.mode = mode;
		break;
	case HDT_MODE:
		hdt_cli.mode = mode;
		snprintf(hdt_cli.prompt, sizeof(hdt_cli.prompt), "%s> ",
			 CLI_HDT);
		break;
	case PXE_MODE:
		if (hardware->sv->filesystem != SYSLINUX_FS_PXELINUX) {
			printf("You are not currently using PXELINUX\n");
			break;
		}
		hdt_cli.mode = mode;
		snprintf(hdt_cli.prompt, sizeof(hdt_cli.prompt), "%s> ",
			 CLI_PXE);
		break;
	case KERNEL_MODE:
		detect_pci(hardware);
		hdt_cli.mode = mode;
		snprintf(hdt_cli.prompt, sizeof(hdt_cli.prompt), "%s> ",
			 CLI_KERNEL);
		break;
	case SYSLINUX_MODE:
		hdt_cli.mode = mode;
		snprintf(hdt_cli.prompt, sizeof(hdt_cli.prompt), "%s> ",
			 CLI_SYSLINUX);
		break;
	case VESA_MODE:
		hdt_cli.mode = mode;
		snprintf(hdt_cli.prompt, sizeof(hdt_cli.prompt), "%s> ",
			 CLI_VESA);
		break;
	case PCI_MODE:
		hdt_cli.mode = mode;
		snprintf(hdt_cli.prompt, sizeof(hdt_cli.prompt), "%s> ",
			 CLI_PCI);
		if (!hardware->pci_detection)
			cli_detect_pci(hardware);
		break;
	case CPU_MODE:
		hdt_cli.mode = mode;
		snprintf(hdt_cli.prompt, sizeof(hdt_cli.prompt), "%s> ",
			 CLI_CPU);
		if (!hardware->dmi_detection)
			detect_dmi(hardware);
		if (!hardware->cpu_detection)
			cpu_detect(hardware);
		break;
	case DMI_MODE:
		detect_dmi(hardware);
		if (!hardware->is_dmi_valid) {
			printf("No valid DMI table found, exiting.\n");
			break;
		}
		hdt_cli.mode = mode;
		snprintf(hdt_cli.prompt, sizeof(hdt_cli.prompt), "%s> ",
			 CLI_DMI);
		break;
	case DISK_MODE:
		detect_disks(hardware);
		hdt_cli.mode = mode;
		snprintf(hdt_cli.prompt, sizeof(hdt_cli.prompt), "%s> ",
			 CLI_DISK);
		break;
	case VPD_MODE:
		detect_vpd(hardware);
		if (!hardware->is_vpd_valid) {
			printf("No valid VPD table found, exiting.\n");
			break;
		}
		hdt_cli.mode = mode;
		snprintf(hdt_cli.prompt, sizeof(hdt_cli.prompt), "%s> ",
			 CLI_VPD);
		break;
	case MEMORY_MODE:
		hdt_cli.mode = mode;
		snprintf(hdt_cli.prompt, sizeof(hdt_cli.prompt), "%s> ",
			 CLI_MEMORY);
		break;
	default:
		/* Invalid mode */
		printf("Unknown mode, please choose among:\n");
		while (list_modes[i]) {
			printf("\t%s\n", list_modes[i]->name);
			i++;
		}
	}

	find_cli_mode_descr(hdt_cli.mode, &current_mode);
	/* There is not cli_mode_descr struct for the exit mode */
	if (current_mode == NULL && hdt_cli.mode != EXIT_MODE) {
		/* Shouldn't get here... */
		printf("!!! BUG: Mode '%d' unknown.\n", hdt_cli.mode);
	}
}

/**
 * mode_s_to_mode_t - given a mode string, return the cli_mode_t representation
 **/
cli_mode_t mode_s_to_mode_t(char *name)
{
	int i = 0;

	while (list_modes[i]) {
		if (!strncmp(name, list_modes[i]->name,
			     sizeof(list_modes[i]->name)))
			break;
		i++;
	}

	if (!list_modes[i])
		return INVALID_MODE;
	else
		return list_modes[i]->mode;
}

/**
 * find_cli_mode_descr - find the cli_mode_descr struct associated to a mode
 * @mode:	mode to look for
 * @mode_found:	store the mode if found, NULL otherwise
 *
 * Given a mode name, return a pointer to the associated cli_mode_descr
 * structure.
 * Note: the current mode name is stored in hdt_cli.mode.
 **/
void find_cli_mode_descr(cli_mode_t mode, struct cli_mode_descr **mode_found)
{
	int i = 0;

	while (list_modes[i] &&
	       list_modes[i]->mode != mode)
		i++;

	/* Shouldn't get here... */
	if (!list_modes[i])
		*mode_found = NULL;
	else
		*mode_found = list_modes[i];
}

/**
 * expand_aliases - resolve aliases mapping
 * @line:	command line to parse
 * @command:	first token in the line
 * @module:	second token in the line
 * @argc:	number of arguments
 * @argv:	array of arguments
 *
 * We maintain a small list of static alises to enhance user experience.
 * Only commands can be aliased (first token). Otherwise it can become really hairy...
 **/
static void expand_aliases(char *line __unused, char **command, char **module,
			   int *argc, char **argv)
{
	struct cli_mode_descr *mode;
	int i, j;

	find_cli_mode_descr(mode_s_to_mode_t(*command), &mode);
	if (mode != NULL && *module == NULL) {
		/*
		 * The user specified a mode instead of `set mode...', e.g.
		 * `dmi' instead of `set mode dmi'
		 */

		/* *argv is NULL since *module is NULL */
		*argc = 1;
		*argv = malloc(*argc * sizeof(char *));
		argv[0] = malloc((sizeof(*command) + 1) * sizeof(char));
		strncpy(argv[0], *command, sizeof(*command) + 1);
		dprintf("CLI DEBUG: ALIAS %s ", *command);

		strncpy(*command, CLI_SET, sizeof(CLI_SET));	/* set */

		*module = malloc(sizeof(CLI_MODE) * sizeof(char));
		strncpy(*module, CLI_MODE, sizeof(CLI_MODE));	/* mode */

		dprintf("--> %s %s %s\n", *command, *module, argv[0]);
		goto out;
	}

	/* Simple aliases mapping a single command to another one */
	for (i = 0; i < MAX_ALIASES; i++) {
		for (j = 0; j < hdt_aliases[i].nb_aliases; j++) {
			if (!strncmp(*command, hdt_aliases[i].aliases[j],
			    sizeof(hdt_aliases[i].aliases[j]))) {
				dprintf("CLI DEBUG: ALIAS %s ", *command);
				strncpy(*command, hdt_aliases[i].command,
					sizeof(hdt_aliases[i].command) + 1);
				dprintf("--> %s\n", *command);
				goto out; /* Don't allow chaining aliases */
			}
		}
	}
	return;

out:
	dprintf("CLI DEBUG: New parameters:\n");
	dprintf("CLI DEBUG: command = %s\n", *command);
	dprintf("CLI DEBUG: module  = %s\n", *module);
	dprintf("CLI DEBUG: argc    = %d\n", *argc);
	for (i = 0; i < *argc; i++)
		dprintf("CLI DEBUG: argv[%d] = %s\n", i, argv[0]);
	return;
}

/**
 * parse_command_line - low level parser for the command line
 * @line:	command line to parse
 * @command:	first token in the line
 * @module:	second token in the line
 * @argc:	number of arguments
 * @argv:	array of arguments
 *
 * The format of the command line is:
 *	<main command> [<module on which to operate> [<args>]]
 **/
static void parse_command_line(char *line, char **command, char **module,
			       int *argc, char **argv)
{
	int argc_iter = 0, args_pos = 0, token_found = 0, token_len = 0;
	int args_len = 0;
	char *pch = NULL, *pch_next = NULL, *tmp_pch_next = NULL;

	*command = NULL;
	*module = NULL;
	*argc = 0;

	pch = line;
	while (pch != NULL) {
		pch_next = strchr(pch + 1, ' ');
		tmp_pch_next = pch_next;

		/*
		 * Skip whitespaces if the user entered
		 * 'set   mode        foo' for 'set mode foo'
		 *  ^   ^
		 *  |___|___ pch
		 *      |___ pch_next <- wrong!
		 *
		 *  We still keep the position into tmp_pch_next to compute
		 *  the lenght of the current token.
		 */
		while (pch_next != NULL && !strncmp(pch_next, CLI_SPACE, 1))
			pch_next++;

		/* End of line guaranteed to be zeroed */
		if (pch_next == NULL) {
			token_len = (int) (strchr(pch + 1, '\0') - pch);
			args_len = token_len;
		}
		else {
			token_len = (int) (tmp_pch_next - pch);
			args_len = (int) (pch_next - pch);
		}

		if (token_found == 0) {
			/* Main command to execute */
			*command = malloc((token_len + 1) * sizeof(char));
			strncpy(*command, pch, token_len);
			(*command)[token_len] = '\0';
			dprintf("CLI DEBUG: command = %s\n", *command);
			args_pos += args_len;
		} else if (token_found == 1) {
			/* Module */
			*module = malloc((token_len + 1) * sizeof(char));
			strncpy(*module, pch, token_len);
			(*module)[token_len] = '\0';
			dprintf("CLI DEBUG: module  = %s\n", *module);
			args_pos += args_len;
		} else
			(*argc)++;

		token_found++;
		pch = pch_next;
	}
	dprintf("CLI DEBUG: argc    = %d\n", *argc);

	/* Skip arguments handling if none is supplied */
	if (!*argc)
		return;

	/* Transform the arguments string into an array */
	*argv = malloc(*argc * sizeof(char *));
	pch = strtok(line + args_pos, CLI_SPACE);
	while (pch != NULL) {
		dprintf("CLI DEBUG: argv[%d] = %s\n", argc_iter, pch);
		argv[argc_iter] = malloc(sizeof(pch) * sizeof(char));
		strncpy(argv[argc_iter], pch, sizeof(pch));
		argc_iter++;
		pch = strtok(NULL, CLI_SPACE);
		/*
		 * strtok(NULL, CLI_SPACE) over a stream of spaces
		 * will return an empty string
		 */
		while (pch != NULL && !strncmp(pch, "", 1))
			pch = strtok(NULL, CLI_SPACE);
	}
}

/**
 * find_cli_callback_descr - find a callback in a list of modules
 * @module_name:	Name of the module to find
 * @modules_list:	Lits of modules among which to find @module_name
 * @module_found:	Pointer to the matched module, NULL if not found
 *
 * Given a module name and a list of possible modules, find the corresponding
 * module structure that matches the module name and store it in @module_found.
 **/
void find_cli_callback_descr(const char* module_name,
			     struct cli_module_descr* modules_list,
			     struct cli_callback_descr** module_found)
{
	int modules_iter = 0;

	if (modules_list == NULL)
		goto not_found;

	/* Find the callback to execute */
	while (modules_list->modules[modules_iter].name &&
	       strcmp(module_name,
		      modules_list->modules[modules_iter].name) != 0)
		modules_iter++;

	if (modules_list->modules[modules_iter].name) {
		*module_found = &(modules_list->modules[modules_iter]);
		dprintf("CLI DEBUG: module %s found\n", (*module_found)->name);
		return;
	}

not_found:
	*module_found = NULL;
	return;
}

/**
 * autocomplete_command - print matching commands
 * @command:	Beginning of the command
 *
 * Given a string @command, print all availables commands starting with
 * @command. Commands are found within the list of commands for the current
 * mode and the hdt mode (if the current mode is not hdt).
 **/
static void autocomplete_command(char *command)
{
	int j = 0;
	struct cli_callback_descr* associated_module = NULL;

	/* First take care of the two special commands: 'show' and 'set' */
	if (strncmp(CLI_SHOW, command, strlen(command)) == 0) {
		printf("%s\n", CLI_SHOW);
		autocomplete_add_token_to_list(CLI_SHOW);
	}
	if (strncmp(CLI_SET, command, strlen(command)) == 0) {
		printf("%s\n", CLI_SET);
		autocomplete_add_token_to_list(CLI_SET);
	}

	/*
	 * Then, go through the modes for the special case
	 *	'<mode>' -> 'set mode <mode>'
	 */
	while (list_modes[j]) {
		if (strncmp(list_modes[j]->name, command, strlen(command)) == 0) {
			printf("%s\n", list_modes[j]->name);
			autocomplete_add_token_to_list(list_modes[j]->name);
		}
		j++;
	}

	/*
	 * Let's go now through the list of default_modules for the current mode
	 * (single token commands for the current_mode)
	 */
	j = 0;
	while (current_mode->default_modules->modules[j].name) {
		if (strncmp(current_mode->default_modules->modules[j].name,
			    command,
			    strlen(command)) == 0) {
			printf("%s\n",
				current_mode->default_modules->modules[j].name);
			autocomplete_add_token_to_list(current_mode->default_modules->modules[j].name);
		}
		j++;
	}

	/*
	 * Finally, if the current_mode is not hdt, list the available
	 * default_modules of hdt (these are always available from any mode).
	 */
	if (current_mode->mode == HDT_MODE)
		return;

	j = 0;
	while (hdt_mode.default_modules->modules[j].name) {
		/*
		 * Any default command that is present in hdt mode but
		 * not in the current mode is available. A default
		 * command can be redefined in the current mode though.
		 * This next call tests this use case: if it is
		 * overwritten, do not print it again.
		 */
		find_cli_callback_descr(hdt_mode.default_modules->modules[j].name,
					current_mode->default_modules,
					&associated_module);
		if (associated_module == NULL &&
		    strncmp(command,
			    hdt_mode.default_modules->modules[j].name,
			    strlen(command)) == 0) {
			printf("%s\n",
				hdt_mode.default_modules->modules[j].name);
			autocomplete_add_token_to_list(hdt_mode.default_modules->modules[j].name);
		}
		j++;
	}
}

/**
 * autocomplete_module - print matching modules
 * @command:	Command on the command line (not NULL)
 * @module:	Beginning of the module
 *
 * Given a command @command and a string @module, print all availables modules
 * starting with @module for command @command. Commands are found within the
 * list of commands for the current mode and the hdt mode (if the current mode
 * is not hdt).
 **/
static void autocomplete_module(char *command, char* module)
{
	int j = 0;
	char autocomplete_full_line[MAX_LINE_SIZE];

	if (strncmp(CLI_SHOW, command, strlen(command)) == 0) {
		while (current_mode->show_modules->modules[j].name) {
			if (strncmp(current_mode->show_modules->modules[j].name,
				    module,
				    strlen(module)) == 0) {
				printf("%s\n",
					current_mode->show_modules->modules[j].name);
				sprintf(autocomplete_full_line, "%s %s",
					CLI_SHOW, current_mode->show_modules->modules[j].name);
				autocomplete_add_token_to_list(autocomplete_full_line);
			}
		j++;
		}
	} else if (strncmp(CLI_SET, command, strlen(command)) == 0) {
		j = 0;
		while (current_mode->set_modules->modules[j].name) {
			if (strncmp(current_mode->set_modules->modules[j].name,
				    module,
				    strlen(module)) == 0) {
				printf("%s\n",
					current_mode->set_modules->modules[j].name);
				sprintf(autocomplete_full_line, "%s %s",
					CLI_SET, current_mode->set_modules->modules[j].name);
				autocomplete_add_token_to_list(autocomplete_full_line);
			}
			j++;
		}
	}
}

/**
 * autocomplete - find possible matches for a command line
 * @line:	command line to parse
 **/
static void autocomplete(char *line)
{
	int i;
	int argc = 0;
	char *command = NULL, *module = NULL;
	char **argv = NULL;

	parse_command_line(line, &command, &module, &argc, argv);

	/* If the user specified arguments, there is nothing we can complete */
	if (argc != 0)
		goto out;

	/* No argument, (the start of) a module has been specified */
	if (module != NULL) {
		autocomplete_module(command, module);
		goto out;
	}

	/* No argument, no module, (the start of) a command has been specified */
	if (command != NULL) {
		autocomplete_command(command);
		goto out;
	}

	/* Nothing specified, list available commands */
	//autocomplete_commands();

out:
	/* Let's not forget to clean ourselves */
	free(command);
	free(module);
	for (i = 0; i < argc; i++)
		free(argv[i]);
	free(argv);
	return;
}


/**
 * exec_command - main logic to map the command line to callbacks
 **/
static void exec_command(char *line,
			 struct s_hardware *hardware)
{
	int argc, i = 0;
	char *command = NULL, *module = NULL;
	char **argv = NULL;
	struct cli_callback_descr* current_module = NULL;

	/* This will allocate memory that will need to be freed */
	parse_command_line(line, &command, &module, &argc, argv);

	/* Expand shortcuts, if needed */
	expand_aliases(line, &command, &module, &argc, argv);

	if (module == NULL) {
		dprintf("CLI DEBUG: single command detected\n", CLI_SHOW);
		/*
		 * A single word was specified: look at the list of default
		 * commands in the current mode to see if there is a match.
		 * If not, it may be a generic function (exit, help, ...). These
		 * are stored in the list of default commands of the hdt mode.
		 */
		find_cli_callback_descr(command, current_mode->default_modules,
				        &current_module);
		if (current_module != NULL)
			return current_module->exec(argc, argv, hardware);
		else if (!strncmp(command, CLI_SHOW, sizeof(CLI_SHOW) - 1) &&
			 current_mode->show_modules != NULL &&
			 current_mode->show_modules->default_callback != NULL)
			return current_mode->show_modules
					   ->default_callback(argc,
							      argv,
							      hardware);
		else if (!strncmp(command, CLI_SET, sizeof(CLI_SET) - 1) &&
			 current_mode->set_modules != NULL &&
			 current_mode->set_modules->default_callback != NULL)
			return current_mode->set_modules
					   ->default_callback(argc,
							      argv,
							      hardware);
		else {
			find_cli_callback_descr(command, hdt_mode.default_modules,
					        &current_module);
			if (current_module != NULL)
				return current_module->exec(argc, argv, hardware);
		}

		printf("unknown command: '%s'\n", command);
		return;
	}

	/*
	 * A module has been specified! We now need to find the type of command.
	 *
	 * The syntax of the cli is the following:
	 *    <type of command> <module on which to operate> <args>
	 * e.g.
	 *    dmi> show system
	 *    dmi> show bank 1
	 *    dmi> show memory 0 1
	 *    pci> show device 12
	 *    hdt> set mode dmi
	 */
	if (!strncmp(command, CLI_SHOW, sizeof(CLI_SHOW) - 1)) {
		dprintf("CLI DEBUG: %s command detected\n", CLI_SHOW);
		find_cli_callback_descr(module, current_mode->show_modules,
					&current_module);
		/* Execute the callback */
		if (current_module != NULL)
			return current_module->exec(argc, argv, hardware);
		else {
			find_cli_callback_descr(module, hdt_mode.show_modules,
					        &current_module);
			if (current_module != NULL)
				return current_module->exec(argc, argv, hardware);
		}

		printf("unknown module: '%s'\n", module);
		return;

	} else if (!strncmp(command, CLI_SET, sizeof(CLI_SET) - 1)) {
		dprintf("CLI DEBUG: %s command detected\n", CLI_SET);
		find_cli_callback_descr(module, current_mode->set_modules,
					&current_module);
		/* Execute the callback */
		if (current_module != NULL)
			return current_module->exec(argc, argv, hardware);
		else {
			find_cli_callback_descr(module, hdt_mode.set_modules,
					        &current_module);
			if (current_module != NULL)
				return current_module->exec(argc, argv, hardware);
		}

		printf("unknown module: '%s'\n", module);
		return;

	}

	printf("I don't understand: '%s'. Try 'help'.\n", line);

	/* Let's not forget to clean ourselves */
	free(command);
	free(module);
	for (i = 0; i < argc; i++)
		free(argv[i]);
	free(argv);
}

static void reset_prompt()
{
	/* No need to display the prompt if we exit */
	if (hdt_cli.mode != EXIT_MODE) {
		printf("%s", hdt_cli.prompt);
		/* Reset the line */
		memset(hdt_cli.input, '\0', MAX_LINE_SIZE);
		hdt_cli.cursor_pos = 0;
	}
}

void start_auto_mode(struct s_hardware *hardware)
{
        char *mypch;
	int nb_commands=0;
	char *commands[MAX_NB_AUTO_COMMANDS];

	if (!quiet)
        	more_printf("\nEntering Auto mode\n");

	/* Protecting the auto_label from the strtok modifications */
        char *.g.p i.mol/
   _mode->show_moduleuto_la;sL i.mautoco Autorrdescr(mo	/* PrloOvApng
   _mtions */
        char *.g. else de->shol:R(NDS];
trompt);
	se )
		r.mautods=0;
	c mpt);
	se ules
 * staes
 * stNDS];
trompt);s the folULL)
				return current_module->exec(d Auto NDS];
tro Autoci_callback_descr(command, hdt_mode.default_modules,
					        &current_module);
			if (cur e strtok mo.cursor_pos) {
		dpro_e_full_liyrI     abel fr of) a  the strtok modifications */
    	if (hdt_cli.mul*nS m@commnchar *.g.iS mnt_LINE_cursor_pos
 =0;
	c mpt);
	se ules
 * staes
 * stNDS];
trompt);sv = NULL;OvApng
   _mtions */
        char *.g. else de->shol:R(NDS];
trompt);
	se )
		r.mautods=0;
	c mpt);
	se ules
 * staes
 * stNDS];
trompt);s the folUurn currentIEI	}
eo dicurren,;

	if 	_pos) {
		dpro_e_full_liyrI   ;s the ff
		dprro_e_dicu_descr(command, >set_mi
    	My
}

she line utoeuto_lo Npene utoeut_ypt);s tinrau.ode_descr(mode_s_to_mode_t(*command), &mode);
	i&		    urrent_mot	#f;	rnnw t);sv = NULL;O
	se )
_lo NpetIEI	}   _mti
    	MyLTRa (hdt_cl	#f;	rn&
			 crthated_modul,yTtrtoeut_ypt);s -s	 crthated_modul,yTg
   _IEI	}   _mti
    	My;
		els >s >s >s >sL0/i cl crtha ls,
					&cRNDS];
t>s >s >s >sL0LI_SHOW);
		find_cli_callbao		find_clii_callba_ pch
				        &current_mod(hated__f the cli isylabel from   	My;
		els >s >s >s >sL0/i cl crtha0/i ->sL0/i cl crtha0/i ();s ths
	se )
	d, &module, &argc, argv);

	/* If the user specified arguments, there is nothing we Ikgt	#fs noth""""""""""""""e
	/*     	Mec = %d\n", d  	Mec_mode,
	&kernel_mw   	Mec = %d\n", des[* A mV(char *command)
{
	int j = 0;
	struct cli_callback_descr* associated_module = NULL;

	/* FirstnsIh    _moo(dproow   	dproow  truct cli_callback_descr* associ    charis nothing weM_gs_len;
		-?  k]eo);
}

static h
		 *      |___ pi cl c;
	spi cl w  ts >eHete_full_line[MAtewcNitc = %d\n",aroyproow .T\n",ac(d Autrchr(tnSben;
		-?  kd_cli_cRnee		t>s >sL_ypt);s tinrau.ode_descr(mode_s_to_mode_t(*co_.);

	/* Litc = %d\n",aroyproow .T\n",ac(d Autrchr(tnSbch
	ee clirooypt)rompuser spwck_dgINE_cursor_tYheai",arzeof(CLI_SHOW)}FirstSbenesoci    chahearcli. ", &benesoci   NULLmode_tltm,ode_descrWtarcli.. ", &benociated_module = NULL;

	/* FirstnsIh    _moo(dproow   	 find a callbac\n", d  	Mec_mode,
	&kernel_vt>s >sm clisep spwck_dgINE_Onociated_ciatn = N&ed  	Mec_1nociciatn = Nl_vt>s >sm  kd_clatn =;

	/* latnYoAE*proow   	dproow  tN
nYoAE*poypt)rompusvStltn =odulesware);
		}temS.rstSbende_desc12
fn=(cotSbeiDS];
trompts
 rT*ie"NE_Ock_dgIdule,
iwdesc12
fn=(cotSn
 * staes
 * stNrMl_callk_descr* associated_module = NULL;

	/llk_dende_d
		}temS.pt)rompusvStltn =odulVe )
	d, & U	frlNULL;

	/llk_dende_heai",t);
	se )
		r.mautod     LtT se )rljoR"lse if (!Mec_mo to cleg 
fn=(cotSn
 (. g 
fn=(cotSn
dicu_d =;

   ;fn=(cotSn
dicu_d =;

   ;fn=(cotSucotSnfcotSucotSnfcotSucotSnfcotSucotSnfcotSucotSnfcotSucotSnfcotSucotSnfcotSucotSnfcotSucotSnfcotSucotSnfcotSucotSnfcoAodule)liyre = NU)rljoR"v = gtSucotSnfcotSucSucotSnfcotSucSuc	/*
	 * Let's go now throt)romp	/*
 |sociated_ms
 rT*ie"NE_O]cursor_pos = 0;
	
spewEcotSucotSnfcotSucotSnf(xwheacSucmoR"v = gtSu)ucotSrite_copgv);

	/* If tthe fe_to_mOn=(cotSbeiDYrcotucotSnf(xwheacre **lewEcotSucotSnfcotSucotSnf(xwheacSucmoRSnfcotSI

	/utSucohtNrMlAxwho now throt)romatnYoA;oewEotSnfcotSucotirM_gs_len;
		ucotSnfcotSurn;
lLI_NrMlAxwho nnfcolLI_A) eociate
-sVse d asst& r(mode
-sVse d ass _len se )rljoR"lse if (!MecdOntf(hdt_cl12
fn=Dv(uotSnfc_li].namerMlAxhS,
 
fn=Dv(uotSnfc_li)rljt	#fs noth""""De* If :tSnfcotsVse dRT*,mot	#f;	rnnw t);sv = N	rnnw t)rot_NB_A	/*2ucotSnfcotSu",t = gtSu)ucotSriucohtt_modules->moT*,mot	#f;	rnnw t);sv = N	rnnw t)rot_NB_A	/*2ucotSnfcotSu",t = gtSu)ucotSriucohtt_modules->moT*,mot	#f;	rnnw t);sv = N	rnnw t)rot_NB_A	/*2ucotSnfcotSu",t = gtSu)ucotSriucohtntf("		 current_mode->show_modules != NULL &&
			 currenthing j(*coTRa (hdt_allbaSnfcot,mot	#f;	rne nnw t);sv = 	/*2ucmot	#fnt_module);
		/* Exe,-(a Dv(uotSnfc_li)rljtu	 currentcotSrxec(roOdules,Fpu*otSnfc_LIh.yhLL;

	/llk_dende_htSnfc_liSnfc]eASRurrentcotSrxeTa (hdt_alU stNrMl_alU s'ee nse )rrentcotSrxI_SHOW]_prentcotS]>kSu)ucotSriucohtntf("		 current_mode->htnmtr.tu	 cPnnwts)ucotSriucohtLI DEB	rnnw  ueh"}

	ucotS    &c t);sdu_dl

	u.vyTcotSrx* IfhtnLvor_poHOW]_pr*,mot	#f;LI DE * s	u.vyT* staes
 * s	u.obRx(EOW]>htnmtr.tu	 cPnnwts)ucbRx(E b(xw	ucotSn)wEc_mti
 ck_l&!= NULL NULL;g	&c t);sdu_ , & U	frlNULL;

	/llk_cur      s) {
		dpro_el&!= NULL ornnwtn cuesiuco(NULL }rentcotSucotSsiuco(NULrent s)    thW=argc, argv, hardware);
		ocothW=argc, arSriOt_module);
			if tocomps >sm clisPenS m@crljtu	 curres >sm cliljtu	 curreculet * s	u.obRx(EOW]>hc	u.obRx,;s	u.obRx(EOW]>hc	ueentco'ucotSrihi	_moode->default_modulNe = NU)rljoR"v = gt__iijgExHlAxwhoUdulNe = N-mps entf(".obRxcutelAxwho now throt)romatnYoA;oewEotSnfcotSucotirM_gs_len;
		ucotSnfcotSurn;
lLI_NrMlAxwho nnfcolLI_A) eociate
-sVse d asst& r(mode
-sVse d ass _len se )rljoR"lse if (!MecdOntf(hdt_cl12
fn=Dv(uotSnfc_li].nj'help_aliases12
eardwLL ornnwtn cuesiuco(NULL }rTliasR"lse if (!Mo now throt)romatnYoA;oewE R"lse if (!MecdOnu	 S]utelA
fnlrMlAxhS,
 
fnnwtf (!Memd(curre
dnw t);sv = NMAxhS,
 e arSrdaRT*,mot	#f;	eoc_cur  rdaRT*,mot	r  rdaRT; curres >sm e_descr(moOss _lwtn cuesiucoR;w>rcs >snOot,ss _lwtn cuesiucoRmoo(dproow   	dp fs _lwtn cuesiudende_d\n"l = N
-sVren].nameEnesoci   NULLmode_tltm,ode_des* stNDS]SN	ucot"l = N
-sVr NULL ornnwtgldescule, (the stadescLSu)cur 	LI_SET)r NU] stadescLSu)cur 	sack_descr* associated_modescLSu)irM0e_d\n"l = wcurreassociat)irc_cur  rdaRT*,mk_descr*Dpes,F man cLSu)irM0esdescL[k)irM0e_d\k]eoeUtsw>defamodescydescr* assoc_modesicuesiu#fnt_massoc_modesicuesiu#fnt_massoc_modecurrent_msckfc_AmM scr*Dpes,F man cfule, (the staelsed_modescLSu) == 0)dltm,od	/*Ne-a)romputhdt_alitc = %d\	} e'

	/llkIint_massstaes
 * s	u.obRx(,cotSrxeTastlh1ed_modscLSu) == * s	u. -mps enmeEnesoci   NULLmode_tltm,od
	LI;s	u.obRx(,cotSrxeTastlh1ed_modscLSu) == * s	u. -mps enmeEnesoci   NULLmode_tltm,od
	LI;s	u.obRx(,cotSrxeTastlh1e_]wLI;saLmstlhkrrnnw  ueh"}

	u%d\	} e'
,od	/*Ne-a)romu

statenmeEnesoci   NUL_SHOW,  cotSucotSnfcotSucotSnfcotSuucotkd_clatn =;
>modules[j].[od	/*Ne-a)romputhdt_alitc = %d\	} e'

	/llkIinlh1ed_d tthfull_lB'Minlh1ed_d )roDia-iltCBlabelcurrene tthfull_lB'MiUfv, har(elcurSu)rintfck_da.sW]_pr*,mot	#f;LI DedR
intfck_da.sW]_pr*,m/ n
intfck_dRlLI_A)Eneso e'

	/ll* Leter;
	ucotSI	}*Ne-a)de);
ESscotSucbucbu associated_modes	ucotSnfceng j(*con *moddul,yTljtu	 chS,
e le/*;s ths
	_alitc = %d\	} e'

	/llkIinlh1ed_d tthfull_lB'Minlh1ed_d )roDia-iltCBlabelcurrene tthfull_lB'MiUjcu_d w
intfck_datCBlabelcurrenli].nameR
intfck_
	j = 0;
	escr* *TS
intfck_dRlLI_@WjoReR
it)romSscotSucb,lh1ed_d tthfull&lLI_A)Eneso e'

	/Ytclas icomine tthfull_lB'Mi.n cues)Enrre> the sc_mode	ull&lLI_rrene*argPon o callbacks
 **/
staticEnt mode tho,meEnesoci   Ns/oci i].M-dde	ull&lLlas icos j(*con *moddul,yTcastlh1ed_mo)
     iodul)*,me"hS]>kSustlh1efc_liSnfcR, har(elcurSu)rintfck_dle
	 *Su)e,
		 I-har(el'MiUjcu_d w
int aels.nameR
intfck_
	j = 0;
	escr* *TS
intfck_dRlLI_@WjoReR
it)romSsco_mti
 ck_e->modee*f("		 c
intfck_
	j = 0;
	escbucbu associated_modes	ulIit_modes[i])
		*mod!ulIpSl,yTlRles[joD*rnesiucoR;w>rcs >snOot,ss _lwtnucbu associated_sco_mti
 ck_e->modee2ucotSnfcocSucmoR"v = gtSu)ucotSrhhTmdde	ull&lLlas ico 	dproow  pr currenthinas ico 	dprucotSlt)mw   	Mec =is as ico dprocotSnwnwts;w>rcs >sucotSlt)mw   	Mec =is as la (hdt_alset_ A=lB'Mi.n cueilB'Mi.n cueilB'MinfcocSucintfck_dRlLI_A)Eneso e'

	/;
	}
	ico 	s la (hdteB'Mi.no 	dpbe freed */
	parse_obRx(,cotSrxeTastlh1ed_modscLSu) == * s	                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                          