/* Ergo, version 3.8.2, a program for linear scaling electronic structure
 * calculations.
 * Copyright (C) 2023 Elias Rudberg, Emanuel H. Rubensson, Pawel Salek,
 * and Anastasia Kruchinina.
 * 
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * 
 * Primary academic reference:
 * Ergo: An open-source program for linear-scaling electronic structure
 * calculations,
 * Elias Rudberg, Emanuel H. Rubensson, Pawel Salek, and Anastasia
 * Kruchinina,
 * SoftwareX 7, 107 (2018),
 * <http://dx.doi.org/10.1016/j.softx.2018.03.005>
 * 
 * For further information about Ergo, see <http://www.ergoscf.org>.
 */

/** @file random_matrices.h

    @brief Header file containing declarations of functions required for testing purposes.
    Functions include generation of the random dense matrices, random sparse symmetric matrices, initialization of the 
    hierarchical matrix structure, work with files and printing matrix to the screen.

    @author Anastasia Kruchinina 
    @sa random_matrices.cc
*/


#ifndef HEADER_RANDOM_MATRICES
#define HEADER_RANDOM_MATRICES


#include "matrix_typedefs.h" // definitions of matrix types and interval type (source)
#include "realtype.h"   // definitions of types (utilities_basic)
#include "matrix_utilities.h"
#include "integral_matrix_wrappers.h"
#include "SizesAndBlocks.h"
#include "Matrix.h"
#include "Vector.h"
#include "MatrixSymmetric.h"
#include "MatrixTriangular.h"
#include "MatrixGeneral.h"
#include "VectorGeneral.h"
#include "output.h"

#include "files_dense.h"
#include "files_sparse.h"

#include <iostream>
#include <fstream>
#include <sstream>
#include <string.h>


using namespace std;



typedef intervalType IntervalType;
typedef symmMatrix MatrixTypeInner;
typedef triangMatrix TriangMatrixType;
typedef normalMatrix MatrixGeneral;

typedef std::vector<int> VectorTypeInt;

#ifdef PRECISION_QUAD_FLT128
#define MAX_DOUBLE FLT128_MAX 
#define MIN_DOUBLE FLT128_MIN
#else 
#define MAX_DOUBLE std::numeric_limits<ergo_real>::max()
#define MIN_DOUBLE std::numeric_limits<ergo_real>::min()
#endif


#define PI 3.14159265 // needed for sprandsym


void print_ergo_matrix(const MatrixTypeInner &F);
void print_matrix(std::vector<ergo_real> const &A);
template<typename Matrix>
void init_matrix(Matrix &X, const int N, int blockSizesMultuple = 4);
void get_random_symm_matrix(int N, MatrixTypeInner &X, int blockSizesMultuple = 4);
void get_random_sparse_blocks_matrix(int N, MatrixTypeInner &X, int blockSizesMultuple = 4, ergo_real probabilityBeingZero = 0);
void get_all_eigenvalues_of_matrix(std::vector<ergo_real> & eigvalList, const MatrixTypeInner & M);
void sprandsym(int N, MatrixTypeInner &X, MatrixGeneral &Q, vector<ergo_real> &D, const double MATRIX_SPARSITY);
int get_matrix_from_sparse(char *filename, MatrixTypeInner &X);
int get_matrix_from_sparse_vec(char *filename, std::vector<int> &I, std::vector<int> &J, std::vector<ergo_real> &val);
int get_matrix_from_binary(char *filename, MatrixTypeInner &X);
int get_matrix_from_binary_vec(char *filename, std::vector<int> &I, std::vector<int> &J, std::vector<ergo_real> &val, int &N);
int get_matrix_from_full(char * filename, MatrixTypeInner &X);


/**
 *  \brief Create hierarchical matrix structure.
 *
 *  \tparam Matrix    type of the matrix (ex. symmMatrix)
 */
template<typename Matrix>
void init_matrix(Matrix &X, const int N, int blockSizesMultuple /*=4*/)
{
  /********** Initialization of SizesAndBlocks */
  int size = N;
  int nlevels = 5; //!!!
  std::vector<int> blockSizes(nlevels);
  blockSizes[nlevels - 1] = 1; // should always be one
    for (int ind = nlevels - 2; ind >= 0; ind--)
      blockSizes[ind] = blockSizes[ind + 1] * blockSizesMultuple;
  mat::SizesAndBlocks rows(blockSizes, size);
  mat::SizesAndBlocks cols(blockSizes, size);
  /********************************************/  
  X.resetSizesAndBlocks(rows,cols);
}


#endif // HEADER_RANDOM_MATRICES
