/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */

 /* (C) Marcin Kwadrans <quarter@users.sourceforge.net> */

#include <cstdio>
#include "include/support.h"
#include "project.h"
#include "environment.h"

/*! \brief Konstruktor

	Tworzy projekt
*/
LWProject::LWProject (): file_name (NULL)
{
	world = new LWBoard (LW_TYPE_WORLD);
	world->setSize (10, 10);
	
	program = new LWBoard (LW_TYPE_PROGRAM);
}

/*! \brief Destruktor

	Niszczy zbiór ikon
*/	
LWProject::~LWProject ()
{
	delete (world);
	delete (program);
	
	if (file_name != NULL)
		g_free (file_name);
	
	if (this == LWEnvironment::getProject())
		LWEnvironment::unsetProject ();
}

/*! \brief Ładowanie pliku projektu

	Ładuje projekt z pliku
	\param a_file_name Nazwa pliku z projektem
	\return TRUE if success, otherwise FALSE.
*/	
gboolean LWProject::load (const gchar *a_file_name)
{
	/* Parse the file and get the DOM */
    xmlDoc *doc = xmlParseFile(a_file_name);

	if (doc == NULL) return FALSE;

    /* Get root node */
    xmlNode *node = xmlDocGetRootElement(doc);

	g_return_val_if_fail (!xmlStrcasecmp (node->name, (xmlChar *) "LittleWizardProject"), FALSE);
	
	/* Version check */
	xmlChar *version = xmlGetProp (node, (xmlChar *) "version");
	if (version != NULL) {
		unsigned int promajor=0, prominor=0, envminor=0, envmajor=0;
		sscanf ((char *) version, "%u.%u", &promajor, &prominor);
		sscanf (VERSION, "%u.%u", &envmajor, &envminor);
		xmlFree (version);
		g_return_val_if_fail (promajor < envmajor || (promajor == envmajor && prominor <= envminor), FALSE);
	}
	
#if 1
	/* Workaround GTK bug */
	gtk_widget_hide (world->getWidget());
	gtk_widget_hide (program->getWidget());
#endif
	
	world->restoreFromXML (node->children);
	program->restoreFromXML (node->children->next);
	
#if 1
	gtk_widget_show (world->getWidget());
	gtk_widget_show (program->getWidget());
#endif
	
	xmlFreeDoc (doc);
	
	setFileName (a_file_name);	
	
	return TRUE;
}

/*! \brief Zapisywanie projektu do pliku

	Zapisuje projekt do pliku
	\param a_file_name Nazwa pliku, pod którą ma być zapisany projekt
	\return TRUE if success, otherwise FALSE.
*/	
gboolean LWProject::save (const gchar *a_file_name)
{
    xmlDoc *doc = xmlNewDoc(BAD_CAST "1.0");
    xmlNode *node = xmlNewNode(NULL, BAD_CAST "LittleWizardProject");

	/* Setting proper version for project */
    gchar **versionarr = g_strsplit (VERSION, ".", 3);
	gchar *version = g_strjoin (".", versionarr[0], versionarr[1], NULL);
	g_strfreev (versionarr);
	xmlNewProp (node, (xmlChar *) "version", (xmlChar *) version);
	g_free (version);
	
	xmlDocSetRootElement(doc, node);
	
	world->storeToXML (node);
	program->storeToXML (node);

	int r = xmlSaveFile (a_file_name, doc);
	
	xmlFreeDoc (doc);
	
	setFileName (a_file_name);
	
	return (r > -1) ? TRUE : FALSE;
}

/*! \brief Ustawia nazwę pliku

	Ustawia nazwę pliku pod którą został zapisany projekt	
	\param a_file_name Nazwa pliku, pod którą został zapisany projekt
*/	
void LWProject::setFileName (const gchar *a_file_name)
{
	g_return_if_fail (a_file_name != NULL);
	
	if (file_name != NULL)
		g_free (file_name);
	
	file_name = g_strdup (a_file_name);
}

/*! \brief Pobierz nazwę pliku

	Pobiera nazwę pliku pod którą został zapisany projekt	
	\return Nazwa pliku, pod którą został zapisany projekt.
	Należy ją zwolnić samodzielnie
*/	
gchar *LWProject::getFileName ()
{
	return (file_name != NULL) ? g_strdup (file_name) : NULL;
}

/*! \brief Pobierz świat

	Pobiera planszę ze światem wchodzącą w skład projektu
	\return Plansza ze światem
*/
LWBoard *LWProject::getWorld ()
{
	return world;
}

/*! \brief Pobierz program

	Pobiera planszę z programem wchodzącą w skład projektu
	\return Plansza z programem
*/

LWBoard *LWProject::getProgram ()
{
	return program;
}
