/* 
 * $Id: su.c,v 1.12 1997/06/24 14:18:06 morgan Exp morgan $
 *
 * su.c
 *
 * ( based on an implementation of `su' by
 *
 *     Peter Orbaek  <poe@daimi.aau.dk>
 *
 * obtained from ftp://ftp.daimi.aau.dk/pub/linux/poe/ )
 *
 * Rewritten for Linux-PAM by Andrew Morgan <morgan@linux.kernel.org>
 *
 * $Log: su.c,v $
 * Revision 1.12  1997/06/24 14:18:06  morgan
 * update for .55 release
 *
 * Revision 1.11  1997/02/24 06:02:03  morgan
 * major overhaul, read diff
 *
 * Revision 1.10  1997/01/29 03:35:39  morgan
 * update for release
 *
 * Revision 1.9  1996/12/15 16:27:27  morgan
 * root can ignore account management
 *
 * Revision 1.8  1996/12/01 00:54:05  morgan
 * extra checks for no user shell
 */

#define ROOT_UID                  0
#define DEFAULT_HOME              "/"
#define DEFAULT_SHELL             "/bin/sh"
#define SLEEP_TO_KILL_CHILDREN    3  /* seconds to wait after SIGTERM before
					SIGKILL */
#define SU_FAIL_DELAY     2000000    /* usec on authentication failure */

#define _BSD_SOURCE                  /* for setgroups() and other calls */
#include <stdlib.h>
#include <signal.h>
#include <termios.h>
#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <unistd.h>
#include <pwd.h>
#include <grp.h>
#include <sys/file.h>
#include <string.h>
#include <stdarg.h>
#include <syslog.h>

#include <security/pam_appl.h>
#include <security/pam_misc.h>

#ifdef HAVE_PWDB
#include <pwdb/pwdb_public.h>
#endif

#include "../inc/make_env.-c"
#include "../inc/setcred.-c"
#include "../inc/shell_args.-c"
#include "../inc/wait4shell.-c"
#include "../inc/wtmp.-c"

/* -------------------------------------------- */
/* ------ declarations ------------------------ */
/* -------------------------------------------- */

static int is_terminal = 0;
static pam_handle_t *pamh = NULL;
static int state;

#define SU_STATE_PAM_INITIALIZED     1
#define SU_STATE_AUTHENTICATED       2
#define SU_STATE_AUTHORIZED          3
#define SU_STATE_SESSION_OPENED      4
#define SU_STATE_CREDENTIALS_GOTTEN  5
#define SU_STATE_PROCESS_UNKILLABLE  6
#define SU_STATE_TERMINAL_REOWNED    7
#define SU_STATE_FORKED              8

static void store_terminal_modes(void);
static int reset_terminal_modes(void);
static void disable_terminal_signals(void);
static void enable_terminal_signals(void);
static int change_terminal_owner(uid_t uid, int is_login, int command_present
	, const char **callname, const char **err_descr);
static void restore_terminal_owner(void);
static int make_process_unkillable(const char **callname
        , const char **err_descr);
static void make_process_killable(void);
static void exit_now(int exit_code, const char *format, ...);
static void exit_child_now(int exit_code, const char *format, ...);
static void usage(void);
static void parse_command_line(int argc, char *argv[]
	, int *is_login, const char **user, const char **command);
static void do_pam_init(const char *user, int is_login);
static void su_exec_shell(const char *shell, uid_t uid, int is_login
			  , const char *command, const char *user);

/* -------------------------------------------- */
/* ------ the application itself -------------- */
/* -------------------------------------------- */

void main(int argc, char *argv[])
{
    int retval, is_login, status, final_retval;
    const char *command, *user;
    const char *shell=NULL;
    pid_t child;
    uid_t uid;
    const char *place, *err_descr;

    /*
     * store terminal modes for later
     */
    store_terminal_modes();

    /*
     * Turn off terminal signals - this is to be sure that su gets a
     * chance to call pam_end() in spite of the frustrated user
     * pressing Ctrl-C. (Only the superuser is exempt in the case that
     * they are trying to run su without a controling tty).
     */

    disable_terminal_signals();

    /* ------------ parse the argument list ----------- */

#ifdef HAVE_PWDB
    retval = pwdb_start();
    if (retval != PWDB_SUCCESS)
	exit_now(1, "su: failed\n");
#endif

    parse_command_line(argc, argv, &is_login, &user, &command);

    /* ------ initialize the Linux-PAM interface ------ */

    do_pam_init(user, is_login);      /* call pam_start and set PAM items */
    user = NULL;                      /* get this info later (it may change) */

    /*
     * Note. We have forgotten everything about the user. We will get
     * this info back after the user has been authenticated..
     */

    /*
     * Starting from here all changes to the process and environment
     * state are reflected in the change of "state".
     * Random exits are strictly prohibited :-)  (SAW)
     */
    status = 1;
    err_descr = NULL;
    state = SU_STATE_PAM_INITIALIZED;

    do {                              /* abuse loop to avoid using goto... */

	place = "pam_authenticate";
        retval = pam_authenticate(pamh, 0);	   /* authenticate the user */
	if (retval != PAM_SUCCESS)
            break;
	state = SU_STATE_AUTHENTICATED;

	/*
	 * The user is valid, but should they have access at this
	 * time?
	 */

	place = "pam_acct_mgmt";
        retval = pam_acct_mgmt(pamh, 0);
	if (retval != PAM_SUCCESS) {
	    if (getuid() == 0) {
		(void) fprintf(stderr, "Account management:- %s\n(Ignored)\n"
                               , pam_strerror(pamh,retval));
	    } else {
		break;
	    }
	}
	state = SU_STATE_AUTHORIZED;

	/* open the su-session */

	place = "pam_open_session";
        retval = pam_open_session(pamh,0);      /* Must take care to close */
	if (retval != PAM_SUCCESS)
            break;
	state = SU_STATE_SESSION_OPENED;

	/*
	 * We obtain all of the new credentials of the user.
	 */

	place = "set_user_credentials";
        retval = set_user_credentials(pamh, is_login, &user, &uid, &shell);
	if (retval != PAM_SUCCESS) {
	    (void) pam_close_session(pamh,retval);
	    break;
	}
	state = SU_STATE_CREDENTIALS_GOTTEN;
	
        /*
         * XXX: Because of the way Linux, Solaris and others(?) handle
         * process control, it is possible for another process (owned by
         * the invoking user) to kill this (setuid) program during its
         * execution. This breaks the integrety of the PAM model (which
         * assumes that pam_end() is correctly called in all cases).  As
         * of 1997/3/8, this still needs to be addressed.  At this time,
         * is not clear if this is an issue for the kernel, or requires
         * some enhancements to the PAM spec. (AGM)
         *
         * I see only one way to protect us from unexpected killing --
         * changing our uid. So we should choose between
         *     a) an authentication with changed uid, and
         *     b) a possibility to be killed before pam_end().
         * I suppose that uid assumptions should be added to the PAM spec
         * and in the future to the PAM RFC. (SAW)
         *
	 * If "su" is installed suid-root (as usually)
	 * from now only root can kill the process.
	 */
	if (make_process_unkillable(&place, &err_descr) != 0)
	    break;
	state = SU_STATE_PROCESS_UNKILLABLE;

        retval = change_terminal_owner(uid, is_login, command != NULL
                , &place, &err_descr);
	if (retval > 0) {
	    (void) fprintf(stderr, "su: %s: %s\n", place, err_descr);
	    err_descr = NULL;
	}else if (retval < 0)
	    break;
	state = SU_STATE_TERMINAL_REOWNED;

	/* this is where we execute the user's shell */
	{
	    int fds[2];        /* this is a hack to avoid a race condition */

	    if (is_login && pipe(fds) == -1) {
                place = "pipe";
		err_descr = strerror(errno);
		break;
	    }

	    child = fork();
	    if (child == -1) {
                place = "fork";
		err_descr = strerror(errno);
		break;
	    }
	    state = SU_STATE_FORKED;

	    if (child == 0) {       /* child exec's shell */

		/* avoid the race condition here.. wait for parent. */
		if (is_login) {
		    char buf[1];

		    /* block until we get something */

		    close(fds[1]);
		    read(fds[0], buf, 1);
		    close(fds[0]);

		    /* The race condition is only insofar as the user does
		       something like getlogin() and the utmp file has not
		       been written yet (by the parent). */
		}

		su_exec_shell(shell, uid, is_login, command, user);
		/* never reached */
	    }

	    prepare_for_job_control(child, command != NULL);
            if (is_login) {
		char buf_c = '\0';

		utmp_open_session(pamh, child);  /* new logname of terminal */

		close(fds[0]);
		write(fds[1], &buf_c, 1);
		close(fds[1]);
	    }
	}

	/* wait for child to terminate */

	status = wait_for_child(child);
	if (status != 0) {
	    D(("shell returned %d", status));
	}

	if (is_login) {
	    utmp_close_session(pamh, child);   /* retrieve original logname */
	}

    }while (0);                       /* abuse loop to avoid using goto... */

    if (retval != PAM_SUCCESS) {
	(void) fprintf(stderr, "su: %s\n", pam_strerror(pamh,retval));
	final_retval = PAM_ABORT;
    }else if (err_descr != NULL) {
	(void) fprintf(stderr, "su: %s: %s\n", place, err_descr);
	final_retval = PAM_ABORT;
    }else
	final_retval = PAM_SUCCESS;

    /* return terminal to local control */
    if (state >= SU_STATE_TERMINAL_REOWNED)
	restore_terminal_owner();

    /*
     * My impression is that PAM expects real uid to be restored.
     * Effective uid of the process is kept
     * unchanged: superuser.  (SAW)
     */
    if (state >= SU_STATE_PROCESS_UNKILLABLE)
	make_process_killable();

    if (state >= SU_STATE_CREDENTIALS_GOTTEN) {
	D(("setcred"));
	/* Delete the user's credentials. */
	retval = pam_setcred(pamh, PAM_DELETE_CRED);
	if (retval != PAM_SUCCESS) {
	    (void) fprintf(stderr, "WARNING: could not delete credentials\n\t%s\n"
		    , pam_strerror(pamh,retval));
	}
    }

    if (state >= SU_STATE_SESSION_OPENED) {
	D(("session"));
	D(("%p", pamh));

	/* close down */
	retval = pam_close_session(pamh,0);
	if (retval != PAM_SUCCESS)
	    (void) fprintf(stderr, "WARNING: could not close session\n\t%s\n"
                           , pam_strerror(pamh,retval));
    }

    /* clean up */
    D(("all done"));
    (void) pam_end(pamh, final_retval);
    pamh = NULL;

#ifdef HAVE_PWDB
    while ( pwdb_end() == PWDB_SUCCESS );
#endif

    /* reset the terminal */
    if (reset_terminal_modes() != 0 && !status)
	status = 1;

    exit(status);                 /* transparent exit */
}

/* -------------------------------------------- */
/* ------ some local (static) functions ------- */
/* -------------------------------------------- */

/* ------ terminal mode ----------------------- */

static struct termios stored_mode;        /* initial terminal mode settings */

/* should be called once at the beginning */
static void store_terminal_modes()
{
    if (isatty(STDIN_FILENO)) {
	is_terminal = 1;
	if (tcgetattr(STDIN_FILENO, &stored_mode) != 0) {
	    (void) fprintf(stderr, "su: couldn't copy terminal mode");
	    exit(1);
	}
    } else if (getuid()) {
	(void) fprintf(stderr, "su: must be run from a terminal\n");
	exit(1);
    } else {
	D(("process was run by superuser; assume you know what you're doing"));
	is_terminal = 0;
    }
}

/*
 * Returns:
 *   0     ok
 * !=0     error
 */
static int reset_terminal_modes()
{
    if (is_terminal && tcsetattr(STDIN_FILENO, TCSAFLUSH, &stored_mode) != 0) {
	(void) fprintf(stderr, "su: cannot reset terminal mode: %s\n"
		       , strerror(errno));
	return 1;
    }else
	return 0;
}

/* ------ unexpected signals ------------------ */

static struct sigaction old_int_act, old_quit_act, old_tstp_act;

static void disable_terminal_signals()
{
    /* 
     * Protect the process from dangerous terminal signals.
     * The protection is implemented via sigaction() because
     * the signals are sent regardless of the process' uid.
     */
    struct sigaction act;

    act.sa_handler = SIG_IGN;  /* ignore the signal */
    sigemptyset(&act.sa_mask); /* no signal blocking on handler
				  call needed */
    act.sa_flags = SA_RESTART; /* do not reset after first signal
				  arriving, restart interrupted
				  system calls if possible */
    sigaction(SIGINT, &act, &old_int_act);
    sigaction(SIGQUIT, &act, &old_quit_act);
    /*
     * Ignore SIGTSTP signals. Why? attacker could otherwise stop
     * a process and a. kill it, or b. wait for the system to
     * shutdown - either way, nothing appears in syslogs.
     */
    sigaction(SIGTSTP, &act, &old_tstp_act);
}

static void enable_terminal_signals()
{
    sigaction(SIGINT, &old_int_act, NULL);
    sigaction(SIGQUIT, &old_quit_act, NULL);
    sigaction(SIGTSTP, &old_tstp_act, NULL);
}

/* ------ terminal ownership ------------------ */

static uid_t terminal_uid = (uid_t) -1;

/*
 * Change the ownership of STDIN if needed.
 * Returns:
 *   0     ok,
 *  -1     fatal error (continue of the work is impossible),
 *   1     non-fatal error.
 * In the case of an error "err_descr" is set to the error message
 * and "callname" to the name of the failed call.
 */
static int change_terminal_owner(uid_t uid, int is_login, int command_present
	, const char **callname, const char **err_descr)
{
    /* determine who owns the terminal line */
    if (is_terminal && is_login) {
	struct stat stat_buf;

	if (fstat(STDIN_FILENO,&stat_buf) != 0) {
            *callname = "fstat to STDIN";
	    *err_descr = strerror(errno);
	    return -1;
	}
	if(fchown(STDIN_FILENO, uid, -1) != 0) {
	    *callname = "fchown to STDIN";
            *err_descr = strerror(errno);
	    return 1;
	}
	terminal_uid = stat_buf.st_uid;
    }
    return 0;
}

static void restore_terminal_owner()
{
    if (terminal_uid != (uid_t) -1) {
        if(fchown(STDIN_FILENO, terminal_uid, -1) != 0) {
            openlog("su", LOG_CONS | LOG_PERROR | LOG_PID, LOG_AUTHPRIV);
	    syslog(LOG_ALERT
		    , "Terminal owner hasn\'t been restored: %s"
		    , strerror(errno));
	    closelog();
        }
        terminal_uid = (uid_t) -1;
    }
}

/* ------ Process Death Control (tm) ---------- */

static uid_t invoked_uid;

/*
 * Make the process unkillable by an user invoked it.
 * Returns:
 *   0     ok,
 *  -1     fatal error (continue of the work is impossible),
 *   1     non-fatal error.
 * In the case of an error "err_descr" is set to the error message
 * and "callname" to the name of the failed call.
 */
static int make_process_unkillable(const char **callname
        , const char **err_descr)
{
    invoked_uid = getuid();
    if(setuid(geteuid()) != 0) {
        *callname = "setuid";
	*err_descr = strerror(errno);
	return -1;
    }else
	return 0;
}

static void make_process_killable()
{
    setreuid(invoked_uid, -1);
}

/* ------ abnormal termination ---------------- */

static void exit_now(int exit_code, const char *format, ...)
{
    va_list args;

    va_start(args,format);
    vfprintf(stderr, format, args);
    va_end(args);

    if (pamh != NULL)
	pam_end(pamh, exit_code ? PAM_ABORT:PAM_SUCCESS);

#ifdef HAVE_PWDB
    while (pwdb_end() == PWDB_SUCCESS);                       /* clean up */
#endif /* HAVE_PWDB */

    /* USER's shell may have completely broken terminal settings
       restore the sane(?) initial conditions */
    reset_terminal_modes();

    exit(exit_code);
}

static void exit_child_now(int exit_code, const char *format, ...)
{
    va_list args;

    va_start(args,format);
    vfprintf(stderr, format, args);
    va_end(args);

    if (pamh != NULL)
	pam_end(pamh, (exit_code ? PAM_ABORT:PAM_SUCCESS)
#ifdef PAM_DATA_QUIET
		     | PAM_DATA_QUIET
#endif
	);

#ifdef HAVE_PWDB
    while (pwdb_end() == PWDB_SUCCESS);                       /* clean up */
#endif /* HAVE_PWDB */

    exit(exit_code);
}

/* ------ command line parser ----------------- */

static void usage()
{
    (void) fprintf(stderr,"usage: su [-] [-c \"command\"] [username]\n");
    exit(1);
}

static void parse_command_line(int argc, char *argv[]
	, int *is_login, const char **user, const char **command)
{
    int username_present, command_present;

    *is_login = 0;
    *user = NULL;
    *command = NULL;
    username_present = command_present = 0;

    while ( --argc > 0 ) {
	const char *token;

	token = *++argv;
	if (*token == '-') {
	    switch (*++token) {
	    case '\0':             /* su as a login shell for the user */
		if (*is_login)
		    usage();
		*is_login = 1;
		break;
	    case 'c':
		if (command_present) {
		    usage();
		} else {               /* indicate we are running commands */
		    if (*++token != '\0') {
			command_present = 1;
			*command = token;
		    } else if (--argc > 0) {
			command_present = 1;
			*command = *++argv;
		    } else
			usage();
		}
		break;
	    default:
		usage();
	    }
	} else {                       /* must be username */
	    if (username_present)
		usage();
	    username_present = 1;
	    *user = *argv;
	}
    }

    if (!username_present) {           /* default user is superuser */
	const struct passwd *pw;

	pw = getpwuid(ROOT_UID);
	if (pw == NULL) {                              /* No ROOT_UID!? */
	    exit_now(1,"su: no access to superuser identity!? (%d)\n"
		     , ROOT_UID);
	}
	*user = x_strdup(pw->pw_name);
    }
}

/* ------ PAM setup --------------------------- */

static struct pam_conv conv = {
    misc_conv,                   /* defined in <pam_misc/libmisc.h> */
    NULL
};

static void do_pam_init(const char *user, int is_login)
{
    int retval;

    retval = pam_start("su", user, &conv, &pamh);
    if (retval != PAM_SUCCESS) {
	/*
	 * From my point of view failing of pam_start() means that
	 * pamh isn't a valid handler. Without a handler
	 * we couldn't call pam_strerror :-(   1998/03/29 (SAW)
	 */
	(void) fprintf(stderr, "su: pam_start failed with code %d\n", retval);
	exit(1);
    }

    /*
     * Fill in some blanks
     */

    retval = make_environment(pamh, !is_login);
    D(("made_environment returned: %s", pam_strerror(pamh,retval)));

    if (retval == PAM_SUCCESS && is_terminal) {
	const char *terminal = ttyname(STDIN_FILENO);
	if (terminal) {
	    retval = pam_set_item(pamh, PAM_TTY, (const void *)terminal);
	} else {
	    retval = PAM_PERM_DENIED;                /* how did we get here? */
	}
	terminal = NULL;
    }

    if (retval == PAM_SUCCESS && is_terminal) {
	const char *ruser = getlogin();      /* Who is running this program? */
	if (ruser) {
	    retval = pam_set_item(pamh, PAM_RUSER, (const void *)ruser);
	} else {
	    retval = PAM_PERM_DENIED;             /* must be known to system */
	}
	ruser = NULL;
    }

    if (retval == PAM_SUCCESS) {
	retval = pam_set_item(pamh, PAM_RHOST, (const void *)"localhost");
    }

    if (retval != PAM_SUCCESS) {
	exit_now(1, "su: problem establishing environment\n");
    }

#ifdef HAVE_PAM_FAIL_DELAY
    /* have to pause on failure. At least this long (doubles..) */
    retval = pam_fail_delay(pamh, SU_FAIL_DELAY);
    if (retval != PAM_SUCCESS) {
	exit_now(1, "su: problem initializing failure delay\n");
    }
#endif /* HAVE_PAM_FAIL_DELAY */
}

/* ------ shell invoker ----------------------- */

static void su_exec_shell(const char *shell, uid_t uid, int is_login
			  , const char *command, const char *user)
{
    char * const * shell_args;
    char * const * shell_env;
    const char *pw_dir;
    int retval;

    /*
     * Now, find the home directory for the user
     */

    pw_dir = pam_getenv(pamh, "HOME");
    if ( !pw_dir || pw_dir[0] == '\0' ) {
	/* Not set so far, so we get it now. */
	struct passwd *pwd;

	pwd = getpwnam(user);
	if (pwd != NULL && pwd->pw_name != NULL) {
	    pw_dir = x_strdup(pwd->pw_name);
	}

	/* Last resort, take default directory.. */
	if ( !pw_dir || pw_dir[0] == '\0') {
	    (void) fprintf(stderr, "setting home directory for %s to %s\n"
                           , user, DEFAULT_HOME);
	    pw_dir = DEFAULT_HOME;
	}
    }

    /*
     * We may wish to change the current directory.
     */

    if (is_login && chdir(pw_dir)) {
	exit_child_now(1, "%s not available; exiting\n", pw_dir);
    }

    /*
     * If it is a login session, we should set the environment
     * accordingly.
     */

    if (is_login
	&& pam_misc_setenv(pamh, "HOME", pw_dir, 0) != PAM_SUCCESS) {
	D(("failed to set $HOME"));
	(void) fprintf(stderr
                       , "Warning: unable to set HOME environment variable\n");
    }

    /*
     * Break up the shell command into a command and arguments
     */

    shell_args = build_shell_args(shell, is_login, command);
    if (shell_args == NULL) {
	exit_child_now(1, "su: could not identify appropriate shell\n");
    }

    /*
     * and now copy the environment for non-PAM use
     */

    shell_env = pam_getenvlist(pamh);
    if (shell_env == NULL) {
	exit_child_now(1, "su: corrupt environment\n");
    }

    /*
     * close PAM (quietly = this is a forked process so ticket files
     * should *not* be deleted logs should not be written - the parent
     * will take care of this)
     */

    D(("pam_end"));
    retval = pam_end(pamh, PAM_SUCCESS
#ifdef PAM_DATA_QUIET
		     | PAM_DATA_QUIET
#endif
	);
    pamh = NULL;
    user = NULL;                            /* user's name not valid now */
    if (retval != PAM_SUCCESS) {
	exit_child_now(1, "su: failed to release authenticator\n");
    }

#ifdef HAVE_PWDB
    while ( pwdb_end() == PWDB_SUCCESS );            /* forget all */
#endif

    /* assume user's identity */
    if (setuid(uid) != 0) {
	exit_child_now(1, "su: cannot assume uid\n");
    }

    /*
     * Restore a signal status: information if the signal is ingored
     * is inherited accross exec() call.  (SAW)
     */
    enable_terminal_signals();

    execve(shell_args[0], shell_args+1, shell_env);
    exit_child_now(1, "su: exec failed\n");
}
