/*
 * Portions of this file taken from the Linux kernel,
 * Copyright 1991-2009 Linus Torvalds and contributors
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
*/

#include <stdio.h>
#include <string.h>
#include "cpuid.h"

struct cpu_dev * cpu_devs[X86_VENDOR_NUM] = {};

/*
* CPUID functions returning a single datum
*/

/* Probe for the CPUID instruction */
static int have_cpuid_p(void)
{
        return cpu_has_eflag(X86_EFLAGS_ID);
}

static struct cpu_dev amd_cpu_dev = {
        .c_vendor       = "AMD",
        .c_ident        = { "AuthenticAMD" }
};

static struct cpu_dev intel_cpu_dev = {
        .c_vendor       = "Intel",
        .c_ident        = { "GenuineIntel" }
};

static struct cpu_dev cyrix_cpu_dev = {
        .c_vendor       = "Cyrix",
        .c_ident        = { "CyrixInstead" }
};

static struct cpu_dev umc_cpu_dev = {
        .c_vendor       = "UMC",
	.c_ident        = { "UMC UMC UMC" }

};

static struct cpu_dev nexgen_cpu_dev = {
        .c_vendor       = "Nexgen",
        .c_ident        = { "NexGenDriven" }
};

static struct cpu_dev centaur_cpu_dev = {
        .c_vendor       = "Centaur",
        .c_ident        = { "CentaurHauls" }
};

static struct cpu_dev rise_cpu_dev = {
        .c_vendor       = "Rise",
        .c_ident        = { "RiseRiseRise" }
};

static struct cpu_dev transmeta_cpu_dev = {
        .c_vendor       = "Transmeta",
        .c_ident        = { "GenuineTMx86", "TransmetaCPU" }
};

void init_cpu_devs(void)
{
	cpu_devs[X86_VENDOR_INTEL] = &intel_cpu_dev;
	cpu_devs[X86_VENDOR_CYRIX] = &cyrix_cpu_dev;
	cpu_devs[X86_VENDOR_AMD] = &amd_cpu_dev;
	cpu_devs[X86_VENDOR_UMC] = &umc_cpu_dev;
	cpu_devs[X86_VENDOR_NEXGEN] = &nexgen_cpu_dev;
	cpu_devs[X86_VENDOR_CENTAUR] = &centaur_cpu_dev;
	cpu_devs[X86_VENDOR_RISE] = &rise_cpu_dev;
	cpu_devs[X86_VENDOR_TRANSMETA] = &transmeta_cpu_dev;
}

void get_cpu_vendor(struct cpuinfo_x86 *c)
{
        char *v = c->x86_vendor_id;
        int i;
	init_cpu_devs();
        for (i = 0; i < X86_VENDOR_NUM; i++) {
                if (cpu_devs[i]) {
                        if (!strcmp(v,cpu_devs[i]->c_ident[0]) ||
                            (cpu_devs[i]->c_ident[1] &&
                             !strcmp(v,cpu_devs[i]->c_ident[1]))) {
                                c->x86_vendor = i;
                                return;
                        }
                }
        }

        c->x86_vendor = X86_VENDOR_UNKNOWN;
}

int get_model_name(struct cpuinfo_x86 *c)
{
        unsigned int *v;
        char *p, *q;

        if (cpuid_eax(0x80000000) < 0x80000004)
                return 0;

        v = (unsigned int *) c->x86_model_id;
        cpuid(0x80000002, &v[0], &v[1], &v[2], &v[3]);
        cpuid(0x80000003, &v[4], &v[5], &v[6], &v[7]);
        cpuid(0x80000004, &v[8], &v[9], &v[10], &v[11]);
        c->x86_model_id[48] = 0;

        /* Intel chips right-justify this string for some dumb reason;
           undo that brain damage */
        p = q = &c->x86_model_id[0];
        while ( *p == ' ' )
             p++;
        if ( p != q ) {
             while ( *p )
                  *q++ = *p++;
             while ( q <= &c->x86_model_id[48] )
                  *q++ = '\0';  /* Zero-pad the rest */
        }

        return 1;
}

void generic_identify(struct cpuinfo_x86 *c)
{
	uint32_t tfms, xlvl;
	unsigned int ebx;

	/* Get vendor name */
	cpuid(0x00000000,
	      (uint32_t *)&c->cpuid_level,
              (uint32_t *)&c->x86_vendor_id[0],
              (uint32_t *)&c->x86_vendor_id[8],
              (uint32_t *)&c->x86_vendor_id[4]);

        get_cpu_vendor(c);
        /* Intel-defined flags: level 0x00000001 */
        if ( c->cpuid_level >= 0x00000001 ) {
		uint32_t capability, excap;
                cpuid(0x00000001, &tfms, &ebx, &excap, &capability);
                c->x86_capability[0] = capability;
                c->x86_capability[4] = excap;
                c->x86 = (tfms >> 8) & 15;
                c->x86_model = (tfms >> 4) & 15;
                if (c->x86 == 0xf)
                        c->x86 += (tfms >> 20) & 0xff;
                if (c->x86 >= 0x6)
                        c->x86_model += ((tfms >> 16) & 0xF) << 4;
                c->x86_mask = tfms & 15;
                if (cpu_has(c, X86_FEATURE_CLFLSH))
                        c->x86_clflush_size = ((ebx >> 8) & 0xff) * 8;
              } else {
                      /* Have CPUID level 0 only - unheard of */
                      c->x86 = 4;
	}

        /* AMD-defined flags: level 0x80000001 */
        xlvl = cpuid_eax(0x80000000);
        if ( (xlvl & 0xffff0000) == 0x80000000 ) {
               if ( xlvl >= 0x80000001 ) {
                     c->x86_capability[1] = cpuid_edx(0x80000001);
                     c->x86_capability[6] = cpuid_ecx(0x80000001);
               }
               if ( xlvl >= 0x80000004 )
                     get_model_name(c); /* Default name */
       }
}

/*
 * Checksum an MP configuration block.
 */

static int mpf_checksum(unsigned char *mp, int len)
{
        int sum = 0;

        while (len--)
                sum += *mp++;

        return sum & 0xFF;
}

static int smp_scan_config (unsigned long base, unsigned long length)
{
        unsigned long *bp = (unsigned long *)base;
        struct intel_mp_floating *mpf;

//        printf("Scan SMP from %p for %ld bytes.\n", bp,length);
        if (sizeof(*mpf) != 16) {
                printf("Error: MPF size\n");
		return 0;
	}

        while (length > 0) {
                mpf = (struct intel_mp_floating *)bp;
                if ((*bp == SMP_MAGIC_IDENT) &&
                        (mpf->mpf_length == 1) &&
                        !mpf_checksum((unsigned char *)bp, 16) &&
                        ((mpf->mpf_specification == 1)
                                || (mpf->mpf_specification == 4)) ) {
                        return 1;
                }
                bp += 4;
                length -= 16;
        }
        return 0;
}

int find_smp_config (void)
{
//        unsigned int address;

        /*
         * FIXME: Linux assumes you have 640K of base ram..
         * this continues the error...
         *
         * 1) Scan the bottom 1K for a signature
         * 2) Scan the top 1K of base RAM
         * 3) Scan the 64K of bios
         */
        if (smp_scan_config(0x0,0x400) ||
                smp_scan_config(639*0x400,0x400) ||
                        smp_scan_config(0xF0000,0x10000))
                return 1;
        /*
         * If it is an SMP machine we should know now, unless the
         * configuration is in an EISA/MCA bus machine with an
         * extended bios data area.
         *
         * there is a real-mode segmented pointer pointing to the
         * 4K EBDA area at 0x40E, calculate and scan it here.
         *
         * NOTE! There are Linux loaders that will corrupt the EBDA
         * area, and as such this kind of SMP config may be less
         * trustworthy, simply because the SMP table may have been
         * stomped on during early boot. These loaders are buggy and
         * should be fixed.
         *
         * MP1.4 SPEC                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                