; -*- fundamental -*- (asm-mode sucks)
; $Id: isolinux.asm,v 1.31 2001/08/04 06:12:54 hpa Exp $
; ****************************************************************************
;
;  isolinux.asm
;
;  A program to boot Linux kernels off a CD-ROM using the El Torito
;  boot standard in "no emulation" mode, making the entire filesystem
;  available.  It is based on the SYSLINUX boot loader for MS-DOS
;  floppies.
;
;   Copyright (C) 1994-2001  H. Peter Anvin
;
;  This program is free software; you can redistribute it and/or modify
;  it under the terms of the GNU General Public License as published by
;  the Free Software Foundation, Inc., 675 Mass Ave, Cambridge MA 02139,
;  USA; either version 2 of the License, or (at your option) any later
;  version; incorporated herein by reference.
; 
; ****************************************************************************

; Note: The Makefile builds one version with DEBUG_MESSAGES automatically.
; %define DEBUG_TRACERS			; Uncomment to get debugging tracers
; %define DEBUG_MESSAGES		; Uncomment to get debugging messages

%ifdef DEBUG_TRACERS

%macro TRACER	1
	call debug_tracer
	db %1
%endmacro

%else	; DEBUG_TRACERS

%macro	TRACER	1
%endmacro

%endif	; DEBUG_TRACERS

;
; Some semi-configurable constants... change on your own risk.  Most are imposed
; by the kernel.
;
max_cmd_len	equ 255			; Must be odd; 255 is the kernel limit
FILENAME_MAX_LG2 equ 8			; log2(Max filename size Including final null)
FILENAME_MAX	equ (1 << FILENAME_MAX_LG2)
HIGHMEM_MAX	equ 038000000h		; Highest address for an initrd
HIGHMEM_SLOP	equ 128*1024		; Avoid this much memory near the top
DEFAULT_BAUD	equ 9600		; Default baud rate for serial port
BAUD_DIVISOR	equ 115200		; Serial port parameter
MAX_OPEN_LG2	equ 6			; log2(Max number of open files)
MAX_OPEN	equ (1 << MAX_OPEN_LG2)
SECTORSIZE_LG2	equ 11			; 2048 bytes/sector (El Torito requirement)
SECTORSIZE	equ (1 << SECTORSIZE_LG2)

;
; Should be updated with every release to avoid bootsector/SYS file mismatch
;
%define	version_str	VERSION		; Must be 4 characters long!
%define date		DATE_STR	; Defined from the Makefile
%define	year		'2001'
;
; Debgging stuff
;
; %define debug 1			; Uncomment to enable debugging
;
; ID for SYSLINUX (reported to kernel)
;
syslinux_id	equ 033h		; SYSLINUX (3) 3 = ISOLINUX

;
; Segments used by Linux
;
real_mode_seg	equ 5000h
fake_setup_seg	equ real_mode_seg+020h

		struc real_mode_seg_t
		resb 20h-($-$$)		; org 20h
kern_cmd_magic	resw 1			; 0020 Magic # for command line
kern_cmd_offset resw 1			; 0022 Offset for kernel command line
		resb 497-($-$$)		; org 497d
bs_setupsecs	resb 1			; 01F1 Sectors for setup code (0 -> 4)
bs_rootflags	resw 1			; 01F2 Root readonly flag
bs_syssize	resw 1			; 01F4
bs_swapdev	resw 1			; 01F6 Swap device (obsolete)
bs_ramsize	resw 1			; 01F8 Ramdisk flags, formerly ramdisk size
bs_vidmode	resw 1			; 01FA Video mode
bs_rootdev	resw 1			; 01FC Root device
bs_bootsign	resw 1			; 01FE Boot sector signature (0AA55h)
su_jump		resb 1			; 0200 0EBh
su_jump2	resb 1			; 0201 Size of following header
su_header	resd 1			; 0202 New setup code: header
su_version	resw 1			; 0206 See linux/arch/i386/boot/setup.S
su_switch	resw 1			; 0208
su_setupseg	resw 1			; 020A
su_startsys	resw 1			; 020C
su_kver		resw 1			; 020E Kernel version pointer
su_loader	resb 1			; 0210 Loader ID
su_loadflags	resb 1			; 0211 Load high flag
su_movesize	resw 1			; 0212
su_code32start	resd 1			; 0214 Start of code loaded high
su_ramdiskat	resd 1			; 0218 Start of initial ramdisk
su_ramdisklen	equ $			; Length of initial ramdisk
su_ramdisklen1	resw 1			; 021C
su_ramdisklen2	resw 1			; 021E
su_bsklugeoffs	resw 1			; 0220
su_bsklugeseg	resw 1			; 0222
su_heapend	resw 1			; 0224
su_pad1		resw 1			; 0226
su_cmd_line_ptr	resd 1			; 0228
		resb (9000h-12)-($-$$)	; Were bootsect.S puts it...
linux_stack	equ $			; 8FF4
linux_fdctab	equ $
		resb 9000h-($-$$)
cmd_line_here	equ $			; 9000 Should be out of the way
		endstruc

;
; Kernel command line signature
;
CMD_MAGIC	equ 0A33Fh		; Command line magic

;
; Magic number of su_header field
;
HEADER_ID       equ 'HdrS'		; HdrS (in littleendian hex)

;
; Flags for the su_loadflags field
;
LOAD_HIGH	equ 01h			; Large kernel, load high
CAN_USE_HEAP    equ 80h                 ; Boot loader reports heap size

;
; The following structure is used for "virtual kernels"; i.e. LILO-style
; option labels.  The options we permit here are `kernel' and `append
; Since there is no room in the bottom 64K for all of these, we
; stick them at vk_seg:0000 and copy them down before we need them.
;
; Note: this structure can be added to, but it must 
;
%define vk_power	6		; log2(max number of vkernels)
%define	max_vk		(1 << vk_power)	; Maximum number of vkernels
%define vk_shift	(16-vk_power)	; Number of bits to shift
%define vk_size		(1 << vk_shift)	; Size of a vkernel buffer

		struc vkernel
vk_vname:	resb FILENAME_MAX	; Virtual name **MUST BE FIRST!**
vk_rname:	resb FILENAME_MAX	; Real name
vk_appendlen:	resw 1
		alignb 4
vk_append:	resb max_cmd_len+1	; Command line
		alignb 4
vk_end:		equ $			; Should be <= vk_size
		endstruc

%if (vk_end > vk_size) || (vk_size*max_vk > 65536)
%error "Too many vkernels defined, reduce vk_power"
%endif

;
; Segment assignments in the bottom 640K
; 0000h - main code/data segment (and BIOS segment)
; 5000h - real_mode_seg
;
vk_seg          equ 4000h		; Virtual kernels
xfer_buf_seg	equ 3000h		; Bounce buffer for I/O to high mem
comboot_seg	equ 2000h		; COMBOOT image loading zone

;
; File structure.  This holds the information for each currently open file.
;
		struc open_file_t
file_sector	resd 1			; Sector pointer (0 = structure free)
file_left	resd 1			; Number of sectors left
		endstruc

%if (open_file_t_size & (open_file_t_size-1))
%error "open_file_t is not a power of 2"
%endif

;
; For our convenience: define macros for jump-over-unconditinal jumps
;
%macro	jmpz	1
	jnz %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpnz	1
	jz %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpe	1
	jne %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpne	1
	je %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpc	1
	jnc %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpnc	1
	jc %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpb	1
	jnb %%skip
	jmp %1
%%skip:
%endmacro

%macro	jmpnb	1
	jb %%skip
	jmp %1
%%skip:
%endmacro

;
; Macros similar to res[bwd], but which works in the code segment (after
; section .text)
;
%macro	zb	1
	times %1 db 0
%endmacro

%macro	zw	1
	times %1 dw 0
%endmacro

%macro	zd	1
	times %1 dd 0
%endmacro

; ---------------------------------------------------------------------------
;   BEGIN THE BIOS/CODE/DATA SEGMENT
; ---------------------------------------------------------------------------

		absolute 0400h
serial_base	resw 4			; Base addresses for 4 serial ports
		absolute 0413h
BIOS_fbm	resw 1			; Free Base Memory (kilobytes)
		absolute 046Ch
BIOS_timer	resw 1			; Timer ticks
		absolute 0472h
BIOS_magic	resw 1			; BIOS reset magic
                absolute 0484h
BIOS_vidrows    resb 1			; Number of screen rows

;
; Memory below this point is reserved for the BIOS and the MBR
;
 		absolute 1000h
trackbuf	resb 8192		; Track buffer goes here
trackbufsize	equ $-trackbuf
;		trackbuf ends at 3000h


;
; Constants for the xfer_buf_seg
;
; The xfer_buf_seg is also used to store message file buffers.  We
; need two trackbuffers (text and graphics), plus a work buffer
; for the graphics decompressor.
;
xbs_textbuf	equ 0			; Also hard-coded, do not change
xbs_vgabuf	equ trackbufsize
xbs_vgatmpbuf	equ 2*trackbufsize

		struc dir_t
dir_lba		resd 1			; Directory start (LBA)
dir_len		resd 1			; Length in bytes
dir_clust	resd 1			; Length in clusters
		endstruc

                absolute 5000h          ; Here we keep our BSS stuff
VKernelBuf:	resb vk_size		; "Current" vkernel
		alignb 4
AppendBuf       resb max_cmd_len+1	; append=
KbdMap		resb 256		; Keyboard map
FKeyName	resb 10*FILENAME_MAX	; File names for F-key help
NumBuf		resb 15			; Buffer to load number
NumBufEnd	resb 1			; Last byte in NumBuf
ISOFileName	resb 64			; ISO filename canonicalization buffer
ISOFileNameEnd	equ $
		alignb 32
KernelName      resb FILENAME_MAX       ; Mangled name for kernel
KernelCName     resb FILENAME_MAX	; Unmangled kernel name
InitRDCName     resb FILENAME_MAX       ; Unmangled initrd name
MNameBuf	resb FILENAME_MAX
InitRD		resb FILENAME_MAX
PartInfo	resb 16			; Partition table entry
E820Buf		resd 5			; INT 15:E820 data buffer
InitRDat	resd 1			; Load address (linear) for initrd
HiLoadAddr      resd 1			; Address pointer for high load loop
HighMemSize	resd 1			; End of memory pointer (bytes)
KernelSize	resd 1			; Size of kernel (bytes)
RootDir		resb dir_t_size		; Root directory
CurDir		resb dir_t_size		; Current directory
SavedSSSP	resd 1			; Our SS:SP while running a COMBOOT image
InitStack	resd 1			; Initial stack pointer (SS:SP)
FirstSecSum	resd 1			; Checksum of bytes 64-2048
ImageDwords	resd 1			; isolinux.bin size, dwords
FBytes		equ $			; Used by open/getc
FBytes1		resw 1
FBytes2		resw 1
KernelClust	resw 1			; Kernel size in clusters
InitRDClust	resw 1			; Ramdisk size in clusters
FClust		resw 1			; Number of clusters in open/getc file
FNextClust	resw 1			; Pointer to next cluster in d:o
FPtr		resw 1			; Pointer to next char in buffer
CmdOptPtr       resw 1			; Pointer to first option on cmd line
KernelCNameLen  resw 1			; Length of unmangled kernel name
InitRDCNameLen  resw 1			; Length of unmangled initrd name
NextCharJump    resw 1			; Routine to interpret next print char
SetupSecs	resw 1			; Number of setup sectors
A20Test		resw 1			; Counter for testing status of A20
CmdLineLen	resw 1			; Length of command line including null
GraphXSize	resw 1			; Width of splash screen file
VGAPos		resw 1			; Pointer into VGA memory
VGACluster	resw 1			; Cluster pointer for VGA image file
VGAFilePtr	resw 1			; Pointer into VGAFileBuf
ConfigFile	resw 1			; Socket for config file
PktTimeout	resw 1			; Timeout for current packet
KernelExtPtr	resw 1			; During search, final null pointer
LocalBootType	resw 1			; Local boot return code
ImageSectors	resw 1			; isolinux.bin size, sectors
TextAttrBX      equ $
TextAttribute   resb 1			; Text attribute for message file
TextPage        resb 1			; Active display page
CursorDX        equ $
CursorCol       resb 1			; Cursor column for message file
CursorRow       resb 1			; Cursor row for message file
ScreenSize      equ $
VidCols         resb 1			; Columns on screen-1
VidRows         resb 1			; Rows on screen-1
RetryCount      resb 1			; Used for disk access retries
KbdFlags	resb 1			; Check for keyboard escapes
LoadFlags	resb 1			; Loadflags from kernel
A20Tries	resb 1			; Times until giving up on A20
FuncFlag	resb 1			; == 1 if <Ctrl-F> pressed
DisplayMask	resb 1			; Display modes mask
ISOFlags	resb 1			; Flags for ISO directory search
DiskError	resb 1			; Error code for disk I/O
DriveNo		resb 1			; CD-ROM BIOS drive number
TextColorReg	resb 17			; VGA color registers for text mode
VGAFileBuf	resb FILENAME_MAX	; Unmangled VGA image name
VGAFileBufEnd	equ $
VGAFileMBuf	resb FILENAME_MAX	; Mangled VGA image name

		alignb open_file_t_size
Files		resb MAX_OPEN*open_file_t_size

		section .text
                org 7C00h
;;
;; Primary entry point.  Because BIOSes are buggy, we only load the first
;; CD-ROM sector (2K) of the file, so the number one priority is actually
;; loading the rest.
;;
bootsec		equ $
_start:		; Far jump makes sure we canonicalize the address
		cli
		jmp 0:_start1
		times 8-($-$$) nop		; Pad to file offset 8

		; This table gets filled in by mkisofs using the
		; -boot-info-table option
bi_pvd:		dd 0xdeadbeef			; LBA of primary volume descriptor
bi_file:	dd 0xdeadbeef			; LBA of boot file
bi_length:	dd 0xdeadbeef			; Length of boot file
bi_csum:	dd 0xdeadbeef			; Checksum of boot file
bi_reserved:	times 10 dd 0xdeadbeef		; Reserved

_start1:	mov [cs:InitStack],sp		; Save initial stack pointer
		mov ax,ss
		mov [cs:InitStack+2],ax
		xor ax,ax
		mov ss,ax
		mov sp,_start			; Set up stack
		mov ds,ax
		mov es,ax
		mov fs,ax
		mov gs,ax
		sti

		cld
		; Show signs of life
		mov si,isolinux_banner
		call writestr
%ifdef DEBUG_MESSAGES
		mov si,copyright_str
		call writestr
%endif

		;
		; Before modifying any memory, get the checksum of bytes
		; 64-2048
		;
initial_csum:	xor edi,edi
		mov si,_start1
		mov cx,(SECTORSIZE-64) >> 2
.loop:		lodsd
		add edi,eax
		loop .loop
		mov [FirstSecSum],edi

		; Set up boot file sizes
		mov eax,[bi_length]
		sub eax,SECTORSIZE-3
		shr eax,2			; bytes->dwords
		mov [ImageDwords],eax		; boot file dwords
		add eax,(2047 >> 2)
		shr eax,9			; dwords->sectors
		mov [ImageSectors],ax		; boot file sectors

		mov [DriveNo],dl
%ifdef DEBUG_MESSAGES
		mov si,startup_msg
		call writemsg
		mov al,dl
		call writehex2
		call crlf
%endif

		; Now figure out what we're actually doing
		; Note: use passed-in DL value rather than 7Fh because
		; at least some BIOSes will get the wrong value otherwise
		mov ax,4B01h			; Get disk emulation status
		mov dl,[DriveNo]
		mov si,spec_packet
		int 13h
		jc near spec_query_failed	; Shouldn't happen (BIOS bug)
		mov dl,[DriveNo]
		cmp [sp_drive],dl		; Should contain the drive number
		jne near spec_query_failed

%ifdef DEBUG_MESSAGES
		mov si,spec_ok_msg
		call writemsg
		mov al,byte [sp_drive]
		call writehex2
		call crlf
%endif

found_drive:
		; Get drive information
		mov ah,48h
		mov dl,[DriveNo]
		mov si,drive_params
		int 13h
		jnc params_ok

		mov si,nosecsize_msg
		call writemsg

params_ok:
		; Check for the sector size (should be 2048, but
		; some BIOSes apparently think we're 512-byte media)
		;
		; FIX: We need to check what the proper behaviour
		; is for getlinsec when the BIOS thinks the sector
		; size is 512!!!  For that, we need such a BIOS, though...
%ifdef DEBUG_MESSAGES
		mov si,secsize_msg
		call writemsg
		mov ax,[dp_secsize]
		call writehex4
		call crlf
%endif

load_image:
		; Some BIOSes apparently have limitations on the size 
		; that may be loaded (despite the El Torito spec being very
		; clear on the fact that it must all be loaded.)  Therefore,
		; we load it ourselves, and *bleep* the BIOS.

		mov eax,[bi_file]		; Address of code to load
		inc eax				; Don't reload bootstrap code
%ifdef DEBUG_MESSAGES
		mov si,offset_msg
		call writemsg
		call writehex8
		call crlf
%endif

		; Just in case some BIOSes have problems with
		; segment wraparound, use the normalized address
		mov bx,((7C00h+2048) >> 4)
		mov es,bx
		xor bx,bx
		mov bp,[ImageSectors]
%ifdef DEBUG_MESSAGES
		push ax
		mov si,size_msg
		call writemsg
		mov ax,bp
		call writehex4
		call crlf
		pop ax
%endif
		call getlinsec

		push ds
		pop es

%ifdef DEBUG_MESSAGES
		mov si,loaded_msg
		call writemsg
%endif

		; Verify the checksum on the loaded image.
verify_image:
		mov si,7C00h+2048
		mov bx,es
		mov ecx,[ImageDwords]
		mov edi,[FirstSecSum]		; First sector checksum
.loop		es lodsd
		add edi,eax
		dec ecx
		jz .done
		and si,si
		jnz .loop
		; SI wrapped around, advance ES
		add bx,1000h
		mov es,bx
		jmp short .loop
.done:		mov ax,ds
		mov es,ax
		cmp [bi_csum],edi
		je integrity_ok

		mov si,checkerr_msg
		call writemsg
		jmp kaboom

integrity_ok:
%ifdef DEBUG_MESSAGES
		mov si,allread_msg
		call writemsg
%endif
		jmp all_read			; Jump to main code

		; INT 13h, AX=4B01h, DL=<passed in value> failed.
		; Try to scan the entire 80h-FFh from the end.
spec_query_failed:
		mov si,spec_err_msg
		call writemsg

		mov dl,0FFh
.test_loop:	pusha
		mov ax,4B01h
		mov si,spec_packet
		mov byte [si],13		; Size of buffer
		int 13h
		popa
		jc .still_broken

		mov si,maybe_msg
		call writemsg
		mov al,dl
		call writehex2
		call crlf

		cmp byte [sp_drive],dl
		jne .maybe_broken

		; Okay, good enough...
		mov si,alright_msg
		call writemsg
		mov [DriveNo],dl
.found_drive:	jmp found_drive

		; Award BIOS 4.51 apparently passes garbage in sp_drive,
		; but if this was the drive number originally passed in
		; DL then consider it "good enough"
.maybe_broken:
		cmp byte [DriveNo],dl
		je .found_drive

.still_broken:	dec dx
		cmp dl, 80h
		jnb .test_loop

fatal_error:
		mov si,nothing_msg
		call writemsg

.norge:		jmp short .norge

		; Information message (DS:SI) output
		; Prefix with "isolinux: "
		;
writemsg:	push ax
		push si
		mov si,isolinux_str
		call writestr
		pop si
		call writestr
		pop ax				
		ret

;
; crlf: Print a newline
;
crlf:		mov si,crlf_msg
		; Fall through

;
; cwritestr: write a null-terminated string to the console, saving
;            registers on entry.
;
; Note: writestr and cwritestr are distinct in SYSLINUX, not in ISOLINUX
;
cwritestr:
		pushfd
                pushad
.top:		lodsb
		and al,al
                jz .end
		call writechr
                jmp short .top
.end:		popad
		popfd
                ret

writestr	equ cwritestr

;
; writehex[248]: Write a hex number in (AL, AX, EAX) to the console
;
writehex2:
		pushfd
		pushad
		shl eax,24
		mov cx,2
		jmp short writehex_common
writehex4:
		pushfd
		pushad
		shl eax,16
		mov cx,4
		jmp short writehex_common
writehex8:
		pushfd
		pushad
		mov cx,8
writehex_common:
.loop:		rol eax,4
		push eax
		and al,0Fh
		cmp al,10
		jae .high
.low:		add al,'0'
		jmp short .ischar
.high:		add al,'A'-10
.ischar:	call writechr
		pop eax
		loop .loop
		popad
		popfd
		ret

;
; Write a character to the screen.  There is a more "sophisticated"
; version of this in the subsequent code, so we patch the pointer
; when appropriate.
;

writechr:
		jmp near writechr_simple	; NOT "short"!!!!

writechr_simple:
		pushfd
		pushad
		mov ah,0Eh
		xor bx,bx
		int 10h
		popad
		popfd
		ret

;
; Get one sector.  Convenience entry point.
;
getonesec:
		mov bp,1
		; Fall through to getlinsec

;
; Get linear sectors - EBIOS LBA addressing, 2048-byte sectors.
;
; Note that we can't always do this as a single request, because at least
; Phoenix BIOSes has a 127-sector limit.  To be on the safe side, stick
; to 32 sectors (64K) per request.
;
; Input:
;	EAX	- Linear sector number
;	ES:BX	- Target buffer
;	BP	- Sector count
;
getlinsec:
		mov si,dapa			; Load up the DAPA
		mov [si+4],bx
		mov bx,es
		mov [si+6],bx
		mov [si+8],eax
.loop:
		push bp				; Sectors left
		cmp bp,byte 32
		jbe .bp_ok
		mov bp,32
.bp_ok:
		mov [si+2],bp
		push si
		mov dl,[DriveNo]
		mov ah,42h			; Extended Read
		call xint13
		pop si
		pop bp
		movzx eax,word [si+2]		; Sectors we read
		add [si+8],eax			; Advance sector pointer
		sub bp,ax			; Sectors left
		shl ax,SECTORSIZE_LG2-4		; 2048-byte sectors -> segment
		add [si+6],ax			; Advance buffer pointer
		and bp,bp
		jnz .loop
		mov eax,[si+8]			; Next sector
		ret

		; INT 13h with retry
xint13:		mov byte [RetryCount], 6
.try:		pushad
		int 13h
		jc .error
		add sp,byte 8*4			; Clean up stack
		ret
.error:		mov [DiskError],ah		; Save error code
		popad
		dec byte [RetryCount]
		jnz .try

.real_error:	mov si,diskerr_msg
		call writemsg
		mov al,[DiskError]
		call writehex2
		mov si,ondrive_str
		call writestr
		mov al,dl
		call writehex2
		call crlf
		; Fall through to kaboom

;
; kaboom: write a message and bail out.  Wait for a user keypress,
;	  then do a hard reboot.
;
kaboom:
		lss sp,[cs:Stack]
		mov ax,cs
		mov ds,ax
		mov es,ax
		mov fs,ax
		mov gs,ax
		sti
		mov si,err_bootfailed
		call cwritestr
		call getchar
		cli
		mov word [BIOS_magic],0	; Cold reboot
		jmp 0F000h:0FFF0h	; Reset vector address

;
; Data that needs to be in the first sector
;
isolinux_banner	db CR, LF, 'ISOLINUX ', version_str, ' ', date, ' ', 0
copyright_str   db ' Copyright (C) 1994-', year, ' H. Peter Anvin'
		db CR, LF, 0
isolinux_str	db 'isolinux: ', 0
%ifdef DEBUG_MESSAGES
startup_msg:	db 'Starting up, DL = ', 0
spec_ok_msg:	db 'Loaded spec packet OK, drive = ', 0
secsize_msg:	db 'Sector size appears to be ', 0
offset_msg:	db 'Loading main image from LBA = ', 0
size_msg:	db 'Sectors to load = ', 0
loaded_msg:	db 'Loaded boot image, verifying...', CR, LF, 0
verify_msg:	db 'Image checksum verified.', CR, LF, 0
allread_msg	db 'Main image read, jumping to main code...', CR, LF, 0
%endif
spec_err_msg:	db 'Loading spec packet failed, trying to wing it...', CR, LF, 0
maybe_msg:	db 'Found something at drive = ', 0
alright_msg:	db 'Looks like it might be right, continuing...', CR, LF, 0
nosecsize_msg:	db 'Failed to get sector size, assuming 0800', CR, LF, 0
diskerr_msg:	db 'Disk error ', 0
ondrive_str:	db ', drive ', 0
nothing_msg:	db 'Failed to locate CD-ROM device; boot failed.', CR, LF, 0
checkerr_msg:	db 'Image checksum error, sorry...', CR, LF, 0

err_bootfailed	db CR, LF, 'Boot failed: press a key to retry...'
bailmsg		equ err_bootfailed
crlf_msg	db CR, LF, 0

;
; El Torito spec packet
;
		align 8, db 0
spec_packet:	db 13				; Size of packet
sp_media:	db 0				; Media type
sp_drive:	db 0				; Drive number
sp_controller:	db 0				; Controller index
sp_lba:		dd 0				; LBA for emulated disk image
sp_devspec:	dw 0				; IDE/SCSI information
sp_buffer:	dw 0				; User-provided buffer
sp_loadseg:	dw 0				; Load segment
sp_sectors:	dw 0				; Sector count
sp_chs:		db 0,0,0			; Simulated CHS geometry

;
; EBIOS drive parameter packet
;
		align 8, db 0
drive_params:	dw 30				; Buffer size
dp_flags:	dw 0				; Information flags
dp_cyl:		dd 0				; Physical cylinders
dp_head:	dd 0				; Physical heads
dp_sec:		dd 0				; Physical sectors/track
dp_totalsec:	dd 0,0				; Total sectors
dp_secsize:	dw 0				; Bytes per sector
dp_dpte:	dd 0				; Device Parameter Table
dp_dpi_key:	dw 0				; 0BEDDh if rest valid
dp_dpi_len:	db 0				; DPI len
		db 0
		dw 0
dp_bus:		times 4 db 0			; Host bus type
dp_interface:	times 8 db 0			; Interface type
db_i_path:	dd 0,0				; Interface path
db_d_path:	dd 0,0				; Device path
		db 0
db_dpi_csum:	db 0				; Checksum for DPI info

;
; EBIOS disk address packet
;
		align 8, db 0
dapa:		dw 16				; Packet size
.count:		dw 0				; Block count
.off:		dw 0				; Offset of buffer
.seg:		dw 0				; Segment of buffer
.lba:		dd 0				; LBA (LSW)
		dd 0				; LBA (MSW)

		alignb 4, db 0
Stack		dw _start, 0		; SS:SP for stack reset

rl_checkpt	equ $				; Must be <= 800h

rl_checkpt_off	equ ($-$$)
%if rl_checkpt_off > 0x800
%error "Sector 0 overflow"
%endif

; ----------------------------------------------------------------------------
;  End of code and data that have to be in the first sector
; ----------------------------------------------------------------------------

all_read:
;
; Initialize screen (if we're using one)
;
		; Now set up screen parameters
		call adjust_screen

		; Patch the writechr routine to point to the full code
		mov word [writechr+1], writechr_full-(writechr+3)

; Tell the user we got this far...
%ifndef DEBUG_MESSAGES			; Gets messy with debugging on
		mov si,copyright_str
		call writestr
%endif

; Test tracers
		TRACER 'T'
		TRACER '>'

;
; Clear Files structures
;
		mov di,Files
		mov cx,(MAX_OPEN*open_file_t_size)/4
		xor eax,eax
		rep stosd

; 
; Check that no moron is trying to boot Linux on a 286 or so.  According
; to Intel, the way to check is to see if the high 4 bits of the FLAGS
; register are either all stuck at 1 (8086/8088) or all stuck at 0
; (286 in real mode), if not it is a 386 or higher.  They didn't
; say how to check for a 186/188, so I *hope* it falls out as a 8086
; or 286 in this test.
;
; Also, provide an escape route in case it doesn't work.
;
check_escapes:
		mov ah,02h			; Check keyboard flags
		int 16h
		mov [KbdFlags],al		; Save for boot prompt check
		test al,04h			; Ctrl->skip 386 check
		jnz skip_checks
test_8086:
		pushf				; Get flags
		pop ax
		and ax,0FFFh			; Clear top 4 bits
		push ax				; Load into FLAGS
		popf
		pushf				; And load back
		pop ax
		and ax,0F000h			; Get top 4 bits
		cmp ax,0F000h			; If set -> 8086/8088
		je not_386
test_286:
		pushf				; Get flags
		pop ax
		or ax,0F000h			; Set top 4 bits
		push ax
		popf
		pushf
		pop ax
		and ax,0F000h			; Get top 4 bits
		jnz is_386			; If not clear -> 386
not_386:
		mov si,err_not386
		call writestr
		jmp kaboom
is_386:
		; Now we know it's a 386 or higher
;
; Now check that there is at least 384K of low (DOS) memory
;
		int 12h
		cmp ax,384
		jae enough_ram
		mov si,err_noram
		call writestr
		jmp kaboom
enough_ram:
skip_checks:

;
; Check if we're 386 (as opposed to 486+); if so we need to blank out
; the WBINVD instruction
;
; We check for 486 by setting EFLAGS.AC
;
		pushfd				; Save the good flags
		pushfd
		pop eax
		mov ebx,eax
		xor eax,(1 << 18)		; AC bit
		push eax
		popfd
		pushfd
		pop eax
		popfd				; Restore the original flags
		xor eax,ebx
		jnz is_486
;
; 386 - Looks like we better blot out the WBINVD instruction
;
		mov byte [try_wbinvd],0c3h		; Near RET		
is_486:

;
; Now we're all set to start with our *real* business.	First load the
; configuration file (if any) and parse it.
;
; In previous versions I avoided using 32-bit registers because of a
; rumour some BIOSes clobbered the upper half of 32-bit registers at
; random.  I figure, though, that if there are any of those still left
; they probably won't be trying to install Linux on them...
;
; The code is still ripe with 16-bitisms, though.  Not worth the hassle
; to take'm out.  In fact, we may want to put them back if we're going
; to boot ELKS at some point.
;
		mov si,linuxauto_cmd		; Default command: "linux auto"
		mov di,default_cmd
                mov cx,linuxauto_len
		rep movsb

		mov di,KbdMap			; Default keymap 1:1
		xor al,al
		mov cx,256
mkkeymap:	stosb
		inc al
		loop mkkeymap

;
; Now, we need to sniff out the actual filesystem data structures.
; mkisofs gave us a pointer to the primary volume descriptor
; (which will be at 16 only for a single-session disk!); from the PVD
; we should be able to find the rest of what we need to know.
; 
get_fs_structures:
		mov eax,[bi_pvd]
		mov bx,trackbuf
		call getonesec

		mov eax,[trackbuf+156+2]
		mov [RootDir+dir_lba],eax
		mov [CurDir+dir_lba],eax
%ifdef DEBUG_MESSAGES
		mov si,dbg_rootdir_msg
		call writemsg
		call writehex8
		call crlf
%endif
		mov eax,[trackbuf+156+10]
		mov [RootDir+dir_len],eax		
		mov [CurDir+dir_len],eax
		add eax,SECTORSIZE-1
		shr eax,SECTORSIZE_LG2
		mov [RootDir+dir_clust],eax
		mov [CurDir+dir_clust],eax

		; Look for an "isolinux" directory, and if found,
		; make it the current directory instead of the root
		; directory.
		mov di,isolinux_dir
		mov al,02h			; Search for a directory
		call searchdir_iso
		jz .no_isolinux_dir
		mov [CurDir+dir_len],eax
		mov eax,[si+file_left]
		mov [CurDir+dir_clust],eax
		xor eax,eax			; Free this file pointer entry
		xchg eax,[si+file_sector]
		mov [CurDir+dir_lba],eax
%ifdef DEBUG_MESSAGES
		push si
		mov si,dbg_isodir_msg
		call writemsg
		pop si
		call writehex8
		call crlf
%endif
.no_isolinux_dir:

;
; Locate the configuration file
;
load_config:
%ifdef DEBUG_MESSAGES
		mov si,dbg_config_msg
		call writemsg
%endif

		mov di,isolinux_cfg
		call open
		jz near no_config_file		; Not found or empty

%ifdef DEBUG_MESSAGES
		mov si,dbg_configok_msg
		call writemsg
%endif

;
; Now we have the config file open
;
parse_config:
		call getkeyword
                jc near end_config_file		; Config file loaded
		cmp ax,'de'			; DEfault
		je pc_default
		cmp ax,'ap'			; APpend
		je pc_append
		cmp ax,'ti'			; TImeout
		je near pc_timeout
		cmp ax,'pr'			; PRompt
		je near pc_prompt
		cmp ax,'fo'			; FOnt
		je near pc_font
		cmp ax,'kb'			; KBd
		je near pc_kbd
		cmp ax,'di'			; DIsplay
		je near pc_display
		cmp ax,'la'			; LAbel
		je near pc_label
		cmp ax,'ke'			; KErnel
		je near pc_kernel
                cmp ax,'im'                     ; IMplicit
                je near pc_implicit
		cmp ax,'se'			; SErial
		je near pc_serial
		cmp ax,'sa'			; SAy
		je near pc_say
		cmp ax,'lo'			; LOcalboot
		je pc_localboot
		cmp al,'f'			; F-key
		jne parse_config
		jmp pc_fkey

pc_default:	mov di,default_cmd		; "default" command
		call getline
		xor al,al
		stosb				; null-terminate
		jmp short parse_config

pc_append:      cmp word [VKernelCtr],byte 0	; "append" command
		ja pc_append_vk
                mov di,AppendBuf
		call getline
                sub di,AppendBuf
pc_app1:        mov [AppendLen],di
                jmp short parse_config_2
pc_append_vk:	mov di,VKernelBuf+vk_append	; "append" command (vkernel)
		call getline
		sub di,VKernelBuf+vk_append
                cmp di,byte 2
                jne pc_app2
                cmp byte [VKernelBuf+vk_append],'-'
                jne pc_app2
                mov di,0                        ; If "append -" -> null string
pc_app2:        mov [VKernelBuf+vk_appendlen],di
		jmp short parse_config_2	

pc_localboot:	call getint			; "localboot" command
		cmp word [VKernelCtr],byte 0	; ("label" section only)
		je parse_config_2
		mov [VKernelBuf+vk_rname], byte 0	; Null kernel name
		mov [VKernelBuf+vk_rname+1], bx	; Return type
		jmp short parse_config_2

pc_kernel:	cmp word [VKernelCtr],byte 0	; "kernel" command
		je parse_config_2		; ("label" section only)
		mov di,trackbuf
		push di
		call getline
		pop si
		mov di,VKernelBuf+vk_rname
		call mangle_name
		jmp short parse_config_2

pc_timeout:	call getint			; "timeout" command
		jc parse_config_2
		mov ax,0D215h			; There are approx 1.D215h
		mul bx				; clock ticks per 1/10 s
		add bx,dx
		mov [KbdTimeOut],bx
parse_config_2:	jmp parse_config

pc_display:	call pc_getfile			; "display" command
		jz parse_config_2		; File not found?
		call get_msg_file		; Load and display file
		jmp short parse_config_2

pc_prompt:	call getint			; "prompt" command
		jc parse_config_2
		mov [ForcePrompt],bx
		jmp short parse_config_2

pc_implicit:    call getint                     ; "implicit" command
                jc parse_config_2
                mov [AllowImplicit],bx
                jmp short parse_config_2

pc_serial:	call getint			; "serial" command
		jc parse_config_2
		push bx				; Serial port #
		call skipspace
		jc parse_config_2
		call ungetc
		call getint
		jnc .valid_baud
		mov ebx,DEFAULT_BAUD		; No baud rate given
.valid_baud:	pop di				; Serial port #
		cmp ebx,byte 75
		jb parse_config_2		; < 75 baud == bogus
		mov eax,BAUD_DIVISOR
		cdq
		div ebx
		push ax				; Baud rate divisor
		mov dx,di
		shl di,1
		mov ax,[di+serial_base]
		mov [SerialPort],ax
		push ax				; Serial port base
		mov ax,00e3h			; INT 14h init parameters
		int 14h				; Init serial port
		pop bx				; Serial port base
		lea dx,[bx+3]
		mov al,83h			; Enable DLAB
		call slow_out
		pop ax				; Divisor
		mov dx,bx
		call slow_out
		inc dx
		mov al,ah
		call slow_out
		mov al,03h			; Disable DLAB
		add dx,byte 2
		call slow_out
		sub dx,byte 2
		xor al,al			; IRQ disable
		call slow_out

		; Show some life
		mov si,isolinux_banner
		call write_serial_str
		mov si,copyright_str
		call write_serial_str

		jmp short parse_config_3

pc_fkey:	sub ah,'1'
		jnb pc_fkey1
		mov ah,9			; F10
pc_fkey1:	xor cx,cx
		mov cl,ah
		push cx
		mov ax,1
		shl ax,cl
		or [FKeyMap], ax		; Mark that we have this loaded
		mov di,trackbuf
		push di
		call getline			; Get filename to display
		pop si
		pop di
		shl di,FILENAME_MAX_LG2		; Convert to offset
		add di,FKeyName
		call mangle_name		; Mangle file name
		jmp short parse_config_3

pc_label:	call commit_vk			; Commit any current vkernel
		mov di,trackbuf			; Get virtual filename
		push di
		call getline
		pop si
		mov di,VKernelBuf+vk_vname
		call mangle_name		; Mangle virtual name
		inc word [VKernelCtr]		; One more vkernel
		mov si,VKernelBuf+vk_vname 	; By default, rname == vname
		mov di,VKernelBuf+vk_rname
		mov cx,FILENAME_MAX
		rep movsb
                mov si,AppendBuf         	; Default append==global append
                mov di,VKernelBuf+vk_append
                mov cx,[AppendLen]
                mov [VKernelBuf+vk_appendlen],cx
                rep movsb
		jmp near parse_config_3

pc_font:	call pc_getfile			; "font" command
		jz parse_config_3		; File not found?
		call loadfont			; Load and install font
		jmp short parse_config_3

pc_kbd:		call pc_getfile			; "kbd" command
		jz parse_config_3
		call loadkeys
parse_config_3:	jmp parse_config

pc_say:		mov di,trackbuf			; "say" command
		push di
		call getline
		xor al,al
		stosb				; Null-terminate
		pop si
		call writestr
		call crlf
		jmp short parse_config_3

;
; pc_getfile:	For command line options that take file argument, this
; 		routine decodes the file argument and runs it through searchdir
;
pc_getfile:	mov di,trackbuf
		push di
		call getline
		pop si
		mov di,MNameBuf
		push di
		call mangle_name
		pop di
		jmp searchdir			; Tailcall

;
; commit_vk: Store the current VKernelBuf into buffer segment
;
commit_vk:
		cmp word [VKernelCtr],byte 0
		je cvk_ret			; No VKernel = return
		cmp word [VKernelCtr],max_vk	; Above limit?
		ja cvk_overflow
		mov di,[VKernelCtr]
		dec di
		shl di,vk_shift
		mov si,VKernelBuf
		mov cx,(vk_size >> 2)
		push es
		push word vk_seg
		pop es
		rep movsd			; Copy to buffer segment
		pop es
cvk_ret:	ret
cvk_overflow:	mov word [VKernelCtr],max_vk	; No more than max_vk, please
		ret

;
; End of configuration file
;
end_config_file:
		call commit_vk			; Commit any current vkernel
no_config_file:
;
; Check whether or not we are supposed to display the boot prompt.
;
check_for_key:
		cmp word [ForcePrompt],byte 0	; Force prompt?
		jnz enter_command
		test byte [KbdFlags],5Bh	; Caps, Scroll, Shift, Alt
		jz near auto_boot		; If neither, default boot

enter_command:
		mov si,boot_prompt
		call cwritestr

		mov byte [FuncFlag],0		; <Ctrl-F> not pressed
		mov di,command_line
;
; get the very first character -- we can either time
; out, or receive a character press at this time.  Some dorky BIOSes stuff
; a return in the buffer on bootup, so wipe the keyboard buffer first.
;
clear_buffer:	mov ah,1			; Check for pending char
		int 16h
		jz get_char_time
		xor ax,ax			; Get char
		int 16h
		jmp short clear_buffer
get_char_time:	
		call vgashowcursor
		mov cx,[KbdTimeOut]
		and cx,cx
		jz get_char			; Timeout == 0 -> no timeout
		inc cx				; The first loop will happen
						; immediately as we don't
						; know the appropriate DX value
time_loop:	push cx
tick_loop:	push dx
		call pollchar
		jnz get_char_pop
		mov dx,[BIOS_timer]		; Get time "of day"
		pop ax
		cmp dx,ax			; Has the timer advanced?
		je tick_loop
		pop cx
		loop time_loop			; If so, decrement counter
		call vgahidecursor
		jmp command_done		; Timeout!

get_char_pop:	pop eax				; Clear stack
get_char:
		call vgashowcursor
		call getchar
		call vgahidecursor
		and al,al
		jz func_key

got_ascii:	cmp al,7Fh			; <DEL> == <BS>
		je backspace
		cmp al,' '			; ASCII?
		jb not_ascii
		ja enter_char
		cmp di,command_line		; Space must not be first
		je get_char
enter_char:	test byte [FuncFlag],1
		jz .not_ctrl_f
		mov byte [FuncFlag],0
		cmp al,'0'
		jb .not_ctrl_f
		je ctrl_f_0
		cmp al,'9'
		jbe ctrl_f
.not_ctrl_f:	cmp di,max_cmd_len+command_line ; Check there's space
		jnb get_char
		stosb				; Save it
		call writechr			; Echo to screen
get_char_2:	jmp short get_char
not_ascii:	mov byte [FuncFlag],0
		cmp al,0Dh			; Enter
		je near command_done
		cmp al,06h			; <Ctrl-F>
		je set_func_flag
		cmp al,08h			; Backspace
		jne get_char
backspace:	cmp di,command_line		; Make sure there is anything
		je get_char			; to erase
		dec di				; Unstore one character
		mov si,wipe_char		; and erase it from the screen
		call cwritestr
		jmp short get_char_2

set_func_flag:
		mov byte [FuncFlag],1
		jmp short get_char_2

ctrl_f_0:	add al,10			; <Ctrl-F>0 == F10
ctrl_f:		push di
		sub al,'1'
		xor ah,ah
		jmp short show_help

func_key:
		push di
		cmp ah,68			; F10
		ja get_char_2
		sub ah,59			; F1
		jb get_char_2
		shr ax,8
show_help:	; AX = func key # (0 = F1, 9 = F10)
		mov cl,al
		shl ax,FILENAME_MAX_LG2		; Convert to offset
		mov bx,1
		shl bx,cl
		and bx,[FKeyMap]
		jz get_char_2			; Undefined F-key
		mov di,ax
		add di,FKeyName
		call searchdir
		jz fk_nofile
		push si
		call crlf
		pop si
		call get_msg_file
		jmp short fk_wrcmd
fk_nofile:
		call crlf
fk_wrcmd:
		mov si,boot_prompt
		call cwritestr
		pop di				; Command line write pointer
		push di
		mov byte [di],0			; Null-terminate command line
		mov si,command_line
		call cwritestr			; Write command line so far
		pop di
		jmp short get_char_2
auto_boot:
		mov si,default_cmd
		mov di,command_line
		mov cx,(max_cmd_len+4) >> 2
		rep movsd
		jmp short load_kernel
command_done:
		call crlf
		cmp di,command_line		; Did we just hit return?
		je auto_boot
		xor al,al			; Store a final null
		stosb

load_kernel:					; Load the kernel now
;
; First we need to mangle the kernel name the way DOS would...
;
		mov si,command_line
                mov di,KernelName
                push si
                push di
		call mangle_name
		pop di
                pop si
;
; Fast-forward to first option (we start over from the beginning, since
; mangle_name doesn't necessarily return a consistent ending state.)
;
clin_non_wsp:   lodsb
                cmp al,' '
                ja clin_non_wsp
clin_is_wsp:    and al,al
                jz clin_opt_ptr
                lodsb
                cmp al,' '
                jbe clin_is_wsp
clin_opt_ptr:   dec si                          ; Point to first nonblank
                mov [CmdOptPtr],si		; Save ptr to first option
;
; Now check if it is a "virtual kernel"
;
		mov cx,[VKernelCtr]
		push ds
		push word vk_seg
		pop ds
		cmp cx,byte 0
		je not_vk
		xor si,si			; Point to first vkernel
vk_check:	pusha
		mov cx,FILENAME_MAX
		repe cmpsb			; Is this it?
		je near vk_found
		popa
		add si,vk_size
		loop vk_check
not_vk:		pop ds
;
; Not a "virtual kernel" - check that's OK and construct the command line
;
                cmp word [AllowImplicit],byte 0
                je bad_implicit
                push es
                push si
                push di
                mov di,real_mode_seg
                mov es,di
                mov si,AppendBuf
                mov di,cmd_line_here
                mov cx,[AppendLen]
                rep movsb
                mov [CmdLinePtr],di
                pop di
                pop si
                pop es
;
; Find the kernel on disk
;
get_kernel:     mov byte [KernelName+FILENAME_MAX],0	; Zero-terminate filename/extension
		mov di,KernelName
		xor al,al
		mov cx,FILENAME_MAX-5		; Need 4 chars + null
		repne scasb			; Scan for final null
		jne .no_skip
		dec di				; Point to final null 
.no_skip:	mov [KernelExtPtr],di
		mov bx,exten_table
.search_loop:	push bx
                mov di,KernelName	      	; Search on disk
                call searchdir
		pop bx
                jnz near kernel_good
		mov eax,[bx]			; Try a different extension
		mov si,[KernelExtPtr]
		mov [si],eax
		mov byte [si+4],0
		add bx,byte 4
		cmp bx,exten_table_end
		jna .search_loop		; allow == case (final case)
bad_kernel:     
		mov si,KernelName
                mov di,KernelCName
		push di
                call unmangle_name              ; Get human form
		mov si,err_notfound		; Complain about missing kernel
		call cwritestr
		pop si				; KernelCName
                call cwritestr
                mov si,crlf_msg
                jmp abort_load                  ; Ask user for clue
;
; bad_implicit: The user entered a nonvirtual kernel name, with "implicit 0"
;
bad_implicit:   mov si,KernelName		; For the error message
                mov di,KernelCName
                call unmangle_name
                jmp short bad_kernel
;
; vk_found: We *are* using a "virtual kernel"
;
vk_found:	popa
		push di
		mov di,VKernelBuf
		mov cx,vk_size >> 2
		rep movsd
		push es				; Restore old DS
		pop ds
		push es
		push word real_mode_seg
		pop es
		mov di,cmd_line_here
		mov si,VKernelBuf+vk_append
		mov cx,[VKernelBuf+vk_appendlen]
		rep movsb
		mov [CmdLinePtr],di		; Where to add rest of cmd
		pop es
                pop di                          ; DI -> KernelName
		push di	
		mov si,VKernelBuf+vk_rname
		mov cx,FILENAME_MAX		; We need ECX == CX later
		rep movsb
		pop di
		xor bx,bx			; Try only one version

		; Is this a "localboot" pseudo-kernel?
		cmp byte [VKernelBuf+vk_rname], 0
		jne near get_kernel		; No, it's real, go get it

		mov ax, [VKernelBuf+vk_rname+1]
		jmp local_boot
;
; kernel_corrupt: Called if the kernel file does not seem healthy
;
kernel_corrupt: mov si,err_notkernel
                jmp abort_load
;
; This is it!  We have a name (and location on the disk)... let's load
; that sucker!!  First we have to decide what kind of file this is; base
; that decision on the file extension.  The following extensions are
; recognized; case insensitive:
;
; .com 	- COMBOOT image
; .cbt	- COMBOOT image
; .bs	- Boot sector
; .bss	- Boot sector, but transfer over DOS superblock
;
; Boot sectors are currently not supported by PXELINUX.
;
; Anything else is assumed to be a Linux kernel.
;
kernel_good:
		pusha
		mov si,KernelName
		mov di,KernelCName
		call unmangle_name
		sub di,KernelCName
		mov [KernelCNameLen],di
		popa
		
		push di
		push ax
		mov di,KernelName
		xor al,al
		mov cx,FILENAME_MAX
		repne scasb
		jne .one_step
		dec di
.one_step:	mov ecx,[di-4]			; 4 bytes before end
		pop ax
		pop di

		or ecx,20202000h		; Force lower case

		cmp ecx,'.com'
		je near is_comboot_image
		cmp ecx,'.cbt'
		je near is_comboot_image
		cmp ecx,'.bss'
		je near is_bss_sector
		and ecx, 00ffffffh
		cmp ecx,'.bs'
		je near is_bootsector
		; Otherwise Linux kernel
;
; A Linux kernel consists of three parts: boot sector, setup code, and
; kernel code.	The boot sector is never executed when using an external
; booting utility, but it contains some status bytes that are necessary.
;
; First check that our kernel is at least 64K and less than 8M (if it is
; more than 8M, we need to change the logic for loading it anyway...)
;
is_linux_kernel:
                cmp dx,80h			; 8 megs
		ja kernel_corrupt
		and dx,dx
		jz kernel_corrupt
kernel_sane:	push ax
		push dx
		push si
		mov si,loading_msg
                call cwritestr
;
; Now start transferring the kernel
;
		push word real_mode_seg
		pop es

		push ax
		push dx
		div word [ClustSize]		; # of clusters total
		and dx,dx			; Round up
		setnz dl
		movzx dx,dl
		add ax,dx
                mov [KernelClust],ax
		pop dx
		pop ax
		mov [KernelSize],ax
		mov [KernelSize+2],dx
;
; Now, if we transfer these straight, we'll hit 64K boundaries.	 Hence we
; have to see if we're loading more than 64K, and if so, load it step by
; step.
;
		mov dx,1			; 10000h
		xor ax,ax
		div word [ClustSize]
		mov [ClustPerMoby],ax		; Clusters/64K
;
; Start by loading the bootsector/setup code, to see if we need to
; do something funky.  It should fit in the first 32K (loading 64K won't
; work since we might have funny stuff up near the end of memory).
; If we have larger than 32K clusters, yes, we're hosed.
;
		call abort_check		; Check for abort key
		mov cx,[ClustPerMoby]
		shr cx,1			; Half a moby
		sub [KernelClust],cx
		xor bx,bx
                pop si                          ; Cluster pointer on stack
		call getfssec
		jc near kernel_corrupt		; Failure in first 32K
                cmp word [es:bs_bootsign],0AA55h
		jne near kernel_corrupt		; Boot sec signature missing
;
; Get the BIOS' idea of what the size of high memory is.
;
		push si				; Save our cluster pointer!
;
; First, try INT 15:E820 (get BIOS memory map)
;
get_e820:
		push es
		xor ebx,ebx			; Start with first record
		mov es,bx			; Need ES = DS = 0 for now
		jmp short .do_e820		; Skip "at end" check first time!
.int_loop:	and ebx,ebx			; If we're back at beginning...
		jz no_e820			; ... bail; nothing found
.do_e820:	mov eax,0000E820h
		mov edx,534D4150h		; "SMAP" backwards
		mov ecx,20
		mov di,E820Buf
		int 15h
		jc no_e820
		cmp eax,534D4150h
		jne no_e820
;
; Look for a memory block starting at <= 1 MB and continuing upward
;
		cmp dword [E820Buf+4], byte 0
		ja .int_loop			; Start >= 4 GB?
		mov edx, (1 << 20)
		sub edx, [E820Buf]
		jb .int_loop			; Start >= 1 MB?
		mov eax, 0FFFFFFFFh
		cmp dword [E820Buf+12], byte 0
		ja .huge			; Size >= 4 GB
		mov eax, [E820Buf+8]
.huge:		sub eax, edx			; Adjust size to start at 1 MB
		jbe .int_loop			; Completely below 1 MB?

		; Now EAX contains the size of memory 1 MB...up
		cmp dword [E820Buf+16], byte 1
		jne near err_nohighmem		; High memory isn't usable memory!!!!

		; We're good!
		pop es
		jmp short got_highmem_add1mb	; Still need to add low 1 MB

;
; INT 15:E820 failed.  Try INT 15:E801.
;
no_e820:	pop es

		mov ax,0e801h			; Query high memory (semi-recent)
		int 15h
		jc no_e801
		cmp ax,3c00h
		ja no_e801			; > 3C00h something's wrong with this call
		jb e801_hole			; If memory hole we can only use low part

		mov ax,bx
		shl eax,16			; 64K chunks
		add eax,(16 << 20)		; Add first 16M
		jmp short got_highmem				

;
; INT 15:E801 failed.  Try INT 15:88.
;
no_e801:
		mov ah,88h			; Query high memory (oldest)
		int 15h
		cmp ax,14*1024			; Don't trust memory >15M
		jna e801_hole
		mov ax,14*1024
e801_hole:
		and eax,0ffffh
		shl eax,10			; Convert from kilobytes
got_highmem_add1mb:
		add eax,(1 << 20)		; First megabyte
got_highmem:
		sub eax,HIGHMEM_SLOP
		mov [HighMemSize],eax

;
; Construct the command line (append options have already been copied)
;
construct_cmdline:
		mov di,[CmdLinePtr]
                mov si,boot_image        	; BOOT_IMAGE=
                mov cx,boot_image_len
                rep movsb
                mov si,KernelCName       	; Unmangled kernel name
                mov cx,[KernelCNameLen]
                rep movsb
                mov al,' '                      ; Space
                stosb

.noipappend:
                mov si,[CmdOptPtr]              ; Options from user input
		mov cx,(kern_cmd_len+3) >> 2
		rep movsd
;
; Scan through the command line for anything that looks like we might be
; interested in.  The original version of this code automatically assumed
; the first option was BOOT_IMAGE=, but that is no longer certain.
;
		mov si,cmd_line_here
                mov byte [initrd_flag],0
                push es				; Set DS <- real_mode_seg
                pop ds
get_next_opt:   lodsb
		and al,al
		jz near cmdline_end
		cmp al,' '
		jbe get_next_opt
		dec si
                mov eax,[si]
                cmp eax,'vga='
		je is_vga_cmd
                cmp eax,'mem='
		je is_mem_cmd
                push es                         ; Save ES -> real_mode_seg
                push cs
                pop es                          ; Set ES <- normal DS
                mov di,initrd_cmd
		mov cx,initrd_cmd_len
		repe cmpsb
                jne not_initrd
		mov di,InitRD
                push si                         ; mangle_dir mangles si
                call mangle_name                ; Mangle ramdisk name
                pop si
		cmp byte [es:InitRD],0		; Null filename?
                seta byte [es:initrd_flag]	; Set flag if not
not_initrd:	pop es                          ; Restore ES -> real_mode_seg
skip_this_opt:  lodsb                           ; Load from command line
                cmp al,' '
                ja skip_this_opt
                dec si
                jmp short get_next_opt
is_vga_cmd:
                add si,byte 4
                mov eax,[si]
                mov bx,-1
                cmp eax, 'norm'                 ; vga=normal
                je vc0
                and eax,0ffffffh		; 3 bytes
                mov bx,-2
                cmp eax, 'ext'                  ; vga=ext
                je vc0
                mov bx,-3
                cmp eax, 'ask'                  ; vga=ask
                je vc0
                call parseint                   ; vga=<number>
		jc skip_this_opt		; Not an integer
vc0:		mov [bs_vidmode],bx		; Set video mode
		jmp short skip_this_opt
is_mem_cmd:
                add si,byte 4
                call parseint
		jc skip_this_opt		; Not an integer
		sub ebx,HIGHMEM_SLOP
		mov [cs:HighMemSize],ebx
		jmp short skip_this_opt
cmdline_end:
                push cs                         ; Restore standard DS
                pop ds
		sub si,cmd_line_here
		mov [CmdLineLen],si		; Length including final null
;
; Now check if we have a large kernel, which needs to be loaded high
;
		cmp dword [es:su_header],HEADER_ID	; New setup code ID
		jne near old_kernel		; Old kernel, load low
		cmp word [es:su_version],0200h	; Setup code version 2.0
		jb near old_kernel		; Old kernel, load low
                cmp word [es:su_version],0201h	; Version 2.01+?
                jb new_kernel                   ; If 2.00, skip this step
                mov word [es:su_heapend],linux_stack	; Set up the heap
                or byte [es:su_loadflags],80h	; Let the kernel know we care
;
; We definitely have a new-style kernel.  Let the kernel know who we are,
; and that we are clueful
;
new_kernel:
		mov byte [es:su_loader],syslinux_id	; Show some ID
		movzx ax,byte [es:bs_setupsecs]	; Variable # of setup sectors
		mov [SetupSecs],ax
;
; About to load the kernel.  This is a modern kernel, so use the boot flags
; we were provided.
;
                mov al,[es:su_loadflags]
		mov [LoadFlags],al
;
; Load the kernel.  We always load it at 100000h even if we're supposed to
; load it "low"; for a "low" load we copy it down to low memory right before
; jumping to it.
;
read_kernel:
                mov si,KernelCName		; Print kernel name part of
                call cwritestr                  ; "Loading" message
                mov si,dotdot_msg		; Print dots
                call cwritestr

                mov eax,[HighMemSize]
		sub eax,100000h			; Load address
		cmp eax,[KernelSize]
		jb near no_high_mem		; Not enough high memory
;
; Move the stuff beyond the setup code to high memory at 100000h
;
		movzx esi,word [SetupSecs]	; Setup sectors
		inc esi				; plus 1 boot sector
                shl esi,9			; Convert to bytes
                mov ecx,108000h			; 108000h = 1M + 32K
                sub ecx,esi			; Adjust pointer to 2nd block
                mov [HiLoadAddr],ecx
		sub ecx,100000h			; Turn into a counter
		shr ecx,2			; Convert to dwords
		add esi,(real_mode_seg << 4)	; Pointer to source
                mov edi,100000h                 ; Copy to address 100000h
                call bcopy			; Transfer to high memory

                push word xfer_buf_seg		; Transfer buffer segment
                pop es
high_load_loop: 
                mov si,dot_msg			; Progress report
                call cwritestr
                call abort_check
                mov cx,[KernelClust]
		cmp cx,[ClustPerMoby]
		jna high_last_moby
		mov cx,[ClustPerMoby]
high_last_moby:
		sub [KernelClust],cx
		xor bx,bx			; Load at offset 0
                pop si                          ; Restore cluster pointer
		call getfssec
                push si                         ; Save cluster pointer
                pushf                           ; Save EOF
                xor bx,bx
		mov esi,(xfer_buf_seg << 4)
                mov edi,[HiLoadAddr]		; Destination address
                mov ecx,4000h			; Cheating - transfer 64K
                call bcopy			; Transfer to high memory
		mov [HiLoadAddr],edi		; Point to next target area
                popf                            ; Restore EOF
                jc high_load_done               ; If EOF we are done
                cmp word [KernelClust],byte 0	; Are we done?
		jne high_load_loop		; Apparently not
high_load_done:
		pop si				; No longer needed
                mov ax,real_mode_seg		; Set to real mode seg
                mov es,ax

                mov si,dot_msg
                call cwritestr

		call crlf
;
; Now see if we have an initial RAMdisk; if so, do requisite computation
; We know we have a new kernel; the old_kernel code already will have objected
; if we tried to load initrd using an old kernel
;
load_initrd:
                test byte [initrd_flag],1
                jz nk_noinitrd
                push es                         ; ES->real_mode_seg
                push ds
                pop es                          ; We need ES==DS
                mov si,InitRD
                mov di,InitRDCName
                call unmangle_name              ; Create human-readable name
                sub di,InitRDCName
                mov [InitRDCNameLen],di
                mov di,InitRD
                call searchdir                  ; Look for it in directory
                pop es
		jz initrd_notthere
		mov [initrd_ptr],si		; Save cluster pointer
		mov [es:su_ramdisklen1],ax	; Ram disk length
		mov [es:su_ramdisklen2],dx
		div word [ClustSize]
		and dx,dx			; Round up
		setnz dl
		movzx dx,dl
		add ax,dx
		mov [InitRDClust],ax		; Ramdisk clusters
		mov edx,[HighMemSize]		; End of memory
		mov eax,HIGHMEM_MAX		; Limit imposed by kernel
		cmp edx,eax
		jna memsize_ok
		mov edx,eax			; Adjust to fit inside limit
memsize_ok:
		sub edx,[es:su_ramdisklen]	; Subtract size of ramdisk
                xor dx,dx			; Round down to 64K boundary
                mov [InitRDat],edx		; Load address
		call loadinitrd			; Load initial ramdisk
		jmp short initrd_end

initrd_notthere:
                mov si,err_noinitrd
                call cwritestr
                mov si,InitRDCName
                call cwritestr
                mov si,crlf_msg
                jmp abort_load

no_high_mem:    mov si,err_nohighmem		; Error routine
                jmp abort_load

initrd_end:
nk_noinitrd:
;
; Abandon hope, ye that enter here!  We do no longer permit aborts.
;
                call abort_check        	; Last chance!!

		mov si,ready_msg
		call cwritestr

		call vgaclearmode		; We can't trust ourselves after this

		cli
		xor ax,ax
		mov ss,ax
		mov sp,7C00h			; Set up a more normal stack
		
;
; Now, if we were supposed to load "low", copy the kernel down to 10000h
; and the real mode stuff to 90000h.  We assume that all bzImage kernels are
; capable of starting their setup from a different address.
;
		mov bx,real_mode_seg		; Real mode segment
		mov fs,bx			; FS -> real_mode_seg
;
; Copy command line.  Unfortunately, the kernel boot protocol requires
; the command line to exist in the 9xxxxh range even if the rest of the
; setup doesn't.
;
		cli				; In case of hooked interrupts
		test byte [LoadFlags],LOAD_HIGH
		jz need_high_cmdline
		cmp word [fs:su_version],0202h	; Support new cmdline protocol?
		jb need_high_cmdline
		; New cmdline protocol
		; Store 32-bit (flat) pointer to command line
		mov dword [fs:su_cmd_line_ptr],(real_mode_seg << 4) + cmd_line_here
		jmp short in_proper_place

need_high_cmdline:
;
; Copy command line up to 90000h
;
		mov ax,9000h
		mov es,ax
		mov si,cmd_line_here
		mov di,si
		mov [fs:kern_cmd_magic],word CMD_MAGIC ; Store magic
		mov [fs:kern_cmd_offset],di	; Store pointer

		mov cx,[CmdLineLen]
		add cx,byte 3
		shr cx,2			; Convert to dwords
		fs rep movsd

		test byte [LoadFlags],LOAD_HIGH
		; Note bx -> real_mode_seg still
		jnz in_proper_place		; If high load, we're done

;
; Loading low; we can't assume it's safe to run in place.
;
; Copy real_mode stuff up to 90000h
;
		mov ax,real_mode_seg
		mov fs,ax
		mov ax,9000h
		mov es,ax
		mov cx,[SetupSecs]
		inc cx				; Setup + boot sector
		shl cx,7			; Sectors -> dwords
		xor si,si
		xor di,di
		fs rep movsd			; Copy setup + boot sector
;
; Some kernels in the 1.2 ballpark but pre-bzImage have more than 4
; setup sectors, but the boot protocol had not yet been defined.  They
; rely on a signature to figure out if they need to copy stuff from
; the "protected mode" kernel area.  Unfortunately, we used that area
; as a transfer buffer, so it's going to find the signature there.
; Hence, zero the low 32K beyond the setup area.
;
		mov di,[SetupSecs]
		inc di				; Setup + boot sector
		mov cx,32768/512		; Sectors/32K
		sub cx,di			; Remaining sectors
		shl di,9			; Sectors -> bytes
		shl cx,7			; Sectors -> dwords
		xor eax,eax
		rep stosd			; Clear region
;
		mov ecx,[KernelSize]
		add ecx,3			; Round upwards
		shr ecx,2			; Bytes -> dwords
		mov esi,100000h
		mov edi,10000h
		call bcopy

		mov bx,9000h			; Real mode segment

;
; Now everything is where it needs to be...
;
in_proper_place:
		mov es,bx			; Real mode segment
;
; If the default root device is set to FLOPPY (0000h), change to
; /dev/fd0 (0200h)
;
		cmp word [es:bs_rootdev],byte 0
		jne root_not_floppy
		mov word [es:bs_rootdev],0200h
root_not_floppy:
;
; Copy the disk table to high memory, then re-initialize the floppy
; controller
;
; This needs to be moved before the copy
;
%if 0
		push ds
		push bx
		lds si,[fdctab]
		mov di,linux_fdctab
		mov cx,3			; 12 bytes
		push di
		rep movsd
		pop di
		mov [fdctab1],di		; Save new floppy tab pos
		mov [fdctab2],es
		xor ax,ax
		xor dx,dx
		int 13h
		pop bx
		pop ds
%endif
;
; Linux wants the floppy motor shut off before starting the kernel,
; at least bootsect.S seems to imply so
;
kill_motor:
		mov dx,03F2h
		xor al,al
		call slow_out
;
; If we're debugging, wait for a keypress so we can read any debug messages
;
%ifdef debug
                xor ax,ax
                int 16h
%endif
;
; Set up segment registers and the Linux real-mode stack
; Note: bx == the real mode segment
;
		cli
		; es is already == real mode segment
		mov ds,bx
		mov fs,bx
		mov gs,bx
		mov ss,bx
		mov sp,linux_stack
;
; We're done... now RUN THAT KERNEL!!!!
; Setup segment == real mode segment + 020h; we need to jump to offset
; zero in the real mode segment.
;
		add bx,020h
		push bx
		push word 0h
		retf

;
; Load an older kernel.  Older kernels always have 4 setup sectors, can't have
; initrd, and are always loaded low.
;
old_kernel:
                test byte [initrd_flag],1	; Old kernel can't have initrd
                jz load_old_kernel
                mov si,err_oldkernel
                jmp abort_load
load_old_kernel:
		mov word [SetupSecs],4		; Always 4 setup sectors
		mov byte [LoadFlags],0		; Always low
		jmp read_kernel

;
; Load a COMBOOT image.  A COMBOOT image is basically a DOS .COM file,
; except that it may, of course, not contain any DOS system calls.  We
; do, however, allow the execution of INT 20h to return to SYSLINUX.
;
is_comboot_image:
		and dx,dx
		jnz near comboot_too_large
		cmp ax,0ff00h		; Max size in bytes
		jae comboot_too_large

		;
		; Set up the DOS vectors in the IVT (INT 20h-3fh)
		;
		mov dword [4*0x20],comboot_return	; INT 20h vector
		mov eax,comboot_bogus
		mov di,4*0x21
		mov cx,31		; All remaining DOS vectors
		rep stosd
	
		mov cx,comboot_seg
		mov es,cx

		mov bx,100h		; Load at <seg>:0100h

		mov cx,[ClustPerMoby]	; Absolute maximum # of clusters
		call getfssec

		xor di,di
		mov cx,64		; 256 bytes (size of PSP)
		xor eax,eax		; Clear PSP
		rep stosd

		mov word [es:0], 020CDh	; INT 20h instruction
		; First non-free paragraph
		mov word [es:02h], comboot_seg+1000h

		; Copy the command line from high memory
		mov cx,125		; Max cmdline len (minus space and CR)
		mov si,[CmdOptPtr]
		mov di,081h		; Offset in PSP for command line
		mov al,' '		; DOS command lines begin with a space
		stosb

comboot_cmd_cp:	lodsb
		and al,al
		jz comboot_end_cmd
		stosb
		loop comboot_cmd_cp
comboot_end_cmd: mov al,0Dh		; CR after last character
		stosb
		mov al,126		; Include space but not CR
		sub al,cl
		mov [es:80h], al	; Store command line length

		mov [SavedSSSP],sp
		mov ax,ss		; Save away SS:SP
		mov [SavedSSSP+2],ax

		call vgaclearmode	; Reset video

		mov ax,es
		mov ds,ax
		mov ss,ax
		xor sp,sp
		push word 0		; Return to address 0 -> exit

		jmp comboot_seg:100h	; Run it

; Looks like a COMBOOT image but too large
comboot_too_large:
		mov si,err_comlarge
		call cwritestr
cb_enter:	jmp enter_command

; Proper return vector
comboot_return:	cli			; Don't trust anyone
		xor ax,ax
		mov ds,ax
		mov es,ax
		lss sp,[SavedSSSP]
		sti
		cld
		jmp short cb_enter

; Attempted to execute DOS system call
comboot_bogus:	cli			; Don't trust anyone
		xor ax,ax
		mov ds,ax
		mov es,ax
		lss sp,[SavedSSSP]
		sti
		cld
		mov si,KernelCName
		call cwritestr
		mov si,err_notdos
		call cwritestr
		jmp short cb_enter

;
; Load a boot sector
;
is_bootsector:
is_bss_sector:
		; Can't load these from the network, dang it!
.badness:	jmp short .badness

;
; Boot a specified local disk.  AX specifies the BIOS disk number; or
; 0xFFFF in case we should execute INT 18h ("next device.")
;
local_boot:
		call vgaclearmode
		lss sp,[cs:Stack]		; Restore stack pointer
		xor dx,dx
		mov ds,dx
		mov es,dx
		mov fs,dx
		mov gs,dx
		mov si,localboot_msg
		call writestr
		cmp ax,-1
		je .int18
		
		; Load boot sector from the specified BIOS device and jump to it.
		mov dl,al
		xor dh,dh
		push dx
		xor ax,ax			; Reset drive
		call xint13
		mov ax,0201h			; Read one sector
		mov cx,0001h			; C/H/S = 0/0/1 (first sector)
		mov bx,trackbuf
		call xint13
		pop dx
		cli				; Abandon hope, ye who enter here
		mov si,trackbuf
		mov di,07C00h
		mov cx,512			; Probably overkill, but should be safe
		rep movsd
		lss sp,[cs:InitStack]
		jmp 0:07C00h			; Jump to new boot sector

.int18:
		int 18h				; Hope this does the right thing...
		jmp kaboom			; If we returned, oh boy...

;
; 32-bit bcopy routine for real mode
;
; We enter protected mode, set up a flat 32-bit environment, run rep movsd
; and then exit.  IMPORTANT: This code assumes cs == ss == 0.
;
; This code is probably excessively anal-retentive in its handling of
; segments, but this stuff is painful enough as it is without having to rely
; on everything happening "as it ought to."
;
		align 4
bcopy_gdt:	dw bcopy_gdt_size-1	; Null descriptor - contains GDT
		dd bcopy_gdt		; pointer for LGDT instruction
		dw 0
		dd 0000ffffh		; Code segment, use16, readable,
		dd 00009b00h		; present, dpl 0, cover 64K
		dd 0000ffffh		; Data segment, use16, read/write,
		dd 008f9300h		; present, dpl 0, cover all 4G
		dd 0000ffffh		; Data segment, use16, read/write,
		dd 00009300h		; present, dpl 0, cover 64K
bcopy_gdt_size:	equ $-bcopy_gdt

bcopy:		push eax
		pushf			; Saves, among others, the IF flag
		push gs
		push fs
		push ds
		push es
		mov [cs:SavedSSSP],sp
		mov ax,ss
		mov [cs:SavedSSSP+2],ax

		cli
		call enable_a20

		o32 lgdt [bcopy_gdt]
		mov eax,cr0
		or al,1
		mov cr0,eax		; Enter protected mode
		jmp 08h:.in_pm

.in_pm:		mov ax,10h		; Data segment selector
		mov es,ax
		mov ds,ax

		mov al,18h		; "Real-mode-like" data segment
		mov ss,ax
		mov fs,ax
		mov gs,ax	
	
		a32 rep movsd		; Do our business
		
		mov es,ax		; Set to "real-mode-like"
		mov ds,ax
	
		mov eax,cr0
		and al,~1
		mov cr0,eax		; Disable protected mode
		jmp 0:.in_rm

.in_rm:		; Back in real mode
		lss sp,[cs:SavedSSSP]
		pop es
		pop ds
		pop fs
		pop gs
		call disable_a20

		popf			; Re-enables interrupts
		pop eax
		ret

;
; Routines to enable and disable (yuck) A20.  These routines are gathered
; from tips from a couple of sources, including the Linux kernel and
; http://www.x86.org/.  The need for the delay to be as large as given here
; is indicated by Donnie Barnes of RedHat, the problematic system being an
; IBM ThinkPad 760EL.
;
; We typically toggle A20 twice for every 64K transferred.
; 
%define	io_delay	call _io_delay
%define IO_DELAY_PORT	80h		; Invalid port (we hope!)
%define disable_wait 	32		; How long to wait for a disable

%define A20_DUNNO	0		; A20 type unknown
%define A20_NONE	1		; A20 always on?
%define A20_BIOS	2		; A20 BIOS enable
%define A20_KBC		3		; A20 through KBC
%define A20_FAST	4		; A20 through port 92h

slow_out:	out dx, al		; Fall through

_io_delay:	out IO_DELAY_PORT,al
		out IO_DELAY_PORT,al
		ret

enable_a20:
		pushad
		mov byte [cs:A20Tries],255 ; Times to try to make this work

try_enable_a20:
;
; Flush the caches
;
;		call try_wbinvd

;
; If the A20 type is known, jump straight to type
;
		mov bp,[cs:A20Type]
		add bp,bp			; Convert to word offset
		jmp word [cs:bp+A20List]

;
; First, see if we are on a system with no A20 gate
;
a20_dunno:
a20_none:
		mov byte [cs:A20Type], A20_NONE
		call a20_test
		jnz a20_done

;
; Next, try the BIOS (INT 15h AX=2401h)
;
a20_bios:
		mov byte [cs:A20Type], A20_BIOS
		mov ax,2401h
		pushf				; Some BIOSes muck with IF
		int 15h
		popf

		call a20_test
		jnz a20_done

;
; Enable the keyboard controller A20 gate
;
a20_kbc:
		mov dl, 1			; Allow early exit
		call empty_8042
		jnz a20_done			; A20 live, no need to use KBC

		mov byte [cs:A20Type], A20_KBC	; Starting KBC command sequence

		mov al,0D1h			; Command write
		out 064h, al
		call empty_8042_uncond

		mov al,0DFh			; A20 on
		out 060h, al
		call empty_8042_uncond

		; Verify that A20 actually is enabled.  Do that by
		; observing a word in low memory and the same word in
		; the HMA until they are no longer coherent.  Note that
		; we don't do the same check in the disable case, because
		; we don't want to *require* A20 masking (SYSLINUX should
		; work fine without it, if the BIOS does.)
.kbc_wait:	push cx
		xor cx,cx
.kbc_wait_loop:
		call a20_test
		jnz a20_done_pop
		loop .kbc_wait_loop

		pop cx
;
; Running out of options here.  Final attempt: enable the "fast A20 gate"
;
a20_fast:
		mov byte [cs:A20Type], A20_FAST	; Haven't used the KBC yet
		in al, 092h
		or al,02h
		and al,~01h			; Don't accidentally reset the machine!
		out 092h, al

.fast_wait:	push cx
		xor cx,cx
.fast_wait_loop:
		call a20_test
		jnz a20_done_pop
		loop .fast_wait_loop

		pop cx

;
; Oh bugger.  A20 is not responding.  Try frobbing it again; eventually give up
; and report failure to the user.
;


		dec byte [cs:A20Tries]
		jnz try_enable_a20ov cx,[CmdLsh eax-20ov cx,[C:     jz lorobbingmemory and the scx

;
; Oh bugger. 

;
; Oh breDELAY_Puit_Z0_done_pop
		loop .fast_wait_loop

		p		loop .it_Z0ax,
es:batee1,03F2h
er.
;


		dec byte    memory,err_notdos
		ax
	sy
; step.
 16h
onger c_gdt_bx
 e by
; :ALoper ret:sent, dpl 0, 			; Ri03F2h
eready wi
onvert toCt
		call writechr			; Echohu cluster pointer
		call getfssec
                push si                  check in image
mdLsh ea:su_raca		pop

		ov [Saveca		pop

		ovcall shl     Fltfsso		or dir
		pop bx
   _cmd_l-ov [Savecbase
 di,07C003h as it it ha up
all getfssec
                push si          w part

	aa instthe 	mov amroughte bad_implie
mdLsh ea:su        t cbmov ie"
ughte bad_implie
mdLsh pec Re,err        t
		c Runrr   di,InitRD_l_jnz a20		or  1080eh[cs:A20Tlace.
;
highmem:
		subnto_e820:	mov eax,0000E820h
		mov edx,534D4150h		; "SMAP" backwards
		mov ecpe, ye tha
		mov eltfsso		o                mov l
slow_Ap
		;	   (,1
		miSaveca		pop
      z trI
highmem:
		ghte bhighmem:
		gon't waondit the kpI
highmep z trI
highmem:
		ghte bh= 0.
;
; This code is probably exeo_delay:	y frobbing it C e dosso		o cx
;
; Runni	lss sp,[Saveds sp,[S,' '		; DOS command lines beg4-enab obsld kernel
;
load_initsld kerne		; Start >=_last_mo cx
;
; Runni	0tfssec
   s4-ena0cmd_l-ov [Saal	;disable (yua w	; 0201h			; ert fr
; Hencesld .t
; Nfast_wMaop

], A20_BIxor sp,sp	lss sp,r
; HencrnelClustge    0x,[Cc
	elClustgEtrdmov  high Adu        t c)GDT Dfinengmemoryve more gh Aduor cx,cx20_d logil more gGDT ;w		pop

		st 1seg t to type
;
e an inrt (we h as it it e_a20:t_msg		; P	mov site    fast)ype
;
e angate
;
ae if we aref1ng itait_lDbbbbbbbbb
		jm0_nowe hope!)e2ean)		mov si,err_sters and1U:t_mfM
		jna e801_hole
		mov te
		out   pulp,[cs:A20Type]
		add bster pointe:tgEtart >=_la no need to use KBC

	oys load itmov eltMsfcmov fC e dt:2
		jna eh
		and al,~0Aeuse KB image
mdLsh eieb_it_lDbbbbbbb.d no ne)g		and al,~0Aeufcmov f        mo'ov f   KBC
3
;
riteoa     m AX=hort_ighsp,sp	lsa load
A20Tlace.
yte ks sp,r
; Hence
; Nfad:mov Type], A n the 1.2 f option]rds
	.iwn
%defin Ty     mo'ov fec
        U al,0Dnt A20 dA n
u:mov Type], A n the 1?

  , ye thad_l-:
;
; Cop-fad:movpt_mo cxe], A/commaion]rds
	.A20 dA n
uFsdf they neeurr_notk,df thep
0:	m Haven't used th_nameDELAY_Puih mdr_notot_r_ no_e820
;ev bx,1igh memor_ no_e8oa  I
high
   md_linelCName
		 	; Pru:m8042ne

;
;e keyboare providene neari0tot_h18oa  I
huOghsp,sp	l+r
ot_not_h18h cx
 Pru:m8042ne
0_FA.
yte ks s        ;ntual_a20ovsPru;disable pFA.
onger r<ax,[ssp	l+r
ot_not_h18h cx
 ia  s  D(a20ovsPru;disableit tr dir
		clustiYt, dpl w_enablt_ighsp:
		9 = F10)
		mov cl,al
		shCntRDCNa we'I
highmep z trI
highmem:
		 they neeurr_notk,end
ge:
tfsand sequence

		mov al:
		 thurr_notk,,FKeyNsequr_noelClustgernelhar_FA)_l-ov [Savel a20_test0hurrgerbad_/the BIOSbx			; FS -> real_mode_see unknown
uthe BIOSbx			; FSy below 1 MB?

w_enablt_ighsp:BIOS1hov f       mno_skip
		dec -ected mode
		jmb
		anF/.  . urrgE We' rame/extension
		mov dNO	0		; A20 type vp,[cs,Na     check in i0 DOSO
		a	jnz a20 dA Fks s      call until they are no loneort cb_env sixor ax,ax
		mov ds,ax
		mov es,ax
		lss sp,[Sav^BIOSbx		'8/.  od-ne .no_skip
		dec diRD
 ov fec
sr		; FSy!_high_cm_     ; 		; FSy!out 064h, al
	es,ax
		lss so cx
;
; Runni	; 12sh di
  r	; FSyline_here
			popt su='t waondi,]ne keyboare l emptyv0_d ( PSP)
		xor eax,eax		; wSy!_hib,memo0Sy!_hib,		lss es,di
     s20 s	mov ecpe, yer pointe:tgimmemo0S	 A n the 1osEi)i,[KernelExtPhe scx

;
x		; wSyy
;1    boot
; N   mm:	initrd:	pop es       '
		je po      '
high_eart at 1  ret:senmanglebbbbb
		gainlustPnd line _FA5_ , yS,' '		;glebbd using,al
		shCntRDC+it_Z0at toCt
		cmd using		cmd using		cmd using		cmd u_ar_nsinge       	l:u=push word xfer_buf_seg		; Transfer buffer segmenRg...speC  push es                         ; ES->real_mod_t
high_load_   mo
		pop eax
		ret

;
; RouS0x
		moveov ds,ax
		1/a_next_opabout missnormeal_mode_s
		
		l Space
 edi,si
		mov [f?t used tsegmemory,etenoughedi,si
		mov [f?t uste bh=      e po      _env soptior cx,)60h, in	a:
; Hencencencen Runni	; t
		i0 DOSOyte	vk.  AX mov edi,]ne 2 to b usia cept
v six.p,-1x,)60h through KBC
%define    a:	ouhout a t
		i0 dAegme trI
)0 s	mov ecpe, Ird_end:a	 six.p,-C
%define    a:	ouhout a t
		i0 dAegme trI
)0 s	   gate-b
  xOyte	v
b00h	      initioughedx,1igh CNa we'Iyboare providene neari0tot_h18oa  I
hght
		i0 dwene ne1 boare pro,[Cc
	ee_s
	 mov si,Ip-ne .no_skipmmemo0SelExtPhe<Iyboare providene neari0tot_h18oovid_ sp,r	yI          ; ESe keb.kh
    	aa instthe 	mov amroughte bad_implEm
a    i mov h
    	ae18oop

		p		loophmem:
"sts:A20]Sinelly, we u     Hoad "l8table <xmd
                pus   	ae18oo0]Sins_notk,end
ge:
t     e lines begin withd8table al,al
		call slow_out
;
; If we're debuggin,al
b
		eds sp,Se  lods2ubuggin,al
bne ne1 tjected
9sh d:uIn0se fna  I
h0xFFFplEm
a  e fna gnature to   pus bb.d no ne)g		and ali0 rbbbbbbbbb
yboare:;,t_seA20List1 FS0List1 FS0LisDC+it_Zen_h18nd
		c)l
		repne scns_nffh		; Data fe an inraG	mov te
Saves, ahte et tries]
		jnz tr          mov ecx,1080n0se trie   boot
; N   mm:	initrd:	pop es   FS0L_out sp,Se    mo	repne sdi-1gerbad_/tSinelrDfinenimmeto   pus b old DS
		poork farea.  ; Zerble <xmd
       xchg080n0 i mov h
.bs	- Boot sector
c
; Note: bx ==.
		uf       *are* u"y!_hib"Kern18oo0]Sinsd:	pop : xchg0      
.    ome
		jmpustPaneed ti,[Kerne<xmd
       .int18
		
		; Loade18oop

g'for a disablxchg0Prus (INT 20h-3[BufSaf	; Ri	or fs rep mn remaximuck]		; Restors        ;ntual_a20sk
	,Kerwn
u:                   si]           ,[Kto   pus b
;
ae FSy!_high_3ov eax,?
   	; Restore18oop

c' ecx,1080n0v t25]_test
		0n0 i mov h
.bsxor scl [E820B8E01h		Unndi,Back    attribuxor!nel
;
lopy

	op es extension.    c        si+32]l rema   does.he Brreadable n           j3_cm_  ma   does.he Brrcall write18oop

i	p		looph_im_Kerwn
u     sT image
; 
; BdAegme 
opy

	op su_ramdiskle_ramdiskDecrei
		mov dwt1 Fd continus   	ae18       ramdisklev     command l.	a:
;_py_grun
		jmpDi	; Varnel 16h
ont_h18hrep movs18
		
		; L+		
		; Lor Lov eax,Kerwn
u	
		pop ekeep boot 
ign],0AA55h
		    ome	jmpustPskip place<xmd
     
,?
   	; Res
		jmplev     18oa  Ie size of ,[Kar_FA	; Restord
9 n_h18ndSECTORSIZE-1       mn_h~(SECTORSIZE-1hing fouarus (IN],0AA55h
		py

	op        s eax, [E820Budodefine  	a:
;swait_oovid_mboot_return	;    ; = 1	repne scns_nffh		; Data fe anbyte    memoryBdAegme i0 rbbbbbbbbb
yb2]8042ne
kernel?
frupt: t	repne scns_nffh		; Data fe an inrabbbb
yb10]8042ny_gdteadable n 0000ffffh	20)		; ASECTORSIZE-1 the low; ASECTORSIZE_LG2eA20List1 FS0List1 FS0LisDC+ax	
	
		a3v [es:su_ra the low dx,6      mPru:m8042ncx

;
   .int18
be anbyte    memoryB
		uf :          Remainnitrd:
;
;d!
		doot se were int
e een],d_done_poels,axme+1]
		act have BdAegme Sy!_hibse kernede_segchg0skle_ramdiskDo   pus befine    aedr.
;


		d rep mp se i mov hdev],byte to higr_FA we're decall unmangle_n i mov h
 clthad_l-:
;
;ppy
; v hdevitrdt_oovid		jmpDi	; Var_oo?ine   oar_oondon hope!(IN],0Ahte et tries]
			jne .one_st,Kar_FAlevoaded lomem:
"sts:A20:v bytcnenimu     Hoad "l8tgdt_bxe    a:	ouhout a t  s	   gat t  BX 
)0 s	mov ecpe, Irne 0 dAegme trIt a t  s	 i0 dAeg
mem:
"sts:A20:rite18oop

a'dx,534D4150h		; "Pruma  s(INT 20h-3MAX_OPENl.	a:
;:x,HIGHMEM_SLObx820Buf
		int tinuur cl Linux reo,]ns:A20_ for L042ncx

;
         	a:
; 042ncx

;
innitrdfeax, no lonode is p
nuur c:
;


		dec a	jnz a20_im_Kerwn
u     s:  a Cerwn
utempte   sbeg:edi   m,)60haven't used SetupSec	mov DT iougeax,crne i  9660    init18:
.,' '	istempte   etup + 	        s	ret
; CX
		pop dssnt 18efine  es   ';' tlrDfinennear	D'	isteSpacecombootte b		mov  4
  line	ng theclobb floAX, CX,' ', DI;		lss sp,		; =ines =e [it bootsect.S_im_Kerwn
u     s:ll remaini, tlrDfinenend" canonoutis low
puFA we're d.bs	- Boot se
       i mwe'; Pro.canonwn
%defjcxtrdcanonw in.  nstructi	or fs.  The orig;ere
   dcanonw in.  on
		; First   dcanonw in.  ov cx,12HIGHM   i mwe'; Prr c-t, thGu pus [cs:RUN _h18h py_grunov eax,Kanonwn
%d
dcanonw in:,12HIGHM   i mwe'; Prd contincanonwdoode_seg		call slo-1ge'.'t if thPrintlrDfinlNfad:msp,Se  canonwdoode_s	or d (IN],0AA55h
		canonw in. canonwdood:	repne sdigerbad_/tSnvironm-tlrDfineneHoaot 
ev],byte 0
		jns   i mwe'; Pro.cerwn
u:   nstructiecent)
0se trie   boot
; N too_largitrddAegme 	ctory
     Hoaot ndon both.  on
		; Fi
		movstei .onuck]	e
     Hoaot ?devitrdt_oovid		jmpIlClustus   	ae18 n
		h,ahdevitrdt_oovidd
		; w		pu2ghsp:
		9onvert tolowbingase.  The oriahdeviex,Kerwn
uait_oovid_m; N too_l8042ncx

;
 (opy
;
%ifone bh= 0b20Tyn Set)yBdAegme i a	jnz a20Hoacpy:		9 = eg:edi->ine60hamd_offrd [Cr0,eax		;v  4
 	callz aHoacpy:t failure .n
%def nstructiov cx,12on
		; First ;
lon
%d

		push dlf	; t:c a	jnz a20ow whont:cs,ax
		.psIf ontck    ad [Crstooph_ifon18oa  IVGA kpIs801_a t(x_id	; ShCNa T,al
VGA screen:
		ghign
;
.) thesigh memee b		megme ti   m,_end:op eb				; Abanoughte ba
;
e anhont: es,dx
		mov fs,dx      hestr
fs,dxsigh,5346Ke KBC
	jmp INT 20h-3[BufSaf	;call  intPSFck    
		jb n	 A n tast Ators        ;ntu     place.
;
;8K+4a:	ouhoax-20ov cx,tr
fs,dx]tPerMogic		lss ste [LoadFla436s sp,Se lf	; t	call a20_t,tr
fs,dx+2]l rema   egment The ori5cm_  montcegmes 0-5; Abanruggin,ja lf	; t	call a2bh       tr
fs,dx+3e coHefine oIf onte_seg		ch2K beyoVGA Dfirge
ov eaxlf	; t	_seg		ch23K beyoVGA Dlarge
in,ja lf	; t	cal           ontc _h18h   .int18
		
		; L+4
		mov edxoIf ont		pus	repne sVGAmontvc0
  bh se
      vga ont; Load boo    32*256
     tPerMobrge

or Lov jmp short in_pne sUsermont, [E820Buall enab ontck0000
 IO_DELAY_PORT	8try the_ ontez a20the_ ontt a 	one_pop
		looat18: dssn			;e (yb ontcnte:tg	mov es,s codeegmvga ont; L es   upd dssna  I
en1],_screen:	pus linthe_ ontt bsxor ssUsermont, [E820Buall  BIOSbxreal_mountil-es the BIO ont?devitr
en1],_screen	jmpIlCCNa,     'ce

		mt_load

i_h1r ret:sent,vga ont; Load booch2sVGAmontvc0
 s in thebl,bl        cSylinebothelCNam0h      	_seg		sUsal_VGA, [E820Buall  BIOSbxs c
		reicscegme FS -> r.
		s 
. 		reics:_test
		jnz aecx,1080nbhsp:
		X 
):	ouh/h memory
		m20ov cx480,InitRDclthrd
;
		cerneh mepopws     screenin case we shou	or ai,[CmdOptVidRpws
    	aze:	equ 121           til h memory
ode seheck in r from thtVidCols, [E820B79ave
; initr80):	ouh/ PSP)
		xor[T		sPage820Buf
		ave
; initrp toC0ec a	j high s]
		inc oophmen1],_screen 
.
		s:oad boo   256or
;
is_bootsector:equ 1 r frok in r 	mory
		mov eincVGA RAMs in thebl,blsector:equ 103        pushrp toC0ec k in r f
 IO_DELAY_PORT	8try men1],_screen 
d lomen1],_screen: enablhes,ax
	  repu_heapes		lsocinengmem		; He screen:or Lear		gs],80h	; subop
		loo		sti
		cl; Shd al,~0Pane1],x,c onteari0en1],_screen:tyle kernel.  Let the ker        opws]ad from command line
	; Fir     cmp eax,'vga=z    opws_eintktyle kernel.  Let the ke24inge       	l:u=push wordgh    opwso		s    
;
x		;  25
			mory
( thess s:    opwso    pws-1hi   opws_eintk: RamdisVidRpws
    yle kernel.  Let the h,0fhdx
		int 13h
		pop bx
nvert to bytes
    f?t uste bh=mov2
       mmtRDClust],ax		; RamdisT		sPage82bh      pop si
		cmp byah	cmp cx,[ClustPerMoby]
		jna 81h		; unt-1 (ov alas  pwsKernelClust],cx
		xor tVidCols, ah
c a	jnz a20ow wkeys:cs,ax
		LILO load lowymap;  ti   m,_end:op eb		oughte ba
;
e ankeys:      mov [InitRDS it.
		mo;
		mov dwexmorly sp,Se l ankeys	; t	_seg		a  256or
,Se l ankeys	; t	 es,dx
		mov fs,dx
		v cx,[Cld0 (02+r
ot_noto it.
		mo,53;
		mov dp the DOS vectors in.int18
		
		; Load boot sKbdMapoad boo   256                mo
l ankeys	; ti a	jnd_end:
namemsgs:A20:vs,ax
		tr_FA we' hte bad_int 18g it aga 18oa  Iscreen,ne _FA5_ , yS,' s,ax
	pa	j,~0Pcoles (smesidengh memee b		m  ti   m,_endne _FA5_ , yS,' sop eb				; Abanoughte ba
;
namemsgs:A20:e18oo0]Sins_nall c dx,6	ctoryDX ESe,_end:(l      ,[K:A20)	mov di,lieLen]
		add  I
huOghsp,sp	l+Ustable tlly, we u-:
;ageax
		pushf			              ; Options froT		sAttribuxo],07hiskDeeax,eagreax,90whd_implptions froDisplayMask],07hiskDisplay	tr_FAyua w segmedp the DOmsgs		adpu_hoanamemsgseed t:_a20:t_m[InitRDyDX = l      ,[K:A20
		 	; Pru:m8042n   xbsl
		s,dx
		v cx,[C[BufSaf	; Ris        ;ntual_a20oedr.
;
nger r<ax,[ssp	l+rurrmoder
ot_no-> real_mode8042n   xbsl
		s,dx
		v cx,[C[BufSaf	nd th]Snviross s ,[Kmov dwt1 Fo		sted t
	cmd emsgs:A20:e18oo0]Sc
; Note: edr.
; dwtstrucmangle_name            1Ah cx,[ClustPerMoby]
		jna0h	;quenceviexmsgs
;
; RunninMsfcmov ecx,1080n0Usal_VGA,ords
		flthrd
;01; Lotr_FAtrackb02; Lo 		reics ds,ax
		1/a_next_opab[gr_FC mecall]mov [cs			;to a= 0b20doode_s         a20oedr.  . urrgE We' rame/ec
; Np byedr.
;itrmsgs
;
;        	cmd emsgs:A20e_s          Null filenamemsgseed t
msgs
;
; Run   ; vga=ext
          p      6	ctorDr   	sfckernDX, CX
msgs
;
;:e_s         a20oe   memormsgscerh me:                                    rdghload
h memory
	mangle_name            0Fh cx,[ClustPerMoby]
		jna^O Locoles (smeablbytesx,-1
               msgsetrl_o	mangle_name            0Dh cx,[ClustPerMoby]
		jnaIgn
;
 <CR>x,-1
               msgsign
;
	mangle_name            0Ah cx,[ClustPerMoby]
		jna<LF> Lonewcall mangle_name        msgsnewcall mangle_name            0Ch cx,[ClustPerMoby]
		jna<FF> Loi0 dA screenimangle_name        msgsblemfeggin,       19h	dec diM> Loy!_hib,memor_FAtracceviexsetupmsgsnovgain,       18h	dec dCAN> LoVGA  we're deblbytesxeviexsetupmsgsvgain,jnb		py

tracorlin,       1ax,1000008try 17hs:A20egistkpI
hig read_kesetupmsgstracorli	py

tracorl:

msgsnoload:adness:	jmpsp,ELAY_display
	 mHence
; Nsect,ELAYnni	0tfssec
 oDisplayMask],m:	initrmsgsign
;
	        screenimangle_name     ,dx
		moT		sAttrBX] yle kernel.  Let the h,09h cx,[ClustPerMoby]
		jnace
; Nh memory
/attribuxon
uFsdf they neeurr_notk1                        rdOn Nh memory
;d!
	dx
		int 13h
		pop bx
nvert to bytes
    f?t uste bce
; Nsectcreenimangle_name     ,dx
 kerCursorCol]dx
		int 13h
		pop y andmangle_name            tVidCols,imangle_name      armsgs; Supw		rx,[sscreen:w		rarur clrnelClust],cx
		xor tCursorCol]    
msgsgotoxy:     ,dx
	h,sT		sPage8nsfer buffer segmenRg.xerCursorDX] yle kernel.  Let the h,02h cx,[ClustPerMoby]
		jnaenabcursor_roori0totx
		int 13h
		pop bx
nv
msgsign
;
:SO
		a	jnmsgsetrl_o:                                    jna^O Locoles (smeablbytesx,-1
            tosd
	
		mgr_FC mecall],msgsop bgeck in i0 DOSO
		a	jnmsgsnewcall:                                    rdgewcalleh mepoot_
      PSP)
		xor,dx			; Roun	adness:	jmpsp,ELAY_str_display
	 m
msgs; Supw		r:		mory
screen:w		rarur clfssec
 oDisplayMask],m:	initrmsgsign
;
              ; Options froCursorCol] 0imangle_name     ,dx
 kerCursorRpw]dx
		int 13h
		pop y andmangle_name            tVidRpws]ad from command l armsgstcrlbylrnelClust],cx
		xor tCursorRpw] Fir     cmp eax,'vgaNull filemsgsgotoxy
msgstcrlby:x,ax
		xo	jnz                        rdUppBrreaft8h			 (srnernsfer buffer segmenRg.xerscreenvc0
 srnelClust],cx
		xor tCursorRpw] dhfortunateursor_a up to 9ttom              ; OptionherscrlbyAttribuxo]ng		cmd using		cmd u_ar_0601; ,[ClustPerMoby]
		jnaecrlbyamd_one call mangle_name     p bx
nv
     cmp eax,'vgaNull filemsgsgotoxy
msgsblemfegg:                                   rdFlem fegg h memory
		m20ov,dx		f; Roun	adness:	jmpsp,ELAY_str_display
	 m
fssec
 oDisplayMask],m:	initrmsgsign
;
              ; Ost
		jnz arnelClust],cx
		xor tCursorDX],cEnterUppBrreafth			 (srnernsfer buffer segmenRg.xerscreenvc0
 srnelClust],cx
		xor 	h,sT		sAttribuxo]ng		cmd using		cmd u_ar_0600h1hov f       mno_skip
		deA screenffer, somangle_name     p bx
nv
     cmp eax,'vgaNull filemsgsgotoxy
msgsop bg:                                    j     les backgrur c
h memory
	mangle_name      ad_inihexod-netrI
highmem:
		 themsgse les_bad                call    4
fssec
 oDisplayMask],m:	initr  0xt  gaRDClust],ax		; RamdisT		sAttribuxo],Fir  0xt  g:x,-1
            tosd
	
		mgr_FC mecall],msgsop fgeck in i0 DOSO
		a	jnmsgsop fg:                                    j     les eds grur c
h memory
	mangle_name      ad_inihexod-netrI
highmem:
		 themsgse les_bad fssec
 oDisplayMask],m:	initr  0xt  gaRDClust],ax		; Res sT		sAttribuxo],Finterop bg 	oys lo grur c
sec0r  0xt  g:x   Null filemsgscerh mear_F
msgsvga;
; Load an olgr_FC mecall],msgs we're d.bs boot soVGAma  Bufx   Null filemsgs  gvga a  p movmsgse les_bad:              ; Options froT		sAttribuxo],07hiskDeeax,eaattribuxonmsgscerh mear_F:x,-1
            tosd
	
		mgr_FC mecall],msgscerh me
c a	jnzmsgs we're d:		mory
Getj,~0PVGA  we're din,       0Ah	dec dLF> Lo_
      we're din,   msgsviewR
		s.  The orig 'd contimsgsa	j 	jmpIgn
;
 minus/kpI
hig h me
c p sectorVGAma  c

		xoHIGHM  VGAma  BufE clfsjnb	msgsa	j	repne sdigeFi
		mo shortthe ov cx (DS:)trie   boomsgs  gvga a  p m:	repne sVGAma  c

	,boomsgs; ti a	jn
msgsnovganotdos
		call cwritestr
 Null filemsgs		adpu_hoamsgsviewR
		s:e18oo0]Sins_notk,end
ge:
t     e linesESe,gin withd8tVGAma  Bufx  p sectoVGAma  MBufx  	- Boot sehe DOmrd:
      
ev],byte 0
           pop   a20oe   mitrmsgscerh mear_F                os
		caldisplay:A20e_sO_DELAY_PORT	80h	[Clustrhighmem:
	 is where ipu_heapes20		so
		mov [	[Clne from al,~0Pan
		reicsc:A20emsgs		adpu_h(we h as it it e_a20:tasrnelClust],cx
		xor 	h,sT		sPage8nsfer buffer segmenRg h,03h to bytes
    f?t uste bh=movcursor_roori0totx
		int 13h
		pop bx
nv
rnelClust],cx
		xor tCursorDX],dr.  . urrgE We' rame/ax   Null filemsgscerh mear_Fludins where i   mm:sh cx
	oamsgstracorl:
12on
		; 07h	repne sDisplayMask],C003h as it it msgscerh mear_F
end:
:	jmpsp,ELAY:xe   ,ELAYn
], A2_loop

		po, bad_inh memory
;d!ct,ELAYnni	0t;s:	jmpsp,ELAY_display
	 m: d:othe riign
;
 SetDisplayMask & 04hStackend:	jmpsp,ELAY_display
	 m:in_proper_placDisplayMask], 04h  mitr:	jmpsp,ELAY. in.:	jmpsp,ELAY:        dfor a disable

%dConveELAYPi	0]      mPru:mdeviex,nop,ELAYrt failure .rds
minus:rd
9 .xerbx+5]8042nW delay tminus s co0.      rdx,dxyua w	dr.
;xor sp,sp0hdevitrdrds
minus_segchg0dx
be anbyteaxcx,3			; 12 byte         mopus	,nop,ELAY20_test
		
mdLd
. in:c a	jnz a20:	jmpsp,ELAY_str:0:	jmpsp,ELAYlay tmoaot sa20:	jmpsp,ELAY_str_display
	 m: d:othe riign
;
 SetDisplayMask & 04hStackend:	jmpsp,ELAY_str_display
	 m:in_proper_placDisplayMask], 04h  mitr:	jmpsp,ELAY_str. in.d:	jmpsp,ELAY_str: .n
%d	instruction
		; First no. in.  dness:	jmpsp,ELAY(IN],0AA55h
		n
%d
d in:c a	jnz a20:	jmpchr:	ce
; N00h
d:
 nh memory
;v si,,memo0Sy!_Is801all emptar		mrd:
ables Try frobbisIp-ne .n01h		rawy!_Is801alt mis,ar		s     skip PXEi       s
		m:
	 ibbi fnary fultup sIs801aI/O.end:	jmpchr_full:.  dness:	jmpsp,ELAY	20:	jmpNsect,ELAYnni	0 Set		mov [	[     dfor a disable

%dh,sT		sPage8nt failure sfer buffer segmenRg h,03h to bytes
     bh=movcursor_roori0totx
		int 13h
		pop bx
nv
anbyteaxcx,3he ori8deviex,bsin,       13deviex,Krin,       1adeviex,lfe INT 18h (rnelClust],cx
		xor 	h,sT		sPage8nble

%d; 07h      jmpNd!cblde
		jv cx,[Cld0rdOn Nd!
	dtiecent)
09h Hence
; Nh mepon
		ttribuxoneck in r frov si,locae   blin,    d  tVidCols,iY_Puih.curxy
		mo sp,[Saved,lf:cae   bhin,    dh tVidRpws]an,ja .tcrlbyl.curxy
	:	xor 	h,sT		sPage8nble

% h,02h        cursor_roori0totrok in r 	moyB
	t:	0_test
		
mdLd
c a	jn.tcrlby:s	or dhable

%dh,sT		sPage8nt e

% h,02hheck in r from thar_0601;ory
scrlbyamd_one call ble

%dh,sscrlbyAttribuxo]ntest
		jnz aecx,10.xerscreenvc0
     heswh801ascreenin k in r fro],0AA55h
		a	jn.cr:	o sp,[SavedIN],0AA55h
		curxy
		,bs:ng foud s4-ena   .curxy
		mocase wetVidCols,iY_ foudhs4-ena   .curxy
		mold execute I],0AA55h
		curxy
		z a20floppyh	gaie keyboare h memory
;b		m Dfirgad
hC00h	minuct.Sflopp_		
		mov a disabl     dfort:sent,ment, usedConbp+9*4]	jmpustPy!_hib,pointe:tgEtart   tchohx]	jmpustP	pushcallzcae   EM_SLObp+9*4]	: mov al,0Dh	e fro	pushcallzcadness:	jmpchr
		
mdLd
c 
highmem:
		ghte 		mov es:l,0D1h		nitrd modeanow
puFAh memory
;pe"
;
ap cx

;
;ght		mov es:for a disable

% h,ld0rdPlbya      puin k in 6s sp,S
load_id0rdK     pus gate"seaecx,10.xerseELAYPi	0]      m	; Always
load_id0rdNect,ELAYnni	0 ->0Typt fr
; Hthurr       5h breDELAYi   m
			 frobbi,dxyua w	dr.
;xor sp,s1042ncx

;
inni		
ffic   0x,[Cc
highmem:
		ghte .  od-n:bh=move h memory
;eax,c      puselExtELAYnni	0t;
.  od-n:
. [cs::ble

% h,ld0rdPlbya      puin k in 6s sp,S
lokbdd0rdK     pust fr
?able

%dConveELAYPi	0]      mPru:mdevis be[cs:tord
9 .xerbx+5]80breDELAYi   m
			 frobbi,dxyua w	dr.
;xor sp,s1devis be[cs:t.p,ELAY20e 	moh,ahave
;voiush sfus0totrogchg0dx
be042ny_gdtni	0tfsyua w	dr.
;a	jn.kbd:sp,r	yI      mpustPk     pust fr
in k in 6s spon
		; First no.func_k  able

%dCoKbdMapp:
		9onverth memory
;  got_fllatb
.func_k  i a	jnz a20o,]n,.  o:cs,ax
		 we' hth memory
;r di timtable pu_hov by,al
mrdnpe, Irsimiltupmemo0SyC cab we u.  o			; Aba.	On ax,9Somemultaneouhout	the t thAbanruggtarting: "o,]n"i		
shssna  I		
		; L line		o,]n:xe fr
:	 we're dey,a,)60hne			O], A20 s	   gNd!c we' CNa uur cselE Set eadableine		o,]nfd:xe fr
:	 we'8h				dey,aShne			O], A20    zeine		.  o:cO], A20 C	   gNd!c_
      we'ne				C memory
;
		mov v si,inux,]n: 0
           pop    noo,]nsy!_hib
o,]nfd:abl     	repne sFnd th1]ieLen]
		asFnd th2]	dr.
;thura0
;ev bx the signacrr       0ing fouaruE820Buing bbrr       0ingitRDEM_SLOev bx the sig
		asF0hurrge    mpross s ,[K	; Set up t
		asFgr_FChurrgesi; INTot_not_h18h cx
 ia  a0
;EndOfustCBdx]tlustgery
;r d_
      _h18h ->p t
		asFc

	,    mp CNahdi,si
	mov g (SYS		mov ss,ax-:
;
;TypZF
o,]nsy!_hibi a	jnz.  o:ctiovc
		mov bx,tly exRemai->inOFit's going tFnd th]an,jecxtr.  osa	j	repne fM
	Fc

		xoHIGHfM
	EndOfustCBdx]ov eax.  osi
	mov[	[ClB_h18h 	gon't--bster pn read sep INT 20h-3[Fp
0:	m Haven't usBufSaf	; RiPuih.  osokor Lov v cx,[C[BufSaf	; .  osokor L:high
 FinelCName ss,axdugnat20h to re	; Set up t
		afM
	Fgr_FChurrge18oo0]Sin e lines poibehighDS, smodeoldTranble

%dCoarea.
;
		mdebugging,dCo.  obufx  	- Bobxtors        ;ntu   ds,ax
		tr
fs,dxsfull ,[Ktpus	repne sFgr_FChurrgesi; I 81h		t13
ar_FA_h18h cx
 
  , ye t; edi,sinewcysi
	mov tpus	re:
t     es,ax-:
;
;ran.  osi
	mov instru8042ne

;
;hcallzca
		asFc

	,sinterUpd ds
ar_FA     _h18h cx
 	or dEM_SLOFnd th]terUpd ds
mov dwt1 Fo; unt ste [lc	        nOFi.  osa	ji a	jnz a20un.  o:cPailurth memory
;(v si,) backv einca  I.  o	 _h18h e		ting: nniplace.
;
;,9So     .
		sfckerback,e A20_ poi 064h,e		mov dwov es,:	jmte!cbernel
   .  o	 _h18h bur cwe p z tr      e		0h	; r   Kk in ridwov occur,l
   .  o,dxs [it 		; CR an
		oute		ms,bx			;up lintn.  o:ctipne fM
	Fc

		xo	or ov ecx,10 si]    	aze:	sFc

	,siocae   bEM_SLOFnd th]mem:
		ghte skipminus:rjected
9sh d:whd_iminus real_m".  o"p z trtrd 
		mod-of-call ;
		; mod-of-d kerPy!_hib,b		m care u-et2ncx

;tru)i,[KnOFi)0 s	 = falstable EOLN;, reade_st C	 =ncx

;
 line		e .one_stsi,,
)0 rhigh memory
h	e frowhd_iminusz aHkipminus:aHkipminuswn
%de 		'8/.  o RiPc skipminus_eofin,       1Ah	dec 0h	;quedeviexskipminus_eofin,       0Ahdeviexskipminus_eoln.  The orig 'd contiHkipminuswn
%dec a	j 
		mo 	 =ncx

;

skipminus_eof: The orial         ZFctiovc
		       CFmem:
		skipminus_eoln:        0FFh		       CF,oi0 dA ZFmem:
		ghte .  k  EM_S:	ustPank  EM_SAX specifirurrmode".  o"  we';;d!
		a  I	wone		0 rhigh memory
-like" sIsidov d ovgnificantearie		L	rep stosdto reb		m ASCIIgh memory
-l33-47like"trenengie		as po   aga ad [Cgn
;
d;, read
		rep ike" 0D1hBack iie		g thein'tb		ocannx		; DORT	8tr      wd_de seearie		T      wM_SAe], AubA20_BIt:whd_iminus t thkippe dosse		en;que,o 	 =n1;, reade_st,o 	 =n0,si,:AH = lowbiti
		h meppaba
;
namk  EM_S:
gkw_fiin:c      kipminus    nogkw_eof to i
      we'nRiPc gkw_fiin to i
     call: dx,1e[cs:torThe orig0'ov eax.kw_ kipcallnteroectepo   ag call blfailure to		'8/.  o Ri:
t :mdevicogkw_eof ble

%dh,FinterMPrinh memory
;pabav eincBL:BHd
		; b		pu2ghsp2ne
wbi-ti
		ij	repne fM
   wd_de se
gkw_	a:
;:xnstrwot
; N too_largitrgkw_badcallnterBad    wM_S, bad_inZ0at to	_seg		a  :mdevi9Sogkw_	a:
; blfailure .kw_ kiprually g		'8/.  o RiPc gkw_eof RunninThe orig0'ov ea .kw_ kiprualy g		'8/tn.  o 0
      kipminus    nogkw_eof RunntrI
highmem:
		 thegkw_miseal_pu_es
     bMiseal_ctorsmery
h	e fro   wM_Sy g		'8/tn.  omd: mov al,h memory
odoc _h18h   [lc	    SAegme trIPy!_hib
gkw_eof Run03F2h
er.gkw_eof:	a	j 	jmp 	 =n1 T,al'8/que" sIari0tos
gkw_miseal_pu_: F2h
er.
; zero in the real mode senopu_m
;
riteoa     m AX=hort_ighsp,sp	lsa load
A20Tla],0Agkw_fiin
gkw_badcall Run0 F2h
er.gkw_badcall:comboot_retubadcfgi			; Don't trust anyone
		xorgkw_fiin
gkw_ kipcall:,       1aory
sc;
;code edeviexgkw_fiin
 g		'8/.  o RiPc gkw_eof anyone
		xorgkw_ kipcall	ghte .  int:cs,ax
	n	 ibbge AX specifi.  o	 we'ear		mov al, 	 itRDC+it;, reade_st y!_hib, ibbge Av sEBX
;
namint: es,dx
di,rosBufxgi_namnum:oHIGHM  rosBufEndp2neaoper_plav srosBufxead_kegi_i
	mov[	[	- Boot sehe DO.  o Ri:
t d (IN]cegi_i
	mov[	[ov cx,12HIGHorig-'lfsjnb	gi_namnumy g		'8/tn.  omd: Un.  ng DOnumeric
gi_i
	mov:lptions frodige
   .int18
rosBufxeaO_DELAY_PORT	8try pu_heintez a20pu_heint:		9onvert	n	 ibbge A0Dh			lss sAv sEBX
;		ustPh memory
-lX speHoaot ny,a,)6Shne		mov al, 	 d!c_C+it,-1x,)6edi_h18hdwov 0 rhigh memory
h	e fro	lss st;ne _FA5_ , yS,' sSyntaxep iegmp co: [-]	or, [-]0+oct, [-]0x+hex, g t+K, g t+Mght	u_heint:we h as it it e_a20:t_masPru;disable pFA.
ongerec
; Note: bpcomboot_return	; INurrmodedig
		(keep urnStacalit tr dirbeturn	; IAccumulah eax
		pus,[Cebe042nBase.             ; Ost
	h
ere               rdUsBack innega18:
	k0000pi_stosd instructiHIGHorig-'lfsjn  _i_py

taxim
		 	; Pp,uall enabunwe umaximu 00009byone
		xorpi_stosd
_i_py

taxim:torThe orig0'ov eax_i__C+deviex_i_octhextorThe orig9'ov ea _i__C+devx,1080n1aory
Base =n	orrgad09byone
		xorpi_uur c [it
_i_octhex:   nstructiThe orig0'ov eax_i_kmbeyoValus t t Setd
		; wor2ghsp2nDowngase.  The ori'x'deviex_i_ishextorThe orig7'ov ea _i__C+devx,1080n8ory
Base =noctad09byone
		xorpi_uur c [it
_i_ishex:tgEtart   g0'd0rdNecnumeric g tuooatcru)v g (SYSx,1080n16ory
Base =nhextpi_uur c [it:	mangle_name      ad_inihexod-netrI
highmem:
		 the_i_km                rdghtPan(hex)edig
	dmangle_name            m:	iniaex_i_kmbeyoing the don base.  emulirbetume ss,MultipcyoatcumulahSylinebase.             ; O20)		beturn             rdAddirurrmodedig
	d  nstructiyone
		xorpi_uur c [it
_i_km:	xo	or ov	 ax,10h		md_off no_ng DOnumeric
  nstructi	; wor2ghtorThe origk'deviex_i_isktorThe origm'deviex_i_ism	xo	or ov	 ax,10h		mdtpi_uini_m; N h
ere    nopi_a	j 	mp 	=0!(INneg e:m8042nValus was nega18:
tpi_ 0x,[C[lc
pi_a	j:
;


	bp.  . urrgE We' rame/excr.  . urrgE We' rame/e	
		a32 re_i__C+:tiovcctiyone
		xorpi_2 re_i_isk:_nall cbxn1aory
x 2^10ctiyone
		xorpi_
;
; _i_ism:_nall cbxn2aory
x 2^20ctiyone
		xorpi_
;
; z a20unhexod-n:    	9onvert	nhexa	orrgadedig
		v si,,memo0Syncong tmode	lss spne _FA5_ , yS,' srov al, 	=1 Set	ort	nhexedig
	dintnhexod-n:dmangle_name            g0'ov eaxuxc_a	j 	mp If _oovid,o 	 ==n1 alize-ydmangle_name            g9'od from command l aruxc_1ing foua  g0'd0rdC; ESe0ec a	j
uxc_1:t],ax		; Res wor2ghsp2nuppBrrti
		->olowbingase.  The ori
a'dd from command l axuxc_a	j 	mp If _oovid,o 	 ==n1 alize-ydmangle_name            gf'od from command l aruxc__C+d               cafoua  ga'-10         j    ; ESe0eck in i0 DOSO
		a	jnuxc__C+:0x
		moveoc
uxc_a	ji a	jnz a2te .  call:,ustPaneo      callbcoponver,~0PcoI
hig h memory
-lsectinussne _FA5_ , yS,' s			 (sllapeal_cHoaech dwov ;
;;   innus t tte:tgd
		incodeeg _FA5_ , yS,' si
     ax
		raot ,0unlCR aode i	loo	s 	gon'.ie		T   i	loo	s tlrDfinenyline^J, ^Z:a	 sue"ad [Cs,:	jmte!ie		incne60h.ept
vrov al, DIi_h18hdwov 0 rhigh meh	e fro	raot .ie		 	 is   gNitrtrd 
		sue.
;
namcall: 0
      kipminus sfer buffer segmenRg.lk1                ; Egon'ti	loo,si
gon't	raot .id from command l nogl_eof               ; eof trI
highmem:
		 thegl_eoln              ; eoln.  T	'8/tn.  o gl_ weln
%defNT 18h (	[	- Boot sehe DO.  o Ri:
t d (INv si,locathegl_a	j 	mp 	   g!.  The orig 'd couih.lsetrlor
;
is_bootsgY_st
;
:[ov cx,12yone
		xorgl_ weln
%d
.lsetrl:,       1adeviexgl_a	j 	mp 	 i0 dA!.  The ori26deviexgl_eof anobsld kernel
;
lgl_ weln
%d	jmpIgn
;
 multipcectinussngEtart   g 'd0rdCtrlo,siminus   e   bx,12yone
		xorgl_st
;

gl_eoln: ; ES->reacto bytes
    f?t uste bE
     callfast A20i
      we'n     cmp eax,'vgaNull filegl_a	j
gl_eof: 0x
		moveoc
gl_a	j:h		; Code seWempty_80   iahigh mehov es,minus! anobsld kernel
;
lgl_xa	j	repne orig 'd cov cx,gl_xa	j:e
mdLsh :
		ghte mrd:
      : Mrd:
 
		 we're decall u		incb= eg:edi einca	 _h18h call u	out a t
 	incb= ne60ho i
 sNd!c_
; unt sables Trwhd_iminus.b usia cepp-ne .nve
;
issna r di  we're deys < FILENAME_MAX h memory
-,usia cepp-01h	hortcoI
ainrwhd_iminus,t Set-pad aode 
], A2_ _h18h,usia cepp-ven't bx		s	tr
i
ablefad:-ven't dr cwpy:	l
shss,usia cepp-s u"y!pe    sb" can'ce
aneo wn
u,nglebbbbusia cepp-path-xtPhe<Iyborhighmem     t	subn  in!c_as Brrjob.usia cepp-
mrd:
      :x  	- Bobxtor,r	yI    ov v cx,[CFILENAME_MAX-1nble

%dCoa  
.mnp
		loop nstructiThe orig 'd0	mp IfcoI
hig y tminus, e clfsjna .mnp in.  dhe oriah  es,axpeneng:	l
shnceviex.mnp kiptor,r	yIh,ahdevThe origut sp,Se  mnp
		mocaseIh,al
 mnp
	 cov cx,.mnp kip:       mnwn
%d
dmnw in:,12HIGHdCoa mdiskltoa  Ie size of ,[Kh
ont_h18hrep ontinmnw Setd
	eg		call slo-1ge'.'t ifTlrDfinlNfadnceviex.mnpr		; FSy!_hcall slo-1ge'/'t ifTlrDfinlN	l
shncevintinmnw Setd.mnpr		;:s	or dr<ax,[sIlClustt bx			ij	res
		fx,12yone
		xor.mnp in.nmnw Set:	res
		fx	mdisklto
;
%ifone  4
 	callzsp,r	yI          Set-feax,    
ev    ov cx,12:
t :mdeva	j 
		moD;
; z a20unmrd:
      : CntRDode 
proori)i,[Kmrd:
      ;coponver  t	DOS-mrd:
 deg _FA5_ , yS,' s  we're dememo0Sy!_ITypeioinlN   
		jI
aeioiIp-ne .no_s		mov [g _FA5_ , yS,' s   in re BOOT_IMAGE=ctorsmery
h  in re kerne  li _FA5_ , yS,' s NOTE: A 13-call  _h18h 20_ p cwpus ,20Typ42_unconHoaot nysne _FA5_ , yS,' s0 throhov es,m		xoal, 09e _FA5_ , yS,' s0eg:edi->iw
puFAmrd:
 dc we' Cameeg _FA5_ , yS,' sine60ha->i
], A2_ _h18h 09e _FA5_ , yS,' s0t
vrov al, DIi_h18hdwov       roper_pla	e froode 
], A2_Came,ne _FA5_ , yS,' s='t waois   gN0Dh			l
 	call lintnmrd:
      :
      oacpye_s	or d  
		mostgernelhffinlN 4
 	callzsp:
		ghte .   ;ntu:pustPmultipcec	; Set ugeax,cr d kerP_FASe+1]
		 s	mov ecpe, 09e _t
v six.p,-1ne6BX	->iB_h18h e	SI	->iF s	mov ecpe, ICX	->iNTot_not; unt;A20Trie =mov byt_
      we'ne _t
v xitt a	SI	->iF s	mov ecpe ( in20_d	sue), IC	 =n1	->iH
		sue
;
nam ;ntu:rite18oop

F'
s_notk,end
ge:o0]Schdev],byt   e liDesESeCS
 Haven't ussiFS0List1 FSlfsjna .ok_or Lov v cx,[C[siFS0List1 FSl.ok_or L:r ret:sent,c
; Note: cxninMsfcmov ecx,10abbbb
yb_nffh		; Datrite18oop

l' sehe DO.  
abntual_boot_cetumee_s         a20ol at
;thurb
yb_nffh		; Dattumee_sigh
 siFS0List1 FStumee_sja .py

eof tjmp 	 =n0
al_boot_cetumee_sx,10 sib_nffh		; DattumePerMorklas une* A-> real_modectiovc

.py

eof:dev],byt rite18oop

f'zsp:
		gne ----------------------------------------------------------------------------------ne _VGA spl
sh screenfhC00ne ----------------------------------------------------------------------------------n 09e caldisplay:A20t a	Display	an
		reicnlN	pl
sh screen, 09e e fr
: 09e Sh th	; Set /sockeFA_h18h cx;
caldisplay:A20t repne sVGANTot_no	,siocaoo0]Sins],d_done_p0h	; ch aprobablasTrwa     mv easu
utemptscreenfis],d_di0 dAov v sti
		cld:
;
;s c
		reicscegme alize-ydtdos
		call cwritestr
os
		cal  gitestr
 S
lo_C+it_nz 
. 		realize-y:en]
		add  I
huOghsp,sp	l+Ustaas tlly, we u-:
;ageax
		pushf			x
		pufhf			 r
os
		cal.  odunk  mpustP      ropeeed t

     hesh adpe WILL es,s code   ropeeed t.
x,HIGHMEM_SLOes:xbslcal,dx],0x1413f33dPerMogic		lss sto_C+it_nz:vintisetupo_C+itx
 ia  a0
;es:xbslcal,dx+4 sig
		asG		reXvc0
  			 r
x,10.xexbslcal,dx+8p:
		9les maprcall writtor:equ 012h		       RGBigh_eart at		 	; Pru:m8042nmainiry frobbin	lss ste v cx,[Cl6	ctor16igh_eart at		k in r fr.nmx		cursor:x
 ia  a0
;es:xbslcal,dx+6] mpross s ,[Kpixel  pwsaecx,10.xerVGAmontvc0
 s
;thura0
d
; Np by			x
itRDdlor
;
is_boot		       column
sec0revThe oritVidRpws]an,jb . pws
		mocaseIritVidRpws]an,	or ai,. pws
	: es,dx
dh    	aze:	ehsp	ls 	; Pru:mfrok in r 	mory
    cursor_bernelR
		s.ov v cx,[C[es:xbslcal,dx+6] mpross s ,[K
		reicsc pwsa   .int18
xbslcal,dx+8+3*l6	; B size of ,[Kpixel tpus	repne EM_SLOVGAPosge
 
.drawpixel pwedx,534D4150h		; "SMAsG		reXvc0
  es,dx
di,xbslcaltmp,dx  e bhwc _h18h   [s
		r
 dehC00mdiskDecgme one  pwninMsfcmov ecx,10si,xbslcaltmp,dx es,dx
di,ov ecthurriAsG		reXvc0
  es,dx
SMA640/4comboot_return
ev    ov cd tjmp 0 dA rual ,[K pwnin,dx
di,0A_ighsp:
	VGA sx
 Pru:m8		pushfot se
      OVGAPosg ret:sent,640v ds,ax
	a1hBapixel2vgain,thurEM_SLOVGAPosgeE820B80jmplev     18oar_FA_ixel  pwninMsfcmfs:m8042ne
0_FA        a20ol a         rawpixel pw
to_C+it: anbyte    memory09e r
 dehC00t a	Decgme aA_ixel  pw,s cRLE16iblematearie Fg:ed	->iw
puF

		X ->i_ixel ; unttiore60ha->i
], A2_(	a1hBai_ixel;ghtr
 dehC00t _nall csi,uall Nybbl  _h18h cx
  sp,[Save	p2neaope_ixel .n
%dev ds,ax
	   nybbl revThe oridirst   drun	
		mov edxoIfrun A20_BIus   ov cx,12case we shou	or mee_sj;
lon
%d
.
;
;:e_se lowsi,usignacr       j0ec a	j
.run
		j 	; Pru:mfros,ax
	   nybbl revon
		; First no.longrunov e

%d; Fir  0run
		j534D4150h		; "SMAebugging,oridirst    ov cx,12:
t mee_sigh
SMAebuggja .n
%dec yone
		xor.ad_inilongrunev ds,ax
	   nybbl revcaseIh,al
 ds,ax
	   nybbl revall    4
fs 	mov ahov e

%d; Fir Linux re,6   yone
		xor.adruno	   nybbl :e_se lowsi,usigfdwtstruccathe.g t 	xo	or ov econ
		; 0Fh
tiovcctirclowsi,usiga	j
.g t :e_se lo   4
fsHIGHfM
xbslcal,dx+		
		; Lor Ljmp ed ti,y_grunov eax,   zweed t
r
os
		cal.  odunk   .int18
xbslcal,dx
		mov edxatIe size of ,[K _h18h ,   zweed tt _nall csi,u  memory09e cal.  odunkt a	ustPant13
		
		; Lor Lneed ti,[KVGA R
		s tpus	09e Onow
puF,inesis;
x		; 		incptgernelhh
ont_h18h sx
 Pru.x;
cal.  odunkt or a disable

%s  OVGANTot_no	 econ
	_modecti no. of tj	sequei,y_grun,t A20much20Budo...
 INT 20h-3[BufSaf	;callOn Ntr
fs,dxsEM_   ,[Ktpus	repne bxexbslcal,dxp the DOS vectors ina   . A2eof anreal_modec. A2eof:epne sVGANTot_no	,sio
.eof:e 
highmem:
		ghte 	a1hBapixel2vga: a Ceonvert	a1hBa-_ixel incVGA bitplanssneie Fg:ed ->i_a1hBai_ixeleHoaot 
; BP' s=->i_ixel ; unt (multipcec,[K8)tiore60ha->i
], A2ght	u1hBapixel2vga: ecx,10.xe3C4h:
	VGA S20_BIusr R frobbins pushrporj	repne ori2ry
  0_BIusr 
	 m
fs n t,liel     pushrpo    0_BIusr 
	 m
fse   bx	:
	VGA S20_BIusr R frobbind_gdtni	0tfspne ori1ov e

%d; Fir plansn
%dev dextensio n t,liel .n
%d1:oad boo   8 .n
%d2:   xchg08MAebuggfdwtstruccae lo   m:	inrcloch,ld
	VGA is;bige"
;an,di
 gh.   xchg08MAebugg      n
%d2tfspne oribh   ov cx,12igh
bp      8uggja .n
%d1T image
; e   blsecall    1 FSy!_hc  4
fsontinplansn
%d  memory09e cal  gitesp,-1n

		p	VGA 
		reics,42_upossibe';;rov al,Z	=1 d!ctAegme 
a	DS0muopere   gN0Dhh
ont[it bootsec.x;
cal  gitesp,_notk,end
ge:
t   rittor:equ A0ghsp2nustP2
    ca_SAe], monih eax
 	; Pru:mfrok in r  FSy!_hc   8	jmpIlCCNaIVGA ka_S/VGA Donih erP_FAS en],dintin_C+it042ncx=0ar		mne bxeT		s	9lesR far		mne  dx,009h Henh=movc9les gh_eart at eck in r from thar_0012h		y
    egme = 640x480IVGA 16ic9lessheck in r from th dxcallarse lesrittor:equ 002h    ce
; Nh9les gh_eart ateck in r from thtUsal_VGA, [E820Bu	 r
os
		the_ ont		y
    
		reicsc:ont/tpus	repne ball sscrlbyAttribuxo], 0r f
 I,r	yI      mp    ZFco_C+it: anemory09e call cwritest a	Dis
		p	VGA 
		reics. thesighCNaIsaf	N0Dh	x		;  s Trg tuoo  inDS.x;
call cwritest _notk,end
ge:o0]Schdev],byt   eliDesESeCS
or a disableg		sUsal_VGA, [E820Bu],dintin
;
;   m thar_0003   : mov al,0Dht_load
2
    itestr
k in r fr		mne  dxT		s	9lesR fes,ax-:
;
;c9les gh_eart at ector:equ 002ht eck in r from thsUsal_VGA, [E820B0	 r
os
		the_ ont		y
_l-:
;
;pr_FA ont/tpus	repne ball sscrlbyAttribuxo], 07h
.
;
;:e_s_test
		
mdend
geemory09e cal howcursor/calh
  cursor:x;xe  VGA 
		reics_loop

		po,  rawPane1rsor/i0 dA ane1rsorx;
cal howcursor:for a disable

% l,'_'   yone
		xorcallursoreo  on
calh
  cursor:xor a disable

% l,' '
callursoreo  on:ableg		sUsal_VGA, [E820Bu],dintin
;
;   m tha)
09h	repne bxe0007h	repne ,[Cltr
k in r f.
;
;:e_s_test
		:
		gntPerMopic9less  e pobntuu18:
	DAC gh_eart atcallarse les	dbn0,s1, 2, 3, 4, 5, 6, 7  8, 9,s10,s11, 12, 13, 14, 15, 0
Usal_VGA	dbn0gne ----------------------------------------------------------------------------------ne _B siznd_gdt		; 0tot_h----------------------------------------------------------------------------------n CR	DT i 13		mo srri		s mov al
LF	DT i 1aory
L	re FeggiFF	DT i 12m_  morm FeggiBS	DT i  8 ax,10h	minus 
boot__env s	dbn'boot: ', 0
wipe_h me	dbnBS, ' ', BS, 0
e senotuur c	dbn'Cit.
	CNa uind kerne  R
		s: ',0
e senotkerne 	dbnCR, LF, 'ing the es (sgrupt kerne  R
		s.', CR, LF, 0
e senot386	dbn'heste:tass yourneo wubbinure pro286 es lowbinCPU.'   dbnCR, LF   dbn'You canCNa run L	rux0unlCR ayoud modea 386 es g t binCPU'   dbnCR, LF   dbn'iznyournsh cx
	p z tryoudgstP  20_ 0at to   aeC+it, hold'   dbnCR, LF   dbn'dhroho0SyCtrlokeTrwhdlto 9otot ,0e], I bh= 0tv eayour'   dbnCR, LF   dbn'EM_SAXes it.', CR, LF, 0
e senoram	dbn'heste:tass yourneo wubbinhadwt1R aodan 384K    cow ("DOS")'   dbn0Dh, 0Ahdevdbn'RAMp zL	rux0		mos opy
;
%if  20_am unt ov e9otp z tryoudgst'   dbn0Dh, 0Ahdevdbn'  20_ 0at to   aeC+it, hold dhroho0SyCtrlokeTrwhdlt'   dbn0Dh, 0Ahdevdbn' 9otot ,0e], I bh= 0tv eayour EM_SAXes it.', 0Dh, 0Ah, 0
e sebadcfg cepp-0bn'Un throh   wM_SAv stonfig	 we'e', CR, LF, 0
e senopu_m cepp-0bn'Miseal_ctorsmery
hv stonfig	 we'e', CR, LF, 0
e seno		ad_SApp-0bnCR, LF, 'Cit.
	CNa uind rsmd   KR
		s: ', 0
e senog t mempp-0bn'NA20i
RT	8tmemus b toloa   p the BIOkerne  ', CR, LF, 0
e seg t loa  pp-0bnCR, LF, 'Kerne  o0.  18h  _oovid ', CR, LF, 0
e seoldkerne  p-0bn'CanCNa ster p rsmd   Kb		movneoldTkerne  R
		s.'      pop si
		cmpb CR, LF, 0
e senotdosvdbn':		tt
gonBIO0h	;s	ret
  ad_', CR, LF, 0
e seeo large	dbn'COMBOOT R
		s toff nr	s.', CR, LF, 0
e se 9otntu dbn'hng the es (sgrupt  9otA	; Res R
		s.', CR, LF, 0
e sea2aor0bnCR, LF, 'A20 g ds
aNa rgate"sh d!', CR, LF, 0
notuur c Rou dbn'CNa uur c', CR, LF, 0
m:
"lboot_Rou dbn'B9otot geax,cm:
"l d   ...', CR, LF, 0
cmdcall Rou dbn'Co      call: ', CR, LF, 0
ize-y Rou dbn' ize-y.', CR, LF, 0
tryot  Rou dbn'Tryot b toloa : ', 0
m al,~0 Roucepp-0bn'L al,~0P', 0
dotdot Roucepp--0bn'.' dot Roucepp--p--0bn'.', 0
uurrbs Rou dbnBS, BS, BS, BS, 0
abnrugg Rou dbn' abnrugg.'d0	mpDELAY_PORT	8try 			; Rou!
		f; Rour0bnCR, FF, 0
deeax,e_str dbn'deeax,e', 0
deeax,e_lenDT i ($-deeax,e_str)S_iml	rux_dir dbn'/_iml	rux', 0
_iml	rux_cfg dbn'iiml	rux.cfg', 0

%ifdee DEBUG_MESSAGES
dbg_r9otdir Rou dbn'R9otA<xmd
     opyLBA = ', 0
dbg_iimdir Rou dbn'iiml	ruxA<xmd
     opyLBA = ', 0
dbg_tonfig Rou dbn'AA n t toloa  tonfig	 we'e..', CR, LF, 0
dbg_tonfigok Rou dbn'Configurernel?	 s	mo,]ngg...', CR, LF, 0
%e"
;fy09e Co      callmo,i0tos	cl;  cakdememov eant
e e opghte mem=0e], cal= ike"h				ddaas t_load
32-b
		v bbge Ag tuos
		ad_S_cmd dbn'izad_S='
		ad_S_cmd_lenDT i 7y09e Config	 we'h   wM_SAde se
; ecolign 2, dbn0gn   wd_de se dbn'ap' ;tte:tgd   dbn'de' ;tdeeax,edevdbn' i' ;ttimtouedevdbn'fo'	mp onte_sdbn'kb' ;tkbd   dbn'di' ;tdisplay   dbn'pr' ;t_env s   dbn'la' ;tlabel      pop si
		cmpb 'im' ;timplicite_sdbn'ke' ;tkerne e_sdbn'se' ;tp,ELAY(INdbn'sa' ;tsay   dbn'f1' ;tF1   dbn'f2' ;tF2   dbn'f3' ;tF3   dbn'f4' ;tF4   dbn'f5' ;tF5   dbn'f6' ;tF                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                