#-*- coding: utf-8 -*-

# Copyright 2010-2012 Calculate Ltd. http://www.calculate-linux.org
#
#  Licensed under the Apache License, Version 2.0 (the "License");
#  you may not use this file except in compliance with the License.
#  You may obtain a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0 #
#  Unless required by applicable law or agreed to in writing, software
#  distributed under the License is distributed on an "AS IS" BASIS,
#  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
#  See the License for the specific language governing permissions and
#  limitations under the License.

from calculate.lib.datavars import Variable,ReadonlyVariable,VariableError
import sys, re,os
from os import path

from calculate.lib.cl_lang import setLocalTranslate
from calculate.core.server.decorators import Dec
#from calculate.api.cert_cmd import find_id_cert
setLocalTranslate('cl_core',sys.modules[__name__])

from calculate.lib.utils.files import listDirectory
from calculate.lib.utils.portage import isPkgInstalled,reVerSplitToPV
from calculate.lib.utils.common import getTupleVersion
from itertools import *

class VariableClCorePkgName(Variable):
    """
    Package name
    """
    type = "choiceedit"
    opt = ["--pkg-name"]
    metavalue = "PN"
    untrusted = True

    def init(self):
        self.label = _("Package name")
        self.help = _("package name")

    def check(self,value):
        if not value: 
            raise VariableError(_("Please, choose package"))
        if not os.environ.get("EBUILD_PHASE","") and not value in self.choice():
            raise VariableError(_("Package not found"))

    def choice(self):
        return [""]+\
            sorted(set(chain(*map(lambda x:map(lambda y:reVerSplitToPV(y)['PN'],
                            listDirectory(x)),
               listDirectory('/var/db/pkg',onlyDir=True,fullPath=True)))))

class VariableClCorePkgVersion(Variable):
    """
    Package version
    """
    type = "choiceedit"
    opt = ["--pkg-version"]
    metavalue = "PVR"
    untrusted = True

    def init(self):
        self.label = _("Package version")
        self.help = _("package version number with the revision")

    def get(self):
        pkg = isPkgInstalled("%s/%s"%(self.Get('cl_core_pkg_category'),
                                      self.Get('cl_core_pkg_name')))
        if pkg:
            return sorted(pkg,
                   key=lambda x:getTupleVersion(x['PVR']))[-1]['PVR']
        return ""

    def check(self,value):
        pkg_name = self.Get('cl_core_pkg_name')
        if not value and pkg_name and pkg_name in self.choice():
            raise VariableError(_("Please, choose version"))

class VariableClCorePkgCategory(Variable):
    """
    Package category
    """
    type = "choiceedit"
    opt = ["--pkg-category"]
    metavalue = "CATEGORY"
    untrusted = True

    def init(self):
        self.label = _("Package category")
        self.help = _("package category name")

    def choice(self):
        pkgName = self.Get('cl_core_pkg_name')
        return map(lambda x:x.rpartition('/')[2],
               filter(lambda x:pkgName in listDirectory(x,onlyDir=True),
               listDirectory('/usr/portage',fullPath=True)))

    def get(self):
        category = isPkgInstalled(self.Get('cl_core_pkg_name'))
        if category:
            category = category[0].rpartition('/')[2]
        else:
            choice = self.choice()
            if len(choice) == 1:
                return choice[0]
            else:
                category = ""
        return category

    def check(self,value):
        pkg_name = self.Get('cl_core_pkg_name')
        if not value and pkg_name and pkg_name in self.choice():
            raise VariableError(_("Please, choose category"))

class VariableClCorePkgPath(Variable):
    """
    Package configure path
    """

    value = "/"
    opt = ["--pkg-path"]
    metavalue = "PATH"

    def init(self):
        self.label = _("Path for configuration")
        self.help = _("root path for saving the updated configuration files")

    def check(self,value):
        if not path.isdir(value):
            raise VariableError(_("Directory %s not found")%value)

class VariableClCorePkgSystemSet(Variable):
    """
    Package configure system
    """

    type = "bool"
    value = "on"
    opt = ["--no-system"]
    metavalue = "off"

    def init(self):
        self.label = _("Configure the system")
        self.help = _("update system configuration files")

class VariableClCorePkgDesktopSet(Variable):
    """
    Package configure desktop
    """

    type = "bool"
    value = "on"
    opt = ["--no-desktop"]
    metavalue = "off"

    def init(self):
        self.label = _("Configure users")
        self.help = _("update desktop (user) configuration files")

class VariableClCorePkgRootSet(Variable):
    """
    Other packages (specified by merge=) will configured in /
    """
    type = "bool"
    value = "on"
    metavalue = "off"

    opt = ["---no-depend-in-root"]

    def init(self):
        self.label = _("Configure dependent packages in root")
        self.help = _("configure the dependent packages in the same path as "
                      "the specified package, instead of root")
