*     JLdL 10Mar07.
*     
*     Copyright (C) 2007 by Jorge L. deLyra.
*     Mail: <delyra@latt.if.usp.br>. Web: "http://latt.if.usp.br".
*     This program may be copied and/or distributed freely. See the
*     _ terms and conditions in the files in the doc/ subdirectory.
*     
*     This is an implementation of the random number generator ran2,
*     _ the algorithm and original code for which were published by
*     _ W. H. Press and S. A. Teukolsky, in Computers in Physics,
*     _ Vol. 6, No. 5, Sep/Oct 1992, p. 522; the algorithm for the
*     _ generator is also given in section 7.1 of the book Numerical
*     _ Recipes, Cambridge University Press, second edition (1992).
*     
*     In this implementation the name of the function was changed to
*     _ ranr, and it was written as a single-precision function;
*     _ the code is extensively documented with comments.
*     
*     This is a very long-period (larger than 2.0e18) random number
*     _ generator, after the design of L'Ecuyer, with the addition
*     _ of a Bays-Durham shuffle; it achieves its long period by the
*     _ combination of two multiplicative congruential generators,
*     _ with carefully chosen integer parameters, giving slightly
*     _ different periods; this combination also tends to eliminate
*     _ the correlations along the final random sequence, and the
*     _ additional shuffle algorithm helps to ensure the absence of
*     _ any short-range correlations.
*     
*     Since the iteration of a multiplicative congruential generator
*     _ involves an integer (truncated-division) calculation such as
*     _
*     _   iran_next = mod(mult*iran,mbig)
*     _             = mult*iran-mbig*[(mult*iran)/mbig]
*     _
*     _ for a very large mbig, where iran is typically also a large
*     _ number within the interval [0,mbig), the product mult*iran
*     _ will very often produce an integer overflow; in order to
*     _ compute the mod above without any such overflows, the routine
*     _ uses an algorithm due to Schrage, which involves two integers,
*     _ mquo and mrem, that constitute an approximate factorization
*     _ of the large integer mbig, given by:
*     _
*     _   mbig = mult*mquo+mrem
*     _
*     _ The Schrage algorithm for the calculation of the mod of such
*     _ a large product of two integers, avoiding all the 31-bit
*     _ integer overflows, is given by:
*     _
*     _   iran_next=mult*(iran-mquo*(iran/mquo))-mrem*(iran/mquo)
*     _
*     _ except that, if the result is negative, then one must cycle
*     _ it back to positive values by the addition of mbig:
*     _
*     _   if (iran_next.lt.0) iran_next=iran_next+mbig
*     
*     The generator returns a uniform random float in (0.0,1.0), with
*     _ the exclusion of the two end-point values; one can call the
*     _ function with any integer seed, except for the value 0, which
*     _ is forbidden and will be mapped to 1 by the function; all the
*     _ integer random numbers produced are positive and the sign of
*     _ the submitted seed will be ignored by the function; if the
*     _ seed is changed externally, while the generator is working on
*     _ a given random sequence, the code will detect this and will
*     _ then automatically re-initialize its internal shuffle table;
*     _ one may use this as a means of producing a new random sequence,
*     _ independent from the first one; the initialization will also
*     _ be executed in the first call to the function.
*     
*     In order to produce real random numbers with the maximum possible
*     _ granularity, or random integers with the widest possible range,
*     _ as the case may be, mbig should be a very large integer; note
*     _ that the maximum value of a 31-bit (signed) integer is given by
*     _
*     _   mbig = 2**31-1 = 2147483647
*     _
*     _ but this mbig is not used in the code; since a single-precision
*     _ real number has 23 significant bits and a 32-bit integer has 31
*     _ significant bits, it is possible to use numbers a bit smaller
*     _ than mbig and still achieve good results for the real numbers.
*     
*     Note that the granularity of the random real numbers generated
*     _ by this function is that of single-precision floating point
*     _ numbers, which corresponds to 23 random bits, and is given
*     _ approximately by 1.2e-7; one can calculate this more
*     _ precisely, to get the result:
*     _
*     _   delta = 1.0/[2.0**23-1.0] = 1.19209304e-7
*     _
*     _ Note that this is larger than the granularity associated to the
*     _ large integer mbig, and that because of this it is possible that
*     _ a real number in (0.0,1.0) created from mbig may turn out to be
*     _ rounded off to exactly 1.0 or even slightly above it, during the
*     _ process of transformation from integer to real format; in fact,
*     _ there are 234 cases of iran near the top (mbig-1) of its range
*     _ for which this does indeed happen.
*     
      FUNCTION RANR (nseed)
*     
*     Explanation of the defining fixed parameters; all these parameters
*     _ have the default Fortran data types implied by their names, and
*     _ they are not explicitly declared.
*     
*     Since there are two concurrent generators in action within this
*     _ code, the digits 1 and 2 in the names of the parameters below
*     _ indicate that the variables marked in this way belong to one or
*     _ the other of these two generators; mbig is the large integer
*     _ defining a generator, mult is its multiplication constant, and
*     _ mquo and mrem are the two numbers involved in the approximate
*     _ factorization of mbig, which are used in the Schrage algorithm;
*     _ the values given for all these numbers are essential for the
*     _ proper operation of the generator, and should not be changed
*     _ under any circumstances.
*     
*     Descriptions:
*     _ mbig1: the large integer for generator 1; this is mbig-84.
*     _ mbig2: the large integer for generator 2; this is mbig-248.
*     _ mult1: the multiplier for generator 1.
*     _ mult2: the multiplier for generator 2.
*     _ mquo1: the Schrage quotient constant for generator 1.
*     _ mquo2: the Schrage quotient constant for generator 2.
*     _ mrem1: the Schrage remainder constant for generator 1.
*     _ mrem2: the Schrage remainder constant for generator 2.
*     _ mshff: the dimension of the Bays-Durham shuffle table.
*     _ delta: the minimum possible variation of a single-precision real
*     _        number, which includes about 7 significant decimal places;
*     _        since this will be subtracted from 1.0, there is no point
*     _        in using here the full possible precision for delta, so
*     _        we round it off to only two significant digits.
      parameter
     *     (mbig1=2147483563
     *     ,mbig2=2147483399
     *     ,mult1=40014
     *     ,mult2=40692
     *     ,mquo1=53668
     *     ,mquo2=52774
     *     ,mrem1=12211
     *     ,mrem2=3791
     *     ,mshff=32
     *     ,delta=1.2e-7)
*     
*     Explanation of the calculated fixed parameters; these parameters
*     _ also have the default Fortran data types implied by their names,
*     _ and they also are not explicitly declared.
*     
*     The values of these parameters follow from those of the defining
*     _ parameters described above, and are defined here mainly for
*     _ efficiency reasons, since they are used repeatedly in the code.
*     
*     Descriptions:
*     _ mmax1: this is mbig1-1, the largest integer that can be produced
*     _        by the first generator, which is used to cycle the final
*     _        random integer iran back to positive values, when needed.
*     _ mdeno: the denominator used to divide a random integer in order
*     _        to produce the random index to be used to retrieve a
*     _        shuffled integer from the array nshtb(mshff).
*     _ rdelt: the float granularity associated to the large integer
*     _        mbig1, which is used to define the single-precision
*     _        real values returned by the function; those values
*     _        will range from rdelt to 1.0-rdelt.
*     _ rrmax: the largest single-precision real number that is still
*     _        less that 1.0; this is the largest real value that the
*     _        function may return.
      parameter
     *     (mmax1=mbig1-1
     *     ,mdeno=1+mmax1/mshff
     *     ,rdelt=1.0/mbig1
     *     ,rrmax=1.0-delta)
*     
*     Explanation of the variables present in the code; most of the
*     _ variables have the default Fortran data types implied by
*     _ their names, in which case they are not explicitly declared.
*     
*     Descriptions:
*     _ ranr:  the single-precision real random number that is produced
*     _        from iran and returned by the function in its name.
*     _ nseed: the initial seed, entered as the argument of the function
*     _        and kept updated with the value of iran1.
*     _ iran:  the resulting random integer, produced from the two
*     _        random integers iran1 and iran2.
*     _ iran1: the random integer of the first generator, which is
*     _        seeded from outside, that is, from the argument nseed
*     _        with which the function is called.
*     _ iran2: the random integer of the second generator, which is
*     _        also seeded by the nseed argument.
*     _ nshtb: the shuffle array, to hold the shuffle table, where
*     _        several values of the first random integer iran1 will
*     _        be kept, to be used later in an arbitrary order which
*     _        is not their sequential order of production.
*     _ ishff: a local temporary variable to serve as an index for
*     _        the array nshtb(mshff).
*     _ ntdiv: a local temporary variable to hold the result of the
*     _        truncated division in the Schrage algorithm.
      dimension nshtb(mshff)
*     
*     The values of these variables must be saved when the function
*     _ exits; they are the internal variables that constitute the
*     _ memory within a given random sequence, and that must be
*     _ kept between successive calls to the function.
      save iran1,iran2,iran,nshtb
*     
*     Basic one-time initialization; this is executed only once, when
*     _ the function is first called within a given program; the initial
*     _ value of zero for iran1, as shown below, is used for comparison
*     _ with the initial value of nseed submitted as the argument, so
*     _ that the function will initialize itself at the first call.
      data iran1/0/
*     
*     Since a null seed will perpetuate itself in a multiplicative
*     _ congruential generator such as the ones used here, make sure
*     _ that the submitted value of nseed in not kept at zero.
      if (nseed.eq.0) nseed=1
*     
*     The following if-block contains the per-seed initialization code;
*     _ it will be executed when the function is first called and when
*     _ the variable nseed is changed externally during the execution
*     _ of the code; since nseed cannot be zero and iran1 is internally
*     _ initialized at zero, this will always be executed during the
*     _ first call to the function.
      if (iabs(nseed).ne.iran1) then
*     
*     Pass the submitted seed to the first generator.
         iran1=iabs(nseed)
*     
*     Pass the seed on to the second generator.
         iran2=iran1
*     
*     Run over the shuffle array from the top down, while propagating
*     _ the first generator, and in this way load the shuffle table
*     _ with random integers from the first generator, after 8
*     _ initial cycles which are discarded.
         do ishff=mshff+8,1,-1
*     
*     Propagate the random integer iran1, that is, iterate the code of
*     _ the first random number generator; this means that we must
*     _ calculate iran1=mod(mult1*iran1,mbig1); compute the mod
*     _ without integer overflows, using the Schrage algorithm.
            ntdiv=iran1/mquo1
            iran1=mult1*(iran1-ntdiv*mquo1)-ntdiv*mrem1
            if (iran1.lt.0) iran1=iran1+mbig1
*     
*     Insert the value of the random integer into the shuffle array.
            if (ishff.le.mshff) nshtb(ishff)=iran1
         end do
*     
*     Start iran as the first element of the array, it will be used
*     _ later to define the first index for the shuffle array to be
*     _ used along the random sequence.
         iran=nshtb(1)
      end if
*     
*     This is the entry point of this routine when it is not being
*     _ initialized by the submission of a new seed.
*     
*     Propagate the random integer iran1, that is, iterate the code of
*     _ the first random number generator; this means that we must
*     _ calculate iran1=mod(mult1*iran1,mbig1); compute the mod
*     _ without integer overflows, using the Schrage algorithm.
      ntdiv=iran1/mquo1
      iran1=mult1*(iran1-ntdiv*mquo1)-ntdiv*mrem1
      if (iran1.lt.0) iran1=iran1+mbig1
*     
*     Propagate the random integer iran2, that is, iterate the code of
*     _ the second random number generator; this means that we must
*     _ calculate iran2=mod(mult2*iran2,mbig2); compute the mod
*     _ without integer overflows, using the Schrage algorithm.
      ntdiv=iran2/mquo2
      iran2=mult2*(iran2-ntdiv*mquo2)-ntdiv*mrem2
      if (iran2.lt.0) iran2=iran2+mbig2
*     
*     Using the previous value of iran, calculate a random index for
*     _ the shuffle array nshtb(1:mshff); the value of ishff will be
*     _ in the interval [1,mshff].
      ishff=1+iran/mdeno
*     
*     This is the point where iran1 is shuffled, and iran1 and iran2
*     _ are combined to generate the final random variable iran.
*     
*     Combine a value of iran1, previously stored in the shuffle
*     _ array, with the current value of iran2, to generate the
*     _ output integer variable iran.
      iran=nshtb(ishff)-iran2
*     
*     If the resulting value of iran is not strictly positive, cycle
*     _ it back to strictly positive values.
      if (iran.lt.1) iran=iran+mmax1
*     
*     Insert the current value of iran1 into the slot of the
*     _ shuffle array which has just been used.
      nshtb(ishff)=iran1
*     
*     Return the current value of iran1 in the argument of the
*     _ function.
      nseed=iran1
*     
*     Now calculate the output real random number as rdelt*iran, but
*     _ then truncate the result at rrmax, so that the function will
*     _ never return exactly 1.0; note that, since iran is never zero,
*     _ ranr will never be zero either; it is commonly expected that
*     _ a random number generator will never return the two end-point
*     _ values, a behavior which is sometimes useful in order to
*     _ avoid singularities at these points, in functions that
*     _ will take the random number as an argument.
      ranr=min(rdelt*float(iran),rrmax)
*     
      return
*     
      END
