/* $Id$ 
 *
 * AOffset: Calculate array offsets (intermediate opcode).
 *
 * Copyright (C) 2008-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#ifndef __A_OFFSET_HPP_INCLUDED
#define __A_OFFSET_HPP_INCLUDED

#include <cassert>
#include "intermediate/operands/Register.hpp"
#include "intermediate/container/Type.hpp"


namespace intermediate {

//! Calculate the array offset to a given base pointer and a given type.
/** This class will calculate the array offset for a given base
 *  pointer and a given type.
 *
 *  Read operands: base, offset,
 *  Meta information operand: rtype
 *  Write operands: dst
 *
 *  Operation: dst = b + array offset(rtype[offset])
 */
class AOffset : public OpCode {
public:
	//! c'tor
	/** @param b base address of the array.
	 *  @param off index offset (0=first elem, 1=second...)
	 *  @param t corresponding type declaration.
	 *  @param destination destination operand
	 */
	AOffset(
		Operand *b,
		Operand *off,
		Type *t,
		Register *destination
		) :	base(b),
			offset(off),
			atype(t),
			dst(destination) {

		assert(b->type == OP_TYPE_POINTER);
		assert(off->type == OP_TYPE_INTEGER);
	}

	//! Accept a Visitor.
 	/** All intermediate code nodes need to implement this method.
         *
         *  @param v the Visitor that can visit this node.
         */
	virtual void accept(Visitor& v) {
		v.visit(*this);
	}

	/** base pointer to array */
	Operand *base;
	/** integral array position (0=first element, 1=second..) */
	Operand *offset;
	/** referred array type */
	Type *atype;
	/** destination operand */
	Register *dst;

protected:
	virtual ~AOffset() {
		util::MiscUtil::terminate(this->base);
		util::MiscUtil::terminate(this->offset);
		util::MiscUtil::terminate(this->atype);
		util::MiscUtil::terminate(this->dst);
	}

};

}; /* namespace intermediate */

#endif /* __A_OFFSET_HPP_INCLUDED */
