/*
 * Copyright (C) 2003-2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include <assert.h>
#include <inttypes.h>
#include <stdio.h>
#include <stdlib.h>

#include "fixme.h"
#include "glue.h"

#define INCLUDE

#include "arch_dma.c"

#undef INCLUDE

#include "chip_intel_8237A.h"

struct cpssp {
	/*
	 * Config
	 */

	/*
	 * Signals
	 */
	struct sig_cs *port_cs;
	struct sig_isa_bus *port_bus;
	struct sig_isa_dma *port_dma0;
	struct sig_isa_dma *port_dma1;
	struct sig_isa_dma *port_dma2;
	struct sig_isa_dma *port_dma3;

	/*
	 * State
	 */
	unsigned int state_power;

#define STATE

#define NAME		dma
#define NAME_(x)	dma_ ## x
#define SNAME		"dma"
#include "arch_dma.c"
#undef SNAME
#undef NAME_
#undef NAME

#undef STATE
};

/* Write transfer. */
static void
dma_ack_in(
	struct cpssp *cpssp,
	unsigned short offset,
	unsigned char chan,
	unsigned int tc
)
{
	struct sig_isa_dma *port_dma;

	assert(/* 0 <= chan && */ chan < 4);
	switch (chan) {
	case 0:
		port_dma = cpssp->port_dma0;
		break;
	case 1:
		port_dma = cpssp->port_dma1;
		break;
	case 2:
		port_dma = cpssp->port_dma2;
		break;
	case 3:
		port_dma = cpssp->port_dma3;
		break;
	default:
		assert(0);	/* Cannot happen. */
	}

	sig_isa_dma_ack_in(port_dma, cpssp, offset, tc);
}

/* Read transfer. */
static void
dma_ack_out(
	struct cpssp *cpssp,
	unsigned short offset,
	unsigned char chan,
	unsigned int tc
)
{
	struct sig_isa_dma *port_dma;

	assert(/* 0 <= chan && */ chan < 4);
	switch (chan) {
	case 0:
		port_dma = cpssp->port_dma0;
		break;
	case 1:
		port_dma = cpssp->port_dma1;
		break;
	case 2:
		port_dma = cpssp->port_dma2;
		break;
	case 3:
		port_dma = cpssp->port_dma3;
		break;
	default:
		assert(0);	/* Cannot happen. */
	}

	sig_isa_dma_ack_out(port_dma, cpssp, offset, tc);
}

/* Verify transfer. */
static void
dma_ack_verify(
	struct cpssp *cpssp,
	unsigned char chan,
	unsigned int tc
)
{
	struct sig_isa_dma *port_dma;

	assert(/* 0 <= chan && */ chan < 4);
	switch (chan) {
	case 0:
		port_dma = cpssp->port_dma0;
		break;
	case 1:
		port_dma = cpssp->port_dma1;
		break;
	case 2:
		port_dma = cpssp->port_dma2;
		break;
	case 3:
		port_dma = cpssp->port_dma3;
		break;
	default:
		assert(0);	/* Cannot happen. */
	}

	sig_isa_dma_ack_verify(port_dma, cpssp, tc);
}

#define BEHAVIOR

#define NAME		dma
#define NAME_(x)	dma_ ## x
#define SNAME		"dma"
#include "arch_dma.c"
#undef SNAME
#undef NAME_
#undef NAME

#undef BEHAVIOR

static void
chip_intel_8237A_power_set(void *_css, unsigned int val)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	cpssp->state_power = val;
}

static void
chip_intel_8237A_n_reset_set(void *_css, unsigned int val)
{

	struct cpssp *cpssp = (struct cpssp *) _css;

	dma_reset(cpssp);
}

static int
chip_intel_8237A_req0(void *_css)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	dma_req(cpssp, 0);

	return 0;
}

static int
chip_intel_8237A_req1(void *_css)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	dma_req(cpssp, 1);

	return 0;
}

static int
chip_intel_8237A_req2(void *_css)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	dma_req(cpssp, 2);

	return 0;
}

static int
chip_intel_8237A_req3(void *_css)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	dma_req(cpssp, 3);

	return 0;
}

static int
chip_intel_8237A_inb(void *_css, uint32_t port, uint8_t *valp)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	dma_inb(cpssp, valp, port);

	return 0;
}

static int
chip_intel_8237A_outb(void *_css, uint32_t port, uint8_t value)
{
	struct cpssp *cpssp = (struct cpssp *) _css;

	dma_outb(cpssp, value, port);

	return 0;
}

void *
chip_intel_8237A_create(
	const char *name,
	struct sig_manage *port_manage,
	struct sig_std_logic *port_power,
	struct sig_std_logic *port_reset_hash_,
	struct sig_cs *port_cs,
	struct sig_isa_bus *port_bus,
	struct sig_isa_dma *port_dma0,
	struct sig_isa_dma *port_dma1,
	struct sig_isa_dma *port_dma2,
	struct sig_isa_dma *port_dma3
)
{
	static const struct sig_std_logic_funcs power_funcs = {
		.boolean_or_set = chip_intel_8237A_power_set,
	};
	static const struct sig_std_logic_funcs reset_hash__funcs = {
		.boolean_or_set = chip_intel_8237A_n_reset_set,
	};
	static const struct sig_cs_funcs cs_funcs = {
		.readb = chip_intel_8237A_inb,
		.writeb = chip_intel_8237A_outb,
	};
	static const struct sig_isa_dma_funcs dma0_funcs = {
		.req = chip_intel_8237A_req0,
	};
	static const struct sig_isa_dma_funcs dma1_funcs = {
		.req = chip_intel_8237A_req1,
	};
	static const struct sig_isa_dma_funcs dma2_funcs = {
		.req = chip_intel_8237A_req2,
	};
	static const struct sig_isa_dma_funcs dma3_funcs = {
		.req = chip_intel_8237A_req3,
	};
	struct cpssp *cpssp;

	cpssp = shm_alloc(sizeof(*cpssp));
	assert(cpssp);

	dma_init(cpssp);

	/* Out */
	/* Call */
	cpssp->port_cs = port_cs;
	sig_cs_connect(port_cs, cpssp, &cs_funcs);

	cpssp->port_dma0 = port_dma0;
	sig_isa_dma_connect(port_dma0, cpssp, &dma0_funcs);

	cpssp->port_dma1 = port_dma1;
	sig_isa_dma_connect(port_dma1, cpssp, &dma1_funcs);

	cpssp->port_dma2 = port_dma2;
	sig_isa_dma_connect(port_dma2, cpssp, &dma2_funcs);

	cpssp->port_dma3 = port_dma3;
	sig_isa_dma_connect(port_dma3, cpssp, &dma3_funcs);

	/* In */
	cpssp->state_power = 0;
	sig_std_logic_connect_in(port_power, cpssp, &power_funcs);

	sig_std_logic_connect_in(port_reset_hash_, cpssp, &reset_hash__funcs);

	return cpssp;
}

void
chip_intel_8237A_destroy(void *_cpssp)
{
	struct cpssp *cpssp = _cpssp;

	shm_free(cpssp);
}

void
chip_intel_8237A_suspend(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_suspend(cpssp, sizeof(*cpssp), fp);
}

void
chip_intel_8237A_resume(void *_cpssp, FILE *fp)
{
	struct cpssp *cpssp = _cpssp;
	
	generic_resume(cpssp, sizeof(*cpssp), fp);
}
