/*
 * Copyright (C) 2017 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

#include "lib/glue-gui-gtk-console.h"

#include <gdk/gdkkeysyms.h>
#include <string.h>
#include "lib/string_tools.h"

static const char *PROMPT_POSTFIX = "> ";

/* GTK helper functions begin */
static void scroll_to_bottom(GtkScrolledWindow *scrolling)
{
	GtkAdjustment *scroll_adjust;
	gdouble upper;
	gdouble page_size;

	if (scrolling == NULL) {
		return;
	}

	scroll_adjust = gtk_scrolled_window_get_vadjustment(scrolling);
	upper = gtk_adjustment_get_upper(scroll_adjust);
	page_size = gtk_adjustment_get_page_size(scroll_adjust);
	gtk_adjustment_set_value(scroll_adjust, upper - page_size);
}

void textbuf_append(GtkTextBuffer *buffer, const gchar *text, gint len,
                    GtkScrolledWindow *scrolling)
{
	GtkTextIter text_end;

	gtk_text_buffer_get_end_iter(buffer, &text_end);
	gtk_text_buffer_place_cursor(buffer, &text_end);
	gtk_text_buffer_insert_at_cursor(buffer, text, len);
	gtk_text_buffer_get_end_iter(buffer, &text_end);
	gtk_text_buffer_place_cursor(buffer, &text_end);

	/* scroll to the bottom */
	scroll_to_bottom(scrolling);
}

static void textbuf_backspace(GtkTextBuffer *buffer, GtkTextMark *cursor_mark,
                              GtkScrolledWindow *scrolling,
                              size_t prompt_length)
{
	GtkTextIter cursor_iter;

	/* scroll to the bottom */
	scroll_to_bottom(scrolling);

	gtk_text_buffer_get_iter_at_mark(buffer, &cursor_iter, cursor_mark);

	if (gtk_text_iter_get_line_offset(&cursor_iter) > prompt_length) {
		gtk_text_buffer_backspace(buffer, &cursor_iter, TRUE, TRUE);
	}
}

static void textbuf_insert_text(GtkTextBuffer *buffer,
                                GtkTextMark *cursor_mark,
                                GtkScrolledWindow *scrolling,
                                const gchar *text, gint len)
{
	GtkTextIter cursor_iter;

	gtk_text_buffer_get_iter_at_mark(buffer, &cursor_iter, cursor_mark);

	/* insert text */
	gtk_text_buffer_insert(buffer, &cursor_iter, text, len);

	/* really necessary? */
        #if 0
	gtk_text_buffer_move_mark(buffer, cursor_mark, &cursor_iter);
	gtk_text_buffer_place_cursor(buffer, &cursor_iter);
        #endif

	/* scroll to the bottom */
	scroll_to_bottom(scrolling);
}

static gboolean textbuf_moveleft(GtkTextBuffer *buffer,
                                 GtkTextMark *cursor_mark,
                                 GtkScrolledWindow *scrolling,
                                 size_t prompt_length)
{
	GtkTextIter cursor_iter;

	/* scroll to the bottom */
	scroll_to_bottom(scrolling);

	gtk_text_buffer_get_iter_at_mark(buffer, &cursor_iter, cursor_mark);

	if (gtk_text_iter_get_line_offset(&cursor_iter) > prompt_length) {
		gtk_text_iter_backward_cursor_position(&cursor_iter);
		gtk_text_buffer_move_mark(buffer, cursor_mark, &cursor_iter);
		gtk_text_buffer_place_cursor(buffer, &cursor_iter);
		return TRUE;
	}
	return FALSE;
}

static gboolean textbuf_moveright(GtkTextBuffer *buffer,
                                  GtkTextMark *cursor_mark,
                                  GtkScrolledWindow *scrolling)
{
	GtkTextIter cursor_iter;

	/* scroll to the bottom */
	scroll_to_bottom(scrolling);

	gtk_text_buffer_get_iter_at_mark(buffer, &cursor_iter, cursor_mark);

	if (gtk_text_iter_ends_line(&cursor_iter) == FALSE) {
		gtk_text_iter_forward_cursor_position(&cursor_iter);
		gtk_text_buffer_move_mark(buffer, cursor_mark, &cursor_iter);
		gtk_text_buffer_place_cursor(buffer, &cursor_iter);
		return TRUE;
	}
	return FALSE;
}

static void textbuf_delete(GtkTextBuffer *buffer,
                           GtkTextMark *cursor_mark,
                           GtkScrolledWindow *scrolling,
                           size_t prompt_length)
{
	if (textbuf_moveright(buffer, cursor_mark, scrolling)) {
		textbuf_backspace(buffer, cursor_mark, scrolling, prompt_length);
	}
}

static void on_scroll_changed(GtkAdjustment *adjustment,
                              gpointer       user_data)
{
	GtkScrolledWindow *scroll_ctx = (GtkScrolledWindow *)user_data;
	scroll_to_bottom(scroll_ctx);
}

void insp_console_init(insp_console_context *ctx) {
	ctx->buffer = NULL;
	ctx->cursor = NULL;
	ctx->scroll_ctx = NULL;
	ctx->clipboard = NULL;

	ctx->prompt_length = snprintf(ctx->prompt_buffer,
	                              MAX_PROMPT_LENGTH,
	                              "inspect:/%s", PROMPT_POSTFIX);
}

void insp_console_insert_text(insp_console_context *ctx,
                              const gchar *text, gint len)
{
	textbuf_insert_text(ctx->buffer,ctx->cursor, ctx->scroll_ctx,
	                    text, len);
}

GtkClipboard* insp_console_get_clipboard(insp_console_context *ctx)
{
	return ctx->clipboard;
}

GtkTextBuffer* insp_console_get_buffer(insp_console_context *ctx)
{
	return ctx->buffer;
}

GtkTextMark* insp_console_get_cursor(insp_console_context *ctx)
{
	return ctx->cursor;
}

GtkScrolledWindow* insp_console_get_scroll_ctx(insp_console_context *ctx)
{
	return ctx->scroll_ctx;
}

void insp_console_set_buffer(insp_console_context *ctx, GtkTextBuffer *buf)
{
	GtkTextIter iter;
	if (ctx == NULL || buf == NULL) {
		return;
	}
	gtk_text_buffer_get_end_iter(buf, &iter);
	ctx->cursor = gtk_text_buffer_create_mark(buf, "cursor", &iter, FALSE);
	ctx->buffer = buf;
}

void insp_console_set_scroll_ctx(insp_console_context *ctx,
                                 GtkScrolledWindow *scroll_ctx)
{
	GtkAdjustment *scroll_adjust;

	ctx->scroll_ctx = scroll_ctx;

	scroll_adjust = gtk_scrolled_window_get_vadjustment(scroll_ctx);
	g_signal_connect(GTK_WIDGET(scroll_adjust), "changed",
	                 G_CALLBACK(on_scroll_changed), scroll_ctx);
}

void insp_console_set_clipboard(insp_console_context *ctx,
                                GtkClipboard *clipboard)
{
	ctx->clipboard = clipboard;
}

void insp_console_handle_backspace(insp_console_context *ctx)
{
	textbuf_backspace(ctx->buffer, ctx->cursor, ctx->scroll_ctx,
	                  insp_console_get_prompt_length(ctx));
}

void insp_console_handle_delete(insp_console_context *ctx)
{
	textbuf_delete(ctx->buffer, ctx->cursor, ctx->scroll_ctx,
	               insp_console_get_prompt_length(ctx));
}

void insp_console_handle_left(insp_console_context *ctx)
{
	textbuf_moveleft(ctx->buffer, ctx->cursor, ctx->scroll_ctx,
	                 insp_console_get_prompt_length(ctx));
}

void insp_console_handle_right(insp_console_context *ctx)
{
	textbuf_moveright(ctx->buffer, ctx->cursor, ctx->scroll_ctx);
}

void insp_console_handle_letter(insp_console_context *ctx, char c)
{
	if (GDK_KEY_space > c || c > GDK_KEY_asciitilde) {
		return;
	}
	insp_console_insert_text(ctx, &c, 1);
}

void insp_console_handle_return(insp_console_context *ctx, inspector_context *inspector)
{
	GtkTextIter start_iter;
	GtkTextIter cursor_iter;
	gchar *cmd = NULL;

	/* move start iterator */
	gtk_text_buffer_get_end_iter(ctx->buffer, &start_iter);
	gtk_text_iter_set_line_offset(&start_iter,
	                              insp_console_get_prompt_length(ctx));

	/* move cursor to end */
	gtk_text_buffer_get_end_iter(ctx->buffer, &cursor_iter);
	gtk_text_buffer_move_mark(ctx->buffer, ctx->cursor, &cursor_iter);
	gtk_text_buffer_place_cursor(ctx->buffer, &cursor_iter);

	/* get command string */
	cmd = gtk_text_buffer_get_text(ctx->buffer, &start_iter, &cursor_iter, FALSE);

	/* newline */
	insp_console_insert_text(ctx, "\n", -1);

	/* execute command */
	inspector_exec_command(inspector, cmd, -1);
	g_free(cmd);
	cmd = NULL;

	/* display prompt */
	insp_console_insert_text(ctx,
	                         insp_console_get_prompt_text(ctx),
	                         -1);
#if 0
	scroll_to_bottom(ctx->scroll_ctx);
#endif
}

const char* insp_console_get_prompt_text(insp_console_context *ctx)
{
	return ctx->prompt_buffer;
}

int insp_console_get_prompt_length(insp_console_context *ctx)
{
	return ctx->prompt_length;
}

void insp_console_push_to_prompt(insp_console_context *ctx,
                                 const char *name, size_t len)
{
	size_t offset = ctx->prompt_length - strlen(PROMPT_POSTFIX);
	snprintf(ctx->prompt_buffer + offset,
	         MAX_PROMPT_LENGTH - offset,
	         "%s/%s", name, PROMPT_POSTFIX);
	ctx->prompt_length = strlen(ctx->prompt_buffer);
}

void insp_console_pop_from_prompt(insp_console_context *ctx)
{
	char *pos = str_nrchr_mutable(ctx->prompt_buffer, '/',
	                      ctx->prompt_length - strlen(PROMPT_POSTFIX) - 1);
	if (pos == NULL) {
		return;
	}
	snprintf(pos + 1, strlen(PROMPT_POSTFIX) + 1, "%s", PROMPT_POSTFIX);
	ctx->prompt_length = strlen(ctx->prompt_buffer);
}
