/*
 * Copyright (C) 2007-2009 FAUmachine Team <info@faumachine.org>.
 * This program is free software. You can redistribute it and/or modify it
 * under the terms of the GNU General Public License, either version 2 of
 * the License, or (at your option) any later version. See COPYING.
 */

/* ********************* exported definitions ************************** */

#ifdef DEFINITIONS

#define	TERM_MAX_WIDTH	256
#define TERM_MAX_HEIGHT	128

#define FONT_WIDTH	8
#define FONT_HEIGHT	16

#define MAX_PATTERN_LEN 128

struct pattern {
	bool active;
	bool last_event;
	char pattern[MAX_PATTERN_LEN];
	void *_cpssp;
	struct sig_boolean* result;
};

#undef MAX_PATTERN_LEN
#endif /* DEFINITIONS */

/* ***************************** STATE ********************************* */

#ifdef STATE

#define MAX_PATTERN_NUM 4

struct {
	/* state */

	/* pattern slots */
	struct pattern patterns[MAX_PATTERN_NUM];

	/* pixel screen buffer */
	uint32_t pixel_buffer[MAX_HEIGHT][MAX_WIDTH];
	/* terminal width/heigth */
	unsigned int width;
	unsigned int height;

	/* ascii screen buffer */
	unsigned char screen8[MAX_HEIGHT / 8][MAX_WIDTH / 8 + 1];
	unsigned char dirty8[MAX_HEIGHT / 8][MAX_WIDTH / 8 + 1];

	unsigned char screen14[MAX_HEIGHT / 14][MAX_WIDTH / 8 + 1];
	unsigned char dirty14[MAX_HEIGHT / 14][MAX_WIDTH / 8 + 1];

	unsigned char screen16[MAX_HEIGHT / 16][MAX_WIDTH / 8 + 1];
	unsigned char dirty16[MAX_HEIGHT / 16][MAX_WIDTH / 8 + 1];

	unsigned int sync_count;
	bool check_necessary;
	bool match_necessary;
} NAME;

#undef MAX_PATTERN_NUM
#endif /* STATE */

/* ************************** BEHAVIOR ********************************* */

#ifdef BEHAVIOR

/* determine number of elements in an array. */
#define ARRAY_SIZE(a) (sizeof(a) / sizeof(a[0]))

/* what to store if no character could be detected? */
#define CHAR_UNKNOWN	255

static const
#include "glyphs_8x8.c"
static const
#include "glyphs_8x14.c"
static const
#include "glyphs_8x16.c"


static inline int
NAME_(check4_one_pattern)(
	const struct cpssp * cpssp,
	const struct pattern *p
)
{
	unsigned int y;

	if (! p->active) {
		return 0;
	}

	for (y = 0; y < cpssp->NAME.height / 16; y++) {
		if (strstr(cpssp->NAME.screen16[y], p->pattern) != NULL) {
			return 1;
		}
	}
	for (y = 0; y < cpssp->NAME.height / 14; y++) {
		if (strstr(cpssp->NAME.screen14[y], p->pattern) != NULL) {
			return 1;
		}
	}
	for (y = 0; y < cpssp->NAME.height / 8; y++) {
		if (strstr(cpssp->NAME.screen8[y], p->pattern) != NULL) {
			return 1;
		}
	}
	return 0;
}

static inline void
NAME_(check4pattern)(struct cpssp * cpssp)
{
	int i;
	struct pattern *p;
	int match;

	for (i = 0; i < ARRAY_SIZE(cpssp->NAME.patterns); i++) {
		p = &cpssp->NAME.patterns[i];
		match = NAME_(check4_one_pattern)(cpssp, p);

		if (match != p->last_event) {
			/*
			 * Match changed.
			 */

			/* Remember last match value. */
			p->last_event = match;

			sig_boolean_set(p->result, p, match);
		}
	}
}

static inline bool
NAME_(check_tile)(
	struct cpssp * cpssp,
	unsigned int x,
	unsigned int y,
	unsigned int height
)
{
	unsigned char bitmap[16], rbitmap[16];
	uint32_t fg, bg;
	int found;
	unsigned int x1;
	unsigned int y1;
	unsigned int i;

	memset(bitmap, 0, sizeof(bitmap));
	memset(rbitmap, 0, sizeof(bitmap));

	/* find out foreground/background color */

	bg = cpssp->NAME.pixel_buffer[y][x];
	for (y1 = 0; y1 < height; y1++) {
		for (x1 = 0; x1 < 8; x1++) {
			if (cpssp->NAME.pixel_buffer[y + y1][x + x1] != bg)
				goto found;
		}
	}
found:	;
	if (y1 != height) {
		fg = cpssp->NAME.pixel_buffer[y + y1][x + x1];
	} else {
		/* only one color */
		fg = ~bg;
	}

	/* setup black/white bitmap of character */

	for (y1 = 0; y1 < height; y1++) {
		for (x1 = 0; x1 < 8; x1++) {
			bitmap[y1] <<= 1;
			rbitmap[y1] <<= 1;
			if (cpssp->NAME.pixel_buffer[y + y1][x + x1] == fg) {
				bitmap[y1] |= 1;
			} else if (cpssp->NAME.pixel_buffer[y + y1][x + x1] == bg) {
				rbitmap[y1] |= 1;
			} else {
				/* more than 2 colors, probably in GFX mode */
				return false;
			}
		}
	}

	/* search for the character in font */
	switch (height) {
	case 8:
		for (i = 0; ; i++) {
			if (i == sizeof(glyph_8x8) / sizeof(glyph_8x8[0])) {
				found = CHAR_UNKNOWN;
				break;
			}
			if (memcmp(bitmap, glyph_8x8[i].pattern, 8) == 0
			 || memcmp(rbitmap, glyph_8x8[i].pattern, 8) == 0) {
				found = glyph_8x8[i].ord;
				break;
			}
		}

		cpssp->NAME.screen8[y / 8][x / 8] = found;
		break;
	case 14:
		for (i = 0; ; i++) {
			if (i == sizeof(glyph_8x14) / sizeof(glyph_8x14[0])) {
				found = CHAR_UNKNOWN;
				break;
			}
			if (memcmp(bitmap, glyph_8x14[i].pattern, 14) == 0
			 || memcmp(rbitmap, glyph_8x14[i].pattern, 14) == 0) {
				found = glyph_8x14[i].ord;
				break;
			}
		}

		cpssp->NAME.screen14[y / 14][x / 8] = found;
		break;
	case 16:
		for (i = 0; ; i++) {
			if (i == sizeof(glyph_8x16) / sizeof(glyph_8x16[0])) {
				found = CHAR_UNKNOWN;
				break;
			}
			if (memcmp(bitmap, glyph_8x16[i].pattern, 16) == 0
			 || memcmp(rbitmap, glyph_8x16[i].pattern, 16) == 0) {
				found = glyph_8x16[i].ord;
				break;
			}
		}

		cpssp->NAME.screen16[y / 16][x / 8] = found;
		break;
	default:
		assert(0); /* Cannot happen. */
	}
	return true;
}

static inline bool
NAME_(find_glyphs)(struct cpssp * cpssp)
{
	bool found;
	unsigned int x;
	unsigned int y;
	unsigned int xt;
	unsigned int yt;

	found = false;

	for (y = 0, yt = 0; y < cpssp->NAME.height; y += 16, yt++) {
		for (x = 0, xt = 0; x < cpssp->NAME.width; x += 8, xt++) {
			if (cpssp->NAME.dirty16[yt][xt]) {
				cpssp->NAME.dirty16[yt][xt] = 0;
				if (NAME_(check_tile)(cpssp, x, y, 16)) {
					found = true;
				}
			}
		}
	}

	for (y = 0, yt = 0; y < cpssp->NAME.height; y += 14, yt++) {
		for (x = 0, xt = 0; x < cpssp->NAME.width; x += 8, xt++) {
			if (cpssp->NAME.dirty14[yt][xt]) {
				cpssp->NAME.dirty14[yt][xt] = 0;
				if (NAME_(check_tile)(cpssp, x, y, 14)) {
					found = true;
				}
			}
		}
	}

	for (y = 0, yt = 0; y < cpssp->NAME.height; y += 8, yt++) {
		for (x = 0, xt = 0; x < cpssp->NAME.width; x += 8, xt++) {
			if (cpssp->NAME.dirty8[yt][xt]) {
				cpssp->NAME.dirty8[yt][xt] = 0;
				if (NAME_(check_tile)(cpssp, x, y, 8)) {
					found = true;
				}
			}
		}
	}

	return found;
}

static void
NAME_(size_set)(struct cpssp *cpssp, unsigned int w, unsigned int h)
{
	int y;

	assert(w <= MAX_WIDTH);
	assert(h <= MAX_HEIGHT);

	/*
	 * Clear old '\0' terminations.
	 */
	for (y = 0; y < cpssp->NAME.height / 16; y++) {
		cpssp->NAME.screen16[y][cpssp->NAME.width / FONT_WIDTH]
				= CHAR_UNKNOWN;
	}
	for (y = 0; y < cpssp->NAME.height / 14; y++) {
		cpssp->NAME.screen14[y][cpssp->NAME.width / FONT_WIDTH]
				= CHAR_UNKNOWN;
	}
	for (y = 0; y < cpssp->NAME.height / 8; y++) {
		cpssp->NAME.screen8[y][cpssp->NAME.width / FONT_WIDTH]
				= CHAR_UNKNOWN;
	}

	/*
	 * Set new size.
	 */
	cpssp->NAME.width = w;
	cpssp->NAME.height = h;

	/*
	 * Set new '\0' terminations.
	 */
	for (y = 0; y < cpssp->NAME.height / 16; y++) {
		cpssp->NAME.screen16[y][cpssp->NAME.width / FONT_WIDTH] = '\0';
	}
	for (y = 0; y < cpssp->NAME.height / 14; y++) {
		cpssp->NAME.screen14[y][cpssp->NAME.width / FONT_WIDTH] = '\0';
	}
	for (y = 0; y < cpssp->NAME.height / 8; y++) {
		cpssp->NAME.screen8[y][cpssp->NAME.width / FONT_WIDTH] = '\0';
	}

	/*
	 * Re-match everything...
	 */
	cpssp->NAME.match_necessary = true;
}

static void
NAME_(pixel_set)(
	struct cpssp *cpssp,
	unsigned int x, unsigned int y,
	uint8_t r, uint8_t g, uint8_t b
)
{
	assert(x < MAX_WIDTH);
	assert(y < MAX_HEIGHT);

	cpssp->NAME.pixel_buffer[y][x] = (r << 16) | (g << 8) | (b << 0);

	cpssp->NAME.dirty16[y / 16][x / 8] = 1;
	cpssp->NAME.dirty14[y / 14][x / 8] = 1;
	cpssp->NAME.dirty8[y /  8][x / 8] = 1;

	cpssp->NAME.match_necessary = true;
}

static void
NAME_(sync)(struct cpssp *cpssp)
{
	if (cpssp->NAME.match_necessary) {
		cpssp->NAME.match_necessary = false;
		if (NAME_(find_glyphs)(cpssp)) {
			cpssp->NAME.check_necessary = true;
		}
	}

	cpssp->NAME.sync_count++;

	if (2 <= cpssp->NAME.sync_count) {
		cpssp->NAME.sync_count = 0;
		if (cpssp->NAME.check_necessary) {
			cpssp->NAME.check_necessary = false;
			NAME_(check4pattern)(cpssp);
		}
	}
}

static void
NAME_(string_set)(void *s, const char* str)
{
	struct pattern* p = (struct pattern*)s;
	struct cpssp * cpssp = (struct cpssp *)p->_cpssp;
	int found;
	assert(str);

	if (*str) {
		if (sizeof(p->pattern) <= strlen(str)) {
			faum_log(FAUM_LOG_WARNING, "patternm", "asc",
				"watch string %s too long, ignoring\n", str);
		}

		strncpy(p->pattern, str, sizeof(p->pattern) - 1);

		p->pattern[sizeof(p->pattern) - 1] = '\0';
		p->active = true;

		found = NAME_(check4_one_pattern)(cpssp, p);
		p->last_event = found;
		sig_boolean_set(p->result, p, found);
	} else {
		p->pattern[0] = '\0';
		p->active = false;
		p->last_event = false;
		sig_boolean_set(p->result, p, 0);
	}
}

static void
NAME_(init)(
	struct sig_string *port_asc_text0,
	struct sig_boolean *port_asc_text0_state,
	struct sig_string *port_asc_text1,
	struct sig_boolean *port_asc_text1_state,
	struct sig_string *port_asc_text2,
	struct sig_boolean *port_asc_text2_state,
	struct sig_string *port_asc_text3,
	struct sig_boolean *port_asc_text3_state,
	struct cpssp * cpssp
)
{
	static const struct sig_string_funcs sf = {
		.set = NAME_(string_set)
	};
	int y;

	cpssp->NAME.width = 640;
	cpssp->NAME.height = 400;
	cpssp->NAME.check_necessary = false;
	cpssp->NAME.match_necessary = true;
	cpssp->NAME.sync_count = 0;

	/* initialize screen buffer */
	memset(cpssp->NAME.screen16, CHAR_UNKNOWN, sizeof(cpssp->NAME.screen16));
	memset(cpssp->NAME.screen14, CHAR_UNKNOWN, sizeof(cpssp->NAME.screen14));
	memset(cpssp->NAME.screen8, CHAR_UNKNOWN, sizeof(cpssp->NAME.screen8));

	/* initialize pixel buffer */
	memset(cpssp->NAME.pixel_buffer, 0, sizeof(cpssp->NAME.pixel_buffer));

	/* initialize dirty tile arrays
	 * (We have to look at each tile at least once
	 * to get a proper initial state of the screen.) */
	memset(cpssp->NAME.dirty16, 1, sizeof(cpssp->NAME.dirty16));
	memset(cpssp->NAME.dirty14, 1, sizeof(cpssp->NAME.dirty14));
	memset(cpssp->NAME.dirty8, 1, sizeof(cpssp->NAME.dirty8));

	/* add terminating '\0' to end of line (screen has one extra space in 
	 * x direction for this) */
	for (y = 0; y < cpssp->NAME.height / 16; y++) {
		cpssp->NAME.screen16[y][cpssp->NAME.width / 8] = '\0';
	}
	for (y = 0; y < cpssp->NAME.height / 14; y++) {
		cpssp->NAME.screen14[y][cpssp->NAME.width / 8] = '\0';
	}
	for (y = 0; y < cpssp->NAME.height / 8; y++) {
		cpssp->NAME.screen8[y][cpssp->NAME.width / 8] = '\0';
	}

	/* initialize members */
	for (y = 0; y < ARRAY_SIZE(cpssp->NAME.patterns); y++) {
		cpssp->NAME.patterns[y].active = false;
		cpssp->NAME.patterns[y].last_event = false;
		cpssp->NAME.patterns[y].pattern[0] = '\0';
		cpssp->NAME.patterns[y]._cpssp = cpssp;
	}

	cpssp->NAME.patterns[0].result = port_asc_text0_state;
	cpssp->NAME.patterns[1].result = port_asc_text1_state;
	cpssp->NAME.patterns[2].result = port_asc_text2_state;
	cpssp->NAME.patterns[3].result = port_asc_text3_state;

	sig_string_connect(port_asc_text0, &cpssp->NAME.patterns[0], &sf);
	sig_string_connect(port_asc_text1, &cpssp->NAME.patterns[1], &sf);
	sig_string_connect(port_asc_text2, &cpssp->NAME.patterns[2], &sf);
	sig_string_connect(port_asc_text3, &cpssp->NAME.patterns[3], &sf);
}

#undef ARRAY_SIZE
#undef CHAR_UNKNOWN
#undef TERM_MAX_WIDTH
#undef TERM_MAX_HEIGHT
#undef FONT_WIDTH
#undef FONT_HEIGHT

#endif /* BEHAVIOR */
