%csmethodmodifiers getLibSBMLVersion "
/**
 * Returns the version number of this copy of libSBML as an integer.
 *
 * @return the libSBML version as an integer; version 1.2.3 becomes 10203.
 */ public
";


%csmethodmodifiers getLibSBMLDottedVersion "
/**
 * Returns the version number of this copy of libSBML as a string.
 *
 * @return the libSBML version as a string; version 1.2.3 becomes
 * \'1.2.3\'.
 *
 * @see getLibSBMLVersionString()
 */ public
";


%csmethodmodifiers getLibSBMLVersionString "
/**
 * Returns the version number of this copy of libSBML as a string without
 * periods.
 *
 * @return the libSBML version as a string: version 1.2.3 becomes \'10203\'.
 *
 * @see getLibSBMLDottedVersion()
 */ public
";


%csmethodmodifiers isLibSBMLCompiledWith(const char* option) "
/**
 * Returns an indication whether libSBML has been compiled with
 * against a specific library. 
 *
 * @param option the library to test against, this can be one of
 *        \'expat\', \'libxml\', \'xerces-c\', \'bzip2\', \'zip\'
 * 
 * @return 0 in case the libSBML has not been compiled against 
 *         that library and non-zero otherwise (for libraries 
 *         that define an integer version number that number will 
 *         be returned).
 *
 * @see getLibSBMLDependencyVersionOf(string option)
 */ public
";


%csmethodmodifiers getLibSBMLDependencyVersionOf(const char* option) "
/**
 * Returns the version string for the dependency library used. 
 *
 * @param option the library for which the version
 *        should be retrieved, this can be one of
 *        \'expat\', \'libxml\', \'xerces-c\', \'bzip2\', \'zip\'
 * 
 * @return null in case libSBML has not been compiled against 
 *         that library and a version string otherwise.
 *
 * @see isLibSBMLCompiledWith(string option)
 */ public
";


%csmethodmodifiers OperationReturnValue_toString(int returnValue) "
/**
 * This method takes an SBML operation return value and returns a string representing
 * the code.
 *
 * @param returnValue the operation return value to convert to a string.
 *
 * @return a human readable name for the given
 * @if clike #OperationReturnValues_t value@else operation return value @endif.
 *
 * @note The caller does not own the returned string and is therefore not
 * allowed to modify it.
 */ public
";


%typemap(csimports) IdList "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html Maintains a list of SIds.
 * @internal
 */
"


%csmethodmodifiers IdList::append(const std::string& id) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers IdList::contains(const std::string& id) const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers IdList::empty() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers IdList::removeIdsBefore(const std::string& id) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers IdList::size() const "
/** */ /* libsbml-internal */ public
";


%typemap(csimports) IdentifierTransformer "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html Base class for identifier transformers.
 * @internal
 */
"


%typemap(csimports) ElementFilter "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html Base class for filter functions.
 *
 * @htmlinclude libsbml-facility-only-warning.html
 *
 * Some libSBML objects provide the ability to return lists of components.
 * To provide callers with greater control over exactly what is
 * returned, these methods take optional arguments in the form of filters.
 * The ElementFilter class is the parent class for these filters.
 */
"


%csmethodmodifiers ElementFilter::ElementFilter "
/**
   * Creates a new ElementFilter object.
   */ public
";


%csmethodmodifiers ElementFilter::filter(const SBase* element) "
/**
   * Predicate to test elements.
   *
   * This is the central predicate of the ElementFilter class.  In subclasses
   * of ElementFilter, callers should implement this method such that it
   * returns @c true for @p element arguments that are \'desirable\' and
   * @c false for those that are \'undesirable\' in whatever filtering context the
   * ElementFilter subclass is designed to be used.
   *
   * @param element the element to be tested.
   *
   * @return @c true if the @p element is desirable or should be kept,
   * @c false otherwise.
   */ public virtual
";


%csmethodmodifiers ElementFilter::getUserData "
/**
   * Returns the user data that has been previously set via setUserData().
   *
   * Callers can attach private data to ElementFilter objects using
   * setUserData().  This user data can be used by an application to store
   * custom information to be accessed by the ElementFilter in its work.  In
   * case of a deep copy, the data will passed as it is.  The attribute will
   * never be interpreted by libSBML.
   *
   * @return the user data of this node, or @c null if no user data has been
   * set.
   *
   * @warning This <em>user data</em> is specific to an ElementFilter object
   * instance, and is not the same as the user data that may be attached to
   * an SBML object using SBase::setUserData().
   *
   * @see setUserData()
   */ public
";


%csmethodmodifiers ElementFilter::setUserData(void* userData) "
/**
   * Sets the user data of this element.
   *
   * Callers can attach private data to ElementFilter objects using this
   * method, and retrieve them using getUserData().  Such user data can be
   * used by an application to store information to be accessed by the
   * ElementFilter in its work.  In case of a deep copy, this data will
   * passed as it is.  The attribute will never be interpreted by libSBML.
   *
   * @param userData specifies the new user data.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @warning This <em>user data</em> is specific to an ElementFilter object
   * instance, and is not the same as the user data that may be attached to
   * an SBML object using SBase::setUserData().
   *
   * @see getUserData()
   */ public
";


%typemap(csimports) SBMLReader "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html File and text-string SBML reader.
 *
 * @htmlinclude not-sbml-warning.html
 *
 * The SBMLReader class provides the main interface for reading SBML content
 * from files and strings.  The methods for reading SBML all return an
 * SBMLDocument object representing the results.  In the case of failures
 * (such as if the SBML contains errors or a file cannot be read), the errors
 * will be recorded with the SBMLErrorLog object kept in the SBMLDocument
 * returned by SBMLReader.  Consequently, immediately after calling a method
 * on SBMLReader, callers should always check for errors and warnings using
 * the methods for this purpose provided by SBMLDocument.
 *
 * For convenience as well as easy access from other languages besides C++,
 * this file also defines two global functions,
 * @sbmlglobalfunction{readSBML, String} and
 * @sbmlglobalfunction{readSBMLFromString, String}.  They are
 * equivalent to creating an SBMLReader object and then calling the
 * @if python @link SBMLReader::readSBML() SBMLReader.readSBML()@endlink@endif@if java SBMLReader::readSBML(String)@endif@if cpp SBMLReader::readSBML()@endif@if csharp SBMLReader.readSBML()@endif and
 * @if python @link SBMLReader::readSBMLFromString() SBMLReader.readSBMLFromString()@endlink@endif@if java SBMLReader::readSBMLFromString(String)@endif@if cpp SBMLReader::readSBMLFromString()@endif@if csharp SBMLReader.readSBMLFromString()@endif methods, respectively.
 *
 * @section compression Support for reading compressed files
 *
 * LibSBML provides support for reading (as well as writing) compressed
 * SBML files.  The process is transparent to the calling
 * application---the application does not need to do anything
 * deliberate to invoke the functionality.  If a given SBML filename ends
 * with an extension for the @em gzip, @em zip or @em bzip2 compression
 * formats (respectively, @c .gz, @c .zip, or @c .bz2), then the methods
 * @if python @link SBMLReader::readSBML() SBMLReader.readSBML()@endlink@endif@if java @link SBMLReader::readSBML(String) SBMLReader.readSBML(String)@endlink@endif@if cpp SBMLReader::readSBML()@endif@if csharp SBMLReader.readSBML()@endif and
 * @if python @link SBMLWriter::writeSBML() SBMLWriter.writeSBML()@endlink@endif@if java @link SBMLWriter::writeSBML(String) SBMLWriter.writeSBML(String)@endlink@endif@if cpp SBMLWriter::writeSBML()@endif@if csharp SBMLWriter.writeSBML()@endif
 * will automatically decompress and compress the file while reading and
 * writing it.  If the filename has no such extension, it will be read and
 * written uncompressed as normal.
 *
 * The compression feature requires that the @em zlib (for @em gzip and @em
 * zip formats) and/or @em bzip2 (for @em bzip2 format) be available on the
 * system running libSBML, and that libSBML was configured with their
 * support compiled-in.  Please see the libSBML
 * @if java <a href=\'../../../libsbml-installation.html\'>installation instructions</a> @else <a href=\'libsbml-installation.html\'>installation instructions</a>@endif
 * for more information about this.  The methods
 * @if java SBMLReader::hasZlib()@else hasZlib()@endif and
 * @if java SBMLReader::hasBzip2()@else hasBzip2()@endif
 * can be used by an application to query at run-time whether support
 * for the compression libraries is available in the present copy of
 * libSBML.
 *
 * Support for compression is not mandated by the SBML standard, but
 * applications may find it helpful, particularly when large SBML models
 * are being communicated across data links of limited bandwidth.
 */
"


%csmethodmodifiers SBMLReader::SBMLReader "
/**
   * Creates a new SBMLReader object and returns it.
   *
   * The libSBML SBMLReader object offers methods for reading SBML in
   * XML form from files and text strings.
   */ public
";


%csmethodmodifiers SBMLReader::readSBML(const std::string& filename) "
/**
   *
 * Reads an SBML document from the given file.
 *
 * If the file named @p filename does not exist or its content is not valid
 * SBML, one or more errors will be logged with the SBMLDocument object
 * returned by this method.  Callers can use the methods on SBMLDocument such
 * as
 * @if python @link libsbml.SBMLDocument.getNumErrors() SBMLDocument.getNumErrors()@endlink@endif,
 * @if conly SBMLDocument_getNumErrors() @else SBMLDocument::getNumErrors()@endif
 * and
 * @if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
 * @if java SBMLDocument::getError(long)@endif
 * @if cpp SBMLDocument::getError()@endif
 * @if csharp SBMLDocument::getError()@endif
 * @if conly SBMLDocument_getError()@endif
 * to get the errors.  The object returned by
 * @if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
 * @if java SBMLDocument::getError(long)@endif
 * @if cpp SBMLDocument::getError()@endif
 * @if csharp SBMLDocument::getError()@endif
 * @if conly SBMLDocument_getError()@endif
 * is an SBMLError object, and it has methods to get the error code,
 * category, and severity level of the problem, as well as a textual
 * description of the problem.  The possible severity levels range from
 * informational messages to fatal errors; see the documentation for
 * @if conly SBMLError_t @else SBMLError@endif
 * for more information.
 *
 * If the file @p filename could not be read, the file-reading error will
 * appear first.  The error code @if clike (a value drawn from the
 * enumeration #XMLErrorCode_t)@endif can provide a clue about what
 * happened.  For example, a file might be unreadable (either because it does
 * not actually exist or because the user does not have the necessary access
 * privileges to read it) or some sort of file operation error may have been
 * reported by the underlying operating system.  Callers can check for these
 * situations using a program fragment such as the following:
 * @if cpp
@code{.cpp}
SBMLReader reader;
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc->getNumErrors() > 0)
{
  if (doc->getError(0)->getErrorId() == XMLError::XMLFileUnreadable)
  {
    // Handle case of unreadable file here.
  }
  else if (doc->getError(0)->getErrorId() == XMLError::XMLFileOperationError)
  {
    // Handle case of other file operation error here.
  }
  else
  {
    // Handle other cases -- see error codes defined in XMLErrorCode_t
    // for other possible cases to check.
  }
}
@endcode
@endif
@if conly
@code{.c}
SBMLReader_t   *sr;
SBMLDocument_t *d;

sr = SBMLReader_create();

d = SBMLReader_readSBML(sr, filename);

if (SBMLDocument_getNumErrors(d) > 0)
{
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_FILE_NOT_FOUND)
  {
     ...
  }
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_NOT_SBML)
  {
     ...
  }
}
@endcode
@endif
@if java
@code{.java}
SBMLReader reader = new SBMLReader();
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbml.libsbml.XMLFileUnreadable)
    {
        // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbml.libsbml.XMLFileOperationError)
    {
        // Handle case of other file operation error here.
    }
    else
    {
        // Handle other error cases.
    }
}
@endcode
@endif
@if python
@code{.py}
reader = SBMLReader()
if reader == None:
  # Handle the truly exceptional case of no object created here.

doc = reader.readSBMLFromFile(filename)
if doc.getNumErrors() > 0:
  if doc.getError(0).getErrorId() == XMLFileUnreadable:
    # Handle case of unreadable file here.
  elif doc.getError(0).getErrorId() == XMLFileOperationError:
    # Handle case of other file error here.
  else:
    # Handle other error cases here.
@endcode
@endif
@if csharp
@code{.cs}
SBMLReader reader = new SBMLReader();
SBMLDocument doc = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileUnreadable)
    {
         // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileOperationError)
    {
         // Handle case of other file operation error here.
    }
    else
    {
         // Handle other cases -- see error codes defined in XMLErrorCode_t
         // for other possible cases to check.
    }
 }
@endcode
@endif
 *
 *
 * 
 * If the given filename ends with the suffix @c \'.gz\' (for example,
 * @c \'myfile.xml.gz\'), the file is assumed to be compressed in @em gzip
 * format and will be automatically decompressed upon reading.
 * Similarly, if the given filename ends with @c \'.zip\' or @c \'.bz2\', the
 * file is assumed to be compressed in @em zip or @em bzip2 format
 * (respectively).  Files whose names lack these suffixes will be read
 * uncompressed.  Note that if the file is in @em zip format but the
 * archive contains more than one file, only the first file in the
 * archive will be read and the rest ignored.
 *
 *
 *
 *
 * 
 * To read a gzip/zip file, libSBML needs to be configured and linked with the
 * <a target=\'_blank\' href=\'http://www.zlib.net/\'>zlib</a> library at compile
 * time.  It also needs to be linked with the <a target=\'_blank\'
 * href=\'\'>bzip2</a> library to read files in <em>bzip2</em> format.  (Both of
 * these are the default configurations for libSBML.)  Errors about unreadable
 * files will be logged if a compressed filename is given and libSBML was
 * <em>not</em> linked with the corresponding required library.
 *
 *
 *
 *
   *
   * This method is identical to
   * @if python @link SBMLReader::readSBMLFromFile() SBMLReader.readSBMLFromFile()@endlink@endif@if java @link SBMLReader::readSBMLFromFile(String) SBMLReader.readSBMLFromFile(String)@endlink@endif@if cpp SBMLReader::readSBMLFromFile()@endif@if csharp SBMLReader.readSBMLFromFile()@endif.
   *
   * @param filename the name or full pathname of the file to be read.
   *
   * @return a pointer to the SBMLDocument object created from the SBML
   * content in @p filename.
   *
   *
 * @note LibSBML versions 2.x and later versions behave differently in
 * error handling in several respects.  One difference is how early some
 * errors are caught and whether libSBML continues processing a file in
 * the face of some early errors.  In general, libSBML versions after 2.x
 * stop parsing SBML inputs sooner than libSBML version 2.x in the face
 * of XML errors, because the errors may invalidate any further SBML
 * content.  For example, a missing XML declaration at the beginning of
 * the file was ignored by libSBML 2.x but in version 3.x and later, it
 * will cause libSBML to stop parsing the rest of the input altogether.
 * While this behavior may seem more severe and intolerant, it was
 * necessary in order to provide uniform behavior regardless of which
 * underlying XML parser (Expat, Xerces, libxml2) is being used by
 * libSBML.  The XML parsers themselves behave differently in their error
 * reporting, and sometimes libSBML has to resort to the lowest common
 * denominator.
 *
 *
   *
   * @see readSBMLFromString(@if java String@endif)
   * @see SBMLError
   * @see SBMLDocument
   */ public
";


%csmethodmodifiers SBMLReader::readSBMLFromFile(const std::string& filename) "
/**
   *
 * Reads an SBML document from the given file.
 *
 * If the file named @p filename does not exist or its content is not valid
 * SBML, one or more errors will be logged with the SBMLDocument object
 * returned by this method.  Callers can use the methods on SBMLDocument such
 * as
 * @if python @link libsbml.SBMLDocument.getNumErrors() SBMLDocument.getNumErrors()@endlink@endif,
 * @if conly SBMLDocument_getNumErrors() @else SBMLDocument::getNumErrors()@endif
 * and
 * @if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
 * @if java SBMLDocument::getError(long)@endif
 * @if cpp SBMLDocument::getError()@endif
 * @if csharp SBMLDocument::getError()@endif
 * @if conly SBMLDocument_getError()@endif
 * to get the errors.  The object returned by
 * @if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
 * @if java SBMLDocument::getError(long)@endif
 * @if cpp SBMLDocument::getError()@endif
 * @if csharp SBMLDocument::getError()@endif
 * @if conly SBMLDocument_getError()@endif
 * is an SBMLError object, and it has methods to get the error code,
 * category, and severity level of the problem, as well as a textual
 * description of the problem.  The possible severity levels range from
 * informational messages to fatal errors; see the documentation for
 * @if conly SBMLError_t @else SBMLError@endif
 * for more information.
 *
 * If the file @p filename could not be read, the file-reading error will
 * appear first.  The error code @if clike (a value drawn from the
 * enumeration #XMLErrorCode_t)@endif can provide a clue about what
 * happened.  For example, a file might be unreadable (either because it does
 * not actually exist or because the user does not have the necessary access
 * privileges to read it) or some sort of file operation error may have been
 * reported by the underlying operating system.  Callers can check for these
 * situations using a program fragment such as the following:
 * @if cpp
@code{.cpp}
SBMLReader reader;
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc->getNumErrors() > 0)
{
  if (doc->getError(0)->getErrorId() == XMLError::XMLFileUnreadable)
  {
    // Handle case of unreadable file here.
  }
  else if (doc->getError(0)->getErrorId() == XMLError::XMLFileOperationError)
  {
    // Handle case of other file operation error here.
  }
  else
  {
    // Handle other cases -- see error codes defined in XMLErrorCode_t
    // for other possible cases to check.
  }
}
@endcode
@endif
@if conly
@code{.c}
SBMLReader_t   *sr;
SBMLDocument_t *d;

sr = SBMLReader_create();

d = SBMLReader_readSBML(sr, filename);

if (SBMLDocument_getNumErrors(d) > 0)
{
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_FILE_NOT_FOUND)
  {
     ...
  }
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_NOT_SBML)
  {
     ...
  }
}
@endcode
@endif
@if java
@code{.java}
SBMLReader reader = new SBMLReader();
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbml.libsbml.XMLFileUnreadable)
    {
        // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbml.libsbml.XMLFileOperationError)
    {
        // Handle case of other file operation error here.
    }
    else
    {
        // Handle other error cases.
    }
}
@endcode
@endif
@if python
@code{.py}
reader = SBMLReader()
if reader == None:
  # Handle the truly exceptional case of no object created here.

doc = reader.readSBMLFromFile(filename)
if doc.getNumErrors() > 0:
  if doc.getError(0).getErrorId() == XMLFileUnreadable:
    # Handle case of unreadable file here.
  elif doc.getError(0).getErrorId() == XMLFileOperationError:
    # Handle case of other file error here.
  else:
    # Handle other error cases here.
@endcode
@endif
@if csharp
@code{.cs}
SBMLReader reader = new SBMLReader();
SBMLDocument doc = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileUnreadable)
    {
         // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileOperationError)
    {
         // Handle case of other file operation error here.
    }
    else
    {
         // Handle other cases -- see error codes defined in XMLErrorCode_t
         // for other possible cases to check.
    }
 }
@endcode
@endif
 *
 *
 * 
 * If the given filename ends with the suffix @c \'.gz\' (for example,
 * @c \'myfile.xml.gz\'), the file is assumed to be compressed in @em gzip
 * format and will be automatically decompressed upon reading.
 * Similarly, if the given filename ends with @c \'.zip\' or @c \'.bz2\', the
 * file is assumed to be compressed in @em zip or @em bzip2 format
 * (respectively).  Files whose names lack these suffixes will be read
 * uncompressed.  Note that if the file is in @em zip format but the
 * archive contains more than one file, only the first file in the
 * archive will be read and the rest ignored.
 *
 *
 *
 *
 * 
 * To read a gzip/zip file, libSBML needs to be configured and linked with the
 * <a target=\'_blank\' href=\'http://www.zlib.net/\'>zlib</a> library at compile
 * time.  It also needs to be linked with the <a target=\'_blank\'
 * href=\'\'>bzip2</a> library to read files in <em>bzip2</em> format.  (Both of
 * these are the default configurations for libSBML.)  Errors about unreadable
 * files will be logged if a compressed filename is given and libSBML was
 * <em>not</em> linked with the corresponding required library.
 *
 *
 *
 *
   *
   * This method is identical to
   * @if python @link SBMLReader::readSBML() SBMLReader.readSBML()@endlink@endif@if java @link SBMLReader::readSBML(String) SBMLReader.readSBML(String)@endlink@endif@if cpp SBMLReader::readSBML()@endif@if csharp SBMLReader.readSBML()@endif.
   *
   * @param filename the name or full pathname of the file to be read.
   *
   * @return a pointer to the SBMLDocument object created from the SBML
   * content in @p filename.
   *
   *
 * @note LibSBML versions 2.x and later versions behave differently in
 * error handling in several respects.  One difference is how early some
 * errors are caught and whether libSBML continues processing a file in
 * the face of some early errors.  In general, libSBML versions after 2.x
 * stop parsing SBML inputs sooner than libSBML version 2.x in the face
 * of XML errors, because the errors may invalidate any further SBML
 * content.  For example, a missing XML declaration at the beginning of
 * the file was ignored by libSBML 2.x but in version 3.x and later, it
 * will cause libSBML to stop parsing the rest of the input altogether.
 * While this behavior may seem more severe and intolerant, it was
 * necessary in order to provide uniform behavior regardless of which
 * underlying XML parser (Expat, Xerces, libxml2) is being used by
 * libSBML.  The XML parsers themselves behave differently in their error
 * reporting, and sometimes libSBML has to resort to the lowest common
 * denominator.
 *
 *
   *
   * @see readSBMLFromString(@if java String@endif)
   * @see SBMLError
   * @see SBMLDocument
   */ public
";


%csmethodmodifiers SBMLReader::readSBMLFromString(const std::string& xml) "
/**
   *
 * Reads an SBML document from a text string.
 *
 * This method is flexible with respect to the presence of an XML
 * declaration at the beginning of the string.  In particular, if the
 * string in @p xml does not begin with the XML declaration
 * @verbatim
<?xml version=\'1.0\' encoding=\'UTF-8\'?>
@endverbatim
 * then this method will automatically prepend the declaration
 * to @p xml.
 *
 * This method will log a fatal error if the content given in the parameter
 * @p xml is not in SBML format.  See the method documentation for
 * @if conly SBMLReader_readSBML() 
 * @elseif java SBMLReader::readSBML( String )
 * @else SBMLReader::readSBML()
 * @endif
 * for an example of code for testing the returned error code.
 *
 *
   *
   * @param xml a string containing a full SBML model.
   *
   * @return a pointer to the SBMLDocument created from the SBML content,
   * or a null pointer if @p xml is @c null.
   *
   *
 * @note When using this method to read an SBMLDocument that uses the SBML
 * Level&nbsp;3 Hierarchical %Model Composition package (comp) the document
 * location cannot be set automatically. Thus, if the model contains
 * references to ExternalModelDefinition objects, it will be necessary to
 * manually set the document URI location
 * (@if conly SBMLDocument_setLocationURI()
 * @elseif java SBMLDocument::setLocationURI( String ) 
 * @else SBMLDocument::setLocationURI()
 * @endif
 * ) in order to facilitate resolving these models.
   *
   * @see SBMLReader::readSBML(@if java String@endif)
   */ public
";


%csmethodmodifiers SBMLReader::hasZlib "
/**
   * Static method; returns @c true if this copy of libSBML supports
   * <i>gzip</I> and <i>zip</i> format compression.
   *
   * @return @c true if libSBML has been linked with the <i>zlib</i>
   * library, @c false otherwise.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   *
   * @see @if clike hasBzip2() @else SBMLReader::hasBzip2()@endif
   */ public
";


%csmethodmodifiers SBMLReader::hasBzip2 "
/**
   * Static method; returns @c true if this copy of libSBML supports
   * <i>bzip2</i> format compression.
   *
   * @return @c true if libSBML is linked with the <i>bzip2</i>
   * libraries, @c false otherwise.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   *
   * @see @if clike hasZlib() @else SBMLReader::hasZlib()@endif
   */ public
";


%csmethodmodifiers SBMLReader::readInternal(const char* content, bool isFile = true) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers readSBML(const char *filename) "
/**
 *
 * Reads an SBML document from the given file.
 *
 * If the file named @p filename does not exist or its content is not valid
 * SBML, one or more errors will be logged with the SBMLDocument object
 * returned by this method.  Callers can use the methods on SBMLDocument such
 * as
 * @if python @link libsbml.SBMLDocument.getNumErrors() SBMLDocument.getNumErrors()@endlink@endif,
 * @if conly SBMLDocument_getNumErrors() @else SBMLDocument::getNumErrors()@endif
 * and
 * @if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
 * @if java SBMLDocument::getError(long)@endif
 * @if cpp SBMLDocument::getError()@endif
 * @if csharp SBMLDocument::getError()@endif
 * @if conly SBMLDocument_getError()@endif
 * to get the errors.  The object returned by
 * @if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
 * @if java SBMLDocument::getError(long)@endif
 * @if cpp SBMLDocument::getError()@endif
 * @if csharp SBMLDocument::getError()@endif
 * @if conly SBMLDocument_getError()@endif
 * is an SBMLError object, and it has methods to get the error code,
 * category, and severity level of the problem, as well as a textual
 * description of the problem.  The possible severity levels range from
 * informational messages to fatal errors; see the documentation for
 * @if conly SBMLError_t @else SBMLError@endif
 * for more information.
 *
 * If the file @p filename could not be read, the file-reading error will
 * appear first.  The error code @if clike (a value drawn from the
 * enumeration #XMLErrorCode_t)@endif can provide a clue about what
 * happened.  For example, a file might be unreadable (either because it does
 * not actually exist or because the user does not have the necessary access
 * privileges to read it) or some sort of file operation error may have been
 * reported by the underlying operating system.  Callers can check for these
 * situations using a program fragment such as the following:
 * @if cpp
@code{.cpp}
SBMLReader reader;
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc->getNumErrors() > 0)
{
  if (doc->getError(0)->getErrorId() == XMLError::XMLFileUnreadable)
  {
    // Handle case of unreadable file here.
  }
  else if (doc->getError(0)->getErrorId() == XMLError::XMLFileOperationError)
  {
    // Handle case of other file operation error here.
  }
  else
  {
    // Handle other cases -- see error codes defined in XMLErrorCode_t
    // for other possible cases to check.
  }
}
@endcode
@endif
@if conly
@code{.c}
SBMLReader_t   *sr;
SBMLDocument_t *d;

sr = SBMLReader_create();

d = SBMLReader_readSBML(sr, filename);

if (SBMLDocument_getNumErrors(d) > 0)
{
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_FILE_NOT_FOUND)
  {
     ...
  }
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_NOT_SBML)
  {
     ...
  }
}
@endcode
@endif
@if java
@code{.java}
SBMLReader reader = new SBMLReader();
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbml.libsbml.XMLFileUnreadable)
    {
        // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbml.libsbml.XMLFileOperationError)
    {
        // Handle case of other file operation error here.
    }
    else
    {
        // Handle other error cases.
    }
}
@endcode
@endif
@if python
@code{.py}
reader = SBMLReader()
if reader == None:
  # Handle the truly exceptional case of no object created here.

doc = reader.readSBMLFromFile(filename)
if doc.getNumErrors() > 0:
  if doc.getError(0).getErrorId() == XMLFileUnreadable:
    # Handle case of unreadable file here.
  elif doc.getError(0).getErrorId() == XMLFileOperationError:
    # Handle case of other file error here.
  else:
    # Handle other error cases here.
@endcode
@endif
@if csharp
@code{.cs}
SBMLReader reader = new SBMLReader();
SBMLDocument doc = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileUnreadable)
    {
         // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileOperationError)
    {
         // Handle case of other file operation error here.
    }
    else
    {
         // Handle other cases -- see error codes defined in XMLErrorCode_t
         // for other possible cases to check.
    }
 }
@endcode
@endif
 *
 *
 * 
 * If the given filename ends with the suffix @c \'.gz\' (for example,
 * @c \'myfile.xml.gz\'), the file is assumed to be compressed in @em gzip
 * format and will be automatically decompressed upon reading.
 * Similarly, if the given filename ends with @c \'.zip\' or @c \'.bz2\', the
 * file is assumed to be compressed in @em zip or @em bzip2 format
 * (respectively).  Files whose names lack these suffixes will be read
 * uncompressed.  Note that if the file is in @em zip format but the
 * archive contains more than one file, only the first file in the
 * archive will be read and the rest ignored.
 *
 *
 *
 *
 * 
 * To read a gzip/zip file, libSBML needs to be configured and linked with the
 * <a target=\'_blank\' href=\'http://www.zlib.net/\'>zlib</a> library at compile
 * time.  It also needs to be linked with the <a target=\'_blank\'
 * href=\'\'>bzip2</a> library to read files in <em>bzip2</em> format.  (Both of
 * these are the default configurations for libSBML.)  Errors about unreadable
 * files will be logged if a compressed filename is given and libSBML was
 * <em>not</em> linked with the corresponding required library.
 *
 *
 *
 *
 *
 * @if conly
 * @memberof SBMLReader_t
 * @endif
 */ public
";


%csmethodmodifiers readSBMLFromFile(const char *filename) "
/**
 *
 * Reads an SBML document from the given file.
 *
 * If the file named @p filename does not exist or its content is not valid
 * SBML, one or more errors will be logged with the SBMLDocument object
 * returned by this method.  Callers can use the methods on SBMLDocument such
 * as
 * @if python @link libsbml.SBMLDocument.getNumErrors() SBMLDocument.getNumErrors()@endlink@endif,
 * @if conly SBMLDocument_getNumErrors() @else SBMLDocument::getNumErrors()@endif
 * and
 * @if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
 * @if java SBMLDocument::getError(long)@endif
 * @if cpp SBMLDocument::getError()@endif
 * @if csharp SBMLDocument::getError()@endif
 * @if conly SBMLDocument_getError()@endif
 * to get the errors.  The object returned by
 * @if python @link libsbml.SBMLDocument.getError() SBMLDocument.getError()@endlink@endif
 * @if java SBMLDocument::getError(long)@endif
 * @if cpp SBMLDocument::getError()@endif
 * @if csharp SBMLDocument::getError()@endif
 * @if conly SBMLDocument_getError()@endif
 * is an SBMLError object, and it has methods to get the error code,
 * category, and severity level of the problem, as well as a textual
 * description of the problem.  The possible severity levels range from
 * informational messages to fatal errors; see the documentation for
 * @if conly SBMLError_t @else SBMLError@endif
 * for more information.
 *
 * If the file @p filename could not be read, the file-reading error will
 * appear first.  The error code @if clike (a value drawn from the
 * enumeration #XMLErrorCode_t)@endif can provide a clue about what
 * happened.  For example, a file might be unreadable (either because it does
 * not actually exist or because the user does not have the necessary access
 * privileges to read it) or some sort of file operation error may have been
 * reported by the underlying operating system.  Callers can check for these
 * situations using a program fragment such as the following:
 * @if cpp
@code{.cpp}
SBMLReader reader;
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc->getNumErrors() > 0)
{
  if (doc->getError(0)->getErrorId() == XMLError::XMLFileUnreadable)
  {
    // Handle case of unreadable file here.
  }
  else if (doc->getError(0)->getErrorId() == XMLError::XMLFileOperationError)
  {
    // Handle case of other file operation error here.
  }
  else
  {
    // Handle other cases -- see error codes defined in XMLErrorCode_t
    // for other possible cases to check.
  }
}
@endcode
@endif
@if conly
@code{.c}
SBMLReader_t   *sr;
SBMLDocument_t *d;

sr = SBMLReader_create();

d = SBMLReader_readSBML(sr, filename);

if (SBMLDocument_getNumErrors(d) > 0)
{
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_FILE_NOT_FOUND)
  {
     ...
  }
  if (XMLError_getId(SBMLDocument_getError(d, 0))
      == SBML_READ_ERROR_NOT_SBML)
  {
     ...
  }
}
@endcode
@endif
@if java
@code{.java}
SBMLReader reader = new SBMLReader();
SBMLDocument doc  = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbml.libsbml.XMLFileUnreadable)
    {
        // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbml.libsbml.XMLFileOperationError)
    {
        // Handle case of other file operation error here.
    }
    else
    {
        // Handle other error cases.
    }
}
@endcode
@endif
@if python
@code{.py}
reader = SBMLReader()
if reader == None:
  # Handle the truly exceptional case of no object created here.

doc = reader.readSBMLFromFile(filename)
if doc.getNumErrors() > 0:
  if doc.getError(0).getErrorId() == XMLFileUnreadable:
    # Handle case of unreadable file here.
  elif doc.getError(0).getErrorId() == XMLFileOperationError:
    # Handle case of other file error here.
  else:
    # Handle other error cases here.
@endcode
@endif
@if csharp
@code{.cs}
SBMLReader reader = new SBMLReader();
SBMLDocument doc = reader.readSBMLFromFile(filename);

if (doc.getNumErrors() > 0)
{
    if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileUnreadable)
    {
         // Handle case of unreadable file here.
    }
    else if (doc.getError(0).getErrorId() == libsbmlcs.libsbml.XMLFileOperationError)
    {
         // Handle case of other file operation error here.
    }
    else
    {
         // Handle other cases -- see error codes defined in XMLErrorCode_t
         // for other possible cases to check.
    }
 }
@endcode
@endif
 *
 *
 * 
 * If the given filename ends with the suffix @c \'.gz\' (for example,
 * @c \'myfile.xml.gz\'), the file is assumed to be compressed in @em gzip
 * format and will be automatically decompressed upon reading.
 * Similarly, if the given filename ends with @c \'.zip\' or @c \'.bz2\', the
 * file is assumed to be compressed in @em zip or @em bzip2 format
 * (respectively).  Files whose names lack these suffixes will be read
 * uncompressed.  Note that if the file is in @em zip format but the
 * archive contains more than one file, only the first file in the
 * archive will be read and the rest ignored.
 *
 *
 *
 *
 * 
 * To read a gzip/zip file, libSBML needs to be configured and linked with the
 * <a target=\'_blank\' href=\'http://www.zlib.net/\'>zlib</a> library at compile
 * time.  It also needs to be linked with the <a target=\'_blank\'
 * href=\'\'>bzip2</a> library to read files in <em>bzip2</em> format.  (Both of
 * these are the default configurations for libSBML.)  Errors about unreadable
 * files will be logged if a compressed filename is given and libSBML was
 * <em>not</em> linked with the corresponding required library.
 *
 *
 *
 *
 *
 * @param filename the name or full pathname of the file to be read.
 *
 * @return a pointer to the SBMLDocument structure created from the SBML
 * content in @p filename.
 *
 * @if conly
 * @memberof SBMLReader_t
 * @endif
 */ public
";


%csmethodmodifiers readSBMLFromString(const char *xml) "
/**
 *
 * Reads an SBML document from a text string.
 *
 * This method is flexible with respect to the presence of an XML
 * declaration at the beginning of the string.  In particular, if the
 * string in @p xml does not begin with the XML declaration
 * @verbatim
<?xml version=\'1.0\' encoding=\'UTF-8\'?>
@endverbatim
 * then this method will automatically prepend the declaration
 * to @p xml.
 *
 * This method will log a fatal error if the content given in the parameter
 * @p xml is not in SBML format.  See the method documentation for
 * @if conly SBMLReader_readSBML() 
 * @elseif java SBMLReader::readSBML( String )
 * @else SBMLReader::readSBML()
 * @endif
 * for an example of code for testing the returned error code.
 *
 *
 *
 * @param xml a string containing a full SBML model.
 *
 * @return a pointer to the SBMLDocument structure created from the SBML
 * content in @p xml.
 *
 *
 * @note When using this method to read an SBMLDocument that uses the SBML
 * Level&nbsp;3 Hierarchical %Model Composition package (comp) the document
 * location cannot be set automatically. Thus, if the model contains
 * references to ExternalModelDefinition objects, it will be necessary to
 * manually set the document URI location
 * (@if conly SBMLDocument_setLocationURI()
 * @elseif java SBMLDocument::setLocationURI( String ) 
 * @else SBMLDocument::setLocationURI()
 * @endif
 * ) in order to facilitate resolving these models.
 *
 * @if conly
 * @memberof SBMLReader_t
 * @endif
 */ public
";


%typemap(csimports) SBMLWriter "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html File and text-string SBML writer.
 *
 * @htmlinclude not-sbml-warning.html
 *
 * The SBMLWriter class is the converse of SBMLReader, and provides the
 * main interface for serializing SBML models into XML and writing the
 * result to an output stream or to files and text strings.  The methods 
 * for writing SBML all take an SBMLDocument object and a destination.  
 * They return a boolean or integer value to indicate success or failure.
 *
 * @section sbmlwriter-compression Support for writing compressed files
 *
 * LibSBML provides support for writing (as well as reading) compressed
 * SBML files.  The process is transparent to the calling
 * application---the application does not need to do anything
 * deliberate to invoke the functionality.  If a given SBML filename ends
 * with an extension for the @em gzip, @em zip or @em bzip2 compression
 * formats (respectively, <code>&quot;.gz&quot;</code>,
 * <code>&quot;.zip&quot;</code>, or <code>&quot;.bz2&quot;</code>),
 * then the methods
 * SBMLWriter::writeSBML(@if java SBMLDocument, String@endif)
 * and SBMLReader::readSBML(@if java String@endif)
 * will automatically compress and decompress the file while writing and
 * reading it.  If the filename has no such extension, it
 * will be written and read uncompressed as normal.
 *
 * The compression feature requires that the @em zlib (for @em gzip and @em
 * zip formats) and/or @em bzip2 (for @em bzip2 format) be available on the
 * system running libSBML, and that libSBML was configured with their
 * support compiled-in.  Please see the libSBML @if clike <a href=\'libsbml-installation.html\'>installation instructions</a>@endif@if python <a href=\'libsbml-installation.html\'>installation instructions</a>@endif@if java  <a href=\'../../../libsbml-installation.html\'>installation instructions</a>@endif for 
 * more information about this.  The methods
 * SBMLWriter::hasZlib() and
 * SBMLWriter::hasBzip2()
 * can be used by an application to query at run-time whether support
 * for the compression libraries is available in the present copy of
 * libSBML.
 *
 * Support for compression is not mandated by the SBML standard, but
 * applications may find it helpful, particularly when large SBML models
 * are being communicated across data links of limited bandwidth.
 */
"


%csmethodmodifiers SBMLWriter::SBMLWriter "
/**
   * Creates a new SBMLWriter.
   *
   * The libSBML SBMLWriter objects offer methods for writing SBML in
   * XML form to files and text strings.
   */ public
";


%csmethodmodifiers SBMLWriter::setProgramName(const std::string& name) "
/**
   * Sets the name of this program, i.e., the program that is about to
   * write out the SBMLDocument.
   *
   * If the program name and version are set (see
   * SBMLWriter::setProgramVersion(@if java String@endif)), the
   * following XML comment, intended for human consumption, will be written
   * at the beginning of the XML document:
   * @verbatim
 <!-- Created by <program name> version <program version>
 on yyyy-MM-dd HH:mm with libSBML version <libsbml version>. -->
 @endverbatim
   *
   * If the program name and version are not set at some point before
   * calling the writeSBML() methods, no such comment is written out.
   *
   * @param name the name of this program (where \'this program\' refers to
   * program in which libSBML is embedded, not libSBML itself!)
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif This particular
 * function only does one thing irrespective of user input or 
 * object state, and thus will only return a single value:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * 
   * @see setProgramVersion(string version)
   */ public
";


%csmethodmodifiers SBMLWriter::setProgramVersion(const std::string& version) "
/**
   * Sets the version of this program, i.e., the program that is about to
   * write out the SBMLDocument.
   *
   * If the program version and name are set (see
   * SBMLWriter::setProgramName(@if java String@endif)), the
   * following XML comment, intended for human consumption, will be written
   * at the beginning of the document:
   * @verbatim
 <!-- Created by <program name> version <program version>
 on yyyy-MM-dd HH:mm with libSBML version <libsbml version>. -->
 @endverbatim
   *
   * If the program version and name are not set at some point before
   * calling the writeSBML() methods, no such comment is written out.
   *
   * @param version the version of this program (where \'this program\'
   * refers to program in which libSBML is embedded, not libSBML itself!)
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif This particular
 * function only does one thing irrespective of user input or 
 * object state, and thus will only return a single value:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   *
   * @see setProgramName(string name)
   */ public
";


%csmethodmodifiers SBMLWriter::writeSBML(const SBMLDocument* d, const std::string& filename) "
/**
   * Writes the given SBML document to filename.
   *
   * @htmlinclude assuming-compressed-file.html
   *
   * @param d the SBML document to be written.
   *
   * @param filename the name or full pathname of the file where the SBML
   * is to be written. 
   *
   * @return @c true on success and @c false if the filename could not be
   * opened for writing.
   *
   * @note @htmlinclude note-writing-zipped-files.html
   * 
   * @see setProgramVersion(string version)
   * @see setProgramName(string name)
   */ public
";


%csmethodmodifiers SBMLWriter::writeSBML(const SBMLDocument* d, std::ostream& stream) "
/**
   * Writes the given SBML document to the output stream.
   *
   * @param d the SBML document to be written.
   *
   * @param stream the stream object where the SBML is to be written.
   *
   * @return @c true on success and @c false if one of the underlying
   * parser components fail (rare).
   * 
   * @see setProgramVersion(string version)
   * @see setProgramName(string name)
   */ public
";


%csmethodmodifiers SBMLWriter::writeToString(const SBMLDocument* d) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLWriter::writeSBMLToFile(const SBMLDocument* d, const std::string& filename) "
/**
   * Writes the given SBML document to filename.
   *
   * @htmlinclude assuming-compressed-file.html
   *
   * @param d the SBML document to be written.
   *
   * @param filename the name or full pathname of the file where the SBML
   * is to be written. 
   *
   * @return @c true on success and @c false if the filename could not be
   * opened for writing.
   *
   * @note @htmlinclude note-writing-zipped-files.html
   * 
   * @see setProgramVersion(string version)
   * @see setProgramName(string name)
   */ public
";


%csmethodmodifiers SBMLWriter::writeSBMLToString(const SBMLDocument* d) "
/**
   * Writes the given SBML document to an in-memory string and returns a
   * pointer to it.
   *
   * The string is owned by the caller and should be freed (with @c free())
   * when no longer needed.
   *
   * @param d the SBML document to be written.
   *
   * @return the string on success and @c 0 if one of the underlying parser
   * components fail.
   * 
   * @see setProgramVersion(string version)
   * @see setProgramName(string name)
   */ public
";


%csmethodmodifiers SBMLWriter::hasZlib "
/**
   * Predicate returning @c true if this copy of libSBML has been linked
   * with the <em>zlib</em> library.
   *
   * LibSBML supports reading and writing files compressed with either
   * bzip2 or zip/gzip compression.  The facility depends on libSBML having
   * been compiled with the necessary support libraries.  This method
   * allows a calling program to inquire whether that is the case for the
   * copy of libSBML it is using.
   *
   * @return @c true if libSBML is linked with zlib, @c false otherwise.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   *
   * @see @if clike hasBzip2() @else SBMLWriter::hasBzip2() @endif
   */ public
";


%csmethodmodifiers SBMLWriter::hasBzip2 "
/**
   * Predicate returning @c true if this copy of libSBML has been linked
   * with the <em>bzip2</em> library.
   *
   * LibSBML supports reading and writing files compressed with either
   * bzip2 or zip/gzip compression.  The facility depends on libSBML having
   * been compiled with the necessary support libraries.  This method
   * allows a calling program to inquire whether that is the case for the
   * copy of libSBML it is using.
   *
   * @return @c true if libSBML is linked with bzip2, @c false otherwise.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   *
   * @see @if clike hasZlib() @else SBMLWriter::hasZlib() @endif
   */ public
";


%csmethodmodifiers writeSBML(const SBMLDocument_t *d, const char *filename) "
/**
 * Writes the given SBML document @p d to the file named by @p filename.
 * This convenience function is functionally equivalent to:
 *
 *   SBMLWriter_writeSBML(SBMLWriter_create(), d, filename);
 *
 * @htmlinclude assuming-compressed-file.html
 *
 * @param d the SBMLDocument object to be written out in XML format.
 * 
 * @param filename a string giving the path to a file where the XML
 * content is to be written.
 *
 * @return @c 1 (true) on success and @c 0 (false) if @p filename could not be
 * written.  Some possible reasons for failure include (a) being unable to
 * open the file, and (b) using a filename that indicates a compressed SBML
 * file (i.e., a filename ending in <code>&quot;.zip&quot;</code> or
 * similar) when the compression functionality has not been enabled in
 * the underlying copy of libSBML.
 *
 * @see @if conly SBMLWriter_hasZlib() @else SBMLWriter::hasZlib() @endif
 * @see @if conly SBMLWriter_hasBzip2() @else SBMLWriter::hasBzip2() @endif
 *
 * @if conly
 * @memberof SBMLWriter_t
 * @endif
 */ public
";


%csmethodmodifiers writeSBMLToString(const SBMLDocument_t *d) "
/**
 * Writes the given SBML document @p d to an in-memory string and returns a
 * pointer to it.  The string is owned by the caller and should be freed
 * (with free()) when no longer needed.  This convenience function is
 * functionally equivalent to:
 *
 *   SBMLWriter_writeSBMLToString(SBMLWriter_create(), d);
 *
 * but does not require the caller to create an SBMLWriter object first.
 *
 * @param d an SBMLDocument object to be written out in XML format.
 *
 * @return the string on success and @c null if one of the underlying parser
 * components fail.
 *
 * @if clike @warning Note that the string is owned by the caller and
 * should be freed after it is no longer needed.@endif
 *
 * @if conly
 * @memberof SBMLWriter_t
 * @endif
 */ public
";


%csmethodmodifiers writeSBMLToFile(const SBMLDocument_t *d, const char *filename) "
/**
 * Writes the given SBML document @p d to the file @p filename.
 * This convenience function is functionally equivalent to:
 *
 *   SBMLWriter_writeSBMLToFile(SBMLWriter_create(), d, filename);
 *
 * but that does not require the caller to create an SBMLWriter object first.
 *
 * @htmlinclude assuming-compressed-file.html
 * 
 * @param d an SBMLDocument object to be written out in XML format.
 *
 * @param filename a string giving the path to a file where the XML
 * content is to be written.
 *
 * @return @c 1 (true) on success and @c 0 (false) if @p filename could not be
 * written.  Some possible reasons for failure include (a) being unable to
 * open the file, and (b) using a filename that indicates a compressed SBML
 * file (i.e., a filename ending in <code>&quot;.zip&quot;</code> or
 * similar) when the compression functionality has not been enabled in
 * the underlying copy of libSBML.
 *
 * @if clike @warning Note that the string is owned by the caller and
 * should be freed (with the normal string <code>free()</code> C++
 * function) after it is no longer needed.@endif
 *
 * @see SBMLWriter::hasZlib()
 * @see SBMLWriter::hasBzip2()
 *
 * @if conly
 * @memberof SBMLWriter_t
 * @endif
 */ public
";


%csmethodmodifiers SBMLTypeCode_toString(int tc, const char* pkgName) "
/**
 * This method takes an SBML type code and returns a string representing
 * the code.
 *
 * @if clike LibSBML attaches an identifying code to every kind of SBML
 * object.  These are known as <em>SBML type codes</em>.  The set of
 * possible type codes is defined in the enumeration #SBMLTypeCode_t.
 * The names of the type codes all begin with the characters
 * @c SBML_. @endif@if java LibSBML attaches an identifying code to every
 * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
 * other languages, the set of type codes is stored in an enumeration; in
 * the Java language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class {@link
 * libsbmlConstants}.  The names of the type codes all begin with the
 * characters @c SBML_. @endif@if python LibSBML attaches an identifying
 * code to every kind of SBML object.  These are known as <em>SBML type
 * codes</em>.  In the Python language interface for libSBML, the type
 * codes are defined as static integer constants in the interface class
 * @link libsbml@endlink.  The names of the type codes all begin with the
 * characters @c SBML_. @endif@if csharp LibSBML attaches an identifying
 * code to every kind of SBML object.  These are known as <em>SBML type
 * codes</em>.  In the C# language interface for libSBML, the type codes
 * are defined as static integer constants in the interface class @link
 * libsbml@endlink.  The names of the type codes all begin with
 * the characters @c SBML_. @endif
 *
 * @return a human readable name for the given
 * @if clike #SBMLTypeCode_t value@else SBML type code@endif.
 *
 * @note The caller does not own the returned string and is therefore not
 * allowed to modify it.
 *
 */ public
";


%typemap(csimports) SBase "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html SBML\'s <em>%SBase</em> class, base class of most SBML objects.
 *
 * Most components in SBML are derived from a single abstract base type,
 * SBase.  In addition to serving as the parent class for most other
 * classes of objects in SBML, this base type is designed to allow a
 * modeler or a software package to attach arbitrary information to each
 * major element or list in an SBML model.
 *
 * SBase has an optional subelement called \'notes\'.  It is intended to
 * serve as a place for storing optional information intended to be seen by
 * humans.  An example use of the \'notes\' element would be to contain
 * formatted user comments about the model element in which the \'notes\'
 * element is enclosed.  There are certain conditions on the XHTML content
 * permitted inside the \'notes\' element; please consult the <a
 * target=\'_blank\' href=\'http://sbml.org/Documents/Specifications\'>SBML
 * specification document</a> corresponding to the SBML Level and Version
 * of your model for more information about the requirements for \'notes\'
 * content.
 *
 * SBase has another optional subelement called \'annotation\'.  Whereas the
 * \'notes\' element described above is a container for content to be shown
 * directly to humans, the \'annotation\' element is a container for optional
 * software-generated content @em not meant to be shown to humans.  The
 * element\'s content type is <a target=\'_blank\'
 * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
 * \'any\'</a>, allowing essentially arbitrary data content.  SBML places
 * only a few restrictions on the organization of the content; these are
 * intended to help software tools read and write the data as well as help
 * reduce conflicts between annotations added by different tools.  As is
 * the case with \'notes\', it is important to refer to the <a
 * target=\'_blank\' href=\'http://sbml.org/Documents/Specifications\'>SBML
 * specification document</a> corresponding to the SBML Level and Version
 * of your model for more information about the requirements for
 * \'annotation\' content.
 * 
 * It is worth pointing out that the \'annotation\' element in the definition
 * of SBase exists in order that software developers may attach optional
 * application-specific data to the elements in an SBML model.  However, it
 * is important that this facility not be misused.  In particular, it is
 * <em>critical</em> that data essential to a model definition or that can
 * be encoded in existing SBML elements is <em>not</em> stored in
 * \'annotation\'. Parameter values, functional dependencies between model
 * elements, etc., should not be recorded as annotations.  It is crucial to
 * keep in mind the fact that data placed in annotations can be freely
 * ignored by software applications.  If such data affects the
 * interpretation of a model, then software interoperability is greatly
 * impeded.
 *
 * SBML Level 2 introduced an optional SBase attribute named \'metaid\' for
 * supporting metadata annotations using RDF (<a target=\'_blank\'
 * href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>). The
 * attribute value has the data type <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>, the XML identifier
 * type, which means each \'metaid\' value must be globally unique within an
 * SBML file.  (Importantly, this uniqueness criterion applies across any
 * attribute with type <a href=\'http://www.w3.org/TR/REC-xml/#id\'>XML
 * ID</a>, not just the \'metaid\' attribute used by SBML---something to
 * be aware of if your application-specific XML content inside the
 * \'annotation\' subelement happens to use <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML ID</a>.)  The \'metaid\' value
 * serves to identify a model component for purposes such as referencing
 * that component from metadata placed within \'annotation\' subelements.
 *
 * Beginning with SBML Level 2 Version 2, SBase has an optional
 * attribute named \'sboTerm\' for supporting the use of the Systems Biology
 * Ontology.  In SBML proper, the data type of the attribute is a string of
 * the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a seven digit integer number;
 * libSBML simplifies the representation by only storing the \'NNNNNNN\'
 * integer portion.  Thus, in libSBML, the \'sboTerm\' attribute on SBase has
 * data type @c int, and SBO identifiers are stored simply as integers.
 * (For convenience, SBase offers methods for returning both the integer
 * form and a text-string form of the SBO identifier.)  SBO terms are a
 * type of optional annotation, and each different class of SBML object
 * derived from SBase imposes its own requirements about the values
 * permitted for \'sboTerm\'.  More details can be found in SBML specifications
 * for Level&nbsp;2 Version&nbsp;2 and above.
 *
 * Finally, note that, in the list of methods on SBase, there is no public
 * constructor because SBase is an abstract class.  The constructors reside
 * in the subclasses derived from SBase.
 *
 *
 * @section sbase-miriam Standard format for annotations linking data resources
 *
 * SBML Level 2 Versions 2, 3 and 4, and Level&nbsp;3, define a proposed
 * regular format for encoding two particular categories of annotations:
 * (a) references to controlled vocabulary terms and database identifiers
 * which define and describe biological and biochemical entities in a
 * model; and (b) descriptions of the provenance of a model, including its
 * author(s) and modification history.
 */
"


%csmethodmodifiers SBase::accept(SBMLVisitor& v) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::clone() const "
/**
   * Creates and returns a deep copy of this SBase object.
   *
   * @return the (deep) copy of this SBase object.
   */ public new
";


%csmethodmodifiers SBase::getElementBySId(const std::string& id) "
/**
   * Returns the first child element found that has the given @p id in the
   * model-wide @c SId namespace, or @c null if no such object is found.
   *
   * @param id string representing the \'id\' attribute value of the object
   * to find.
   *
   * @return pointer to the first element found with the given identifier.
   */ public new
";


%csmethodmodifiers SBase::getElementBySId(const std::string& id) const "
/**
   * Returns the first child element found that has the given @p id in the
   * model-wide @c SId namespace, or @c null if no such object is found.
   *
   * @param id string representing the \'id\' attribute value of the object
   * to find.
   *
   * @return pointer to the first element found with the given identifier.
   */ public
";


%csmethodmodifiers SBase::getElementByMetaId(const std::string& metaid) "
/**
   * Returns the first child element it can find with a specific \'metaid\'
   * attribute value, or @c null if no such object is found.
   *
   *
 * 
 * The optional attribute named \'metaid\', present on every major SBML
 * component type, is for supporting metadata annotations using RDF (<a
 * href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>).  The
 * attribute value has the data type <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a>, the XML
 * identifier type, which means each \'metaid\' value must be globally unique
 * within an SBML file.  The latter point is important, because the
 * uniqueness criterion applies across <em>any</em> attribute with type
 * <code>ID</code> anywhere in the file, not just the \'metaid\' attribute used
 * by SBML---something to be aware of if your application-specific XML
 * content inside the \'annotation\' subelement happens to use the XML
 * <code>ID</code> type.  Although SBML itself specifies the use of <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a> only for
 * the \'metaid\' attribute, SBML-compatible applications should be careful if
 * they use XML <code>ID</code>\'s in XML portions of a model that are not
 * defined by SBML, such as in the application-specific content of the
 * \'annotation\' subelement.  Finally, note that LibSBML does not provide an
 * explicit XML <code>ID</code> data type; it uses ordinary character
 * strings, which is easier for applications to support.
 *
 *
   *
   * @param metaid string representing the \'metaid\' attribute value of the
   * object to find.
   *
   * @return pointer to the first element found with the given meta-identifier.
   */ public new
";


%csmethodmodifiers SBase::getElementByMetaId(const std::string& metaid) const "
/**
   * Returns the first child element it can find with a specific \'metaid\'
   * attribute value, or @c null if no such object is found.
   *
   *
 * 
 * The optional attribute named \'metaid\', present on every major SBML
 * component type, is for supporting metadata annotations using RDF (<a
 * href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>).  The
 * attribute value has the data type <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a>, the XML
 * identifier type, which means each \'metaid\' value must be globally unique
 * within an SBML file.  The latter point is important, because the
 * uniqueness criterion applies across <em>any</em> attribute with type
 * <code>ID</code> anywhere in the file, not just the \'metaid\' attribute used
 * by SBML---something to be aware of if your application-specific XML
 * content inside the \'annotation\' subelement happens to use the XML
 * <code>ID</code> type.  Although SBML itself specifies the use of <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a> only for
 * the \'metaid\' attribute, SBML-compatible applications should be careful if
 * they use XML <code>ID</code>\'s in XML portions of a model that are not
 * defined by SBML, such as in the application-specific content of the
 * \'annotation\' subelement.  Finally, note that LibSBML does not provide an
 * explicit XML <code>ID</code> data type; it uses ordinary character
 * strings, which is easier for applications to support.
 *
 *
   *
   * @param metaid string representing the \'metaid\' attribute value of the
   * object to find.
   *
   * @return pointer to the first element found with the given meta-identifier.
   */ public
";


%csmethodmodifiers SBase::getAllElements(ElementFilter* filter=NULL) "
/**
   * Returns a List of all child SBase objects, including those nested to
   * an arbitrary depth.
   *
   * @param filter a pointer to an ElementFilter, which causes the function 
   * to return only elements that match a particular set of constraints.  
   * If null (the default), the function will return all child objects.
   *
   * @return a pointer to a List of pointers to all objects that are children
   * of this object.
   */ public new
";


%csmethodmodifiers SBase::renameSIdRefs(const std::string& oldid, const std::string& newid) "
/**
   *
 * Replaces all uses of a given @c SIdRef type attribute value with another
 * value.
 *
 *
 * 

 * In SBML, object identifiers are of a data type called <code>SId</code>.
 * In SBML Level&nbsp;3, an explicit data type called <code>SIdRef</code> was
 * introduced for attribute values that refer to <code>SId</code> values; in
 * previous Levels of SBML, this data type did not exist and attributes were
 * simply described to as \'referring to an identifier\', but the effective
 * data type was the same as <code>SIdRef</code> in Level&nbsp;3.  These and
 * other methods of libSBML refer to the type <code>SIdRef</code> for all
 * Levels of SBML, even if the corresponding SBML specification did not
 * explicitly name the data type.
 *
 *
 *
 * This method works by looking at all attributes and (if appropriate)
 * mathematical formulas in MathML content, comparing the referenced
 * identifiers to the value of @p oldid.  If any matches are found, the
 * matching values are replaced with @p newid.  The method does @em not
 * descend into child elements.
 *
 * @param oldid the old identifier.
 * @param newid the new identifier.
 *
 *
   */ public new
";


%csmethodmodifiers SBase::renameMetaIdRefs(const std::string& oldid, const std::string& newid) "
/**
   *
 * Replaces all uses of a given meta identifier attribute value with
 * another value.
 *
 *
 * 
 * In SBML, object \'meta\' identifiers are of the XML data type <code>ID</code>;
 * the SBML object attribute itself is typically named <code>metaid</code>.  All
 * attributes that hold values <em>referring</em> to values of type
 * <code>ID</code> are of the XML data type <code>IDREF</code>.  They are also
 * sometimes informally referred to as \'metaid refs\', in analogy to the
 * SBML-defined type <code>SIdRef</code>.
 *
 *
 *
 * This method works by looking at all meta-identifier attribute values,
 * comparing the identifiers to the value of @p oldid.  If any matches are
 * found, the matching identifiers are replaced with @p newid.  The method
 * does @em not descend into child elements.
 *
 * @param oldid the old identifier.
 * @param newid the new identifier.
 *
 *
   */ public new
";


%csmethodmodifiers SBase::renameUnitSIdRefs(const std::string& oldid, const std::string& newid) "
/**
   *
 * Replaces all uses of a given @c UnitSIdRef type attribute value with
 * another value.
 *
 *
 * 
 * In SBML, unit definitions have identifiers of type <code>UnitSId</code>.  In
 * SBML Level&nbsp;3, an explicit data type called <code>UnitSIdRef</code> was
 * introduced for attribute values that refer to <code>UnitSId</code> values; in
 * previous Levels of SBML, this data type did not exist and attributes were
 * simply described to as \'referring to a unit identifier\', but the effective
 * data type was the same as <code>UnitSIdRef</code> in Level&nbsp;3.  These and
 * other methods of libSBML refer to the type <code>UnitSIdRef</code> for all
 * Levels of SBML, even if the corresponding SBML specification did not
 * explicitly name the data type.
 *
 *
 *
 * This method works by looking at all unit identifier attribute values
 * (including, if appropriate, inside mathematical formulas), comparing the
 * referenced unit identifiers to the value of @p oldid.  If any matches
 * are found, the matching values are replaced with @p newid.  The method
 * does @em not descend into child elements.
 *
 * @param oldid the old identifier.
 * @param newid the new identifier.
 *
 *
   */ public new
";


%csmethodmodifiers SBase::replaceSIDWithFunction(const std::string& id, const ASTNode* function) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::divideAssignmentsToSIdByFunction(const std::string& id, const ASTNode* function) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::multiplyAssignmentsToSIdByFunction(const std::string& id, const ASTNode* function) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::getElementFromPluginsBySId(std::string id) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::getElementFromPluginsByMetaId(std::string metaid) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::hasNonstandardIdentifierBeginningWith(const std::string& prefix) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::prependStringToAllIdentifiers(const std::string& prefix) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::transformIdentifiers(IdentifierTransformer* idTransformer) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::getAllElementsFromPlugins(ElementFilter* filter=NULL) "
/**
   * Returns a List of all child SBase objects contained in SBML package
   * plug-ins.
   *
   *
 * 
 * SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
 * via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
 * that it uses one or more SBML packages, and likewise, a software tool may be
 * able to support one or more packages.  LibSBML does not come preconfigured
 * with all possible packages included and enabled, in part because not all
 * package specifications have been finalized.  To support the ability for
 * software systems to enable support for the Level&nbsp;3 packages they choose,
 * libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
 * package is implemented in a separate code plug-in that can be enabled by the
 * application to support working with that SBML package.  A given SBML model
 * may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
 * objects created by libSBML plug-ins supporting additional Level&nbsp;3
 * packages.
 *
 *
   *
   * This method walks down the list of all SBML Level&nbsp;3 packages used
   * by this object and returns all child objects defined by those packages.
   *
   * @param filter a pointer to an ElementFilter, which causes the function 
   * to return only elements that match a particular set of constraints.  
   * If null (the default), the function will return all child objects.
   *
   * @return a pointer to a List of pointers to all children objects from
   * plug-ins.
   *
   * @ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif
   */ public new
";


%csmethodmodifiers SBase::getMetaId() const "
/**
   * Returns the value of the \'metaid\' attribute of this SBML object.
   *
   *
 * 
 * The optional attribute named \'metaid\', present on every major SBML
 * component type, is for supporting metadata annotations using RDF (<a
 * href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>).  The
 * attribute value has the data type <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a>, the XML
 * identifier type, which means each \'metaid\' value must be globally unique
 * within an SBML file.  The latter point is important, because the
 * uniqueness criterion applies across <em>any</em> attribute with type
 * <code>ID</code> anywhere in the file, not just the \'metaid\' attribute used
 * by SBML---something to be aware of if your application-specific XML
 * content inside the \'annotation\' subelement happens to use the XML
 * <code>ID</code> type.  Although SBML itself specifies the use of <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a> only for
 * the \'metaid\' attribute, SBML-compatible applications should be careful if
 * they use XML <code>ID</code>\'s in XML portions of a model that are not
 * defined by SBML, such as in the application-specific content of the
 * \'annotation\' subelement.  Finally, note that LibSBML does not provide an
 * explicit XML <code>ID</code> data type; it uses ordinary character
 * strings, which is easier for applications to support.
 *
 *
   *  
   * @return the meta-identifier of this SBML object.
   *
   * @see isSetMetaId()
   * @see setMetaId(string metaid)
   */ public
";


%csmethodmodifiers SBase::getMetaId "
/**
   * Returns the value of the \'metaid\' attribute of this SBML object.
   *
   *
 * 
 * The optional attribute named \'metaid\', present on every major SBML
 * component type, is for supporting metadata annotations using RDF (<a
 * href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>).  The
 * attribute value has the data type <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a>, the XML
 * identifier type, which means each \'metaid\' value must be globally unique
 * within an SBML file.  The latter point is important, because the
 * uniqueness criterion applies across <em>any</em> attribute with type
 * <code>ID</code> anywhere in the file, not just the \'metaid\' attribute used
 * by SBML---something to be aware of if your application-specific XML
 * content inside the \'annotation\' subelement happens to use the XML
 * <code>ID</code> type.  Although SBML itself specifies the use of <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a> only for
 * the \'metaid\' attribute, SBML-compatible applications should be careful if
 * they use XML <code>ID</code>\'s in XML portions of a model that are not
 * defined by SBML, such as in the application-specific content of the
 * \'annotation\' subelement.  Finally, note that LibSBML does not provide an
 * explicit XML <code>ID</code> data type; it uses ordinary character
 * strings, which is easier for applications to support.
 *
 *
   *  
   * @return the meta-identifier of this SBML object, as a string.
   *
   * @see isSetMetaId()
   * @see setMetaId(string metaid)
   */ public
";


%csmethodmodifiers SBase::getId() const "
/**
   * Returns the value of the \'id\' attribute of this SBML object, if it has one, 
   * or the \'variable\' attribute of a Rule, or the \'symbol\' attribute of
   * an InitialAssignment.
   *
   *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
   *
   * @return the id of this SBML object, or the \'variable\' if the object is a
   * Rule, or the \'symbol\' if the object is an InitialAssignment.
   *
   * @note Because of the inconsistent behavior of this function with respect
   * to assignments and rules, callers should use getIdAttribute() instead.
   *
   * @see getIdAttribute()
   * @see setIdAttribute(string sid)
   * @see isSetIdAttribute()
   * @see unsetIdAttribute()
   */ public new
";


%csmethodmodifiers SBase::getIdAttribute() const "
/**
   * Returns the value of the \'id\' attribute of this SBML object.
   *
   *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
   *
   * @return the id of this SBML object, if set and valid for this
   * level and version of SBML; an empty string otherwise.
   *
   * @note Because of the inconsistent behavior of this function with respect
   * to assignments and rules, callers should use getIdAttribute() instead.
   *
   * @see setIdAttribute(string sid)
   * @see isSetIdAttribute()
   * @see unsetIdAttribute()
   */ public
";


%csmethodmodifiers SBase::getName() const "
/**
   * Returns the value of the \'name\' attribute of this SBML object.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 *
 * @return the name of this SBML object, or the empty string if not set or unsettable.
 *
 * @see getIdAttribute()
 * @see isSetName()
 * @see setName(string sid)
 * @see unsetName()
 * 
 *
   */ public new
";


%csmethodmodifiers SBase::getNotes "
/**
   * Returns the content of the \'notes\' subelement of this object as
   * a tree of XMLNode objects.
   *
   *
 * 
 * The optional SBML element named \'notes\', present on every major SBML
 * component type (and in SBML Level&nbsp;3, the \'message\' subelement of
 * Constraint), is intended as a place for storing optional information
 * intended to be seen by humans.  An example use of the \'notes\' element
 * would be to contain formatted user comments about the model element in
 * which the \'notes\' element is enclosed.  Every object derived directly or
 * indirectly from type SBase can have a separate value for \'notes\', allowing
 * users considerable freedom when adding comments to their models.
 *
 * The format of \'notes\' elements conform to the definition of <a
 * target=\'_blank\' href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.
 * However, the content cannot be @em entirely free-form; it must satisfy
 * certain requirements defined in the <a target=\'_blank\'
 * href=\'http://sbml.org/Documents/Specifications\'>SBML specifications</a>
 * for specific SBML Levels.  To help verify the formatting of \'notes\'
 * content, libSBML provides the static utility method
 * SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif); this
 * method implements a verification process that lets callers check whether
 * the content of a given XMLNode object conforms to the SBML requirements
 * for \'notes\' and \'message\' structure.  Developers are urged to consult the
 * appropriate <a target=\'_blank\'
 * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
 * document</a> for the Level and Version of their model for more in-depth
 * explanations of using \'notes\' in SBML.  The SBML Level&nbsp;2 and&nbsp;3
 * specifications have considerable detail about how \'notes\' element content
 * must be structured.
 *
 * 
   *
   * The \'notes\' element content returned by this method will be in XML
   * form, but libSBML does not provide an object model specifically for
   * the content of notes.  Callers will need to traverse the XML tree
   * structure using the facilities available on XMLNode and related
   * objects.  For an alternative method of accessing the notes, see
   * getNotesString().
   *
   * @return the content of the \'notes\' subelement of this SBML object as a
   * tree structure composed of XMLNode objects.
   *
   * @see getNotesString()
   * @see isSetNotes()
   * @see setNotes(XMLNode notes)
   * @see setNotes(string notes, bool addXHTMLMarkup)
   * @see appendNotes(XMLNode notes)
   * @see appendNotes(string notes)
   * @see unsetNotes()
   * @see SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif)
   */ public
";


%csmethodmodifiers SBase::getNotes() const "
/**
   * Returns the content of the \'notes\' subelement of this object as
   * a tree of XMLNode objects.
   *
   *
 * 
 * The optional SBML element named \'notes\', present on every major SBML
 * component type (and in SBML Level&nbsp;3, the \'message\' subelement of
 * Constraint), is intended as a place for storing optional information
 * intended to be seen by humans.  An example use of the \'notes\' element
 * would be to contain formatted user comments about the model element in
 * which the \'notes\' element is enclosed.  Every object derived directly or
 * indirectly from type SBase can have a separate value for \'notes\', allowing
 * users considerable freedom when adding comments to their models.
 *
 * The format of \'notes\' elements conform to the definition of <a
 * target=\'_blank\' href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.
 * However, the content cannot be @em entirely free-form; it must satisfy
 * certain requirements defined in the <a target=\'_blank\'
 * href=\'http://sbml.org/Documents/Specifications\'>SBML specifications</a>
 * for specific SBML Levels.  To help verify the formatting of \'notes\'
 * content, libSBML provides the static utility method
 * SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif); this
 * method implements a verification process that lets callers check whether
 * the content of a given XMLNode object conforms to the SBML requirements
 * for \'notes\' and \'message\' structure.  Developers are urged to consult the
 * appropriate <a target=\'_blank\'
 * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
 * document</a> for the Level and Version of their model for more in-depth
 * explanations of using \'notes\' in SBML.  The SBML Level&nbsp;2 and&nbsp;3
 * specifications have considerable detail about how \'notes\' element content
 * must be structured.
 *
 * 
   *
   * The \'notes\' element content returned by this method will be in XML
   * form, but libSBML does not provide an object model specifically for
   * the content of notes.  Callers will need to traverse the XML tree
   * structure using the facilities available on XMLNode and related
   * objects.  For an alternative method of accessing the notes, see
   * getNotesString().
   *
   * @return the content of the \'notes\' subelement of this SBML object as a
   * tree structure composed of XMLNode objects.
   *
   * @see getNotesString()
   * @see isSetNotes()
   * @see setNotes(XMLNode notes)
   * @see setNotes(string notes, bool addXHTMLMarkup)
   * @see appendNotes(XMLNode notes)
   * @see appendNotes(string notes)
   * @see unsetNotes()
   * @see SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif)
   */ public
";


%csmethodmodifiers SBase::getNotesString "
/**
   * Returns the content of the \'notes\' subelement of this object as a
   * string.
   *
   *
 * 
 * The optional SBML element named \'notes\', present on every major SBML
 * component type (and in SBML Level&nbsp;3, the \'message\' subelement of
 * Constraint), is intended as a place for storing optional information
 * intended to be seen by humans.  An example use of the \'notes\' element
 * would be to contain formatted user comments about the model element in
 * which the \'notes\' element is enclosed.  Every object derived directly or
 * indirectly from type SBase can have a separate value for \'notes\', allowing
 * users considerable freedom when adding comments to their models.
 *
 * The format of \'notes\' elements conform to the definition of <a
 * target=\'_blank\' href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.
 * However, the content cannot be @em entirely free-form; it must satisfy
 * certain requirements defined in the <a target=\'_blank\'
 * href=\'http://sbml.org/Documents/Specifications\'>SBML specifications</a>
 * for specific SBML Levels.  To help verify the formatting of \'notes\'
 * content, libSBML provides the static utility method
 * SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif); this
 * method implements a verification process that lets callers check whether
 * the content of a given XMLNode object conforms to the SBML requirements
 * for \'notes\' and \'message\' structure.  Developers are urged to consult the
 * appropriate <a target=\'_blank\'
 * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
 * document</a> for the Level and Version of their model for more in-depth
 * explanations of using \'notes\' in SBML.  The SBML Level&nbsp;2 and&nbsp;3
 * specifications have considerable detail about how \'notes\' element content
 * must be structured.
 *
 * 
   *
   * For an alternative method of accessing the notes, see getNotes(),
   * which returns the content as an XMLNode tree structure.  Depending on
   * an application\'s needs, one or the other method may be more
   * convenient.
   *
   * @return the content of the \'notes\' subelement of this SBML object as a
   * string.
   *
   * @see getNotes()
   * @see isSetNotes()
   * @see setNotes(XMLNode notes)
   * @see setNotes(string notes, bool addXHTMLMarkup)
   * @see appendNotes(XMLNode notes)
   * @see appendNotes(string notes)
   * @see unsetNotes()
   * @see SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif)
   */ public
";


%csmethodmodifiers SBase::getNotesString() const "
/**
   * Returns the content of the \'notes\' subelement of this object as a
   * string.
   *
   *
 * 
 * The optional SBML element named \'notes\', present on every major SBML
 * component type (and in SBML Level&nbsp;3, the \'message\' subelement of
 * Constraint), is intended as a place for storing optional information
 * intended to be seen by humans.  An example use of the \'notes\' element
 * would be to contain formatted user comments about the model element in
 * which the \'notes\' element is enclosed.  Every object derived directly or
 * indirectly from type SBase can have a separate value for \'notes\', allowing
 * users considerable freedom when adding comments to their models.
 *
 * The format of \'notes\' elements conform to the definition of <a
 * target=\'_blank\' href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.
 * However, the content cannot be @em entirely free-form; it must satisfy
 * certain requirements defined in the <a target=\'_blank\'
 * href=\'http://sbml.org/Documents/Specifications\'>SBML specifications</a>
 * for specific SBML Levels.  To help verify the formatting of \'notes\'
 * content, libSBML provides the static utility method
 * SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif); this
 * method implements a verification process that lets callers check whether
 * the content of a given XMLNode object conforms to the SBML requirements
 * for \'notes\' and \'message\' structure.  Developers are urged to consult the
 * appropriate <a target=\'_blank\'
 * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
 * document</a> for the Level and Version of their model for more in-depth
 * explanations of using \'notes\' in SBML.  The SBML Level&nbsp;2 and&nbsp;3
 * specifications have considerable detail about how \'notes\' element content
 * must be structured.
 *
 * 
   *
   * For an alternative method of accessing the notes, see getNotes(),
   * which returns the content as an XMLNode tree structure.  Depending on
   * an application\'s needs, one or the other method may be more
   * convenient.
   *
   * @return the content of the \'notes\' subelement of this SBML object as a
   * string.
   *
   * @see getNotes()
   * @see isSetNotes()
   * @see setNotes(XMLNode notes)
   * @see setNotes(string notes, bool addXHTMLMarkup)
   * @see appendNotes(XMLNode notes)
   * @see appendNotes(string notes)
   * @see unsetNotes()
   * @see SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif)
   */ public
";


%csmethodmodifiers SBase::getAnnotation "
/**
   * Returns the content of the \'annotation\' subelement of this object as
   * a tree of XMLNode objects.
   *
   *
 * 
 * Whereas the SBML \'notes\' subelement is a container for content to be
 * shown directly to humans, the \'annotation\' element is a container for
 * optional software-generated content @em not meant to be shown to
 * humans.  Every object derived from SBase can have its own value for
 * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
 * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
 * \'any\'</a>, allowing essentially arbitrary well-formed XML data
 * content.
 *
 * SBML places a few restrictions on the organization of the content of
 * annotations; these are intended to help software tools read and write
 * the data as well as help reduce conflicts between annotations added by
 * different tools.  Please see the SBML specifications for more details.
 *
 * 
   *
   * The annotations returned by this method will be in XML form.  LibSBML
   * provides an object model and related interfaces for certain specific
   * kinds of annotations, namely model history information and RDF
   * content.  See the ModelHistory, CVTerm and RDFAnnotationParser classes
   * for more information about the facilities available.
   *
   * @return the annotation of this SBML object as a tree of XMLNode objects.
   *
   * @see getAnnotationString()
   * @see isSetAnnotation()
   * @see setAnnotation(XMLNode annotation)
   * @see setAnnotation(string annotation)
   * @see appendAnnotation(XMLNode annotation)
   * @see appendAnnotation(string annotation)
   * @see unsetAnnotation()
   */ public
";


%csmethodmodifiers SBase::getAnnotation() const "
/**
   * Returns the content of the \'annotation\' subelement of this object as
   * a tree of XMLNode objects.
   *
   *
 * 
 * Whereas the SBML \'notes\' subelement is a container for content to be
 * shown directly to humans, the \'annotation\' element is a container for
 * optional software-generated content @em not meant to be shown to
 * humans.  Every object derived from SBase can have its own value for
 * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
 * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
 * \'any\'</a>, allowing essentially arbitrary well-formed XML data
 * content.
 *
 * SBML places a few restrictions on the organization of the content of
 * annotations; these are intended to help software tools read and write
 * the data as well as help reduce conflicts between annotations added by
 * different tools.  Please see the SBML specifications for more details.
 *
 * 
   *
   * The annotations returned by this method will be in XML form.  LibSBML
   * provides an object model and related interfaces for certain specific
   * kinds of annotations, namely model history information and RDF
   * content.  See the ModelHistory, CVTerm and RDFAnnotationParser classes
   * for more information about the facilities available.
   *
   * @return the annotation of this SBML object as a tree of XMLNode objects.
   *
   * @see getAnnotationString()
   * @see isSetAnnotation()
   * @see setAnnotation(XMLNode annotation)
   * @see setAnnotation(string annotation)
   * @see appendAnnotation(XMLNode annotation)
   * @see appendAnnotation(string annotation)
   * @see unsetAnnotation()
   */ public
";


%csmethodmodifiers SBase::getAnnotationString "
/**
   * Returns the content of the \'annotation\' subelement of this object as a
   * character string.
   *
   *
 * 
 * Whereas the SBML \'notes\' subelement is a container for content to be
 * shown directly to humans, the \'annotation\' element is a container for
 * optional software-generated content @em not meant to be shown to
 * humans.  Every object derived from SBase can have its own value for
 * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
 * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
 * \'any\'</a>, allowing essentially arbitrary well-formed XML data
 * content.
 *
 * SBML places a few restrictions on the organization of the content of
 * annotations; these are intended to help software tools read and write
 * the data as well as help reduce conflicts between annotations added by
 * different tools.  Please see the SBML specifications for more details.
 *
 * 
   *
   * The annotations returned by this method will be in string form.  See the
   * method getAnnotation() for a version that returns annotations in XML form.
   *
   * @return the annotation of this SBML object as a character string.
   *
   * @see getAnnotation()
   * @see isSetAnnotation()
   * @see setAnnotation(XMLNode annotation)
   * @see setAnnotation(string annotation)
   * @see appendAnnotation(XMLNode annotation)
   * @see appendAnnotation(string annotation)
   * @see unsetAnnotation()
   */ public
";


%csmethodmodifiers SBase::getAnnotationString() const "
/**
   * Returns the content of the \'annotation\' subelement of this object as a
   * character string.
   *
   *
 * 
 * Whereas the SBML \'notes\' subelement is a container for content to be
 * shown directly to humans, the \'annotation\' element is a container for
 * optional software-generated content @em not meant to be shown to
 * humans.  Every object derived from SBase can have its own value for
 * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
 * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
 * \'any\'</a>, allowing essentially arbitrary well-formed XML data
 * content.
 *
 * SBML places a few restrictions on the organization of the content of
 * annotations; these are intended to help software tools read and write
 * the data as well as help reduce conflicts between annotations added by
 * different tools.  Please see the SBML specifications for more details.
 *
 *
   *
   * The annotations returned by this method will be in string form.  See the
   * method getAnnotation() for a version that returns annotations in XML form.
   *
   * @return the annotation of this SBML object as a character string.
   *
   * @see getAnnotation()
   * @see isSetAnnotation()
   * @see setAnnotation(XMLNode annotation)
   * @see setAnnotation(string annotation)
   * @see appendAnnotation(XMLNode annotation)
   * @see appendAnnotation(string annotation)
   * @see unsetAnnotation()
   */ public
";


%csmethodmodifiers SBase::getNamespaces() const "
/**
   * Returns a list of the XML Namespaces declared on the SBML document
   * owning this object.
   *
   * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and (in
   * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   *
   * @return the XML Namespaces associated with this SBML object, or @c null
   * in certain very usual circumstances where a namespace is not set.
   *
   * @see getLevel()
   * @see getVersion()
   */ public new
";


%csmethodmodifiers SBase::getSBMLDocument() const "
/**
   * Returns the SBMLDocument object containing this object instance.
   *
   *
 * 
 * LibSBML uses the class SBMLDocument as a top-level container for
 * storing SBML content and data associated with it (such as warnings and
 * error messages).  An SBML model in libSBML is contained inside an
 * SBMLDocument object.  SBMLDocument corresponds roughly to the class
 * <i>SBML</i> defined in the SBML Level&nbsp;3 and Level&nbsp;2
 * specifications, but it does not have a direct correspondence in SBML
 * Level&nbsp;1.  (But, it is created by libSBML no matter whether the
 * model is Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)
 *
 *
   *
   * This method allows the caller to obtain the SBMLDocument for the
   * current object.
   * 
   * @return the parent SBMLDocument object of this SBML object.
   *
   * @see getParentSBMLObject()
   * @see getModel()
   */ public
";


%csmethodmodifiers SBase::getSBMLDocument "
/**
   * Returns the SBMLDocument object containing this object instance.
   *
   *
 * 
 * LibSBML uses the class SBMLDocument as a top-level container for
 * storing SBML content and data associated with it (such as warnings and
 * error messages).  An SBML model in libSBML is contained inside an
 * SBMLDocument object.  SBMLDocument corresponds roughly to the class
 * <i>SBML</i> defined in the SBML Level&nbsp;3 and Level&nbsp;2
 * specifications, but it does not have a direct correspondence in SBML
 * Level&nbsp;1.  (But, it is created by libSBML no matter whether the
 * model is Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)
 *
 *
   *
   * This method allows the caller to obtain the SBMLDocument for the
   * current object.
   * 
   * @return the parent SBMLDocument object of this SBML object.
   *
   * @see getParentSBMLObject()
   * @see getModel()
   */ public
";


%csmethodmodifiers SBase::getParentSBMLObject "
/**
   * Returns the parent SBML object containing this object.
   *
   * This returns the immediately-containing object.  This method is
   * convenient when holding an object nested inside other objects in an
   * SBML model.  
   * 
   * @return the parent SBML object of this SBML object.
   *
   * @see getSBMLDocument()
   * @see getModel()
   */ public
";


%csmethodmodifiers SBase::getParentSBMLObject() const "
/**
   * Returns the parent SBML object containing this object.
   *
   * This returns the immediately-containing object.  This method is
   * convenient when holding an object nested inside other objects in an
   * SBML model.  
   * 
   * @return the parent SBML object of this SBML object.
   *
   * @see getSBMLDocument()
   * @see getModel()
   */ public
";


%csmethodmodifiers SBase::getAncestorOfType(int type, const std::string pkgName = "core") "
/**
   * Returns the first ancestor object that has the given SBML type code from the given package.
   *
   * @if clike LibSBML attaches an identifying code to every kind of SBML
   * object.  These are known as <em>SBML type codes</em>.  The set of
   * possible type codes is defined in the enumeration #SBMLTypeCode_t.
   * The names of the type codes all begin with the characters
   * @c SBML_. @endif@if java LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters @c SBML_. @endif@if python LibSBML attaches an identifying
   * code to every kind of SBML object.  These are known as <em>SBML type
   * codes</em>.  In the Python language interface for libSBML, the type
   * codes are defined as static integer constants in the interface class
   * @link libsbml@endlink.  The names of the type codes all begin with the
   * characters @c SBML_. @endif@if csharp LibSBML attaches an identifying
   * code to every kind of SBML object.  These are known as <em>SBML type
   * codes</em>.  In the C# language interface for libSBML, the type codes
   * are defined as static integer constants in the interface class @link
   * libsbmlcs.libsbml libsbml@endlink.  The names of the type codes all begin with
   * the characters @c SBML_. @endif
   *
   * This method searches the tree of objects that are parents of this
   * object, and returns the first one that has the given SBML type code from 
   * the given @p pkgName.
   *
   * @param type the SBML type code of the object sought.
   *
   * @param pkgName (optional) the short name of an SBML Level&nbsp;3
   * package to which the sought-after object must belong.
   * 
   * @return the ancestor SBML object of this SBML object that corresponds
   * to the given @if clike #SBMLTypeCode_t value@else SBML object type
   * code@endif, or @c null if no ancestor exists.
   *
   * @warning The optional argument @p pkgName must be used for all type codes
   * from SBML Level&nbsp;3 packages.  Otherwise, the function will search the 
   * \'core\' namespace alone, not find any corresponding elements, and return 
   * null.
   *
   * @ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif
   */ public
";


%csmethodmodifiers SBase::getAncestorOfType(int type, const std::string pkgName = "core") const "
/**
   * Returns the first ancestor object that has the given SBML type code from the given package.
   *
   * @if clike LibSBML attaches an identifying code to every kind of SBML
   * object.  These are known as <em>SBML type codes</em>.  The set of
   * possible type codes is defined in the enumeration #SBMLTypeCode_t.
   * The names of the type codes all begin with the characters
   * @c SBML_. @endif@if java LibSBML attaches an identifying code to every
   * kind of SBML object.  These are known as <em>SBML type codes</em>.  In
   * other languages, the set of type codes is stored in an enumeration; in
   * the Java language interface for libSBML, the type codes are defined as
   * static integer constants in the interface class {@link
   * libsbmlConstants}.  The names of the type codes all begin with the
   * characters @c SBML_. @endif@if python LibSBML attaches an identifying
   * code to every kind of SBML object.  These are known as <em>SBML type
   * codes</em>.  In the Python language interface for libSBML, the type
   * codes are defined as static integer constants in the interface class
   * @link libsbml@endlink.  The names of the type codes all begin with the
   * characters @c SBML_. @endif@if csharp LibSBML attaches an identifying
   * code to every kind of SBML object.  These are known as <em>SBML type
   * codes</em>.  In the C# language interface for libSBML, the type codes
   * are defined as static integer constants in the interface class @link
   * libsbmlcs.libsbml libsbml@endlink.  The names of the type codes all begin with
   * the characters @c SBML_. @endif
   *
   * This method searches the tree of objects that are parents of this
   * object, and returns the first one that has the given SBML type code from 
   * the given @p pkgName.
   *
   * @param type the SBML type code of the object sought.
   *
   * @param pkgName (optional) the short name of an SBML Level&nbsp;3
   * package to which the sought-after object must belong.
   * 
   * @return the ancestor SBML object of this SBML object that corresponds
   * to the given @if clike #SBMLTypeCode_t value@else SBML object type
   * code@endif, or @c null if no ancestor exists.
   *
   * @warning The optional argument @p pkgName must be used for all type codes
   * from SBML Level&nbsp;3 packages.  Otherwise, the function will search the 
   * \'core\' namespace alone, not find any corresponding elements, and return 
   * null.
   *
   * @ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif
   */ public
";


%csmethodmodifiers SBase::getSBOTerm() const "
/**
   * Returns the integer portion of the value of the \'sboTerm\' attribute of
   * this object.
   *
   *
 * 
 * Beginning with SBML Level 2 Version 2, objects derived from SBase have
 * an optional attribute named \'sboTerm\' for supporting the use of the
 * Systems Biology Ontology.  In SBML proper, the data type of the
 * attribute is a string of the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a
 * seven digit integer number; libSBML simplifies the representation by
 * only storing the \'NNNNNNN\' integer portion.  Thus, in libSBML, the
 * \'sboTerm\' attribute on SBase has data type @c int, and SBO identifiers
 * are stored simply as integers.
 *
 *
   *
 * 
 * SBO terms are a type of optional annotation, and each different class
 * of SBML object derived from SBase imposes its own requirements about
 * the values permitted for \'sboTerm\'. More details can be found in SBML
 * specifications for Level&nbsp;2 Version&nbsp;2 and above.
 *
 *
   *
   * @return the value of the \'sboTerm\' attribute as an integer, or @c -1
   * if the value is not set.
   */ public
";


%csmethodmodifiers SBase::getSBOTermID() const "
/**
   * Returns the string representation of the \'sboTerm\' attribute of
   * this object.
   *
   *
 * 
 * Beginning with SBML Level 2 Version 2, objects derived from SBase have
 * an optional attribute named \'sboTerm\' for supporting the use of the
 * Systems Biology Ontology.  In SBML proper, the data type of the
 * attribute is a string of the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a
 * seven digit integer number; libSBML simplifies the representation by
 * only storing the \'NNNNNNN\' integer portion.  Thus, in libSBML, the
 * \'sboTerm\' attribute on SBase has data type @c int, and SBO identifiers
 * are stored simply as integers.
 *
 *
   *
 * 
 * SBO terms are a type of optional annotation, and each different class
 * of SBML object derived from SBase imposes its own requirements about
 * the values permitted for \'sboTerm\'. More details can be found in SBML
 * specifications for Level&nbsp;2 Version&nbsp;2 and above.
 *
 *
   *
   * @return the value of the \'sboTerm\' attribute as a string (its value
   * will be of the form \'SBO:NNNNNNN\'), or an empty string if
   * the value is not set.
   */ public
";


%csmethodmodifiers SBase::getSBOTermAsURL() const "
/**
   * Returns the URL representation of the \'sboTerm\' attribute of this
   * object.
   *
   * This method returns the entire SBO identifier as a text string in the
   * form <code style=\'margin-right:0; padding-right:0\'>http</code><code style=\'margin-left:0; padding-left:0\'>://identifiers.org/biomodels.sbo/SBO:NNNNNNN\'</code>.
   *
   *
 * 
 * SBO terms are a type of optional annotation, and each different class
 * of SBML object derived from SBase imposes its own requirements about
 * the values permitted for \'sboTerm\'. More details can be found in SBML
 * specifications for Level&nbsp;2 Version&nbsp;2 and above.
 *
 *
   *
   * @return the value of the \'sboTerm\' attribute as an identifiers.org URL,
   * or an empty string if the value is not set.
   */ public
";


%csmethodmodifiers SBase::getLine() const "
/**
   * Returns the line number where this object first appears in the XML
   * representation of the SBML document.
   *
   * @return the line number of this SBML object.  If this object was
   * created programmatically and not read from a file, this method will
   * return the value @c 0.
   *
   * @note The line number for each construct in an SBML model is set upon
   * reading the model.  The accuracy of the line number depends on the
   * correctness of the XML representation of the model, and on the
   * particular XML parser library being used.  The former limitation
   * relates to the following problem: if the model is actually invalid
   * XML, then the parser may not be able to interpret the data correctly
   * and consequently may not be able to establish the real line number.
   * The latter limitation is simply that different parsers seem to have
   * their own accuracy limitations, and out of all the parsers supported
   * by libSBML, none have been 100% accurate in all situations. (At this
   * time, libSBML supports the use of <a target=\'_blank\'
   * href=\'http://xmlsoft.org\'>libxml2</a>, <a target=\'_blank\'
   * href=\'http://expat.sourceforge.net/\'>Expat</a> and <a target=\'_blank\'
   * href=\'http://xerces.apache.org/xerces-c/\'>Xerces</a>.)
   *
   * @see getColumn()
   */ public
";


%csmethodmodifiers SBase::getColumn() const "
/**
   * Returns the column number where this object first appears in the XML
   * representation of the SBML document.
   *
   * @return the column number of this SBML object.  If this object was
   * created programmatically and not read from a file, this method will
   * return the value @c 0.
   *
   * @note The column number for each construct in an SBML model is set
   * upon reading the model.  The accuracy of the column number depends on
   * the correctness of the XML representation of the model, and on the
   * particular XML parser library being used.  The former limitation
   * relates to the following problem: if the model is actually invalid
   * XML, then the parser may not be able to interpret the data correctly
   * and consequently may not be able to establish the real column number.
   * The latter limitation is simply that different parsers seem to have
   * their own accuracy limitations, and out of all the parsers supported
   * by libSBML, none have been 100% accurate in all situations. (At this
   * time, libSBML supports the use of <a target=\'_blank\'
   * href=\'http://xmlsoft.org\'>libxml2</a>, <a target=\'_blank\'
   * href=\'http://expat.sourceforge.net/\'>Expat</a> and <a target=\'_blank\'
   * href=\'http://xerces.apache.org/xerces-c/\'>Xerces</a>.)
   * 
   * @see getLine()
   */ public
";


%csmethodmodifiers SBase::getModelHistory() const "
/**
   * Returns the ModelHistory object, if any, attached to this object.
   *
   * @return the ModelHistory object attached to this object, or @c null if
   * none exist.
   * 
   * @note In SBML Level&nbsp;2, model history annotations were only
   * permitted on the Model element.  In SBML Level&nbsp;3, they are
   * permitted on all SBML components derived from SBase.
   */ public
";


%csmethodmodifiers SBase::getModelHistory "
/**
   * Returns the ModelHistory object, if any, attached to this object.
   * 
   * @return the ModelHistory object attached to this object, or @c null if
   * none exist.
   * 
   * @note In SBML Level&nbsp;2, model history annotations were only
   * permitted on the Model element.  In SBML Level&nbsp;3, they are
   * permitted on all SBML components derived from SBase.
   */ public
";


%csmethodmodifiers SBase::isSetMetaId() const "
/**
   * Predicate returning @c true if this object\'s \'metaid\' attribute is set.
   *
   *
 * 
 * The optional attribute named \'metaid\', present on every major SBML
 * component type, is for supporting metadata annotations using RDF (<a
 * href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>).  The
 * attribute value has the data type <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a>, the XML
 * identifier type, which means each \'metaid\' value must be globally unique
 * within an SBML file.  The latter point is important, because the
 * uniqueness criterion applies across <em>any</em> attribute with type
 * <code>ID</code> anywhere in the file, not just the \'metaid\' attribute used
 * by SBML---something to be aware of if your application-specific XML
 * content inside the \'annotation\' subelement happens to use the XML
 * <code>ID</code> type.  Although SBML itself specifies the use of <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a> only for
 * the \'metaid\' attribute, SBML-compatible applications should be careful if
 * they use XML <code>ID</code>\'s in XML portions of a model that are not
 * defined by SBML, such as in the application-specific content of the
 * \'annotation\' subelement.  Finally, note that LibSBML does not provide an
 * explicit XML <code>ID</code> data type; it uses ordinary character
 * strings, which is easier for applications to support.
 *
 * 
   *
   * @return @c true if the \'metaid\' attribute of this SBML object is
   * set, @c false otherwise.
   *
   * @see getMetaId()
   * @see setMetaId(string metaid)
   */ public
";


%csmethodmodifiers SBase::isSetId() const "
/**
   * Predicate returning @c true if a call to getId() returns a
   * non-empty string.
   *
   * For most objects, this function will return @c true if its \'id\'
   * attribute is set, and @c false if it is not, or if the object has no
   * \'id\' attribute at all.  However, for an EventAssignment or a Rule,
   * isSetId() checks whether the \'variable\' attribute is set, and for an
   * InitialAssignment, it checks whether the \'symbol\' attribute is set.
   * Because those elements will also have an \'id\' attribute in SBML
   * Level&nbsp;3 Version&nbsp;2 which isSetId() will not check, the function
   * itself is deprecated, and it is recommended to use isSetIdAttribute() in
   * all cases where one needs to know whether the \'id\' attribute is set, and
   * to use EventAssignment::isSetVariable(), Rule::isSetVariable() and
   * InitialAssignment::isSetSymbol() when the status of the \'variable\' or
   * \'symbol\' attributes need to be checked.
   *
   *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 *
 * @return @c true if the \'id\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @note Because of the inconsistent behavior of this function with
 * respect to assignments and rules, it is recommended that callers
 * use isSetIdAttribute() instead.
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see unsetIdAttribute()
 * @see isSetIdAttribute()
 *
 *
   */ public new
";


%csmethodmodifiers SBase::isSetIdAttribute() const "
/**
   * Predicate returning @c true if this object\'s \'id\' attribute is set.
   *
   *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
   * 
   * @return @c true if the \'id\' attribute of this SBML object is
   * set, @c false otherwise.
   *
   * @see getIdAttribute()
   * @see setIdAttribute(string sid)
   * @see unsetIdAttribute()
   */ public
";


%csmethodmodifiers SBase::isSetName() const "
/**
   * Predicate returning @c true if this
   * object\'s \'name\' attribute is set.
   * 
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 * @return @c true if the \'name\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @see getName()
 * @see setName(string sid)
 * @see unsetName()
 *
 *
   */ public new
";


%csmethodmodifiers SBase::isSetNotes() const "
/**
   * Predicate returning @c true if this
   * object\'s \'notes\' subelement exists and has content.
   *
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type SBase can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   *
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode xhtml@endif); however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and&nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   *
   * @return @c true if a \'notes\' subelement exists, @c false otherwise.
   * 
   * @see getNotes()
   * @see getNotesString()
   * @see setNotes(XMLNode notes)
   * @see setNotes(string notes, bool addXHTMLMarkup)
   * @see appendNotes(XMLNode notes)
   * @see appendNotes(string notes)
   * @see unsetNotes()
   * @see SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif)
   */ public
";


%csmethodmodifiers SBase::isSetAnnotation() const "
/**
   * Predicate returning @c true if this
   * object\'s \'annotation\' subelement exists and has content.
   *
   * Whereas the SBase \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content @em not meant to be shown to
   * humans.  Every object derived from SBase can have its own value for
   * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
   * \'any\'</a>, allowing essentially arbitrary well-formed XML data
   * content.
   *
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   *
   * @return @c true if a \'annotation\' subelement exists, @c false
   * otherwise.
   * 
   * @see getAnnotation()
   * @see getAnnotationString()
   * @see setAnnotation(XMLNode annotation)
   * @see setAnnotation(string annotation)
   * @see appendAnnotation(XMLNode annotation)
   * @see appendAnnotation(string annotation)
   * @see unsetAnnotation()
   */ public
";


%csmethodmodifiers SBase::isSetSBOTerm() const "
/**
   * Predicate returning @c true if this
   * object\'s \'sboTerm\' attribute is set.
   *
   * @return @c true if the \'sboTerm\' attribute of this SBML object is
   * set, @c false otherwise.
   */ public
";


%csmethodmodifiers SBase::setMetaId(const std::string& metaid) "
/**
   * Sets the value of the meta-identifier attribute of this SBML object.
   *
   *
 * 
 * The optional attribute named \'metaid\', present on every major SBML
 * component type, is for supporting metadata annotations using RDF (<a
 * href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>).  The
 * attribute value has the data type <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a>, the XML
 * identifier type, which means each \'metaid\' value must be globally unique
 * within an SBML file.  The latter point is important, because the
 * uniqueness criterion applies across <em>any</em> attribute with type
 * <code>ID</code> anywhere in the file, not just the \'metaid\' attribute used
 * by SBML---something to be aware of if your application-specific XML
 * content inside the \'annotation\' subelement happens to use the XML
 * <code>ID</code> type.  Although SBML itself specifies the use of <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a> only for
 * the \'metaid\' attribute, SBML-compatible applications should be careful if
 * they use XML <code>ID</code>\'s in XML portions of a model that are not
 * defined by SBML, such as in the application-specific content of the
 * \'annotation\' subelement.  Finally, note that LibSBML does not provide an
 * explicit XML <code>ID</code> data type; it uses ordinary character
 * strings, which is easier for applications to support.
 *
 * 
   *
   * The string @p metaid is copied.  
   *
   * @param metaid the identifier string to use as the value of the
   * \'metaid\' attribute.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * 
   * @see getMetaId()
   * @see isSetMetaId()
   */ public
";


%csmethodmodifiers SBase::isSetModelHistory "
/**
   * Predicate returning @c true if this
   * object has a ModelHistory object attached to it.
   *
   * @return @c true if the ModelHistory of this object is set,
   * @c false otherwise.
   * 
   * @note In SBML Level&nbsp;2, model history annotations were only
   * permitted on the Model element.  In SBML Level&nbsp;3, they are
   * permitted on all SBML components derived from SBase.
   */ public
";


%csmethodmodifiers SBase::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this SBML object.
   *
   *
 * 
 * The string @p sid is copied.
 *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 * 
 * @param sid the string to use as the identifier of this object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see isSetIdAttribute()
 * @see unsetIdAttribute()
 * 
 *
   */ public new
";


%csmethodmodifiers SBase::setIdAttribute(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this SBML object.
   *
   *
 * 
 * The string @p sid is copied.
 *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 * 
 * @param sid the string to use as the identifier of this object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see isSetIdAttribute()
 * @see unsetIdAttribute()
 * 
 *
   */ public new
";


%csmethodmodifiers SBase::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this SBML object.
   *
   *
 * 
 *
 * The string in @p name is copied.
 *
 * @param name the new name for the SBML object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 *
 *
   */ public new
";


%csmethodmodifiers SBase::setAnnotation(const XMLNode* annotation) "
/**
   * Sets the value of the \'annotation\' subelement of this SBML object.
   *
   * The content of @p annotation is copied, and any previous content of
   * this object\'s \'annotation\' subelement is deleted.
   * 
   * Whereas the SBase \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content @em not meant to be shown to
   * humans.  Every object derived from SBase can have its own value for
   * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
   * \'any\'</a>, allowing essentially arbitrary well-formed XML data
   * content.
   *
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   *
   * Call this method will result in any existing content of the
   * \'annotation\' subelement to be discarded.  Unless you have taken steps
   * to first copy and reconstitute any existing annotations into the @p
   * annotation that is about to be assigned, it is likely that performing
   * such wholesale replacement is unfriendly towards other software
   * applications whose annotations are discarded.  An alternative may be
   * to use SBase::appendAnnotation(XMLNode annotation) or
   * SBase::appendAnnotation(string annotation).
   *
   * @param annotation an XML structure that is to be used as the new content
   * of the \'annotation\' subelement of this object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif This particular
 * function only does one thing irrespective of user input or 
 * object state, and thus will only return a single value:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   *
   * @see getAnnotationString()
   * @see isSetAnnotation()
   * @see setAnnotation(string annotation)
   * @see appendAnnotation(XMLNode annotation)
   * @see appendAnnotation(string annotation)
   * @see unsetAnnotation()
   */ public new
";


%csmethodmodifiers SBase::setAnnotation(const std::string& annotation) "
/**
   * Sets the value of the \'annotation\' subelement of this SBML object.
   *
   * The content of @p annotation is copied, and any previous content of
   * this object\'s \'annotation\' subelement is deleted.
   * 
   * Whereas the SBase \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content @em not meant to be shown to
   * humans.  Every object derived from SBase can have its own value for
   * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
   * \'any\'</a>, allowing essentially arbitrary well-formed XML data
   * content.
   *
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   *
   * Call this method will result in any existing content of the
   * \'annotation\' subelement to be discarded.  Unless you have taken steps
   * to first copy and reconstitute any existing annotations into the @p
   * annotation that is about to be assigned, it is likely that performing
   * such wholesale replacement is unfriendly towards other software
   * applications whose annotations are discarded.  An alternative may be
   * to use SBase::appendAnnotation(XMLNode annotation) or
   * SBase::appendAnnotation(string annotation).
   *
   * @param annotation an XML string that is to be used as the content
   * of the \'annotation\' subelement of this object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @see getAnnotationString()
   * @see isSetAnnotation()
   * @see setAnnotation(XMLNode annotation)
   * @see appendAnnotation(XMLNode annotation)
   * @see appendAnnotation(string annotation)
   * @see unsetAnnotation()
   */ public new
";


%csmethodmodifiers SBase::appendAnnotation(const XMLNode* annotation) "
/**
   * Appends the given @p annotation to the \'annotation\' subelement of this
   * object.
   * 
   * Whereas the SBase \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content @em not meant to be shown to
   * humans.  Every object derived from SBase can have its own value for
   * \'annotation\'.  The element\'s content type is <a
   * target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type \'any\'</a>,
   * allowing essentially arbitrary well-formed XML data content.
   *
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   *
   * Unlike SBase::setAnnotation(XMLNode annotation) or
   * SBase::setAnnotation(string annotation), this method
   * allows other annotations to be preserved when an application adds its
   * own data.
   *
   * @param annotation an XML structure that is to be copied and appended
   * to the content of the \'annotation\' subelement of this object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * (if the parent element does not have the \'metaid\' attribute set)
   * @li @link libsbml#LIBSBML_DUPLICATE_ANNOTATION_NS LIBSBML_DUPLICATE_ANNOTATION_NS@endlink
   * (if the parent was already annotated with the annotation in question)
   *
   * @see getAnnotationString()
   * @see isSetAnnotation()
   * @see setAnnotation(XMLNode annotation)
   * @see setAnnotation(string annotation)
   * @see appendAnnotation(string annotation)
   * @see unsetAnnotation()
   */ public new
";


%csmethodmodifiers SBase::appendAnnotation(const std::string& annotation) "
/**
   * Appends the given @p annotation to the \'annotation\' subelement of this
   * object.
   *
   * Whereas the SBase \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content @em not meant to be shown to
   * humans.  Every object derived from SBase can have its own value for
   * \'annotation\'.  The element\'s content type is <a
   * target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type \'any\'</a>,
   * allowing essentially arbitrary well-formed XML data content.
   *
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   *
   * Unlike SBase::setAnnotation(XMLNode annotation) or
   * SBase::setAnnotation(string annotation), this method
   * allows other annotations to be preserved when an application adds its
   * own data.
   *
   * @param annotation an XML string that is to be copied and appended
   * to the content of the \'annotation\' subelement of this object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * (if the parent element does not have the \'metaid\' attribute set)
   * @li @link libsbml#LIBSBML_DUPLICATE_ANNOTATION_NS LIBSBML_DUPLICATE_ANNOTATION_NS@endlink
   * (if the parent was already annotated with the annotation in question)
   *
   * @see getAnnotationString()
   * @see isSetAnnotation()
   * @see setAnnotation(XMLNode annotation)
   * @see setAnnotation(string annotation)
   * @see appendAnnotation(XMLNode annotation)
   * @see unsetAnnotation()
   */ public new
";


%csmethodmodifiers SBase::removeTopLevelAnnotationElement(const std::string elementName, const std::string elementURI = "", bool removeEmpty = true) "
/**
   * Removes the top-level element within the \'annotation\' subelement of this
   * SBML object with the given name and optional URI.
   *
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   *
   * Calling this method allows a particular annotation element to be removed
   * whilst the remaining annotations remain intact.
   *
   * @param elementName a string representing the name of the top level
   * annotation element that is to be removed.
   * @param elementURI an optional string that is used to check both the name
   * and URI of the top level element to be removed.
   * @param removeEmpty if after removing of the element, the annotation is 
   * empty, and the removeEmpty argument is true, the annotation node will be 
   * deleted (default). 
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   * @li @link libsbml#LIBSBML_ANNOTATION_NAME_NOT_FOUND LIBSBML_ANNOTATION_NAME_NOT_FOUND@endlink
   * @li @link libsbml#LIBSBML_ANNOTATION_NS_NOT_FOUND LIBSBML_ANNOTATION_NS_NOT_FOUND@endlink
   *
   * @see replaceTopLevelAnnotationElement(XMLNode )
   * @see replaceTopLevelAnnotationElement(string)
   */ public
";


%csmethodmodifiers SBase::replaceTopLevelAnnotationElement(const XMLNode* annotation) "
/**
   * Replaces the given top-level element within the \'annotation\' 
   * subelement of this SBML object and with the annotation element supplied.
   *
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   *
   * This method determines the name of the element to be replaced from the
   * annotation argument. Functionally it is equivalent to calling <code>
   * removeTopLevelAnnotationElement(name)</code> followed by calling
   * <code>appendAnnotation(annotation_with_name)</code>, with the exception
   * that the placement of the annotation element remains the same.
   *
   * @param annotation XMLNode representing the replacement top level annotation.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   *
   * @see removeTopLevelAnnotationElement(string elementName, string elementURI, bool removeEmpty)
   * @see replaceTopLevelAnnotationElement(string)
   */ public
";


%csmethodmodifiers SBase::replaceTopLevelAnnotationElement(const std::string& annotation) "
/**
   * Replaces the given top-level element within the \'annotation\' 
   * subelement of this SBML object and with the annotation element supplied.
   *
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   *
   * This method determines the name of the element to be replaced from the
   * annotation argument. Functionally it is equivalent to calling <code>
   * removeTopLevelAnnotationElement(name)</code> followed by calling
   * <code>appendAnnotation(annotation_with_name)</code>, with the exception
   * that the placement of the annotation element remains the same.
   *
   * @param annotation string representing the replacement top level annotation.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   *
   * @see removeTopLevelAnnotationElement(string elementName, string elementURI)
   * @see replaceTopLevelAnnotationElement(XMLNode)
   */ public
";


%csmethodmodifiers SBase::setNotes(const XMLNode* notes) "
/**
   * Sets the value of the \'notes\' subelement of this SBML object.
   *
   * The content of @p notes is copied, and any existing content of this
   * object\'s \'notes\' subelement is deleted.
   *
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type SBase can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   *
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode xhtml@endif); however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and&nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   *
   * @param notes an XML structure that is to be used as the content of the
   * \'notes\' subelement of this object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   *
   * @see getNotesString()
   * @see isSetNotes()
   * @see setNotes(string notes, bool addXHTMLMarkup)
   * @see appendNotes(XMLNode notes)
   * @see appendNotes(string notes)
   * @see unsetNotes()
   * @see SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif)
   */ public
";


%csmethodmodifiers SBase::setNotes(const std::string& notes, bool addXHTMLMarkup = false) "
/**
   * Sets the value of the \'notes\' subelement of this SBML object to a copy
   * of the string @p notes.
   *
   * The content of @p notes is copied, and any existing content of this
   * object\'s \'notes\' subelement is deleted.
   *
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type SBase can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   *
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif); however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and&nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   *
   * The following code illustrates a very simple way of setting the notes
   * using this method.  Here, the object being annotated is the whole SBML
   * document, but that is for illustration purposes only; you could of
   * course use this same approach to annotate any other SBML component.
   * @if cpp
@code{.cpp}
SBMLDocument s = new SBMLDocument(3, 1);
s->setNotes(\'<body xmlns=\'http://www.w3.org/1999/xhtml\'><p>here is my note</p></body>\');
@endcode
@endif
@if java
@code{.java}
SBMLDocument s = new SBMLDocument(3, 1);
s.setNotes(\'<body xmlns=\'http://www.w3.org/1999/xhtml\'><p>here is my note</p></body>\');
@endcode
@endif
@if python
@code{.py}
try:
  sbmlDoc = SBMLDocument(3, 1)
except ValueError:
  print(\'Could not create SBMLDocument object\')
  sys.exit(1)

note = \'<body xmlns=\'http://www.w3.org/1999/xhtml\'><p>here is my note</p></body>\'

status = sbmlDoc.setNotes(note)
if status != LIBSBML_OPERATION_SUCCESS:
  # Do something to handle the error here.
  print(\'Unable to set notes on the SBML document object\')
  sys.exit(1)
@endcode
@endif
@if csharp
@code
SBMLDocument s = new SBMLDocument(3, 1);
s.setNotes(\'<body xmlns=\'http://www.w3.org/1999/xhtml\'><p>here is my note</p></body>\');
@endcode
@endif
   *
   * @param notes an XML string that is to be used as the content of the
   * \'notes\' subelement of this object.
   *
   * @param addXHTMLMarkup a boolean indicating whether to wrap the contents
   * of the @p notes argument with XHTML paragraph (<code>&lt;p&gt;</code>)
   * tags.  This is appropriate when the string in @p notes does not already
   * containg the appropriate XHTML markup.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @see getNotesString()
   * @see isSetNotes()
   * @see setNotes(XMLNode notes)
   * @see appendNotes(XMLNode notes)
   * @see appendNotes(string notes)
   * @see unsetNotes()
   * @see SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif)
   */ public
";


%csmethodmodifiers SBase::appendNotes(const XMLNode* notes) "
/**
   * Appends the given @p notes to the \'notes\' subelement of this object.
   *
   * The content of @p notes is copied.
   *
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type SBase can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   *
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif); however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and&nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   * 
   * @param notes an XML node structure that is to appended to the content
   * of the \'notes\' subelement of this object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @see getNotesString()
   * @see isSetNotes()
   * @see setNotes(XMLNode notes)
   * @see setNotes(string notes, bool addXHTMLMarkup)
   * @see appendNotes(string notes)
   * @see unsetNotes()
   * @see SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif)
   */ public
";


%csmethodmodifiers SBase::appendNotes(const std::string& notes) "
/**
   * Appends the given @p notes to the \'notes\' subelement of this object.
   *
   * The content of the parameter @p notes is copied.
   *
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type SBase can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   *
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif); however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and&nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   *
   * @param notes an XML string that is to appended to the content of
   * the \'notes\' subelement of this object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @see getNotesString()
   * @see isSetNotes()
   * @see setNotes(XMLNode notes)
   * @see setNotes(string notes, bool addXHTMLMarkup)
   * @see appendNotes(XMLNode notes)
   * @see unsetNotes()
   * @see SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif)
   */ public
";


%csmethodmodifiers SBase::setModelHistory(ModelHistory * history) "
/**
   * Sets the ModelHistory of this object.
   *
   * The content of @p history is copied, and this object\'s existing model
   * history content is deleted.
   *
   * @param history ModelHistory of this object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * 
   * @note In SBML Level&nbsp;2, model history annotations were only
   * permitted on the Model element.  In SBML Level&nbsp;3, they are
   * permitted on all SBML components derived from SBase.
   */ public
";


%csmethodmodifiers SBase::setSBMLDocument(SBMLDocument* d) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::connectToParent(SBase* parent) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::setSBOTerm(int value) "
/**
   * Sets the value of the \'sboTerm\' attribute.
   *
   *
 * 
 * Beginning with SBML Level 2 Version 2, objects derived from SBase have
 * an optional attribute named \'sboTerm\' for supporting the use of the
 * Systems Biology Ontology.  In SBML proper, the data type of the
 * attribute is a string of the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a
 * seven digit integer number; libSBML simplifies the representation by
 * only storing the \'NNNNNNN\' integer portion.  Thus, in libSBML, the
 * \'sboTerm\' attribute on SBase has data type @c int, and SBO identifiers
 * are stored simply as integers.
 *
 *
   *
 * 
 * SBO terms are a type of optional annotation, and each different class
 * of SBML object derived from SBase imposes its own requirements about
 * the values permitted for \'sboTerm\'. More details can be found in SBML
 * specifications for Level&nbsp;2 Version&nbsp;2 and above.
 *
 *
   *
   * @param value the NNNNNNN integer portion of the SBO identifier.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   * @see setSBOTerm(string &sboid)
   */ public new
";


%csmethodmodifiers SBase::setSBOTerm(const std::string &sboid) "
/**
   * Sets the value of the \'sboTerm\' attribute by string.
   *
   *
 * 
 * Beginning with SBML Level 2 Version 2, objects derived from SBase have
 * an optional attribute named \'sboTerm\' for supporting the use of the
 * Systems Biology Ontology.  In SBML proper, the data type of the
 * attribute is a string of the form \'SBO:NNNNNNN\', where \'NNNNNNN\' is a
 * seven digit integer number; libSBML simplifies the representation by
 * only storing the \'NNNNNNN\' integer portion.  Thus, in libSBML, the
 * \'sboTerm\' attribute on SBase has data type @c int, and SBO identifiers
 * are stored simply as integers.
 *
 *
   *
 * 
 * SBO terms are a type of optional annotation, and each different class
 * of SBML object derived from SBase imposes its own requirements about
 * the values permitted for \'sboTerm\'. More details can be found in SBML
 * specifications for Level&nbsp;2 Version&nbsp;2 and above.
 *
 *
   *
   * @param sboid the SBO identifier string of the form \'SBO:NNNNNNN\'.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   * @see setSBOTerm(int value)
   */ public new
";


%csmethodmodifiers SBase::setNamespaces(XMLNamespaces* xmlns) "
/**
   * Sets the namespaces relevant of this SBML object.
   *
   * The content of @p xmlns is copied, and this object\'s existing
   * namespace content is deleted.
   *
   * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
   * information.  It is used to communicate the SBML Level, Version, and
   * (in Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.
   * 
   * @param xmlns the namespaces to set.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   */ public
";


%csmethodmodifiers SBase::unsetMetaId "
/**
   * Unsets the value of the \'metaid\' attribute of this SBML object.
   *
   *
 * 
 * The optional attribute named \'metaid\', present on every major SBML
 * component type, is for supporting metadata annotations using RDF (<a
 * href=\'http://www.w3.org/RDF/\'>Resource Description Format</a>).  The
 * attribute value has the data type <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a>, the XML
 * identifier type, which means each \'metaid\' value must be globally unique
 * within an SBML file.  The latter point is important, because the
 * uniqueness criterion applies across <em>any</em> attribute with type
 * <code>ID</code> anywhere in the file, not just the \'metaid\' attribute used
 * by SBML---something to be aware of if your application-specific XML
 * content inside the \'annotation\' subelement happens to use the XML
 * <code>ID</code> type.  Although SBML itself specifies the use of <a
 * href=\'http://www.w3.org/TR/REC-xml/#id\'>XML <code>ID</code></a> only for
 * the \'metaid\' attribute, SBML-compatible applications should be careful if
 * they use XML <code>ID</code>\'s in XML portions of a model that are not
 * defined by SBML, such as in the application-specific content of the
 * \'annotation\' subelement.  Finally, note that LibSBML does not provide an
 * explicit XML <code>ID</code> data type; it uses ordinary character
 * strings, which is easier for applications to support.
 *
 * 
   *  
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   */ public
";


%csmethodmodifiers SBase::unsetId "
/**
   * Unsets the value of the \'id\' attribute of this SBML object.
   *
   *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 * 
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see isSetIdAttribute()
 * @see unsetIdAttribute()
 *
 *
   */ public new
";


%csmethodmodifiers SBase::unsetIdAttribute "
/**
   * Unsets the value of the \'id\' attribute of this SBML object.
   *
   * Most (but not all) objects in SBML include two common attributes: \'id\'
   * and \'name\'.  The identifier given by an object\'s \'id\' attribute value
   * is used to identify the object within the SBML model definition.
   * Other objects can refer to the component using this identifier.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @see getIdAttribute()
   * @see setIdAttribute(string sid)
   * @see isSetIdAttribute()
   */ public
";


%csmethodmodifiers SBase::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this SBML object.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
 *
 * @see getName()
 * @see setName(string sid)
 * @see isSetName()
 *
 *
   */ public new
";


%csmethodmodifiers SBase::unsetNotes "
/**
   * Unsets the value of the \'notes\' subelement of this SBML object.
   *
   * The optional SBML element named \'notes\', present on every major SBML
   * component type, is intended as a place for storing optional
   * information intended to be seen by humans.  An example use of the
   * \'notes\' element would be to contain formatted user comments about the
   * model element in which the \'notes\' element is enclosed.  Every object
   * derived directly or indirectly from type SBase can have a separate
   * value for \'notes\', allowing users considerable freedom when adding
   * comments to their models.
   *
   * The format of \'notes\' elements must be <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/xhtml1/\'>XHTML&nbsp;1.0</a>.  To help
   * verify the formatting of \'notes\' content, libSBML provides the static
   * utility method SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif); however,
   * readers are urged to consult the appropriate <a target=\'_blank\'
   * href=\'http://sbml.org/Documents/Specifications\'>SBML specification
   * document</a> for the Level and Version of their model for more
   * in-depth explanations.  The SBML Level&nbsp;2 and&nbsp;3
   * specifications have considerable detail about how \'notes\' element
   * content must be structured.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif This particular
 * function only does one thing irrespective of user input or 
 * object state, and thus will only return a single value:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   *
   * @see getNotesString()
   * @see isSetNotes()
   * @see setNotes(XMLNode notes)
   * @see setNotes(string notes, bool addXHTMLMarkup)
   * @see appendNotes(XMLNode notes)
   * @see appendNotes(string notes)
   * @see SyntaxChecker::hasExpectedXHTMLSyntax(@if java XMLNode@endif)
   */ public
";


%csmethodmodifiers SBase::unsetAnnotation "
/**
   * Unsets the value of the \'annotation\' subelement of this SBML object.
   *
   * Whereas the SBase \'notes\' subelement is a container for content to be
   * shown directly to humans, the \'annotation\' element is a container for
   * optional software-generated content @em not meant to be shown to
   * humans.  Every object derived from SBase can have its own value for
   * \'annotation\'.  The element\'s content type is <a target=\'_blank\'
   * href=\'http://www.w3.org/TR/2004/REC-xml-20040204/#elemdecls\'>XML type
   * \'any\'</a>, allowing essentially arbitrary well-formed XML data
   * content.
   *
   * SBML places a few restrictions on the organization of the content of
   * annotations; these are intended to help software tools read and write
   * the data as well as help reduce conflicts between annotations added by
   * different tools.  Please see the SBML specifications for more details.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif This particular
 * function only does one thing irrespective of user input or 
 * object state, and thus will only return a single value:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   *
   * @see getAnnotation()
   * @see getAnnotationString()
   * @see isSetAnnotation()
   * @see setAnnotation(XMLNode annotation)
   * @see setAnnotation(string annotation)
   * @see appendAnnotation(XMLNode annotation)
   * @see appendAnnotation(string annotation)
   */ public
";


%csmethodmodifiers SBase::unsetSBOTerm "
/**
   * Unsets the value of the \'sboTerm\' attribute of this SBML object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   */ public
";


%csmethodmodifiers SBase::addCVTerm(CVTerm * term, bool newBag = false) "
/**
   * Adds a copy of the given CVTerm object to this SBML object.
   *
   * @param term the CVTerm to assign.
   *
   * @param newBag if @c true, creates a new RDF bag with the same identifier
   * as a previous bag, and if @c false, adds the term to an existing
   * RDF bag with the same type of qualifier as the term being added.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink, if
   * this object lacks a \'metaid\' attribute
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   *
   * @note Since the CV Term uses the \'metaid\' attribute of the object as a
   * reference, if the object has no \'metaid\' attribute value set, then the
   * CVTerm will not be added.
   *
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 *
   * 
   * @ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif
   */ public
";


%csmethodmodifiers SBase::getCVTerms "
/**
   * Returns a list of CVTerm objects in the annotations of this SBML
   * object.
   * 
   * @return the list of CVTerms for this SBML object.
   */ public
";


%csmethodmodifiers SBase::getCVTerms() const "
/**
   * Returns a list of CVTerm objects in the annotations of this SBML
   * object.
   * 
   * @return the list of CVTerms for this SBML object.
   */ public
";


%csmethodmodifiers SBase::getNumCVTerms "
/**
   * Returns the number of CVTerm objects in the annotations of this SBML
   * object.
   * 
   * @return the number of CVTerms for this SBML object.
   */ public
";


%csmethodmodifiers SBase::getCVTerm(unsigned int n) "
/**
   * Returns the nth CVTerm in the list of CVTerms of this SBML
   * object.
   * 
   * @param n unsigned int the index of the CVTerm to retrieve.
   *
   * @return the nth CVTerm in the list of CVTerms for this SBML object.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers SBase::unsetCVTerms "
/**
   * Clears the list of CVTerm objects attached to this SBML object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   */ public
";


%csmethodmodifiers SBase::unsetModelHistory "
/**
   * Unsets the ModelHistory object attached to this object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   * 
   * @note In SBML Level&nbsp;2, model history annotations were only
   * permitted on the Model element.  In SBML Level&nbsp;3, they are
   * permitted on all SBML components derived from SBase.
   */ public
";


%csmethodmodifiers SBase::getResourceBiologicalQualifier(std::string resource) "
/**
   * Returns the MIRIAM <em>biological qualifier</em> associated with the
   * given resource.
   *
   * In <a target=\'_blank\' href=\'http://biomodels.net/miriam\'>MIRIAM</a>,
   * qualifiers are an optional means of indicating the relationship
   * between a model component and its annotations.  There are two broad
   * kinds of annotations: <em>model</em> and <em>biological</em>.  The
   * latter kind is used to qualify the relationship between a model
   * component and a biological entity which it represents.  Examples of
   * relationships include \'is\' and \'has part\', but many others are
   * possible.  MIRIAM defines <a target=\'_blank\'
   * href=\'http://www.ebi.ac.uk/miriam/main/qualifiers/\'>numerous
   * relationship qualifiers</a> to enable different software tools to
   * qualify biological annotations in the same standardized way.  In
   * libSBML, the MIRIAM controlled-vocabulary annotations on an SBML model
   * element are represented using lists of CVTerm objects, and the
   * the MIRIAM biological qualifiers are represented using
   * values @if clike from the enumeration
   * type #BiolQualifierType_t.@endif@if python whose
   * names begin with <code>BQB_</code> in the interface class
   * @link libsbml libsbml@endlink.@endif@if java whose
   * names begin with <code>BQB_</code> in the interface class
   * {@link libsbmlConstants}.@endif@if csharp whose
   * names begin with <code>BQB_</code> in the interface class
   * @link libsbmlcs.libsbml libsbml@endlink.@endif
   *
   * This method searches the controlled-vocabulary annotations
   * (i.e., the list of CVTerm objects) on the present object, then out of
   * those that have biological qualifiers, looks for an annotation to the
   * given @p resource.  If such an annotation is found, it returns the
   * type of biological qualifier associated with that resource as a 
   * value @if clike from the enumeration type
   * #BiolQualifierType_t.@endif@if python whose name begins with
   * <code>BQB_</code> from the interface
   * class @link libsbml libsbml@endlink.@endif@if java whose name
   * begins with <code>BQB_</code> from the interface
   * class {@link libsbmlConstants}.@endif@if csharp whose
   * names begin with <code>BQB_</code> in the interface class
   * @link libsbmlcs.libsbml libsbml@endlink.@endif
   *
   * @param resource string representing the resource; e.g.,
   * <code>\'http://www.geneontology.org/#GO:0005892\'</code>.
   *
   * @return the qualifier associated with the resource,
   * or @link libsbml#BQB_UNKNOWN BQB_UNKNOWN@endlink if the
   * resource does not exist.
   *
   * @if clike
   * @note The set of MIRIAM biological qualifiers grows over
   * time, although relatively slowly.  The values in the enumeration
   * #BiolQualifierType_t are up to date with MIRIAM at the time of a given
   * libSBML release.  The set of values may be expanded in later libSBML
   * releases, to match the values defined by MIRIAM at that later time.
   * @endif@if python
   * @note The set of MIRIAM biological qualifiers grows over
   * time, although relatively slowly.  The values are up to date with
   * MIRIAM at the time of a given libSBML release.  The set of values in
   * list of <code>BQB_</code> constants defined in @link libsbml
   * libsbml@endlink may be expanded in later libSBML releases, to match
   * the values defined by MIRIAM at that later time.
   * @endif@if java
   * @note The set of MIRIAM biological qualifiers grows over
   * time, although relatively slowly.  The values are up to date with
   * MIRIAM at the time of a given libSBML release.  The set of values in
   * list of <code>BQB_</code> constants defined in {@link libsbmlConstants}
   * may be expanded in later libSBML releases, to match
   * the values defined by MIRIAM at that later time.
   * @endif@if csharp
   * @note The set of MIRIAM biological qualifiers grows over
   * time, although relatively slowly.  The values are up to date with
   * MIRIAM at the time of a given libSBML release.  The set of values in
   * list of <code>BQB_</code> constants defined in @link libsbmlcs.libsbml libsbml@endlink
   * may be expanded in later libSBML releases, to match
   * the values defined by MIRIAM at that later time.
   * @endif
   */ public
";


%csmethodmodifiers SBase::getResourceModelQualifier(std::string resource) "
/**
   * Returns the MIRIAM <em>model qualifier</em> associated with the
   * given resource.
   *
   * In <a target=\'_blank\' href=\'http://biomodels.net/miriam\'>MIRIAM</a>,
   * qualifiers are an optional means of indicating the relationship
   * between a model component and its annotations.  There are two broad
   * kinds of annotations: <em>model</em> and <em>biological</em>.  The
   * former kind is used to qualify the relationship between a model
   * component and another modeling object.  An example qualifier is
   * \'isDerivedFrom\', to indicate that a given component of the model is
   * derived from the modeling object represented by the referenced
   * resource.  MIRIAM defines <a target=\'_blank\'
   * href=\'http://www.ebi.ac.uk/miriam/main/qualifiers/\'>numerous
   * relationship qualifiers</a> to enable different software tools to
   * qualify model annotations in the same standardized way.  In libSBML,
   * the MIRIAM controlled-vocabulary annotations on an SBML model element
   * are represented using lists of CVTerm objects, and the 
   * the MIRIAM model qualifiers are represented using
   * values @if clike from the enumeration
   * type #ModelQualifierType_t.@endif@if python whose
   * names begin with <code>BQM_</code> in the interface class
   * @link libsbml libsbml@endlink.@endif@if java whose
   * names begin with <code>BQM_</code> in the interface class
   * {@link libsbmlConstants}.@endif@if csharp whose
   * names begin with <code>BQB_</code> in the interface class
   * @link libsbmlcs.libsbml libsbml@endlink.@endif
   *
   * This method method searches the controlled-vocabulary annotations
   * (i.e., the list of CVTerm objects) on the present object, then out of
   * those that have model qualifiers, looks for an annotation to the given
   * @p resource.  If such an annotation is found, it returns the type of
   * type of model qualifier associated with that resource as a 
   * value @if clike from the enumeration type
   * #ModelQualifierType_t.@endif@if python whose name begins with
   * <code>BQM_</code> from the interface
   * class @link libsbml libsbml@endlink.@endif@if java whose name
   * begins with <code>BQM_</code> from the interface
   * class {@link libsbmlConstants}.@endif@if csharp whose
   * names begin with <code>BQB_</code> in the interface class
   * @link libsbmlcs.libsbml libsbml@endlink.@endif
   *
   * @param resource string representing the resource; e.g.,
   * <code>\'http://www.geneontology.org/#GO:0005892\'</code>.
   *
   * @return the @if clike #ModelQualifierType_t value@else model qualifier
   * type@endif associated with the resource, or @link libsbml#BQM_UNKNOWN BQM_UNKNOWN@endlink if the resource does not exist.
   *
   * @if clike
   * @note The set of MIRIAM biological qualifiers grows over
   * time, although relatively slowly.  The values in the enumeration
   * #ModelQualifierType_t are up to date with MIRIAM at the time of a given
   * libSBML release.  The set of values may be expanded in later libSBML
   * releases, to match the values defined by MIRIAM at that later time.
   * @endif@if python
   * @note The set of MIRIAM model qualifiers grows over
   * time, although relatively slowly.  The values are up to date with
   * MIRIAM at the time of a given libSBML release.  The set of values in
   * list of <code>BQM_</code> constants defined in @link libsbml
   * libsbml@endlink may be expanded in later libSBML releases, to match
   * the values defined by MIRIAM at that later time.
   * @endif@if java
   * @note The set of MIRIAM model qualifiers grows over
   * time, although relatively slowly.  The values are up to date with
   * MIRIAM at the time of a given libSBML release.  The set of values in
   * list of <code>BQM_</code> constants defined in {@link libsbmlConstants}
   * may be expanded in later libSBML releases, to match
   * the values defined by MIRIAM at that later time.
   * @endif@if csharp
   * @note The set of MIRIAM model qualifiers grows over
   * time, although relatively slowly.  The values are up to date with
   * MIRIAM at the time of a given libSBML release.  The set of values in
   * list of <code>BQM_</code> constants defined in @link libsbmlcs.libsbml libsbml@endlink
   * may be expanded in later libSBML releases, to match
   * the values defined by MIRIAM at that later time.
   * @endif
   */ public
";


%csmethodmodifiers SBase::getModel() const "
/**
   * Returns the Model object for the SBML Document in which the current object is located.
   * 
   * @return the Model object for the SBML Document of this SBML object.
   *
   * @see getParentSBMLObject()
   * @see getSBMLDocument()
   */ public
";


%csmethodmodifiers SBase::getLevel() const "
/**
   * Returns the SBML Level of the SBMLDocument object containing @em this
   * object.
   * 
   *
 * 
 * LibSBML uses the class SBMLDocument as a top-level container for
 * storing SBML content and data associated with it (such as warnings and
 * error messages).  An SBML model in libSBML is contained inside an
 * SBMLDocument object.  SBMLDocument corresponds roughly to the class
 * <i>SBML</i> defined in the SBML Level&nbsp;3 and Level&nbsp;2
 * specifications, but it does not have a direct correspondence in SBML
 * Level&nbsp;1.  (But, it is created by libSBML no matter whether the
 * model is Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)
 *
 *
   * 
   * @return the SBML level of this SBML object.
   * 
   * @see getVersion()
   * @see getNamespaces()
   * @see getPackageVersion()
   */ public
";


%csmethodmodifiers SBase::getVersion() const "
/**
   * Returns the Version within the SBML Level of the SBMLDocument object
   * containing @em this object.
   * 
   *
 * 
 * LibSBML uses the class SBMLDocument as a top-level container for
 * storing SBML content and data associated with it (such as warnings and
 * error messages).  An SBML model in libSBML is contained inside an
 * SBMLDocument object.  SBMLDocument corresponds roughly to the class
 * <i>SBML</i> defined in the SBML Level&nbsp;3 and Level&nbsp;2
 * specifications, but it does not have a direct correspondence in SBML
 * Level&nbsp;1.  (But, it is created by libSBML no matter whether the
 * model is Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)
 *
 *
   * 
   * @return the SBML version of this SBML object.
   *
   * @see getLevel()
   * @see getNamespaces()
   */ public
";


%csmethodmodifiers SBase::getPackageCoreVersion() const "
/**
  * Returns the SBML Core Version within the SBML Level of the actual object.
  *
  *
 * 
 * LibSBML uses the class SBMLDocument as a top-level container for
 * storing SBML content and data associated with it (such as warnings and
 * error messages).  An SBML model in libSBML is contained inside an
 * SBMLDocument object.  SBMLDocument corresponds roughly to the class
 * <i>SBML</i> defined in the SBML Level&nbsp;3 and Level&nbsp;2
 * specifications, but it does not have a direct correspondence in SBML
 * Level&nbsp;1.  (But, it is created by libSBML no matter whether the
 * model is Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)
 *
 *
  *
  * @return the SBML core version of this SBML object.
  */ public
";


%csmethodmodifiers SBase::getPackageVersion() const "
/**
   * Returns the Version of the SBML Level&nbsp;3 package to which this
   * element belongs to.
   *
   * @return the version of the SBML Level&nbsp;3 package to which this
   * element belongs. The value @c 0 will be returned if this element
   * belongs to the SBML Level&nbsp;3 Core package.
   *
   * @see getLevel()
   * @see getVersion()
   */ public
";


%csmethodmodifiers SBase::getPackageName() const "
/**
   * Returns the name of the SBML Level&nbsp;3 package in which this element
   * is defined.
   *
   * @return the name of the SBML package in which this element is defined.
   * The string <code>&quot;core&quot;</code> will be returned if this
   * element is defined in SBML Level&nbsp;3 Core. The string
   * <code>&quot;unknown&quot;</code> will be returned if this element is
   * not defined in any SBML package.
   */ public
";


%csmethodmodifiers SBase::getTypeCode() const "
/**
   * Returns the libSBML type code for this object.
   *
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   *
 * 
 * Here follow some additional general information about libSBML type codes:
 *
 * @li The codes are the possible return values (integers) for the following
 * functions:
 * <ul>
 *     <li> SBase::getTypeCode()
 *     <li> ListOf::getItemTypeCode()
 * </ul>
 * (Compatibility note: in libSBML 5, the type of return values of these
 * functions changed from an enumeration to an integer for extensibility
 * in the face of different programming languages.)
 *
 * @li Each package extension must define similar sets of values for each
 * SBase subclass (e.g. <code>SBMLLayoutTypeCode_t</code> for the SBML Level&nbsp;3
 * %Layout extension, <code>SBMLFbcTypeCode_t</code> for the SBML Level&nbsp;3 Flux
 * Balance Constraints extension, etc.).
 *
 * @li The value of each package-specific type code can be duplicated between
 * those of different packages.  (This is necessary because the development
 * of libSBML extensions for different SBML packages may be undertaken by
 * different developers at different times; requiring the developers to
 * coordinate their use of type codes would be nettlesome and probably
 * doomed to failure.)
 *
 * @li To distinguish between the type codes of different packages, both the
 * return value of SBase::getTypeCode() and SBase::getPackageName() must be
 * checked.  This is particularly important for functions that take an SBML
 * type code as an argument, such as
 * SBase::getAncestorOfType(@if java int, String@endif), which by
 * default assumes you are handing it a core type, and will return @c null if
 * the value you give it is actually from a package.
 *
 * The following example code illustrates the combined use of
 * SBase::getPackageName() and SBase::getTypeCode():
 * @if cpp
 * @code{.cpp}
 void example (SBase sb)
 {
   cons string pkgName = sb->getPackageName();
   if (pkgName == \'core\')
   {
     switch (sb->getTypeCode())
     {
       case SBML_MODEL:
          ....
          break;
       case SBML_REACTION:
          ....
     }
   }
   else if (pkgName == \'layout\')
   {
     switch (sb->getTypeCode())
     {
       case SBML_LAYOUT_LAYOUT:
          ....
          break;
       case SBML_LAYOUT_REACTIONGLYPH:
          ....
     }
   }
   ...
 }
@endcode
@endif
@if python
@code{.py}
def example(item):
  pkg_name  = item.getPackageName()
  type_code = item.getTypeCode()
  if pkg_name == \'core\':
    print(\'Got a \' + SBMLTypeCode_toString(type_code, \'core\') + \' object\')
    if type_code == SBML_MODEL:
      print(\'This is a very, very nice model\')
      # Do whatever the application wants to do with the model.
    elif type_code == SBML_COMPARTMENT:
      print(\'This is a very, very nice compartment\')
      # Do whatever the application wants to do with the compartment.
    elif type_code == SBML_SPECIES:
      print(\'This is a very, very nice species\')
      # Do whatever the application wants to do with the species.
    elif ...
      ...
  elif pkg_name == \'layout\':
    print(\'Got a \' + SBMLTypeCode_toString(type_code, \'layout\') + \' object\')
    if type_code == SBML_LAYOUT_POINT:
      print(\'This is a very, very nice layout point\')
      # Do whatever the application wants to do with the layout point.
    elif type_code == SBML_LAYOUT_BOUNDINGBOX:
      print(\'This is a very, very nice layout bounding box\')
      # Do whatever the application wants to do with the layout bounding box.
    elif ...
      ...
  elif pkg_name == \'unknown\':
    print(\'Something went wrong -- libSBML did not recognize the object type\')
    # Handle errors
@endcode
@endif
@if java
@code{.java}
void example (SBase sb)
{
  String pkgName = sb.getPackageName();
  if (pkgName.equals(\'core\'))
  {
    switch (sb.getTypeCode())
    {
      case libsbml.SBML_MODEL:
         ....
         break;
      case libsbml.SBML_REACTION:
         ....
    }
  }
  else if (pkgName.equals(\'layout\'))
  {
    switch (sb.getTypeCode())
    {
      case libsbml.SBML_LAYOUT_LAYOUT:
         ....
         break;
      case libsbml.SBML_LAYOUT_REACTIONGLYPH:
         ....
    }
  }
  ...
}
@endcode
@endif
 *
 *
   *
   * @return the @if clike #SBMLTypeCode_t value@else SBML object type
   * code@endif of this SBML object, or @link libsbml#SBML_UNKNOWN SBML_UNKNOWN@endlink (the default).  The latter is possible because
   * subclasses of SBase are not required to implement this method to return
   * a type code.
   *
   *
 * @warning <span class=\'warning\'>The specific integer values of the possible
 * type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
 * packages,  To fully identify the correct code, <strong>it is necessary to
 * invoke both getPackageName() and getTypeCode()</strong> (or 
 * ListOf::getItemTypeCode()).</span>
 *
 *
   *
   * @see getPackageName()
   * @see getElementName()
   */ public new
";


%csmethodmodifiers SBase::hasValidLevelVersionNamespaceCombination "
/**
   * Predicate returning @c true if this object\'s level/version and namespace
   * values correspond to a valid SBML specification.
   *
   * The valid combinations of SBML Level, Version and Namespace as of this
   * release of libSBML are the following:
   * <ul>
   * <li> Level&nbsp;1 Version&nbsp;2: &quot;<code style=\'margin-right:0; padding-right:0\'>http</code><code style=\'margin-left:0; padding-left:0\'>://www.sbml.org/sbml/level1</code>&quot;
   * <li> Level&nbsp;2 Version&nbsp;1: &quot;<code style=\'margin-right:0; padding-right:0\'>http</code><code style=\'margin-left:0; padding-left:0\'>://www.sbml.org/sbml/level2</code>&quot;
   * <li> Level&nbsp;2 Version&nbsp;2: &quot;<code style=\'margin-right:0; padding-right:0\'>http</code><code style=\'margin-left:0; padding-left:0\'>://www.sbml.org/sbml/level2/version2</code>&quot;
   * <li> Level&nbsp;2 Version&nbsp;3: &quot;<code style=\'margin-right:0; padding-right:0\'>http</code><code style=\'margin-left:0; padding-left:0\'>://www.sbml.org/sbml/level2/version3</code>&quot;
   * <li> Level&nbsp;2 Version&nbsp;4: &quot;<code style=\'margin-right:0; padding-right:0\'>http</code><code style=\'margin-left:0; padding-left:0\'>://www.sbml.org/sbml/level2/version4</code>&quot;
   * <li> Level&nbsp;2 Version&nbsp;5: &quot;<code style=\'margin-right:0; padding-right:0\'>http</code><code style=\'margin-left:0; padding-left:0\'>://www.sbml.org/sbml/level2/version5</code>&quot;
   * <li> Level&nbsp;3 Version&nbsp;1 Core: &quot;<code style=\'margin-right:0; padding-right:0\'>http</code><code style=\'margin-left:0; padding-left:0\'>://www.sbml.org/sbml/level3/version1/core</code>&quot;
   * <li> Level&nbsp;3 Version&nbsp;2 Core: &quot;<code style=\'margin-right:0; padding-right:0\'>http</code><code style=\'margin-left:0; padding-left:0\'>://www.sbml.org/sbml/level3/version2/core</code>&quot;
   * </ul>
   *
   * @return @c true if the level, version and namespace values of this 
   * SBML object correspond to a valid set of values, @c false otherwise.
   */ public
";


%csmethodmodifiers SBase::getElementName() const "
/**
   * Returns the XML element name of this object.
   *
   * This is overridden by subclasses to return a string appropriate to the
   * SBML component.  For example, Model defines it as returning
   * @c \'model\', CompartmentType defines it as returning @c \'compartmentType\',
   * and so on.
   */ public new
";


%csmethodmodifiers SBase::toSBML "
/**
   * Returns a string consisting of a partial SBML corresponding to just
   * this object.
   * 
   * @return the partial SBML that describes this SBML object.
   *
   * @warning <span class=\'warning\'>This is primarily provided for testing
   * and debugging purposes.  It may be removed in a future version of
   * libSBML.</span>
   */ public
";


%csmethodmodifiers SBase::toXMLNode "
/**
   * Returns this element as an XMLNode.
   *
   * @return this element as an XMLNode.
   *
   * @warning <span class=\'warning\'>This operation is computationally
   * expensive, because the element has to be fully serialized to a string
   * and then parsed into the XMLNode structure.  Attempting to convert a
   * large tree structure (e.g., a large Model) may consume significant
   * computer memory and time.</span>
   */ public
";


%csmethodmodifiers SBase::read(XMLNode& node, XMLErrorSeverityOverride_t flag = LIBSBML_OVERRIDE_DISABLED) "
/**
   * Reads (initializes) this SBML object by reading from the given XMLNode.
   * 
   * @param node the XMLNode to read from.
   *
   * @param flag an optional flag that determines how how errors are logged
   * during the reading process.
   *
   * @warning <span class=\'warning\'>This method is computationally expensive,
   * because the given node has to be serialized to a string first.
   * Attempting to serialize a large tree structure (e.g., a large Model) may
   * consume significant computer memory and time.</span>
   */ public
";


%csmethodmodifiers SBase::getPlugin(const std::string& package) "
/**
   * Returns a plug-in object (extension interface) for an SBML Level&nbsp;3
   * package extension with the given package name or URI.
   * The returned plug-in will be the appropriate type of plugin requested:
   * calling Model::getPlugin(\'fbc\') will return an FbcModelPlugin; calling
   * Parameter::getPlugin(\'comp\') will return CompSBasePlugin, etc.
   *
   * If no such plugin exists, null is returned.
   *
   *
 * 
 * SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
 * via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
 * that it uses one or more SBML packages, and likewise, a software tool may be
 * able to support one or more packages.  LibSBML does not come preconfigured
 * with all possible packages included and enabled, in part because not all
 * package specifications have been finalized.  To support the ability for
 * software systems to enable support for the Level&nbsp;3 packages they choose,
 * libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
 * package is implemented in a separate code plug-in that can be enabled by the
 * application to support working with that SBML package.  A given SBML model
 * may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
 * objects created by libSBML plug-ins supporting additional Level&nbsp;3
 * packages.
 *
 *
   *
   * @param package the name or URI of the package.
   *
   * @return the plug-in object (the libSBML extension interface) of
   * a package extension with the given package name or URI.
   *
   * @see getPlugin(unsigned int n)
   */ public
";


%csmethodmodifiers SBase::getPlugin(const std::string& package) const "
/**
   * Returns a plug-in object (extension interface) for an SBML Level&nbsp;3
   * package extension with the given package name or URI.
   * The returned plug-in will be the appropriate type of plugin requested:
   * calling Model::getPlugin(\'fbc\') will return an FbcModelPlugin; calling
   * Parameter::getPlugin(\'comp\') will return CompSBasePlugin, etc.
   *
   * If no such plugin exists, null is returned.
   *
   *
 * 
 * SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
 * via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
 * that it uses one or more SBML packages, and likewise, a software tool may be
 * able to support one or more packages.  LibSBML does not come preconfigured
 * with all possible packages included and enabled, in part because not all
 * package specifications have been finalized.  To support the ability for
 * software systems to enable support for the Level&nbsp;3 packages they choose,
 * libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
 * package is implemented in a separate code plug-in that can be enabled by the
 * application to support working with that SBML package.  A given SBML model
 * may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
 * objects created by libSBML plug-ins supporting additional Level&nbsp;3
 * packages.
 *
 *
   *
   * @param package the name or URI of the package.
   *
   * @return the plug-in object (the libSBML extension interface) of a
   * package extension with the given package name or URI.
   *
   * @see getPlugin(unsigned int n)
   */ public
";


%csmethodmodifiers SBase::getPlugin(unsigned int n) "
/**
   * Returns the nth plug-in object (extension interface) for an SBML Level&nbsp;3
   * package extension.
   * The returned plug-in will be the appropriate type of plugin requested:
   * calling Model::getPlugin(\'fbc\') will return an FbcModelPlugin; calling
   * Parameter::getPlugin(\'comp\') will return CompSBasePlugin, etc.
   *
   * If no such plugin exists, null is returned.
   *
   *
 * 
 * SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
 * via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
 * that it uses one or more SBML packages, and likewise, a software tool may be
 * able to support one or more packages.  LibSBML does not come preconfigured
 * with all possible packages included and enabled, in part because not all
 * package specifications have been finalized.  To support the ability for
 * software systems to enable support for the Level&nbsp;3 packages they choose,
 * libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
 * package is implemented in a separate code plug-in that can be enabled by the
 * application to support working with that SBML package.  A given SBML model
 * may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
 * objects created by libSBML plug-ins supporting additional Level&nbsp;3
 * packages.
 *
 *
   *
   * @param n the index of the plug-in to return.
   *
   * @return the nth plug-in object (the libSBML extension interface) of a
   * package extension.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see getNumPlugins()
   * @see getPlugin(string package)
   */ public
";


%csmethodmodifiers SBase::getPlugin "
/**
   * Returns the nth plug-in object (extension interface) for an SBML Level&nbsp;3
   * package extension.
   * The returned plug-in will be the appropriate type of plugin requested:
   * calling Model::getPlugin(\'fbc\') will return an FbcModelPlugin; calling
   * Parameter::getPlugin(\'comp\') will return CompSBasePlugin, etc.
   *
   * If no such plugin exists, null is returned.
   *
   *
 * 
 * SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
 * via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
 * that it uses one or more SBML packages, and likewise, a software tool may be
 * able to support one or more packages.  LibSBML does not come preconfigured
 * with all possible packages included and enabled, in part because not all
 * package specifications have been finalized.  To support the ability for
 * software systems to enable support for the Level&nbsp;3 packages they choose,
 * libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
 * package is implemented in a separate code plug-in that can be enabled by the
 * application to support working with that SBML package.  A given SBML model
 * may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
 * objects created by libSBML plug-ins supporting additional Level&nbsp;3
 * packages.
 *
 *
   *
   * @param n the index of the plug-in to return.
   *
   * @return the nth plug-in object (the libSBML extension interface) of a
   * package extension.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see getNumPlugins()
   * @see getPlugin(string package)
   */ public
";


%csmethodmodifiers SBase::getDisabledPlugin(unsigned int n) "
/**
   * Returns the nth disabled plug-in object (extension interface) for an SBML Level&nbsp;3
   * package extension.  If no such plugin exists, null is returned.
   *
   *
 * 
 * SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
 * via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
 * that it uses one or more SBML packages, and likewise, a software tool may be
 * able to support one or more packages.  LibSBML does not come preconfigured
 * with all possible packages included and enabled, in part because not all
 * package specifications have been finalized.  To support the ability for
 * software systems to enable support for the Level&nbsp;3 packages they choose,
 * libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
 * package is implemented in a separate code plug-in that can be enabled by the
 * application to support working with that SBML package.  A given SBML model
 * may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
 * objects created by libSBML plug-ins supporting additional Level&nbsp;3
 * packages.
 *
 *
   *
   *
 * 
 * If a plugin is <em>disabled</em>, the package information it contains is
 * no longer considered to be part of the SBML document for the purposes of
 * searching the document or writing out the document.  However, the information
 * is still retained, so if the plugin is enabled again, the same information
 * will once again be available, and will be written out to the final model.
 *
 *
   *
   * @param n the index of the disabled plug-in to return.
   *
   * @return the nth disabled plug-in object (the libSBML extension interface) of a
   * package extension.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see getNumDisabledPlugins()
   * @see getPlugin(string package)
   */ public
";


%csmethodmodifiers SBase::getDisabledPlugin "
/**
   * Returns the nth disabled plug-in object (extension interface) for an SBML Level&nbsp;3
   * package extension.  If no such plugin exists, null is returned.
   *
   *
 * 
 * SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
 * via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
 * that it uses one or more SBML packages, and likewise, a software tool may be
 * able to support one or more packages.  LibSBML does not come preconfigured
 * with all possible packages included and enabled, in part because not all
 * package specifications have been finalized.  To support the ability for
 * software systems to enable support for the Level&nbsp;3 packages they choose,
 * libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
 * package is implemented in a separate code plug-in that can be enabled by the
 * application to support working with that SBML package.  A given SBML model
 * may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
 * objects created by libSBML plug-ins supporting additional Level&nbsp;3
 * packages.
 *
 *
   *
   *
 * 
 * If a plugin is <em>disabled</em>, the package information it contains is
 * no longer considered to be part of the SBML document for the purposes of
 * searching the document or writing out the document.  However, the information
 * is still retained, so if the plugin is enabled again, the same information
 * will once again be available, and will be written out to the final model.
 *
 *
   *
   * @param n the index of the disabled plug-in to return.
   *
   * @return the nth disabled plug-in object (the libSBML extension interface) of a
   * package extension.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see getNumDisabledPlugins()
   * @see getPlugin(string package)
   */ public
";


%csmethodmodifiers SBase::getNumPlugins() const "
/**
   * Returns the number of plug-in objects (extenstion interfaces) for SBML
   * Level&nbsp;3 package extensions known.
   *
   *
 * 
 * SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
 * via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
 * that it uses one or more SBML packages, and likewise, a software tool may be
 * able to support one or more packages.  LibSBML does not come preconfigured
 * with all possible packages included and enabled, in part because not all
 * package specifications have been finalized.  To support the ability for
 * software systems to enable support for the Level&nbsp;3 packages they choose,
 * libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
 * package is implemented in a separate code plug-in that can be enabled by the
 * application to support working with that SBML package.  A given SBML model
 * may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
 * objects created by libSBML plug-ins supporting additional Level&nbsp;3
 * packages.
 *
 *
   *
   * @return the number of plug-in objects (extension interfaces) of
   * package extensions known by this instance of libSBML.
   *
   * @see getPlugin(unsigned int n)
   */ public
";


%csmethodmodifiers SBase::getNumDisabledPlugins() const "
/**
   * Returns the number of disabled plug-in objects (extension interfaces) 
   * for SBML Level&nbsp;3 package extensions known.
   *
   *
 * 
 * SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
 * via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
 * that it uses one or more SBML packages, and likewise, a software tool may be
 * able to support one or more packages.  LibSBML does not come preconfigured
 * with all possible packages included and enabled, in part because not all
 * package specifications have been finalized.  To support the ability for
 * software systems to enable support for the Level&nbsp;3 packages they choose,
 * libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
 * package is implemented in a separate code plug-in that can be enabled by the
 * application to support working with that SBML package.  A given SBML model
 * may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
 * objects created by libSBML plug-ins supporting additional Level&nbsp;3
 * packages.
 *
 *
   *
   *
 * 
 * If a plugin is <em>disabled</em>, the package information it contains is
 * no longer considered to be part of the SBML document for the purposes of
 * searching the document or writing out the document.  However, the information
 * is still retained, so if the plugin is enabled again, the same information
 * will once again be available, and will be written out to the final model.
 *
 *
   *
   * @return the number of disabled plug-in objects (extension interfaces) 
   * of package extensions known by this instance of libSBML.
   *
   */ public
";


%csmethodmodifiers SBase::deleteDisabledPlugins(bool recursive=true) "
/**
   * Deletes all information stored in disabled plugins.  If the plugin is 
   * re-enabled later, it will then not have any previously-stored information.
   *
   *
 * 
 * SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
 * via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
 * that it uses one or more SBML packages, and likewise, a software tool may be
 * able to support one or more packages.  LibSBML does not come preconfigured
 * with all possible packages included and enabled, in part because not all
 * package specifications have been finalized.  To support the ability for
 * software systems to enable support for the Level&nbsp;3 packages they choose,
 * libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
 * package is implemented in a separate code plug-in that can be enabled by the
 * application to support working with that SBML package.  A given SBML model
 * may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
 * objects created by libSBML plug-ins supporting additional Level&nbsp;3
 * packages.
 *
 *
   *
   *
 * 
 * If a plugin is <em>disabled</em>, the package information it contains is
 * no longer considered to be part of the SBML document for the purposes of
 * searching the document or writing out the document.  However, the information
 * is still retained, so if the plugin is enabled again, the same information
 * will once again be available, and will be written out to the final model.
 *
 *
   *
   * @param recursive if @c true, the disabled information will be deleted
   * also from all child elements, otherwise only from this SBase element.
   *
   * @see getNumDisabledPlugins()
   */ public
";


%csmethodmodifiers SBase::enablePackage(const std::string& pkgURI, const std::string& pkgPrefix, bool flag) "
/**
   * Enables or disables the given SBML Level&nbsp;3 package on this object.
   *
   * This method enables the specified package on this object and other
   * objects connected by child-parent links in the same SBMLDocument object.
   * This method is the converse of
   * SBase::disablePackage(string pkgURI, string pkgPrefix).
   *
   * @param pkgURI the URI of the package.
   * 
   * @param pkgPrefix the XML prefix of the package.
   * 
   * @param flag whether to enable (@c true) or disable (@c false) the package.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_PKG_UNKNOWN LIBSBML_PKG_UNKNOWN@endlink
   * @li @link libsbml#LIBSBML_PKG_VERSION_MISMATCH LIBSBML_PKG_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_PKG_CONFLICTED_VERSION LIBSBML_PKG_CONFLICTED_VERSION@endlink
   *
   * @see disablePackage(string pkgURI, string pkgPrefix)
   */ public
";


%csmethodmodifiers SBase::disablePackage(const std::string& pkgURI, const std::string& pkgPrefix) "
/**
   * Disables the given SBML Level&nbsp;3 package on this object.
   *
   * This method disables the specified package on this object
   * and other objects connected by child-parent links in the same
   * SBMLDocument object.
   * 
   * An example of when this may be useful is during construction of model
   * components when mixing existing and new models.  Suppose your
   * application read an SBML document containing a model that used the SBML
   * Hierarchical %Model Composition (&ldquo;comp&rdquo;) package, and
   * extracted parts of that model in order to construct a new model in
   * memory.  The new, in-memory model will not accept a component drawn from
   * an other SBMLDocument with different package namespace declarations.
   * You could reconstruct the same namespaces in the in-memory model first,
   * but as a shortcut, you could also disable the package namespace on the
   * object being added.  Here is a code example to help clarify this:
   * @if cpp
@code{.cpp}
// We read in an SBML L3V1 model that uses the \'comp\'
// package namespace.
doc = readSBML(\'sbml-file-with-comp-elements.xml\');

// We extract one of the species from the model.
Species s1 = doc->getModel()->getSpecies(0);

// We construct a new model.  This model does not use the
// \'comp\' package.
Model  newModel = new Model(3,1);

// The following will fail with an error, because addSpecies()
// will first check that the parent of the given object has
// namespaces declared, and will discover that s1 does but
// newModel does not.

// newModel->addSpecies(s1);

// However, if we disable the \'comp\' package on s1, then
// the call to addSpecies will work.

s1->disablePackage(\'http://www.sbml.org/sbml/level3/version1/comp/version1\',
                   \'comp\');
newModel->addSpecies(s1);
@endcode
@endif
@if python
@code{.py}
import sys
import os.path
from libsbml import *

# We read an SBML L3V1 model that uses the \'comp\' package.

doc = readSBML(\'sbml-file-with-comp-elements.xml\');
if doc.getNumErrors() > 0:
  print(\'readSBML encountered errors while reading the file.\')
  doc.printErrors()
  sys.exit(1)

# We extract one of the species from the model.

model = doc.getModel()
if model == None:
  print(\'Unable to retrieve Model object\')
  sys.exit(1)

s1 = model.getSpecies(0)
if s1 == None:
  print(\'Unable to retrieve Species object\')
  sys.exit(1)

# We construct a new model.
# This model does not use the \'comp\' package.

try:
  newDoc = SBMLDocument(3, 1)
except ValueError:
  print(\'Could not create SBMLDocument object\')
  sys.exit(1)

newModel = newDoc.createModel()
if newModel == None:
  print(\'Unable to create new Model object\')
  sys.exit(1)

# The following would normally fail with an error, because
# addSpecies() would first check that the parent of the given
# object has namespaces declared, and will discover that s1
# does but newModel does not.

#   newModel.addSpecies(s1)

# However, if we disable the \'comp\' package on s1, then the
# call to addSpecies will work.

compNS = \'http://www.sbml.org/sbml/level3/version1/comp/version1\'
status = s1.disablePackage(compNS, \'comp\')
if status != LIBSBML_OPERATION_SUCCESS:
  print(\'Unable to disable package.\')
  sys.exit(1)

newSpecies = newModel.addSpecies(s1)   # This will work now.
if newSpecies == None:
  print(\'Could not add Species\')       # (This will not happen,
  sys.exit(1)                          # but always check errors.)
@endcode
@endif
@if java
@code{.java}
// We read in an SBML L3V1 model that uses the \'comp\' package namespace
SBMLReader reader = new SBMLReader();
SBMLDocument doc = reader.readSBML(\'sbml-file-with-comp-elements.xml\');

// We extract one of the species from the model we just read in.
Species s1 = doc.getModel().getSpecies(0);

// We construct a new model.  This model does not use the \'comp\' package.
Model newModel = new Model(3,1);

// The following will fail with an error, because addSpecies() will
// first check that the parent of the given object has namespaces
// declared, and will discover that s1 does but newModel does not.

// newModel->addSpecies(s1);

// However, if we disable the \'comp\' package on s1, then the call
// to addSpecies will work.

s1->disablePackage(\'http://www.sbml.org/sbml/level3/version1/comp/version1\',
                   \'comp\');
newModel.addSpecies(s1);
@endcode
@endif
   *
   * @param pkgURI the URI of the package.
   * 
   * @param pkgPrefix the XML prefix of the package.
   * 
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_PKG_UNKNOWN LIBSBML_PKG_UNKNOWN@endlink
   * @li @link libsbml#LIBSBML_PKG_VERSION_MISMATCH LIBSBML_PKG_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_PKG_CONFLICTED_VERSION LIBSBML_PKG_CONFLICTED_VERSION@endlink
   *
   * @see enablePackage(string pkgURI, string pkgPrefix, bool flag)
   */ public
";


%csmethodmodifiers SBase::enablePackageInternal(const std::string& pkgURI, const std::string& pkgPrefix, bool flag) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::isPackageURIEnabled(const std::string& pkgURI) const "
/**
   * Predicate returning @c true if an SBML Level&nbsp;3 package with the
   * given URI is enabled with this object.
   *
   * @param pkgURI the URI of the package.
   *
   * @return @c true if the given package is enabled within this object, @c 
   * false otherwise.
   *
   * @see isPackageEnabled(@if java String@endif)
   */ public
";


%csmethodmodifiers SBase::isPackageEnabled(const std::string& pkgName) const "
/**
   * Predicate returning @c true if the given SBML Level&nbsp;3 package is
   * enabled with this object.
   *
   * The search ignores the package version.
   *
   * @param pkgName the name of the package.
   *
   * @return @c true if the given package is enabled within this object,
   * @c false otherwise.
   *
   * @see isPackageURIEnabled(@if java String@endif)
   */ public
";


%csmethodmodifiers SBase::isPkgURIEnabled(const std::string& pkgURI) const "
/**
   * Predicate returning @c true if an SBML Level&nbsp;3 package with the
   * given URI is enabled with this object.
   *
   * @param pkgURI the URI of the package.
   *
   * @return @c true if the given package is enabled within this object, @c 
   * false otherwise.
   *
   * @deprecated Replaced in libSBML 5.2.0 by
   * SBase::isPackageURIEnabled(@if java String@endif).
   *
   * @see isPkgEnabled(@if java String@endif)
   */ public
";


%csmethodmodifiers SBase::isPkgEnabled(const std::string& pkgName) const "
/**
   * Predicate returning @c true if the given SBML Level&nbsp;3 package is
   * enabled with this object.
   *
   * The search ignores the package version.
   *
   * @param pkgName the name of the package.
   *
   * @return @c true if the given package is enabled within this object,
   * @c false otherwise.
   *
   * @deprecated Replaced in libSBML 5.2.0 by
   * SBase::isPackageEnabled(@if java String@endif).
   *
   * @see isPkgURIEnabled(@if java String@endif)
   */ public
";


%csmethodmodifiers SBase::writeExtensionElements(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::read(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::write(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::writeElements(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::hasRequiredAttributes() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::hasRequiredElements() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::checkCompatibility(const SBase * object) const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::setSBMLNamespaces(SBMLNamespaces * sbmlns) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::setSBMLNamespacesAndOwn(SBMLNamespaces * disownedNs) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::getSBMLNamespaces() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::removeDuplicateAnnotations "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::checkMathMLNamespace(const XMLToken elem) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::getDerivedUnitDefinition "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::containsUndeclaredUnits "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::removeFromParentAndDelete "
/**
   * Removes this object from its parent.
   *
   * If the parent was storing this object as a pointer, it is deleted.  If
   * not, it is simply cleared (as in ListOf objects).  This is a pure
   * virtual method, as every SBase element has different parents, and
   * therefore different methods of removing itself.  Will fail (and not
   * delete itself) if it has no parent object.  This function is designed to
   * be overridden, but for all objects whose parent is of the class ListOf,
   * the default implementation will work.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   */ public new
";


%csmethodmodifiers SBase::matchesSBMLNamespaces(const SBase * sb) "
/**
   * Returns @c true if this object\'s set of XML namespaces are the same
   * as the given object\'s XML namespaces.
   *
   *
 * 
 * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
 * information.  It is used to communicate the SBML Level, Version, and (in
 * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
 * common approach to using libSBML\'s SBMLNamespaces facilities is to create an
 * SBMLNamespaces object somewhere in a program once, then hand that object
 * as needed to object constructors that accept SBMLNamespaces as arguments.
 *
 *
   *
   * @param sb an object to compare with respect to namespaces.
   *
   * @return boolean, @c true if this object\'s collection of namespaces is
   * the same as @p sb\'s, @c false otherwise.
   */ public
";


%csmethodmodifiers SBase::matchesSBMLNamespaces(const SBase * sb) const "
/**
   * Returns @c true if this object\'s set of XML namespaces are the same
   * as the given object\'s XML namespaces.
   *
   *
 * 
 * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
 * information.  It is used to communicate the SBML Level, Version, and (in
 * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
 * common approach to using libSBML\'s SBMLNamespaces facilities is to create an
 * SBMLNamespaces object somewhere in a program once, then hand that object
 * as needed to object constructors that accept SBMLNamespaces as arguments.
 *
 *
   *
   * @param sb an object to compare with respect to namespaces.
   *
   * @return boolean, @c true if this object\'s collection of namespaces is
   * the same as @p sb\'s, @c false otherwise.
   */ public
";


%csmethodmodifiers SBase::matchesRequiredSBMLNamespacesForAddition(const SBase * sb) "
/**
   * Returns @c true if this object\'s set of XML namespaces are a subset
   * of the given object\'s XML namespaces.
   *
   *
 * 
 * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
 * information.  It is used to communicate the SBML Level, Version, and (in
 * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
 * common approach to using libSBML\'s SBMLNamespaces facilities is to create an
 * SBMLNamespaces object somewhere in a program once, then hand that object
 * as needed to object constructors that accept SBMLNamespaces as arguments.
 *
 *
   *
   * @param sb an object to compare with respect to namespaces.
   *
   * @return boolean, @c true if this object\'s collection of namespaces is
   * a subset of @p sb\'s, @c false otherwise.
   */ public
";


%csmethodmodifiers SBase::matchesRequiredSBMLNamespacesForAddition(const SBase * sb) const "
/**
   * Returns @c true if this object\'s set of XML namespaces are a subset
   * of the given object\'s XML namespaces.
   *
   *
 * 
 * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
 * information.  It is used to communicate the SBML Level, Version, and (in
 * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
 * common approach to using libSBML\'s SBMLNamespaces facilities is to create an
 * SBMLNamespaces object somewhere in a program once, then hand that object
 * as needed to object constructors that accept SBMLNamespaces as arguments.
 *
 *
   *
   * @param sb an object to compare with respect to namespaces.
   *
   * @return boolean, @c true if this object\'s collection of namespaces is
   * a subset of @p sb\'s, @c false otherwise.
   */ public
";


%csmethodmodifiers SBase::setUserData(void *userData) "
/**
   * Sets the user data of this element.
   *
   *
 * 
 * The user data associated with an SBML object can be used by an application
 * developer to attach custom information to that object in the model.  In case
 * of a deep copy, this data will passed as-is.  The data attribute will never
 * be interpreted by libSBML.
   *
   * @param userData specifies the new user data.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   */ public
";


%csmethodmodifiers SBase::*getUserData() const "
/**
   * Returns the user data that has been previously set via setUserData().
   *
   *
 * 
 * The user data associated with an SBML object can be used by an application
 * developer to attach custom information to that object in the model.  In case
 * of a deep copy, this data will passed as-is.  The data attribute will never
 * be interpreted by libSBML.
   *
   * @return the user data of this node, or @c null if no user data has been set.
   *
   * @if clike
   * @see ASTNode::setUserData(void *userData)
   * @endif
   */ public
";


%csmethodmodifiers SBase::isSetUserData() const "
/**
   * Predicate returning true or false depending on whether
   * the user data of this element has been set.
   *
   *
 * 
 * The user data associated with an SBML object can be used by an application
 * developer to attach custom information to that object in the model.  In case
 * of a deep copy, this data will passed as-is.  The data attribute will never
 * be interpreted by libSBML.
   *
   * @return boolean, @c true if this object\'s user data has been set,
   * @c false otherwise.
   */ public
";


%csmethodmodifiers SBase::unsetUserData "
/**
   * Unsets the user data of this element.
   *
   *
 * 
 * The user data associated with an SBML object can be used by an application
 * developer to attach custom information to that object in the model.  In case
 * of a deep copy, this data will passed as-is.  The data attribute will never
 * be interpreted by libSBML.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   */ public
";


%csmethodmodifiers SBase::getURI() const "
/**
   * Gets the namespace URI to which this element belongs to.
   *
   * For example, all elements that belong to SBML Level&nbsp;3 Version&nbsp;1 Core
   * must would have the URI <code>\'http://www.sbml.org/sbml/level3/version1/core\'</code>;
   * all elements that belong to Layout Extension Version 1 for SBML Level&nbsp;3
   * Version&nbsp;1 Core must would have the URI
   * <code>\'http://www.sbml.org/sbml/level3/version1/layout/version1\'</code>.
   *
   * This function first returns the URI for this element by looking into the
   * SBMLNamespaces object of the document with the its package name.  If not
   * found, it will then look for the namespace associated with the element
   * itself.
   *
   * @return the URI of this element, as a text string.
   *
   * @see getSBMLDocument()
   * @see getPackageName()
   */ public
";


%csmethodmodifiers SBase::getPrefix() const "
/**
   * Returns the XML namespace prefix of this element.
   *
   * This reports the XML namespace prefix chosen for this class of object in
   * the current SBML document.  This may be an empty string if the component
   * has no explicit prefix (for instance, if it is a core SBML object placed
   * in the default SBML namespace of the document).  If it is not empty, then
   * it corresponds to the XML namespace prefix used set the object, whatever
   * that may be in a given SBML document.
   *
   * @return a text string representing the XML namespace prefix.
   */ public
";


%csmethodmodifiers SBase::hasOptionalAttributes() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::hasOptionalElements() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::updateSBMLNamespace(const std::string& package, unsigned int level, unsigned int version) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::setElementText(const std::string &text) "
/** 
   * When overridden allows SBase elements to use the text included in between
   * the elements tags. The default implementation does nothing.
   * 
   * @param text the text string found between the element tags.
   */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::matchesCoreSBMLNamespace(const SBase * sb) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::matchesCoreSBMLNamespace(const SBase * sb) const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::SBase(unsigned int level, unsigned int version) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::SBase(SBMLNamespaces* sbmlns) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::SBase(const SBase& orig) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::createObject(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::hasValidLevelVersionNamespaceCombination(int typecode, XMLNamespaces *xmlns) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::readOtherXML(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::getElementPosition() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::getErrorLog "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::logError(  unsigned int       id , const unsigned int level   = 2 , const unsigned int version = 3 , const std::string& details = "" ) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::logUnknownAttribute( const std::string& attribute, const unsigned int level, const unsigned int version, const std::string& element, const std::string& prefix="") "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::logUnknownElement( const std::string& element, const unsigned int level, const unsigned int version ) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::logEmptyString( const std::string& attribute, const unsigned int level, const unsigned int version, const std::string& element) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::addExpectedAttributes(ExpectedAttributes& attributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::writeAttributes(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::writeXMLNS(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::syncAnnotation "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::reconstructRDFAnnotation "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::checkOrderAndLogError(SBase* object, int expected) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::checkListOfPopulated(SBase* object) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::checkDefaultNamespace(const XMLNamespaces* xmlns, const std::string& elementName, const std::string& prefix = "") "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::checkAnnotation "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::checkXHTML(const XMLNode *) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::loadPlugins(SBMLNamespaces *sbmlns) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::createExtensionObject(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::setElementNamespace(const std::string &uri) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::getElementNamespace() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::readExtensionAttributes(const XMLAttributes& attributes, const ExpectedAttributes* expectedAttributes = NULL) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::writeExtensionAttributes(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBase::storeUnknownExtAttribute(const std::string& element, const XMLAttributes& xattr, unsigned int index) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::storeUnknownExtElement(XMLInputStream &stream) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::getSBMLPrefix() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::getRootElement "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::getHasBeenDeleted() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::setSBaseFields(const XMLToken& element) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::readAnnotation(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::removeDuplicatedResources(CVTerm *term, QualifierType_t type) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::addTermToExistingBag(CVTerm *term, QualifierType_t type) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBase::readNotes(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public
";


%typemap(csimports) ListOf "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html Parent class for libSBML\'s \'ListOfXYZ\' classes.
 *
 * @htmlinclude not-sbml-warning.html
 *
 * The ListOf class in libSBML is a utility class that serves as the parent
 * class for implementing the ListOf__ classes.  It provides methods for
 * working generically with the various SBML lists of objects in a program.
 * LibSBML uses this separate list class rather than ordinary
 * @if conly C@endif@if cpp C++; @endif@if java Java@endif@if python Python@endif lists,
 * so that it can provide the methods and features associated with SBase.
 *
 * Whether a given ListOf element may be empty or not depends on the 
 * element in question, and on what level and version of SBML it
 * is being used for.  For ListOf elements in SBML Level&nbsp;3
 * Version&nbsp;1 and prior, no core list and few package lists could
 * be empty.  As of SBML Level&nbsp;3 Version&nbsp;2, the rules
 * were relaxed, and lists were allowed to be empty.  In libSBML,
 * documents created for Level&nbsp;3 Version&nbsp;2 will be written
 * with empty ListOf\'s if that ListOf contains some other \'extra\'
 * information: an attribute such as metaid or sboTerm, a child
 * \'&lt;notes&gt;\' or \'&lt;annotation&gt;\', or information from a SBML 
 * Level&nbsp;3 package.
 *
 *
 * 
 * The various ListOf___ @if conly structures @else classes@endif in SBML
 * are merely containers used for organizing the main components of an SBML
 * model.  In libSBML\'s implementation, ListOf___
 * @if conly data structures @else classes@endif are derived from the
 * intermediate utility @if conly structure @else class@endif ListOf, which
 * is not defined by the SBML specifications but serves as a useful
 * programmatic construct.  ListOf is itself is in turn derived from SBase,
 * which provides all of the various ListOf___
 * @if conly data structures @else classes@endif with common features
 * defined by the SBML specification, such as \'metaid\' attributes and
 * annotations.
 *
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 *
 * @htmlinclude listof-illustration.html
 *
 * SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
 * Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
 * Version&nbsp;2 allows
 * containers to contain zero or more of the relevant object, instead of 
 * requiring at least one.  As such, libsbml will write out an 
 * otherwise-empty ListOf___ element that has any optional attribute set 
 * (such as \'id\' or \'metaid\'), that has an optional child (such 
 * as a \'notes\' or \'annotation\'), or that has attributes or children set
 * from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
 * any other children.
 *
 * Readers may wonder about the motivations for using the ListOf___
 * containers in SBML.  A simpler approach in XML might be to place the
 * components all directly at the top level of the model definition.  The
 * choice made in SBML is to group them within XML elements named after
 * %ListOf<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from SBase means that software tools can add information @em about
 * the lists themselves into each list container\'s \'annotation\'.
 *
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 *
 * @if conly
 * @note In the C API for libSBML, functions that in other language APIs
 * would be inherited by the various ListOf___ structures not shown in the
 * pages for the individual ListOf___\'s.  Instead, the functions are defined
 * on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
 * the many common functions available for manipulating ListOf___
 * structures</strong>.  The documentation for the individual ListOf___
 * structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
 * all of the functionality available. @endif
 *
 *
 */
"


%csmethodmodifiers ListOf::ListOf(unsigned int level   = SBML_DEFAULT_LEVEL, unsigned int version = SBML_DEFAULT_VERSION) "
/**
   * Creates a new ListOf object.
   *
   * @param level the SBML Level; if not assigned, defaults to the
   * value of SBMLDocument::getDefaultLevel().
   *
   * @param version the Version within the SBML Level; if not assigned,
   * defaults to the value of SBMLDocument::getDefaultVersion().
   *
   *
 * @note Bare ListOf @if conly structures @else objects@endif are
 * impossible to add to SBML models.  The ListOf
 * @if conly structure type@else class@endif is simply the base
 * of <em>other</em> @if conly structure types @else classes@endif in
 * libSBML.  Calling programs are not intended to create bare ListOf
 * @if conly structures @else objects@endif themselves.
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   *
   * @ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif
   */ public
";


%csmethodmodifiers ListOf::ListOf(SBMLNamespaces* sbmlns) "
/**
   * Creates a new ListOf with a given SBMLNamespaces object.
   *
   * @param sbmlns the set of SBML namespaces that this ListOf should
   * contain.
   *
   *
 * @note Bare ListOf @if conly structures @else objects@endif are
 * impossible to add to SBML models.  The ListOf
 * @if conly structure type@else class@endif is simply the base
 * of <em>other</em> @if conly structure types @else classes@endif in
 * libSBML.  Calling programs are not intended to create bare ListOf
 * @if conly structures @else objects@endif themselves.
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOf::ListOf(const ListOf& orig) "
/**
   * Copy constructor; creates a copy of this ListOf.
   *
   * @param orig the ListOf instance to copy.
   */ public
";


%csmethodmodifiers ListOf::accept(SBMLVisitor& v) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOf::clone() const "
/**
   * Creates and returns a deep copy of this ListOf object.
   *
   * @return the (deep) copy of this ListOf object.
   */ public new
";


%csmethodmodifiers ListOf::append(const SBase* item) "
/**
   * Adds an item to the end of this ListOf\'s list of items.
   *
   * This method makes a clone of the @p item handed to it.  This means that
   * when the ListOf object is destroyed, the original items will not be
   * destroyed.  For a method with an alternative ownership behavior, see the
   * ListOf::appendAndOwn(@if java SBase@endif) method.
   *
   * @param item the item to be added to the list.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   *
   * @see appendAndOwn(SBase disownedItem)
   * @see appendFrom(ListOf list)
   */ public
";


%csmethodmodifiers ListOf::appendAndOwn(SBase* disownedItem) "
/**
   * Adds an item to the end of this ListOf\'s list of items.
   *
   * This method does not clone the @p disownedItem handed to it; instead, it assumes
   * ownership of it.  This means that when the ListOf is destroyed, the item
   * will be destroyed along with it.  For a method with an alternative
   * ownership behavior, see the ListOf::append(SBase item) method.
   *
   * @param disownedItem the item to be added to the list.
   * Will become a child of the parent list.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   *
   * @see append(SBase item)
   * @see appendFrom(ListOf list)
   */ public
";


%csmethodmodifiers ListOf::appendFrom(const ListOf* list) "
/**
   * Adds a clone of a list of items to this ListOf\'s list.
   *
   * Note that because this clones the objects handed to it, the original
   * items will not be destroyed when this ListOf object is destroyed.
   *
   * @param list a list of items to be added.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   *
   * @see append(SBase item)
   * @see appendAndOwn(SBase disownedItem)
   */ public new
";


%csmethodmodifiers ListOf::insert(int location, const SBase* item) "
/**
   * Inserts an item at a given position in this ListOf\'s list of items.
   *
   * This variant of the method makes a clone of the @p item handed to it.
   * This means that when the ListOf is destroyed, the original @p item will
   * <em>not</em> be destroyed.
   *
   * @param location the location in the list where to insert the item.
   * @param item the item to be inserted to the list.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   *
   * @see insertAndOwn(int location, SBase item)
   */ public
";


%csmethodmodifiers ListOf::insertAndOwn(int location, SBase* disownedItem) "
/**
   * Inserts an item at a given position in this ListOf\'s list of items.
   *
   * This variant of the method does not make a clone of the @p disownedItem handed to it.
   * This means that when the ListOf is destroyed, the original @p item
   * <em>will</em> be destroyed.
   *
   * @param location the location where to insert the item.
   * @param disownedItem the item to be inserted to the list.
   * Will become a child of the parent list.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   *
   * @see insert(int location, SBase item)
   */ public
";


%csmethodmodifiers ListOf::get "
/**
   * Get an item from the list.
   *
   * @param n the index number of the item to get.
   *
   * @return the <em>n</em>th item in this ListOf items, or a null pointer if
   * the index number @p n refers to a nonexistent position in this list.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOf::get(unsigned int n) "
/**
   * Get an item from the list.
   *
   * @param n the index number of the item to get.
   * 
   * @return the <em>n</em>th item in this ListOf items, or a null pointer if
   * the index number @p n refers to a nonexistent position in this list.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOf::getElementBySId(const std::string& id) "
/**
   * Returns the first child element found that has the given identifier.
   *
   * This method searches this ListOf\'s list of items for SBML objects based
   * on their \'id\' attribute value in the model-wide <code>SId</code>
   * identifier namespace.
   *
   * @param id string representing the id of the object to find.
   *
   * @return the first element found with the given @p id, or @c null if no
   * such object is found.
   */ public new
";


%csmethodmodifiers ListOf::getElementByMetaId(const std::string& metaid) "
/**
   * Returns the first child element found with the given meta-identifier.
   *
   * @param metaid string representing the \'metaid\' attribute of the object
   * to find.
   *
   * @return the first element found with the given @p metaid, or @c null if
   * no such object is found.
   */ public new
";


%csmethodmodifiers ListOf::getAllElements(ElementFilter* filter=NULL) "
/**
   * Returns a List of all child SBase objects.
   *
   * The values returned include all children of the objects in this ListOf
   * list, nested to an arbitrary depth.
   *
   * @param filter a pointer to an ElementFilter, which causes the function 
   * to return only elements that match a particular set of constraints.  
   * If null (the default), the function will return all child objects.
   *
   * @return a List of pointers to all child objects.
   */ public new
";


%csmethodmodifiers ListOf::clear(bool doDelete = true) "
/**
   * Removes all items in this ListOf object.
   *
   * If parameter @p doDelete is @c true (default), all items in this ListOf
   * object are deleted and cleared, and thus the caller doesn\'t have to
   * delete those items.  Otherwise, all items are cleared only from this
   * ListOf object; the caller is still responsible for deleting the actual
   * items.  (In the latter case, callers are advised to store pointers to
   * all items elsewhere before calling this function.)
   *
   * @param doDelete if @c true (default), all items are deleted and cleared.
   * Otherwise, all items are just cleared and not deleted.
   *
   * @ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif
   */ public
";


%csmethodmodifiers ListOf::removeFromParentAndDelete "
/**
   * Removes all items in this ListOf object and deletes its properties too.
   *
   * This performs a call to clear() with an argument of @c true (thus removing
   * all the child objects in the list), followed by calls to various libSBML
   * <code>unset<em>Foo</em></code> methods to delete everything else: CVTerm
   * objects, model history objects, etc.
   *
   * @if cpp Implementations of subclasses of ListOf may need to override
   * this method if different handling of child objects is needed.@endif
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   */ public new
";


%csmethodmodifiers ListOf::remove(unsigned int n) "
/**
   * Removes the <em>n</em>th item from this ListOf list of items and returns
   * it.
   *
   * The caller owns the returned item and is responsible for deleting it.
   *
   * @param n the index of the item to remove.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOf::size() const "
/**
   * Returns number of items in this ListOf list.
   *
   * @return the number of items in this ListOf items.
   */ public
";


%csmethodmodifiers ListOf::setSBMLDocument(SBMLDocument* d) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOf::getTypeCode() const "
/**
   * Returns the libSBML type code for this object, namely,
   * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink.
   * 
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for this object:
   * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink (default).
   *
   * @note The various ListOf classes mostly differ from each other in what they
   * contain.  Hence, one must call getItemTypeCode() to fully determine the
   * class of this SBML object.
   *
   *
 * @warning <span class=\'warning\'>The specific integer values of the possible
 * type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
 * packages,  To fully identify the correct code, <strong>it is necessary to
 * invoke both getPackageName() and getTypeCode()</strong> (or 
 * ListOf::getItemTypeCode()).</span>
 *
 *
   *
   * @see getItemTypeCode()
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers ListOf::getItemTypeCode() const "
/**
   * Get the type code of the objects contained in this ListOf.
   *
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * Classes that inherit from the ListOf class should override this method
   * to return the SBML type code for the objects contained in this ListOf.
   * If they do not, this method will return
   * @link libsbml#SBML_UNKNOWN SBML_UNKNOWN@endlink
   *
   * @return The ListOf base class contains no SBML objects, and therefore
   * this method returns @link libsbml#SBML_UNKNOWN SBML_UNKNOWN@endlink.
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers ListOf::getElementName() const "
/**
   * Returns the XML element name of this object, which for ListOf, is
   * always @c \'listOf\'.
   *
   * @return the XML name of this element.
   */ public new
";


%csmethodmodifiers ListOf::writeElements(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOf::enablePackageInternal(const std::string& pkgURI, const std::string& pkgPrefix, bool flag) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOf::updateSBMLNamespace(const std::string& package, unsigned int level, unsigned int version) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOf::hasOptionalElements() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOf::isExplicitlyListed() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers ListOf::setExplicitlyListed(bool value=true) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers ListOf::addExpectedAttributes(ExpectedAttributes& attributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOf::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOf::writeAttributes(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOf::isValidTypeForList(SBase * item) "
/** */ /* libsbml-internal */ public new
";


%typemap(csimports) Model "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html An SBML model.
 *
 * In an SBML model definition, a single object of class Model serves as
 * the overall container for the lists of the various model components.
 * All of the lists are optional, but if a given list container is present
 * within the model, the list must not be empty; that is, it must have
 * length one or more.  The following are the components and lists
 * permitted in different Levels and Versions of SBML in
 * version @htmlinclude libsbml-version.html
 * of libSBML:
 * <ul>
 * <li> In SBML Level 1, the components are: UnitDefinition, Compartment,
 * Species, Parameter, Rule, and Reaction.  Instances of the classes are
 * placed inside instances of classes ListOfUnitDefinitions,
 * ListOfCompartments, ListOfSpecies, ListOfParameters, ListOfRules, and
 * ListOfReactions.
 *
 * <li> In SBML Level 2 Version 1, the components are: FunctionDefinition,
 * UnitDefinition, Compartment, Species, Parameter, Rule, Reaction and
 * Event.  Instances of the classes are placed inside instances of classes
 * ListOfFunctionDefinitions, ListOfUnitDefinitions, ListOfCompartments,
 * ListOfSpecies, ListOfParameters, ListOfRules, ListOfReactions, and
 * ListOfEvents.
 *
 * <li> In SBML Level 2 Versions 2, 3 and 4, the components are:
 * FunctionDefinition, UnitDefinition, CompartmentType, SpeciesType,
 * Compartment, Species, Parameter, InitialAssignment, Rule, Constraint,
 * Reaction and Event.  Instances of the classes are placed inside
 * instances of classes ListOfFunctionDefinitions, ListOfUnitDefinitions,
 * ListOfCompartmentTypes, ListOfSpeciesTypes, ListOfCompartments,
 * ListOfSpecies, ListOfParameters, ListOfInitialAssignments, ListOfRules,
 * ListOfConstraints, ListOfReactions, and ListOfEvents.
 *
 * <li> In SBML Level 3 Version 1, the components are: FunctionDefinition,
 * UnitDefinition, Compartment, Species, Parameter, InitialAssignment,
 * Rule, Constraint, Reaction and Event.  Instances of the classes are
 * placed inside instances of classes ListOfFunctionDefinitions,
 * ListOfUnitDefinitions, ListOfCompartments, ListOfSpecies,
 * ListOfParameters, ListOfInitialAssignments, ListOfRules,
 * ListOfConstraints, ListOfReactions, and ListOfEvents.  
 * </ul>
 *
 * Although all the lists are optional, there are dependencies between SBML
 * components such that defining some components requires defining others.
 * An example is that defining a species requires defining a compartment,
 * and defining a reaction requires defining a species.  The dependencies
 * are explained in more detail in the SBML specifications.
 *
 * In addition to the above lists and attributes, the Model class in both
 * SBML Level&nbsp;2 and Level&nbsp;3 has the usual two attributes of \'id\'
 * and \'name\', and both are optional.  As is the case for other SBML
 * components with \'id\' and \'name\' attributes, they must be used according
 * to the guidelines described in the SBML specifications.  (Within the
 * frameworks of SBML Level&nbsp;2 and Level&nbsp;3, a
 * Model object identifier has no assigned meaning, but extension packages
 * planned for SBML Level&nbsp;3 are likely to make use of this
 * identifier.)
 *
 * Finally, SBML Level&nbsp;3 has introduced a number of additional Model
 * attributes.  They are discussed in a separate section below.
 *
 *
 * @section approaches Approaches to creating objects using the libSBML API
 *
 * LibSBML provides two main mechanisms for creating objects: class
 * constructors
 * (e.g., @if java <a href=\'org/sbml/libsbml/Species.html\'>Species()</a> @else Species::Species() @endif), 
 * and <code>create<span class=\'placeholder-nospace\'><em>Object</em></span>()</code>
 * methods (such as Model::createSpecies()) provided by certain <span
 * class=\'placeholder-nospace\'><em>Object</em></span> classes such as Model.  These
 * multiple mechanisms are provided by libSBML for flexibility and to
 * support different use-cases, but they also have different implications
 * for the overall model structure.
 *
 * In general, the recommended approach is to use the <code>create<span
 * class=\'placeholder-nospace\'><em>Object</em></span>()</code> methods.  These
 * methods both create an object @em and link it to the parent in one step.
 * Here is an example:@if clike
 * @code{.cpp}
// Create an SBMLDocument object in Level 3 Version 1 format:

SBMLDocument sbmlDoc = new SBMLDocument(3, 1);

// Create a Model object inside the SBMLDocument object and set
// its identifier.  The call returns a pointer to the Model object
// created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).

Model model = sbmlDoc->createModel();
model->setId(\'BestModelEver\');

// Create a Species object inside the Model and set its identifier.
// Similar to the lines above, this call returns a pointer to the Species
// object created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).

Species sp = model->createSpecies();
sp->setId(\'MySpecies\');
@endcode
 * @endif@if java
@code{.java}
// Create an SBMLDocument object in Level 3 Version 1 format:

SBMLDocument sbmlDoc = new SBMLDocument(3, 1);

// Create a Model object inside the SBMLDocument object and set
// its identifier.  The call returns a pointer to the Model object
// created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).  Note that
// the call to setId() returns a status code, and a real program
// should check this status code to make sure everything went okay.

Model model = sbmlDoc.createModel();
model.setId(&#34;BestModelEver&#34;);

// Create a Species object inside the Model and set its identifier.
// Similar to the lines above, this call returns a pointer to the Species
// object created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).  Note that, like
// with Model, the call to setId() returns a status code, and a real program
// should check this status code to make sure everything went okay.

Species sp = model.createSpecies();
sp.setId(&#34;BestSpeciesEver&#34;);
@endcode
 * @endif@if python
@code{.py}
# Create an SBMLDocument object in Level 3 Version 1 format.
# Make sure to check for possible failures.

try:
  sbmlDoc = SBMLDocument(3, 1)
except ValueError:
  print(\'Could not create SBMLDocument object\')
  sys.exit(1)

# Create a Model object inside the SBMLDocument object and set its
# identifier, checking the returned values.  The call to setId() returns a
# status code to indicate whether the assignment was successful.

model = sbmlDoc.createModel()
if model == None:
  # Do something to handle the error here.
  print(\'Unable to create Model object.\')
  sys.exit(1)

status = model.setId(\'BestModelEver\')
if status != LIBSBML_OPERATION_SUCCESS:
  # Do something to handle the error here.
  print(\'Unable to set identifier on the Model object\')
  sys.exit(1)

# Create a Species object inside the Model and set its identifier.
# Again, the setId() returns a status code to indicate whether the
# assignment was successful.

sp = model.createSpecies()
if sp == None:
  # Do something to handle the error here.
  print(\'Unable to create Species object.\')
  sys.exit(1)

status = sp.setId(\'BestSpeciesEver\')
if status != LIBSBML_OPERATION_SUCCESS:
  # Do something to handle the error here.
  print(\'Unable to set identifier on the Species object\')
  sys.exit(1)
@endcode
 * @endif@if csharp
@code
// Create an SBMLDocument object in Level 3 Version 1 format:

SBMLDocument sbmlDoc = new SBMLDocument(3, 1);

// Create a Model object inside the SBMLDocument object and set
// its identifier.  The call returns a pointer to the Model object
// created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).

Model model = sbmlDoc.createModel();
model.setId(\'BestModelEver\');

// Create a Species object inside the Model and set its identifier.
// Similar to the lines above, this call returns a pointer to the Species
// object created, and methods called on that object affect the attributes
// of the object attached to the model (as expected).

Species sp = model.createSpecies();
sp.setId(\'MySpecies\');
@endcode
 * @endif
 *
 * The <code>create<span
 * class=\'placeholder-nospace\'><em>Object</em></span>()</code> methods return a
 * pointer to the object created, but they also add the object to the
 * relevant list of object instances contained in the parent.  (These lists
 * become the <code>&lt;listOf<span
 * class=\'placeholder-nospace\'><em>Object</em></span>s&gt;</code> elements in the
 * finished XML rendition of SBML.)  In the example above,
 * Model::createSpecies() adds the created species directly to the
 * <code>&lt;listOfSpeciesgt;</code> list in the model.  Subsequently,
 * methods called on the species change the species in the model (which is
 * what is expected in most situations).
 *
 * @section model-checking Consistency and adherence to SBML specifications
 *
 * To make it easier for applications to do whatever they need,
 * libSBML version @htmlinclude libsbml-version.html
 * is relatively lax when it comes to enforcing correctness and
 * completeness of models @em during model construction and editing.
 * Essentially, libSBML @em will @em not in most cases check automatically
 * that a model\'s components have valid attribute values, or that the
 * overall model is consistent and free of errors---even obvious errors
 * such as duplication of identifiers.  This allows applications great
 * leeway in how they build their models, but it means that software
 * authors must take deliberate steps to ensure that the model will be, in
 * the end, valid SBML.  These steps include such things as keeping track
 * of the identifiers used in a model, manually performing updates in
 * certain situations where an entity is referenced in more than one place
 * (e.g., a species that is referenced by multiple SpeciesReference
 * objects), and so on.
 *
 * That said, libSBML does provide powerful features for deliberately
 * performing validation of SBML when an application decides it is time to
 * do so.  The interfaces to these facilities are on the SBMLDocument
 * class, in the form of SBMLDocument::checkInternalConsistency() and
 * SBMLDocument::checkConsistency().  Please refer to the documentation for
 * SBMLDocument for more information about this.
 *
 * While applications may play fast and loose and live like free spirits
 * during the construction and editing of SBML models, they should always
 * make sure to call SBMLDocument::checkInternalConsistency() and/or
 * SBMLDocument::checkConsistency() before writing out the final version of
 * an SBML model.
 *
 *
 * @section model-l3-attrib Model attributes introduced in SBML Level&nbsp;3
 *
 * As mentioned above, the Model class has a number of optional attributes
 * in SBML Level&nbsp;3.  These are \'substanceUnits\',
 * \'timeUnits\', \'volumeUnits\', \'areaUnits\', \'lengthUnits\', \'extentUnits\',
 * and \'conversionFactor.  The following provide more information about
 * them.
 *
 * @subsection model-l3-substanceunits The \'substanceUnits\' attribute
 *
 * The \'substanceUnits\' attribute is used to specify the unit of
 * measurement associated with substance quantities of Species objects that
 * do not specify units explicitly.  If a given Species object definition
 * does not specify its unit of substance quantity via the \'substanceUnits\'
 * attribute on the Species object instance, then that species inherits the
 * value of the Model \'substanceUnits\' attribute.  If the Model does not
 * define a value for this attribute, then there is no unit to inherit, and
 * all species that do not specify individual \'substanceUnits\' attribute
 * values then have <em>no</em> declared units for their quantities.  The
 * SBML Level&nbsp;3 specifications provide more details.
 *
 * Note that when the identifier of a species appears in a model\'s
 * mathematical expressions, the unit of measurement associated with that
 * identifier is <em>not solely determined</em> by setting \'substanceUnits\'
 * on Model or Species.  Please see the discussion about units given in
 * the documentation for the Species class.
 *
 *
 * @subsection model-l3-timeunits The \'timeUnits\' attribute
 *
 * The \'timeUnits\' attribute on SBML Level&nbsp;3\'s Model object is used to
 * specify the unit in which time is measured in the model.  This attribute
 * on Model is the <em>only</em> way to specify a unit for time in a model.
 * It is a global attribute; time is measured in the model everywhere in
 * the same way.  This is particularly relevant to Reaction and RateRule
 * objects in a model: all Reaction and RateRule objects in SBML define
 * per-time values, and the unit of time is given by the \'timeUnits\'
 * attribute on the Model object instance.  If the Model \'timeUnits\'
 * attribute has no value, it means that the unit of time is not defined
 * for the model\'s reactions and rate rules.  Leaving it unspecified in an
 * SBML model does not result in an invalid model in SBML Level&nbsp;3;
 * however, as a matter of best practice, we strongly recommend that all
 * models specify units of measurement for time.
 *
 *
 * @subsection model-l3-voletc The \'volumeUnits\', \'areaUnits\', and \'lengthUnits\' attributes
 *
 * The attributes \'volumeUnits\', \'areaUnits\' and \'lengthUnits\' together are
 * used to set the units of measurements for the sizes of Compartment
 * objects in an SBML Level&nbsp;3 model when those objects do not
 * otherwise specify units.  The three attributes correspond to the most
 * common cases of compartment dimensions: \'volumeUnits\' for compartments
 * having a \'spatialDimensions\' attribute value of @c \'3\', \'areaUnits\' for
 * compartments having a \'spatialDimensions\' attribute value of @c \'2\', and
 * \'lengthUnits\' for compartments having a \'spatialDimensions\' attribute
 * value of @c \'1\'.  The attributes are not applicable to compartments
 * whose \'spatialDimensions\' attribute values are @em not one of @c \'1\',
 * @c \'2\' or @c \'3\'.
 *
 * If a given Compartment object instance does not provide a value for its
 * \'units\' attribute, then the unit of measurement of that compartment\'s
 * size is inherited from the value specified by the Model \'volumeUnits\',
 * \'areaUnits\' or \'lengthUnits\' attribute, as appropriate based on the
 * Compartment object\'s \'spatialDimensions\' attribute value.  If the Model
 * object does not define the relevant attribute, then there are no units
 * to inherit, and all Compartment objects that do not set a value for
 * their \'units\' attribute then have <em>no</em> units associated with
 * their compartment sizes.
 *
 * The use of three separate attributes is a carry-over from SBML
 * Level&nbsp;2.  Note that it is entirely possible for a model to define a
 * value for two or more of the attributes \'volumeUnits\', \'areaUnits\' and
 * \'lengthUnits\' simultaneously, because SBML models may contain
 * compartments with different numbers of dimensions.
 *
 *
 * @subsection model-l3-extentunits The \'extentUnits\' attribute
 *
 * Reactions are processes that occur over time.  These processes involve
 * events of some sort, where a single ``reaction event\'\' is one in which
 * some set of entities (known as reactants, products and modifiers in
 * SBML) interact, once.  The <em>extent</em> of a reaction is a measure of
 * how many times the reaction has occurred, while the time derivative of
 * the extent gives the instantaneous rate at which the reaction is
 * occurring.  Thus, what is colloquially referred to as the \'rate of the
 * reaction\' is in fact equal to the rate of change of reaction extent.
 *
 * In SBML Level&nbsp;3, the combination of \'extentUnits\' and \'timeUnits\'
 * defines the units of kinetic laws in SBML and establishes how the
 * numerical value of each KineticLaw object\'s mathematical formula is
 * meant to be interpreted in a model.  The units of the kinetic laws are
 * taken to be \'extentUnits\' divided by \'timeUnits\'.
 *
 * Note that this embodies an important principle in SBML Level&nbsp;3
 * models: <em>all reactions in an SBML model must have the same units</em>
 * for the rate of change of extent.  In other words, the units of all
 * reaction rates in the model <em>must be the same</em>.  There is only
 * one global value for \'extentUnits\' and one global value for \'timeUnits\'.
 *
 *
 * @subsection model-l3-convfactor The \'conversionFactor\' attribute
 *
 * The attribute \'conversionFactor\' in SBML Level&nbsp;3\'s Model object
 * defines a global value inherited by all Species object instances that do
 * not define separate values for their \'conversionFactor\' attributes.  The
 * value of this attribute must refer to a Parameter object instance
 * defined in the model.  The Parameter object in question must be a
 * constant; ie it must have its \'constant\' attribute value set to
 * @c \'true\'.
 *
 * If a given Species object definition does not specify a conversion
 * factor via the \'conversionFactor\' attribute on Species, then the species
 * inherits the conversion factor specified by the Model \'conversionFactor\'
 * attribute.  If the Model does not define a value for this attribute,
 * then there is no conversion factor to inherit.  More information about
 * conversion factors is provided in the SBML Level&nbsp;3
 * specifications.
 */
"


%csmethodmodifiers Model::Model(unsigned int level, unsigned int version) "
/**
   * Creates a new Model using the given SBML @p level and @p version
   * values.
   *
   * @param level a long integer, the SBML Level to assign to this Model.
   *
   * @param version a long integer, the SBML Version to assign to this
   * Model.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers Model::Model(SBMLNamespaces* sbmlns) "
/**
   * Creates a new Model using the given SBMLNamespaces object
   * @p sbmlns.
   *
   *
 * 
 * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
 * information.  It is used to communicate the SBML Level, Version, and (in
 * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
 * common approach to using libSBML\'s SBMLNamespaces facilities is to create an
 * SBMLNamespaces object somewhere in a program once, then hand that object
 * as needed to object constructors that accept SBMLNamespaces as arguments.
 *
 *
   *
   * @param sbmlns an SBMLNamespaces object.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers Model::Model(const Model& orig) "
/**
   * Copy constructor; creates a (deep) copy of the given Model object.
   *
   * @param orig the object to copy.
   */ public
";


%csmethodmodifiers Model::accept(SBMLVisitor& v) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::clone() const "
/**
   * Creates and returns a deep copy of this Model object.
   *
   * @return the (deep) copy of this Model object.
   */ public new
";


%csmethodmodifiers Model::getElementBySId(const std::string& id) "
/**
   * Returns the first child element found that has the given @p id.
   *
   * This operation searches the model-wide <code>SId</code> identifier
   * type namespace
   *
   * @param id string representing the id of the object to find.
   *
   * @return pointer to the first element found with the given @p id, or
   * @c null if no such object is found.
   */ public new
";


%csmethodmodifiers Model::getElementByMetaId(const std::string& metaid) "
/**
   * Returns the first child element it can find with the given @p metaid.
   *
   * @param metaid string representing the meta-identifier of the object to
   * find.
   *
   * @return pointer to the first element found with the given @p metaid, or
   * null if no such object is found.
   */ public new
";


%csmethodmodifiers Model::getAllElements(ElementFilter* filter=NULL) "
/**
   * Returns a List of all child objects.
   *
   * This returns all child objects nested to an arbitrary depth.  If an
   * optional element filter is provided, only those objects for which the
   * @p filter\'s <code>filter()</code> method returns true will be added to 
   * the list.
   *
   * @param filter a pointer to an ElementFilter, which causes the function 
   * to return only elements that match a particular set of constraints.  
   * If null (the default), the function will return all child objects.
   *
   * @return a List of pointers to all children objects.
   */ public new
";


%csmethodmodifiers Model::getId() const "
/**
   * Returns the value of the \'id\' attribute of this Model.
   *
   * @note Because of the inconsistent behavior of this function with 
   * respect to assignments and rules, it is now recommended to
   * use the getIdAttribute() function instead.
   *
   *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
   *
   * @return the id of this Model.
   *
   * @see getIdAttribute()
   * @see setIdAttribute(string sid)
   * @see isSetIdAttribute()
   * @see unsetIdAttribute()
   */ public new
";


%csmethodmodifiers Model::getName() const "
/**
   * Returns the value of the \'name\' attribute of this Model object.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 *
 * @return the name of this SBML object, or the empty string if not set or unsettable.
 *
 * @see getIdAttribute()
 * @see isSetName()
 * @see setName(string sid)
 * @see unsetName()
 * 
 *
   */ public new
";


%csmethodmodifiers Model::getSubstanceUnits() const "
/**
   * Returns the value of the \'substanceUnits\' attribute of this Model.
   *
   * @return the substanceUnits of this Model.
   *
   * @note The \'substanceUnits\' attribute is available in
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::getTimeUnits() const "
/**
   * Returns the value of the \'timeUnits\' attribute of this Model.
   *
   * @return the timeUnits of this Model.
   *
   * @note The \'timeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::getVolumeUnits() const "
/**
   * Returns the value of the \'volumeUnits\' attribute of this Model.
   *
   * @return the volumeUnits of this Model.
   *
   * @note The \'volumeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::getAreaUnits() const "
/**
   * Returns the value of the \'areaUnits\' attribute of this Model.
   *
   * @return the areaUnits of this Model.
   *
   * @note The \'areaUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::getLengthUnits() const "
/**
   * Returns the value of the \'lengthUnits\' attribute of this Model.
   *
   * @return the lengthUnits of this Model.
   *
   * @note The \'lengthUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::getExtentUnits() const "
/**
   * Returns the value of the \'extentUnits\' attribute of this Model.
   *
   * @return the extentUnits of this Model.
   *
   * @note The \'extentUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::getConversionFactor() const "
/**
   * Returns the value of the \'conversionFactor\' attribute of this Model.
   *
   * @return the conversionFactor of this Model.
   *
   * @note The \'conversionFactor\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::isSetId() const "
/**
   * Predicate returning @c true if this
   * Model\'s \'id\' attribute is set.
   *
   *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 *
 * @return @c true if the \'id\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @note Because of the inconsistent behavior of this function with
 * respect to assignments and rules, it is recommended that callers
 * use isSetIdAttribute() instead.
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see unsetIdAttribute()
 * @see isSetIdAttribute()
 *
 *
   */ public new
";


%csmethodmodifiers Model::isSetName() const "
/**
   * Predicate returning @c true if this
   * Model\'s \'name\' attribute is set.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 * @return @c true if the \'name\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @see getName()
 * @see setName(string sid)
 * @see unsetName()
 *
 *
   */ public new
";


%csmethodmodifiers Model::isSetSubstanceUnits() const "
/**
   * Predicate returning @c true if this
   * Model\'s \'substanceUnits\' attribute is set.
   *
   * @return @c true if the \'substanceUnits\' attribute of this Model is
   * set, @c false otherwise.
   *
   * @note The \'substanceUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::isSetTimeUnits() const "
/**
   * Predicate returning @c true if this
   * Model\'s \'timeUnits\' attribute is set.
   *
   * @return @c true if the \'timeUnits\' attribute of this Model is
   * set, @c false otherwise.
   *
   * @note The \'substanceUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::isSetVolumeUnits() const "
/**
   * Predicate returning @c true if this
   * Model\'s \'volumeUnits\' attribute is set.
   *
   * @return @c true if the \'volumeUnits\' attribute of this Model is
   * set, @c false otherwise.
   *
   * @note The \'volumeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::isSetAreaUnits() const "
/**
   * Predicate returning @c true if this
   * Model\'s \'areaUnits\' attribute is set.
   *
   * @return @c true if the \'areaUnits\' attribute of this Model is
   * set, @c false otherwise.
   *
   * @note The \'areaUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::isSetLengthUnits() const "
/**
   * Predicate returning @c true if this
   * Model\'s \'lengthUnits\' attribute is set.
   *
   * @return @c true if the \'lengthUnits\' attribute of this Model is
   * set, @c false otherwise.
   *
   * @note The \'lengthUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::isSetExtentUnits() const "
/**
   * Predicate returning @c true if this
   * Model\'s \'extentUnits\' attribute is set.
   *
   * @return @c true if the \'extentUnits\' attribute of this Model is
   * set, @c false otherwise.
   *
   * @note The \'extentUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::isSetConversionFactor() const "
/**
   * Predicate returning @c true if this
   * Model\'s \'conversionFactor\' attribute is set.
   *
   * @return @c true if the \'conversionFactor\' attribute of this Model is
   * set, @c false otherwise.
   *
   * @note The \'conversionFactor\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this Model.
   *
   *
 * 
 * The string @p sid is copied.
 *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 * 
 * @param sid the string to use as the identifier of this object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see isSetIdAttribute()
 * @see unsetIdAttribute()
 * 
 *
   */ public new
";


%csmethodmodifiers Model::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this Model.
   *
   *
 * 
 *
 * The string in @p name is copied.
 *
 * @param name the new name for the SBML object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 *
 *
   */ public new
";


%csmethodmodifiers Model::setSubstanceUnits(const std::string& units) "
/**
   * Sets the value of the \'substanceUnits\' attribute of this Model.
   *
   * The string in @p units is copied.
   *
   * @param units the new substanceUnits for the Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   *
   * @note The \'substanceUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::setTimeUnits(const std::string& units) "
/**
   * Sets the value of the \'timeUnits\' attribute of this Model.
   *
   * The string in @p units is copied.
   *
   * @param units the new timeUnits for the Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   *
   * @note The \'timeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::setVolumeUnits(const std::string& units) "
/**
   * Sets the value of the \'volumeUnits\' attribute of this Model.
   *
   * The string in @p units is copied.
   *
   * @param units the new volumeUnits for the Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   *
   * @note The \'volumeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::setAreaUnits(const std::string& units) "
/**
   * Sets the value of the \'areaUnits\' attribute of this Model.
   *
   * The string in @p units is copied.
   *
   * @param units the new areaUnits for the Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   *
   * @note The \'areaUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::setLengthUnits(const std::string& units) "
/**
   * Sets the value of the \'lengthUnits\' attribute of this Model.
   *
   * The string in @p units is copied.
   *
   * @param units the new lengthUnits for the Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   *
   * @note The \'lengthUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::setExtentUnits(const std::string& units) "
/**
   * Sets the value of the \'extentUnits\' attribute of this Model.
   *
   * The string in @p units is copied.
   *
   * @param units the new extentUnits for the Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   *
   * @note The \'extentUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::setConversionFactor(const std::string& units) "
/**
   * Sets the value of the \'conversionFactor\' attribute of this Model.
   *
   * The string in @p units is copied.
   *
   * @param units the new conversionFactor for the Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   *
   * @note The \'conversionFactor\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::unsetId "
/**
   * Unsets the value of the \'id\' attribute of this Model.
   *
   *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 * 
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see isSetIdAttribute()
 * @see unsetIdAttribute()
 *
 *
   */ public new
";


%csmethodmodifiers Model::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this Model.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
 *
 * @see getName()
 * @see setName(string sid)
 * @see isSetName()
 *
 *
   */ public new
";


%csmethodmodifiers Model::unsetSubstanceUnits "
/**
   * Unsets the value of the \'substanceUnits\' attribute of this Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @note The \'substanceUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::unsetTimeUnits "
/**
   * Unsets the value of the \'timeUnits\' attribute of this Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @note The \'timeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::unsetVolumeUnits "
/**
   * Unsets the value of the \'volumeUnits\' attribute of this Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @note The \'volumeUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::unsetAreaUnits "
/**
   * Unsets the value of the \'areaUnits\' attribute of this Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @note The \'areaUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::unsetLengthUnits "
/**
   * Unsets the value of the \'lengthUnits\' attribute of this Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @note The \'lengthUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::unsetExtentUnits "
/**
   * Unsets the value of the \'extentUnits\' attribute of this Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @note The \'extentUnits\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::unsetConversionFactor "
/**
   * Unsets the value of the \'conversionFactor\' attribute of this Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @note The \'conversionFactor\' attribute is available in 
   * SBML Level&nbsp;3 but is not present on Model in lower Levels of SBML.
   */ public
";


%csmethodmodifiers Model::addFunctionDefinition(const FunctionDefinition* fd) "
/**
   * Adds a copy of the given FunctionDefinition object to this Model.
   *
   * @param fd the FunctionDefinition to add.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 * 
   *
   * @see createFunctionDefinition()
   */ public
";


%csmethodmodifiers Model::addUnitDefinition(const UnitDefinition* ud) "
/**
   * Adds a copy of the given UnitDefinition object to this Model.
   *
   * @param ud the UnitDefinition object to add.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 * 
   *
   * @see createUnitDefinition()
   */ public
";


%csmethodmodifiers Model::addCompartmentType(const CompartmentType* ct) "
/**
   * Adds a copy of the given CompartmentType object to this Model.
   *
   * @param ct the CompartmentType object to add.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 * 
   *
   * @note The CompartmentType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   *
   * @see createCompartmentType()
   */ public
";


%csmethodmodifiers Model::addSpeciesType(const SpeciesType* st) "
/**
   * Adds a copy of the given SpeciesType object to this Model.
   *
   * @param st the SpeciesType object to add.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 * 
   *
   * @note The SpeciesType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   *
   * @see createSpeciesType()
   */ public
";


%csmethodmodifiers Model::addCompartment(const Compartment* c) "
/**
   * Adds a copy of the given Compartment object to this Model.
   *
   * @param c the Compartment object to add.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 * 
   *
   * @see createCompartment()
   */ public
";


%csmethodmodifiers Model::addSpecies(const Species* s) "
/**
   * Adds a copy of the given Species object to this Model.
   *
   * @param s the Species object to add.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 * 
   *
   * @see createSpecies()
   */ public
";


%csmethodmodifiers Model::addParameter(const Parameter* p) "
/**
   * Adds a copy of the given Parameter object to this Model.
   *
   * @param p the Parameter object to add.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 * 
   *
   * @see createParameter()
   */ public
";


%csmethodmodifiers Model::addInitialAssignment(const InitialAssignment* ia) "
/**
   * Adds a copy of the given InitialAssignment object to this Model.
   *
   * @param ia the InitialAssignment object to add.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 * 
   *
   * @see createInitialAssignment()
   */ public
";


%csmethodmodifiers Model::addRule(const Rule* r) "
/**
   * Adds a copy of the given Rule object to this Model.
   *
   * @param r the Rule object to add.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 * 
   *
   * @see createAlgebraicRule()
   * @see createAssignmentRule()
   * @see createRateRule()
   */ public
";


%csmethodmodifiers Model::addConstraint(const Constraint* c) "
/**
   * Adds a copy of the given Constraint object to this Model.
   *
   * @param c the Constraint object to add.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 * 
   *
   * @see createConstraint()
   */ public
";


%csmethodmodifiers Model::addReaction(const Reaction* r) "
/**
   * Adds a copy of the given Reaction object to this Model.
   *
   * @param r the Reaction object to add.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 * 
   *
   * @see createReaction()
   */ public
";


%csmethodmodifiers Model::addEvent(const Event* e) "
/**
   * Adds a copy of the given Event object to this Model.
   *
   * @param e the Event object to add.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_DUPLICATE_OBJECT_ID LIBSBML_DUPLICATE_OBJECT_ID@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 * 
   *
   * @see createEvent()
   */ public
";


%csmethodmodifiers Model::createFunctionDefinition "
/**
   * Creates a new FunctionDefinition inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the FunctionDefinition object created.
   *
   * @see addFunctionDefinition(FunctionDefinition fd)
   */ public
";


%csmethodmodifiers Model::createUnitDefinition "
/**
   * Creates a new UnitDefinition inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the UnitDefinition object created.
   *
   * @see addUnitDefinition(UnitDefinition ud)
   */ public
";


%csmethodmodifiers Model::createUnit "
/**
   * Creates a new Unit object within the last UnitDefinition object
   * created in this model and returns a pointer to it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * The mechanism by which the UnitDefinition was created is not
   * significant.  If a UnitDefinition object does not exist in this model,
   * a new Unit is @em not created and @c null is returned instead.
   *
   * @return the Unit object created.
   *
   * @see addUnitDefinition(UnitDefinition ud)
   */ public
";


%csmethodmodifiers Model::createCompartmentType "
/**
   * Creates a new CompartmentType inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the CompartmentType object created.
   *
   * @note The CompartmentType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   *
   * @see addCompartmentType(CompartmentType ct)
   */ public
";


%csmethodmodifiers Model::createSpeciesType "
/**
   * Creates a new SpeciesType inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the SpeciesType object created.
   *
   * @note The SpeciesType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   *
   * @see addSpeciesType(SpeciesType st)
   */ public
";


%csmethodmodifiers Model::createCompartment "
/**
   * Creates a new Compartment inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the Compartment object created.
   *
   * @see addCompartment(Compartment c)
   */ public
";


%csmethodmodifiers Model::createSpecies "
/**
   * Creates a new Species inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the Species object created.
   *
   * @see addSpecies(Species s)
   */ public
";


%csmethodmodifiers Model::createParameter "
/**
   * Creates a new Parameter inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the Parameter object created.
   *
   * @see addParameter(Parameter p)
   */ public
";


%csmethodmodifiers Model::createInitialAssignment "
/**
   * Creates a new InitialAssignment inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the InitialAssignment object created.
   *
   * @see addInitialAssignment(InitialAssignment ia)
   */ public
";


%csmethodmodifiers Model::createAlgebraicRule "
/**
   * Creates a new AlgebraicRule inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the AlgebraicRule object created.
   *
   * @see addRule(Rule r)
   */ public
";


%csmethodmodifiers Model::createAssignmentRule "
/**
   * Creates a new AssignmentRule inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the AssignmentRule object created.
   *
   * @see addRule(Rule r)
   */ public
";


%csmethodmodifiers Model::createRateRule "
/**
   * Creates a new RateRule inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the RateRule object created.
   *
   * @see addRule(Rule r)
   */ public
";


%csmethodmodifiers Model::createConstraint "
/**
   * Creates a new Constraint inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the Constraint object created.
   *
   * @see addConstraint(Constraint c)
   */ public
";


%csmethodmodifiers Model::createReaction "
/**
   * Creates a new Reaction inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the Reaction object created.
   *
   * @see addReaction(Reaction r)
   */ public
";


%csmethodmodifiers Model::createReactant "
/**
   * Creates a new SpeciesReference object for a reactant inside the last
   * Reaction object in this Model, and returns a pointer to it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   *
 * 
 * The mechanism by which the last Reaction object was created and added
 * to this Model is not significant.  It could have been created in a
 * variety of ways, for example using createReaction().  If a Reaction
 * does not exist for this model, a new SpeciesReference is @em not
 * created and @c null is returned instead.
 *
 *
   *
   * @return the SpeciesReference object created.  If a Reaction does not
   * exist for this model, a new SpeciesReference is @em not created and
   * @c null is returned.
   */ public
";


%csmethodmodifiers Model::createProduct "
/**
   * Creates a new SpeciesReference object for a product inside the last
   * Reaction object in this Model, and returns a pointer to it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   *
 * 
 * The mechanism by which the last Reaction object was created and added
 * to this Model is not significant.  It could have been created in a
 * variety of ways, for example using createReaction().  If a Reaction
 * does not exist for this model, a new SpeciesReference is @em not
 * created and @c null is returned instead.
 *
 *
   *
   * @return the SpeciesReference object created. If a Reaction does not
   * exist for this model, a new SpeciesReference is @em not created and
   * @c null is returned.
   */ public
";


%csmethodmodifiers Model::createModifier "
/**
   * Creates a new ModifierSpeciesReference object for a modifier species
   * inside the last Reaction object in this Model, and returns a pointer
   * to it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   *
 * 
 * The mechanism by which the last Reaction object was created and added
 * to this Model is not significant.  It could have been created in a
 * variety of ways, for example using createReaction().  If a Reaction
 * does not exist for this model, a new SpeciesReference is @em not
 * created and @c null is returned instead.
 *
 *
   *
   * @return the SpeciesReference object created.  If a Reaction does not
   * exist for this model, a new SpeciesReference is @em not created and
   * @c null is returned.
   */ public
";


%csmethodmodifiers Model::createKineticLaw "
/**
   * Creates a new KineticLaw inside the last Reaction object created in
   * this Model, and returns a pointer to it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   *
 * 
 * The mechanism by which the last Reaction object was created and added
 * to this Model is not significant.  It could have been created in a
 * variety of ways, for example using createReaction().  If a Reaction
 * does not exist for this model, a new SpeciesReference is @em not
 * created and @c null is returned instead.
 *
 *
   *
   * @return the KineticLaw object created.  If a Reaction does not exist for
   * this model, or a Reaction does exist but already has a KineticLaw, a new
   * KineticLaw is @em not created and @c null is returned.
   */ public
";


%csmethodmodifiers Model::createKineticLawParameter "
/**
   * Creates a new local Parameter inside the KineticLaw object of the last
   * Reaction created inside this Model, and returns a pointer to it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   *
 * 
 * The last KineticLaw object in this Model could have been created in a
 * variety of ways.  For example, it could have been added using
 * createKineticLaw(), or it could be the result of using
 * Reaction::createKineticLaw() on the Reaction object created by a
 * createReaction().  If a Reaction does not exist for this model, or the
 * last Reaction does not contain a KineticLaw object, a new Parameter is
 * @em not created and @c null is returned instead.
 *
 *
   *
   * @return the Parameter object created.  If a Reaction does not exist for
   * this model, or a KineticLaw for the Reaction does not exist, a new
   * Parameter is @em not created and @c null is returned.
   */ public
";


%csmethodmodifiers Model::createKineticLawLocalParameter "
/**
   * Creates a new LocalParameter inside the KineticLaw object of the last
   * Reaction created inside this Model, and returns a pointer to it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   *
 * 
 * The last KineticLaw object in this Model could have been created in a
 * variety of ways.  For example, it could have been added using
 * createKineticLaw(), or it could be the result of using
 * Reaction::createKineticLaw() on the Reaction object created by a
 * createReaction().  If a Reaction does not exist for this model, or the
 * last Reaction does not contain a KineticLaw object, a new Parameter is
 * @em not created and @c null is returned instead.
 *
 *
   *
   * @return the Parameter object created.  If a Reaction does not exist for
   * this model, or a KineticLaw for the Reaction does not exist, a new
   * Parameter is @em not created and @c null is returned.
   */ public
";


%csmethodmodifiers Model::createEvent "
/**
   * Creates a new Event inside this Model and returns it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   * @return the Event object created.
   */ public
";


%csmethodmodifiers Model::createEventAssignment "
/**
   * Creates a new EventAssignment inside the last Event object created in
   * this Model, and returns a pointer to it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   *
 * 
 * The mechanism by which the last Event object in this model was created
 * is not significant.  It could have been created in a variety of ways,
 * for example by using createEvent().  If no Event object exists in this
 * Model object, a new EventAssignment is @em not created and @c null is
 * returned instead.
   *
   * @return the EventAssignment object created.
   */ public
";


%csmethodmodifiers Model::createTrigger "
/**
   * Creates a new Trigger inside the last Event object created in
   * this Model, and returns a pointer to it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   *
 * 
 * The mechanism by which the last Event object in this model was created
 * is not significant.  It could have been created in a variety of ways,
 * for example by using createEvent().  If no Event object exists in this
 * Model object, a new EventAssignment is @em not created and @c null is
 * returned instead.
   *
   * @return the Trigger object created.
   */ public
";


%csmethodmodifiers Model::createDelay "
/**
   * Creates a new Delay inside the last Event object created in
   * this Model, and returns a pointer to it.
   *
   * The SBML Level and Version of the enclosing Model object, as well as
   * any SBML package namespaces, are used to initialize this
   * object\'s corresponding attributes.
   *
   *
 * 
 * The mechanism by which the last Event object in this model was created
 * is not significant.  It could have been created in a variety of ways,
 * for example by using createEvent().  If no Event object exists in this
 * Model object, a new EventAssignment is @em not created and @c null is
 * returned instead.
   *
   * @return the Delay object created.
   */ public
";


%csmethodmodifiers Model::setAnnotation(const XMLNode* annotation) "
/**
   * Sets the value of the \'annotation\' subelement of this SBML object to a
   * copy of @p annotation.
   *
   * Any existing content of the \'annotation\' subelement is discarded.
   * Unless you have taken steps to first copy and reconstitute any
   * existing annotations into the @p annotation that is about to be
   * assigned, it is likely that performing such wholesale replacement is
   * unfriendly towards other software applications whose annotations are
   * discarded.  An alternative may be to use appendAnnotation().
   *
   * @param annotation an XML structure that is to be used as the content
   * of the \'annotation\' subelement of this object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif This particular
 * function only does one thing irrespective of user input or 
 * object state, and thus will only return a single value:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   *
   * @see appendAnnotation(XMLNode annotation)
   */ public new
";


%csmethodmodifiers Model::setAnnotation(const std::string& annotation) "
/**
   * Sets the value of the \'annotation\' subelement of this SBML object to a
   * copy of @p annotation.
   *
   * Any existing content of the \'annotation\' subelement is discarded.
   * Unless you have taken steps to first copy and reconstitute any
   * existing annotations into the @p annotation that is about to be
   * assigned, it is likely that performing such wholesale replacement is
   * unfriendly towards other software applications whose annotations are
   * discarded.  An alternative may be to use appendAnnotation().
   *
   * @param annotation an XML string that is to be used as the content
   * of the \'annotation\' subelement of this object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @see appendAnnotation(string annotation)
   */ public new
";


%csmethodmodifiers Model::appendAnnotation(const XMLNode* annotation) "
/**
   * Appends annotation content to any existing content in the \'annotation\'
   * subelement of this object.
   *
   * The content in @p annotation is copied.  Unlike setAnnotation(), this
   * method allows other annotations to be preserved when an application
   * adds its own data.
   *
   * @param annotation an XML structure that is to be copied and appended
   * to the content of the \'annotation\' subelement of this object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @see setAnnotation(XMLNode annotation)
   */ public new
";


%csmethodmodifiers Model::appendAnnotation(const std::string& annotation) "
/**
   * Appends annotation content to any existing content in the \'annotation\'
   * subelement of this object.
   *
   * The content in @p annotation is copied.  Unlike setAnnotation(), this 
   * method allows other annotations to be preserved when an application
   * adds its own data.
   *
   * @param annotation an XML string that is to be copied and appended
   * to the content of the \'annotation\' subelement of this object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @see setAnnotation(string annotation)
   */ public new
";


%csmethodmodifiers Model::getListOfFunctionDefinitions() const "
/**
   * Get the ListOfFunctionDefinitions object in this Model.
   *
   * @return the list of FunctionDefinitions for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfFunctionDefinitions "
/**
   * Get the ListOfFunctionDefinitions object in this Model.
   *
   * @return the list of FunctionDefinitions for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfUnitDefinitions() const "
/**
   * Get the ListOfUnitDefinitions object in this Model.
   *
   * @return the list of UnitDefinitions for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfUnitDefinitions "
/**
   * Get the ListOfUnitDefinitions object in this Model.
   *
   * @return the list of UnitDefinitions for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfCompartmentTypes() const "
/**
   * Get the ListOfCompartmentTypes object in this Model.
   *
   * @return the list of CompartmentTypes for this Model.
   *
   * @note The CompartmentType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getListOfCompartmentTypes "
/**
   * Get the ListOfCompartmentTypes object in this Model.
   *
   * @return the list of CompartmentTypes for this Model.
   *
   * @note The CompartmentType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getListOfSpeciesTypes() const "
/**
   * Get the ListOfSpeciesTypes object in this Model.
   *
   * @return the list of SpeciesTypes for this Model.
   *
   * @note The SpeciesType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getListOfSpeciesTypes "
/**
   * Get the ListOfSpeciesTypes object in this Model.
   *
   * @return the list of SpeciesTypes for this Model.
   *
   * @note The SpeciesType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getListOfCompartments() const "
/**
   * Get the ListOfCompartments object in this Model.
   *
   * @return the list of Compartments for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfCompartments "
/**
   * Get the ListOfCompartments object in this Model.
   *
   * @return the list of Compartments for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfSpecies() const "
/**
   * Get the ListOfSpecies object in this Model.
   *
   * @return the list of Species for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfSpecies "
/**
   * Get the ListOfSpecies object in this Model.
   *
   * @return the list of Species for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfParameters() const "
/**
   * Get the ListOfParameters object in this Model.
   *
   * @return the list of Parameters for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfParameters "
/**
   * Get the ListOfParameters object in this Model.
   *
   * @return the list of Parameters for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfInitialAssignments() const "
/**
   * Get the ListOfInitialAssignments object in this Model.
   *
   * @return the list of InitialAssignments for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfInitialAssignments "
/**
   * Get the ListOfInitialAssignments object in this Model.
   *
   * @return the list of InitialAssignment for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfRules() const "
/**
   * Get the ListOfRules object in this Model.
   *
   * @return the list of Rules for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfRules "
/**
   * Get the ListOfRules object in this Model.
   *
   * @return the list of Rules for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfConstraints() const "
/**
   * Get the ListOfConstraints object in this Model.
   *
   * @return the list of Constraints for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfConstraints "
/**
   * Get the ListOfConstraints object in this Model.
   *
   * @return the list of Constraints for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfReactions() const "
/**
   * Get the ListOfReactions object in this Model.
   *
   * @return the list of Reactions for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfReactions "
/**
   * Get the ListOfReactions object in this Model.
   *
   * @return the list of Reactions for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfEvents() const "
/**
   * Get the ListOfEvents object in this Model.
   *
   * @return the list of Events for this Model.
   */ public
";


%csmethodmodifiers Model::getListOfEvents "
/**
   * Get the ListOfEvents object in this Model.
   *
   * @return the list of Events for this Model.
   */ public
";


%csmethodmodifiers Model::getFunctionDefinition "
/**
   * Get the nth FunctionDefinitions object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth FunctionDefinition of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getFunctionDefinition(unsigned int n) "
/**
   * Get the nth FunctionDefinitions object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth FunctionDefinition of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getFunctionDefinition(const std::string& sid) const "
/**
   * Get a FunctionDefinition object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the FunctionDefinition in this Model with the identifier
   * @p sid or @c null if no such FunctionDefinition exists.
   */ public
";


%csmethodmodifiers Model::getFunctionDefinition(const std::string& sid) "
/**
   * Get a FunctionDefinition object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the FunctionDefinition in this Model with the identifier
   * @p sid or @c null if no such FunctionDefinition exists.
   */ public
";


%csmethodmodifiers Model::getUnitDefinition "
/**
   * Get the nth UnitDefinition object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth UnitDefinition of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getUnitDefinition(unsigned int n) "
/**
   * Get the nth UnitDefinition object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth UnitDefinition of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getUnitDefinition(const std::string& sid) const "
/**
   * Get a UnitDefinition based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the UnitDefinition in this Model with the identifier @p sid or
   * @c null if no such UnitDefinition exists.
   */ public
";


%csmethodmodifiers Model::getUnitDefinition(const std::string& sid) "
/**
   * Get a UnitDefinition based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the UnitDefinition in this Model with the identifier @p sid or
   * @c null if no such UnitDefinition exists.
   */ public
";


%csmethodmodifiers Model::getCompartmentType "
/**
   * Get the nth CompartmentType object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth CompartmentType of this Model.
   * If the index @p n is invalid, @c null is returned.
   *
   * @note The CompartmentType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getCompartmentType(unsigned int n) "
/**
   * Get the nth CompartmentType object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth CompartmentType of this Model.
   * If the index @p n is invalid, @c null is returned.
   *
   * @note The CompartmentType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getCompartmentType(const std::string& sid) const "
/**
   * Get a CompartmentType object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the CompartmentType in this Model with the identifier @p sid
   * or @c null if no such CompartmentType exists.
   *
   * @note The CompartmentType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getCompartmentType(const std::string& sid) "
/**
   * Get a CompartmentType object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the CompartmentType in this Model with the identifier @p sid
   * or @c null if no such CompartmentType exists.
   *
   * @note The CompartmentType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getSpeciesType "
/**
   * Get the nth SpeciesType object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth SpeciesType of this Model.
   * If the index @p n is invalid, @c null is returned.
   *
   * @note The SpeciesType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getSpeciesType(unsigned int n) "
/**
   * Get the nth SpeciesType object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth SpeciesType of this Model.
   * If the index @p n is invalid, @c null is returned.
   *
   * @note The SpeciesType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getSpeciesType(const std::string& sid) const "
/**
   * Get a SpeciesType object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the SpeciesType in this Model with the identifier @p sid or
   * @c null if no such SpeciesType exists.
   *
   * @note The SpeciesType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getSpeciesType(const std::string& sid) "
/**
   * Get a SpeciesType object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the SpeciesType in this Model with the identifier @p sid or
   * @c null if no such SpeciesType exists.
   *
   * @note The SpeciesType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getCompartment "
/**
   * Get the nth Compartment object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Compartment of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getCompartment(unsigned int n) "
/**
   * Get the nth Compartment object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Compartment of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getCompartment(const std::string& sid) const "
/**
   * Get a Compartment object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the Compartment in this Model with the identifier @p sid or
   * @c null if no such Compartment exists.
   */ public
";


%csmethodmodifiers Model::getCompartment(const std::string& sid) "
/**
   * Get a Compartment object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the Compartment in this Model with the identifier @p sid or
   * @c null if no such Compartment exists.
   */ public
";


%csmethodmodifiers Model::getSpecies "
/**
   * Get the nth Species object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Species of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getSpecies(unsigned int n) "
/**
   * Get the nth Species object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Species of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getSpecies(const std::string& sid) const "
/**
   * Get a Species object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the Species in this Model with the identifier @p sid or @c null
   * if no such Species exists.
   */ public
";


%csmethodmodifiers Model::getSpecies(const std::string& sid) "
/**
   * Get a Species object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the Species in this Model with the identifier @p sid or @c null
   * if no such Species exists.
   */ public
";


%csmethodmodifiers Model::getParameter "
/**
   * Get the nth Parameter object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Parameter of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getParameter(unsigned int n) "
/**
   * Get the nth Parameter object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Parameter of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getParameter(const std::string& sid) const "
/**
   * Get a Parameter object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the Parameter in this Model with the identifier @p sid or @c null
   * if no such Parameter exists.
   */ public
";


%csmethodmodifiers Model::getParameter(const std::string& sid) "
/**
   * Get a Parameter object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the Parameter in this Model with the identifier @p sid or @c null
   * if no such Parameter exists.
   */ public
";


%csmethodmodifiers Model::getInitialAssignment "
/**
   * Get the nth InitialAssignment object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth InitialAssignment of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getInitialAssignment(unsigned int n) "
/**
   * Get the nth InitialAssignment object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth InitialAssignment of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getInitialAssignment(const std::string& symbol) const "
/**
   * Get an InitialAssignment object based on the symbol to which it
   * assigns a value.
   *
   * @param symbol the symbol to search for.
   *
   * @return the InitialAssignment in this Model with the given \'symbol\'
   * attribute value or @c null if no such InitialAssignment exists.
   */ public
";


%csmethodmodifiers Model::getInitialAssignmentBySymbol(const std::string& symbol) const "
/**
   * Get an InitialAssignment object based on the symbol to which it
   * assigns a value.
   *
   * @param symbol the symbol to search for.
   *
   * @return the InitialAssignment in this Model with the given \'symbol\'
   * attribute value or @c null if no such InitialAssignment exists.
   */ public
";


%csmethodmodifiers Model::getInitialAssignment(const std::string& symbol) "
/**
   * Get an InitialAssignment object based on the symbol to which it
   * assigns a value.
   *
   * @param symbol the symbol to search for.
   *
   * @return the InitialAssignment in this Model with the given \'symbol\'
   * attribute value or @c null if no such InitialAssignment exists.
   */ public
";


%csmethodmodifiers Model::getInitialAssignmentBySymbol(const std::string& symbol) "
/**
   * Get an InitialAssignment object based on the symbol to which it
   * assigns a value.
   *
   * @param symbol the symbol to search for.
   *
   * @return the InitialAssignment in this Model with the given \'symbol\'
   * attribute value or @c null if no such InitialAssignment exists.
   */ public
";


%csmethodmodifiers Model::getRule "
/**
   * Get the nth Rule object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Rule of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getRule(unsigned int n) "
/**
   * Get the nth Rule object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Rule of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getRule(const std::string& variable) const "
/**
   * Get a Rule object based on the variable to which it assigns a value.
   *
   * @param variable the variable to search for.
   *
   * @return the Rule in this Model with the given \'variable\' attribute
   * value or @c null if no such Rule exists.
   */ public
";


%csmethodmodifiers Model::getRule(const std::string& variable) "
/**
   * Get a Rule object based on the variable to which it assigns a value.
   *
   * @param variable the variable to search for.
   *
   * @return the Rule in this Model with the given \'variable\' attribute
   * value or @c null if no such Rule exists.
   */ public
";


%csmethodmodifiers Model::getRuleByVariable(const std::string& variable) const "
/**
   * Get a Rule object based on the variable to which it assigns a value.
   *
   * @param variable the variable to search for.
   *
   * @return the Rule in this Model with the given \'variable\' attribute
   * value or @c null if no such Rule exists.
   */ public
";


%csmethodmodifiers Model::getRuleByVariable(const std::string& variable) "
/**
   * Get a Rule object based on the variable to which it assigns a value.
   *
   * @param variable the variable to search for.
   *
   * @return the Rule in this Model with the given \'variable\' attribute
   * value or @c null if no such Rule exists.
   */ public
";


%csmethodmodifiers Model::getAssignmentRule(const std::string& variable) const "
/**
   * Get a Rule object based on the variable to which it assigns a value.
   *
   * @param variable the variable to search for.
   *
   * @return the Rule in this Model with the given \'variable\' attribute
   * value or @c null if no such Rule exists.
   */ public
";


%csmethodmodifiers Model::getAssignmentRule(const std::string& variable) "
/**
   * Get a Rule object based on the variable to which it assigns a value.
   *
   * @param variable the variable to search for.
   *
   * @return the Rule in this Model with the given \'variable\' attribute
   * value or @c null if no such Rule exists.
   */ public
";


%csmethodmodifiers Model::getRateRule(const std::string& variable) const "
/**
   * Get a Rule object based on the variable to which it assigns a value.
   *
   * @param variable the symbol to search for.
   *
   * @return the Rule in this Model with the given \'variable\' attribute
   * value or @c null if no such Rule exists.
   */ public
";


%csmethodmodifiers Model::getRateRule(const std::string& variable) "
/**
   * Get a Rule object based on the variable to which it assigns a value.
   *
   * @param variable the variable to search for.
   *
   * @return the Rule in this Model with the given \'variable\' attribute
   * value or @c null if no such Rule exists.
   */ public
";


%csmethodmodifiers Model::getAssignmentRuleByVariable(const std::string& variable) const "
/**
   * Get a Rule object based on the variable to which it assigns a value.
   *
   * @param variable the variable to search for.
   *
   * @return the Rule in this Model with the given \'variable\' attribute
   * value or @c null if no such Rule exists.
   */ public
";


%csmethodmodifiers Model::getAssignmentRuleByVariable(const std::string& variable) "
/**
   * Get a Rule object based on the variable to which it assigns a value.
   *
   * @param variable the variable to search for.
   *
   * @return the Rule in this Model with the given \'variable\' attribute
   * value or @c null if no such Rule exists.
   */ public
";


%csmethodmodifiers Model::getRateRuleByVariable(const std::string& variable) const "
/**
   * Get a Rule object based on the variable to which it assigns a value.
   *
   * @param variable the variable to search for.
   *
   * @return the Rule in this Model with the given \'variable\' attribute
   * value or @c null if no such Rule exists.
   */ public
";


%csmethodmodifiers Model::getRateRuleByVariable(const std::string& variable) "
/**
   * Get a Rule object based on the variable to which it assigns a value.
   *
   * @param variable the variable to search for.
   *
   * @return the Rule in this Model with the given \'variable\' attribute
   * value or @c null if no such Rule exists.
   */ public
";


%csmethodmodifiers Model::getConstraint "
/**
   * Get the nth Constraint object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Constraint of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getConstraint(unsigned int n) "
/**
   * Get the nth Constraint object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Constraint of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getReaction "
/**
   * Get the nth Reaction object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Reaction of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getReaction(unsigned int n) "
/**
   * Get the nth Reaction object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Reaction of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getReaction(const std::string& sid) const "
/**
   * Get a Reaction object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the Reaction in this Model with the identifier @p sid or @c null
   * if no such Reaction exists.
   */ public
";


%csmethodmodifiers Model::getReaction(const std::string& sid) "
/**
   * Get a Reaction object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the Reaction in this Model with the identifier @p sid or @c null
   * if no such Reaction exists.
   */ public
";


%csmethodmodifiers Model::getSpeciesReference(const std::string& sid) "
/**
   * Get a SpeciesReference object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the SpeciesReference in this Model with the identifier @p sid or @c null
   * if no such SpeciesReference exists.
   */ public
";


%csmethodmodifiers Model::getSpeciesReference(const std::string& sid) const "
/**
   * Get a SpeciesReference object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the SpeciesReference in this Model with the identifier @p sid or @c null
   * if no such SpeciesReference exists.
   */ public
";


%csmethodmodifiers Model::getModifierSpeciesReference(const std::string& sid) "
/**
   * Get a ModifierSpeciesReference object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the ModifierSpeciesReference in this Model with the 
   * identifier @p sid or @c null
   * if no such ModifierSpeciesReference exists.
   */ public
";


%csmethodmodifiers Model::getModifierSpeciesReference(const std::string& sid) const "
/**
   * Get a ModifierSpeciesReference object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the ModifierSpeciesReference in this Model with the 
   * identifier @p sid or @c null
   * if no such ModifierSpeciesReference exists.
   */ public
";


%csmethodmodifiers Model::getEvent "
/**
   * Get the nth Event object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Event of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getEvent(unsigned int n) "
/**
   * Get the nth Event object in this Model.
   *
   * @param n the index of the object to return.
   *
   * @return the nth Event of this Model.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers Model::getEvent(const std::string& sid) const "
/**
   * Get an Event object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the Event in this Model with the identifier @p sid or @c null if
   * no such Event exists.
   */ public
";


%csmethodmodifiers Model::getEvent(const std::string& sid) "
/**
   * Get an Event object based on its identifier.
   *
   * @param sid the identifier to search for.
   *
   * @return the Event in this Model with the identifier @p sid or @c null if
   * no such Event exists.
   */ public
";


%csmethodmodifiers Model::getNumFunctionDefinitions() const "
/**
   * Get the number of FunctionDefinition objects in this Model.
   *
   * @return the number of FunctionDefinitions in this Model.
   */ public
";


%csmethodmodifiers Model::getNumUnitDefinitions() const "
/**
   * Get the number of UnitDefinition objects in this Model.
   *
   * @return the number of UnitDefinitions in this Model.
   */ public
";


%csmethodmodifiers Model::getNumCompartmentTypes() const "
/**
   * Get the number of CompartmentType objects in this Model.
   *
   * @return the number of CompartmentTypes in this Model.
   *
   * @note The CompartmentType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getNumSpeciesTypes() const "
/**
   * Get the number of SpeciesType objects in this Model.
   *
   * @return the number of SpeciesTypes in this Model.
   *
   * @note The SpeciesType object class is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.  It is not available in
   * Level&nbsp;1 nor Level&nbsp;3.
   */ public
";


%csmethodmodifiers Model::getNumCompartments() const "
/**
   * Get the number of Compartment objects in this Model.
   *
   * @return the number of Compartments in this Model.
   */ public
";


%csmethodmodifiers Model::getNumSpecies() const "
/**
   * Get the number of Species objects in this Model.
   *
   * @return the number of Species in this Model.
   */ public
";


%csmethodmodifiers Model::getNumSpeciesWithBoundaryCondition() const "
/**
   * Get the number of Species in this Model having their
   * \'boundaryCondition\' attribute value set to @c true.
   *
   * @return the number of Species in this Model with boundaryCondition set
   * to true.
   */ public
";


%csmethodmodifiers Model::getNumParameters() const "
/**
   * Get the number of Parameter objects in this Model.
   *
   * @return the number of Parameters in this Model.  Parameters defined in
   * KineticLaws are not included.
   */ public
";


%csmethodmodifiers Model::getNumInitialAssignments() const "
/**
   * Get the number of InitialAssignment objects in this Model.
   *
   * @return the number of InitialAssignments in this Model.
   */ public
";


%csmethodmodifiers Model::getNumRules() const "
/**
   * Get the number of Rule objects in this Model.
   *
   * @return the number of Rules in this Model.
   */ public
";


%csmethodmodifiers Model::getNumConstraints() const "
/**
   * Get the number of Constraint objects in this Model.
   *
   * @return the number of Constraints in this Model.
   */ public
";


%csmethodmodifiers Model::getNumReactions() const "
/**
   * Get the number of Reaction objects in this Model.
   *
   * @return the number of Reactions in this Model.
   */ public
";


%csmethodmodifiers Model::getNumEvents() const "
/**
   * Get the number of Event objects in this Model.
   *
   * @return the number of Events in this Model.
   */ public
";


%csmethodmodifiers Model::removeFromParentAndDelete "
/**
   * Remove this Model from its parent SBMLDocument object.
   *
   * This works by finding this Model\'s parent SBMLDocument and then calling
   * <code>setModel(null)</code> on it, indirectly deleting itself.
   * Overridden from the SBase function since the parent is not a ListOf.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   */ public new
";


%csmethodmodifiers Model::renameAllIds(IdentifierTransformer* idTransformer, ElementFilter* filter=NULL) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::renameIDs(List* elements, IdentifierTransformer* idTransformer) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::renameSIdRefs(const std::string& oldid, const std::string& newid) "
/**
   *
 * Replaces all uses of a given @c SIdRef type attribute value with another
 * value.
 *
 *
 * 

 * In SBML, object identifiers are of a data type called <code>SId</code>.
 * In SBML Level&nbsp;3, an explicit data type called <code>SIdRef</code> was
 * introduced for attribute values that refer to <code>SId</code> values; in
 * previous Levels of SBML, this data type did not exist and attributes were
 * simply described to as \'referring to an identifier\', but the effective
 * data type was the same as <code>SIdRef</code> in Level&nbsp;3.  These and
 * other methods of libSBML refer to the type <code>SIdRef</code> for all
 * Levels of SBML, even if the corresponding SBML specification did not
 * explicitly name the data type.
 *
 *
 *
 * This method works by looking at all attributes and (if appropriate)
 * mathematical formulas in MathML content, comparing the referenced
 * identifiers to the value of @p oldid.  If any matches are found, the
 * matching values are replaced with @p newid.  The method does @em not
 * descend into child elements.
 *
 * @param oldid the old identifier.
 * @param newid the new identifier.
 *
 *
   */ public new
";


%csmethodmodifiers Model::renameUnitSIdRefs(const std::string& oldid, const std::string& newid) "
/**
   *
 * Replaces all uses of a given @c UnitSIdRef type attribute value with
 * another value.
 *
 *
 * 
 * In SBML, unit definitions have identifiers of type <code>UnitSId</code>.  In
 * SBML Level&nbsp;3, an explicit data type called <code>UnitSIdRef</code> was
 * introduced for attribute values that refer to <code>UnitSId</code> values; in
 * previous Levels of SBML, this data type did not exist and attributes were
 * simply described to as \'referring to a unit identifier\', but the effective
 * data type was the same as <code>UnitSIdRef</code> in Level&nbsp;3.  These and
 * other methods of libSBML refer to the type <code>UnitSIdRef</code> for all
 * Levels of SBML, even if the corresponding SBML specification did not
 * explicitly name the data type.
 *
 *
 *
 * This method works by looking at all unit identifier attribute values
 * (including, if appropriate, inside mathematical formulas), comparing the
 * referenced unit identifiers to the value of @p oldid.  If any matches
 * are found, the matching values are replaced with @p newid.  The method
 * does @em not descend into child elements.
 *
 * @param oldid the old identifier.
 * @param newid the new identifier.
 *
 *
   */ public new
";


%csmethodmodifiers Model::isBoolean(const ASTNode* node) const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::convertL1ToL2 "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::convertL1ToL3(bool addDefaultUnits = true) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::convertL2ToL3(bool strict = false, bool addDefaultUnits = true) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::convertL2ToL1(bool strict = false) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::convertL3ToL1(bool strict = false) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::convertL3ToL2(bool strict = false) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::convertFromL3V2(bool strict = false) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::dealWithFast "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::dealWithL3Fast(unsigned int targetVersion) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::addModifiers "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::addConstantAttribute "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::setSpatialDimensions(double dims = 3.0) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::addDefinitionsForDefaultUnits "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::dealWithDefaultValues "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::convertParametersToLocals(unsigned int level, unsigned int version) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::setSpeciesReferenceConstantValueAndStoichiometry "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::removeMetaId "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::removeSBOTerms(bool strict) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::removeHasOnlySubstanceUnits "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::removeSBOTermsNotInL2V2(bool strict) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::removeDuplicateTopLevelAnnotations "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::removeParameterRuleUnits(bool strict) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::convertStoichiometryMath "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::assignRequiredValues "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::dealWithModelUnits(bool strict = false) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::dealWithStoichiometry "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::dealWithEvents(bool strict) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::removeSpeciesTypes "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::removeCompartmentTypes "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::setSBMLDocument(SBMLDocument* d) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::getTypeCode() const "
/**
   * Returns the libSBML type code for this %SBML object.
   *
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for this object:
   * @link libsbml#SBML_MODEL SBML_MODEL@endlink (default).
   *
   *
 * @warning <span class=\'warning\'>The specific integer values of the possible
 * type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
 * packages,  To fully identify the correct code, <strong>it is necessary to
 * invoke both getPackageName() and getTypeCode()</strong> (or 
 * ListOf::getItemTypeCode()).</span>
 *
 *
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers Model::getElementName() const "
/**
   * Returns the XML element name of this object, which for Model, is
   * always @c \'model\'.
   *
   * @return the name of this element, i.e., @c \'model\'.
   */ public new
";


%csmethodmodifiers Model::getElementPosition() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::writeElements(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::populateListFormulaUnitsData "
/**
   * Populates the internal list of derived units for this Model object.
   *
   * This method tells libSBML to (re)calculate all units for all components
   * of the enclosing Model object.  The result is stored in an internal list
   * of unit data.  Users can access the resulting data by calling the method
   * SBase::getDerivedUnitDefinition() available on most objects.  (The name
   * \'formula units data\' is drawn from the name of the internal objects
   * libSBML uses to store the data; note that these internal objects are not
   * exposed to callers, because callers can interact with the results using
   * the ordinary SBML unit objects.)
   *
   * This method is used by libSBML itself in the validator concerned with
   * unit consistency.  The unit consistency validator (like all other
   * validators in libSBML) is invoked by using
   * SBMLDocument::checkConsistency(), with the consistency checks for the
   * category @link libsbml#LIBSBML_CAT_UNITS_CONSISTENCY LIBSBML_CAT_UNITS_CONSISTENCY@endlink turned on.  The method
   * populateListFormulaUnitsData() does not need to be called prior to
   * invoking the validator if unit consistency checking has not been turned
   * off.  This method is only provided for cases when callers have a special
   * need to force the unit data to be recalculated.  For instance, during
   * construction of a model, a caller may want to interrogate libSBML\'s
   * inferred units without invoking full-blown model validation; this is a
   * scenario in which calling populateListFormulaUnitsData() may be useful.
   *
   * @warning Computing and inferring units is a time-consuming operation.
   * Callers may want to call isPopulatedListFormulaUnitsData() to determine
   * whether the units may already have been computed, to save themselves the
   * need of invoking unit inference unnecessarily.
   *
   * @see isPopulatedListFormulaUnitsData()
   */ public
";


%csmethodmodifiers Model::isPopulatedListFormulaUnitsData "
/**
   * Predicate returning @c true if libSBML has derived units for the
   * components of this model.
   *
   * LibSBML can infer the units of measurement associated with different
   * elements of a model.  When libSBML does that, it builds a complex
   * internal structure during a resource-intensive operation.  This is done
   * automatically only when callers invoke validation (via
   * SBMLDocument::checkConsistency()) and have not turned off the unit
   * validation option.
   *
   * Callers can force units to be recalculated by calling
   * populateListFormulaUnitsData().  To avoid calling that method
   * unnecessarily, calling programs may first want to invoke this method
   * (isPopulatedListFormulaUnitsData()) to determine whether it is even
   * necessary.
   *
   * @return @c true if the units have already been computed, @c false
   * otherwise.
   */ public
";


%csmethodmodifiers Model::addFormulaUnitsData(const FormulaUnitsData* fud) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createFormulaUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createFormulaUnitsData(const std::string& id, int typecode) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getFormulaUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getFormulaUnitsData(unsigned int n) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getFormulaUnitsData(const std::string& sid, int typecode) const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getFormulaUnitsData(const std::string& sid, int) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getFormulaUnitsDataForVariable(const std::string& sid) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getFormulaUnitsDataForAssignment(const std::string& sid) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getNumFormulaUnitsData() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getListFormulaUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getListFormulaUnitsData() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::populateAllElementIdList "
/**
   * Populates the internal list of the identifiers of all elements within this Model object.
   *
   * This method tells libSBML to retrieve the identifiers of all elements
   * of the enclosing Model object.  The result is stored in an internal list
   * of ids.  Users can access the resulting data by calling the method
   * getAllElementIdList().
   *
   * @warning Retrieving all elements within a model is a time-consuming operation.
   * Callers may want to call isPopulatedAllElementIdList() to determine
   * whether the id list may already have been populated.
   *
   * @see isPopulatedAllElementIdList()
   */ public
";


%csmethodmodifiers Model::isPopulatedAllElementIdList() const "
/**
   * Predicate returning @c true if libSBML has a list of the ids of all 
   * components of this model.
   *
   * @return @c true if the id list has already been populated, @c false
   * otherwise.
   */ public
";


%csmethodmodifiers Model::getAllElementIdList() const "
/**
   * Returns the internal list of the identifiers of all elements within this Model object.
   *
   * @return an IdList of all the identifiers in the model.
   *
   * @see populateAllElementIdList()
   * @see isPopulatedAllElementIdList()
   */ public
";


%csmethodmodifiers Model::clearAllElementIdList "
/**
   * Clears the internal list of the identifiers of all elements within this Model object.
   *
   * @see populateAllElementIdList()
   * @see isPopulatedAllElementIdList()
   */ public
";


%csmethodmodifiers Model::populateAllElementMetaIdList "
/**
   * Populates the internal list of the metaids of all elements within this Model object.
   *
   * This method tells libSBML to retrieve the identifiers of all elements
   * of the enclosing Model object.  The result is stored in an internal list
   * of metaids.  Users can access the resulting data by calling the method
   * getAllElementMetaIdList().
   *
   * @warning Retrieving all elements within a model is a time-consuming operation.
   * Callers may want to call isPopulatedAllElementMetaIdList() to determine
   * whether the metaid list may already have been populated.
   *
   * @see isPopulatedAllElementMetaIdList()
   */ public
";


%csmethodmodifiers Model::isPopulatedAllElementMetaIdList() const "
/**
   * Predicate returning @c true if libSBML has a list of the metaids of all 
   * components of this model.
   *
   * @return @c true if the metaid list has already been populated, @c false
   * otherwise.
   */ public
";


%csmethodmodifiers Model::getAllElementMetaIdList() const "
/**
   * Returns the internal list of the metaids of all elements within this Model object.
   *
   * @return an IdList of all the metaids in the model.
   *
   * @see populateAllElementMetaIdList()
   * @see isPopulatedAllElementMetaIdList()
   */ public
";


%csmethodmodifiers Model::clearAllElementMetaIdList "
/**
   * Clears the internal list of the metaids of all elements within this Model object.
   *
   * @see populateAllElementMetaIdList()
   * @see isPopulatedAllElementMetaIdList()
   */ public
";


%csmethodmodifiers Model::hasRequiredElements() const "
/**
   * Predicate returning @c true if all the required elements for this Model
   * object have been set.
   *
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */ public new
";


%csmethodmodifiers Model::removeFunctionDefinition(unsigned int n) "
/**
   * Removes the nth FunctionDefinition object from this Model object and 
   * returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param n the index of the FunctionDefinition object to remove.
   *
   * @return the FunctionDefinition object removed, or @c null if the given
   * index is out of range.
   *
   */ public
";


%csmethodmodifiers Model::removeFunctionDefinition(const std::string& sid) "
/**
   * Removes the FunctionDefinition object with the given identifier from this Model 
   * object and returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param sid the identifier of the FunctionDefinition object to remove.
   *
   * @return the FunctionDefinition object removed, or @c null if no
   * FunctionDefinition object with the identifier exists in this Model
   * object.
   */ public
";


%csmethodmodifiers Model::removeUnitDefinition(unsigned int n) "
/**
   * Removes the nth UnitDefinition object from this Model object and
   * returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param n the index of the UnitDefinition object to remove.
   *
   * @return the UnitDefinition object removed., or @c null if the given
   * index is out of range.
   *
   */ public
";


%csmethodmodifiers Model::removeUnitDefinition(const std::string& sid) "
/**
   * Removes the UnitDefinition object with the given identifier from this Model
   * object and returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param sid the identifier of the UnitDefinition object to remove.
   *
   * @return the UnitDefinition object removed, or @c null if no
   * UnitDefinition object with the identifier exists in this Model object.
   */ public
";


%csmethodmodifiers Model::removeCompartmentType(unsigned int n) "
/**
   * Removes the nth CompartmentType object from this Model object and
   * returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param n the index of the CompartmentType object to remove.
   *
   * @return the ComapartmentType object removed, or @c null if the given
   * index is out of range.
   *
   */ public
";


%csmethodmodifiers Model::removeCompartmentType(const std::string& sid) "
/**
   * Removes the CompartmentType object with the given identifier from this Model
   * object and returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param sid the identifier of the object to remove.
   *
   * @return the CompartmentType object removed, or @c null if no
   * CompartmentType object with the identifier exists in this Model object.
   */ public
";


%csmethodmodifiers Model::removeSpeciesType(unsigned int n) "
/**
   * Removes the nth SpeciesType object from this Model object and
   * returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param n the index of the SpeciesType object to remove.
   *
   * @return the SpeciesType object removed, or @c null if the given index is
   * out of range.
   *
   */ public
";


%csmethodmodifiers Model::removeSpeciesType(const std::string& sid) "
/**
   * Removes the SpeciesType object with the given identifier from this Model
   * object and returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param sid the identifier of the SpeciesType object to remove.
   *
   * @return the SpeciesType object removed, or @c null if no SpeciesType
   * object with the identifier exists in this Model object.
   *
   */ public
";


%csmethodmodifiers Model::removeCompartment(unsigned int n) "
/**
   * Removes the nth Compartment object from this Model object and
   * returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param n the index of the Compartment object to remove.
   *
   * @return the Compartment object removed, or @c null if the given index is
   * out of range.
   *
   */ public
";


%csmethodmodifiers Model::removeCompartment(const std::string& sid) "
/**
   * Removes the Compartment object with the given identifier from this Model
   * object and returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param sid the identifier of the Compartment object to remove.
   *
   * @return the Compartment object removed, or @c null if no Compartment
   * object with the identifier exists in this Model object.
   */ public
";


%csmethodmodifiers Model::removeSpecies(unsigned int n) "
/**
   * Removes the nth Species object from this Model object and
   * returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param n the index of the Species object to remove.
   *
   * @return the Species object removed, or @c null if the given index is out
   * of range.
   *
   */ public
";


%csmethodmodifiers Model::removeSpecies(const std::string& sid) "
/**
   * Removes the Species object with the given identifier from this Model
   * object and returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param sid the identifier of the Species object to remove.
   *
   * @return the Species object removed, or @c null if no Species object with
   * the identifier exists in this Model object.
   *
   */ public
";


%csmethodmodifiers Model::removeParameter(unsigned int n) "
/**
   * Removes the nth Parameter object from this Model object and
   * returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param n the index of the Parameter object to remove.
   *
   * @return the Parameter object removed, or @c null if the given index is
   * out of range.
   *
   */ public
";


%csmethodmodifiers Model::removeParameter(const std::string& sid) "
/**
   * Removes the Parameter object with the given identifier from this Model
   * object and returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param sid the identifier of the Parameter object to remove.
   *
   * @return the Parameter object removed, or @c null if no Parameter object
   * with the identifier exists in this Model object.
   */ public
";


%csmethodmodifiers Model::removeInitialAssignment(unsigned int n) "
/**
   * Removes the nth InitialAssignment object from this Model object and
   * returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param n the index of the InitialAssignment object to remove.
   *
   * @return the InitialAssignment object removed, or @c null if the given
   * index is out of range.
   *
   */ public
";


%csmethodmodifiers Model::removeInitialAssignment(const std::string& symbol) "
/**
   * Removes the InitialAssignment object with the given \'symbol\' attribute 
   * from this Model object and returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param symbol the \'symbol\' attribute of the InitialAssignment object to remove.
   *
   * @return the InitialAssignment object removed, or @c null if no
   * InitialAssignment object with the \'symbol\' attribute exists in this
   * Model object.
   */ public
";


%csmethodmodifiers Model::removeRule(unsigned int n) "
/**
   * Removes the nth Rule object from this Model object and
   * returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param n the index of the Rule object to remove.
   *
   * @return the Rule object removed, or @c null if the given index is out of
   * range.
   *
   */ public
";


%csmethodmodifiers Model::removeRule(const std::string& variable) "
/**
   * Removes the Rule object with the given \'variable\' attribute from this Model 
   * object and returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param variable the \'variable\' attribute of the Rule object to remove.
   *
   * @return the Rule object removed, or @c null if no Rule object with the
   * \'variable\' attribute exists in this Model object.
   */ public
";


%csmethodmodifiers Model::removeRuleByVariable(const std::string& variable) "
/**
   * Removes the Rule object with the given \'variable\' attribute from this Model 
   * object and returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param variable the \'variable\' attribute of the Rule object to remove.
   *
   * @return the Rule object removed, or @c null if no Rule object with the
   * \'variable\' attribute exists in this Model object.
   */ public
";


%csmethodmodifiers Model::removeConstraint(unsigned int n) "
/**
   * Removes the nth Constraint object from this Model object and
   * returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param n the index of the Constraint object to remove.
   *
   * @return the Constraint object removed, or @c null if the given index is
   * out of range.
   *
   */ public
";


%csmethodmodifiers Model::removeReaction(unsigned int n) "
/**
   * Removes the nth Reaction object from this Model object and
   * returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param n the index of the Reaction object to remove.
   *
   * @return the Reaction object removed, or @c null if the given index is
   * out of range.
   *
   */ public
";


%csmethodmodifiers Model::removeReaction(const std::string& sid) "
/**
   * Removes the Reaction object with the given identifier from this Model
   * object and returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param sid the identifier of the Reaction object to remove.
   *
   * @return the Reaction object removed, or @c null if no Reaction object
   * with the identifier exists in this Model object.
   *
   */ public
";


%csmethodmodifiers Model::removeEvent(unsigned int n) "
/**
   * Removes the nth Event object from this Model object and
   * returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param n the index of the Event object to remove.
   *
   * @return the Event object removed, or @c null if the given index is out
   * of range.
   *
   */ public
";


%csmethodmodifiers Model::removeEvent(const std::string& sid) "
/**
   * Removes the Event object with the given identifier from this Model
   * object and returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param sid the identifier of the Event object to remove.
   *
   * @return the Event object removed, or @c null if no Event object with the
   * identifier exists in this Model object.
   *
   */ public
";


%csmethodmodifiers Model::appendFrom(const Model* model) "
/**
   * Copies a given Model object\'s subcomponents and appends the copies to
   * the appropriate places in this Model.
   *
   * This method also calls the <code>appendFrom</code> method on all libSBML
   * plug-in objects.
   *
   *
 * 
 * SBML Level&nbsp;3 consists of a <em>Core</em> definition that can be extended
 * via optional SBML Level&nbsp;3 <em>packages</em>.  A given model may indicate
 * that it uses one or more SBML packages, and likewise, a software tool may be
 * able to support one or more packages.  LibSBML does not come preconfigured
 * with all possible packages included and enabled, in part because not all
 * package specifications have been finalized.  To support the ability for
 * software systems to enable support for the Level&nbsp;3 packages they choose,
 * libSBML features a <em>plug-in</em> mechanism.  Each SBML Level&nbsp;3
 * package is implemented in a separate code plug-in that can be enabled by the
 * application to support working with that SBML package.  A given SBML model
 * may thus contain not only objects defined by SBML Level&nbsp;3 Core, but also
 * objects created by libSBML plug-ins supporting additional Level&nbsp;3
 * packages.
 *
 *
   *
   * @param model the Model to merge with this one.
   *
   */ public new
";


%csmethodmodifiers Model::enablePackageInternal(const std::string& pkgURI, const std::string& pkgPrefix, bool flag) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::updateSBMLNamespace(const std::string& package, unsigned int level, unsigned int version) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::readOtherXML(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::createObject(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::addExpectedAttributes(ExpectedAttributes& attributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::readL1Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::readL2Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::readL3Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::writeAttributes(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::syncAnnotation "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Model::populatePerTimeUnitDefinition(FormulaUnitsData *fud) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createSpeciesReferenceUnitsData(SpeciesReference* sr, UnitFormulaFormatter* unitFormatter) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::removeListFormulaUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createSubstanceUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getL3SubstanceUD "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getSubstanceUD "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createTimeUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getL3TimeUD "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getTimeUD "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createVolumeUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getL3VolumeUD "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getVolumeUD "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createAreaUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getL3AreaUD "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getAreaUD "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createLengthUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getL3LengthUD "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getLengthUD "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createExtentUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getL3ExtentUD "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createSubstancePerTimeUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getL3SubstancePerTimeUD(FormulaUnitsData* fud) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::getSubstancePerTimeUD "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createCompartmentUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createSpeciesUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createL3SpeciesUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createParameterUnitsData "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createUnitsDataFromMath(UnitFormulaFormatter * unitFormatter, FormulaUnitsData * fud, const ASTNode * math) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createInitialAssignmentUnitsData(UnitFormulaFormatter * unitFormatter) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createConstraintUnitsData(UnitFormulaFormatter * unitFormatter) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createRuleUnitsData(UnitFormulaFormatter * unitFormatter) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createReactionUnitsData(UnitFormulaFormatter * unitFormatter) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createEventUnitsData(UnitFormulaFormatter * unitFormatter) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createDelayUnitsData(UnitFormulaFormatter* unitFormatter, Event * e, const std::string& eventId) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createTriggerUnitsData(UnitFormulaFormatter* unitFormatter, Event * e, const std::string& eventId) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createPriorityUnitsData(UnitFormulaFormatter* unitFormatter, Priority * p, const std::string& eventId) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createEventAssignmentUnitsData(UnitFormulaFormatter* unitFormatter, EventAssignment * ea, const std::string& eventId) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Model::createLocalParameterUnitsData(KineticLaw * kl, UnitFormulaFormatter * unitFormatter) "
/** */ /* libsbml-internal */ public
";


%typemap(csimports) SBMLDocument "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html Overall SBML container object.
 *
 * @if clike LibSBML uses the class SBMLDocument as a
 * top-level container for storing SBML content and data associated with it
 * (such as warnings and error messages).  The two primary means of reading
 * an SBML model, SBMLReader::readSBML() and
 * SBMLReader::readSBMLFromString(), both return a pointer to an
 * SBMLDocument object.  From there, callers can inquire about any errors
 * encountered (e.g., using SBMLDocument::getNumErrors()), access the Model
 * object, and perform other actions such as consistency-checking and model
 * translation.
 * @endif@if python LibSBML uses the class SBMLDocument as a
 * top-level container for storing SBML content and data associated with it
 * (such as warnings and error messages).  The two primary means of reading
 * an SBML model, SBMLReader::readSBML() and
 * SBMLReader::readSBMLFromString(), both return a pointer to an
 * SBMLDocument object.  From there, callers can inquire about any errors
 * encountered (e.g., using SBMLDocument::getNumErrors()), access the Model
 * object, and perform other actions such as consistency-checking and model
 * translation.
 * @endif@if java LibSBML uses the class SBMLDocument as a top-level
 * container for storing SBML content and data associated with it (such as
 * warnings and error messages).  The two primary means of reading an SBML
 * model, SBMLReader::readSBML(String filename) and
 * SBMLReader::readSBMLFromString(String xml), both return an SBMLDocument
 * object.  From there, callers can inquire about any errors encountered
 * (e.g., using SBMLDocument::getNumErrors()), access the Model object, and
 * perform other actions such as consistency-checking and model
 * translation.
 * @endif
 * 
 * When creating fresh models programmatically, the starting point is
 * typically the creation of an SBMLDocument object instance.  The
 * SBMLDocument constructor accepts arguments for the SBML Level and
 * Version of the model to be created.  After creating the SBMLDocument
 * object, calling programs then typically call SBMLDocument::createModel()
 * almost immediately, and then proceed to call the methods on the Model
 * object to fill out the model\'s contents.
 *
 * SBMLDocument corresponds roughly to the class <i>Sbml</i> defined in the
 * SBML Level&nbsp;2 specification and <i>SBML</i> in the Level&nbsp;3
 * specification.  It does not have a direct correspondence in SBML
 * Level&nbsp;1.  (However, to make matters simpler for applications,
 * libSBML creates an SBMLDocument no matter whether the model is
 * Level&nbsp;1, Level&nbsp;2 or Level&nbsp;3.)  In its barest form, when written out in
 * XML format for (e.g.) SBML Level&nbsp;2 Version&nbsp;4, the corresponding
 * structure is the following:
 * @verbatim
<sbml xmlns=\'http://www.sbml.org/sbml/level2/version4\' level=\'2\' version=\'4\'>
  ...
</sbml>@endverbatim
 * 
 * SBMLDocument is derived from SBase, and therefore contains the usual SBase
 * attributes (in SBML Level&nbsp;2 and Level&nbsp;3) of \'metaid\' and \'sboTerm\', as
 * well as the subelements \'notes\' and \'annotation\'.  It also contains the
 * attributes \'level\' and \'version\' indicating the Level and Version of the
 * SBML data structure.  These can be accessed using the methods defined by
 * the SBase class for that purpose.
 *
 * @section checking Checking consistency and adherence to SBML specifications
 *
 * One of the most important features of libSBML is its ability to perform
 * SBML validation to ensure that a model adheres to the SBML specification
 * for whatever Level+Version combination the model uses.  SBMLDocument
 * provides the methods for running consistency-checking and validation
 * rules on the SBML content.
 *
 * First, a brief explanation of the rationale is in order.  In libSBML
 * versions up to and including the version&nbsp;3.3.x series, the
 * individual methods for creating and setting attributes and other
 * components were quite lenient, and allowed a caller to compose SBML
 * entities that might not, in the end, represent valid SBML.  This allowed
 * applications the freedom to do things such as save incomplete models
 * (which is useful when models are being developed over long periods of
 * time).  In the version&nbsp;4.x series, libSBML is somewhat stricter,
 * but still permits structures to be created independently and the results
 * to be combined in a separate step.  In all these cases, it means that a
 * separate validation step is necessary when a calling program finally
 * wants to finish a complete SBML document.
 *
 * The primary interface to this validation facility is SBMLDocument\'s
 * SBMLDocument::checkInternalConsistency() and
 * SBMLDocument::checkConsistency().  The former verifies the basic
 * internal consistency and syntax of an SBML document, and the latter
 * implements more elaborate validation rules (both those defined by the
 * SBML specifications, as well as additional rules offered by libSBML).
 *
 * @if clike The checks performed by SBMLDocument::checkInternalConsistency() are
 * hardwired and cannot be changed by calling programs, but the validation
 * performed by SBMLDocument::checkConsistency() is under program control
 * using the method SBMLDocument::setConsistencyChecks().  Applications can
 * selectively disable specific kinds of checks that they may not be
 * interested in, by calling SBMLDocument::setConsistencyChecks() with
 * appropriate parameters.
 * @endif@if python The checks performed by SBMLDocument::checkInternalConsistency() are
 * hardwired and cannot be changed by calling programs, but the validation
 * performed by SBMLDocument::checkConsistency() is under program control
 * using the method SBMLDocument::setConsistencyChecks().  Applications can
 * selectively disable specific kinds of checks that they may not be
 * interested in, by calling SBMLDocument::setConsistencyChecks() with
 * appropriate parameters.
 * @endif@if java The checks performed by SBMLDocument::checkInternalConsistency() are
 * hardwired and cannot be changed by calling programs, but the validation
 * performed by SBMLDocument::checkConsistency() is under program control
 * using the method SBMLDocument::setConsistencyChecks(int categ, boolean
 * onoff).  Applications can selectively disable specific kinds of checks
 * that they may not be interested by calling
 * SBMLDocument::setConsistencyChecks(int categ, boolean onoff) with
 * appropriate parameters.
 * @endif
 *
 * These methods have slightly different relevance depending on whether a
 * model is created programmaticaly from scratch, or whether it is read in
 * from a file or data stream.  The following list summarizes the possible
 * scenarios.
 *
 * <em>Scenario 1: Creating a model from scratch</em>.  Before writing out
 * the model, 
 *
 * @li Call SBMLDocument::checkInternalConsistency(), then inquire about
 * the results by calling SBMLDocument::getNumErrors()
 *
 * @li Call @if java SBMLDocument::setConsistencyChecks(int categ, boolean
 * onoff) @else SBMLDocument::setConsistencyChecks() @endif to configure
 * which checks will be performed by SBMLDocument::checkConsistency()
 *
 * @li Call SBMLDocument::checkConsistency(), then inquire about the results by
 * calling SBMLDocument::getNumErrors()
 *
 * <em>Scenario 2: Reading a model from a file or data stream.</em> After
 * reading the model,
 * 
 * @li Basic consistency checks will have been performed automatically by
 * libSBML upon reading the content, so you only need to inquire about the
 * results by using SBMLDocument::getNumErrors()
 * 
 * @li Call @if java SBMLDocument::setConsistencyChecks(int categ, boolean
 * onoff) @else SBMLDocument::setConsistencyChecks() @endif to configure
 * which checks are performed by SBMLDocument::checkConsistency()
 * 
 * @li Call SBMLDocument::checkConsistency(), then inquire about the results
 * by calling SBMLDocument::getNumErrors()
 *
 * @if clike An example of using the consistency-checking
 * and validation facilities is provided in this manual in the
 * section @ref libsbml-example. @endif
 *
 * It should be noted that as of SBML Level&nbsp;3 Version&nbsp;2, the Model
 * became an optional child of SBMLDocument, instead of being required.  This
 * means that one can no longer use SBMLDocument::getModel() as a cheap method
 * of checking if an SBML document was read in properly: the more robust
 * getError methods detailed above must be used instead.
 * 
 * @section converting Converting documents between Levels and Versions of SBML
 *
 * LibSBML provides facilities for limited translation of SBML between
 * Levels and Versions of the SBML specifications.  The method for doing is
 * is @if java SBMLDocument::setLevelAndVersion(long lev, long ver, boolean strict) @else setLevelAndVersion() @endif.  In 
 * general, models can be converted upward without difficulty (e.g., from
 * SBML Level&nbsp;1 to Level&nbsp;2, or from an earlier Version of
 * Level&nbsp;2 to the latest Version of Level&nbsp;2).  Sometimes models
 * can be translated downward as well, if they do not use constructs
 * specific to more advanced Levels of SBML.
 *
 * Calling @if java SBMLDocument::setLevelAndVersion(long lev, long ver, boolean strict) @else SBMLDocument::setLevelAndVersion() @endif will not @em necessarily lead
 * to a successful conversion.  The method will return a boolean value
 * to indicate success or failure.  Callers must check the error log (see 
 * next section) attached to the SBMLDocument object after calling
 * @if java SBMLDocument::setLevelAndVersion(long lev, long ver) @else SBMLDocument::setLevelAndVersion() @endif in order to assess whether any
 * problems arose.
 *
 * If an application is interested in translating to a lower Level and/or
 * Version of SBML within a Level, the following methods allow for prior
 * assessment of whether there is sufficient compatibility to make a
 * translation possible:
 *
 * @li SBMLDocument::checkL1Compatibility(),
 * @li SBMLDocument::checkL2v1Compatibility(),
 * @li SBMLDocument::checkL2v2Compatibility(),
 * @li SBMLDocument::checkL2v3Compatibility(), 
 * @li SBMLDocument::checkL2v4Compatibility(),
 * @li SBMLDocument::checkL2v5Compatibility(), and
 * @li SBMLDocument::checkL3v1Compatibility().
 * 
 * Some changes between Versions of SBML Level&nbsp;2 may lead to
 * unexpected behaviors when attempting conversions in either direction.
 * For example, SBML Level&nbsp;2 Version&nbsp;4 relaxed the requirement
 * for consistency in units of measurement between expressions annd
 * quantities in a model.  As a result, a model written in Version&nbsp;4,
 * if converted to Version&nbsp;3 with no other changes, may fail
 * validation as a Version&nbsp;3 model because Version&nbsp;3 imposed
 * stricter requirements on unit consistency.
 *
 * Other changes between SBML Level 2 and Level 3 make downward conversions
 * challenging.  In some cases, it means that a model converted to
 * Level&nbsp;2 from Level&nbsp;3 will contain attributes that were not
 * explicitly given in the Level&nbsp;3 model, because in Level&nbsp;2
 * these attributes may have been optional or have default values.
 * 
 * @section errors Error handling
 *
 * Upon reading a model, SBMLDocument logs any problems encountered while
 * reading the model from the file or data stream.  The log contains
 * objects that record diagnostic information about any notable issues that
 * arose.  Whether the problems are warnings or errors, they are both
 * reported through a single common interface involving the object class
 * SBMLError.
 *
 * The methods SBMLDocument::getNumErrors(), @if java SBMLDocument::getError(long n) @else SBMLDocument::getError() @endif and
 * SBMLDocument::printErrors() allow callers to interact with the warnings
 * or errors logged.  Alternatively, callers may retrieve the entire log as
 * an SBMLErrorLog object using the method SBMLDocument::getErrorLog().
 * The SBMLErrorLog object provides some alternative methods for
 * interacting with the set of errors and warnings.  In either case,
 * applications typically should first call SBMLDocument::getNumErrors() to
 * find out if any issues have been logged after specific libSBML
 * operations such as the ones discussed in the sections above.  If they
 * have, then an application will should proceed to inspect the individual
 * reports using either the direct interfaces on SBMLDocument or using the
 * methods on the SBMLErrorLog object.
 *
 * @if clike An example of using the error facility is
 * provided in this manual in the
 * section @ref libsbml-example. @endif
 * 
 */
"


%csmethodmodifiers SBMLDocument::getDefaultLevel "
/**
   * The default SBML Level of new SBMLDocument objects.
   *
   *
 * 
 * This \'default Level\' corresponds to the most recent SBML specification
 * Level available at the time libSBML version @htmlinclude
 * libsbml-version.html was released.  The default Level is used by
 * SBMLDocument if no Level is explicitly specified at the time of the
 * construction of an SBMLDocument instance.
 *
   *
   * @return an integer indicating the most recent SBML specification Level.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   *
   * @see @if clike getDefaultVersion() @else SBMLDocument::getDefaultVersion() @endif
   */ public
";


%csmethodmodifiers SBMLDocument::getDefaultVersion "
/**
   * The default Version of new SBMLDocument objects.
   *
   *
 * 
 * This \'default Version\' corresponds to the most recent Version within the
 * most recent Level of SBML available at the time libSBML version
 * @htmlinclude libsbml-version.html was released.  The default Version is
 * used by SBMLDocument if no Version is explicitly specified at the time of
 * the construction of an SBMLDocument instance.
 * 
   *
   * @return an integer indicating the most recent SBML specification
   * Version.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   *
   * @see @if clike getDefaultLevel() @else SBMLDocument::getDefaultLevel() @endif
   */ public
";


%csmethodmodifiers SBMLDocument::SBMLDocument(unsigned int level = 0, unsigned int version = 0) "
/**
   * Creates a new SBMLDocument, optionally with given values for the SBML
   * Level and Version.
   *
   * If <em>both</em> the SBML Level and Version attributes are not
   * specified, the SBML document is treated as having the latest Level and
   * Version of SBML as determined by SBMLDocument::getDefaultLevel() and
   * SBMLDocument::getDefaultVersion(); <em>however</em>, the SBMLDocument
   * object is otherwise left blank.  In particular, the blank SBMLDocument
   * object has no associated XML attributes, including (but not limited
   * to) an XML namespace declaration.  The XML namespace declaration is
   * not added until the model is written out, <em>or</em> the method
   * SBMLDocument::setLevelAndVersion(@if java long, long, boolean@endif)
   * is called.  This may be important to keep in mind
   * if an application needs to add additional XML namespace declarations
   * on the <code>&lt;sbml&gt;</code> element.  Application writers should
   * either provide values for @p level and @p version on the call to this
   * constructor, or else call
   * SBMLDocument::setLevelAndVersion(@if java long, long, boolean@endif)
   * shortly after creating the SBMLDocument object.
   *
   * @param level an integer for the SBML Level.
   *
   * @param version an integer for the Version within the SBML Level.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   * @ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif
   *
   * @see SBMLDocument::setLevelAndVersion(@if java long, long, boolean@endif)
   * @see getDefaultLevel()
   * @see getDefaultVersion()
   */ public
";


%csmethodmodifiers SBMLDocument::SBMLDocument(SBMLNamespaces* sbmlns) "
/**
   * Creates a new SBMLDocument using the given SBMLNamespaces object 
   * @p sbmlns.
   *
   *
 * 
 * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
 * information.  It is used to communicate the SBML Level, Version, and (in
 * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
 * common approach to using libSBML\'s SBMLNamespaces facilities is to create an
 * SBMLNamespaces object somewhere in a program once, then hand that object
 * as needed to object constructors that accept SBMLNamespaces as arguments.
 *
 * 
   *
   * @param sbmlns an SBMLNamespaces object.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   */ public
";


%csmethodmodifiers SBMLDocument::SBMLDocument(const SBMLDocument& orig) "
/**
   * Copy constructor; creates a copy of this SBMLDocument.
   *
   * @param orig the object to copy.
   */ public
";


%csmethodmodifiers SBMLDocument::accept(SBMLVisitor& v) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBMLDocument::clone() const "
/**
   * Creates and returns a deep copy of this SBMLDocument object.
   *
   * @return the (deep) copy of this SBMLDocument object.
   */ public new
";


%csmethodmodifiers SBMLDocument::isSetModel() const "
/**
  * Returns @c true if the Model object has been set, otherwise 
  * returns @c false.
  *
  * @return @c true if the Model object has been set
  */ public
";


%csmethodmodifiers SBMLDocument::getModel() const "
/**
   * Returns the Model object stored in this SBMLDocument.
   *
   * It is important to note that this method <em>does not create</em> a
   * Model instance.  The model in the SBMLDocument must have been created
   * at some prior time, for example using SBMLDocument::createModel() 
   * or SBMLDocument::setModel(@if java Model@endif).
   * This method returns @c null if a model does not yet exist.
   * 
   * @return the Model contained in this SBMLDocument, or @c null if no such model exists.
   *
   * @see createModel()
   */ public new
";


%csmethodmodifiers SBMLDocument::getModel "
/**
   * Returns the Model object stored in this SBMLDocument.
   *
   * It is important to note that this method <em>does not create</em> a
   * Model instance.  The model in the SBMLDocument must have been created
   * at some prior time, for example using SBMLDocument::createModel() 
   * or SBMLDocument::setModel(@if java Model@endif).
   * This method returns @c null if a model does not yet exist.
   * 
   * @return the Model contained in this SBMLDocument.
   *
   * @see createModel()
   */ public new
";


%csmethodmodifiers SBMLDocument::getElementBySId(const std::string& id) "
/**
   * Returns the first child element found that has the given @p id in the
   * model-wide SId namespace, or @c null if no such object is found.
   *
   * @param id string representing the id of the object to find.
   *
   * @return pointer to the first element found with the given @p id.
   */ public new
";


%csmethodmodifiers SBMLDocument::getElementByMetaId(const std::string& metaid) "
/**
   * Returns the first child element it can find with the given @p metaid, or
   * itself if it has the given @p metaid, or @c null if no such object is
   * found.
   *
   * @param metaid string representing the metaid of the object to find.
   *
   * @return pointer to the first element found with the given @p metaid.
   */ public new
";


%csmethodmodifiers SBMLDocument::getAllElements(ElementFilter* filter=NULL) "
/**
   * Returns a List of all child SBase objects, including those nested to an
   * arbitrary depth
   *
   * @param filter a pointer to an ElementFilter, which causes the function 
   * to return only elements that match a particular set of constraints.  
   * If null (the default), the function will return all child objects.
   *
   * @return a List of pointers to all children objects.
   */ public new
";


%csmethodmodifiers SBMLDocument::expandFunctionDefinitions "
/**
   * Removes FunctionDefinition constructs from the document and expands
   * any instances of their use within <code>&lt;math&gt;</code> elements.
   *
   * For example, suppose a Model contains a FunctionDefinition with
   * identifier @c \'f\' representing the math expression: <em>f(x, y) = x *
   * y</em>.  Suppose further that there is a reaction in which the
   * <code>&lt;math&gt;</code> element of the KineticLaw object contains
   * <code>f(s, p)</code>, where @c s and @c p are other identifiers
   * defined in the model.  The outcome of invoking this method is that the
   * <code>&lt;math&gt;</code> of the KineticLaw now represents the
   * expression <em>s * p</em> and the model no longer contains any
   * FunctionDefinition objects.
   * 
   * @return bool @c true if the transformation was successful, 
   * @c false, otherwise.
   *
   * @note This function will check the consistency of a model before
   * attemptimg the transformation.  If the model is not valid SBML, the
   * transformation will not be performed and the function will return
   * @c false.
   */ public
";


%csmethodmodifiers SBMLDocument::expandInitialAssignments "
/**
   * Removes InitialAssignment constructs from the document and
   * replaces them with appropriate values.
   *
   * For example, suppose a Model contains a InitialAssignment to a symbol
   * @c \'k\' where @c \'k\' is the identifier of a Parameter.  The outcome of
   * invoking this method is that the \'value\' attribute of the Parameter
   * definition is set to the result calculated using the InitialAssignment
   * object\'s <code>&lt;math&gt;</code> formula, and the corresponding
   * InitialAssignment is then removed from the Model.
   * 
   * @return bool @c true if the transformation was successful, 
   * @c false, otherwise.
   *
   * @note This function will check the consistency of a model before
   * attemptimg the transformation.  If the model is not valid SBML, the
   * transformation will not be performed and the function will return
   * @c false.  As part of that process, this method will check that it has
   * values for any components referred to by the <code>&lt;math&gt;</code>
   * elements of InitialAssignment objects.  In cases where not all of the
   * values have been declared (e.g., if the mathematical expression refers
   * to model entities that have no declared values), the InitialAssignment
   * in question will @em not be removed and this method will return
   * @c false.
   */ public
";


%csmethodmodifiers SBMLDocument::setLevelAndVersion(unsigned int level, unsigned int version, bool strict = true, bool ignorePackages = false) "
/**
   * Sets the SBML Level and Version of this SBMLDocument instance,
   * attempting to convert the model as needed.
   *
   * This method is the principal way in libSBML to convert models between
   * Levels and Versions of SBML.  Generally, models can be converted
   * upward without difficulty (e.g., from SBML Level&nbsp;1 to
   * Level&nbsp;2, or from an earlier Version of Level&nbsp;2 to the latest
   * Version of Level&nbsp;2).  Sometimes models can be translated downward
   * as well, if they do not use constructs specific to more advanced
   * Levels of SBML.
   *
   * Before calling this method, callers may check compatibility directly
   * using the methods SBMLDocument::checkL1Compatibility(),
   * SBMLDocument::checkL2v1Compatibility(),
   * SBMLDocument::checkL2v2Compatibility(),
   * SBMLDocument::checkL2v3Compatibility(),
   * SBMLDocument::checkL2v4Compatibility(),
   * SBMLDocument::checkL2v5Compatibility(),
   * SBMLDocument::checkL3v1Compatibility(), and
   * SBMLDocument::checkL3v2Compatibility().
   * 
   * The valid combinations of SBML Level and Version as of this release
   * of libSBML are the following: 
   * <ul>
   * <li> Level&nbsp;1 Version&nbsp;2
   * <li> Level&nbsp;2 Version&nbsp;1
   * <li> Level&nbsp;2 Version&nbsp;2
   * <li> Level&nbsp;2 Version&nbsp;3
   * <li> Level&nbsp;2 Version&nbsp;4
   * <li> Level&nbsp;2 Version&nbsp;5
   * <li> Level&nbsp;3 Version&nbsp;1
   * <li> Level&nbsp;3 Version&nbsp;2
   * </ul>
   * 
   * Strict conversion applies the additional criteria that both the
   * source and the target model must be consistent SBML.  Users can
   * control the consistency checks that are applied using the
   * SBMLDocument::setConsistencyChecksForConversion(@if java int, boolean@endif) method.  If either
   * the source or the potential target model have validation errors, the
   * conversion is not performed.  When a strict conversion is successful,
   * the underlying SBML object model is altered to reflect the new level
   * and version.  Thus, information that cannot be converted
   * (e.g. sboTerms) will be lost.
   * 
   * @param level the desired SBML Level.
   *  
   * @param version the desired Version within the SBML Level.
   *
   * @param strict boolean indicating whether to check consistency
   * of both the source and target model when performing
   * conversion (defaults to <code> true </code>).
   *
   * @param ignorePackages boolean indicating whether the presence of
   * packages should be ignored by the conversion routine 
   * (defaults to <code> false </code>).
   *
   * @return @c true if the level and version of the document were
   * successfully set to the requested values (which may have required
   * conversion of the model), @c false otherwise.
   *
   * @note Calling this method will not @em necessarily lead to a successful
   * conversion.  If the conversion fails, it will be logged in the error
   * list associated with this SBMLDocument.  Callers should consult
   * getNumErrors() to find out if the conversion succeeded without
   * problems.  For conversions from Level&nbsp;2 to Level&nbsp;1, callers
   * can also check the Level of the model after calling this method to
   * find out whether it is Level&nbsp;1.  (If the conversion to
   * Level&nbsp;1 failed, the Level of this model will be left unchanged.)
   *
   * @ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif
   * 
   * @see checkL1Compatibility()
   * @see checkL2v1Compatibility()
   * @see checkL2v2Compatibility()
   * @see checkL2v3Compatibility()
   * @see checkL2v4Compatibility()
   * @see checkL2v5Compatibility()
   * @see checkL3v1Compatibility()
   */ public
";


%csmethodmodifiers SBMLDocument::updateSBMLNamespace(const std::string& package, unsigned int level, unsigned int version) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::setModel(const Model* m) "
/**
   * Sets the Model for this SBMLDocument to a copy of the given Model.
   *
   * @param m the new Model to use.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   *
   * @see createModel()
   * @see getModel()
   */ public
";


%csmethodmodifiers SBMLDocument::createModel(const std::string& sid = "") "
/**
   * Creates a new Model inside this SBMLDocument, and returns a pointer to
   * it.
   *
   * In SBML Level&nbsp;2, the use of an identifier on a Model object is
   * optional.  This method takes an optional argument, @p sid, for setting
   * the identifier.  If not supplied, the identifier attribute on the
   * Model instance is not set.
   *
   * @param sid the identifier of the new Model to create.
   *
   * @ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif
   *
   * @see getModel()
   * @see SBMLDocument::setModel(@if java Model@endif)
   */ public
";


%csmethodmodifiers SBMLDocument::setLocationURI(const std::string& uri) "
/**
   * Sets the location of this SBMLDocument.
   *
   * Called automatically when readSBMLFromFile is used, but may be set
   * manually as well.
   */ public
";


%csmethodmodifiers SBMLDocument::getLocationURI() const "
/**
   * Get the location of this SBMLDocument.
   *
   * If this document was read from a file or had its location set manually,
   * that filename or set location will be returned, otherwise, an empty
   * string is returned.
   */ public
";


%csmethodmodifiers SBMLDocument::getLocationURI "
/**
   * Get the location of this SBMLDocument.
   *
   * If this document was read from a file or had its location set manually,
   * that filename or set location will be returned, otherwise, an empty
   * string is returned.
   */ public
";


%csmethodmodifiers SBMLDocument::setConsistencyChecks(SBMLErrorCategory_t category, bool apply) "
/**
   * Controls the consistency checks that are performed when
   * SBMLDocument::checkConsistency() is called.
   *
   * This method works by adding or subtracting consistency checks from the
   * set of all possible checks that SBMLDocument::checkConsistency() knows
   * how to perform.  This method may need to be called multiple times in
   * order to achieve the desired combination of checks.  The first
   * argument (@p category) in a call to this method indicates the category
   * of consistency/error checks that are to be turned on or off, and the
   * second argument (@p apply, a boolean) indicates whether to turn it on
   * (value of @c true) or off (value of @c false).
   *
   * @if clike
   * The possible categories (values to the argument @p category) are the
   * set of values from the enumeration #SBMLErrorCategory_t.
   * The following are the possible choices:
   * @endif@if java
   * The possible categories (values to the argument @p category) are the
   * set of constants whose names begin with the characters <code>LIBSBML_CAT_</code>
   * in the interface class {@link libsbmlConstants}.
   * The following are the possible choices:
   * @endif@if python 
   * The possible categories (values to the argument @p category) are the
   * set of constants whose names begin with the characters <code>LIBSBML_CAT_</code>
   * in the interface class @link libsbml libsbml@endlink.
   * The following are the possible choices:
   * @endif
   * <ul>
   * <li> @link libsbml#LIBSBML_CAT_GENERAL_CONSISTENCY LIBSBML_CAT_GENERAL_CONSISTENCY@endlink:
   * Correctness and consistency of specific SBML language constructs.
   * Performing this set of checks is highly recommended.  With respect to
   * the SBML specification, these concern failures in applying the
   * validation rules numbered 2xxxx in the Level&nbsp;2
   * Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.
   * 
   * <li> @link libsbml#LIBSBML_CAT_IDENTIFIER_CONSISTENCY LIBSBML_CAT_IDENTIFIER_CONSISTENCY@endlink:
   * Correctness and consistency of identifiers used for model entities.  An
   * example of inconsistency would be using a species identifier in a
   * reaction rate formula without first having declared the species.  With
   * respect to the SBML specification, these concern failures in applying
   * the validation rules numbered 103xx in the Level&nbsp;2
   * Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.
   * 
   * <li> @link libsbml#LIBSBML_CAT_UNITS_CONSISTENCY LIBSBML_CAT_UNITS_CONSISTENCY@endlink:
   * Consistency of measurement units associated with quantities in a model.
   * With respect to the SBML specification, these concern failures in
   * applying the validation rules numbered 105xx in the Level&nbsp;2
   * Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.
   * 
   * <li> @link libsbml#LIBSBML_CAT_MATHML_CONSISTENCY LIBSBML_CAT_MATHML_CONSISTENCY@endlink:
   * Syntax of MathML constructs.  With respect to the SBML specification,
   * these concern failures in applying the validation rules numbered 102xx
   * in the Level&nbsp;2 Versions&nbsp;2&ndash;4 and Level&nbsp;3
   * Versions&nbsp;1&ndash;2 specifications.
   * 
   * <li> @link libsbml#LIBSBML_CAT_SBO_CONSISTENCY LIBSBML_CAT_SBO_CONSISTENCY@endlink:
   * Consistency and validity of %SBO identifiers (if any) used in the model.
   * With respect to the SBML specification, these concern failures in
   * applying the validation rules numbered 107xx in the Level&nbsp;2
   * Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.
   * 
   * <li> @link libsbml#LIBSBML_CAT_OVERDETERMINED_MODEL LIBSBML_CAT_OVERDETERMINED_MODEL@endlink:
   * Static analysis of whether the system of equations implied by a model is
   * mathematically overdetermined.  With respect to the SBML specification,
   * this is validation rule #10601 in the Level&nbsp;2
   * Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.
   * 
   * <li> @link libsbml#LIBSBML_CAT_MODELING_PRACTICE LIBSBML_CAT_MODELING_PRACTICE@endlink:
   * Additional checks for recommended good modeling practice. (These are
   * tests performed by libSBML and do not have equivalent SBML validation
   * rules.)  </ul>
   * 
   * <em>By default, all validation checks are applied</em> to the model in
   * an SBMLDocument object @em unless
   * SBMLDocument::setConsistencyChecks(@if java int categ, boolean onoff@endif)
   * is called to indicate that only a subset should be applied.  Further,
   * this default (i.e., performing all checks) applies separately to
   * <em>each new SBMLDocument object</em> created.  In other words, each
   * time a model is read using SBMLReader::readSBML(@if java String filename@endif),
   * SBMLReader::readSBMLFromString(@if java String xml@endif),
   * or the global functions readSBML() and readSBMLFromString(), a new
   * SBMLDocument is created and for that document, a call to
   * SBMLDocument::checkConsistency() will default to applying all possible checks.
   * Calling programs must invoke
   * SBMLDocument::setConsistencyChecks(@if java int categ, boolean onoff@endif)
   * for each such new model if they wish to change the consistency checks
   * applied.
   * 
   * @param category a value drawn from @if clike #SBMLErrorCategory_t@else
   * the set of SBML error categories@endif indicating the
   * consistency checking/validation to be turned on or off.
   *
   * @param apply a boolean indicating whether the checks indicated by
   * @p category should be applied or not.
   *
   * @see SBMLDocument::checkConsistency()
   */ public
";


%csmethodmodifiers SBMLDocument::setConsistencyChecksForConversion(SBMLErrorCategory_t category, bool apply) "
/**
   * Controls the consistency checks that are performed when
   * SBMLDocument::setLevelAndVersion(@if java long, long, boolean@endif) is called.
   *
   * This method works by adding or subtracting consistency checks from the
   * set of all possible checks that may be performed to avoid conversion
   * to or from an invalid document.  This method may need to be called 
   * multiple times in
   * order to achieve the desired combination of checks.  The first
   * argument (@p category) in a call to this method indicates the category
   * of consistency/error checks that are to be turned on or off, and the
   * second argument (@p apply, a boolean) indicates whether to turn it on
   * (value of @c true) or off (value of @c false).
   *
   * @if clike
   * The possible categories (values to the argument @p category) are the
   * set of values from the enumeration #SBMLErrorCategory_t.
   * The following are the possible choices:
   * @endif@if java
   * The possible categories (values to the argument @p category) are the
   * set of constants whose names begin with the characters <code>LIBSBML_CAT_</code>
   * in the interface class {@link libsbmlConstants}.
   * The following are the possible choices:
   * @endif@if python 
   * The possible categories (values to the argument @p category) are the
   * set of constants whose names begin with the characters <code>LIBSBML_CAT_</code>
   * in the interface class @link libsbml libsbml@endlink.
   * The following are the possible choices:
   * @endif
   * <ul>
   * <li> @link libsbml#LIBSBML_CAT_GENERAL_CONSISTENCY LIBSBML_CAT_GENERAL_CONSISTENCY@endlink:
   * Correctness and consistency of specific SBML language constructs.
   * Performing this set of checks is highly recommended.  With respect to
   * the SBML specification, these concern failures in applying the
   * validation rules numbered 2xxxx in the Level&nbsp;2
   * Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.
   * 
   * <li> @link libsbml#LIBSBML_CAT_IDENTIFIER_CONSISTENCY LIBSBML_CAT_IDENTIFIER_CONSISTENCY@endlink:
   * Correctness and consistency of identifiers used for model entities.  An
   * example of inconsistency would be using a species identifier in a
   * reaction rate formula without first having declared the species.  With
   * respect to the SBML specification, these concern failures in applying
   * the validation rules numbered 103xx in the Level&nbsp;2
   * Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.
   * 
   * <li> @link libsbml#LIBSBML_CAT_UNITS_CONSISTENCY LIBSBML_CAT_UNITS_CONSISTENCY@endlink:

   * Consistency of measurement units associated with quantities in a model.
   * With respect to the SBML specification, these concern failures in
   * applying the validation rules numbered 105xx in the Level&nbsp;2
   * Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.
   * 
   * <li> @link libsbml#LIBSBML_CAT_MATHML_CONSISTENCY LIBSBML_CAT_MATHML_CONSISTENCY@endlink:
   * Syntax of MathML constructs.  With respect to the SBML specification,
   * these concern failures in applying the validation rules numbered 102xx
   * in the Level&nbsp;2 Versions&nbsp;2&ndash;4 and Level&nbsp;3
   * Versions&nbsp;1&ndash;2 specifications.
   * 
   * <li> @link libsbml#LIBSBML_CAT_SBO_CONSISTENCY LIBSBML_CAT_SBO_CONSISTENCY@endlink:
   * Consistency and validity of %SBO identifiers (if any) used in the model.
   * With respect to the SBML specification, these concern failures in
   * applying the validation rules numbered 107xx in the Level&nbsp;2
   * Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.
   * 
   * <li> @link libsbml#LIBSBML_CAT_OVERDETERMINED_MODEL LIBSBML_CAT_OVERDETERMINED_MODEL@endlink:
   * Static analysis of whether the system of equations implied by a model is
   * mathematically overdetermined.  With respect to the SBML specification,
   * this is validation rule #10601 in the Level&nbsp;2
   * Versions&nbsp;2&ndash;4 and Level&nbsp;3 Versions&nbsp;1&ndash;2 specifications.
   * 
   * <li> @link libsbml#LIBSBML_CAT_MODELING_PRACTICE LIBSBML_CAT_MODELING_PRACTICE@endlink:
   * Additional checks for recommended good modeling practice. (These are
   * tests performed by libSBML and do not have equivalent SBML validation
   * rules.)
   * </ul>
   *
   * <em>By default, all validation checks are applied</em> to the model in
   * an SBMLDocument object @em unless
   * SBMLDocument::setConsistencyChecks(@if java int, boolean@endif)
   * is called to indicate that only a subset should be applied.  Further,
   * this default (i.e., performing all checks) applies separately to
   * <em>each new SBMLDocument object</em> created.  In other words, each
   * time a model is read using SBMLReader::readSBML(@if java String@endif),
   * SBMLReader::readSBMLFromString(@if java String@endif),
   * or the global functions readSBML() and readSBMLFromString(), a new
   * SBMLDocument is created and for that document, a call to
   * SBMLDocument::checkConsistency() will default to applying all possible checks.
   * Calling programs must invoke
   * SBMLDocument::setConsistencyChecks(@if java int, boolean@endif)
   * for each such new model if they wish to change the consistency checks
   * applied.
   * 
   * @param category a value drawn from @if clike #SBMLErrorCategory_t@else
   * the set of SBML error categories@endif indicating the consistency
   * checking/validation to be turned on or off.
   *
   * @param apply a boolean indicating whether the checks indicated by
   * @p category should be applied or not.
   *
   * @see SBMLDocument::setLevelAndVersion(@if java long, long, boolean@endif)
   */ public
";


%csmethodmodifiers SBMLDocument::checkConsistency "
/**
   * Performs consistency checking and validation on this SBML document.
   *
   * If this method returns a nonzero value (meaning, one or more
   * consistency checks have failed for SBML document), the failures may be
   * due to warnings @em or errors.  Callers should inspect the severity
   * flag in the individual SBMLError objects returned by
   * SBMLDocument::getError(@if java long@endif) to determine the nature of the failures.
   *
   * @return the number of failed checks (errors) encountered.
   *
   * @see SBMLDocument::checkInternalConsistency()
   */ public
";


%csmethodmodifiers SBMLDocument::checkConsistencyWithStrictUnits "
/**
   * Performs consistency checking and validation on this SBML document
   * using the ultra strict units validator that assumes that there
   * are no hidden numerical conversion factors.
   *
   * If this method returns a nonzero value (meaning, one or more
   * consistency checks have failed for SBML document), the failures may be
   * due to warnings @em or errors.  Callers should inspect the severity
   * flag in the individual SBMLError objects returned by
   * SBMLDocument::getError(@if java long@endif) to determine the nature of the failures.
   *
   * @return the number of failed checks (errors) encountered.
   *
   * @see SBMLDocument::checkInternalConsistency()
   */ public
";


%csmethodmodifiers SBMLDocument::validateSBML "
/**
   * Performs consistency checking and validation on this SBML document.
   *
   * If this method returns a nonzero value (meaning, one or more
   * consistency checks have failed for SBML document), the failures may be
   * due to warnings @em or errors.  Callers should inspect the severity
   * flag in the individual SBMLError objects returned by
   * SBMLDocument::getError(@if java long@endif) to determine the nature of the failures.
   *
   * @note unlike checkConsistency this method will write the document
   *       in order to determine all errors for the document. This will 
   *       also clear the error log. 
   *
   * @return the number of failed checks (errors) encountered.
   *
   * @see SBMLDocument::checkConsistency()
   */ public
";


%csmethodmodifiers SBMLDocument::checkInternalConsistency "
/**
   * Performs consistency checking on libSBML\'s internal representation of 
   * an SBML Model.
   *
   * Callers should query the results of the consistency check by calling
   * SBMLDocument::getError(@if java long@endif).
   *
   * @return the number of failed checks (errors) encountered.
   *
   * The distinction between this method and
   * SBMLDocument::checkConsistency() is that this method reports on
   * fundamental syntactic and structural errors that violate the XML
   * Schema for SBML; by contrast, SBMLDocument::checkConsistency()
   * performs more elaborate model verifications and also validation
   * according to the validation rules written in the appendices of the
   * SBML Level&nbsp;2 Versions&nbsp;2&ndash;4 specification documents.
   * 
   * @see SBMLDocument::checkConsistency()
   */ public
";


%csmethodmodifiers SBMLDocument::checkL1Compatibility(bool inConversion = false) "
/**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;1 and can be converted
   * to Level&nbsp;1.
   *
   * Callers should query the results of the consistency check by calling
   * SBMLDocument::getError(@if java long@endif).
   *
   * @return the number of failed checks (errors) encountered.
   */ public
";


%csmethodmodifiers SBMLDocument::checkL2v1Compatibility(bool inConversion = false) "
/**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;2 Version&nbsp;1 and can
   * be converted to Level&nbsp;2 Version&nbsp;1.
   *
   * Callers should query the results of the consistency check by calling
   * SBMLDocument::getError(@if java long@endif).
   *
   * @return the number of failed checks (errors) encountered.
   */ public
";


%csmethodmodifiers SBMLDocument::checkL2v2Compatibility(bool inConversion = false) "
/**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;2 Version&nbsp;2 and can
   * be converted to Level&nbsp;2 Version&nbsp;2.
   *
   * Callers should query the results of the consistency check by calling
   * SBMLDocument::getError(@if java long@endif).
   *
   * @return the number of failed checks (errors) encountered.
   */ public
";


%csmethodmodifiers SBMLDocument::checkL2v3Compatibility(bool inConversion = false) "
/**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;2 Version&nbsp;3 and can
   * be converted to Level&nbsp;2 Version&nbsp;3.
   *
   * Callers should query the results of the consistency check by calling
   * SBMLDocument::getError(@if java long@endif).
   *
   * @return the number of failed checks (errors) encountered.
   */ public
";


%csmethodmodifiers SBMLDocument::checkL2v4Compatibility "
/**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;2 Version&nbsp;4 and can
   * be converted to Level&nbsp;2 Version&nbsp;4.
   *
   * Callers should query the results of the consistency check by calling
   * SBMLDocument::getError(@if java long@endif).
   *
   * @return the number of failed checks (errors) encountered.
   */ public
";


%csmethodmodifiers SBMLDocument::checkL2v5Compatibility "
/**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;2 Version&nbsp;5 and can
   * be converted to Level&nbsp;2 Version&nbsp;5.
   *
   * Callers should query the results of the consistency check by calling
   * SBMLDocument::getError(@if java long@endif).
   *
   * @return the number of failed checks (errors) encountered.
   */ public
";


%csmethodmodifiers SBMLDocument::checkL3v1Compatibility "
/**
   * Performs a set of consistency checks on the document to establish
   * whether it is compatible with SBML Level&nbsp;3 Version&nbsp;1 and can
   * be converted to Level&nbsp;3 Version&nbsp;1.
   *
   * Callers should query the results of the consistency check by calling
   * SBMLDocument::getError(@if java long@endif).
   *
   * @return the number of failed checks (errors) encountered.
   */ public
";


%csmethodmodifiers SBMLDocument::checkL3v2Compatibility "
/**
  * Performs a set of consistency checks on the document to establish
  * whether it is compatible with SBML Level&nbsp;3 Version&nbsp;2 and can
  * be converted to Level&nbsp;3 Version&nbsp;2.
  *
  * Callers should query the results of the consistency check by calling
  * SBMLDocument::getError(@if java long@endif).
  *
  * @return the number of failed checks (errors) encountered.
  */ public
";


%csmethodmodifiers SBMLDocument::getError "
/**
   * Returns the nth error or warning encountered during parsing,
   * consistency checking, or attempted translation of this model.
   *
   * Callers can use method XMLError::getSeverity() on the result to assess
   * the severity of the problem.  The possible severity levels range from
   * informational messages to fatal errors.
   *
   * @return the error or warning indexed by integer @p n, or return
   * @c null if <code>n &gt; (getNumErrors() - 1)</code>.
   *
   * @param n the integer index of the error sought.
   *
   * @see SBMLDocument::getNumErrors()
   */ public
";


%csmethodmodifiers SBMLDocument::getErrorWithSeverity "
/**
   * Returns the nth error or warning with the given severity
   * encountered during parsing, consistency checking, or attempted
   * translation of this model.
   *
   * @return the error or warning indexed by integer @p n, or return
   * @c null if <code>n &gt; (getNumErrors(severity) - 1)</code>.
   *
   * @param n the integer index of the error sought.
   * @param severity the severity of the error sought.
   *
   * @see SBMLDocument::getNumErrors()
   */ public
";


%csmethodmodifiers SBMLDocument::getNumErrors() const "
/**
   * Returns the number of errors or warnings encountered during parsing,
   * consistency checking, or attempted translation of this model.
   *
   * @return the number of errors or warnings encountered.
   *
   * @see SBMLDocument::getError(unsigned int n)
   */ public
";


%csmethodmodifiers SBMLDocument::getNumErrors "
/**
   * Returns the number of errors or warnings encountered with the given 
   * severity during parsing,
   * consistency checking, or attempted translation of this model.
   *
   * @param severity the severity of the error sought.
   *
   * @return the number of errors or warnings encountered.
   *
   * @see SBMLDocument::getError(unsigned int n)
   */ public
";


%csmethodmodifiers SBMLDocument::printErrors(std::ostream& stream = std::cerr) const "
/**
   * Prints all the errors or warnings encountered trying to parse,
   * check, or translate this SBML document.
   *
   * It prints the text to the stream given by the optional parameter @p
   * stream.  If no parameter is given, it prints the output to the
   * standard error stream.
   *
   * If no errors have occurred, i.e., <code>getNumErrors() == 0</code>, no
   * output will be sent to the stream.
   *
   * The format of the output is:
   * @verbatim
   N error(s):
     line NNN: (id) message
 @endverbatim
   *
   * @param stream the ostream or ostringstream object indicating where
   * the output should be printed.
   *
   * @ifnot hasDefaultArgs @htmlinclude warn-default-args-in-docs.html @endif
   *
   * @see getNumErrors()
   * @see getErrorLog()
   * @see SBMLDocument::getError(unsigned int n)
   */ public
";


%csmethodmodifiers SBMLDocument::printErrors "
/**
    * Prints all the errors or warnings with the given severity encountered 
    * trying to parse, check, or translate this SBML document.
    *
    * It prints the text to the stream given by the parameter @p
    * stream.  
    *
    * If no errors have occurred, i.e., <code>getNumErrors(severity) == 0</code>, no
    * output will be sent to the stream.
    *
    * The format of the output is:
    * @verbatim
    N error(s):
      line NNN: (id) message
@endverbatim
    *
    * @param stream the ostream or ostringstream object indicating where
    * the output should be printed.
    * @param severity of the errors sought.
    *
    * @see getNumErrors(unsigned int severity)
    * @see getErrorLog()
    * @see SBMLDocument::getErrorWithSeverity(unsigned int n, unsigned int severity)
    */ public
";


%csmethodmodifiers SBMLDocument::setSBMLDocument(SBMLDocument* d) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBMLDocument::convert(const ConversionProperties& props) "
/**
   * Converts this document using the converter that best matches
   * the given conversion properties. 
   * 
   * @param props the conversion properties to use.
   * 
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   * @li @link libsbml#LIBSBML_CONV_CONVERSION_NOT_AVAILABLE LIBSBML_CONV_CONVERSION_NOT_AVAILABLE@endlink
   */ public new
";


%csmethodmodifiers SBMLDocument::enablePackageInternal(const std::string& pkgURI,const std::string& pkgPrefix, bool flag) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBMLDocument::getTypeCode() const "
/**
   * Returns the libSBML type code for this %SBML object.
   * 
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for this object:
   * @link libsbml#SBML_DOCUMENT SBML_DOCUMENT@endlink (default).
   *
   *
 * @warning <span class=\'warning\'>The specific integer values of the possible
 * type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
 * packages,  To fully identify the correct code, <strong>it is necessary to
 * invoke both getPackageName() and getTypeCode()</strong> (or 
 * ListOf::getItemTypeCode()).</span>
 *
 *
   *
   * @see SBMLDocument::getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers SBMLDocument::getElementName() const "
/**
   * Returns the XML element name of this object, which for SBMLDocument,
   * is always @c \'sbml\'.
   * 
   * @return the name of this element, i.e., @c \'sbml\'.
   */ public new
";


%csmethodmodifiers SBMLDocument::getErrorLog "
/**
   * Returns the list of errors or warnings logged during parsing, 
   * consistency checking, or attempted translation of this model.
   * 
   * @return the SBMLErrorLog used for this SBMLDocument.
   * 
   * @see SBMLDocument::getNumErrors()
   */ public
";


%csmethodmodifiers SBMLDocument::getErrorLog() const "
/**
   * Returns a constant pointer to the list of errors or warnings 
   * logged during parsing, consistency checking, or attempted translation 
   * of this model.
   * 
   * @return the SBMLErrorLog used for this SBMLDocument.
   * 
   * @see SBMLDocument::getNumErrors()
   */ public
";


%csmethodmodifiers SBMLDocument::getNamespaces() const "
/**
   * Returns a list of XML Namespaces associated with the XML content
   * of this SBML document.
   * 
   * @return the XML Namespaces associated with this SBML object.
   */ public new
";


%csmethodmodifiers SBMLDocument::enableDefaultNS(const std::string& package, bool flag) "
/**
   * Set/unset default namespace to each top-level element defined in the
   * given package extension.
   *
   * This works by adding a <code>xmlns=&quot;...&quot;</code> attribute.  No
   * prefix will be written when writing elements defined in the given
   * package extension if @c true is given as second argument.
   *
   * @param package the name or URI of the package extension.
   * @param flag boolean value to indicate whether to write a namespace
   * prefix.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_PKG_UNKNOWN_VERSION LIBSBML_PKG_UNKNOWN_VERSION@endlink
   */ public
";


%csmethodmodifiers SBMLDocument::isEnabledDefaultNS(const std::string& package) "
/**
   * Returns @c true if a default namespace is added to each top-level
   * element defined in the given package extension, otherwise returns
   * @c false.
   *
   * This basically checks if the attribute
   * <code>xmlns=&quot;...&quot;</code> is present.
   *   
   * @param package the name or URI of the package extension.
   *
   * @return a boolean indicating whether the given package\'s default namespace is enabled.
   */ public
";


%csmethodmodifiers SBMLDocument::setPackageRequired(const std::string& package, bool flag) "
/**
   * Sets the <code>required</code> attribute value of the given package
   * extension.
   *
   * @param package the name or URI of the package extension.
   * @param flag Boolean value indicating whether the package is required.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_PKG_UNKNOWN_VERSION LIBSBML_PKG_UNKNOWN_VERSION@endlink
   */ public
";


%csmethodmodifiers SBMLDocument::getPackageRequired(const std::string& package) "
/**
   * Returns the <code>required</code> attribute of the given package
   * extension.
   *
   * @param package the name or URI of the package extension.
   *
   * @return Boolean flag indicating whether the package is flagged as
   * being required.
   */ public
";


%csmethodmodifiers SBMLDocument::isSetPackageRequired(const std::string& package) "
/**
   * Returns @c true if the required attribute of the given package extension
   * is defined, otherwise returns @c false.
   *
   * @param package the name or URI of the package extension.
   *
   * @return a Boolean indicating whether the package\'s \'required\' flag is set.
   */ public
";


%csmethodmodifiers SBMLDocument::isIgnoredPackage(const std::string& pkgURI) "
/**
   * Returns @c true if the given package extension is one of an ignored
   * packages, otherwise returns @c false.
   *
   * An ignored package is one that is defined to be used in this SBML
   * document, but the package is not enabled in this copy of libSBML.
   *
   * @param pkgURI the URI of the package extension.
   *
   * @return a Boolean, @c true if the package is being ignored and
   * @c false otherwise.
   */ public
";


%csmethodmodifiers SBMLDocument::isDisabledIgnoredPackage(const std::string& pkgURI) "
/**
   * Returns @c true if the given package extension is one of an ignored
   * packages that has been disabled, otherwise returns @c false.
   *
   * An ignored package is one that is defined to be used in this SBML
   * document, but the package is not enabled in this copy of libSBML.
   * It may have been disabled to avoid reproducing the package
   * information when writing out the file.
   *
   * @param pkgURI the URI of the package extension.
   *
   * @return a Boolean, @c true if the package is being ignored and
   * @c false otherwise.
   */ public
";


%csmethodmodifiers SBMLDocument::setPkgRequired(const std::string& package, bool flag) "
/**
   * Sets the value of the <code>required</code> attribute for the given
   * package.
   *
   * @param package the name or URI of the package extension.
   * @param flag a Boolean value.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_PKG_UNKNOWN_VERSION LIBSBML_PKG_UNKNOWN_VERSION@endlink
   *
   * @deprecated Replaced in libSBML 5.2.0 by
   * setPackageRequired(@if java String, boolean@endif)
   */ public
";


%csmethodmodifiers SBMLDocument::getPkgRequired(const std::string& package) "
/**
   * Returns the <code>required</code> attribute of the given package
   * extension.
   *
   * @param package the name or URI of the package extension.
   *
   * @return a Boolean value indicating whether the package is flagged as
   * being required in this SBML document.
   *
   * @deprecated Replaced in libSBML 5.2.0 by
   * getPackageRequired(@if java String@endif)
   */ public
";


%csmethodmodifiers SBMLDocument::isSetPkgRequired(const std::string& package) "
/**
   * Returns @c true if the required attribute of the given package extension
   * is defined, otherwise returns @c false.
   *
   * @param package the name or URI of the package extension.
   *
   * @return a Boolean value.
   *
   * @deprecated Replaced in libSBML 5.2.0 by
   * isSetPackageRequired(@if java String@endif)
   */ public
";


%csmethodmodifiers SBMLDocument::isIgnoredPkg(const std::string& pkgURI) "
/**
   * Returns @c true if the given package extension is one of ignored
   * packages, otherwise returns @c false.
   *
   * An ignored package is one that is defined to be used in this SBML
   * document, but the package is not enabled in this copy of libSBML.
   *
   * @param pkgURI the URI of the package extension.
   *
   * @return a boolean indicating whether the given package is being ignored.
   *
   * @deprecated Replaced in libSBML 5.2.0 by
   * isIgnoredPackage(@if java String@endif)
   */ public
";


%csmethodmodifiers SBMLDocument::getElementPosition() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::writeElements(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBMLDocument::getApplicableValidators() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::getConversionValidators() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::setApplicableValidators(unsigned char appl) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::setConversionValidators(unsigned char appl) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::getNumValidators() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::clearValidators "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::addValidator(const SBMLValidator* validator) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::getValidator(unsigned int index) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::addUnknownPackageRequired(const std::string& pkgURI, const std::string& prefix, bool flag) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::hasUnknownPackage(const std::string& pkgURI) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::getNumUnknownPackages() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::getUnknownPackageURI(int index) const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::getUnknownPackagePrefix(int index) const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SBMLDocument::createObject(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBMLDocument::addExpectedAttributes(ExpectedAttributes& attributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBMLDocument::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBMLDocument::writeAttributes(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBMLDocument::writeXMLNS(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SBMLDocument::setInvalidLevel "
/** */ /* libsbml-internal */ public
";


%typemap(csimports) FunctionDefinition "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html A user-defined function in an SBML model.
 *
 * The FunctionDefinition structure associates an identifier with a
 * function definition.  This identifier can then be used as the function
 * called in subsequent MathML content elsewhere in an SBML model.
 * 
 * FunctionDefinition has one required attribute, \'id\', to give the
 * function a unique identifier by which other parts of an SBML model
 * definition can refer to it.  A FunctionDefinition instance can also have
 * an optional \'name\' attribute of type @c string.  Identifiers and names
 * must be used according to the guidelines described in the %SBML
 * specification (e.g., Section 3.3 in the Level 2 Version 4
 * specification).
 * 
 * FunctionDefinition has a \'math\' subelement containing a MathML
 * expression defining the function body.  In SBML Level&nbsp;2 and SBML
 * Level&nbsp;3 Version&nbsp;1, that \'math\' subelement is required;
 * in SBML Level&nbsp;3 Version&nbsp;2, this restriction was relaxed,
 * making the \'math\' subelement optional.  The content of this element can
 * only be a MathML \'lambda\' element.  The \'lambda\' element must begin with
 * zero or more \'bvar\' elements, followed by any other of the elements in
 * the MathML subset allowed in SBML Level 2 @em except \'lambda\' (i.e., a
 * \'lambda\' element cannot contain another \'lambda\' element).  This is the
 * only place in SBML where a \'lambda\' element can be used.  The function
 * defined by a FunctionDefinition is only available for use in other
 * MathML elements that @em follow the FunctionDefinition definition in the
 * model.  (These restrictions prevent recursive and mutually-recursive
 * functions from being expressed.)
 *
 * A further restriction on the content of \'math\' is that it cannot contain
 * references to variables other than the variables declared to the
 * \'lambda\' itself.  That is, the contents of MathML \'ci\' elements inside
 * the body of the \'lambda\' can only be the variables declared by its
 * \'bvar\' elements, or the identifiers of other FunctionDefinition
 * instances in the model.  This means must be written so that all
 * variables or parameters used in the MathML content are passed to them
 * via their function parameters.  In SBML Level&nbsp;2, this restriction
 * applies also to the MathML @c csymbol elements for @em time and @em
 * delay; in SBML Level&nbsp;3, it additionally applies to the @c csymbol
 * element for @em avogadro.
 *
 * In SBML Level&nbsp;3 Version&nbsp;2, if no math element is present in 
 * the FunctionDefinition, the function has no mathematical meaning 
 * defined in SBML Level&nbsp;3 Core. This situation may arise when models 
 * are incomplete, or when additional meanings are provided by an SBML 
 * Level&nbsp;3 package.
 *
 * @note Function definitions (also informally known as user-defined
 * functions) were introduced in SBML Level 2.  They have purposefully
 * limited capabilities.  A function cannot reference parameters or other
 * model quantities outside of itself; values must be passed as parameters
 * to the function.  Moreover, recursive and mutually-recursive functions
 * are not permitted.  The purpose of these limitations is to balance power
 * against complexity of implementation.  With the restrictions as they
 * are, function definitions could be implemented as textual
 * substitutions---they are simply macros.  Software implementations
 * therefore do not need the full function-definition machinery typically
 * associated with programming languages.
 * <br><br>
 * Another important point to note is FunctionDefinition does not
 * have a separate attribute for defining the units of the value returned
 * by the function.  The units associated with the function\'s return value,
 * when the function is called from within MathML expressions elsewhere in
 * SBML, are simply the overall units of the expression in
 * FunctionDefinition\'s \'math\' subelement when applied to the arguments
 * supplied in the call to the function.  Ascertaining these units requires
 * performing dimensional analysis on the expression.  (Readers may wonder
 * why there is no attribute.  The reason is that having a separate
 * attribute for declaring the units would not only be redundant, but also
 * lead to the potential for having conflicting information.  In the case
 * of a conflict between the declared units and those of the value actually
 * returned by the function, the only logical resolution rule would be to
 * assume that the correct units are those of the expression anyway.)
 * 
 * 
 *
 */
"


%typemap(csimports) ListOfFunctionDefinitions "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html A list of FunctionDefinition objects.
 *
 *
 * 
 * The various ListOf___ @if conly structures @else classes@endif in SBML
 * are merely containers used for organizing the main components of an SBML
 * model.  In libSBML\'s implementation, ListOf___
 * @if conly data structures @else classes@endif are derived from the
 * intermediate utility @if conly structure @else class@endif ListOf, which
 * is not defined by the SBML specifications but serves as a useful
 * programmatic construct.  ListOf is itself is in turn derived from SBase,
 * which provides all of the various ListOf___
 * @if conly data structures @else classes@endif with common features
 * defined by the SBML specification, such as \'metaid\' attributes and
 * annotations.
 *
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 *
 * @htmlinclude listof-illustration.html
 *
 * SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
 * Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
 * Version&nbsp;2 allows
 * containers to contain zero or more of the relevant object, instead of 
 * requiring at least one.  As such, libsbml will write out an 
 * otherwise-empty ListOf___ element that has any optional attribute set 
 * (such as \'id\' or \'metaid\'), that has an optional child (such 
 * as a \'notes\' or \'annotation\'), or that has attributes or children set
 * from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
 * any other children.
 *
 * Readers may wonder about the motivations for using the ListOf___
 * containers in SBML.  A simpler approach in XML might be to place the
 * components all directly at the top level of the model definition.  The
 * choice made in SBML is to group them within XML elements named after
 * %ListOf<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from SBase means that software tools can add information @em about
 * the lists themselves into each list container\'s \'annotation\'.
 *
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 *
 * @if conly
 * @note In the C API for libSBML, functions that in other language APIs
 * would be inherited by the various ListOf___ structures not shown in the
 * pages for the individual ListOf___\'s.  Instead, the functions are defined
 * on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
 * the many common functions available for manipulating ListOf___
 * structures</strong>.  The documentation for the individual ListOf___
 * structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
 * all of the functionality available. @endif
 *
 *
 */
"


%csmethodmodifiers FunctionDefinition::FunctionDefinition(unsigned int level, unsigned int version) "
/**
   * Creates a new FunctionDefinition using the given SBML @p level and @p version
   * values.
   *
   * @param level a long integer, the SBML Level to assign to this FunctionDefinition.
   *
   * @param version a long integer, the SBML Version to assign to this
   * FunctionDefinition.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers FunctionDefinition::FunctionDefinition(SBMLNamespaces* sbmlns) "
/**
   * Creates a new FunctionDefinition using the given SBMLNamespaces object
   * @p sbmlns.
   *
   *
 * 
 * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
 * information.  It is used to communicate the SBML Level, Version, and (in
 * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
 * common approach to using libSBML\'s SBMLNamespaces facilities is to create an
 * SBMLNamespaces object somewhere in a program once, then hand that object
 * as needed to object constructors that accept SBMLNamespaces as arguments.
 *
 * 
   *
   * @param sbmlns an SBMLNamespaces object.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers FunctionDefinition::FunctionDefinition(const FunctionDefinition& orig) "
/**
   * Copy constructor; creates a copy of this FunctionDefinition.
   *
   * @param orig the object to copy.
   */ public
";


%csmethodmodifiers FunctionDefinition::accept(SBMLVisitor& v) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers FunctionDefinition::clone() const "
/**
   * Creates and returns a deep copy of this FunctionDefinition object.
   *
   * @return the (deep) copy of this FunctionDefinition object.
   */ public new
";


%csmethodmodifiers FunctionDefinition::getId() const "
/**
   * Returns the value of the \'id\' attribute of this FunctionDefinition.
   *
   * @note Because of the inconsistent behavior of this function with 
   * respect to assignments and rules, it is now recommended to
   * use the getIdAttribute() function instead.
   *
   *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
   *
   * @return the id of this FunctionDefinition.
   *
   * @see getIdAttribute()
   * @see setIdAttribute(string sid)
   * @see isSetIdAttribute()
   * @see unsetIdAttribute()
   */ public new
";


%csmethodmodifiers FunctionDefinition::getName() const "
/**
   * Returns the value of the \'name\' attribute of this FunctionDefinition object.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 *
 * @return the name of this SBML object, or the empty string if not set or unsettable.
 *
 * @see getIdAttribute()
 * @see isSetName()
 * @see setName(string sid)
 * @see unsetName()
 * 
 *
   */ public new
";


%csmethodmodifiers FunctionDefinition::getMath() const "
/**
   * Get the mathematical formula of this FunctionDefinition.
   *
   * @return an ASTNode, the value of the \'math\' subelement of this
   * FunctionDefinition, or @c null if the math is not set.
   */ public new
";


%csmethodmodifiers FunctionDefinition::isSetId() const "
/**
   * Predicate returning @c true if this
   * FunctionDefinition\'s \'id\' attribute is set.
   *
   *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 *
 * @return @c true if the \'id\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @note Because of the inconsistent behavior of this function with
 * respect to assignments and rules, it is recommended that callers
 * use isSetIdAttribute() instead.
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see unsetIdAttribute()
 * @see isSetIdAttribute()
 *
 *
   */ public new
";


%csmethodmodifiers FunctionDefinition::isSetName() const "
/**
   * Predicate returning @c true if this
   * FunctionDefinition\'s \'name\' attribute is set.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 * @return @c true if the \'name\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @see getName()
 * @see setName(string sid)
 * @see unsetName()
 *
 *
   */ public new
";


%csmethodmodifiers FunctionDefinition::isSetMath() const "
/**
   * Predicate returning @c true if this
   * FunctionDefinition\'s \'math\' subelement contains a value.
   * 
   * @return @c true if the \'math\' for this FunctionDefinition is set,
   * @c false otherwise.
   */ public
";


%csmethodmodifiers FunctionDefinition::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this FunctionDefinition.
   *
   *
 * 
 * The string @p sid is copied.
 *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 * 
 * @param sid the string to use as the identifier of this object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see isSetIdAttribute()
 * @see unsetIdAttribute()
 * 
 *
   */ public new
";


%csmethodmodifiers FunctionDefinition::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this FunctionDefinition.
   *
   *
 * 
 *
 * The string in @p name is copied.
 *
 * @param name the new name for the SBML object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 *
 *
   */ public new
";


%csmethodmodifiers FunctionDefinition::setMath(const ASTNode* math) "
/**
   * Sets the \'math\' subelement of this FunctionDefinition to the Abstract
   * Syntax Tree given in @p math.
   *
   * @param math an AST containing the mathematical expression to
   * be used as the formula for this FunctionDefinition.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   */ public new
";


%csmethodmodifiers FunctionDefinition::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this FunctionDefinition.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
 *
 * @see getName()
 * @see setName(string sid)
 * @see isSetName()
 *
 *
   */ public new
";


%csmethodmodifiers FunctionDefinition::getArgument "
/**
   * Get the <code>n</code>th argument to this function.
   *
   * Callers should first find out the number of arguments to the function
   * by calling getNumArguments().
   *
   * @param n an integer index for the argument sought.
   * 
   * @return the nth argument (bound variable) passed to this
   * FunctionDefinition.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see getNumArguments()
   */ public
";


%csmethodmodifiers FunctionDefinition::getArgument(const std::string& name) const "
/**
   * Get the argument named @p name to this FunctionDefinition.
   *
   * @param name the exact name (case-sensitive) of the sought-after
   * argument.
   * 
   * @return the argument (bound variable) having the given name, or @c null if
   * no such argument exists.
   */ public
";


%csmethodmodifiers FunctionDefinition::getBody() const "
/**
   * Get the mathematical expression that is the body of this
   * FunctionDefinition object.
   * 
   * @return the body of this FunctionDefinition as an Abstract Syntax
   * Tree, or @c null if no body is defined.
   */ public
";


%csmethodmodifiers FunctionDefinition::getBody "
/**
   * Get the mathematical expression that is the body of this
   * FunctionDefinition object.
   * 
   * @return the body of this FunctionDefinition as an Abstract Syntax
   * Tree, or @c null if no body is defined.
   */ public
";


%csmethodmodifiers FunctionDefinition::isSetBody() const "
/**
   * Predicate returning @c true if the body of this
   * FunctionDefinition has set.
   *
   * @return @c true if the body of this FunctionDefinition is 
   * set, @c false otherwise.
   */ public
";


%csmethodmodifiers FunctionDefinition::getNumArguments() const "
/**
   * Get the number of arguments (bound variables) taken by this
   * FunctionDefinition.
   *
   * @return the number of arguments (bound variables) that must be passed
   * to this FunctionDefinition.
   */ public
";


%csmethodmodifiers FunctionDefinition::getTypeCode() const "
/**
   * Returns the libSBML type code for this %SBML object.
   * 
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for this object:
   * @link libsbml#SBML_FUNCTION_DEFINITION SBML_FUNCTION_DEFINITION@endlink (default).
   *
   *
 * @warning <span class=\'warning\'>The specific integer values of the possible
 * type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
 * packages,  To fully identify the correct code, <strong>it is necessary to
 * invoke both getPackageName() and getTypeCode()</strong> (or 
 * ListOf::getItemTypeCode()).</span>
 *
 *
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers FunctionDefinition::getElementName() const "
/**
   * Returns the XML element name of this object, which for
   * FunctionDefinition, is always @c \'functionDefinition\'.
   * 
   * @return the name of this element, i.e., @c \'functionDefinition\'.
   */ public new
";


%csmethodmodifiers FunctionDefinition::writeElements(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers FunctionDefinition::hasRequiredAttributes() const "
/**
   * Predicate returning @c true if
   * all the required attributes for this FunctionDefinition object
   * have been set.
   *
   * The required attributes for a FunctionDefinition object are:
   * @li \'id\'
   *
   * @return @c true if the required attributes have been set, @c false
   * otherwise.
   */ public new
";


%csmethodmodifiers FunctionDefinition::hasRequiredElements() const "
/**
   * Predicate returning @c true if
   * all the required elements for this FunctionDefinition object
   * have been set.
   *
   * @note The required elements for a FunctionDefinition object are:
   * @li \'math\' inSBML Level&nbsp;2 and Level&nbsp;3 Version&nbsp;1.  
   *     (In SBML Level&nbsp;3 Version&nbsp;2+, it is no longer required.)
   *
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */ public new
";


%csmethodmodifiers FunctionDefinition::renameUnitSIdRefs(const std::string& oldid, const std::string& newid) "
/**
   * Renames all the @c UnitSIdRef attributes on this element.
   *
   *
 * 
 * In SBML, unit definitions have identifiers of type <code>UnitSId</code>.  In
 * SBML Level&nbsp;3, an explicit data type called <code>UnitSIdRef</code> was
 * introduced for attribute values that refer to <code>UnitSId</code> values; in
 * previous Levels of SBML, this data type did not exist and attributes were
 * simply described to as \'referring to a unit identifier\', but the effective
 * data type was the same as <code>UnitSIdRef</code> in Level&nbsp;3.  These and
 * other methods of libSBML refer to the type <code>UnitSIdRef</code> for all
 * Levels of SBML, even if the corresponding SBML specification did not
 * explicitly name the data type.
 *
 *
   *
   * This method works by looking at all unit identifier attribute values
   * (including, if appropriate, inside mathematical formulas), comparing the
   * unit identifiers to the value of @p oldid.  If any matches are found,
   * the matching identifiers are replaced with @p newid.  The method does
   * @em not descend into child elements.
   * 
   * @param oldid the old identifier.
   * @param newid the new identifier.
   */ public new
";


%csmethodmodifiers FunctionDefinition::readOtherXML(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers FunctionDefinition::addExpectedAttributes(ExpectedAttributes& attributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers FunctionDefinition::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers FunctionDefinition::readL2Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers FunctionDefinition::readL3Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers FunctionDefinition::writeAttributes(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOfFunctionDefinitions::ListOfFunctionDefinitions(unsigned int level, unsigned int version) "
/**
   * Creates a new ListOfFunctionDefinitions object.
   *
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   *
   * @param level the SBML Level.
   * 
   * @param version the Version within the SBML Level.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfFunctionDefinitions::ListOfFunctionDefinitions(SBMLNamespaces* sbmlns) "
/**
   * Creates a new ListOfFunctionDefinitions object.
   *
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the SBMLNamespaces object in @p
   * sbmlns.
   *
   * @param sbmlns an SBMLNamespaces object that is used to determine the
   * characteristics of the ListOfFunctionDefinitions object to be created.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfFunctionDefinitions::clone() const "
/**
   * Creates and returns a deep copy of this ListOfFunctionDefinitions object.
   *
   * @return the (deep) copy of this ListOfFunctionDefinitions object.
   */ public new
";


%csmethodmodifiers ListOfFunctionDefinitions::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this ListOf
   * (i.e., FunctionDefinition objects, if the list is non-empty).
   *
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for the objects contained in this ListOf:
   * @link libsbml#SBML_FUNCTION_DEFINITION SBML_FUNCTION_DEFINITION@endlink (default).
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers ListOfFunctionDefinitions::getElementName() const "
/**
   * Returns the XML element name of this object.
   *
   * For ListOfFunctionDefinitions, the XML element name is
   * @c \'listOfFunctionDefinitions\'.
   * 
   * @return the name of this element, i.e., @c \'listOfFunctionDefinitions\'.
   */ public new
";


%csmethodmodifiers ListOfFunctionDefinitions::get(unsigned int n) "
/**
   * Get a FunctionDefinition from the ListOfFunctionDefinitions.
   *
   * @param n the index number of the FunctionDefinition to get.
   * 
   * @return the nth FunctionDefinition in this ListOfFunctionDefinitions.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfFunctionDefinitions::get "
/**
   * Get a FunctionDefinition from the ListOfFunctionDefinitions.
   *
   * @param n the index number of the FunctionDefinition to get.
   * 
   * @return the nth FunctionDefinition in this ListOfFunctionDefinitions.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfFunctionDefinitions::get(const std::string& sid) "
/**
   * Get a FunctionDefinition from the ListOfFunctionDefinitions
   * based on its identifier.
   *
   * @param sid a string representing the identifier 
   * of the FunctionDefinition to get.
   * 
   * @return FunctionDefinition in this ListOfFunctionDefinitions
   * with the given @p sid or @c null if no such
   * FunctionDefinition exists.
   *
   * @see get(unsigned int n)
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfFunctionDefinitions::get(const std::string& sid) const "
/**
   * Get a FunctionDefinition from the ListOfFunctionDefinitions
   * based on its identifier.
   *
   * @param sid a string representing the identifier 
   * of the FunctionDefinition to get.
   * 
   * @return FunctionDefinition in this ListOfFunctionDefinitions
   * with the given @p sid or @c null if no such
   * FunctionDefinition exists.
   *
   * @see get(unsigned int n)
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfFunctionDefinitions::remove(unsigned int n) "
/**
   * Removes the nth item from this ListOfFunctionDefinitions items and returns a pointer to
   * it.
   *
   * The caller owns the returned item and is responsible for deleting it.
   *
   * @param n the index of the item to remove.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfFunctionDefinitions::remove(const std::string& sid) "
/**
   * Removes item in this ListOfFunctionDefinitions items with the given identifier.
   *
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier @p sid, then
   * @c null is returned.
   *
   * @param sid the identifier of the item to remove.
   *
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public new
";


%csmethodmodifiers ListOfFunctionDefinitions::getElementPosition() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOfFunctionDefinitions::createObject(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers UnitKind_equals(UnitKind_t uk1, UnitKind_t uk2) "
/**
 * Tests for logical equality between two given <code>UNIT_KIND_</code>
 * code values.
 *
 * This function behaves exactly like C\'s <code>==</code> operator, except
 * for the following two cases:
 * <ul>
  * <li>@link libsbml#UNIT_KIND_LITER UNIT_KIND_LITER@endlink <code>==</code> @link libsbml#UNIT_KIND_LITRE UNIT_KIND_LITRE@endlink
 * <li>@link libsbml#UNIT_KIND_METER UNIT_KIND_METER@endlink <code>==</code> @link libsbml#UNIT_KIND_METRE UNIT_KIND_METRE@endlink
 * </ul>
 *
 * In the two cases above, C equality comparison would yield @c 0 (false)
 * (because each of the above is a distinct enumeration value), but
 * this function returns @c true.
 *
 * @param uk1 a <code>UNIT_KIND_</code> value.
 * @param uk2 a second <code>UNIT_KIND_</code> value to compare to @p uk1.
 *
 * @return @c 1 (true) if @p uk1 is logically equivalent to @p
 * uk2, @c 0 (false) otherwise.
 *
 * @note For more information about the libSBML unit codes, please refer to
 * the class documentation for Unit.
 *
 * @if conly
 * @memberof Unit_t
 * @endif
 */ public
";


%csmethodmodifiers UnitKind_forName(const char *name) "
/**
 * Converts a text string naming a kind of unit to its corresponding
 * libSBML <code>UNIT_KIND_</code> constant/enumeration value.
 *
 * @param name a string, the name of a predefined base unit in SBML.
 * 
 * @return @if clike a value from UnitKind_t corresponding to the given
 * string @p name (determined in a case-insensitive manner).
 * @endif@if python a value the set of <code>UNIT_KIND_</code> codes
 * defined in class @link libsbml libsbml@endlink, corresponding to the
 * string @p name (determined in a case-insensitive
 * manner).@endif@if java a value the set of <code>UNIT_KIND_</code> codes
 * defined in class {@link libsbmlConstants}, corresponding to the string
 * @p name (determined in a case-insensitive manner).@endif
 *
 * @note For more information about the libSBML unit codes, please refer to
 * the class documentation for Unit.
 *
 * @if conly
 * @memberof Unit_t
 * @endif
 */ public
";


%csmethodmodifiers UnitKind_toString(UnitKind_t uk) "
/**
 * Converts a unit code to a text string equivalent.
 *
 * @param uk @if clike a value from the UnitKind_t enumeration
 * @endif@if python a value from the set of <code>UNIT_KIND_</code> codes
 * defined in the class @link libsbml libsbml@endlink
 * @endif@if java a value from the set of <code>UNIT_KIND_</code> codes
 * defined in the class {@link libsbmlConstants}
 * @endif
 *
 * @return the name corresponding to the given unit code.
 *
 * @note For more information about the libSBML unit codes, please refer to
 * the class documentation for Unit.
 * 
 * @warning The string returned is a static data value.  The caller does not
 * own the returned string and is therefore not allowed to modify it.
 *
 * @if conly
 * @memberof Unit_t
 * @endif
 */ public
";


%csmethodmodifiers UnitKind_isValidUnitKindString(const char *str, unsigned int level, unsigned int version) "
/**
 * Predicate for testing whether a given string corresponds to a
 * predefined libSBML unit code.
 *
 * @param str a text string naming a base unit defined by SBML.
 * @param level the Level of SBML.
 * @param version the Version within the Level of SBML.
 *
 * @return 1 (true) if string is the name of a valid
 * <code>UNIT_KIND_</code> value, @c 0 (false) otherwise.
 *
 * @note For more information about the libSBML unit codes, please refer to
 * the class documentation for Unit.
 *
 * @if conly
 * @memberof Unit_t
 * @endif
 */ public
";


%typemap(csimports) Unit "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html A single unit referenced in an SBML <em>unit definition</em>.
 *
 * The SBML unit definition facility uses two classes of objects,
 * UnitDefinition and Unit.  The approach to defining units in %SBML is
 * compositional; for example, <em>meter second<sup> &ndash;2</sup></em> is
 * constructed by combining a Unit object representing <em>meter</em> with
 * another Unit object representing <em>second<sup> &ndash;2</sup></em>.
 * The combination is wrapped inside a UnitDefinition, which provides for
 * assigning an identifier and optional name to the combination.  The
 * identifier can then be referenced from elsewhere in a model.  Thus, the
 * UnitDefinition class is the container, and Unit instances are placed
 * inside UnitDefinition instances.
 *
 * A Unit has four attributes named \'kind\', \'exponent\', \'scale\'
 * and \'multiplier\'.  It represents a (possibly transformed) reference to a
 * base unit.  The attribute \'kind\' on Unit indicates the chosen base unit.
 * Its value must be one of the text strings listed below; this list
 * corresponds to SBML Level&nbsp;3:
 *
 *
 * 
<table border=\'0\' class=\'centered text-table width80 normal-font code\'
       style=\'border: none !important\'>
<tr>
<td>ampere</td><td>farad</td><td>joule</td><td>lux</td><td>radian</td><td>volt</td>
</tr>
<tr>
<td>avogadro</td><td>gram</td><td>katal</td><td>metre</td><td>second</td><td>watt</td>
</tr>
<tr>
<td>becquerel</td><td>gray</td><td>kelvin</td><td>mole</td><td>siemens</td><td>weber</td>
</tr>
<tr>
<td>candela</td><td>henry</td><td>kilogram</td><td>newton</td><td>sievert</td>
</tr>
<tr>
<td>coulomb</td><td>hertz</td><td>litre</td><td>ohm</td><td>steradian</td>
</tr>
<tr>
<td>dimensionless</td><td>item</td><td>lumen</td><td>pascal</td><td>tesla</td>
</tr>
</table>
 *
 *
 *
 * A few small differences exist between the Level&nbsp;3 list of base
 * units and the list defined in other Level/Version combinations of SBML.
 * Specifically, Levels of SBML before Level&nbsp;3 do not define
 * @c avogadro; conversely, Level&nbsp;2 Version&nbsp;1 defines @c Celsius,
 * and Level&nbsp;1 defines @c celsius, @c meter, and @c liter, none of
 * which are available in Level&nbsp;3.  In libSBML, each of the predefined
 * base unit names is represented by an enumeration value @if clike in
 * #UnitKind_t@else whose name begins with the characters
 * <code>UNIT_KIND_</code>@endif, discussed in a separate section below.
 *
 * The attribute named \'exponent\' on Unit represents an exponent on the
 * unit.  In SBML Level&nbsp;2, the attribute is optional and has a default
 * value of @c 1 (one); in SBML Level&nbsp;3, the attribute is mandatory
 * and there is no default value.  A Unit also has an attribute
 * called \'scale\'; its value must be an integer exponent for a power-of-ten
 * multiplier used to set the scale of the unit.  For example, a unit
 * having a \'kind\' value of @c gram and a \'scale\' value of @c -3 signifies
 * 10<sup>&nbsp;&ndash;3</sup> * gram, or milligrams.  In SBML
 * Level&nbsp;2, the attribute is optional and has a default value of @c 0
 * (zero), because 10<sup> 0</sup> = 1; in SBML Level&nbsp;3, the attribute
 * is mandatory and has no default value.  Lastly, the attribute named
 * \'multiplier\' can be used to multiply the unit by a real-numbered factor;
 * this enables the definition of units that are not power-of-ten multiples
 * of SI units.  For instance, a multiplier of 0.3048 could be used to
 * define @c foot as a measure of length in terms of a @c metre.  The
 * \'multiplier\' attribute is optional in SBML Level&nbsp;2, where it has a
 * default value of @c 1 (one); in SBML Level&nbsp;3, the attribute is
 * mandatory and has no default value.
 *
 * @if clike
 * <h3><a class=\'anchor\' name=\'UnitKind_t\'>UnitKind_t</a></h3>
 * @else
 * <h3><a class=\'anchor\' name=\'UnitKind_t\'>%Unit identification codes</a></h3>
 * @endif
 *
 * As discussed above, SBML defines a set of base units which serves as the
 * starting point for new unit definitions.  This set of base units
 * consists of the SI units and a small number of additional convenience
 * units.
 * 
 * @if clike Until SBML Level&nbsp;2 Version&nbsp;3, there
 * existed a data type in the SBML specifications called @c UnitKind,
 * enumerating the possible SBML base units.  Although SBML Level&nbsp;2
 * Version&nbsp;3 removed this type from the language specification,
 * libSBML maintains the corresponding enumeration type #UnitKind_t as a
 * convenience and as a way to provide backward compatibility to previous
 * SBML Level/Version specifications.  (The removal in SBML Level&nbsp;2
 * Version&nbsp;3 of the enumeration @c UnitKind was also accompanied by
 * the redefinition of the data type @c UnitSId to include the previous
 * @c UnitKind values as reserved symbols in the @c UnitSId space.  This
 * change has no net effect on permissible models, their representation or
 * their syntax.  The purpose of the change in the SBML specification was
 * simply to clean up an inconsistency about the contexts in which these
 * values were usable.)
 * @endif@if java In SBML Level&nbsp;2 Versions before
 * Version&nbsp;3, there existed an enumeration of units called
 * @c UnitKind.  In Version&nbsp;3, this enumeration was removed and the
 * identifier class @c UnitSId redefined to include the previous
 * @c UnitKind values as reserved symbols.  This change has no net effect on
 * permissible models, their representation or their syntax.  The purpose
 * of the change in the SBML specification was simply to clean up an
 * inconsistency about the contexts in which these values were usable.
 * However, libSBML maintains UnitKind in the form of of a set of static
 * integer constants whose names begin with the characters
 * <code>UNIT_KIND_</code>.  These constants are defined in the class
 * <code><a href=\'libsbml.libsbml.html\'>libsbmlConstants</a></code>.
 * @endif@if python In SBML Level&nbsp;2 Versions before
 * Version&nbsp;3, there existed an enumeration of units called
 * @c UnitKind.  In Version&nbsp;3, this enumeration was removed and the
 * identifier class @c UnitSId redefined to include the previous
 * @c UnitKind values as reserved symbols.  This change has no net effect on
 * permissible models, their representation or their syntax.  The purpose
 * of the change in the SBML specification was simply to clean up an
 * inconsistency about the contexts in which these values were usable.
 * However, libSBML maintains UnitKind in the form of of a set of static
 * integer constants whose names begin with the characters
 * <code>UNIT_KIND_</code>.  These constants are defined in the class
 * @link libsbml libsbml@endlink.
 * @endif
 *
 * As a consequence of the fact that libSBML supports models in all Levels
 * and Versions of SBML, libSBML\'s set of @c UNIT_KIND_ values is a union
 * of all the possible base unit names defined in the different SBML
 * specifications.  However, not every base unit is allowed in every
 * Level+Version combination of SBML.  Note in particular the following
 * exceptions:
 * <ul>
 * <li> The alternate spelling @c \'meter\' is included in
 * addition to the official SI spelling @c \'metre\'.  This spelling is only
 * permitted in SBML Level&nbsp;1 models.
 *
 * <li> The alternate spelling @c \'liter\' is included in addition to the
 * official SI spelling @c \'litre\'.  This spelling is only permitted in
 * SBML Level&nbsp;1 models.
 *
 * <li> The unit @c \'Celsius\' is included because of its presence in
 * specifications of SBML prior to SBML Level&nbsp;2 Version&nbsp;2.
 *
 * <li> The unit @c avogadro was introduced in SBML Level&nbsp;3, and
 * is only permitted for use in SBML Level&nbsp;3 models.
 * </ul>
 *
 * @if clike The table below lists the symbols defined in the
 * @c UnitKind_t enumeration, and their
 * meanings. @else The table below lists the unit
 * constants defined in libSBML, and their meanings. @endif
 *
 * @htmlinclude unitkind-table.html
 * 
 * 
 */
"


%typemap(csimports) ListOfUnits "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html A list of Unit objects.
 * 
 * ListOfUnits is entirely contained within UnitDefinition.
 *
 *
 * 
 * The various ListOf___ @if conly structures @else classes@endif in SBML
 * are merely containers used for organizing the main components of an SBML
 * model.  In libSBML\'s implementation, ListOf___
 * @if conly data structures @else classes@endif are derived from the
 * intermediate utility @if conly structure @else class@endif ListOf, which
 * is not defined by the SBML specifications but serves as a useful
 * programmatic construct.  ListOf is itself is in turn derived from SBase,
 * which provides all of the various ListOf___
 * @if conly data structures @else classes@endif with common features
 * defined by the SBML specification, such as \'metaid\' attributes and
 * annotations.
 *
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 *
 * @htmlinclude listof-illustration.html
 *
 * SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
 * Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
 * Version&nbsp;2 allows
 * containers to contain zero or more of the relevant object, instead of 
 * requiring at least one.  As such, libsbml will write out an 
 * otherwise-empty ListOf___ element that has any optional attribute set 
 * (such as \'id\' or \'metaid\'), that has an optional child (such 
 * as a \'notes\' or \'annotation\'), or that has attributes or children set
 * from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
 * any other children.
 *
 * Readers may wonder about the motivations for using the ListOf___
 * containers in SBML.  A simpler approach in XML might be to place the
 * components all directly at the top level of the model definition.  The
 * choice made in SBML is to group them within XML elements named after
 * %ListOf<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from SBase means that software tools can add information @em about
 * the lists themselves into each list container\'s \'annotation\'.
 *
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 *
 * @if conly
 * @note In the C API for libSBML, functions that in other language APIs
 * would be inherited by the various ListOf___ structures not shown in the
 * pages for the individual ListOf___\'s.  Instead, the functions are defined
 * on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
 * the many common functions available for manipulating ListOf___
 * structures</strong>.  The documentation for the individual ListOf___
 * structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
 * all of the functionality available. @endif
 *
 *
 */
"


%csmethodmodifiers Unit::Unit(unsigned int level, unsigned int version) "
/**
   * Creates a new Unit using the given SBML @p level and @p version
   * values.
   *
   * @param level a long integer, the SBML Level to assign to this Unit.
   *
   * @param version a long integer, the SBML Version to assign to this
   * Unit.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers Unit::Unit(SBMLNamespaces* sbmlns) "
/**
   * Creates a new Unit using the given SBMLNamespaces object
   * @p sbmlns.
   *
   *
 * 
 * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
 * information.  It is used to communicate the SBML Level, Version, and (in
 * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
 * common approach to using libSBML\'s SBMLNamespaces facilities is to create an
 * SBMLNamespaces object somewhere in a program once, then hand that object
 * as needed to object constructors that accept SBMLNamespaces as arguments.
 *
 *
   * 
   * @param sbmlns an SBMLNamespaces object.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers Unit::Unit(const Unit& orig) "
/**
   * Copy constructor; creates a copy of this Unit.
   *
   * @param orig the object to copy.
   */ public
";


%csmethodmodifiers Unit::accept(SBMLVisitor& v) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Unit::clone() const "
/**
   * Creates and returns a deep copy of this Unit object.
   *
   * @return the (deep) copy of this Unit object.
   */ public new
";


%csmethodmodifiers Unit::initDefaults "
/**
   * Initializes the fields of this Unit object to \'typical\' default
   * values.
   *
   * The SBML Unit component has slightly different aspects and default
   * attribute values in different SBML Levels and Versions.  This method
   * sets the values to certain common defaults, based mostly on what they
   * are in SBML Level&nbsp;2.  Specifically:
   * <ul>
   * <li> Sets attribute \'exponent\' to @c 1
   * <li> Sets attribute \'scale\' to @c 0
   * <li> Sets attribute \'multiplier\' to @c 1.0
   * </ul>
   *
   * The \'kind\' attribute is left unchanged.
   */ public
";


%csmethodmodifiers Unit::getKind() const "
/**
   * Returns the \'kind\' of Unit this is.
   * 
   * @if clike
   * @return the value of the \'kind\' attribute of this Unit as a
   * value from the <a class=\'el\' href=\'#UnitKind_t\'>UnitKind_t</a> enumeration.
   * @endif@if java
   * @return the value of the \'kind\' attribute of this Unit as a
   * value from the set of constants whose names begin
   * with <code>UNIT_KIND_</code> defined in the class
   * <code><a href=\'libsbml.libsbml.html\'>libsbmlConstants</a></code>.
   * @endif@if python
   * @return the value of the \'kind\' attribute of this Unit as a
   * value from the set of constants whose names begin
   * with <code>UNIT_KIND_</code> defined in the class
   * @link libsbml libsbml@endlink.
   * @endif
   */ public
";


%csmethodmodifiers Unit::getExponent() const "
/**
   * Returns the value of the \'exponent\' attribute of this unit.
   * 
   * @return the \'exponent\' value of this Unit, as an integer.
   */ public
";


%csmethodmodifiers Unit::getExponentAsDouble() const "
/**
   * Returns the value of the \'exponent\' attribute of this unit.
   * 
   * @return the \'exponent\' value of this Unit, as a double.
   */ public
";


%csmethodmodifiers Unit::getScale() const "
/**
   * Returns the value of the \'scale\' attribute of this unit.
   * 
   * @return the \'scale\' value of this Unit, as an integer.
   */ public
";


%csmethodmodifiers Unit::getMultiplier() const "
/**
   * Returns the value of the \'multiplier\' attribute of this Unit.
   * 
   * @return the \'multiplier\' value of this Unit, as a double.
   */ public
";


%csmethodmodifiers Unit::getOffset() const "
/**
   * Returns the value of the \'offset\' attribute of this Unit.
   * 
   * @return the \'offset\' value of this Unit, as a double.
   *
   *
 * @warning <span class=\'warning\'>The \'offset\' attribute is only available in
 * SBML Level&nbsp;2 Version&nbsp;1.  This attribute is not present in SBML
 * Level&nbsp;2 Version&nbsp;2 or above.  When producing SBML models using
 * these later specifications, modelers and software tools need to account
 * for units with offsets explicitly.  The %SBML specification document
 * offers a number of suggestions for how to achieve this.  LibSBML methods
 * such as this one related to \'offset\' are retained for compatibility with
 * earlier versions of SBML Level&nbsp;2, but their use is strongly
 * discouraged.</span>
 *
   */ public
";


%csmethodmodifiers Unit::isAmpere() const "
/**
   * Predicate for testing whether this Unit is of the kind @c ampere.
   * 
   * @return @c true if the kind of this Unit is @c ampere, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isAvogadro() const "
/**
   * Predicate for testing whether this Unit is of the kind @c avogadro.
   * 
   * @return @c true if the kind of this Unit is @c avogadro, @c false
   * otherwise.
   *
   * @note The unit @c avogadro was introduced in SBML Level&nbsp;3, and
   * is only permitted for use in SBML Level&nbsp;3 models.
   */ public
";


%csmethodmodifiers Unit::isBecquerel() const "
/**
   * Predicate for testing whether this Unit is of the kind @c becquerel
   *
   * @return @c true if the kind of this Unit is @c becquerel, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isCandela() const "
/**
   * Predicate for testing whether this Unit is of the kind @c candela
   *
   * @return @c true if the kind of this Unit is @c candela, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isCelsius() const "
/**
   * Predicate for testing whether this Unit is of the kind @c Celsius
   *
   * @return @c true if the kind of this Unit is @c Celsius, @c false
   * otherwise. 
   *
   * @warning <span class=\'warning\'>The predefined unit @c Celsius was
   * removed from the list of predefined units in SBML Level&nbsp;2
   * Version&nbsp;2 at the same time that the \'offset\' attribute was removed
   * from Unit definitions.  LibSBML methods such as this one related to
   * @c Celsius are retained in order to support SBML Level&nbsp;2
   * Version&nbsp;1, but their use is strongly discouraged.</span>
   */ public
";


%csmethodmodifiers Unit::isCoulomb() const "
/**
   * Predicate for testing whether this Unit is of the kind @c coulomb
   *
   * @return @c true if the kind of this Unit is @c coulomb, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isDimensionless() const "
/**
   * Predicate for testing whether this Unit is of the kind
   * @c dimensionless.
   *
   * @return @c true if the kind of this Unit is @c dimensionless, @c false
   * otherwise.
   */ public
";


%csmethodmodifiers Unit::isFarad() const "
/**
   * Predicate for testing whether this Unit is of the kind @c farad
   *
   * @return @c true if the kind of this Unit is @c farad, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isGram() const "
/**
   * Predicate for testing whether this Unit is of the kind @c gram
   *
   * @return @c true if the kind of this Unit is @c gram, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isGray() const "
/**
   * Predicate for testing whether this Unit is of the kind @c gray
   *
   * @return @c true if the kind of this Unit is @c gray, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isHenry() const "
/**
   * Predicate for testing whether this Unit is of the kind @c henry
   *
   * @return @c true if the kind of this Unit is @c henry, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isHertz() const "
/**
   * Predicate for testing whether this Unit is of the kind @c hertz
   *
   * @return @c true if the kind of this Unit is @c hertz, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isItem() const "
/**
   * Predicate for testing whether this Unit is of the kind @c item
   *
   * @return @c true if the kind of this Unit is @c item, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isJoule() const "
/**
   * Predicate for testing whether this Unit is of the kind @c joule
   *
   * @return @c true if the kind of this Unit is @c joule, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isKatal() const "
/**
   * Predicate for testing whether this Unit is of the kind @c katal
   *
   * @return @c true if the kind of this Unit is @c katal, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isKelvin() const "
/**
   * Predicate for testing whether this Unit is of the kind @c kelvin
   *
   * @return @c true if the kind of this Unit is @c kelvin, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isKilogram() const "
/**
   * Predicate for testing whether this Unit is of the kind @c kilogram
   *
   * @return @c true if the kind of this Unit is @c kilogram, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isLitre() const "
/**
   * Predicate for testing whether this Unit is of the kind @c litre
   *
   * @return @c true if the kind of this Unit is @c litre or \'liter\',
   * @c false otherwise.
   */ public
";


%csmethodmodifiers Unit::isLumen() const "
/**
   * Predicate for testing whether this Unit is of the kind @c lumen
   *
   * @return @c true if the kind of this Unit is @c lumen, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isLux() const "
/**
   * Predicate for testing whether this Unit is of the kind @c lux
   *
   * @return @c true if the kind of this Unit is @c lux, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isMetre() const "
/**
   * Predicate for testing whether this Unit is of the kind @c metre
   *
   * @return @c true if the kind of this Unit is @c metre or \'meter\',
   * @c false 
   * otherwise.
   */ public
";


%csmethodmodifiers Unit::isMole() const "
/**
   * Predicate for testing whether this Unit is of the kind @c mole
   *
   * @return @c true if the kind of this Unit is @c mole, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isNewton() const "
/**
   * Predicate for testing whether this Unit is of the kind @c newton
   *
   * @return @c true if the kind of this Unit is @c newton, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isOhm() const "
/**
   * Predicate for testing whether this Unit is of the kind @c ohm
   *
   * @return @c true if the kind of this Unit is @c ohm, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isPascal() const "
/**
   * Predicate for testing whether this Unit is of the kind @c pascal
   *
   * @return @c true if the kind of this Unit is @c pascal, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isRadian() const "
/**
   * Predicate for testing whether this Unit is of the kind @c radian
   *
   * @return @c true if the kind of this Unit is @c radian, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isSecond() const "
/**
   * Predicate for testing whether this Unit is of the kind @c second
   *
   * @return @c true if the kind of this Unit is @c second, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isSiemens() const "
/**
   * Predicate for testing whether this Unit is of the kind @c siemens
   *
   * @return @c true if the kind of this Unit is @c siemens, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isSievert() const "
/**
   * Predicate for testing whether this Unit is of the kind @c sievert
   *
   * @return @c true if the kind of this Unit is @c sievert, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isSteradian() const "
/**
   * Predicate for testing whether this Unit is of the kind @c steradian
   *
   * @return @c true if the kind of this Unit is @c steradian, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isTesla() const "
/**
   * Predicate for testing whether this Unit is of the kind @c tesla
   *
   * @return @c true if the kind of this Unit is @c tesla, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isVolt() const "
/**
   * Predicate for testing whether this Unit is of the kind @c volt
   *
   * @return @c true if the kind of this Unit is @c volt, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isWatt() const "
/**
   * Predicate for testing whether this Unit is of the kind @c watt
   *
   * @return @c true if the kind of this Unit is @c watt, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isWeber() const "
/**
   * Predicate for testing whether this Unit is of the kind @c weber
   *
   * @return @c true if the kind of this Unit is @c weber, @c false
   * otherwise. 
   */ public
";


%csmethodmodifiers Unit::isSetKind() const "
/**
   * Predicate to test whether the \'kind\' attribute of this Unit is set.
   * 
   * @return @c true if the \'kind\' attribute of this Unit is set,
   * @c false otherwise.
   */ public
";


%csmethodmodifiers Unit::isSetExponent() const "
/**
   * Predicate to test whether the \'exponent\' attribute of this Unit 
   * is set.
   * 
   * @return @c true if the \'exponent\' attribute of this Unit is set, 
   * @c false otherwise.
   */ public
";


%csmethodmodifiers Unit::isSetScale() const "
/**
   * Predicate to test whether the \'scale\' attribute of this Unit 
   * is set.
   * 
   * @return @c true if the \'scale\' attribute of this Unit is set, 
   * @c false otherwise.
   */ public
";


%csmethodmodifiers Unit::isSetMultiplier() const "
/**
   * Predicate to test whether the \'multiplier\' attribute of this Unit 
   * is set.
   * 
   * @return @c true if the \'multiplier\' attribute of this Unit is set, 
   * @c false otherwise.
   */ public
";


%csmethodmodifiers Unit::isSetOffset() const "
/**
   * Predicate to test whether the \'offset\' attribute of this Unit 
   * is set.
   * 
   * @return @c true if the \'offset\' attribute of this Unit is set, 
   * @c false otherwise.
   *
   *
 * @warning <span class=\'warning\'>The \'offset\' attribute is only available in
 * SBML Level&nbsp;2 Version&nbsp;1.  This attribute is not present in SBML
 * Level&nbsp;2 Version&nbsp;2 or above.  When producing SBML models using
 * these later specifications, modelers and software tools need to account
 * for units with offsets explicitly.  The %SBML specification document
 * offers a number of suggestions for how to achieve this.  LibSBML methods
 * such as this one related to \'offset\' are retained for compatibility with
 * earlier versions of SBML Level&nbsp;2, but their use is strongly
 * discouraged.</span>
 *
   */ public
";


%csmethodmodifiers Unit::setKind(UnitKind_t kind) "
/**
   * Sets the \'kind\' attribute value of this Unit.
   *
   * @if clike
   * @param kind a value from the <a class=\'el\'
   * href=\'#UnitKind_t\'>UnitKind_t</a> enumeration.
   * @endif@if java
   * @param kind a unit identifier chosen from the set of constants whose
   * names begin with <code>UNIT_KIND_</code> in <code><a
   * href=\'libsbml.libsbml.html\'>libsbmlConstants</a></code>.
   * @endif@if python
   * @param kind a unit identifier chosen from the set of constants whose
   * names begin with <code>UNIT_KIND_</code> in @link libsbml libsbml@endlink.
   * @endif
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   */ public
";


%csmethodmodifiers Unit::setExponent(int value) "
/**
   * Sets the \'exponent\' attribute value of this Unit.
   *
   * @param value the integer to which the attribute \'exponent\' should be set.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   */ public
";


%csmethodmodifiers Unit::setExponent(double value) "
/**
   * Sets the \'exponent\' attribute value of this Unit.
   *
   * @param value the double to which the attribute \'exponent\' should be set.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   */ public
";


%csmethodmodifiers Unit::setScale(int value) "
/**
   * Sets the \'scale\' attribute value of this Unit.
   *
   * @param value the integer to which the attribute \'scale\' should be set.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   */ public
";


%csmethodmodifiers Unit::setMultiplier(double value) "
/**
   * Sets the \'multipler\' attribute value of this Unit.
   *
   * @param value the floating-point value to which the attribute
   * \'multiplier\' should be set.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   */ public
";


%csmethodmodifiers Unit::setOffset(double value) "
/**
   * Sets the \'offset\' attribute value of this Unit.
   *
   * @param value the floating point value to which the attribute \'offset\'
   * should set.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   *
 * @warning <span class=\'warning\'>The \'offset\' attribute is only available in
 * SBML Level&nbsp;2 Version&nbsp;1.  This attribute is not present in SBML
 * Level&nbsp;2 Version&nbsp;2 or above.  When producing SBML models using
 * these later specifications, modelers and software tools need to account
 * for units with offsets explicitly.  The %SBML specification document
 * offers a number of suggestions for how to achieve this.  LibSBML methods
 * such as this one related to \'offset\' are retained for compatibility with
 * earlier versions of SBML Level&nbsp;2, but their use is strongly
 * discouraged.</span>
 *
   */ public
";


%csmethodmodifiers Unit::unsetKind "
/**
   * Unsets the \'kind\' attribute value of this Unit.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   */ public
";


%csmethodmodifiers Unit::unsetExponent "
/**
   * Unsets the \'exponent\' attribute value of this Unit.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   */ public
";


%csmethodmodifiers Unit::unsetScale "
/**
   * Unsets the \'scale\' attribute value of this Unit.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   */ public
";


%csmethodmodifiers Unit::unsetMultiplier "
/**
   * Unsets the \'multipler\' attribute value of this Unit.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   */ public
";


%csmethodmodifiers Unit::unsetOffset "
/**
   * Unsets the \'offset\' attribute value of this Unit.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   *
 * @warning <span class=\'warning\'>The \'offset\' attribute is only available in
 * SBML Level&nbsp;2 Version&nbsp;1.  This attribute is not present in SBML
 * Level&nbsp;2 Version&nbsp;2 or above.  When producing SBML models using
 * these later specifications, modelers and software tools need to account
 * for units with offsets explicitly.  The %SBML specification document
 * offers a number of suggestions for how to achieve this.  LibSBML methods
 * such as this one related to \'offset\' are retained for compatibility with
 * earlier versions of SBML Level&nbsp;2, but their use is strongly
 * discouraged.</span>
 *
   */ public
";


%csmethodmodifiers Unit::getTypeCode() const "
/**
   * Returns the libSBML type code of this object instance.
   *
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for this object:
   * @link libsbml#SBML_UNIT SBML_UNIT@endlink (default).
   *
   *
 * @warning <span class=\'warning\'>The specific integer values of the possible
 * type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
 * packages,  To fully identify the correct code, <strong>it is necessary to
 * invoke both getPackageName() and getTypeCode()</strong> (or 
 * ListOf::getItemTypeCode()).</span>
 *
 *
   *
   * @see getPackageName()
   * @see getElementName()
   */ public new
";


%csmethodmodifiers Unit::getElementName() const "
/**
   * Returns the XML element name of this object, which for Unit, is
   * always @c \'unit\'.
   * 
   * @return the name of this element, i.e., @c \'unit\'. 
   */ public new
";


%csmethodmodifiers Unit::writeElements(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Unit::isBuiltIn(const std::string& name, unsigned int level) "
/**
   * Predicate to test whether a given string is the name of a built-in SBML
   * unit, depending on the SBML level, since new predefined units were added
   * between level 2 versions 1 and 2, and then all predefined units were removed
   * again in SBML Level 3.
   *
   * @param name a string to be tested against the built-in unit names.
   * @param level the level of SBML one is checking.
   *
   * @return @c true if @p name is one of @c \'substance\', @c \'volume\',
   * or @c \'time\' and the @p level is @c 1; or if @p name is one of
   * @c \'substance\', @c \'volume\', @c \'area\', @c \'length\', or @c \'time\' and
   * the @p level is @c 2; @c false otherwise (including all values when
   * @p level is @c 3).
   *
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   */ public
";


%csmethodmodifiers Unit::isUnitKind(const std::string& name, unsigned int level, unsigned int version) "
/**
   * Predicate to test whether a given string is the name of a valid
   * base unit in SBML (such as @c \'gram\' or @c \'mole\').
   *
   * This method exists because prior to SBML Level&nbsp;2 Version&nbsp;3,
   * an enumeration called @c UnitKind was defined by SBML.  This enumeration
   * was removed in SBML Level&nbsp;2 Version&nbsp;3 and its values were
   * folded into the space of values of a type called @c UnitSId.  This method
   * therefore has less significance in SBML Level&nbsp;2 Version&nbsp;3
   * and Level&nbsp;2 Version&nbsp;4, but remains for backward
   * compatibility and support for reading models in older Versions of
   * Level&nbsp;2.
   *
   * @param name a string to be tested.
   * 
   * @param level a long integer representing the SBML specification
   * Level.
   * 
   * @param version a long integer representing the SBML specification
   * Version.
   * 
   * @return @c true if name is a valid SBML UnitKind, @c false otherwise.
   *
   * @note The allowed unit names differ between SBML Levels&nbsp;1
   * and&nbsp;2 and again slightly between Level&nbsp;2 Versions&nbsp;1
   * and&nbsp;2.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   */ public
";


%csmethodmodifiers Unit::areIdentical(Unit * unit1, Unit * unit2) "
/**
   * Predicate returning @c true if two
   * Unit objects are identical.
   *
   * Two Unit objects are considered to be @em identical if they match in
   * all attributes.  (Contrast this to the method areEquivalent(@if java
   * Unit u1, %Unit u2@endif), which compares Unit objects only with respect
   * to certain attributes.)
   *
   * @param unit1 the first Unit object to compare.
   * @param unit2 the second Unit object to compare.
   *
   * @return @c true if all the attributes of unit1 are identical
   * to the attributes of unit2, @c false otherwise.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   *
   * @see @if clike areEquivalent() @else Unit::areEquivalent(Unit u1, %Unit u2) @endif
   */ public
";


%csmethodmodifiers Unit::areEquivalent(Unit * unit1, Unit * unit2) "
/**
   * Predicate returning @c true if 
   * Unit objects are equivalent.
   *
   * Two Unit objects are considered to be @em equivalent either if (1) both
   * have a \'kind\' attribute value of @c dimensionless, or (2) their \'kind\',
   * \'exponent\' and (for SBML Level&nbsp;2 Version&nbsp;1) \'offset\'
   * attribute values are equal. (Contrast this to the method
   * areIdentical(@if java Unit, %Unit@endif), which compares Unit objects with respect to all
   * attributes, not just the \'kind\' and \'exponent\'.)
   *
   * @param unit1 the first Unit object to compare.
   * @param unit2 the second Unit object to compare.
   *
   * @return @c true if the \'kind\' and \'exponent\' attributes of unit1 are
   * identical to the kind and exponent attributes of unit2, or if the kind
   * attributes of both are @c dimensionless; @c false otherwise.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   * 
   * @see @if clike areIdentical() @else Unit::areIdentical(Unit u1, %Unit u2) @endif
   */ public
";


%csmethodmodifiers Unit::removeScale(Unit * unit) "
/**
   * Manipulates the attributes of the Unit to express the unit with the 
   * value of the scale attribute reduced to zero.
   *
   * For example, 1 millimetre can be expressed as a Unit with kind=
   * @c \'metre\' multiplier=@c \'1\' scale=@c \'-3\' exponent=@c \'1\'. It can also be
   * expressed as a Unit with kind=@c \'metre\'
   * multiplier=<code>\'0.001\'</code> scale=@c \'0\' exponent=@c \'1\'.
   *
   * @param unit the Unit object to manipulate.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif This particular
 * function only does one thing irrespective of user input or 
 * object state, and thus will only return a single value:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   *
   * @see @if clike convertToSI() @else Unit::convertToSI(Unit u) @endif
   * @see @if clike merge() @else Unit::merge(Unit u1, Unit u2) @endif
   */ public
";


%csmethodmodifiers Unit::merge(Unit * unit1, Unit * unit2) "
/**
   * Merges two Unit objects with the same \'kind\' attribute value into a
   * single Unit.
   *
   * For example, the following,
   * @verbatim
 <unit kind=\'metre\' exponent=\'2\'/>
 <unit kind=\'metre\' exponent=\'1\'/>
 @endverbatim
   * would be merged to become
   * @verbatim
 <unit kind=\'metre\' exponent=\'3\'/>
 @endverbatim
   *
   * @param unit1 the first Unit object; the result of the operation is
   * left as a new version of this unit, modified in-place.  Not modified if
   * the two units have different kinds.
   * 
   * @param unit2 the second Unit object to merge with the first.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   * 
   * @see @if clike convertToSI() @else Unit::convertToSI(Unit u) @endif
   * @see @if clike removeScale() @else Unit::removeScale(Unit u) @endif
   */ public
";


%csmethodmodifiers Unit::convertToSI(const Unit * unit) "
/**
   * Returns a UnitDefinition object containing the given @p unit converted
   * to the appropriate SI unit.
   *
   * This method exists because some units can be expressed in terms of
   * others when the same physical dimension is involved.  For example, one
   * hertz is identical to 1&nbsp;sec<sup>-1</sup>, one litre is equivalent
   * to 1 cubic decametre, and so on.
   *
   * @param unit the Unit object to convert to SI.
   *
   * @return a UnitDefinition object containing the SI unit.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   *
   * @see @if clike merge() @else Unit::merge(Unit u1, Unit u2) @endif
   */ public
";


%csmethodmodifiers Unit::hasRequiredAttributes() const "
/**
   * Predicate returning @c true if
   * all the required attributes for this Unit object
   * have been set.
   *
   * The required attributes for a Unit object are:
   * @li \'kind\'
   * @li \'exponent\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * @li \'multiplier\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   * @li \'scale\' (required in SBML Level&nbsp;3; optional in Level&nbsp;2)
   *
   * @return @c true if the required attributes have been set, @c false
   * otherwise.
   */ public new
";


%csmethodmodifiers Unit::setExponentUnitChecking(double value) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::getExponentUnitChecking "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::getExponentUnitChecking() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::isUnitChecking "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::isUnitChecking() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::addExpectedAttributes(ExpectedAttributes& attributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Unit::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Unit::readL1Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::readL2Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::readL3Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::writeAttributes(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Unit::isL1UnitKind(const std::string& name) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::isL2V1UnitKind(const std::string& name) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::isL2UnitKind(const std::string& name) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::isL3UnitKind(const std::string& name) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::isExplicitlySetExponent() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::isExplicitlySetMultiplier() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::isExplicitlySetScale() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Unit::isExplicitlySetOffset() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers ListOfUnits::ListOfUnits(unsigned int level, unsigned int version) "
/**
   * Creates a new ListOfUnits object.
   *
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   *
   * @param level the SBML Level.
   * 
   * @param version the Version within the SBML Level.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfUnits::ListOfUnits(SBMLNamespaces* sbmlns) "
/**
   * Creates a new ListOfUnits object.
   *
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the SBMLNamespaces object in @p
   * sbmlns.
   *
   * @param sbmlns an SBMLNamespaces object that is used to determine the
   * characteristics of the ListOfUnits object to be created.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfUnits::clone() const "
/**
   * Creates and returns a deep copy of this ListOfUnits object.
   *
   * @return the (deep) copy of this ListOfUnits object.
   */ public new
";


%csmethodmodifiers ListOfUnits::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this ListOf
   * (i.e., Unit objects, if the list is non-empty).
   * 
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for objects contained in this list:
   * @link libsbml#SBML_UNIT SBML_UNIT@endlink (default).
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers ListOfUnits::getElementName() const "
/**
   * Returns the XML element name of this object.
   *
   * For ListOfUnits, the XML element name is @c \'listOfUnits\'.
   * 
   * @return the name of this element, i.e., @c \'listOfUnits\'.
   */ public new
";


%csmethodmodifiers ListOfUnits::get(unsigned int n) "
/**
   * Get a Unit from the ListOfUnits.
   *
   * @param n the index number of the Unit to get.
   * 
   * @return the nth Unit in this ListOfUnits.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfUnits::get "
/**
   * Get a Unit from the ListOfUnits.
   *
   * @param n the index number of the Unit to get.
   * 
   * @return the nth Unit in this ListOfUnits.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfUnits::remove(unsigned int n) "
/**
   * Removes the nth item from this ListOfUnits items and returns a pointer to
   * it.
   *
   * The caller owns the returned item and is responsible for deleting it.
   *
   * @param n the index of the item to remove.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfUnits::getElementPosition() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOfUnits::createObject(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%typemap(csimports) UnitDefinition "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html A definition of a unit used in an SBML model.
 *
 * Units of measurement may be supplied in a number of contexts in an SBML
 * model.  The SBML unit definition facility uses two classes of objects,
 * UnitDefinition and Unit.  The approach to defining units in SBML is
 * compositional; for example, <em>meter second<sup> &ndash;2</sup></em> is
 * constructed by combining a Unit object representing <em>meter</em> with
 * another Unit object representing <em>second<sup> &ndash;2</sup></em>.
 * The combination is wrapped inside a UnitDefinition, which provides for
 * assigning an identifier and optional name to the combination.  The
 * identifier can then be referenced from elsewhere in a model.  Thus, the
 * UnitDefinition class is the container, and Unit instances are placed
 * inside UnitDefinition instances.
 *
 * Two points are worth discussing in the context of SBML units.  First,
 * unit declarations in SBML models are @em optional.  The consequence of
 * this is that a model must be numerically self-consistent independently
 * of unit declarations, for the benefit of software tools that cannot
 * interpret or manipulate units.  Unit declarations in SBML are thus more
 * akin to a type of annotation; they can indicate intentions, and can be
 * used by model readers for checking the consistency of the model,
 * labeling simulation output, etc., but any transformations of values
 * implied by different units must be incorporated @em explicitly into a
 * model.
 * 
 * Second, the vast majority of situations that require new SBML unit
 * definitions involve simple multiplicative combinations of base units and
 * factors.  An example is <em>moles per litre per second</em>.  What
 * distinguishes these sorts of unit definitions from more complex ones is
 * that they may be expressed without the use of an additive offset from a
 * zero point.  The use of offsets complicates all unit definition systems,
 * yet in the domain of SBML, the real-life cases requiring offsets are few
 * (and in fact, to the best of our knowledge, only involve temperature).
 * Consequently, the SBML unit system has been consciously designed to
 * simplify implementation of unit support for the most common cases in
 * systems biology.  The cost of this simplification is to require units
 * with offsets to be handled explicitly by the modeler.
 *
 * @section unitdef-summary Summary of the UnitDefinition construct
 *
 * UnitDefinition has two attributes and one subelement.  The two
 * attributes are \'id\' and \'name\', and the subelement is ListOfUnits.
 *
 * The required attribute \'id\' and optional attribute \'name\' are both
 * strings.  The \'id\' attribute is used to give the defined unit a unique
 * identifier by which other parts of an SBML model definition can refer to
 * it.  The \'name\' attribute is intended to be used for giving the unit
 * definition an optional human-readable name.  Please see the <a
 * href=\'#unitdef-id\'>next section</a> for information about the values
 * permitted for \'id\'.
 *
 * A UnitDefinition may contain exactly one ListOfUnits, and this list
 * may contain one or more Unit definitions; see the definitions of these
 * other object classes for more information about them.  In SBML 
 * Level&nbsp;2 and SBML Level&nbsp;3 Version&nbsp;1, if the ListOfUnits
 * was present, it must have one or more Unit definitions.  In SBML
 * Level&nbsp;3 Version&nbsp;2, this restriction was relaxed, and 
 * a ListOfUnits was allowed to be empty.  In either case, if a
 * UnitDefinition had no child Unit elements, the unit was considered
 * to be undefined.
 *
 * The following
 * example illustrates a complete unit definition (when written in XML)
 * when all the pieces are combined together.  This defines \'mmls\'
 * to be millimoles per litre per second.
 * @verbatim
 <listOfUnitDefinitions>
     <unitDefinition id=\'mmls\'>
         <listOfUnits>
             <unit kind=\'mole\'   scale=\'-3\'/>
             <unit kind=\'litre\'  exponent=\'-1\'/>
             <unit kind=\'second\' exponent=\'-1\'/>
         </listOfUnits>
     </unitDefinition>
 </listOfUnitDefinitions>
 @endverbatim
 *
 * @section unitdef-id Special considerations for Unit object identifiers
 *
 * The attribute \'id\' in UnitDefinition cannot be given simply any value,
 * and the precise details of the values permitted differ slightly between
 * Levels of SBML:
 * <ul>
 *
 * <li> The \'id\' of a UnitDefinition must @em not contain a value from the
 * list of SBML\'s predefined base unit names (i.e., the strings @c gram,
 * @c litre, etc.).  In SBML Level&nbsp;3, this list consists of the
 * following:
 * 
 *
 * 
<table border=\'0\' class=\'centered text-table width80 normal-font code\'
       style=\'border: none !important\'>
<tr>
<td>ampere</td><td>farad</td><td>joule</td><td>lux</td><td>radian</td><td>volt</td>
</tr>
<tr>
<td>avogadro</td><td>gram</td><td>katal</td><td>metre</td><td>second</td><td>watt</td>
</tr>
<tr>
<td>becquerel</td><td>gray</td><td>kelvin</td><td>mole</td><td>siemens</td><td>weber</td>
</tr>
<tr>
<td>candela</td><td>henry</td><td>kilogram</td><td>newton</td><td>sievert</td>
</tr>
<tr>
<td>coulomb</td><td>hertz</td><td>litre</td><td>ohm</td><td>steradian</td>
</tr>
<tr>
<td>dimensionless</td><td>item</td><td>lumen</td><td>pascal</td><td>tesla</td>
</tr>
</table>
 *
 *
 *
 * This list of predefined base units is nearly identical in SBML
 * Level&nbsp;2 Version&nbsp;4, the exception being that Level&nbsp;2 does
 * not define @c avogadro.  SBML Level&nbsp;2 Version&nbsp;1 (and @em only
 * this Level+Version combination) provides an additional predefined unit
 * name, @c Celsius, not available in Level&nbsp;3.  Finally, SBML
 * Level&nbsp;1 Versions&nbsp;2&ndash;3 provide two more additional
 * predefined unit names, @c meter and @c liter.  This is explained in
 * somewhat greater detail in the description of the Unit class.
 *
 * <ul>
 * <li> In SBML Level&nbsp;2 (all Versions), there is an additional set of
 * reserved identifiers: @c substance, @c volume, @c area, @c length, and
 * @c time.  Using one of these values for the attribute \'id\' of a
 * UnitDefinition has the effect of redefining the model-wide default units
 * for the corresponding quantities.  The list of special unit names in
 * SBML Level&nbsp;2 is given in the table below:
 * </ul>
 *
 *   @htmlinclude predefined-units.html
 *
 * Also, SBML Level&nbsp;2 imposes two limitations on redefining the
 * predefined unit @c substance, @c volume, @c area, @c length, and
 * @c time: (1) The UnitDefinition of a predefined SBML unit can only contain
 * a single Unit object within it.  (2) The value of the \'kind\' attribute
 * in a Unit instance must be drawn from one of the values in the second
 * column of the table above.
 *
 * The special unit names @c substance, @c volume, @c area, @c length, and
 * @c time are not defined by SBML Level&nbsp;3, which uses a different
 * approach to setting model-wide inherited units.
 *
 *
 * @section sbml-units-limits Further comments about SBML\'s unit definition system
 * 
 * The vast majority of modeling situations requiring new SBML unit
 * definitions involve simple multiplicative combinations of base units and
 * factors.  An example of this might be <em>moles per litre per
 * second</em>.  What distinguishes these sorts of simpler unit definitions
 * from more complex ones is that they may be expressed without the use of
 * an additive offset from a zero point.  The use of offsets complicates
 * all unit definition systems, yet in the domain of SBML the real-life
 * cases requiring offsets are few (and in fact, to the best of our
 * knowledge, only involve temperature).  Consequently, the SBML unit
 * system has been consciously designed in a way that attempts to simplify
 * implementation of unit support for the most common cases in systems
 * biology.
 *
 * As of SBML Level&nbsp;2 Version&nbsp;2, Unit no longer has the
 * attribute called \'offset\' introduced in SBML Level&nbsp;2
 * Version&nbsp;1.  It turned out that the general case involving units
 * with offsets was incorrectly defined, and few (if any) developers even
 * attempted to support offset-based units in their software.  In the
 * development of Level&nbsp;2 Version&nbsp;2, a consensus among SBML
 * developers emerged that a fully generalized unit scheme is @em so
 * confusing and complicated that it actually @em impedes interoperability.
 * SBML Level&nbsp;2 Version&nbsp;2, Version&nbsp;3 and Version&nbsp;4 acknowledge this
 * reality by reducing and simplifying the unit system, specifically by
 * removing the \'offset\' attribute on Unit and @c Celsius as a pre-defined
 * unit.
 *
 * The following guidelines suggest methods for handling units that do
 * require the use of zero offsets for their definitions:
 * <ul>
 * <li> <em>Handling Celsius</em>.  A model in which certain quantities are
 *   temperatures measured in degrees Celsius can be converted
 *   straightforwardly to a model in which those temperatures are in
 *   kelvin.  A software tool could do this by performing a straightforward
 *   substitution using the following relationship: <em>T<sub> kelvin</sub> =
 *   T<sub>Celsius</sub> + 273.15</em>.  In every mathematical formula of the
 *   model where a quantity (call it @em x) in degrees Celsius appears,
 *   replace @em x with <em>x<sub>k</sub>+ 273.15</em>, where
 *   <em>x<sub>k</sub></em> is now in kelvin.  An alternative approach would 
 *   be to use a FunctionDefinition object to define a function encapsulating this
 *   relationship above and then using that in the rest of the model as
 *   needed.  Since Celsius is a commonly-used unit, software tools could
 *   help users by providing users with the ability to express temperatures
 *   in Celsius in the tools\' interfaces, and making substitutions
 *   automatically when writing out the SBML.
 *
 * <li> <em>Other units requiring offsets</em>.  One approach to handling
 *   other kinds of units is to use a FunctionDefinition to define a function
 *   encapsulating the necessary mathematical relationship, then
 *   substituting a call to this function wherever the original quantity
 *   appeared in the model.  For example, here is a possible definition for
 *   converting Fahrenheit to Celsius degrees:
 *   @verbatim
 <functionDefinition id=\'Fahrenheit_to_kelvin\'>
     <math xmlns=\'http://www.w3.org/1998/Math/MathML\'>
         <lambda>
             <bvar><ci> temp_in_fahrenheit </ci></bvar>
             <apply>
                 <divide/>
                 <apply>
                     <plus/>
                     <ci> temp_in_fahrenheit </ci>
                     <cn> 459.67 </cn>
                 </apply>
                 <cn> 1.8 </cn>
             </apply>
         </lambda>
     </math>
 </functionDefinition>
 @endverbatim
 *     
 * <li> An alternative approach not requiring the use of function definitions
 *   is to use an AssignmentRule for each variable in Fahrenheit units.
 *   The AssignmentRule could compute the conversion from Fahrenheit to
 *   (say) kelvin, assign its value to a variable (in Kelvin units), and
 *   then that variable could be used elsewhere in the model.
 *
 * <li> Still another approach is to rewrite the mathematical formulas of a
 *   model to directly incorporate the conversion formula wherever the
 *   original quantity appeared.
 * </ul>
 * 
 * Please consult the SBML specifications for more information about this
 * and other issues involving units.
 *  
 * 
 */
"


%typemap(csimports) ListOfUnitDefinitions "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html A list of UnitDefinition objects.
 * 
 *
 * 
 * The various ListOf___ @if conly structures @else classes@endif in SBML
 * are merely containers used for organizing the main components of an SBML
 * model.  In libSBML\'s implementation, ListOf___
 * @if conly data structures @else classes@endif are derived from the
 * intermediate utility @if conly structure @else class@endif ListOf, which
 * is not defined by the SBML specifications but serves as a useful
 * programmatic construct.  ListOf is itself is in turn derived from SBase,
 * which provides all of the various ListOf___
 * @if conly data structures @else classes@endif with common features
 * defined by the SBML specification, such as \'metaid\' attributes and
 * annotations.
 *
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 *
 * @htmlinclude listof-illustration.html
 *
 * SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
 * Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
 * Version&nbsp;2 allows
 * containers to contain zero or more of the relevant object, instead of 
 * requiring at least one.  As such, libsbml will write out an 
 * otherwise-empty ListOf___ element that has any optional attribute set 
 * (such as \'id\' or \'metaid\'), that has an optional child (such 
 * as a \'notes\' or \'annotation\'), or that has attributes or children set
 * from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
 * any other children.
 *
 * Readers may wonder about the motivations for using the ListOf___
 * containers in SBML.  A simpler approach in XML might be to place the
 * components all directly at the top level of the model definition.  The
 * choice made in SBML is to group them within XML elements named after
 * %ListOf<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from SBase means that software tools can add information @em about
 * the lists themselves into each list container\'s \'annotation\'.
 *
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 *
 * @if conly
 * @note In the C API for libSBML, functions that in other language APIs
 * would be inherited by the various ListOf___ structures not shown in the
 * pages for the individual ListOf___\'s.  Instead, the functions are defined
 * on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
 * the many common functions available for manipulating ListOf___
 * structures</strong>.  The documentation for the individual ListOf___
 * structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
 * all of the functionality available. @endif
 *
 *
 */
"


%csmethodmodifiers UnitDefinition::UnitDefinition(unsigned int level, unsigned int version) "
/**
   * Creates a new UnitDefinition using the given SBML @p level and @p version
   * values.
   *
   * @param level a long integer, the SBML Level to assign to this UnitDefinition.
   *
   * @param version a long integer, the SBML Version to assign to this
   * UnitDefinition.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers UnitDefinition::UnitDefinition(SBMLNamespaces* sbmlns) "
/**
   * Creates a new UnitDefinition using the given SBMLNamespaces object
   * @p sbmlns.
   *
   *
 * 
 * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
 * information.  It is used to communicate the SBML Level, Version, and (in
 * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
 * common approach to using libSBML\'s SBMLNamespaces facilities is to create an
 * SBMLNamespaces object somewhere in a program once, then hand that object
 * as needed to object constructors that accept SBMLNamespaces as arguments.
 *
 * 
   *
   * @param sbmlns an SBMLNamespaces object.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers UnitDefinition::UnitDefinition(const UnitDefinition& orig) "
/**
   * Copy constructor; creates a copy of this UnitDefinition.
   *
   * @param orig the object to copy.
   */ public
";


%csmethodmodifiers UnitDefinition::accept(SBMLVisitor& v) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers UnitDefinition::clone() const "
/**
   * Creates and returns a deep copy of this UnitDefinition object.
   *
   * @return the (deep) copy of this UnitDefinition object.
   */ public new
";


%csmethodmodifiers UnitDefinition::getElementBySId(const std::string& id) "
/**
   * Returns the first child element found that has the given @p id in the
   * model-wide SId namespace, or @c null if no such object is found.
   *
   * @param id string representing the id of the object to find.
   *
   * @return pointer to the first element found with the given @p id.
   */ public new
";


%csmethodmodifiers UnitDefinition::getElementByMetaId(const std::string& metaid) "
/**
   * Returns the first child element it can find with the given @p metaid, or
   * @c null if no such object is found.
   *
   * @param metaid string representing the metaid of the object to find.
   *
   * @return pointer to the first element found with the given @p metaid.
   */ public new
";


%csmethodmodifiers UnitDefinition::getAllElements(ElementFilter* filter=NULL) "
/**
   * Returns a List of all child SBase objects, including those nested to an
   * arbitrary depth
   *
   * @param filter a pointer to an ElementFilter, which causes the function 
   * to return only elements that match a particular set of constraints.  
   * If null (the default), the function will return all child objects.
   *
   * @return a List of pointers to all children objects.
   */ public new
";


%csmethodmodifiers UnitDefinition::getId() const "
/**
   * Returns the value of the \'id\' attribute of this UnitDefinition.
   *
   * @note Because of the inconsistent behavior of this function with 
   * respect to assignments and rules, it is now recommended to
   * use the getIdAttribute() function instead.
   *
   *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
   *
   * @return the id of this UnitDefinition.
   *
   * @see getIdAttribute()
   * @see setIdAttribute(string sid)
   * @see isSetIdAttribute()
   * @see unsetIdAttribute()
   */ public new
";


%csmethodmodifiers UnitDefinition::getName() const "
/**
   * Returns the value of the \'name\' attribute of this UnitDefinition object.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 *
 * @return the name of this SBML object, or the empty string if not set or unsettable.
 *
 * @see getIdAttribute()
 * @see isSetName()
 * @see setName(string sid)
 * @see unsetName()
 * 
 *
   */ public new
";


%csmethodmodifiers UnitDefinition::isSetId() const "
/**
   * Predicate returning @c true if this
   * UnitDefinition\'s \'id\' attribute is set.
   *
   *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 *
 * @return @c true if the \'id\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @note Because of the inconsistent behavior of this function with
 * respect to assignments and rules, it is recommended that callers
 * use isSetIdAttribute() instead.
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see unsetIdAttribute()
 * @see isSetIdAttribute()
 *
 *
   */ public new
";


%csmethodmodifiers UnitDefinition::isSetName() const "
/**
   * Predicate returning @c true if this
   * UnitDefinition\'s \'name\' attribute is set.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 * @return @c true if the \'name\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @see getName()
 * @see setName(string sid)
 * @see unsetName()
 *
 *
   */ public new
";


%csmethodmodifiers UnitDefinition::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this UnitDefinition.
   *
   *
 * 
 * The string @p sid is copied.
 *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 * 
 * @param sid the string to use as the identifier of this object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see isSetIdAttribute()
 * @see unsetIdAttribute()
 * 
 *
   */ public new
";


%csmethodmodifiers UnitDefinition::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this UnitDefinition.
   *
   *
 * 
 *
 * The string in @p name is copied.
 *
 * @param name the new name for the SBML object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 *
 *
   */ public new
";


%csmethodmodifiers UnitDefinition::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this UnitDefinition.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
 *
 * @see getName()
 * @see setName(string sid)
 * @see isSetName()
 *
 *
   */ public new
";


%csmethodmodifiers UnitDefinition::isVariantOfArea(bool relaxed = false) const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier @c \'area\'.
   *
   * @return @c true if this UnitDefinition is a variant of the predefined
   * unit @c area, meaning square metres with only arbitrary variations
   * in scale or multiplier values; @c false otherwise.
   */ public
";


%csmethodmodifiers UnitDefinition::isVariantOfLength(bool relaxed = false) const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier @c \'length\'.
   *
   * @return @c true if this UnitDefinition is a variant of the predefined
   * unit @c length, meaning metres with only arbitrary variations in scale
   * or multiplier values; @c false otherwise.
   */ public
";


%csmethodmodifiers UnitDefinition::isVariantOfSubstance(bool relaxed = false) const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier @c \'substance\'.
   *
   * @return @c true if this UnitDefinition is a variant of the predefined
   * unit @c substance, meaning moles or items (and grams or kilograms from
   * SBML Level&nbsp;2 Version&nbsp;2 onwards) with only arbitrary variations
   * in scale or multiplier values; @c false otherwise.
   */ public
";


%csmethodmodifiers UnitDefinition::isVariantOfTime(bool relaxed = false) const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier @c \'time\'.
   *
   * @return @c true if this UnitDefinition is a variant of the predefined
   * unit @c time, meaning seconds with only arbitrary variations in scale or
   * multiplier values; @c false otherwise.
   */ public
";


%csmethodmodifiers UnitDefinition::isVariantOfVolume(bool relaxed = false) const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier @c \'volume\'.
   *
   * @return @c true if this UnitDefinition is a variant of the predefined
   * unit @c volume, meaning litre or cubic metre with only arbitrary
   * variations in scale or multiplier values; @c false otherwise.
   */ public
";


%csmethodmodifiers UnitDefinition::isVariantOfDimensionless(bool relaxed = false) const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the unit @c \'dimensionless\'.
   *
   * @return @c true if this UnitDefinition is a variant of
   * @c dimensionless, meaning dimensionless with only arbitrary variations in
   * scale or multiplier values; @c false otherwise.
   */ public
";


%csmethodmodifiers UnitDefinition::isVariantOfMass(bool relaxed = false) const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit identifier @c \'mass\'.
   *
   * @return @c true if this UnitDefinition is a variant of mass units,
   * meaning gram or kilogram with only arbitrary variations in scale or
   * multiplier values; @c false otherwise.
   */ public
";


%csmethodmodifiers UnitDefinition::isVariantOfSubstancePerTime(bool relaxed = false) const "
/**
   * Convenience function for testing if a given unit definition is a
   * variant of the predefined unit @c \'substance\' divided by the predefined
   * unit @c \'time\'.
   *
   * @return @c true if this UnitDefinition is a variant of the predefined
   * unit @c substance per predefined unit @c time, meaning it contains two
   * units one of which is a variant of substance and the other is a
   * variant of time which an exponent of -1; @c false otherwise.
   */ public
";


%csmethodmodifiers UnitDefinition::addUnit(const Unit* u) "
/**
   * Adds a copy of the given Unit to this UnitDefinition.
   *
   * @param u the Unit instance to add to this UnitDefinition.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_LEVEL_MISMATCH LIBSBML_LEVEL_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_VERSION_MISMATCH LIBSBML_VERSION_MISMATCH@endlink
   * @li @link libsbml#LIBSBML_INVALID_OBJECT LIBSBML_INVALID_OBJECT@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   * 
   *
 * @note This method should be used with some caution.  The fact that this
 * method @em copies the object passed to it means that the caller will be
 * left holding a physically different object instance than the one contained
 * inside this object.  Changes made to the original object instance (such as
 * resetting attribute values) will <em>not affect the instance in this
 * object</em>.  In addition, the caller should make sure to free the
 * original object if it is no longer being used, or else a memory leak will
 * result.  Please see other methods on this class (particularly a
 * corresponding method whose name begins with the word <code>create</code>)
 * for alternatives that do not lead to these issues.
 *
 * 
   *
   * @see createUnit()
   */ public
";


%csmethodmodifiers UnitDefinition::createUnit "
/**
   * Creates a new and empty Unit, adds it to this UnitDefinition\'s list of
   * units, and returns it.
   *
   * @return a newly constructed (and empty) Unit instance.
   * 
   * @note It is worth emphasizing that the attribute \'kind\' value of a
   * Unit is a required attribute for a valid Unit definition.  The
   * createUnit() method does not assign a valid kind to the constructed
   * unit (instead, it sets the \'kind\' to @link libsbml#UNIT_KIND_INVALID UNIT_KIND_INVALID@endlink).
   * Callers are cautioned to set the newly-constructed Unit\'s kind using
   * Unit::setKind(@if java int@endif) soon after calling this method.
   *
   * @see addUnit(Unit u)
   */ public
";


%csmethodmodifiers UnitDefinition::getListOfUnits() const "
/**
   * Returns the list of Units for this UnitDefinition instance.
   * @return the ListOfUnits value for this UnitDefinition.
   */ public
";


%csmethodmodifiers UnitDefinition::getListOfUnits "
/**
   * Returns the list of Units for this UnitDefinition instance.
   * @return the ListOfUnits value for this UnitDefinition.
   */ public
";


%csmethodmodifiers UnitDefinition::getUnit(unsigned int n) "
/**
   * Returns a specific Unit instance belonging to this UnitDefinition.
   *
   * @param n an integer, the index of the Unit to be returned.
   * 
   * @return the nth Unit of this UnitDefinition.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see getNumUnits()
   */ public
";


%csmethodmodifiers UnitDefinition::getUnit "
/**
   * Returns a specific Unit instance belonging to this UnitDefinition.
   *
   * @param n an integer, the index of the Unit to be returned.
   * 
   * @return the nth Unit of this UnitDefinition.
   * If the index @p n is invalid, @c null is returned.
   */ public
";


%csmethodmodifiers UnitDefinition::getNumUnits() const "
/**
   * Returns the number of Unit objects contained within this
   * UnitDefinition.
   * 
   * @return an integer representing the number of Units in this
   * UnitDefinition.
   */ public
";


%csmethodmodifiers UnitDefinition::removeUnit(unsigned int n) "
/**
   * Removes the nth Unit object from this UnitDefinition object and
   * returns a pointer to it.
   *
   * The caller owns the returned object and is responsible for deleting it.
   *
   * @param n the index of the Unit object to remove.
   *
   * @return the Unit object removed, or @c null if the given index 
   * is out of range.
   *
   */ public
";


%csmethodmodifiers UnitDefinition::setSBMLDocument(SBMLDocument* d) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers UnitDefinition::enablePackageInternal(const std::string& pkgURI, const std::string& pkgPrefix, bool flag) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers UnitDefinition::updateSBMLNamespace(const std::string& package, unsigned int level, unsigned int version) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers UnitDefinition::getTypeCode() const "
/**
   * Returns the libSBML type code for this object instance.
   * 
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for this object:
   * @link libsbml#SBML_UNIT_DEFINITION SBML_UNIT_DEFINITION@endlink (default).
   *
   *
 * @warning <span class=\'warning\'>The specific integer values of the possible
 * type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
 * packages,  To fully identify the correct code, <strong>it is necessary to
 * invoke both getPackageName() and getTypeCode()</strong> (or 
 * ListOf::getItemTypeCode()).</span>
 *
 *
   *
   * @see getPackageName()
   * @see getElementName()
   */ public new
";


%csmethodmodifiers UnitDefinition::getElementName() const "
/**
   * Returns the XML element name of this object, which for UnitDefinition,
   * is always @c \'unitDefinition\'.
   * 
   * @return the name of this element, i.e., @c \'unitDefinition\'.
   */ public new
";


%csmethodmodifiers UnitDefinition::simplify(UnitDefinition * ud) "
/**
   * Simplifies the UnitDefinition such that any given kind of Unit object
   * occurs only once in the ListOfUnits.
   *
   * For example, the following definition,
   * @verbatim
 <unitDefinition>
  <listOfUnits>
    <unit kind=\'metre\' exponent=\'1\'/>
    <unit kind=\'metre\' exponent=\'2\'/>
  </listOfUnits>
 <unitDefinition>
 @endverbatim
   * will be simplified to 
   * @verbatim
 <unitDefinition>
   <listOfUnits>
     <unit kind=\'metre\' exponent=\'3\'/>
   </listOfUnits>
 <unitDefinition>
 @endverbatim
   *
   * @param ud the UnitDefinition object to be simplified.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   */ public
";


%csmethodmodifiers UnitDefinition::reorder(UnitDefinition * ud) "
/**
   * Alphabetically orders the Unit objects within the ListOfUnits of a
   * UnitDefinition.
   *
   * @param ud the UnitDefinition object whose units are to be reordered.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   */ public
";


%csmethodmodifiers UnitDefinition::convertToSI(const UnitDefinition *ud) "
/**
   * Convert a given UnitDefinition into a new UnitDefinition object
   * that uses SI units.
   * 
   * @param ud the UnitDefinition object to convert to SI.
   *
   * @return a new UnitDefinition object representing the results of the
   * conversion.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   */ public
";


%csmethodmodifiers UnitDefinition::areIdentical(const UnitDefinition * ud1, const UnitDefinition * ud2) "
/**
   * Predicate returning @c true if two
   * UnitDefinition objects are identical.
   *
   * For the purposes of performing this comparison, two UnitDefinition
   * objects are considered identical when they contain identical lists of
   * Unit objects.  Pairs of Unit objects in the lists are in turn
   * considered identical if they satisfy the predicate
   * Unit::areIdentical(@if java Unit, %Unit@endif).
   * The predicate compares every attribute of the
   * Unit objects.
   *
   * @param ud1 the first UnitDefinition object to compare.
   * @param ud2 the second UnitDefinition object to compare.
   *
   * @return @c true if all the Unit objects in @p ud1 are identical to the
   * Unit objects of @p ud2, @c false otherwise.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   *
   * @see UnitDefinition::areEquivalent(UnitDefinition  ud1, %UnitDefinition  ud2)
   * @see Unit::areIdentical(Unit  unit1, %Unit  unit2)
   */ public
";


%csmethodmodifiers UnitDefinition::areEquivalent(const UnitDefinition *ud1 , const UnitDefinition * ud2) "
/**
   * Predicate returning @c true if two
   * UnitDefinition objects are equivalent.
   *
   * For the purposes of performing this comparison, two UnitDefinition
   * objects are considered equivalent when they contain @em equivalent
   * list of Unit objects.  Unit objects are in turn considered equivalent
   * if they satisfy the predicate
   * Unit::areEquivalent(@if java Unit, %Unit@endif).
   * The predicate tests a subset of the objects\'s attributes.
   *
   * @param ud1 the first UnitDefinition object to compare.
   * 
   * @param ud2 the second UnitDefinition object to compare.
   *
   * @return @c true if all the Unit objects in @p ud1 are equivalent
   * to the Unit objects in @p ud2, @c false otherwise.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   *
   * @see UnitDefinition::areIdentical(UnitDefinition  ud1, %UnitDefinition  ud2)
   * @see Unit::areEquivalent(Unit  unit1, %Unit  unit2)
   */ public
";


%csmethodmodifiers UnitDefinition::areIdenticalSIUnits(const UnitDefinition * ud1, const UnitDefinition * ud2) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers UnitDefinition::combine(UnitDefinition * ud1, UnitDefinition * ud2) "
/**
   * Combines two UnitDefinition objects into a single UnitDefinition.
   *
   * This takes UnitDefinition objects @p ud1 and @p ud2, and creates a
   * UnitDefinition object that expresses the product of the units of @p
   * ud1 and @p ud2.
   *
   * @param ud1 the first UnitDefinition object.
   * @param ud2 the second UnitDefinition object.
   *
   * @return a UnitDefinition which represents the product of the 
   * units of the two argument UnitDefinitions.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   */ public
";


%csmethodmodifiers UnitDefinition::divide(UnitDefinition * ud1, UnitDefinition * ud2) "
/**
   * Combines two UnitDefinition objects into a single UnitDefinition as
   * a division.
   *
   * This takes UnitDefinition objects @p ud1 and @p ud2, and creates a
   * UnitDefinition object that expresses the division of the units of @p
   * ud1 and @p ud2.
   *
   * @param ud1 the first UnitDefinition object.
   * @param ud2 the second UnitDefinition object.
   *
   * @return a UnitDefinition which represents the division of the 
   * units of the two argument UnitDefinitions.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   */ public
";


%csmethodmodifiers UnitDefinition::printUnits(const UnitDefinition * ud, bool compact = false) "
/**
   * Expresses the given definition in a plain-text form.
   *
   * For example,
   * UnitDefinition::printUnits(@if java UnitDefinition@endif)
   * applied to
   * @verbatim
 <unitDefinition>
  <listOfUnits>
    <unit kind=\'metre\' exponent=\'1\'/>
    <unit kind=\'second\' exponent=\'-2\'/>
  </listOfUnits>
 <unitDefinition>
 @endverbatim
   * will return the string <code>\'metre (exponent = 1, multiplier = 1,
   * scale = 0) second (exponent = -2, multiplier = 1, scale = 0)\'</code>
   * or, if the optional parameter @p compact is given the value @c true,
   * the string <code>\'(1 metre)^1 (1 second)^-2\'</code>.  This method may
   * be useful for printing unit information to human users, or in
   * debugging software, or other situations.
   *
   * @param ud the UnitDefinition object.
   * @param compact boolean indicating whether the compact form
   * should be used (defaults to false).
   *
   * @return a string expressing the unit definition defined by the given
   * UnitDefinition object @p ud.
   *
   *
 * @if python @note Because this is a static method on a class, the Python
 * language interface for libSBML will contain two variants.  One will be the
 * expected, normal static method on the class (i.e., a regular
 * <em>methodName</em>), and the other will be a standalone top-level
 * function with the name <em>ClassName_methodName()</em>. This is merely an
 * artifact of how the language interfaces are created in libSBML.  The
 * methods are functionally identical. @endif
 *
 *
   */ public
";


%csmethodmodifiers UnitDefinition::writeElements(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers UnitDefinition::hasRequiredAttributes() const "
/**
   * Predicate returning @c true if
   * all the required attributes for this UnitDefinition object
   * have been set.
   *
   * The required attributes for a UnitDefinition object are:
   * @li \'id\'
   *
   * @return @c true if the required attributes have been set, @c false
   * otherwise.
   */ public new
";


%csmethodmodifiers UnitDefinition::hasRequiredElements() const "
/**
   * Predicate returning @c true if
   * all the required elements for this UnitDefinition object
   * have been set.
   *
   * @note The required elements for a Constraint object are:
   * @li \'listOfUnits\' (required in SBML Level&nbsp;2 only, optional in Level&nbsp;3)
   *
   * @return a boolean value indicating whether all the required
   * elements for this object have been defined.
   */ public new
";


%csmethodmodifiers UnitDefinition::createObject(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers UnitDefinition::addExpectedAttributes(ExpectedAttributes& attributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers UnitDefinition::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers UnitDefinition::readL1Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers UnitDefinition::readL2Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers UnitDefinition::readL3Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers UnitDefinition::writeAttributes(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOfUnitDefinitions::ListOfUnitDefinitions(unsigned int level, unsigned int version) "
/**
   * Creates a new ListOfUnitDefinitions object.
   *
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   *
   * @param level the SBML Level.
   * 
   * @param version the Version within the SBML Level.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfUnitDefinitions::ListOfUnitDefinitions(SBMLNamespaces* sbmlns) "
/**
   * Creates a new ListOfUnitDefinitions object.
   *
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the SBMLNamespaces object in @p
   * sbmlns.
   *
   * @param sbmlns an SBMLNamespaces object that is used to determine the
   * characteristics of the ListOfUnitDefinitions object to be created.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfUnitDefinitions::clone() const "
/**
   * Creates and returns a deep copy of this ListOfUnitDefinitions object.
   *
   * @return the (deep) copy of this ListOfUnitDefinitions object.
   */ public new
";


%csmethodmodifiers ListOfUnitDefinitions::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this ListOf
   * (i.e., UnitDefinition objects, if the list is non-empty).
   *
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for objects contained in this list:
   * @link libsbml#SBML_UNIT_DEFINITION SBML_UNIT_DEFINITION@endlink (default).
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers ListOfUnitDefinitions::getElementName() const "
/**
   * Returns the XML element name of this object.
   *
   * For ListOfUnitDefinitions, the XML element name is
   * @c \'listOfUnitDefinitions\'.
   * 
   * @return the name of this element, i.e., @c \'listOfUnitDefinitions\'.
   */ public new
";


%csmethodmodifiers ListOfUnitDefinitions::get(unsigned int n) "
/**
   * Get a UnitDefinition from the ListOfUnitDefinitions.
   *
   * @param n the index number of the UnitDefinition to get.
   * 
   * @return the nth UnitDefinition in this ListOfUnitDefinitions.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfUnitDefinitions::get "
/**
   * Get a UnitDefinition from the ListOfUnitDefinitions.
   *
   * @param n the index number of the UnitDefinition to get.
   * 
   * @return the nth UnitDefinition in this ListOfUnitDefinitions.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfUnitDefinitions::get(const std::string& sid) "
/**
   * Get a UnitDefinition from the ListOfUnitDefinitions
   * based on its identifier.
   *
   * @param sid a string representing the identifier 
   * of the UnitDefinition to get.
   * 
   * @return UnitDefinition in this ListOfUnitDefinitions
   * with the given @p sid or @c null if no such
   * UnitDefinition exists.
   *
   * @see get(unsigned int n)
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfUnitDefinitions::get(const std::string& sid) const "
/**
   * Get a UnitDefinition from the ListOfUnitDefinitions
   * based on its identifier.
   *
   * @param sid a string representing the identifier 
   * of the UnitDefinition to get.
   * 
   * @return UnitDefinition in this ListOfUnitDefinitions
   * with the given @p sid or @c null if no such
   * UnitDefinition exists.
   *
   * @see get(unsigned int n)
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfUnitDefinitions::getElementBySId(const std::string& id) "
/**
   * Returns the first child element found that has the given @p id in the
   * model-wide SId namespace, or @c null if no such object is found.
   *
   * Note that UnitDefinitions themselves are in the UnitId namespace, not
   * the SId namespace, so no UnitDefinition object will be returned from
   * this function (and is the reason we override the base
   * ListOf::getElementBySId function here).
   *
   * @param id string representing the id of the object to find.
   *
   * @return pointer to the first element found with the given @p id.
   */ public new
";


%csmethodmodifiers ListOfUnitDefinitions::remove(unsigned int n) "
/**
   * Removes the nth item from this ListOfUnitDefinitions items and returns a pointer to
   * it.
   *
   * The caller owns the returned item and is responsible for deleting it.
   *
   * @param n the index of the item to remove.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfUnitDefinitions::remove(const std::string& sid) "
/**
   * Removes item in this ListOfUnitDefinitions items with the given identifier.
   *
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier @p sid, then
   * @c null is returned.
   *
   * @param sid the identifier of the item to remove.
   *
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public new
";


%csmethodmodifiers ListOfUnitDefinitions::getElementPosition() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOfUnitDefinitions::createObject(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%typemap(csimports) CompartmentType "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html A <em>compartment type</em> in SBML Level&nbsp;2.
 *
 * SBML Level&nbsp;2 Versions&nbsp;2&ndash;4 provide the <em>compartment
 * type</em> as a grouping construct that can be used to establish a
 * relationship between multiple Compartment objects.  A CompartmentType
 * object only has an identity, and this identity can only be used to
 * indicate that particular Compartment objects in the model belong to this
 * type.  This may be useful for conveying a modeling intention, such as
 * when a model contains many similar compartments, either by their
 * biological function or the reactions they carry.  Without a compartment
 * type construct, it would be impossible within SBML itself to indicate
 * that all of the compartments share an underlying conceptual relationship
 * because each SBML compartment must be given a unique and separate
 * identity.  A CompartmentType has no mathematical meaning in
 * SBML---it has no effect on a model\'s mathematical interpretation.
 * Simulators and other numerical analysis software may ignore
 * CompartmentType definitions and references to them in a model.
 * 
 * There is no mechanism in SBML Level 2 for representing hierarchies of
 * compartment types.  One CompartmentType instance cannot be the subtype
 * of another CompartmentType instance; SBML provides no means of defining
 * such relationships.
 * 
 * As with other major structures in SBML, CompartmentType has a mandatory
 * attribute, \'id\', used to give the compartment type an identifier.  The
 * identifier must be a text %string conforming to the identifer syntax
 * permitted in SBML.  CompartmentType also has an optional \'name\'
 * attribute, of type @c string.  The \'id\' and \'name\' must be used
 * according to the guidelines described in the SBML specification (e.g.,
 * Section 3.3 in the Level 2 Version 4 specification).
 *
 * CompartmentType was introduced in SBML Level 2 Version 2.  It is not
 * available in SBML Level&nbsp;1 nor in Level&nbsp;3.
 *
 * @see Compartment
 * @see ListOfCompartmentTypes
 * @see SpeciesType
 * @see ListOfSpeciesTypes
 *
 * 
 * 
 */
"


%typemap(csimports) ListOfCompartmentTypes "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html A list of CompartmentType objects.
 * 
 *
 * 
 * The various ListOf___ @if conly structures @else classes@endif in SBML
 * are merely containers used for organizing the main components of an SBML
 * model.  In libSBML\'s implementation, ListOf___
 * @if conly data structures @else classes@endif are derived from the
 * intermediate utility @if conly structure @else class@endif ListOf, which
 * is not defined by the SBML specifications but serves as a useful
 * programmatic construct.  ListOf is itself is in turn derived from SBase,
 * which provides all of the various ListOf___
 * @if conly data structures @else classes@endif with common features
 * defined by the SBML specification, such as \'metaid\' attributes and
 * annotations.
 *
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 *
 * @htmlinclude listof-illustration.html
 *
 * SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
 * Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
 * Version&nbsp;2 allows
 * containers to contain zero or more of the relevant object, instead of 
 * requiring at least one.  As such, libsbml will write out an 
 * otherwise-empty ListOf___ element that has any optional attribute set 
 * (such as \'id\' or \'metaid\'), that has an optional child (such 
 * as a \'notes\' or \'annotation\'), or that has attributes or children set
 * from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
 * any other children.
 *
 * Readers may wonder about the motivations for using the ListOf___
 * containers in SBML.  A simpler approach in XML might be to place the
 * components all directly at the top level of the model definition.  The
 * choice made in SBML is to group them within XML elements named after
 * %ListOf<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from SBase means that software tools can add information @em about
 * the lists themselves into each list container\'s \'annotation\'.
 *
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 *
 * @if conly
 * @note In the C API for libSBML, functions that in other language APIs
 * would be inherited by the various ListOf___ structures not shown in the
 * pages for the individual ListOf___\'s.  Instead, the functions are defined
 * on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
 * the many common functions available for manipulating ListOf___
 * structures</strong>.  The documentation for the individual ListOf___
 * structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
 * all of the functionality available. @endif
 *
 *
 */
"


%csmethodmodifiers CompartmentType::CompartmentType(unsigned int level, unsigned int version) "
/**
   * Creates a new CompartmentType object using the given SBML @p level and
   * @p version values.
   *
   * @param level a long integer, the SBML Level to assign to this
   * CompartmentType.
   *
   * @param version a long integer, the SBML Version to assign to this
   * CompartmentType.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers CompartmentType::CompartmentType(SBMLNamespaces* sbmlns) "
/**
   * Creates a new CompartmentType object using the given SBMLNamespaces
   * object @p sbmlns.
   *
   *
 * 
 * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
 * information.  It is used to communicate the SBML Level, Version, and (in
 * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
 * common approach to using libSBML\'s SBMLNamespaces facilities is to create an
 * SBMLNamespaces object somewhere in a program once, then hand that object
 * as needed to object constructors that accept SBMLNamespaces as arguments.
 *
 *
   *
   * It is worth emphasizing that although this constructor does not take an
   * identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a CompartmentType object is required to have a
   * value.  Thus, callers are cautioned to assign a value after calling this
   * constructor.  Setting the identifier can be accomplished using the
   * method setId(@if java String@endif).
   *
   * @param sbmlns an SBMLNamespaces object.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers CompartmentType::CompartmentType(const CompartmentType& orig) "
/**
   * Copy constructor; creates a copy of this CompartmentType object.
   *
   * @param orig the object to copy.
   */ public
";


%csmethodmodifiers CompartmentType::accept(SBMLVisitor& v) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers CompartmentType::clone() const "
/**
   * Creates and returns a deep copy of this CompartmentType object.
   *
   * @return the (deep) copy of this CompartmentType object.
   */ public new
";


%csmethodmodifiers CompartmentType::getId() const "
/**
   * Returns the value of the \'id\' attribute of this CompartmentType.
   *
   * @note Because of the inconsistent behavior of this function with 
   * respect to assignments and rules, it is now recommended to
   * use the getIdAttribute() function instead.
   *
   *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
   *
   * @return the id of this CompartmentType.
   *
   * @see getIdAttribute()
   * @see setIdAttribute(string sid)
   * @see isSetIdAttribute()
   * @see unsetIdAttribute()
   */ public new
";


%csmethodmodifiers CompartmentType::getName() const "
/**
   * Returns the value of the \'name\' attribute of this CompartmentType object.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 *
 * @return the name of this SBML object, or the empty string if not set or unsettable.
 *
 * @see getIdAttribute()
 * @see isSetName()
 * @see setName(string sid)
 * @see unsetName()
 * 
 *
   */ public new
";


%csmethodmodifiers CompartmentType::isSetId() const "
/**
   * Predicate returning @c true if this CompartmentType object\'s \'id\'
   * attribute is set.
   *
   *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 *
 * @return @c true if the \'id\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @note Because of the inconsistent behavior of this function with
 * respect to assignments and rules, it is recommended that callers
 * use isSetIdAttribute() instead.
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see unsetIdAttribute()
 * @see isSetIdAttribute()
 *
 *
   */ public new
";


%csmethodmodifiers CompartmentType::isSetName() const "
/**
   * Predicate returning @c true if this CompartmentType object\'s \'name\'
   * attribute is set.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 * @return @c true if the \'name\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @see getName()
 * @see setName(string sid)
 * @see unsetName()
 *
 *
   */ public new
";


%csmethodmodifiers CompartmentType::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this CompartmentType.
   *
   *
 * 
 * The string @p sid is copied.
 *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 * 
 * @param sid the string to use as the identifier of this object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see isSetIdAttribute()
 * @see unsetIdAttribute()
 * 
 *
   */ public new
";


%csmethodmodifiers CompartmentType::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this CompartmentType.
   *
   *
 * 
 *
 * The string in @p name is copied.
 *
 * @param name the new name for the SBML object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 *
 *
   */ public new
";


%csmethodmodifiers CompartmentType::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this CompartmentType object.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
 *
 * @see getName()
 * @see setName(string sid)
 * @see isSetName()
 *
 *
   */ public new
";


%csmethodmodifiers CompartmentType::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   *
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for this object:
   * @link libsbml#SBML_COMPARTMENT_TYPE SBML_COMPARTMENT_TYPE@endlink (default).
   *
   *
 * @warning <span class=\'warning\'>The specific integer values of the possible
 * type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
 * packages,  To fully identify the correct code, <strong>it is necessary to
 * invoke both getPackageName() and getTypeCode()</strong> (or 
 * ListOf::getItemTypeCode()).</span>
 *
 *
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers CompartmentType::getElementName() const "
/**
   * Returns the XML element name of this object
   *
   * For CompartmentType, the element name is always @c \'compartmentType\'.
   *
   * @return the name of this element.
   *
   * @see getTypeCode()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers CompartmentType::writeElements(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers CompartmentType::hasRequiredAttributes() const "
/**
   * Predicate returning @c true if all the required attributes for this
   * CompartmentType object have been set.
   *
   * The required attributes for a CompartmentType object are:
   * @li \'id\'
   *
   * @return @c true if the required attributes have been set, @c false
   * otherwise.
   */ public new
";


%csmethodmodifiers CompartmentType::addExpectedAttributes(ExpectedAttributes& attributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers CompartmentType::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers CompartmentType::readL2Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers CompartmentType::writeAttributes(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOfCompartmentTypes::ListOfCompartmentTypes(unsigned int level, unsigned int version) "
/**
   * Creates a new ListOfCompartmentTypes object.
   *
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   *
   * @param level the SBML Level.
   *
   * @param version the Version within the SBML Level.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfCompartmentTypes::ListOfCompartmentTypes(SBMLNamespaces* sbmlns) "
/**
   * Creates a new ListOfCompartmentTypes object.
   *
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the SBMLNamespaces object in @p
   * sbmlns.
   *
   * @param sbmlns an SBMLNamespaces object that is used to determine the
   * characteristics of the ListOfCompartmentTypes object to be created.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfCompartmentTypes::clone() const "
/**
   * Creates and returns a deep copy of this ListOfCompartmentTypes object.
   *
   * @return the (deep) copy of this ListOfCompartmentTypes object.
   */ public new
";


%csmethodmodifiers ListOfCompartmentTypes::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this ListOf
   * (i.e., CompartmentType objects, if the list is non-empty).
   *
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for the objects contained in this ListOf
   * instance: @link libsbml#SBML_COMPARTMENT_TYPE SBML_COMPARTMENT_TYPE@endlink (default).
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers ListOfCompartmentTypes::getElementName() const "
/**
   * Returns the XML element name of this object.
   *
   * For ListOfCompartmentTypes, the XML element name is
   * @c \'listOfCompartmentTypes\'.
   *
   * @return the name of this element, i.e., @c \'listOfCompartmentTypes\'.
   */ public new
";


%csmethodmodifiers ListOfCompartmentTypes::get(unsigned int n) "
/**
   * Get a CompartmentType object from the ListOfCompartmentTypes.
   *
   * @param n the index number of the CompartmentType object to get.
   *
   * @return the nth CompartmentType object in this ListOfCompartmentTypes.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfCompartmentTypes::get "
/**
   * Get a CompartmentType object from the ListOfCompartmentTypes.
   *
   * @param n the index number of the CompartmentType object to get.
   *
   * @return the nth CompartmentType object in this ListOfCompartmentTypes.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfCompartmentTypes::get(const std::string& sid) "
/**
   * Get a CompartmentType object from the ListOfCompartmentTypes
   * based on its identifier.
   *
   * @param sid a string representing the identifier
   * of the CompartmentType object to get.
   *
   * @return CompartmentType object in this ListOfCompartmentTypes
   * with the given @p sid or @c null if no such
   * CompartmentType object exists.
   *
   * @see get(unsigned int n)
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfCompartmentTypes::get(const std::string& sid) const "
/**
   * Get a CompartmentType object from the ListOfCompartmentTypes
   * based on its identifier.
   *
   * @param sid a string representing the identifier
   * of the CompartmentType object to get.
   *
   * @return CompartmentType object in this ListOfCompartmentTypes
   * with the given @p sid or @c null if no such
   * CompartmentType object exists.
   *
   * @see get(unsigned int n)
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfCompartmentTypes::remove(unsigned int n) "
/**
   * Removes the nth item from this ListOfCompartmentTypes items
   * and returns a pointer to it.
   *
   * The caller owns the returned item and is responsible for deleting it.
   *
   * @param n the index of the item to remove.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfCompartmentTypes::remove(const std::string& sid) "
/**
   * Removes item in this ListOfCompartmentTypes items with the given identifier.
   *
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier @p sid, then
   * @c null is returned.
   *
   * @param sid the identifier of the item to remove.
   *
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public new
";


%csmethodmodifiers ListOfCompartmentTypes::getElementPosition() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOfCompartmentTypes::createObject(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%typemap(csimports) SpeciesType "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html A <em>species type</em> in SBML Level 2.
 *
 * The term @em species @em type refers to reacting entities independent of
 * location.  These include simple ions (e.g., protons, calcium), simple
 * molecules (e.g., glucose, ATP), large molecules (e.g., RNA,
 * polysaccharides, and proteins), and others.
 * 
 * SBML Level&nbsp;2 Versions&nbsp;2&ndash;4 provide an explicit
 * SpeciesType class of object to enable Species objects of the same type
 * to be related together.  SpeciesType is a conceptual construct; the
 * existence of SpeciesType objects in a model has no effect on the model\'s
 * numerical interpretation.  Except for the requirement for uniqueness of
 * species/species type combinations located in compartments, simulators
 * and other numerical analysis software may ignore SpeciesType definitions
 * and references to them in a model.
 * 
 * There is no mechanism in SBML Level 2 for representing hierarchies of
 * species types.  One SpeciesType object cannot be the subtype of another
 * SpeciesType object; SBML provides no means of defining such
 * relationships.
 * 
 * As with other major structures in SBML, SpeciesType has a mandatory
 * attribute, \'id\', used to give the species type an identifier.  The
 * identifier must be a text string conforming to the identifer syntax
 * permitted in SBML.  SpeciesType also has an optional \'name\' attribute,
 * of type @c string.  The \'id\' and \'name\' must be used according to the
 * guidelines described in the SBML specification (e.g., Section 3.3 in
 * the Level&nbsp;2 Version&nbsp;4 specification).
 *
 * SpeciesType was introduced in SBML Level 2 Version 2.  It is not
 * available in SBML Level&nbsp;1 nor in Level&nbsp;3.
 *
 * @see Species
 * @see ListOfSpeciesTypes
 * @see CompartmentType
 * @see ListOfCompartmentTypes
 * 
 * 
 */
"


%typemap(csimports) ListOfSpeciesTypes "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html A list of SpeciesType objects.
 *
 *
 * 
 * The various ListOf___ @if conly structures @else classes@endif in SBML
 * are merely containers used for organizing the main components of an SBML
 * model.  In libSBML\'s implementation, ListOf___
 * @if conly data structures @else classes@endif are derived from the
 * intermediate utility @if conly structure @else class@endif ListOf, which
 * is not defined by the SBML specifications but serves as a useful
 * programmatic construct.  ListOf is itself is in turn derived from SBase,
 * which provides all of the various ListOf___
 * @if conly data structures @else classes@endif with common features
 * defined by the SBML specification, such as \'metaid\' attributes and
 * annotations.
 *
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 *
 * @htmlinclude listof-illustration.html
 *
 * SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
 * Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
 * Version&nbsp;2 allows
 * containers to contain zero or more of the relevant object, instead of 
 * requiring at least one.  As such, libsbml will write out an 
 * otherwise-empty ListOf___ element that has any optional attribute set 
 * (such as \'id\' or \'metaid\'), that has an optional child (such 
 * as a \'notes\' or \'annotation\'), or that has attributes or children set
 * from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
 * any other children.
 *
 * Readers may wonder about the motivations for using the ListOf___
 * containers in SBML.  A simpler approach in XML might be to place the
 * components all directly at the top level of the model definition.  The
 * choice made in SBML is to group them within XML elements named after
 * %ListOf<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from SBase means that software tools can add information @em about
 * the lists themselves into each list container\'s \'annotation\'.
 *
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 *
 * @if conly
 * @note In the C API for libSBML, functions that in other language APIs
 * would be inherited by the various ListOf___ structures not shown in the
 * pages for the individual ListOf___\'s.  Instead, the functions are defined
 * on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
 * the many common functions available for manipulating ListOf___
 * structures</strong>.  The documentation for the individual ListOf___
 * structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
 * all of the functionality available. @endif
 *
 *
 */
"


%csmethodmodifiers SpeciesType::SpeciesType(unsigned int level, unsigned int version) "
/**
   * Creates a new SpeciesType using the given SBML @p level and @p version
   * values.
   *
   * @param level a long integer, the SBML Level to assign to this SpeciesType.
   *
   * @param version a long integer, the SBML Version to assign to this
   * SpeciesType.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers SpeciesType::SpeciesType(SBMLNamespaces* sbmlns) "
/**
   * Creates a new SpeciesType using the given SBMLNamespaces object
   * @p sbmlns.
   *
   *
 * 
 * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
 * information.  It is used to communicate the SBML Level, Version, and (in
 * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
 * common approach to using libSBML\'s SBMLNamespaces facilities is to create an
 * SBMLNamespaces object somewhere in a program once, then hand that object
 * as needed to object constructors that accept SBMLNamespaces as arguments.
 *
 * 
   *
   * It is worth emphasizing that although this constructor does not take
   * an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a SpeciesType object is required to have a value.
   * Thus, callers are cautioned to assign a value after calling this
   * constructor.  Setting the identifier can be accomplished using the
   * method SBase::setId(@if java String@endif).
   *
   * @param sbmlns an SBMLNamespaces object.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers SpeciesType::SpeciesType(const SpeciesType& orig) "
/**
   * Copy constructor; creates a copy of this SpeciesType.
   *
   * @param orig the object to copy.
   */ public
";


%csmethodmodifiers SpeciesType::accept(SBMLVisitor& v) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SpeciesType::clone() const "
/**
   * Creates and returns a deep copy of this SpeciesType object.
   *
   * @return the (deep) copy of this SpeciesType object.
   */ public new
";


%csmethodmodifiers SpeciesType::getId() const "
/**
   * Returns the value of the \'id\' attribute of this SpeciesType.
   *
   * @note Because of the inconsistent behavior of this function with 
   * respect to assignments and rules, it is now recommended to
   * use the getIdAttribute() function instead.
   *
   *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
   *
   * @return the id of this SpeciesType.
   *
   * @see getIdAttribute()
   * @see setIdAttribute(string sid)
   * @see isSetIdAttribute()
   * @see unsetIdAttribute()
   */ public new
";


%csmethodmodifiers SpeciesType::getName() const "
/**
   * Returns the value of the \'name\' attribute of this SpeciesType object.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 *
 * @return the name of this SBML object, or the empty string if not set or unsettable.
 *
 * @see getIdAttribute()
 * @see isSetName()
 * @see setName(string sid)
 * @see unsetName()
 * 
 *
   */ public new
";


%csmethodmodifiers SpeciesType::isSetId() const "
/**
   * Predicate returning @c true if this
   * SpeciesType\'s \'id\' attribute is set.
   *
   *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 *
 * @return @c true if the \'id\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @note Because of the inconsistent behavior of this function with
 * respect to assignments and rules, it is recommended that callers
 * use isSetIdAttribute() instead.
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see unsetIdAttribute()
 * @see isSetIdAttribute()
 *
 *
   */ public new
";


%csmethodmodifiers SpeciesType::isSetName() const "
/**
   * Predicate returning @c true if this
   * SpeciesType\'s \'name\' attribute is set.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 * @return @c true if the \'name\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @see getName()
 * @see setName(string sid)
 * @see unsetName()
 *
 *
   */ public new
";


%csmethodmodifiers SpeciesType::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this SpeciesType.
   *
   *
 * 
 * The string @p sid is copied.
 *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 * 
 * @param sid the string to use as the identifier of this object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see isSetIdAttribute()
 * @see unsetIdAttribute()
 * 
 *
   */ public new
";


%csmethodmodifiers SpeciesType::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this SpeciesType.
   *
   *
 * 
 *
 * The string in @p name is copied.
 *
 * @param name the new name for the SBML object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 *
 *
   */ public new
";


%csmethodmodifiers SpeciesType::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this SpeciesType.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
 *
 * @see getName()
 * @see setName(string sid)
 * @see isSetName()
 *
 *
   */ public new
";


%csmethodmodifiers SpeciesType::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * 
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for this object:
   * @link libsbml#SBML_SPECIES_TYPE SBML_SPECIES_TYPE@endlink (default).
   *
   *
 * @warning <span class=\'warning\'>The specific integer values of the possible
 * type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
 * packages,  To fully identify the correct code, <strong>it is necessary to
 * invoke both getPackageName() and getTypeCode()</strong> (or 
 * ListOf::getItemTypeCode()).</span>
 *
 *
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers SpeciesType::getElementName() const "
/**
   * Returns the XML element name of this object, which for
   * SpeciesType, is always @c \'compartmentType\'.
   * 
   * @return the name of this element, i.e., @c \'compartmentType\'.
   */ public new
";


%csmethodmodifiers SpeciesType::writeElements(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SpeciesType::hasRequiredAttributes() const "
/**
   * Predicate returning @c true if
   * all the required attributes for this SpeciesType object
   * have been set.
   *
   * The required attributes for a SpeciesType object are:
   * @li \'id\'
   *
   * @return @c true if the required attributes have been set, @c false
   * otherwise.
   */ public new
";


%csmethodmodifiers SpeciesType::addExpectedAttributes(ExpectedAttributes& attributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SpeciesType::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers SpeciesType::readL2Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers SpeciesType::writeAttributes(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOfSpeciesTypes::ListOfSpeciesTypes(unsigned int level, unsigned int version) "
/**
   * Creates a new ListOfSpeciesTypes object.
   *
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   *
   * @param level the SBML Level.
   * 
   * @param version the Version within the SBML Level.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfSpeciesTypes::ListOfSpeciesTypes(SBMLNamespaces* sbmlns) "
/**
   * Creates a new ListOfSpeciesTypes object.
   *
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the SBMLNamespaces object in @p
   * sbmlns.
   *
   * @param sbmlns an SBMLNamespaces object that is used to determine the
   * characteristics of the ListOfSpeciesTypes object to be created.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfSpeciesTypes::clone() const "
/**
   * Creates and returns a deep copy of this ListOfSpeciesTypes object.
   *
   * @return the (deep) copy of this ListOfSpeciesTypes object.
   */ public new
";


%csmethodmodifiers ListOfSpeciesTypes::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this ListOf
   * (i.e., SpeciesType objects, if the list is non-empty).
   *
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for objects contained in this list:
   * @link libsbml#SBML_SPECIES_TYPE SBML_SPECIES_TYPE@endlink (default).
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers ListOfSpeciesTypes::getElementName() const "
/**
   * Returns the XML element name of this object.
   *
   * For ListOfSpeciesTypes, the XML element name is
   * @c \'listOfSpeciesTypes\'.
   * 
   * @return the name of this element, i.e., @c \'listOfSpeciesTypes\'.
   */ public new
";


%csmethodmodifiers ListOfSpeciesTypes::get(unsigned int n) "
/**
   * Get a SpeciesType from the ListOfSpeciesTypes.
   *
   * @param n the index number of the SpeciesType to get.
   * 
   * @return the nth SpeciesType in this ListOfSpeciesTypes.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfSpeciesTypes::get "
/**
   * Get a SpeciesType from the ListOfSpeciesTypes.
   *
   * @param n the index number of the SpeciesType to get.
   * 
   * @return the nth SpeciesType in this ListOfSpeciesTypes.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfSpeciesTypes::get(const std::string& sid) "
/**
   * Get a SpeciesType from the ListOfSpeciesTypes
   * based on its identifier.
   *
   * @param sid a string representing the identifier 
   * of the SpeciesType to get.
   * 
   * @return SpeciesType in this ListOfSpeciesTypes
   * with the given @p sid or @c null if no such
   * SpeciesType exists.
   *
   * @see get(unsigned int n)
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfSpeciesTypes::get(const std::string& sid) const "
/**
   * Get a SpeciesType from the ListOfSpeciesTypes
   * based on its identifier.
   *
   * @param sid a string representing the identifier 
   * of the SpeciesType to get.
   * 
   * @return SpeciesType in this ListOfSpeciesTypes
   * with the given @p sid or @c null if no such
   * SpeciesType exists.
   *
   * @see get(unsigned int n)
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfSpeciesTypes::remove(unsigned int n) "
/**
   * Removes the nth item from this ListOfSpeciesTypes items and returns a pointer to
   * it.
   *
   * The caller owns the returned item and is responsible for deleting it.
   *
   * @param n the index of the item to remove.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfSpeciesTypes::remove(const std::string& sid) "
/**
   * Removes item in this ListOfSpeciesTypes items with the given identifier.
   *
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier @p sid, then
   * @c null is returned.
   *
   * @param sid the identifier of the item to remove.
   *
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public new
";


%csmethodmodifiers ListOfSpeciesTypes::getElementPosition() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOfSpeciesTypes::createObject(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%typemap(csimports) Compartment "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html An SBML compartment, where species are located.
 *
 * A compartment in SBML represents a bounded space in which species are
 * located.  Compartments do not necessarily have to correspond to actual
 * structures inside or outside of a biological cell.
 *
 * It is important to note that although compartments are optional in the
 * overall definition of Model, every species in an SBML model must be
 * located in a compartment.  This in turn means that if a model defines
 * any species, the model must also define at least one compartment.  The
 * reason is simply that species represent physical things, and therefore
 * must exist @em somewhere.  Compartments represent the @em somewhere.
 *
 * Compartment has one required attribute, \'id\', to give the compartment a
 * unique identifier by which other parts of an SBML model definition can
 * refer to it.  A compartment can also have an optional \'name\' attribute
 * of type @c string.  Identifiers and names must be used according to the
 * guidelines described in the SBML specifications.
 *
 * Compartment also has an optional attribute \'spatialDimensions\' that is
 * used to indicate the number of spatial dimensions possessed by the
 * compartment.  Most modeling scenarios involve compartments with integer
 * values of \'spatialDimensions\' of @c 3 (i.e., a three-dimensional
 * compartment, which is to say, a volume), or 2 (a two-dimensional
 * compartment, a surface), or @c 1 (a one-dimensional compartment, a
 * line).  In SBML Level&nbsp;3, the type of this attribute is @c double,
 * there are no restrictions on the permitted values of the
 * \'spatialDimensions\' attribute, and there are no default values.  In SBML
 * Level&nbsp;2, the value must be a positive @c integer, and the default
 * value is @c 3; the permissible values in SBML Level&nbsp;2 are @c 3,
 * @c 2, @c 1, and @c 0 (for a point).
 *
 * Another optional attribute on Compartment is \'size\', representing the @em
 * initial total size of that compartment in the model.  The \'size\' attribute
 * must be a floating-point value and may represent a volume (if the
 * compartment is a three-dimensional one), or an area (if the compartment is
 * two-dimensional), or a length (if the compartment is one-dimensional).
 * There is no default value of compartment size in SBML Level&nbsp;2 or
 * Level&nbsp;3.  In particular, a missing \'size\' value <em>does not imply
 * that the compartment size is 1</em>.  (This is unlike the definition of
 * compartment \'volume\' in SBML Level&nbsp;1.)  When the compartment\'s
 * \'spatialDimensions\' attribute does not have a value of @c 0, a missing
 * value of \'size\' for a given compartment signifies that the value either is
 * unknown, or to be obtained from an external source, or determined by an
 * InitialAssignment, AssignmentRule, AlgebraicRule or RateRule
 * @if conly structure @else object@endif elsewhere in the model.  In SBML
 * Level&nbsp;2, there are additional special requirements on the values of
 * \'size\'; we discuss them in a <a href=\'#comp-l2\'>separate section
 * below</a>.
 *
 * The units associated with a compartment\'s \'size\' attribute value may be
 * set using the optional attribute \'units\'.  The rules for setting and
 * using compartment size units differ between SBML Level&nbsp;2 and
 * Level&nbsp;3, and are discussed separately below.
 *
 * Finally, the Compartment attribute named \'constant\' is used to
 * indicate whether the compartment\'s size stays constant after simulation
 * begins.  A value of @c true indicates the compartment\'s \'size\' cannot be
 * changed by any other construct except InitialAssignment; a value of
 * @c false indicates the compartment\'s \'size\' can be changed by other
 * constructs in SBML.  In SBML Level&nbsp;2, there is an additional
 * explicit restriction that if \'spatialDimensions\'=@c \'0\', the value
 * cannot be changed by InitialAssignment either.  Further, in
 * Level&nbsp;2, \'constant\' is optional, and has a default value of @c true.  In SBML
 * Level&nbsp;3, there is no default value for the \'constant\' attribute,
 * and it is required.
 *
 *
 * @section comp-l2 Additional considerations in SBML Level&nbsp;2
 *
 * In SBML Level&nbsp;2, the default units of compartment size, and the kinds
 * of units allowed as values of the attribute \'units\', interact with the
 * number of spatial dimensions of the compartment.  The value of the \'units\'
 * attribute of a Compartment @if conly structure @else object@endif must
 * be one of the base units (see Unit), or the predefined unit identifiers
 * @c volume, @c area, @c length or @c dimensionless, or a new unit defined by a
 * UnitDefinition @if conly structure @else object@endif in the enclosing
 * Model, subject to the restrictions detailed in the following table:
 *
 * <table border=\'0\' class=\'centered text-table width80 normal-font alt-row-colors\'
 *        style=\'padding-bottom: 0.5em\'>
 * <caption class=\'top-caption\'>Restrictions on values permitted for
 * compartment <code>size</code> and <code>units</code> attributes.</caption>
 *  <tr>
 *      <th align=\'left\' valign=\'bottom\'>
 *        Value of<br><code>spatialDimensions</code>
 *      </th>
 *      <th align=\'left\' valign=\'bottom\'>
 *        <code>size</code><br>allowed?
 *      </th>
 *      <th align=\'left\' valign=\'bottom\'>
 *        <code>units</code><br>allowed?
 *      </th>
 *      <th align=\'left\' valign=\'bottom\'>
 * 	  Allowable kinds of units
 *      </th>
 *      <th align=\'left\' valign=\'bottom\'>
 * 	  Default value of attribute <code>units</code>
 *      </th>
 *  </tr>
 *  <tr>
 *      <td><code>3</code></td>
 *      <td>yes</td>
 *      <td>yes</td>
 *      <td>units of volume, or <code>dimensionless</code></td>
 *      <td><code>volume</code></td>
 *  </tr>
 *  <tr>
 *      <td><code>2</code></td>
 *      <td>yes</td>
 *      <td>yes</td>
 *      <td>units of area, or <code>dimensionless</code></td>
 *      <td><code>area</code></td>
 *  </tr>
 *  <tr>
 *      <td><code>1</code></td>
 *      <td>yes</td>
 *      <td>yes</td>
 *      <td>units of length, or <code>dimensionless</code></td>
 *      <td><code>length</code></td>
 *  </tr>
 *  <tr>
 *      <td><code>0</code></td>
 *      <td>no</td>
 *      <td>no</td>
 *      <td>(no units allowed)</td>
 *      <td></td>
 *  </tr>
 *  </tr>
 * </table>
 *
 * In SBML Level&nbsp;2, the units of the compartment size, as defined by the
 * \'units\' attribute or (if \'units\' is not set) the default value listed in
 * the table above, are used in the following ways when the compartment has
 * a \'spatialDimensions\' value greater than @c 0:
 * <ul>
 * <li> The value of the \'units\' attribute is used as the units of the
 * compartment identifier when the identifier appears as a numerical
 * quantity in a mathematical formula expressed in MathML.
 *
 * <li> The @c math element of an AssignmentRule or InitialAssignment
 * referring to this compartment @em must (in Level&nbsp;2 Versions&nbsp;1-3)
 * or @em should (in Level&nbsp;2 Version 4) have identical units.
 *
 * <li> In RateRule objects that set the rate of change of the compartment\'s
 * size, the units of the rule\'s @c math element @em must (in Level&nbsp;2 
 * Versions&nbsp;1&ndash;3) or @em should (in Level&nbsp;2 Version 4) be identical to the
 * compartment\'s units (whether defined by the \'units\' attribute or by taking the 
 * default value from the Model) divided by the default @em time units.
 * (In other words, the units for the rate of change of compartment size
 * are <em>compartment size</em>/<em>time</em> units.
 *
 * <li> When a Species is to be treated in terms of concentrations or
 * density, the units of the spatial size portion of the concentration
 * value (i.e., the denominator in the units formula @em substance/@em
 * size) are those indicated by the value of the \'units\' attribute on the
 * compartment in which the species is located.
 * </ul>
 *
 * Compartments with \'spatialDimensions\'=@c 0 require special treatment in
 * this framework.  As implied above, the \'size\' attribute must not have a
 * value on an SBML Level&nbsp;2 Compartment
 * @if conly structure @else object@endif if the \'spatialDimensions\'
 * attribute has a value of @c 0.  An additional related restriction is that
 * the \'constant\' attribute must default to or be set to @c true if the value
 * of the \'spatialDimensions\' attribute is @c 0, because a zero-dimensional
 * compartment cannot ever have a size.
 *
 * If a compartment has no size or dimensional units, how should such a
 * compartment\'s identifier be interpreted when it appears in mathematical
 * formulas?  The answer is that such a compartment\'s identifier should not
 * appear in mathematical formulas in the first place---it has no
 * value, and its value cannot change.  Note also that a zero-dimensional
 * compartment is a point, and species located at points can only be
 * described in terms of amounts, not spatially-dependent measures such as
 * concentration.  Since SBML KineticLaw formulas are already in terms of
 * @em substance/@em time and not (say) @em concentration/@em time, volume
 * or other factors in principle are not needed for species located in
 * zero-dimensional compartments.
 *
 * Finally, in SBML Level&nbsp;2 Versions 2&ndash;4, each compartment in a
 * model may optionally be designated as belonging to a particular
 * compartment @em type.  The optional attribute \'compartmentType\' is used
 * identify the compartment type represented by the Compartment structure.
 * The \'compartmentType\' attribute\'s value must be the identifier of a
 * CompartmentType instance defined in the model.  If the \'compartmentType\'
 * attribute is not present on a particular compartment definition, a
 * unique virtual compartment type is assumed for that compartment, and no
 * other compartment can belong to that compartment type.  The values of
 * \'compartmentType\' attributes on compartments have no effect on the
 * numerical interpretation of a model.  Simulators and other numerical
 * analysis software may ignore \'compartmentType\' attributes.  The
 * \'compartmentType\' attribute and the CompartmentType
 * @if conly structures @else class of objects@endif are
 * not present in SBML Level&nbsp;3 Core nor in SBML Level&nbsp;1.
 *
 *
 * @section comp-l3 Additional considerations in SBML Level&nbsp;3
 *
 * One difference between SBML Level&nbsp;3 and lower Levels of SBML is
 * that there are no restrictions on the permissible values of the
 * \'spatialDimensions\' attribute, and there is no default value defined for
 * the attribute.  The value of \'spatialDimensions\' does not have to be an
 * integer, either; this is to allow for the possibility of representing
 * structures with fractal dimensions.
 *
 * The number of spatial dimensions possessed by a compartment cannot enter
 * into mathematical formulas, and therefore cannot directly alter the
 * numerical interpretation of a model.  However, the value of
 * \'spatialDimensions\' @em does affect the interpretation of the units
 * associated with a compartment\'s size.  Specifically, the value of
 * \'spatialDimensions\' is used to select among the Model attributes
 * \'volumeUnits\', \'areaUnits\' and \'lengthUnits\' when a Compartment
 * @if conly object @else structure@endif does not define a value for its
 * \'units\' attribute.
 *
 * The \'units\' attribute may be left unspecified for a given compartment in a
 * model; in that case, the compartment inherits the unit of measurement
 * specified by one of the attributes on the enclosing Model
 * @if conly structure @else object@endif instance.  The applicable
 * attribute on Model depends on the value of the compartment\'s
 * \'spatialDimensions\' attribute; the relationship is shown in the table
 * below.  If the Model @if conly structure @else object@endif does not
 * define the relevant attribute (\'volumeUnits\', \'areaUnits\' or
 * \'lengthUnits\') for a given \'spatialDimensions\' value, the unit associated
 * with that Compartment @if conly structure @else object@endif\'s size is
 * undefined.  If a given Compartment\'s \'units\' are left unset and 
 * the \'spatialDimensions\' either has a value other than @c 1, @c 2, or 
 * @c 3 or is left unset itself (as it has no default value),
 * then no unit can be chosen from among the Model\'s \'volumeUnits\',
 * \'areaUnits\' or \'lengthUnits\' attributes (even if the Model instance
 * provides values for those attributes), because there is no basis to select
 * between them.
 * Leaving the units of compartments\' sizes undefined in an SBML model does
 * not render the model invalid; however, as a matter of best practice, we
 * strongly recommend that all models specify the units of measurement for
 * all compartment sizes.
 *
 * <table border=\'0\' class=\'centered text-table width80 normal-font alt-row-colors\'
 *        style=\'padding-bottom: 0.5em\'>
 * <caption class=\'top-caption\'>Interpretation of the Compartment \'units\' attribute.</caption>
 *  <tr>
 *      <th align=\'left\' valign=\'bottom\'>
 *        Value of attribute<br>\'spatialDimensions\'
 *      </th>
 *      <th align=\'left\' valign=\'bottom\'>
 *        Attribute of Model used<br>for inheriting the unit
 *      </th>
 *      <th align=\'left\' valign=\'bottom\'>
 *        Recommended candidate units
 *      </th>
 *  </tr>
 *  <tr>
 *      <td><code>3</code></td>
 *      <td>&quot;volumeUnits&quot;</td>
 *      <td>units of volume, or <code>dimensionless</code></td>
 *  </tr>
 *  <tr>
 *      <td><code>2</code></td>
 *      <td>&quot;areaUnits&quot;</td>
 *      <td>units of area, or <code>dimensionless</code></td>
 *  </tr>
 *  <tr>
 *      <td><code>1</code></td>
 *      <td>&quot;lengthUnits&quot;</td>
 *      <td>units of length, or <code>dimensionless</code></td>
 *  </tr>
 *  <tr>
 *      <td><em>other</em></td>
 *      <td><em>no units inherited</em></td>
 *      <td><em>no specific recommendations</em></td>
 *  </tr>
 *  </tr>
 * </table>
 *
 * The unit of measurement associated with a compartment\'s size, as defined
 * by the \'units\' attribute or (if \'units\' is not set) the inherited value
 * from Model according to the table above, is used in the following ways:
 *
 * <ul>
 *
 * <li> When the identifier of the compartment appears as a numerical
 * quantity in a mathematical formula expressed in MathML, it represents
 * the size of the compartment, and the unit associated with the size is
 * the value of the \'units\' attribute.
 *
 * <li> When a Species is to be treated in terms of concentrations or
 * density, the unit associated with the spatial size portion of the
 * concentration value (i.e., the denominator in the formula
 * <em>amount</em>/<em>size</em>) is specified by the value of the \'units\'
 * attribute on the compartment in which the species is located.
 *
 * <li> The \'math\' elements of AssignmentRule, InitialAssignment and
 * EventAssignment @if conly structures @else objects@endif setting the
 * value of the compartment size should all have the same units as the unit
 * associated with the compartment\'s size.
 *
 * <li> In a RateRule @if conly structure @else object@endif that defines a
 * rate of change for a compartment\'s size, the unit of the rule\'s \'math\'
 * element should be identical to the compartment\'s \'units\' attribute divided
 * by the model-wide unit of <em>time</em>.  (In other words, {<em>unit of
 * compartment size</em>}/{<em>unit of time</em>}.)
 *
 * </ul>
 *
 *
 * @section comp-other Other aspects of Compartment
 *
 * In SBML Level&nbsp;1 and Level&nbsp;2, Compartment has an optional
 * attribute named \'outside\', whose value can be the identifier of another
 * Compartment @if conly structure @else object@endif defined in the
 * enclosing Model @if conly structure @else object@endif.  Doing so means
 * that the other compartment contains it or is outside of it.  This enables
 * the representation of simple topological relationships between
 * compartments, for those simulation systems that can make use of the
 * information (e.g., for drawing simple diagrams of compartments).  It is
 * worth noting that in SBML, there is no relationship between compartment
 * sizes when compartment positioning is expressed using the \'outside\'
 * attribute.  The size of a given compartment does not in any sense include
 * the sizes of other compartments having it as the value of their \'outside\'
 * attributes.  In other words, if a compartment @em B has the identifier of
 * compartment @em A as its \'outside\' attribute value, the size of @em A does
 * not include the size of @em B.  The compartment sizes are separate.
 *
 * In Level&nbsp;2, there are two restrictions on the \'outside\' attribute.
 * First, because a compartment with \'spatialDimensions\' of @c 0 has no
 * size, such a compartment cannot act as the container of any other
 * compartment @em except compartments that @em also have
 * \'spatialDimensions\' values of @c 0.  Second, the directed graph formed
 * by representing Compartment structures as vertexes and the \'outside\'
 * attribute values as edges must be acyclic.  The latter condition is
 * imposed to prevent a compartment from being contained inside itself.  In
 * the absence of a value for \'outside\', compartment definitions in SBML
 * Level&nbsp;2 do not have any implied spatial relationships between each
 * other.
 *
 *
 * 
 */
"


%typemap(csimports) ListOfCompartments "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html A list of Compartment objects.
 *
 *
 * 
 * The various ListOf___ @if conly structures @else classes@endif in SBML
 * are merely containers used for organizing the main components of an SBML
 * model.  In libSBML\'s implementation, ListOf___
 * @if conly data structures @else classes@endif are derived from the
 * intermediate utility @if conly structure @else class@endif ListOf, which
 * is not defined by the SBML specifications but serves as a useful
 * programmatic construct.  ListOf is itself is in turn derived from SBase,
 * which provides all of the various ListOf___
 * @if conly data structures @else classes@endif with common features
 * defined by the SBML specification, such as \'metaid\' attributes and
 * annotations.
 *
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 *
 * @htmlinclude listof-illustration.html
 *
 * SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
 * Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
 * Version&nbsp;2 allows
 * containers to contain zero or more of the relevant object, instead of 
 * requiring at least one.  As such, libsbml will write out an 
 * otherwise-empty ListOf___ element that has any optional attribute set 
 * (such as \'id\' or \'metaid\'), that has an optional child (such 
 * as a \'notes\' or \'annotation\'), or that has attributes or children set
 * from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
 * any other children.
 *
 * Readers may wonder about the motivations for using the ListOf___
 * containers in SBML.  A simpler approach in XML might be to place the
 * components all directly at the top level of the model definition.  The
 * choice made in SBML is to group them within XML elements named after
 * %ListOf<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from SBase means that software tools can add information @em about
 * the lists themselves into each list container\'s \'annotation\'.
 *
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 *
 * @if conly
 * @note In the C API for libSBML, functions that in other language APIs
 * would be inherited by the various ListOf___ structures not shown in the
 * pages for the individual ListOf___\'s.  Instead, the functions are defined
 * on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
 * the many common functions available for manipulating ListOf___
 * structures</strong>.  The documentation for the individual ListOf___
 * structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
 * all of the functionality available. @endif
 *
 *
 */
"


%csmethodmodifiers Compartment::Compartment(unsigned int level, unsigned int version) "
/**
   * Creates a new Compartment object using the given SBML @p level and @p
   * version values.
   *
   * @param level a long integer, the SBML Level to assign to this Compartment.
   *
   * @param version a long integer, the SBML Version to assign to this
   * Compartment.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers Compartment::Compartment(SBMLNamespaces* sbmlns) "
/**
   * Creates a new Compartment object using the given SBMLNamespaces object
   * @p sbmlns.
   *
   *
 * 
 * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
 * information.  It is used to communicate the SBML Level, Version, and (in
 * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
 * common approach to using libSBML\'s SBMLNamespaces facilities is to create an
 * SBMLNamespaces object somewhere in a program once, then hand that object
 * as needed to object constructors that accept SBMLNamespaces as arguments.
 *
 *
   *
   * It is worth emphasizing that although this constructor does not take an
   * identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a Compartment object is required to have a
   * value.  Thus, callers are cautioned to assign a value after calling this
   * constructor.  Setting the identifier can be accomplished using the
   * method @if java Compartment::setId(String id)@else setId()@endif.
   *
   * @param sbmlns an SBMLNamespaces object.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers Compartment::Compartment(const Compartment& orig) "
/**
   * Copy constructor.
   *
   * This creates a copy of a Compartment object.
   *
   * @param orig the Compartment instance to copy.
   */ public
";


%csmethodmodifiers Compartment::accept(SBMLVisitor& v) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Compartment::clone() const "
/**
   * Creates and returns a deep copy of this Compartment object.
   *
   * @return the (deep) copy of this Compartment object.
   */ public new
";


%csmethodmodifiers Compartment::initDefaults "
/**
   * Initializes the fields of this Compartment object to \'typical\' default
   * values.
   *
   * The SBML Compartment component has slightly different aspects and
   * default attribute values in different SBML Levels and Versions.
   * This method sets the values to certain common defaults, based
   * mostly on what they are in SBML Level&nbsp;2.  Specifically:
   *
   * @li Sets attribute \'spatialDimensions\' to @c 3
   * @li Sets attribute \'constant\' to @c true
   * @li (Applies to Level&nbsp;1 models only) Sets attribute \'volume\' to @c 1.0
   * @li (Applies to Level&nbsp;3 models only) Sets attribute \'units\' to @c litre
   */ public
";


%csmethodmodifiers Compartment::getId() const "
/**
   * Returns the value of the \'id\' attribute of this Compartment.
   *
   * @note Because of the inconsistent behavior of this function with 
   * respect to assignments and rules, it is now recommended to
   * use the getIdAttribute() function instead.
   *
   *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
   *
   * @return the id of this Compartment.
   *
   * @see getIdAttribute()
   * @see setIdAttribute(string sid)
   * @see isSetIdAttribute()
   * @see unsetIdAttribute()
   */ public new
";


%csmethodmodifiers Compartment::getName() const "
/**
   * Returns the value of the \'name\' attribute of this Compartment object.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 *
 * @return the name of this SBML object, or the empty string if not set or unsettable.
 *
 * @see getIdAttribute()
 * @see isSetName()
 * @see setName(string sid)
 * @see unsetName()
 * 
 *
   */ public new
";


%csmethodmodifiers Compartment::getCompartmentType() const "
/**
   * Get the value of the \'compartmentType\' attribute of this Compartment
   * object.
   *
   * @return the value of the \'compartmentType\' attribute of this
   * Compartment object as a string.
   *
   * @note The \'compartmentType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   *
   * @see isSetCompartmentType()
   * @see setCompartmentType(@if java String@endif)
   * @see unsetCompartmentType()
   */ public
";


%csmethodmodifiers Compartment::getSpatialDimensions() const "
/**
   * Get the number of spatial dimensions of this Compartment object.
   *
   * @return the value of the \'spatialDimensions\' attribute of this
   * Compartment object as a long integereger.
   *
   *
 * @note In SBML Level&nbsp;3, the data type of the \'spatialDimensions\'
 * attribute is @c double, whereas in Level&nbsp;2, it is @c integer.  To
 * avoid backward compatibility issues, libSBML provides two separate methods
 * for obtaining the value as either an integer or a type @c double, for
 * models where it is relevant.
 *
 *
   *
   * @see getSpatialDimensionsAsDouble()
   * @see setSpatialDimensions(@if java unsigned int@endif)
   * @see isSetSpatialDimensions()
   * @see unsetSpatialDimensions()
   */ public
";


%csmethodmodifiers Compartment::getSpatialDimensionsAsDouble() const "
/**
   * Get the number of spatial dimensions of this Compartment object,
   * as a double.
   *
   * @return the value of the \'spatialDimensions\' attribute of this
   * Compartment object as a double, or @c NaN if this model is not in SBML
   * Level&nbsp;3 format.
   *
   *
 * @note In SBML Level&nbsp;3, the data type of the \'spatialDimensions\'
 * attribute is @c double, whereas in Level&nbsp;2, it is @c integer.  To
 * avoid backward compatibility issues, libSBML provides two separate methods
 * for obtaining the value as either an integer or a type @c double, for
 * models where it is relevant.
 *
 *
   *
   * @see getSpatialDimensions()
   * @see setSpatialDimensions(@if java unsigned int@endif)
   * @see isSetSpatialDimensions()
   * @see unsetSpatialDimensions()
   */ public
";


%csmethodmodifiers Compartment::getSize() const "
/**
   * Get the size of this Compartment object.
   *
   *
 * 
 * In SBML Level&nbsp;1, compartments are always three-dimensional
 * constructs and only have volumes, whereas in SBML Level&nbsp;2 and higher,
 * compartments may be other than three-dimensional, and therefore the
 * \'volume\' attribute is named \'size\' in Level&nbsp;2 and above.  LibSBML
 * provides both @if conly Compartment_getSize() and Compartment_getVolume()
 * @else getSize() and getVolume()@endif
 * for easier support of different SBML Levels.
 *
 *
   *
   * @return the value of the \'size\' attribute (\'volume\' in Level&nbsp;1) of
   * this Compartment object as a floating-point number.
   *
   * @note This method is identical to
   * @if java Compartment::getVolume()@else getVolume()@endif.
   *
   * @see getVolume()
   * @see isSetSize()
   * @see setSize(@if java double@endif)
   * @see unsetSize()
   */ public
";


%csmethodmodifiers Compartment::getVolume() const "
/**
   * Get the volume of this Compartment object.
   *
   *
 * 
 * In SBML Level&nbsp;1, compartments are always three-dimensional
 * constructs and only have volumes, whereas in SBML Level&nbsp;2 and higher,
 * compartments may be other than three-dimensional, and therefore the
 * \'volume\' attribute is named \'size\' in Level&nbsp;2 and above.  LibSBML
 * provides both @if conly Compartment_getSize() and Compartment_getVolume()
 * @else getSize() and getVolume()@endif
 * for easier support of different SBML Levels.
 *
 *
   *
   * @return the value of the \'volume\' attribute (\'size\' in Level&nbsp;2) of
   * this Compartment object, as a floating-point number.
   *
   *
 * @note The attribute \'volume\' only exists by that name in SBML
 * Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
 * named \'size\'.  In SBML Level&nbsp;1, a compartment\'s volume has a default
 * value (@c 1.0) and therefore methods such as
 * @if conly Compartment_isSetVolume() @else isSetVolume()@endif
 * will always return @c true for a Level&nbsp;1 model.  In Level&nbsp;2, a
 * compartment\'s size (the equivalent of SBML Level&nbsp;1\'s \'volume\') is
 * optional and has no default value, and therefore may or may not be set.
 *
 *
   *
   * @note This method is identical to
   * @if java Compartment::getSize()@else getSize()@endif.
   *
   * @see getSize()
   * @see isSetVolume()
   * @see setVolume(@if java double@endif)
   * @see unsetVolume()
   */ public
";


%csmethodmodifiers Compartment::getUnits() const "
/**
   * Get the units of this Compartment object\'s size.
   *
   * The value of an SBML compartment\'s \'units\' attribute establishes the
   * unit of measurement associated with the compartment\'s size.
   *
   * @return the value of the \'units\' attribute of this Compartment object,
   * as a string.  An empty string indicates that no units have been assigned
   * to the value of the size.
   *
   *
 * @note There is an important distinction to be made between <em>no units
 * assigned</em>, and assuming a value without units has any specific unit
 * such as <code>dimensionless</code>.  In SBML, default units are never
 * attributed to numbers, and numbers without units are not automatically
 * assumed to have the unit <code>dimensionless</code>.  Please consult the
 * relevant SBML specification document for a more in-depth explanation of
 * this topic and the SBML unit system.
 *
 *
   *
   * @see isSetUnits()
   * @see setUnits(@if java String@endif)
   * @see unsetUnits()
   */ public
";


%csmethodmodifiers Compartment::getOutside() const "
/**
   * Get the identifier, if any, of the Compartment object that is designated
   * as being outside of @em this one.
   *
   * @return the value of the \'outside\' attribute of this Compartment object.
   *
   * @note The \'outside\' attribute is defined in SBML Level&nbsp;1 and
   * Level&nbsp;2, but does not exist in SBML Level&nbsp;3.
   *
   * @see isSetOutside()
   * @see setOutside(@if java String@endif)
   * @see unsetOutside()
   */ public
";


%csmethodmodifiers Compartment::getConstant() const "
/**
   * Get the value of the \'constant\' attribute of this Compartment object.
   *
   * @return @c true if this Compartment object\'s size is flagged as being
   * constant, @c false otherwise.
   *
   * @see isSetConstant()
   * @see setConstant(@if java bool@endif)
   */ public
";


%csmethodmodifiers Compartment::isSetId() const "
/**
   * Predicate returning @c true if this Compartment object\'s \'id\' attribute
   * is set.
   *
   *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 *
 * @return @c true if the \'id\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @note Because of the inconsistent behavior of this function with
 * respect to assignments and rules, it is recommended that callers
 * use isSetIdAttribute() instead.
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see unsetIdAttribute()
 * @see isSetIdAttribute()
 *
 *
   */ public new
";


%csmethodmodifiers Compartment::isSetName() const "
/**
   * Predicate returning @c true if this Compartment object\'s \'name\'
   * attribute is set.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 * @return @c true if the \'name\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @see getName()
 * @see setName(string sid)
 * @see unsetName()
 *
 *
   */ public new
";


%csmethodmodifiers Compartment::isSetCompartmentType() const "
/**
   * Predicate returning @c true if this Compartment object\'s
   * \'compartmentType\' attribute is set.
   *
   * @return @c true if the \'compartmentType\' attribute of this Compartment
   * is set, @c false otherwise.
   *
   * @note The \'compartmentType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   *
   * @see setCompartmentType(@if java String@endif)
   * @see getCompartmentType()
   * @see unsetCompartmentType()
   */ public
";


%csmethodmodifiers Compartment::isSetSize() const "
/**
   * Predicate returning @c true if this Compartment object\'s \'size\'
   * attribute is set.
   *
   * This method is similar but not identical to
   * @if java Compartment::isSetVolume()@else isSetVolume()@endif.  The latter
   * should be used in the context of SBML Level&nbsp;1 models instead of
   * @if java Compartment::isSetSize()@else isSetSize()@endif
   * because @if java Compartment::isSetVolume()@else isSetVolume()@endif
   * performs extra processing to take into account the difference in
   * default values between SBML Levels&nbsp;1 and&nbsp;2.
   *
   * @return @c true if the \'size\' attribute (\'volume\' in Level&nbsp;2) of
   * this Compartment object is set, @c false otherwise.
   *
   * @see isSetVolume()
   * @see setSize(@if java double@endif)
   * @see getSize()
   * @see unsetSize()
   */ public
";


%csmethodmodifiers Compartment::isSetVolume() const "
/**
   * Predicate returning @c true if this Compartment object\'s \'volume\'
   * attribute is set.
   *
   * This method is similar but not identical to
   * @if java Compartment::isSetSize()@else isSetSize()@endif.  The latter
   * should not be used in the context of SBML Level&nbsp;1 models because the
   * present method performs extra processing to take into account
   * the difference in default values between SBML Levels&nbsp;1 and&nbsp;2.
   *
   * @return @c true if the \'volume\' attribute (\'size\' in Level&nbsp;2 and
   * above) of this Compartment object is set, @c false otherwise.
   *
   *
 * @note The attribute \'volume\' only exists by that name in SBML
 * Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
 * named \'size\'.  In SBML Level&nbsp;1, a compartment\'s volume has a default
 * value (@c 1.0) and therefore methods such as
 * @if conly Compartment_isSetVolume() @else isSetVolume()@endif
 * will always return @c true for a Level&nbsp;1 model.  In Level&nbsp;2, a
 * compartment\'s size (the equivalent of SBML Level&nbsp;1\'s \'volume\') is
 * optional and has no default value, and therefore may or may not be set.
 *
 *
   *
   * @see isSetSize()
   * @see getVolume()
   * @see setVolume(@if java double@endif)
   * @see unsetVolume()
   */ public
";


%csmethodmodifiers Compartment::isSetUnits() const "
/**
   * Predicate returning @c true if this Compartment object\'s \'units\'
   * attribute is set.
   *
   * @return @c true if the \'units\' attribute of this Compartment object is
   * set, @c false otherwise.
   *
   *
 * @note There is an important distinction to be made between <em>no units
 * assigned</em>, and assuming a value without units has any specific unit
 * such as <code>dimensionless</code>.  In SBML, default units are never
 * attributed to numbers, and numbers without units are not automatically
 * assumed to have the unit <code>dimensionless</code>.  Please consult the
 * relevant SBML specification document for a more in-depth explanation of
 * this topic and the SBML unit system.
 *
 *
   *
   * @see setUnits(@if java String@endif)
   * @see getUnits()
   * @see unsetUnits()
   */ public
";


%csmethodmodifiers Compartment::isSetOutside() const "
/**
   * Predicate returning @c true if this Compartment object\'s \'outside\'
   * attribute is set.
   *
   * @return @c true if the \'outside\' attribute of this Compartment object is
   * set, @c false otherwise.
   *
   * @note The \'outside\' attribute is defined in SBML Level&nbsp;1 and
   * Level&nbsp;2, but does not exist in SBML Level&nbsp;3.
   *
   * @see getOutside()
   * @see setOutside(@if java String@endif)
   * @see unsetOutside()
   */ public
";


%csmethodmodifiers Compartment::isSetSpatialDimensions() const "
/**
   * Predicate returning @c true if this Compartment object\'s
   * \'spatialDimensions\' attribute is set.
   *
   * @return @c true if the \'spatialDimensions\' attribute of this
   * Compartment object is set, @c false otherwise.
   *
   * @see getSpatialDimensions()
   * @see setSpatialDimensions(@if java unsigned int@endif)
   * @see unsetSpatialDimensions()
   */ public
";


%csmethodmodifiers Compartment::isSetConstant() const "
/**
   * Predicate returning @c true if this Compartment object\'s \'constant\'
   * attribute is set.
   *
   * @return @c true if the \'constant\' attribute of this Compartment object is
   * set, @c false otherwise.
   *
   * @see getConstant()
   * @see setConstant(@if java bool@endif)
   */ public
";


%csmethodmodifiers Compartment::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this Compartment object.
   *
   * The string @p sid is copied.
   *
   *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
   *
   * @param sid the string to use as the identifier of this Compartment object. If
   * the string is @c null, this method will return
   * @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   *
   * @see getId()
   * @see unsetId()
   * @see isSetId()
   */ public new
";


%csmethodmodifiers Compartment::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this Compartment object.
   *
   *
 * 
 *
 * The string in @p name is copied.
 *
 * @param name the new name for the SBML object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 *
 *
   */ public new
";


%csmethodmodifiers Compartment::setCompartmentType(const std::string& sid) "
/**
   * Sets the \'compartmentType\' attribute of this Compartment object.
   *
   * @param sid the identifier of a CompartmentType object defined elsewhere
   * in this Model. If the string is @c null, this method will return
   * @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   * @note The \'compartmentType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   *
   * @see isSetCompartmentType()
   * @see getCompartmentType()
   * @see unsetCompartmentType()
   */ public
";


%csmethodmodifiers Compartment::setSpatialDimensions(unsigned int value) "
/**
   * Sets the \'spatialDimensions\' attribute of this Compartment object.
   *
   * @param value a long integereger indicating the number of dimensions
   * of this compartment.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   * @see getSpatialDimensions()
   * @see isSetSpatialDimensions()
   * @see unsetSpatialDimensions()
   */ public
";


%csmethodmodifiers Compartment::setSpatialDimensions(double value) "
/**
   * Sets the \'spatialDimensions\' attribute of this Compartment object as a double.
   *
   * @param value a double indicating the number of dimensions
   * of this compartment.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   * @see getSpatialDimensions()
   * @see isSetSpatialDimensions()
   * @see unsetSpatialDimensions()
   */ public
";


%csmethodmodifiers Compartment::setSize(double value) "
/**
   * Sets the \'size\' attribute (or \'volume\' in SBML Level&nbsp;1) of this
   * Compartment object.
   *
   * @param value a @c double representing the size of this compartment
   * instance in whatever units are in effect for the compartment.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif This particular
 * function only does one thing irrespective of user input or 
 * object state, and thus will only return a single value:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   *
   * @note This method is identical to
   * @if java Compartment::setVolume(double value)@else setVolume()@endif.
   *
   * @see setVolume(@if java double@endif)
   * @see getSize()
   * @see isSetSize()
   * @see unsetSize()
   */ public
";


%csmethodmodifiers Compartment::setVolume(double value) "
/**
   * Sets the \'volume\' attribute (or \'size\' in SBML Level&nbsp;2) of this
   * Compartment object.
   *
   * This method is identical to
   * @if java Compartment::setSize(double value)@else setSize()@endif
   * and is provided for compatibility between SBML Level&nbsp;1 and
   * higher Levels of SBML.
   *
   * @param value a @c double representing the volume of this compartment
   * instance in whatever units are in effect for the compartment.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif This particular
 * function only does one thing irrespective of user input or 
 * object state, and thus will only return a single value:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   *
   *
 * @note The attribute \'volume\' only exists by that name in SBML
 * Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
 * named \'size\'.  In SBML Level&nbsp;1, a compartment\'s volume has a default
 * value (@c 1.0) and therefore methods such as
 * @if conly Compartment_isSetVolume() @else isSetVolume()@endif
 * will always return @c true for a Level&nbsp;1 model.  In Level&nbsp;2, a
 * compartment\'s size (the equivalent of SBML Level&nbsp;1\'s \'volume\') is
 * optional and has no default value, and therefore may or may not be set.
 *
 *
   *
   * @see setSize(@if java double@endif)
   * @see getVolume()
   * @see isSetVolume()
   * @see unsetVolume()
   */ public
";


%csmethodmodifiers Compartment::setUnits(const std::string& sid) "
/**
   * Sets the \'units\' attribute of this Compartment object.
   *
   * @param sid the identifier of the defined units to use.  If @p sid is
   * @c null, then this method will return
   * @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   *
   * @see isSetUnits()
   * @see getUnits()
   * @see unsetUnits()
   */ public
";


%csmethodmodifiers Compartment::setOutside(const std::string& sid) "
/**
   * Sets the \'outside\' attribute of this Compartment object.
   *
   * @param sid the identifier of a compartment that encloses this one. If @p
   * sid is @c null, then this method will return
   * @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   *
   * @note The \'outside\' attribute is defined in SBML Level&nbsp;1 and
   * Level&nbsp;2, but does not exist in SBML Level&nbsp;3.
   *
   * @see isSetOutside()
   * @see getOutside()
   * @see unsetOutside()
   */ public
";


%csmethodmodifiers Compartment::setConstant(bool value) "
/**
   * Sets the value of the \'constant\' attribute of this Compartment object.
   *
   * @param value a boolean indicating whether the size/volume of this
   * compartment should be considered constant (@c true) or variable
   * (@c false).
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   * @see isSetConstant()
   * @see getConstant()
   */ public
";


%csmethodmodifiers Compartment::renameSIdRefs(const std::string& oldid, const std::string& newid) "
/**
   *
 * Replaces all uses of a given @c SIdRef type attribute value with another
 * value.
 *
 *
 * 

 * In SBML, object identifiers are of a data type called <code>SId</code>.
 * In SBML Level&nbsp;3, an explicit data type called <code>SIdRef</code> was
 * introduced for attribute values that refer to <code>SId</code> values; in
 * previous Levels of SBML, this data type did not exist and attributes were
 * simply described to as \'referring to an identifier\', but the effective
 * data type was the same as <code>SIdRef</code> in Level&nbsp;3.  These and
 * other methods of libSBML refer to the type <code>SIdRef</code> for all
 * Levels of SBML, even if the corresponding SBML specification did not
 * explicitly name the data type.
 *
 *
 *
 * This method works by looking at all attributes and (if appropriate)
 * mathematical formulas in MathML content, comparing the referenced
 * identifiers to the value of @p oldid.  If any matches are found, the
 * matching values are replaced with @p newid.  The method does @em not
 * descend into child elements.
 *
 * @param oldid the old identifier.
 * @param newid the new identifier.
 *
 *
   */ public new
";


%csmethodmodifiers Compartment::renameUnitSIdRefs(const std::string& oldid, const std::string& newid) "
/**
   *
 * Replaces all uses of a given @c UnitSIdRef type attribute value with
 * another value.
 *
 *
 * 
 * In SBML, unit definitions have identifiers of type <code>UnitSId</code>.  In
 * SBML Level&nbsp;3, an explicit data type called <code>UnitSIdRef</code> was
 * introduced for attribute values that refer to <code>UnitSId</code> values; in
 * previous Levels of SBML, this data type did not exist and attributes were
 * simply described to as \'referring to a unit identifier\', but the effective
 * data type was the same as <code>UnitSIdRef</code> in Level&nbsp;3.  These and
 * other methods of libSBML refer to the type <code>UnitSIdRef</code> for all
 * Levels of SBML, even if the corresponding SBML specification did not
 * explicitly name the data type.
 *
 *
 *
 * This method works by looking at all unit identifier attribute values
 * (including, if appropriate, inside mathematical formulas), comparing the
 * referenced unit identifiers to the value of @p oldid.  If any matches
 * are found, the matching values are replaced with @p newid.  The method
 * does @em not descend into child elements.
 *
 * @param oldid the old identifier.
 * @param newid the new identifier.
 *
 *
   */ public new
";


%csmethodmodifiers Compartment::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this Compartment object.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
 *
 * @see getName()
 * @see setName(string sid)
 * @see isSetName()
 *
 *
   */ public new
";


%csmethodmodifiers Compartment::unsetCompartmentType "
/**
   * Unsets the value of the \'compartmentType\' attribute of this Compartment object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   * @note The \'compartmentType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   *
   * @see setCompartmentType(@if java String@endif)
   * @see isSetCompartmentType()
   * @see getCompartmentType()
   */ public
";


%csmethodmodifiers Compartment::unsetConstant "
/**
   * Unsets the value of the \'constant\' attribute of this Compartment object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @see isSetConstant()
   * @see setConstant(@if java bool@endif)
   * @see getConstant()
   */ public
";


%csmethodmodifiers Compartment::unsetSize "
/**
   * Unsets the value of the \'size\' attribute of this Compartment object.
   *
   * In SBML Level&nbsp;1, a compartment\'s volume has a default value
   * (@c 1.0) and therefore <em>should always be set</em>.  Calling this method
   * on a Level&nbsp;1 model resets the value to @c 1.0 rather than actually
   * unsetting it.  In Level&nbsp;2, a compartment\'s \'size\' is optional with
   * no default value, and unsetting it will result in the compartment having
   * no defined size.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @note This method is identical to
   * @if java Compartment::unsetVolume()@else unsetVolume()@endif.
   *
   * @see unsetVolume()
   * @see getSize()
   * @see isSetSize()
   * @see setSize(@if java double@endif)
   */ public
";


%csmethodmodifiers Compartment::unsetVolume "
/**
   * Unsets the value of the \'volume\' attribute of this Compartment object.
   *
   * This method is identical to
   * @if java Compartment::unsetSize()@else unsetSize()@endif.  Please refer
   * to that method\'s documentation for more information about its behavior.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note The attribute \'volume\' only exists by that name in SBML
 * Level&nbsp;1.  In Level&nbsp;2 and above, the equivalent attribute is
 * named \'size\'.  In SBML Level&nbsp;1, a compartment\'s volume has a default
 * value (@c 1.0) and therefore methods such as
 * @if conly Compartment_isSetVolume() @else isSetVolume()@endif
 * will always return @c true for a Level&nbsp;1 model.  In Level&nbsp;2, a
 * compartment\'s size (the equivalent of SBML Level&nbsp;1\'s \'volume\') is
 * optional and has no default value, and therefore may or may not be set.
 *
 *
   *
   * @see unsetSize()
   * @see getVolume()
   * @see setVolume(@if java double@endif)
   * @see isSetVolume()
   */ public
";


%csmethodmodifiers Compartment::unsetUnits "
/**
   * Unsets the value of the \'units\' attribute of this Compartment object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @see isSetUnits()
   * @see setUnits(@if java String@endif)
   * @see getUnits()
   */ public
";


%csmethodmodifiers Compartment::unsetOutside "
/**
   * Unsets the value of the \'outside\' attribute of this Compartment object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @note The \'outside\' attribute is defined in SBML Level&nbsp;1 and
   * Level&nbsp;2, but does not exist in SBML Level&nbsp;3.
   *
   * @see isSetOutside()
   * @see getOutside()
   * @see setOutside(@if java String@endif)
   */ public
";


%csmethodmodifiers Compartment::unsetSpatialDimensions "
/**
   * Unsets the value of the \'spatialDimensions\' attribute of this
   * Compartment object.
   *
   * In SBML Levels prior to Level&nbsp;3, compartments must always have a
   * value for the number of dimensions.  Consequently, calling this method
   * on a model of SBML Level 1&ndash;2 will result in a return value of
   * @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   * @note This function is only valid for SBML Level&nbsp;3.
   *
   * @see getSpatialDimensions()
   * @see isSetSpatialDimensions()
   * @see setSpatialDimensions(@if java unsigned int@endif)
   */ public
";


%csmethodmodifiers Compartment::getDerivedUnitDefinition "
/**
   * Constructs and returns a UnitDefinition that corresponds to the units
   * of this Compartment object\'s designated size.
   *
   *
 * 
 * Compartments in SBML have an attribute (\'units\') for declaring the
 * units of measurement intended for the value of the compartment\'s size.  In
 * the absence of a value given for this attribute, the units are inherited
 * from values either defined on the enclosing Model (in SBML Level&nbsp;3)
 * or in defaults (in SBML Level&nbsp;2).  This method returns a
 * @if conly UnitDefinition_t structure @else UnitDefinition object@endif
 * based on how this compartment\'s units are interpreted according to the
 * relevant SBML guidelines, or it returns @c null if no units have been
 * declared and no defaults are defined by the relevant SBML specification.
 *
 * Note that unit declarations for
 * @if conly Compartment_t structures @else Compartment objects @endif
 * are specified in terms of the @em identifier of a unit,
 * but @em this method returns @if conly a structure @else an object @endif,
 * not a unit identifier.  It does this by constructing an appropriate
 * @if conly UnitDefinition_t structure @else UnitDefinition object@endif.  For
 * SBML Level&nbsp;2 models, it will do this even when the value of the
 * \'units\' attribute is one of the special SBML Level&nbsp;2 unit identifiers
 * @c \'substance\', @c \'volume\', @c \'area\', @c \'length\' or @c \'time\'.  Callers
 * may find this useful in conjunction with the helper methods provided by
 * the @if conly UnitDefinition_t structure @else UnitDefinition
 * class@endif for comparing different UnitDefinition objects.
 *
 *
   *
   * @return a UnitDefinition that expresses the units of this
   * Compartment object, or @c null if one cannot be constructed.
   *
   *
 * @note The libSBML system for unit analysis depends on the model as a
 * whole.  In cases where the
 * @if conly Compartment_t structure @else Compartment object@endif has not
 * yet been added to a model, or the model itself is incomplete, unit
 * analysis is not possible, and consequently this method will return
 * @c null.
   *
   * @see isSetUnits()
   * @see getUnits()
   */ public
";


%csmethodmodifiers Compartment::getDerivedUnitDefinition() const "
/**
   * Constructs and returns a UnitDefinition that corresponds to the units
   * of this Compartment object\'s designated size.
   *
   *
 * 
 * Compartments in SBML have an attribute (\'units\') for declaring the
 * units of measurement intended for the value of the compartment\'s size.  In
 * the absence of a value given for this attribute, the units are inherited
 * from values either defined on the enclosing Model (in SBML Level&nbsp;3)
 * or in defaults (in SBML Level&nbsp;2).  This method returns a
 * @if conly UnitDefinition_t structure @else UnitDefinition object@endif
 * based on how this compartment\'s units are interpreted according to the
 * relevant SBML guidelines, or it returns @c null if no units have been
 * declared and no defaults are defined by the relevant SBML specification.
 *
 * Note that unit declarations for
 * @if conly Compartment_t structures @else Compartment objects @endif
 * are specified in terms of the @em identifier of a unit,
 * but @em this method returns @if conly a structure @else an object @endif,
 * not a unit identifier.  It does this by constructing an appropriate
 * @if conly UnitDefinition_t structure @else UnitDefinition object@endif.  For
 * SBML Level&nbsp;2 models, it will do this even when the value of the
 * \'units\' attribute is one of the special SBML Level&nbsp;2 unit identifiers
 * @c \'substance\', @c \'volume\', @c \'area\', @c \'length\' or @c \'time\'.  Callers
 * may find this useful in conjunction with the helper methods provided by
 * the @if conly UnitDefinition_t structure @else UnitDefinition
 * class@endif for comparing different UnitDefinition objects.
 *
 *
   *
   * @return a UnitDefinition that expresses the units of this
   * Compartment object, or @c null if one cannot be constructed.
   *
   *
 * @note The libSBML system for unit analysis depends on the model as a
 * whole.  In cases where the
 * @if conly Compartment_t structure @else Compartment object@endif has not
 * yet been added to a model, or the model itself is incomplete, unit
 * analysis is not possible, and consequently this method will return
 * @c null.
   *
   * @see isSetUnits()
   * @see getUnits()
   */ public
";


%csmethodmodifiers Compartment::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   *
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for this object:
   * @link libsbml#SBML_COMPARTMENT SBML_COMPARTMENT@endlink (default).
   *
   *
 * @warning <span class=\'warning\'>The specific integer values of the possible
 * type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
 * packages,  To fully identify the correct code, <strong>it is necessary to
 * invoke both getPackageName() and getTypeCode()</strong> (or 
 * ListOf::getItemTypeCode()).</span>
 *
 *
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers Compartment::getElementName() const "
/**
   * Returns the XML element name of this object
   *
   * For Compartment, the XML element name is always @c \'compartment\'.
   *
   * @return the name of this element.
   */ public new
";


%csmethodmodifiers Compartment::writeElements(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Compartment::hasRequiredAttributes() const "
/**
   * Predicate returning @c true if all the required attributes for this
   * Compartment object have been set.
   *
   * The required attributes for a Compartment object are:
   * @li \'id\' (or \'name\' in SBML Level&nbsp;1)
   * @li \'constant\' (in SBML Level&nbsp;3 only)
   *
   * @return @c true if the required attributes have been set, @c false
   * otherwise.
   */ public new
";


%csmethodmodifiers Compartment::addExpectedAttributes(ExpectedAttributes& attributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Compartment::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Compartment::readL1Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Compartment::readL2Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Compartment::readL3Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Compartment::writeAttributes(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Compartment::isExplicitlySetSpatialDimensions() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Compartment::isExplicitlySetConstant() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers ListOfCompartments::ListOfCompartments(unsigned int level, unsigned int version) "
/**
   * Creates a new ListOfCompartments object.
   *
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   *
   * @param level the SBML Level.
   *
   * @param version the Version within the SBML Level.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfCompartments::ListOfCompartments(SBMLNamespaces* sbmlns) "
/**
   * Creates a new ListOfCompartments object.
   *
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the SBMLNamespaces object in @p
   * sbmlns.
   *
   * @param sbmlns an SBMLNamespaces object that is used to determine the
   * characteristics of the ListOfCompartments object to be created.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfCompartments::clone() const "
/**
   * Creates and returns a deep copy of this ListOfCompartments object.
   *
   * @return the (deep) copy of this ListOfCompartments object.
   */ public new
";


%csmethodmodifiers ListOfCompartments::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this ListOf
   * (i.e., Compartment objects, if the list is non-empty).
   *
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for the objects contained in this ListOf
   * instance: @link libsbml#SBML_COMPARTMENT SBML_COMPARTMENT@endlink (default).
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers ListOfCompartments::getElementName() const "
/**
   * Returns the XML element name of this object.
   *
   * For ListOfCompartments, the XML element name is always
   * @c \'listOfCompartments\'.
   *
   * @return the name of this element.
   */ public new
";


%csmethodmodifiers ListOfCompartments::get(unsigned int n) "
/**
   * Get a Compartment object from the ListOfCompartments.
   *
   * @param n the index number of the Compartment object to get.
   *
   * @return the nth Compartment object in this ListOfCompartments.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfCompartments::get "
/**
   * Get a Compartment object from the ListOfCompartments.
   *
   * @param n the index number of the Compartment object to get.
   *
   * @return the nth Compartment object in this ListOfCompartments.
   * If the index @p n is invalid, @c null is returned.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfCompartments::get(const std::string& sid) "
/**
   * Get a Compartment object from the ListOfCompartments
   * based on its identifier.
   *
   * @param sid a string representing the identifier
   * of the Compartment object to get.
   *
   * @return Compartment object in this ListOfCompartments
   * with the given @p sid or @c null if no such
   * Compartment object exists.
   *
   * @see get(unsigned int n)
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfCompartments::get(const std::string& sid) const "
/**
   * Get a Compartment object from the ListOfCompartments
   * based on its identifier.
   *
   * @param sid a string representing the identifier
   * of the Compartment object to get.
   *
   * @return Compartment object in this ListOfCompartments
   * with the given @p sid or @c null if no such
   * Compartment object exists.
   *
   * @see get(unsigned int n)
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfCompartments::remove(unsigned int n) "
/**
   * Removes the nth item from this ListOfCompartments items and returns a pointer to
   * it.
   *
   * The caller owns the returned item and is responsible for deleting it.
   *
   * @param n the index of the item to remove.
   *
   * @see size()
   */ public new
";


%csmethodmodifiers ListOfCompartments::remove(const std::string& sid) "
/**
   * Removes item in this ListOfCompartments items with the given identifier.
   *
   * The caller owns the returned item and is responsible for deleting it.
   * If none of the items in this list have the identifier @p sid, then
   * @c null is returned.
   *
   * @param sid the identifier of the item to remove.
   *
   * @return the item removed.  As mentioned above, the caller owns the
   * returned item.
   */ public new
";


%csmethodmodifiers ListOfCompartments::getElementPosition() const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers ListOfCompartments::createObject(XMLInputStream& stream) "
/** */ /* libsbml-internal */ public new
";


%typemap(csimports) Species "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html An SBML <em>species</em> &ndash; a pool of entities.
 *
 * A @em species in SBML refers to a pool of entities that (a) are
 * considered indistinguishable from each other for the purposes of the
 * model, (b) participate in reactions, and (c) are located in a specific
 * @em compartment.  The SBML Species object class is intended to represent
 * these pools.
 *
 * As with other major constructs in SBML, Species has a mandatory
 * attribute, \'id\', used to give the species type an identifier in the
 * model.  The identifier must be a text string conforming to the identifer
 * syntax permitted in SBML.  Species also has an optional \'name\'
 * attribute, of type @c string.  The \'id\' and \'name\' must be used
 * according to the guidelines described in the SBML specifications.
 *
 * The required attribute \'compartment\' is used to identify the compartment
 * in which the species is located.  The attribute\'s value must be the
 * identifier of an existing Compartment object.  It is important to note
 * that there is no default value for the \'compartment\' attribute on
 * Species; every species in an SBML model must be assigned a compartment
 * @em explicitly.  (This also implies that every model with one or more
 * Species objects must define at least one Compartment object.)
 *
 * 
 * @section species-amounts The initial amount and concentration of a species
 *
 * The optional attributes \'initialAmount\' and \'initialConcentration\', both
 * having a data type of @c double, can be used to set the @em initial
 * quantity of the species in the compartment where the species is located.
 * These attributes are mutually exclusive; i.e., <em>only one</em> can
 * have a value on any given instance of a Species object.  Missing
 * \'initialAmount\' and \'initialConcentration\' values implies that their
 * values either are unknown, or to be obtained from an external source, or
 * determined by an InitialAssignment or other SBML construct elsewhere in
 * the model.
 *
 * A species\' initial quantity in SBML is set by the \'initialAmount\' or
 * \'initialConcentration\' attribute exactly once.  If the \'constant\'
 * attribute is @c true, then the value of the species\' quantity is fixed
 * and cannot be changed except by an InitialAssignment.  These methods
 * differ in that the \'initialAmount\' and \'initialConcentration\' attributes
 * can only be used to set the species quantity to a literal floating-point
 * number, whereas the use of an InitialAssignment object allows the value
 * to be set using an arbitrary mathematical expression (which, thanks to
 * MathML\'s expressiveness, may evaluate to a rational number).  If the
 * species\' \'constant\' attribute is @c false, the species\' quantity value
 * may be overridden by an InitialAssignment or changed by AssignmentRule
 * or AlgebraicRule, and in addition, for <em>t &gt; 0</em>, it may also be
 * changed by a RateRule, Event objects, and as a result of being a
 * reactant or product in one or more Reaction objects.  (However, some
 * constructs are mutually exclusive; see the SBML specifications for the
 * precise details.)  It is not an error to define \'initialAmount\' or
 * \'initialConcentration\' on a species and also redefine the value using an
 * InitialAssignment, but the \'initialAmount\' or \'initialConcentration\'
 * setting in that case is ignored.  The SBML specifications provide
 * additional information about the semantics of assignments, rules and
 * values for simulation time <em>t</em> <= <em>0</em>.
 * 
 * SBML Level&nbsp;2 additionally stipulates that in cases where a species\'
 * compartment has a \'spatialDimensions\' value of @c 0 (zero), the species
 * cannot have a value for \'initialConcentration\' because the concepts of
 * concentration and density break down when a container has zero
 * dimensions.
 *
 * @section species-units The units of a species\' amount or concentration
 * 
 * When the attribute \'initialAmount\' is set, the unit of measurement
 * associated with the value of \'initialAmount\' is specified by the Species
 * attribute \'substanceUnits\'.  When the \'initialConcentration\' attribute
 * is set, the unit of measurement associated with this concentration value
 * is {<em>unit of amount</em>} divided by {<em>unit of size</em>}, where
 * the {<em>unit of amount</em>} is specified by the Species
 * \'substanceUnits\' attribute, and the {<em>unit of size</em>} is specified
 * by the \'units\' attribute of the Compartment object in which the species
 * is located.  Note that in either case, a unit of <em>amount</em> is
 * involved and determined by the \'substanceUnits\' attribute.  Note
 * <strong>these two attributes alone do not determine the units of the
 * species when the species identifier appears in a mathematical
 * expression</strong>; <em>that</em> aspect is determined by the attribute
 * \'hasOnlySubstanceUnits\' discussed below.
 * 
 * In SBML Level&nbsp;3, if the \'substanceUnits\' attribute is not set on a
 * given Species object instance, then the unit of <em>amount</em> for that
 * species is inherited from the \'substanceUnits\' attribute on the
 * enclosing Model object instance.  If that attribute on Model is not set
 * either, then the unit associated with the species\' quantity is
 * undefined.
 *
 * In SBML Level&nbsp;2, if the \'substanceUnits\' attribute is not set on a
 * given Species object instance, then the unit of <em>amount</em> for that
 * species is taken from the predefined SBML unit identifier
 * @c \'substance\'.  The value assigned to \'substanceUnits\' must be chosen from
 * one of the following possibilities: one of the base unit identifiers
 * defined in SBML, the built-in unit identifier @c \'substance\', or the
 * identifier of a new unit defined in the list of unit definitions in the
 * enclosing Model object.  The chosen units for \'substanceUnits\' must be
 * be @c \'dimensionless\', @c \'mole\', @c \'item\', @c \'kilogram\', @c \'gram\',
 * or units derived from these.
 * 
 * As noted at the beginning of this section, simply setting
 * \'initialAmount\' or \'initialConcentration\' alone does @em not determine
 * whether a species identifier represents an amount or a concentration
 * when it appears elsewhere in an SBML model.  The role of the attribute
 * \'hasOnlySubstanceUnits\' is to indicate whether the units of the species,
 * when the species identifier appears in mathematical formulas, are
 * intended to be concentration or amount.  The attribute takes on a
 * boolean value.  In SBML Level&nbsp;3, the attribute has no default value
 * and must always be set in a model; in SBML Level&nbsp;2, it has a
 * default value of @c false.
 *
 * The <em>units of the species</em> are used in the following ways:
 * <ul>

 * <li> When the species\' identifier appears in a MathML formula, it
 * represents the species\' quantity, and the unit of measurement associated
 * with the quantity is as described above.
 * 
 * <li> The \'math\' elements of AssignmentRule, InitialAssignment and
 * EventAssignment objects referring to this species should all have the
 * same units as the unit of measurement associated with the species
 * quantity.
 * 
 * <li> In a RateRule object that defines the rate of change of the
 * species\' quantity, the unit associated with the rule\'s \'math\' element
 * should be equal to the unit of the species\' quantity divided by the
 * model-wide unit of <em>time</em>; in other words, {<em>unit of species
 * quantity</em>}/{<em>unit of time</em>}.
 * 
 * </ul>
 *
 *
 * @section species-constant The \'constant\' and \'boundaryCondition\' attributes
 *
 * The Species object class has two boolean attributes named \'constant\' and
 * \'boundaryCondition\', used to indicate whether and how the quantity of
 * that species can vary during a simulation.  In SBML Level&nbsp;2 they
 * are optional; in SBML Level&nbsp;3 they are mandatory.  The following
 * table shows how to interpret the combined values of these attributes.
 *
 * @htmlinclude species-boundarycondition.html
 * 
 * By default, when a species is a product or reactant of one or more
 * reactions, its quantity is determined by those reactions.  In SBML, it
 * is possible to indicate that a given species\' quantity is <em>not</em>
 * determined by the set of reactions even when that species occurs as a
 * product or reactant; i.e., the species is on the <em>boundary</em> of
 * the reaction system, and its quantity is not determined by the
 * reactions.  The boolean attribute \'boundaryCondition\' can be used to
 * indicate this.  A value of @c false indicates that the species @em is
 * part of the reaction system.  In SBML Level&nbsp;2, the attribute has a
 * default value of @c false, while in SBML Level&nbsp;3, it has no
 * default.
 *
 * The \'constant\' attribute indicates whether the species\' quantity can be
 * changed at all, regardless of whether by reactions, rules, or constructs
 * other than InitialAssignment.  A value of @c false indicates that the
 * species\' quantity can be changed.  (This is also a common value because
 * the purpose of most simulations is precisely to calculate changes in
 * species quantities.)  In SBML Level&nbsp;2, the attribute has a default
 * value of @c false, while in SBML Level&nbsp;3, it has no default.  Note
 * that the initial quantity of a species can be set by an
 * InitialAssignment irrespective of the value of the \'constant\' attribute.
 *
 * In practice, a \'boundaryCondition\' value of @c true means a differential
 * equation derived from the reaction definitions should not be generated
 * for the species.  However, the species\' quantity may still be changed by
 * AssignmentRule, RateRule, AlgebraicRule, Event, and InitialAssignment
 * constructs if its \'constant\' attribute is @c false.  Conversely, if the
 * species\' \'constant\' attribute is @c true, then its value cannot be
 * changed by anything except InitialAssignment.
 *
 * A species having \'boundaryCondition\'=@c false and \'constant\'=@c false
 * can appear as a product and/or reactant of one or more reactions in the
 * model.  If the species is a reactant or product of a reaction, it must
 * @em not also appear as the target of any AssignmentRule or RateRule
 * object in the model.  If instead the species has \'boundaryCondition\'=
 * @c false and \'constant\'=@c true, then it cannot appear as a reactant or
 * product, or as the target of any AssignmentRule, RateRule or
 * EventAssignment object in the model.
 *
 * Finally, it is worth clarifying that while the constant and 
 * boundaryCondition attributes restrict whether and how the species 
 * amount changes, the same is not true of a species\' concentration. In 
 * SBML, the concentration of a species is a quantity that depends on the 
 * size of the compartment in which it is located. A compartment\'s size 
 * may change, and therefore, so can the concentration of a species even 
 * if the amount of the species remains unchanged. A species\' concentration 
 * may therefore vary even if the Species object\'s constant attribute is 
 * set to @c true in a model.
 *
 * @section species-l2-convfactor The conversionFactor attribute in SBML Level&nbsp;3
 * 
 * In SBML Level&nbsp;3, Species has an additional optional attribute,
 * \'conversionFactor\', that defines a conversion factor that applies to a
 * particular species.  The value must be the identifier of a Parameter
 * object instance defined in the model.  That Parameter object must be a
 * constant, meaning its \'constant\' attribute must be set to @c true.
 * If a given Species object definition defines a value for its
 * \'conversionFactor\' attribute, it takes precedence over any factor
 * defined by the Model object\'s \'conversionFactor\' attribute.
 * 
 * The unit of measurement associated with a species\' quantity can be
 * different from the unit of extent of reactions in the model.  SBML
 * Level&nbsp;3 avoids implicit unit conversions by providing an explicit
 * way to indicate any unit conversion that might be required.  The use of
 * a conversion factor in computing the effects of reactions on a species\'
 * quantity is explained in detail in the SBML Level&nbsp;3 specification
 * document.  Because the value of the \'conversionFactor\' attribute is the
 * identifier of a Parameter object, and because parameters can have units
 * attached to them, the transformation from reaction extent units to
 * species units can be completely specified using this approach.
 * 
 * Note that the unit conversion factor is <strong>only applied when
 * calculating the effect of a reaction on a species</strong>.  It is not
 * used in any rules or other SBML constructs that affect the species, and
 * it is also not used when the value of the species is referenced in a
 * mathematical expression.
 * 
 *
 * @section species-l2-type The speciesType attribute in SBML Level&nbsp;2 Versions&nbsp;2&ndash;4
 *
 * In SBML Level&nbsp;2 Versions&nbsp;2&ndash;4, each species in a model
 * may optionally be designated as belonging to a particular species type.
 * The optional attribute \'speciesType\' is used to identify the species
 * type of the chemical entities that make up the pool represented by the
 * Species objects.  The attribute\'s value must be the identifier of an
 * existing SpeciesType object in the model.  If the \'speciesType\'
 * attribute is not present on a particular species definition, it means
 * the pool contains chemical entities of a type unique to that pool; in
 * effect, a virtual species type is assumed for that species, and no other
 * species can belong to that species type.  The value of \'speciesType\'
 * attributes on species have no effect on the numerical interpretation of
 * a model; simulators and other numerical analysis software may ignore
 * \'speciesType\' attributes.
 * 
 * There can be only one species of a given species type in any given
 * compartment of a model.  More specifically, for all Species objects
 * having a value for the \'speciesType\' attribute, the pair
 * <center>
 * (\'speciesType\' attribute value, \'compartment\' attribute value)
 * </center>
 * 
 * must be unique across the set of all Species object in a model.
 *
 * 
 * @section species-other The spatialSizeUnits attribute in SBML Level&nbsp;2 Versions&nbsp;1&ndash;2
 *
 * In versions of SBML Level&nbsp;2 before Version&nbsp;3, the class
 * Species included an attribute called \'spatialSizeUnits\', which allowed
 * explicitly setting the units of size for initial concentration.  LibSBML
 * retains this attribute for compatibility with older definitions of
 * Level&nbsp;2, but its use is strongly discouraged because many software
 * tools do no properly interpret this unit declaration and it is
 * incompatible with all SBML specifications after Level&nbsp;2
 * Version&nbsp;3.
 *
 * 
 * @section species-math Additional considerations for interpreting the numerical value of a species
 * 
 * Species are unique in SBML in that they have a kind of duality: a
 * species identifier may stand for either substance amount (meaning, a
 * count of the number of individual entities) or a concentration or
 * density (meaning, amount divided by a compartment size).  The previous
 * sections explain the meaning of a species identifier when it is
 * referenced in a mathematical formula or in rules or other SBML
 * constructs; however, it remains to specify what happens to a species
 * when the compartment in which it is located changes in size.
 * 
 * When a species definition has a \'hasOnlySubstanceUnits\' attribute value
 * of @c false and the size of the compartment in which the species is
 * located changes, the default in SBML is to assume that it is the
 * concentration that must be updated to account for the size change.  This
 * follows from the principle that, all other things held constant, if a
 * compartment simply changes in size, the size change does not in itself
 * cause an increase or decrease in the number of entities of any species
 * in that compartment.  In a sense, the default is that the @em amount of
 * a species is preserved across compartment size changes.  Upon such size
 * changes, the value of the concentration or density must be recalculated
 * from the simple relationship <em>concentration = amount / size</em> if
 * the value of the concentration is needed (for example, if the species
 * identifier appears in a mathematical formula or is otherwise referenced
 * in an SBML construct).  There is one exception: if the species\' quantity
 * is determined by an AssignmentRule, RateRule, AlgebraicRule, or an
 * EventAssignment and the species has a \'hasOnlySubstanceUnits\' attribute
 * value of @c false, it means that the <em>concentration</em> is assigned
 * by the rule or event; in that case, the <em>amount</em> must be
 * calculated when the compartment size changes.  (Events also require
 * additional care in this situation, because an event with multiple
 * assignments could conceivably reassign both a species quantity and a
 * compartment size simultaneously.  Please refer to the SBML
 * specifications for the details.)
 * 
 * Note that the above only matters if a species has a
 * \'hasOnlySubstanceUnits\' attribute value of @c false, meaning that the
 * species identifier refers to a concentration wherever the identifier
 * appears in a mathematical formula.  If instead the attribute\'s value is
 * @c true, then the identifier of the species <em>always</em> stands for
 * an amount wherever it appears in a mathematical formula or is referenced
 * by an SBML construct.  In that case, there is never a question about
 * whether an assignment or event is meant to affect the amount or
 * concentration: it is always the amount.
 * 
 * A particularly confusing situation can occur when the species has
 * \'constant\' attribute value of @c true in combination with a
 * \'hasOnlySubstanceUnits\' attribute value of @c false.  Suppose this
 * species is given a value for \'initialConcentration\'.  Does a \'constant\'
 * value of @c true mean that the concentration is held constant if the
 * compartment size changes?  No; it is still the amount that is kept
 * constant across a compartment size change.  The fact that the species
 * was initialized using a concentration value is irrelevant.
 * 
 *
 * 
 */
"


%typemap(csimports) ListOfSpecies "
 using System;
 using System.Runtime.InteropServices;

/** 
 * @sbmlpackage{core}
 *
@htmlinclude pkg-marker-core.html A list of Species objects.
 * 
 *
 * 
 * The various ListOf___ @if conly structures @else classes@endif in SBML
 * are merely containers used for organizing the main components of an SBML
 * model.  In libSBML\'s implementation, ListOf___
 * @if conly data structures @else classes@endif are derived from the
 * intermediate utility @if conly structure @else class@endif ListOf, which
 * is not defined by the SBML specifications but serves as a useful
 * programmatic construct.  ListOf is itself is in turn derived from SBase,
 * which provides all of the various ListOf___
 * @if conly data structures @else classes@endif with common features
 * defined by the SBML specification, such as \'metaid\' attributes and
 * annotations.
 *
 * The relationship between the lists and the rest of an SBML model is
 * illustrated by the following (for SBML Level&nbsp;2 Version&nbsp;4):
 *
 * @htmlinclude listof-illustration.html
 *
 * SBML Level&nbsp;3 Version&nbsp;1 has essentially the same structure as 
 * Level&nbsp;2 Version&nbsp;4, depicted above, but SBML Level&nbsp;3 
 * Version&nbsp;2 allows
 * containers to contain zero or more of the relevant object, instead of 
 * requiring at least one.  As such, libsbml will write out an 
 * otherwise-empty ListOf___ element that has any optional attribute set 
 * (such as \'id\' or \'metaid\'), that has an optional child (such 
 * as a \'notes\' or \'annotation\'), or that has attributes or children set
 * from any SBML Level&nbsp;3 package, whether or not the ListOf___ has 
 * any other children.
 *
 * Readers may wonder about the motivations for using the ListOf___
 * containers in SBML.  A simpler approach in XML might be to place the
 * components all directly at the top level of the model definition.  The
 * choice made in SBML is to group them within XML elements named after
 * %ListOf<em>Classname</em>, in part because it helps organize the
 * components.  More importantly, the fact that the container classes are
 * derived from SBase means that software tools can add information @em about
 * the lists themselves into each list container\'s \'annotation\'.
 *
 * @see ListOfFunctionDefinitions
 * @see ListOfUnitDefinitions
 * @see ListOfCompartmentTypes
 * @see ListOfSpeciesTypes
 * @see ListOfCompartments
 * @see ListOfSpecies
 * @see ListOfParameters
 * @see ListOfInitialAssignments
 * @see ListOfRules
 * @see ListOfConstraints
 * @see ListOfReactions
 * @see ListOfEvents
 *
 * @if conly
 * @note In the C API for libSBML, functions that in other language APIs
 * would be inherited by the various ListOf___ structures not shown in the
 * pages for the individual ListOf___\'s.  Instead, the functions are defined
 * on ListOf_t.  <strong>Please consult the documentation for ListOf_t for
 * the many common functions available for manipulating ListOf___
 * structures</strong>.  The documentation for the individual ListOf___
 * structures (ListOfCompartments_t, ListOfReactions_t, etc.) does not reveal
 * all of the functionality available. @endif
 *
 *
 */
"


%csmethodmodifiers Species::Species(unsigned int level, unsigned int version) "
/**
   * Creates a new Species using the given SBML @p level and @p version
   * values.
   *
   * @param level a long integer, the SBML Level to assign to this Species.
   *
   * @param version a long integer, the SBML Version to assign to this
   * Species.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers Species::Species(SBMLNamespaces* sbmlns) "
/**
   * Creates a new Species using the given SBMLNamespaces object
   * @p sbmlns.
   *
   *
 * 
 * The SBMLNamespaces object encapsulates SBML Level/Version/namespaces
 * information.  It is used to communicate the SBML Level, Version, and (in
 * Level&nbsp;3) packages used in addition to SBML Level&nbsp;3 Core.  A
 * common approach to using libSBML\'s SBMLNamespaces facilities is to create an
 * SBMLNamespaces object somewhere in a program once, then hand that object
 * as needed to object constructors that accept SBMLNamespaces as arguments.
 *
 * 
   *
   * It is worth emphasizing that although this constructor does not take
   * an identifier argument, in SBML Level&nbsp;2 and beyond, the \'id\'
   * (identifier) attribute of a Species is required to have a value.
   * Thus, callers are cautioned to assign a value after calling this
   * constructor.  Setting the identifier can be accomplished using the
   * method Species::setId(@if java String@endif).
   *
   * @param sbmlns an SBMLNamespaces object.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers Species::Species(const Species& orig) "
/**
   * Copy constructor; creates a copy of this Species object.
   *
   * @param orig the object to copy.
   */ public
";


%csmethodmodifiers Species::accept(SBMLVisitor& v) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Species::clone() const "
/**
   * Creates and returns a deep copy of this Species object.
   *
   * @return the (deep) copy of this Species object.
   */ public new
";


%csmethodmodifiers Species::initDefaults "
/**
   * Initializes the fields of this Species object to \'typical\' defaults
   * values.
   *
   * The SBML Species component has slightly different aspects and
   * default attribute values in different SBML Levels and Versions.
   * This method sets the values to certain common defaults, based
   * mostly on what they are in SBML Level&nbsp;2.  Specifically:
   * <ul>
   * <li> Sets \'boundaryCondition\' to @c false
   * <li> Sets \'constant\' to @c false
   * <li> sets \'hasOnlySubstanceUnits\' to @c false
   * <li> (Applies to Level&nbsp;3 models only) Sets attribute \'substanceUnits\' to @c mole
   * </ul>
   */ public
";


%csmethodmodifiers Species::getId() const "
/**
   * Returns the value of the \'id\' attribute of this Species.
   *
   * @note Because of the inconsistent behavior of this function with 
   * respect to assignments and rules, it is now recommended to
   * use the getIdAttribute() function instead.
   *
   *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
   *
   * @return the id of this Species.
   *
   * @see getIdAttribute()
   * @see setIdAttribute(string sid)
   * @see isSetIdAttribute()
   * @see unsetIdAttribute()
   */ public new
";


%csmethodmodifiers Species::getName() const "
/**
   * Returns the value of the \'name\' attribute of this Species object.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 *
 * @return the name of this SBML object, or the empty string if not set or unsettable.
 *
 * @see getIdAttribute()
 * @see isSetName()
 * @see setName(string sid)
 * @see unsetName()
 * 
 *
   */ public new
";


%csmethodmodifiers Species::getSpeciesType() const "
/**
   * Get the type of this Species object object.
   * 
   * @return the value of the \'speciesType\' attribute of this
   * Species as a string.
   * 
   * @note The \'speciesType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   */ public
";


%csmethodmodifiers Species::getCompartment() const "
/**
   * Get the compartment in which this species is located.
   *
   * The compartment is designated by its identifier.
   * 
   * @return the value of the \'compartment\' attribute of this Species
   * object, as a string.
   */ public
";


%csmethodmodifiers Species::getInitialAmount() const "
/**
   * Get the value of the \'initialAmount\' attribute.
   * 
   * @return the initialAmount of this Species, as a floating point number.
   */ public
";


%csmethodmodifiers Species::getInitialConcentration() const "
/**
   * Get the value of the \'initialConcentration\' attribute.
   * 
   * @return the initialConcentration of this Species,, as a floating point
   * number.
   *
   * @note The attribute \'initialConcentration\' is only available in SBML
   * Level&nbsp;2 and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
   */ public
";


%csmethodmodifiers Species::getSubstanceUnits() const "
/**
   * Get the value of the \'substanceUnits\' attribute.
   * 
   * @return the value of the \'substanceUnits\' attribute of this Species,
   * as a string.  An empty string indicates that no units have been
   * assigned.
   *
   *
 * @note There is an important distinction to be made between <em>no units
 * assigned</em>, and assuming a value without units has any specific unit
 * such as <code>dimensionless</code>.  In SBML, default units are never
 * attributed to numbers, and numbers without units are not automatically
 * assumed to have the unit <code>dimensionless</code>.  Please consult the
 * relevant SBML specification document for a more in-depth explanation of
 * this topic and the SBML unit system.
 *
 *
   *
   * @see isSetSubstanceUnits()
   * @see setSubstanceUnits(string sid)
   */ public
";


%csmethodmodifiers Species::getSpatialSizeUnits() const "
/**
   * Get the value of the \'spatialSizeUnits\' attribute.
   * 
   * @return the value of the \'spatialSizeUnits\' attribute of this Species
   * object, as a string.
   *
   *
 * @warning <span class=\'warning\'>In versions of SBML Level&nbsp;2 before
 * Version&nbsp;3, the class Species included an attribute called
 * \'spatialSizeUnits\', which allowed explicitly setting the units of size for
 * initial concentration.  This attribute was removed in SBML Level&nbsp;2
 * Version&nbsp;3.  LibSBML retains this attribute for compatibility with
 * older definitions of Level&nbsp;2, but its use is strongly discouraged
 * because it is incompatible with levels and versions of SBML beyond
 * Level&nbsp;2 Version&nbsp;2.</span>
 *
   */ public
";


%csmethodmodifiers Species::getUnits() const "
/**
   * Get the value of the \'units\' attribute.
   * 
   * @return the units of this Species (L1 only).
   *
   *
 * @note The \'units\' attribute is defined only in SBML Level&nbsp;1.  In
 * SBML Level&nbsp;2 and Level&nbsp;3, it has been replaced by a
 * combination of \'substanceUnits\' and the units of the Compartment
 * object in which a species is located.  In SBML Level&nbsp;2
 * Versions&nbsp;1&ndash;2, an additional attribute \'spatialSizeUnits\'
 * helps determine the units of the species quantity, but this attribute
 * was removed in later versions of SBML Level&nbsp;2.
 * 
   */ public
";


%csmethodmodifiers Species::getHasOnlySubstanceUnits() const "
/**
   * Get the value of the \'hasOnlySubstanceUnits\' attribute.
   * 
   * @return @c true if this Species\' \'hasOnlySubstanceUnits\' attribute
   * value is @c true, @c false otherwise.
   *
   * @note The \'hasOnlySubstanceUnits\' attribute does not exist in SBML
   * Level&nbsp;1.
   */ public
";


%csmethodmodifiers Species::getBoundaryCondition() const "
/**
   * Get the value of the \'boundaryCondition\' attribute.
   * 
   * @return @c true if this Species\' \'boundaryCondition\' attribute value
   * is @c true, @c false otherwise.
   */ public
";


%csmethodmodifiers Species::getCharge() const "
/**
   * Get the value of the \'charge\' attribute.
   * 
   * @return the charge of this Species object.
   *
   *
 * @note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
 * attribute on Species is deprecated and in SBML Level&nbsp;3 it does
 * not exist at all.  Its use strongly discouraged.  Its presence is
 * considered a misfeature in earlier definitions of SBML because its
 * implications for the mathematics of a model were never defined, and in
 * any case, no known modeling system ever used it.  Instead, models take
 * account of charge values directly in their definitions of species by
 * (for example) having separate species identities for the charged and
 * uncharged versions of the same species.  This allows the condition to
 * affect model mathematics directly.  LibSBML retains this method for
 * easier compatibility with SBML Level&nbsp;1.
   */ public
";


%csmethodmodifiers Species::getConstant() const "
/**
   * Get the value of the \'constant\' attribute.
   * 
   * @return @c true if this Species\'s \'constant\' attribute value is
   * @c true, @c false otherwise.
   *
   * @note The attribute \'constant\' is only available in SBML Levels&nbsp;2
   * and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
   */ public
";


%csmethodmodifiers Species::getConversionFactor() const "
/**
   * Get the value of the \'conversionFactor\' attribute.
   * 
   * @return the conversionFactor of this Species, as a string.
   * 
   * @note The \'conversionFactor\' attribute was introduced in SBML
   * Level&nbsp;3.  It does not exist on Species in SBML Levels&nbsp;1
   * and&nbsp;2.
   */ public
";


%csmethodmodifiers Species::isSetId() const "
/**
   * Predicate returning @c true if this
   * Species object\'s \'id\' attribute is set.
   *
   *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 *
 * @return @c true if the \'id\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @note Because of the inconsistent behavior of this function with
 * respect to assignments and rules, it is recommended that callers
 * use isSetIdAttribute() instead.
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see unsetIdAttribute()
 * @see isSetIdAttribute()
 *
 *
   */ public new
";


%csmethodmodifiers Species::isSetName() const "
/**
   * Predicate returning @c true if this
   * Species object\'s \'name\' attribute is set.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 * @return @c true if the \'name\' attribute of this SBML object is
 * set, @c false otherwise.
 *
 * @see getName()
 * @see setName(string sid)
 * @see unsetName()
 *
 *
   */ public new
";


%csmethodmodifiers Species::isSetSpeciesType() const "
/**
   * Predicate returning @c true if this Species object\'s
   * \'speciesType\' attribute is set.
   *
   * @return @c true if the \'speciesType\' attribute of this Species is
   * set, @c false otherwise.
   * 
   * @note The \'speciesType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   */ public
";


%csmethodmodifiers Species::isSetCompartment() const "
/**
   * Predicate returning @c true if this
   * Species object\'s \'compartment\' attribute is set.
   *
   * @return @c true if the \'compartment\' attribute of this Species is
   * set, @c false otherwise.
   */ public
";


%csmethodmodifiers Species::isSetInitialAmount() const "
/**
   * Predicate returning @c true if this
   * Species object\'s \'initialAmount\' attribute is set.
   *
   * @return @c true if the \'initialAmount\' attribute of this Species is
   * set, @c false otherwise.
   *
   * @note In SBML Level&nbsp;1, Species\' \'initialAmount\' is required and
   * therefore <em>should always be set</em>.  (However, in Level&nbsp;1, the
   * attribute has no default value either, so this method will not return
   * @c true until a value has been assigned.)  In SBML Level&nbsp;2,
   * \'initialAmount\' is optional and as such may or may not be set.
   */ public
";


%csmethodmodifiers Species::isSetInitialConcentration() const "
/**
   * Predicate returning @c true if this
   * Species object\'s \'initialConcentration\' attribute is set.
   *
   * @return @c true if the \'initialConcentration\' attribute of this Species is
   * set, @c false otherwise.
   *
   * @note The attribute \'initialConcentration\' is only available in SBML
   * Level&nbsp;2 and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
   */ public
";


%csmethodmodifiers Species::isSetSubstanceUnits() const "
/**
   * Predicate returning @c true if this
   * Species object\'s \'substanceUnits\' attribute is set.
   *
   * @return @c true if the \'substanceUnits\' attribute of this Species is
   * set, @c false otherwise.
   */ public
";


%csmethodmodifiers Species::isSetSpatialSizeUnits() const "
/**
   * Predicate returning @c true if this
   * Species object\'s \'spatialSizeUnits\' attribute is set.
   *
   * @return @c true if the \'spatialSizeUnits\' attribute of this Species is
   * set, @c false otherwise.
   *
   *
 * @warning <span class=\'warning\'>In versions of SBML Level&nbsp;2 before
 * Version&nbsp;3, the class Species included an attribute called
 * \'spatialSizeUnits\', which allowed explicitly setting the units of size for
 * initial concentration.  This attribute was removed in SBML Level&nbsp;2
 * Version&nbsp;3.  LibSBML retains this attribute for compatibility with
 * older definitions of Level&nbsp;2, but its use is strongly discouraged
 * because it is incompatible with levels and versions of SBML beyond
 * Level&nbsp;2 Version&nbsp;2.</span>
 *
   */ public
";


%csmethodmodifiers Species::isSetUnits() const "
/**
   * Predicate returning @c true if
   * this Species object\'s \'units\' attribute is set.
   *
   * @return @c true if the \'units\' attribute of this Species is
   * set, @c false otherwise.
   */ public
";


%csmethodmodifiers Species::isSetCharge() const "
/**
   * Predicate returning @c true if this
   * Species object\'s \'charge\' attribute is set.
   *
   * @return @c true if the \'charge\' attribute of this Species is
   * set, @c false otherwise.
   *
   *
 * @note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
 * attribute on Species is deprecated and in SBML Level&nbsp;3 it does
 * not exist at all.  Its use strongly discouraged.  Its presence is
 * considered a misfeature in earlier definitions of SBML because its
 * implications for the mathematics of a model were never defined, and in
 * any case, no known modeling system ever used it.  Instead, models take
 * account of charge values directly in their definitions of species by
 * (for example) having separate species identities for the charged and
 * uncharged versions of the same species.  This allows the condition to
 * affect model mathematics directly.  LibSBML retains this method for
 * easier compatibility with SBML Level&nbsp;1. 
   */ public
";


%csmethodmodifiers Species::isSetConversionFactor() const "
/**
   * Predicate returning @c true if this
   * Species object\'s \'conversionFactor\' attribute is set.
   *
   * @return @c true if the \'conversionFactor\' attribute of this Species is
   * set, @c false otherwise.
   * 
   * @note The \'conversionFactor\' attribute was introduced in SBML
   * Level&nbsp;3.  It does not exist on Species in SBML Levels&nbsp;1
   * and&nbsp;2.
   */ public
";


%csmethodmodifiers Species::isSetBoundaryCondition() const "
/**
   * Predicate returning @c true if this
   * Species object\'s \'boundaryCondition\' attribute is set.
   *
   * @return @c true if the \'boundaryCondition\' attribute of this Species is
   * set, @c false otherwise.
   */ public
";


%csmethodmodifiers Species::isSetHasOnlySubstanceUnits() const "
/**
   * Predicate returning @c true if this
   * Species object\'s \'hasOnlySubstanceUnits\' attribute is set.
   *
   * @return @c true if the \'hasOnlySubstanceUnits\' attribute of this Species is
   * set, @c false otherwise.
   *
   * @note The \'hasOnlySubstanceUnits\' attribute does not exist in SBML
   * Level&nbsp;1.
   */ public
";


%csmethodmodifiers Species::isSetConstant() const "
/**
   * Predicate returning @c true if this
   * Species object\'s \'constant\' attribute is set.
   *
   * @return @c true if the \'constant\' attribute of this Species is
   * set, @c false otherwise.
   *
   * @note The attribute \'constant\' is only available in SBML Levels&nbsp;2
   * and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
   */ public
";


%csmethodmodifiers Species::setId(const std::string& sid) "
/**
   * Sets the value of the \'id\' attribute of this Species.
   *
   *
 * 
 * The string @p sid is copied.
 *
 *
 * 
 * The identifier given by an object\'s \'id\' attribute value
 * is used to identify the object within the SBML model definition.
 * Other objects can refer to the component using this identifier.  The
 * data type of \'id\' is always <code>SId</code> or a type derived
 * from that, such as <code>UnitSId</code>, depending on the object in 
 * question.  All data types are defined as follows:
 * <pre style=\'margin-left: 2em; border: none; font-weight: bold; color: black\'>
 *   letter ::= \'a\'..\'z\',\'A\'..\'Z\'
 *   digit  ::= \'0\'..\'9\'
 *   idChar ::= letter | digit | \'_\'
 *   SId    ::= ( letter | \'_\' ) idChar*
 * </pre>
 * The characters <code>(</code> and <code>)</code> are used for grouping,
 * the character <code>*</code> \'zero or more times\', and the character
 * <code>|</code> indicates logical \'or\'.  The equality of SBML identifiers
 * is determined by an exact character sequence match; i.e., comparisons must
 * be performed in a case-sensitive manner.  This applies to all uses of
 * <code>SId</code>, <code>SIdRef</code>, and derived types.
 *
 * Users need to be aware of some important API issues that are the result of
 * the history of SBML and libSBML.  Prior to SBML Level&nbsp;3
 * Version&nbsp;2, SBML defined \'id\' and \'name\' attributes on only a subset
 * of SBML objects.  To simplify the work of programmers, libSBML\'s API
 * provided get, set, check, and unset on the SBase object class itself
 * instead of on individual subobject classes. This made the
 * get/set/etc. methods uniformly available on all objects in the libSBML
 * API.  LibSBML simply returned empty strings or otherwise did not act when
 * the methods were applied to SBML objects that were not defined by the SBML
 * specification to have \'id\' or \'name\' attributes.  Additional complications
 * arose with the rule and assignment objects: InitialAssignment,
 * EventAssignment, AssignmentRule, and RateRule.  In early versions of SBML,
 * the rule object hierarchy was different, and in addition, then as now,
 * they possess different attributes: \'variable\' (for the rules and event
 * assignments), \'symbol\' (for initial assignments), or neither (for
 * algebraic rules).  Prior to SBML Level&nbsp;3 Version&nbsp;2, getId()
 * would always return an empty string, and isSetId() would always return @c
 * false for objects of these classes.
 *
 * With the addition of \'id\' and \'name\' attributes on SBase in Level&nbsp;3
 * Version&nbsp;2, it became necessary to introduce a new way to interact
 * with the attributes more consistently in libSBML to avoid breaking
 * backward compatibility in the behavior of the original \'id\' methods.  For
 * this reason, libSBML provides four functions (getIdAttribute(),
 * setIdAttribute(@if java String@endif), isSetIdAttribute(), and
 * unsetIdAttribute()) that always act on the actual \'id\' attribute inherited
 * from SBase, regardless of the object\'s type.  <strong>These new methods
 * should be used instead of the older getId()/setId()/etc. methods</strong>
 * unless the old behavior is somehow necessary.  Regardless of the Level and
 * Version of the SBML, these functions allow client applications to use more
 * generalized code in some situations (for instance, when manipulating
 * objects that are all known to have identifiers).  If the object in
 * question does not posess an \'id\' attribute according to the SBML
 * specification for the Level and Version in use, libSBML will not allow the
 * identifier to be set, nor will it read or write \'id\' attributes for those
 * objects.
 *
 *
 * 
 * @param sid the string to use as the identifier of this object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
 *
 * @see getIdAttribute()
 * @see setIdAttribute(string sid)
 * @see isSetIdAttribute()
 * @see unsetIdAttribute()
 * 
 *
   */ public new
";


%csmethodmodifiers Species::setName(const std::string& name) "
/**
   * Sets the value of the \'name\' attribute of this Species.
   *
   *
 * 
 *
 * The string in @p name is copied.
 *
 * @param name the new name for the SBML object.
 *
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
 *
 *
   */ public new
";


%csmethodmodifiers Species::setSpeciesType(const std::string& sid) "
/**
   * Sets the \'speciesType\' attribute of this Species object.
   *
   * @param sid the identifier of a SpeciesType object defined elsewhere
   * in this Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * 
   * @note The \'speciesType\' attribute is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   */ public
";


%csmethodmodifiers Species::setCompartment(const std::string& sid) "
/**
   * Sets the \'compartment\' attribute of this Species object.
   *
   * @param sid the identifier of a Compartment object defined elsewhere
   * in this Model.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   */ public
";


%csmethodmodifiers Species::setInitialAmount(double value) "
/**
   * Sets the \'initialAmount\' attribute of this Species and marks the field
   * as set.
   *
   * This method also unsets the \'initialConcentration\' attribute.
   *
   * @param value the value to which the \'initialAmount\' attribute should
   * be set.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   */ public
";


%csmethodmodifiers Species::setInitialConcentration(double value) "
/**
   * Sets the \'initialConcentration\' attribute of this Species and marks
   * the field as set.
   *
   * This method also unsets the \'initialAmount\' attribute.
   *
   * @param value the value to which the \'initialConcentration\' attribute
   * should be set.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   * @note The attribute \'initialConcentration\' is only available in SBML
   * Level&nbsp;2 and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
   */ public
";


%csmethodmodifiers Species::setSubstanceUnits(const std::string& sid) "
/**
   * Sets the \'substanceUnits\' attribute of this Species object.
   *
   * @param sid the identifier of the unit to use.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   */ public
";


%csmethodmodifiers Species::setSpatialSizeUnits(const std::string& sid) "
/**
   * (SBML Level&nbsp;2 Versions&nbsp;1&ndash;2) Sets the \'spatialSizeUnits\' attribute of this Species object.
   *
   * @param sid the identifier of the unit to use.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   *
 * @warning <span class=\'warning\'>In versions of SBML Level&nbsp;2 before
 * Version&nbsp;3, the class Species included an attribute called
 * \'spatialSizeUnits\', which allowed explicitly setting the units of size for
 * initial concentration.  This attribute was removed in SBML Level&nbsp;2
 * Version&nbsp;3.  LibSBML retains this attribute for compatibility with
 * older definitions of Level&nbsp;2, but its use is strongly discouraged
 * because it is incompatible with levels and versions of SBML beyond
 * Level&nbsp;2 Version&nbsp;2.</span>
 *
   */ public
";


%csmethodmodifiers Species::setUnits(const std::string& sname) "
/**
   * (SBML Level&nbsp;1 only) Sets the units of this Species object.
   *
   * @param sname the identifier of the unit to use.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
  */ public
";


%csmethodmodifiers Species::setHasOnlySubstanceUnits(bool value) "
/**
   * Sets the \'hasOnlySubstanceUnits\' attribute of this Species object.
   *
   * @param value boolean value for the \'hasOnlySubstanceUnits\' attribute.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   * @note The \'hasOnlySubstanceUnits\' attribute does not exist in SBML
   * Level&nbsp;1.
   */ public
";


%csmethodmodifiers Species::setBoundaryCondition(bool value) "
/**
   * Sets the \'boundaryCondition\' attribute of this Species object.
   *
   * @param value boolean value for the \'boundaryCondition\' attribute.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   */ public
";


%csmethodmodifiers Species::setCharge(int value) "
/**
   * Sets the \'charge\' attribute of this Species object.
   *
   * @param value an integer to which to set the \'charge\' to.
   *
   *
 * @note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
 * attribute on Species is deprecated and in SBML Level&nbsp;3 it does
 * not exist at all.  Its use strongly discouraged.  Its presence is
 * considered a misfeature in earlier definitions of SBML because its
 * implications for the mathematics of a model were never defined, and in
 * any case, no known modeling system ever used it.  Instead, models take
 * account of charge values directly in their definitions of species by
 * (for example) having separate species identities for the charged and
 * uncharged versions of the same species.  This allows the condition to
 * affect model mathematics directly.  LibSBML retains this method for
 * easier compatibility with SBML Level&nbsp;1. 
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   */ public
";


%csmethodmodifiers Species::setConstant(bool value) "
/**
   * Sets the \'constant\' attribute of this Species object.
   *
   * @param value a boolean value for the \'constant\' attribute.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   *
   * @note The attribute \'constant\' is only available in SBML Levels&nbsp;2
   * and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
   */ public
";


%csmethodmodifiers Species::setConversionFactor(const std::string& sid) "
/**
   * Sets the value of the \'conversionFactor\' attribute of this Species object.
   *
   * The string in @p sid is copied.
   *
   * @param sid the new conversionFactor for the Species.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_INVALID_ATTRIBUTE_VALUE LIBSBML_INVALID_ATTRIBUTE_VALUE@endlink
   * 
   * @note The \'conversionFactor\' attribute was introduced in SBML
   * Level&nbsp;3.  It does not exist on Species in SBML Levels&nbsp;1
   * and&nbsp;2.
   */ public
";


%csmethodmodifiers Species::unsetName "
/**
   * Unsets the value of the \'name\' attribute of this Species object.
   *
   *
 *
 * 
 * In SBML Level&nbsp;3 Version&nbsp;2, the \'id\' and \'name\' attributes were
 * moved to SBase directly, instead of being defined individually for many
 * (but not all) objects.  LibSBML has for a long time provided functions
 * defined on SBase itself to get, set, and unset those attributes, which 
 * would fail or otherwise return empty strings if executed on any object 
 * for which those attributes were not defined.  Now that all SBase objects 
 * define those attributes, those functions now succeed for any object with 
 * the appropriate level and version.
 *
 * The \'name\' attribute is
 * optional and is not intended to be used for cross-referencing purposes
 * within a model.  Its purpose instead is to provide a human-readable
 * label for the component.  The data type of \'name\' is the type
 * <code>string</code> defined in XML Schema.  SBML imposes no
 * restrictions as to the content of \'name\' attributes beyond those
 * restrictions defined by the <code>string</code> type in XML Schema.
 *
 * The recommended practice for handling \'name\' is as follows.  If a
 * software tool has the capability for displaying the content of \'name\'
 * attributes, it should display this content to the user as a
 * component\'s label instead of the component\'s \'id\'.  If the user
 * interface does not have this capability (e.g., because it cannot
 * display or use special characters in symbol names), or if the \'name\'
 * attribute is missing on a given component, then the user interface
 * should display the value of the \'id\' attribute instead.  (Script
 * language interpreters are especially likely to display \'id\' instead of
 * \'name\'.)
 * 
 * As a consequence of the above, authors of systems that automatically
 * generate the values of \'id\' attributes should be aware some systems
 * may display the \'id\'\'s to the user.  Authors therefore may wish to
 * take some care to have their software create \'id\' values that are: (a)
 * reasonably easy for humans to type and read; and (b) likely to be
 * meaningful, for example by making the \'id\' attribute be an abbreviated
 * form of the name attribute value.
 * 
 * An additional point worth mentioning is although there are
 * restrictions on the uniqueness of \'id\' values, there are no
 * restrictions on the uniqueness of \'name\' values in a model.  This
 * allows software applications leeway in assigning component identifiers.
 *
 * Regardless of the level and version of the SBML, these functions allow
 * client applications to use more generalized code in some situations 
 * (for instance, when manipulating objects that are all known to have 
 * names).  If the object in question does not posess a \'name\' attribute 
 * according to the SBML specification for the Level and Version in use,
 * libSBML will not allow the name to be set, nor will it read or 
 * write \'name\' attributes for those objects.
 *
 *
 * 
 *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
 * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
 *
 * @see getName()
 * @see setName(string sid)
 * @see isSetName()
 *
 *
   */ public new
";


%csmethodmodifiers Species::unsetConstant "
/**
   * Unsets the value of the \'constant\' attribute of this Species object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @see isSetConstant()
   * @see setConstant(@if java boolean@endif)
   * @see getConstant()
   */ public
";


%csmethodmodifiers Species::unsetSpeciesType "
/**
   * Unsets the \'speciesType\' attribute value of this Species object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @note The attribute \'speciesType\' is only available in SBML
   * Level&nbsp;2 Versions&nbsp;2&ndash;4.
   */ public
";


%csmethodmodifiers Species::unsetInitialAmount "
/**
   * Unsets the \'initialAmount\' attribute value of this Species object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   */ public
";


%csmethodmodifiers Species::unsetInitialConcentration "
/**
   * Unsets the \'initialConcentration\' attribute value of this Species object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   * @note The attribute \'initialConcentration\' is only available in SBML
   * Level&nbsp;2 and&nbsp;3.  It does not exist on Species in Level&nbsp;1.
   */ public
";


%csmethodmodifiers Species::unsetSubstanceUnits "
/**
   * Unsets the \'substanceUnits\' attribute value of this Species object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   */ public
";


%csmethodmodifiers Species::unsetSpatialSizeUnits "
/**
   * Unsets the \'spatialSizeUnits\' attribute value of this Species object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @warning <span class=\'warning\'>In versions of SBML Level&nbsp;2 before
 * Version&nbsp;3, the class Species included an attribute called
 * \'spatialSizeUnits\', which allowed explicitly setting the units of size for
 * initial concentration.  This attribute was removed in SBML Level&nbsp;2
 * Version&nbsp;3.  LibSBML retains this attribute for compatibility with
 * older definitions of Level&nbsp;2, but its use is strongly discouraged
 * because it is incompatible with levels and versions of SBML beyond
 * Level&nbsp;2 Version&nbsp;2.</span>
 *
   */ public
";


%csmethodmodifiers Species::unsetUnits "
/**
   * Unsets the \'units\' attribute value of this Species object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   */ public
";


%csmethodmodifiers Species::unsetCharge "
/**
   * Unsets the \'charge\' attribute
   * value of this Species object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   *
   *
 * @note Beginning in SBML Level&nbsp;2 Version&nbsp;2, the \'charge\'
 * attribute on Species is deprecated and in SBML Level&nbsp;3 it does
 * not exist at all.  Its use strongly discouraged.  Its presence is
 * considered a misfeature in earlier definitions of SBML because its
 * implications for the mathematics of a model were never defined, and in
 * any case, no known modeling system ever used it.  Instead, models take
 * account of charge values directly in their definitions of species by
 * (for example) having separate species identities for the charged and
 * uncharged versions of the same species.  This allows the condition to
 * affect model mathematics directly.  LibSBML retains this method for
 * easier compatibility with SBML Level&nbsp;1. 
   */ public
";


%csmethodmodifiers Species::unsetConversionFactor "
/**
   * Unsets the \'conversionFactor\' attribute value of this Species object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_UNEXPECTED_ATTRIBUTE LIBSBML_UNEXPECTED_ATTRIBUTE@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   * 
   * @note The \'conversionFactor\' attribute was introduced in SBML
   * Level&nbsp;3.  It does not exist on Species in SBML Levels&nbsp;1
   * and&nbsp;2.
   */ public
";


%csmethodmodifiers Species::unsetCompartment "
/**
   * Unsets the \'compartment\' attribute value of this Species object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   */ public
";


%csmethodmodifiers Species::unsetBoundaryCondition "
/**
   * Unsets the \'boundaryCondition\' attribute value of this Species object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   */ public
";


%csmethodmodifiers Species::unsetHasOnlySubstanceUnits "
/**
   * Unsets the \'hasOnlySubstanceUnits\' attribute value of this Species object.
   *
   *
 * @return integer value indicating success/failure of the
 * function.  @if clike The value is drawn from the
 * enumeration #OperationReturnValues_t. @endif The possible values
 * returned by this function are:
 * @li @link libsbml#LIBSBML_OPERATION_SUCCESS LIBSBML_OPERATION_SUCCESS@endlink
   * @li @link libsbml#LIBSBML_OPERATION_FAILED LIBSBML_OPERATION_FAILED@endlink
   */ public
";


%csmethodmodifiers Species::getDerivedUnitDefinition "
/**
   * Constructs and returns a UnitDefinition that corresponds to the units
   * of this Species\' amount or concentration.
   *
   * Species in SBML have an attribute (\'substanceUnits\') for declaring the
   * units of measurement intended for the species\' amount or concentration
   * (depending on which one applies).  In the absence of a value given for
   * \'substanceUnits\', the units are taken from the enclosing Model\'s
   * definition of @c \'substance\' or @c \'substance\'/<em>(size of the
   * compartment)</em> in which the species is located, or finally, if
   * these are not redefined by the Model, the relevant SBML default units
   * for those quantities.  Following that procedure, the method
   * @if java Species::getDerivedUnitDefinition()@else getDerivedUnitDefinition()@endif
   * returns a UnitDefinition based on the
   * interpreted units of this species\'s amount or concentration.
   *
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return @c null.
   *
   * Note also that unit declarations for Species are in terms of the @em
   * identifier of a unit, but this method returns a UnitDefinition object,
   * not a unit identifier.  It does this by constructing an appropriate
   * UnitDefinition.  Callers may find this particularly useful when used
   * in conjunction with the helper methods on UnitDefinition for comparing
   * different UnitDefinition objects.
   * 
   * In SBML Level&nbsp;2 specifications prior to Version&nbsp;3, Species
   * includes an additional attribute named \'spatialSizeUnits\', which
   * allows explicitly setting the units of size for initial concentration.
   * The @if java Species::getDerivedUnitDefinition()@else getDerivedUnitDefinition()@endif
   * takes this into account for models
   * expressed in SBML Level&nbsp;2 Versions&nbsp;1 and&nbsp;2.
   *
   * @return a UnitDefinition that expresses the units of this 
   * Species, or @c null if one cannot be constructed.
   *
   * @see getSubstanceUnits()
   */ public
";


%csmethodmodifiers Species::getDerivedUnitDefinition() const "
/**
   * Constructs and returns a UnitDefinition that corresponds to the units
   * of this Species\' amount or concentration.
   *
   * Species in SBML have an attribute (\'substanceUnits\') for declaring the
   * units of measurement intended for the species\' amount or concentration
   * (depending on which one applies).  In the absence of a value given for
   * \'substanceUnits\', the units are taken from the enclosing Model\'s
   * definition of @c \'substance\' or @c \'substance\'/<em>(size of the
   * compartment)</em> in which the species is located, or finally, if
   * these are not redefined by the Model, the relevant SBML default units
   * for those quantities.  Following that procedure, the method
   * @if java Species::getDerivedUnitDefinition()@else getDerivedUnitDefinition()@endif
   * returns a UnitDefinition based on the
   * interpreted units of this species\'s amount or concentration.
   *
   * Note that the functionality that facilitates unit analysis depends 
   * on the model as a whole.  Thus, in cases where the object has not 
   * been added to a model or the model itself is incomplete,
   * unit analysis is not possible and this method will return @c null.
   *
   * Note also that unit declarations for Species are in terms of the @em
   * identifier of a unit, but this method returns a UnitDefinition object,
   * not a unit identifier.  It does this by constructing an appropriate
   * UnitDefinition.  Callers may find this particularly useful when used
   * in conjunction with the helper methods on UnitDefinition for comparing
   * different UnitDefinition objects.
   * 
   * In SBML Level&nbsp;2 specifications prior to Version&nbsp;3, Species
   * includes an additional attribute named \'spatialSizeUnits\', which
   * allows explicitly setting the units of size for initial concentration.
   * The @if java Species::getDerivedUnitDefinition()@else getDerivedUnitDefinition()@endif
   * takes this into account for models
   * expressed in SBML Level&nbsp;2 Versions&nbsp;1 and&nbsp;2.
   *
   * @return a UnitDefinition that expresses the units of this 
   * Species, or @c null if one cannot be constructed.
   *
   * @see getSubstanceUnits()
   */ public
";


%csmethodmodifiers Species::getTypeCode() const "
/**
   * Returns the libSBML type code for this SBML object.
   * 
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for this object:
   * @link libsbml#SBML_SPECIES SBML_SPECIES@endlink (default).
   *
   *
 * @warning <span class=\'warning\'>The specific integer values of the possible
 * type codes may be reused by different libSBML plug-ins for SBML Level&nbsp;3.
 * packages,  To fully identify the correct code, <strong>it is necessary to
 * invoke both getPackageName() and getTypeCode()</strong> (or 
 * ListOf::getItemTypeCode()).</span>
 *
 *
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers Species::getElementName() const "
/**
   * Returns the XML element name of this object, which for Species, is
   * always @c \'species\'.
   * 
   * @return the name of this element, i.e., @c \'species\'.
   */ public new
";


%csmethodmodifiers Species::writeElements(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Species::hasRequiredAttributes() const "
/**
   * Predicate returning @c true if
   * all the required attributes for this Species object
   * have been set.
   *
   * The required attributes for a Species object are:
   * @li \'id\' (or \'name\' in SBML Level&nbsp;1)
   * @li \'compartment\'
   * @li \'initialAmount\' (required in SBML Level&nbsp;1 only; optional otherwise)
   * @li \'hasOnlySubstanceUnits\' (required in SBML Level&nbsp;3; optional in SBML Level&nbsp;2)
   * @li \'boundaryCondition\' (required in SBML Level&nbsp;3; optional in Levels&nbsp;1 and&nbsp;2)
   * @li \'constant\' (required in SBML Level&nbsp;3; optional in SBML Level&nbsp;2)
   *
   * @return @c true if the required attributes have been set, @c false
   * otherwise.
   */ public new
";


%csmethodmodifiers Species::renameSIdRefs(const std::string& oldid, const std::string& newid) "
/**
   *
 * Replaces all uses of a given @c SIdRef type attribute value with another
 * value.
 *
 *
 * 

 * In SBML, object identifiers are of a data type called <code>SId</code>.
 * In SBML Level&nbsp;3, an explicit data type called <code>SIdRef</code> was
 * introduced for attribute values that refer to <code>SId</code> values; in
 * previous Levels of SBML, this data type did not exist and attributes were
 * simply described to as \'referring to an identifier\', but the effective
 * data type was the same as <code>SIdRef</code> in Level&nbsp;3.  These and
 * other methods of libSBML refer to the type <code>SIdRef</code> for all
 * Levels of SBML, even if the corresponding SBML specification did not
 * explicitly name the data type.
 *
 *
 *
 * This method works by looking at all attributes and (if appropriate)
 * mathematical formulas in MathML content, comparing the referenced
 * identifiers to the value of @p oldid.  If any matches are found, the
 * matching values are replaced with @p newid.  The method does @em not
 * descend into child elements.
 *
 * @param oldid the old identifier.
 * @param newid the new identifier.
 *
 *
   */ public new
";


%csmethodmodifiers Species::renameUnitSIdRefs(const std::string& oldid, const std::string& newid) "
/**
   *
 * Replaces all uses of a given @c UnitSIdRef type attribute value with
 * another value.
 *
 *
 * 
 * In SBML, unit definitions have identifiers of type <code>UnitSId</code>.  In
 * SBML Level&nbsp;3, an explicit data type called <code>UnitSIdRef</code> was
 * introduced for attribute values that refer to <code>UnitSId</code> values; in
 * previous Levels of SBML, this data type did not exist and attributes were
 * simply described to as \'referring to a unit identifier\', but the effective
 * data type was the same as <code>UnitSIdRef</code> in Level&nbsp;3.  These and
 * other methods of libSBML refer to the type <code>UnitSIdRef</code> for all
 * Levels of SBML, even if the corresponding SBML specification did not
 * explicitly name the data type.
 *
 *
 *
 * This method works by looking at all unit identifier attribute values
 * (including, if appropriate, inside mathematical formulas), comparing the
 * referenced unit identifiers to the value of @p oldid.  If any matches
 * are found, the matching values are replaced with @p newid.  The method
 * does @em not descend into child elements.
 *
 * @param oldid the old identifier.
 * @param newid the new identifier.
 *
 *
   */ public new
";


%csmethodmodifiers Species::addExpectedAttributes(ExpectedAttributes& attributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Species::readAttributes(const XMLAttributes& attributes, const ExpectedAttributes& expectedAttributes) "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Species::readL1Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Species::readL2Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Species::readL3Attributes(const XMLAttributes& attributes) "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Species::writeAttributes(XMLOutputStream& stream) const "
/** */ /* libsbml-internal */ public new
";


%csmethodmodifiers Species::isExplicitlySetBoundaryCondition() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Species::isExplicitlySetConstant() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers Species::isExplicitlySetHasOnlySubsUnits() const "
/** */ /* libsbml-internal */ public
";


%csmethodmodifiers ListOfSpecies::ListOfSpecies(unsigned int level, unsigned int version) "
/**
   * Creates a new ListOfSpecies object.
   *
   * The object is constructed such that it is valid for the given SBML
   * Level and Version combination.
   *
   * @param level the SBML Level.
   * 
   * @param version the Version within the SBML Level.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p level and @p version combination are invalid
 * or if this object is incompatible with the given level and version.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfSpecies::ListOfSpecies(SBMLNamespaces* sbmlns) "
/**
   * Creates a new ListOfSpecies object.
   *
   * The object is constructed such that it is valid for the SBML Level and
   * Version combination determined by the SBMLNamespaces object in @p
   * sbmlns.
   *
   * @param sbmlns an SBMLNamespaces object that is used to determine the
   * characteristics of the ListOfSpecies object to be created.
   *
   *
 * @throws SBMLConstructorException
 * Thrown if the given @p sbmlns is inconsistent or incompatible
 * with this object.
 *
 *
   *
   *
 * @note Attempting to add an object to an SBMLDocument having a different
 * combination of SBML Level, Version and XML namespaces than the object
 * itself will result in an error at the time a caller attempts to make the
 * addition.  A parent object must have compatible Level, Version and XML
 * namespaces.  (Strictly speaking, a parent may also have more XML
 * namespaces than a child, but the reverse is not permitted.)  The
 * restriction is necessary to ensure that an SBML model has a consistent
 * overall structure.  This requires callers to manage their objects
 * carefully, but the benefit is increased flexibility in how models can be
 * created by permitting callers to create objects bottom-up if desired.  In
 * situations where objects are not yet attached to parents (e.g.,
 * SBMLDocument), knowledge of the intented SBML Level and Version help
 * libSBML determine such things as whether it is valid to assign a
 * particular value to an attribute.
 *
 *
   */ public
";


%csmethodmodifiers ListOfSpecies::clone() const "
/**
   * Creates and returns a deep copy of this ListOfSpecies object.
   *
   * @return the (deep) copy of this ListOfSpecies object.
   */ public new
";


%csmethodmodifiers ListOfSpecies::getItemTypeCode() const "
/**
   * Returns the libSBML type code for the objects contained in this ListOf
   * (i.e., Species objects, if the list is non-empty).
   * 
   *
 * 
 * LibSBML attaches an identifying code to every kind of SBML object.  These
 * are integer constants known as <em>SBML type codes</em>.  The names of all
 * the codes begin with the characters <code>SBML_</code>.
 * @if clike The set of possible type codes for core elements is defined in
 * the enumeration #SBMLTypeCode_t, and in addition, libSBML plug-ins for
 * SBML Level&nbsp;3 packages define their own extra enumerations of type
 * codes (e.g., #SBMLLayoutTypeCode_t for the Level&nbsp;3 Layout
 * package).@endif@if java In the Java language interface for libSBML, the
 * type codes are defined as static integer constants in the interface class
 * {@link libsbmlConstants}.  @endif@if python In the Python language
 * interface for libSBML, the type codes are defined as static integer
 * constants in the interface class @link libsbml@endlink.@endif@if csharp In
 * the C# language interface for libSBML, the type codes are defined as
 * static integer constants in the interface class
 * @link libsbmlcs.libsbml@endlink.@endif  Note that different Level&nbsp;3
 * package plug-ins may use overlapping type codes; to identify the package
 * to which a given object belongs, call the 
 * <code>@if conly SBase_getPackageName()
 * @else SBase::getPackageName()
 * @endif</code>
 * method on the object.
 *
 * The exception to this is lists:  all SBML-style list elements have the type 
 * @link libsbml#SBML_LIST_OF SBML_LIST_OF@endlink, regardless of what package they 
 * are from.
 *
 *
   *
   * @return the SBML type code for objects contained in this list:
   * @link libsbml#SBML_SPECIES SBML_SPECIES@endlink (default).
   *
   * @see getElementName()
   * @see getPackageName()
   */ public new
";


%csmethodmodifiers ListOfSpecies::getElementName() const "
/**
   * Returns the XML element name of this object.
   *
   * For ListOfSpeciess, the XML element name is @c \'listOfSpeciess\'.
   * 
   * @return the name of this element, i.e