/*
 * Copyright (C) 2020 Uniontech Technology Co., Ltd.
 *
 * Author:     xinbo wang <wangxinbo@uniontech.com>
 *
 * Maintainer: xinbo wang <wangxinbo@uniontech.com>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "log.h"
#include "dtk_screenace_wayland.h"
#include "dtk_screen_ace.h"
#include <wayland-client.h>
#include "wayland-dde-restrict-client-protocol.h"

static void handle_global(void *data, struct wl_registry *wl_registry, uint32_t name, const char *interface, uint32_t version)
{
    WaylandClientBackendPtr pWayland = (WaylandClientBackendPtr)data;
    if (strcmp(interface, dde_restrict_interface.name) == 0) {
        pWayland->restrict_interface = wl_registry_bind(wl_registry, name, &dde_restrict_interface, version);
    }
}

static void handle_global_remove(void *data, struct wl_registry *wl_registry, uint32_t name)
{
    // Who cares?
}

static const struct wl_registry_listener registry_listener = {
    .global = handle_global,
    .global_remove = handle_global_remove,
};

static void *wayland_dispatch(void *arg)
{
    WaylandClientBackendPtr pWayland = (WaylandClientBackendPtr *)arg;
    if (!pWayland)
        return NULL;

    if (!pWayland->display)
        return NULL;

    while (pWayland->lock) {
        wl_display_dispatch(pWayland->display);
    }

    return NULL;
}

int init_wayland_client()
{
    if (!pScreenAce->backend) {
        log_error("wayland backend has been destroyed \n");
        return -1;
    }

    WaylandClientBackendPtr pWayland = ((WaylandClientBackend *)pScreenAce->backend);

    pWayland->lock = true;

    pWayland->display = wl_display_connect(NULL);
    if (pWayland->display == NULL) {
        log_error("failed to create display \n");
        return -1;
    }

    struct wl_registry *registry = wl_display_get_registry(pWayland->display);
    wl_registry_add_listener(registry, &registry_listener, pWayland);
    wl_display_roundtrip(pWayland->display);

    pthread_mutex_init(&pWayland->cond_lock, NULL);
    pthread_create(&pWayland->dispatch, NULL, wayland_dispatch, pWayland);

    return 0;
}

void destory_wayland_client()
{
    if (!pScreenAce->backend) {
        log_error("wayland backend has been destroyed \n");
        return;
    }

    WaylandClientBackendPtr pWayland = ((WaylandClientBackend *)pScreenAce->backend);

    pthread_mutex_lock(&pWayland->cond_lock);
    pWayland->lock = false;
    pthread_mutex_unlock(&pWayland->cond_lock);

    // cancel pthread
    pthread_exit(pWayland->dispatch);

    // display disconnect
    wl_display_disconnect(pWayland->display);

    free(pWayland);
    memset(pWayland, 0, sizeof(WaylandClientBackend));
}

int initWaylandScreenAce()
{
    if (!pScreenAce) {
        log_error("need init dtkdisplay content \n");
        return -1;
    }

    WaylandClientBackendPtr pWayland = (WaylandClientBackendPtr)malloc(sizeof(WaylandClientBackend));
    memset(pWayland, 0, sizeof(WaylandClientBackend));
    if (!pWayland) {
        fprintf(stderr, "malloc wayland backend failed \n");
        return -1;
    }

    pScreenAce->backend = pWayland;

    return init_wayland_client();
}

void destoryWaylandScreenAce()
{
    if (!pScreenAce->backend) {
        log_error("wayland backend has been destroyed \n");
        return;
    }

    destory_wayland_client();
}

bool wSetProtectedWindow(int32_t window)
{
    if (!pScreenAce->backend) {
        log_error("wayland backend has been destroyed \n");
        return false;
    }

    WaylandClientBackendPtr pWayland = ((WaylandClientBackend *)pScreenAce->backend);

    dde_restrict_set_protected_window(pWayland->restrict_interface, window);
    wl_display_flush(pWayland->display);
    return true;
}

bool wRemoveProtectedWindow(int32_t window)
{
    if (!pScreenAce->backend) {
        log_error("wayland backend has been destroyed \n");
        return false;
    }

    WaylandClientBackendPtr pWayland = ((WaylandClientBackend *)pScreenAce->backend);

    dde_restrict_remove_protected_window(pWayland->restrict_interface, window);
    wl_display_flush(pWayland->display);
    return true;
}


bool wSetScreenShotTools(const char *name)
{
    log_warn("wSetScreenShotTools is not available \n");
}

bool wRemoveScreenShotTools(char *name)
{
    log_warn("wRemoveScreenShotTools is not available \n");
}

void wFreeScreenShotTools()
{
    log_warn("wFreeScreenShotTools is not available \n");
}

void wSetProhibited(const char *white_lists, bool prohibit)
{
    if (!pScreenAce->backend) {
        log_error("wayland backend has been destroyed \n");
        return;
    }

    WaylandClientBackendPtr pWayland = ((WaylandClientBackend *)pScreenAce->backend);

    auto state = prohibit ? DDE_RESTRICT_SWITCH_FLAG_OFF : DDE_RESTRICT_SWITCH_FLAG_ON;
    dde_restrict_switch_screencast(pWayland->restrict_interface, state);

    log_warn("current wayland white list: %s\n", white_lists);
    dde_restrict_client_whitelist(pWayland->restrict_interface, white_lists);

    wl_display_flush(pWayland->display);
}