/*
 * Copyright (c) 2024
 *	Tim Woodall. All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN NO EVENT SHALL THE
 * REGENTS OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS;
 * OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
 * WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR
 * OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF
 * ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

/*
 * file-to-tape. This reads a file created by dump and writes it to a faketape
 * recreating the blocking as if it had been written by dump
 */

#include "faketape-lib.h"

#include <err.h>	// errx
#include <fcntl.h>	// O_WRONLY

// Messy. Ideally we'd use readblock_tape from restore/readtape.c but there are
// too many issues with global variables that we don't use but need to link in.


FakeTape	tape;

std::vector<uint8_t> databuffer;

void usage(const char* prog) {
	std::cerr << "Usage " << prog << " [ -b max block size ] [-r] faketape-img dump-img" << std::endl;
	std::cerr << "       -r  write as max block size blocks ignoring original block sizes." << std::endl;
	exit(1);
}

int main(int argc, char* argv[]) {

	const char* prog = argv[0];

	int ch;
	ssize_t block_size = 0;
	bool reblock = false;
	while (( ch = getopt(argc, argv, "b:r")) != -1)
		switch(ch) {
			case 'b': {
				block_size = atoi(optarg);
				if (block_size < 1024 || block_size > 16*1024*1024)
					errx(1, "Invalid blocksize, must be between 1024 and %d", 16*1024*1024);
				tape.setMaxBlockSize(block_size);
				break;
			}
			case 'r':
				reblock = true;
				break;
			default:
				usage(prog);
		}
	argc -= optind;
	argv += optind;

	if (argc < 2)
		usage(prog);

	/* First check that the target tape image does not exist */
	if (open(argv[0], O_RDONLY) != -1)
		errx(1, "%s exists, won't overwrite.",  argv[0]);

	tape.open(argv[0], O_WRONLY);
	argv++;
	argc--;

	while(argc--) {
		/* Now open the dump image */
		int dumpfd;
		warnx("Starting %s", argv[0]);
		dumpfd = open(argv[0], O_RDONLY);
		if (dumpfd == -1) {
			err(1, "%s does not exist.",  argv[0]);
		}

		std::vector<uint8_t> buffer(1024);

		/* Read the first block */
		if(int rsize = read(dumpfd, (void*)buffer.data(), 1024); rsize != 1024)
			err(1, "Failed to read first block from dump image (%d)", rsize);

		uint32_t ntrec;
		bool compressed;

		// Manually copied from protocols/dumprestore.h
		int32_t* c_magic = (int32_t*)(buffer.data()+24);
		int32_t* c_flags = (int32_t*)(buffer.data()+888);
		uint32_t* c_ntrec = (uint32_t*)(buffer.data()+896);
#define NFS_MAGIC   	(int)60012
#define DR_COMPRESSED	0x0080	/* dump tape is compressed */

		// No bswap etc. Assume it's compatible
		if (*c_magic == NFS_MAGIC) {
			compressed = *c_flags & DR_COMPRESSED;
			ntrec = *c_ntrec;
		} else
			errx(1, "This doesn't appear to be a dump tape, or was written on a system with byteswapped values");

		if(block_size == 0) {
			if (ntrec)
				block_size = ntrec * (size_t)1024;
			else
				errx(1, "Old dump tapes do not have blocksize in header. You must specify -b <blocksize> on command line");
		}

		if(ntrec && block_size != (ssize_t)ntrec*(ssize_t)1024)
			warnx("Tape appears to have been written with a blocksize of %zu but commandline specified %zu", ntrec*(size_t)1024, block_size);

		buffer.resize(block_size + 1024);
		/* Read in the rest of the first block */
		if (int rsize = read(dumpfd, buffer.data() + 1024, block_size - 1024); rsize != block_size-1024)
			err(1, "Failed to read the rest of the first block from dump image (%d)", rsize);

		/* Write first block to tape */
		if(tape.writeData(buffer.data(), block_size) != FakeTape::Response::OK)
			err(1, "Write of first block failed");

		while(true) {
			int rsize;
			if (!reblock && compressed) {
				rsize = read(dumpfd, (void*)buffer.data(), sizeof(uint32_t));
				if (rsize == 0)
					break;
				if (rsize != sizeof(uint32_t))
					err(1, "Failed to read compress header from dump image (%d)", rsize);
				int bsize = *(uint32_t*)buffer.data() >> 4;
				rsize = read(dumpfd, (void*)(buffer.data()+sizeof(uint32_t)), bsize);
				if (rsize != bsize)
					err(1, "Failed to read compressed block from dump image (%d)", rsize);
				rsize += 4;
			} else {
				rsize = read(dumpfd, (void*)buffer.data(), block_size);
				if (rsize == 0)
					break;
				if (rsize == -1)
					err(1, "Failed to read block from dump image");
				if (rsize < block_size)
					warnx("Short block read. This could be due to an incorrect -b setting");
			}
			if (rsize == block_size+4) {
				warnx("Splitting oversized block");
				tape.writeData(buffer.data(), rsize-4);
				tape.writeData(buffer.data()+rsize-4, 4);
			} else {
				tape.writeData(buffer.data(), rsize);
			}
		}
		close(dumpfd);
		warnx("Dump %s written successfully", argv[0]);
		argv++;
	}
	tape.close();
	warnx("Finished");
}

/* vim: set sw=8 sts=8 ts=8 noexpandtab: */
