#include <QCommandLineParser>
#include <QDebug>
#include <iostream>
#include "UImgOperate.h"
#include "utils/environments.h"

int main(int argc, char **argv)
{
    const char kAppDesc[] = "uimg-installer tool";

    QCoreApplication app(argc, argv);
    app.setApplicationVersion(VERSION);

    QCommandLineOption op_mode("m", "check: check img file\ninstall: install img file");
    op_mode.setValueName("mode");

    QCommandLineOption op_file("f", "img file path");
    op_file.setValueName("imgfile");

    QCommandLineOption op_device("d", "device list");
    op_device.setValueName("devices");

    // init command line
    QCommandLineParser parser;
    parser.setApplicationDescription(kAppDesc);
    parser.addHelpOption();      // 添加帮助选项 （"-h" 或 "--help"）
    parser.addVersionOption();   // 添加版本选项 ("-v" 或 "--version")
    parser.addOption(op_mode);
    parser.addOption(op_file);
    parser.addOption(op_device);
    parser.process(app);

    if (!parser.parse(app.arguments())) {
        parser.showHelp(0);
    }

    QString modeStr = parser.value(op_mode).toLocal8Bit();
    UImgOperate uImgOperate;
    if (!modeStr.compare("check")) {
        // 0---可以安装、1---谨慎安装（架构不同、CPU型号不匹配、显卡不匹配） 、2---不能安装（磁盘大小不足）
        QString deviceListStr = parser.value(op_device).toLocal8Bit();
#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
        QStringList deviceList = deviceListStr.split(";", Qt::SkipEmptyParts);
#else
        QStringList deviceList = deviceListStr.split(";", QString::SkipEmptyParts);
#endif
        QString imgFile = parser.value(op_file).toLocal8Bit();
        if (deviceList.size() == 0) {
            qInfo()<<"deviceList is empty";
            exit(UImgOperate::CheckFileExitCode::CanNotInstall);
        }

        if (!uImgOperate.mountFile(imgFile)) {
            exit(UImgOperate::CheckFileExitCode::CanNotInstall);
        }

        UImgOperate::CheckFileExitCode exitcode = uImgOperate.checkImgFile(imgFile, deviceList);

        uImgOperate.umountFile();
        exit(exitcode);

    } else if (!modeStr.compare("install")) {
        // 0---安装成功、1---MD5校验失败、2---解压失败、3---拷贝失败、4---未知错误
        QString deviceListStr = parser.value(op_device).toLocal8Bit();
#if QT_VERSION >= QT_VERSION_CHECK(5,15,0)
        QStringList deviceList = deviceListStr.split(";", Qt::SkipEmptyParts);
#else
        QStringList deviceList = deviceListStr.split(";", QString::SkipEmptyParts);
#endif
        QString imgFile = parser.value(op_file).toLocal8Bit();
        if (deviceList.size() == 0) {
            exit(UImgOperate::InstallFileExitCode::UnknowError);
        }

        if (!uImgOperate.mountFile(imgFile)) {
            exit(UImgOperate::InstallFileExitCode::UnknowError);
        }

        UImgOperate::InstallFileExitCode exitcode = uImgOperate.installImgFile(imgFile, deviceList);

        if (exitcode != UImgOperate::InstallFileExitCode::Installing) {
            uImgOperate.umountFile();
            exit(exitcode);
        }

        exitcode = (UImgOperate::InstallFileExitCode)app.exec();
        uImgOperate.umountFile();
        exit(exitcode);

    } else {
        parser.showHelp(0);
    }
}


