
/*
 * This file is part of Codecrypt.
 *
 * Copyright (C) 2013-2016 Mirek Kratochvil <exa.exa@gmail.com>
 *
 * Codecrypt is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or (at
 * your option) any later version.
 *
 * Codecrypt is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public
 * License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with Codecrypt. If not, see <http://www.gnu.org/licenses/>.
 */

#include "xsynd.h"

//computes Y=Phi(X)*A. Array sizes are fixed to 512, 16, 16.
static void xsynd_multiply (const uint64_t*A, const uint64_t*X, uint64_t*Y)
{
	int i, j;

	//ensure this is unsigned!
	const byte*Xreg = (const byte*) X;

	//clear space for the result
	for (i = 0; i < 16; ++i) Y[i] = 0;

	//cycle through all word-regular parts
	for (i = 0; i < 128; ++i) {
		int col = 128 * i + Xreg[i],
		    qc = col / 1024, //qc block
		    qcoff = qc * (1024 / 64), //block offset
		    rot = col % 1024, //rotation in the block
		    wrot = rot / 64, //rotate words
		    brot = rot % 64; //rotate bits

		//make the shifted word from two parts
		// [......(oo][oooooo)..][........]
		for (j = 0; j < 16; ++j)
			Y[j] = Y[j] ^ (
			           (A[qcoff + ( (j + wrot) % 16)]
			            >> brot)
			           |
			           (A[qcoff + ( (j + wrot + 1) % 16)]
			            << (64 - brot))
			       );
	}
}

//sizes: 512, 512, 16, 16. If R2==NULL, don't output anything.
static void xsynd_round (const uint64_t*A1, const uint64_t*A2,
                         uint64_t*R1, uint64_t*R2)
{
	if (R2) xsynd_multiply (A2, R1, R2);
	uint64_t tmp[16];
	for (int i = 0; i < 16; ++i) tmp[i] = R1[i];
	xsynd_multiply (A1, tmp, R1);
}

/*
 * A1 and A2 matrices. A1 is binary representation of fractional part of
 * sqrt(2), A2 of sqrt(3).
 */

const uint64_t CA1[512] = {
	0x6a09e667f3bcc908, 0xb2fb1366ea957d3e, 0x3adec17512775099,
	0xda2f590b0667322a, 0x95f9060875714587, 0x5163fcdfb907b672,
	0x1ee950bc8738f694, 0xf0090e6c7bf44ed1, 0xa4405d0e855e3e9c,
	0xa60b38c0237866f7, 0x956379222d108b14, 0x8c1578e45ef89c67,
	0x8dab5147176fd3b9, 0x9654c68663e7909b, 0xea5e241f06dcb05d,
	0xd549411320819495, 0x0272956db1fa1dfb, 0xe9a74059d7927c18,
	0x84c9b579aa516ca3, 0x719e6836df046d8e, 0x0209b803fc646a5e,
	0x6654bd3ef7b43d7f, 0xed437c7f9444260f, 0xbd40c483ef550385,
	0x83f97bbd45efb866, 0x3107145d5febe765, 0xa49e94ec7f597105,
	0xfbfc2e1fa763ef01, 0xf3599c82f2fe500b, 0x848cf0bd252ae046,
	0xbf9f1ef7947d4676, 0x9af8c14bcc67c7c2, 0x90be76929b0578c1,
	0x0b584fb487c924f5, 0xb71f82dcd2903609, 0xdee8912983d4eaad,
	0x0eea321f7489f46a, 0x7e9030be20fb7694, 0xefb58c9984cdd70a,
	0x1da9045c3d133a06, 0x8423d6e38303d901, 0xba9da3476684796c,
	0x5cd5972dc0ff3540, 0xc3412942d6406101, 0xef6fc6de9114a2b4,
	0xf248c689c600bb40, 0xa8b56b041fd5de6e, 0x0dd0c66d4831fe7f,
	0xff5757e4710980cd, 0xbd5c268485da5e91, 0xb3e2f205b72725b9,
	0x71d60a1f888f08a0, 0xa6e100ccedc2ce5b, 0xd98aee71e42e268d,
	0x37a6072f22023461, 0x3ffc22453439ea97, 0xa999b6c9e3ce71f9,
	0x4d6092ace120ab8e, 0x550e0d5511688631, 0x778cf60350d02fe8,
	0x5f29ec8be5c72b80, 0x7af5771b825b30a0, 0xe78376a91c08c6a7,
	0xf0f8f323b36281d2, 0x25689c0b5a82047d, 0xb989f63a8a64e851,
	0x9bc0d0c1e2228048, 0x4d94f4f9bb3d4b31, 0xd489d75231b5c633,
	0xc480c96be549bf5d, 0x96678b4d2c4dda86, 0x7bd8e48029fea8c8,
	0x173567c2ba3d3ce9, 0xdfe0bd1b4dd77117, 0x8057b695b7eaf1b0,
	0x5c22c8d5feebd077, 0xfec96db8f778fc1c, 0x2bbbce1b49ebf5af,
	0x4460882958add01c, 0xa7f1b6bc0b7ec1bc, 0x6e0a6edbc67f85b2,
	0x74e0861b3a137571, 0xb16549873d211c6a, 0xae69d801e579445b,
	0xc60a3e0a4fd8968e, 0xb794bdc702d6945d, 0xa94b04a440cabc94,
	0x387c3d26fd0f3be8, 0xaf6305a53a177288, 0xaca13fb406c98291,
	0x5d83ba0d3558d81d, 0xd1159e9643eaa27e, 0xb7757a2052975b6f,
	0x4a889e3d092bb1c6, 0x85480dc2e99947b3, 0x72deda05e2192f95,
	0xb1b926512b404c33, 0x181d64a359a89b6f, 0x8864f3d575319359,
	0xaa386257a93a57a5, 0x977547fa0f606cc3, 0x2eea84ef7103d2d2,
	0xd57b6c153a5f37c2, 0xc77fe2928d321b29, 0x470eae4158b3cedc,
	0x64ade1e431a138bb, 0x7be7305b46f7fe0b, 0xc754ff3ec042bdc8,
	0x5b325996522a87b8, 0x643e3d17870f8c25, 0xb5978060d55b82fd,
	0x4ed3f3e15892b5b5, 0x236182cb2831f44c, 0xa27ad1a66fd56fdc,
	0x29a7c7a93e4279aa, 0x12d460a0d49495b6, 0xbe4dc73ddf96ee80,
	0xa9dfc22d8d385ea2, 0xecc801d740c1990c, 0x7bf28458480527bb,
	0x8ba8109d2770c3ff, 0xde121a7be434f83d, 0x0aabfbde531f74ba,
	0xfdbc2dafbece02e6, 0x5bb77b8fbb8cdb4a, 0xd8e4b9ef01e2c90f,
	0x8069d23ecfb0c8e2, 0xe27d647daa43762a, 0x64b9c7d745344fdb,
	0x516fc56fa1eae6a9, 0x5874fb501f69b17b, 0x66b81dfba11ed524,
	0x15d7ee3e53f95859, 0x46bb208161780599, 0x6805bb4f9da796f6,
	0x33c9d13f6c172c3a, 0xfe3401a32e99dd6a, 0x247e279600a3f5c0,
	0x9991a6554a35000f, 0xfe090abfbc2d8a4a, 0x881459ed127a54cc,
	0xecb63cf6373a711c, 0x9cf91566c397b282, 0x78a7f0bb8bbc4516,
	0xddcab0f6e9557448, 0xf93eeb80f69f34b5, 0x25c760f3380ef9ed,
	0xb8aceefa8db3c7ce, 0x26e42aa5984d4db9, 0x5612ec528a245fcf,
	0x43db3757f0a0a1fc, 0x24fe218f4b630720, 0x904044b232f38c40,
	0x32f1c3ab8971dd4f, 0x7966dd97614bae8d, 0x797a0848b1d5cfdb,
	0x137d0a181b20ce77, 0x5d4dbae51393bd57, 0x2a26cc5c5e0bde6e,
	0x4233efeadad3a08d, 0x0ae2119ac0ca4e07, 0xd3415082e44acc3f,
	0x0fb34acb07c96fcb, 0x5d0f4b7d8979ea9c, 0x8e7f637d661618b9,
	0x78a7544bb02fe085, 0x51a28d25dbcd92e4, 0xa3285029fdb62cc7,
	0xe5ee2c9b1a30428a, 0x5749176e085a29b5, 0xde32bd6ad845bbe2,
	0xfa16520ceaa88b5b, 0x0b39c961cc2da368, 0x3743767f97335c3f,
	0xc59571282d338c39, 0xf37e129611e11746, 0xfeb754ceb376163c,
	0x52a10a2d23e68a2b, 0xc7d3287d1a7271c5, 0xef90550b07054a63,
	0x354c86b6c938f45a, 0xe8d2fd8c2356d7c5, 0xaa21b24576372fe3,
	0x1bd41f9836af6744, 0x1fd07246d841de5c, 0xae84610c55a12ced,
	0x62f4cb51ff3a3e1b, 0xd5187e208e40e01e, 0xcc30a263c54d3b9b,
	0x3c9a1d7b2ad9c333, 0x062b1dcc099d918e, 0x32bbfea90f208344,
	0x22206e9f02dfd751, 0xa473ca66d474b10c, 0xb7c1763240b55515,
	0x970400bf860a9895, 0x14b1446e8ce42cbb, 0x1a6f70dd2fdf48d3,
	0x02f6355bb1d1a50a, 0x0514269a7fa408ea, 0xede6c207150a6840,
	0xa8c6d27783efa8f3, 0x13c9bc7cacfb1ebb, 0xc9d8f3866935e8fb,
	0xe49120a5fd11d464, 0x8dfad3e81392d736, 0xde1a29a42d8e65a7,
	0x34af485d295fb0c1, 0x1c913a06af409b12, 0x786ec8f4d8e781f3,
	0xebb855e687318d3d, 0x713dad46d3abda87, 0x79cbbe7a012a21ba,
	0x9a49f939d78fa6ce, 0xcdcc8709eb7959f5, 0x24e2ac80897af7d9,
	0xc02055be6723e024, 0x2f77940aac6be91f, 0x0dc1b6dc014de067,
	0x975807a446561678, 0xd79927bf63b06963, 0x4882675a3d495305,
	0x3f6d6750a1ce0c19, 0x667052d1ecefe273, 0x914bd42578f257ba,
	0xd3ebc167360e91f1, 0x14828ee192acbd48, 0xda1bc5373b60b7f8,
	0x105519701b014b68, 0xbe45113983838b6e, 0xc65224edaffc7ec0,
	0xa87cccc8dbb14abe, 0xcc2f91a49a967945, 0x074dee2756862f08,
	0x55ffef477ed846d2, 0xf343c69aa2b3cee3, 0xfba920abd1760f54,
	0x4936c7ab392b5ead, 0x6198df92823a890e, 0xe787badabe2a72e9,
	0x956f56bf4dede970, 0xa0c55a3f4e250b8e, 0xd9e169b1d35f9899,
	0x8f4ed78dd99aa7a2, 0x2647213b9b1e92d4, 0xf7ab0f168d317e05,
	0x8dd4d8610e03af7c, 0x63c280d956143d30, 0xa6a70be9ae917c17,
	0xa1881bd5a4a51cbb, 0xd47da2eec968d79b, 0xe8dcba5ed091a2b8,
	0x44f9ccb1604e941d, 0xb45d47b8eca63026, 0xfbb643e19b2f7d1d,
	0xf8872431f73b7517, 0xc7770f0c97347440, 0xd4f938a56fb5b13e,
	0x5a1c355764bdb351, 0xbd69d0fb7ea37f42, 0xd628a1238e53ce7b,
	0x89fbaabf02b63dbd, 0x9d8b7e4129188054, 0x42bb968c6955689d,
	0xf1f9e4252ac04260, 0x9e9afd7abbb2f4d7, 0x5390ead235a6b107,
	0x553ab8d71a5b4562, 0xdc928fc3b6f7c43c, 0xa2b2f93ae8b91b7f,
	0x06639fd55ee26372, 0x7a2516e0e3827a42, 0x61ca8cf0ed0ac84b,
	0x57bb0320af8d4a6f, 0x60911eb38aa4ce78, 0x2ba4fe9fc49892b9,
	0x07e802a3aa0d68fb, 0x98254f654ae84a40, 0xef078aa234c4c51c,
	0xf5c5d2525d1e3985, 0xf70e870a4ce3695b, 0xf0b0ad809606d072,
	0xf0d3d2bc3012e843, 0x7da12f1510876a48, 0xa12d84dc10ccea92,
	0x2a267e932b23ee26, 0xa12b053edb5aeaa8, 0x5624cfd35401f01e,
	0xca15b0f120f91af1, 0xd3b42a940db5daeb, 0x7f4ce153bfe8da57,
	0x84da8386b56ee33e, 0x9325b715b9dcbf39, 0x2df9cb2f84b5d7be,
	0x6ba241ab5798099f, 0xea94377a4622f55f, 0x48de74ed082e1594,
	0x8c2cc8eddaf2e7dc, 0x99aaa9d245d1d70d, 0x91eaed5914f70e68,
	0xe3c74e0cb10c1ad0, 0x7cf2854b9b5e5a50, 0x6984ad617818906c,
	0x7c4df3af36ac763a, 0x94189f3c67b47ebf, 0xde2d681f951c3b56,
	0x4cdda4d6ebe8b73e, 0x2c8ab135b82993c1, 0xa40dababce8336d4,
	0x3306f30f056ff27d, 0x9608fa19490ac2ac, 0xf9df3c24328b8c0d,
	0x2fc087b0c211aeb4, 0x2ccd7c1fefb34b81, 0x802bfdd348991ec7,
	0x4572c2ae2e10ce78, 0xc953eb8941abf74b, 0x88c738af788a571a,
	0xe3dbe8ac3587d5a3, 0xa557abdaeb9d8716, 0x71d6d8691dc17fa5,
	0x8f075ad46c39866d, 0x520d964b4b6db362, 0xa2831eeb936d08e1,
	0x0a0511ce0470be20, 0x7dc664b3d6888851, 0x8fe1c18c90379890,
	0x15d3d576714fce03, 0x90eabfb117dc1338, 0xaab69e8b87da1351,
	0xbca775a412ba7191, 0x17688839130b8fad, 0xf2a40992f540dfc0,
	0x26ee24bef3f4d016, 0x99dd2eb816868730, 0x6187491b27dba726,
	0xfade5099eaaf3339, 0x8f818af86971c1e5, 0xfb194a927d46ebe9,
	0xf2c456771bd07d89, 0x22e43828b5ff033b, 0x74611d82aead6e85,
	0xc607a7625beba50b, 0xa025b14121f890c5, 0x3b28afeba7909953,
	0x6f788e2c6235f2e2, 0xd5414859b14079e0, 0x0e1d43b3399ae361,
	0x8d3d0357d66a8c5f, 0x78ecc82914ba2db2, 0x1522a79f48041070,
	0xec0043201c4098a2, 0xb1c92d6819c45b70, 0x0ba30738a8a92ae1,
	0xf2af5d8060ef6f79, 0x1ec70786a7da8ae8, 0x477080fb0d504817,
	0xb35d4d3586a21763, 0x0998db4e75b501c4, 0xbea39530a46e169c,
	0x636a0ab13853f6ca, 0x624dd63f6dd21fcc, 0x8cda1b385fac62b0,
	0xbca8368422b0fc47, 0x2dc789d83df4f9f9, 0x24a2600c77a44504,
	0x9c4702ca9cdd50ba, 0xff429b0cc5b26106, 0x1ae5666f11ccd227,
	0x070399c3b1a3a42f, 0xa63f263a4d750290, 0xa2dfbbd2e076dc29,
	0xf1ea5a18e178ee34, 0x386a66da6a0aec1e, 0xaeb8574a134e613f,
	0x00003f75ab4d5b4e, 0x7dc912f991560f2f, 0x1345f1a36866980a,
	0x12bade6acd3b25fa, 0x541a7799449e3763, 0x10272d25e1079733,
	0x1f50ad4dab237316, 0x0602cd06d8f1dff0, 0xccfb02d19cac446a,
	0x2dd41f8fc3e35d32, 0xb262a6bdda94334a, 0x043d74d95de5886c,
	0xb3b882f8560dc83d, 0xf5c6cc9cef0097e7, 0x01cac74be639299f,
	0xc787e31be4add3f5, 0xe79ad479263ad484, 0x83955b92fac3ed06,
	0xffd7210d4b59bd3c, 0xb703ae584e4ffbd8, 0xf13a540e8c33f3a2,
	0x312363d8f09eadbc, 0xc74034ec318bcc44, 0xe8ba8692cc5b1dce,
	0x1dd47b145c7d750d, 0xa8db89c0e90817d0, 0x99c6385e0b3b1919,
	0x1e8136ac5131ca4d, 0x79b6ca6e3944179f, 0x1ea3e72b6738fc7e,
	0x24ca6fee73c674c3, 0x51e089d185906a48, 0x135c20f9b6c5496a,
	0xb58603b7b62ca529, 0xe9e1dcf720c289e1, 0xc492821aaa6fe6ae,
	0x37704ada5f25613b, 0xece8c442dbdda8c2, 0x5a18315984d36a36,
	0x354e817e1cc5bfa2, 0xe249688ef99828d5, 0xef670e646ba82406,
	0xafad63c56469f671, 0x6e5b4e35b99cfa03, 0xbff0133537b2a286,
	0x8a95bfcf410b25d2, 0xbf4ea7c72571741f, 0x6d8557b11d2fee8c,
	0x10e1a7c90ec00dcd, 0x67eae5d6f9e57996, 0x4e671a00e079d26b,
	0x16f885f9ffd8074f, 0xabd28b2ddc836d70, 0xa8c41adbc5344755,
	0xaff9ac24350db604, 0xebe8e44643a4c97a, 0x93df9084927d2466,
	0xaf6888810c6ed5d2, 0xb1b079dd70ff654b, 0x84aa5a86b69bcf39,
	0x0104f907c76af253, 0xcf62c803cbbfc295, 0xc7782d742bf92b02,
	0x45f0457b62e040e7, 0xcb1561b662f0e7de, 0x73c65b2fea0d3f70,
	0x56883b37b792a030, 0x533be03d9a1f359a, 0xd8408de3c8299a58,
	0x6bbeee30b5b3e205, 0xba13ba42a649b51c, 0xe0047a9bbb08c040,
	0x1b55592f274f0134, 0x1bd087d9cb591265, 0x914afb4732b4333a,
	0x47d811c9fbd2c79c, 0xaad09a90781d4b97, 0x67733721e545efea,
	0x20a8fe48eb5b1d49, 0x434a21ca029076ab, 0x54bb4efe6f1e8709,
	0x6c0c401e36298dcb, 0xf3b0e14f044c7837, 0x07e6fdc9a744ff81,
	0xb0b6b27af6f3271e, 0x9401f05316649d4d, 0x269d224a38053e04,
	0x07bad5005bd087a9, 0x64a96def976b9e4c, 0xfeec45becb0acc0d,
	0x183d782bb545e7b2, 0x13080b10e55bca97, 0x052081cbc42b938a,
	0x7d01f316310f358a, 0xf72c16484846ce7a, 0x69e81dccf0775a1f,
	0x6c21ad27cbc843dc, 0x88a5ae6145208498, 0x2251400aeb931f75,
	0x793535f979b101b2, 0xe16be96ef0e0e085, 0x3ab8a88a72aa664f,
	0x8b2394aeb3edee04, 0xba9cb92195016df8, 0x1aaecedddec77534,
	0x63c2649cd58c8692, 0x0c9dc931caa88398, 0xbb43bd097c3689ef,
	0x73dc8f2c94f219c6, 0xe5728ed6eb0c9fd5, 0xc53cf4f1feb73a90,
	0x8f28231e351f9acf, 0x5ec7bd191a9c89d1, 0x2fa41e99608df08f,
	0x5868919838e0274c, 0xaab64c2ec97f2ea9, 0xd3b565a3b4a9349e,
	0x75c70ea80e51609a, 0x35ae7f8ee25c5d19, 0x348a4d21ba7cea51,
	0x7e8e1170e591b580, 0x76f822a8c7e638f5, 0xb07bebd1775cbcb4,
	0xf398b87ae784e328, 0xd0ce27ed1bf74aab, 0x05e205420670dcb8,
	0x2f3e8fe1fdda4570, 0xa9f1b36a492e0b8c, 0xd919eca65e0a71ec,
	0xb5e398001cc8b31a, 0xd7c8fda76565a7bd, 0x8a564610a59921bf,
	0xe49239c77378d070, 0x0291ac2192439cb2, 0xcdc0ced254ca0853,
	0x3566ef36b7339c11, 0xad088cea07ccdd86
};

const uint64_t CA2[512] = {
	0xbb67ae8584caa73b, 0x25742d7078b83b89, 0x25d834cc53da4798,
	0xc720a6486e45a6e2, 0x490bcfd95ef15dbd, 0xa9930aae12228f87,
	0xcc4cf24da3a1ec68, 0xd0cd33a01ad9a383, 0xb9e122e6138c3ae6,
	0xde5ede3bd42db730, 0x1b6bf553af7b09fd, 0x6ebef33a9a9fe577,
	0x29426f30e5892ab5, 0x72816ccefc589935, 0x5f7f11c3e24f3768,
	0xa5c7cb90d5530250, 0x5df014c70adb1ec1, 0xbaf58f084433c0d5,
	0x6696a039d962d843, 0x4194aab164d642f7, 0x6e2c8f7dbed64b39,
	0xdad2f9f7726c6930, 0xc49d48ffcdc21114, 0xa6c36b2ab7018af6,
	0x5c10f806a8fcf418, 0x8ef377d2d5b5fd19, 0x42e6fd41f9d23b66,
	0x7813df420b68d313, 0xb387ae00d7a1ddd4, 0x0aa69bb5352d3522,
	0x529a7269af1ace62, 0xc8325ced0d18c246, 0x35d661cfb468be1b,
	0x096fe677a78756ff, 0x46b5506d202bbab5, 0xe3351351776080e7,
	0x51e7a6ed2ecbd5ca, 0xb0343f677bc2f4d3, 0x020b12c5a3dd5417,
	0x0514f1d910c64a78, 0x6d9f4fdde306dfce, 0x0de8466b0c0b1097,
	0x801348a9497aca75, 0xbdd1b2cb8a042330, 0xfb4a53f02fbe02b0,
	0xbd65b382e53877c9, 0xb3b518f837f09b9e, 0xa3eada4de774744c,
	0x667ee778a7de2100, 0xef3847573d88bf59, 0x3779af61a446d5cf,
	0x76c39e9b5e3a8e82, 0xeadb3bab8bbed573, 0xc82332c9b692ad88,
	0x926393687c4cfd26, 0x8bf8c11f4f394903, 0x816010414c231d5a,
	0xb4e954cca351a0b9, 0x7b9c1dfc9c263bc1, 0x8fe1b0899efae7dc,
	0x41d63fa0d29a5e5d, 0x541f52617ab4671c, 0xded759436e30da23,
	0x4491b192dc6d190f, 0xe6b15e72e0a142ca, 0x532c3971e0e57a50,
	0xd517325b52415ccf, 0x1e4feb9efba3bf09, 0x1f9c3ffb3a9b243b,
	0xb001a5fa8c9c9272, 0x115c5b9f6cf09325, 0xecc659bb47a3b2db,
	0xf660c7bfd404681a, 0x5c00c10bc4c7978e, 0x5255fc1aa1230c77,
	0x68f631ccba5fe873, 0x14527436fc1e1851, 0xffcf0709060141b5,
	0xe7c80d2f80c57e3f, 0x4f1242645dc47a2e, 0x85bcbe4b8d7a8185,
	0xe7000f7655499a68, 0x1467d71000bd0524, 0x4d47947991e01d82,
	0x23c817c45780617c, 0x2a1fed7f14f33461, 0x5af5cd378b298c36,
	0xafe497ed4e6d74b8, 0xf87312cc7d9c03b1, 0xb260ae50eae83b12,
	0x1f59686686a44115, 0x094562752ce8e7ee, 0xf2baf5383aaf67c4,
	0xb9cc3e12d6edf242, 0x4fab1e11fdea7d43, 0x3bca44d5022ed471,
	0xd20ae60d74b953c3, 0x0bbc0a8ca668f86c, 0x230ff4b42e14ee60,
	0xfa2752e462642ceb, 0xb5c383da047309ef, 0xec6fff1c9ab3b796,
	0xef8d1b446f6b7b87, 0x453acf3c77dcdab7, 0xd5eb310fb8b07a01,
	0x4e9c03e0f1e0fcdd, 0x841bc45e2186afe5, 0x8e091e66d9d71395,
	0x2fbf2cef88598adf, 0x4ce26f02044d52ef, 0x69dca5124d91adc0,
	0x08a9375da7ec5b47, 0x9530ff2146c81ab0, 0x95fe0a995a418ed5,
	0xeef9ab0b044f0636, 0x9d3e37b1b97baf10, 0xee167701aa2bdcde,
	0x28e3b3ab1924e9f9, 0x3affd5e402c83fbc, 0x35c9c1ebe0b497b1,
	0xeb1fb1333d19373e, 0x5b7327b1da58ef99, 0xc201deee108a885f,
	0x9d569d37843e775a, 0x3b7a6ebe41c3b08f, 0x740482934519c3d8,
	0x2cabd6bc17498ad9, 0xd484526b2af59e94, 0x8b13fc009a7ae97d,
	0x7ed1dd29de6c5cbe, 0xff0deb3f702f4b10, 0x5873857f2fa74515,
	0xcca70f8453dafb5b, 0xd7104a15ad925982, 0x6c47c17ac69a50cb,
	0x6280457aa7ae6a4d, 0xcf359618816b665a, 0x5949609815ffb3b9,
	0xa5ca3db6941ab0fa, 0xba197ed8870640a0, 0x714bcc395db140d3,
	0xab37df75a5c13af7, 0xfe4d8b784d8594a5, 0x9a664cb01ecb6910,
	0xbe6712b1f42d8028, 0x86f52cd61fc24a81, 0x3d04b79a53bf7bd7,
	0xc5a328606a507d61, 0xc98e40551ed0923b, 0xe10c7927cfc334b9,
	0xa72486dfcdf21739, 0x5cf771132201a0c1, 0x814249a0edd77719,
	0x33fa5a194818792f, 0x12eb0aed7c164b6a, 0x95c3786ff646361d,
	0x752dd714cbfba17c, 0xa1785eefda2e66d5, 0xda16cf72dbe0fcbb,
	0x8e7b3b066b66a5e6, 0x6c106a0d18f1c6c6, 0x3041968139d8c437,
	0xa31cea71a56170da, 0xa56d11c17e683bb6, 0x7b55a010a19ac485,
	0x3cb8ed2e56d6924f, 0x97b1313acceb628f, 0x3c430ec231e587a8,
	0x7a6bf63c254ff386, 0x8d8d74cfd9702918, 0x19e4f2593dd71b39,
	0x9146e7e378881b56, 0x3a95013f68996d54, 0xf0890c1b93c3dc17,
	0xf5b9a289d85c235c, 0xe3aab24c232c8e8c, 0x3483ce5452dae805,
	0xae5e29e5a75b0629, 0x8827c4e4b84f0be3, 0xa708c90c6d144e76,
	0x375015956fb3ffef, 0xcdb5b7965cf64926, 0x1edac6d3ac41852d,
	0x0ec3c8277c18c8bc, 0xc7b93e3310f50acb, 0x2276333d4f5dcdc7,
	0xfbc25e19c5b5ff14, 0xbb742f36b89c56a1, 0x114d058e61efccc0,
	0x311d5c693cca4fb2, 0xc1a1b30add704d78, 0xdfa3ad4ed7b1d1af,
	0x56d6f624b7d76678, 0xa4b0777234b2bf2a, 0xd8ed62270917d367,
	0x354cd4f495a91827, 0x0f27b037e5ccc339, 0x313d991a20420786,
	0x504e1f0f62b1f759, 0xc15c765e5ae61900, 0x25e2014fc8513695,
	0x55b8ff8873dd1860, 0x6189c46ce3b91b8f, 0xc44d12a89c4166ad,
	0xe4da17292ec76ef5, 0x5568868456a2077c, 0x1b9afc7727ee0eec,
	0x048caa1420236abc, 0x89b05689b798fdd4, 0x6f55281584c13f06,
	0x9471d815bd032dbd, 0xc48f14d595cd6e98, 0xbe3ccaf4c498b111,
	0x850b963a4320f5cd, 0x5af5b17b310b49a9, 0xf87722b047bcef70,
	0xa1202e1e4dbd59e4, 0xebde109e28176599, 0x8c2ea10ab5242e63,
	0x11b11c600b6bbb10, 0xf39e98fd0961f708, 0xcf37e66ea1fe4a3b,
	0xfaf81b99c6e624b8, 0x062cc413d0abe1b0, 0x64daed20edd6cb40,
	0x2512fecfa1323e05, 0x25ed207ad599a0fa, 0xa4e37569a618d100,
	0xf707f0fbcf6683c9, 0x9e848575cfafa93e, 0x238b42cfca171f20,
	0xf2f82c6bdc9cc38a, 0x491a346735081293, 0xfc8610ac1de32afc,
	0xbebae5e95a093550, 0xbbdae1c4a792149b, 0x4866f5aab49fe65b,
	0x62f707087654416b, 0x588c9f98ae9e7e76, 0x20a889d0ec2c5e5f,
	0xb87e5e2f45bbc157, 0x4663205bc72b79a2, 0x1f0c3bc1a89a2de6,
	0x243398a44e9c1783, 0xfd67d24d093e3980, 0xd4d427cf109923e3,
	0x3e9896165115401c, 0x5eea6e65c2653454, 0x586e6020b53e2b6f,
	0xaf98f24136527f3e, 0x9f4763b8b36d5710, 0x9d2aec5448539a33,
	0x2902b6e300fa4b9d, 0x083136497e112f34, 0xf8fbfc78c68ba8f8,
	0x4d90155d3db318b8, 0x93e32ac7679468df, 0xd032ce36edd00029,
	0x7aa51a4a3f6e016b, 0x5c88a32265427ba6, 0xf15c1b8584167057,
	0xf21e20a716fa1e62, 0x56d22ededaaa2e49, 0xaf6af936dce853bb,
	0x60f498927b6ba233, 0xc4a0d5df750a3b36, 0x6c622ff0cc56ae6f,
	0x3ebec2f5ceb9e76a, 0xa3a32bad562440f4, 0xa0bf324f6a25c11e,
	0xd5c77222c97560bc, 0x5405210ccaff16aa, 0xae4ae9fcc901bf36,
	0x7823a1654b7b03b1, 0x2e857800ce2256cc, 0xd55df151d8c9487b,
	0x17c106bfc2f3e5c3, 0xaf2559034e2f6e8c, 0x2e90de72e9cd5207,
	0x62da9b8507d945a4, 0xf2fbf74f9db7633d, 0xe4e5b2ec97222c29,
	0x47822f884fad5917, 0x8b364afdd6d313f0, 0x51c3ffe2c3cf0cc0,
	0xd03707201a01f7b5, 0xddd69b4f109f619e, 0x82e5ca8ecc6c16e2,
	0x21319f674c7381ed, 0x5fd1dab9db2f633d, 0x5ef5e41a0421c915,
	0x82bebf28ff27c81f, 0xa926a46d53380594, 0x0bb32c42154599f3,
	0x2093b3387038ae0b, 0x9b1a609134a61458, 0xd12be410e0e442b3,
	0x59eade83ea9df147, 0x0136e61dd8795ce8, 0xa84d30fdef6e3dec,
	0x4307d7a9719f9baf, 0x043fa429c6430172, 0x010caa94b2ad17be,
	0x49c73c64622661e8, 0x13ce89c8611c04ca, 0xfa9b06360dfd9905,
	0xefb3fee513f79b0f, 0xfac86449980a78ce, 0x63538274a8fefb28,
	0x2f89782cb1a2efeb, 0xef175f1be1c5b209, 0x04fa14cedcd60e9c,
	0x0f3e9453c0a22f78, 0xfd45d4e6300deae7, 0xa28c86e2f1b2641d,
	0x5100dfc0ed6e41b1, 0xd9ced6425f5cf812, 0x7e2b8c3dcc4b3fe3,
	0x935c0463d17ef0cf, 0x778734a6c835becc, 0x523113b7f91079a6,
	0xe7656c508234c683, 0x0fc955a6c04d29b6, 0xaa8d27488491ecdc,
	0x9197ebc13706ccb2, 0xcc5a86bf16f26598, 0xdfa64249760a0c4f,
	0x15b4a042ef0d2fa5, 0x3c7ade1fbd675961, 0x1aee36499587db99,
	0x733b86c4f37b2d0a, 0x0cc466edaeb42190, 0x60a1e5d56e29dbd4,
	0x4cb46424fefc7c23, 0xe989eb732f40fc67, 0x691a4b3783395d52,
	0x632b31aee8a18c80, 0xd177384c4c29eef5, 0x31ae3cf77bedaf3d,
	0x9830c698cab68eea, 0x9af34c5cc5dd8e94, 0x8eb0770a68084e5d,
	0x34c9b94c3b30e00f, 0x9a0a943a15d5ef5c, 0x805a883ee7defbdb,
	0xe7e6be23882ef11f, 0x8912e12a55c75b9e, 0x690ffe1d22df0b68,
	0x338da484928c5cfb, 0x5369e25ecefab635, 0x39d386f0fcf56880,
	0xed9817745553b87c, 0xe1881a59c43da44d, 0xdd1b0b0c8c814bd3,
	0x0fa7bd89b0242d44, 0x478fd2f07388c8ab, 0x09c0fdb850c1a5de,
	0x997eaadb50b2b3c6, 0x35920f070d75f971, 0x560b00e78c96665f,
	0x2f401c27abec5d79, 0x3211eaafe6d4cbb1, 0x0102a5badaa6740a,
	0x0fd0b547be022ec8, 0x538b529117b0fad6, 0x49d30e16525f3c37,
	0xd951bf082ea41c55, 0xcc0679ae8a3e0af7, 0x35614cdcdb2fc498,
	0xd2637d5a64fc5efb, 0x29548667d635c234, 0x23c538802295853d,
	0x039a73efbbb4129f, 0xa830760d33966231, 0x043bf6e13b5d228f,
	0xdc0ad03260eba422, 0x8a7789e54c1be4df, 0xa899111bb25f18d0,
	0x1999a50de05156a0, 0x83ec10e0da75fee7, 0xf41d91f970df8107,
	0xf59743507788b2b8, 0x32567533c6312e4b, 0xda019414c9b7a4c1,
	0x7fad78fb01265431, 0x03f6790b4e3eb27b, 0x3b7492703078f1d3,
	0x84998fddba8d8099, 0x4641ab75e6583293, 0xb6f019ce340c9c4b,
	0xfb6126f0e34b3617, 0xf67ccdab918a0553, 0x3375566fd52cade5,
	0xf96b836ccf02199e, 0xbd537529f60219f9, 0x1e64789238d72b2f,
	0x8a1f262b3b27e486, 0x22f56d03977a399a, 0x27a6ba765ef139db,
	0x700016e2025fe95c, 0xb6f221df91ff12e5, 0x169c45826da7eaff,
	0x543c2daf78217247, 0xd38a64d292ceb5d1, 0x850ec7f114845fb1,
	0xe5aa8eb9e9f2cb46, 0x50fb215547d26351, 0x45ee579fae7f9c7a,
	0x4ff5956bfc002bef, 0x367809d0c0ac3105, 0x6f46237ef990f5da,
	0xc86abc0b9d061aa7, 0xcc2e56e9bc79a198, 0x9ec45239e1ef9b70,
	0xa0f7415053dd9254, 0x59caa2f7e390c6ad, 0x386e517452aa1278,
	0xae95d3b07abfadec, 0xf6348ef430450959, 0x9cbc9cd7702ff63d,
	0xb2174c3a0f766bc3, 0x068f13cf540aaeab, 0xbe60fd30f79d943d,
	0xcfa521e168b9a821, 0xae53f437f51ae10a, 0x72d310290346a162,
	0xc5304323fcc19383, 0x39f1253eed01ac8b, 0x8abe53c3045e1db9,
	0x0e07d0a2d83e13e0, 0x5ee849ecb13bd785, 0x57949d96d12f5bdc,
	0x6bfbfc37a55e1e00, 0x0d9c472fc3019db8, 0x80e367aee43db558,
	0xdb9746ec804b9725, 0xd1d63747c740fdb7, 0x2011c63695a8064e,
	0x184d18a7fc430f3d, 0x6e68ed0f15e9dd32, 0xa2a95157c2f7548f,
	0x7d8f84c337236815, 0x2a55d04dd9e4b2ba, 0xb74dfa08df915885,
	0x4ea84f23987ffe76, 0x019a5e143a1e187b, 0x7d9fce40044fd69b,
	0x0fb11e63770fb8a4, 0xb33e71114bc13d87, 0x3141f7caf907665f,
	0xb66435e774a2e9c1, 0xf54e6665d9bfd597, 0xb7705519960e43fd,
	0x6306f5e1f5b5fbba, 0x7a84ed45826810e3, 0x3decf7f70caedd4e,
	0xe84cf65d68f832b7, 0x5d21dc5cf9e72c23, 0x95dce11f1dd9559c,
	0xa960b6c3e0f8b99f, 0xf7ca959534038f82, 0xbda8490a7d0fa4e6,
	0xb7670df4d93c337d, 0xe155da669c08c3fc, 0x46af190a58fe44f8,
	0xd6b23faa0c9365c0, 0x4ffebda527d89a2f, 0x84c605b17c325ff5,
	0xed918de4b52b2d53, 0xe9ae913c54ba04ba, 0x6196eee9c1197cea,
	0x23224ce502404ff7, 0xbbb3c461e3405a94, 0x900438a835348ef1,
	0x9eef3e72de08fe4b, 0x30e88f1673ad8007, 0x814ffc9c0ae46a01,
	0x4d79530c3ee5eb0a, 0x566420532a65b22c, 0x46e1d4a8f734e26e,
	0x1287f3481bdeae31, 0x1228c2f59964df8a, 0x946847d0fe2591a6,
	0xdc119b7d59d88747, 0xf0cc3d6d33bb43f5, 0x762cf928a93542f8,
	0x2ad829b09fee0b56, 0x097e4991eed0a709, 0xce9f77c42f714d8e,
	0x51e7f432f961a9e8, 0x0df11df566a860cb, 0xae93a38e69d45ef1,
	0x49dd849bf207bd78, 0x081c95fc96f55b97, 0xe960143cfbecff5d,
	0xa0d09b7dacb59dad, 0x6af78f8ce2a967e5, 0x3fea9dc34a01015a,
	0xe13bc3803470d13b, 0x4c40725ff8b8d999, 0xf4809a4ebf954314,
	0x5a9661101a9f8dd2, 0x731786b980c78070, 0xa5ca90d239ebbbf6,
	0x2ccc4999c8ca83e7, 0x6393903abe4baeca, 0xb45d603845f4e15f,
	0xe0378d772f826829, 0x1a6fa23cf76431fb, 0xa6650b6ac20dd700,
	0x42ef1cf26ef68537, 0x0b5820c47d832f94
};

void xsynd::init()
{
	int i;

	for (i = 0; i < 16; ++i) R1[i] = 0;
	for (i = 0; i < 128; ++i) block[i] = 0;
	blockpos = 128;
}

void xsynd::load_key (const byte*begin, const byte*end)
{
	uint64_t Y[16];

	//fill the key

	if (begin >= end) return; //invalid usage

	byte *ckey = (byte*) R1;
	byte *kp = ckey;
	const byte *b = begin;

	for (; b < end; ++b) { //stuff in whole key
		*kp = *b ^ *kp;
		if (++kp > ckey + 128) kp = ckey;
	}

	b = begin;
	for (; kp < ckey + 128; ++kp) { //fill up the rest
		*kp = *b ^*kp;
		if (++b == end) b = begin;
	}

	//run the XIni
	xsynd_multiply (CA1, R1, Y);
	for (int i = 0; i < 16; ++i) Y[i] = R1[i] ^ Y[i];
	xsynd_multiply (CA2, Y, R1);
	for (int i = 0; i < 16; ++i) R1[i] = Y[i] ^ R1[i];
}

byte xsynd::gen()
{
	byte r;
	gen (1, &r);
	return r;
}

void xsynd::gen (size_t n, byte*out)
{
	while (n && blockpos < 128) {
		if (out) * (out++) = block[blockpos++];
		else blockpos++;
		--n;
	}

	while (n >= 128) {
		xsynd_round (CA1, CA2, R1, (uint64_t*) out);
		out += 128;
		n -= 128;
	}

	if (!n) return;
	blockpos = 0;
	xsynd_round (CA1, CA2, R1, (uint64_t*) block);

	while (n) {
		if (out) * (out++) = block[blockpos++];
		else blockpos++;
		--n;
	}
}
