# -*- coding: utf-8 -*-
# report_html.py
"""report_html.py"""

# Copyright (C) 2012-2014 Pierluigi Villani, Federico Brega

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 3
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

import codecs

TEMPLATE_CSS = """
<style type="text/css">
    h2 {
            margin-left:100px;
            color: #000066;
            font-size: 24px
       }

    h3 {
            margin-left:150px;
            color: #000000;
            font-size: 16px
       }

    p  {
            margin-left:150px;
            margin-right:250px;
            color: #000000;
       }

    b  {
            font-size: 18px
       }
</style>
"""

MAP_SCRIPT = """
<script type="text/javascript" src="http://www.openlayers.org/api/OpenLayers.js"></script>
<script type="text/javascript">

vectorlayer = new OpenLayers.Layer.Vector("Vector Layer", {style: {strokeColor: "blue", strokeWidth: 5, strokeOpacity: 0.6}});

var map = new OpenLayers.Map({
    div: "map",
    layers: [
        new OpenLayers.Layer.OSM(
            "OpenCycleMap",
            ["http://a.tile.opencyclemap.org/cycle/${z}/${x}/${y}.png",
                "http://b.tile.opencyclemap.org/cycle/${z}/${x}/${y}.png",
                "http://c.tile.opencyclemap.org/cycle/${z}/${x}/${y}.png"],
            { type: 'png', displayOutsideMaxExtent: true,
                transitionEffect: 'resize'}
        ),
        vectorlayer
    ],
projection: new OpenLayers.Projection("EPSG:4326"),
displayProjection: new OpenLayers.Projection("EPSG:4326")
});

wkt = new OpenLayers.Format.WKT();
var features = wkt.read("LINESTRING(%s)");

if(features){
    features.geometry.transform(map.displayProjection, map.getProjectionObject());
    if(features.constructor != Array){
        features = [features];
    }
    var bounds = features[0].geometry.getBounds();
    for(var i=1; i<features.length; ++i){
        bounds.extend(features[i].geometry.getBounds());
    }
    vectorlayer.addFeatures(features);
    map.zoomToExtent(bounds);
}

</script>
"""


class Report_html():
    """html report"""
    def __init__(self, slopename):
        self.reportdata = '<html>\n<head>\n<meta http-equiv="Content-Type" content="text/html; charset=utf-8" />\n'
        if slopename != '':
            self.reportdata += '<title>' + slopename + '</title>\n'
        self.reportdata += TEMPLATE_CSS
        self.reportdata += '\n</head>\n<body>\n'

    def add_image(self, svgstring):
        """Add image"""
        self.reportdata += '<div style="margin-left:50px;">\n' + svgstring

    def add_map(self, slope, width, height):
        """Add map with route"""
        mapstring = '<div id="map" class="smallmap" style="width: %spx; height: %spx;"></div>\n'
        self.reportdata += mapstring % (str(width), str(height))
        (lat, lng) = slope.coords[0]
        stringc = "%s %s" % (lng, lat)
        for i in range(1, len(slope.coords)):
            (lat, lng) = slope.coords[i]
            stringc += ",%s %s" % (lng, lat)
        # print(stringc)
        self.reportdata += MAP_SCRIPT % (stringc,)

    def addtext(self, label, value):
        """Add label+value"""
        self.reportdata += '\n<h3><b>' + label + '</b>' + value + '</h3>'

    def addtitle(self, text):
        """Add group title"""
        self.reportdata += '\n<h2>' + text + '</h2>'

    def addnote(self, text):
        """Add comments"""
        self.reportdata += '\n<p>' + text + '</p>'

    def save(self, filepath):
        """Save html report to file"""
        self.reportdata += '\n</div>\n</body>\n</html>\n'
        with codecs.open(filepath, encoding='utf-8', mode='w') as fid:
            fid.write(self.reportdata)

# vim:sw=4:softtabstop=4:expandtab
