/*
 * Copyright (c) 2017 Kernel Labs Inc. All Rights Reserved
 *
 * Address: Kernel Labs Inc., PO Box 745, St James, NY. 11780
 * Contact: sales@kernellabs.com
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include <stdio.h>
#include <stdlib.h>
#include <libklvanc/vanc.h>

/* Normally we don't use a global, but we know our test harness will never be
   multi-threaded, and this is a really easy way to get the results out of the
   callback for comparison */
static uint16_t vancResult[16384];
static size_t vancResultCount;
static int passCount = 0;
static int failCount = 0;

#define SHOW_DETAIL 1

/* CALLBACKS for message notification */
static int cb_eia_708(void *callback_context, struct klvanc_context_s *ctx,
		      struct klvanc_packet_eia_708b_s *pkt)
{
	int ret = -1;

#ifdef SHOW_DETAIL
	/* Have the library display some debug */
	printf("Asking libklvanc to dump a struct\n");
	ret = klvanc_dump_EIA_708B(ctx, pkt);
	if (ret != 0) {
		fprintf(stderr, "Error dumping EIA-708 packet!\n");
		return -1;
	}
#endif

	uint16_t *words;
	uint16_t wordCount;
	ret = klvanc_convert_EIA_708B_to_words(pkt, &words, &wordCount);
	if (ret != 0) {
		fprintf(stderr, "Failed to convert 104 to words: %d\n", ret);
		return -1;
	}

	memcpy(vancResult, words, wordCount * sizeof(uint16_t));
	vancResultCount = wordCount;
	free(words);

	return 0;
}

static struct klvanc_callbacks_s callbacks =
{
	.eia_708b		= cb_eia_708,
};
/* END - CALLBACKS for message notification */

static unsigned char test1[] = {
	0x00, 0x00, 0x03, 0xff, 0x03, 0xff, 0x01, 0x61, 0x01, 0x01, 0x01, 0x52, 0x02, 0x96, 0x02, 0x69, 0x01, 0x52, 0x01, 0x4f, 0x02, 0x77, 0x01, 0xbc, 0x02, 0x95, 0x02, 0x72, 0x01, 0xf4, 0x02, 0xfc, 0x01, 0x80, 0x01, 0x80, 0x01, 0xfd, 0x01, 0x80, 0x01, 0x80, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x02, 0xfa, 0x02, 0x00, 0x02, 0x00, 0x01, 0x73, 0x02, 0xd1, 0x01, 0xe0, 0x02, 0x00, 0x02, 0x00, 0x02, 0x00, 0x02, 0x00, 0x02, 0x00, 0x02, 0x00, 0x02, 0x74, 0x01, 0xbc, 0x02, 0x95, 0x01, 0xbc, 0x01, 0xb4

};

/* svcinfo_present flag set but section missing */
unsigned short test_data_708_1[] = {
  0x0000, 0x03ff, 0x03ff, 0x0161, 0x0101, 0x022b, 0x0296, 0x0269, 0x022b,
  0x017f, 0x0277, 0x01bf, 0x022e, 0x0272, 0x01ea, 0x01fd, 0x0180, 0x0180,
  0x02fa, 0x0200, 0x0200, 0x02fa, 0x0200, 0x0200, 0x02fa, 0x0200, 0x0200,
  0x02fa, 0x0200, 0x0200, 0x02fa, 0x0200, 0x0200, 0x02fa, 0x0200, 0x0200,
  0x02fa, 0x0200, 0x0200, 0x02fa, 0x0200, 0x0200, 0x02fa, 0x0200, 0x0200,
  0x0274, 0x01bf, 0x022e, 0x026f, 0x028d
};

/* svcinfo_present flag set but section missing (2) */
unsigned short test_data_708_2[] = {
  0x0000, 0x03ff, 0x03ff, 0x0161, 0x0101, 0x022b, 0x0296, 0x0269, 0x022b,
  0x017f, 0x0277, 0x0263, 0x016d, 0x0272, 0x01ea, 0x01fd, 0x01d3, 0x01d3,
  0x02ff, 0x02c3, 0x0123, 0x01fe, 0x0255, 0x0154, 0x01fe, 0x0203, 0x0200,
  0x02fa, 0x0200, 0x0200, 0x02fa, 0x0200, 0x0200, 0x02fa, 0x0200, 0x0200,
  0x02fa, 0x0200, 0x0200, 0x02fa, 0x0200, 0x0200, 0x02fa, 0x0200, 0x0200,
  0x0274, 0x0263, 0x016d, 0x0164, 0x018d, 0x0040, 0x0040, 0x0040, 0x0040,
};

/* Includes Service Info */
static unsigned short test_data_708_3[] = {
	0x0000, 0x03ff, 0x03ff, 0x0161, 0x0101, 0x0259, 0x0296, 0x0269,
	0x0259, 0x014f, 0x0277, 0x01fb, 0x0168, 0x0272, 0x01f4, 0x02fc,
	0x01ec, 0x01e9, 0x01fd, 0x0173, 0x0161, 0x02ff, 0x0192, 0x0167,
	0x01fe, 0x0198, 0x0230, 0x01fe, 0x012a, 0x028b, 0x01fe, 0x0140,
	0x0205, 0x01fe, 0x0200, 0x0259, 0x01fe, 0x0266, 0x0167, 0x01fe,
	0x0120, 0x0230, 0x01fe, 0x022d, 0x0131, 0x01fe, 0x022d, 0x0230,
	0x01fe, 0x0120, 0x0173, 0x01fe, 0x026f, 0x026c, 0x01fe, 0x0269,
	0x0164, 0x01fe, 0x010d, 0x0162, 0x01fe, 0x0167, 0x0120, 0x01fe,
	0x0131, 0x022d, 0x01fe, 0x0233, 0x022d, 0x01fe, 0x0131, 0x0120,
	0x01fe, 0x0274, 0x0200, 0x0173, 0x02d2, 0x02e1, 0x0265, 0x016e,
	0x0167, 0x01c1, 0x017f, 0x02ff, 0x02e2, 0x0173, 0x0170, 0x0161,
	0x01c2, 0x01bf, 0x02ff, 0x0274, 0x01fb, 0x0168, 0x02de, 0x02bb
};

/* Includes Timecode */
static unsigned short test_data_708_4[] = {
	0x0000, 0x03ff, 0x03ff, 0x0161, 0x0101, 0x024e, 0x0296, 0x0269,
	0x024e, 0x014f, 0x02c3, 0x018c, 0x026f, 0x0271, 0x01e0, 0x02c6,
	0x0185, 0x02a6, 0x0272, 0x01f4, 0x02fc, 0x0180, 0x0180, 0x01fd,
	0x01b3, 0x0120, 0x02ff, 0x0151, 0x029f, 0x01fe, 0x018c, 0x01fe,
	0x01fe, 0x0180, 0x0230, 0x01fe, 0x0278, 0x0233, 0x01fe, 0x0233,
	0x0120, 0x01fe, 0x0110, 0x0233, 0x01fe, 0x0120, 0x0230, 0x01fe,
	0x0278, 0x0233, 0x01fe, 0x0134, 0x0120, 0x01fe, 0x0110, 0x0134,
	0x01fe, 0x0120, 0x0230, 0x01fe, 0x0278, 0x0233, 0x01fe, 0x0235,
	0x0120, 0x01fe, 0x0110, 0x0235, 0x01fe, 0x0120, 0x0230, 0x01fe,
	0x0278, 0x0233, 0x01fe, 0x0239, 0x0200, 0x02fa, 0x0200, 0x0200,
	0x0274, 0x018c, 0x026f, 0x0123, 0x01b0
};

static int test_eia_708_u16(struct klvanc_context_s *ctx, const unsigned short *arr, int items)
{
	int mismatch = 0;

	printf("\nParsing a new EIA-708 VANC packet (%d words)......\n", items);

	/* Clear out any previous results in case the callback never fires */
	vancResultCount = 0;

	printf("Original Input\n");
	for (int i = 0; i < items; i++) {
		printf("%04x ", arr[i]);
	}
	printf("\n");

	int ret = klvanc_packet_parse(ctx, 9, arr, items);

	printf("Final output\n");
	for (int i = 0; i < vancResultCount; i++) {
		printf("%04x ", vancResult[i]);
	}
	printf("\n");


	for (int i = 0; i < vancResultCount; i++) {
		if (arr[i] != vancResult[i]) {
			fprintf(stderr, "Mismatch starting at offset 0x%02x\n", i);
			mismatch = 1;
			break;
		}
	}
	if (vancResultCount == 0) {
		/* No output at all.  This is usually because the VANC parser choked
		   on the VANC checksum and thus the parser never ran */
		fprintf(stderr, "No output generated\n");
		mismatch = 1;
	}

	if (mismatch) {
		printf("Printing mismatched structure:\n");
		failCount++;
		ret = klvanc_packet_parse(ctx, 13, vancResult, vancResultCount);
	} else {
		printf("Original and generated versions match!\n");
		passCount++;
	}

	return ret;
}

static int test_eia_708(struct klvanc_context_s *ctx, const uint8_t *buf, size_t bufSize)
{
	int numWords = bufSize / 2;
	int ret;


	uint16_t *arr = malloc(bufSize);
	if (arr == NULL)
		return -1;

	for (int i = 0; i < numWords; i++) {
		arr[i] = buf[i * 2] << 8 | buf[i * 2 + 1];
	}

	ret = test_eia_708_u16(ctx, arr, numWords);

	free(arr);

	return ret;
}

int eia708_main(int argc, char *argv[])
{
	struct klvanc_context_s *ctx;
	int ret;

	if (klvanc_context_create(&ctx) < 0) {
		fprintf(stderr, "Error initializing library context\n");
		exit(1);
	}
#ifdef SHOW_DETAIL
	ctx->verbose = 1;
#endif
	ctx->callbacks = &callbacks;
	printf("Library initialized.\n");

	ret = test_eia_708(ctx, test1, sizeof(test1));
	if (ret < 0)
		fprintf(stderr, "EIA-708B failed to parse\n");

	ret = test_eia_708_u16(ctx, test_data_708_1, sizeof(test_data_708_1) / sizeof(unsigned short));
	if (ret < 0)
		fprintf(stderr, "EIA-708B-1 failed to parse\n");

	ret = test_eia_708_u16(ctx, test_data_708_2, sizeof(test_data_708_2) / sizeof(unsigned short));
	if (ret < 0)
		fprintf(stderr, "EIA-708B-2 failed to parse\n");

	ret = test_eia_708_u16(ctx, test_data_708_3, sizeof(test_data_708_3) / sizeof(unsigned short));
	if (ret < 0)
		fprintf(stderr, "EIA-708B-3 failed to parse\n");

	ret = test_eia_708_u16(ctx, test_data_708_4, sizeof(test_data_708_4) / sizeof(unsigned short));
	if (ret < 0)
		fprintf(stderr, "EIA-708B-4 failed to parse\n");

	klvanc_context_destroy(ctx);
	printf("Library destroyed.\n");

	printf("Final result: PASS: %d/%d, Failures: %d\n",
	       passCount, passCount + failCount, failCount);
	if (failCount != 0)
		return 1;
	return 0;
}
