/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Code to provide mutexes to protect non-thread-safe libraries and functions
 */

#include "spettro.h"
#include "lock.h"

/*
 * Define the lock type and the locking and unlocking functions
 * according to the system we're using
 */

#if ECORE_LOCKS
# include <Ecore.h>
  typedef Eina_Lock lock_t;
# define do_lock(lockp)   (eina_lock_take(lockp) == EINA_LOCK_SUCCEED)
# define do_unlock(lockp) (eina_lock_release(lockp) == EINA_LOCK_SUCCEED)
#elif SDL_LOCKS
# include <SDL.h>
# include <SDL_thread.h>
  typedef SDL_mutex *lock_t;
# define do_lock(lockp)   (SDL_LockMutex(*lockp) == 0)
# define do_unlock(lockp) (SDL_UnlockMutex(*lockp) == 0)
# define do_trylock(lockp) (SDL_TryLockMutex(*lockp) == 0)
#else
# error "Define one of ECORE_LOCKS and SDL_LOCKS"
#endif

/* The lock initialization function */

static bool
initialize(lock_t *lockp, bool *initp)
{
    if (!*initp) {
#if ECORE_LOCKS
	if (eina_lock_new(lockp) != EINA_TRUE)
#elif SDL_LOCKS
	if ( (*lockp = SDL_CreateMutex()) == NULL )
#endif
	    return(FALSE);
	*initp = TRUE;
    }
    return TRUE;
}

/*
 * Private data and public functions for the locks
 */

static lock_t fftw3_lock;
static bool fftw3_lock_is_initialized = FALSE;
static lock_t audio_cache_lock;
static bool audio_cache_lock_is_initialized = FALSE;
static lock_t audio_file_lock;
static bool audio_file_lock_is_initialized = FALSE;
static lock_t work_lock;
static bool work_lock_is_initialized = FALSE;
static lock_t window_lock;
static bool window_lock_is_initialized = FALSE;
static lock_t buffer_lock;
static bool buffer_lock_is_initialized = FALSE;

void
lock_fftw3()
{
    if (!initialize(&fftw3_lock, &fftw3_lock_is_initialized) ||
	!do_lock(&fftw3_lock)) {
	fprintf(stderr, "Cannot lock FFTW3\n");
	abort();
    }
}

void
unlock_fftw3()
{
    if (!do_unlock(&fftw3_lock)) {
	fprintf(stderr, "Cannot unlock FFTW3\n");
    }
}

void
lock_audio_cache()
{
    if (!initialize(&audio_cache_lock, &audio_cache_lock_is_initialized) ||
	!do_lock(&audio_cache_lock)) {
	fprintf(stderr, "Cannot lock audio_cache\n");
	abort();
    }
}

void
unlock_audio_cache()
{
    if (!do_unlock(&audio_cache_lock)) {
	fprintf(stderr, "Cannot unlock audio_cache\n");
    }
}

void
lock_audio_file()
{
    if (!initialize(&audio_file_lock, &audio_file_lock_is_initialized) ||
	!do_lock(&audio_file_lock)) {
	fprintf(stderr, "Cannot lock audio_file\n");
	abort();
    }
}

void
unlock_audio_file()
{
    if (!do_unlock(&audio_file_lock)) {
	fprintf(stderr, "Cannot unlock audio_file\n");
    }
}

bool
lock_work()
{
    if (!initialize(&work_lock, &work_lock_is_initialized))
	return FALSE;
    else
	return do_lock(&work_lock);
}

bool
unlock_work()
{
    return do_unlock(&work_lock);
}

bool
lock_window()
{
    if (!initialize(&window_lock, &window_lock_is_initialized))
	return FALSE;
    else
	return do_lock(&window_lock);
}

bool
unlock_window()
{
    return do_unlock(&window_lock);
}

bool
lock_buffer()
{
    if (!initialize(&buffer_lock, &buffer_lock_is_initialized))
	return FALSE;
    else
	return do_lock(&buffer_lock);
}

bool
unlock_buffer()
{
    return do_unlock(&buffer_lock);
}
