/*	Copyright (C) 2018-2024 Martin Guy <martinwguy@gmail.com>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation, either version 3 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* ui.h - Header for the ui.c and its callers
 * and default values of its settings.
 */

#include "gui.h"	/* for color_t */
#include "window.h"

/* UI state variables */

/* Size of display area in pixels */
extern unsigned disp_width;
extern unsigned disp_height;
#define DEFAULT_DISP_WIDTH	640
#define DEFAULT_DISP_HEIGHT	480

/* Range of frequencies to display */
extern freq_t min_freq, max_freq;
/* Default values: 9 octaves from A0 to A9 */
#define DEFAULT_MIN_FREQ	27.5
#define DEFAULT_MAX_FREQ	14080.0

/* Dynamic range of color map (values below minus this are black) */
extern float dyn_range;
#define DEFAULT_DYN_RANGE	((float)96.0)
extern float logmax;
#define DEFAULT_LOGMAX		((float)0.0)	/* 0 = log10(1.0) */

/* Screen-scroll frequency and number of pixel columns per second of audio */
extern freq_t fps;
extern freq_t max_fps;
extern freq_t ppsec;
#define DEFAULT_FPS	25.0
#define DEFAULT_PPSEC	100.0

/* The "FFT frequency": 1/fft_freq seconds of audio are windowed and FFT-ed */
extern freq_t fft_freq;
#define DEFAULT_FFT_FREQ 5.0
#define MIN_FFT_FREQ 0.1	/* Makes spettro use about 1GB */

/* Which window functions to apply to each audio sample before FFt-ing it */
extern window_function_t window_function;
#define DEFAULT_WINDOW_FUNCTION KAISER

/* The -t/--start time parameter */
extern secs_t start_time;

extern bool piano_lines;	/* Draw lines where piano keys fall? */
extern bool staff_lines;	/* Draw manuscript score staff lines? */
extern bool guitar_lines;	/* Draw guitar string lines? */
#define STAFF_LINE_COLOR  white
#define GUITAR_LINE_COLOR white

extern bool show_frequency_axes;/* Are we to show/showing the v. axis? */
extern bool show_time_axes;	/* Are we to show/showing the h. axis? */
extern bool use_tiny_font;	/* Draw axes using the tiny builtint font? */

/* Markers for the start and end of one bar, measured in seconds from
 * the start of the piece. */
extern secs_t left_bar_time, right_bar_time;
/* Number of beats per bar. If >1, the bar lines become 3 pixels wide. */
extern int beats_per_bar;
#define DEFAULT_BEATS_PER_BAR 1

/* Other option flags */
extern bool autoplay_mode;	/* !-p Start playing on startup */
extern bool mute_mode;		/* -m  Be mute */
extern bool shuffle_mode;	/* -S  Play files in random order */
extern bool fullscreen_mode;	/* -F  Start up in fullscreen mode?
				 *     and are-we-in-fullscreen-mode? */
extern bool start_up_minimized; /* Start with window minimized? */
extern bool start_up_fit;	/* Start showing whole audio file? */
extern bool green_line_off;	/* Should we omit it when repainting? */
extern double softvol;
extern double softvol_double;	/* Increase softvol so it doubles in n secs */
extern int max_threads;		/* 0 means use default (the number of CPUs) */
extern char *output_file;	/* Image file to write to */

/* End of option flags. Derived and calculated parameters follow */

/* Where is time and space is the current playing position on the screen? */
extern secs_t disp_time;	/* When in the audio file is the crosshair? */
extern void set_disp_time(secs_t when); /* Only modify it with this function */

extern int disp_offset;	/* Crosshair is in which display column? */

extern int frequency_axis_width;	/* Left axis area */
extern int note_name_axis_width;	/* Right axis area */
extern int top_margin, bottom_margin;	/* Top and bottom axes' heights */

/*
 * Values derived from the above
 */

/* Edges of graph in display coords */
#define min_x (show_frequency_axes ? frequency_axis_width : 0)
#define max_x (disp_width - 1 - (show_frequency_axes ? note_name_axis_width : 0))
#define min_y (show_time_axes ? bottom_margin : 0)
#define max_y ((disp_height - 1) - (show_time_axes ? top_margin : 0))

#define secpp	(1 / ppsec)		/* time step per column */
#define maglen	(max_y - min_y + 1)	/* Size of logarithmic spectral data
					 * == height of graph in pixels */

/*
 * Variables to control the main loop from the depths of do_key.c
 */
extern bool quitting;		/* Have they asked to quit the program? */
extern bool nexting;		/* Have they asked for the next file? */

extern void set_disp_offset(void);
