/*
 * Unix SMB/CIFS implementation.
 *
 * Copyright (C) 2022 Andreas Schneider <asn@samba.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */


#include <stdarg.h>
#include <stddef.h>
#include <stdint.h>
#include <setjmp.h>
#include <cmocka.h>

#include "libcli/auth/smbencrypt.c"

/* The following hexdumps are from a Windows Server 2022 time trace */
static const uint8_t plaintext_data[] = {
	0x14, 0x00, 0x50, 0x00, 0x61, 0x00, 0x24, 0x00,
	0x24, 0x00, 0x77, 0x00, 0x30, 0x00, 0x72, 0x00,
	0x64, 0x00, 0x40, 0x00, 0x32, 0x00, 0xc2, 0x34,
	0x7d, 0x21, 0x79, 0x05, 0xef, 0x88, 0xd7, 0x11,
	0xec, 0xe2, 0xce, 0xb5, 0xd4, 0x4d, 0x64, 0x2d,
	0x15, 0x79, 0x01, 0x39, 0xb8, 0xb9, 0x89, 0x5c,
	0x4e, 0x71, 0xbd, 0xf0, 0x14, 0x0c, 0x87, 0x72,
	0xa5, 0xfa, 0x90, 0xbe, 0x62, 0x55, 0xad, 0x7f,
	0xe9, 0x7f, 0x0d, 0x20, 0x19, 0x3a, 0x76, 0xbe,
	0xb2, 0x14, 0x6d, 0x5b, 0x25, 0x1c, 0x67, 0x3a,
	0x23, 0x45, 0x1f, 0x7e, 0x36, 0xa0, 0x95, 0xb7,
	0xa7, 0xb1, 0x33, 0xe1, 0xc4, 0xb6, 0xe6, 0x2d,
	0xd8, 0x2f, 0xe7, 0xdf, 0x01, 0xe8, 0xba, 0x02,
	0x54, 0x36, 0xe9, 0xb6, 0x5e, 0x00, 0x52, 0x9e,
	0x64, 0x00, 0xcb, 0x3c, 0x6d, 0x05, 0x43, 0x7d,
	0x01, 0x9c, 0x22, 0x18, 0x92, 0xe7, 0xa3, 0x55,
	0x65, 0x6d, 0x2e, 0xa3, 0x53, 0x6e, 0xc0, 0x67,
	0x26, 0xac, 0xaa, 0x98, 0xa4, 0xcb, 0xb4, 0x49,
	0x13, 0x60, 0xd4, 0x33, 0x2c, 0x77, 0x58, 0x5e,
	0x50, 0x45, 0xaa, 0x1e, 0x05, 0x15, 0x18, 0x59,
	0x55, 0xca, 0x14, 0x37, 0x31, 0xac, 0x63, 0xfc,
	0x63, 0xa8, 0x2a, 0xa9, 0x99, 0xec, 0x49, 0x87,
	0x64, 0x1d, 0x4e, 0xdd, 0xa3, 0xd0, 0xdc, 0x08,
	0x00, 0x17, 0xf4, 0x2f, 0x9c, 0x4a, 0x17, 0xc7,
	0xbd, 0x30, 0xb7, 0x0e, 0x81, 0xe4, 0xd5, 0x94,
	0x31, 0xff, 0xd6, 0xcc, 0xc6, 0xbb, 0x39, 0xcd,
	0x72, 0xfe, 0xa6, 0x3d, 0x0d, 0x88, 0x68, 0x40,
	0xf8, 0x51, 0x2b, 0xe6, 0xc9, 0xaa, 0x84, 0xf3,
	0xf4, 0x6e, 0x55, 0x37, 0xbf, 0x5d, 0x87, 0xce,
	0xa6, 0x80, 0x4f, 0x8f, 0x8f, 0x7b, 0xe8, 0x30,
	0xc3, 0x2e, 0x24, 0xc7, 0x3e, 0xf1, 0x9f, 0xa6,
	0x77, 0xca, 0x04, 0xbe, 0xb5, 0xe1, 0x40, 0x59,
	0x43, 0xc5, 0x30, 0xc8, 0xe7, 0xbf, 0xab, 0xfa,
	0x86, 0x62, 0xd9, 0x3a, 0x8e, 0xa9, 0x34, 0x73,
	0x20, 0x7b, 0x61, 0x1b, 0x0e, 0xca, 0x98, 0xec,
	0xa1, 0xc1, 0x78, 0xa9, 0xa7, 0x6c, 0x8c, 0xe3,
	0x21, 0x7d, 0xb9, 0x90, 0xe2, 0x73, 0x1a, 0x99,
	0x1d, 0x44, 0xa8, 0xd5, 0x7f, 0x0a, 0x59, 0x47,
	0xd0, 0xf5, 0x6c, 0x14, 0xff, 0x4a, 0x29, 0x20,
	0xb5, 0xfc, 0xe9, 0xf0, 0xa5, 0x35, 0x9e, 0x1c,
	0xa1, 0x4c, 0xec, 0xb5, 0x7d, 0x2d, 0x27, 0xff,
	0x7a, 0x42, 0x18, 0xb8, 0x53, 0x4e, 0xfb, 0xec,
	0xb1, 0xc1, 0x65, 0x2d, 0xa4, 0x69, 0x85, 0x56,
	0x61, 0x6d, 0x21, 0x66, 0x88, 0x31, 0xdf, 0xba,
	0x28, 0xc6, 0x9a, 0xf8, 0xb7, 0xf6, 0x2a, 0x43,
	0xba, 0x9b, 0x84, 0x14, 0xce, 0xa9, 0xc9, 0xf5,
	0x85, 0x6f, 0x31, 0x89, 0x8d, 0xfc, 0x25, 0x2e,
	0x98, 0x25, 0x5a, 0x03, 0xf0, 0xb8, 0x5d, 0x4a,
	0xd4, 0x4c, 0xc8, 0x62, 0x4e, 0xeb, 0x07, 0xc8,
	0x5c, 0x9e, 0x63, 0x30, 0xfe, 0x9f, 0x0f, 0x96,
	0xd0, 0xd7, 0x70, 0xad, 0xcd, 0x84, 0xbc, 0x1e,
	0x48, 0xa0, 0x20, 0x14, 0x10, 0xa4, 0xb1, 0x5b,
	0x05, 0x36, 0x9a, 0x6d, 0xb0, 0x10, 0x98, 0xbd,
	0x8d, 0xa2, 0xd1, 0xb2, 0xfa, 0x23, 0x37, 0xeb,
	0xb0, 0x04, 0x53, 0xcb, 0xa1, 0xa9, 0xc4, 0x88,
	0xdd, 0xf9, 0x80, 0xf5, 0xa9, 0xcd, 0x7b, 0xf8,
	0x77, 0x0b, 0x19, 0x84, 0x4c, 0xef, 0x2c, 0x14,
	0xa1, 0xdc, 0x9f, 0x2f, 0x41, 0xd0, 0xd0, 0x33,
	0x29, 0x8a, 0xb9, 0x39, 0x7e, 0xc9, 0x7f, 0xe7,
	0x63, 0x64, 0xa4, 0x7b, 0x4a, 0x6a, 0x33, 0xa7,
	0xaa, 0xf6, 0xca, 0x98, 0xc4, 0x9b, 0x62, 0x5b,
	0xcd, 0x53, 0x82, 0xbf, 0xf0, 0x0b, 0x9c, 0xe7,
	0xb2, 0x44, 0x1b, 0x88, 0x71, 0x61, 0xa1, 0x36,
	0x9e, 0x7a, 0x0a, 0x3c, 0x20, 0xd8, 0xff, 0xa1,
	0x23, 0x66
};

static void torture_encode_pwd_buffer514_from_str(void **state)
{
	uint8_t data[514] = {0};
	bool ok;

	ok = encode_pwd_buffer514_from_str(data, "Pa$$w0rd@2", STR_UNICODE);
	assert_true(ok);

	/* We can only compare the first 22 bytes as the rest is random data */
	assert_memory_equal(data, plaintext_data, 22);
	assert_memory_not_equal(data + 22,
				plaintext_data + 22,
				sizeof(plaintext_data) - 22);
}

static void torture_extract_pwd_blob_from_buffer514(void **state)
{
	DATA_BLOB new_password = {
		.length = 0,
	};
	bool ok;

	ok = extract_pwd_blob_from_buffer514(NULL, plaintext_data, &new_password);
	assert_true(ok);
	assert_int_equal(new_password.length, 20);
	assert_memory_equal(new_password.data,
			    plaintext_data + 2,
			    new_password.length);
	data_blob_free(&new_password);
}

static void torture_decode_pwd_string_from_buffer514(void **state)
{
	DATA_BLOB decoded_password = {
		.length = 0,
	};
	bool ok;

	ok = decode_pwd_string_from_buffer514(NULL,
					   plaintext_data,
				           CH_UTF16,
					   &decoded_password);
	assert_true(ok);
	assert_memory_equal(decoded_password.data, "Pa$$w0rd@2", decoded_password.length);
	data_blob_free(&decoded_password);
}

int main(int argc, char *argv[])
{
	int rc;
	const struct CMUnitTest tests[] = {
		cmocka_unit_test(torture_encode_pwd_buffer514_from_str),
		cmocka_unit_test(torture_extract_pwd_blob_from_buffer514),
		cmocka_unit_test(torture_decode_pwd_string_from_buffer514),
	};

	if (argc == 2) {
		cmocka_set_test_filter(argv[1]);
	}
	cmocka_set_message_output(CM_OUTPUT_SUBUNIT);

	rc = cmocka_run_group_tests(tests, NULL, NULL);

	return rc;
}
