/*
 * Copyright (c) 2021 Bitdefender
 * SPDX-License-Identifier: Apache-2.0
 */
//! Mnemonics.

use super::decode_error::DecodeError;
use core::convert::TryFrom;

/// Uniquely identifies an instruction.
#[derive(Copy, Clone, Eq, PartialEq, Hash, Debug)]
pub enum Mnemonic {
    Aaa,
    Aad,
    Aadd,
    Aam,
    Aand,
    Aas,
    Adc,
    Adcx,
    Add,
    Addpd,
    Addps,
    Addsd,
    Addss,
    Addsubpd,
    Addsubps,
    Adox,
    Aesdec,
    Aesdec128kl,
    Aesdec256kl,
    Aesdeclast,
    Aesdecwide128kl,
    Aesdecwide256kl,
    Aesenc,
    Aesenc128kl,
    Aesenc256kl,
    Aesenclast,
    Aesencwide128kl,
    Aesencwide256kl,
    Aesimc,
    Aeskeygenassist,
    Altinst,
    And,
    Andn,
    Andnpd,
    Andnps,
    Andpd,
    Andps,
    Aor,
    Arpl,
    Axor,
    Bextr,
    Blcfill,
    Blci,
    Blcic,
    Blcmsk,
    Blcs,
    Blendpd,
    Blendps,
    Blendvpd,
    Blendvps,
    Blsfill,
    Blsi,
    Blsic,
    Blsmsk,
    Blsr,
    Bndcl,
    Bndcn,
    Bndcu,
    Bndldx,
    Bndmk,
    Bndmov,
    Bndstx,
    Bound,
    Bsf,
    Bsr,
    Bswap,
    Bt,
    Btc,
    Btr,
    Bts,
    Bzhi,
    Callfd,
    Callfi,
    Callni,
    Callnr,
    Cbw,
    Cdq,
    Cdqe,
    Clac,
    Clc,
    Cld,
    Cldemote,
    Clevict0,
    Clevict1,
    Clflush,
    Clflushopt,
    Clgi,
    Cli,
    Clrssbsy,
    Clts,
    Clui,
    Clwb,
    Clzero,
    Cmc,
    Cmovcc,
    Cmp,
    Cmpbexadd,
    Cmpcxadd,
    Cmplexadd,
    Cmplxadd,
    Cmpnbexadd,
    Cmpncxadd,
    Cmpnlexadd,
    Cmpnlxadd,
    Cmpnoxadd,
    Cmpnpxadd,
    Cmpnsxadd,
    Cmpnzxadd,
    Cmpoxadd,
    Cmppd,
    Cmpps,
    Cmppxadd,
    Cmps,
    Cmpsd,
    Cmpss,
    Cmpsxadd,
    Cmpxchg,
    Cmpxchg16b,
    Cmpxchg8b,
    Cmpzxadd,
    Comisd,
    Comiss,
    Cpuid,
    CpuRead,
    CpuWrite,
    Cqo,
    Crc32,
    Cvtdq2pd,
    Cvtdq2ps,
    Cvtpd2dq,
    Cvtpd2pi,
    Cvtpd2ps,
    Cvtpi2pd,
    Cvtpi2ps,
    Cvtps2dq,
    Cvtps2pd,
    Cvtps2pi,
    Cvtsd2si,
    Cvtsd2ss,
    Cvtsi2sd,
    Cvtsi2ss,
    Cvtss2sd,
    Cvtss2si,
    Cvttpd2dq,
    Cvttpd2pi,
    Cvttps2dq,
    Cvttps2pi,
    Cvttsd2si,
    Cvttss2si,
    Cwd,
    Cwde,
    Daa,
    Das,
    Dec,
    Delay,
    Div,
    Divpd,
    Divps,
    Divsd,
    Divss,
    Dmint,
    Dppd,
    Dpps,
    Emms,
    Encls,
    Enclu,
    Enclv,
    Encodekey128,
    Encodekey256,
    Endbr,
    Enqcmd,
    Enqcmds,
    Enter,
    Erets,
    Eretu,
    Extractps,
    Extrq,
    F2xm1,
    Fabs,
    Fadd,
    Faddp,
    Fbld,
    Fbstp,
    Fchs,
    Fcmovb,
    Fcmovbe,
    Fcmove,
    Fcmovnb,
    Fcmovnbe,
    Fcmovne,
    Fcmovnu,
    Fcmovu,
    Fcom,
    Fcomi,
    Fcomip,
    Fcomp,
    Fcompp,
    Fcos,
    Fdecstp,
    Fdiv,
    Fdivp,
    Fdivr,
    Fdivrp,
    Femms,
    Ffree,
    Ffreep,
    Fiadd,
    Ficom,
    Ficomp,
    Fidiv,
    Fidivr,
    Fild,
    Fimul,
    Fincstp,
    Fist,
    Fistp,
    Fisttp,
    Fisub,
    Fisubr,
    Fld,
    Fld1,
    Fldcw,
    Fldenv,
    Fldl2e,
    Fldl2t,
    Fldlg2,
    Fldln2,
    Fldpi,
    Fldz,
    Fmul,
    Fmulp,
    Fnclex,
    Fndisi,
    Fninit,
    Fnop,
    Fnsave,
    Fnstcw,
    Fnstenv,
    Fnstsw,
    Fpatan,
    Fprem,
    Fprem1,
    Fptan,
    Frinear,
    Frndint,
    Frstor,
    Fscale,
    Fsin,
    Fsincos,
    Fsqrt,
    Fst,
    Fstdw,
    Fstp,
    Fstpnce,
    Fstsg,
    Fsub,
    Fsubp,
    Fsubr,
    Fsubrp,
    Ftst,
    Fucom,
    Fucomi,
    Fucomip,
    Fucomp,
    Fucompp,
    Fxam,
    Fxch,
    Fxrstor,
    Fxrstor64,
    Fxsave,
    Fxsave64,
    Fxtract,
    Fyl2x,
    Fyl2xp1,
    Getsec,
    Gf2p8affineinvqb,
    Gf2p8affineqb,
    Gf2p8mulb,
    Haddpd,
    Haddps,
    Hlt,
    Hreset,
    Hsubpd,
    Hsubps,
    Idiv,
    Imul,
    In,
    Inc,
    Incssp,
    Ins,
    Insertps,
    Insertq,
    Int,
    Int1,
    Int3,
    Into,
    Invd,
    Invept,
    Invlpg,
    Invlpga,
    Invlpgb,
    Invpcid,
    Invvpid,
    Iret,
    Jmpe,
    Jmpfd,
    Jmpfi,
    Jmpni,
    Jmpnr,
    Jcc,
    Jrcxz,
    Kadd,
    Kand,
    Kandn,
    Kmerge2l1h,
    Kmerge2l1l,
    Kmov,
    Knot,
    Kor,
    Kortest,
    Kshiftl,
    Kshiftr,
    Ktest,
    Kunpckbw,
    Kunpckdq,
    Kunpckwd,
    Kxnor,
    Kxor,
    Lahf,
    Lar,
    Lddqu,
    Ldmxcsr,
    Lds,
    Ldtilecfg,
    Lea,
    Leave,
    Les,
    Lfence,
    Lfs,
    Lgdt,
    Lgs,
    Lidt,
    Lkgs,
    Lldt,
    Llwpcb,
    Lmsw,
    Loadiwkey,
    Lods,
    Loop,
    Loopnz,
    Loopz,
    Lsl,
    Lss,
    Ltr,
    Lwpins,
    Lwpval,
    Lzcnt,
    Maskmovdqu,
    Maskmovq,
    Maxpd,
    Maxps,
    Maxsd,
    Maxss,
    Mcommit,
    Mfence,
    Minpd,
    Minps,
    Minsd,
    Minss,
    Monitor,
    Monitorx,
    Montmul,
    Mov,
    Movapd,
    Movaps,
    Movbe,
    Movd,
    Movddup,
    Movdir64b,
    Movdiri,
    Movdq2q,
    Movdqa,
    Movdqu,
    Movhlps,
    Movhpd,
    Movhps,
    Movlhps,
    Movlpd,
    Movlps,
    Movmskpd,
    Movmskps,
    Movntdq,
    Movntdqa,
    Movnti,
    Movntpd,
    Movntps,
    Movntq,
    Movntsd,
    Movntss,
    Movq,
    Movq2dq,
    Movs,
    Movsd,
    Movshdup,
    Movsldup,
    Movss,
    Movsx,
    Movsxd,
    Movupd,
    Movups,
    Movzx,
    MovCr,
    MovDr,
    MovTr,
    Mpsadbw,
    Mul,
    Mulpd,
    Mulps,
    Mulsd,
    Mulss,
    Mulx,
    Mwait,
    Mwaitx,
    Neg,
    Nop,
    Not,
    Or,
    Orpd,
    Orps,
    Out,
    Outs,
    Pabsb,
    Pabsd,
    Pabsw,
    Packssdw,
    Packsswb,
    Packusdw,
    Packuswb,
    Paddb,
    Paddd,
    Paddq,
    Paddsb,
    Paddsw,
    Paddusb,
    Paddusw,
    Paddw,
    Palignr,
    Pand,
    Pandn,
    Pause,
    Pavgb,
    Pavgusb,
    Pavgw,
    Pblendvb,
    Pblendw,
    Pclmulqdq,
    Pcmpeqb,
    Pcmpeqd,
    Pcmpeqq,
    Pcmpeqw,
    Pcmpestri,
    Pcmpestrm,
    Pcmpgtb,
    Pcmpgtd,
    Pcmpgtq,
    Pcmpgtw,
    Pcmpistri,
    Pcmpistrm,
    Pconfig,
    Pdep,
    Pext,
    Pextrb,
    Pextrd,
    Pextrq,
    Pextrw,
    Pf2id,
    Pf2iw,
    Pfacc,
    Pfadd,
    Pfcmpeq,
    Pfcmpge,
    Pfcmpgt,
    Pfmax,
    Pfmin,
    Pfmul,
    Pfnacc,
    Pfpnacc,
    Pfrcp,
    Pfrcpit1,
    Pfrcpit2,
    Pfrcpv,
    Pfrsqit1,
    Pfrsqrt,
    Pfrsqrtv,
    Pfsub,
    Pfsubr,
    Phaddd,
    Phaddsw,
    Phaddw,
    Phminposuw,
    Phsubd,
    Phsubsw,
    Phsubw,
    Pi2fd,
    Pi2fw,
    Pinsrb,
    Pinsrd,
    Pinsrq,
    Pinsrw,
    Pmaddubsw,
    Pmaddwd,
    Pmaxsb,
    Pmaxsd,
    Pmaxsw,
    Pmaxub,
    Pmaxud,
    Pmaxuw,
    Pminsb,
    Pminsd,
    Pminsw,
    Pminub,
    Pminud,
    Pminuw,
    Pmovmskb,
    Pmovsxbd,
    Pmovsxbq,
    Pmovsxbw,
    Pmovsxdq,
    Pmovsxwd,
    Pmovsxwq,
    Pmovzxbd,
    Pmovzxbq,
    Pmovzxbw,
    Pmovzxdq,
    Pmovzxwd,
    Pmovzxwq,
    Pmuldq,
    Pmulhrsw,
    Pmulhrw,
    Pmulhuw,
    Pmulhw,
    Pmulld,
    Pmullw,
    Pmuludq,
    Pop,
    Popa,
    Popad,
    Popcnt,
    Popf,
    Por,
    Prefetch,
    Prefetche,
    Prefetchit0,
    Prefetchit1,
    Prefetchm,
    Prefetchnta,
    Prefetcht0,
    Prefetcht1,
    Prefetcht2,
    Prefetchw,
    Prefetchwt1,
    Psadbw,
    Pshufb,
    Pshufd,
    Pshufhw,
    Pshuflw,
    Pshufw,
    Psignb,
    Psignd,
    Psignw,
    Pslld,
    Pslldq,
    Psllq,
    Psllw,
    Psmash,
    Psrad,
    Psraw,
    Psrld,
    Psrldq,
    Psrlq,
    Psrlw,
    Psubb,
    Psubd,
    Psubq,
    Psubsb,
    Psubsw,
    Psubusb,
    Psubusw,
    Psubw,
    Pswapd,
    Ptest,
    Ptwrite,
    Punpckhbw,
    Punpckhdq,
    Punpckhqdq,
    Punpckhwd,
    Punpcklbw,
    Punpckldq,
    Punpcklqdq,
    Punpcklwd,
    Push,
    Pusha,
    Pushad,
    Pushf,
    Pvalidate,
    Pxor,
    Rcl,
    Rcpps,
    Rcpss,
    Rcr,
    Rdfsbase,
    Rdgsbase,
    Rdmsr,
    Rdmsrlist,
    Rdpid,
    Rdpkru,
    Rdpmc,
    Rdpru,
    Rdrand,
    Rdseed,
    Rdshr,
    Rdtsc,
    Rdtscp,
    Retf,
    Retn,
    Rmpadjust,
    Rmpquery,
    Rmpupdate,
    Rol,
    Ror,
    Rorx,
    Roundpd,
    Roundps,
    Roundsd,
    Roundss,
    Rsdc,
    Rsldt,
    Rsm,
    Rsqrtps,
    Rsqrtss,
    Rsssp,
    Rstorssp,
    Rsts,
    Sahf,
    Sal,
    Salc,
    Sar,
    Sarx,
    Saveprevssp,
    Sbb,
    Scas,
    Seamcall,
    Seamops,
    Seamret,
    Senduipi,
    Serialize,
    Setssbsy,
    Setcc,
    Sfence,
    Sgdt,
    Sha1msg1,
    Sha1msg2,
    Sha1nexte,
    Sha1rnds4,
    Sha256msg1,
    Sha256msg2,
    Sha256rnds2,
    Shl,
    Shld,
    Shlx,
    Shr,
    Shrd,
    Shrx,
    Shufpd,
    Shufps,
    Sidt,
    Skinit,
    Sldt,
    Slwpcb,
    Smint,
    Smsw,
    Spflt,
    Sqrtpd,
    Sqrtps,
    Sqrtsd,
    Sqrtss,
    Stac,
    Stc,
    Std,
    Stgi,
    Sti,
    Stmxcsr,
    Stos,
    Str,
    Sttilecfg,
    Stui,
    Sub,
    Subpd,
    Subps,
    Subsd,
    Subss,
    Svdc,
    Svldt,
    Svts,
    Swapgs,
    Syscall,
    Sysenter,
    Sysexit,
    Sysret,
    T1mskc,
    Tcmmimfp16ps,
    Tcmmrlfp16ps,
    Tdcall,
    Tdpbf16ps,
    Tdpbssd,
    Tdpbsud,
    Tdpbusd,
    Tdpbuud,
    Tdpfp16ps,
    Test,
    Testui,
    Tileloadd,
    Tileloaddt1,
    Tilerelease,
    Tilestored,
    Tilezero,
    Tlbsync,
    Tpause,
    Tzcnt,
    Tzmsk,
    Ucomisd,
    Ucomiss,
    Ud0,
    Ud1,
    Ud2,
    Uiret,
    Umonitor,
    Umwait,
    Unpckhpd,
    Unpckhps,
    Unpcklpd,
    Unpcklps,
    V4fmaddps,
    V4fmaddss,
    V4fnmaddps,
    V4fnmaddss,
    Vaddpd,
    Vaddph,
    Vaddps,
    Vaddsd,
    Vaddsh,
    Vaddss,
    Vaddsubpd,
    Vaddsubps,
    Vaesdec,
    Vaesdeclast,
    Vaesenc,
    Vaesenclast,
    Vaesimc,
    Vaeskeygenassist,
    Valignd,
    Valignq,
    Vandnpd,
    Vandnps,
    Vandpd,
    Vandps,
    Vbcstnebf162ps,
    Vbcstnesh2ps,
    Vblendmpd,
    Vblendmps,
    Vblendpd,
    Vblendps,
    Vblendvpd,
    Vblendvps,
    Vbroadcastf128,
    Vbroadcastf32x2,
    Vbroadcastf32x4,
    Vbroadcastf32x8,
    Vbroadcastf64x2,
    Vbroadcastf64x4,
    Vbroadcasti128,
    Vbroadcasti32x2,
    Vbroadcasti32x4,
    Vbroadcasti32x8,
    Vbroadcasti64x2,
    Vbroadcasti64x4,
    Vbroadcastsd,
    Vbroadcastss,
    Vcmppd,
    Vcmpph,
    Vcmpps,
    Vcmpsd,
    Vcmpsh,
    Vcmpss,
    Vcomisd,
    Vcomish,
    Vcomiss,
    Vcompresspd,
    Vcompressps,
    Vcvtdq2pd,
    Vcvtdq2ph,
    Vcvtdq2ps,
    Vcvtne2ps2bf16,
    Vcvtneebf162ps,
    Vcvtneeph2ps,
    Vcvtneobf162ps,
    Vcvtneoph2ps,
    Vcvtneps2bf16,
    Vcvtpd2dq,
    Vcvtpd2ph,
    Vcvtpd2ps,
    Vcvtpd2qq,
    Vcvtpd2udq,
    Vcvtpd2uqq,
    Vcvtph2dq,
    Vcvtph2pd,
    Vcvtph2ps,
    Vcvtph2psx,
    Vcvtph2qq,
    Vcvtph2udq,
    Vcvtph2uqq,
    Vcvtph2uw,
    Vcvtph2w,
    Vcvtps2dq,
    Vcvtps2pd,
    Vcvtps2ph,
    Vcvtps2phx,
    Vcvtps2qq,
    Vcvtps2udq,
    Vcvtps2uqq,
    Vcvtqq2pd,
    Vcvtqq2ph,
    Vcvtqq2ps,
    Vcvtsd2sh,
    Vcvtsd2si,
    Vcvtsd2ss,
    Vcvtsd2usi,
    Vcvtsh2sd,
    Vcvtsh2si,
    Vcvtsh2ss,
    Vcvtsh2usi,
    Vcvtsi2sd,
    Vcvtsi2sh,
    Vcvtsi2ss,
    Vcvtss2sd,
    Vcvtss2sh,
    Vcvtss2si,
    Vcvtss2usi,
    Vcvttpd2dq,
    Vcvttpd2qq,
    Vcvttpd2udq,
    Vcvttpd2uqq,
    Vcvttph2dq,
    Vcvttph2qq,
    Vcvttph2udq,
    Vcvttph2uqq,
    Vcvttph2uw,
    Vcvttph2w,
    Vcvttps2dq,
    Vcvttps2qq,
    Vcvttps2udq,
    Vcvttps2uqq,
    Vcvttsd2si,
    Vcvttsd2usi,
    Vcvttsh2si,
    Vcvttsh2usi,
    Vcvttss2si,
    Vcvttss2usi,
    Vcvtudq2pd,
    Vcvtudq2ph,
    Vcvtudq2ps,
    Vcvtuqq2pd,
    Vcvtuqq2ph,
    Vcvtuqq2ps,
    Vcvtusi2sd,
    Vcvtusi2sh,
    Vcvtusi2ss,
    Vcvtuw2ph,
    Vcvtw2ph,
    Vdbpsadbw,
    Vdivpd,
    Vdivph,
    Vdivps,
    Vdivsd,
    Vdivsh,
    Vdivss,
    Vdpbf16ps,
    Vdppd,
    Vdpps,
    Verr,
    Verw,
    Vexp2pd,
    Vexp2ps,
    Vexpandpd,
    Vexpandps,
    Vextractf128,
    Vextractf32x4,
    Vextractf32x8,
    Vextractf64x2,
    Vextractf64x4,
    Vextracti128,
    Vextracti32x4,
    Vextracti32x8,
    Vextracti64x2,
    Vextracti64x4,
    Vextractps,
    Vfcmaddcph,
    Vfcmaddcsh,
    Vfcmulcph,
    Vfcmulcsh,
    Vfixupimmpd,
    Vfixupimmps,
    Vfixupimmsd,
    Vfixupimmss,
    Vfmadd132pd,
    Vfmadd132ph,
    Vfmadd132ps,
    Vfmadd132sd,
    Vfmadd132sh,
    Vfmadd132ss,
    Vfmadd213pd,
    Vfmadd213ph,
    Vfmadd213ps,
    Vfmadd213sd,
    Vfmadd213sh,
    Vfmadd213ss,
    Vfmadd231pd,
    Vfmadd231ph,
    Vfmadd231ps,
    Vfmadd231sd,
    Vfmadd231sh,
    Vfmadd231ss,
    Vfmaddcph,
    Vfmaddcsh,
    Vfmaddpd,
    Vfmaddps,
    Vfmaddsd,
    Vfmaddss,
    Vfmaddsub132pd,
    Vfmaddsub132ph,
    Vfmaddsub132ps,
    Vfmaddsub213pd,
    Vfmaddsub213ph,
    Vfmaddsub213ps,
    Vfmaddsub231pd,
    Vfmaddsub231ph,
    Vfmaddsub231ps,
    Vfmaddsubpd,
    Vfmaddsubps,
    Vfmsub132pd,
    Vfmsub132ph,
    Vfmsub132ps,
    Vfmsub132sd,
    Vfmsub132sh,
    Vfmsub132ss,
    Vfmsub213pd,
    Vfmsub213ph,
    Vfmsub213ps,
    Vfmsub213sd,
    Vfmsub213sh,
    Vfmsub213ss,
    Vfmsub231pd,
    Vfmsub231ph,
    Vfmsub231ps,
    Vfmsub231sd,
    Vfmsub231sh,
    Vfmsub231ss,
    Vfmsubadd132pd,
    Vfmsubadd132ph,
    Vfmsubadd132ps,
    Vfmsubadd213pd,
    Vfmsubadd213ph,
    Vfmsubadd213ps,
    Vfmsubadd231pd,
    Vfmsubadd231ph,
    Vfmsubadd231ps,
    Vfmsubaddpd,
    Vfmsubaddps,
    Vfmsubpd,
    Vfmsubps,
    Vfmsubsd,
    Vfmsubss,
    Vfmulcph,
    Vfmulcsh,
    Vfnmadd132pd,
    Vfnmadd132ph,
    Vfnmadd132ps,
    Vfnmadd132sd,
    Vfnmadd132sh,
    Vfnmadd132ss,
    Vfnmadd213pd,
    Vfnmadd213ph,
    Vfnmadd213ps,
    Vfnmadd213sd,
    Vfnmadd213sh,
    Vfnmadd213ss,
    Vfnmadd231pd,
    Vfnmadd231ph,
    Vfnmadd231ps,
    Vfnmadd231sd,
    Vfnmadd231sh,
    Vfnmadd231ss,
    Vfnmaddpd,
    Vfnmaddps,
    Vfnmaddsd,
    Vfnmaddss,
    Vfnmsub132pd,
    Vfnmsub132ph,
    Vfnmsub132ps,
    Vfnmsub132sd,
    Vfnmsub132sh,
    Vfnmsub132ss,
    Vfnmsub213pd,
    Vfnmsub213ph,
    Vfnmsub213ps,
    Vfnmsub213sd,
    Vfnmsub213sh,
    Vfnmsub213ss,
    Vfnmsub231pd,
    Vfnmsub231ph,
    Vfnmsub231ps,
    Vfnmsub231sd,
    Vfnmsub231sh,
    Vfnmsub231ss,
    Vfnmsubpd,
    Vfnmsubps,
    Vfnmsubsd,
    Vfnmsubss,
    Vfpclasspd,
    Vfpclassph,
    Vfpclassps,
    Vfpclasssd,
    Vfpclasssh,
    Vfpclassss,
    Vfrczpd,
    Vfrczps,
    Vfrczsd,
    Vfrczss,
    Vgatherdpd,
    Vgatherdps,
    Vgatherpf0dpd,
    Vgatherpf0dps,
    Vgatherpf0qpd,
    Vgatherpf0qps,
    Vgatherpf1dpd,
    Vgatherpf1dps,
    Vgatherpf1qpd,
    Vgatherpf1qps,
    Vgatherqpd,
    Vgatherqps,
    Vgetexppd,
    Vgetexpph,
    Vgetexpps,
    Vgetexpsd,
    Vgetexpsh,
    Vgetexpss,
    Vgetmantpd,
    Vgetmantph,
    Vgetmantps,
    Vgetmantsd,
    Vgetmantsh,
    Vgetmantss,
    Vgf2p8affineinvqb,
    Vgf2p8affineqb,
    Vgf2p8mulb,
    Vhaddpd,
    Vhaddps,
    Vhsubpd,
    Vhsubps,
    Vinsertf128,
    Vinsertf32x4,
    Vinsertf32x8,
    Vinsertf64x2,
    Vinsertf64x4,
    Vinserti128,
    Vinserti32x4,
    Vinserti32x8,
    Vinserti64x2,
    Vinserti64x4,
    Vinsertps,
    Vlddqu,
    Vldmxcsr,
    Vmaskmovdqu,
    Vmaskmovpd,
    Vmaskmovps,
    Vmaxpd,
    Vmaxph,
    Vmaxps,
    Vmaxsd,
    Vmaxsh,
    Vmaxss,
    Vmcall,
    Vmclear,
    Vmfunc,
    Vmgexit,
    Vminpd,
    Vminph,
    Vminps,
    Vminsd,
    Vminsh,
    Vminss,
    Vmlaunch,
    Vmload,
    Vmmcall,
    Vmovapd,
    Vmovaps,
    Vmovd,
    Vmovddup,
    Vmovdqa,
    Vmovdqa32,
    Vmovdqa64,
    Vmovdqu,
    Vmovdqu16,
    Vmovdqu32,
    Vmovdqu64,
    Vmovdqu8,
    Vmovhlps,
    Vmovhpd,
    Vmovhps,
    Vmovlhps,
    Vmovlpd,
    Vmovlps,
    Vmovmskpd,
    Vmovmskps,
    Vmovntdq,
    Vmovntdqa,
    Vmovntpd,
    Vmovntps,
    Vmovq,
    Vmovsd,
    Vmovsh,
    Vmovshdup,
    Vmovsldup,
    Vmovss,
    Vmovupd,
    Vmovups,
    Vmovw,
    Vmpsadbw,
    Vmptrld,
    Vmptrst,
    Vmread,
    Vmresume,
    Vmrun,
    Vmsave,
    Vmulpd,
    Vmulph,
    Vmulps,
    Vmulsd,
    Vmulsh,
    Vmulss,
    Vmwrite,
    Vmxoff,
    Vmxon,
    Vorpd,
    Vorps,
    Vp2intersectd,
    Vp2intersectq,
    Vp4dpwssd,
    Vp4dpwssds,
    Vpabsb,
    Vpabsd,
    Vpabsq,
    Vpabsw,
    Vpackssdw,
    Vpacksswb,
    Vpackusdw,
    Vpackuswb,
    Vpaddb,
    Vpaddd,
    Vpaddq,
    Vpaddsb,
    Vpaddsw,
    Vpaddusb,
    Vpaddusw,
    Vpaddw,
    Vpalignr,
    Vpand,
    Vpandd,
    Vpandn,
    Vpandnd,
    Vpandnq,
    Vpandq,
    Vpavgb,
    Vpavgw,
    Vpblendd,
    Vpblendmb,
    Vpblendmd,
    Vpblendmq,
    Vpblendmw,
    Vpblendvb,
    Vpblendw,
    Vpbroadcastb,
    Vpbroadcastd,
    Vpbroadcastmb2q,
    Vpbroadcastmw2d,
    Vpbroadcastq,
    Vpbroadcastw,
    Vpclmulqdq,
    Vpcmov,
    Vpcmpb,
    Vpcmpd,
    Vpcmpeqb,
    Vpcmpeqd,
    Vpcmpeqq,
    Vpcmpeqw,
    Vpcmpestri,
    Vpcmpestrm,
    Vpcmpgtb,
    Vpcmpgtd,
    Vpcmpgtq,
    Vpcmpgtw,
    Vpcmpistri,
    Vpcmpistrm,
    Vpcmpq,
    Vpcmpub,
    Vpcmpud,
    Vpcmpuq,
    Vpcmpuw,
    Vpcmpw,
    Vpcomb,
    Vpcomd,
    Vpcompressb,
    Vpcompressd,
    Vpcompressq,
    Vpcompressw,
    Vpcomq,
    Vpcomub,
    Vpcomud,
    Vpcomuq,
    Vpcomuw,
    Vpcomw,
    Vpconflictd,
    Vpconflictq,
    Vpdpbssd,
    Vpdpbssds,
    Vpdpbsud,
    Vpdpbsuds,
    Vpdpbusd,
    Vpdpbusds,
    Vpdpbuud,
    Vpdpbuuds,
    Vpdpwssd,
    Vpdpwssds,
    Vperm2f128,
    Vperm2i128,
    Vpermb,
    Vpermd,
    Vpermi2b,
    Vpermi2d,
    Vpermi2pd,
    Vpermi2ps,
    Vpermi2q,
    Vpermi2w,
    Vpermil2pd,
    Vpermil2ps,
    Vpermilpd,
    Vpermilps,
    Vpermpd,
    Vpermps,
    Vpermq,
    Vpermt2b,
    Vpermt2d,
    Vpermt2pd,
    Vpermt2ps,
    Vpermt2q,
    Vpermt2w,
    Vpermw,
    Vpexpandb,
    Vpexpandd,
    Vpexpandq,
    Vpexpandw,
    Vpextrb,
    Vpextrd,
    Vpextrq,
    Vpextrw,
    Vpgatherdd,
    Vpgatherdq,
    Vpgatherqd,
    Vpgatherqq,
    Vphaddbd,
    Vphaddbq,
    Vphaddbw,
    Vphaddd,
    Vphadddq,
    Vphaddsw,
    Vphaddubd,
    Vphaddubq,
    Vphaddubw,
    Vphaddudq,
    Vphadduwd,
    Vphadduwq,
    Vphaddw,
    Vphaddwd,
    Vphaddwq,
    Vphminposuw,
    Vphsubbw,
    Vphsubd,
    Vphsubdq,
    Vphsubsw,
    Vphsubw,
    Vphsubwd,
    Vpinsrb,
    Vpinsrd,
    Vpinsrq,
    Vpinsrw,
    Vplzcntd,
    Vplzcntq,
    Vpmacsdd,
    Vpmacsdqh,
    Vpmacsdql,
    Vpmacssdd,
    Vpmacssdqh,
    Vpmacssdql,
    Vpmacsswd,
    Vpmacssww,
    Vpmacswd,
    Vpmacsww,
    Vpmadcsswd,
    Vpmadcswd,
    Vpmadd52huq,
    Vpmadd52luq,
    Vpmaddubsw,
    Vpmaddwd,
    Vpmaskmovd,
    Vpmaskmovq,
    Vpmaxsb,
    Vpmaxsd,
    Vpmaxsq,
    Vpmaxsw,
    Vpmaxub,
    Vpmaxud,
    Vpmaxuq,
    Vpmaxuw,
    Vpminsb,
    Vpminsd,
    Vpminsq,
    Vpminsw,
    Vpminub,
    Vpminud,
    Vpminuq,
    Vpminuw,
    Vpmovb2m,
    Vpmovd2m,
    Vpmovdb,
    Vpmovdw,
    Vpmovm2b,
    Vpmovm2d,
    Vpmovm2q,
    Vpmovm2w,
    Vpmovmskb,
    Vpmovq2m,
    Vpmovqb,
    Vpmovqd,
    Vpmovqw,
    Vpmovsdb,
    Vpmovsdw,
    Vpmovsqb,
    Vpmovsqd,
    Vpmovsqw,
    Vpmovswb,
    Vpmovsxbd,
    Vpmovsxbq,
    Vpmovsxbw,
    Vpmovsxdq,
    Vpmovsxwd,
    Vpmovsxwq,
    Vpmovusdb,
    Vpmovusdw,
    Vpmovusqb,
    Vpmovusqd,
    Vpmovusqw,
    Vpmovuswb,
    Vpmovw2m,
    Vpmovwb,
    Vpmovzxbd,
    Vpmovzxbq,
    Vpmovzxbw,
    Vpmovzxdq,
    Vpmovzxwd,
    Vpmovzxwq,
    Vpmuldq,
    Vpmulhrsw,
    Vpmulhuw,
    Vpmulhw,
    Vpmulld,
    Vpmullq,
    Vpmullw,
    Vpmultishiftqb,
    Vpmuludq,
    Vpopcntb,
    Vpopcntd,
    Vpopcntq,
    Vpopcntw,
    Vpor,
    Vpord,
    Vporq,
    Vpperm,
    Vprold,
    Vprolq,
    Vprolvd,
    Vprolvq,
    Vprord,
    Vprorq,
    Vprorvd,
    Vprorvq,
    Vprotb,
    Vprotd,
    Vprotq,
    Vprotw,
    Vpsadbw,
    Vpscatterdd,
    Vpscatterdq,
    Vpscatterqd,
    Vpscatterqq,
    Vpshab,
    Vpshad,
    Vpshaq,
    Vpshaw,
    Vpshlb,
    Vpshld,
    Vpshldd,
    Vpshldq,
    Vpshldvd,
    Vpshldvq,
    Vpshldvw,
    Vpshldw,
    Vpshlq,
    Vpshlw,
    Vpshrdd,
    Vpshrdq,
    Vpshrdvd,
    Vpshrdvq,
    Vpshrdvw,
    Vpshrdw,
    Vpshufb,
    Vpshufbitqmb,
    Vpshufd,
    Vpshufhw,
    Vpshuflw,
    Vpsignb,
    Vpsignd,
    Vpsignw,
    Vpslld,
    Vpslldq,
    Vpsllq,
    Vpsllvd,
    Vpsllvq,
    Vpsllvw,
    Vpsllw,
    Vpsrad,
    Vpsraq,
    Vpsravd,
    Vpsravq,
    Vpsravw,
    Vpsraw,
    Vpsrld,
    Vpsrldq,
    Vpsrlq,
    Vpsrlvd,
    Vpsrlvq,
    Vpsrlvw,
    Vpsrlw,
    Vpsubb,
    Vpsubd,
    Vpsubq,
    Vpsubsb,
    Vpsubsw,
    Vpsubusb,
    Vpsubusw,
    Vpsubw,
    Vpternlogd,
    Vpternlogq,
    Vptest,
    Vptestmb,
    Vptestmd,
    Vptestmq,
    Vptestmw,
    Vptestnmb,
    Vptestnmd,
    Vptestnmq,
    Vptestnmw,
    Vpunpckhbw,
    Vpunpckhdq,
    Vpunpckhqdq,
    Vpunpckhwd,
    Vpunpcklbw,
    Vpunpckldq,
    Vpunpcklqdq,
    Vpunpcklwd,
    Vpxor,
    Vpxord,
    Vpxorq,
    Vrangepd,
    Vrangeps,
    Vrangesd,
    Vrangess,
    Vrcp14pd,
    Vrcp14ps,
    Vrcp14sd,
    Vrcp14ss,
    Vrcp28pd,
    Vrcp28ps,
    Vrcp28sd,
    Vrcp28ss,
    Vrcpph,
    Vrcpps,
    Vrcpsh,
    Vrcpss,
    Vreducepd,
    Vreduceph,
    Vreduceps,
    Vreducesd,
    Vreducesh,
    Vreducess,
    Vrndscalepd,
    Vrndscaleph,
    Vrndscaleps,
    Vrndscalesd,
    Vrndscalesh,
    Vrndscaless,
    Vroundpd,
    Vroundps,
    Vroundsd,
    Vroundss,
    Vrsqrt14pd,
    Vrsqrt14ps,
    Vrsqrt14sd,
    Vrsqrt14ss,
    Vrsqrt28pd,
    Vrsqrt28ps,
    Vrsqrt28sd,
    Vrsqrt28ss,
    Vrsqrtph,
    Vrsqrtps,
    Vrsqrtsh,
    Vrsqrtss,
    Vscalefpd,
    Vscalefph,
    Vscalefps,
    Vscalefsd,
    Vscalefsh,
    Vscalefss,
    Vscatterdpd,
    Vscatterdps,
    Vscatterpf0dpd,
    Vscatterpf0dps,
    Vscatterpf0qpd,
    Vscatterpf0qps,
    Vscatterpf1dpd,
    Vscatterpf1dps,
    Vscatterpf1qpd,
    Vscatterpf1qps,
    Vscatterqpd,
    Vscatterqps,
    Vshuff32x4,
    Vshuff64x2,
    Vshufi32x4,
    Vshufi64x2,
    Vshufpd,
    Vshufps,
    Vsqrtpd,
    Vsqrtph,
    Vsqrtps,
    Vsqrtsd,
    Vsqrtsh,
    Vsqrtss,
    Vstmxcsr,
    Vsubpd,
    Vsubph,
    Vsubps,
    Vsubsd,
    Vsubsh,
    Vsubss,
    Vtestpd,
    Vtestps,
    Vucomisd,
    Vucomish,
    Vucomiss,
    Vunpckhpd,
    Vunpckhps,
    Vunpcklpd,
    Vunpcklps,
    Vxorpd,
    Vxorps,
    Vzeroall,
    Vzeroupper,
    Wait,
    Wbinvd,
    Wbnoinvd,
    Wrfsbase,
    Wrgsbase,
    Wrmsr,
    Wrmsrlist,
    Wrmsrns,
    Wrpkru,
    Wrshr,
    Wrss,
    Wruss,
    Xabort,
    Xadd,
    Xbegin,
    Xchg,
    Xcryptcbc,
    Xcryptcfb,
    Xcryptctr,
    Xcryptecb,
    Xcryptofb,
    Xend,
    Xgetbv,
    Xlatb,
    Xor,
    Xorpd,
    Xorps,
    Xresldtrk,
    Xrstor,
    Xrstors,
    Xsave,
    Xsavec,
    Xsaveopt,
    Xsaves,
    Xsetbv,
    Xsha1,
    Xsha256,
    Xstore,
    Xsusldtrk,
    Xtest,
}

#[doc(hidden)]
impl TryFrom<ffi::ND_INS_CLASS> for Mnemonic {
    type Error = DecodeError;

    fn try_from(value: ffi::ND_INS_CLASS) -> Result<Self, Self::Error> {
        match value {
            ffi::_ND_INS_CLASS::ND_INS_INVALID => Err(DecodeError::InternalError(value as u64)),
            ffi::_ND_INS_CLASS::ND_INS_AAA => Ok(Mnemonic::Aaa),
            ffi::_ND_INS_CLASS::ND_INS_AAD => Ok(Mnemonic::Aad),
            ffi::_ND_INS_CLASS::ND_INS_AADD => Ok(Mnemonic::Aadd),
            ffi::_ND_INS_CLASS::ND_INS_AAM => Ok(Mnemonic::Aam),
            ffi::_ND_INS_CLASS::ND_INS_AAND => Ok(Mnemonic::Aand),
            ffi::_ND_INS_CLASS::ND_INS_AAS => Ok(Mnemonic::Aas),
            ffi::_ND_INS_CLASS::ND_INS_ADC => Ok(Mnemonic::Adc),
            ffi::_ND_INS_CLASS::ND_INS_ADCX => Ok(Mnemonic::Adcx),
            ffi::_ND_INS_CLASS::ND_INS_ADD => Ok(Mnemonic::Add),
            ffi::_ND_INS_CLASS::ND_INS_ADDPD => Ok(Mnemonic::Addpd),
            ffi::_ND_INS_CLASS::ND_INS_ADDPS => Ok(Mnemonic::Addps),
            ffi::_ND_INS_CLASS::ND_INS_ADDSD => Ok(Mnemonic::Addsd),
            ffi::_ND_INS_CLASS::ND_INS_ADDSS => Ok(Mnemonic::Addss),
            ffi::_ND_INS_CLASS::ND_INS_ADDSUBPD => Ok(Mnemonic::Addsubpd),
            ffi::_ND_INS_CLASS::ND_INS_ADDSUBPS => Ok(Mnemonic::Addsubps),
            ffi::_ND_INS_CLASS::ND_INS_ADOX => Ok(Mnemonic::Adox),
            ffi::_ND_INS_CLASS::ND_INS_AESDEC => Ok(Mnemonic::Aesdec),
            ffi::_ND_INS_CLASS::ND_INS_AESDEC128KL => Ok(Mnemonic::Aesdec128kl),
            ffi::_ND_INS_CLASS::ND_INS_AESDEC256KL => Ok(Mnemonic::Aesdec256kl),
            ffi::_ND_INS_CLASS::ND_INS_AESDECLAST => Ok(Mnemonic::Aesdeclast),
            ffi::_ND_INS_CLASS::ND_INS_AESDECWIDE128KL => Ok(Mnemonic::Aesdecwide128kl),
            ffi::_ND_INS_CLASS::ND_INS_AESDECWIDE256KL => Ok(Mnemonic::Aesdecwide256kl),
            ffi::_ND_INS_CLASS::ND_INS_AESENC => Ok(Mnemonic::Aesenc),
            ffi::_ND_INS_CLASS::ND_INS_AESENC128KL => Ok(Mnemonic::Aesenc128kl),
            ffi::_ND_INS_CLASS::ND_INS_AESENC256KL => Ok(Mnemonic::Aesenc256kl),
            ffi::_ND_INS_CLASS::ND_INS_AESENCLAST => Ok(Mnemonic::Aesenclast),
            ffi::_ND_INS_CLASS::ND_INS_AESENCWIDE128KL => Ok(Mnemonic::Aesencwide128kl),
            ffi::_ND_INS_CLASS::ND_INS_AESENCWIDE256KL => Ok(Mnemonic::Aesencwide256kl),
            ffi::_ND_INS_CLASS::ND_INS_AESIMC => Ok(Mnemonic::Aesimc),
            ffi::_ND_INS_CLASS::ND_INS_AESKEYGENASSIST => Ok(Mnemonic::Aeskeygenassist),
            ffi::_ND_INS_CLASS::ND_INS_ALTINST => Ok(Mnemonic::Altinst),
            ffi::_ND_INS_CLASS::ND_INS_AND => Ok(Mnemonic::And),
            ffi::_ND_INS_CLASS::ND_INS_ANDN => Ok(Mnemonic::Andn),
            ffi::_ND_INS_CLASS::ND_INS_ANDNPD => Ok(Mnemonic::Andnpd),
            ffi::_ND_INS_CLASS::ND_INS_ANDNPS => Ok(Mnemonic::Andnps),
            ffi::_ND_INS_CLASS::ND_INS_ANDPD => Ok(Mnemonic::Andpd),
            ffi::_ND_INS_CLASS::ND_INS_ANDPS => Ok(Mnemonic::Andps),
            ffi::_ND_INS_CLASS::ND_INS_AOR => Ok(Mnemonic::Aor),
            ffi::_ND_INS_CLASS::ND_INS_ARPL => Ok(Mnemonic::Arpl),
            ffi::_ND_INS_CLASS::ND_INS_AXOR => Ok(Mnemonic::Axor),
            ffi::_ND_INS_CLASS::ND_INS_BEXTR => Ok(Mnemonic::Bextr),
            ffi::_ND_INS_CLASS::ND_INS_BLCFILL => Ok(Mnemonic::Blcfill),
            ffi::_ND_INS_CLASS::ND_INS_BLCI => Ok(Mnemonic::Blci),
            ffi::_ND_INS_CLASS::ND_INS_BLCIC => Ok(Mnemonic::Blcic),
            ffi::_ND_INS_CLASS::ND_INS_BLCMSK => Ok(Mnemonic::Blcmsk),
            ffi::_ND_INS_CLASS::ND_INS_BLCS => Ok(Mnemonic::Blcs),
            ffi::_ND_INS_CLASS::ND_INS_BLENDPD => Ok(Mnemonic::Blendpd),
            ffi::_ND_INS_CLASS::ND_INS_BLENDPS => Ok(Mnemonic::Blendps),
            ffi::_ND_INS_CLASS::ND_INS_BLENDVPD => Ok(Mnemonic::Blendvpd),
            ffi::_ND_INS_CLASS::ND_INS_BLENDVPS => Ok(Mnemonic::Blendvps),
            ffi::_ND_INS_CLASS::ND_INS_BLSFILL => Ok(Mnemonic::Blsfill),
            ffi::_ND_INS_CLASS::ND_INS_BLSI => Ok(Mnemonic::Blsi),
            ffi::_ND_INS_CLASS::ND_INS_BLSIC => Ok(Mnemonic::Blsic),
            ffi::_ND_INS_CLASS::ND_INS_BLSMSK => Ok(Mnemonic::Blsmsk),
            ffi::_ND_INS_CLASS::ND_INS_BLSR => Ok(Mnemonic::Blsr),
            ffi::_ND_INS_CLASS::ND_INS_BNDCL => Ok(Mnemonic::Bndcl),
            ffi::_ND_INS_CLASS::ND_INS_BNDCN => Ok(Mnemonic::Bndcn),
            ffi::_ND_INS_CLASS::ND_INS_BNDCU => Ok(Mnemonic::Bndcu),
            ffi::_ND_INS_CLASS::ND_INS_BNDLDX => Ok(Mnemonic::Bndldx),
            ffi::_ND_INS_CLASS::ND_INS_BNDMK => Ok(Mnemonic::Bndmk),
            ffi::_ND_INS_CLASS::ND_INS_BNDMOV => Ok(Mnemonic::Bndmov),
            ffi::_ND_INS_CLASS::ND_INS_BNDSTX => Ok(Mnemonic::Bndstx),
            ffi::_ND_INS_CLASS::ND_INS_BOUND => Ok(Mnemonic::Bound),
            ffi::_ND_INS_CLASS::ND_INS_BSF => Ok(Mnemonic::Bsf),
            ffi::_ND_INS_CLASS::ND_INS_BSR => Ok(Mnemonic::Bsr),
            ffi::_ND_INS_CLASS::ND_INS_BSWAP => Ok(Mnemonic::Bswap),
            ffi::_ND_INS_CLASS::ND_INS_BT => Ok(Mnemonic::Bt),
            ffi::_ND_INS_CLASS::ND_INS_BTC => Ok(Mnemonic::Btc),
            ffi::_ND_INS_CLASS::ND_INS_BTR => Ok(Mnemonic::Btr),
            ffi::_ND_INS_CLASS::ND_INS_BTS => Ok(Mnemonic::Bts),
            ffi::_ND_INS_CLASS::ND_INS_BZHI => Ok(Mnemonic::Bzhi),
            ffi::_ND_INS_CLASS::ND_INS_CALLFD => Ok(Mnemonic::Callfd),
            ffi::_ND_INS_CLASS::ND_INS_CALLFI => Ok(Mnemonic::Callfi),
            ffi::_ND_INS_CLASS::ND_INS_CALLNI => Ok(Mnemonic::Callni),
            ffi::_ND_INS_CLASS::ND_INS_CALLNR => Ok(Mnemonic::Callnr),
            ffi::_ND_INS_CLASS::ND_INS_CBW => Ok(Mnemonic::Cbw),
            ffi::_ND_INS_CLASS::ND_INS_CDQ => Ok(Mnemonic::Cdq),
            ffi::_ND_INS_CLASS::ND_INS_CDQE => Ok(Mnemonic::Cdqe),
            ffi::_ND_INS_CLASS::ND_INS_CLAC => Ok(Mnemonic::Clac),
            ffi::_ND_INS_CLASS::ND_INS_CLC => Ok(Mnemonic::Clc),
            ffi::_ND_INS_CLASS::ND_INS_CLD => Ok(Mnemonic::Cld),
            ffi::_ND_INS_CLASS::ND_INS_CLDEMOTE => Ok(Mnemonic::Cldemote),
            ffi::_ND_INS_CLASS::ND_INS_CLEVICT0 => Ok(Mnemonic::Clevict0),
            ffi::_ND_INS_CLASS::ND_INS_CLEVICT1 => Ok(Mnemonic::Clevict1),
            ffi::_ND_INS_CLASS::ND_INS_CLFLUSH => Ok(Mnemonic::Clflush),
            ffi::_ND_INS_CLASS::ND_INS_CLFLUSHOPT => Ok(Mnemonic::Clflushopt),
            ffi::_ND_INS_CLASS::ND_INS_CLGI => Ok(Mnemonic::Clgi),
            ffi::_ND_INS_CLASS::ND_INS_CLI => Ok(Mnemonic::Cli),
            ffi::_ND_INS_CLASS::ND_INS_CLRSSBSY => Ok(Mnemonic::Clrssbsy),
            ffi::_ND_INS_CLASS::ND_INS_CLTS => Ok(Mnemonic::Clts),
            ffi::_ND_INS_CLASS::ND_INS_CLUI => Ok(Mnemonic::Clui),
            ffi::_ND_INS_CLASS::ND_INS_CLWB => Ok(Mnemonic::Clwb),
            ffi::_ND_INS_CLASS::ND_INS_CLZERO => Ok(Mnemonic::Clzero),
            ffi::_ND_INS_CLASS::ND_INS_CMC => Ok(Mnemonic::Cmc),
            ffi::_ND_INS_CLASS::ND_INS_CMOVcc => Ok(Mnemonic::Cmovcc),
            ffi::_ND_INS_CLASS::ND_INS_CMP => Ok(Mnemonic::Cmp),
            ffi::_ND_INS_CLASS::ND_INS_CMPBEXADD => Ok(Mnemonic::Cmpbexadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPCXADD => Ok(Mnemonic::Cmpcxadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPLEXADD => Ok(Mnemonic::Cmplexadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPLXADD => Ok(Mnemonic::Cmplxadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPNBEXADD => Ok(Mnemonic::Cmpnbexadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPNCXADD => Ok(Mnemonic::Cmpncxadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPNLEXADD => Ok(Mnemonic::Cmpnlexadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPNLXADD => Ok(Mnemonic::Cmpnlxadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPNOXADD => Ok(Mnemonic::Cmpnoxadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPNPXADD => Ok(Mnemonic::Cmpnpxadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPNSXADD => Ok(Mnemonic::Cmpnsxadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPNZXADD => Ok(Mnemonic::Cmpnzxadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPOXADD => Ok(Mnemonic::Cmpoxadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPPD => Ok(Mnemonic::Cmppd),
            ffi::_ND_INS_CLASS::ND_INS_CMPPS => Ok(Mnemonic::Cmpps),
            ffi::_ND_INS_CLASS::ND_INS_CMPPXADD => Ok(Mnemonic::Cmppxadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPS => Ok(Mnemonic::Cmps),
            ffi::_ND_INS_CLASS::ND_INS_CMPSD => Ok(Mnemonic::Cmpsd),
            ffi::_ND_INS_CLASS::ND_INS_CMPSS => Ok(Mnemonic::Cmpss),
            ffi::_ND_INS_CLASS::ND_INS_CMPSXADD => Ok(Mnemonic::Cmpsxadd),
            ffi::_ND_INS_CLASS::ND_INS_CMPXCHG => Ok(Mnemonic::Cmpxchg),
            ffi::_ND_INS_CLASS::ND_INS_CMPXCHG16B => Ok(Mnemonic::Cmpxchg16b),
            ffi::_ND_INS_CLASS::ND_INS_CMPXCHG8B => Ok(Mnemonic::Cmpxchg8b),
            ffi::_ND_INS_CLASS::ND_INS_CMPZXADD => Ok(Mnemonic::Cmpzxadd),
            ffi::_ND_INS_CLASS::ND_INS_COMISD => Ok(Mnemonic::Comisd),
            ffi::_ND_INS_CLASS::ND_INS_COMISS => Ok(Mnemonic::Comiss),
            ffi::_ND_INS_CLASS::ND_INS_CPUID => Ok(Mnemonic::Cpuid),
            ffi::_ND_INS_CLASS::ND_INS_CPU_READ => Ok(Mnemonic::CpuRead),
            ffi::_ND_INS_CLASS::ND_INS_CPU_WRITE => Ok(Mnemonic::CpuWrite),
            ffi::_ND_INS_CLASS::ND_INS_CQO => Ok(Mnemonic::Cqo),
            ffi::_ND_INS_CLASS::ND_INS_CRC32 => Ok(Mnemonic::Crc32),
            ffi::_ND_INS_CLASS::ND_INS_CVTDQ2PD => Ok(Mnemonic::Cvtdq2pd),
            ffi::_ND_INS_CLASS::ND_INS_CVTDQ2PS => Ok(Mnemonic::Cvtdq2ps),
            ffi::_ND_INS_CLASS::ND_INS_CVTPD2DQ => Ok(Mnemonic::Cvtpd2dq),
            ffi::_ND_INS_CLASS::ND_INS_CVTPD2PI => Ok(Mnemonic::Cvtpd2pi),
            ffi::_ND_INS_CLASS::ND_INS_CVTPD2PS => Ok(Mnemonic::Cvtpd2ps),
            ffi::_ND_INS_CLASS::ND_INS_CVTPI2PD => Ok(Mnemonic::Cvtpi2pd),
            ffi::_ND_INS_CLASS::ND_INS_CVTPI2PS => Ok(Mnemonic::Cvtpi2ps),
            ffi::_ND_INS_CLASS::ND_INS_CVTPS2DQ => Ok(Mnemonic::Cvtps2dq),
            ffi::_ND_INS_CLASS::ND_INS_CVTPS2PD => Ok(Mnemonic::Cvtps2pd),
            ffi::_ND_INS_CLASS::ND_INS_CVTPS2PI => Ok(Mnemonic::Cvtps2pi),
            ffi::_ND_INS_CLASS::ND_INS_CVTSD2SI => Ok(Mnemonic::Cvtsd2si),
            ffi::_ND_INS_CLASS::ND_INS_CVTSD2SS => Ok(Mnemonic::Cvtsd2ss),
            ffi::_ND_INS_CLASS::ND_INS_CVTSI2SD => Ok(Mnemonic::Cvtsi2sd),
            ffi::_ND_INS_CLASS::ND_INS_CVTSI2SS => Ok(Mnemonic::Cvtsi2ss),
            ffi::_ND_INS_CLASS::ND_INS_CVTSS2SD => Ok(Mnemonic::Cvtss2sd),
            ffi::_ND_INS_CLASS::ND_INS_CVTSS2SI => Ok(Mnemonic::Cvtss2si),
            ffi::_ND_INS_CLASS::ND_INS_CVTTPD2DQ => Ok(Mnemonic::Cvttpd2dq),
            ffi::_ND_INS_CLASS::ND_INS_CVTTPD2PI => Ok(Mnemonic::Cvttpd2pi),
            ffi::_ND_INS_CLASS::ND_INS_CVTTPS2DQ => Ok(Mnemonic::Cvttps2dq),
            ffi::_ND_INS_CLASS::ND_INS_CVTTPS2PI => Ok(Mnemonic::Cvttps2pi),
            ffi::_ND_INS_CLASS::ND_INS_CVTTSD2SI => Ok(Mnemonic::Cvttsd2si),
            ffi::_ND_INS_CLASS::ND_INS_CVTTSS2SI => Ok(Mnemonic::Cvttss2si),
            ffi::_ND_INS_CLASS::ND_INS_CWD => Ok(Mnemonic::Cwd),
            ffi::_ND_INS_CLASS::ND_INS_CWDE => Ok(Mnemonic::Cwde),
            ffi::_ND_INS_CLASS::ND_INS_DAA => Ok(Mnemonic::Daa),
            ffi::_ND_INS_CLASS::ND_INS_DAS => Ok(Mnemonic::Das),
            ffi::_ND_INS_CLASS::ND_INS_DEC => Ok(Mnemonic::Dec),
            ffi::_ND_INS_CLASS::ND_INS_DELAY => Ok(Mnemonic::Delay),
            ffi::_ND_INS_CLASS::ND_INS_DIV => Ok(Mnemonic::Div),
            ffi::_ND_INS_CLASS::ND_INS_DIVPD => Ok(Mnemonic::Divpd),
            ffi::_ND_INS_CLASS::ND_INS_DIVPS => Ok(Mnemonic::Divps),
            ffi::_ND_INS_CLASS::ND_INS_DIVSD => Ok(Mnemonic::Divsd),
            ffi::_ND_INS_CLASS::ND_INS_DIVSS => Ok(Mnemonic::Divss),
            ffi::_ND_INS_CLASS::ND_INS_DMINT => Ok(Mnemonic::Dmint),
            ffi::_ND_INS_CLASS::ND_INS_DPPD => Ok(Mnemonic::Dppd),
            ffi::_ND_INS_CLASS::ND_INS_DPPS => Ok(Mnemonic::Dpps),
            ffi::_ND_INS_CLASS::ND_INS_EMMS => Ok(Mnemonic::Emms),
            ffi::_ND_INS_CLASS::ND_INS_ENCLS => Ok(Mnemonic::Encls),
            ffi::_ND_INS_CLASS::ND_INS_ENCLU => Ok(Mnemonic::Enclu),
            ffi::_ND_INS_CLASS::ND_INS_ENCLV => Ok(Mnemonic::Enclv),
            ffi::_ND_INS_CLASS::ND_INS_ENCODEKEY128 => Ok(Mnemonic::Encodekey128),
            ffi::_ND_INS_CLASS::ND_INS_ENCODEKEY256 => Ok(Mnemonic::Encodekey256),
            ffi::_ND_INS_CLASS::ND_INS_ENDBR => Ok(Mnemonic::Endbr),
            ffi::_ND_INS_CLASS::ND_INS_ENQCMD => Ok(Mnemonic::Enqcmd),
            ffi::_ND_INS_CLASS::ND_INS_ENQCMDS => Ok(Mnemonic::Enqcmds),
            ffi::_ND_INS_CLASS::ND_INS_ENTER => Ok(Mnemonic::Enter),
            ffi::_ND_INS_CLASS::ND_INS_ERETS => Ok(Mnemonic::Erets),
            ffi::_ND_INS_CLASS::ND_INS_ERETU => Ok(Mnemonic::Eretu),
            ffi::_ND_INS_CLASS::ND_INS_EXTRACTPS => Ok(Mnemonic::Extractps),
            ffi::_ND_INS_CLASS::ND_INS_EXTRQ => Ok(Mnemonic::Extrq),
            ffi::_ND_INS_CLASS::ND_INS_F2XM1 => Ok(Mnemonic::F2xm1),
            ffi::_ND_INS_CLASS::ND_INS_FABS => Ok(Mnemonic::Fabs),
            ffi::_ND_INS_CLASS::ND_INS_FADD => Ok(Mnemonic::Fadd),
            ffi::_ND_INS_CLASS::ND_INS_FADDP => Ok(Mnemonic::Faddp),
            ffi::_ND_INS_CLASS::ND_INS_FBLD => Ok(Mnemonic::Fbld),
            ffi::_ND_INS_CLASS::ND_INS_FBSTP => Ok(Mnemonic::Fbstp),
            ffi::_ND_INS_CLASS::ND_INS_FCHS => Ok(Mnemonic::Fchs),
            ffi::_ND_INS_CLASS::ND_INS_FCMOVB => Ok(Mnemonic::Fcmovb),
            ffi::_ND_INS_CLASS::ND_INS_FCMOVBE => Ok(Mnemonic::Fcmovbe),
            ffi::_ND_INS_CLASS::ND_INS_FCMOVE => Ok(Mnemonic::Fcmove),
            ffi::_ND_INS_CLASS::ND_INS_FCMOVNB => Ok(Mnemonic::Fcmovnb),
            ffi::_ND_INS_CLASS::ND_INS_FCMOVNBE => Ok(Mnemonic::Fcmovnbe),
            ffi::_ND_INS_CLASS::ND_INS_FCMOVNE => Ok(Mnemonic::Fcmovne),
            ffi::_ND_INS_CLASS::ND_INS_FCMOVNU => Ok(Mnemonic::Fcmovnu),
            ffi::_ND_INS_CLASS::ND_INS_FCMOVU => Ok(Mnemonic::Fcmovu),
            ffi::_ND_INS_CLASS::ND_INS_FCOM => Ok(Mnemonic::Fcom),
            ffi::_ND_INS_CLASS::ND_INS_FCOMI => Ok(Mnemonic::Fcomi),
            ffi::_ND_INS_CLASS::ND_INS_FCOMIP => Ok(Mnemonic::Fcomip),
            ffi::_ND_INS_CLASS::ND_INS_FCOMP => Ok(Mnemonic::Fcomp),
            ffi::_ND_INS_CLASS::ND_INS_FCOMPP => Ok(Mnemonic::Fcompp),
            ffi::_ND_INS_CLASS::ND_INS_FCOS => Ok(Mnemonic::Fcos),
            ffi::_ND_INS_CLASS::ND_INS_FDECSTP => Ok(Mnemonic::Fdecstp),
            ffi::_ND_INS_CLASS::ND_INS_FDIV => Ok(Mnemonic::Fdiv),
            ffi::_ND_INS_CLASS::ND_INS_FDIVP => Ok(Mnemonic::Fdivp),
            ffi::_ND_INS_CLASS::ND_INS_FDIVR => Ok(Mnemonic::Fdivr),
            ffi::_ND_INS_CLASS::ND_INS_FDIVRP => Ok(Mnemonic::Fdivrp),
            ffi::_ND_INS_CLASS::ND_INS_FEMMS => Ok(Mnemonic::Femms),
            ffi::_ND_INS_CLASS::ND_INS_FFREE => Ok(Mnemonic::Ffree),
            ffi::_ND_INS_CLASS::ND_INS_FFREEP => Ok(Mnemonic::Ffreep),
            ffi::_ND_INS_CLASS::ND_INS_FIADD => Ok(Mnemonic::Fiadd),
            ffi::_ND_INS_CLASS::ND_INS_FICOM => Ok(Mnemonic::Ficom),
            ffi::_ND_INS_CLASS::ND_INS_FICOMP => Ok(Mnemonic::Ficomp),
            ffi::_ND_INS_CLASS::ND_INS_FIDIV => Ok(Mnemonic::Fidiv),
            ffi::_ND_INS_CLASS::ND_INS_FIDIVR => Ok(Mnemonic::Fidivr),
            ffi::_ND_INS_CLASS::ND_INS_FILD => Ok(Mnemonic::Fild),
            ffi::_ND_INS_CLASS::ND_INS_FIMUL => Ok(Mnemonic::Fimul),
            ffi::_ND_INS_CLASS::ND_INS_FINCSTP => Ok(Mnemonic::Fincstp),
            ffi::_ND_INS_CLASS::ND_INS_FIST => Ok(Mnemonic::Fist),
            ffi::_ND_INS_CLASS::ND_INS_FISTP => Ok(Mnemonic::Fistp),
            ffi::_ND_INS_CLASS::ND_INS_FISTTP => Ok(Mnemonic::Fisttp),
            ffi::_ND_INS_CLASS::ND_INS_FISUB => Ok(Mnemonic::Fisub),
            ffi::_ND_INS_CLASS::ND_INS_FISUBR => Ok(Mnemonic::Fisubr),
            ffi::_ND_INS_CLASS::ND_INS_FLD => Ok(Mnemonic::Fld),
            ffi::_ND_INS_CLASS::ND_INS_FLD1 => Ok(Mnemonic::Fld1),
            ffi::_ND_INS_CLASS::ND_INS_FLDCW => Ok(Mnemonic::Fldcw),
            ffi::_ND_INS_CLASS::ND_INS_FLDENV => Ok(Mnemonic::Fldenv),
            ffi::_ND_INS_CLASS::ND_INS_FLDL2E => Ok(Mnemonic::Fldl2e),
            ffi::_ND_INS_CLASS::ND_INS_FLDL2T => Ok(Mnemonic::Fldl2t),
            ffi::_ND_INS_CLASS::ND_INS_FLDLG2 => Ok(Mnemonic::Fldlg2),
            ffi::_ND_INS_CLASS::ND_INS_FLDLN2 => Ok(Mnemonic::Fldln2),
            ffi::_ND_INS_CLASS::ND_INS_FLDPI => Ok(Mnemonic::Fldpi),
            ffi::_ND_INS_CLASS::ND_INS_FLDZ => Ok(Mnemonic::Fldz),
            ffi::_ND_INS_CLASS::ND_INS_FMUL => Ok(Mnemonic::Fmul),
            ffi::_ND_INS_CLASS::ND_INS_FMULP => Ok(Mnemonic::Fmulp),
            ffi::_ND_INS_CLASS::ND_INS_FNCLEX => Ok(Mnemonic::Fnclex),
            ffi::_ND_INS_CLASS::ND_INS_FNDISI => Ok(Mnemonic::Fndisi),
            ffi::_ND_INS_CLASS::ND_INS_FNINIT => Ok(Mnemonic::Fninit),
            ffi::_ND_INS_CLASS::ND_INS_FNOP => Ok(Mnemonic::Fnop),
            ffi::_ND_INS_CLASS::ND_INS_FNSAVE => Ok(Mnemonic::Fnsave),
            ffi::_ND_INS_CLASS::ND_INS_FNSTCW => Ok(Mnemonic::Fnstcw),
            ffi::_ND_INS_CLASS::ND_INS_FNSTENV => Ok(Mnemonic::Fnstenv),
            ffi::_ND_INS_CLASS::ND_INS_FNSTSW => Ok(Mnemonic::Fnstsw),
            ffi::_ND_INS_CLASS::ND_INS_FPATAN => Ok(Mnemonic::Fpatan),
            ffi::_ND_INS_CLASS::ND_INS_FPREM => Ok(Mnemonic::Fprem),
            ffi::_ND_INS_CLASS::ND_INS_FPREM1 => Ok(Mnemonic::Fprem1),
            ffi::_ND_INS_CLASS::ND_INS_FPTAN => Ok(Mnemonic::Fptan),
            ffi::_ND_INS_CLASS::ND_INS_FRINEAR => Ok(Mnemonic::Frinear),
            ffi::_ND_INS_CLASS::ND_INS_FRNDINT => Ok(Mnemonic::Frndint),
            ffi::_ND_INS_CLASS::ND_INS_FRSTOR => Ok(Mnemonic::Frstor),
            ffi::_ND_INS_CLASS::ND_INS_FSCALE => Ok(Mnemonic::Fscale),
            ffi::_ND_INS_CLASS::ND_INS_FSIN => Ok(Mnemonic::Fsin),
            ffi::_ND_INS_CLASS::ND_INS_FSINCOS => Ok(Mnemonic::Fsincos),
            ffi::_ND_INS_CLASS::ND_INS_FSQRT => Ok(Mnemonic::Fsqrt),
            ffi::_ND_INS_CLASS::ND_INS_FST => Ok(Mnemonic::Fst),
            ffi::_ND_INS_CLASS::ND_INS_FSTDW => Ok(Mnemonic::Fstdw),
            ffi::_ND_INS_CLASS::ND_INS_FSTP => Ok(Mnemonic::Fstp),
            ffi::_ND_INS_CLASS::ND_INS_FSTPNCE => Ok(Mnemonic::Fstpnce),
            ffi::_ND_INS_CLASS::ND_INS_FSTSG => Ok(Mnemonic::Fstsg),
            ffi::_ND_INS_CLASS::ND_INS_FSUB => Ok(Mnemonic::Fsub),
            ffi::_ND_INS_CLASS::ND_INS_FSUBP => Ok(Mnemonic::Fsubp),
            ffi::_ND_INS_CLASS::ND_INS_FSUBR => Ok(Mnemonic::Fsubr),
            ffi::_ND_INS_CLASS::ND_INS_FSUBRP => Ok(Mnemonic::Fsubrp),
            ffi::_ND_INS_CLASS::ND_INS_FTST => Ok(Mnemonic::Ftst),
            ffi::_ND_INS_CLASS::ND_INS_FUCOM => Ok(Mnemonic::Fucom),
            ffi::_ND_INS_CLASS::ND_INS_FUCOMI => Ok(Mnemonic::Fucomi),
            ffi::_ND_INS_CLASS::ND_INS_FUCOMIP => Ok(Mnemonic::Fucomip),
            ffi::_ND_INS_CLASS::ND_INS_FUCOMP => Ok(Mnemonic::Fucomp),
            ffi::_ND_INS_CLASS::ND_INS_FUCOMPP => Ok(Mnemonic::Fucompp),
            ffi::_ND_INS_CLASS::ND_INS_FXAM => Ok(Mnemonic::Fxam),
            ffi::_ND_INS_CLASS::ND_INS_FXCH => Ok(Mnemonic::Fxch),
            ffi::_ND_INS_CLASS::ND_INS_FXRSTOR => Ok(Mnemonic::Fxrstor),
            ffi::_ND_INS_CLASS::ND_INS_FXRSTOR64 => Ok(Mnemonic::Fxrstor64),
            ffi::_ND_INS_CLASS::ND_INS_FXSAVE => Ok(Mnemonic::Fxsave),
            ffi::_ND_INS_CLASS::ND_INS_FXSAVE64 => Ok(Mnemonic::Fxsave64),
            ffi::_ND_INS_CLASS::ND_INS_FXTRACT => Ok(Mnemonic::Fxtract),
            ffi::_ND_INS_CLASS::ND_INS_FYL2X => Ok(Mnemonic::Fyl2x),
            ffi::_ND_INS_CLASS::ND_INS_FYL2XP1 => Ok(Mnemonic::Fyl2xp1),
            ffi::_ND_INS_CLASS::ND_INS_GETSEC => Ok(Mnemonic::Getsec),
            ffi::_ND_INS_CLASS::ND_INS_GF2P8AFFINEINVQB => Ok(Mnemonic::Gf2p8affineinvqb),
            ffi::_ND_INS_CLASS::ND_INS_GF2P8AFFINEQB => Ok(Mnemonic::Gf2p8affineqb),
            ffi::_ND_INS_CLASS::ND_INS_GF2P8MULB => Ok(Mnemonic::Gf2p8mulb),
            ffi::_ND_INS_CLASS::ND_INS_HADDPD => Ok(Mnemonic::Haddpd),
            ffi::_ND_INS_CLASS::ND_INS_HADDPS => Ok(Mnemonic::Haddps),
            ffi::_ND_INS_CLASS::ND_INS_HLT => Ok(Mnemonic::Hlt),
            ffi::_ND_INS_CLASS::ND_INS_HRESET => Ok(Mnemonic::Hreset),
            ffi::_ND_INS_CLASS::ND_INS_HSUBPD => Ok(Mnemonic::Hsubpd),
            ffi::_ND_INS_CLASS::ND_INS_HSUBPS => Ok(Mnemonic::Hsubps),
            ffi::_ND_INS_CLASS::ND_INS_IDIV => Ok(Mnemonic::Idiv),
            ffi::_ND_INS_CLASS::ND_INS_IMUL => Ok(Mnemonic::Imul),
            ffi::_ND_INS_CLASS::ND_INS_IN => Ok(Mnemonic::In),
            ffi::_ND_INS_CLASS::ND_INS_INC => Ok(Mnemonic::Inc),
            ffi::_ND_INS_CLASS::ND_INS_INCSSP => Ok(Mnemonic::Incssp),
            ffi::_ND_INS_CLASS::ND_INS_INS => Ok(Mnemonic::Ins),
            ffi::_ND_INS_CLASS::ND_INS_INSERTPS => Ok(Mnemonic::Insertps),
            ffi::_ND_INS_CLASS::ND_INS_INSERTQ => Ok(Mnemonic::Insertq),
            ffi::_ND_INS_CLASS::ND_INS_INT => Ok(Mnemonic::Int),
            ffi::_ND_INS_CLASS::ND_INS_INT1 => Ok(Mnemonic::Int1),
            ffi::_ND_INS_CLASS::ND_INS_INT3 => Ok(Mnemonic::Int3),
            ffi::_ND_INS_CLASS::ND_INS_INTO => Ok(Mnemonic::Into),
            ffi::_ND_INS_CLASS::ND_INS_INVD => Ok(Mnemonic::Invd),
            ffi::_ND_INS_CLASS::ND_INS_INVEPT => Ok(Mnemonic::Invept),
            ffi::_ND_INS_CLASS::ND_INS_INVLPG => Ok(Mnemonic::Invlpg),
            ffi::_ND_INS_CLASS::ND_INS_INVLPGA => Ok(Mnemonic::Invlpga),
            ffi::_ND_INS_CLASS::ND_INS_INVLPGB => Ok(Mnemonic::Invlpgb),
            ffi::_ND_INS_CLASS::ND_INS_INVPCID => Ok(Mnemonic::Invpcid),
            ffi::_ND_INS_CLASS::ND_INS_INVVPID => Ok(Mnemonic::Invvpid),
            ffi::_ND_INS_CLASS::ND_INS_IRET => Ok(Mnemonic::Iret),
            ffi::_ND_INS_CLASS::ND_INS_JMPE => Ok(Mnemonic::Jmpe),
            ffi::_ND_INS_CLASS::ND_INS_JMPFD => Ok(Mnemonic::Jmpfd),
            ffi::_ND_INS_CLASS::ND_INS_JMPFI => Ok(Mnemonic::Jmpfi),
            ffi::_ND_INS_CLASS::ND_INS_JMPNI => Ok(Mnemonic::Jmpni),
            ffi::_ND_INS_CLASS::ND_INS_JMPNR => Ok(Mnemonic::Jmpnr),
            ffi::_ND_INS_CLASS::ND_INS_Jcc => Ok(Mnemonic::Jcc),
            ffi::_ND_INS_CLASS::ND_INS_JrCXZ => Ok(Mnemonic::Jrcxz),
            ffi::_ND_INS_CLASS::ND_INS_KADD => Ok(Mnemonic::Kadd),
            ffi::_ND_INS_CLASS::ND_INS_KAND => Ok(Mnemonic::Kand),
            ffi::_ND_INS_CLASS::ND_INS_KANDN => Ok(Mnemonic::Kandn),
            ffi::_ND_INS_CLASS::ND_INS_KMERGE2L1H => Ok(Mnemonic::Kmerge2l1h),
            ffi::_ND_INS_CLASS::ND_INS_KMERGE2L1L => Ok(Mnemonic::Kmerge2l1l),
            ffi::_ND_INS_CLASS::ND_INS_KMOV => Ok(Mnemonic::Kmov),
            ffi::_ND_INS_CLASS::ND_INS_KNOT => Ok(Mnemonic::Knot),
            ffi::_ND_INS_CLASS::ND_INS_KOR => Ok(Mnemonic::Kor),
            ffi::_ND_INS_CLASS::ND_INS_KORTEST => Ok(Mnemonic::Kortest),
            ffi::_ND_INS_CLASS::ND_INS_KSHIFTL => Ok(Mnemonic::Kshiftl),
            ffi::_ND_INS_CLASS::ND_INS_KSHIFTR => Ok(Mnemonic::Kshiftr),
            ffi::_ND_INS_CLASS::ND_INS_KTEST => Ok(Mnemonic::Ktest),
            ffi::_ND_INS_CLASS::ND_INS_KUNPCKBW => Ok(Mnemonic::Kunpckbw),
            ffi::_ND_INS_CLASS::ND_INS_KUNPCKDQ => Ok(Mnemonic::Kunpckdq),
            ffi::_ND_INS_CLASS::ND_INS_KUNPCKWD => Ok(Mnemonic::Kunpckwd),
            ffi::_ND_INS_CLASS::ND_INS_KXNOR => Ok(Mnemonic::Kxnor),
            ffi::_ND_INS_CLASS::ND_INS_KXOR => Ok(Mnemonic::Kxor),
            ffi::_ND_INS_CLASS::ND_INS_LAHF => Ok(Mnemonic::Lahf),
            ffi::_ND_INS_CLASS::ND_INS_LAR => Ok(Mnemonic::Lar),
            ffi::_ND_INS_CLASS::ND_INS_LDDQU => Ok(Mnemonic::Lddqu),
            ffi::_ND_INS_CLASS::ND_INS_LDMXCSR => Ok(Mnemonic::Ldmxcsr),
            ffi::_ND_INS_CLASS::ND_INS_LDS => Ok(Mnemonic::Lds),
            ffi::_ND_INS_CLASS::ND_INS_LDTILECFG => Ok(Mnemonic::Ldtilecfg),
            ffi::_ND_INS_CLASS::ND_INS_LEA => Ok(Mnemonic::Lea),
            ffi::_ND_INS_CLASS::ND_INS_LEAVE => Ok(Mnemonic::Leave),
            ffi::_ND_INS_CLASS::ND_INS_LES => Ok(Mnemonic::Les),
            ffi::_ND_INS_CLASS::ND_INS_LFENCE => Ok(Mnemonic::Lfence),
            ffi::_ND_INS_CLASS::ND_INS_LFS => Ok(Mnemonic::Lfs),
            ffi::_ND_INS_CLASS::ND_INS_LGDT => Ok(Mnemonic::Lgdt),
            ffi::_ND_INS_CLASS::ND_INS_LGS => Ok(Mnemonic::Lgs),
            ffi::_ND_INS_CLASS::ND_INS_LIDT => Ok(Mnemonic::Lidt),
            ffi::_ND_INS_CLASS::ND_INS_LKGS => Ok(Mnemonic::Lkgs),
            ffi::_ND_INS_CLASS::ND_INS_LLDT => Ok(Mnemonic::Lldt),
            ffi::_ND_INS_CLASS::ND_INS_LLWPCB => Ok(Mnemonic::Llwpcb),
            ffi::_ND_INS_CLASS::ND_INS_LMSW => Ok(Mnemonic::Lmsw),
            ffi::_ND_INS_CLASS::ND_INS_LOADIWKEY => Ok(Mnemonic::Loadiwkey),
            ffi::_ND_INS_CLASS::ND_INS_LODS => Ok(Mnemonic::Lods),
            ffi::_ND_INS_CLASS::ND_INS_LOOP => Ok(Mnemonic::Loop),
            ffi::_ND_INS_CLASS::ND_INS_LOOPNZ => Ok(Mnemonic::Loopnz),
            ffi::_ND_INS_CLASS::ND_INS_LOOPZ => Ok(Mnemonic::Loopz),
            ffi::_ND_INS_CLASS::ND_INS_LSL => Ok(Mnemonic::Lsl),
            ffi::_ND_INS_CLASS::ND_INS_LSS => Ok(Mnemonic::Lss),
            ffi::_ND_INS_CLASS::ND_INS_LTR => Ok(Mnemonic::Ltr),
            ffi::_ND_INS_CLASS::ND_INS_LWPINS => Ok(Mnemonic::Lwpins),
            ffi::_ND_INS_CLASS::ND_INS_LWPVAL => Ok(Mnemonic::Lwpval),
            ffi::_ND_INS_CLASS::ND_INS_LZCNT => Ok(Mnemonic::Lzcnt),
            ffi::_ND_INS_CLASS::ND_INS_MASKMOVDQU => Ok(Mnemonic::Maskmovdqu),
            ffi::_ND_INS_CLASS::ND_INS_MASKMOVQ => Ok(Mnemonic::Maskmovq),
            ffi::_ND_INS_CLASS::ND_INS_MAXPD => Ok(Mnemonic::Maxpd),
            ffi::_ND_INS_CLASS::ND_INS_MAXPS => Ok(Mnemonic::Maxps),
            ffi::_ND_INS_CLASS::ND_INS_MAXSD => Ok(Mnemonic::Maxsd),
            ffi::_ND_INS_CLASS::ND_INS_MAXSS => Ok(Mnemonic::Maxss),
            ffi::_ND_INS_CLASS::ND_INS_MCOMMIT => Ok(Mnemonic::Mcommit),
            ffi::_ND_INS_CLASS::ND_INS_MFENCE => Ok(Mnemonic::Mfence),
            ffi::_ND_INS_CLASS::ND_INS_MINPD => Ok(Mnemonic::Minpd),
            ffi::_ND_INS_CLASS::ND_INS_MINPS => Ok(Mnemonic::Minps),
            ffi::_ND_INS_CLASS::ND_INS_MINSD => Ok(Mnemonic::Minsd),
            ffi::_ND_INS_CLASS::ND_INS_MINSS => Ok(Mnemonic::Minss),
            ffi::_ND_INS_CLASS::ND_INS_MONITOR => Ok(Mnemonic::Monitor),
            ffi::_ND_INS_CLASS::ND_INS_MONITORX => Ok(Mnemonic::Monitorx),
            ffi::_ND_INS_CLASS::ND_INS_MONTMUL => Ok(Mnemonic::Montmul),
            ffi::_ND_INS_CLASS::ND_INS_MOV => Ok(Mnemonic::Mov),
            ffi::_ND_INS_CLASS::ND_INS_MOVAPD => Ok(Mnemonic::Movapd),
            ffi::_ND_INS_CLASS::ND_INS_MOVAPS => Ok(Mnemonic::Movaps),
            ffi::_ND_INS_CLASS::ND_INS_MOVBE => Ok(Mnemonic::Movbe),
            ffi::_ND_INS_CLASS::ND_INS_MOVD => Ok(Mnemonic::Movd),
            ffi::_ND_INS_CLASS::ND_INS_MOVDDUP => Ok(Mnemonic::Movddup),
            ffi::_ND_INS_CLASS::ND_INS_MOVDIR64B => Ok(Mnemonic::Movdir64b),
            ffi::_ND_INS_CLASS::ND_INS_MOVDIRI => Ok(Mnemonic::Movdiri),
            ffi::_ND_INS_CLASS::ND_INS_MOVDQ2Q => Ok(Mnemonic::Movdq2q),
            ffi::_ND_INS_CLASS::ND_INS_MOVDQA => Ok(Mnemonic::Movdqa),
            ffi::_ND_INS_CLASS::ND_INS_MOVDQU => Ok(Mnemonic::Movdqu),
            ffi::_ND_INS_CLASS::ND_INS_MOVHLPS => Ok(Mnemonic::Movhlps),
            ffi::_ND_INS_CLASS::ND_INS_MOVHPD => Ok(Mnemonic::Movhpd),
            ffi::_ND_INS_CLASS::ND_INS_MOVHPS => Ok(Mnemonic::Movhps),
            ffi::_ND_INS_CLASS::ND_INS_MOVLHPS => Ok(Mnemonic::Movlhps),
            ffi::_ND_INS_CLASS::ND_INS_MOVLPD => Ok(Mnemonic::Movlpd),
            ffi::_ND_INS_CLASS::ND_INS_MOVLPS => Ok(Mnemonic::Movlps),
            ffi::_ND_INS_CLASS::ND_INS_MOVMSKPD => Ok(Mnemonic::Movmskpd),
            ffi::_ND_INS_CLASS::ND_INS_MOVMSKPS => Ok(Mnemonic::Movmskps),
            ffi::_ND_INS_CLASS::ND_INS_MOVNTDQ => Ok(Mnemonic::Movntdq),
            ffi::_ND_INS_CLASS::ND_INS_MOVNTDQA => Ok(Mnemonic::Movntdqa),
            ffi::_ND_INS_CLASS::ND_INS_MOVNTI => Ok(Mnemonic::Movnti),
            ffi::_ND_INS_CLASS::ND_INS_MOVNTPD => Ok(Mnemonic::Movntpd),
            ffi::_ND_INS_CLASS::ND_INS_MOVNTPS => Ok(Mnemonic::Movntps),
            ffi::_ND_INS_CLASS::ND_INS_MOVNTQ => Ok(Mnemonic::Movntq),
            ffi::_ND_INS_CLASS::ND_INS_MOVNTSD => Ok(Mnemonic::Movntsd),
            ffi::_ND_INS_CLASS::ND_INS_MOVNTSS => Ok(Mnemonic::Movntss),
            ffi::_ND_INS_CLASS::ND_INS_MOVQ => Ok(Mnemonic::Movq),
            ffi::_ND_INS_CLASS::ND_INS_MOVQ2DQ => Ok(Mnemonic::Movq2dq),
            ffi::_ND_INS_CLASS::ND_INS_MOVS => Ok(Mnemonic::Movs),
            ffi::_ND_INS_CLASS::ND_INS_MOVSD => Ok(Mnemonic::Movsd),
            ffi::_ND_INS_CLASS::ND_INS_MOVSHDUP => Ok(Mnemonic::Movshdup),
            ffi::_ND_INS_CLASS::ND_INS_MOVSLDUP => Ok(Mnemonic::Movsldup),
            ffi::_ND_INS_CLASS::ND_INS_MOVSS => Ok(Mnemonic::Movss),
            ffi::_ND_INS_CLASS::ND_INS_MOVSX => Ok(Mnemonic::Movsx),
            ffi::_ND_INS_CLASS::ND_INS_MOVSXD => Ok(Mnemonic::Movsxd),
            ffi::_ND_INS_CLASS::ND_INS_MOVUPD => Ok(Mnemonic::Movupd),
            ffi::_ND_INS_CLASS::ND_INS_MOVUPS => Ok(Mnemonic::Movups),
            ffi::_ND_INS_CLASS::ND_INS_MOVZX => Ok(Mnemonic::Movzx),
            ffi::_ND_INS_CLASS::ND_INS_MOV_CR => Ok(Mnemonic::MovCr),
            ffi::_ND_INS_CLASS::ND_INS_MOV_DR => Ok(Mnemonic::MovDr),
            ffi::_ND_INS_CLASS::ND_INS_MOV_TR => Ok(Mnemonic::MovTr),
            ffi::_ND_INS_CLASS::ND_INS_MPSADBW => Ok(Mnemonic::Mpsadbw),
            ffi::_ND_INS_CLASS::ND_INS_MUL => Ok(Mnemonic::Mul),
            ffi::_ND_INS_CLASS::ND_INS_MULPD => Ok(Mnemonic::Mulpd),
            ffi::_ND_INS_CLASS::ND_INS_MULPS => Ok(Mnemonic::Mulps),
            ffi::_ND_INS_CLASS::ND_INS_MULSD => Ok(Mnemonic::Mulsd),
            ffi::_ND_INS_CLASS::ND_INS_MULSS => Ok(Mnemonic::Mulss),
            ffi::_ND_INS_CLASS::ND_INS_MULX => Ok(Mnemonic::Mulx),
            ffi::_ND_INS_CLASS::ND_INS_MWAIT => Ok(Mnemonic::Mwait),
            ffi::_ND_INS_CLASS::ND_INS_MWAITX => Ok(Mnemonic::Mwaitx),
            ffi::_ND_INS_CLASS::ND_INS_NEG => Ok(Mnemonic::Neg),
            ffi::_ND_INS_CLASS::ND_INS_NOP => Ok(Mnemonic::Nop),
            ffi::_ND_INS_CLASS::ND_INS_NOT => Ok(Mnemonic::Not),
            ffi::_ND_INS_CLASS::ND_INS_OR => Ok(Mnemonic::Or),
            ffi::_ND_INS_CLASS::ND_INS_ORPD => Ok(Mnemonic::Orpd),
            ffi::_ND_INS_CLASS::ND_INS_ORPS => Ok(Mnemonic::Orps),
            ffi::_ND_INS_CLASS::ND_INS_OUT => Ok(Mnemonic::Out),
            ffi::_ND_INS_CLASS::ND_INS_OUTS => Ok(Mnemonic::Outs),
            ffi::_ND_INS_CLASS::ND_INS_PABSB => Ok(Mnemonic::Pabsb),
            ffi::_ND_INS_CLASS::ND_INS_PABSD => Ok(Mnemonic::Pabsd),
            ffi::_ND_INS_CLASS::ND_INS_PABSW => Ok(Mnemonic::Pabsw),
            ffi::_ND_INS_CLASS::ND_INS_PACKSSDW => Ok(Mnemonic::Packssdw),
            ffi::_ND_INS_CLASS::ND_INS_PACKSSWB => Ok(Mnemonic::Packsswb),
            ffi::_ND_INS_CLASS::ND_INS_PACKUSDW => Ok(Mnemonic::Packusdw),
            ffi::_ND_INS_CLASS::ND_INS_PACKUSWB => Ok(Mnemonic::Packuswb),
            ffi::_ND_INS_CLASS::ND_INS_PADDB => Ok(Mnemonic::Paddb),
            ffi::_ND_INS_CLASS::ND_INS_PADDD => Ok(Mnemonic::Paddd),
            ffi::_ND_INS_CLASS::ND_INS_PADDQ => Ok(Mnemonic::Paddq),
            ffi::_ND_INS_CLASS::ND_INS_PADDSB => Ok(Mnemonic::Paddsb),
            ffi::_ND_INS_CLASS::ND_INS_PADDSW => Ok(Mnemonic::Paddsw),
            ffi::_ND_INS_CLASS::ND_INS_PADDUSB => Ok(Mnemonic::Paddusb),
            ffi::_ND_INS_CLASS::ND_INS_PADDUSW => Ok(Mnemonic::Paddusw),
            ffi::_ND_INS_CLASS::ND_INS_PADDW => Ok(Mnemonic::Paddw),
            ffi::_ND_INS_CLASS::ND_INS_PALIGNR => Ok(Mnemonic::Palignr),
            ffi::_ND_INS_CLASS::ND_INS_PAND => Ok(Mnemonic::Pand),
            ffi::_ND_INS_CLASS::ND_INS_PANDN => Ok(Mnemonic::Pandn),
            ffi::_ND_INS_CLASS::ND_INS_PAUSE => Ok(Mnemonic::Pause),
            ffi::_ND_INS_CLASS::ND_INS_PAVGB => Ok(Mnemonic::Pavgb),
            ffi::_ND_INS_CLASS::ND_INS_PAVGUSB => Ok(Mnemonic::Pavgusb),
            ffi::_ND_INS_CLASS::ND_INS_PAVGW => Ok(Mnemonic::Pavgw),
            ffi::_ND_INS_CLASS::ND_INS_PBLENDVB => Ok(Mnemonic::Pblendvb),
            ffi::_ND_INS_CLASS::ND_INS_PBLENDW => Ok(Mnemonic::Pblendw),
            ffi::_ND_INS_CLASS::ND_INS_PCLMULQDQ => Ok(Mnemonic::Pclmulqdq),
            ffi::_ND_INS_CLASS::ND_INS_PCMPEQB => Ok(Mnemonic::Pcmpeqb),
            ffi::_ND_INS_CLASS::ND_INS_PCMPEQD => Ok(Mnemonic::Pcmpeqd),
            ffi::_ND_INS_CLASS::ND_INS_PCMPEQQ => Ok(Mnemonic::Pcmpeqq),
            ffi::_ND_INS_CLASS::ND_INS_PCMPEQW => Ok(Mnemonic::Pcmpeqw),
            ffi::_ND_INS_CLASS::ND_INS_PCMPESTRI => Ok(Mnemonic::Pcmpestri),
            ffi::_ND_INS_CLASS::ND_INS_PCMPESTRM => Ok(Mnemonic::Pcmpestrm),
            ffi::_ND_INS_CLASS::ND_INS_PCMPGTB => Ok(Mnemonic::Pcmpgtb),
            ffi::_ND_INS_CLASS::ND_INS_PCMPGTD => Ok(Mnemonic::Pcmpgtd),
            ffi::_ND_INS_CLASS::ND_INS_PCMPGTQ => Ok(Mnemonic::Pcmpgtq),
            ffi::_ND_INS_CLASS::ND_INS_PCMPGTW => Ok(Mnemonic::Pcmpgtw),
            ffi::_ND_INS_CLASS::ND_INS_PCMPISTRI => Ok(Mnemonic::Pcmpistri),
            ffi::_ND_INS_CLASS::ND_INS_PCMPISTRM => Ok(Mnemonic::Pcmpistrm),
            ffi::_ND_INS_CLASS::ND_INS_PCONFIG => Ok(Mnemonic::Pconfig),
            ffi::_ND_INS_CLASS::ND_INS_PDEP => Ok(Mnemonic::Pdep),
            ffi::_ND_INS_CLASS::ND_INS_PEXT => Ok(Mnemonic::Pext),
            ffi::_ND_INS_CLASS::ND_INS_PEXTRB => Ok(Mnemonic::Pextrb),
            ffi::_ND_INS_CLASS::ND_INS_PEXTRD => Ok(Mnemonic::Pextrd),
            ffi::_ND_INS_CLASS::ND_INS_PEXTRQ => Ok(Mnemonic::Pextrq),
            ffi::_ND_INS_CLASS::ND_INS_PEXTRW => Ok(Mnemonic::Pextrw),
            ffi::_ND_INS_CLASS::ND_INS_PF2ID => Ok(Mnemonic::Pf2id),
            ffi::_ND_INS_CLASS::ND_INS_PF2IW => Ok(Mnemonic::Pf2iw),
            ffi::_ND_INS_CLASS::ND_INS_PFACC => Ok(Mnemonic::Pfacc),
            ffi::_ND_INS_CLASS::ND_INS_PFADD => Ok(Mnemonic::Pfadd),
            ffi::_ND_INS_CLASS::ND_INS_PFCMPEQ => Ok(Mnemonic::Pfcmpeq),
            ffi::_ND_INS_CLASS::ND_INS_PFCMPGE => Ok(Mnemonic::Pfcmpge),
            ffi::_ND_INS_CLASS::ND_INS_PFCMPGT => Ok(Mnemonic::Pfcmpgt),
            ffi::_ND_INS_CLASS::ND_INS_PFMAX => Ok(Mnemonic::Pfmax),
            ffi::_ND_INS_CLASS::ND_INS_PFMIN => Ok(Mnemonic::Pfmin),
            ffi::_ND_INS_CLASS::ND_INS_PFMUL => Ok(Mnemonic::Pfmul),
            ffi::_ND_INS_CLASS::ND_INS_PFNACC => Ok(Mnemonic::Pfnacc),
            ffi::_ND_INS_CLASS::ND_INS_PFPNACC => Ok(Mnemonic::Pfpnacc),
            ffi::_ND_INS_CLASS::ND_INS_PFRCP => Ok(Mnemonic::Pfrcp),
            ffi::_ND_INS_CLASS::ND_INS_PFRCPIT1 => Ok(Mnemonic::Pfrcpit1),
            ffi::_ND_INS_CLASS::ND_INS_PFRCPIT2 => Ok(Mnemonic::Pfrcpit2),
            ffi::_ND_INS_CLASS::ND_INS_PFRCPV => Ok(Mnemonic::Pfrcpv),
            ffi::_ND_INS_CLASS::ND_INS_PFRSQIT1 => Ok(Mnemonic::Pfrsqit1),
            ffi::_ND_INS_CLASS::ND_INS_PFRSQRT => Ok(Mnemonic::Pfrsqrt),
            ffi::_ND_INS_CLASS::ND_INS_PFRSQRTV => Ok(Mnemonic::Pfrsqrtv),
            ffi::_ND_INS_CLASS::ND_INS_PFSUB => Ok(Mnemonic::Pfsub),
            ffi::_ND_INS_CLASS::ND_INS_PFSUBR => Ok(Mnemonic::Pfsubr),
            ffi::_ND_INS_CLASS::ND_INS_PHADDD => Ok(Mnemonic::Phaddd),
            ffi::_ND_INS_CLASS::ND_INS_PHADDSW => Ok(Mnemonic::Phaddsw),
            ffi::_ND_INS_CLASS::ND_INS_PHADDW => Ok(Mnemonic::Phaddw),
            ffi::_ND_INS_CLASS::ND_INS_PHMINPOSUW => Ok(Mnemonic::Phminposuw),
            ffi::_ND_INS_CLASS::ND_INS_PHSUBD => Ok(Mnemonic::Phsubd),
            ffi::_ND_INS_CLASS::ND_INS_PHSUBSW => Ok(Mnemonic::Phsubsw),
            ffi::_ND_INS_CLASS::ND_INS_PHSUBW => Ok(Mnemonic::Phsubw),
            ffi::_ND_INS_CLASS::ND_INS_PI2FD => Ok(Mnemonic::Pi2fd),
            ffi::_ND_INS_CLASS::ND_INS_PI2FW => Ok(Mnemonic::Pi2fw),
            ffi::_ND_INS_CLASS::ND_INS_PINSRB => Ok(Mnemonic::Pinsrb),
            ffi::_ND_INS_CLASS::ND_INS_PINSRD => Ok(Mnemonic::Pinsrd),
            ffi::_ND_INS_CLASS::ND_INS_PINSRQ => Ok(Mnemonic::Pinsrq),
            ffi::_ND_INS_CLASS::ND_INS_PINSRW => Ok(Mnemonic::Pinsrw),
            ffi::_ND_INS_CLASS::ND_INS_PMADDUBSW => Ok(Mnemonic::Pmaddubsw),
            ffi::_ND_INS_CLASS::ND_INS_PMADDWD => Ok(Mnemonic::Pmaddwd),
            ffi::_ND_INS_CLASS::ND_INS_PMAXSB => Ok(Mnemonic::Pmaxsb),
            ffi::_ND_INS_CLASS::ND_INS_PMAXSD => Ok(Mnemonic::Pmaxsd),
            ffi::_ND_INS_CLASS::ND_INS_PMAXSW => Ok(Mnemonic::Pmaxsw),
            ffi::_ND_INS_CLASS::ND_INS_PMAXUB => Ok(Mnemonic::Pmaxub),
            ffi::_ND_INS_CLASS::ND_INS_PMAXUD => Ok(Mnemonic::Pmaxud),
            ffi::_ND_INS_CLASS::ND_INS_PMAXUW => Ok(Mnemonic::Pmaxuw),
            ffi::_ND_INS_CLASS::ND_INS_PMINSB => Ok(Mnemonic::Pminsb),
            ffi::_ND_INS_CLASS::ND_INS_PMINSD => Ok(Mnemonic::Pminsd),
            ffi::_ND_INS_CLASS::ND_INS_PMINSW => Ok(Mnemonic::Pminsw),
            ffi::_ND_INS_CLASS::ND_INS_PMINUB => Ok(Mnemonic::Pminub),
            ffi::_ND_INS_CLASS::ND_INS_PMINUD => Ok(Mnemonic::Pminud),
            ffi::_ND_INS_CLASS::ND_INS_PMINUW => Ok(Mnemonic::Pminuw),
            ffi::_ND_INS_CLASS::ND_INS_PMOVMSKB => Ok(Mnemonic::Pmovmskb),
            ffi::_ND_INS_CLASS::ND_INS_PMOVSXBD => Ok(Mnemonic::Pmovsxbd),
            ffi::_ND_INS_CLASS::ND_INS_PMOVSXBQ => Ok(Mnemonic::Pmovsxbq),
            ffi::_ND_INS_CLASS::ND_INS_PMOVSXBW => Ok(Mnemonic::Pmovsxbw),
            ffi::_ND_INS_CLASS::ND_INS_PMOVSXDQ => Ok(Mnemonic::Pmovsxdq),
            ffi::_ND_INS_CLASS::ND_INS_PMOVSXWD => Ok(Mnemonic::Pmovsxwd),
            ffi::_ND_INS_CLASS::ND_INS_PMOVSXWQ => Ok(Mnemonic::Pmovsxwq),
            ffi::_ND_INS_CLASS::ND_INS_PMOVZXBD => Ok(Mnemonic::Pmovzxbd),
            ffi::_ND_INS_CLASS::ND_INS_PMOVZXBQ => Ok(Mnemonic::Pmovzxbq),
            ffi::_ND_INS_CLASS::ND_INS_PMOVZXBW => Ok(Mnemonic::Pmovzxbw),
            ffi::_ND_INS_CLASS::ND_INS_PMOVZXDQ => Ok(Mnemonic::Pmovzxdq),
            ffi::_ND_INS_CLASS::ND_INS_PMOVZXWD => Ok(Mnemonic::Pmovzxwd),
            ffi::_ND_INS_CLASS::ND_INS_PMOVZXWQ => Ok(Mnemonic::Pmovzxwq),
            ffi::_ND_INS_CLASS::ND_INS_PMULDQ => Ok(Mnemonic::Pmuldq),
            ffi::_ND_INS_CLASS::ND_INS_PMULHRSW => Ok(Mnemonic::Pmulhrsw),
            ffi::_ND_INS_CLASS::ND_INS_PMULHRW => Ok(Mnemonic::Pmulhrw),
            ffi::_ND_INS_CLASS::ND_INS_PMULHUW => Ok(Mnemonic::Pmulhuw),
            ffi::_ND_INS_CLASS::ND_INS_PMULHW => Ok(Mnemonic::Pmulhw),
            ffi::_ND_INS_CLASS::ND_INS_PMULLD => Ok(Mnemonic::Pmulld),
            ffi::_ND_INS_CLASS::ND_INS_PMULLW => Ok(Mnemonic::Pmullw),
            ffi::_ND_INS_CLASS::ND_INS_PMULUDQ => Ok(Mnemonic::Pmuludq),
            ffi::_ND_INS_CLASS::ND_INS_POP => Ok(Mnemonic::Pop),
            ffi::_ND_INS_CLASS::ND_INS_POPA => Ok(Mnemonic::Popa),
            ffi::_ND_INS_CLASS::ND_INS_POPAD => Ok(Mnemonic::Popad),
            ffi::_ND_INS_CLASS::ND_INS_POPCNT => Ok(Mnemonic::Popcnt),
            ffi::_ND_INS_CLASS::ND_INS_POPF => Ok(Mnemonic::Popf),
            ffi::_ND_INS_CLASS::ND_INS_POR => Ok(Mnemonic::Por),
            ffi::_ND_INS_CLASS::ND_INS_PREFETCH => Ok(Mnemonic::Prefetch),
            ffi::_ND_INS_CLASS::ND_INS_PREFETCHE => Ok(Mnemonic::Prefetche),
            ffi::_ND_INS_CLASS::ND_INS_PREFETCHIT0 => Ok(Mnemonic::Prefetchit0),
            ffi::_ND_INS_CLASS::ND_INS_PREFETCHIT1 => Ok(Mnemonic::Prefetchit1),
            ffi::_ND_INS_CLASS::ND_INS_PREFETCHM => Ok(Mnemonic::Prefetchm),
            ffi::_ND_INS_CLASS::ND_INS_PREFETCHNTA => Ok(Mnemonic::Prefetchnta),
            ffi::_ND_INS_CLASS::ND_INS_PREFETCHT0 => Ok(Mnemonic::Prefetcht0),
            ffi::_ND_INS_CLASS::ND_INS_PREFETCHT1 => Ok(Mnemonic::Prefetcht1),
            ffi::_ND_INS_CLASS::ND_INS_PREFETCHT2 => Ok(Mnemonic::Prefetcht2),
            ffi::_ND_INS_CLASS::ND_INS_PREFETCHW => Ok(Mnemonic::Prefetchw),
            ffi::_ND_INS_CLASS::ND_INS_PREFETCHWT1 => Ok(Mnemonic::Prefetchwt1),
            ffi::_ND_INS_CLASS::ND_INS_PSADBW => Ok(Mnemonic::Psadbw),
            ffi::_ND_INS_CLASS::ND_INS_PSHUFB => Ok(Mnemonic::Pshufb),
            ffi::_ND_INS_CLASS::ND_INS_PSHUFD => Ok(Mnemonic::Pshufd),
            ffi::_ND_INS_CLASS::ND_INS_PSHUFHW => Ok(Mnemonic::Pshufhw),
            ffi::_ND_INS_CLASS::ND_INS_PSHUFLW => Ok(Mnemonic::Pshuflw),
            ffi::_ND_INS_CLASS::ND_INS_PSHUFW => Ok(Mnemonic::Pshufw),
            ffi::_ND_INS_CLASS::ND_INS_PSIGNB => Ok(Mnemonic::Psignb),
            ffi::_ND_INS_CLASS::ND_INS_PSIGND => Ok(Mnemonic::Psignd),
            ffi::_ND_INS_CLASS::ND_INS_PSIGNW => Ok(Mnemonic::Psignw),
            ffi::_ND_INS_CLASS::ND_INS_PSLLD => Ok(Mnemonic::Pslld),
            ffi::_ND_INS_CLASS::ND_INS_PSLLDQ => Ok(Mnemonic::Pslldq),
            ffi::_ND_INS_CLASS::ND_INS_PSLLQ => Ok(Mnemonic::Psllq),
            ffi::_ND_INS_CLASS::ND_INS_PSLLW => Ok(Mnemonic::Psllw),
            ffi::_ND_INS_CLASS::ND_INS_PSMASH => Ok(Mnemonic::Psmash),
            ffi::_ND_INS_CLASS::ND_INS_PSRAD => Ok(Mnemonic::Psrad),
            ffi::_ND_INS_CLASS::ND_INS_PSRAW => Ok(Mnemonic::Psraw),
            ffi::_ND_INS_CLASS::ND_INS_PSRLD => Ok(Mnemonic::Psrld),
            ffi::_ND_INS_CLASS::ND_INS_PSRLDQ => Ok(Mnemonic::Psrldq),
            ffi::_ND_INS_CLASS::ND_INS_PSRLQ => Ok(Mnemonic::Psrlq),
            ffi::_ND_INS_CLASS::ND_INS_PSRLW => Ok(Mnemonic::Psrlw),
            ffi::_ND_INS_CLASS::ND_INS_PSUBB => Ok(Mnemonic::Psubb),
            ffi::_ND_INS_CLASS::ND_INS_PSUBD => Ok(Mnemonic::Psubd),
            ffi::_ND_INS_CLASS::ND_INS_PSUBQ => Ok(Mnemonic::Psubq),
            ffi::_ND_INS_CLASS::ND_INS_PSUBSB => Ok(Mnemonic::Psubsb),
            ffi::_ND_INS_CLASS::ND_INS_PSUBSW => Ok(Mnemonic::Psubsw),
            ffi::_ND_INS_CLASS::ND_INS_PSUBUSB => Ok(Mnemonic::Psubusb),
            ffi::_ND_INS_CLASS::ND_INS_PSUBUSW => Ok(Mnemonic::Psubusw),
            ffi::_ND_INS_CLASS::ND_INS_PSUBW => Ok(Mnemonic::Psubw),
            ffi::_ND_INS_CLASS::ND_INS_PSWAPD => Ok(Mnemonic::Pswapd),
            ffi::_ND_INS_CLASS::ND_INS_PTEST => Ok(Mnemonic::Ptest),
            ffi::_ND_INS_CLASS::ND_INS_PTWRITE => Ok(Mnemonic::Ptwrite),
            ffi::_ND_INS_CLASS::ND_INS_PUNPCKHBW => Ok(Mnemonic::Punpckhbw),
            ffi::_ND_INS_CLASS::ND_INS_PUNPCKHDQ => Ok(Mnemonic::Punpckhdq),
            ffi::_ND_INS_CLASS::ND_INS_PUNPCKHQDQ => Ok(Mnemonic::Punpckhqdq),
            ffi::_ND_INS_CLASS::ND_INS_PUNPCKHWD => Ok(Mnemonic::Punpckhwd),
            ffi::_ND_INS_CLASS::ND_INS_PUNPCKLBW => Ok(Mnemonic::Punpcklbw),
            ffi::_ND_INS_CLASS::ND_INS_PUNPCKLDQ => Ok(Mnemonic::Punpckldq),
            ffi::_ND_INS_CLASS::ND_INS_PUNPCKLQDQ => Ok(Mnemonic::Punpcklqdq),
            ffi::_ND_INS_CLASS::ND_INS_PUNPCKLWD => Ok(Mnemonic::Punpcklwd),
            ffi::_ND_INS_CLASS::ND_INS_PUSH => Ok(Mnemonic::Push),
            ffi::_ND_INS_CLASS::ND_INS_PUSHA => Ok(Mnemonic::Pusha),
            ffi::_ND_INS_CLASS::ND_INS_PUSHAD => Ok(Mnemonic::Pushad),
            ffi::_ND_INS_CLASS::ND_INS_PUSHF => Ok(Mnemonic::Pushf),
            ffi::_ND_INS_CLASS::ND_INS_PVALIDATE => Ok(Mnemonic::Pvalidate),
            ffi::_ND_INS_CLASS::ND_INS_PXOR => Ok(Mnemonic::Pxor),
            ffi::_ND_INS_CLASS::ND_INS_RCL => Ok(Mnemonic::Rcl),
            ffi::_ND_INS_CLASS::ND_INS_RCPPS => Ok(Mnemonic::Rcpps),
            ffi::_ND_INS_CLASS::ND_INS_RCPSS => Ok(Mnemonic::Rcpss),
            ffi::_ND_INS_CLASS::ND_INS_RCR => Ok(Mnemonic::Rcr),
            ffi::_ND_INS_CLASS::ND_INS_RDFSBASE => Ok(Mnemonic::Rdfsbase),
            ffi::_ND_INS_CLASS::ND_INS_RDGSBASE => Ok(Mnemonic::Rdgsbase),
            ffi::_ND_INS_CLASS::ND_INS_RDMSR => Ok(Mnemonic::Rdmsr),
            ffi::_ND_INS_CLASS::ND_INS_RDMSRLIST => Ok(Mnemonic::Rdmsrlist),
            ffi::_ND_INS_CLASS::ND_INS_RDPID => Ok(Mnemonic::Rdpid),
            ffi::_ND_INS_CLASS::ND_INS_RDPKRU => Ok(Mnemonic::Rdpkru),
            ffi::_ND_INS_CLASS::ND_INS_RDPMC => Ok(Mnemonic::Rdpmc),
            ffi::_ND_INS_CLASS::ND_INS_RDPRU => Ok(Mnemonic::Rdpru),
            ffi::_ND_INS_CLASS::ND_INS_RDRAND => Ok(Mnemonic::Rdrand),
            ffi::_ND_INS_CLASS::ND_INS_RDSEED => Ok(Mnemonic::Rdseed),
            ffi::_ND_INS_CLASS::ND_INS_RDSHR => Ok(Mnemonic::Rdshr),
            ffi::_ND_INS_CLASS::ND_INS_RDTSC => Ok(Mnemonic::Rdtsc),
            ffi::_ND_INS_CLASS::ND_INS_RDTSCP => Ok(Mnemonic::Rdtscp),
            ffi::_ND_INS_CLASS::ND_INS_RETF => Ok(Mnemonic::Retf),
            ffi::_ND_INS_CLASS::ND_INS_RETN => Ok(Mnemonic::Retn),
            ffi::_ND_INS_CLASS::ND_INS_RMPADJUST => Ok(Mnemonic::Rmpadjust),
            ffi::_ND_INS_CLASS::ND_INS_RMPQUERY => Ok(Mnemonic::Rmpquery),
            ffi::_ND_INS_CLASS::ND_INS_RMPUPDATE => Ok(Mnemonic::Rmpupdate),
            ffi::_ND_INS_CLASS::ND_INS_ROL => Ok(Mnemonic::Rol),
            ffi::_ND_INS_CLASS::ND_INS_ROR => Ok(Mnemonic::Ror),
            ffi::_ND_INS_CLASS::ND_INS_RORX => Ok(Mnemonic::Rorx),
            ffi::_ND_INS_CLASS::ND_INS_ROUNDPD => Ok(Mnemonic::Roundpd),
            ffi::_ND_INS_CLASS::ND_INS_ROUNDPS => Ok(Mnemonic::Roundps),
            ffi::_ND_INS_CLASS::ND_INS_ROUNDSD => Ok(Mnemonic::Roundsd),
            ffi::_ND_INS_CLASS::ND_INS_ROUNDSS => Ok(Mnemonic::Roundss),
            ffi::_ND_INS_CLASS::ND_INS_RSDC => Ok(Mnemonic::Rsdc),
            ffi::_ND_INS_CLASS::ND_INS_RSLDT => Ok(Mnemonic::Rsldt),
            ffi::_ND_INS_CLASS::ND_INS_RSM => Ok(Mnemonic::Rsm),
            ffi::_ND_INS_CLASS::ND_INS_RSQRTPS => Ok(Mnemonic::Rsqrtps),
            ffi::_ND_INS_CLASS::ND_INS_RSQRTSS => Ok(Mnemonic::Rsqrtss),
            ffi::_ND_INS_CLASS::ND_INS_RSSSP => Ok(Mnemonic::Rsssp),
            ffi::_ND_INS_CLASS::ND_INS_RSTORSSP => Ok(Mnemonic::Rstorssp),
            ffi::_ND_INS_CLASS::ND_INS_RSTS => Ok(Mnemonic::Rsts),
            ffi::_ND_INS_CLASS::ND_INS_SAHF => Ok(Mnemonic::Sahf),
            ffi::_ND_INS_CLASS::ND_INS_SAL => Ok(Mnemonic::Sal),
            ffi::_ND_INS_CLASS::ND_INS_SALC => Ok(Mnemonic::Salc),
            ffi::_ND_INS_CLASS::ND_INS_SAR => Ok(Mnemonic::Sar),
            ffi::_ND_INS_CLASS::ND_INS_SARX => Ok(Mnemonic::Sarx),
            ffi::_ND_INS_CLASS::ND_INS_SAVEPREVSSP => Ok(Mnemonic::Saveprevssp),
            ffi::_ND_INS_CLASS::ND_INS_SBB => Ok(Mnemonic::Sbb),
            ffi::_ND_INS_CLASS::ND_INS_SCAS => Ok(Mnemonic::Scas),
            ffi::_ND_INS_CLASS::ND_INS_SEAMCALL => Ok(Mnemonic::Seamcall),
            ffi::_ND_INS_CLASS::ND_INS_SEAMOPS => Ok(Mnemonic::Seamops),
            ffi::_ND_INS_CLASS::ND_INS_SEAMRET => Ok(Mnemonic::Seamret),
            ffi::_ND_INS_CLASS::ND_INS_SENDUIPI => Ok(Mnemonic::Senduipi),
            ffi::_ND_INS_CLASS::ND_INS_SERIALIZE => Ok(Mnemonic::Serialize),
            ffi::_ND_INS_CLASS::ND_INS_SETSSBSY => Ok(Mnemonic::Setssbsy),
            ffi::_ND_INS_CLASS::ND_INS_SETcc => Ok(Mnemonic::Setcc),
            ffi::_ND_INS_CLASS::ND_INS_SFENCE => Ok(Mnemonic::Sfence),
            ffi::_ND_INS_CLASS::ND_INS_SGDT => Ok(Mnemonic::Sgdt),
            ffi::_ND_INS_CLASS::ND_INS_SHA1MSG1 => Ok(Mnemonic::Sha1msg1),
            ffi::_ND_INS_CLASS::ND_INS_SHA1MSG2 => Ok(Mnemonic::Sha1msg2),
            ffi::_ND_INS_CLASS::ND_INS_SHA1NEXTE => Ok(Mnemonic::Sha1nexte),
            ffi::_ND_INS_CLASS::ND_INS_SHA1RNDS4 => Ok(Mnemonic::Sha1rnds4),
            ffi::_ND_INS_CLASS::ND_INS_SHA256MSG1 => Ok(Mnemonic::Sha256msg1),
            ffi::_ND_INS_CLASS::ND_INS_SHA256MSG2 => Ok(Mnemonic::Sha256msg2),
            ffi::_ND_INS_CLASS::ND_INS_SHA256RNDS2 => Ok(Mnemonic::Sha256rnds2),
            ffi::_ND_INS_CLASS::ND_INS_SHL => Ok(Mnemonic::Shl),
            ffi::_ND_INS_CLASS::ND_INS_SHLD => Ok(Mnemonic::Shld),
            ffi::_ND_INS_CLASS::ND_INS_SHLX => Ok(Mnemonic::Shlx),
            ffi::_ND_INS_CLASS::ND_INS_SHR => Ok(Mnemonic::Shr),
            ffi::_ND_INS_CLASS::ND_INS_SHRD => Ok(Mnemonic::Shrd),
            ffi::_ND_INS_CLASS::ND_INS_SHRX => Ok(Mnemonic::Shrx),
            ffi::_ND_INS_CLASS::ND_INS_SHUFPD => Ok(Mnemonic::Shufpd),
            ffi::_ND_INS_CLASS::ND_INS_SHUFPS => Ok(Mnemonic::Shufps),
            ffi::_ND_INS_CLASS::ND_INS_SIDT => Ok(Mnemonic::Sidt),
            ffi::_ND_INS_CLASS::ND_INS_SKINIT => Ok(Mnemonic::Skinit),
            ffi::_ND_INS_CLASS::ND_INS_SLDT => Ok(Mnemonic::Sldt),
            ffi::_ND_INS_CLASS::ND_INS_SLWPCB => Ok(Mnemonic::Slwpcb),
            ffi::_ND_INS_CLASS::ND_INS_SMINT => Ok(Mnemonic::Smint),
            ffi::_ND_INS_CLASS::ND_INS_SMSW => Ok(Mnemonic::Smsw),
            ffi::_ND_INS_CLASS::ND_INS_SPFLT => Ok(Mnemonic::Spflt),
            ffi::_ND_INS_CLASS::ND_INS_SQRTPD => Ok(Mnemonic::Sqrtpd),
            ffi::_ND_INS_CLASS::ND_INS_SQRTPS => Ok(Mnemonic::Sqrtps),
            ffi::_ND_INS_CLASS::ND_INS_SQRTSD => Ok(Mnemonic::Sqrtsd),
            ffi::_ND_INS_CLASS::ND_INS_SQRTSS => Ok(Mnemonic::Sqrtss),
            ffi::_ND_INS_CLASS::ND_INS_STAC => Ok(Mnemonic::Stac),
            ffi::_ND_INS_CLASS::ND_INS_STC => Ok(Mnemonic::Stc),
            ffi::_ND_INS_CLASS::ND_INS_STD => Ok(Mnemonic::Std),
            ffi::_ND_INS_CLASS::ND_INS_STGI => Ok(Mnemonic::Stgi),
            ffi::_ND_INS_CLASS::ND_INS_STI => Ok(Mnemonic::Sti),
            ffi::_ND_INS_CLASS::ND_INS_STMXCSR => Ok(Mnemonic::Stmxcsr),
            ffi::_ND_INS_CLASS::ND_INS_STOS => Ok(Mnemonic::Stos),
            ffi::_ND_INS_CLASS::ND_INS_STR => Ok(Mnemonic::Str),
            ffi::_ND_INS_CLASS::ND_INS_STTILECFG => Ok(Mnemonic::Sttilecfg),
            ffi::_ND_INS_CLASS::ND_INS_STUI => Ok(Mnemonic::Stui),
            ffi::_ND_INS_CLASS::ND_INS_SUB => Ok(Mnemonic::Sub),
            ffi::_ND_INS_CLASS::ND_INS_SUBPD => Ok(Mnemonic::Subpd),
            ffi::_ND_INS_CLASS::ND_INS_SUBPS => Ok(Mnemonic::Subps),
            ffi::_ND_INS_CLASS::ND_INS_SUBSD => Ok(Mnemonic::Subsd),
            ffi::_ND_INS_CLASS::ND_INS_SUBSS => Ok(Mnemonic::Subss),
            ffi::_ND_INS_CLASS::ND_INS_SVDC => Ok(Mnemonic::Svdc),
            ffi::_ND_INS_CLASS::ND_INS_SVLDT => Ok(Mnemonic::Svldt),
            ffi::_ND_INS_CLASS::ND_INS_SVTS => Ok(Mnemonic::Svts),
            ffi::_ND_INS_CLASS::ND_INS_SWAPGS => Ok(Mnemonic::Swapgs),
            ffi::_ND_INS_CLASS::ND_INS_SYSCALL => Ok(Mnemonic::Syscall),
            ffi::_ND_INS_CLASS::ND_INS_SYSENTER => Ok(Mnemonic::Sysenter),
            ffi::_ND_INS_CLASS::ND_INS_SYSEXIT => Ok(Mnemonic::Sysexit),
            ffi::_ND_INS_CLASS::ND_INS_SYSRET => Ok(Mnemonic::Sysret),
            ffi::_ND_INS_CLASS::ND_INS_T1MSKC => Ok(Mnemonic::T1mskc),
            ffi::_ND_INS_CLASS::ND_INS_TCMMIMFP16PS => Ok(Mnemonic::Tcmmimfp16ps),
            ffi::_ND_INS_CLASS::ND_INS_TCMMRLFP16PS => Ok(Mnemonic::Tcmmrlfp16ps),
            ffi::_ND_INS_CLASS::ND_INS_TDCALL => Ok(Mnemonic::Tdcall),
            ffi::_ND_INS_CLASS::ND_INS_TDPBF16PS => Ok(Mnemonic::Tdpbf16ps),
            ffi::_ND_INS_CLASS::ND_INS_TDPBSSD => Ok(Mnemonic::Tdpbssd),
            ffi::_ND_INS_CLASS::ND_INS_TDPBSUD => Ok(Mnemonic::Tdpbsud),
            ffi::_ND_INS_CLASS::ND_INS_TDPBUSD => Ok(Mnemonic::Tdpbusd),
            ffi::_ND_INS_CLASS::ND_INS_TDPBUUD => Ok(Mnemonic::Tdpbuud),
            ffi::_ND_INS_CLASS::ND_INS_TDPFP16PS => Ok(Mnemonic::Tdpfp16ps),
            ffi::_ND_INS_CLASS::ND_INS_TEST => Ok(Mnemonic::Test),
            ffi::_ND_INS_CLASS::ND_INS_TESTUI => Ok(Mnemonic::Testui),
            ffi::_ND_INS_CLASS::ND_INS_TILELOADD => Ok(Mnemonic::Tileloadd),
            ffi::_ND_INS_CLASS::ND_INS_TILELOADDT1 => Ok(Mnemonic::Tileloaddt1),
            ffi::_ND_INS_CLASS::ND_INS_TILERELEASE => Ok(Mnemonic::Tilerelease),
            ffi::_ND_INS_CLASS::ND_INS_TILESTORED => Ok(Mnemonic::Tilestored),
            ffi::_ND_INS_CLASS::ND_INS_TILEZERO => Ok(Mnemonic::Tilezero),
            ffi::_ND_INS_CLASS::ND_INS_TLBSYNC => Ok(Mnemonic::Tlbsync),
            ffi::_ND_INS_CLASS::ND_INS_TPAUSE => Ok(Mnemonic::Tpause),
            ffi::_ND_INS_CLASS::ND_INS_TZCNT => Ok(Mnemonic::Tzcnt),
            ffi::_ND_INS_CLASS::ND_INS_TZMSK => Ok(Mnemonic::Tzmsk),
            ffi::_ND_INS_CLASS::ND_INS_UCOMISD => Ok(Mnemonic::Ucomisd),
            ffi::_ND_INS_CLASS::ND_INS_UCOMISS => Ok(Mnemonic::Ucomiss),
            ffi::_ND_INS_CLASS::ND_INS_UD0 => Ok(Mnemonic::Ud0),
            ffi::_ND_INS_CLASS::ND_INS_UD1 => Ok(Mnemonic::Ud1),
            ffi::_ND_INS_CLASS::ND_INS_UD2 => Ok(Mnemonic::Ud2),
            ffi::_ND_INS_CLASS::ND_INS_UIRET => Ok(Mnemonic::Uiret),
            ffi::_ND_INS_CLASS::ND_INS_UMONITOR => Ok(Mnemonic::Umonitor),
            ffi::_ND_INS_CLASS::ND_INS_UMWAIT => Ok(Mnemonic::Umwait),
            ffi::_ND_INS_CLASS::ND_INS_UNPCKHPD => Ok(Mnemonic::Unpckhpd),
            ffi::_ND_INS_CLASS::ND_INS_UNPCKHPS => Ok(Mnemonic::Unpckhps),
            ffi::_ND_INS_CLASS::ND_INS_UNPCKLPD => Ok(Mnemonic::Unpcklpd),
            ffi::_ND_INS_CLASS::ND_INS_UNPCKLPS => Ok(Mnemonic::Unpcklps),
            ffi::_ND_INS_CLASS::ND_INS_V4FMADDPS => Ok(Mnemonic::V4fmaddps),
            ffi::_ND_INS_CLASS::ND_INS_V4FMADDSS => Ok(Mnemonic::V4fmaddss),
            ffi::_ND_INS_CLASS::ND_INS_V4FNMADDPS => Ok(Mnemonic::V4fnmaddps),
            ffi::_ND_INS_CLASS::ND_INS_V4FNMADDSS => Ok(Mnemonic::V4fnmaddss),
            ffi::_ND_INS_CLASS::ND_INS_VADDPD => Ok(Mnemonic::Vaddpd),
            ffi::_ND_INS_CLASS::ND_INS_VADDPH => Ok(Mnemonic::Vaddph),
            ffi::_ND_INS_CLASS::ND_INS_VADDPS => Ok(Mnemonic::Vaddps),
            ffi::_ND_INS_CLASS::ND_INS_VADDSD => Ok(Mnemonic::Vaddsd),
            ffi::_ND_INS_CLASS::ND_INS_VADDSH => Ok(Mnemonic::Vaddsh),
            ffi::_ND_INS_CLASS::ND_INS_VADDSS => Ok(Mnemonic::Vaddss),
            ffi::_ND_INS_CLASS::ND_INS_VADDSUBPD => Ok(Mnemonic::Vaddsubpd),
            ffi::_ND_INS_CLASS::ND_INS_VADDSUBPS => Ok(Mnemonic::Vaddsubps),
            ffi::_ND_INS_CLASS::ND_INS_VAESDEC => Ok(Mnemonic::Vaesdec),
            ffi::_ND_INS_CLASS::ND_INS_VAESDECLAST => Ok(Mnemonic::Vaesdeclast),
            ffi::_ND_INS_CLASS::ND_INS_VAESENC => Ok(Mnemonic::Vaesenc),
            ffi::_ND_INS_CLASS::ND_INS_VAESENCLAST => Ok(Mnemonic::Vaesenclast),
            ffi::_ND_INS_CLASS::ND_INS_VAESIMC => Ok(Mnemonic::Vaesimc),
            ffi::_ND_INS_CLASS::ND_INS_VAESKEYGENASSIST => Ok(Mnemonic::Vaeskeygenassist),
            ffi::_ND_INS_CLASS::ND_INS_VALIGND => Ok(Mnemonic::Valignd),
            ffi::_ND_INS_CLASS::ND_INS_VALIGNQ => Ok(Mnemonic::Valignq),
            ffi::_ND_INS_CLASS::ND_INS_VANDNPD => Ok(Mnemonic::Vandnpd),
            ffi::_ND_INS_CLASS::ND_INS_VANDNPS => Ok(Mnemonic::Vandnps),
            ffi::_ND_INS_CLASS::ND_INS_VANDPD => Ok(Mnemonic::Vandpd),
            ffi::_ND_INS_CLASS::ND_INS_VANDPS => Ok(Mnemonic::Vandps),
            ffi::_ND_INS_CLASS::ND_INS_VBCSTNEBF162PS => Ok(Mnemonic::Vbcstnebf162ps),
            ffi::_ND_INS_CLASS::ND_INS_VBCSTNESH2PS => Ok(Mnemonic::Vbcstnesh2ps),
            ffi::_ND_INS_CLASS::ND_INS_VBLENDMPD => Ok(Mnemonic::Vblendmpd),
            ffi::_ND_INS_CLASS::ND_INS_VBLENDMPS => Ok(Mnemonic::Vblendmps),
            ffi::_ND_INS_CLASS::ND_INS_VBLENDPD => Ok(Mnemonic::Vblendpd),
            ffi::_ND_INS_CLASS::ND_INS_VBLENDPS => Ok(Mnemonic::Vblendps),
            ffi::_ND_INS_CLASS::ND_INS_VBLENDVPD => Ok(Mnemonic::Vblendvpd),
            ffi::_ND_INS_CLASS::ND_INS_VBLENDVPS => Ok(Mnemonic::Vblendvps),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTF128 => Ok(Mnemonic::Vbroadcastf128),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTF32X2 => Ok(Mnemonic::Vbroadcastf32x2),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTF32X4 => Ok(Mnemonic::Vbroadcastf32x4),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTF32X8 => Ok(Mnemonic::Vbroadcastf32x8),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTF64X2 => Ok(Mnemonic::Vbroadcastf64x2),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTF64X4 => Ok(Mnemonic::Vbroadcastf64x4),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTI128 => Ok(Mnemonic::Vbroadcasti128),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTI32X2 => Ok(Mnemonic::Vbroadcasti32x2),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTI32X4 => Ok(Mnemonic::Vbroadcasti32x4),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTI32X8 => Ok(Mnemonic::Vbroadcasti32x8),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTI64X2 => Ok(Mnemonic::Vbroadcasti64x2),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTI64X4 => Ok(Mnemonic::Vbroadcasti64x4),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTSD => Ok(Mnemonic::Vbroadcastsd),
            ffi::_ND_INS_CLASS::ND_INS_VBROADCASTSS => Ok(Mnemonic::Vbroadcastss),
            ffi::_ND_INS_CLASS::ND_INS_VCMPPD => Ok(Mnemonic::Vcmppd),
            ffi::_ND_INS_CLASS::ND_INS_VCMPPH => Ok(Mnemonic::Vcmpph),
            ffi::_ND_INS_CLASS::ND_INS_VCMPPS => Ok(Mnemonic::Vcmpps),
            ffi::_ND_INS_CLASS::ND_INS_VCMPSD => Ok(Mnemonic::Vcmpsd),
            ffi::_ND_INS_CLASS::ND_INS_VCMPSH => Ok(Mnemonic::Vcmpsh),
            ffi::_ND_INS_CLASS::ND_INS_VCMPSS => Ok(Mnemonic::Vcmpss),
            ffi::_ND_INS_CLASS::ND_INS_VCOMISD => Ok(Mnemonic::Vcomisd),
            ffi::_ND_INS_CLASS::ND_INS_VCOMISH => Ok(Mnemonic::Vcomish),
            ffi::_ND_INS_CLASS::ND_INS_VCOMISS => Ok(Mnemonic::Vcomiss),
            ffi::_ND_INS_CLASS::ND_INS_VCOMPRESSPD => Ok(Mnemonic::Vcompresspd),
            ffi::_ND_INS_CLASS::ND_INS_VCOMPRESSPS => Ok(Mnemonic::Vcompressps),
            ffi::_ND_INS_CLASS::ND_INS_VCVTDQ2PD => Ok(Mnemonic::Vcvtdq2pd),
            ffi::_ND_INS_CLASS::ND_INS_VCVTDQ2PH => Ok(Mnemonic::Vcvtdq2ph),
            ffi::_ND_INS_CLASS::ND_INS_VCVTDQ2PS => Ok(Mnemonic::Vcvtdq2ps),
            ffi::_ND_INS_CLASS::ND_INS_VCVTNE2PS2BF16 => Ok(Mnemonic::Vcvtne2ps2bf16),
            ffi::_ND_INS_CLASS::ND_INS_VCVTNEEBF162PS => Ok(Mnemonic::Vcvtneebf162ps),
            ffi::_ND_INS_CLASS::ND_INS_VCVTNEEPH2PS => Ok(Mnemonic::Vcvtneeph2ps),
            ffi::_ND_INS_CLASS::ND_INS_VCVTNEOBF162PS => Ok(Mnemonic::Vcvtneobf162ps),
            ffi::_ND_INS_CLASS::ND_INS_VCVTNEOPH2PS => Ok(Mnemonic::Vcvtneoph2ps),
            ffi::_ND_INS_CLASS::ND_INS_VCVTNEPS2BF16 => Ok(Mnemonic::Vcvtneps2bf16),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPD2DQ => Ok(Mnemonic::Vcvtpd2dq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPD2PH => Ok(Mnemonic::Vcvtpd2ph),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPD2PS => Ok(Mnemonic::Vcvtpd2ps),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPD2QQ => Ok(Mnemonic::Vcvtpd2qq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPD2UDQ => Ok(Mnemonic::Vcvtpd2udq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPD2UQQ => Ok(Mnemonic::Vcvtpd2uqq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPH2DQ => Ok(Mnemonic::Vcvtph2dq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPH2PD => Ok(Mnemonic::Vcvtph2pd),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPH2PS => Ok(Mnemonic::Vcvtph2ps),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPH2PSX => Ok(Mnemonic::Vcvtph2psx),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPH2QQ => Ok(Mnemonic::Vcvtph2qq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPH2UDQ => Ok(Mnemonic::Vcvtph2udq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPH2UQQ => Ok(Mnemonic::Vcvtph2uqq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPH2UW => Ok(Mnemonic::Vcvtph2uw),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPH2W => Ok(Mnemonic::Vcvtph2w),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPS2DQ => Ok(Mnemonic::Vcvtps2dq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPS2PD => Ok(Mnemonic::Vcvtps2pd),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPS2PH => Ok(Mnemonic::Vcvtps2ph),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPS2PHX => Ok(Mnemonic::Vcvtps2phx),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPS2QQ => Ok(Mnemonic::Vcvtps2qq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPS2UDQ => Ok(Mnemonic::Vcvtps2udq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTPS2UQQ => Ok(Mnemonic::Vcvtps2uqq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTQQ2PD => Ok(Mnemonic::Vcvtqq2pd),
            ffi::_ND_INS_CLASS::ND_INS_VCVTQQ2PH => Ok(Mnemonic::Vcvtqq2ph),
            ffi::_ND_INS_CLASS::ND_INS_VCVTQQ2PS => Ok(Mnemonic::Vcvtqq2ps),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSD2SH => Ok(Mnemonic::Vcvtsd2sh),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSD2SI => Ok(Mnemonic::Vcvtsd2si),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSD2SS => Ok(Mnemonic::Vcvtsd2ss),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSD2USI => Ok(Mnemonic::Vcvtsd2usi),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSH2SD => Ok(Mnemonic::Vcvtsh2sd),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSH2SI => Ok(Mnemonic::Vcvtsh2si),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSH2SS => Ok(Mnemonic::Vcvtsh2ss),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSH2USI => Ok(Mnemonic::Vcvtsh2usi),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSI2SD => Ok(Mnemonic::Vcvtsi2sd),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSI2SH => Ok(Mnemonic::Vcvtsi2sh),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSI2SS => Ok(Mnemonic::Vcvtsi2ss),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSS2SD => Ok(Mnemonic::Vcvtss2sd),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSS2SH => Ok(Mnemonic::Vcvtss2sh),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSS2SI => Ok(Mnemonic::Vcvtss2si),
            ffi::_ND_INS_CLASS::ND_INS_VCVTSS2USI => Ok(Mnemonic::Vcvtss2usi),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPD2DQ => Ok(Mnemonic::Vcvttpd2dq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPD2QQ => Ok(Mnemonic::Vcvttpd2qq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPD2UDQ => Ok(Mnemonic::Vcvttpd2udq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPD2UQQ => Ok(Mnemonic::Vcvttpd2uqq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPH2DQ => Ok(Mnemonic::Vcvttph2dq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPH2QQ => Ok(Mnemonic::Vcvttph2qq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPH2UDQ => Ok(Mnemonic::Vcvttph2udq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPH2UQQ => Ok(Mnemonic::Vcvttph2uqq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPH2UW => Ok(Mnemonic::Vcvttph2uw),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPH2W => Ok(Mnemonic::Vcvttph2w),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPS2DQ => Ok(Mnemonic::Vcvttps2dq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPS2QQ => Ok(Mnemonic::Vcvttps2qq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPS2UDQ => Ok(Mnemonic::Vcvttps2udq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTPS2UQQ => Ok(Mnemonic::Vcvttps2uqq),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTSD2SI => Ok(Mnemonic::Vcvttsd2si),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTSD2USI => Ok(Mnemonic::Vcvttsd2usi),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTSH2SI => Ok(Mnemonic::Vcvttsh2si),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTSH2USI => Ok(Mnemonic::Vcvttsh2usi),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTSS2SI => Ok(Mnemonic::Vcvttss2si),
            ffi::_ND_INS_CLASS::ND_INS_VCVTTSS2USI => Ok(Mnemonic::Vcvttss2usi),
            ffi::_ND_INS_CLASS::ND_INS_VCVTUDQ2PD => Ok(Mnemonic::Vcvtudq2pd),
            ffi::_ND_INS_CLASS::ND_INS_VCVTUDQ2PH => Ok(Mnemonic::Vcvtudq2ph),
            ffi::_ND_INS_CLASS::ND_INS_VCVTUDQ2PS => Ok(Mnemonic::Vcvtudq2ps),
            ffi::_ND_INS_CLASS::ND_INS_VCVTUQQ2PD => Ok(Mnemonic::Vcvtuqq2pd),
            ffi::_ND_INS_CLASS::ND_INS_VCVTUQQ2PH => Ok(Mnemonic::Vcvtuqq2ph),
            ffi::_ND_INS_CLASS::ND_INS_VCVTUQQ2PS => Ok(Mnemonic::Vcvtuqq2ps),
            ffi::_ND_INS_CLASS::ND_INS_VCVTUSI2SD => Ok(Mnemonic::Vcvtusi2sd),
            ffi::_ND_INS_CLASS::ND_INS_VCVTUSI2SH => Ok(Mnemonic::Vcvtusi2sh),
            ffi::_ND_INS_CLASS::ND_INS_VCVTUSI2SS => Ok(Mnemonic::Vcvtusi2ss),
            ffi::_ND_INS_CLASS::ND_INS_VCVTUW2PH => Ok(Mnemonic::Vcvtuw2ph),
            ffi::_ND_INS_CLASS::ND_INS_VCVTW2PH => Ok(Mnemonic::Vcvtw2ph),
            ffi::_ND_INS_CLASS::ND_INS_VDBPSADBW => Ok(Mnemonic::Vdbpsadbw),
            ffi::_ND_INS_CLASS::ND_INS_VDIVPD => Ok(Mnemonic::Vdivpd),
            ffi::_ND_INS_CLASS::ND_INS_VDIVPH => Ok(Mnemonic::Vdivph),
            ffi::_ND_INS_CLASS::ND_INS_VDIVPS => Ok(Mnemonic::Vdivps),
            ffi::_ND_INS_CLASS::ND_INS_VDIVSD => Ok(Mnemonic::Vdivsd),
            ffi::_ND_INS_CLASS::ND_INS_VDIVSH => Ok(Mnemonic::Vdivsh),
            ffi::_ND_INS_CLASS::ND_INS_VDIVSS => Ok(Mnemonic::Vdivss),
            ffi::_ND_INS_CLASS::ND_INS_VDPBF16PS => Ok(Mnemonic::Vdpbf16ps),
            ffi::_ND_INS_CLASS::ND_INS_VDPPD => Ok(Mnemonic::Vdppd),
            ffi::_ND_INS_CLASS::ND_INS_VDPPS => Ok(Mnemonic::Vdpps),
            ffi::_ND_INS_CLASS::ND_INS_VERR => Ok(Mnemonic::Verr),
            ffi::_ND_INS_CLASS::ND_INS_VERW => Ok(Mnemonic::Verw),
            ffi::_ND_INS_CLASS::ND_INS_VEXP2PD => Ok(Mnemonic::Vexp2pd),
            ffi::_ND_INS_CLASS::ND_INS_VEXP2PS => Ok(Mnemonic::Vexp2ps),
            ffi::_ND_INS_CLASS::ND_INS_VEXPANDPD => Ok(Mnemonic::Vexpandpd),
            ffi::_ND_INS_CLASS::ND_INS_VEXPANDPS => Ok(Mnemonic::Vexpandps),
            ffi::_ND_INS_CLASS::ND_INS_VEXTRACTF128 => Ok(Mnemonic::Vextractf128),
            ffi::_ND_INS_CLASS::ND_INS_VEXTRACTF32X4 => Ok(Mnemonic::Vextractf32x4),
            ffi::_ND_INS_CLASS::ND_INS_VEXTRACTF32X8 => Ok(Mnemonic::Vextractf32x8),
            ffi::_ND_INS_CLASS::ND_INS_VEXTRACTF64X2 => Ok(Mnemonic::Vextractf64x2),
            ffi::_ND_INS_CLASS::ND_INS_VEXTRACTF64X4 => Ok(Mnemonic::Vextractf64x4),
            ffi::_ND_INS_CLASS::ND_INS_VEXTRACTI128 => Ok(Mnemonic::Vextracti128),
            ffi::_ND_INS_CLASS::ND_INS_VEXTRACTI32X4 => Ok(Mnemonic::Vextracti32x4),
            ffi::_ND_INS_CLASS::ND_INS_VEXTRACTI32X8 => Ok(Mnemonic::Vextracti32x8),
            ffi::_ND_INS_CLASS::ND_INS_VEXTRACTI64X2 => Ok(Mnemonic::Vextracti64x2),
            ffi::_ND_INS_CLASS::ND_INS_VEXTRACTI64X4 => Ok(Mnemonic::Vextracti64x4),
            ffi::_ND_INS_CLASS::ND_INS_VEXTRACTPS => Ok(Mnemonic::Vextractps),
            ffi::_ND_INS_CLASS::ND_INS_VFCMADDCPH => Ok(Mnemonic::Vfcmaddcph),
            ffi::_ND_INS_CLASS::ND_INS_VFCMADDCSH => Ok(Mnemonic::Vfcmaddcsh),
            ffi::_ND_INS_CLASS::ND_INS_VFCMULCPH => Ok(Mnemonic::Vfcmulcph),
            ffi::_ND_INS_CLASS::ND_INS_VFCMULCSH => Ok(Mnemonic::Vfcmulcsh),
            ffi::_ND_INS_CLASS::ND_INS_VFIXUPIMMPD => Ok(Mnemonic::Vfixupimmpd),
            ffi::_ND_INS_CLASS::ND_INS_VFIXUPIMMPS => Ok(Mnemonic::Vfixupimmps),
            ffi::_ND_INS_CLASS::ND_INS_VFIXUPIMMSD => Ok(Mnemonic::Vfixupimmsd),
            ffi::_ND_INS_CLASS::ND_INS_VFIXUPIMMSS => Ok(Mnemonic::Vfixupimmss),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD132PD => Ok(Mnemonic::Vfmadd132pd),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD132PH => Ok(Mnemonic::Vfmadd132ph),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD132PS => Ok(Mnemonic::Vfmadd132ps),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD132SD => Ok(Mnemonic::Vfmadd132sd),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD132SH => Ok(Mnemonic::Vfmadd132sh),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD132SS => Ok(Mnemonic::Vfmadd132ss),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD213PD => Ok(Mnemonic::Vfmadd213pd),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD213PH => Ok(Mnemonic::Vfmadd213ph),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD213PS => Ok(Mnemonic::Vfmadd213ps),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD213SD => Ok(Mnemonic::Vfmadd213sd),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD213SH => Ok(Mnemonic::Vfmadd213sh),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD213SS => Ok(Mnemonic::Vfmadd213ss),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD231PD => Ok(Mnemonic::Vfmadd231pd),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD231PH => Ok(Mnemonic::Vfmadd231ph),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD231PS => Ok(Mnemonic::Vfmadd231ps),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD231SD => Ok(Mnemonic::Vfmadd231sd),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD231SH => Ok(Mnemonic::Vfmadd231sh),
            ffi::_ND_INS_CLASS::ND_INS_VFMADD231SS => Ok(Mnemonic::Vfmadd231ss),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDCPH => Ok(Mnemonic::Vfmaddcph),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDCSH => Ok(Mnemonic::Vfmaddcsh),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDPD => Ok(Mnemonic::Vfmaddpd),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDPS => Ok(Mnemonic::Vfmaddps),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDSD => Ok(Mnemonic::Vfmaddsd),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDSS => Ok(Mnemonic::Vfmaddss),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDSUB132PD => Ok(Mnemonic::Vfmaddsub132pd),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDSUB132PH => Ok(Mnemonic::Vfmaddsub132ph),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDSUB132PS => Ok(Mnemonic::Vfmaddsub132ps),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDSUB213PD => Ok(Mnemonic::Vfmaddsub213pd),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDSUB213PH => Ok(Mnemonic::Vfmaddsub213ph),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDSUB213PS => Ok(Mnemonic::Vfmaddsub213ps),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDSUB231PD => Ok(Mnemonic::Vfmaddsub231pd),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDSUB231PH => Ok(Mnemonic::Vfmaddsub231ph),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDSUB231PS => Ok(Mnemonic::Vfmaddsub231ps),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDSUBPD => Ok(Mnemonic::Vfmaddsubpd),
            ffi::_ND_INS_CLASS::ND_INS_VFMADDSUBPS => Ok(Mnemonic::Vfmaddsubps),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB132PD => Ok(Mnemonic::Vfmsub132pd),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB132PH => Ok(Mnemonic::Vfmsub132ph),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB132PS => Ok(Mnemonic::Vfmsub132ps),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB132SD => Ok(Mnemonic::Vfmsub132sd),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB132SH => Ok(Mnemonic::Vfmsub132sh),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB132SS => Ok(Mnemonic::Vfmsub132ss),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB213PD => Ok(Mnemonic::Vfmsub213pd),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB213PH => Ok(Mnemonic::Vfmsub213ph),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB213PS => Ok(Mnemonic::Vfmsub213ps),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB213SD => Ok(Mnemonic::Vfmsub213sd),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB213SH => Ok(Mnemonic::Vfmsub213sh),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB213SS => Ok(Mnemonic::Vfmsub213ss),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB231PD => Ok(Mnemonic::Vfmsub231pd),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB231PH => Ok(Mnemonic::Vfmsub231ph),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB231PS => Ok(Mnemonic::Vfmsub231ps),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB231SD => Ok(Mnemonic::Vfmsub231sd),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB231SH => Ok(Mnemonic::Vfmsub231sh),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUB231SS => Ok(Mnemonic::Vfmsub231ss),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBADD132PD => Ok(Mnemonic::Vfmsubadd132pd),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBADD132PH => Ok(Mnemonic::Vfmsubadd132ph),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBADD132PS => Ok(Mnemonic::Vfmsubadd132ps),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBADD213PD => Ok(Mnemonic::Vfmsubadd213pd),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBADD213PH => Ok(Mnemonic::Vfmsubadd213ph),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBADD213PS => Ok(Mnemonic::Vfmsubadd213ps),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBADD231PD => Ok(Mnemonic::Vfmsubadd231pd),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBADD231PH => Ok(Mnemonic::Vfmsubadd231ph),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBADD231PS => Ok(Mnemonic::Vfmsubadd231ps),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBADDPD => Ok(Mnemonic::Vfmsubaddpd),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBADDPS => Ok(Mnemonic::Vfmsubaddps),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBPD => Ok(Mnemonic::Vfmsubpd),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBPS => Ok(Mnemonic::Vfmsubps),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBSD => Ok(Mnemonic::Vfmsubsd),
            ffi::_ND_INS_CLASS::ND_INS_VFMSUBSS => Ok(Mnemonic::Vfmsubss),
            ffi::_ND_INS_CLASS::ND_INS_VFMULCPH => Ok(Mnemonic::Vfmulcph),
            ffi::_ND_INS_CLASS::ND_INS_VFMULCSH => Ok(Mnemonic::Vfmulcsh),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD132PD => Ok(Mnemonic::Vfnmadd132pd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD132PH => Ok(Mnemonic::Vfnmadd132ph),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD132PS => Ok(Mnemonic::Vfnmadd132ps),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD132SD => Ok(Mnemonic::Vfnmadd132sd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD132SH => Ok(Mnemonic::Vfnmadd132sh),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD132SS => Ok(Mnemonic::Vfnmadd132ss),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD213PD => Ok(Mnemonic::Vfnmadd213pd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD213PH => Ok(Mnemonic::Vfnmadd213ph),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD213PS => Ok(Mnemonic::Vfnmadd213ps),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD213SD => Ok(Mnemonic::Vfnmadd213sd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD213SH => Ok(Mnemonic::Vfnmadd213sh),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD213SS => Ok(Mnemonic::Vfnmadd213ss),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD231PD => Ok(Mnemonic::Vfnmadd231pd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD231PH => Ok(Mnemonic::Vfnmadd231ph),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD231PS => Ok(Mnemonic::Vfnmadd231ps),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD231SD => Ok(Mnemonic::Vfnmadd231sd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD231SH => Ok(Mnemonic::Vfnmadd231sh),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADD231SS => Ok(Mnemonic::Vfnmadd231ss),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADDPD => Ok(Mnemonic::Vfnmaddpd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADDPS => Ok(Mnemonic::Vfnmaddps),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADDSD => Ok(Mnemonic::Vfnmaddsd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMADDSS => Ok(Mnemonic::Vfnmaddss),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB132PD => Ok(Mnemonic::Vfnmsub132pd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB132PH => Ok(Mnemonic::Vfnmsub132ph),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB132PS => Ok(Mnemonic::Vfnmsub132ps),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB132SD => Ok(Mnemonic::Vfnmsub132sd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB132SH => Ok(Mnemonic::Vfnmsub132sh),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB132SS => Ok(Mnemonic::Vfnmsub132ss),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB213PD => Ok(Mnemonic::Vfnmsub213pd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB213PH => Ok(Mnemonic::Vfnmsub213ph),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB213PS => Ok(Mnemonic::Vfnmsub213ps),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB213SD => Ok(Mnemonic::Vfnmsub213sd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB213SH => Ok(Mnemonic::Vfnmsub213sh),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB213SS => Ok(Mnemonic::Vfnmsub213ss),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB231PD => Ok(Mnemonic::Vfnmsub231pd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB231PH => Ok(Mnemonic::Vfnmsub231ph),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB231PS => Ok(Mnemonic::Vfnmsub231ps),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB231SD => Ok(Mnemonic::Vfnmsub231sd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB231SH => Ok(Mnemonic::Vfnmsub231sh),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUB231SS => Ok(Mnemonic::Vfnmsub231ss),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUBPD => Ok(Mnemonic::Vfnmsubpd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUBPS => Ok(Mnemonic::Vfnmsubps),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUBSD => Ok(Mnemonic::Vfnmsubsd),
            ffi::_ND_INS_CLASS::ND_INS_VFNMSUBSS => Ok(Mnemonic::Vfnmsubss),
            ffi::_ND_INS_CLASS::ND_INS_VFPCLASSPD => Ok(Mnemonic::Vfpclasspd),
            ffi::_ND_INS_CLASS::ND_INS_VFPCLASSPH => Ok(Mnemonic::Vfpclassph),
            ffi::_ND_INS_CLASS::ND_INS_VFPCLASSPS => Ok(Mnemonic::Vfpclassps),
            ffi::_ND_INS_CLASS::ND_INS_VFPCLASSSD => Ok(Mnemonic::Vfpclasssd),
            ffi::_ND_INS_CLASS::ND_INS_VFPCLASSSH => Ok(Mnemonic::Vfpclasssh),
            ffi::_ND_INS_CLASS::ND_INS_VFPCLASSSS => Ok(Mnemonic::Vfpclassss),
            ffi::_ND_INS_CLASS::ND_INS_VFRCZPD => Ok(Mnemonic::Vfrczpd),
            ffi::_ND_INS_CLASS::ND_INS_VFRCZPS => Ok(Mnemonic::Vfrczps),
            ffi::_ND_INS_CLASS::ND_INS_VFRCZSD => Ok(Mnemonic::Vfrczsd),
            ffi::_ND_INS_CLASS::ND_INS_VFRCZSS => Ok(Mnemonic::Vfrczss),
            ffi::_ND_INS_CLASS::ND_INS_VGATHERDPD => Ok(Mnemonic::Vgatherdpd),
            ffi::_ND_INS_CLASS::ND_INS_VGATHERDPS => Ok(Mnemonic::Vgatherdps),
            ffi::_ND_INS_CLASS::ND_INS_VGATHERPF0DPD => Ok(Mnemonic::Vgatherpf0dpd),
            ffi::_ND_INS_CLASS::ND_INS_VGATHERPF0DPS => Ok(Mnemonic::Vgatherpf0dps),
            ffi::_ND_INS_CLASS::ND_INS_VGATHERPF0QPD => Ok(Mnemonic::Vgatherpf0qpd),
            ffi::_ND_INS_CLASS::ND_INS_VGATHERPF0QPS => Ok(Mnemonic::Vgatherpf0qps),
            ffi::_ND_INS_CLASS::ND_INS_VGATHERPF1DPD => Ok(Mnemonic::Vgatherpf1dpd),
            ffi::_ND_INS_CLASS::ND_INS_VGATHERPF1DPS => Ok(Mnemonic::Vgatherpf1dps),
            ffi::_ND_INS_CLASS::ND_INS_VGATHERPF1QPD => Ok(Mnemonic::Vgatherpf1qpd),
            ffi::_ND_INS_CLASS::ND_INS_VGATHERPF1QPS => Ok(Mnemonic::Vgatherpf1qps),
            ffi::_ND_INS_CLASS::ND_INS_VGATHERQPD => Ok(Mnemonic::Vgatherqpd),
            ffi::_ND_INS_CLASS::ND_INS_VGATHERQPS => Ok(Mnemonic::Vgatherqps),
            ffi::_ND_INS_CLASS::ND_INS_VGETEXPPD => Ok(Mnemonic::Vgetexppd),
            ffi::_ND_INS_CLASS::ND_INS_VGETEXPPH => Ok(Mnemonic::Vgetexpph),
            ffi::_ND_INS_CLASS::ND_INS_VGETEXPPS => Ok(Mnemonic::Vgetexpps),
            ffi::_ND_INS_CLASS::ND_INS_VGETEXPSD => Ok(Mnemonic::Vgetexpsd),
            ffi::_ND_INS_CLASS::ND_INS_VGETEXPSH => Ok(Mnemonic::Vgetexpsh),
            ffi::_ND_INS_CLASS::ND_INS_VGETEXPSS => Ok(Mnemonic::Vgetexpss),
            ffi::_ND_INS_CLASS::ND_INS_VGETMANTPD => Ok(Mnemonic::Vgetmantpd),
            ffi::_ND_INS_CLASS::ND_INS_VGETMANTPH => Ok(Mnemonic::Vgetmantph),
            ffi::_ND_INS_CLASS::ND_INS_VGETMANTPS => Ok(Mnemonic::Vgetmantps),
            ffi::_ND_INS_CLASS::ND_INS_VGETMANTSD => Ok(Mnemonic::Vgetmantsd),
            ffi::_ND_INS_CLASS::ND_INS_VGETMANTSH => Ok(Mnemonic::Vgetmantsh),
            ffi::_ND_INS_CLASS::ND_INS_VGETMANTSS => Ok(Mnemonic::Vgetmantss),
            ffi::_ND_INS_CLASS::ND_INS_VGF2P8AFFINEINVQB => Ok(Mnemonic::Vgf2p8affineinvqb),
            ffi::_ND_INS_CLASS::ND_INS_VGF2P8AFFINEQB => Ok(Mnemonic::Vgf2p8affineqb),
            ffi::_ND_INS_CLASS::ND_INS_VGF2P8MULB => Ok(Mnemonic::Vgf2p8mulb),
            ffi::_ND_INS_CLASS::ND_INS_VHADDPD => Ok(Mnemonic::Vhaddpd),
            ffi::_ND_INS_CLASS::ND_INS_VHADDPS => Ok(Mnemonic::Vhaddps),
            ffi::_ND_INS_CLASS::ND_INS_VHSUBPD => Ok(Mnemonic::Vhsubpd),
            ffi::_ND_INS_CLASS::ND_INS_VHSUBPS => Ok(Mnemonic::Vhsubps),
            ffi::_ND_INS_CLASS::ND_INS_VINSERTF128 => Ok(Mnemonic::Vinsertf128),
            ffi::_ND_INS_CLASS::ND_INS_VINSERTF32X4 => Ok(Mnemonic::Vinsertf32x4),
            ffi::_ND_INS_CLASS::ND_INS_VINSERTF32X8 => Ok(Mnemonic::Vinsertf32x8),
            ffi::_ND_INS_CLASS::ND_INS_VINSERTF64X2 => Ok(Mnemonic::Vinsertf64x2),
            ffi::_ND_INS_CLASS::ND_INS_VINSERTF64X4 => Ok(Mnemonic::Vinsertf64x4),
            ffi::_ND_INS_CLASS::ND_INS_VINSERTI128 => Ok(Mnemonic::Vinserti128),
            ffi::_ND_INS_CLASS::ND_INS_VINSERTI32X4 => Ok(Mnemonic::Vinserti32x4),
            ffi::_ND_INS_CLASS::ND_INS_VINSERTI32X8 => Ok(Mnemonic::Vinserti32x8),
            ffi::_ND_INS_CLASS::ND_INS_VINSERTI64X2 => Ok(Mnemonic::Vinserti64x2),
            ffi::_ND_INS_CLASS::ND_INS_VINSERTI64X4 => Ok(Mnemonic::Vinserti64x4),
            ffi::_ND_INS_CLASS::ND_INS_VINSERTPS => Ok(Mnemonic::Vinsertps),
            ffi::_ND_INS_CLASS::ND_INS_VLDDQU => Ok(Mnemonic::Vlddqu),
            ffi::_ND_INS_CLASS::ND_INS_VLDMXCSR => Ok(Mnemonic::Vldmxcsr),
            ffi::_ND_INS_CLASS::ND_INS_VMASKMOVDQU => Ok(Mnemonic::Vmaskmovdqu),
            ffi::_ND_INS_CLASS::ND_INS_VMASKMOVPD => Ok(Mnemonic::Vmaskmovpd),
            ffi::_ND_INS_CLASS::ND_INS_VMASKMOVPS => Ok(Mnemonic::Vmaskmovps),
            ffi::_ND_INS_CLASS::ND_INS_VMAXPD => Ok(Mnemonic::Vmaxpd),
            ffi::_ND_INS_CLASS::ND_INS_VMAXPH => Ok(Mnemonic::Vmaxph),
            ffi::_ND_INS_CLASS::ND_INS_VMAXPS => Ok(Mnemonic::Vmaxps),
            ffi::_ND_INS_CLASS::ND_INS_VMAXSD => Ok(Mnemonic::Vmaxsd),
            ffi::_ND_INS_CLASS::ND_INS_VMAXSH => Ok(Mnemonic::Vmaxsh),
            ffi::_ND_INS_CLASS::ND_INS_VMAXSS => Ok(Mnemonic::Vmaxss),
            ffi::_ND_INS_CLASS::ND_INS_VMCALL => Ok(Mnemonic::Vmcall),
            ffi::_ND_INS_CLASS::ND_INS_VMCLEAR => Ok(Mnemonic::Vmclear),
            ffi::_ND_INS_CLASS::ND_INS_VMFUNC => Ok(Mnemonic::Vmfunc),
            ffi::_ND_INS_CLASS::ND_INS_VMGEXIT => Ok(Mnemonic::Vmgexit),
            ffi::_ND_INS_CLASS::ND_INS_VMINPD => Ok(Mnemonic::Vminpd),
            ffi::_ND_INS_CLASS::ND_INS_VMINPH => Ok(Mnemonic::Vminph),
            ffi::_ND_INS_CLASS::ND_INS_VMINPS => Ok(Mnemonic::Vminps),
            ffi::_ND_INS_CLASS::ND_INS_VMINSD => Ok(Mnemonic::Vminsd),
            ffi::_ND_INS_CLASS::ND_INS_VMINSH => Ok(Mnemonic::Vminsh),
            ffi::_ND_INS_CLASS::ND_INS_VMINSS => Ok(Mnemonic::Vminss),
            ffi::_ND_INS_CLASS::ND_INS_VMLAUNCH => Ok(Mnemonic::Vmlaunch),
            ffi::_ND_INS_CLASS::ND_INS_VMLOAD => Ok(Mnemonic::Vmload),
            ffi::_ND_INS_CLASS::ND_INS_VMMCALL => Ok(Mnemonic::Vmmcall),
            ffi::_ND_INS_CLASS::ND_INS_VMOVAPD => Ok(Mnemonic::Vmovapd),
            ffi::_ND_INS_CLASS::ND_INS_VMOVAPS => Ok(Mnemonic::Vmovaps),
            ffi::_ND_INS_CLASS::ND_INS_VMOVD => Ok(Mnemonic::Vmovd),
            ffi::_ND_INS_CLASS::ND_INS_VMOVDDUP => Ok(Mnemonic::Vmovddup),
            ffi::_ND_INS_CLASS::ND_INS_VMOVDQA => Ok(Mnemonic::Vmovdqa),
            ffi::_ND_INS_CLASS::ND_INS_VMOVDQA32 => Ok(Mnemonic::Vmovdqa32),
            ffi::_ND_INS_CLASS::ND_INS_VMOVDQA64 => Ok(Mnemonic::Vmovdqa64),
            ffi::_ND_INS_CLASS::ND_INS_VMOVDQU => Ok(Mnemonic::Vmovdqu),
            ffi::_ND_INS_CLASS::ND_INS_VMOVDQU16 => Ok(Mnemonic::Vmovdqu16),
            ffi::_ND_INS_CLASS::ND_INS_VMOVDQU32 => Ok(Mnemonic::Vmovdqu32),
            ffi::_ND_INS_CLASS::ND_INS_VMOVDQU64 => Ok(Mnemonic::Vmovdqu64),
            ffi::_ND_INS_CLASS::ND_INS_VMOVDQU8 => Ok(Mnemonic::Vmovdqu8),
            ffi::_ND_INS_CLASS::ND_INS_VMOVHLPS => Ok(Mnemonic::Vmovhlps),
            ffi::_ND_INS_CLASS::ND_INS_VMOVHPD => Ok(Mnemonic::Vmovhpd),
            ffi::_ND_INS_CLASS::ND_INS_VMOVHPS => Ok(Mnemonic::Vmovhps),
            ffi::_ND_INS_CLASS::ND_INS_VMOVLHPS => Ok(Mnemonic::Vmovlhps),
            ffi::_ND_INS_CLASS::ND_INS_VMOVLPD => Ok(Mnemonic::Vmovlpd),
            ffi::_ND_INS_CLASS::ND_INS_VMOVLPS => Ok(Mnemonic::Vmovlps),
            ffi::_ND_INS_CLASS::ND_INS_VMOVMSKPD => Ok(Mnemonic::Vmovmskpd),
            ffi::_ND_INS_CLASS::ND_INS_VMOVMSKPS => Ok(Mnemonic::Vmovmskps),
            ffi::_ND_INS_CLASS::ND_INS_VMOVNTDQ => Ok(Mnemonic::Vmovntdq),
            ffi::_ND_INS_CLASS::ND_INS_VMOVNTDQA => Ok(Mnemonic::Vmovntdqa),
            ffi::_ND_INS_CLASS::ND_INS_VMOVNTPD => Ok(Mnemonic::Vmovntpd),
            ffi::_ND_INS_CLASS::ND_INS_VMOVNTPS => Ok(Mnemonic::Vmovntps),
            ffi::_ND_INS_CLASS::ND_INS_VMOVQ => Ok(Mnemonic::Vmovq),
            ffi::_ND_INS_CLASS::ND_INS_VMOVSD => Ok(Mnemonic::Vmovsd),
            ffi::_ND_INS_CLASS::ND_INS_VMOVSH => Ok(Mnemonic::Vmovsh),
            ffi::_ND_INS_CLASS::ND_INS_VMOVSHDUP => Ok(Mnemonic::Vmovshdup),
            ffi::_ND_INS_CLASS::ND_INS_VMOVSLDUP => Ok(Mnemonic::Vmovsldup),
            ffi::_ND_INS_CLASS::ND_INS_VMOVSS => Ok(Mnemonic::Vmovss),
            ffi::_ND_INS_CLASS::ND_INS_VMOVUPD => Ok(Mnemonic::Vmovupd),
            ffi::_ND_INS_CLASS::ND_INS_VMOVUPS => Ok(Mnemonic::Vmovups),
            ffi::_ND_INS_CLASS::ND_INS_VMOVW => Ok(Mnemonic::Vmovw),
            ffi::_ND_INS_CLASS::ND_INS_VMPSADBW => Ok(Mnemonic::Vmpsadbw),
            ffi::_ND_INS_CLASS::ND_INS_VMPTRLD => Ok(Mnemonic::Vmptrld),
            ffi::_ND_INS_CLASS::ND_INS_VMPTRST => Ok(Mnemonic::Vmptrst),
            ffi::_ND_INS_CLASS::ND_INS_VMREAD => Ok(Mnemonic::Vmread),
            ffi::_ND_INS_CLASS::ND_INS_VMRESUME => Ok(Mnemonic::Vmresume),
            ffi::_ND_INS_CLASS::ND_INS_VMRUN => Ok(Mnemonic::Vmrun),
            ffi::_ND_INS_CLASS::ND_INS_VMSAVE => Ok(Mnemonic::Vmsave),
            ffi::_ND_INS_CLASS::ND_INS_VMULPD => Ok(Mnemonic::Vmulpd),
            ffi::_ND_INS_CLASS::ND_INS_VMULPH => Ok(Mnemonic::Vmulph),
            ffi::_ND_INS_CLASS::ND_INS_VMULPS => Ok(Mnemonic::Vmulps),
            ffi::_ND_INS_CLASS::ND_INS_VMULSD => Ok(Mnemonic::Vmulsd),
            ffi::_ND_INS_CLASS::ND_INS_VMULSH => Ok(Mnemonic::Vmulsh),
            ffi::_ND_INS_CLASS::ND_INS_VMULSS => Ok(Mnemonic::Vmulss),
            ffi::_ND_INS_CLASS::ND_INS_VMWRITE => Ok(Mnemonic::Vmwrite),
            ffi::_ND_INS_CLASS::ND_INS_VMXOFF => Ok(Mnemonic::Vmxoff),
            ffi::_ND_INS_CLASS::ND_INS_VMXON => Ok(Mnemonic::Vmxon),
            ffi::_ND_INS_CLASS::ND_INS_VORPD => Ok(Mnemonic::Vorpd),
            ffi::_ND_INS_CLASS::ND_INS_VORPS => Ok(Mnemonic::Vorps),
            ffi::_ND_INS_CLASS::ND_INS_VP2INTERSECTD => Ok(Mnemonic::Vp2intersectd),
            ffi::_ND_INS_CLASS::ND_INS_VP2INTERSECTQ => Ok(Mnemonic::Vp2intersectq),
            ffi::_ND_INS_CLASS::ND_INS_VP4DPWSSD => Ok(Mnemonic::Vp4dpwssd),
            ffi::_ND_INS_CLASS::ND_INS_VP4DPWSSDS => Ok(Mnemonic::Vp4dpwssds),
            ffi::_ND_INS_CLASS::ND_INS_VPABSB => Ok(Mnemonic::Vpabsb),
            ffi::_ND_INS_CLASS::ND_INS_VPABSD => Ok(Mnemonic::Vpabsd),
            ffi::_ND_INS_CLASS::ND_INS_VPABSQ => Ok(Mnemonic::Vpabsq),
            ffi::_ND_INS_CLASS::ND_INS_VPABSW => Ok(Mnemonic::Vpabsw),
            ffi::_ND_INS_CLASS::ND_INS_VPACKSSDW => Ok(Mnemonic::Vpackssdw),
            ffi::_ND_INS_CLASS::ND_INS_VPACKSSWB => Ok(Mnemonic::Vpacksswb),
            ffi::_ND_INS_CLASS::ND_INS_VPACKUSDW => Ok(Mnemonic::Vpackusdw),
            ffi::_ND_INS_CLASS::ND_INS_VPACKUSWB => Ok(Mnemonic::Vpackuswb),
            ffi::_ND_INS_CLASS::ND_INS_VPADDB => Ok(Mnemonic::Vpaddb),
            ffi::_ND_INS_CLASS::ND_INS_VPADDD => Ok(Mnemonic::Vpaddd),
            ffi::_ND_INS_CLASS::ND_INS_VPADDQ => Ok(Mnemonic::Vpaddq),
            ffi::_ND_INS_CLASS::ND_INS_VPADDSB => Ok(Mnemonic::Vpaddsb),
            ffi::_ND_INS_CLASS::ND_INS_VPADDSW => Ok(Mnemonic::Vpaddsw),
            ffi::_ND_INS_CLASS::ND_INS_VPADDUSB => Ok(Mnemonic::Vpaddusb),
            ffi::_ND_INS_CLASS::ND_INS_VPADDUSW => Ok(Mnemonic::Vpaddusw),
            ffi::_ND_INS_CLASS::ND_INS_VPADDW => Ok(Mnemonic::Vpaddw),
            ffi::_ND_INS_CLASS::ND_INS_VPALIGNR => Ok(Mnemonic::Vpalignr),
            ffi::_ND_INS_CLASS::ND_INS_VPAND => Ok(Mnemonic::Vpand),
            ffi::_ND_INS_CLASS::ND_INS_VPANDD => Ok(Mnemonic::Vpandd),
            ffi::_ND_INS_CLASS::ND_INS_VPANDN => Ok(Mnemonic::Vpandn),
            ffi::_ND_INS_CLASS::ND_INS_VPANDND => Ok(Mnemonic::Vpandnd),
            ffi::_ND_INS_CLASS::ND_INS_VPANDNQ => Ok(Mnemonic::Vpandnq),
            ffi::_ND_INS_CLASS::ND_INS_VPANDQ => Ok(Mnemonic::Vpandq),
            ffi::_ND_INS_CLASS::ND_INS_VPAVGB => Ok(Mnemonic::Vpavgb),
            ffi::_ND_INS_CLASS::ND_INS_VPAVGW => Ok(Mnemonic::Vpavgw),
            ffi::_ND_INS_CLASS::ND_INS_VPBLENDD => Ok(Mnemonic::Vpblendd),
            ffi::_ND_INS_CLASS::ND_INS_VPBLENDMB => Ok(Mnemonic::Vpblendmb),
            ffi::_ND_INS_CLASS::ND_INS_VPBLENDMD => Ok(Mnemonic::Vpblendmd),
            ffi::_ND_INS_CLASS::ND_INS_VPBLENDMQ => Ok(Mnemonic::Vpblendmq),
            ffi::_ND_INS_CLASS::ND_INS_VPBLENDMW => Ok(Mnemonic::Vpblendmw),
            ffi::_ND_INS_CLASS::ND_INS_VPBLENDVB => Ok(Mnemonic::Vpblendvb),
            ffi::_ND_INS_CLASS::ND_INS_VPBLENDW => Ok(Mnemonic::Vpblendw),
            ffi::_ND_INS_CLASS::ND_INS_VPBROADCASTB => Ok(Mnemonic::Vpbroadcastb),
            ffi::_ND_INS_CLASS::ND_INS_VPBROADCASTD => Ok(Mnemonic::Vpbroadcastd),
            ffi::_ND_INS_CLASS::ND_INS_VPBROADCASTMB2Q => Ok(Mnemonic::Vpbroadcastmb2q),
            ffi::_ND_INS_CLASS::ND_INS_VPBROADCASTMW2D => Ok(Mnemonic::Vpbroadcastmw2d),
            ffi::_ND_INS_CLASS::ND_INS_VPBROADCASTQ => Ok(Mnemonic::Vpbroadcastq),
            ffi::_ND_INS_CLASS::ND_INS_VPBROADCASTW => Ok(Mnemonic::Vpbroadcastw),
            ffi::_ND_INS_CLASS::ND_INS_VPCLMULQDQ => Ok(Mnemonic::Vpclmulqdq),
            ffi::_ND_INS_CLASS::ND_INS_VPCMOV => Ok(Mnemonic::Vpcmov),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPB => Ok(Mnemonic::Vpcmpb),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPD => Ok(Mnemonic::Vpcmpd),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPEQB => Ok(Mnemonic::Vpcmpeqb),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPEQD => Ok(Mnemonic::Vpcmpeqd),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPEQQ => Ok(Mnemonic::Vpcmpeqq),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPEQW => Ok(Mnemonic::Vpcmpeqw),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPESTRI => Ok(Mnemonic::Vpcmpestri),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPESTRM => Ok(Mnemonic::Vpcmpestrm),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPGTB => Ok(Mnemonic::Vpcmpgtb),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPGTD => Ok(Mnemonic::Vpcmpgtd),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPGTQ => Ok(Mnemonic::Vpcmpgtq),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPGTW => Ok(Mnemonic::Vpcmpgtw),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPISTRI => Ok(Mnemonic::Vpcmpistri),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPISTRM => Ok(Mnemonic::Vpcmpistrm),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPQ => Ok(Mnemonic::Vpcmpq),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPUB => Ok(Mnemonic::Vpcmpub),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPUD => Ok(Mnemonic::Vpcmpud),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPUQ => Ok(Mnemonic::Vpcmpuq),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPUW => Ok(Mnemonic::Vpcmpuw),
            ffi::_ND_INS_CLASS::ND_INS_VPCMPW => Ok(Mnemonic::Vpcmpw),
            ffi::_ND_INS_CLASS::ND_INS_VPCOMB => Ok(Mnemonic::Vpcomb),
            ffi::_ND_INS_CLASS::ND_INS_VPCOMD => Ok(Mnemonic::Vpcomd),
            ffi::_ND_INS_CLASS::ND_INS_VPCOMPRESSB => Ok(Mnemonic::Vpcompressb),
            ffi::_ND_INS_CLASS::ND_INS_VPCOMPRESSD => Ok(Mnemonic::Vpcompressd),
            ffi::_ND_INS_CLASS::ND_INS_VPCOMPRESSQ => Ok(Mnemonic::Vpcompressq),
            ffi::_ND_INS_CLASS::ND_INS_VPCOMPRESSW => Ok(Mnemonic::Vpcompressw),
            ffi::_ND_INS_CLASS::ND_INS_VPCOMQ => Ok(Mnemonic::Vpcomq),
            ffi::_ND_INS_CLASS::ND_INS_VPCOMUB => Ok(Mnemonic::Vpcomub),
            ffi::_ND_INS_CLASS::ND_INS_VPCOMUD => Ok(Mnemonic::Vpcomud),
            ffi::_ND_INS_CLASS::ND_INS_VPCOMUQ => Ok(Mnemonic::Vpcomuq),
            ffi::_ND_INS_CLASS::ND_INS_VPCOMUW => Ok(Mnemonic::Vpcomuw),
            ffi::_ND_INS_CLASS::ND_INS_VPCOMW => Ok(Mnemonic::Vpcomw),
            ffi::_ND_INS_CLASS::ND_INS_VPCONFLICTD => Ok(Mnemonic::Vpconflictd),
            ffi::_ND_INS_CLASS::ND_INS_VPCONFLICTQ => Ok(Mnemonic::Vpconflictq),
            ffi::_ND_INS_CLASS::ND_INS_VPDPBSSD => Ok(Mnemonic::Vpdpbssd),
            ffi::_ND_INS_CLASS::ND_INS_VPDPBSSDS => Ok(Mnemonic::Vpdpbssds),
            ffi::_ND_INS_CLASS::ND_INS_VPDPBSUD => Ok(Mnemonic::Vpdpbsud),
            ffi::_ND_INS_CLASS::ND_INS_VPDPBSUDS => Ok(Mnemonic::Vpdpbsuds),
            ffi::_ND_INS_CLASS::ND_INS_VPDPBUSD => Ok(Mnemonic::Vpdpbusd),
            ffi::_ND_INS_CLASS::ND_INS_VPDPBUSDS => Ok(Mnemonic::Vpdpbusds),
            ffi::_ND_INS_CLASS::ND_INS_VPDPBUUD => Ok(Mnemonic::Vpdpbuud),
            ffi::_ND_INS_CLASS::ND_INS_VPDPBUUDS => Ok(Mnemonic::Vpdpbuuds),
            ffi::_ND_INS_CLASS::ND_INS_VPDPWSSD => Ok(Mnemonic::Vpdpwssd),
            ffi::_ND_INS_CLASS::ND_INS_VPDPWSSDS => Ok(Mnemonic::Vpdpwssds),
            ffi::_ND_INS_CLASS::ND_INS_VPERM2F128 => Ok(Mnemonic::Vperm2f128),
            ffi::_ND_INS_CLASS::ND_INS_VPERM2I128 => Ok(Mnemonic::Vperm2i128),
            ffi::_ND_INS_CLASS::ND_INS_VPERMB => Ok(Mnemonic::Vpermb),
            ffi::_ND_INS_CLASS::ND_INS_VPERMD => Ok(Mnemonic::Vpermd),
            ffi::_ND_INS_CLASS::ND_INS_VPERMI2B => Ok(Mnemonic::Vpermi2b),
            ffi::_ND_INS_CLASS::ND_INS_VPERMI2D => Ok(Mnemonic::Vpermi2d),
            ffi::_ND_INS_CLASS::ND_INS_VPERMI2PD => Ok(Mnemonic::Vpermi2pd),
            ffi::_ND_INS_CLASS::ND_INS_VPERMI2PS => Ok(Mnemonic::Vpermi2ps),
            ffi::_ND_INS_CLASS::ND_INS_VPERMI2Q => Ok(Mnemonic::Vpermi2q),
            ffi::_ND_INS_CLASS::ND_INS_VPERMI2W => Ok(Mnemonic::Vpermi2w),
            ffi::_ND_INS_CLASS::ND_INS_VPERMIL2PD => Ok(Mnemonic::Vpermil2pd),
            ffi::_ND_INS_CLASS::ND_INS_VPERMIL2PS => Ok(Mnemonic::Vpermil2ps),
            ffi::_ND_INS_CLASS::ND_INS_VPERMILPD => Ok(Mnemonic::Vpermilpd),
            ffi::_ND_INS_CLASS::ND_INS_VPERMILPS => Ok(Mnemonic::Vpermilps),
            ffi::_ND_INS_CLASS::ND_INS_VPERMPD => Ok(Mnemonic::Vpermpd),
            ffi::_ND_INS_CLASS::ND_INS_VPERMPS => Ok(Mnemonic::Vpermps),
            ffi::_ND_INS_CLASS::ND_INS_VPERMQ => Ok(Mnemonic::Vpermq),
            ffi::_ND_INS_CLASS::ND_INS_VPERMT2B => Ok(Mnemonic::Vpermt2b),
            ffi::_ND_INS_CLASS::ND_INS_VPERMT2D => Ok(Mnemonic::Vpermt2d),
            ffi::_ND_INS_CLASS::ND_INS_VPERMT2PD => Ok(Mnemonic::Vpermt2pd),
            ffi::_ND_INS_CLASS::ND_INS_VPERMT2PS => Ok(Mnemonic::Vpermt2ps),
            ffi::_ND_INS_CLASS::ND_INS_VPERMT2Q => Ok(Mnemonic::Vpermt2q),
            ffi::_ND_INS_CLASS::ND_INS_VPERMT2W => Ok(Mnemonic::Vpermt2w),
            ffi::_ND_INS_CLASS::ND_INS_VPERMW => Ok(Mnemonic::Vpermw),
            ffi::_ND_INS_CLASS::ND_INS_VPEXPANDB => Ok(Mnemonic::Vpexpandb),
            ffi::_ND_INS_CLASS::ND_INS_VPEXPANDD => Ok(Mnemonic::Vpexpandd),
            ffi::_ND_INS_CLASS::ND_INS_VPEXPANDQ => Ok(Mnemonic::Vpexpandq),
            ffi::_ND_INS_CLASS::ND_INS_VPEXPANDW => Ok(Mnemonic::Vpexpandw),
            ffi::_ND_INS_CLASS::ND_INS_VPEXTRB => Ok(Mnemonic::Vpextrb),
            ffi::_ND_INS_CLASS::ND_INS_VPEXTRD => Ok(Mnemonic::Vpextrd),
            ffi::_ND_INS_CLASS::ND_INS_VPEXTRQ => Ok(Mnemonic::Vpextrq),
            ffi::_ND_INS_CLASS::ND_INS_VPEXTRW => Ok(Mnemonic::Vpextrw),
            ffi::_ND_INS_CLASS::ND_INS_VPGATHERDD => Ok(Mnemonic::Vpgatherdd),
            ffi::_ND_INS_CLASS::ND_INS_VPGATHERDQ => Ok(Mnemonic::Vpgatherdq),
            ffi::_ND_INS_CLASS::ND_INS_VPGATHERQD => Ok(Mnemonic::Vpgatherqd),
            ffi::_ND_INS_CLASS::ND_INS_VPGATHERQQ => Ok(Mnemonic::Vpgatherqq),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDBD => Ok(Mnemonic::Vphaddbd),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDBQ => Ok(Mnemonic::Vphaddbq),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDBW => Ok(Mnemonic::Vphaddbw),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDD => Ok(Mnemonic::Vphaddd),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDDQ => Ok(Mnemonic::Vphadddq),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDSW => Ok(Mnemonic::Vphaddsw),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDUBD => Ok(Mnemonic::Vphaddubd),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDUBQ => Ok(Mnemonic::Vphaddubq),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDUBW => Ok(Mnemonic::Vphaddubw),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDUDQ => Ok(Mnemonic::Vphaddudq),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDUWD => Ok(Mnemonic::Vphadduwd),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDUWQ => Ok(Mnemonic::Vphadduwq),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDW => Ok(Mnemonic::Vphaddw),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDWD => Ok(Mnemonic::Vphaddwd),
            ffi::_ND_INS_CLASS::ND_INS_VPHADDWQ => Ok(Mnemonic::Vphaddwq),
            ffi::_ND_INS_CLASS::ND_INS_VPHMINPOSUW => Ok(Mnemonic::Vphminposuw),
            ffi::_ND_INS_CLASS::ND_INS_VPHSUBBW => Ok(Mnemonic::Vphsubbw),
            ffi::_ND_INS_CLASS::ND_INS_VPHSUBD => Ok(Mnemonic::Vphsubd),
            ffi::_ND_INS_CLASS::ND_INS_VPHSUBDQ => Ok(Mnemonic::Vphsubdq),
            ffi::_ND_INS_CLASS::ND_INS_VPHSUBSW => Ok(Mnemonic::Vphsubsw),
            ffi::_ND_INS_CLASS::ND_INS_VPHSUBW => Ok(Mnemonic::Vphsubw),
            ffi::_ND_INS_CLASS::ND_INS_VPHSUBWD => Ok(Mnemonic::Vphsubwd),
            ffi::_ND_INS_CLASS::ND_INS_VPINSRB => Ok(Mnemonic::Vpinsrb),
            ffi::_ND_INS_CLASS::ND_INS_VPINSRD => Ok(Mnemonic::Vpinsrd),
            ffi::_ND_INS_CLASS::ND_INS_VPINSRQ => Ok(Mnemonic::Vpinsrq),
            ffi::_ND_INS_CLASS::ND_INS_VPINSRW => Ok(Mnemonic::Vpinsrw),
            ffi::_ND_INS_CLASS::ND_INS_VPLZCNTD => Ok(Mnemonic::Vplzcntd),
            ffi::_ND_INS_CLASS::ND_INS_VPLZCNTQ => Ok(Mnemonic::Vplzcntq),
            ffi::_ND_INS_CLASS::ND_INS_VPMACSDD => Ok(Mnemonic::Vpmacsdd),
            ffi::_ND_INS_CLASS::ND_INS_VPMACSDQH => Ok(Mnemonic::Vpmacsdqh),
            ffi::_ND_INS_CLASS::ND_INS_VPMACSDQL => Ok(Mnemonic::Vpmacsdql),
            ffi::_ND_INS_CLASS::ND_INS_VPMACSSDD => Ok(Mnemonic::Vpmacssdd),
            ffi::_ND_INS_CLASS::ND_INS_VPMACSSDQH => Ok(Mnemonic::Vpmacssdqh),
            ffi::_ND_INS_CLASS::ND_INS_VPMACSSDQL => Ok(Mnemonic::Vpmacssdql),
            ffi::_ND_INS_CLASS::ND_INS_VPMACSSWD => Ok(Mnemonic::Vpmacsswd),
            ffi::_ND_INS_CLASS::ND_INS_VPMACSSWW => Ok(Mnemonic::Vpmacssww),
            ffi::_ND_INS_CLASS::ND_INS_VPMACSWD => Ok(Mnemonic::Vpmacswd),
            ffi::_ND_INS_CLASS::ND_INS_VPMACSWW => Ok(Mnemonic::Vpmacsww),
            ffi::_ND_INS_CLASS::ND_INS_VPMADCSSWD => Ok(Mnemonic::Vpmadcsswd),
            ffi::_ND_INS_CLASS::ND_INS_VPMADCSWD => Ok(Mnemonic::Vpmadcswd),
            ffi::_ND_INS_CLASS::ND_INS_VPMADD52HUQ => Ok(Mnemonic::Vpmadd52huq),
            ffi::_ND_INS_CLASS::ND_INS_VPMADD52LUQ => Ok(Mnemonic::Vpmadd52luq),
            ffi::_ND_INS_CLASS::ND_INS_VPMADDUBSW => Ok(Mnemonic::Vpmaddubsw),
            ffi::_ND_INS_CLASS::ND_INS_VPMADDWD => Ok(Mnemonic::Vpmaddwd),
            ffi::_ND_INS_CLASS::ND_INS_VPMASKMOVD => Ok(Mnemonic::Vpmaskmovd),
            ffi::_ND_INS_CLASS::ND_INS_VPMASKMOVQ => Ok(Mnemonic::Vpmaskmovq),
            ffi::_ND_INS_CLASS::ND_INS_VPMAXSB => Ok(Mnemonic::Vpmaxsb),
            ffi::_ND_INS_CLASS::ND_INS_VPMAXSD => Ok(Mnemonic::Vpmaxsd),
            ffi::_ND_INS_CLASS::ND_INS_VPMAXSQ => Ok(Mnemonic::Vpmaxsq),
            ffi::_ND_INS_CLASS::ND_INS_VPMAXSW => Ok(Mnemonic::Vpmaxsw),
            ffi::_ND_INS_CLASS::ND_INS_VPMAXUB => Ok(Mnemonic::Vpmaxub),
            ffi::_ND_INS_CLASS::ND_INS_VPMAXUD => Ok(Mnemonic::Vpmaxud),
            ffi::_ND_INS_CLASS::ND_INS_VPMAXUQ => Ok(Mnemonic::Vpmaxuq),
            ffi::_ND_INS_CLASS::ND_INS_VPMAXUW => Ok(Mnemonic::Vpmaxuw),
            ffi::_ND_INS_CLASS::ND_INS_VPMINSB => Ok(Mnemonic::Vpminsb),
            ffi::_ND_INS_CLASS::ND_INS_VPMINSD => Ok(Mnemonic::Vpminsd),
            ffi::_ND_INS_CLASS::ND_INS_VPMINSQ => Ok(Mnemonic::Vpminsq),
            ffi::_ND_INS_CLASS::ND_INS_VPMINSW => Ok(Mnemonic::Vpminsw),
            ffi::_ND_INS_CLASS::ND_INS_VPMINUB => Ok(Mnemonic::Vpminub),
            ffi::_ND_INS_CLASS::ND_INS_VPMINUD => Ok(Mnemonic::Vpminud),
            ffi::_ND_INS_CLASS::ND_INS_VPMINUQ => Ok(Mnemonic::Vpminuq),
            ffi::_ND_INS_CLASS::ND_INS_VPMINUW => Ok(Mnemonic::Vpminuw),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVB2M => Ok(Mnemonic::Vpmovb2m),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVD2M => Ok(Mnemonic::Vpmovd2m),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVDB => Ok(Mnemonic::Vpmovdb),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVDW => Ok(Mnemonic::Vpmovdw),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVM2B => Ok(Mnemonic::Vpmovm2b),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVM2D => Ok(Mnemonic::Vpmovm2d),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVM2Q => Ok(Mnemonic::Vpmovm2q),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVM2W => Ok(Mnemonic::Vpmovm2w),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVMSKB => Ok(Mnemonic::Vpmovmskb),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVQ2M => Ok(Mnemonic::Vpmovq2m),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVQB => Ok(Mnemonic::Vpmovqb),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVQD => Ok(Mnemonic::Vpmovqd),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVQW => Ok(Mnemonic::Vpmovqw),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVSDB => Ok(Mnemonic::Vpmovsdb),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVSDW => Ok(Mnemonic::Vpmovsdw),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVSQB => Ok(Mnemonic::Vpmovsqb),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVSQD => Ok(Mnemonic::Vpmovsqd),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVSQW => Ok(Mnemonic::Vpmovsqw),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVSWB => Ok(Mnemonic::Vpmovswb),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVSXBD => Ok(Mnemonic::Vpmovsxbd),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVSXBQ => Ok(Mnemonic::Vpmovsxbq),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVSXBW => Ok(Mnemonic::Vpmovsxbw),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVSXDQ => Ok(Mnemonic::Vpmovsxdq),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVSXWD => Ok(Mnemonic::Vpmovsxwd),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVSXWQ => Ok(Mnemonic::Vpmovsxwq),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVUSDB => Ok(Mnemonic::Vpmovusdb),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVUSDW => Ok(Mnemonic::Vpmovusdw),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVUSQB => Ok(Mnemonic::Vpmovusqb),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVUSQD => Ok(Mnemonic::Vpmovusqd),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVUSQW => Ok(Mnemonic::Vpmovusqw),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVUSWB => Ok(Mnemonic::Vpmovuswb),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVW2M => Ok(Mnemonic::Vpmovw2m),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVWB => Ok(Mnemonic::Vpmovwb),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVZXBD => Ok(Mnemonic::Vpmovzxbd),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVZXBQ => Ok(Mnemonic::Vpmovzxbq),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVZXBW => Ok(Mnemonic::Vpmovzxbw),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVZXDQ => Ok(Mnemonic::Vpmovzxdq),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVZXWD => Ok(Mnemonic::Vpmovzxwd),
            ffi::_ND_INS_CLASS::ND_INS_VPMOVZXWQ => Ok(Mnemonic::Vpmovzxwq),
            ffi::_ND_INS_CLASS::ND_INS_VPMULDQ => Ok(Mnemonic::Vpmuldq),
            ffi::_ND_INS_CLASS::ND_INS_VPMULHRSW => Ok(Mnemonic::Vpmulhrsw),
            ffi::_ND_INS_CLASS::ND_INS_VPMULHUW => Ok(Mnemonic::Vpmulhuw),
            ffi::_ND_INS_CLASS::ND_INS_VPMULHW => Ok(Mnemonic::Vpmulhw),
            ffi::_ND_INS_CLASS::ND_INS_VPMULLD => Ok(Mnemonic::Vpmulld),
            ffi::_ND_INS_CLASS::ND_INS_VPMULLQ => Ok(Mnemonic::Vpmullq),
            ffi::_ND_INS_CLASS::ND_INS_VPMULLW => Ok(Mnemonic::Vpmullw),
            ffi::_ND_INS_CLASS::ND_INS_VPMULTISHIFTQB => Ok(Mnemonic::Vpmultishiftqb),
            ffi::_ND_INS_CLASS::ND_INS_VPMULUDQ => Ok(Mnemonic::Vpmuludq),
            ffi::_ND_INS_CLASS::ND_INS_VPOPCNTB => Ok(Mnemonic::Vpopcntb),
            ffi::_ND_INS_CLASS::ND_INS_VPOPCNTD => Ok(Mnemonic::Vpopcntd),
            ffi::_ND_INS_CLASS::ND_INS_VPOPCNTQ => Ok(Mnemonic::Vpopcntq),
            ffi::_ND_INS_CLASS::ND_INS_VPOPCNTW => Ok(Mnemonic::Vpopcntw),
            ffi::_ND_INS_CLASS::ND_INS_VPOR => Ok(Mnemonic::Vpor),
            ffi::_ND_INS_CLASS::ND_INS_VPORD => Ok(Mnemonic::Vpord),
            ffi::_ND_INS_CLASS::ND_INS_VPORQ => Ok(Mnemonic::Vporq),
            ffi::_ND_INS_CLASS::ND_INS_VPPERM => Ok(Mnemonic::Vpperm),
            ffi::_ND_INS_CLASS::ND_INS_VPROLD => Ok(Mnemonic::Vprold),
            ffi::_ND_INS_CLASS::ND_INS_VPROLQ => Ok(Mnemonic::Vprolq),
            ffi::_ND_INS_CLASS::ND_INS_VPROLVD => Ok(Mnemonic::Vprolvd),
            ffi::_ND_INS_CLASS::ND_INS_VPROLVQ => Ok(Mnemonic::Vprolvq),
            ffi::_ND_INS_CLASS::ND_INS_VPRORD => Ok(Mnemonic::Vprord),
            ffi::_ND_INS_CLASS::ND_INS_VPRORQ => Ok(Mnemonic::Vprorq),
            ffi::_ND_INS_CLASS::ND_INS_VPRORVD => Ok(Mnemonic::Vprorvd),
            ffi::_ND_INS_CLASS::ND_INS_VPRORVQ => Ok(Mnemonic::Vprorvq),
            ffi::_ND_INS_CLASS::ND_INS_VPROTB => Ok(Mnemonic::Vprotb),
            ffi::_ND_INS_CLASS::ND_INS_VPROTD => Ok(Mnemonic::Vprotd),
            ffi::_ND_INS_CLASS::ND_INS_VPROTQ => Ok(Mnemonic::Vprotq),
            ffi::_ND_INS_CLASS::ND_INS_VPROTW => Ok(Mnemonic::Vprotw),
            ffi::_ND_INS_CLASS::ND_INS_VPSADBW => Ok(Mnemonic::Vpsadbw),
            ffi::_ND_INS_CLASS::ND_INS_VPSCATTERDD => Ok(Mnemonic::Vpscatterdd),
            ffi::_ND_INS_CLASS::ND_INS_VPSCATTERDQ => Ok(Mnemonic::Vpscatterdq),
            ffi::_ND_INS_CLASS::ND_INS_VPSCATTERQD => Ok(Mnemonic::Vpscatterqd),
            ffi::_ND_INS_CLASS::ND_INS_VPSCATTERQQ => Ok(Mnemonic::Vpscatterqq),
            ffi::_ND_INS_CLASS::ND_INS_VPSHAB => Ok(Mnemonic::Vpshab),
            ffi::_ND_INS_CLASS::ND_INS_VPSHAD => Ok(Mnemonic::Vpshad),
            ffi::_ND_INS_CLASS::ND_INS_VPSHAQ => Ok(Mnemonic::Vpshaq),
            ffi::_ND_INS_CLASS::ND_INS_VPSHAW => Ok(Mnemonic::Vpshaw),
            ffi::_ND_INS_CLASS::ND_INS_VPSHLB => Ok(Mnemonic::Vpshlb),
            ffi::_ND_INS_CLASS::ND_INS_VPSHLD => Ok(Mnemonic::Vpshld),
            ffi::_ND_INS_CLASS::ND_INS_VPSHLDD => Ok(Mnemonic::Vpshldd),
            ffi::_ND_INS_CLASS::ND_INS_VPSHLDQ => Ok(Mnemonic::Vpshldq),
            ffi::_ND_INS_CLASS::ND_INS_VPSHLDVD => Ok(Mnemonic::Vpshldvd),
            ffi::_ND_INS_CLASS::ND_INS_VPSHLDVQ => Ok(Mnemonic::Vpshldvq),
            ffi::_ND_INS_CLASS::ND_INS_VPSHLDVW => Ok(Mnemonic::Vpshldvw),
            ffi::_ND_INS_CLASS::ND_INS_VPSHLDW => Ok(Mnemonic::Vpshldw),
            ffi::_ND_INS_CLASS::ND_INS_VPSHLQ => Ok(Mnemonic::Vpshlq),
            ffi::_ND_INS_CLASS::ND_INS_VPSHLW => Ok(Mnemonic::Vpshlw),
            ffi::_ND_INS_CLASS::ND_INS_VPSHRDD => Ok(Mnemonic::Vpshrdd),
            ffi::_ND_INS_CLASS::ND_INS_VPSHRDQ => Ok(Mnemonic::Vpshrdq),
            ffi::_ND_INS_CLASS::ND_INS_VPSHRDVD => Ok(Mnemonic::Vpshrdvd),
            ffi::_ND_INS_CLASS::ND_INS_VPSHRDVQ => Ok(Mnemonic::Vpshrdvq),
            ffi::_ND_INS_CLASS::ND_INS_VPSHRDVW => Ok(Mnemonic::Vpshrdvw),
            ffi::_ND_INS_CLASS::ND_INS_VPSHRDW => Ok(Mnemonic::Vpshrdw),
            ffi::_ND_INS_CLASS::ND_INS_VPSHUFB => Ok(Mnemonic::Vpshufb),
            ffi::_ND_INS_CLASS::ND_INS_VPSHUFBITQMB => Ok(Mnemonic::Vpshufbitqmb),
            ffi::_ND_INS_CLASS::ND_INS_VPSHUFD => Ok(Mnemonic::Vpshufd),
            ffi::_ND_INS_CLASS::ND_INS_VPSHUFHW => Ok(Mnemonic::Vpshufhw),
            ffi::_ND_INS_CLASS::ND_INS_VPSHUFLW => Ok(Mnemonic::Vpshuflw),
            ffi::_ND_INS_CLASS::ND_INS_VPSIGNB => Ok(Mnemonic::Vpsignb),
            ffi::_ND_INS_CLASS::ND_INS_VPSIGND => Ok(Mnemonic::Vpsignd),
            ffi::_ND_INS_CLASS::ND_INS_VPSIGNW => Ok(Mnemonic::Vpsignw),
            ffi::_ND_INS_CLASS::ND_INS_VPSLLD => Ok(Mnemonic::Vpslld),
            ffi::_ND_INS_CLASS::ND_INS_VPSLLDQ => Ok(Mnemonic::Vpslldq),
            ffi::_ND_INS_CLASS::ND_INS_VPSLLQ => Ok(Mnemonic::Vpsllq),
            ffi::_ND_INS_CLASS::ND_INS_VPSLLVD => Ok(Mnemonic::Vpsllvd),
            ffi::_ND_INS_CLASS::ND_INS_VPSLLVQ => Ok(Mnemonic::Vpsllvq),
            ffi::_ND_INS_CLASS::ND_INS_VPSLLVW => Ok(Mnemonic::Vpsllvw),
            ffi::_ND_INS_CLASS::ND_INS_VPSLLW => Ok(Mnemonic::Vpsllw),
            ffi::_ND_INS_CLASS::ND_INS_VPSRAD => Ok(Mnemonic::Vpsrad),
            ffi::_ND_INS_CLASS::ND_INS_VPSRAQ => Ok(Mnemonic::Vpsraq),
            ffi::_ND_INS_CLASS::ND_INS_VPSRAVD => Ok(Mnemonic::Vpsravd),
            ffi::_ND_INS_CLASS::ND_INS_VPSRAVQ => Ok(Mnemonic::Vpsravq),
            ffi::_ND_INS_CLASS::ND_INS_VPSRAVW => Ok(Mnemonic::Vpsravw),
            ffi::_ND_INS_CLASS::ND_INS_VPSRAW => Ok(Mnemonic::Vpsraw),
            ffi::_ND_INS_CLASS::ND_INS_VPSRLD => Ok(Mnemonic::Vpsrld),
            ffi::_ND_INS_CLASS::ND_INS_VPSRLDQ => Ok(Mnemonic::Vpsrldq),
            ffi::_ND_INS_CLASS::ND_INS_VPSRLQ => Ok(Mnemonic::Vpsrlq),
            ffi::_ND_INS_CLASS::ND_INS_VPSRLVD => Ok(Mnemonic::Vpsrlvd),
            ffi::_ND_INS_CLASS::ND_INS_VPSRLVQ => Ok(Mnemonic::Vpsrlvq),
            ffi::_ND_INS_CLASS::ND_INS_VPSRLVW => Ok(Mnemonic::Vpsrlvw),
            ffi::_ND_INS_CLASS::ND_INS_VPSRLW => Ok(Mnemonic::Vpsrlw),
            ffi::_ND_INS_CLASS::ND_INS_VPSUBB => Ok(Mnemonic::Vpsubb),
            ffi::_ND_INS_CLASS::ND_INS_VPSUBD => Ok(Mnemonic::Vpsubd),
            ffi::_ND_INS_CLASS::ND_INS_VPSUBQ => Ok(Mnemonic::Vpsubq),
            ffi::_ND_INS_CLASS::ND_INS_VPSUBSB => Ok(Mnemonic::Vpsubsb),
            ffi::_ND_INS_CLASS::ND_INS_VPSUBSW => Ok(Mnemonic::Vpsubsw),
            ffi::_ND_INS_CLASS::ND_INS_VPSUBUSB => Ok(Mnemonic::Vpsubusb),
            ffi::_ND_INS_CLASS::ND_INS_VPSUBUSW => Ok(Mnemonic::Vpsubusw),
            ffi::_ND_INS_CLASS::ND_INS_VPSUBW => Ok(Mnemonic::Vpsubw),
            ffi::_ND_INS_CLASS::ND_INS_VPTERNLOGD => Ok(Mnemonic::Vpternlogd),
            ffi::_ND_INS_CLASS::ND_INS_VPTERNLOGQ => Ok(Mnemonic::Vpternlogq),
            ffi::_ND_INS_CLASS::ND_INS_VPTEST => Ok(Mnemonic::Vptest),
            ffi::_ND_INS_CLASS::ND_INS_VPTESTMB => Ok(Mnemonic::Vptestmb),
            ffi::_ND_INS_CLASS::ND_INS_VPTESTMD => Ok(Mnemonic::Vptestmd),
            ffi::_ND_INS_CLASS::ND_INS_VPTESTMQ => Ok(Mnemonic::Vptestmq),
            ffi::_ND_INS_CLASS::ND_INS_VPTESTMW => Ok(Mnemonic::Vptestmw),
            ffi::_ND_INS_CLASS::ND_INS_VPTESTNMB => Ok(Mnemonic::Vptestnmb),
            ffi::_ND_INS_CLASS::ND_INS_VPTESTNMD => Ok(Mnemonic::Vptestnmd),
            ffi::_ND_INS_CLASS::ND_INS_VPTESTNMQ => Ok(Mnemonic::Vptestnmq),
            ffi::_ND_INS_CLASS::ND_INS_VPTESTNMW => Ok(Mnemonic::Vptestnmw),
            ffi::_ND_INS_CLASS::ND_INS_VPUNPCKHBW => Ok(Mnemonic::Vpunpckhbw),
            ffi::_ND_INS_CLASS::ND_INS_VPUNPCKHDQ => Ok(Mnemonic::Vpunpckhdq),
            ffi::_ND_INS_CLASS::ND_INS_VPUNPCKHQDQ => Ok(Mnemonic::Vpunpckhqdq),
            ffi::_ND_INS_CLASS::ND_INS_VPUNPCKHWD => Ok(Mnemonic::Vpunpckhwd),
            ffi::_ND_INS_CLASS::ND_INS_VPUNPCKLBW => Ok(Mnemonic::Vpunpcklbw),
            ffi::_ND_INS_CLASS::ND_INS_VPUNPCKLDQ => Ok(Mnemonic::Vpunpckldq),
            ffi::_ND_INS_CLASS::ND_INS_VPUNPCKLQDQ => Ok(Mnemonic::Vpunpcklqdq),
            ffi::_ND_INS_CLASS::ND_INS_VPUNPCKLWD => Ok(Mnemonic::Vpunpcklwd),
            ffi::_ND_INS_CLASS::ND_INS_VPXOR => Ok(Mnemonic::Vpxor),
            ffi::_ND_INS_CLASS::ND_INS_VPXORD => Ok(Mnemonic::Vpxord),
            ffi::_ND_INS_CLASS::ND_INS_VPXORQ => Ok(Mnemonic::Vpxorq),
            ffi::_ND_INS_CLASS::ND_INS_VRANGEPD => Ok(Mnemonic::Vrangepd),
            ffi::_ND_INS_CLASS::ND_INS_VRANGEPS => Ok(Mnemonic::Vrangeps),
            ffi::_ND_INS_CLASS::ND_INS_VRANGESD => Ok(Mnemonic::Vrangesd),
            ffi::_ND_INS_CLASS::ND_INS_VRANGESS => Ok(Mnemonic::Vrangess),
            ffi::_ND_INS_CLASS::ND_INS_VRCP14PD => Ok(Mnemonic::Vrcp14pd),
            ffi::_ND_INS_CLASS::ND_INS_VRCP14PS => Ok(Mnemonic::Vrcp14ps),
            ffi::_ND_INS_CLASS::ND_INS_VRCP14SD => Ok(Mnemonic::Vrcp14sd),
            ffi::_ND_INS_CLASS::ND_INS_VRCP14SS => Ok(Mnemonic::Vrcp14ss),
            ffi::_ND_INS_CLASS::ND_INS_VRCP28PD => Ok(Mnemonic::Vrcp28pd),
            ffi::_ND_INS_CLASS::ND_INS_VRCP28PS => Ok(Mnemonic::Vrcp28ps),
            ffi::_ND_INS_CLASS::ND_INS_VRCP28SD => Ok(Mnemonic::Vrcp28sd),
            ffi::_ND_INS_CLASS::ND_INS_VRCP28SS => Ok(Mnemonic::Vrcp28ss),
            ffi::_ND_INS_CLASS::ND_INS_VRCPPH => Ok(Mnemonic::Vrcpph),
            ffi::_ND_INS_CLASS::ND_INS_VRCPPS => Ok(Mnemonic::Vrcpps),
            ffi::_ND_INS_CLASS::ND_INS_VRCPSH => Ok(Mnemonic::Vrcpsh),
            ffi::_ND_INS_CLASS::ND_INS_VRCPSS => Ok(Mnemonic::Vrcpss),
            ffi::_ND_INS_CLASS::ND_INS_VREDUCEPD => Ok(Mnemonic::Vreducepd),
            ffi::_ND_INS_CLASS::ND_INS_VREDUCEPH => Ok(Mnemonic::Vreduceph),
            ffi::_ND_INS_CLASS::ND_INS_VREDUCEPS => Ok(Mnemonic::Vreduceps),
            ffi::_ND_INS_CLASS::ND_INS_VREDUCESD => Ok(Mnemonic::Vreducesd),
            ffi::_ND_INS_CLASS::ND_INS_VREDUCESH => Ok(Mnemonic::Vreducesh),
            ffi::_ND_INS_CLASS::ND_INS_VREDUCESS => Ok(Mnemonic::Vreducess),
            ffi::_ND_INS_CLASS::ND_INS_VRNDSCALEPD => Ok(Mnemonic::Vrndscalepd),
            ffi::_ND_INS_CLASS::ND_INS_VRNDSCALEPH => Ok(Mnemonic::Vrndscaleph),
            ffi::_ND_INS_CLASS::ND_INS_VRNDSCALEPS => Ok(Mnemonic::Vrndscaleps),
            ffi::_ND_INS_CLASS::ND_INS_VRNDSCALESD => Ok(Mnemonic::Vrndscalesd),
            ffi::_ND_INS_CLASS::ND_INS_VRNDSCALESH => Ok(Mnemonic::Vrndscalesh),
            ffi::_ND_INS_CLASS::ND_INS_VRNDSCALESS => Ok(Mnemonic::Vrndscaless),
            ffi::_ND_INS_CLASS::ND_INS_VROUNDPD => Ok(Mnemonic::Vroundpd),
            ffi::_ND_INS_CLASS::ND_INS_VROUNDPS => Ok(Mnemonic::Vroundps),
            ffi::_ND_INS_CLASS::ND_INS_VROUNDSD => Ok(Mnemonic::Vroundsd),
            ffi::_ND_INS_CLASS::ND_INS_VROUNDSS => Ok(Mnemonic::Vroundss),
            ffi::_ND_INS_CLASS::ND_INS_VRSQRT14PD => Ok(Mnemonic::Vrsqrt14pd),
            ffi::_ND_INS_CLASS::ND_INS_VRSQRT14PS => Ok(Mnemonic::Vrsqrt14ps),
            ffi::_ND_INS_CLASS::ND_INS_VRSQRT14SD => Ok(Mnemonic::Vrsqrt14sd),
            ffi::_ND_INS_CLASS::ND_INS_VRSQRT14SS => Ok(Mnemonic::Vrsqrt14ss),
            ffi::_ND_INS_CLASS::ND_INS_VRSQRT28PD => Ok(Mnemonic::Vrsqrt28pd),
            ffi::_ND_INS_CLASS::ND_INS_VRSQRT28PS => Ok(Mnemonic::Vrsqrt28ps),
            ffi::_ND_INS_CLASS::ND_INS_VRSQRT28SD => Ok(Mnemonic::Vrsqrt28sd),
            ffi::_ND_INS_CLASS::ND_INS_VRSQRT28SS => Ok(Mnemonic::Vrsqrt28ss),
            ffi::_ND_INS_CLASS::ND_INS_VRSQRTPH => Ok(Mnemonic::Vrsqrtph),
            ffi::_ND_INS_CLASS::ND_INS_VRSQRTPS => Ok(Mnemonic::Vrsqrtps),
            ffi::_ND_INS_CLASS::ND_INS_VRSQRTSH => Ok(Mnemonic::Vrsqrtsh),
            ffi::_ND_INS_CLASS::ND_INS_VRSQRTSS => Ok(Mnemonic::Vrsqrtss),
            ffi::_ND_INS_CLASS::ND_INS_VSCALEFPD => Ok(Mnemonic::Vscalefpd),
            ffi::_ND_INS_CLASS::ND_INS_VSCALEFPH => Ok(Mnemonic::Vscalefph),
            ffi::_ND_INS_CLASS::ND_INS_VSCALEFPS => Ok(Mnemonic::Vscalefps),
            ffi::_ND_INS_CLASS::ND_INS_VSCALEFSD => Ok(Mnemonic::Vscalefsd),
            ffi::_ND_INS_CLASS::ND_INS_VSCALEFSH => Ok(Mnemonic::Vscalefsh),
            ffi::_ND_INS_CLASS::ND_INS_VSCALEFSS => Ok(Mnemonic::Vscalefss),
            ffi::_ND_INS_CLASS::ND_INS_VSCATTERDPD => Ok(Mnemonic::Vscatterdpd),
            ffi::_ND_INS_CLASS::ND_INS_VSCATTERDPS => Ok(Mnemonic::Vscatterdps),
            ffi::_ND_INS_CLASS::ND_INS_VSCATTERPF0DPD => Ok(Mnemonic::Vscatterpf0dpd),
            ffi::_ND_INS_CLASS::ND_INS_VSCATTERPF0DPS => Ok(Mnemonic::Vscatterpf0dps),
            ffi::_ND_INS_CLASS::ND_INS_VSCATTERPF0QPD => Ok(Mnemonic::Vscatterpf0qpd),
            ffi::_ND_INS_CLASS::ND_INS_VSCATTERPF0QPS => Ok(Mnemonic::Vscatterpf0qps),
            ffi::_ND_INS_CLASS::ND_INS_VSCATTERPF1DPD => Ok(Mnemonic::Vscatterpf1dpd),
            ffi::_ND_INS_CLASS::ND_INS_VSCATTERPF1DPS => Ok(Mnemonic::Vscatterpf1dps),
            ffi::_ND_INS_CLASS::ND_INS_VSCATTERPF1QPD => Ok(Mnemonic::Vscatterpf1qpd),
            ffi::_ND_INS_CLASS::ND_INS_VSCATTERPF1QPS => Ok(Mnemonic::Vscatterpf1qps),
            ffi::_ND_INS_CLASS::ND_INS_VSCATTERQPD => Ok(Mnemonic::Vscatterqpd),
            ffi::_ND_INS_CLASS::ND_INS_VSCATTERQPS => Ok(Mnemonic::Vscatterqps),
            ffi::_ND_INS_CLASS::ND_INS_VSHUFF32X4 => Ok(Mnemonic::Vshuff32x4),
            ffi::_ND_INS_CLASS::ND_INS_VSHUFF64X2 => Ok(Mnemonic::Vshuff64x2),
            ffi::_ND_INS_CLASS::ND_INS_VSHUFI32X4 => Ok(Mnemonic::Vshufi32x4),
            ffi::_ND_INS_CLASS::ND_INS_VSHUFI64X2 => Ok(Mnemonic::Vshufi64x2),
            ffi::_ND_INS_CLASS::ND_INS_VSHUFPD => Ok(Mnemonic::Vshufpd),
            ffi::_ND_INS_CLASS::ND_INS_VSHUFPS => Ok(Mnemonic::Vshufps),
            ffi::_ND_INS_CLASS::ND_INS_VSQRTPD => Ok(Mnemonic::Vsqrtpd),
            ffi::_ND_INS_CLASS::ND_INS_VSQRTPH => Ok(Mnemonic::Vsqrtph),
            ffi::_ND_INS_CLASS::ND_INS_VSQRTPS => Ok(Mnemonic::Vsqrtps),
            ffi::_ND_INS_CLASS::ND_INS_VSQRTSD => Ok(Mnemonic::Vsqrtsd),
            ffi::_ND_INS_CLASS::ND_INS_VSQRTSH => Ok(Mnemonic::Vsqrtsh),
            ffi::_ND_INS_CLASS::ND_INS_VSQRTSS => Ok(Mnemonic::Vsqrtss),
            ffi::_ND_INS_CLASS::ND_INS_VSTMXCSR => Ok(Mnemonic::Vstmxcsr),
            ffi::_ND_INS_CLASS::ND_INS_VSUBPD => Ok(Mnemonic::Vsubpd),
            ffi::_ND_INS_CLASS::ND_INS_VSUBPH => Ok(Mnemonic::Vsubph),
            ffi::_ND_INS_CLASS::ND_INS_VSUBPS => Ok(Mnemonic::Vsubps),
            ffi::_ND_INS_CLASS::ND_INS_VSUBSD => Ok(Mnemonic::Vsubsd),
            ffi::_ND_INS_CLASS::ND_INS_VSUBSH => Ok(Mnemonic::Vsubsh),
            ffi::_ND_INS_CLASS::ND_INS_VSUBSS => Ok(Mnemonic::Vsubss),
            ffi::_ND_INS_CLASS::ND_INS_VTESTPD => Ok(Mnemonic::Vtestpd),
            ffi::_ND_INS_CLASS::ND_INS_VTESTPS => Ok(Mnemonic::Vtestps),
            ffi::_ND_INS_CLASS::ND_INS_VUCOMISD => Ok(Mnemonic::Vucomisd),
            ffi::_ND_INS_CLASS::ND_INS_VUCOMISH => Ok(Mnemonic::Vucomish),
            ffi::_ND_INS_CLASS::ND_INS_VUCOMISS => Ok(Mnemonic::Vucomiss),
            ffi::_ND_INS_CLASS::ND_INS_VUNPCKHPD => Ok(Mnemonic::Vunpckhpd),
            ffi::_ND_INS_CLASS::ND_INS_VUNPCKHPS => Ok(Mnemonic::Vunpckhps),
            ffi::_ND_INS_CLASS::ND_INS_VUNPCKLPD => Ok(Mnemonic::Vunpcklpd),
            ffi::_ND_INS_CLASS::ND_INS_VUNPCKLPS => Ok(Mnemonic::Vunpcklps),
            ffi::_ND_INS_CLASS::ND_INS_VXORPD => Ok(Mnemonic::Vxorpd),
            ffi::_ND_INS_CLASS::ND_INS_VXORPS => Ok(Mnemonic::Vxorps),
            ffi::_ND_INS_CLASS::ND_INS_VZEROALL => Ok(Mnemonic::Vzeroall),
            ffi::_ND_INS_CLASS::ND_INS_VZEROUPPER => Ok(Mnemonic::Vzeroupper),
            ffi::_ND_INS_CLASS::ND_INS_WAIT => Ok(Mnemonic::Wait),
            ffi::_ND_INS_CLASS::ND_INS_WBINVD => Ok(Mnemonic::Wbinvd),
            ffi::_ND_INS_CLASS::ND_INS_WBNOINVD => Ok(Mnemonic::Wbnoinvd),
            ffi::_ND_INS_CLASS::ND_INS_WRFSBASE => Ok(Mnemonic::Wrfsbase),
            ffi::_ND_INS_CLASS::ND_INS_WRGSBASE => Ok(Mnemonic::Wrgsbase),
            ffi::_ND_INS_CLASS::ND_INS_WRMSR => Ok(Mnemonic::Wrmsr),
            ffi::_ND_INS_CLASS::ND_INS_WRMSRLIST => Ok(Mnemonic::Wrmsrlist),
            ffi::_ND_INS_CLASS::ND_INS_WRMSRNS => Ok(Mnemonic::Wrmsrns),
            ffi::_ND_INS_CLASS::ND_INS_WRPKRU => Ok(Mnemonic::Wrpkru),
            ffi::_ND_INS_CLASS::ND_INS_WRSHR => Ok(Mnemonic::Wrshr),
            ffi::_ND_INS_CLASS::ND_INS_WRSS => Ok(Mnemonic::Wrss),
            ffi::_ND_INS_CLASS::ND_INS_WRUSS => Ok(Mnemonic::Wruss),
            ffi::_ND_INS_CLASS::ND_INS_XABORT => Ok(Mnemonic::Xabort),
            ffi::_ND_INS_CLASS::ND_INS_XADD => Ok(Mnemonic::Xadd),
            ffi::_ND_INS_CLASS::ND_INS_XBEGIN => Ok(Mnemonic::Xbegin),
            ffi::_ND_INS_CLASS::ND_INS_XCHG => Ok(Mnemonic::Xchg),
            ffi::_ND_INS_CLASS::ND_INS_XCRYPTCBC => Ok(Mnemonic::Xcryptcbc),
            ffi::_ND_INS_CLASS::ND_INS_XCRYPTCFB => Ok(Mnemonic::Xcryptcfb),
            ffi::_ND_INS_CLASS::ND_INS_XCRYPTCTR => Ok(Mnemonic::Xcryptctr),
            ffi::_ND_INS_CLASS::ND_INS_XCRYPTECB => Ok(Mnemonic::Xcryptecb),
            ffi::_ND_INS_CLASS::ND_INS_XCRYPTOFB => Ok(Mnemonic::Xcryptofb),
            ffi::_ND_INS_CLASS::ND_INS_XEND => Ok(Mnemonic::Xend),
            ffi::_ND_INS_CLASS::ND_INS_XGETBV => Ok(Mnemonic::Xgetbv),
            ffi::_ND_INS_CLASS::ND_INS_XLATB => Ok(Mnemonic::Xlatb),
            ffi::_ND_INS_CLASS::ND_INS_XOR => Ok(Mnemonic::Xor),
            ffi::_ND_INS_CLASS::ND_INS_XORPD => Ok(Mnemonic::Xorpd),
            ffi::_ND_INS_CLASS::ND_INS_XORPS => Ok(Mnemonic::Xorps),
            ffi::_ND_INS_CLASS::ND_INS_XRESLDTRK => Ok(Mnemonic::Xresldtrk),
            ffi::_ND_INS_CLASS::ND_INS_XRSTOR => Ok(Mnemonic::Xrstor),
            ffi::_ND_INS_CLASS::ND_INS_XRSTORS => Ok(Mnemonic::Xrstors),
            ffi::_ND_INS_CLASS::ND_INS_XSAVE => Ok(Mnemonic::Xsave),
            ffi::_ND_INS_CLASS::ND_INS_XSAVEC => Ok(Mnemonic::Xsavec),
            ffi::_ND_INS_CLASS::ND_INS_XSAVEOPT => Ok(Mnemonic::Xsaveopt),
            ffi::_ND_INS_CLASS::ND_INS_XSAVES => Ok(Mnemonic::Xsaves),
            ffi::_ND_INS_CLASS::ND_INS_XSETBV => Ok(Mnemonic::Xsetbv),
            ffi::_ND_INS_CLASS::ND_INS_XSHA1 => Ok(Mnemonic::Xsha1),
            ffi::_ND_INS_CLASS::ND_INS_XSHA256 => Ok(Mnemonic::Xsha256),
            ffi::_ND_INS_CLASS::ND_INS_XSTORE => Ok(Mnemonic::Xstore),
            ffi::_ND_INS_CLASS::ND_INS_XSUSLDTRK => Ok(Mnemonic::Xsusldtrk),
            ffi::_ND_INS_CLASS::ND_INS_XTEST => Ok(Mnemonic::Xtest),
        }
    }
}
