"use strict";
/**
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var __extends = (this && this.__extends) || (function () {
    var extendStatics = function (d, b) {
        extendStatics = Object.setPrototypeOf ||
            ({ __proto__: [] } instanceof Array && function (d, b) { d.__proto__ = b; }) ||
            function (d, b) { for (var p in b) if (Object.prototype.hasOwnProperty.call(b, p)) d[p] = b[p]; };
        return extendStatics(d, b);
    };
    return function (d, b) {
        if (typeof b !== "function" && b !== null)
            throw new TypeError("Class extends value " + String(b) + " is not a constructor or null");
        extendStatics(d, b);
        function __() { this.constructor = d; }
        d.prototype = b === null ? Object.create(b) : (__.prototype = b.prototype, new __());
    };
})();
Object.defineProperty(exports, "__esModule", { value: true });
exports.EMPTY_CONNECTION_HOLDER = exports.ReadOnlyConnectionHolder = exports.ConnectionHolder = void 0;
var error_1 = require("../error");
var util_1 = require("./util");
var constants_1 = require("./constants");
var bookmark_1 = require("./bookmark");
/**
 * Utility to lazily initialize connections and return them back to the pool when unused.
 * @private
 */
var ConnectionHolder = /** @class */ (function () {
    /**
     * @constructor
     * @param {object} params
     * @property {string} params.mode - the access mode for new connection holder.
     * @property {string} params.database - the target database name.
     * @property {Bookmark} params.bookmark - the last bookmark
     * @property {ConnectionProvider} params.connectionProvider - the connection provider to acquire connections from.
     * @property {string?} params.impersonatedUser - the user which will be impersonated
     * @property {function(databaseName:string)} params.onDatabaseNameResolved - callback called when the database name is resolved
     */
    function ConnectionHolder(_a) {
        var _b = _a === void 0 ? {} : _a, _c = _b.mode, mode = _c === void 0 ? constants_1.ACCESS_MODE_WRITE : _c, _d = _b.database, database = _d === void 0 ? '' : _d, bookmark = _b.bookmark, connectionProvider = _b.connectionProvider, impersonatedUser = _b.impersonatedUser, onDatabaseNameResolved = _b.onDatabaseNameResolved;
        this._mode = mode;
        this._database = database ? (0, util_1.assertString)(database, 'database') : '';
        this._bookmark = bookmark || bookmark_1.Bookmark.empty();
        this._connectionProvider = connectionProvider;
        this._impersonatedUser = impersonatedUser;
        this._referenceCount = 0;
        this._connectionPromise = Promise.resolve();
        this._onDatabaseNameResolved = onDatabaseNameResolved;
    }
    ConnectionHolder.prototype.mode = function () {
        return this._mode;
    };
    ConnectionHolder.prototype.database = function () {
        return this._database;
    };
    ConnectionHolder.prototype.setDatabase = function (database) {
        this._database = database;
    };
    ConnectionHolder.prototype.bookmark = function () {
        return this._bookmark;
    };
    ConnectionHolder.prototype.connectionProvider = function () {
        return this._connectionProvider;
    };
    ConnectionHolder.prototype.referenceCount = function () {
        return this._referenceCount;
    };
    ConnectionHolder.prototype.initializeConnection = function () {
        if (this._referenceCount === 0 && this._connectionProvider) {
            this._connectionPromise = this._connectionProvider.acquireConnection({
                accessMode: this._mode,
                database: this._database,
                bookmarks: this._bookmark,
                impersonatedUser: this._impersonatedUser,
                onDatabaseNameResolved: this._onDatabaseNameResolved
            });
        }
        else {
            this._referenceCount++;
            return false;
        }
        this._referenceCount++;
        return true;
    };
    ConnectionHolder.prototype.getConnection = function () {
        return this._connectionPromise;
    };
    ConnectionHolder.prototype.releaseConnection = function () {
        if (this._referenceCount === 0) {
            return this._connectionPromise;
        }
        this._referenceCount--;
        if (this._referenceCount === 0) {
            return this._releaseConnection();
        }
        return this._connectionPromise;
    };
    ConnectionHolder.prototype.close = function () {
        if (this._referenceCount === 0) {
            return this._connectionPromise;
        }
        this._referenceCount = 0;
        return this._releaseConnection();
    };
    /**
     * Return the current pooled connection instance to the connection pool.
     * We don't pool Session instances, to avoid users using the Session after they've called close.
     * The `Session` object is just a thin wrapper around Connection anyway, so it makes little difference.
     * @return {Promise} - promise resolved then connection is returned to the pool.
     * @private
     */
    ConnectionHolder.prototype._releaseConnection = function () {
        this._connectionPromise = this._connectionPromise
            .then(function (connection) {
            if (connection) {
                if (connection.isOpen()) {
                    return connection
                        .resetAndFlush()
                        .catch(ignoreError)
                        .then(function () { return connection._release(); });
                }
                return connection._release();
            }
            else {
                return Promise.resolve();
            }
        })
            .catch(ignoreError);
        return this._connectionPromise;
    };
    return ConnectionHolder;
}());
exports.ConnectionHolder = ConnectionHolder;
/**
 * Provides a interaction with a ConnectionHolder without change it state by
 * releasing or initilizing
 */
var ReadOnlyConnectionHolder = /** @class */ (function (_super) {
    __extends(ReadOnlyConnectionHolder, _super);
    /**
     * Contructor
     * @param {ConnectionHolder} connectionHolder the connection holder which will treat the requests
     */
    function ReadOnlyConnectionHolder(connectionHolder) {
        var _this = _super.call(this, {
            mode: connectionHolder.mode(),
            database: connectionHolder.database(),
            bookmark: connectionHolder.bookmark(),
            connectionProvider: connectionHolder.connectionProvider()
        }) || this;
        _this._connectionHolder = connectionHolder;
        return _this;
    }
    /**
     * Return the true if the connection is suppose to be initilized with the command.
     *
     * @return {boolean}
     */
    ReadOnlyConnectionHolder.prototype.initializeConnection = function () {
        if (this._connectionHolder.referenceCount() === 0) {
            return false;
        }
        return true;
    };
    /**
     * Get the current connection promise.
     * @return {Promise<Connection>} promise resolved with the current connection.
     */
    ReadOnlyConnectionHolder.prototype.getConnection = function () {
        return this._connectionHolder.getConnection();
    };
    /**
     * Get the current connection promise, doesn't performs the release
     * @return {Promise<Connection>} promise with the resolved current connection
     */
    ReadOnlyConnectionHolder.prototype.releaseConnection = function () {
        return this._connectionHolder.getConnection().catch(function () { return Promise.resolve(); });
    };
    /**
     * Get the current connection promise, doesn't performs the connection close
     * @return {Promise<Connection>} promise with the resolved current connection
     */
    ReadOnlyConnectionHolder.prototype.close = function () {
        return this._connectionHolder.getConnection().catch(function () { return Promise.resolve(); });
    };
    return ReadOnlyConnectionHolder;
}(ConnectionHolder));
exports.ReadOnlyConnectionHolder = ReadOnlyConnectionHolder;
exports.default = ReadOnlyConnectionHolder;
var EmptyConnectionHolder = /** @class */ (function (_super) {
    __extends(EmptyConnectionHolder, _super);
    function EmptyConnectionHolder() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    EmptyConnectionHolder.prototype.mode = function () {
        return undefined;
    };
    EmptyConnectionHolder.prototype.database = function () {
        return undefined;
    };
    EmptyConnectionHolder.prototype.initializeConnection = function () {
        // nothing to initialize
        return true;
    };
    EmptyConnectionHolder.prototype.getConnection = function () {
        return Promise.reject((0, error_1.newError)('This connection holder does not serve connections'));
    };
    EmptyConnectionHolder.prototype.releaseConnection = function () {
        return Promise.resolve();
    };
    EmptyConnectionHolder.prototype.close = function () {
        return Promise.resolve();
    };
    return EmptyConnectionHolder;
}(ConnectionHolder));
/**
 * Connection holder that does not manage any connections.
 * @type {ConnectionHolder}
 * @private
 */
var EMPTY_CONNECTION_HOLDER = new EmptyConnectionHolder();
exports.EMPTY_CONNECTION_HOLDER = EMPTY_CONNECTION_HOLDER;
// eslint-disable-next-line handle-callback-err
function ignoreError(error) { }
