"use strict";
var __awaiter = (this && this.__awaiter) || function (thisArg, _arguments, P, generator) {
    function adopt(value) { return value instanceof P ? value : new P(function (resolve) { resolve(value); }); }
    return new (P || (P = Promise))(function (resolve, reject) {
        function fulfilled(value) { try { step(generator.next(value)); } catch (e) { reject(e); } }
        function rejected(value) { try { step(generator["throw"](value)); } catch (e) { reject(e); } }
        function step(result) { result.done ? resolve(result.value) : adopt(result.value).then(fulfilled, rejected); }
        step((generator = generator.apply(thisArg, _arguments || [])).next());
    });
};
var __generator = (this && this.__generator) || function (thisArg, body) {
    var _ = { label: 0, sent: function() { if (t[0] & 1) throw t[1]; return t[1]; }, trys: [], ops: [] }, f, y, t, g;
    return g = { next: verb(0), "throw": verb(1), "return": verb(2) }, typeof Symbol === "function" && (g[Symbol.iterator] = function() { return this; }), g;
    function verb(n) { return function (v) { return step([n, v]); }; }
    function step(op) {
        if (f) throw new TypeError("Generator is already executing.");
        while (_) try {
            if (f = 1, y && (t = op[0] & 2 ? y["return"] : op[0] ? y["throw"] || ((t = y["return"]) && t.call(y), 0) : y.next) && !(t = t.call(y, op[1])).done) return t;
            if (y = 0, t) op = [op[0] & 2, t.value];
            switch (op[0]) {
                case 0: case 1: t = op; break;
                case 4: _.label++; return { value: op[1], done: false };
                case 5: _.label++; y = op[1]; op = [0]; continue;
                case 7: op = _.ops.pop(); _.trys.pop(); continue;
                default:
                    if (!(t = _.trys, t = t.length > 0 && t[t.length - 1]) && (op[0] === 6 || op[0] === 2)) { _ = 0; continue; }
                    if (op[0] === 3 && (!t || (op[1] > t[0] && op[1] < t[3]))) { _.label = op[1]; break; }
                    if (op[0] === 6 && _.label < t[1]) { _.label = t[1]; t = op; break; }
                    if (t && _.label < t[2]) { _.label = t[2]; _.ops.push(op); break; }
                    if (t[2]) _.ops.pop();
                    _.trys.pop(); continue;
            }
            op = body.call(thisArg, _);
        } catch (e) { op = [6, e]; y = 0; } finally { f = t = 0; }
        if (op[0] & 5) throw op[1]; return { value: op[0] ? op[1] : void 0, done: true };
    }
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * Copyright (c) "Neo4j"
 * Neo4j Sweden AB [http://neo4j.com]
 *
 * This file is part of Neo4j.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
var observers_1 = require("./internal/observers");
var util_1 = require("./internal/util");
var error_1 = require("./error");
var result_1 = __importDefault(require("./result"));
var transaction_1 = __importDefault(require("./transaction"));
var connection_holder_1 = require("./internal/connection-holder");
var constants_1 = require("./internal/constants");
var transaction_executor_1 = require("./internal/transaction-executor");
var bookmark_1 = require("./internal/bookmark");
var tx_config_1 = require("./internal/tx-config");
/**
 * A Session instance is used for handling the connection and
 * sending queries through the connection.
 * In a single session, multiple queries will be executed serially.
 * In order to execute parallel queries, multiple sessions are required.
 * @access public
 */
var Session = /** @class */ (function () {
    /**
     * @constructor
     * @protected
     * @param {Object} args
     * @param {string} args.mode the default access mode for this session.
     * @param {ConnectionProvider} args.connectionProvider - The connection provider to acquire connections from.
     * @param {Bookmark} args.bookmark - The initial bookmark for this session.
     * @param {string} args.database the database name
     * @param {Object} args.config={} - This driver configuration.
     * @param {boolean} args.reactive - Whether this session should create reactive streams
     * @param {number} args.fetchSize - Defines how many records is pulled in each pulling batch
     * @param {string} args.impersonatedUser - The username which the user wants to impersonate for the duration of the session.
     */
    function Session(_a) {
        var mode = _a.mode, connectionProvider = _a.connectionProvider, bookmark = _a.bookmark, database = _a.database, config = _a.config, reactive = _a.reactive, fetchSize = _a.fetchSize, impersonatedUser = _a.impersonatedUser;
        this._mode = mode;
        this._database = database;
        this._reactive = reactive;
        this._fetchSize = fetchSize;
        this._onDatabaseNameResolved = this._onDatabaseNameResolved.bind(this);
        this._readConnectionHolder = new connection_holder_1.ConnectionHolder({
            mode: constants_1.ACCESS_MODE_READ,
            database: database,
            bookmark: bookmark,
            connectionProvider: connectionProvider,
            impersonatedUser: impersonatedUser,
            onDatabaseNameResolved: this._onDatabaseNameResolved
        });
        this._writeConnectionHolder = new connection_holder_1.ConnectionHolder({
            mode: constants_1.ACCESS_MODE_WRITE,
            database: database,
            bookmark: bookmark,
            connectionProvider: connectionProvider,
            impersonatedUser: impersonatedUser,
            onDatabaseNameResolved: this._onDatabaseNameResolved
        });
        this._open = true;
        this._hasTx = false;
        this._impersonatedUser = impersonatedUser;
        this._lastBookmark = bookmark || bookmark_1.Bookmark.empty();
        this._transactionExecutor = _createTransactionExecutor(config);
        this._onComplete = this._onCompleteCallback.bind(this);
        this._databaseNameResolved = this._database !== '';
    }
    /**
     * Run Cypher query
     * Could be called with a query object i.e.: `{text: "MATCH ...", parameters: {param: 1}}`
     * or with the query and parameters as separate arguments.
     *
     * @public
     * @param {mixed} query - Cypher query to execute
     * @param {Object} parameters - Map with parameters to use in query
     * @param {TransactionConfig} [transactionConfig] - Configuration for the new auto-commit transaction.
     * @return {Result} New Result.
     */
    Session.prototype.run = function (query, parameters, transactionConfig) {
        var _this = this;
        var _a = (0, util_1.validateQueryAndParameters)(query, parameters), validatedQuery = _a.validatedQuery, params = _a.params;
        var autoCommitTxConfig = transactionConfig
            ? new tx_config_1.TxConfig(transactionConfig)
            : tx_config_1.TxConfig.empty();
        return this._run(validatedQuery, params, function (connection) {
            _this._assertSessionIsOpen();
            return connection.protocol().run(validatedQuery, params, {
                bookmark: _this._lastBookmark,
                txConfig: autoCommitTxConfig,
                mode: _this._mode,
                database: _this._database,
                impersonatedUser: _this._impersonatedUser,
                afterComplete: _this._onComplete,
                reactive: _this._reactive,
                fetchSize: _this._fetchSize
            });
        });
    };
    Session.prototype._run = function (query, parameters, customRunner) {
        var connectionHolder = this._connectionHolderWithMode(this._mode);
        var observerPromise;
        if (!this._open) {
            observerPromise = Promise.resolve(new observers_1.FailedObserver({
                error: (0, error_1.newError)('Cannot run query in a closed session.')
            }));
        }
        else if (!this._hasTx && connectionHolder.initializeConnection()) {
            observerPromise = connectionHolder
                .getConnection()
                .then(function (connection) { return customRunner(connection); })
                .catch(function (error) { return Promise.resolve(new observers_1.FailedObserver({ error: error })); });
        }
        else {
            observerPromise = Promise.resolve(new observers_1.FailedObserver({
                error: (0, error_1.newError)('Queries cannot be run directly on a ' +
                    'session with an open transaction; either run from within the ' +
                    'transaction or use a different session.')
            }));
        }
        return new result_1.default(observerPromise, query, parameters, connectionHolder);
    };
    Session.prototype._acquireConnection = function (connectionConsumer) {
        return __awaiter(this, void 0, void 0, function () {
            var promise, connectionHolder;
            var _this = this;
            return __generator(this, function (_a) {
                connectionHolder = this._connectionHolderWithMode(this._mode);
                if (!this._open) {
                    promise = Promise.reject((0, error_1.newError)('Cannot run query in a closed session.'));
                }
                else if (!this._hasTx && connectionHolder.initializeConnection()) {
                    promise = connectionHolder
                        .getConnection()
                        .then(function (connection) { return connectionConsumer(connection); })
                        .then(function (result) { return __awaiter(_this, void 0, void 0, function () {
                        return __generator(this, function (_a) {
                            switch (_a.label) {
                                case 0: return [4 /*yield*/, connectionHolder.releaseConnection()];
                                case 1:
                                    _a.sent();
                                    return [2 /*return*/, result];
                            }
                        });
                    }); });
                }
                else {
                    promise = Promise.reject((0, error_1.newError)('Queries cannot be run directly on a ' +
                        'session with an open transaction; either run from within the ' +
                        'transaction or use a different session.'));
                }
                return [2 /*return*/, promise];
            });
        });
    };
    /**
     * Begin a new transaction in this session. A session can have at most one transaction running at a time, if you
     * want to run multiple concurrent transactions, you should use multiple concurrent sessions.
     *
     * While a transaction is open the session cannot be used to run queries outside the transaction.
     *
     * @param {TransactionConfig} [transactionConfig] - Configuration for the new auto-commit transaction.
     * @returns {Transaction} New Transaction.
     */
    Session.prototype.beginTransaction = function (transactionConfig) {
        // this function needs to support bookmarks parameter for backwards compatibility
        // parameter was of type {string|string[]} and represented either a single or multiple bookmarks
        // that's why we need to check parameter type and decide how to interpret the value
        var arg = transactionConfig;
        var txConfig = tx_config_1.TxConfig.empty();
        if (arg) {
            txConfig = new tx_config_1.TxConfig(arg);
        }
        return this._beginTransaction(this._mode, txConfig);
    };
    Session.prototype._beginTransaction = function (accessMode, txConfig) {
        if (!this._open) {
            throw (0, error_1.newError)('Cannot begin a transaction on a closed session.');
        }
        if (this._hasTx) {
            throw (0, error_1.newError)('You cannot begin a transaction on a session with an open transaction; ' +
                'either run from within the transaction or use a different session.');
        }
        var mode = Session._validateSessionMode(accessMode);
        var connectionHolder = this._connectionHolderWithMode(mode);
        connectionHolder.initializeConnection();
        this._hasTx = true;
        var tx = new transaction_1.default({
            connectionHolder: connectionHolder,
            impersonatedUser: this._impersonatedUser,
            onClose: this._transactionClosed.bind(this),
            onBookmark: this._updateBookmark.bind(this),
            onConnection: this._assertSessionIsOpen.bind(this),
            reactive: this._reactive,
            fetchSize: this._fetchSize
        });
        tx._begin(this._lastBookmark, txConfig);
        return tx;
    };
    /**
     * @private
     * @returns {void}
     */
    Session.prototype._assertSessionIsOpen = function () {
        if (!this._open) {
            throw (0, error_1.newError)('You cannot run more transactions on a closed session.');
        }
    };
    /**
     * @private
     * @returns {void}
     */
    Session.prototype._transactionClosed = function () {
        this._hasTx = false;
    };
    /**
     * Return the bookmark received following the last completed {@link Transaction}.
     *
     * @return {string[]} A reference to a previous transaction.
     */
    Session.prototype.lastBookmark = function () {
        return this._lastBookmark.values();
    };
    /**
     * Execute given unit of work in a {@link READ} transaction.
     *
     * Transaction will automatically be committed unless the given function throws or returns a rejected promise.
     * Some failures of the given function or the commit itself will be retried with exponential backoff with initial
     * delay of 1 second and maximum retry time of 30 seconds. Maximum retry time is configurable via driver config's
     * `maxTransactionRetryTime` property in milliseconds.
     *
     * @param {function(tx: Transaction): Promise} transactionWork - Callback that executes operations against
     * a given {@link Transaction}.
     * @param {TransactionConfig} [transactionConfig] - Configuration for all transactions started to execute the unit of work.
     * @return {Promise} Resolved promise as returned by the given function or rejected promise when given
     * function or commit fails.
     */
    Session.prototype.readTransaction = function (transactionWork, transactionConfig) {
        var config = new tx_config_1.TxConfig(transactionConfig);
        return this._runTransaction(constants_1.ACCESS_MODE_READ, config, transactionWork);
    };
    /**
     * Execute given unit of work in a {@link WRITE} transaction.
     *
     * Transaction will automatically be committed unless the given function throws or returns a rejected promise.
     * Some failures of the given function or the commit itself will be retried with exponential backoff with initial
     * delay of 1 second and maximum retry time of 30 seconds. Maximum retry time is configurable via driver config's
     * `maxTransactionRetryTime` property in milliseconds.
     *
     * @param {function(tx: Transaction): Promise} transactionWork - Callback that executes operations against
     * a given {@link Transaction}.
     * @param {TransactionConfig} [transactionConfig] - Configuration for all transactions started to execute the unit of work.
     * @return {Promise} Resolved promise as returned by the given function or rejected promise when given
     * function or commit fails.
     */
    Session.prototype.writeTransaction = function (transactionWork, transactionConfig) {
        var config = new tx_config_1.TxConfig(transactionConfig);
        return this._runTransaction(constants_1.ACCESS_MODE_WRITE, config, transactionWork);
    };
    Session.prototype._runTransaction = function (accessMode, transactionConfig, transactionWork) {
        var _this = this;
        return this._transactionExecutor.execute(function () { return _this._beginTransaction(accessMode, transactionConfig); }, transactionWork);
    };
    /**
     * Sets the resolved database name in the session context.
     * @private
     * @param {string|undefined} database The resolved database name
     * @returns {void}
     */
    Session.prototype._onDatabaseNameResolved = function (database) {
        if (!this._databaseNameResolved) {
            var normalizedDatabase = database || '';
            this._database = normalizedDatabase;
            this._readConnectionHolder.setDatabase(normalizedDatabase);
            this._writeConnectionHolder.setDatabase(normalizedDatabase);
            this._databaseNameResolved = true;
        }
    };
    /**
     * Update value of the last bookmark.
     * @private
     * @param {Bookmark} newBookmark - The new bookmark.
     * @returns {void}
     */
    Session.prototype._updateBookmark = function (newBookmark) {
        if (newBookmark && !newBookmark.isEmpty()) {
            this._lastBookmark = newBookmark;
        }
    };
    /**
     * Close this session.
     * @return {Promise}
     */
    Session.prototype.close = function () {
        return __awaiter(this, void 0, void 0, function () {
            return __generator(this, function (_a) {
                switch (_a.label) {
                    case 0:
                        if (!this._open) return [3 /*break*/, 3];
                        this._open = false;
                        this._transactionExecutor.close();
                        return [4 /*yield*/, this._readConnectionHolder.close()];
                    case 1:
                        _a.sent();
                        return [4 /*yield*/, this._writeConnectionHolder.close()];
                    case 2:
                        _a.sent();
                        _a.label = 3;
                    case 3: return [2 /*return*/];
                }
            });
        });
    };
    Session.prototype._connectionHolderWithMode = function (mode) {
        if (mode === constants_1.ACCESS_MODE_READ) {
            return this._readConnectionHolder;
        }
        else if (mode === constants_1.ACCESS_MODE_WRITE) {
            return this._writeConnectionHolder;
        }
        else {
            throw (0, error_1.newError)('Unknown access mode: ' + mode);
        }
    };
    /**
     * @private
     * @param {Object} meta Connection metadatada
     * @returns {void}
     */
    Session.prototype._onCompleteCallback = function (meta) {
        this._updateBookmark(new bookmark_1.Bookmark(meta.bookmark));
    };
    /**
     * @protected
     */
    Session._validateSessionMode = function (rawMode) {
        var mode = rawMode || constants_1.ACCESS_MODE_WRITE;
        if (mode !== constants_1.ACCESS_MODE_READ && mode !== constants_1.ACCESS_MODE_WRITE) {
            throw (0, error_1.newError)('Illegal session mode ' + mode);
        }
        return mode;
    };
    return Session;
}());
/**
 * @private
 * @param {object} config
 * @returns {TransactionExecutor} The transaction executor
 */
function _createTransactionExecutor(config) {
    var maxRetryTimeMs = config && config.maxTransactionRetryTime
        ? config.maxTransactionRetryTime
        : null;
    return new transaction_executor_1.TransactionExecutor(maxRetryTimeMs);
}
exports.default = Session;
