/* TETRA TEA2 keystream generator implementation */
/*
 * Copyright (C) 2023 Midnight Blue B.V.
 *
 * Author: Wouter Bokslag <w.bokslag [ ] midnightblue [ ] nl>
 *
 * SPDX-License-Identifier: AGPL-3.0+
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Affero General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 * See the COPYING file in the main directory for details.
 */

#include <stdio.h>
#include <inttypes.h>
#include <string.h>

#include "tea2.h"


const uint16_t g_abTea2LutA[8] = { 0x2579, 0x86E5, 0xB6C8, 0x31D6, 0x7394, 0x934D, 0x638E, 0xC68B };
const uint16_t g_abTea2LutB[8] = { 0xD68A, 0x97A1, 0xB2C9, 0x239E, 0x9C71, 0x36E8, 0xC9B2, 0x6CD1 };
const uint8_t g_abTea2Sbox[256] = {
	0x62, 0xDA, 0xFD, 0xB6, 0xBB, 0x9C, 0xD8, 0x2A, 0xAB, 0x28, 0x6E, 0x42, 0xE7, 0x1C, 0x78, 0x9E,
	0xFC, 0xCA, 0x81, 0x8E, 0x32, 0x3B, 0xB4, 0xEF, 0x9F, 0x8B, 0xDB, 0x94, 0x0F, 0x9A, 0xA2, 0x96,
	0x1B, 0x7A, 0xFF, 0xAA, 0xC5, 0xD6, 0xBC, 0x24, 0xDF, 0x44, 0x03, 0x09, 0x0B, 0x57, 0x90, 0xBA,
	0x7F, 0x1F, 0xCF, 0x71, 0x98, 0x07, 0xF8, 0xA1, 0x60, 0xF7, 0x52, 0x8D, 0xE5, 0xD7, 0x69, 0x87,
	0x14, 0xED, 0x92, 0xEB, 0xB3, 0x2F, 0xE9, 0x3D, 0xC6, 0x50, 0x5A, 0xA7, 0x45, 0x18, 0x11, 0xC4,
	0xCE, 0xAC, 0xF4, 0x1D, 0x82, 0x54, 0x3E, 0x49, 0xD5, 0xEE, 0x84, 0x35, 0x41, 0x3A, 0xEC, 0x34,
	0x17, 0xE0, 0xC9, 0xFE, 0xE8, 0xCB, 0xE6, 0xAE, 0x68, 0xE2, 0x6B, 0x46, 0xC8, 0x47, 0xB2, 0xE3,
	0x97, 0x10, 0x0E, 0xB8, 0x76, 0x5B, 0xBE, 0xF5, 0xA6, 0x3C, 0x8F, 0xF6, 0xD1, 0xAF, 0xC0, 0x5E,
	0x7E, 0xCD, 0x7C, 0x51, 0x6D, 0x74, 0x2C, 0x16, 0xF2, 0xA5, 0x65, 0x64, 0x58, 0x72, 0x1E, 0xF1,
	0x04, 0xA8, 0x13, 0x53, 0x31, 0xB1, 0x20, 0xD3, 0x75, 0x5F, 0xA4, 0x56, 0x06, 0x8A, 0x8C, 0xD9,
	0x70, 0x12, 0x29, 0x61, 0x4F, 0x4C, 0x15, 0x05, 0xD2, 0xBD, 0x7D, 0x9B, 0x99, 0x83, 0x2B, 0x25,
	0xD0, 0x23, 0x48, 0x3F, 0xB0, 0x2E, 0x0D, 0x0C, 0xC7, 0xCC, 0xB7, 0x5C, 0xF0, 0xBF, 0x2D, 0x4E,
	0x40, 0x39, 0x9D, 0x21, 0x37, 0x77, 0x73, 0x4B, 0x4D, 0x5D, 0xFA, 0xDE, 0x00, 0x80, 0x85, 0x6F,
	0x22, 0x91, 0xDC, 0x26, 0x38, 0xE4, 0x4A, 0x79, 0x6A, 0x67, 0x93, 0xF3, 0xFB, 0x19, 0xA0, 0x7B,
	0xF9, 0x95, 0x89, 0x66, 0xB9, 0xD4, 0xC1, 0xDD, 0x63, 0x33, 0xE1, 0xC3, 0xB5, 0xA3, 0xC2, 0x27,
	0x0A, 0x88, 0xA9, 0x1A, 0x6C, 0x43, 0xEA, 0xAD, 0x30, 0x86, 0x36, 0x59, 0x08, 0x55, 0x01, 0x02
};


static uint64_t tea2_expand_iv(uint32_t dwFrameNumbers)
{
	uint32_t dwXorred = dwFrameNumbers ^ 0x5A6E3278;
	dwXorred = (dwXorred << 8) | (dwXorred >> 24);
	uint64_t qwIv = ((uint64_t)dwFrameNumbers << 32) | dwXorred;
	return (qwIv >> 8) | (qwIv << 56);
}

static uint8_t tea2_state_word_to_newbyte(uint16_t wSt, const uint16_t *awLut)
{

	uint8_t bSt0 = wSt;
	uint8_t bSt1 = wSt >> 8;
	uint8_t bDist;
	uint8_t bOut = 0;

	for (int i = 0; i < 8; i++) {
		/* taps on bit 0,2 for bSt0 and bit 0,7 for bSt1 */
		bDist = ((bSt0 >> 1) & 0x1) | ((bSt0 >> 1) & 0x2) | ((bSt1 >> 5) & 0x4) | ((bSt1 << 3) & 0x8);
		if (awLut[i] & (1 << bDist))
			bOut |= 1 << i;

		/* rotate one position */
		bSt0 = ((bSt0 >> 1) | (bSt0 << 7));
		bSt1 = ((bSt1 >> 1) | (bSt1 << 7));
	}

	return bOut;
}

static uint8_t tea2_reorder_state_byte(uint8_t bStByte)
{
	/* simple re-ordering of bits */
	uint8_t bOut = 0;
	bOut |= ((bStByte << 6) & 0x40);
	bOut |= ((bStByte << 3) & 0x10);
	bOut |= ((bStByte >> 2) & 0x01);
	bOut |= ((bStByte << 2) & 0x20);
	bOut |= ((bStByte << 3) & 0x80);
	bOut |= ((bStByte >> 4) & 0x02);
	bOut |= ((bStByte >> 3) & 0x08);
	bOut |= ((bStByte >> 5) & 0x04);
	return bOut;
}

void tea2(uint32_t dwFrameNumbers, uint8_t *lpKey, uint32_t dwNumKsBytes, uint8_t *lpKsOut)
{
	uint8_t abKeyReg[10];
	uint32_t dwNumSkipRounds = 51;

	/* init registers */
	uint64_t qwIvReg = tea2_expand_iv(dwFrameNumbers);
	memcpy(abKeyReg, lpKey, 10);

	for (int i = 0; i < dwNumKsBytes; i++) {
		for (int j = 0; j < dwNumSkipRounds; j++) {
			/* Step 1: Derive a non-linear feedback byte through sbox and feed back into key register */
			uint8_t bSboxOut = g_abTea2Sbox[abKeyReg[0] ^ abKeyReg[7]];
			memmove(abKeyReg, abKeyReg + 1, 9);
			abKeyReg[9] = bSboxOut;

			/* Step 2: Compute 3 bytes derived from current state */
			uint8_t bDerivByte01 = tea2_state_word_to_newbyte((qwIvReg >>  0) & 0xffff, g_abTea2LutA);
			uint8_t bDerivByte34 = tea2_state_word_to_newbyte((qwIvReg >> 24) & 0xffff, g_abTea2LutB);
			uint8_t bReordByte5  = tea2_reorder_state_byte((qwIvReg >> 40) & 0xff);

			/* Step 3: Combine current state with state derived values, and xor in key derived sbox output */
			uint8_t bNewByte = ((qwIvReg >> 56) ^ (qwIvReg >> 16) ^ bReordByte5 ^ bDerivByte01 ^ bSboxOut) & 0xff;
			uint8_t bMixByte = bDerivByte34;

			/* Step 4: Update lfsr: leftshift 8, feed/mix in previously generated bytes */
			qwIvReg = ((qwIvReg << 8) ^ ((uint64_t)bMixByte << 24)) | bNewByte;
		}

		lpKsOut[i] = qwIvReg >> 56;
		dwNumSkipRounds = 19;
	}
}
