import datetime as dt

import pytest
import pytz

import stix2

from .constants import VULNERABILITY_ID

EXPECTED = """{
    "type": "vulnerability",
    "spec_version": "2.1",
    "id": "vulnerability--0c7b5b88-8ff7-4a4d-aa9d-feb398cd0061",
    "created": "2016-05-12T08:17:27.000Z",
    "modified": "2016-05-12T08:17:27.000Z",
    "name": "CVE-2016-1234",
    "external_references": [
        {
            "source_name": "cve",
            "external_id": "CVE-2016-1234"
        }
    ]
}"""


def test_vulnerability_example():
    vulnerability = stix2.v21.Vulnerability(
        id=VULNERABILITY_ID,
        created="2016-05-12T08:17:27.000Z",
        modified="2016-05-12T08:17:27.000Z",
        name="CVE-2016-1234",
        external_references=[
            stix2.ExternalReference(
                source_name='cve',
                external_id="CVE-2016-1234",
            ),
        ],
    )

    assert vulnerability.serialize(pretty=True) == EXPECTED


@pytest.mark.parametrize(
    "data", [
        EXPECTED,
        {
            "created": "2016-05-12T08:17:27Z",
            "external_references": [
                {
                    "external_id": "CVE-2016-1234",
                    "source_name": "cve",
                },
            ],
            "id": VULNERABILITY_ID,
            "modified": "2016-05-12T08:17:27Z",
            "name": "CVE-2016-1234",
            "spec_version": "2.1",
            "type": "vulnerability",
        },
    ],
)
def test_parse_vulnerability(data):
    vuln = stix2.parse(data, version="2.1")

    assert vuln.type == 'vulnerability'
    assert vuln.spec_version == '2.1'
    assert vuln.id == VULNERABILITY_ID
    assert vuln.created == dt.datetime(2016, 5, 12, 8, 17, 27, tzinfo=pytz.utc)
    assert vuln.modified == dt.datetime(2016, 5, 12, 8, 17, 27, tzinfo=pytz.utc)
    assert vuln.name == "CVE-2016-1234"
    assert vuln.external_references[0].external_id == "CVE-2016-1234"
    assert vuln.external_references[0].source_name == "cve"

# TODO: Add other examples
