// SPDX-License-Identifier: Apache-2.0
// Copyright Authors of Tetragon

// Code generated by protoc-gen-go-tetragon. DO NOT EDIT

package eventchecker

import (
	list "container/list"
	json "encoding/json"
	fmt "fmt"
	tetragon "github.com/cilium/tetragon/api/v1/tetragon"
	bytesmatcher "github.com/cilium/tetragon/pkg/matchers/bytesmatcher"
	listmatcher "github.com/cilium/tetragon/pkg/matchers/listmatcher"
	stringmatcher "github.com/cilium/tetragon/pkg/matchers/stringmatcher"
	timestampmatcher "github.com/cilium/tetragon/pkg/matchers/timestampmatcher"
	logrus "github.com/sirupsen/logrus"
	yaml "sigs.k8s.io/yaml"
	strings "strings"
)

// MultiEventChecker is an interface for checking multiple Tetragon events
type MultiEventChecker interface {
	// NextEventCheck checks an event and returns a boolean value indicating
	// whether the checker has concluded, and an error indicating whether the
	// check was successful. The boolean value allows short-circuiting checks.
	//
	// Specifically:
	// (false,  nil): this event check was successful, but need to check more events
	// (false, !nil): this event check not was successful, but need to check more events
	// (true,   nil): checker was successful, no need to check more events
	// (true,  !nil): checker failed, no need to check more events
	NextEventCheck(Event, *logrus.Logger) (bool, error)

	// FinalCheck indicates that the sequence of events has ended, and
	// asks the checker to make a final decision. Once this function is
	// called, the checker is expected to return to its initial state so
	// that it can be reused. Hence, this function should only be called
	// once for each stream of events.
	FinalCheck(*logrus.Logger) error
}

// NextResponseCheck checks the next response
func NextResponseCheck(c MultiEventChecker, res *tetragon.GetEventsResponse, l *logrus.Logger) (bool, error) {
	event, err := EventFromResponse(res)
	if err != nil {
		return false, err
	}
	return c.NextEventCheck(event, l)
}

// OrderedEventChecker checks a series of events in order
type OrderedEventChecker struct {
	checks []EventChecker
	idx    int
}

// NewOrderedEventChecker creates a new OrderedEventChecker
func NewOrderedEventChecker(checks ...EventChecker) *OrderedEventChecker {
	return &OrderedEventChecker{
		checks: checks,
		idx:    0,
	}
}

// NextEventCheck implements the MultiEventChecker interface
func (checker *OrderedEventChecker) NextEventCheck(event Event, logger *logrus.Logger) (bool, error) {
	if checker.idx >= len(checker.checks) {
		return true, nil
	}

	err := checker.checks[checker.idx].CheckEvent(event)
	if err != nil {
		return false, err
	}

	checker.idx++
	if checker.idx == len(checker.checks) {
		if logger != nil {
			logger.Infof("OrderedEventChecker: all %d checks matched", len(checker.checks))
		}
		return true, nil
	}

	if logger != nil {
		logger.Infof("OrderedEventChecker: %d/%d matched", checker.idx, len(checker.checks))
	}
	return false, nil
}

// FinalCheck implements the MultiEventChecker interface
func (checker *OrderedEventChecker) FinalCheck(logger *logrus.Logger) error {
	idx := checker.idx
	checker.idx = 0

	if idx >= len(checker.checks) {
		return nil
	}

	return fmt.Errorf("OrderedEventChecker: only %d/%d matched", idx, len(checker.checks))
}

// AddChecks adds one or more checks to the end of this event checker
func (checker *OrderedEventChecker) AddChecks(checks ...EventChecker) {
	for _, check := range checks {
		checker.checks = append(checker.checks, check)
	}
}

// GetChecks returns this checker's list of checks
func (checker *OrderedEventChecker) GetChecks() []EventChecker {
	return checker.checks
}

// GetRemainingChecks returns this checker's list of remaining checks
func (checker *OrderedEventChecker) GetRemainingChecks() []EventChecker {
	return checker.checks[checker.idx:]
}

// UnorderedEventChecker checks a series of events in arbitrary order
type UnorderedEventChecker struct {
	pendingChecks *list.List
	totalChecks   int
	allChecks     *list.List
}

// NewUnorderedEventChecker creates a new UnorderedEventChecker
func NewUnorderedEventChecker(checks ...EventChecker) *UnorderedEventChecker {
	allList := list.New()
	for _, c := range checks {
		allList.PushBack(c)
	}

	pendingList := list.New()
	pendingList.PushBackList(allList)

	return &UnorderedEventChecker{
		allChecks:     allList,
		pendingChecks: pendingList,
		totalChecks:   len(checks),
	}
}

// NextEventCheck implements the MultiEventChecker interface
func (checker *UnorderedEventChecker) NextEventCheck(event Event, logger *logrus.Logger) (bool, error) {
	pending := checker.pendingChecks.Len()
	if pending == 0 {
		return true, nil
	}

	totalMatched := checker.totalChecks - pending
	if logger != nil {
		logger.Infof("UnorderedEventChecker: checking event with %d/%d total matched", totalMatched, checker.totalChecks)
	}
	idx := 1

	for e := checker.pendingChecks.Front(); e != nil; e = e.Next() {
		check := e.Value.(EventChecker)
		err := check.CheckEvent(event)
		if err == nil {
			totalMatched++
			if logger != nil {
				logger.Infof("UnorderedEventChecker: successfully matched %d/%d", totalMatched, checker.totalChecks)
			}
			checker.pendingChecks.Remove(e)
			pending--
			if pending > 0 {
				return false, nil
			}

			if logger != nil {
				logger.Infof("UnorderedEventChecker: all %d check(s) matched", checker.totalChecks)
			}
			return true, nil
		}
		if logger != nil {
			logger.Infof("UnorderedEventChecker: checking pending %d/%d: failure: %s", idx, pending, err)
		}
		idx++
	}

	return false, fmt.Errorf("UnorderedEventChecker: all %d check(s) failed", pending)
}

// FinalCheck implements the MultiEventChecker interface
func (checker *UnorderedEventChecker) FinalCheck(logger *logrus.Logger) error {
	pending := checker.pendingChecks.Len()
	total := checker.totalChecks

	checker.pendingChecks = list.New()
	checker.pendingChecks.PushBackList(checker.allChecks)
	checker.totalChecks = checker.pendingChecks.Len()

	if pending == 0 {
		return nil
	}

	return fmt.Errorf("UnorderedEventChecker: %d/%d checks remain", pending, total)
}

// AddChecks adds one or more checks to the set of checks in this event checker
func (checker *UnorderedEventChecker) AddChecks(checks ...EventChecker) {
	for _, check := range checks {
		checker.pendingChecks.PushBack(check)
		checker.allChecks.PushBack(check)
		checker.totalChecks++
	}
}

// GetChecks returns this checker's list of checks
func (checker *UnorderedEventChecker) GetChecks() []EventChecker {
	var checks []EventChecker

	for e := checker.allChecks.Front(); e != nil; e = e.Next() {
		if check, ok := e.Value.(EventChecker); ok {
			checks = append(checks, check)
		}
	}

	return checks
}

// GetRemainingChecks returns this checker's list of remaining checks
func (checker *UnorderedEventChecker) GetRemainingChecks() []EventChecker {
	var checks []EventChecker

	for e := checker.pendingChecks.Front(); e != nil; e = e.Next() {
		if check, ok := e.Value.(EventChecker); ok {
			checks = append(checks, check)
		}
	}

	return checks
}

// FnEventChecker checks a series of events using custom-defined functions for
// the MultiEventChecker implementation
type FnEventChecker struct {
	// NextCheckFn checks an event and returns a boolean value indicating
	// whether the checker has concluded, and an error indicating whether the
	// check was successful. The boolean value allows short-circuiting checks.
	//
	// Specifically:
	// (false,  nil): this event check was successful, but need to check more events
	// (false, !nil): this event check not was successful, but need to check more events
	// (true,   nil): checker was successful, no need to check more events
	// (true,  !nil): checker failed, no need to check more events
	NextCheckFn func(Event, *logrus.Logger) (bool, error)
	// FinalCheckFn indicates that the sequence of events has ended, and asks the
	// checker to make a final decision. Any cleanup should also be performed here.
	FinalCheckFn func(*logrus.Logger) error
}

// NextEventCheck implements the MultiEventChecker interface
func (checker *FnEventChecker) NextEventCheck(event Event, logger *logrus.Logger) (bool, error) {
	return checker.NextCheckFn(event, logger)
}

// FinalCheck implements the MultiEventChecker interface
func (checker *FnEventChecker) FinalCheck(logger *logrus.Logger) error {
	return checker.FinalCheckFn(logger)
}

// CheckerFromEvent converts an event into an EventChecker
func CheckerFromEvent(event Event) (EventChecker, error) {
	switch ev := event.(type) {
	case *tetragon.ProcessExec:
		return NewProcessExecChecker("").FromProcessExec(ev), nil
	case *tetragon.ProcessExit:
		return NewProcessExitChecker("").FromProcessExit(ev), nil
	case *tetragon.ProcessKprobe:
		return NewProcessKprobeChecker("").FromProcessKprobe(ev), nil
	case *tetragon.ProcessTracepoint:
		return NewProcessTracepointChecker("").FromProcessTracepoint(ev), nil
	case *tetragon.ProcessUprobe:
		return NewProcessUprobeChecker("").FromProcessUprobe(ev), nil
	case *tetragon.Test:
		return NewTestChecker("").FromTest(ev), nil
	case *tetragon.ProcessLoader:
		return NewProcessLoaderChecker("").FromProcessLoader(ev), nil
	case *tetragon.RateLimitInfo:
		return NewRateLimitInfoChecker("").FromRateLimitInfo(ev), nil

	default:
		return nil, fmt.Errorf("Unhandled event type %T", event)
	}
}

// ResponseToChecker converts a gRPC response into an EventChecker
func CheckerFromResponse(response *tetragon.GetEventsResponse) (EventChecker, error) {
	event, err := EventFromResponse(response)
	if err != nil {
		return nil, err
	}
	return CheckerFromEvent(event)
}

// CheckerLogPrefix is a helper that outputs the log prefix for an event checker,
// which is a combination of the checker type and the checker name if applicable.
func CheckerLogPrefix(checker interface{ GetCheckerType() string }) string {
	type_ := checker.GetCheckerType()

	if withName, ok := checker.(interface{ GetCheckerName() string }); ok {
		name := withName.GetCheckerName()
		if len(name) > 0 {
			return fmt.Sprintf("%s/%s", type_, name)
		}
	}

	return type_
}

// Event is an empty interface used for events like ProcessExec, etc.
type Event tetragon.Event

// EventChecker is an interface for checking a Tetragon event
type EventChecker interface {
	// CheckEvent checks a single event
	CheckEvent(Event) error
	// CheckEvent checks a single gRPC response
	CheckResponse(*tetragon.GetEventsResponse) error
}

// EventFromResponse coerces an event from a Tetragon gRPC response
func EventFromResponse(response *tetragon.GetEventsResponse) (Event, error) {
	switch ev := response.Event.(type) {
	case *tetragon.GetEventsResponse_ProcessExec:
		return ev.ProcessExec, nil
	case *tetragon.GetEventsResponse_ProcessExit:
		return ev.ProcessExit, nil
	case *tetragon.GetEventsResponse_ProcessKprobe:
		return ev.ProcessKprobe, nil
	case *tetragon.GetEventsResponse_ProcessTracepoint:
		return ev.ProcessTracepoint, nil
	case *tetragon.GetEventsResponse_ProcessUprobe:
		return ev.ProcessUprobe, nil
	case *tetragon.GetEventsResponse_Test:
		return ev.Test, nil
	case *tetragon.GetEventsResponse_ProcessLoader:
		return ev.ProcessLoader, nil
	case *tetragon.GetEventsResponse_RateLimitInfo:
		return ev.RateLimitInfo, nil

	default:
		return nil, fmt.Errorf("Unknown event type %T", response.Event)
	}
}

// ProcessExecChecker implements a checker struct to check a ProcessExec event
type ProcessExecChecker struct {
	CheckerName string              `json:"checkerName"`
	Process     *ProcessChecker     `json:"process,omitempty"`
	Parent      *ProcessChecker     `json:"parent,omitempty"`
	Ancestors   *ProcessListMatcher `json:"ancestors,omitempty"`
}

// CheckEvent checks a single event and implements the EventChecker interface
func (checker *ProcessExecChecker) CheckEvent(event Event) error {
	if ev, ok := event.(*tetragon.ProcessExec); ok {
		return checker.Check(ev)
	}
	return fmt.Errorf("%s: %T is not a ProcessExec event", CheckerLogPrefix(checker), event)
}

// CheckResponse checks a single gRPC response and implements the EventChecker interface
func (checker *ProcessExecChecker) CheckResponse(response *tetragon.GetEventsResponse) error {
	event, err := EventFromResponse(response)
	if err != nil {
		return err
	}
	return checker.CheckEvent(event)
}

// NewProcessExecChecker creates a new ProcessExecChecker
func NewProcessExecChecker(name string) *ProcessExecChecker {
	return &ProcessExecChecker{CheckerName: name}
}

// Get the name associated with the checker
func (checker *ProcessExecChecker) GetCheckerName() string {
	return checker.CheckerName
}

// Get the type of the checker as a string
func (checker *ProcessExecChecker) GetCheckerType() string {
	return "ProcessExecChecker"
}

// Check checks a ProcessExec event
func (checker *ProcessExecChecker) Check(event *tetragon.ProcessExec) error {
	if event == nil {
		return fmt.Errorf("%s: ProcessExec event is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Process != nil {
			if err := checker.Process.Check(event.Process); err != nil {
				return fmt.Errorf("Process check failed: %w", err)
			}
		}
		if checker.Parent != nil {
			if err := checker.Parent.Check(event.Parent); err != nil {
				return fmt.Errorf("Parent check failed: %w", err)
			}
		}
		if checker.Ancestors != nil {
			if err := checker.Ancestors.Check(event.Ancestors); err != nil {
				return fmt.Errorf("Ancestors check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithProcess adds a Process check to the ProcessExecChecker
func (checker *ProcessExecChecker) WithProcess(check *ProcessChecker) *ProcessExecChecker {
	checker.Process = check
	return checker
}

// WithParent adds a Parent check to the ProcessExecChecker
func (checker *ProcessExecChecker) WithParent(check *ProcessChecker) *ProcessExecChecker {
	checker.Parent = check
	return checker
}

// WithAncestors adds a Ancestors check to the ProcessExecChecker
func (checker *ProcessExecChecker) WithAncestors(check *ProcessListMatcher) *ProcessExecChecker {
	checker.Ancestors = check
	return checker
}

//FromProcessExec populates the ProcessExecChecker using data from a ProcessExec event
func (checker *ProcessExecChecker) FromProcessExec(event *tetragon.ProcessExec) *ProcessExecChecker {
	if event == nil {
		return checker
	}
	if event.Process != nil {
		checker.Process = NewProcessChecker().FromProcess(event.Process)
	}
	if event.Parent != nil {
		checker.Parent = NewProcessChecker().FromProcess(event.Parent)
	}
	{
		var checks []*ProcessChecker
		for _, check := range event.Ancestors {
			var convertedCheck *ProcessChecker
			if check != nil {
				convertedCheck = NewProcessChecker().FromProcess(check)
			}
			checks = append(checks, convertedCheck)
		}
		lm := NewProcessListMatcher().WithOperator(listmatcher.Ordered).
			WithValues(checks...)
		checker.Ancestors = lm
	}
	return checker
}

// ProcessListMatcher checks a list of *tetragon.Process fields
type ProcessListMatcher struct {
	Operator listmatcher.Operator `json:"operator"`
	Values   []*ProcessChecker    `json:"values"`
}

// NewProcessListMatcher creates a new ProcessListMatcher. The checker defaults to a subset checker unless otherwise specified using WithOperator()
func NewProcessListMatcher() *ProcessListMatcher {
	return &ProcessListMatcher{
		Operator: listmatcher.Subset,
	}
}

// WithOperator sets the match kind for the ProcessListMatcher
func (checker *ProcessListMatcher) WithOperator(operator listmatcher.Operator) *ProcessListMatcher {
	checker.Operator = operator
	return checker
}

// WithValues sets the checkers that the ProcessListMatcher should use
func (checker *ProcessListMatcher) WithValues(values ...*ProcessChecker) *ProcessListMatcher {
	checker.Values = values
	return checker
}

// Check checks a list of *tetragon.Process fields
func (checker *ProcessListMatcher) Check(values []*tetragon.Process) error {
	switch checker.Operator {
	case listmatcher.Ordered:
		return checker.orderedCheck(values)
	case listmatcher.Unordered:
		return checker.unorderedCheck(values)
	case listmatcher.Subset:
		return checker.subsetCheck(values)
	default:
		return fmt.Errorf("Unhandled ListMatcher operator %s", checker.Operator)
	}
}

// orderedCheck checks a list of ordered *tetragon.Process fields
func (checker *ProcessListMatcher) orderedCheck(values []*tetragon.Process) error {
	innerCheck := func(check *ProcessChecker, value *tetragon.Process) error {
		if err := check.Check(value); err != nil {
			return fmt.Errorf("Ancestors check failed: %w", err)
		}
		return nil
	}

	if len(checker.Values) != len(values) {
		return fmt.Errorf("ProcessListMatcher: Wanted %d elements, got %d", len(checker.Values), len(values))
	}

	for i, check := range checker.Values {
		value := values[i]
		if err := innerCheck(check, value); err != nil {
			return fmt.Errorf("ProcessListMatcher: Check failed on element %d: %w", i, err)
		}
	}

	return nil
}

// unorderedCheck checks a list of unordered *tetragon.Process fields
func (checker *ProcessListMatcher) unorderedCheck(values []*tetragon.Process) error {
	if len(checker.Values) != len(values) {
		return fmt.Errorf("ProcessListMatcher: Wanted %d elements, got %d", len(checker.Values), len(values))
	}

	return checker.subsetCheck(values)
}

// subsetCheck checks a subset of *tetragon.Process fields
func (checker *ProcessListMatcher) subsetCheck(values []*tetragon.Process) error {
	innerCheck := func(check *ProcessChecker, value *tetragon.Process) error {
		if err := check.Check(value); err != nil {
			return fmt.Errorf("Ancestors check failed: %w", err)
		}
		return nil
	}

	numDesired := len(checker.Values)
	numMatched := 0

nextCheck:
	for _, check := range checker.Values {
		for _, value := range values {
			if err := innerCheck(check, value); err == nil {
				numMatched += 1
				continue nextCheck
			}
		}
	}

	if numMatched < numDesired {
		return fmt.Errorf("ProcessListMatcher: Check failed, only matched %d elements but wanted %d", numMatched, numDesired)
	}

	return nil
}

// ProcessExitChecker implements a checker struct to check a ProcessExit event
type ProcessExitChecker struct {
	CheckerName string                             `json:"checkerName"`
	Process     *ProcessChecker                    `json:"process,omitempty"`
	Parent      *ProcessChecker                    `json:"parent,omitempty"`
	Signal      *stringmatcher.StringMatcher       `json:"signal,omitempty"`
	Status      *uint32                            `json:"status,omitempty"`
	Time        *timestampmatcher.TimestampMatcher `json:"time,omitempty"`
}

// CheckEvent checks a single event and implements the EventChecker interface
func (checker *ProcessExitChecker) CheckEvent(event Event) error {
	if ev, ok := event.(*tetragon.ProcessExit); ok {
		return checker.Check(ev)
	}
	return fmt.Errorf("%s: %T is not a ProcessExit event", CheckerLogPrefix(checker), event)
}

// CheckResponse checks a single gRPC response and implements the EventChecker interface
func (checker *ProcessExitChecker) CheckResponse(response *tetragon.GetEventsResponse) error {
	event, err := EventFromResponse(response)
	if err != nil {
		return err
	}
	return checker.CheckEvent(event)
}

// NewProcessExitChecker creates a new ProcessExitChecker
func NewProcessExitChecker(name string) *ProcessExitChecker {
	return &ProcessExitChecker{CheckerName: name}
}

// Get the name associated with the checker
func (checker *ProcessExitChecker) GetCheckerName() string {
	return checker.CheckerName
}

// Get the type of the checker as a string
func (checker *ProcessExitChecker) GetCheckerType() string {
	return "ProcessExitChecker"
}

// Check checks a ProcessExit event
func (checker *ProcessExitChecker) Check(event *tetragon.ProcessExit) error {
	if event == nil {
		return fmt.Errorf("%s: ProcessExit event is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Process != nil {
			if err := checker.Process.Check(event.Process); err != nil {
				return fmt.Errorf("Process check failed: %w", err)
			}
		}
		if checker.Parent != nil {
			if err := checker.Parent.Check(event.Parent); err != nil {
				return fmt.Errorf("Parent check failed: %w", err)
			}
		}
		if checker.Signal != nil {
			if err := checker.Signal.Match(event.Signal); err != nil {
				return fmt.Errorf("Signal check failed: %w", err)
			}
		}
		if checker.Status != nil {
			if *checker.Status != event.Status {
				return fmt.Errorf("Status has value %d which does not match expected value %d", event.Status, *checker.Status)
			}
		}
		if checker.Time != nil {
			if err := checker.Time.Match(event.Time); err != nil {
				return fmt.Errorf("Time check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithProcess adds a Process check to the ProcessExitChecker
func (checker *ProcessExitChecker) WithProcess(check *ProcessChecker) *ProcessExitChecker {
	checker.Process = check
	return checker
}

// WithParent adds a Parent check to the ProcessExitChecker
func (checker *ProcessExitChecker) WithParent(check *ProcessChecker) *ProcessExitChecker {
	checker.Parent = check
	return checker
}

// WithSignal adds a Signal check to the ProcessExitChecker
func (checker *ProcessExitChecker) WithSignal(check *stringmatcher.StringMatcher) *ProcessExitChecker {
	checker.Signal = check
	return checker
}

// WithStatus adds a Status check to the ProcessExitChecker
func (checker *ProcessExitChecker) WithStatus(check uint32) *ProcessExitChecker {
	checker.Status = &check
	return checker
}

// WithTime adds a Time check to the ProcessExitChecker
func (checker *ProcessExitChecker) WithTime(check *timestampmatcher.TimestampMatcher) *ProcessExitChecker {
	checker.Time = check
	return checker
}

//FromProcessExit populates the ProcessExitChecker using data from a ProcessExit event
func (checker *ProcessExitChecker) FromProcessExit(event *tetragon.ProcessExit) *ProcessExitChecker {
	if event == nil {
		return checker
	}
	if event.Process != nil {
		checker.Process = NewProcessChecker().FromProcess(event.Process)
	}
	if event.Parent != nil {
		checker.Parent = NewProcessChecker().FromProcess(event.Parent)
	}
	checker.Signal = stringmatcher.Full(event.Signal)
	{
		val := event.Status
		checker.Status = &val
	}
	// NB: We don't want to match timestamps for now
	checker.Time = nil
	return checker
}

// ProcessKprobeChecker implements a checker struct to check a ProcessKprobe event
type ProcessKprobeChecker struct {
	CheckerName  string                       `json:"checkerName"`
	Process      *ProcessChecker              `json:"process,omitempty"`
	Parent       *ProcessChecker              `json:"parent,omitempty"`
	FunctionName *stringmatcher.StringMatcher `json:"functionName,omitempty"`
	Args         *KprobeArgumentListMatcher   `json:"args,omitempty"`
	Return       *KprobeArgumentChecker       `json:"return,omitempty"`
	Action       *KprobeActionChecker         `json:"action,omitempty"`
}

// CheckEvent checks a single event and implements the EventChecker interface
func (checker *ProcessKprobeChecker) CheckEvent(event Event) error {
	if ev, ok := event.(*tetragon.ProcessKprobe); ok {
		return checker.Check(ev)
	}
	return fmt.Errorf("%s: %T is not a ProcessKprobe event", CheckerLogPrefix(checker), event)
}

// CheckResponse checks a single gRPC response and implements the EventChecker interface
func (checker *ProcessKprobeChecker) CheckResponse(response *tetragon.GetEventsResponse) error {
	event, err := EventFromResponse(response)
	if err != nil {
		return err
	}
	return checker.CheckEvent(event)
}

// NewProcessKprobeChecker creates a new ProcessKprobeChecker
func NewProcessKprobeChecker(name string) *ProcessKprobeChecker {
	return &ProcessKprobeChecker{CheckerName: name}
}

// Get the name associated with the checker
func (checker *ProcessKprobeChecker) GetCheckerName() string {
	return checker.CheckerName
}

// Get the type of the checker as a string
func (checker *ProcessKprobeChecker) GetCheckerType() string {
	return "ProcessKprobeChecker"
}

// Check checks a ProcessKprobe event
func (checker *ProcessKprobeChecker) Check(event *tetragon.ProcessKprobe) error {
	if event == nil {
		return fmt.Errorf("%s: ProcessKprobe event is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Process != nil {
			if err := checker.Process.Check(event.Process); err != nil {
				return fmt.Errorf("Process check failed: %w", err)
			}
		}
		if checker.Parent != nil {
			if err := checker.Parent.Check(event.Parent); err != nil {
				return fmt.Errorf("Parent check failed: %w", err)
			}
		}
		if checker.FunctionName != nil {
			if err := checker.FunctionName.Match(event.FunctionName); err != nil {
				return fmt.Errorf("FunctionName check failed: %w", err)
			}
		}
		if checker.Args != nil {
			if err := checker.Args.Check(event.Args); err != nil {
				return fmt.Errorf("Args check failed: %w", err)
			}
		}
		if checker.Return != nil {
			if err := checker.Return.Check(event.Return); err != nil {
				return fmt.Errorf("Return check failed: %w", err)
			}
		}
		if checker.Action != nil {
			if err := checker.Action.Check(&event.Action); err != nil {
				return fmt.Errorf("Action check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithProcess adds a Process check to the ProcessKprobeChecker
func (checker *ProcessKprobeChecker) WithProcess(check *ProcessChecker) *ProcessKprobeChecker {
	checker.Process = check
	return checker
}

// WithParent adds a Parent check to the ProcessKprobeChecker
func (checker *ProcessKprobeChecker) WithParent(check *ProcessChecker) *ProcessKprobeChecker {
	checker.Parent = check
	return checker
}

// WithFunctionName adds a FunctionName check to the ProcessKprobeChecker
func (checker *ProcessKprobeChecker) WithFunctionName(check *stringmatcher.StringMatcher) *ProcessKprobeChecker {
	checker.FunctionName = check
	return checker
}

// WithArgs adds a Args check to the ProcessKprobeChecker
func (checker *ProcessKprobeChecker) WithArgs(check *KprobeArgumentListMatcher) *ProcessKprobeChecker {
	checker.Args = check
	return checker
}

// WithReturn adds a Return check to the ProcessKprobeChecker
func (checker *ProcessKprobeChecker) WithReturn(check *KprobeArgumentChecker) *ProcessKprobeChecker {
	checker.Return = check
	return checker
}

// WithAction adds a Action check to the ProcessKprobeChecker
func (checker *ProcessKprobeChecker) WithAction(check tetragon.KprobeAction) *ProcessKprobeChecker {
	wrappedCheck := KprobeActionChecker(check)
	checker.Action = &wrappedCheck
	return checker
}

//FromProcessKprobe populates the ProcessKprobeChecker using data from a ProcessKprobe event
func (checker *ProcessKprobeChecker) FromProcessKprobe(event *tetragon.ProcessKprobe) *ProcessKprobeChecker {
	if event == nil {
		return checker
	}
	if event.Process != nil {
		checker.Process = NewProcessChecker().FromProcess(event.Process)
	}
	if event.Parent != nil {
		checker.Parent = NewProcessChecker().FromProcess(event.Parent)
	}
	checker.FunctionName = stringmatcher.Full(event.FunctionName)
	{
		var checks []*KprobeArgumentChecker
		for _, check := range event.Args {
			var convertedCheck *KprobeArgumentChecker
			if check != nil {
				convertedCheck = NewKprobeArgumentChecker().FromKprobeArgument(check)
			}
			checks = append(checks, convertedCheck)
		}
		lm := NewKprobeArgumentListMatcher().WithOperator(listmatcher.Ordered).
			WithValues(checks...)
		checker.Args = lm
	}
	if event.Return != nil {
		checker.Return = NewKprobeArgumentChecker().FromKprobeArgument(event.Return)
	}
	checker.Action = NewKprobeActionChecker(event.Action)
	return checker
}

// KprobeArgumentListMatcher checks a list of *tetragon.KprobeArgument fields
type KprobeArgumentListMatcher struct {
	Operator listmatcher.Operator     `json:"operator"`
	Values   []*KprobeArgumentChecker `json:"values"`
}

// NewKprobeArgumentListMatcher creates a new KprobeArgumentListMatcher. The checker defaults to a subset checker unless otherwise specified using WithOperator()
func NewKprobeArgumentListMatcher() *KprobeArgumentListMatcher {
	return &KprobeArgumentListMatcher{
		Operator: listmatcher.Subset,
	}
}

// WithOperator sets the match kind for the KprobeArgumentListMatcher
func (checker *KprobeArgumentListMatcher) WithOperator(operator listmatcher.Operator) *KprobeArgumentListMatcher {
	checker.Operator = operator
	return checker
}

// WithValues sets the checkers that the KprobeArgumentListMatcher should use
func (checker *KprobeArgumentListMatcher) WithValues(values ...*KprobeArgumentChecker) *KprobeArgumentListMatcher {
	checker.Values = values
	return checker
}

// Check checks a list of *tetragon.KprobeArgument fields
func (checker *KprobeArgumentListMatcher) Check(values []*tetragon.KprobeArgument) error {
	switch checker.Operator {
	case listmatcher.Ordered:
		return checker.orderedCheck(values)
	case listmatcher.Unordered:
		return checker.unorderedCheck(values)
	case listmatcher.Subset:
		return checker.subsetCheck(values)
	default:
		return fmt.Errorf("Unhandled ListMatcher operator %s", checker.Operator)
	}
}

// orderedCheck checks a list of ordered *tetragon.KprobeArgument fields
func (checker *KprobeArgumentListMatcher) orderedCheck(values []*tetragon.KprobeArgument) error {
	innerCheck := func(check *KprobeArgumentChecker, value *tetragon.KprobeArgument) error {
		if err := check.Check(value); err != nil {
			return fmt.Errorf("Args check failed: %w", err)
		}
		return nil
	}

	if len(checker.Values) != len(values) {
		return fmt.Errorf("KprobeArgumentListMatcher: Wanted %d elements, got %d", len(checker.Values), len(values))
	}

	for i, check := range checker.Values {
		value := values[i]
		if err := innerCheck(check, value); err != nil {
			return fmt.Errorf("KprobeArgumentListMatcher: Check failed on element %d: %w", i, err)
		}
	}

	return nil
}

// unorderedCheck checks a list of unordered *tetragon.KprobeArgument fields
func (checker *KprobeArgumentListMatcher) unorderedCheck(values []*tetragon.KprobeArgument) error {
	if len(checker.Values) != len(values) {
		return fmt.Errorf("KprobeArgumentListMatcher: Wanted %d elements, got %d", len(checker.Values), len(values))
	}

	return checker.subsetCheck(values)
}

// subsetCheck checks a subset of *tetragon.KprobeArgument fields
func (checker *KprobeArgumentListMatcher) subsetCheck(values []*tetragon.KprobeArgument) error {
	innerCheck := func(check *KprobeArgumentChecker, value *tetragon.KprobeArgument) error {
		if err := check.Check(value); err != nil {
			return fmt.Errorf("Args check failed: %w", err)
		}
		return nil
	}

	numDesired := len(checker.Values)
	numMatched := 0

nextCheck:
	for _, check := range checker.Values {
		for _, value := range values {
			if err := innerCheck(check, value); err == nil {
				numMatched += 1
				continue nextCheck
			}
		}
	}

	if numMatched < numDesired {
		return fmt.Errorf("KprobeArgumentListMatcher: Check failed, only matched %d elements but wanted %d", numMatched, numDesired)
	}

	return nil
}

// ProcessTracepointChecker implements a checker struct to check a ProcessTracepoint event
type ProcessTracepointChecker struct {
	CheckerName string                       `json:"checkerName"`
	Process     *ProcessChecker              `json:"process,omitempty"`
	Parent      *ProcessChecker              `json:"parent,omitempty"`
	Subsys      *stringmatcher.StringMatcher `json:"subsys,omitempty"`
	Event       *stringmatcher.StringMatcher `json:"event,omitempty"`
	Args        *KprobeArgumentListMatcher   `json:"args,omitempty"`
}

// CheckEvent checks a single event and implements the EventChecker interface
func (checker *ProcessTracepointChecker) CheckEvent(event Event) error {
	if ev, ok := event.(*tetragon.ProcessTracepoint); ok {
		return checker.Check(ev)
	}
	return fmt.Errorf("%s: %T is not a ProcessTracepoint event", CheckerLogPrefix(checker), event)
}

// CheckResponse checks a single gRPC response and implements the EventChecker interface
func (checker *ProcessTracepointChecker) CheckResponse(response *tetragon.GetEventsResponse) error {
	event, err := EventFromResponse(response)
	if err != nil {
		return err
	}
	return checker.CheckEvent(event)
}

// NewProcessTracepointChecker creates a new ProcessTracepointChecker
func NewProcessTracepointChecker(name string) *ProcessTracepointChecker {
	return &ProcessTracepointChecker{CheckerName: name}
}

// Get the name associated with the checker
func (checker *ProcessTracepointChecker) GetCheckerName() string {
	return checker.CheckerName
}

// Get the type of the checker as a string
func (checker *ProcessTracepointChecker) GetCheckerType() string {
	return "ProcessTracepointChecker"
}

// Check checks a ProcessTracepoint event
func (checker *ProcessTracepointChecker) Check(event *tetragon.ProcessTracepoint) error {
	if event == nil {
		return fmt.Errorf("%s: ProcessTracepoint event is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Process != nil {
			if err := checker.Process.Check(event.Process); err != nil {
				return fmt.Errorf("Process check failed: %w", err)
			}
		}
		if checker.Parent != nil {
			if err := checker.Parent.Check(event.Parent); err != nil {
				return fmt.Errorf("Parent check failed: %w", err)
			}
		}
		if checker.Subsys != nil {
			if err := checker.Subsys.Match(event.Subsys); err != nil {
				return fmt.Errorf("Subsys check failed: %w", err)
			}
		}
		if checker.Event != nil {
			if err := checker.Event.Match(event.Event); err != nil {
				return fmt.Errorf("Event check failed: %w", err)
			}
		}
		if checker.Args != nil {
			if err := checker.Args.Check(event.Args); err != nil {
				return fmt.Errorf("Args check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithProcess adds a Process check to the ProcessTracepointChecker
func (checker *ProcessTracepointChecker) WithProcess(check *ProcessChecker) *ProcessTracepointChecker {
	checker.Process = check
	return checker
}

// WithParent adds a Parent check to the ProcessTracepointChecker
func (checker *ProcessTracepointChecker) WithParent(check *ProcessChecker) *ProcessTracepointChecker {
	checker.Parent = check
	return checker
}

// WithSubsys adds a Subsys check to the ProcessTracepointChecker
func (checker *ProcessTracepointChecker) WithSubsys(check *stringmatcher.StringMatcher) *ProcessTracepointChecker {
	checker.Subsys = check
	return checker
}

// WithEvent adds a Event check to the ProcessTracepointChecker
func (checker *ProcessTracepointChecker) WithEvent(check *stringmatcher.StringMatcher) *ProcessTracepointChecker {
	checker.Event = check
	return checker
}

// WithArgs adds a Args check to the ProcessTracepointChecker
func (checker *ProcessTracepointChecker) WithArgs(check *KprobeArgumentListMatcher) *ProcessTracepointChecker {
	checker.Args = check
	return checker
}

//FromProcessTracepoint populates the ProcessTracepointChecker using data from a ProcessTracepoint event
func (checker *ProcessTracepointChecker) FromProcessTracepoint(event *tetragon.ProcessTracepoint) *ProcessTracepointChecker {
	if event == nil {
		return checker
	}
	if event.Process != nil {
		checker.Process = NewProcessChecker().FromProcess(event.Process)
	}
	if event.Parent != nil {
		checker.Parent = NewProcessChecker().FromProcess(event.Parent)
	}
	checker.Subsys = stringmatcher.Full(event.Subsys)
	checker.Event = stringmatcher.Full(event.Event)
	{
		var checks []*KprobeArgumentChecker
		for _, check := range event.Args {
			var convertedCheck *KprobeArgumentChecker
			if check != nil {
				convertedCheck = NewKprobeArgumentChecker().FromKprobeArgument(check)
			}
			checks = append(checks, convertedCheck)
		}
		lm := NewKprobeArgumentListMatcher().WithOperator(listmatcher.Ordered).
			WithValues(checks...)
		checker.Args = lm
	}
	return checker
}

// ProcessUprobeChecker implements a checker struct to check a ProcessUprobe event
type ProcessUprobeChecker struct {
	CheckerName string                       `json:"checkerName"`
	Process     *ProcessChecker              `json:"process,omitempty"`
	Parent      *ProcessChecker              `json:"parent,omitempty"`
	Path        *stringmatcher.StringMatcher `json:"path,omitempty"`
	Symbol      *stringmatcher.StringMatcher `json:"symbol,omitempty"`
}

// CheckEvent checks a single event and implements the EventChecker interface
func (checker *ProcessUprobeChecker) CheckEvent(event Event) error {
	if ev, ok := event.(*tetragon.ProcessUprobe); ok {
		return checker.Check(ev)
	}
	return fmt.Errorf("%s: %T is not a ProcessUprobe event", CheckerLogPrefix(checker), event)
}

// CheckResponse checks a single gRPC response and implements the EventChecker interface
func (checker *ProcessUprobeChecker) CheckResponse(response *tetragon.GetEventsResponse) error {
	event, err := EventFromResponse(response)
	if err != nil {
		return err
	}
	return checker.CheckEvent(event)
}

// NewProcessUprobeChecker creates a new ProcessUprobeChecker
func NewProcessUprobeChecker(name string) *ProcessUprobeChecker {
	return &ProcessUprobeChecker{CheckerName: name}
}

// Get the name associated with the checker
func (checker *ProcessUprobeChecker) GetCheckerName() string {
	return checker.CheckerName
}

// Get the type of the checker as a string
func (checker *ProcessUprobeChecker) GetCheckerType() string {
	return "ProcessUprobeChecker"
}

// Check checks a ProcessUprobe event
func (checker *ProcessUprobeChecker) Check(event *tetragon.ProcessUprobe) error {
	if event == nil {
		return fmt.Errorf("%s: ProcessUprobe event is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Process != nil {
			if err := checker.Process.Check(event.Process); err != nil {
				return fmt.Errorf("Process check failed: %w", err)
			}
		}
		if checker.Parent != nil {
			if err := checker.Parent.Check(event.Parent); err != nil {
				return fmt.Errorf("Parent check failed: %w", err)
			}
		}
		if checker.Path != nil {
			if err := checker.Path.Match(event.Path); err != nil {
				return fmt.Errorf("Path check failed: %w", err)
			}
		}
		if checker.Symbol != nil {
			if err := checker.Symbol.Match(event.Symbol); err != nil {
				return fmt.Errorf("Symbol check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithProcess adds a Process check to the ProcessUprobeChecker
func (checker *ProcessUprobeChecker) WithProcess(check *ProcessChecker) *ProcessUprobeChecker {
	checker.Process = check
	return checker
}

// WithParent adds a Parent check to the ProcessUprobeChecker
func (checker *ProcessUprobeChecker) WithParent(check *ProcessChecker) *ProcessUprobeChecker {
	checker.Parent = check
	return checker
}

// WithPath adds a Path check to the ProcessUprobeChecker
func (checker *ProcessUprobeChecker) WithPath(check *stringmatcher.StringMatcher) *ProcessUprobeChecker {
	checker.Path = check
	return checker
}

// WithSymbol adds a Symbol check to the ProcessUprobeChecker
func (checker *ProcessUprobeChecker) WithSymbol(check *stringmatcher.StringMatcher) *ProcessUprobeChecker {
	checker.Symbol = check
	return checker
}

//FromProcessUprobe populates the ProcessUprobeChecker using data from a ProcessUprobe event
func (checker *ProcessUprobeChecker) FromProcessUprobe(event *tetragon.ProcessUprobe) *ProcessUprobeChecker {
	if event == nil {
		return checker
	}
	if event.Process != nil {
		checker.Process = NewProcessChecker().FromProcess(event.Process)
	}
	if event.Parent != nil {
		checker.Parent = NewProcessChecker().FromProcess(event.Parent)
	}
	checker.Path = stringmatcher.Full(event.Path)
	checker.Symbol = stringmatcher.Full(event.Symbol)
	return checker
}

// TestChecker implements a checker struct to check a Test event
type TestChecker struct {
	CheckerName string  `json:"checkerName"`
	Arg0        *uint64 `json:"arg0,omitempty"`
	Arg1        *uint64 `json:"arg1,omitempty"`
	Arg2        *uint64 `json:"arg2,omitempty"`
	Arg3        *uint64 `json:"arg3,omitempty"`
}

// CheckEvent checks a single event and implements the EventChecker interface
func (checker *TestChecker) CheckEvent(event Event) error {
	if ev, ok := event.(*tetragon.Test); ok {
		return checker.Check(ev)
	}
	return fmt.Errorf("%s: %T is not a Test event", CheckerLogPrefix(checker), event)
}

// CheckResponse checks a single gRPC response and implements the EventChecker interface
func (checker *TestChecker) CheckResponse(response *tetragon.GetEventsResponse) error {
	event, err := EventFromResponse(response)
	if err != nil {
		return err
	}
	return checker.CheckEvent(event)
}

// NewTestChecker creates a new TestChecker
func NewTestChecker(name string) *TestChecker {
	return &TestChecker{CheckerName: name}
}

// Get the name associated with the checker
func (checker *TestChecker) GetCheckerName() string {
	return checker.CheckerName
}

// Get the type of the checker as a string
func (checker *TestChecker) GetCheckerType() string {
	return "TestChecker"
}

// Check checks a Test event
func (checker *TestChecker) Check(event *tetragon.Test) error {
	if event == nil {
		return fmt.Errorf("%s: Test event is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Arg0 != nil {
			if *checker.Arg0 != event.Arg0 {
				return fmt.Errorf("Arg0 has value %d which does not match expected value %d", event.Arg0, *checker.Arg0)
			}
		}
		if checker.Arg1 != nil {
			if *checker.Arg1 != event.Arg1 {
				return fmt.Errorf("Arg1 has value %d which does not match expected value %d", event.Arg1, *checker.Arg1)
			}
		}
		if checker.Arg2 != nil {
			if *checker.Arg2 != event.Arg2 {
				return fmt.Errorf("Arg2 has value %d which does not match expected value %d", event.Arg2, *checker.Arg2)
			}
		}
		if checker.Arg3 != nil {
			if *checker.Arg3 != event.Arg3 {
				return fmt.Errorf("Arg3 has value %d which does not match expected value %d", event.Arg3, *checker.Arg3)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithArg0 adds a Arg0 check to the TestChecker
func (checker *TestChecker) WithArg0(check uint64) *TestChecker {
	checker.Arg0 = &check
	return checker
}

// WithArg1 adds a Arg1 check to the TestChecker
func (checker *TestChecker) WithArg1(check uint64) *TestChecker {
	checker.Arg1 = &check
	return checker
}

// WithArg2 adds a Arg2 check to the TestChecker
func (checker *TestChecker) WithArg2(check uint64) *TestChecker {
	checker.Arg2 = &check
	return checker
}

// WithArg3 adds a Arg3 check to the TestChecker
func (checker *TestChecker) WithArg3(check uint64) *TestChecker {
	checker.Arg3 = &check
	return checker
}

//FromTest populates the TestChecker using data from a Test event
func (checker *TestChecker) FromTest(event *tetragon.Test) *TestChecker {
	if event == nil {
		return checker
	}
	{
		val := event.Arg0
		checker.Arg0 = &val
	}
	{
		val := event.Arg1
		checker.Arg1 = &val
	}
	{
		val := event.Arg2
		checker.Arg2 = &val
	}
	{
		val := event.Arg3
		checker.Arg3 = &val
	}
	return checker
}

// ProcessLoaderChecker implements a checker struct to check a ProcessLoader event
type ProcessLoaderChecker struct {
	CheckerName string                       `json:"checkerName"`
	Process     *ProcessChecker              `json:"process,omitempty"`
	Path        *stringmatcher.StringMatcher `json:"path,omitempty"`
	Buildid     *bytesmatcher.BytesMatcher   `json:"buildid,omitempty"`
}

// CheckEvent checks a single event and implements the EventChecker interface
func (checker *ProcessLoaderChecker) CheckEvent(event Event) error {
	if ev, ok := event.(*tetragon.ProcessLoader); ok {
		return checker.Check(ev)
	}
	return fmt.Errorf("%s: %T is not a ProcessLoader event", CheckerLogPrefix(checker), event)
}

// CheckResponse checks a single gRPC response and implements the EventChecker interface
func (checker *ProcessLoaderChecker) CheckResponse(response *tetragon.GetEventsResponse) error {
	event, err := EventFromResponse(response)
	if err != nil {
		return err
	}
	return checker.CheckEvent(event)
}

// NewProcessLoaderChecker creates a new ProcessLoaderChecker
func NewProcessLoaderChecker(name string) *ProcessLoaderChecker {
	return &ProcessLoaderChecker{CheckerName: name}
}

// Get the name associated with the checker
func (checker *ProcessLoaderChecker) GetCheckerName() string {
	return checker.CheckerName
}

// Get the type of the checker as a string
func (checker *ProcessLoaderChecker) GetCheckerType() string {
	return "ProcessLoaderChecker"
}

// Check checks a ProcessLoader event
func (checker *ProcessLoaderChecker) Check(event *tetragon.ProcessLoader) error {
	if event == nil {
		return fmt.Errorf("%s: ProcessLoader event is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Process != nil {
			if err := checker.Process.Check(event.Process); err != nil {
				return fmt.Errorf("Process check failed: %w", err)
			}
		}
		if checker.Path != nil {
			if err := checker.Path.Match(event.Path); err != nil {
				return fmt.Errorf("Path check failed: %w", err)
			}
		}
		if checker.Buildid != nil {
			if err := checker.Buildid.Match(event.Buildid); err != nil {
				return fmt.Errorf("Buildid check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithProcess adds a Process check to the ProcessLoaderChecker
func (checker *ProcessLoaderChecker) WithProcess(check *ProcessChecker) *ProcessLoaderChecker {
	checker.Process = check
	return checker
}

// WithPath adds a Path check to the ProcessLoaderChecker
func (checker *ProcessLoaderChecker) WithPath(check *stringmatcher.StringMatcher) *ProcessLoaderChecker {
	checker.Path = check
	return checker
}

// WithBuildid adds a Buildid check to the ProcessLoaderChecker
func (checker *ProcessLoaderChecker) WithBuildid(check *bytesmatcher.BytesMatcher) *ProcessLoaderChecker {
	checker.Buildid = check
	return checker
}

//FromProcessLoader populates the ProcessLoaderChecker using data from a ProcessLoader event
func (checker *ProcessLoaderChecker) FromProcessLoader(event *tetragon.ProcessLoader) *ProcessLoaderChecker {
	if event == nil {
		return checker
	}
	if event.Process != nil {
		checker.Process = NewProcessChecker().FromProcess(event.Process)
	}
	checker.Path = stringmatcher.Full(event.Path)
	checker.Buildid = bytesmatcher.Full(event.Buildid)
	return checker
}

// RateLimitInfoChecker implements a checker struct to check a RateLimitInfo event
type RateLimitInfoChecker struct {
	CheckerName                  string  `json:"checkerName"`
	NumberOfDroppedProcessEvents *uint64 `json:"numberOfDroppedProcessEvents,omitempty"`
}

// CheckEvent checks a single event and implements the EventChecker interface
func (checker *RateLimitInfoChecker) CheckEvent(event Event) error {
	if ev, ok := event.(*tetragon.RateLimitInfo); ok {
		return checker.Check(ev)
	}
	return fmt.Errorf("%s: %T is not a RateLimitInfo event", CheckerLogPrefix(checker), event)
}

// CheckResponse checks a single gRPC response and implements the EventChecker interface
func (checker *RateLimitInfoChecker) CheckResponse(response *tetragon.GetEventsResponse) error {
	event, err := EventFromResponse(response)
	if err != nil {
		return err
	}
	return checker.CheckEvent(event)
}

// NewRateLimitInfoChecker creates a new RateLimitInfoChecker
func NewRateLimitInfoChecker(name string) *RateLimitInfoChecker {
	return &RateLimitInfoChecker{CheckerName: name}
}

// Get the name associated with the checker
func (checker *RateLimitInfoChecker) GetCheckerName() string {
	return checker.CheckerName
}

// Get the type of the checker as a string
func (checker *RateLimitInfoChecker) GetCheckerType() string {
	return "RateLimitInfoChecker"
}

// Check checks a RateLimitInfo event
func (checker *RateLimitInfoChecker) Check(event *tetragon.RateLimitInfo) error {
	if event == nil {
		return fmt.Errorf("%s: RateLimitInfo event is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.NumberOfDroppedProcessEvents != nil {
			if *checker.NumberOfDroppedProcessEvents != event.NumberOfDroppedProcessEvents {
				return fmt.Errorf("NumberOfDroppedProcessEvents has value %d which does not match expected value %d", event.NumberOfDroppedProcessEvents, *checker.NumberOfDroppedProcessEvents)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithNumberOfDroppedProcessEvents adds a NumberOfDroppedProcessEvents check to the RateLimitInfoChecker
func (checker *RateLimitInfoChecker) WithNumberOfDroppedProcessEvents(check uint64) *RateLimitInfoChecker {
	checker.NumberOfDroppedProcessEvents = &check
	return checker
}

//FromRateLimitInfo populates the RateLimitInfoChecker using data from a RateLimitInfo event
func (checker *RateLimitInfoChecker) FromRateLimitInfo(event *tetragon.RateLimitInfo) *RateLimitInfoChecker {
	if event == nil {
		return checker
	}
	{
		val := event.NumberOfDroppedProcessEvents
		checker.NumberOfDroppedProcessEvents = &val
	}
	return checker
}

// ImageChecker implements a checker struct to check a Image field
type ImageChecker struct {
	Id   *stringmatcher.StringMatcher `json:"id,omitempty"`
	Name *stringmatcher.StringMatcher `json:"name,omitempty"`
}

// NewImageChecker creates a new ImageChecker
func NewImageChecker() *ImageChecker {
	return &ImageChecker{}
}

// Get the type of the checker as a string
func (checker *ImageChecker) GetCheckerType() string {
	return "ImageChecker"
}

// Check checks a Image field
func (checker *ImageChecker) Check(event *tetragon.Image) error {
	if event == nil {
		return fmt.Errorf("%s: Image field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Id != nil {
			if err := checker.Id.Match(event.Id); err != nil {
				return fmt.Errorf("Id check failed: %w", err)
			}
		}
		if checker.Name != nil {
			if err := checker.Name.Match(event.Name); err != nil {
				return fmt.Errorf("Name check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithId adds a Id check to the ImageChecker
func (checker *ImageChecker) WithId(check *stringmatcher.StringMatcher) *ImageChecker {
	checker.Id = check
	return checker
}

// WithName adds a Name check to the ImageChecker
func (checker *ImageChecker) WithName(check *stringmatcher.StringMatcher) *ImageChecker {
	checker.Name = check
	return checker
}

//FromImage populates the ImageChecker using data from a Image field
func (checker *ImageChecker) FromImage(event *tetragon.Image) *ImageChecker {
	if event == nil {
		return checker
	}
	checker.Id = stringmatcher.Full(event.Id)
	checker.Name = stringmatcher.Full(event.Name)
	return checker
}

// ContainerChecker implements a checker struct to check a Container field
type ContainerChecker struct {
	Id             *stringmatcher.StringMatcher       `json:"id,omitempty"`
	Name           *stringmatcher.StringMatcher       `json:"name,omitempty"`
	Image          *ImageChecker                      `json:"image,omitempty"`
	StartTime      *timestampmatcher.TimestampMatcher `json:"startTime,omitempty"`
	Pid            *uint32                            `json:"pid,omitempty"`
	MaybeExecProbe *bool                              `json:"maybeExecProbe,omitempty"`
}

// NewContainerChecker creates a new ContainerChecker
func NewContainerChecker() *ContainerChecker {
	return &ContainerChecker{}
}

// Get the type of the checker as a string
func (checker *ContainerChecker) GetCheckerType() string {
	return "ContainerChecker"
}

// Check checks a Container field
func (checker *ContainerChecker) Check(event *tetragon.Container) error {
	if event == nil {
		return fmt.Errorf("%s: Container field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Id != nil {
			if err := checker.Id.Match(event.Id); err != nil {
				return fmt.Errorf("Id check failed: %w", err)
			}
		}
		if checker.Name != nil {
			if err := checker.Name.Match(event.Name); err != nil {
				return fmt.Errorf("Name check failed: %w", err)
			}
		}
		if checker.Image != nil {
			if err := checker.Image.Check(event.Image); err != nil {
				return fmt.Errorf("Image check failed: %w", err)
			}
		}
		if checker.StartTime != nil {
			if err := checker.StartTime.Match(event.StartTime); err != nil {
				return fmt.Errorf("StartTime check failed: %w", err)
			}
		}
		if checker.Pid != nil {
			if event.Pid == nil {
				return fmt.Errorf("Pid is nil and does not match expected value %v", *checker.Pid)
			}
			if *checker.Pid != event.Pid.Value {
				return fmt.Errorf("Pid has value %v which does not match expected value %v", event.Pid.Value, *checker.Pid)
			}
		}
		if checker.MaybeExecProbe != nil {
			if *checker.MaybeExecProbe != event.MaybeExecProbe {
				return fmt.Errorf("MaybeExecProbe has value %t which does not match expected value %t", event.MaybeExecProbe, *checker.MaybeExecProbe)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithId adds a Id check to the ContainerChecker
func (checker *ContainerChecker) WithId(check *stringmatcher.StringMatcher) *ContainerChecker {
	checker.Id = check
	return checker
}

// WithName adds a Name check to the ContainerChecker
func (checker *ContainerChecker) WithName(check *stringmatcher.StringMatcher) *ContainerChecker {
	checker.Name = check
	return checker
}

// WithImage adds a Image check to the ContainerChecker
func (checker *ContainerChecker) WithImage(check *ImageChecker) *ContainerChecker {
	checker.Image = check
	return checker
}

// WithStartTime adds a StartTime check to the ContainerChecker
func (checker *ContainerChecker) WithStartTime(check *timestampmatcher.TimestampMatcher) *ContainerChecker {
	checker.StartTime = check
	return checker
}

// WithPid adds a Pid check to the ContainerChecker
func (checker *ContainerChecker) WithPid(check uint32) *ContainerChecker {
	checker.Pid = &check
	return checker
}

// WithMaybeExecProbe adds a MaybeExecProbe check to the ContainerChecker
func (checker *ContainerChecker) WithMaybeExecProbe(check bool) *ContainerChecker {
	checker.MaybeExecProbe = &check
	return checker
}

//FromContainer populates the ContainerChecker using data from a Container field
func (checker *ContainerChecker) FromContainer(event *tetragon.Container) *ContainerChecker {
	if event == nil {
		return checker
	}
	checker.Id = stringmatcher.Full(event.Id)
	checker.Name = stringmatcher.Full(event.Name)
	if event.Image != nil {
		checker.Image = NewImageChecker().FromImage(event.Image)
	}
	// NB: We don't want to match timestamps for now
	checker.StartTime = nil
	if event.Pid != nil {
		val := event.Pid.Value
		checker.Pid = &val
	}
	{
		val := event.MaybeExecProbe
		checker.MaybeExecProbe = &val
	}
	return checker
}

// PodChecker implements a checker struct to check a Pod field
type PodChecker struct {
	Namespace *stringmatcher.StringMatcher           `json:"namespace,omitempty"`
	Name      *stringmatcher.StringMatcher           `json:"name,omitempty"`
	Labels    map[string]stringmatcher.StringMatcher `json:"labels,omitempty"`
	Container *ContainerChecker                      `json:"container,omitempty"`
	PodLabels map[string]stringmatcher.StringMatcher `json:"podLabels,omitempty"`
}

// NewPodChecker creates a new PodChecker
func NewPodChecker() *PodChecker {
	return &PodChecker{}
}

// Get the type of the checker as a string
func (checker *PodChecker) GetCheckerType() string {
	return "PodChecker"
}

// Check checks a Pod field
func (checker *PodChecker) Check(event *tetragon.Pod) error {
	if event == nil {
		return fmt.Errorf("%s: Pod field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Namespace != nil {
			if err := checker.Namespace.Match(event.Namespace); err != nil {
				return fmt.Errorf("Namespace check failed: %w", err)
			}
		}
		if checker.Name != nil {
			if err := checker.Name.Match(event.Name); err != nil {
				return fmt.Errorf("Name check failed: %w", err)
			}
		}
		{
			values := make(map[string]string)
			for _, s := range event.Labels {
				// Split out key,value pair
				kv := strings.SplitN(s, "=", 2)
				if len(kv) != 2 {
					// If we wanted to match an invalid label, error out
					if _, ok := checker.Labels[s]; ok {
						return fmt.Errorf("PodChecker: Label %s is in an invalid format (want key=value)", s)
					}
					continue
				}
				values[kv[0]] = kv[1]
			}
			var unmatched []string
			matched := make(map[string]struct{})
			for key, value := range values {
				if len(checker.Labels) > 0 {
					// Attempt to grab the matcher for this key
					if matcher, ok := checker.Labels[key]; ok {
						if err := matcher.Match(value); err != nil {
							return fmt.Errorf("Labels[%s] (%s=%s) check failed: %w", key, key, value, err)
						}
						matched[key] = struct{}{}
					}
				}
			}

			// See if we have any unmatched values that we wanted to match
			if len(matched) != len(checker.Labels) {
				for k := range checker.Labels {
					if _, ok := matched[k]; !ok {
						unmatched = append(unmatched, k)
					}
				}
				return fmt.Errorf("Labels unmatched: %v", unmatched)
			}
		}
		if checker.Container != nil {
			if err := checker.Container.Check(event.Container); err != nil {
				return fmt.Errorf("Container check failed: %w", err)
			}
		}
		{
			var unmatched []string
			matched := make(map[string]struct{})
			for key, value := range event.PodLabels {
				if len(checker.PodLabels) > 0 {
					// Attempt to grab the matcher for this key
					if matcher, ok := checker.PodLabels[key]; ok {
						if err := matcher.Match(value); err != nil {
							return fmt.Errorf("PodLabels[%s] (%s=%s) check failed: %w", key, key, value, err)
						}
						matched[key] = struct{}{}
					}
				}
			}

			// See if we have any unmatched values that we wanted to match
			if len(matched) != len(checker.PodLabels) {
				for k := range checker.PodLabels {
					if _, ok := matched[k]; !ok {
						unmatched = append(unmatched, k)
					}
				}
				return fmt.Errorf("PodLabels unmatched: %v", unmatched)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithNamespace adds a Namespace check to the PodChecker
func (checker *PodChecker) WithNamespace(check *stringmatcher.StringMatcher) *PodChecker {
	checker.Namespace = check
	return checker
}

// WithName adds a Name check to the PodChecker
func (checker *PodChecker) WithName(check *stringmatcher.StringMatcher) *PodChecker {
	checker.Name = check
	return checker
}

// WithLabels adds a Labels check to the PodChecker
func (checker *PodChecker) WithLabels(check map[string]stringmatcher.StringMatcher) *PodChecker {
	checker.Labels = check
	return checker
}

// WithContainer adds a Container check to the PodChecker
func (checker *PodChecker) WithContainer(check *ContainerChecker) *PodChecker {
	checker.Container = check
	return checker
}

// WithPodLabels adds a PodLabels check to the PodChecker
func (checker *PodChecker) WithPodLabels(check map[string]stringmatcher.StringMatcher) *PodChecker {
	checker.PodLabels = check
	return checker
}

//FromPod populates the PodChecker using data from a Pod field
func (checker *PodChecker) FromPod(event *tetragon.Pod) *PodChecker {
	if event == nil {
		return checker
	}
	checker.Namespace = stringmatcher.Full(event.Namespace)
	checker.Name = stringmatcher.Full(event.Name)
	// TODO from labels
	if event.Container != nil {
		checker.Container = NewContainerChecker().FromContainer(event.Container)
	}
	// TODO: implement fromMap
	return checker
}

// CapabilitiesChecker implements a checker struct to check a Capabilities field
type CapabilitiesChecker struct {
	Permitted   *CapabilitiesTypeListMatcher `json:"permitted,omitempty"`
	Effective   *CapabilitiesTypeListMatcher `json:"effective,omitempty"`
	Inheritable *CapabilitiesTypeListMatcher `json:"inheritable,omitempty"`
}

// NewCapabilitiesChecker creates a new CapabilitiesChecker
func NewCapabilitiesChecker() *CapabilitiesChecker {
	return &CapabilitiesChecker{}
}

// Get the type of the checker as a string
func (checker *CapabilitiesChecker) GetCheckerType() string {
	return "CapabilitiesChecker"
}

// Check checks a Capabilities field
func (checker *CapabilitiesChecker) Check(event *tetragon.Capabilities) error {
	if event == nil {
		return fmt.Errorf("%s: Capabilities field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Permitted != nil {
			if err := checker.Permitted.Check(event.Permitted); err != nil {
				return fmt.Errorf("Permitted check failed: %w", err)
			}
		}
		if checker.Effective != nil {
			if err := checker.Effective.Check(event.Effective); err != nil {
				return fmt.Errorf("Effective check failed: %w", err)
			}
		}
		if checker.Inheritable != nil {
			if err := checker.Inheritable.Check(event.Inheritable); err != nil {
				return fmt.Errorf("Inheritable check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithPermitted adds a Permitted check to the CapabilitiesChecker
func (checker *CapabilitiesChecker) WithPermitted(check *CapabilitiesTypeListMatcher) *CapabilitiesChecker {
	checker.Permitted = check
	return checker
}

// WithEffective adds a Effective check to the CapabilitiesChecker
func (checker *CapabilitiesChecker) WithEffective(check *CapabilitiesTypeListMatcher) *CapabilitiesChecker {
	checker.Effective = check
	return checker
}

// WithInheritable adds a Inheritable check to the CapabilitiesChecker
func (checker *CapabilitiesChecker) WithInheritable(check *CapabilitiesTypeListMatcher) *CapabilitiesChecker {
	checker.Inheritable = check
	return checker
}

//FromCapabilities populates the CapabilitiesChecker using data from a Capabilities field
func (checker *CapabilitiesChecker) FromCapabilities(event *tetragon.Capabilities) *CapabilitiesChecker {
	if event == nil {
		return checker
	}
	{
		var checks []*CapabilitiesTypeChecker
		for _, check := range event.Permitted {
			var convertedCheck *CapabilitiesTypeChecker
			convertedCheck = NewCapabilitiesTypeChecker(check)
			checks = append(checks, convertedCheck)
		}
		lm := NewCapabilitiesTypeListMatcher().WithOperator(listmatcher.Ordered).
			WithValues(checks...)
		checker.Permitted = lm
	}
	{
		var checks []*CapabilitiesTypeChecker
		for _, check := range event.Effective {
			var convertedCheck *CapabilitiesTypeChecker
			convertedCheck = NewCapabilitiesTypeChecker(check)
			checks = append(checks, convertedCheck)
		}
		lm := NewCapabilitiesTypeListMatcher().WithOperator(listmatcher.Ordered).
			WithValues(checks...)
		checker.Effective = lm
	}
	{
		var checks []*CapabilitiesTypeChecker
		for _, check := range event.Inheritable {
			var convertedCheck *CapabilitiesTypeChecker
			convertedCheck = NewCapabilitiesTypeChecker(check)
			checks = append(checks, convertedCheck)
		}
		lm := NewCapabilitiesTypeListMatcher().WithOperator(listmatcher.Ordered).
			WithValues(checks...)
		checker.Inheritable = lm
	}
	return checker
}

// CapabilitiesTypeListMatcher checks a list of tetragon.CapabilitiesType fields
type CapabilitiesTypeListMatcher struct {
	Operator listmatcher.Operator       `json:"operator"`
	Values   []*CapabilitiesTypeChecker `json:"values"`
}

// NewCapabilitiesTypeListMatcher creates a new CapabilitiesTypeListMatcher. The checker defaults to a subset checker unless otherwise specified using WithOperator()
func NewCapabilitiesTypeListMatcher() *CapabilitiesTypeListMatcher {
	return &CapabilitiesTypeListMatcher{
		Operator: listmatcher.Subset,
	}
}

// WithOperator sets the match kind for the CapabilitiesTypeListMatcher
func (checker *CapabilitiesTypeListMatcher) WithOperator(operator listmatcher.Operator) *CapabilitiesTypeListMatcher {
	checker.Operator = operator
	return checker
}

// WithValues sets the checkers that the CapabilitiesTypeListMatcher should use
func (checker *CapabilitiesTypeListMatcher) WithValues(values ...*CapabilitiesTypeChecker) *CapabilitiesTypeListMatcher {
	checker.Values = values
	return checker
}

// Check checks a list of tetragon.CapabilitiesType fields
func (checker *CapabilitiesTypeListMatcher) Check(values []tetragon.CapabilitiesType) error {
	switch checker.Operator {
	case listmatcher.Ordered:
		return checker.orderedCheck(values)
	case listmatcher.Unordered:
		return checker.unorderedCheck(values)
	case listmatcher.Subset:
		return checker.subsetCheck(values)
	default:
		return fmt.Errorf("Unhandled ListMatcher operator %s", checker.Operator)
	}
}

// orderedCheck checks a list of ordered tetragon.CapabilitiesType fields
func (checker *CapabilitiesTypeListMatcher) orderedCheck(values []tetragon.CapabilitiesType) error {
	innerCheck := func(check *CapabilitiesTypeChecker, value tetragon.CapabilitiesType) error {
		if err := check.Check(&value); err != nil {
			return fmt.Errorf("Permitted check failed: %w", err)
		}
		return nil
	}

	if len(checker.Values) != len(values) {
		return fmt.Errorf("CapabilitiesTypeListMatcher: Wanted %d elements, got %d", len(checker.Values), len(values))
	}

	for i, check := range checker.Values {
		value := values[i]
		if err := innerCheck(check, value); err != nil {
			return fmt.Errorf("CapabilitiesTypeListMatcher: Check failed on element %d: %w", i, err)
		}
	}

	return nil
}

// unorderedCheck checks a list of unordered tetragon.CapabilitiesType fields
func (checker *CapabilitiesTypeListMatcher) unorderedCheck(values []tetragon.CapabilitiesType) error {
	if len(checker.Values) != len(values) {
		return fmt.Errorf("CapabilitiesTypeListMatcher: Wanted %d elements, got %d", len(checker.Values), len(values))
	}

	return checker.subsetCheck(values)
}

// subsetCheck checks a subset of tetragon.CapabilitiesType fields
func (checker *CapabilitiesTypeListMatcher) subsetCheck(values []tetragon.CapabilitiesType) error {
	innerCheck := func(check *CapabilitiesTypeChecker, value tetragon.CapabilitiesType) error {
		if err := check.Check(&value); err != nil {
			return fmt.Errorf("Permitted check failed: %w", err)
		}
		return nil
	}

	numDesired := len(checker.Values)
	numMatched := 0

nextCheck:
	for _, check := range checker.Values {
		for _, value := range values {
			if err := innerCheck(check, value); err == nil {
				numMatched += 1
				continue nextCheck
			}
		}
	}

	if numMatched < numDesired {
		return fmt.Errorf("CapabilitiesTypeListMatcher: Check failed, only matched %d elements but wanted %d", numMatched, numDesired)
	}

	return nil
}

// NamespaceChecker implements a checker struct to check a Namespace field
type NamespaceChecker struct {
	Inum   *uint32 `json:"inum,omitempty"`
	IsHost *bool   `json:"isHost,omitempty"`
}

// NewNamespaceChecker creates a new NamespaceChecker
func NewNamespaceChecker() *NamespaceChecker {
	return &NamespaceChecker{}
}

// Get the type of the checker as a string
func (checker *NamespaceChecker) GetCheckerType() string {
	return "NamespaceChecker"
}

// Check checks a Namespace field
func (checker *NamespaceChecker) Check(event *tetragon.Namespace) error {
	if event == nil {
		return fmt.Errorf("%s: Namespace field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Inum != nil {
			if *checker.Inum != event.Inum {
				return fmt.Errorf("Inum has value %d which does not match expected value %d", event.Inum, *checker.Inum)
			}
		}
		if checker.IsHost != nil {
			if *checker.IsHost != event.IsHost {
				return fmt.Errorf("IsHost has value %t which does not match expected value %t", event.IsHost, *checker.IsHost)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithInum adds a Inum check to the NamespaceChecker
func (checker *NamespaceChecker) WithInum(check uint32) *NamespaceChecker {
	checker.Inum = &check
	return checker
}

// WithIsHost adds a IsHost check to the NamespaceChecker
func (checker *NamespaceChecker) WithIsHost(check bool) *NamespaceChecker {
	checker.IsHost = &check
	return checker
}

//FromNamespace populates the NamespaceChecker using data from a Namespace field
func (checker *NamespaceChecker) FromNamespace(event *tetragon.Namespace) *NamespaceChecker {
	if event == nil {
		return checker
	}
	{
		val := event.Inum
		checker.Inum = &val
	}
	{
		val := event.IsHost
		checker.IsHost = &val
	}
	return checker
}

// NamespacesChecker implements a checker struct to check a Namespaces field
type NamespacesChecker struct {
	Uts             *NamespaceChecker `json:"uts,omitempty"`
	Ipc             *NamespaceChecker `json:"ipc,omitempty"`
	Mnt             *NamespaceChecker `json:"mnt,omitempty"`
	Pid             *NamespaceChecker `json:"pid,omitempty"`
	PidForChildren  *NamespaceChecker `json:"pidForChildren,omitempty"`
	Net             *NamespaceChecker `json:"net,omitempty"`
	Time            *NamespaceChecker `json:"time,omitempty"`
	TimeForChildren *NamespaceChecker `json:"timeForChildren,omitempty"`
	Cgroup          *NamespaceChecker `json:"cgroup,omitempty"`
	User            *NamespaceChecker `json:"user,omitempty"`
}

// NewNamespacesChecker creates a new NamespacesChecker
func NewNamespacesChecker() *NamespacesChecker {
	return &NamespacesChecker{}
}

// Get the type of the checker as a string
func (checker *NamespacesChecker) GetCheckerType() string {
	return "NamespacesChecker"
}

// Check checks a Namespaces field
func (checker *NamespacesChecker) Check(event *tetragon.Namespaces) error {
	if event == nil {
		return fmt.Errorf("%s: Namespaces field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Uts != nil {
			if err := checker.Uts.Check(event.Uts); err != nil {
				return fmt.Errorf("Uts check failed: %w", err)
			}
		}
		if checker.Ipc != nil {
			if err := checker.Ipc.Check(event.Ipc); err != nil {
				return fmt.Errorf("Ipc check failed: %w", err)
			}
		}
		if checker.Mnt != nil {
			if err := checker.Mnt.Check(event.Mnt); err != nil {
				return fmt.Errorf("Mnt check failed: %w", err)
			}
		}
		if checker.Pid != nil {
			if err := checker.Pid.Check(event.Pid); err != nil {
				return fmt.Errorf("Pid check failed: %w", err)
			}
		}
		if checker.PidForChildren != nil {
			if err := checker.PidForChildren.Check(event.PidForChildren); err != nil {
				return fmt.Errorf("PidForChildren check failed: %w", err)
			}
		}
		if checker.Net != nil {
			if err := checker.Net.Check(event.Net); err != nil {
				return fmt.Errorf("Net check failed: %w", err)
			}
		}
		if checker.Time != nil {
			if err := checker.Time.Check(event.Time); err != nil {
				return fmt.Errorf("Time check failed: %w", err)
			}
		}
		if checker.TimeForChildren != nil {
			if err := checker.TimeForChildren.Check(event.TimeForChildren); err != nil {
				return fmt.Errorf("TimeForChildren check failed: %w", err)
			}
		}
		if checker.Cgroup != nil {
			if err := checker.Cgroup.Check(event.Cgroup); err != nil {
				return fmt.Errorf("Cgroup check failed: %w", err)
			}
		}
		if checker.User != nil {
			if err := checker.User.Check(event.User); err != nil {
				return fmt.Errorf("User check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithUts adds a Uts check to the NamespacesChecker
func (checker *NamespacesChecker) WithUts(check *NamespaceChecker) *NamespacesChecker {
	checker.Uts = check
	return checker
}

// WithIpc adds a Ipc check to the NamespacesChecker
func (checker *NamespacesChecker) WithIpc(check *NamespaceChecker) *NamespacesChecker {
	checker.Ipc = check
	return checker
}

// WithMnt adds a Mnt check to the NamespacesChecker
func (checker *NamespacesChecker) WithMnt(check *NamespaceChecker) *NamespacesChecker {
	checker.Mnt = check
	return checker
}

// WithPid adds a Pid check to the NamespacesChecker
func (checker *NamespacesChecker) WithPid(check *NamespaceChecker) *NamespacesChecker {
	checker.Pid = check
	return checker
}

// WithPidForChildren adds a PidForChildren check to the NamespacesChecker
func (checker *NamespacesChecker) WithPidForChildren(check *NamespaceChecker) *NamespacesChecker {
	checker.PidForChildren = check
	return checker
}

// WithNet adds a Net check to the NamespacesChecker
func (checker *NamespacesChecker) WithNet(check *NamespaceChecker) *NamespacesChecker {
	checker.Net = check
	return checker
}

// WithTime adds a Time check to the NamespacesChecker
func (checker *NamespacesChecker) WithTime(check *NamespaceChecker) *NamespacesChecker {
	checker.Time = check
	return checker
}

// WithTimeForChildren adds a TimeForChildren check to the NamespacesChecker
func (checker *NamespacesChecker) WithTimeForChildren(check *NamespaceChecker) *NamespacesChecker {
	checker.TimeForChildren = check
	return checker
}

// WithCgroup adds a Cgroup check to the NamespacesChecker
func (checker *NamespacesChecker) WithCgroup(check *NamespaceChecker) *NamespacesChecker {
	checker.Cgroup = check
	return checker
}

// WithUser adds a User check to the NamespacesChecker
func (checker *NamespacesChecker) WithUser(check *NamespaceChecker) *NamespacesChecker {
	checker.User = check
	return checker
}

//FromNamespaces populates the NamespacesChecker using data from a Namespaces field
func (checker *NamespacesChecker) FromNamespaces(event *tetragon.Namespaces) *NamespacesChecker {
	if event == nil {
		return checker
	}
	if event.Uts != nil {
		checker.Uts = NewNamespaceChecker().FromNamespace(event.Uts)
	}
	if event.Ipc != nil {
		checker.Ipc = NewNamespaceChecker().FromNamespace(event.Ipc)
	}
	if event.Mnt != nil {
		checker.Mnt = NewNamespaceChecker().FromNamespace(event.Mnt)
	}
	if event.Pid != nil {
		checker.Pid = NewNamespaceChecker().FromNamespace(event.Pid)
	}
	if event.PidForChildren != nil {
		checker.PidForChildren = NewNamespaceChecker().FromNamespace(event.PidForChildren)
	}
	if event.Net != nil {
		checker.Net = NewNamespaceChecker().FromNamespace(event.Net)
	}
	if event.Time != nil {
		checker.Time = NewNamespaceChecker().FromNamespace(event.Time)
	}
	if event.TimeForChildren != nil {
		checker.TimeForChildren = NewNamespaceChecker().FromNamespace(event.TimeForChildren)
	}
	if event.Cgroup != nil {
		checker.Cgroup = NewNamespaceChecker().FromNamespace(event.Cgroup)
	}
	if event.User != nil {
		checker.User = NewNamespaceChecker().FromNamespace(event.User)
	}
	return checker
}

// ProcessChecker implements a checker struct to check a Process field
type ProcessChecker struct {
	ExecId       *stringmatcher.StringMatcher       `json:"execId,omitempty"`
	Pid          *uint32                            `json:"pid,omitempty"`
	Uid          *uint32                            `json:"uid,omitempty"`
	Cwd          *stringmatcher.StringMatcher       `json:"cwd,omitempty"`
	Binary       *stringmatcher.StringMatcher       `json:"binary,omitempty"`
	Arguments    *stringmatcher.StringMatcher       `json:"arguments,omitempty"`
	Flags        *stringmatcher.StringMatcher       `json:"flags,omitempty"`
	StartTime    *timestampmatcher.TimestampMatcher `json:"startTime,omitempty"`
	Auid         *uint32                            `json:"auid,omitempty"`
	Pod          *PodChecker                        `json:"pod,omitempty"`
	Docker       *stringmatcher.StringMatcher       `json:"docker,omitempty"`
	ParentExecId *stringmatcher.StringMatcher       `json:"parentExecId,omitempty"`
	Refcnt       *uint32                            `json:"refcnt,omitempty"`
	Cap          *CapabilitiesChecker               `json:"cap,omitempty"`
	Ns           *NamespacesChecker                 `json:"ns,omitempty"`
	Tid          *uint32                            `json:"tid,omitempty"`
}

// NewProcessChecker creates a new ProcessChecker
func NewProcessChecker() *ProcessChecker {
	return &ProcessChecker{}
}

// Get the type of the checker as a string
func (checker *ProcessChecker) GetCheckerType() string {
	return "ProcessChecker"
}

// Check checks a Process field
func (checker *ProcessChecker) Check(event *tetragon.Process) error {
	if event == nil {
		return fmt.Errorf("%s: Process field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.ExecId != nil {
			if err := checker.ExecId.Match(event.ExecId); err != nil {
				return fmt.Errorf("ExecId check failed: %w", err)
			}
		}
		if checker.Pid != nil {
			if event.Pid == nil {
				return fmt.Errorf("Pid is nil and does not match expected value %v", *checker.Pid)
			}
			if *checker.Pid != event.Pid.Value {
				return fmt.Errorf("Pid has value %v which does not match expected value %v", event.Pid.Value, *checker.Pid)
			}
		}
		if checker.Uid != nil {
			if event.Uid == nil {
				return fmt.Errorf("Uid is nil and does not match expected value %v", *checker.Uid)
			}
			if *checker.Uid != event.Uid.Value {
				return fmt.Errorf("Uid has value %v which does not match expected value %v", event.Uid.Value, *checker.Uid)
			}
		}
		if checker.Cwd != nil {
			if err := checker.Cwd.Match(event.Cwd); err != nil {
				return fmt.Errorf("Cwd check failed: %w", err)
			}
		}
		if checker.Binary != nil {
			if err := checker.Binary.Match(event.Binary); err != nil {
				return fmt.Errorf("Binary check failed: %w", err)
			}
		}
		if checker.Arguments != nil {
			if err := checker.Arguments.Match(event.Arguments); err != nil {
				return fmt.Errorf("Arguments check failed: %w", err)
			}
		}
		if checker.Flags != nil {
			if err := checker.Flags.Match(event.Flags); err != nil {
				return fmt.Errorf("Flags check failed: %w", err)
			}
		}
		if checker.StartTime != nil {
			if err := checker.StartTime.Match(event.StartTime); err != nil {
				return fmt.Errorf("StartTime check failed: %w", err)
			}
		}
		if checker.Auid != nil {
			if event.Auid == nil {
				return fmt.Errorf("Auid is nil and does not match expected value %v", *checker.Auid)
			}
			if *checker.Auid != event.Auid.Value {
				return fmt.Errorf("Auid has value %v which does not match expected value %v", event.Auid.Value, *checker.Auid)
			}
		}
		if checker.Pod != nil {
			if err := checker.Pod.Check(event.Pod); err != nil {
				return fmt.Errorf("Pod check failed: %w", err)
			}
		}
		if checker.Docker != nil {
			if err := checker.Docker.Match(event.Docker); err != nil {
				return fmt.Errorf("Docker check failed: %w", err)
			}
		}
		if checker.ParentExecId != nil {
			if err := checker.ParentExecId.Match(event.ParentExecId); err != nil {
				return fmt.Errorf("ParentExecId check failed: %w", err)
			}
		}
		if checker.Refcnt != nil {
			if *checker.Refcnt != event.Refcnt {
				return fmt.Errorf("Refcnt has value %d which does not match expected value %d", event.Refcnt, *checker.Refcnt)
			}
		}
		if checker.Cap != nil {
			if err := checker.Cap.Check(event.Cap); err != nil {
				return fmt.Errorf("Cap check failed: %w", err)
			}
		}
		if checker.Ns != nil {
			if err := checker.Ns.Check(event.Ns); err != nil {
				return fmt.Errorf("Ns check failed: %w", err)
			}
		}
		if checker.Tid != nil {
			if event.Tid == nil {
				return fmt.Errorf("Tid is nil and does not match expected value %v", *checker.Tid)
			}
			if *checker.Tid != event.Tid.Value {
				return fmt.Errorf("Tid has value %v which does not match expected value %v", event.Tid.Value, *checker.Tid)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithExecId adds a ExecId check to the ProcessChecker
func (checker *ProcessChecker) WithExecId(check *stringmatcher.StringMatcher) *ProcessChecker {
	checker.ExecId = check
	return checker
}

// WithPid adds a Pid check to the ProcessChecker
func (checker *ProcessChecker) WithPid(check uint32) *ProcessChecker {
	checker.Pid = &check
	return checker
}

// WithUid adds a Uid check to the ProcessChecker
func (checker *ProcessChecker) WithUid(check uint32) *ProcessChecker {
	checker.Uid = &check
	return checker
}

// WithCwd adds a Cwd check to the ProcessChecker
func (checker *ProcessChecker) WithCwd(check *stringmatcher.StringMatcher) *ProcessChecker {
	checker.Cwd = check
	return checker
}

// WithBinary adds a Binary check to the ProcessChecker
func (checker *ProcessChecker) WithBinary(check *stringmatcher.StringMatcher) *ProcessChecker {
	checker.Binary = check
	return checker
}

// WithArguments adds a Arguments check to the ProcessChecker
func (checker *ProcessChecker) WithArguments(check *stringmatcher.StringMatcher) *ProcessChecker {
	checker.Arguments = check
	return checker
}

// WithFlags adds a Flags check to the ProcessChecker
func (checker *ProcessChecker) WithFlags(check *stringmatcher.StringMatcher) *ProcessChecker {
	checker.Flags = check
	return checker
}

// WithStartTime adds a StartTime check to the ProcessChecker
func (checker *ProcessChecker) WithStartTime(check *timestampmatcher.TimestampMatcher) *ProcessChecker {
	checker.StartTime = check
	return checker
}

// WithAuid adds a Auid check to the ProcessChecker
func (checker *ProcessChecker) WithAuid(check uint32) *ProcessChecker {
	checker.Auid = &check
	return checker
}

// WithPod adds a Pod check to the ProcessChecker
func (checker *ProcessChecker) WithPod(check *PodChecker) *ProcessChecker {
	checker.Pod = check
	return checker
}

// WithDocker adds a Docker check to the ProcessChecker
func (checker *ProcessChecker) WithDocker(check *stringmatcher.StringMatcher) *ProcessChecker {
	checker.Docker = check
	return checker
}

// WithParentExecId adds a ParentExecId check to the ProcessChecker
func (checker *ProcessChecker) WithParentExecId(check *stringmatcher.StringMatcher) *ProcessChecker {
	checker.ParentExecId = check
	return checker
}

// WithRefcnt adds a Refcnt check to the ProcessChecker
func (checker *ProcessChecker) WithRefcnt(check uint32) *ProcessChecker {
	checker.Refcnt = &check
	return checker
}

// WithCap adds a Cap check to the ProcessChecker
func (checker *ProcessChecker) WithCap(check *CapabilitiesChecker) *ProcessChecker {
	checker.Cap = check
	return checker
}

// WithNs adds a Ns check to the ProcessChecker
func (checker *ProcessChecker) WithNs(check *NamespacesChecker) *ProcessChecker {
	checker.Ns = check
	return checker
}

// WithTid adds a Tid check to the ProcessChecker
func (checker *ProcessChecker) WithTid(check uint32) *ProcessChecker {
	checker.Tid = &check
	return checker
}

//FromProcess populates the ProcessChecker using data from a Process field
func (checker *ProcessChecker) FromProcess(event *tetragon.Process) *ProcessChecker {
	if event == nil {
		return checker
	}
	checker.ExecId = stringmatcher.Full(event.ExecId)
	if event.Pid != nil {
		val := event.Pid.Value
		checker.Pid = &val
	}
	if event.Uid != nil {
		val := event.Uid.Value
		checker.Uid = &val
	}
	checker.Cwd = stringmatcher.Full(event.Cwd)
	checker.Binary = stringmatcher.Full(event.Binary)
	checker.Arguments = stringmatcher.Full(event.Arguments)
	checker.Flags = stringmatcher.Full(event.Flags)
	// NB: We don't want to match timestamps for now
	checker.StartTime = nil
	if event.Auid != nil {
		val := event.Auid.Value
		checker.Auid = &val
	}
	if event.Pod != nil {
		checker.Pod = NewPodChecker().FromPod(event.Pod)
	}
	checker.Docker = stringmatcher.Full(event.Docker)
	checker.ParentExecId = stringmatcher.Full(event.ParentExecId)
	{
		val := event.Refcnt
		checker.Refcnt = &val
	}
	if event.Cap != nil {
		checker.Cap = NewCapabilitiesChecker().FromCapabilities(event.Cap)
	}
	if event.Ns != nil {
		checker.Ns = NewNamespacesChecker().FromNamespaces(event.Ns)
	}
	if event.Tid != nil {
		val := event.Tid.Value
		checker.Tid = &val
	}
	return checker
}

// KprobeSockChecker implements a checker struct to check a KprobeSock field
type KprobeSockChecker struct {
	Family   *stringmatcher.StringMatcher `json:"family,omitempty"`
	Type     *stringmatcher.StringMatcher `json:"type,omitempty"`
	Protocol *stringmatcher.StringMatcher `json:"protocol,omitempty"`
	Mark     *uint32                      `json:"mark,omitempty"`
	Priority *uint32                      `json:"priority,omitempty"`
	Saddr    *stringmatcher.StringMatcher `json:"saddr,omitempty"`
	Daddr    *stringmatcher.StringMatcher `json:"daddr,omitempty"`
	Sport    *uint32                      `json:"sport,omitempty"`
	Dport    *uint32                      `json:"dport,omitempty"`
	Cookie   *uint64                      `json:"cookie,omitempty"`
}

// NewKprobeSockChecker creates a new KprobeSockChecker
func NewKprobeSockChecker() *KprobeSockChecker {
	return &KprobeSockChecker{}
}

// Get the type of the checker as a string
func (checker *KprobeSockChecker) GetCheckerType() string {
	return "KprobeSockChecker"
}

// Check checks a KprobeSock field
func (checker *KprobeSockChecker) Check(event *tetragon.KprobeSock) error {
	if event == nil {
		return fmt.Errorf("%s: KprobeSock field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Family != nil {
			if err := checker.Family.Match(event.Family); err != nil {
				return fmt.Errorf("Family check failed: %w", err)
			}
		}
		if checker.Type != nil {
			if err := checker.Type.Match(event.Type); err != nil {
				return fmt.Errorf("Type check failed: %w", err)
			}
		}
		if checker.Protocol != nil {
			if err := checker.Protocol.Match(event.Protocol); err != nil {
				return fmt.Errorf("Protocol check failed: %w", err)
			}
		}
		if checker.Mark != nil {
			if *checker.Mark != event.Mark {
				return fmt.Errorf("Mark has value %d which does not match expected value %d", event.Mark, *checker.Mark)
			}
		}
		if checker.Priority != nil {
			if *checker.Priority != event.Priority {
				return fmt.Errorf("Priority has value %d which does not match expected value %d", event.Priority, *checker.Priority)
			}
		}
		if checker.Saddr != nil {
			if err := checker.Saddr.Match(event.Saddr); err != nil {
				return fmt.Errorf("Saddr check failed: %w", err)
			}
		}
		if checker.Daddr != nil {
			if err := checker.Daddr.Match(event.Daddr); err != nil {
				return fmt.Errorf("Daddr check failed: %w", err)
			}
		}
		if checker.Sport != nil {
			if *checker.Sport != event.Sport {
				return fmt.Errorf("Sport has value %d which does not match expected value %d", event.Sport, *checker.Sport)
			}
		}
		if checker.Dport != nil {
			if *checker.Dport != event.Dport {
				return fmt.Errorf("Dport has value %d which does not match expected value %d", event.Dport, *checker.Dport)
			}
		}
		if checker.Cookie != nil {
			if *checker.Cookie != event.Cookie {
				return fmt.Errorf("Cookie has value %d which does not match expected value %d", event.Cookie, *checker.Cookie)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithFamily adds a Family check to the KprobeSockChecker
func (checker *KprobeSockChecker) WithFamily(check *stringmatcher.StringMatcher) *KprobeSockChecker {
	checker.Family = check
	return checker
}

// WithType adds a Type check to the KprobeSockChecker
func (checker *KprobeSockChecker) WithType(check *stringmatcher.StringMatcher) *KprobeSockChecker {
	checker.Type = check
	return checker
}

// WithProtocol adds a Protocol check to the KprobeSockChecker
func (checker *KprobeSockChecker) WithProtocol(check *stringmatcher.StringMatcher) *KprobeSockChecker {
	checker.Protocol = check
	return checker
}

// WithMark adds a Mark check to the KprobeSockChecker
func (checker *KprobeSockChecker) WithMark(check uint32) *KprobeSockChecker {
	checker.Mark = &check
	return checker
}

// WithPriority adds a Priority check to the KprobeSockChecker
func (checker *KprobeSockChecker) WithPriority(check uint32) *KprobeSockChecker {
	checker.Priority = &check
	return checker
}

// WithSaddr adds a Saddr check to the KprobeSockChecker
func (checker *KprobeSockChecker) WithSaddr(check *stringmatcher.StringMatcher) *KprobeSockChecker {
	checker.Saddr = check
	return checker
}

// WithDaddr adds a Daddr check to the KprobeSockChecker
func (checker *KprobeSockChecker) WithDaddr(check *stringmatcher.StringMatcher) *KprobeSockChecker {
	checker.Daddr = check
	return checker
}

// WithSport adds a Sport check to the KprobeSockChecker
func (checker *KprobeSockChecker) WithSport(check uint32) *KprobeSockChecker {
	checker.Sport = &check
	return checker
}

// WithDport adds a Dport check to the KprobeSockChecker
func (checker *KprobeSockChecker) WithDport(check uint32) *KprobeSockChecker {
	checker.Dport = &check
	return checker
}

// WithCookie adds a Cookie check to the KprobeSockChecker
func (checker *KprobeSockChecker) WithCookie(check uint64) *KprobeSockChecker {
	checker.Cookie = &check
	return checker
}

//FromKprobeSock populates the KprobeSockChecker using data from a KprobeSock field
func (checker *KprobeSockChecker) FromKprobeSock(event *tetragon.KprobeSock) *KprobeSockChecker {
	if event == nil {
		return checker
	}
	checker.Family = stringmatcher.Full(event.Family)
	checker.Type = stringmatcher.Full(event.Type)
	checker.Protocol = stringmatcher.Full(event.Protocol)
	{
		val := event.Mark
		checker.Mark = &val
	}
	{
		val := event.Priority
		checker.Priority = &val
	}
	checker.Saddr = stringmatcher.Full(event.Saddr)
	checker.Daddr = stringmatcher.Full(event.Daddr)
	{
		val := event.Sport
		checker.Sport = &val
	}
	{
		val := event.Dport
		checker.Dport = &val
	}
	{
		val := event.Cookie
		checker.Cookie = &val
	}
	return checker
}

// KprobeSkbChecker implements a checker struct to check a KprobeSkb field
type KprobeSkbChecker struct {
	Hash        *uint32                      `json:"hash,omitempty"`
	Len         *uint32                      `json:"len,omitempty"`
	Priority    *uint32                      `json:"priority,omitempty"`
	Mark        *uint32                      `json:"mark,omitempty"`
	Saddr       *stringmatcher.StringMatcher `json:"saddr,omitempty"`
	Daddr       *stringmatcher.StringMatcher `json:"daddr,omitempty"`
	Sport       *uint32                      `json:"sport,omitempty"`
	Dport       *uint32                      `json:"dport,omitempty"`
	Proto       *uint32                      `json:"proto,omitempty"`
	SecPathLen  *uint32                      `json:"secPathLen,omitempty"`
	SecPathOlen *uint32                      `json:"secPathOlen,omitempty"`
	Protocol    *stringmatcher.StringMatcher `json:"protocol,omitempty"`
}

// NewKprobeSkbChecker creates a new KprobeSkbChecker
func NewKprobeSkbChecker() *KprobeSkbChecker {
	return &KprobeSkbChecker{}
}

// Get the type of the checker as a string
func (checker *KprobeSkbChecker) GetCheckerType() string {
	return "KprobeSkbChecker"
}

// Check checks a KprobeSkb field
func (checker *KprobeSkbChecker) Check(event *tetragon.KprobeSkb) error {
	if event == nil {
		return fmt.Errorf("%s: KprobeSkb field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Hash != nil {
			if *checker.Hash != event.Hash {
				return fmt.Errorf("Hash has value %d which does not match expected value %d", event.Hash, *checker.Hash)
			}
		}
		if checker.Len != nil {
			if *checker.Len != event.Len {
				return fmt.Errorf("Len has value %d which does not match expected value %d", event.Len, *checker.Len)
			}
		}
		if checker.Priority != nil {
			if *checker.Priority != event.Priority {
				return fmt.Errorf("Priority has value %d which does not match expected value %d", event.Priority, *checker.Priority)
			}
		}
		if checker.Mark != nil {
			if *checker.Mark != event.Mark {
				return fmt.Errorf("Mark has value %d which does not match expected value %d", event.Mark, *checker.Mark)
			}
		}
		if checker.Saddr != nil {
			if err := checker.Saddr.Match(event.Saddr); err != nil {
				return fmt.Errorf("Saddr check failed: %w", err)
			}
		}
		if checker.Daddr != nil {
			if err := checker.Daddr.Match(event.Daddr); err != nil {
				return fmt.Errorf("Daddr check failed: %w", err)
			}
		}
		if checker.Sport != nil {
			if *checker.Sport != event.Sport {
				return fmt.Errorf("Sport has value %d which does not match expected value %d", event.Sport, *checker.Sport)
			}
		}
		if checker.Dport != nil {
			if *checker.Dport != event.Dport {
				return fmt.Errorf("Dport has value %d which does not match expected value %d", event.Dport, *checker.Dport)
			}
		}
		if checker.Proto != nil {
			if *checker.Proto != event.Proto {
				return fmt.Errorf("Proto has value %d which does not match expected value %d", event.Proto, *checker.Proto)
			}
		}
		if checker.SecPathLen != nil {
			if *checker.SecPathLen != event.SecPathLen {
				return fmt.Errorf("SecPathLen has value %d which does not match expected value %d", event.SecPathLen, *checker.SecPathLen)
			}
		}
		if checker.SecPathOlen != nil {
			if *checker.SecPathOlen != event.SecPathOlen {
				return fmt.Errorf("SecPathOlen has value %d which does not match expected value %d", event.SecPathOlen, *checker.SecPathOlen)
			}
		}
		if checker.Protocol != nil {
			if err := checker.Protocol.Match(event.Protocol); err != nil {
				return fmt.Errorf("Protocol check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithHash adds a Hash check to the KprobeSkbChecker
func (checker *KprobeSkbChecker) WithHash(check uint32) *KprobeSkbChecker {
	checker.Hash = &check
	return checker
}

// WithLen adds a Len check to the KprobeSkbChecker
func (checker *KprobeSkbChecker) WithLen(check uint32) *KprobeSkbChecker {
	checker.Len = &check
	return checker
}

// WithPriority adds a Priority check to the KprobeSkbChecker
func (checker *KprobeSkbChecker) WithPriority(check uint32) *KprobeSkbChecker {
	checker.Priority = &check
	return checker
}

// WithMark adds a Mark check to the KprobeSkbChecker
func (checker *KprobeSkbChecker) WithMark(check uint32) *KprobeSkbChecker {
	checker.Mark = &check
	return checker
}

// WithSaddr adds a Saddr check to the KprobeSkbChecker
func (checker *KprobeSkbChecker) WithSaddr(check *stringmatcher.StringMatcher) *KprobeSkbChecker {
	checker.Saddr = check
	return checker
}

// WithDaddr adds a Daddr check to the KprobeSkbChecker
func (checker *KprobeSkbChecker) WithDaddr(check *stringmatcher.StringMatcher) *KprobeSkbChecker {
	checker.Daddr = check
	return checker
}

// WithSport adds a Sport check to the KprobeSkbChecker
func (checker *KprobeSkbChecker) WithSport(check uint32) *KprobeSkbChecker {
	checker.Sport = &check
	return checker
}

// WithDport adds a Dport check to the KprobeSkbChecker
func (checker *KprobeSkbChecker) WithDport(check uint32) *KprobeSkbChecker {
	checker.Dport = &check
	return checker
}

// WithProto adds a Proto check to the KprobeSkbChecker
func (checker *KprobeSkbChecker) WithProto(check uint32) *KprobeSkbChecker {
	checker.Proto = &check
	return checker
}

// WithSecPathLen adds a SecPathLen check to the KprobeSkbChecker
func (checker *KprobeSkbChecker) WithSecPathLen(check uint32) *KprobeSkbChecker {
	checker.SecPathLen = &check
	return checker
}

// WithSecPathOlen adds a SecPathOlen check to the KprobeSkbChecker
func (checker *KprobeSkbChecker) WithSecPathOlen(check uint32) *KprobeSkbChecker {
	checker.SecPathOlen = &check
	return checker
}

// WithProtocol adds a Protocol check to the KprobeSkbChecker
func (checker *KprobeSkbChecker) WithProtocol(check *stringmatcher.StringMatcher) *KprobeSkbChecker {
	checker.Protocol = check
	return checker
}

//FromKprobeSkb populates the KprobeSkbChecker using data from a KprobeSkb field
func (checker *KprobeSkbChecker) FromKprobeSkb(event *tetragon.KprobeSkb) *KprobeSkbChecker {
	if event == nil {
		return checker
	}
	{
		val := event.Hash
		checker.Hash = &val
	}
	{
		val := event.Len
		checker.Len = &val
	}
	{
		val := event.Priority
		checker.Priority = &val
	}
	{
		val := event.Mark
		checker.Mark = &val
	}
	checker.Saddr = stringmatcher.Full(event.Saddr)
	checker.Daddr = stringmatcher.Full(event.Daddr)
	{
		val := event.Sport
		checker.Sport = &val
	}
	{
		val := event.Dport
		checker.Dport = &val
	}
	{
		val := event.Proto
		checker.Proto = &val
	}
	{
		val := event.SecPathLen
		checker.SecPathLen = &val
	}
	{
		val := event.SecPathOlen
		checker.SecPathOlen = &val
	}
	checker.Protocol = stringmatcher.Full(event.Protocol)
	return checker
}

// KprobePathChecker implements a checker struct to check a KprobePath field
type KprobePathChecker struct {
	Mount *stringmatcher.StringMatcher `json:"mount,omitempty"`
	Path  *stringmatcher.StringMatcher `json:"path,omitempty"`
	Flags *stringmatcher.StringMatcher `json:"flags,omitempty"`
}

// NewKprobePathChecker creates a new KprobePathChecker
func NewKprobePathChecker() *KprobePathChecker {
	return &KprobePathChecker{}
}

// Get the type of the checker as a string
func (checker *KprobePathChecker) GetCheckerType() string {
	return "KprobePathChecker"
}

// Check checks a KprobePath field
func (checker *KprobePathChecker) Check(event *tetragon.KprobePath) error {
	if event == nil {
		return fmt.Errorf("%s: KprobePath field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Mount != nil {
			if err := checker.Mount.Match(event.Mount); err != nil {
				return fmt.Errorf("Mount check failed: %w", err)
			}
		}
		if checker.Path != nil {
			if err := checker.Path.Match(event.Path); err != nil {
				return fmt.Errorf("Path check failed: %w", err)
			}
		}
		if checker.Flags != nil {
			if err := checker.Flags.Match(event.Flags); err != nil {
				return fmt.Errorf("Flags check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithMount adds a Mount check to the KprobePathChecker
func (checker *KprobePathChecker) WithMount(check *stringmatcher.StringMatcher) *KprobePathChecker {
	checker.Mount = check
	return checker
}

// WithPath adds a Path check to the KprobePathChecker
func (checker *KprobePathChecker) WithPath(check *stringmatcher.StringMatcher) *KprobePathChecker {
	checker.Path = check
	return checker
}

// WithFlags adds a Flags check to the KprobePathChecker
func (checker *KprobePathChecker) WithFlags(check *stringmatcher.StringMatcher) *KprobePathChecker {
	checker.Flags = check
	return checker
}

//FromKprobePath populates the KprobePathChecker using data from a KprobePath field
func (checker *KprobePathChecker) FromKprobePath(event *tetragon.KprobePath) *KprobePathChecker {
	if event == nil {
		return checker
	}
	checker.Mount = stringmatcher.Full(event.Mount)
	checker.Path = stringmatcher.Full(event.Path)
	checker.Flags = stringmatcher.Full(event.Flags)
	return checker
}

// KprobeFileChecker implements a checker struct to check a KprobeFile field
type KprobeFileChecker struct {
	Mount *stringmatcher.StringMatcher `json:"mount,omitempty"`
	Path  *stringmatcher.StringMatcher `json:"path,omitempty"`
	Flags *stringmatcher.StringMatcher `json:"flags,omitempty"`
}

// NewKprobeFileChecker creates a new KprobeFileChecker
func NewKprobeFileChecker() *KprobeFileChecker {
	return &KprobeFileChecker{}
}

// Get the type of the checker as a string
func (checker *KprobeFileChecker) GetCheckerType() string {
	return "KprobeFileChecker"
}

// Check checks a KprobeFile field
func (checker *KprobeFileChecker) Check(event *tetragon.KprobeFile) error {
	if event == nil {
		return fmt.Errorf("%s: KprobeFile field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Mount != nil {
			if err := checker.Mount.Match(event.Mount); err != nil {
				return fmt.Errorf("Mount check failed: %w", err)
			}
		}
		if checker.Path != nil {
			if err := checker.Path.Match(event.Path); err != nil {
				return fmt.Errorf("Path check failed: %w", err)
			}
		}
		if checker.Flags != nil {
			if err := checker.Flags.Match(event.Flags); err != nil {
				return fmt.Errorf("Flags check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithMount adds a Mount check to the KprobeFileChecker
func (checker *KprobeFileChecker) WithMount(check *stringmatcher.StringMatcher) *KprobeFileChecker {
	checker.Mount = check
	return checker
}

// WithPath adds a Path check to the KprobeFileChecker
func (checker *KprobeFileChecker) WithPath(check *stringmatcher.StringMatcher) *KprobeFileChecker {
	checker.Path = check
	return checker
}

// WithFlags adds a Flags check to the KprobeFileChecker
func (checker *KprobeFileChecker) WithFlags(check *stringmatcher.StringMatcher) *KprobeFileChecker {
	checker.Flags = check
	return checker
}

//FromKprobeFile populates the KprobeFileChecker using data from a KprobeFile field
func (checker *KprobeFileChecker) FromKprobeFile(event *tetragon.KprobeFile) *KprobeFileChecker {
	if event == nil {
		return checker
	}
	checker.Mount = stringmatcher.Full(event.Mount)
	checker.Path = stringmatcher.Full(event.Path)
	checker.Flags = stringmatcher.Full(event.Flags)
	return checker
}

// KprobeTruncatedBytesChecker implements a checker struct to check a KprobeTruncatedBytes field
type KprobeTruncatedBytesChecker struct {
	BytesArg *bytesmatcher.BytesMatcher `json:"bytesArg,omitempty"`
	OrigSize *uint64                    `json:"origSize,omitempty"`
}

// NewKprobeTruncatedBytesChecker creates a new KprobeTruncatedBytesChecker
func NewKprobeTruncatedBytesChecker() *KprobeTruncatedBytesChecker {
	return &KprobeTruncatedBytesChecker{}
}

// Get the type of the checker as a string
func (checker *KprobeTruncatedBytesChecker) GetCheckerType() string {
	return "KprobeTruncatedBytesChecker"
}

// Check checks a KprobeTruncatedBytes field
func (checker *KprobeTruncatedBytesChecker) Check(event *tetragon.KprobeTruncatedBytes) error {
	if event == nil {
		return fmt.Errorf("%s: KprobeTruncatedBytes field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.BytesArg != nil {
			if err := checker.BytesArg.Match(event.BytesArg); err != nil {
				return fmt.Errorf("BytesArg check failed: %w", err)
			}
		}
		if checker.OrigSize != nil {
			if *checker.OrigSize != event.OrigSize {
				return fmt.Errorf("OrigSize has value %d which does not match expected value %d", event.OrigSize, *checker.OrigSize)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithBytesArg adds a BytesArg check to the KprobeTruncatedBytesChecker
func (checker *KprobeTruncatedBytesChecker) WithBytesArg(check *bytesmatcher.BytesMatcher) *KprobeTruncatedBytesChecker {
	checker.BytesArg = check
	return checker
}

// WithOrigSize adds a OrigSize check to the KprobeTruncatedBytesChecker
func (checker *KprobeTruncatedBytesChecker) WithOrigSize(check uint64) *KprobeTruncatedBytesChecker {
	checker.OrigSize = &check
	return checker
}

//FromKprobeTruncatedBytes populates the KprobeTruncatedBytesChecker using data from a KprobeTruncatedBytes field
func (checker *KprobeTruncatedBytesChecker) FromKprobeTruncatedBytes(event *tetragon.KprobeTruncatedBytes) *KprobeTruncatedBytesChecker {
	if event == nil {
		return checker
	}
	checker.BytesArg = bytesmatcher.Full(event.BytesArg)
	{
		val := event.OrigSize
		checker.OrigSize = &val
	}
	return checker
}

// KprobeCredChecker implements a checker struct to check a KprobeCred field
type KprobeCredChecker struct {
	Permitted   *CapabilitiesTypeListMatcher `json:"permitted,omitempty"`
	Effective   *CapabilitiesTypeListMatcher `json:"effective,omitempty"`
	Inheritable *CapabilitiesTypeListMatcher `json:"inheritable,omitempty"`
}

// NewKprobeCredChecker creates a new KprobeCredChecker
func NewKprobeCredChecker() *KprobeCredChecker {
	return &KprobeCredChecker{}
}

// Get the type of the checker as a string
func (checker *KprobeCredChecker) GetCheckerType() string {
	return "KprobeCredChecker"
}

// Check checks a KprobeCred field
func (checker *KprobeCredChecker) Check(event *tetragon.KprobeCred) error {
	if event == nil {
		return fmt.Errorf("%s: KprobeCred field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Permitted != nil {
			if err := checker.Permitted.Check(event.Permitted); err != nil {
				return fmt.Errorf("Permitted check failed: %w", err)
			}
		}
		if checker.Effective != nil {
			if err := checker.Effective.Check(event.Effective); err != nil {
				return fmt.Errorf("Effective check failed: %w", err)
			}
		}
		if checker.Inheritable != nil {
			if err := checker.Inheritable.Check(event.Inheritable); err != nil {
				return fmt.Errorf("Inheritable check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithPermitted adds a Permitted check to the KprobeCredChecker
func (checker *KprobeCredChecker) WithPermitted(check *CapabilitiesTypeListMatcher) *KprobeCredChecker {
	checker.Permitted = check
	return checker
}

// WithEffective adds a Effective check to the KprobeCredChecker
func (checker *KprobeCredChecker) WithEffective(check *CapabilitiesTypeListMatcher) *KprobeCredChecker {
	checker.Effective = check
	return checker
}

// WithInheritable adds a Inheritable check to the KprobeCredChecker
func (checker *KprobeCredChecker) WithInheritable(check *CapabilitiesTypeListMatcher) *KprobeCredChecker {
	checker.Inheritable = check
	return checker
}

//FromKprobeCred populates the KprobeCredChecker using data from a KprobeCred field
func (checker *KprobeCredChecker) FromKprobeCred(event *tetragon.KprobeCred) *KprobeCredChecker {
	if event == nil {
		return checker
	}
	{
		var checks []*CapabilitiesTypeChecker
		for _, check := range event.Permitted {
			var convertedCheck *CapabilitiesTypeChecker
			convertedCheck = NewCapabilitiesTypeChecker(check)
			checks = append(checks, convertedCheck)
		}
		lm := NewCapabilitiesTypeListMatcher().WithOperator(listmatcher.Ordered).
			WithValues(checks...)
		checker.Permitted = lm
	}
	{
		var checks []*CapabilitiesTypeChecker
		for _, check := range event.Effective {
			var convertedCheck *CapabilitiesTypeChecker
			convertedCheck = NewCapabilitiesTypeChecker(check)
			checks = append(checks, convertedCheck)
		}
		lm := NewCapabilitiesTypeListMatcher().WithOperator(listmatcher.Ordered).
			WithValues(checks...)
		checker.Effective = lm
	}
	{
		var checks []*CapabilitiesTypeChecker
		for _, check := range event.Inheritable {
			var convertedCheck *CapabilitiesTypeChecker
			convertedCheck = NewCapabilitiesTypeChecker(check)
			checks = append(checks, convertedCheck)
		}
		lm := NewCapabilitiesTypeListMatcher().WithOperator(listmatcher.Ordered).
			WithValues(checks...)
		checker.Inheritable = lm
	}
	return checker
}

// KprobeCapabilityChecker implements a checker struct to check a KprobeCapability field
type KprobeCapabilityChecker struct {
	Value *int32                       `json:"value,omitempty"`
	Name  *stringmatcher.StringMatcher `json:"name,omitempty"`
}

// NewKprobeCapabilityChecker creates a new KprobeCapabilityChecker
func NewKprobeCapabilityChecker() *KprobeCapabilityChecker {
	return &KprobeCapabilityChecker{}
}

// Get the type of the checker as a string
func (checker *KprobeCapabilityChecker) GetCheckerType() string {
	return "KprobeCapabilityChecker"
}

// Check checks a KprobeCapability field
func (checker *KprobeCapabilityChecker) Check(event *tetragon.KprobeCapability) error {
	if event == nil {
		return fmt.Errorf("%s: KprobeCapability field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Value != nil {
			if event.Value == nil {
				return fmt.Errorf("Value is nil and does not match expected value %v", *checker.Value)
			}
			if *checker.Value != event.Value.Value {
				return fmt.Errorf("Value has value %v which does not match expected value %v", event.Value.Value, *checker.Value)
			}
		}
		if checker.Name != nil {
			if err := checker.Name.Match(event.Name); err != nil {
				return fmt.Errorf("Name check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithValue adds a Value check to the KprobeCapabilityChecker
func (checker *KprobeCapabilityChecker) WithValue(check int32) *KprobeCapabilityChecker {
	checker.Value = &check
	return checker
}

// WithName adds a Name check to the KprobeCapabilityChecker
func (checker *KprobeCapabilityChecker) WithName(check *stringmatcher.StringMatcher) *KprobeCapabilityChecker {
	checker.Name = check
	return checker
}

//FromKprobeCapability populates the KprobeCapabilityChecker using data from a KprobeCapability field
func (checker *KprobeCapabilityChecker) FromKprobeCapability(event *tetragon.KprobeCapability) *KprobeCapabilityChecker {
	if event == nil {
		return checker
	}
	if event.Value != nil {
		val := event.Value.Value
		checker.Value = &val
	}
	checker.Name = stringmatcher.Full(event.Name)
	return checker
}

// KprobeUserNamespaceChecker implements a checker struct to check a KprobeUserNamespace field
type KprobeUserNamespaceChecker struct {
	Level *int32            `json:"level,omitempty"`
	Owner *uint32           `json:"owner,omitempty"`
	Group *uint32           `json:"group,omitempty"`
	Ns    *NamespaceChecker `json:"ns,omitempty"`
}

// NewKprobeUserNamespaceChecker creates a new KprobeUserNamespaceChecker
func NewKprobeUserNamespaceChecker() *KprobeUserNamespaceChecker {
	return &KprobeUserNamespaceChecker{}
}

// Get the type of the checker as a string
func (checker *KprobeUserNamespaceChecker) GetCheckerType() string {
	return "KprobeUserNamespaceChecker"
}

// Check checks a KprobeUserNamespace field
func (checker *KprobeUserNamespaceChecker) Check(event *tetragon.KprobeUserNamespace) error {
	if event == nil {
		return fmt.Errorf("%s: KprobeUserNamespace field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.Level != nil {
			if event.Level == nil {
				return fmt.Errorf("Level is nil and does not match expected value %v", *checker.Level)
			}
			if *checker.Level != event.Level.Value {
				return fmt.Errorf("Level has value %v which does not match expected value %v", event.Level.Value, *checker.Level)
			}
		}
		if checker.Owner != nil {
			if event.Owner == nil {
				return fmt.Errorf("Owner is nil and does not match expected value %v", *checker.Owner)
			}
			if *checker.Owner != event.Owner.Value {
				return fmt.Errorf("Owner has value %v which does not match expected value %v", event.Owner.Value, *checker.Owner)
			}
		}
		if checker.Group != nil {
			if event.Group == nil {
				return fmt.Errorf("Group is nil and does not match expected value %v", *checker.Group)
			}
			if *checker.Group != event.Group.Value {
				return fmt.Errorf("Group has value %v which does not match expected value %v", event.Group.Value, *checker.Group)
			}
		}
		if checker.Ns != nil {
			if err := checker.Ns.Check(event.Ns); err != nil {
				return fmt.Errorf("Ns check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithLevel adds a Level check to the KprobeUserNamespaceChecker
func (checker *KprobeUserNamespaceChecker) WithLevel(check int32) *KprobeUserNamespaceChecker {
	checker.Level = &check
	return checker
}

// WithOwner adds a Owner check to the KprobeUserNamespaceChecker
func (checker *KprobeUserNamespaceChecker) WithOwner(check uint32) *KprobeUserNamespaceChecker {
	checker.Owner = &check
	return checker
}

// WithGroup adds a Group check to the KprobeUserNamespaceChecker
func (checker *KprobeUserNamespaceChecker) WithGroup(check uint32) *KprobeUserNamespaceChecker {
	checker.Group = &check
	return checker
}

// WithNs adds a Ns check to the KprobeUserNamespaceChecker
func (checker *KprobeUserNamespaceChecker) WithNs(check *NamespaceChecker) *KprobeUserNamespaceChecker {
	checker.Ns = check
	return checker
}

//FromKprobeUserNamespace populates the KprobeUserNamespaceChecker using data from a KprobeUserNamespace field
func (checker *KprobeUserNamespaceChecker) FromKprobeUserNamespace(event *tetragon.KprobeUserNamespace) *KprobeUserNamespaceChecker {
	if event == nil {
		return checker
	}
	if event.Level != nil {
		val := event.Level.Value
		checker.Level = &val
	}
	if event.Owner != nil {
		val := event.Owner.Value
		checker.Owner = &val
	}
	if event.Group != nil {
		val := event.Group.Value
		checker.Group = &val
	}
	if event.Ns != nil {
		checker.Ns = NewNamespaceChecker().FromNamespace(event.Ns)
	}
	return checker
}

// KprobeBpfAttrChecker implements a checker struct to check a KprobeBpfAttr field
type KprobeBpfAttrChecker struct {
	ProgType *stringmatcher.StringMatcher `json:"ProgType,omitempty"`
	InsnCnt  *uint32                      `json:"InsnCnt,omitempty"`
	ProgName *stringmatcher.StringMatcher `json:"ProgName,omitempty"`
}

// NewKprobeBpfAttrChecker creates a new KprobeBpfAttrChecker
func NewKprobeBpfAttrChecker() *KprobeBpfAttrChecker {
	return &KprobeBpfAttrChecker{}
}

// Get the type of the checker as a string
func (checker *KprobeBpfAttrChecker) GetCheckerType() string {
	return "KprobeBpfAttrChecker"
}

// Check checks a KprobeBpfAttr field
func (checker *KprobeBpfAttrChecker) Check(event *tetragon.KprobeBpfAttr) error {
	if event == nil {
		return fmt.Errorf("%s: KprobeBpfAttr field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.ProgType != nil {
			if err := checker.ProgType.Match(event.ProgType); err != nil {
				return fmt.Errorf("ProgType check failed: %w", err)
			}
		}
		if checker.InsnCnt != nil {
			if *checker.InsnCnt != event.InsnCnt {
				return fmt.Errorf("InsnCnt has value %d which does not match expected value %d", event.InsnCnt, *checker.InsnCnt)
			}
		}
		if checker.ProgName != nil {
			if err := checker.ProgName.Match(event.ProgName); err != nil {
				return fmt.Errorf("ProgName check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithProgType adds a ProgType check to the KprobeBpfAttrChecker
func (checker *KprobeBpfAttrChecker) WithProgType(check *stringmatcher.StringMatcher) *KprobeBpfAttrChecker {
	checker.ProgType = check
	return checker
}

// WithInsnCnt adds a InsnCnt check to the KprobeBpfAttrChecker
func (checker *KprobeBpfAttrChecker) WithInsnCnt(check uint32) *KprobeBpfAttrChecker {
	checker.InsnCnt = &check
	return checker
}

// WithProgName adds a ProgName check to the KprobeBpfAttrChecker
func (checker *KprobeBpfAttrChecker) WithProgName(check *stringmatcher.StringMatcher) *KprobeBpfAttrChecker {
	checker.ProgName = check
	return checker
}

//FromKprobeBpfAttr populates the KprobeBpfAttrChecker using data from a KprobeBpfAttr field
func (checker *KprobeBpfAttrChecker) FromKprobeBpfAttr(event *tetragon.KprobeBpfAttr) *KprobeBpfAttrChecker {
	if event == nil {
		return checker
	}
	checker.ProgType = stringmatcher.Full(event.ProgType)
	{
		val := event.InsnCnt
		checker.InsnCnt = &val
	}
	checker.ProgName = stringmatcher.Full(event.ProgName)
	return checker
}

// KprobePerfEventChecker implements a checker struct to check a KprobePerfEvent field
type KprobePerfEventChecker struct {
	KprobeFunc  *stringmatcher.StringMatcher `json:"KprobeFunc,omitempty"`
	Type        *stringmatcher.StringMatcher `json:"Type,omitempty"`
	Config      *uint64                      `json:"Config,omitempty"`
	ProbeOffset *uint64                      `json:"ProbeOffset,omitempty"`
}

// NewKprobePerfEventChecker creates a new KprobePerfEventChecker
func NewKprobePerfEventChecker() *KprobePerfEventChecker {
	return &KprobePerfEventChecker{}
}

// Get the type of the checker as a string
func (checker *KprobePerfEventChecker) GetCheckerType() string {
	return "KprobePerfEventChecker"
}

// Check checks a KprobePerfEvent field
func (checker *KprobePerfEventChecker) Check(event *tetragon.KprobePerfEvent) error {
	if event == nil {
		return fmt.Errorf("%s: KprobePerfEvent field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.KprobeFunc != nil {
			if err := checker.KprobeFunc.Match(event.KprobeFunc); err != nil {
				return fmt.Errorf("KprobeFunc check failed: %w", err)
			}
		}
		if checker.Type != nil {
			if err := checker.Type.Match(event.Type); err != nil {
				return fmt.Errorf("Type check failed: %w", err)
			}
		}
		if checker.Config != nil {
			if *checker.Config != event.Config {
				return fmt.Errorf("Config has value %d which does not match expected value %d", event.Config, *checker.Config)
			}
		}
		if checker.ProbeOffset != nil {
			if *checker.ProbeOffset != event.ProbeOffset {
				return fmt.Errorf("ProbeOffset has value %d which does not match expected value %d", event.ProbeOffset, *checker.ProbeOffset)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithKprobeFunc adds a KprobeFunc check to the KprobePerfEventChecker
func (checker *KprobePerfEventChecker) WithKprobeFunc(check *stringmatcher.StringMatcher) *KprobePerfEventChecker {
	checker.KprobeFunc = check
	return checker
}

// WithType adds a Type check to the KprobePerfEventChecker
func (checker *KprobePerfEventChecker) WithType(check *stringmatcher.StringMatcher) *KprobePerfEventChecker {
	checker.Type = check
	return checker
}

// WithConfig adds a Config check to the KprobePerfEventChecker
func (checker *KprobePerfEventChecker) WithConfig(check uint64) *KprobePerfEventChecker {
	checker.Config = &check
	return checker
}

// WithProbeOffset adds a ProbeOffset check to the KprobePerfEventChecker
func (checker *KprobePerfEventChecker) WithProbeOffset(check uint64) *KprobePerfEventChecker {
	checker.ProbeOffset = &check
	return checker
}

//FromKprobePerfEvent populates the KprobePerfEventChecker using data from a KprobePerfEvent field
func (checker *KprobePerfEventChecker) FromKprobePerfEvent(event *tetragon.KprobePerfEvent) *KprobePerfEventChecker {
	if event == nil {
		return checker
	}
	checker.KprobeFunc = stringmatcher.Full(event.KprobeFunc)
	checker.Type = stringmatcher.Full(event.Type)
	{
		val := event.Config
		checker.Config = &val
	}
	{
		val := event.ProbeOffset
		checker.ProbeOffset = &val
	}
	return checker
}

// KprobeBpfMapChecker implements a checker struct to check a KprobeBpfMap field
type KprobeBpfMapChecker struct {
	MapType    *stringmatcher.StringMatcher `json:"MapType,omitempty"`
	KeySize    *uint32                      `json:"KeySize,omitempty"`
	ValueSize  *uint32                      `json:"ValueSize,omitempty"`
	MaxEntries *uint32                      `json:"MaxEntries,omitempty"`
	MapName    *stringmatcher.StringMatcher `json:"MapName,omitempty"`
}

// NewKprobeBpfMapChecker creates a new KprobeBpfMapChecker
func NewKprobeBpfMapChecker() *KprobeBpfMapChecker {
	return &KprobeBpfMapChecker{}
}

// Get the type of the checker as a string
func (checker *KprobeBpfMapChecker) GetCheckerType() string {
	return "KprobeBpfMapChecker"
}

// Check checks a KprobeBpfMap field
func (checker *KprobeBpfMapChecker) Check(event *tetragon.KprobeBpfMap) error {
	if event == nil {
		return fmt.Errorf("%s: KprobeBpfMap field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.MapType != nil {
			if err := checker.MapType.Match(event.MapType); err != nil {
				return fmt.Errorf("MapType check failed: %w", err)
			}
		}
		if checker.KeySize != nil {
			if *checker.KeySize != event.KeySize {
				return fmt.Errorf("KeySize has value %d which does not match expected value %d", event.KeySize, *checker.KeySize)
			}
		}
		if checker.ValueSize != nil {
			if *checker.ValueSize != event.ValueSize {
				return fmt.Errorf("ValueSize has value %d which does not match expected value %d", event.ValueSize, *checker.ValueSize)
			}
		}
		if checker.MaxEntries != nil {
			if *checker.MaxEntries != event.MaxEntries {
				return fmt.Errorf("MaxEntries has value %d which does not match expected value %d", event.MaxEntries, *checker.MaxEntries)
			}
		}
		if checker.MapName != nil {
			if err := checker.MapName.Match(event.MapName); err != nil {
				return fmt.Errorf("MapName check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithMapType adds a MapType check to the KprobeBpfMapChecker
func (checker *KprobeBpfMapChecker) WithMapType(check *stringmatcher.StringMatcher) *KprobeBpfMapChecker {
	checker.MapType = check
	return checker
}

// WithKeySize adds a KeySize check to the KprobeBpfMapChecker
func (checker *KprobeBpfMapChecker) WithKeySize(check uint32) *KprobeBpfMapChecker {
	checker.KeySize = &check
	return checker
}

// WithValueSize adds a ValueSize check to the KprobeBpfMapChecker
func (checker *KprobeBpfMapChecker) WithValueSize(check uint32) *KprobeBpfMapChecker {
	checker.ValueSize = &check
	return checker
}

// WithMaxEntries adds a MaxEntries check to the KprobeBpfMapChecker
func (checker *KprobeBpfMapChecker) WithMaxEntries(check uint32) *KprobeBpfMapChecker {
	checker.MaxEntries = &check
	return checker
}

// WithMapName adds a MapName check to the KprobeBpfMapChecker
func (checker *KprobeBpfMapChecker) WithMapName(check *stringmatcher.StringMatcher) *KprobeBpfMapChecker {
	checker.MapName = check
	return checker
}

//FromKprobeBpfMap populates the KprobeBpfMapChecker using data from a KprobeBpfMap field
func (checker *KprobeBpfMapChecker) FromKprobeBpfMap(event *tetragon.KprobeBpfMap) *KprobeBpfMapChecker {
	if event == nil {
		return checker
	}
	checker.MapType = stringmatcher.Full(event.MapType)
	{
		val := event.KeySize
		checker.KeySize = &val
	}
	{
		val := event.ValueSize
		checker.ValueSize = &val
	}
	{
		val := event.MaxEntries
		checker.MaxEntries = &val
	}
	checker.MapName = stringmatcher.Full(event.MapName)
	return checker
}

// KprobeArgumentChecker implements a checker struct to check a KprobeArgument field
type KprobeArgumentChecker struct {
	StringArg         *stringmatcher.StringMatcher `json:"stringArg,omitempty"`
	IntArg            *int32                       `json:"intArg,omitempty"`
	SkbArg            *KprobeSkbChecker            `json:"skbArg,omitempty"`
	SizeArg           *uint64                      `json:"sizeArg,omitempty"`
	BytesArg          *bytesmatcher.BytesMatcher   `json:"bytesArg,omitempty"`
	PathArg           *KprobePathChecker           `json:"pathArg,omitempty"`
	FileArg           *KprobeFileChecker           `json:"fileArg,omitempty"`
	TruncatedBytesArg *KprobeTruncatedBytesChecker `json:"truncatedBytesArg,omitempty"`
	SockArg           *KprobeSockChecker           `json:"sockArg,omitempty"`
	CredArg           *KprobeCredChecker           `json:"credArg,omitempty"`
	LongArg           *int64                       `json:"longArg,omitempty"`
	BpfAttrArg        *KprobeBpfAttrChecker        `json:"bpfAttrArg,omitempty"`
	PerfEventArg      *KprobePerfEventChecker      `json:"perfEventArg,omitempty"`
	BpfMapArg         *KprobeBpfMapChecker         `json:"bpfMapArg,omitempty"`
	UintArg           *uint32                      `json:"uintArg,omitempty"`
	UserNamespaceArg  *KprobeUserNamespaceChecker  `json:"userNamespaceArg,omitempty"`
	CapabilityArg     *KprobeCapabilityChecker     `json:"capabilityArg,omitempty"`
	Label             *stringmatcher.StringMatcher `json:"label,omitempty"`
}

// NewKprobeArgumentChecker creates a new KprobeArgumentChecker
func NewKprobeArgumentChecker() *KprobeArgumentChecker {
	return &KprobeArgumentChecker{}
}

// Get the type of the checker as a string
func (checker *KprobeArgumentChecker) GetCheckerType() string {
	return "KprobeArgumentChecker"
}

// Check checks a KprobeArgument field
func (checker *KprobeArgumentChecker) Check(event *tetragon.KprobeArgument) error {
	if event == nil {
		return fmt.Errorf("%s: KprobeArgument field is nil", CheckerLogPrefix(checker))
	}

	fieldChecks := func() error {
		if checker.StringArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_StringArg:
				if err := checker.StringArg.Match(event.StringArg); err != nil {
					return fmt.Errorf("StringArg check failed: %w", err)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: StringArg check failed: %T is not a StringArg", event)
			}
		}
		if checker.IntArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_IntArg:
				if *checker.IntArg != event.IntArg {
					return fmt.Errorf("IntArg has value %d which does not match expected value %d", event.IntArg, *checker.IntArg)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: IntArg check failed: %T is not a IntArg", event)
			}
		}
		if checker.SkbArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_SkbArg:
				if err := checker.SkbArg.Check(event.SkbArg); err != nil {
					return fmt.Errorf("SkbArg check failed: %w", err)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: SkbArg check failed: %T is not a SkbArg", event)
			}
		}
		if checker.SizeArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_SizeArg:
				if *checker.SizeArg != event.SizeArg {
					return fmt.Errorf("SizeArg has value %d which does not match expected value %d", event.SizeArg, *checker.SizeArg)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: SizeArg check failed: %T is not a SizeArg", event)
			}
		}
		if checker.BytesArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_BytesArg:
				if err := checker.BytesArg.Match(event.BytesArg); err != nil {
					return fmt.Errorf("BytesArg check failed: %w", err)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: BytesArg check failed: %T is not a BytesArg", event)
			}
		}
		if checker.PathArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_PathArg:
				if err := checker.PathArg.Check(event.PathArg); err != nil {
					return fmt.Errorf("PathArg check failed: %w", err)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: PathArg check failed: %T is not a PathArg", event)
			}
		}
		if checker.FileArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_FileArg:
				if err := checker.FileArg.Check(event.FileArg); err != nil {
					return fmt.Errorf("FileArg check failed: %w", err)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: FileArg check failed: %T is not a FileArg", event)
			}
		}
		if checker.TruncatedBytesArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_TruncatedBytesArg:
				if err := checker.TruncatedBytesArg.Check(event.TruncatedBytesArg); err != nil {
					return fmt.Errorf("TruncatedBytesArg check failed: %w", err)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: TruncatedBytesArg check failed: %T is not a TruncatedBytesArg", event)
			}
		}
		if checker.SockArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_SockArg:
				if err := checker.SockArg.Check(event.SockArg); err != nil {
					return fmt.Errorf("SockArg check failed: %w", err)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: SockArg check failed: %T is not a SockArg", event)
			}
		}
		if checker.CredArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_CredArg:
				if err := checker.CredArg.Check(event.CredArg); err != nil {
					return fmt.Errorf("CredArg check failed: %w", err)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: CredArg check failed: %T is not a CredArg", event)
			}
		}
		if checker.LongArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_LongArg:
				if *checker.LongArg != event.LongArg {
					return fmt.Errorf("LongArg has value %d which does not match expected value %d", event.LongArg, *checker.LongArg)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: LongArg check failed: %T is not a LongArg", event)
			}
		}
		if checker.BpfAttrArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_BpfAttrArg:
				if err := checker.BpfAttrArg.Check(event.BpfAttrArg); err != nil {
					return fmt.Errorf("BpfAttrArg check failed: %w", err)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: BpfAttrArg check failed: %T is not a BpfAttrArg", event)
			}
		}
		if checker.PerfEventArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_PerfEventArg:
				if err := checker.PerfEventArg.Check(event.PerfEventArg); err != nil {
					return fmt.Errorf("PerfEventArg check failed: %w", err)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: PerfEventArg check failed: %T is not a PerfEventArg", event)
			}
		}
		if checker.BpfMapArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_BpfMapArg:
				if err := checker.BpfMapArg.Check(event.BpfMapArg); err != nil {
					return fmt.Errorf("BpfMapArg check failed: %w", err)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: BpfMapArg check failed: %T is not a BpfMapArg", event)
			}
		}
		if checker.UintArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_UintArg:
				if *checker.UintArg != event.UintArg {
					return fmt.Errorf("UintArg has value %d which does not match expected value %d", event.UintArg, *checker.UintArg)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: UintArg check failed: %T is not a UintArg", event)
			}
		}
		if checker.UserNamespaceArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_UserNamespaceArg:
				if err := checker.UserNamespaceArg.Check(event.UserNamespaceArg); err != nil {
					return fmt.Errorf("UserNamespaceArg check failed: %w", err)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: UserNamespaceArg check failed: %T is not a UserNamespaceArg", event)
			}
		}
		if checker.CapabilityArg != nil {
			switch event := event.Arg.(type) {
			case *tetragon.KprobeArgument_CapabilityArg:
				if err := checker.CapabilityArg.Check(event.CapabilityArg); err != nil {
					return fmt.Errorf("CapabilityArg check failed: %w", err)
				}
			default:
				return fmt.Errorf("KprobeArgumentChecker: CapabilityArg check failed: %T is not a CapabilityArg", event)
			}
		}
		if checker.Label != nil {
			if err := checker.Label.Match(event.Label); err != nil {
				return fmt.Errorf("Label check failed: %w", err)
			}
		}
		return nil
	}
	if err := fieldChecks(); err != nil {
		return fmt.Errorf("%s: %w", CheckerLogPrefix(checker), err)
	}
	return nil
}

// WithStringArg adds a StringArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithStringArg(check *stringmatcher.StringMatcher) *KprobeArgumentChecker {
	checker.StringArg = check
	return checker
}

// WithIntArg adds a IntArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithIntArg(check int32) *KprobeArgumentChecker {
	checker.IntArg = &check
	return checker
}

// WithSkbArg adds a SkbArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithSkbArg(check *KprobeSkbChecker) *KprobeArgumentChecker {
	checker.SkbArg = check
	return checker
}

// WithSizeArg adds a SizeArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithSizeArg(check uint64) *KprobeArgumentChecker {
	checker.SizeArg = &check
	return checker
}

// WithBytesArg adds a BytesArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithBytesArg(check *bytesmatcher.BytesMatcher) *KprobeArgumentChecker {
	checker.BytesArg = check
	return checker
}

// WithPathArg adds a PathArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithPathArg(check *KprobePathChecker) *KprobeArgumentChecker {
	checker.PathArg = check
	return checker
}

// WithFileArg adds a FileArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithFileArg(check *KprobeFileChecker) *KprobeArgumentChecker {
	checker.FileArg = check
	return checker
}

// WithTruncatedBytesArg adds a TruncatedBytesArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithTruncatedBytesArg(check *KprobeTruncatedBytesChecker) *KprobeArgumentChecker {
	checker.TruncatedBytesArg = check
	return checker
}

// WithSockArg adds a SockArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithSockArg(check *KprobeSockChecker) *KprobeArgumentChecker {
	checker.SockArg = check
	return checker
}

// WithCredArg adds a CredArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithCredArg(check *KprobeCredChecker) *KprobeArgumentChecker {
	checker.CredArg = check
	return checker
}

// WithLongArg adds a LongArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithLongArg(check int64) *KprobeArgumentChecker {
	checker.LongArg = &check
	return checker
}

// WithBpfAttrArg adds a BpfAttrArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithBpfAttrArg(check *KprobeBpfAttrChecker) *KprobeArgumentChecker {
	checker.BpfAttrArg = check
	return checker
}

// WithPerfEventArg adds a PerfEventArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithPerfEventArg(check *KprobePerfEventChecker) *KprobeArgumentChecker {
	checker.PerfEventArg = check
	return checker
}

// WithBpfMapArg adds a BpfMapArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithBpfMapArg(check *KprobeBpfMapChecker) *KprobeArgumentChecker {
	checker.BpfMapArg = check
	return checker
}

// WithUintArg adds a UintArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithUintArg(check uint32) *KprobeArgumentChecker {
	checker.UintArg = &check
	return checker
}

// WithUserNamespaceArg adds a UserNamespaceArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithUserNamespaceArg(check *KprobeUserNamespaceChecker) *KprobeArgumentChecker {
	checker.UserNamespaceArg = check
	return checker
}

// WithCapabilityArg adds a CapabilityArg check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithCapabilityArg(check *KprobeCapabilityChecker) *KprobeArgumentChecker {
	checker.CapabilityArg = check
	return checker
}

// WithLabel adds a Label check to the KprobeArgumentChecker
func (checker *KprobeArgumentChecker) WithLabel(check *stringmatcher.StringMatcher) *KprobeArgumentChecker {
	checker.Label = check
	return checker
}

//FromKprobeArgument populates the KprobeArgumentChecker using data from a KprobeArgument field
func (checker *KprobeArgumentChecker) FromKprobeArgument(event *tetragon.KprobeArgument) *KprobeArgumentChecker {
	if event == nil {
		return checker
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_StringArg:
		checker.StringArg = stringmatcher.Full(event.StringArg)
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_IntArg:
		{
			val := event.IntArg
			checker.IntArg = &val
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_SkbArg:
		if event.SkbArg != nil {
			checker.SkbArg = NewKprobeSkbChecker().FromKprobeSkb(event.SkbArg)
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_SizeArg:
		{
			val := event.SizeArg
			checker.SizeArg = &val
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_BytesArg:
		checker.BytesArg = bytesmatcher.Full(event.BytesArg)
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_PathArg:
		if event.PathArg != nil {
			checker.PathArg = NewKprobePathChecker().FromKprobePath(event.PathArg)
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_FileArg:
		if event.FileArg != nil {
			checker.FileArg = NewKprobeFileChecker().FromKprobeFile(event.FileArg)
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_TruncatedBytesArg:
		if event.TruncatedBytesArg != nil {
			checker.TruncatedBytesArg = NewKprobeTruncatedBytesChecker().FromKprobeTruncatedBytes(event.TruncatedBytesArg)
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_SockArg:
		if event.SockArg != nil {
			checker.SockArg = NewKprobeSockChecker().FromKprobeSock(event.SockArg)
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_CredArg:
		if event.CredArg != nil {
			checker.CredArg = NewKprobeCredChecker().FromKprobeCred(event.CredArg)
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_LongArg:
		{
			val := event.LongArg
			checker.LongArg = &val
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_BpfAttrArg:
		if event.BpfAttrArg != nil {
			checker.BpfAttrArg = NewKprobeBpfAttrChecker().FromKprobeBpfAttr(event.BpfAttrArg)
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_PerfEventArg:
		if event.PerfEventArg != nil {
			checker.PerfEventArg = NewKprobePerfEventChecker().FromKprobePerfEvent(event.PerfEventArg)
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_BpfMapArg:
		if event.BpfMapArg != nil {
			checker.BpfMapArg = NewKprobeBpfMapChecker().FromKprobeBpfMap(event.BpfMapArg)
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_UintArg:
		{
			val := event.UintArg
			checker.UintArg = &val
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_UserNamespaceArg:
		if event.UserNamespaceArg != nil {
			checker.UserNamespaceArg = NewKprobeUserNamespaceChecker().FromKprobeUserNamespace(event.UserNamespaceArg)
		}
	}
	switch event := event.Arg.(type) {
	case *tetragon.KprobeArgument_CapabilityArg:
		if event.CapabilityArg != nil {
			checker.CapabilityArg = NewKprobeCapabilityChecker().FromKprobeCapability(event.CapabilityArg)
		}
	}
	checker.Label = stringmatcher.Full(event.Label)
	return checker
}

// CapabilitiesTypeChecker checks a tetragon.CapabilitiesType
type CapabilitiesTypeChecker tetragon.CapabilitiesType

// MarshalJSON implements json.Marshaler interface
func (enum CapabilitiesTypeChecker) MarshalJSON() ([]byte, error) {
	if name, ok := tetragon.CapabilitiesType_name[int32(enum)]; ok {
		name = strings.TrimPrefix(name, "CAP_")
		return json.Marshal(name)
	}

	return nil, fmt.Errorf("Unknown CapabilitiesType %d", enum)
}

// UnmarshalJSON implements json.Unmarshaler interface
func (enum *CapabilitiesTypeChecker) UnmarshalJSON(b []byte) error {
	var str string
	if err := yaml.UnmarshalStrict(b, &str); err != nil {
		return err
	}

	// Convert to uppercase if not already
	str = strings.ToUpper(str)

	// Look up the value from the enum values map
	if n, ok := tetragon.CapabilitiesType_value[str]; ok {
		*enum = CapabilitiesTypeChecker(n)
	} else if n, ok := tetragon.CapabilitiesType_value["CAP_"+str]; ok {
		*enum = CapabilitiesTypeChecker(n)
	} else {
		return fmt.Errorf("Unknown CapabilitiesType %s", str)
	}

	return nil
}

// NewCapabilitiesTypeChecker creates a new CapabilitiesTypeChecker
func NewCapabilitiesTypeChecker(val tetragon.CapabilitiesType) *CapabilitiesTypeChecker {
	enum := CapabilitiesTypeChecker(val)
	return &enum
}

// Check checks a CapabilitiesType against the checker
func (enum *CapabilitiesTypeChecker) Check(val *tetragon.CapabilitiesType) error {
	if val == nil {
		return fmt.Errorf("CapabilitiesTypeChecker: CapabilitiesType is nil and does not match expected value %s", tetragon.CapabilitiesType(*enum))
	}
	if *enum != CapabilitiesTypeChecker(*val) {
		return fmt.Errorf("CapabilitiesTypeChecker: CapabilitiesType has value %s which does not match expected value %s", (*val), tetragon.CapabilitiesType(*enum))
	}
	return nil
}

// KprobeActionChecker checks a tetragon.KprobeAction
type KprobeActionChecker tetragon.KprobeAction

// MarshalJSON implements json.Marshaler interface
func (enum KprobeActionChecker) MarshalJSON() ([]byte, error) {
	if name, ok := tetragon.KprobeAction_name[int32(enum)]; ok {
		name = strings.TrimPrefix(name, "KPROBE_ACTION_")
		return json.Marshal(name)
	}

	return nil, fmt.Errorf("Unknown KprobeAction %d", enum)
}

// UnmarshalJSON implements json.Unmarshaler interface
func (enum *KprobeActionChecker) UnmarshalJSON(b []byte) error {
	var str string
	if err := yaml.UnmarshalStrict(b, &str); err != nil {
		return err
	}

	// Convert to uppercase if not already
	str = strings.ToUpper(str)

	// Look up the value from the enum values map
	if n, ok := tetragon.KprobeAction_value[str]; ok {
		*enum = KprobeActionChecker(n)
	} else if n, ok := tetragon.KprobeAction_value["KPROBE_ACTION_"+str]; ok {
		*enum = KprobeActionChecker(n)
	} else {
		return fmt.Errorf("Unknown KprobeAction %s", str)
	}

	return nil
}

// NewKprobeActionChecker creates a new KprobeActionChecker
func NewKprobeActionChecker(val tetragon.KprobeAction) *KprobeActionChecker {
	enum := KprobeActionChecker(val)
	return &enum
}

// Check checks a KprobeAction against the checker
func (enum *KprobeActionChecker) Check(val *tetragon.KprobeAction) error {
	if val == nil {
		return fmt.Errorf("KprobeActionChecker: KprobeAction is nil and does not match expected value %s", tetragon.KprobeAction(*enum))
	}
	if *enum != KprobeActionChecker(*val) {
		return fmt.Errorf("KprobeActionChecker: KprobeAction has value %s which does not match expected value %s", (*val), tetragon.KprobeAction(*enum))
	}
	return nil
}
