# Copyright 2011-2014 OpenStack Foundation
# All Rights Reserved.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

"""
Command-line interface to the OpenStack Cinder API.
"""

import argparse
import collections
import getpass
import logging
import sys
from urllib import parse as urlparse

from keystoneauth1 import discover
from keystoneauth1 import exceptions
from keystoneauth1.identity import v2 as v2_auth
from keystoneauth1.identity import v3 as v3_auth
from keystoneauth1 import loading
from keystoneauth1 import session
from oslo_utils import importutils
import requests

import cinderclient
from cinderclient._i18n import _
from cinderclient import api_versions
from cinderclient import client
from cinderclient import exceptions as exc
from cinderclient import utils

try:
    osprofiler_profiler = importutils.try_import("osprofiler.profiler")
except Exception:
    pass


DEFAULT_MAJOR_OS_VOLUME_API_VERSION = "3"
DEFAULT_CINDER_ENDPOINT_TYPE = 'publicURL'
V3_SHELL = 'cinderclient.v3.shell'
HINT_HELP_MSG = (" [hint: use '--os-volume-api-version' flag to show help "
                 "message for proper version]")

FILTER_CHECK = ["type-list",
                "backup-list",
                "get-pools",
                "list",
                "group-list",
                "group-snapshot-list",
                "message-list",
                "snapshot-list",
                "attachment-list"]

RESOURCE_FILTERS = {
    "list": ["name", "status", "metadata",
             "bootable", "migration_status", "availability_zone",
             "group_id", "size"],
    "backup-list": ["name", "status", "volume_id"],
    "snapshot-list": ["name", "status", "volume_id", "metadata",
                      "availability_zone"],
    "group-list": ["name"],
    "group-snapshot-list": ["name", "status", "group_id"],
    "attachment-list": ["volume_id", "status", "instance_id", "attach_status"],
    "message-list": ["resource_uuid", "resource_type", "event_id",
                     "request_id", "message_level"],
    "get-pools": ["name", "volume_type"],
    "type-list": ["is_public"]
}


logging.basicConfig()
logger = logging.getLogger(__name__)


class CinderClientArgumentParser(argparse.ArgumentParser):

    def __init__(self, *args, **kwargs):
        super(CinderClientArgumentParser, self).__init__(*args, **kwargs)

    def error(self, message):
        """error(message: string)

        Prints a usage message incorporating the message to stderr and
        exits.
        """
        self.print_usage(sys.stderr)
        # FIXME(lzyeval): if changes occur in argparse.ArgParser._check_value
        choose_from = ' (choose from'
        progparts = self.prog.partition(' ')
        self.exit(2, "error: %(errmsg)s\nTry '%(mainp)s help %(subp)s'"
                     " for more information.\n" %
                     {'errmsg': message.split(choose_from)[0],
                      'mainp': progparts[0],
                      'subp': progparts[2]})

    def _get_option_tuples(self, option_string):
        """Avoid ambiguity in argument abbreviation.

        The idea of this method is to override the default behaviour to
        avoid ambiguity in the abbreviation feature of argparse.
        In the case that the ambiguity is generated by 2 or more parameters
        and only one is visible in the help and the others are with
        help=argparse.SUPPRESS, the ambiguity is solved by taking the visible
        one.
        The use case is for parameters that are left hidden for backward
        compatibility.
        """

        result = super(CinderClientArgumentParser, self)._get_option_tuples(
            option_string)

        if len(result) > 1:
            aux = [x for x in result if x[0].help != argparse.SUPPRESS]
            if len(aux) == 1:
                result = aux

        return result


class OpenStackCinderShell(object):

    def __init__(self):
        self.ks_logger = None
        self.client_logger = None
        self.extensions = []

    def get_base_parser(self):
        parser = CinderClientArgumentParser(
            prog='cinder',
            description=__doc__.strip(),
            epilog=_('Run "cinder help SUBCOMMAND" for help on a subcommand.'),
            add_help=False,
            formatter_class=OpenStackHelpFormatter,
        )

        # Global arguments
        parser.add_argument('-h', '--help',
                            action='store_true',
                            help=argparse.SUPPRESS)

        parser.add_argument('--version',
                            action='version',
                            version=cinderclient.__version__)

        parser.add_argument('-d', '--debug',
                            action='store_true',
                            default=utils.env('CINDERCLIENT_DEBUG',
                                              default=False),
                            help=_('Shows debugging output.'))

        parser.add_argument('--service-type',
                            metavar='<service-type>',
                            help=_('Service type. '
                            'For most actions, default is volume.'))
        parser.add_argument('--service_type',
                            help=argparse.SUPPRESS)

        parser.add_argument('--service-name',
                            metavar='<service-name>',
                            default=utils.env('CINDER_SERVICE_NAME'),
                            help=_('Service name. '
                            'Default=env[CINDER_SERVICE_NAME].'))
        parser.add_argument('--service_name',
                            help=argparse.SUPPRESS)

        parser.add_argument('--volume-service-name',
                            metavar='<volume-service-name>',
                            default=utils.env('CINDER_VOLUME_SERVICE_NAME'),
                            help=_('Volume service name. '
                            'Default=env[CINDER_VOLUME_SERVICE_NAME].'))
        parser.add_argument('--volume_service_name',
                            help=argparse.SUPPRESS)

        parser.add_argument('--os-endpoint-type',
                            metavar='<os-endpoint-type>',
                            default=utils.env('CINDER_ENDPOINT_TYPE',
                            default=utils.env('OS_ENDPOINT_TYPE',
                            default=DEFAULT_CINDER_ENDPOINT_TYPE)),
                            help=_('Endpoint type, which is publicURL or '
                            'internalURL. '
                            'Default=env[OS_ENDPOINT_TYPE] or '
                            'nova env[CINDER_ENDPOINT_TYPE] or %s.')
                            % DEFAULT_CINDER_ENDPOINT_TYPE)
        parser.add_argument('--os_endpoint_type',
                            help=argparse.SUPPRESS)

        parser.add_argument('--os-volume-api-version',
                            metavar='<volume-api-ver>',
                            default=utils.env('OS_VOLUME_API_VERSION',
                                              default=None),
                            help=_('Block Storage API version. '
                            'Accepts X, X.Y (where X is major and Y is minor '
                            'part).  NOTE: this client accepts only \'3\' for '
                            'the major version. '
                            'Default=env[OS_VOLUME_API_VERSION].'))
        parser.add_argument('--os_volume_api_version',
                            help=argparse.SUPPRESS)

        parser.add_argument('--os-endpoint',
                            metavar='<os-endpoint>',
                            dest='os_endpoint',
                            default=utils.env('CINDER_ENDPOINT'),
                            help=_("Use this API endpoint instead of the "
                            "Service Catalog. Defaults to "
                            "env[CINDER_ENDPOINT]."))
        parser.add_argument('--os_endpoint',
                            help=argparse.SUPPRESS)

        parser.add_argument('--retries',
                            metavar='<retries>',
                            type=int,
                            default=0,
                            help=_('Number of retries.'))

        parser.set_defaults(func=self.do_help)
        parser.set_defaults(command='')

        if osprofiler_profiler:
            parser.add_argument('--profile',
                                metavar='HMAC_KEY',
                                default=utils.env('OS_PROFILE'),
                                help=_('HMAC key to use for encrypting '
                                'context data for performance profiling '
                                'of operation. This key needs to match the '
                                'one configured on the cinder api server. '
                                'Without key the profiling will not be '
                                'triggered even if osprofiler is enabled '
                                'on server side. Defaults to '
                                'env[OS_PROFILE].'))

        self._append_global_identity_args(parser)

        return parser

    def _append_global_identity_args(self, parser):
        loading.register_session_argparse_arguments(parser)

        # Use "password" auth plugin as default and keep the explicit
        # "--os-token" arguments below for backward compatibility.
        default_auth_plugin = 'password'

        # Passing [] to loading.register_auth_argparse_arguments to avoid
        # the auth_type being overridden by the command line.
        loading.register_auth_argparse_arguments(
            parser, [], default=default_auth_plugin)

        parser.add_argument(
            '--os-auth-strategy', metavar='<auth-strategy>',
            default=utils.env('OS_AUTH_STRATEGY', default='keystone'),
            help=_('Authentication strategy (Env: OS_AUTH_STRATEGY'
                   ', default keystone). For now, any other value will'
                   ' disable the authentication.'))
        parser.add_argument(
            '--os_auth_strategy',
            help=argparse.SUPPRESS)

        # Change os_auth_type default value defined by
        # register_auth_argparse_arguments to be backward compatible
        # with OS_AUTH_SYSTEM.
        env_plugin = utils.env('OS_AUTH_TYPE',
                               'OS_AUTH_PLUGIN',
                               'OS_AUTH_SYSTEM')
        parser.set_defaults(os_auth_type=env_plugin)
        parser.add_argument('--os_auth_type',
                            help=argparse.SUPPRESS)

        parser.set_defaults(os_username=utils.env('OS_USERNAME',
                                                  'CINDER_USERNAME'))
        parser.add_argument('--os_username',
                            help=argparse.SUPPRESS)

        parser.set_defaults(os_password=utils.env('OS_PASSWORD',
                                                  'CINDER_PASSWORD'))
        parser.add_argument('--os_password',
                            help=argparse.SUPPRESS)

        parser.set_defaults(os_project_name=utils.env('OS_PROJECT_NAME',
                                                      'CINDER_PROJECT_ID'))
        parser.add_argument(
            '--os_project_name',
            help=argparse.SUPPRESS)

        parser.set_defaults(os_project_id=utils.env('OS_PROJECT_ID',
                                                    'CINDER_PROJECT_ID'))
        parser.add_argument(
            '--os_project_id',
            help=argparse.SUPPRESS)

        parser.set_defaults(os_auth_url=utils.env('OS_AUTH_URL',
                                                  'CINDER_URL'))
        parser.add_argument('--os_auth_url',
                            help=argparse.SUPPRESS)

        parser.set_defaults(os_user_id=utils.env('OS_USER_ID'))
        parser.add_argument(
            '--os_user_id',
            help=argparse.SUPPRESS)

        parser.set_defaults(
            os_user_domain_id=utils.env('OS_USER_DOMAIN_ID'))
        parser.add_argument(
            '--os_user_domain_id',
            help=argparse.SUPPRESS)

        parser.set_defaults(
            os_user_domain_name=utils.env('OS_USER_DOMAIN_NAME'))
        parser.add_argument(
            '--os_user_domain_name',
            help=argparse.SUPPRESS)

        parser.set_defaults(
            os_project_domain_id=utils.env('OS_PROJECT_DOMAIN_ID'))

        parser.set_defaults(
            os_project_domain_name=utils.env('OS_PROJECT_DOMAIN_NAME'))

        parser.set_defaults(
            os_region_name=utils.env('OS_REGION_NAME',
                                     'CINDER_REGION_NAME'))
        parser.add_argument('--os_region_name',
                            help=argparse.SUPPRESS)

        parser.set_defaults(os_token=utils.env('OS_TOKEN'))
        parser.add_argument(
            '--os_token',
            help=argparse.SUPPRESS)

        parser.add_argument(
            '--os-url', metavar='<url>',
            default=utils.env('OS_URL'),
            help=_('Defaults to env[OS_URL].'))
        parser.add_argument(
            '--os_url',
            help=argparse.SUPPRESS)

        parser.set_defaults(insecure=utils.env('CINDERCLIENT_INSECURE',
                                               default=False))

    def get_subcommand_parser(self, version, do_help=False, input_args=None):
        parser = self.get_base_parser()

        self.subcommands = {}
        subparsers = parser.add_subparsers(metavar='<subcommand>')

        actions_module = importutils.import_module(V3_SHELL)

        self._find_actions(subparsers, actions_module, version, do_help,
                           input_args)
        self._find_actions(subparsers, self, version, do_help, input_args)

        for extension in self.extensions:
            self._find_actions(subparsers, extension.module, version, do_help,
                               input_args)

        self._add_bash_completion_subparser(subparsers)

        return parser

    def _add_bash_completion_subparser(self, subparsers):
        subparser = subparsers.add_parser(
            'bash_completion',
            add_help=False,
            formatter_class=OpenStackHelpFormatter)

        self.subcommands['bash_completion'] = subparser
        subparser.set_defaults(func=self.do_bash_completion)

    def _build_versioned_help_message(self, start_version, end_version):
        if start_version and end_version:
            msg = (_(" (Supported by API versions %(start)s - %(end)s)")
                % {"start": start_version.get_string(),
                   "end": end_version.get_string()})
        elif start_version:
            msg = (_(" (Supported by API version %(start)s and later)")
                % {"start": start_version.get_string()})
        else:
            msg = (_(" (Supported until API version %(end)s)")
                % {"end": end_version.get_string()})
        return str(msg)

    def _find_actions(self, subparsers, actions_module, version,
                      do_help, input_args):
        for attr in (a for a in dir(actions_module) if a.startswith('do_')):
            # I prefer to be hyphen-separated instead of underscores.
            command = attr[3:].replace('_', '-')
            callback = getattr(actions_module, attr)
            desc = callback.__doc__ or ''
            action_help = desc.strip().split('\n')[0]
            if hasattr(callback, "versioned"):
                additional_msg = ""
                subs = api_versions.get_substitutions(
                    utils.get_function_name(callback))
                if do_help:
                    additional_msg = self._build_versioned_help_message(
                        subs[0].start_version, subs[-1].end_version)
                    if version.is_latest():
                        additional_msg += HINT_HELP_MSG
                subs = [versioned_method for versioned_method in subs
                        if version.matches(versioned_method.start_version,
                                           versioned_method.end_version)]
                if not subs:
                    # There is no proper versioned method.
                    continue
                # Use the "latest" substitution.
                callback = subs[-1].func
                desc = callback.__doc__ or desc
                action_help = desc.strip().split('\n')[0]
                action_help += additional_msg

            exclusive_args = getattr(callback, 'exclusive_args', {})
            arguments = getattr(callback, 'arguments', [])

            subparser = subparsers.add_parser(
                command,
                help=action_help,
                description=desc,
                add_help=False,
                formatter_class=OpenStackHelpFormatter)

            subparser.add_argument('-h', '--help',
                                   action='help',
                                   help=argparse.SUPPRESS,)

            self.subcommands[command] = subparser
            self._add_subparser_args(subparser, arguments, version, do_help,
                                     input_args, command)
            self._add_subparser_exclusive_args(subparser, exclusive_args,
                                               version, do_help, input_args,
                                               command)
            subparser.set_defaults(func=callback)

    def _add_subparser_args(self, subparser, arguments, version, do_help,
                            input_args, command):
        # NOTE(ntpttr): We get a counter for each argument in this
        # command here because during the microversion check we only
        # want to raise an exception if no version of the argument
        # matches the current microversion. The exception will only
        # be raised after the last instance of a particular argument
        # fails the check.
        arg_counter = collections.defaultdict(int)
        for (args, kwargs) in arguments:
            arg_counter[args[0]] += 1

        for (args, kwargs) in arguments:
            start_version = kwargs.get("start_version", None)
            start_version = api_versions.APIVersion(start_version)
            end_version = kwargs.get('end_version', None)
            end_version = api_versions.APIVersion(end_version)
            if do_help and (start_version or end_version):
                kwargs["help"] = kwargs.get("help", "") + (
                    self._build_versioned_help_message(start_version,
                                                       end_version))
            if not version.matches(start_version, end_version):
                if args[0] in input_args and command == input_args[0]:
                    if arg_counter[args[0]] == 1:
                        # This is the last version of this argument,
                        # raise the exception.
                        raise exc.UnsupportedAttribute(args[0],
                            start_version, end_version)
                    arg_counter[args[0]] -= 1
                continue
            kw = kwargs.copy()
            kw.pop("start_version", None)
            kw.pop("end_version", None)
            subparser.add_argument(*args, **kw)

    def _add_subparser_exclusive_args(self, subparser, exclusive_args,
                                      version, do_help, input_args, command):
        for group_name, arguments in exclusive_args.items():
            if group_name == '__required__':
                continue
            required = exclusive_args['__required__'][group_name]
            exclusive_group = subparser.add_mutually_exclusive_group(
                required=required)
            self._add_subparser_args(exclusive_group, arguments,
                                     version, do_help, input_args, command)

    def setup_debugging(self, debug):
        if not debug:
            return

        streamhandler = logging.StreamHandler()
        streamformat = "%(levelname)s (%(module)s:%(lineno)d) %(message)s"
        streamhandler.setFormatter(logging.Formatter(streamformat))
        logger.setLevel(logging.DEBUG if debug else logging.WARNING)
        logger.addHandler(streamhandler)

        self.client_logger = logging.getLogger(client.__name__)
        ch = logging.StreamHandler()
        self.client_logger.setLevel(logging.DEBUG)
        self.client_logger.addHandler(ch)
        if hasattr(requests, 'logging'):
            requests.logging.getLogger(requests.__name__).addHandler(ch)

        self.ks_logger = logging.getLogger("keystoneauth")
        self.ks_logger.setLevel(logging.DEBUG)

    def _delimit_metadata_args(self, argv):
        """This function adds -- separator at the appropriate spot
        """
        word = '--metadata'
        tmp = []
        # flag is true in between metadata option and next option
        metadata_options = False
        if word in argv:
            for arg in argv:
                if arg == word:
                    metadata_options = True
                elif metadata_options:
                    if arg.startswith('--'):
                        metadata_options = False
                    elif '=' not in arg:
                        tmp.append(u'--')
                        metadata_options = False
                tmp.append(arg)
            return tmp
        else:
            return argv

    @staticmethod
    def _validate_input_api_version(options):
        if not options.os_volume_api_version:
            api_version = api_versions.APIVersion(api_versions.MAX_VERSION)
        else:
            api_version = api_versions.get_api_version(
                options.os_volume_api_version)
        return api_version

    @staticmethod
    def downgrade_warning(requested, discovered):
        logger.warning("API version %s requested, " % requested.get_string())
        logger.warning("downgrading to %s based on server support." %
                       discovered.get_string())

    def check_duplicate_filters(self, argv, filter):
        resource = RESOURCE_FILTERS[filter]
        filters = []
        for opt in range(len(argv)):
            if argv[opt].startswith('--'):
                if argv[opt] == '--filters':
                    key, __ = argv[opt + 1].split('=')
                    if key in resource:
                        filters.append(key)
                elif argv[opt][2:] in resource:
                    filters.append(argv[opt][2:])

        if len(set(filters)) != len(filters):
            raise exc.CommandError(
                "Filters are only allowed to be passed once.")

    def main(self, argv):
        # Parse args once to find version and debug settings
        for filter in FILTER_CHECK:
            if filter in argv:
                self.check_duplicate_filters(argv, filter)
                break
        parser = self.get_base_parser()
        (options, args) = parser.parse_known_args(argv)
        self.setup_debugging(options.debug)
        api_version_input = True
        self.options = options

        do_help = ('help' in argv) or (
            '--help' in argv) or ('-h' in argv) or not argv

        api_version = self._validate_input_api_version(options)

        # build available subcommands based on version
        major_version_string = "%s" % api_version.ver_major
        self.extensions = client.discover_extensions(major_version_string)
        self._run_extension_hooks('__pre_parse_args__')

        subcommand_parser = self.get_subcommand_parser(api_version,
                                                       do_help, args)
        self.parser = subcommand_parser

        if argv and len(argv) > 1 and '--help' in argv:
            argv = [x for x in argv if x != '--help']
            if argv[0] in self.subcommands:
                self.subcommands[argv[0]].print_help()
                return 0

        if options.help or not argv:
            subcommand_parser.print_help()
            return 0

        argv = self._delimit_metadata_args(argv)
        args = subcommand_parser.parse_args(argv)
        self._run_extension_hooks('__post_parse_args__', args)

        # Short-circuit and deal with help right away.
        if args.func == self.do_help:
            self.do_help(args)
            return 0
        elif args.func == self.do_bash_completion:
            self.do_bash_completion(args)
            return 0

        (os_username, os_password, os_project_name, os_auth_url,
         os_region_name, os_project_id, endpoint_type,
         service_type, service_name, volume_service_name, os_endpoint,
         cacert, os_auth_type) = (
             args.os_username, args.os_password,
             args.os_project_name, args.os_auth_url,
             args.os_region_name, args.os_project_id,
             args.os_endpoint_type,
             args.service_type, args.service_name,
             args.volume_service_name,
             args.os_endpoint, args.os_cacert,
             args.os_auth_type)
        auth_session = None

        if os_auth_type and os_auth_type != "keystone":
            auth_plugin = loading.load_auth_from_argparse_arguments(
                self.options)
            auth_session = loading.load_session_from_argparse_arguments(
                self.options, auth=auth_plugin)
        else:
            auth_plugin = None

        if not service_type:
            service_type = client.SERVICE_TYPES[major_version_string]

        # FIXME(usrleon): Here should be restrict for project id same as
        # for os_username or os_password but for compatibility it is not.

        # V3 stuff
        project_info_provided = ((self.options.os_project_name and
                                  (self.options.os_project_domain_name or
                                   self.options.os_project_domain_id)) or
                                 self.options.os_project_id or
                                 self.options.os_project_name)

        # NOTE(e0ne): if auth_session exists it means auth plugin created
        # session and we don't need to check for password and other
        # authentification-related things.
        if not utils.isunauthenticated(args.func) and not auth_session:
            if not os_password:
                # No password, If we've got a tty, try prompting for it
                if hasattr(sys.stdin, 'isatty') and sys.stdin.isatty():
                    # Check for Ctl-D
                    try:
                        os_password = getpass.getpass('OS Password: ')
                        # Initialize options.os_password with password
                        # input from tty. It is used in _get_keystone_session.
                        options.os_password = os_password
                    except EOFError:
                        pass
                # No password because we didn't have a tty or the
                # user Ctl-D when prompted.
                if not os_password:
                    raise exc.CommandError("You must provide a password "
                                           "through --os-password, "
                                           "env[OS_PASSWORD] "
                                           "or, prompted response.")

            if not project_info_provided:
                raise exc.CommandError(_(
                    "You must provide a project_id or project_name (with "
                    "project_domain_name or project_domain_id) via "
                    "  --os-project-id (env[OS_PROJECT_ID])"
                    "  --os-project-name (env[OS_PROJECT_NAME]),"
                    "  --os-project-domain-id "
                    "(env[OS_PROJECT_DOMAIN_ID])"
                    "  --os-project-domain-name "
                    "(env[OS_PROJECT_DOMAIN_NAME])"
                ))

            if not os_auth_url:
                raise exc.CommandError(
                    "You must provide an authentication URL "
                    "through --os-auth-url or env[OS_AUTH_URL].")

        if not project_info_provided:
            raise exc.CommandError(_(
                "You must provide a project_id or project_name (with "
                "project_domain_name or project_domain_id) via "
                "  --os-project-id (env[OS_PROJECT_ID])"
                "  --os-project-name (env[OS_PROJECT_NAME]),"
                "  --os-project-domain-id "
                "(env[OS_PROJECT_DOMAIN_ID])"
                "  --os-project-domain-name "
                "(env[OS_PROJECT_DOMAIN_NAME])"
            ))

        if not os_auth_url and not auth_plugin:
            raise exc.CommandError(
                "You must provide an authentication URL "
                "through --os-auth-url or env[OS_AUTH_URL].")

        if not auth_session:
            auth_session = self._get_keystone_session()

        # collect_timing is a keystone session option
        if (not isinstance(auth_session, session.Session)
                and getattr(args, 'collect_timing', False) is True):
            raise exc.AuthorizationFailure("Provided auth plugin doesn't "
                                           "support collect_timing option")

        insecure = self.options.insecure

        client_args = dict(
            region_name=os_region_name,
            tenant_id=os_project_id,
            endpoint_type=endpoint_type,
            extensions=self.extensions,
            service_type=service_type,
            service_name=service_name,
            volume_service_name=volume_service_name,
            os_endpoint=os_endpoint,
            retries=options.retries,
            http_log_debug=args.debug,
            insecure=insecure,
            cacert=cacert, auth_system=os_auth_type,
            auth_plugin=auth_plugin,
            session=auth_session,
            logger=self.ks_logger if auth_session else self.client_logger)

        self.cs = client.Client(
            api_version, os_username,
            os_password, os_project_name, os_auth_url,
            **client_args)

        try:
            if not utils.isunauthenticated(args.func):
                self.cs.authenticate()
        except exc.Unauthorized:
            raise exc.CommandError("OpenStack credentials are not valid.")
        except exc.AuthorizationFailure:
            raise exc.CommandError("Unable to authorize user.")

        # FIXME: this section figuring out the api version could use
        # analysis and refactoring.  See
        # https://review.opendev.org/c/openstack/python-cinderclient/+/766882/
        # for some ideas.
        endpoint_api_version = None
        # Try to get the API version from the endpoint URL.  If that fails fall
        # back to trying to use what the user specified via
        # --os-volume-api-version or with the OS_VOLUME_API_VERSION environment
        # variable.  Fail safe is to use the default API setting.
        try:
            endpoint_api_version = \
                self.cs.get_volume_api_version_from_endpoint()
        except exc.UnsupportedVersion:
            endpoint_api_version = options.os_volume_api_version
            # FIXME: api_version_input is initialized as True at the beginning
            # of this function and never modified
            if api_version_input and endpoint_api_version:
                logger.warning("Cannot determine the API version from "
                               "the endpoint URL. Falling back to the "
                               "user-specified version: %s",
                               endpoint_api_version)
            elif endpoint_api_version:
                logger.warning("Cannot determine the API version from the "
                               "endpoint URL or user input. Falling back "
                               "to the default API version: %s",
                               endpoint_api_version)
            else:
                msg = _("Cannot determine API version.  Please specify by "
                        "using --os-volume-api-version option.")
                raise exc.UnsupportedVersion(msg)

        API_MIN_VERSION = api_versions.APIVersion(api_versions.MIN_VERSION)
        # FIXME: the endpoint_api_version[0] can ONLY be '3' now, so the
        # above line should probably be ripped out and this condition removed
        if endpoint_api_version[0] == '3':
            disc_client = client.Client(API_MIN_VERSION,
                                        os_username,
                                        os_password,
                                        os_project_name,
                                        os_auth_url,
                                        **client_args)
            self.cs, discovered_version = self._discover_client(
                disc_client,
                api_version,
                args.os_endpoint_type,
                args.service_type,
                os_username,
                os_password,
                os_project_name,
                os_auth_url,
                client_args)

            if discovered_version < api_version:
                self.downgrade_warning(api_version, discovered_version)

        profile = osprofiler_profiler and options.profile
        if profile:
            osprofiler_profiler.init(options.profile)

        try:
            args.func(self.cs, args)
        finally:
            if profile:
                trace_id = osprofiler_profiler.get().get_base_id()
                print("Trace ID: %s" % trace_id)
                print("To display trace use next command:\n"
                      "osprofiler trace show --html %s " % trace_id)

            if getattr(args, 'collect_timing', False) is True:
                self._print_timings(auth_session)

    def _print_timings(self, session):
        timings = session.get_timings()
        utils.print_list(
            timings,
            fields=('method', 'url', 'seconds'),
            sortby_index=None,
            formatters={'seconds': lambda r: r.elapsed.total_seconds()})

    def _discover_client(self,
                         current_client,
                         os_api_version,
                         os_endpoint_type,
                         os_service_type,
                         os_username,
                         os_password,
                         os_project_name,
                         os_auth_url,
                         client_args):

        discovered_version = api_versions.discover_version(
            current_client,
            os_api_version)

        if not os_endpoint_type:
            os_endpoint_type = DEFAULT_CINDER_ENDPOINT_TYPE

        if not os_service_type:
            os_service_type = self._discover_service_type(discovered_version)

        API_MIN_VERSION = api_versions.APIVersion(api_versions.MIN_VERSION)

        if (discovered_version != API_MIN_VERSION or
                os_service_type != 'volume' or
                os_endpoint_type != DEFAULT_CINDER_ENDPOINT_TYPE):
            client_args['service_type'] = os_service_type
            client_args['endpoint_type'] = os_endpoint_type

            return (client.Client(discovered_version,
                                  os_username,
                                  os_password,
                                  os_project_name,
                                  os_auth_url,
                                  **client_args),
                    discovered_version)
        else:
            return current_client, discovered_version

    def _discover_service_type(self, discovered_version):
        # FIXME: this function is either no longer needed or could use a
        # refactoring.  The official service type is 'block-storage',
        # which isn't even present here.  (Devstack creates 2 service
        # types which it maps to v3: 'block-storage' and 'volumev3'.
        # The default 'catalog_type' in tempest is 'volumev3'.)
        SERVICE_TYPES = {'1': 'volume', '2': 'volumev2', '3': 'volumev3'}
        major_version = discovered_version.get_major_version()
        service_type = SERVICE_TYPES[major_version]
        return service_type

    def _run_extension_hooks(self, hook_type, *args, **kwargs):
        """Runs hooks for all registered extensions."""
        for extension in self.extensions:
            extension.run_hooks(hook_type, *args, **kwargs)

    def do_bash_completion(self, args):
        """Prints arguments for bash_completion.

        Prints all commands and options to stdout so that the
        cinder.bash_completion script does not have to hard code them.
        """
        commands = set()
        options = set()
        for sc_str, sc in list(self.subcommands.items()):
            commands.add(sc_str)
            for option in sc._optionals._option_string_actions:
                options.add(option)

        commands.remove('bash-completion')
        commands.remove('bash_completion')
        print(' '.join(commands | options))

    @utils.arg('command', metavar='<subcommand>', nargs='?',
               help='Shows help for <subcommand>.')
    def do_help(self, args):
        """
        Shows help about this program or one of its subcommands.
        """
        if args.command:
            if args.command in self.subcommands:
                self.subcommands[args.command].print_help()
            else:
                raise exc.CommandError("'%s' is not a valid subcommand" %
                                       args.command)
        else:
            self.parser.print_help()

    def get_v2_auth(self, v2_auth_url):

        username = self.options.os_username
        password = self.options.os_password
        tenant_id = self.options.os_project_id
        tenant_name = self.options.os_project_name

        return v2_auth.Password(
            v2_auth_url,
            username=username,
            password=password,
            tenant_id=tenant_id,
            tenant_name=tenant_name)

    def get_v3_auth(self, v3_auth_url):

        username = self.options.os_username
        user_id = self.options.os_user_id
        user_domain_name = self.options.os_user_domain_name
        user_domain_id = self.options.os_user_domain_id
        password = self.options.os_password
        project_id = self.options.os_project_id
        project_name = self.options.os_project_name
        project_domain_name = self.options.os_project_domain_name
        project_domain_id = self.options.os_project_domain_id

        return v3_auth.Password(
            v3_auth_url,
            username=username,
            password=password,
            user_id=user_id,
            user_domain_name=user_domain_name,
            user_domain_id=user_domain_id,
            project_id=project_id,
            project_name=project_name,
            project_domain_name=project_domain_name,
            project_domain_id=project_domain_id,
        )

    def _discover_auth_versions(self, session, auth_url):
        # discover the API versions the server is supporting based on the
        # given URL
        v2_auth_url = None
        v3_auth_url = None
        try:
            ks_discover = discover.Discover(session=session, url=auth_url)
            v2_auth_url = ks_discover.url_for('2.0')
            v3_auth_url = ks_discover.url_for('3.0')
        except exceptions.DiscoveryFailure:
            # Discovery response mismatch. Raise the error
            raise
        except Exception:
            # Some public clouds throw some other exception or doesn't support
            # discovery. In that case try to determine version from auth_url
            # API version from the original URL
            url_parts = urlparse.urlparse(auth_url)
            (scheme, netloc, path, params, query, fragment) = url_parts
            path = path.lower()
            if path.startswith('/v3'):
                v3_auth_url = auth_url
            elif path.startswith('/v2'):
                v2_auth_url = auth_url
            else:
                raise exc.CommandError('Unable to determine the Keystone'
                                       ' version to authenticate with '
                                       'using the given auth_url.')

        return (v2_auth_url, v3_auth_url)

    def _get_keystone_session(self, **kwargs):
        # first create a Keystone session
        cacert = self.options.os_cacert or None
        cert = self.options.os_cert or None
        if cert and self.options.os_key:
            cert = cert, self.options.os_key

        insecure = self.options.insecure or False

        if insecure:
            verify = False
        else:
            verify = cacert or True

        ks_session = session.Session(verify=verify, cert=cert)
        # discover the supported keystone versions using the given url
        (v2_auth_url, v3_auth_url) = self._discover_auth_versions(
            session=ks_session,
            auth_url=self.options.os_auth_url)

        username = self.options.os_username or None
        user_domain_name = self.options.os_user_domain_name or None
        user_domain_id = self.options.os_user_domain_id or None

        auth = None
        if v3_auth_url and v2_auth_url:
            # support both v2 and v3 auth. Use v3 if possible.
            if username:
                if user_domain_name or user_domain_id:
                    # use v3 auth
                    auth = self.get_v3_auth(v3_auth_url)
                else:
                    # use v2 auth
                    auth = self.get_v2_auth(v2_auth_url)

        elif v3_auth_url:
            # support only v3
            auth = self.get_v3_auth(v3_auth_url)
        elif v2_auth_url:
            # support only v2
            auth = self.get_v2_auth(v2_auth_url)
        else:
            raise exc.CommandError('Unable to determine the Keystone version '
                                   'to authenticate with using the given '
                                   'auth_url.')

        ks_session.auth = auth
        return ks_session

# I'm picky about my shell help.


class OpenStackHelpFormatter(argparse.HelpFormatter):

    def start_section(self, heading):
        # Title-case the headings
        heading = heading.title()
        super(OpenStackHelpFormatter, self).start_section(heading)


def main():
    try:
        OpenStackCinderShell().main(sys.argv[1:])
    except KeyboardInterrupt:
        print("... terminating cinder client", file=sys.stderr)
        sys.exit(130)
    except Exception as e:
        logger.debug(e, exc_info=1)
        print("ERROR: %s" % str(e), file=sys.stderr)
        sys.exit(1)


if __name__ == "__main__":

    main()
