/* ============================================================
 *
 * This file is a part of digiKam project
 * https://www.digikam.org
 *
 * Date        : 2009-06-15
 * Description : multi-languages string editor
 *
 * SPDX-FileCopyrightText: 2009-2024 by Gilles Caulier <caulier dot gilles at gmail dot com>
 *
 * SPDX-License-Identifier: GPL-2.0-or-later
 *
 * ============================================================ */

#include "altlangstredit_p.h"

namespace Digikam
{

/**
 * NOTE: We cannot use KLocale::allLanguagesList() here because KDE only
 * support 2 characters country codes. XMP require 2+2 characters language+country
 * following RFC 3066.
 * The format is based on xx-XX using ISO-639 two-letter code with ISO-3166 two-letter country code.
 *
 * List version 1       : babelwiki.babelzilla.org/index.php?title=Language_codes
 * List version 2 update: xml.coverpages.org/TexinUsingLangID.html
 * List version 3 update: List of ISO 639-1 codes (https://en.wikipedia.org/wiki/List_of_ISO_639-1_codes)
 *                        List of ISO 639-2 codes (https://en.wikipedia.org/wiki/List_of_ISO_639-2_codes)
 *                        List of ISO 639 macrolanguage (https://en.wikipedia.org/wiki/ISO_639_macrolanguage)
 *
 * NOTE: this lists of RFC3066 map is also used un conversion map from DOnlineTranslator class.
 *       Any changes here must be also backported to DOnlineTranslator class.
 */
const LanguageCodeMap s_rfc3066ForXMP =
{
    // The first one from the list is the Default Language code specified by XMP paper.

    { QLatin1String("x-default"), kli18nc("@info: country code", "Default Language")                                                },

    // the rest follow the RFC 3066.

    { QLatin1String("af-ZA"),     kli18nc("@info: country code", "Afrikaans (South Africa)")                                        },
    { QLatin1String("am-ET"),     kli18nc("@info: country code", "Amharic (Ethiopia)")                                              },
    { QLatin1String("apc-TR"),    kli18nc("@info: country code", "North Levantine Arabic (Turkey)")                                 },
    { QLatin1String("ar-AE"),     kli18nc("@info: country code", "Arabic (UAE)")                                                    },
    { QLatin1String("ar-BH"),     kli18nc("@info: country code", "Arabic (Bahrain)")                                                },
    { QLatin1String("ar-DZ"),     kli18nc("@info: country code", "Arabic (Algeria)")                                                },
    { QLatin1String("ar-EG"),     kli18nc("@info: country code", "Arabic (Egypt)")                                                  },
    { QLatin1String("ar-IQ"),     kli18nc("@info: country code", "Arabic (Iraq)")                                                   },
    { QLatin1String("ar-IL"),     kli18nc("@info: country code", "Arabic (Israel)")                                                 },
    { QLatin1String("ar-JO"),     kli18nc("@info: country code", "Arabic (Jordan)")                                                 },
    { QLatin1String("ar-KW"),     kli18nc("@info: country code", "Arabic (Kuwait)")                                                 },
    { QLatin1String("ar-LB"),     kli18nc("@info: country code", "Arabic (Lebanon)")                                                },
    { QLatin1String("ar-LY"),     kli18nc("@info: country code", "Arabic (Libya)")                                                  },
    { QLatin1String("ar-MA"),     kli18nc("@info: country code", "Arabic (Morocco)")                                                },
    { QLatin1String("ar-MR"),     kli18nc("@info: country code", "Arabic (Mauritania)")                                             },
    { QLatin1String("ar-PS"),     kli18nc("@info: country code", "Arabic (Palestinian West Bank and Gaza)")                         },
    { QLatin1String("ar-OM"),     kli18nc("@info: country code", "Arabic (Oman)")                                                   },
    { QLatin1String("ar-QA"),     kli18nc("@info: country code", "Arabic (Qatar)")                                                  },
    { QLatin1String("ar-SA"),     kli18nc("@info: country code", "Arabic (Saudi Arabia)")                                           },
    { QLatin1String("ar-SD"),     kli18nc("@info: country code", "Arabic (Sudan)")                                                  },
    { QLatin1String("ar-SO"),     kli18nc("@info: country code", "Arabic (Somalia)")                                                },
    { QLatin1String("ar-SY"),     kli18nc("@info: country code", "Arabic (Syria)")                                                  },
    { QLatin1String("ar-TD"),     kli18nc("@info: country code", "Arabic (Chad)")                                                   },
    { QLatin1String("ar-TN"),     kli18nc("@info: country code", "Arabic (Tunisia)")                                                },
    { QLatin1String("ar-YE"),     kli18nc("@info: country code", "Arabic (Yemen)")                                                  },
    { QLatin1String("as-IN"),     kli18nc("@info: country code", "Assamese (India)")                                                },
    { QLatin1String("az-AZ"),     kli18nc("@info: country code", "Azerbaijani (Republic of Azerbaijan)")                            },
    { QLatin1String("ba-RU"),     kli18nc("@info: country code", "Bashkir (Russia)")                                                },
    { QLatin1String("bn-BD"),     kli18nc("@info: country code", "Bengali (Bangladesh)")                                            },
    { QLatin1String("bn-IN"),     kli18nc("@info: country code", "Bengali (India)")                                                 },
    { QLatin1String("bn-SG"),     kli18nc("@info: country code", "Bengali (Singapore)")                                             },
    { QLatin1String("be-BY"),     kli18nc("@info: country code", "Belarusian (Belarus)")                                            },
    { QLatin1String("bg-BG"),     kli18nc("@info: country code", "Bulgarian (Bulgaria)")                                            },
    { QLatin1String("bn-IN"),     kli18nc("@info: country code", "Bengali (India)")                                                 },
    { QLatin1String("bo-BT"),     kli18nc("@info: country code", "Tibetan (Bhutan)")                                                },
    { QLatin1String("bo-CN"),     kli18nc("@info: country code", "Tibetan (People's Republic of China)")                            },
    { QLatin1String("br-FR"),     kli18nc("@info: country code", "Breton (France)")                                                 },
    { QLatin1String("bs-BA"),     kli18nc("@info: country code", "Bosnian (Bosnia and Herzegovina)")                                },
    { QLatin1String("bs-BG"),     kli18nc("@info: country code", "Bulgarian (Bulgaria)")                                            },
    { QLatin1String("ca-AD"),     kli18nc("@info: country code", "Catalan (Andorra)")                                               },
    { QLatin1String("ca-ES"),     kli18nc("@info: country code", "Catalan (Spain)")                                                 },
    { QLatin1String("ca-FR"),     kli18nc("@info: country code", "Catalan (France)")                                                },
    { QLatin1String("ceb-PH"),    kli18nc("@info: country code", "Cebuano (Philippines)")                                           },
    { QLatin1String("ch-GU"),     kli18nc("@info: country code", "Chamorro (Guam)")                                                 },
    { QLatin1String("ch-MP"),     kli18nc("@info: country code", "Chamorro (Northern Mariana Islands)")                             },
    { QLatin1String("co-FR"),     kli18nc("@info: country code", "Corsican (France)")                                               },
    { QLatin1String("cs-CZ"),     kli18nc("@info: country code", "Czech (Czechia)")                                                 },
    { QLatin1String("cy-GB"),     kli18nc("@info: country code", "Welsh (United Kingdom)")                                          },
    { QLatin1String("da-DK"),     kli18nc("@info: country code", "Danish (Denmark)")                                                },
    { QLatin1String("da-GL"),     kli18nc("@info: country code", "Danish (Greenland)")                                              },
    { QLatin1String("de-AT"),     kli18nc("@info: country code", "German (Austria)")                                                },
    { QLatin1String("de-CH"),     kli18nc("@info: country code", "German (Switzerland)")                                            },
    { QLatin1String("de-DE"),     kli18nc("@info: country code", "German (Germany)")                                                },
    { QLatin1String("de-DK"),     kli18nc("@info: country code", "German (Denmark)")                                                },
    { QLatin1String("de-LI"),     kli18nc("@info: country code", "German (Liechtenstein)")                                          },
    { QLatin1String("de-LU"),     kli18nc("@info: country code", "German (Luxembourg)")                                             },
    { QLatin1String("el-CY"),     kli18nc("@info: country code", "Greek (Cyprus)")                                                  },
    { QLatin1String("el-GR"),     kli18nc("@info: country code", "Greek (Greece)")                                                  },
    { QLatin1String("en-AG"),     kli18nc("@info: country code", "English (Antigua and Barbuda)")                                   },
    { QLatin1String("en-AI"),     kli18nc("@info: country code", "English (Anguilla)")                                              },
    { QLatin1String("en-AS"),     kli18nc("@info: country code", "English (American Samoa)")                                        },
    { QLatin1String("en-AU"),     kli18nc("@info: country code", "English (Australia)")                                             },
    { QLatin1String("en-BB"),     kli18nc("@info: country code", "English (Barbados)")                                              },
    { QLatin1String("en-BM"),     kli18nc("@info: country code", "English (Bermuda)")                                               },
    { QLatin1String("en-BN"),     kli18nc("@info: country code", "English (Brunei)")                                                },
    { QLatin1String("en-BS"),     kli18nc("@info: country code", "English (Bahamas)")                                               },
    { QLatin1String("en-BW"),     kli18nc("@info: country code", "English (Botswana)")                                              },
    { QLatin1String("en-BZ"),     kli18nc("@info: country code", "English (Belize)")                                                },
    { QLatin1String("en-CA"),     kli18nc("@info: country code", "English (Canada)")                                                },
    { QLatin1String("en-CB"),     kli18nc("@info: country code", "English (Caribbean)")                                             },
    { QLatin1String("en-CK"),     kli18nc("@info: country code", "English (Cook Islands)")                                          },
    { QLatin1String("en-CM"),     kli18nc("@info: country code", "English (Cameroon)")                                              },
    { QLatin1String("en-DM"),     kli18nc("@info: country code", "English (Dominica)")                                              },
    { QLatin1String("en-ER"),     kli18nc("@info: country code", "English (Eritrea)")                                               },
    { QLatin1String("en-ET"),     kli18nc("@info: country code", "English (Ethiopia)")                                              },
    { QLatin1String("en-FJ"),     kli18nc("@info: country code", "English (Fiji)")                                                  },
    { QLatin1String("en-FK"),     kli18nc("@info: country code", "English (Falkland Islands)")                                      },
    { QLatin1String("en-FM"),     kli18nc("@info: country code", "English (Micronesia)")                                            },
    { QLatin1String("en-GB"),     kli18nc("@info: country code", "English (United Kingdom)")                                        },
    { QLatin1String("en-GD"),     kli18nc("@info: country code", "English (Grenada)")                                               },
    { QLatin1String("en-GH"),     kli18nc("@info: country code", "English (Ghana)")                                                 },
    { QLatin1String("en-GI"),     kli18nc("@info: country code", "English (Gibraltar)")                                             },
    { QLatin1String("en-GM"),     kli18nc("@info: country code", "English (Gambia)")                                                },
    { QLatin1String("en-GU"),     kli18nc("@info: country code", "English (Guam)")                                                  },
    { QLatin1String("en-GY"),     kli18nc("@info: country code", "English (Guyana)")                                                },
    { QLatin1String("en-IE"),     kli18nc("@info: country code", "English (Ireland)")                                               },
    { QLatin1String("en-IL"),     kli18nc("@info: country code", "English (Israel)")                                                },
    { QLatin1String("en-IN"),     kli18nc("@info: country code", "English (India)")                                                 },
    { QLatin1String("en-IO"),     kli18nc("@info: country code", "English (British Indian Ocean Territ)")                           },
    { QLatin1String("en-JM"),     kli18nc("@info: country code", "English (Jamaica)")                                               },
    { QLatin1String("en-KR"),     kli18nc("@info: country code", "English (Kenya)")                                                 },
    { QLatin1String("en-KI"),     kli18nc("@info: country code", "English (Kiribati)")                                              },
    { QLatin1String("en-KN"),     kli18nc("@info: country code", "English (St. Kitts-Nevis)")                                       },
    { QLatin1String("en-KY"),     kli18nc("@info: country code", "English (Cayman Islands)")                                        },
    { QLatin1String("en-LC"),     kli18nc("@info: country code", "English (St. Lucia)")                                             },
    { QLatin1String("en-LR"),     kli18nc("@info: country code", "English (Liberia)")                                               },
    { QLatin1String("en-LS"),     kli18nc("@info: country code", "English (Lesotho)")                                               },
    { QLatin1String("en-MP"),     kli18nc("@info: country code", "English (Northern Mariana Islands)")                              },
    { QLatin1String("en-MS"),     kli18nc("@info: country code", "English (Montserrat)")                                            },
    { QLatin1String("en-MT"),     kli18nc("@info: country code", "English (Malta)")                                                 },
    { QLatin1String("en-MU"),     kli18nc("@info: country code", "English (Mauritius)")                                             },
    { QLatin1String("en-MW"),     kli18nc("@info: country code", "English (Malawi)")                                                },
    { QLatin1String("en-MY"),     kli18nc("@info: country code", "English (Malaysia)")                                              },
    { QLatin1String("en-NA"),     kli18nc("@info: country code", "English (Namibia)")                                               },
    { QLatin1String("en-NF"),     kli18nc("@info: country code", "English (Norfolk Island)")                                        },
    { QLatin1String("en-NG"),     kli18nc("@info: country code", "English (Nigeria)")                                               },
    { QLatin1String("en-NR"),     kli18nc("@info: country code", "English (Nauru)")                                                 },
    { QLatin1String("en-NU"),     kli18nc("@info: country code", "English (Niue)")                                                  },
    { QLatin1String("en-NZ"),     kli18nc("@info: country code", "English (New Zealand)")                                           },
    { QLatin1String("en-PG"),     kli18nc("@info: country code", "English (Papua New Guinea)")                                      },
    { QLatin1String("en-PH"),     kli18nc("@info: country code", "English (Philippines)")                                           },
    { QLatin1String("en-PK"),     kli18nc("@info: country code", "English (Pakistan)")                                              },
    { QLatin1String("en-PN"),     kli18nc("@info: country code", "English (Pitcairn)")                                              },
    { QLatin1String("en-PR"),     kli18nc("@info: country code", "English (Puerto Rico)")                                           },
    { QLatin1String("en-PW"),     kli18nc("@info: country code", "English (Palau)")                                                 },
    { QLatin1String("en-RW"),     kli18nc("@info: country code", "English (Rwanda)")                                                },
    { QLatin1String("en-SB"),     kli18nc("@info: country code", "English (Solomon Islands)")                                       },
    { QLatin1String("en-SC"),     kli18nc("@info: country code", "English (Seychelles)")                                            },
    { QLatin1String("en-SG"),     kli18nc("@info: country code", "English (Singapore)")                                             },
    { QLatin1String("en-SH"),     kli18nc("@info: country code", "English (St. Helena)")                                            },
    { QLatin1String("en-SL"),     kli18nc("@info: country code", "English (Sierra Leone)")                                          },
    { QLatin1String("en-SO"),     kli18nc("@info: country code", "English (Somalia)")                                               },
    { QLatin1String("en-SZ"),     kli18nc("@info: country code", "English (Swaziland)")                                             },
    { QLatin1String("en-TC"),     kli18nc("@info: country code", "English (Turks and Caicos Islands)")                              },
    { QLatin1String("en-TK"),     kli18nc("@info: country code", "English (Tokelau)")                                               },
    { QLatin1String("en-TO"),     kli18nc("@info: country code", "English (Tonga)")                                                 },
    { QLatin1String("en-TT"),     kli18nc("@info: country code", "English (Trinidad)")                                              },
    { QLatin1String("en-UG"),     kli18nc("@info: country code", "English (Uganda)")                                                },
    { QLatin1String("en-US"),     kli18nc("@info: country code", "English (United States)")                                         },
    { QLatin1String("en-VC"),     kli18nc("@info: country code", "English (St. Vincent and the Grenadi)")                           },
    { QLatin1String("en-VG"),     kli18nc("@info: country code", "English (British Virgin Islands)")                                },
    { QLatin1String("en-VI"),     kli18nc("@info: country code", "English (U.S. Virgin Islands)")                                   },
    { QLatin1String("en-VU"),     kli18nc("@info: country code", "English (Vanuatu)")                                               },
    { QLatin1String("en-WS"),     kli18nc("@info: country code", "English (Western Samoa)")                                         },
    { QLatin1String("en-ZA"),     kli18nc("@info: country code", "English (South Africa)")                                          },
    { QLatin1String("en-ZM"),     kli18nc("@info: country code", "English (Zambia)")                                                },
    { QLatin1String("en-ZW"),     kli18nc("@info: country code", "English (Zimbabwe)")                                              },
    { QLatin1String("eo-FR"),     kli18nc("@info: country code", "Esperanto (France)")                                              },
    { QLatin1String("es-AR"),     kli18nc("@info: country code", "Spanish (Argentina)")                                             },
    { QLatin1String("es-BO"),     kli18nc("@info: country code", "Spanish (Bolivia)")                                               },
    { QLatin1String("es-CL"),     kli18nc("@info: country code", "Spanish (Chile)")                                                 },
    { QLatin1String("es-CO"),     kli18nc("@info: country code", "Spanish (Colombia)")                                              },
    { QLatin1String("es-CR"),     kli18nc("@info: country code", "Spanish (Costa Rica)")                                            },
    { QLatin1String("es-CU"),     kli18nc("@info: country code", "Spanish (Cuba)")                                                  },
    { QLatin1String("es-DO"),     kli18nc("@info: country code", "Spanish (Dominican Republic)")                                    },
    { QLatin1String("es-EC"),     kli18nc("@info: country code", "Spanish (Ecuador)")                                               },
    { QLatin1String("es-ES"),     kli18nc("@info: country code", "Spanish (Spain)")                                                 },
    { QLatin1String("es-GQ"),     kli18nc("@info: country code", "Spanish (Equatorial Guinea)")                                     },
    { QLatin1String("es-GT"),     kli18nc("@info: country code", "Spanish (Guatemala)")                                             },
    { QLatin1String("es-HN"),     kli18nc("@info: country code", "Spanish (Honduras)")                                              },
    { QLatin1String("es-MX"),     kli18nc("@info: country code", "Spanish (Mexico)")                                                },
    { QLatin1String("es-NI"),     kli18nc("@info: country code", "Spanish (Nicaragua)")                                             },
    { QLatin1String("es-PA"),     kli18nc("@info: country code", "Spanish (Panama)")                                                },
    { QLatin1String("es-PE"),     kli18nc("@info: country code", "Spanish (Peru)")                                                  },
    { QLatin1String("es-PR"),     kli18nc("@info: country code", "Spanish (Puerto Rico)")                                           },
    { QLatin1String("es-PY"),     kli18nc("@info: country code", "Spanish (Paraguay)")                                              },
    { QLatin1String("es-SV"),     kli18nc("@info: country code", "Spanish (El Salvador)")                                           },
    { QLatin1String("es-US"),     kli18nc("@info: country code", "Spanish (United States)")                                         },
    { QLatin1String("es-UY"),     kli18nc("@info: country code", "Spanish (Uruguay)")                                               },
    { QLatin1String("es-VE"),     kli18nc("@info: country code", "Spanish (Venezuela)")                                             },
    { QLatin1String("et-EE"),     kli18nc("@info: country code", "Estonian (Estonia)")                                              },
    { QLatin1String("eu-ES"),     kli18nc("@info: country code", "Basque (Basque Country)")                                         },
    { QLatin1String("fa-IR"),     kli18nc("@info: country code", "Persian (Iran)")                                                  },
    { QLatin1String("fi-FI"),     kli18nc("@info: country code", "Finnish (Finland)")                                               },
    { QLatin1String("fil-PH"),    kli18nc("@info: country code", "Filipino (Philippines)")                                          },
    { QLatin1String("fj-FJ"),     kli18nc("@info: country code", "Fijian (Fiji)")                                                   },
    { QLatin1String("fo-FO"),     kli18nc("@info: country code", "Faeroese (Faero Islands)")                                        },
    { QLatin1String("fr-AD"),     kli18nc("@info: country code", "French (Andorra)")                                                },
    { QLatin1String("fr-BE"),     kli18nc("@info: country code", "French (Belgium)")                                                },
    { QLatin1String("fr-BF"),     kli18nc("@info: country code", "French (Burkina Faso)")                                           },
    { QLatin1String("fr-BI"),     kli18nc("@info: country code", "French (Burundi)")                                                },
    { QLatin1String("fr-BJ"),     kli18nc("@info: country code", "French (Benin)")                                                  },
    { QLatin1String("fr-CA"),     kli18nc("@info: country code", "French (Canada)")                                                 },
    { QLatin1String("fr-CD"),     kli18nc("@info: country code", "French (Democratic Republic of Congo)")                           },
    { QLatin1String("fr-CF"),     kli18nc("@info: country code", "French (Central African Republic)")                               },
    { QLatin1String("fr-CG"),     kli18nc("@info: country code", "French (Congo)")                                                  },
    { QLatin1String("fr-CH"),     kli18nc("@info: country code", "French (Switzerland)")                                            },
    { QLatin1String("fr-CI"),     kli18nc("@info: country code", "French (Cote d'Ivoire)")                                          },
    { QLatin1String("fr-CM"),     kli18nc("@info: country code", "French (Cameroon)")                                               },
    { QLatin1String("fr-DJ"),     kli18nc("@info: country code", "French (Djibouti)")                                               },
    { QLatin1String("fr-FR"),     kli18nc("@info: country code", "French (France)")                                                 },
    { QLatin1String("fr-GA"),     kli18nc("@info: country code", "French (Gabon)")                                                  },
    { QLatin1String("fr-GB"),     kli18nc("@info: country code", "French (United Kingdom)")                                         },
    { QLatin1String("fr-GF"),     kli18nc("@info: country code", "French (French Guiana)")                                          },
    { QLatin1String("fr-GN"),     kli18nc("@info: country code", "French (Guinea)")                                                 },
    { QLatin1String("fr-GP"),     kli18nc("@info: country code", "French (Guadeloupe)")                                             },
    { QLatin1String("fr-HT"),     kli18nc("@info: country code", "French (Haiti)")                                                  },
    { QLatin1String("fr-IT"),     kli18nc("@info: country code", "French (Italy)")                                                  },
    { QLatin1String("fr-KM"),     kli18nc("@info: country code", "French (Comoros Islands)")                                        },
    { QLatin1String("fr-LB"),     kli18nc("@info: country code", "French (Lebanon)")                                                },
    { QLatin1String("fr-LU"),     kli18nc("@info: country code", "French (Luxembourg)")                                             },
    { QLatin1String("fr-MC"),     kli18nc("@info: country code", "French (Monaco)")                                                 },
    { QLatin1String("fr-MG"),     kli18nc("@info: country code", "French (Madagascar)")                                             },
    { QLatin1String("fr-ML"),     kli18nc("@info: country code", "French (Mali)")                                                   },
    { QLatin1String("fr-MQ"),     kli18nc("@info: country code", "French (Martinique)")                                             },
    { QLatin1String("fr-NC"),     kli18nc("@info: country code", "French (New Caledonia)")                                          },
    { QLatin1String("fr-PF"),     kli18nc("@info: country code", "French (French Polynesia)")                                       },
    { QLatin1String("fr-PM"),     kli18nc("@info: country code", "French (St. Pierre and Miquelon)")                                },
    { QLatin1String("fr-RE"),     kli18nc("@info: country code", "French (Reunion)")                                                },
    { QLatin1String("fr-RW"),     kli18nc("@info: country code", "French (Rwanda)")                                                 },
    { QLatin1String("fr-SC"),     kli18nc("@info: country code", "French (Seychelles)")                                             },
    { QLatin1String("fr-TD"),     kli18nc("@info: country code", "French (Chad)")                                                   },
    { QLatin1String("fr-TG"),     kli18nc("@info: country code", "French (Togo)")                                                   },
    { QLatin1String("fr-VU"),     kli18nc("@info: country code", "French (Vanuatu)")                                                },
    { QLatin1String("fr-WF"),     kli18nc("@info: country code", "French (Wallis and Futuna)")                                      },
    { QLatin1String("fr-YT"),     kli18nc("@info: country code", "French (Mayotte)")                                                },
    { QLatin1String("fy-NL"),     kli18nc("@info: country code", "Frisian (Netherlands)")                                           },
    { QLatin1String("ga-GB"),     kli18nc("@info: country code", "Irish (United Kingdom)")                                          },
    { QLatin1String("ga-IE"),     kli18nc("@info: country code", "Irish (Ireland)")                                                 },
    { QLatin1String("gd-GB"),     kli18nc("@info: country code", "Gaelic (United Kingdom)")                                         },
    { QLatin1String("gl-ES"),     kli18nc("@info: country code", "Galician (Galicia)")                                              },
    { QLatin1String("gu-IN"),     kli18nc("@info: country code", "Gujarati (India)")                                                },
    { QLatin1String("gv-GB"),     kli18nc("@info: country code", "Manx (United Kingdom)")                                           },
    { QLatin1String("ha-TD"),     kli18nc("@info: country code", "Hausa (Chad)")                                                    },
    { QLatin1String("haw-US"),    kli18nc("@info: country code", "Hawaiian (US state of Hawaii)")                                   },
    { QLatin1String("he-IL"),     kli18nc("@info: country code", "Modern Hebrew (Israel)")                                          },
    { QLatin1String("hi-IN"),     kli18nc("@info: country code", "Hindi (India)")                                                   },
    { QLatin1String("hmn-LA"),    kli18nc("@info: country code", "Hmong (Laos)")                                                    },
    { QLatin1String("hr-BA"),     kli18nc("@info: country code", "Croatian (Bosnia and Herzegovina, Latin)")                        },
    { QLatin1String("hr-HR"),     kli18nc("@info: country code", "Croatian (Croatia)")                                              },
    { QLatin1String("ht-FR"),     kli18nc("@info: country code", "Haitian Creole (France)")                                         },
    { QLatin1String("hu-HU"),     kli18nc("@info: country code", "Hungarian (Hungary)")                                             },
    { QLatin1String("hu-SI"),     kli18nc("@info: country code", "Hungarian (Slovenia)")                                            },
    { QLatin1String("hy-AM"),     kli18nc("@info: country code", "Armenian (Armenia)")                                              },
    { QLatin1String("id-ID"),     kli18nc("@info: country code", "Indonesian (Bahasa Indonesia)")                                   },
    { QLatin1String("ig-NG"),     kli18nc("@info: country code", "Igbo (Nigeria)")                                                  },
    { QLatin1String("ii-CN"),     kli18nc("@info: country code", "Sichuan Yi (People's Republic of China)")                         },
    { QLatin1String("is-IS"),     kli18nc("@info: country code", "Icelandic (Iceland)")                                             },
    { QLatin1String("it-CH"),     kli18nc("@info: country code", "Italian (Switzerland)")                                           },
    { QLatin1String("it-HR"),     kli18nc("@info: country code", "Italian (Croatia)")                                               },
    { QLatin1String("it-IT"),     kli18nc("@info: country code", "Italian (Italy)")                                                 },
    { QLatin1String("it-SI"),     kli18nc("@info: country code", "Italian (Slovenia)")                                              },
    { QLatin1String("it-SM"),     kli18nc("@info: country code", "Italian (San Marino)")                                            },
    { QLatin1String("iw-IL"),     kli18nc("@info: country code", "Hebrew (Israel)")                                                 },
    { QLatin1String("ja-JP"),     kli18nc("@info: country code", "Japanese (Japan)")                                                },
    { QLatin1String("jv-ID"),     kli18nc("@info: country code", "Modern Javanese (Indonesia)")                                     },
    { QLatin1String("jw-ID"),     kli18nc("@info: country code", "Javanese (Indonesia)")                                            },
    { QLatin1String("ka-GE"),     kli18nc("@info: country code", "Georgian (Georgia)")                                              },
    { QLatin1String("kk-KZ"),     kli18nc("@info: country code", "Kazakh (Kazakhstan)")                                             },
    { QLatin1String("kl-GL"),     kli18nc("@info: country code", "Greenlandic (Greenland)")                                         },
    { QLatin1String("km-KH"),     kli18nc("@info: country code", "Khmer (Cambodia)")                                                },
    { QLatin1String("kn-IN"),     kli18nc("@info: country code", "Kannada (India)")                                                 },
    { QLatin1String("ko-KR"),     kli18nc("@info: country code", "Korean (South Korea)")                                            },
    { QLatin1String("ko-KP"),     kli18nc("@info: country code", "Korean (North Korea)")                                            },
    { QLatin1String("ky-KG"),     kli18nc("@info: country code", "Kyrgyz (Kyrgyzstan)")                                             },
    { QLatin1String("ku-IR"),     kli18nc("@info: country code", "Cornish (United Kingdom)")                                        },
    { QLatin1String("kw-GB"),     kli18nc("@info: country code", "Kurdish (Iran)")                                                  },
    { QLatin1String("la-IT"),     kli18nc("@info: country code", "Latin (Italy)")                                                   },
    { QLatin1String("lb-LU"),     kli18nc("@info: country code", "Luxembourgish (Luxembourg)")                                      },
    { QLatin1String("ln-CD"),     kli18nc("@info: country code", "Lingala (Democratic Republic of Congo)")                          },
    { QLatin1String("ln-CG"),     kli18nc("@info: country code", "Lingala (Congo)")                                                 },
    { QLatin1String("lo-LA"),     kli18nc("@info: country code", "Lao (Lao PDR)")                                                   },
    { QLatin1String("lt-LT"),     kli18nc("@info: country code", "Lithuanian (Lithuania)")                                          },
    { QLatin1String("lv-LV"),     kli18nc("@info: country code", "Latvian (Latvia)")                                                },
    { QLatin1String("mg-MG"),     kli18nc("@info: country code", "Malagasy (Madagascar)")                                           },
    { QLatin1String("mhr-RU"),    kli18nc("@info: country code", "Meadow Mari (Russia)")                                            },
    { QLatin1String("mi-NZ"),     kli18nc("@info: country code", "Maori (New Zealand)")                                             },
    { QLatin1String("mk-MK"),     kli18nc("@info: country code", "Macedonian (Macedonia)")                                          },
    { QLatin1String("ml-IN"),     kli18nc("@info: country code", "Malayalam (India)")                                               },
    { QLatin1String("mn-CN"),     kli18nc("@info: country code", "Mongolian (People's Republic of China)")                          },
    { QLatin1String("mn-MN"),     kli18nc("@info: country code", "Mongolian (Mongolia)")                                            },
    { QLatin1String("mr-IN"),     kli18nc("@info: country code", "Marathi (India)")                                                 },
    { QLatin1String("mrj-RU"),    kli18nc("@info: country code", "Hill Mari (Russia)")                                              },
    { QLatin1String("ms-BN"),     kli18nc("@info: country code", "Malay (Brunei Darussalam)")                                       },
    { QLatin1String("ms-MY"),     kli18nc("@info: country code", "Malay (Malaysia)")                                                },
    { QLatin1String("ms-SG"),     kli18nc("@info: country code", "Malay (Singapore)")                                               },
    { QLatin1String("mt-MT"),     kli18nc("@info: country code", "Maltese (Malta)")                                                 },
    { QLatin1String("my-MM"),     kli18nc("@info: country code", "Burmese (Myanmar)")                                               },
    { QLatin1String("nb-NO"),     kli18nc("@info: country code", "Norwegian Bokmål (Norway)")                                       },
    { QLatin1String("ne-NP"),     kli18nc("@info: country code", "Nepali (Nepal)")                                                  },
    { QLatin1String("nl-AN"),     kli18nc("@info: country code", "Dutch (Netherlands Antilles)")                                    },
    { QLatin1String("nl-AW"),     kli18nc("@info: country code", "Dutch (Aruba)")                                                   },
    { QLatin1String("nl-BE"),     kli18nc("@info: country code", "Dutch (Belgium)")                                                 },
    { QLatin1String("nl-NL"),     kli18nc("@info: country code", "Dutch (Netherlands)")                                             },
    { QLatin1String("nl-SR"),     kli18nc("@info: country code", "Dutch (Suriname)")                                                },
    { QLatin1String("no-NO"),     kli18nc("@info: country code", "Norwegian (Norway)")                                              },
    { QLatin1String("nn-NO"),     kli18nc("@info: country code", "Norwegian Nynorsk (Norway)")                                      },
    { QLatin1String("ns-ZA"),     kli18nc("@info: country code", "Sesotho sa Leboa (South Africa)")                                 },
    { QLatin1String("ny-ZM"),     kli18nc("@info: country code", "Chewa (Zambia)")                                                  },
    { QLatin1String("oc-FR"),     kli18nc("@info: country code", "Occitan (France)")                                                },
    { QLatin1String("or-IN"),     kli18nc("@info: country code", "Oriya (India)")                                                   },
    { QLatin1String("pa-IN"),     kli18nc("@info: country code", "Punjabi (India)")                                                 },
    { QLatin1String("pap-AW"),    kli18nc("@info: country code", "Papiamento (Aruba-Dutch Caribbean)")                              },
    { QLatin1String("pl-PL"),     kli18nc("@info: country code", "Polish (Poland)")                                                 },
    { QLatin1String("ps-AF"),     kli18nc("@info: country code", "Pashto (Afghanistan)")                                            },
    { QLatin1String("pt-AO"),     kli18nc("@info: country code", "Portuguese (Angola)")                                             },
    { QLatin1String("pt-BR"),     kli18nc("@info: country code", "Portuguese (Brazil)")                                             },
    { QLatin1String("pt-CV"),     kli18nc("@info: country code", "Portuguese (Cape Verde Islands)")                                 },
    { QLatin1String("pt-GW"),     kli18nc("@info: country code", "Portuguese (Guinea-Bissau)")                                      },
    { QLatin1String("pt-MZ"),     kli18nc("@info: country code", "Portuguese (Mozambique)")                                         },
    { QLatin1String("pt-PT"),     kli18nc("@info: country code", "Portuguese (Portugal)")                                           },
    { QLatin1String("pt-ST"),     kli18nc("@info: country code", "Portuguese (São Tomé e Principe)")                                },
    { QLatin1String("otq-MX"),    kli18nc("@info: country code", "Otomi Querétaro (Mexico)")                                        },
    { QLatin1String("rm-CH"),     kli18nc("@info: country code", "Romansh (Switzerland)")                                           },
    { QLatin1String("ro-RO"),     kli18nc("@info: country code", "Romanian (Romania)")                                              },
    { QLatin1String("ru-RU"),     kli18nc("@info: country code", "Russian (Russia)")                                                },
    { QLatin1String("rw-RW"),     kli18nc("@info: country code", "Kinyarwanda (Rwanda)")                                            },
    { QLatin1String("sa-IN"),     kli18nc("@info: country code", "Sanskrit (India)")                                                },
    { QLatin1String("sd-IN"),     kli18nc("@info: country code", "Sindhi (India)")                                                  },
    { QLatin1String("sd-PK"),     kli18nc("@info: country code", "Sindhi (Pakistan)")                                               },
    { QLatin1String("se-FI"),     kli18nc("@info: country code", "Sami (Northern, Finland)")                                        },
    { QLatin1String("se-NO"),     kli18nc("@info: country code", "Sami (Northern, Norway)")                                         },
    { QLatin1String("se-SE"),     kli18nc("@info: country code", "Sami (Northern, Sweden)")                                         },
    { QLatin1String("si-LK"),     kli18nc("@info: country code", "Sinhala (Sri Lanka)")                                             },
    { QLatin1String("sk-SK"),     kli18nc("@info: country code", "Slovak (Slovakia)")                                               },
    { QLatin1String("sl-SI"),     kli18nc("@info: country code", "Slovenian (Slovenia)")                                            },
    { QLatin1String("sm-WS"),     kli18nc("@info: country code", "Samoan (Samoa)")                                                  },
    { QLatin1String("sn-ZW"),     kli18nc("@info: country code", "Shona (Zimbabwe)")                                                },
    { QLatin1String("so-SO"),     kli18nc("@info: country code", "Somali  (Somalia)")                                               },
    { QLatin1String("sq-AL"),     kli18nc("@info: country code", "Albanian (Albania)")                                              },
    { QLatin1String("sr-BA"),     kli18nc("@info: country code", "Serbian (Bosnia-Herzegovina)")                                    },
    { QLatin1String("sr-RS"),     kli18nc("@info: country code", "Serbian (Serbia)")                                                },
    { QLatin1String("sr-SP"),     kli18nc("@info: country code", "Serbian (Serbia and Montenegro)")                                 },
    { QLatin1String("sr-YU"),     kli18nc("@info: country code", "Serbian (Yugoslavia)")                                            },
    { QLatin1String("ss-SZ"),     kli18nc("@info: country code", "Swati (Swaziland)")                                               },
    { QLatin1String("ss-ZA"),     kli18nc("@info: country code", "Swati (South Africa)")                                            },
    { QLatin1String("st-LS"),     kli18nc("@info: country code", "Sotho (Lesotho)")                                                 },
    { QLatin1String("su-ID"),     kli18nc("@info: country code", "Sundanese (Indonesia)")                                           },
    { QLatin1String("sv-FI"),     kli18nc("@info: country code", "Swedish (Finland)")                                               },
    { QLatin1String("sv-SE"),     kli18nc("@info: country code", "Swedish (Sweden)")                                                },
    { QLatin1String("sw-KE"),     kli18nc("@info: country code", "Swahili (Kenya)")                                                 },
    { QLatin1String("sw-TZ"),     kli18nc("@info: country code", "Swahili (Tanzania)")                                              },
    { QLatin1String("ta-IN"),     kli18nc("@info: country code", "Tamil (India)")                                                   },
    { QLatin1String("ta-SG"),     kli18nc("@info: country code", "Tamil (Singapore)")                                               },
    { QLatin1String("te-IN"),     kli18nc("@info: country code", "Telugu (India)")                                                  },
    { QLatin1String("tg-TJ"),     kli18nc("@info: country code", "Tajik (Tajikistan)")                                              },
    { QLatin1String("th-TH"),     kli18nc("@info: country code", "Thai (Thailand)")                                                 },
    { QLatin1String("tk-TM"),     kli18nc("@info: country code", "Turkmen (Turkmenistan)")                                          },
    { QLatin1String("tlh-US"),    kli18nc("@info: country code", "Klingon (US state)")                                              },
    { QLatin1String("tl-PH"),     kli18nc("@info: country code", "Tagalog (Philippines)")                                           },
    { QLatin1String("tn-BW"),     kli18nc("@info: country code", "Setswana Tswana (Botswana)")                                      },
    { QLatin1String("tn-ZA"),     kli18nc("@info: country code", "Setswana Tswana (South Africa)")                                  },
    { QLatin1String("to-TO"),     kli18nc("@info: country code", "Tongan (Tonga)")                                                  },
    { QLatin1String("tr-BG"),     kli18nc("@info: country code", "Turkish (Bulgaria)")                                              },
    { QLatin1String("tr-CY"),     kli18nc("@info: country code", "Turkish (Cyprus)")                                                },
    { QLatin1String("tr-TR"),     kli18nc("@info: country code", "Turkish (Turkey)")                                                },
    { QLatin1String("tt-RU"),     kli18nc("@info: country code", "Tatar (Russia)")                                                  },
    { QLatin1String("ty-PF"),     kli18nc("@info: country code", "Tahitian (French Polynesia)")                                     },
    { QLatin1String("udm-RU"),    kli18nc("@info: country code", "Udmurt (Russia)")                                                 },
    { QLatin1String("ug-CN"),     kli18nc("@info: country code", "Uighur (People's Republic of China)")                             },
    { QLatin1String("uk-UA"),     kli18nc("@info: country code", "Ukrainian (Ukraine)")                                             },
    { QLatin1String("ur-IN"),     kli18nc("@info: country code", "Urdu (India)")                                                    },
    { QLatin1String("ur-PK"),     kli18nc("@info: country code", "Urdu (Pakistan)")                                                 },
    { QLatin1String("uz-UZ"),     kli18nc("@info: country code", "Uzbek (Uzbekistan)")                                              },
    { QLatin1String("vi-VN"),     kli18nc("@info: country code", "Vietnamese (Vietnam)")                                            },
    { QLatin1String("wo-SN"),     kli18nc("@info: country code", "Wolof (Senegal)")                                                 },
    { QLatin1String("xh-ZA"),     kli18nc("@info: country code", "IsiXhosa Xhosa (South Africa)")                                   },
    { QLatin1String("yi-DE"),     kli18nc("@info: country code", "Yiddish (Germany)")                                               },
    { QLatin1String("yo-NG"),     kli18nc("@info: country code", "Yoruba (Nigeria)")                                                },
    { QLatin1String("yua-MX"),    kli18nc("@info: country code", "Yucatec Maya (Mexico)")                                           },
    { QLatin1String("zh-CN"),     kli18nc("@info: country code", "Chinese (People's Republic of China)")                            },
    { QLatin1String("zh-HK"),     kli18nc("@info: country code", "Chinese (Hong Kong SAR, People's Republic of China)")             },
    { QLatin1String("zh-MO"),     kli18nc("@info: country code", "Chinese (Macao SAR)")                                             },
    { QLatin1String("zh-SG"),     kli18nc("@info: country code", "Chinese (Singapore)")                                             },
    { QLatin1String("zh-TW"),     kli18nc("@info: country code", "Chinese (Taiwan)")                                                },
    { QLatin1String("zn-CH"),     kli18nc("@info: country code", "Simplified Chinese (People's Republic of China)")                 },
    { QLatin1String("zu-ZA"),     kli18nc("@info: country code", "isiZulu Zulu (South Africa)")                                     }
};

} // namespace Digikam
