/*--------------------------------------------------------------------------*/
/* ALBERTA:  an Adaptive multi Level finite element toolbox using           */
/*           Bisectioning refinement and Error control by Residual          */
/*           Techniques for scientific Applications                         */
/*                                                                          */
/* file:     lagrange_1_2d.c                                                */
/*                                                                          */
/* description:  piecewise linear Lagrange elements in 2d                   */
/*                                                                          */
/*--------------------------------------------------------------------------*/
/*                                                                          */
/*  authors:   Alfred Schmidt                                               */
/*             Zentrum fuer Technomathematik                                */
/*             Fachbereich 3 Mathematik/Informatik                          */
/*             Universitaet Bremen                                          */
/*             Bibliothekstr. 2                                             */
/*             D-28359 Bremen, Germany                                      */
/*                                                                          */
/*             Kunibert G. Siebert                                          */
/*             Institut fuer Mathematik                                     */
/*             Universitaet Augsburg                                        */
/*             Universitaetsstr. 14                                         */
/*             D-86159 Augsburg, Germany                                    */
/*                                                                          */
/*  http://www.mathematik.uni-freiburg.de/IAM/ALBERTA                       */
/*                                                                          */
/*  (c) by A. Schmidt and K.G. Siebert (1996-2003)                          */
/*                                                                          */
/*--------------------------------------------------------------------------*/

static const REAL   bary1_2d[3][N_LAMBDA] = {{1.0, 0.0, 0.0, 0.0},
					     {0.0, 1.0, 0.0, 0.0},
					     {0.0, 0.0, 1.0, 0.0}};

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 0                                               */
/*--------------------------------------------------------------------------*/

static REAL phi1v0_2d(const REAL lambda[N_LAMBDA])
{
  return(lambda[0]);
}

static const REAL *grd_phi1v0_2d(const REAL lambda[N_LAMBDA])
{
  static const REAL  grd[N_LAMBDA] = {1.0, 0.0, 0.0, 0.0};

  return(grd);
}

static const REAL (*D2_phi1v0_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA];

  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 1                                               */
/*--------------------------------------------------------------------------*/

static REAL phi1v1_2d(const REAL lambda[N_LAMBDA])
{
  return(lambda[1]);
}

static const REAL *grd_phi1v1_2d(const REAL lambda[N_LAMBDA])
{
  static const REAL  grd[N_LAMBDA] = {0.0, 1.0, 0.0, 0.0};

  return(grd);
}

static const REAL (*D2_phi1v1_2d(const REAL *lambda))[N_LAMBDA]
{
  static const REAL D2[N_LAMBDA][N_LAMBDA];

  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  basisfunction at vertex 2                                               */
/*--------------------------------------------------------------------------*/

static REAL phi1v2_2d(const REAL lambda[N_LAMBDA])
{
  return(lambda[2]);
}

static const REAL *grd_phi1v2_2d(const REAL lambda[N_LAMBDA])
{
  static const REAL  grd[N_LAMBDA] = {0.0, 0.0, 1.0, 0.0};

  return(grd);
}

static const REAL (*D2_phi1v2_2d(const REAL *lambda))[N_LAMBDA]
{
  static  const REAL D2[N_LAMBDA][N_LAMBDA];

  return(D2);
}

/*--------------------------------------------------------------------------*/
/*  functions for combining basisfunctions with coefficients                */
/*--------------------------------------------------------------------------*/

static const DOF *get_dof_indices1_2d(const EL *el, const DOF_ADMIN *admin,
				      DOF *idof)
{
  static DOF  index_vec[N_VERTICES_2D];
  DOF         *rvec = idof ? idof : index_vec;
  int         i, n0 = admin->n0_dof[VERTEX];
  DOF         **dof = el->dof;

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = dof[i][n0];

  return((const DOF *) rvec);
}

static const S_CHAR *get_bound1_2d(const EL_INFO *el_info, S_CHAR *bound)
{
  FUNCNAME("get_bound1_2d");
  static S_CHAR  bound_vec[N_VERTICES_2D];
  S_CHAR         *rvec = bound ? bound : bound_vec;
  int            i;

  TEST_FLAG(FILL_BOUND, el_info);

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = el_info->vertex_bound[i];

  return((const S_CHAR *) rvec);
}

static const int *get_int_vec1_2d(const EL *el, const DOF_INT_VEC *vec,
				  int *ivec)
{
  FUNCNAME("get_int_vec1_2d");
  int            i, n0;
  static int     local_vec[N_VERTICES_2D];
  int            *v = nil, *rvec = ivec ? ivec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);
  n0 = vec->fe_space->admin->n0_dof[VERTEX];

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];

  return((const int *) rvec);
}

static const REAL *get_real_vec1_2d(const EL *el, const DOF_REAL_VEC *vec,
				    REAL *Rvec)
{
  FUNCNAME("get_real_v1_2d");
  int            i, n0;
  static REAL    local_vec[N_VERTICES_2D];
  REAL           *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);
  n0 = vec->fe_space->admin->n0_dof[VERTEX];

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];

  return((const REAL *) rvec);
}

static const REAL_D *get_real_d_vec1_2d(const EL *el, 
					const DOF_REAL_D_VEC *vec,
					REAL_D *Rvec)
{
  FUNCNAME("get_real_d_v1_2d");
  int            i, k, n0;
  static REAL_D  local_vec[N_VERTICES_2D];
  REAL_D         *v = nil, *rvec = Rvec ? Rvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);
  n0 = vec->fe_space->admin->n0_dof[VERTEX];

  for (i = 0; i < N_VERTICES_2D; i++)
    for (k = 0; k < DIM_OF_WORLD; k++)
      rvec[i][k] = v[dof[i][n0]][k];

  return((const REAL_D *) rvec);
}

static const U_CHAR *get_uchar_vec1_2d(const EL *el, const DOF_UCHAR_VEC *vec,
				       U_CHAR *uvec)
{
  FUNCNAME("get_uchar_vec1_2d");
  int            i, n0;
  static U_CHAR  local_vec[N_VERTICES_2D];
  U_CHAR         *v = nil, *rvec = uvec ? uvec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);
  n0 = vec->fe_space->admin->n0_dof[VERTEX];

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];

  return((const U_CHAR *) rvec);
}

static const S_CHAR *get_schar_vec1_2d(const EL *el, const DOF_SCHAR_VEC *vec,
				       S_CHAR *svec)
{
  FUNCNAME("get_schar_vec1_2d");
  int            i, n0;
  static S_CHAR  local_vec[N_VERTICES_2D];
  S_CHAR         *v = nil, *rvec = svec ? svec : local_vec;
  DOF            **dof = el->dof;

  GET_DOF_VEC(v, vec);
  n0 = vec->fe_space->admin->n0_dof[VERTEX];

  for (i = 0; i < N_VERTICES_2D; i++)
    rvec[i] = v[dof[i][n0]];

  return((const S_CHAR *) rvec);
}

static const REAL *interpol1_2d(const EL_INFO *el_info, int no,
				const int *b_no, REAL (*f)(const REAL_D), 
				REAL (*f_loc)(const EL_INFO *,
					      const REAL [N_LAMBDA]), 
				REAL *vec)
{
  FUNCNAME("interpol1_2d");
  static REAL       inter[N_VERTICES_2D];
  REAL             *rvec = vec ? vec : inter;
  int               i;
  const PARAMETRIC *parametric = el_info->mesh->parametric;


  DEBUG_TEST_EXIT(!b_no || (no > 0 && no <= N_VERTICES_2D), 
		  "not for %d points\n", no);

  if(f_loc)
    for (i = 0; i < N_VERTICES_2D; i++)
      rvec[i] = f_loc(el_info, bary1_2d[i]);
  else {
    if (parametric) {
      REAL_D   world[N_VERTICES_2D];

      parametric->init_element(el_info, parametric);
      parametric->coord_to_world(el_info, nil, N_VERTICES_2D, bary1_2d, world);
	
      for (i = 0; i < N_VERTICES_2D; i++)
	rvec[i] = f(world[i]);
    }
    else { /* Vertex coordinates do not have to be calculated. */
      DEBUG_TEST_FLAG(FILL_COORDS, el_info);
      
      rvec[0] = f(el_info->coord[0]);
      rvec[1] = f(el_info->coord[1]);
      rvec[2] = f(el_info->coord[2]);
    }
  }

  if(b_no) { /* Perform resorting if only certain indices are required. */
    REAL tmp[N_VERTICES_2D];

    memcpy(tmp, rvec, N_VERTICES_2D * sizeof(REAL));
    
    for(i = 0; i < no; i++)
      rvec[i] = tmp[b_no[i]];
  }
  
  return((const REAL *) rvec);
}

static const REAL_D *interpol_d1_2d(const EL_INFO *el_info, int no, 
				    const int *b_no,
				    const REAL *(*f)(const REAL_D, REAL_D),
				    const REAL *(*f_loc)(const EL_INFO *,
							 const REAL [N_LAMBDA],
							 REAL_D),
				    REAL_D *vec)
{
  FUNCNAME("interpol_d1_2d");
  static REAL_D     inter[N_VERTICES_2D];
  REAL_D           *rvec = vec ? vec : inter;
  int               i;
  const PARAMETRIC *parametric = el_info->mesh->parametric;


  DEBUG_TEST_EXIT(!b_no || (no > 0 && no <= N_VERTICES_2D), 
		  "not for %d points\n", no);

  if(f_loc)
    for (i = 0; i < N_VERTICES_2D; i++)
      f_loc(el_info, bary1_2d[i], rvec[i]);
  else {
    if (parametric) {
      REAL_D   world[N_VERTICES_2D];

      parametric->init_element(el_info, parametric);	
      parametric->coord_to_world(el_info, nil, N_VERTICES_2D, bary1_2d, world);
	  
      for (i = 0; i < N_VERTICES_2D; i++)
	f(world[i], rvec[i]);
    }
    else { /* Vertex coordinates do not have to be calculated. */
      DEBUG_TEST_FLAG(FILL_COORDS, el_info);
	
      f(el_info->coord[0], rvec[0]);
      f(el_info->coord[1], rvec[1]);
      f(el_info->coord[2], rvec[2]);
    }
  }

  if(b_no) { /* Perform resorting if only certain indices are required. */
    REAL_D tmp[N_VERTICES_2D];

    memcpy(tmp, rvec, N_VERTICES_2D * sizeof(REAL_D));
    
    for(i = 0; i < no; i++)
      COPY_DOW(tmp[b_no[i]], rvec[i]);
  }
  
  return((const REAL_D *) rvec);
}

/*--------------------------------------------------------------------------*/
/*  functions for interpolation/ restriction during refinement/coarsening   */
/*--------------------------------------------------------------------------*/

static void real_refine_inter1_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_refine_inter1_2d");
  EL      *el;
  REAL    *vec = nil;
  DOF     dof_new, dof0, dof1;
  int     n0;

  if (n < 1) return;
  GET_DOF_VEC(vec, drv);
  n0 = drv->fe_space->admin->n0_dof[VERTEX];
  el = list->el_info.el;
  dof0 = el->dof[0][n0];           /* 1st endpoint of refinement edge */
  dof1 = el->dof[1][n0];           /* 2nd endpoint of refinement edge */
  dof_new = el->child[0]->dof[2][n0];  /*     newest vertex is dim==2 */
  vec[dof_new] = 0.5*(vec[dof0] + vec[dof1]);

  return;
}

/*--------------------------------------------------------------------------*/
/*  linear interpolation during coarsening: do nothing                      */
/*--------------------------------------------------------------------------*/

static void real_coarse_restr1_2d(DOF_REAL_VEC *drv, RC_LIST_EL *list, int n)
{
  FUNCNAME("real_coarse_restr1_2d");
  EL      *el;
  REAL    *vec = nil;
  DOF     dof_new, dof0, dof1;
  int     n0;

  if (n < 1) return;
  GET_DOF_VEC(vec, drv);
  n0 = drv->fe_space->admin->n0_dof[VERTEX];
  el = list->el_info.el;
  dof0 = el->dof[0][n0];           /* 1st endpoint of refinement edge */
  dof1 = el->dof[1][n0];           /* 2nd endpoint of refinement edge */
  dof_new = el->child[0]->dof[2][n0];    /*   newest vertex is dim==2 */
  vec[dof0] += 0.5*vec[dof_new];
  vec[dof1] += 0.5*vec[dof_new];

  return;
}

static void real_d_refine_inter1_2d(DOF_REAL_D_VEC *drdv,
				    RC_LIST_EL *list, int n)
{
  FUNCNAME("real_d_refine_inter1_2d");
  EL      *el;
  REAL_D  *vec = nil;
  DOF     dof_new, dof0, dof1;
  int     n0, j;

  if (n < 1) return;
  GET_DOF_VEC(vec, drdv);
  n0 = drdv->fe_space->admin->n0_dof[VERTEX];
  el = list->el_info.el;
  dof0 = el->dof[0][n0];           /* 1st endpoint of refinement edge */
  dof1 = el->dof[1][n0];           /* 2nd endpoint of refinement edge */
  dof_new = el->child[0]->dof[2][n0];  /*     newest vertex is dim==2 */
  for (j = 0; j < DIM_OF_WORLD; j++)
    vec[dof_new][j] = 0.5*(vec[dof0][j] + vec[dof1][j]);

  return;
}

/*--------------------------------------------------------------------------*/
/*  linear interpolation during coarsening: do nothing                      */
/*--------------------------------------------------------------------------*/

static void real_d_coarse_restr1_2d(DOF_REAL_D_VEC *drdv, RC_LIST_EL *list,
				    int n)
{
  FUNCNAME("real_d_coarse_restr1_2d");
  EL      *el;
  REAL_D  *vec = nil;
  DOF     dof_new, dof0, dof1;
  int     n0, j;

  if (n < 1) return;
  GET_DOF_VEC(vec, drdv);
  n0 = drdv->fe_space->admin->n0_dof[VERTEX];
  el = list->el_info.el;
  dof0 = el->dof[0][n0];           /* 1st endpoint of refinement edge */
  dof1 = el->dof[1][n0];           /* 2nd endpoint of refinement edge */
  dof_new = el->child[0]->dof[2][n0];  /*     newest vertex is dim==2 */
  for (j = 0; j < DIM_OF_WORLD; j++)
  {
    vec[dof0][j] += 0.5 * vec[dof_new][j];
    vec[dof1][j] += 0.5 * vec[dof_new][j];
  }

  return;
}

static BAS_FCT      *phi1_2d[3]     = {phi1v0_2d, phi1v1_2d, phi1v2_2d};
static GRD_BAS_FCT  *grd_phi1_2d[3] = {grd_phi1v0_2d, grd_phi1v1_2d,
				       grd_phi1v2_2d};
static D2_BAS_FCT   *D2_phi1_2d[3]  = {D2_phi1v0_2d, D2_phi1v1_2d,
				       D2_phi1v2_2d};

static BAS_FCTS  lagrange1_2d = {"lagrange1_2d", 2, 3, 1,
				 {1, 0, 0, 0}, /* VERTEX, CENTER, EDGE, FACE */
				 nil,
				 phi1_2d, grd_phi1_2d, D2_phi1_2d, 
				 get_dof_indices1_2d,
				 get_bound1_2d, 
				 interpol1_2d,
				 interpol_d1_2d,
				 get_int_vec1_2d,
				 get_real_vec1_2d,
				 get_real_d_vec1_2d,
				 get_uchar_vec1_2d,
				 get_schar_vec1_2d,
				 real_refine_inter1_2d,
				 nil,
				 real_coarse_restr1_2d,
				 real_d_refine_inter1_2d,
				 nil,
				 real_d_coarse_restr1_2d,
				 bary1_2d, };
