/*
  CoreLinux++ 
  Copyright (C) 2000 CoreLinux Consortium
  
   The CoreLinux++ Library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   The CoreLinux++ Library Library is distributed in the hope that it will 
   be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public
   License along with the GNU C Library; see the file COPYING.LIB.  If not,
   write to the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.  
*/   


#if   !defined(__COMMON_HPP)
#include <Common.hpp>
#endif

#if   !defined(__MAZE_HPP)
#include <Maze.hpp>
#endif

#if   !defined(__ROOM_HPP)
#include <Room.hpp>
#endif

#if   !defined(__DOOR_HPP)
#include <Door.hpp>
#endif

using namespace corelinux;

//
// Default constructor
//

Maze::Maze( RoomPtr aRoom )
   :
   theStartRoom( aRoom ),
   theCurrentRoom( aRoom ),
   theCurrentDirection( NORTH )
{
   theRooms[theCurrentRoom->getRoomNumber()] = theCurrentRoom;
}

//
// Protected default constructor
//

Maze::Maze( void ) throw(Assertion)
   :
   theStartRoom( NULLPTR ),
   theCurrentRoom( NULLPTR ),
   theCurrentDirection( NORTH )
{
   NEVER_GET_HERE;
}

//
// Copy constructor
//

Maze::Maze( MazeCref ) throw(Assertion)
   :
   theStartRoom( NULLPTR ),
   theCurrentRoom( NULLPTR ),
   theCurrentDirection( NORTH )
{
   NEVER_GET_HERE;
}

//
// Virtual destructor
//

Maze::~Maze( void )
{
   theStartRoom = NULLPTR;
   theCurrentRoom = NULLPTR;
   theRooms.clear();
}

//
// Operator assignment
//

MazeRef Maze::operator=( MazeCref ) throw(Assertion)
{
   NEVER_GET_HERE;
   return (*this);
}

//
// Equality operator
//

bool  Maze::operator==( MazeCref aRef ) const
{
   return 
      ( 
         this == &aRef && 
         this->getStartLocation() == aRef.getStartLocation() 
      );
}

//
// Get the starting point
//

RoomCref  Maze::getStartLocation( void ) const
{
   return (*theStartRoom);
}

//
// Get the current location
//

RoomCref Maze::getCurrentLocation( void ) const
{
   return (*theCurrentRoom);
}

//
// When in a room, move in one of four directions
// Walls bounce but we stay here, a door puts
// our current room to what is on the other side
//

void  Maze::walkInDirection( Direction aDirection )
{
   MapSitePtr  aSide( theCurrentRoom->getSide(aDirection) );
   DoorPtr     aDoor( dynamic_cast<DoorPtr>(aSide) );
   
   aSide->enter();

   if( aDoor != NULLPTR )
   {
      theCurrentRoom = aDoor->otherSideFrom( theCurrentRoom );
   }
   else
   {
      ;  // do nothing, already bounced
   }

}

//
// Add a room to the maze
//

void  Maze::addRoom( RoomPtr aRoom )
{
   theRooms[aRoom->getRoomNumber()] = aRoom;
}

//
// Get the room collection
//

RoomMapRef  Maze::getRooms( void )
{
   return theRooms;
}

/*
   Common rcs information do not modify
   $Author: frankc $
   $Revision: 1.1 $
   $Date: 2000/02/06 13:32:31 $
   $Locker:  $
*/

