#include <sstream>
#include <fstream>

#include "ossimBmvStager.h"
#include <ossim/imaging/ossimImageChain.h>
#include <ossim/base/ossimKeywordNames.h>
#include <ossim/base/ossimEndian.h>
#include <ossim/imaging/ossimImageMosaic.h>
#include <ossim/imaging/ossimScalarRemapper.h>
#include <ossim/imaging/ossimCacheTileSource.h>
#include <ossim/imaging/ossimImageRenderer.h>
#include <ossim/imaging/ossimImageChain.h>
#include <ossim/imaging/ossimImageHandler.h>
#include <ossim/imaging/ossimImageHandlerRegistry.h>
#include <ossim/imaging/ossimImageSourceSequencer.h>
#include <ossim/imaging/ossimFilterResampler.h>
#include <ossim/projection/ossimLlxyProjection.h>
#include <ossim/projection/ossimProjectionFactoryRegistry.h>
#include <ossim/base/ossimGpt.h>
#include "ossimArrayHolder.h"

ossimBmvStager::ossimBmvStager(ossim_uint32 tileSize,
                               ossim_uint32 tilesWide,   // tiles wide at level 0
                               ossim_uint32 tilesHigh)
      :theBmvUtil(tileSize,
                  tilesWide,
                  tilesHigh),
       theMosaicChain(0),
       theView(0),
       theResampler("bilinear")
{
   theBuf     = new ossim_uint16[tileSize*tileSize];
   theFileBuf = new ossim_uint16[tileSize*tileSize];
   theView = new ossimLlxyProjection;
}

ossimBmvStager::~ossimBmvStager()
{
   deleteChains();
   if(theView)
   {
      delete theView;
      theView = 0;
   }
   if(theBuf)
   {
      delete [] theBuf;
      theBuf = 0;
   }
   if(theFileBuf)
   {
      delete [] theFileBuf;
      theFileBuf = 0;
   }
}


void ossimBmvStager::addFile(const ossimFilename& filename)
{
   theFilenameList.push_back(filename);
}

void ossimBmvStager::clearFileList()
{
   theFilenameList.clear();
}

bool ossimBmvStager::isOpen()const
{
   return getFilename().exists();
}

bool ossimBmvStager::open()
{
   // will just check if the output directory exists.
   //
   return getFilename().exists();
}

void ossimBmvStager::close()
{
   // nothing to close here
}

bool ossimBmvStager::writeFile()
{
   if(((theFilenameList.size() < 1)&&(!getInput())) ||
      (!getFilename().exists()))
   {
      // will later throw exception but for now just return
      ossimNotify(ossimNotifyLevel_WARN) << "ossimBmvStager::writeFile Need an output directory and input data sources." << std::endl;
      return false;
   }
   ossimDpt gsd;
   // ossim_uint32 i = 0;
   ossim_uint32 level=0;
   ossim_uint32 tileId = 0;
   findFullResolution(level);
   ossimGpt gpt;

   ossim_uint32 currentLevel = 0;
   buildMosaic();
   
   ossimImageSource* mainMosaic = theMosaicChain;
   // ossim_uint32 tilingColumns = 10;
   // ossim_uint32 tilingrows    = 5;
   ossimImageSourceSequencer* sequencer = theInputConnection;
   if(theMosaicChain)
   {
      ossimArrayHolder<ossim_uint16, 256*256> tempBuffer;
      ossim_uint32 row;
      ossim_uint32 col;
      while(currentLevel <= level)
      {
         ossim_float64 gsd = theBmvUtil.getSpacing(currentLevel);
         theView->setLatSpacing(gsd);
         theView->setLonSpacing(gsd);
         theView->setUlGpt(ossimGpt(90, -180));
         ossimGpt ul;
         ossimGpt lr;
         setViewToChains();

         ((ossimSource*)mainMosaic)->initialize();
         ossimIrect bounds = theMosaicChain->getBoundingRect();
         theView->lineSampleToWorld(bounds.ul(), ul);
         theView->lineSampleToWorld(bounds.lr(), lr);

         ossimIrect tempBounds = theBmvUtil.getBounds(currentLevel,
                                                      ul,
                                                      lr);
         bounds.stretchToTileBoundary(ossimIpt(theBmvUtil.getTileSize(),
                                               theBmvUtil.getTileSize()));
                                                 
         bounds = bounds.clipToRect(tempBounds);
         
         sequencer->connectMyInputTo(0, mainMosaic);
         sequencer->initialize();
         sequencer->setTileSize(ossimIpt(256, 256));
         sequencer->setAreaOfInterest(bounds);
         sequencer->setToStartOfSequence();
         
         
         ossimRefPtr<ossimImageData> data = sequencer->getNextTile();
         tileId = 0;
         
         ossim_uint32 tilesLeft = ((bounds.width() / theBmvUtil.getTileSize())*
                                  (bounds.height() / theBmvUtil.getTileSize()));
         while(data.valid())
         {
            --tilesLeft;
            ossimDpt center = data->getImageRectangle().midPoint();
            theView->lineSampleToWorld(center,
                                       gpt);
            theBmvUtil.getTileRowCol(currentLevel, gpt, row, col);
            theBmvUtil.readTile(tempBuffer, getFilename(), currentLevel, row, col);
            
            ossimFilename filename = buildFilename(currentLevel,
                                                   gpt);
            
            ossimNotify(ossimNotifyLevel_INFO)
               << "INFO: Tiles left   " << tilesLeft
               << "\nINFO: Level        " << currentLevel
               << "\nINFO: Filename     " << filename << std::endl;
            
            if(data.valid() && data->getBuf() &&
               (data->getDataObjectStatus()!=OSSIM_EMPTY))
            {
               writeTile(tempBuffer, data, filename);
            }
            
            data = sequencer->getNextTile();
         }
         ++currentLevel;
      }
  }
  
   return true;
}

ossimFilename ossimBmvStager::buildFilename(ossim_uint32 level,
                                            const ossimGpt& gpt)const
{
   ossim_uint32 row;
   ossim_uint32 col;

   theBmvUtil.getTileRowCol(level, gpt, row, col);
   ossimFilename filename = theBmvUtil.getFilename(level, row, col);
   ossimFilename f1;
   
   filename += ".5551";

   f1 = ossimFilename(ossimString::toString(level)).dirCat(filename);
   return ossimFilename(getFilename()).dirCat(f1);
}

void ossimBmvStager::findFullResolution(ossim_uint32& level)
{
   ossim_float64 scale = theBmvUtil.getSpacing(0); // geographic scale
   ossim_uint32 i = 0;
   ossim_float64 gsd = 0.0;

   double degreePerMeter = 1.0/ossimGpt().metersPerDegree().y;
   
   for(i = 0; i < theFilenameList.size(); ++i)
   {
      ossimImageHandler* ih = ossimImageHandlerRegistry::instance()->open(theFilenameList[i]);

      ossimKeywordlist kwl;

      if(ih)
      {
         ih->getImageGeometry(kwl);

         const char* meters_pixel_x = kwl.find(ossimKeywordNames::METERS_PER_PIXEL_X_KW);
         const char* meters_pixel_y = kwl.find(ossimKeywordNames::METERS_PER_PIXEL_Y_KW);
         const char* dd_pixel_lat   = kwl.find(ossimKeywordNames::DECIMAL_DEGREES_PER_PIXEL_LAT);
         const char* dd_pixel_lon   = kwl.find(ossimKeywordNames::DECIMAL_DEGREES_PER_PIXEL_LON);


         if(meters_pixel_x&&meters_pixel_y)
         {
            double mx = ossimString(meters_pixel_x).toDouble();
            double my = ossimString(meters_pixel_y).toDouble();
            
            double deltax = degreePerMeter*mx;
            double deltay = degreePerMeter*my;
            gsd    = (deltax+deltay)/2.0;
         }
         else if(dd_pixel_lat&&dd_pixel_lon)
         {
            gsd = (ossimString(dd_pixel_lat).toDouble() +
                   ossimString(dd_pixel_lon).toDouble())/2.0;
         }
         else
         {
            ossimProjection* proj = ossimProjectionFactoryRegistry::instance()->createProjection(kwl);
            
            if(proj)
            {
               ossimDpt metersPerPixel = proj->getMetersPerPixel();
               double deltax = degreePerMeter*metersPerPixel.x;
               double deltay = degreePerMeter*metersPerPixel.y;
               gsd    = (deltax+deltay)/2.0;

               delete proj;
               proj = 0;
            }
         }

         if(gsd < scale)
         {
            scale = gsd;
         }
         delete ih;
      }
   }

   level = theBmvUtil.findClosestLevel(scale);
}

void ossimBmvStager::writeTile(const ossim_uint16* backgroundBuffer,
                               const ossimRefPtr<ossimImageData>& imageData,
                               const ossimFilename& filename)const
{
   const ossim_uint8* rbuf = 0;
   const ossim_uint8* gbuf = 0;
   const ossim_uint8* bbuf = 0;
   ossimEndian endian;
   ossim_uint32 tileSize = theBmvUtil.getTileSize();
   bool needToUseBackground = true;
   if(imageData->getNumberOfBands() < 3)
   {
      rbuf = (ossim_uint8*)imageData->getBuf(0);
      gbuf = (ossim_uint8*)imageData->getBuf(0);
      bbuf = (ossim_uint8*)imageData->getBuf(0);
   }
   else 
   {
      rbuf = (ossim_uint8*)imageData->getBuf(0);
      gbuf = (ossim_uint8*)imageData->getBuf(1);
      bbuf = (ossim_uint8*)imageData->getBuf(2);
   }
   
   bool fileExists = filename.exists();
   if(fileExists)
   {
      ifstream in(filename.c_str(), ios::in|ios::binary);

      if(in)
      {
         in.read((char*)theFileBuf, tileSize*tileSize*2);
         if(endian.getSystemEndianType() != OSSIM_LITTLE_ENDIAN)
         {
            endian.swap(theFileBuf, tileSize*tileSize);
         }
         needToUseBackground = false;
      }
   }
   if(rbuf&&gbuf&&bbuf)
   {
      ossim_uint32 result;
      ossim_uint32 idx = 0;
      ossim_int32 line = tileSize - 1;
      for(; line >= 0; --line)
      {
         for(idx = 0; idx < tileSize; ++idx)
         {
            ossim_uint16 r = *rbuf;
            ossim_uint16 g = *gbuf;
            ossim_uint16 b = *bbuf;
            if(r&&g&&b)
            {
               result = (((r&0xf8)<<8)|((g&0xf8)<<3)|((b&0xf8)>>2)|1);
            }
            else
            {
               if(needToUseBackground)
               {
                  result = backgroundBuffer[line*tileSize + idx];
               }
               else
               {
                  result = 0;
               }
            }
            theBuf[line*tileSize + idx] = result;
            
            
            ++rbuf;
            ++gbuf;
            ++bbuf;
         }
      }
      ossimFilename tempDir = filename.path();
      if(!tempDir.exists())
      {
         tempDir.createDirectory();
      }

      if(fileExists)
      {
         // merge only non 0 data.

         ossim_uint32 upperBound = tileSize*tileSize;
         for(idx = 0; idx < upperBound; ++idx)
         {
            if(theBuf[idx] !=0)
            {
               theFileBuf[idx] = theBuf[idx];
            }
         }
      }
      ofstream out(filename.c_str(), ios::out|ios::binary);

      if(fileExists)
      {
         if(endian.getSystemEndianType() != OSSIM_LITTLE_ENDIAN)
         {
            endian.swap(theFileBuf, tileSize*tileSize);
         }
         out.write((char*)theFileBuf, tileSize*tileSize*2);
      }
      else
      {
         if(endian.getSystemEndianType() != OSSIM_LITTLE_ENDIAN)
         {
            endian.swap(theBuf, tileSize*tileSize);
         }
         out.write((char*)theBuf, tileSize*tileSize*2);
      }
   }
}

void ossimBmvStager::buildMosaic()
{
   deleteChains();
   
   theMosaicChain           = new ossimImageChain;
   
   ossimImageMosaic* mosaic = new ossimImageMosaic;
   
   ossim_uint32 idx = 0;
   theMosaicChain->addChild(mosaic);


   ossim_uint32 numberOfInputs = getNumberOfInputs();
   if(numberOfInputs>0)
   {
      for(idx = 0; idx < numberOfInputs; ++idx)
      {
         ossimConnectableObject* inputObj = getInput(idx);
         // only add objects if I can set the view of the chain
         //
         if(inputObj)
         {
            ossimConnectableObject* connectable = inputObj->findObjectOfType("ossimViewInterface",
                                                                             ossimConnectableObject::CONNECTABLE_DIRECTION_INPUT,
                                                                             true);
            if(connectable)
            {
               theMosaicChain->connectMyInputTo(getInput(idx));
            }
         }
      }
   }
   
   for(idx = 0; idx < theFilenameList.size(); ++idx)
   {
      ossimImageChain* chain = new ossimImageChain;
      ossimImageHandler* ih = ossimImageHandlerRegistry::instance()->open(theFilenameList[idx]);
      ossimCacheTileSource* cache  = new ossimCacheTileSource;
      ossimCacheTileSource* cache2  = new ossimCacheTileSource;
      ossimImageRenderer* renderer = new ossimImageRenderer;
      renderer->getResampler()->setFilterType("bilinear");
      
      if(ih)
      {
         chain->addChild(ih);
         cache->connectMyInputTo(ih);
         chain->addChild(cache);
         renderer->connectMyInputTo(cache);
         chain->addChild(renderer);
         cache2->connectMyInputTo(renderer);
         chain->addChild(cache2);
      }
      chain->initialize();
      theChainList.push_back(chain);
      theMosaicChain->connectMyInputTo(chain);
   }

   theMosaicChain->addChild(new ossimScalarRemapper);
      
   theMosaicChain->initialize();
}

void ossimBmvStager::setViewToChains()
{
   ossim_uint32 idx = 0;

   for(idx = 0; idx < theChainList.size(); ++idx)
   {
      if(theChainList[idx])
      {
         ossimConnectableObject* connectable = theChainList[idx]->findObjectOfType("ossimViewInterface",
                                                                                   ossimConnectableObject::CONNECTABLE_DIRECTION_INPUT,
                                                                                   true);
         ossimViewInterface *viewInterface = PTR_CAST(ossimViewInterface, connectable);
         
         if(viewInterface)
         {
            viewInterface->setView(theView->dup(), true);
         }
         theChainList[idx]->initialize();
      }
   }
   if(theMosaicChain)
   {
      theMosaicChain->initialize();
   }
}

void ossimBmvStager::deleteChains()
{
   if(theMosaicChain)
   {
      delete theMosaicChain;
      theMosaicChain = 0;
   }
   ossim_uint32 idx = 0;
   for(idx = 0; idx < theChainList.size();++idx)
   {
      if(theChainList[idx])
      {
         delete theChainList[idx];
         theChainList[idx] = 0;
      }
   }
   
   theChainList.clear();
}

void ossimBmvStager::getImageTypeList(std::vector<ossimString>& imageTypeList)const
{
   imageTypeList.push_back(ossimString("bmv"));
}

