///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/gui/mainwnd/MainFrame.h>
#include <core/viewport/Viewport.h>
#include <core/viewport/ViewportManager.h>
#include <core/viewport/snapping/SnappingManager.h>
#include <core/viewport/input/ViewportInputManager.h>
#include <core/viewport/ViewportPanel.h>
#include <core/viewport/SceneRenderer.h>
#include <core/actions/ActionManager.h>
#include <core/data/ObjectSaveStream.h>
#include <core/data/ObjectLoadStream.h>
#include <core/data/DataSetManager.h>
#include <core/scene/animation/AnimManager.h>
#include <core/scene/ObjectNode.h>
#include <core/scene/SceneRoot.h>
#include <core/scene/objects/AbstractCameraObject.h>
#include "ViewportMenu.h"

namespace Core {

/******************************************************************************
* The constructor of the viewport window class.
******************************************************************************/
Viewport::Viewport(Window3DContainer* container) : Window3D(container),
	_completeUpdate(true)
{
	// Create internal settings storage.
	viewportRecord = new ViewportRecord(this);

	_cameraView.aspectRatio = 1;
	_cameraView.isPerspective = false;
	_cameraView.znear = 1;
	_cameraView.zfar = 100;
	_cameraView.fieldOfView = 1;
	_cameraView.viewMatrix = AffineTransformation(IDENTITY);
	_cameraView.inverseViewMatrix = AffineTransformation(IDENTITY);
	_cameraView.projectionMatrix = Matrix4(IDENTITY);
	_cameraView.inverseProjectionMatrix = Matrix4(IDENTITY);
	_cameraView.validityInterval = TimeForever;

	// Set viewport background color.
	setClearColor(getVPColor(COLOR_VIEWPORT_BKG));
	// Initialize grid.
	grid().viewport = this;

	// Setup default lights.
	ambientLight.type = Window3DLight::AMBIENT_LIGHT;
	ambientLight.color = Color(0.2f, 0.2f, 0.2f);

	defaultLight1.type = Window3DLight::DIRECT_LIGHT;
	defaultLight1.color = Color(1.0f, 1.0f, 1.0f);
	defaultLight1.intensity = 0.8f;
	defaultLight1.position = ORIGIN + Normalize(Vector3(1.2f, -1.6f, 1.9f));

	defaultLight2.type = Window3DLight::DIRECT_LIGHT;
	defaultLight2.color = Color(1.0f, 1.0f, 1.0f);
	defaultLight2.intensity = 0.8f;
	defaultLight2.position = ORIGIN + Normalize(-Vector3(1.2f, -1.6f, 1.9f));
}

/******************************************************************************
* Destructor.
******************************************************************************/
Viewport::~Viewport()
{
}

/******************************************************************************
* Sets the zoom of the viewport.
******************************************************************************/
void Viewport::setFieldOfView(FloatType fov)
{
	OVITO_ASSERT_MSG(fov > 0.0, "Viewport::setFieldOfView", "Field of view must be positive.");

	if(currentView().fieldOfView == fov) return;

    _cameraView.fieldOfView = fov;

	// Do a complete scene update when the view has changed.
	updateViewport(true);
}

/******************************************************************************
* Sets the current view matrix. It transforms from world space to view space.
******************************************************************************/
void Viewport::setViewMatrix(const AffineTransformation& tm)
{
	Window3D::setViewMatrix(tm);
	_cameraView.viewMatrix = viewMatrix();
	_cameraView.inverseViewMatrix = inverseViewMatrix();
}

/******************************************************************************
* Renders the 3d contents of the window.
******************************************************************************/
void Viewport::renderWindow()
{
	if(settings()->viewType() == VIEW_NONE || VIEWPORT_MANAGER.isSuspended()) {
		clearBuffer();
		return;
	}

	// We have to do a full update when the camera has moved.
	if(currentView().validityInterval.contains(ANIM_MANAGER.time()) == false)
		_completeUpdate = true;

	if(_completeUpdate) {
		/// Setup projection matrix.
		updateProjectionMatrix();
		grid().updateGrid();
	}

	// Clear background.
	clearBuffer();

	// Render the scene and everything else.
	renderViewportScene();

	// Reset the update flag.
	_completeUpdate = false;
}

/******************************************************************************
* Renders the scene as seen through the viewport.
******************************************************************************/
void Viewport::renderViewportScene()
{
	// Render grid.
	grid().renderGrid();

	// Activate the default lights.
	setLight(0, &ambientLight);
	setLight(1, &defaultLight1);
	setLight(2, &defaultLight2);

	// Render scene.
	SceneRenderer* renderer = SceneRenderer::activeRenderer();
	CHECK_POINTER(renderer);
	renderer->renderViewport(this, ANIM_MANAGER.time());

	// Render input mode overlays.
	Q_FOREACH(const ViewportInputHandler::SmartPtr& handler, VIEWPORT_INPUT_MANAGER.stack()) {
		handler->renderOverlay(this, handler == VIEWPORT_INPUT_MANAGER.currentHandler());
	}

	// Render world axis.
	renderViewOrientationIndicator();

	// Draw viewport caption.
	setDepthTest(false);
	setLightingEnabled(false);
	setRenderingColor(getVPColor(COLOR_VIEWPORT_CAPTION));
	renderText(2 + viewportRectangle().x(), 2 + textAscender() + viewportRectangle().y(), caption());
	// Compute the size of the caption area.
	// This is where the context menu can be activated.
	contextMenuArea = container()->fontMetrics().boundingRect(caption());
	contextMenuArea.translate(2, 2 + textAscender());
	contextMenuArea.translate(viewportRectangle().x(), viewportRectangle().y());
	contextMenuArea.adjust(-3, -6, 3, 6);
}

/******************************************************************************
* Computes the current projection and view matrix.
******************************************************************************/
void Viewport::updateProjectionMatrix()
{
	// Update the viewport rectangle.

	// By default, use whole window as viewport rectangle.
	QRect viewport = QRect(0, 0, width(), height());
	setViewportRectangle(viewport);
	_cameraView.aspectRatio = aspectRatio();

	Box3 sceneExtentsPlusMargin = lastSceneExtent().centerScale(5.0);
	sceneExtentsPlusMargin.addBox(Box3(ORIGIN, 200));

	// Update projection and view matrices.
	_cameraView = getViewDescription(ANIM_MANAGER.time(), _cameraView.aspectRatio, sceneExtentsPlusMargin);

	Window3D::setViewMatrix(_cameraView.viewMatrix);
	setProjectionMatrix(_cameraView.projectionMatrix);
}

/******************************************************************************
* Returns a description the viewport's view at the given animation time.
******************************************************************************/
CameraViewDescription Viewport::getViewDescription(TimeTicks time, FloatType aspectRatio, const Box3& bb) const
{
	return settings()->getViewDescription(time, aspectRatio, bb);
}


/******************************************************************************
* Updates the title text in the viewport based on the
* current view type of the viewport.
******************************************************************************/
void Viewport::updateViewportTitle()
{
	// Load viewport caption string.
	switch(settings()->viewType()) {
		case VIEW_TOP: _caption = tr("Top"); break;
		case VIEW_BOTTOM: _caption = tr("Bottom"); break;
		case VIEW_FRONT: _caption = tr("Front"); break;
		case VIEW_BACK: _caption = tr("Back"); break;
		case VIEW_LEFT: _caption = tr("Left"); break;
		case VIEW_RIGHT: _caption = tr("Right"); break;
		case VIEW_ORTHO: _caption = tr("Ortho"); break;
		case VIEW_PERSPECTIVE: _caption = tr("Perspective"); break;
		case VIEW_SCENENODE:
			if(settings()->viewNode() != NULL)
				_caption = settings()->viewNode()->name();
			else
				_caption = tr("No view node");
		break;
		default: OVITO_ASSERT(false); _caption = ""; // unknown viewport type
	}
}

/// The default colors for viewport drawing.
Color Viewport::viewportColors[Viewport::NUMBER_OF_COLORS] = {
	Color(0.0f, 0.0f, 0.0f),	// COLOR_VIEWPORT_BKG
	Color(0.5f, 0.5f, 0.5f),	// COLOR_GRID
	Color(0.6f, 0.6f, 0.6f),	// COLOR_GRID_INTENS
	Color(0.7f, 0.7f, 0.7f),	// COLOR_GRID_AXIS
	Color(1.0f, 1.0f, 1.0f),	// COLOR_VIEWPORT_CAPTION
	Color(1.0f, 1.0f, 1.0f),	// COLOR_SELECTION
	Color(1.0f, 0.0f, 0.0f),	// COLOR_ACTIVE_AXIS
	Color(0.0f, 0.0f, 0.0f),	// COLOR_INACTIVE_AXIS
	Color(0.5f, 0.5f, 0.5f),	// COLOR_VIEWPORT_BORDER
	Color(1.0f, 1.0f, 0.0f),	// COLOR_ACTIVE_VIEWPORT_BORDER
	Color(0.8f, 1.0f, 1.0f),	// COLOR_SNAPPING_MARKER
	Color(1.0f, 0.0f, 0.0f),	// COLOR_ANIMATION_MODE
	Color(0.0f, 1.0f, 0.0f),	// COLOR_RENDER_FRAME
	Color(0.5f, 0.5f, 1.0f),	// COLOR_CAMERAS
	Color(1.0f, 1.0f, 0.0f),	// COLOR_LIGHTS
};

/******************************************************************************
* Returns color values for drawing in the viewport.
******************************************************************************/
Color Viewport::getVPColor(ViewportColor which)
{
	OVITO_ASSERT(which >= 0 && which < sizeof(viewportColors)/sizeof(viewportColors[0]));
	return viewportColors[which];
}

/******************************************************************************
* Sets color values for drawing in the viewport.
******************************************************************************/
void Viewport::setVPColor(ViewportColor which, const Color& clr)
{
	OVITO_ASSERT(which >= 0 && which < sizeof(viewportColors)/sizeof(viewportColors[0]));
	viewportColors[which] = clr;
}


/******************************************************************************
* Handles the resize events for the viewport window.
******************************************************************************/
void Viewport::resizeEvent(QResizeEvent* event)
{
	updateViewport(true);
}

/******************************************************************************
* Handles mouse down events for the viewport.
* The event is passed on to the active input handler.
******************************************************************************/
void Viewport::mousePressEvent(QMouseEvent* event)
{
	if(isEnabled() == false) return;
	VIEWPORT_MANAGER.setActiveViewport(this);
	container()->setFocus(Qt::MouseFocusReason);
	if(/*event->button() == Qt::RightButton && */contextMenuArea.contains(event->pos())) {
        showViewportMenu(event->pos());
        return;
	}

	if(VIEWPORT_INPUT_MANAGER.currentHandler() != NULL)
		VIEWPORT_INPUT_MANAGER.currentHandler()->onMouseDown(*this, event);
}

/******************************************************************************
* Handles mouse up events for the viewport.
* The event is passed on to the active input handler.
******************************************************************************/
void Viewport::mouseReleaseEvent(QMouseEvent* event)
{
	if(isEnabled() == false) return;
	if(VIEWPORT_INPUT_MANAGER.currentHandler() != NULL)
		VIEWPORT_INPUT_MANAGER.currentHandler()->onMouseUp(*this, event);
}

/******************************************************************************
* Handles mouse move events for the viewport.
* The event is passed on to the active input handler.
******************************************************************************/
void Viewport::mouseMoveEvent(QMouseEvent* event)
{
	if(isEnabled() == false) return;
	if(VIEWPORT_INPUT_MANAGER.currentHandler() != NULL)
		VIEWPORT_INPUT_MANAGER.currentHandler()->onMouseMove(*this, event);
}

/******************************************************************************
* Handles mouse enter events for the viewport.
* The event is passed on to the active input handler.
******************************************************************************/
void Viewport::enterEvent(QEvent* event)
{
	if(isEnabled() == false) return;
	if(VIEWPORT_INPUT_MANAGER.currentHandler() != NULL)
		VIEWPORT_INPUT_MANAGER.currentHandler()->onMouseEnter(*this, event);
}

/******************************************************************************
* Handles mouse leave events for the viewport.
* The event is passed on to the active input handler.
******************************************************************************/
void Viewport::leaveEvent(QEvent* event)
{
	if(isEnabled() == false) return;
	if(VIEWPORT_INPUT_MANAGER.currentHandler() != NULL)
		VIEWPORT_INPUT_MANAGER.currentHandler()->onMouseLeave(*this, event);
}

/******************************************************************************
* Handles mouse wheel events for the viewport.
* The event is passed on to the active input handler.
******************************************************************************/
void Viewport::wheelEvent(QWheelEvent* event)
{
	if(isEnabled() == false) return;

	VIEWPORT_MANAGER.setActiveViewport(this);
	if(VIEWPORT_INPUT_MANAGER.currentHandler() != NULL)
		VIEWPORT_INPUT_MANAGER.currentHandler()->onMouseWheel(*this, event);
}

/******************************************************************************
* Computes the size of an object that should appear always in the
* same size, independent of the zoom/fov of the viewport.
******************************************************************************/
FloatType Viewport::nonScalingSize(const Point3& worldPoint)
{
	if(isPerspectiveProjection()) {
		Point3 p = viewMatrix() * worldPoint;
        if(abs(p.Z) < FLOATTYPE_EPSILON) return 1.0;

        Point3 p1 = projectionMatrix() * p;
		Point3 p2 = projectionMatrix() * (p + Vector3(1,0,0));

		return 0.15 / Distance(p1, p2);
	}
	else {
		if(width() == 0) return 1.0;
		return fieldOfView() / (FloatType)width() * 50.0;
	}
}

/******************************************************************************
* Computes a point in the given coordinate system based on the given screen position
* and the current snap settings.
******************************************************************************/
bool Viewport::snapPoint(const Point2I& screenPoint, Point3& snapPoint, const AffineTransformation& snapSystem)
{
	// Try to find a snap point.
	if(SNAPPING_MANAGER.snapPoint(this, screenPoint, snapPoint, snapSystem)) {
		snapPoint = snapSystem.inverse() * snapPoint;
		return true;
	}

	// Just return the intersection point with the construction plane.
    Ray3 ray = snapSystem.inverse() * screenRay(screenPoint);

    Plane3 plane(Vector3(0,0,1), 0);
	FloatType t = plane.intersectionT(ray, 0.001f);
	if(t == numeric_limits<FloatType>::max()) return false;
	if(isPerspectiveProjection() && t <= 0.0) return false;

	snapPoint = ray.point(t);
	snapPoint.Z = 0.0;
	return true;
}

/******************************************************************************
* Computes a ray in world space going through a pixel (in window coordinates).
******************************************************************************/
Ray3 Viewport::screenRay(const Point2I& screenPoint)
{
	return viewportRay(screenToViewport(screenPoint));
}

/******************************************************************************
* Computes a ray in world space going through a pixel (in viewport coordinates).
******************************************************************************/
Ray3 Viewport::viewportRay(const Point2& viewportPoint)
{
	Matrix4 inverse = screenToWorldMatrix();
	Point3 p1 = inverse * Point3(viewportPoint.X, viewportPoint.Y, 1.0);
	Point3 p2 = inverse * Point3(viewportPoint.X, viewportPoint.Y, 0.0);
	Vector3 dir = Normalize(p1 - p2);
	if(isPerspectiveProjection())
		return Ray3(ORIGIN + inverseViewMatrix().getTranslation(), dir);
	else
		return Ray3(p2 - dir * (currentView().zfar - currentView().znear), dir);
}

/******************************************************************************
* Computes the projection of a world point in window coordinates (pixel units).
*    worldPoint - The world point to project.
*    screenPoint - The resulting viewport point.
* Returns true if the point is visible or false if the point was clipped.
* If the point was clipped, (0,0) is returned as screen point.
******************************************************************************/
bool Viewport::projectWorldPoint(const Point3& worldPoint, Point2& screenPoint)
{
	screenPoint = ORIGIN;
	Vector4 clipPoint = worldToScreenMatrix() * Vector4(worldPoint.X, worldPoint.Y, worldPoint.Z, 1.0);
	if(clipPoint[0] >  clipPoint[3]) return false;
	else if(clipPoint[0] < -clipPoint[3]) return false;
	if(clipPoint[1] >  clipPoint[3]) return false;
	else if(clipPoint[1] < -clipPoint[3]) return false;
	if(clipPoint[2] >  clipPoint[3]) return false;
	else if(clipPoint[2] < -clipPoint[3]) return false;
	FloatType wInv = 1.0 / clipPoint[3];
	screenPoint.X = ((clipPoint[0] * wInv + 1.0) * (FloatType)viewportRectangle().width() * 0.5) + viewportRectangle().x();
	screenPoint.Y = ((-clipPoint[1] * wInv + 1.0) * (FloatType)viewportRectangle().height() * 0.5) + viewportRectangle().y();
	return true;
}

/******************************************************************************
* Renders mesh in the viewport using current shading context and
* material from the given scene node.
******************************************************************************/
void Viewport::renderNodeMesh(const TriMesh& mesh, ObjectNode* contextNode)
{
	CHECK_OBJECT_POINTER(contextNode);

	// Setup shading mode.
	if(settings()->shadingMode() == SHADING_SHADED || settings()->shadingMode() == SHADING_SHADED_WITH_EDGES) {
		setDepthTest(true);
		setBackfaceCulling(true);
		setLightingEnabled(true);
	}
	else {
		setDepthTest(false);
		setBackfaceCulling(true);
		setLightingEnabled(false);
	}

	// Setup rendering color.
	if(contextNode->isSelected() && settings()->shadingMode() == SHADING_WIREFRAME)
		setRenderingColor(getVPColor(COLOR_SELECTION));
	else
		setRenderingColor(contextNode->displayColor());

	if(settings()->shadingMode() == SHADING_WIREFRAME) {
		renderMeshWireframe(mesh);
	}
	else if(settings()->shadingMode() == SHADING_SHADED) {
		renderMeshShaded(mesh);
	}
	else if(settings()->shadingMode() == SHADING_SHADED_WITH_EDGES) {
		// Two rendering passes for shaded-with-edges mode.
		if(isRendering()) {
			glPushAttrib(GL_ENABLE_BIT);
			glEnable(GL_POLYGON_OFFSET_FILL);
			glPolygonOffset(1.0, 1.0);
		}
		renderMeshShaded(mesh);
		if(isRendering()) {
			glPopAttrib();
			renderMeshWireframe(mesh);
		}
	}
}

/******************************************************************************
* Renders a shape in the viewport.
******************************************************************************/
void Viewport::renderNodeBezierShape(const BezierShape& shape, ObjectNode* contextNode)
{
	CHECK_OBJECT_POINTER(contextNode);

	// Setup rendering color.
	if(contextNode->isSelected())
		setRenderingColor(getVPColor(COLOR_SELECTION));
	else
		setRenderingColor(contextNode->displayColor());

	// Do the actual rendering.
	renderBezierShape(shape);
}

/******************************************************************************
* Zooms the given bounding box in world space.
******************************************************************************/
void Viewport::zoomBoundingBox(const Box3& bb)
{
	if(settings()->viewType() == VIEW_SCENENODE) return;	// Cannot reposition the view node.

	if(!isPerspectiveProjection()) {
		AffineTransformation viewMat = viewMatrix();
		viewMat.setTranslation(NULL_VECTOR);
		if(bb.isEmpty()) {
			setFieldOfView(DEFAULT_ORTHOGONAL_FIELD_OF_VIEW);
		}
		else {
			FloatType minX =  numeric_limits<FloatType>::max(), minY =  numeric_limits<FloatType>::max();
			FloatType maxX = -numeric_limits<FloatType>::max(), maxY = -numeric_limits<FloatType>::max();
			for(size_t i=0; i<8; i++) {
				Point3 trans = viewMat * bb[i];
				if(trans.X < minX) minX = trans.X;
				if(trans.X > maxX) maxX = trans.X;
				if(trans.Y < minY) minY = trans.Y;
				if(trans.Y > maxY) maxY = trans.Y;
			}

			Point3 center = viewMat * bb.center();
			FloatType w = maxX - minX;
			FloatType h = maxY - minY;
			if(aspectRatio() < h/w)
				setFieldOfView(h / aspectRatio() * 0.55);
			else
				setFieldOfView(w * 0.55);
			viewMat = AffineTransformation::translation(Vector3(-center.X, -center.Y, -center.Z)) * viewMat;
		}
		setViewMatrix(viewMat);
	}
	else {
		if(bb.isEmpty()) {
			setViewMatrix(AffineTransformation::lookAt(Point3(70,-100,80), ORIGIN, Vector3(0,0,1)));
		}
		else {
			AffineTransformation viewMat = viewMatrix();
			Vector3 viewDir = Normalize(inverseViewMatrix() * Vector3(0,0,1));
			FloatType dist = Length(bb.size()) * 0.5 / tan(fieldOfView() * 0.5);
			Point3 center = bb.center();
			setViewMatrix(AffineTransformation::lookAt(center + dist * viewDir, center, Vector3(0,0,1)));
		}
	}
	// Update the window when the view has changed.
	updateViewport(true);
}

/******************************************************************************
* Zooms to the extents of the scene.
******************************************************************************/
void Viewport::zoomToExtents(SceneRenderer::SceneExtentsMode mode)
{
	Box3 bb;

    SceneRenderer* renderer = SceneRenderer::activeRenderer();
	if(renderer) {
		bb = renderer->sceneExtents(settings().get(), ANIM_MANAGER.time(), mode);
		if(mode != SceneRenderer::ALL && bb.isEmpty())
            bb = renderer->sceneExtents(settings().get(), ANIM_MANAGER.time(), SceneRenderer::ALL);
	}

	zoomBoundingBox(bb);
}

/******************************************************************************
* Displays the context menu for this viewport.
******************************************************************************/
void Viewport::showViewportMenu(const QPoint& pos)
{
	/// The context menu of the viewport.
	ViewportMenu contextMenu(this);

	/// Show menu.
	contextMenu.exec(container()->mapToGlobal(pos + geometry().topLeft()));
}

/******************************************************************************
* Render the axis symbol in the corner of the viewport that shows
* the view orientation.
******************************************************************************/
void Viewport::renderViewOrientationIndicator()
{
	const FloatType tripodSize = 60.0f;		// in pixels
	const FloatType tripodArrowSize = 0.1f; // in percent of the above value.

	// Save old transformation matrices.
	AffineTransformation oldWorldMatrix = worldMatrix();
	AffineTransformation oldViewMatrix = viewMatrix();
	Matrix4 oldProjMatrix = projectionMatrix();

	// Determine the view orientation.
	Vector3 dir = Normalize(viewMatrix() * Vector3(0, 0, 1));
	Vector3 up = Normalize(viewMatrix() * Vector3(0, 1, 0));
	Vector3 right = CrossProduct(up, dir);

	AffineTransformation orientation(IDENTITY);
	orientation.column(0) = right;
	orientation.column(1) = up;
	orientation.column(2) = dir;

	setDepthTest(false);
	setBackfaceCulling(true);
	setLightingEnabled(false);

	FloatType scaling = viewportRectangle().width() / tripodSize;
	Matrix4 orthoMat = Matrix4::ortho(-scaling, scaling, -scaling*aspectRatio(), scaling*aspectRatio(), -2, 2);
	Matrix4 translMat = Matrix4::translation(Vector3(-1.0 + 1.3f/scaling, -1.0 + 1.3f/scaling/aspectRatio(), 0));
	setProjectionMatrix(translMat * orthoMat);
	setWorldMatrix(orientation);
	setViewMatrix(IDENTITY);

	// Render lines of the tripod.
	Point3 verts[6];
	verts[0] = ORIGIN;
	verts[1] = Point3(1,0,0);
	verts[2] = Point3(1,0,0);
	verts[3] = Point3(1.0-tripodArrowSize,tripodArrowSize,0);
	verts[4] = Point3(1,0,0);
	verts[5] = Point3(1.0-tripodArrowSize,-tripodArrowSize,0);
	setRenderingColor(Color(1,0,0));
	renderLines(6, Box3(Point3(-tripodArrowSize), Point3(1)), verts);
	verts[0] = ORIGIN;
	verts[1] = Point3(0,1,0);
	verts[2] = Point3(0,1,0);
	verts[3] = Point3(tripodArrowSize,1.0-tripodArrowSize,0);
	verts[4] = Point3(0,1,0);
	verts[5] = Point3(-tripodArrowSize,1.0-tripodArrowSize,0);
	setRenderingColor(Color(0,1,0));
	renderLines(6, Box3(Point3(-tripodArrowSize), Point3(1)), verts);
	verts[0] = ORIGIN;
	verts[1] = Point3(0,0,1);
	verts[2] = Point3(0,0,1);
	verts[3] = Point3(tripodArrowSize,0,1.0-tripodArrowSize);
	verts[4] = Point3(0,0,1);
	verts[5] = Point3(-tripodArrowSize,0,1.0-tripodArrowSize);
	setRenderingColor(Color(0,0,1));
	renderLines(6, Box3(Point3(-tripodArrowSize), Point3(1)), verts);

	// Render x,y,z labels.
	Point2 sp;
	if(projectObjectPoint(Point3(1.2f, 0,0), sp)) {
		setRenderingColor(Color(1,0,0));
		renderText((int)sp.X, (int)sp.Y, "x");
	}
	if(projectObjectPoint(Point3(0, 1.2f,0), sp)) {
		setRenderingColor(Color(0,1,0));
		renderText((int)sp.X, (int)sp.Y, "y");
	}
	if(projectObjectPoint(Point3(0,0,1.2f), sp)) {
		setRenderingColor(Color(0,0,1));
		renderText((int)sp.X, (int)sp.Y, "z");
	}

	// Restore old transformation matrices.
	setWorldMatrix(oldWorldMatrix);
	setViewMatrix(oldViewMatrix);
	setProjectionMatrix(oldProjMatrix);
}

};
