use ui;
use core:geometry;
use graphics;

/**
 * Create drawings with a more fine-grained layout. Similarly to Tikz in LaTeX.
 *
 * The picture element is created using a custom syntax, which is similar to the regular layout
 * syntax. See 'pictureSyntax.bs' and 'pictureSyntax.bnf' for details on the syntax.
 *
 * The picture itself basically contains a set of nodes, each which has a bounding rectangle. These
 * nodes do their own layout according to the specified properties, much like TikZ. The picture
 * element does, however, keep track of the bounds of the picture so that it is able to fit the
 * picture into the desired layout rectangle.
 */
class Picture extends Element {
	// Scale of the picture.
	Float scale;

	// Function to apply on all nodes (such as styles).
	fn(Node)->void allNodes;

	// Function to apply on all edges (such as styles).
	fn(Edge)->void allEdges;

	// Current bounding rectangle.
	private Rect boundingBox;

	// Elements added to the picture.
	private PElement[] elements;

	// Create.
	init() {
		init() {
			scale = 1;
			allNodes = &noNode;
			allEdges = &noEdge;
		}
	}

	// Dummy functions.
	void noNode(Node n) : static {}
	void noEdge(Edge e) : static {}

	// Add an element.
	void add(PElement e) {
		elements << e;

		if (e as Node)
			allNodes.call(e);
		else if (e as Edge)
			allEdges.call(e);

		if (elements.count == 1) {
			boundingBox = e.boundingBox;
		} else {
			boundingBox = boundingBox.include(e.boundingBox);
		}
	}

	// Minimum size.
	Size minSize() : override {
		boundingBox.size;
	}

	void draw(Graphics g, Nat step, Duration time) {
		if (before(g, step, time)) {
			// Center the block according to the current position and draw everything.
			Point offset = pos.p0 - boundingBox.p0*scale + (pos.size - boundingBox.size*scale)/2;
			g.push();
			g.transform = scale(scale) * translate(offset);

			for (e in elements)
				e.draw(this, g, step, time);

			g.pop();
		}
		after(g, step, time);
	}

	// Draw, never called.
	void draw(Graphics g) : override {}

	// Get animations.
	Animation[] animations() : override {
		Animation[] r = super:animations().clone();
		for (e in elements)
			for (a in e.animations)
				r << a;
		r;
	}

	// Initialize animations.
	void setupAnimations(Presentation p) : override {
		super:setupAnimations(p);
		for (e in elements)
			for (a in e.animations)
				a.setup(p, this);
	}
}


/**
 * An element in a picture.
 */
class PElement on Render {
	// Animations.
	Animation[] animations;
	void animation(Animation a) { animations << a; }

	// Get the bounding box of this element.
	Rect boundingBox() : abstract;

	// Draw this element.
	void draw(Graphics g) : abstract;

	// Draw, consider animations.
	void draw(Element elem, Graphics g, Nat step, Duration time) {
		Bool show = true;
		Nat count = animations.count;
		for (Nat i = 0; i < count; i++)
			show &= animations[i].before(elem, g, step, time);

		if (show)
			draw(g);

		for (Nat i = count; i > 0; i--)
			animations[i - 1].after(elem, g, step, time);
	}
}


/**
 * A node in a picture. The node is specified in terms of a size, an origin and an anchor point.
 */
class Node extends PElement {
	// Create.
	init(TextStyle style) {
		center c;

		init() {
			myFont = style.font;
			textBrush = style.fill;
			borderBrush = SolidBrush(black);
			anchor = c;
			textAnchor = c;
			margin = Size(6, 4);
		}
	}

	// Position.
	Point at;
	void at(Float x, Float y) { at = Point(x, y); }

	// Anchor point.
	Cardinal anchor;

	// Minimum size.
	Size minSize;
	void minSize(Float w, Float h) { minSize = Size(w, h); }

	// Margin around the text.
	Size margin;
	void margin(Float x) { margin = Size(x); }
	void margin(Float w, Float h) { margin = Size(w, h); }

	// Current font.
	private Font myFont;
	void font(Font f) {
		myFont = f;
		if (myText)
			text(myText.text);
	}
	void style(TextStyle t) {
		font(t.font);
		textBrush = t.fill;
	}

	// Text.
	private Text? myText;
	void text(Str t) {
		myText = Text(t, myFont);
	}

	// Text alignment.
	Cardinal textAnchor;

	// Text color.
	Brush textBrush;
	void textColor(Color c) { textBrush = SolidBrush(c); }

	// Border color.
	Brush borderBrush;
	void borderColor(Color c) { borderBrush = SolidBrush(c); }

	// Fill color.
	Brush? fillBrush;
	void fillColor(Color c) { fillBrush = SolidBrush(c); }

	// Get the current size.
	Size size() {
		Size s = minSize;
		if (myText) {
			s = max(s, myText.size + margin*2);
		}
		s;
	}

	// Get this node's position.
	Rect pos() {
		anchor.place(at, size);
	}

	// Bounding box.
	Rect boundingBox() { pos(); }

	// Get different anchor points.
	Point center() { pos.center; }
	Point north() { north.pick(pos); }
	Point northEast() { northEast.pick(pos); }
	Point east() { east.pick(pos); }
	Point southEast() { southEast.pick(pos); }
	Point south() { south.pick(pos); }
	Point southWest() { southWest.pick(pos); }
	Point west() { west.pick(pos); }
	Point northWest() { northWest.pick(pos); }

	// Place at various positions.
	void place(Cardinal dir, Float distance, Point origin) {
		anchor = dir.opposite;
		at = origin + dir.direction*distance;
	}
	void place(Cardinal dir, Float distance, Node origin) {
		place(dir, distance, dir.pick(origin.pos));
	}

	// Get a point on the edge of this node. The point should be aimed towards 'angle'.
	Point atEdge(Angle angle) {
		// Basically the same as for rectangle, but no concern for rounded corners.
		Rect p = pos;
		Size s = p.size/2;
		Point dir = angle.angle;
		Float scale = Point(s).length;
		if (dir.x != 0.0) {
			scale = min(scale, abs(s.w / dir.x));
		}
		if (dir.y != 0.0) {
			scale = min(scale, abs(s.h / dir.y));
		}

		p.center + dir*scale;
	}

	// Draw this node.
	void draw(Graphics g) {
		if (myText) {
			Rect p = pos;
			Point origin = textAnchor.pick(p.shrink(margin));
			Rect textRect = textAnchor.place(origin, myText.size);
			g.draw(myText, textBrush, textRect.p0);
		}
	}
}

// Easy creation.
Node node(Presentation p) on Render {
	Node(p.contentStyle);
}


/**
 * Rectangle node.
 */
class Rectangle extends Node {
	// Rounded corners?
	Size rounded;
	void rounded(Float r) { rounded = Size(r); }

	// Create.
	init(TextStyle style) {
		init(style) {}
	}

	// Get points at the edge.
	Point atEdge(Angle a) : override {
		// TODO: Respect rounded corners.

		Rect p = pos;
		Size s = p.size/2;
		Point dir = a.angle;
		Float scale = Point(s).length - Point(rounded).length * 0.2;
		if (dir.x != 0.0) {
			scale = min(scale, abs(s.w / dir.x));
		}
		if (dir.y != 0.0) {
			scale = min(scale, abs(s.h / dir.y));
		}

		p.center + dir*scale;
	}

	void draw(Graphics g) : override {
		Rect r = pos;
		if (rounded != Size()) {
			if (fillBrush) {
				g.fill(r, rounded, fillBrush);
			}
			g.draw(r, rounded, borderBrush);
		} else {
			if (fillBrush) {
				g.fill(r, fillBrush);
			}
			g.draw(r, borderBrush);
		}
		super:draw(g);
	}
}

// Easy creation.
Rectangle rectangle(Presentation p) on Render {
	Rectangle(p.contentStyle);
}


/**
 * Circle node.
 */
class Circle extends Node {
	// Allowed to be oval?
	Bool allowOval;
	void oval() { allowOval = true; }

	// Create.
	init(TextStyle style) {
		init(style) {
			allowOval = false;
		}
	}

	// Custom size computation to account for the extra space taken up by the circle.
	Size size() : override {
		Size s = super:size();
		if (allowOval) {
			s * 2.0.sqrt;
		} else {
			Size(max(s.w, s.h) * 2.0.sqrt);
		}
	}

	// Get points at the edge.
	Point atEdge(Angle a) : override {
		Rect p = pos;
		if (allowOval) {
			Size sz = p.size/2;
			Point dir = a.angle;
			dir.x /= sz.w;
			dir.y /= sz.h;
			// We're now in unit space, normalize the direction vector and transform it back.
			dir = dir.normalized;
			dir.x *= sz.w;
			dir.y *= sz.h;
			p.center + dir;
		} else {
			p.center + a.angle * p.size.h/2;
		}
	}

	void draw(Graphics g) : override {
		Rect r = pos;

		if (fillBrush) {
			g.fillOval(r, fillBrush);
		}
		g.oval(r, borderBrush);

		super:draw(g);
	}
}

// Easy creation.
Circle circle(Presentation p) on Render {
	Circle(p.contentStyle);
}

// An image inside a picture element.
class ImageNode extends Rectangle {
	// Bitmap we're drawing.
	Bitmap bitmap;

	// Scale of the picture.
	Float scale;

	// Create.
	init(TextStyle style, Bitmap bitmap) {
		init(style) {
			bitmap = bitmap;
			scale = 1.0;
		}

		borderColor(transparent);
	}

	// Size.
	Size size() : override {
		max(super:size(), bitmap.size * scale);
	}

	void draw(Graphics g) : override {
		Rect p = pos;
		Size bSz = bitmap.size * scale;
		g.draw(bitmap, Rect(p.center - bSz/2, bSz));

		super:draw(g);
	}

	// Set width.
	void width(Float w) {
		scale = w / bitmap.size.w;
	}

	// Set height.
	void height(Float h) {
		scale = h / bitmap.size.h;
	}
}

// Easy creation.
ImageNode imageNode(Presentation p, Bitmap b) on Render {
	ImageNode(p.contentStyle, b);
}

// An edge.
class Edge extends PElement {
	// From and to. Always the actual endpoints of the line.
	private Point fromPt;
	private Point toPt;

	// Control point in between 'from' and 'to'.
	Point? through;

	// From or to a node?
	private Node? fromNode;
	private Node? toNode;

	// Arrow style at the beginning and end.
	private Arrow? fromArrow;
	private Arrow? toArrow;

	// Path to draw.
	private Path path;

	// Brush.
	Brush brush;
	void color(Color c) { brush = SolidBrush(c); }

	// Line width.
	Float width;

	// Create.
	package init() {
		init() {
			brush = SolidBrush(black);
			width = 1.2;
		}
	}

	// Draw from a point.
	void from(Point pt) {
		fromPt = pt;
		fromNode = null;
		update();
	}

	// Draw from a node.
	void from(Node node) {
		fromPt = Point();
		fromNode = node;
		update();
	}

	// Draw to a point.
	void to(Point pt) {
		toPt = pt;
		toNode = null;
		update();
	}

	// Draw to a node.
	void to(Node node) {
		toPt = Point();
		toNode = node;
		update();
	}

	// Get the logical starting point.
	Point fromPos() {
		if (fromNode)
			fromNode.pos.center;
		else
			fromPt;
	}

	// Get the logical ending point.
	Point toPos() {
		if (toNode)
			toNode.pos.center;
		else
			toPt;
	}

	// Bend left/right. The amount is expressed as a fraction of the original line length.
	void bendLeft() { bendLeft(0.1); }
	void bendLeft(Float amount) { bendRight(-amount); }

	void bendRight() { bendRight(0.1); }
	void bendRight(Float amount) {
		Point from = fromPos;
		Point dir = toPos - from;
		Point offset = dir.tangent * amount;
		through = from + dir*0.5 + offset;
		update();
	}

	// Set arrow tips.
	void fromArrow(Arrow a) {
		fromArrow = a;
		update();
	}

	void toArrow(Arrow a) {
		toArrow = a;
		update();
	}

	// Compute the bounding box.
	Rect boundingBox() : override {
		Rect(fromPt).include(toPt);
	}

	// Draw.
	void draw(Graphics g) : override {
		g.push();
		g.lineWidth = width;

		g.draw(path, brush);
		if (fromArrow)
			fromArrow.draw(g, brush);
		if (toArrow)
			toArrow.draw(g, brush);

		g.pop();
	}

	// Update points.
	private void update() {
		if (through) {
			Angle fromA = angle(through - fromPos);
			if (fromNode)
				fromPt = fromNode.atEdge(fromA);
			if (fromArrow)
				fromArrow.update(fromPt, fromA);

			Angle toA = angle(through - toPos);
			if (toNode)
				toPt = toNode.atEdge(toA);
			if (toArrow)
				toArrow.update(toPt, toA);


			path.clear();
			path.start(fromPt);
			path.bezier(through, toPt);
		} else {
			Angle a = angle(toPos - fromPos);
			if (fromNode)
				fromPt = fromNode.atEdge(a);
			if (fromArrow)
				fromArrow.update(fromPt, a);
			if (toNode)
				toPt = toNode.atEdge(a.opposite);
			if (toArrow)
				toArrow.update(toPt, a.opposite);

			path.clear();
			path.start(fromPt);
			path.line(toPt);
		}
	}
}

// Create edges.
Edge edge(Point from, Point to) on Render {
	Edge e;
	e.from(from);
	e.to(to);
	e;
}

Edge edge(Point from, Node to) on Render {
	Edge e;
	e.from(from);
	e.to(to);
	e;
}

Edge edge(Node from, Point to) on Render {
	Edge e;
	e.from(from);
	e.to(to);
	e;
}

Edge edge(Node from, Node to) on Render {
	Edge e;
	e.from(from);
	e.to(to);
	e;
}

/**
 * Arrows at the end of an edge.
 */
class Arrow on Render {
	// The path to draw.
	Path path;

	// Update the path.
	void update(Point pt, Angle angle) : abstract;

	// Draw the arrow.
	void draw(Graphics g, Brush b) {
		g.fill(path, b);
	}
}


/**
 * Arrow of two lines.
 */
class PlainArrow extends Arrow {
	Float length;
	Angle angle;

	init() { init() { length = 10; angle = 30 deg; } }
	init(Angle angle) { init() { length = 10; angle = angle; } }
	init(Float length) { init() { length = length; angle = 30 deg; } }
	init(Float length, Angle angle) { init() { length = length; angle = angle; } }

	void update(Point pt, Angle a) {
		path.clear();
		path.start(pt + angle(a - angle)*length);
		path.line(pt);
		path.line(pt + angle(a + angle)*length);
	}

	// Draw the arrow.
	void draw(Graphics g, Brush b) {
		g.draw(path, b);
	}
}


/**
 * Arrow consisting of a filled triangle.
 */
class TriangleArrow extends Arrow {
	Float length;
	Angle angle;

	init() { init() { length = 10; angle = 30 deg; } }
	init(Angle angle) { init() { length = 10; angle = angle; } }
	init(Float length) { init() { length = length; angle = 30 deg; } }
	init(Float length, Angle angle) { init() { length = length; angle = angle; } }

	void update(Point pt, Angle a) {
		path.clear();
		path.start(pt + angle(a - angle)*length);
		path.line(pt);
		path.line(pt + angle(a + angle)*length);
	}
}


/**
 * Fancy, filled arrow.
 */
class FancyArrow extends Arrow {
	Float length;
	Angle angle;

	init() { init() { length = 12; angle = 20 deg; } }
	init(Angle angle) { init() { length = 12; angle = angle; } }
	init(Float length) { init() { length = length; angle = 20 deg; } }
	init(Float length, Angle angle) { init() { length = length; angle = angle; } }

	void update(Point pt, Angle a) {
		path.clear();
		path.start(pt);
		path.line(pt + angle(a - angle)*length);
		path.bezier(pt + angle(a)*length*2/3, pt + angle(a + angle)*length);
		path.close();
	}
}
