use core:io;
use util:serialize;
use core:info;
use lang:bs:macro;

/**
 * This is the base class for all responses from the server to the client.
 */
class Response : serializable {}

/**
 * Error response. Contains some kind of error message.
 */
class ErrorResponse : extends Response, serializable {
	Str message;

	init(Str message) {
		init { message = message; }
	}
}

/**
 * Authentication response. Successful case.
 */
class AuthResponse : extends Response, serializable {
	// User name.
	Str name;

	// Is the user an admin?
	Bool admin = false;

	// Server version.
	Version version;

	// TODO: Other stats?

	init(Str name, Bool admin) {
		init {
			name = name;
			admin = admin;
			version = named{PROGVIS_VERSION}.version;
		}
	}
}

/**
 * Authentication response. The user must sign in separately.
 */
class AuthLoginResponse : extends Response, serializable {
	// URL to redirect the user to.
	Url url;

	init(Url url) {
		init { url = url; }
	}
}

/**
 * Reply for the logout request. No data.
 */
class LogoutResponse : extends Response, serializable {}

/**
 * Reply for the leaderboard request.
 */
class LeaderboardResponse : extends Response, serializable {
	Score[] scores;

	init(Score[] scores) {
		init { scores = scores; }
	}
}

/**
 * Reply to the problem list query.
 */
class ProblemListResponse : extends Response, serializable {
	ProblemInfo[] data;
	init(ProblemInfo[] data) {
		init { data = data; }
	}
}

/**
 * Response with an individual problem.
 */
class ProblemResponse : extends Response, serializable {
	Problem problem;
	init(Problem problem) {
		init { problem = problem; }
	}
}

/**
 * Response for submitting a new item in the database.
 */
class NewItemResponse : extends Response, serializable {
	Int id;

	init(Int id) {
		init { id = id; }
	}
}

/**
 * List of available tests or available implementations.
 */
class ImprovementsResponse : extends Response, serializable {
	Int problemId;

	Improvement[] data;

	init(Int id, Improvement[] data) {
		init {
			problemId = id;
			data = data;
		}
	}
}

/**
 * A piece of code.
 */
class CodeResponse : extends Response, serializable {
	Code code;

	init(Code code) {
		init { code = code; }
	}
}
