package org.apache.maven.surefire.its;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */


import org.apache.maven.it.VerificationException;
import org.apache.maven.it.Verifier;
import org.apache.maven.it.util.ResourceExtractor;
import org.apache.maven.surefire.its.misc.HelperAssertions;

import java.io.File;
import java.util.List;

/**
 * Test running a single test with -Dtest=BasicTest
 *
 * @author <a href="mailto:dfabulich@apache.org">Dan Fabulich</a>
 */
public class CheckSingleTestIT
    extends AbstractSurefireIntegrationTestClass
{
    public void testSingleTest()
        throws Exception
    {
        File testDir = ResourceExtractor.simpleExtractResources( getClass(), "/default-configuration" );

        Verifier verifier = new Verifier( testDir.getAbsolutePath() );
        List<String> goals = this.getInitialGoals();
        goals.add( "test" );
        goals.add( "-Dtest=BasicTest" );
        executeGoals( verifier, goals );
        verifier.verifyErrorFreeLog();
        verifier.resetStreams();

        HelperAssertions.assertTestSuiteResults( 1, 0, 0, 0, testDir );
    }

    public void testSingleTestDotJava()
        throws Exception
    {
        File testDir = ResourceExtractor.simpleExtractResources( getClass(), "/default-configuration" );

        Verifier verifier = new Verifier( testDir.getAbsolutePath() );
        List<String> goals = this.getInitialGoals();
        goals.add( "test" );
        goals.add( "-Dtest=BasicTest.java" );
        executeGoals( verifier, goals );
        verifier.verifyErrorFreeLog();
        verifier.resetStreams();

        HelperAssertions.assertTestSuiteResults( 1, 0, 0, 0, testDir );
    }

    public void testSingleTestNonExistent()
        throws Exception
    {
        File testDir = ResourceExtractor.simpleExtractResources( getClass(), "/default-configuration" );

        Verifier verifier = new Verifier( testDir.getAbsolutePath() );
        List<String> goals = this.getInitialGoals();
        goals.add( "test" );
        goals.add( "-Dtest=DoesNotExist" );

        try
        {
            executeGoals( verifier, goals );
            verifier.verifyErrorFreeLog();
            fail( "Build should have failed" );
        }
        catch ( VerificationException e )
        {
            // as expected
        }
        finally
        {
            verifier.resetStreams();
        }

        File reportsDir = new File( testDir, "target/surefire-reports" );
        assertFalse( "Unexpected reports directory", reportsDir.exists() );
    }

    public void testSingleTestNonExistentOverride()
        throws Exception
    {
        File testDir = ResourceExtractor.simpleExtractResources( getClass(), "/default-configuration" );

        Verifier verifier = new Verifier( testDir.getAbsolutePath() );
        List<String> goals = this.getInitialGoals();
        goals.add( "test" );
        goals.add( "-Dtest=DoesNotExist" );
        goals.add( "-DfailIfNoTests=false" );
        executeGoals( verifier, goals );

        verifier.verifyErrorFreeLog();
        verifier.resetStreams();

        File reportsDir = new File( testDir, "target/surefire-reports" );
        assertFalse( "Unexpected reports directory", reportsDir.exists() );
    }
}
