// @HEADER
// ************************************************************************
//
//               Rapid Optimization Library (ROL) Package
//                 Copyright (2014) Sandia Corporation
//
// Under terms of Contract DE-AC04-94AL85000, there is a non-exclusive
// license for use of this work by or on behalf of the U.S. Government.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are
// met:
//
// 1. Redistributions of source code must retain the above copyright
// notice, this list of conditions and the following disclaimer.
//
// 2. Redistributions in binary form must reproduce the above copyright
// notice, this list of conditions and the following disclaimer in the
// documentation and/or other materials provided with the distribution.
//
// 3. Neither the name of the Corporation nor the names of the
// contributors may be used to endorse or promote products derived from
// this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY SANDIA CORPORATION "AS IS" AND ANY
// EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
// PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL SANDIA CORPORATION OR THE
// CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
// EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
// PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
// PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
// LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
// NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
// SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
//
// Questions? Contact lead developers:
//              Drew Kouri   (dpkouri@sandia.gov) and
//              Denis Ridzal (dridzal@sandia.gov)
//
// ************************************************************************
// @HEADER

#ifndef ROL_GAUSSLEGENDREQUADRATURE_HPP
#define ROL_GAUSSLEGENDREQUADRATURE_HPP

#include "ROL_Quadrature1D.hpp"

namespace ROL {

template<class Real>
class GaussLegendreQuadrature : public Quadrature1D<Real> {
private:
  const int nQuad_;

  void checkInputs(void) const {
    TEUCHOS_TEST_FOR_EXCEPTION((nQuad_ < 1 || nQuad_ > 20),std::invalid_argument,
      ">>> ERROR (ROL::GaussLegendreQuadrature): Requested number of points not implemented!");
  }

public:
  GaussLegendreQuadrature(const int nQuad) : nQuad_(nQuad) {
    // Check inputs
    checkInputs();
    std::vector<Real> pts, wts;
    buildQuadrature(pts,wts);
    Quadrature1D<Real>::set(pts,wts);
  }

  std::vector<std::vector<Real> > test(const bool printToStream = true,
                                       std::ostream &outStream = std::cout) const {
    int deg = 2*nQuad_-1;
    std::vector<Real> tmp(4);
    std::vector<std::vector<Real> > out(deg+1,tmp);
    std::vector<Real> pts, wts;
    Quadrature1D<Real>::get(pts,wts);
    for (int i = 0; i < deg+1; ++i) {
      if (printToStream) {
        if (i==0) {
          outStream << std::right
                    << std::setw(20) << "Poly order"
                    << std::setw(20) << "integral"
                    << std::setw(20) << "quad approx"
                    << std::setw(20) << "abs error"
                    << std::endl;
        }
      }
      out[i][0] = static_cast<Real>(i);
      out[i][1] = ((i%2) ? static_cast<Real>(0)
                         : static_cast<Real>(2)/static_cast<Real>(i+1));
      for (int j = 0; j < nQuad_; ++j) {
        out[i][2] += wts[j]*std::pow(pts[j],out[i][0]);
      }
      out[i][3] = std::abs(out[i][2] - out[i][1]);
      if (printToStream) {
        outStream << std::scientific << std::setprecision(11) << std::right
                  << std::setw(20) << out[i][0]
                  << std::setw(20) << out[i][1]
                  << std::setw(20) << out[i][2]
                  << std::setw(20) << out[i][3]
                  << std::endl;
      }
    }
    return out;
  }

private:
  void buildQuadrature(std::vector<Real> &pts, std::vector<Real> &wts) const {
    pts.resize(nQuad_); wts.resize(nQuad_);
    if ( nQuad_ == 1 ) {
      pts[0] = static_cast<Real>(0);
      wts[0] = static_cast<Real>(2);
    }
    else if ( nQuad_ == 2 ) {
      pts[0] = static_cast<Real>(-0.5773502691896257645091487805019574556476017512701268760186023264839776723029333456937153955857495252252087138051355676766566483649996508262705518373647912161760310773007685273559916067003615583077550051041144223011076288835574182229739459904090157105534559538626730166621791266197964892168);
      pts[1] = static_cast<Real>(0.5773502691896257645091487805019574556476017512701268760186023264839776723029333456937153955857495252252087138051355676766566483649996508262705518373647912161760310773007685273559916067003615583077550051041144223011076288835574182229739459904090157105534559538626730166621791266197964892168);
      wts[0] = static_cast<Real>(1.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000);
      wts[1] = static_cast<Real>(1.000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000);
    }
    else if ( nQuad_ == 3 ) {
      pts[0] = static_cast<Real>(0);
      pts[1] = static_cast<Real>(-0.774596669241483377035853079956479922166584341058318165317514753222696618387395806703857475371734703583260441372189929402637908087832729923135978349224240702213750958202698716256783906245777858513169283405612501838634682531972963691092925710263188052523534528101729260090115562126394576188);
      pts[2] = static_cast<Real>(0.774596669241483377035853079956479922166584341058318165317514753222696618387395806703857475371734703583260441372189929402637908087832729923135978349224240702213750958202698716256783906245777858513169283405612501838634682531972963691092925710263188052523534528101729260090115562126394576188);
      wts[0] = static_cast<Real>(0.8888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888889);
      wts[1] = static_cast<Real>(0.5555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555556);
      wts[2] = static_cast<Real>(0.5555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555555556);
    }
    else if ( nQuad_ == 4 ) {
      pts[0] = static_cast<Real>(-0.3399810435848562648026657591032446872005758697709143525929539768210200304632370344778752804355548115489602395207464932135845003241712491992776363684338328221538611182352836311104158340621521124125023821932864240034767086752629560943410821534146791671405442668508151756169732898924953195536);
      pts[1] = static_cast<Real>(0.3399810435848562648026657591032446872005758697709143525929539768210200304632370344778752804355548115489602395207464932135845003241712491992776363684338328221538611182352836311104158340621521124125023821932864240034767086752629560943410821534146791671405442668508151756169732898924953195536);
      pts[2] = static_cast<Real>(-0.8611363115940525752239464888928095050957253796297176376157219209065294714950488657041623398844793052105769209319781763249637438391157919764084938458618855762872931327441369944290122598469710261906458681564745219362114916066097678053187180580268539141223471780870198639372247416951073770551);
      pts[3] = static_cast<Real>(0.8611363115940525752239464888928095050957253796297176376157219209065294714950488657041623398844793052105769209319781763249637438391157919764084938458618855762872931327441369944290122598469710261906458681564745219362114916066097678053187180580268539141223471780870198639372247416951073770551);
      wts[0] = static_cast<Real>(0.6521451548625461426269360507780005927646513041661064595074706804812481325340896482780162322677418404902018960952364978455755577496740182191429757016783303751407135229556360801973666260481564013273531860737119707353160256000107787211587578617532049337456560923057986412084590467808124974086);
      wts[1] = static_cast<Real>(0.6521451548625461426269360507780005927646513041661064595074706804812481325340896482780162322677418404902018960952364978455755577496740182191429757016783303751407135229556360801973666260481564013273531860737119707353160256000107787211587578617532049337456560923057986412084590467808124974086);
      wts[2] = static_cast<Real>(0.3478548451374538573730639492219994072353486958338935404925293195187518674659103517219837677322581595097981039047635021544244422503259817808570242983216696248592864770443639198026333739518435986726468139262880292646839743999892212788412421382467950662543439076942013587915409532191875025701);
      wts[3] = static_cast<Real>(0.3478548451374538573730639492219994072353486958338935404925293195187518674659103517219837677322581595097981039047635021544244422503259817808570242983216696248592864770443639198026333739518435986726468139262880292646839743999892212788412421382467950662543439076942013587915409532191875025701);
    }
    else if ( nQuad_ == 5 ) {
      pts[0] = static_cast<Real>(0);
      pts[1] = static_cast<Real>(-0.5384693101056830910363144207002088049672866069055599562022316270594711853677552910358036672505709315713670572321043495510816912158744046420683486075627481533978123828583369317846132387526796166796502053799563629878671716361660767584852200097418079241406256057571019602720019270523093750336);
      pts[2] = static_cast<Real>(0.5384693101056830910363144207002088049672866069055599562022316270594711853677552910358036672505709315713670572321043495510816912158744046420683486075627481533978123828583369317846132387526796166796502053799563629878671716361660767584852200097418079241406256057571019602720019270523093750336);
      pts[3] = static_cast<Real>(-0.9061798459386639927976268782993929651256519107625308628737622865437707949166868469411429895535422619115836248167051160932020660084349721915374869570125418659061700540273012086530604091207821562942704193786707298217315368769002376029537907738935528847397895557648103916797868140600953498906);
      pts[4] = static_cast<Real>(0.9061798459386639927976268782993929651256519107625308628737622865437707949166868469411429895535422619115836248167051160932020660084349721915374869570125418659061700540273012086530604091207821562942704193786707298217315368769002376029537907738935528847397895557648103916797868140600953498906);
      wts[0] = static_cast<Real>(0.5688888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888888889);
      wts[1] = static_cast<Real>(0.4786286704993664680412915148356381929122955533431415399727276673338382671525124569755621250616041107794464209474122299742927901670531874220236019762755381069981020199559708433435017355341690324695622104863536721598859262913644828482640505637133513606531929893286127565185389732581634388813);
      wts[2] = static_cast<Real>(0.4786286704993664680412915148356381929122955533431415399727276673338382671525124569755621250616041107794464209474122299742927901670531874220236019762755381069981020199559708433435017355341690324695622104863536721598859262913644828482640505637133513606531929893286127565185389732581634388813);
      wts[3] = static_cast<Real>(0.2369268850561890875142640407199173626432600022124140155828278882217172884030430985799934304939514447761091346081433255812627653885023681335319535792800174485574535355995847122120538200213865230859933450692018833956696292641910727072915049918422041949023625662269427990370165822973921166843);
      wts[4] = static_cast<Real>(0.2369268850561890875142640407199173626432600022124140155828278882217172884030430985799934304939514447761091346081433255812627653885023681335319535792800174485574535355995847122120538200213865230859933450692018833956696292641910727072915049918422041949023625662269427990370165822973921166843);
    }
    else if ( nQuad_ == 6 ) {
      pts[0] = static_cast<Real>(0.6612093864662645136613995950199053470064485643951700708145267058521834966071431009442864037464614564298883716392751466795573467722253804381723198010093367423918538864300079016299442625145884902455718821970386303223620117352321357022187936189069743012315558710642131016398967690135661651261150514997832);
      pts[1] = static_cast<Real>(-0.6612093864662645136613995950199053470064485643951700708145267058521834966071431009442864037464614564298883716392751466795573467722253804381723198010093367423918538864300079016299442625145884902455718821970386303223620117352321357022187936189069743012315558710642131016398967690135661651261150514997832);
      pts[2] = static_cast<Real>(-0.2386191860831969086305017216807119354186106301400213501813951645742749342756398422492244272573491316090722230970106872029554530350772051352628872175189982985139866216812636229030578298770859440976999298617585739469216136216592222334626416400139367778945327871453246721518889993399000945408150514997832);
      pts[3] = static_cast<Real>(0.2386191860831969086305017216807119354186106301400213501813951645742749342756398422492244272573491316090722230970106872029554530350772051352628872175189982985139866216812636229030578298770859440976999298617585739469216136216592222334626416400139367778945327871453246721518889993399000945406150514997832);
      pts[4] = static_cast<Real>(-0.9324695142031520278123015544939946091347657377122898248725496165266135008442001962762887399219259850478636797265728341065879713795116384041921786180750210169211578452038930846310372961174632524612619760497437974074226320896716211721783852305051047442772222093863676553669179038880252326771150514997832);
      pts[5] = static_cast<Real>(0.9324695142031520278123015544939946091347657377122898248725496165266135008442001962762887399219259850478636797265728341065879713795116384041921786180750210169211578452038930846310372961174632524612619760497437974074226320896716211721783852305051047442772222093863676553669179038880252326771150514997832);
      wts[0] = static_cast<Real>(0.3607615730481386075698335138377161116615218927467454822897392402371400378372617183209622019888193479431172091403707985898798902783643210707767872114085818922114502722525757771126000732368828591631602895111800517408136855470744824724861011832599314498172164024255867775267681999309503106873150514997832);
      wts[1] = static_cast<Real>(0.3607615730481386075698335138377161116615218927467454822897392402371400378372617183209622019888193479431172091403707985898798902783643210707767872114085818922114502722525757771126000732368828591631602895111800517408136855470744824724861011832599314498172164024255867775267681999309503106873150514997832);
      wts[2] = static_cast<Real>(0.4679139345726910473898703439895509948116556057692105353116253199639142016203981270311100925847919823047662687897547971009283625541735029545935635592733866593364825926382559018030281273563502536241704619318259000997569870959005334740800746343768244318081732063691741034162617653462927888917150514997832);
      wts[3] = static_cast<Real>(0.4679139345726910473898703439895509948116556057692105353116253199639142016203981270311100925847919823047662687897547971009283625541735029545935635592733866593364825926382559018030281273563502536241704619318259000997569870959005334740800746343768244318081732063691741034162617653462927888917150514997832);
      wts[4] = static_cast<Real>(0.1713244923791703450402961421727328935268225014840439823986354397989457605423401546479277054263886697521165220698744043091917471674621759746296492293180314484520671351091683210843717994067668872126692485569940481594293273570249840534338241823632441183746103912052391190442197035702977497812150514997832);
      wts[5] = static_cast<Real>(0.1713244923791703450402961421727328935268225014840439823986354397989457605423401546479277054263886697521165220698744043091917471674621759746296492293180314484520671351091683210843717994067668872126692485569940481594293273570249840534338241823632441183746103912052391190442197035702977497812150514997832);
    }
    else if ( nQuad_ == 7 ) {
      pts[0] = static_cast<Real>(0);
      pts[1] = static_cast<Real>(0.4058451513773971669066064120769614633473820140993701263870432517946638132261256553283126897277465877652867586660480186780142389774087899602458293459431152403705864850136028192946798646997494188869169765542654505357384603100658598476270710450994883480024599267113885472679490162043321422574150514997832);
      pts[2] = static_cast<Real>(-0.4058451513773971669066064120769614633473820140993701263870432517946638132261256553283126897277465877652867586660480186780142389774087899602458293459431152403705864850136028192946798646997494188869169765542654505357384603100658598476270710450994883480024599267113885472679490162043321422574150514997832);
      pts[3] = static_cast<Real>(-0.7415311855993944398638647732807884070741476471413902601199553519674298746721805137928268323668632470596925180931120142436000543982298353471703857152740498332960747607976107150698769026932844561958151246095962171815950287169821619140709720118875391555834601955414971467103462901278094572097150514997832);
      pts[4] = static_cast<Real>(0.7415311855993944398638647732807884070741476471413902601199553519674298746721805137928268323668632470596925180931120142436000543982298353471703857152740498332960747607976107150698769026932844561958151246095962171815950287169821619140709720118875391555834601955414971467103462901278094572093150514997832);
      pts[5] = static_cast<Real>(-0.949107912342758524526189684047851262400770937670617783548769103913063330354840140805730770027925724144300739666995216194195625811353553118277789915859810085013901000179888247732305040104815148851112904940437420579459979108498442397952261081440138823188704950068274774322776063669713039873415051499783203);
      pts[6] = static_cast<Real>(0.949107912342758524526189684047851262400770937670617783548769103913063330354840140805730770027925724144300739666995216194195625811353553118277789915859810085013901000179888247732305040104815148851112904940437420579459979108498442397952261081440138823188704950068274774322776063669713039873415051499783203);
      wts[0] = static_cast<Real>(0.4179591836734693877551020408163265306122448979591836734693877551020408163265306122448979591836734693877551020408163265306122448979591836734693877551020408163265306122448979591836734693877551020408163265306122448979591836734693877551020408163265306122448979591836734693877551020408163265306);
      wts[1] = static_cast<Real>(0.3818300505051189449503697754889751338783650835338627347510834510307055464341297083486846593440448014503146717645853573344928956776383837562443187566373816994263513750309425122069048082192405967657155458166140211350441276674773890998286086501179973611850420964132316867109449892362441359044150514997832);
      wts[2] = static_cast<Real>(0.3818300505051189449503697754889751338783650835338627347510834510307055464341297083486846593440448014503146717645853573344928956776383837562443187566373816994263513750309425122069048082192405967657155458166140211350441276674773890998286086501179973611850420964132316867109449892362441359044150514997832);
      wts[3] = static_cast<Real>(0.2797053914892766679014677714237795824869250652265987645370140326936188104305626768132409429011976187663233752133720515191356369795631199443713526578123368545563592025336909192643194833502493348267909279642862302108156020621692804889667628308214167200738365553989662119973317061306305537859150514997832);
      wts[4] = static_cast<Real>(0.2797053914892766679014677714237795824869250652265987645370140326936188104305626768132409429011976187663233752133720515191356369795631199443713526578123368545563592025336909192643194833502493348267909279642862302108156020621692804889667628308214167200738365553989662119973317061306305537859150514997832);
      wts[5] = static_cast<Real>(0.1294849661688696932706114326790820183285874022599466639772086387246552349720423087156254181629208450894844020016344278810653448938189044626496347079992610378540241163129175889369389737366325173870853629537936262051606784336186365336536081108973206126186723685959653665979209749176350897029150514997832);
      wts[6] = static_cast<Real>(0.1294849661688696932706114326790820183285874022599466639772086387246552349720423087156254181629208450894844020016344278810653448938189044626496347079992610378540241163129175889369389737366325173870853629537936262051606784336186365336536081108973206126186723685959653665979209749176350897029150514997832);
    }
    else if ( nQuad_ == 8 ) {
      pts[0] = static_cast<Real>(-0.1834346424956498049394761423601839806667578129129737823171884736992044742215421141160682237111233537452676587642867666089196012523876865683788569995160663568104475551617138501966385810764205532370882654749492812314961247764619363562770645716456613159405134052985058171969174306064445289638150514997832);
      pts[1] = static_cast<Real>(0.1834346424956498049394761423601839806667578129129737823171884736992044742215421141160682237111233537452676587642867666089196012523876865683788569995160663568104475551617138501966385810764205532370882654749492812314961247764619363562770645716456613159405134052985058171969174306064445289638150514997832);
      pts[2] = static_cast<Real>(-0.5255324099163289858177390491892463490419642431203928577508570992724548207685612725239614001936319820619096829248252608507108793766638779939805395303668253631119018273032402360060717470006127901479587576756241288895336619643528330825624263470540184224603688817537938539658502113876953598879150514997832);
      pts[3] = static_cast<Real>(0.5255324099163289858177390491892463490419642431203928577508570992724548207685612725239614001936319820619096829248252608507108793766638779939805395303668253631119018273032402360060717470006127901479587576756241288895336619643528330825624263470540184224603688817537938539658502113876953598879150514997832);
      pts[4] = static_cast<Real>(-0.7966664774136267395915539364758304368371717316159648320701702950392173056764730921471519272957259390191974534530973092653656494917010859602772562074621689676153935016290342325645582634205301545856060095727342603557415761265140428851957341933710803722783136113628137267630651413319993338002150514997832);
      pts[5] = static_cast<Real>(0.7966664774136267395915539364758304368371717316159648320701702950392173056764730921471519272957259390191974534530973092653656494917010859602772562074621689676153935016290342325645582634205301545856060095727342603557415761265140428851957341933710803722783136113628137267630651413319993338002150514997832);
      pts[6] = static_cast<Real>(-0.960289856497536231683560868569472990428235234301452038271639777372424897743419284439438959263312268310424392817294176210238958155217128547937364220490969970043398261832663734680878126355334692786735966348087059754254760392931853386656813286884261347489628923208763998895240977248938732425615051499783203);
      pts[7] = static_cast<Real>(0.960289856497536231683560868569472990428235234301452038271639777372424897743419284439438959263312268310424392817294176210238958155217128547937364220490969970043398261832663734680878126355334692786735966348087059754254760392931853386656813286884261347489628923208763998895240977248938732425615051499783203);
      wts[0] = static_cast<Real>(0.3626837833783619829651504492771956121941460398943305405248230675666867347239066773243660420848285095502587699262967065529258215569895173844995576007862076842778350382862546305771007553373269714714894268328780431822779077846722965535548199601402487767505928976560993309027632737537826127502150514997832);
      wts[1] = static_cast<Real>(0.3626837833783619829651504492771956121941460398943305405248230675666867347239066773243660420848285095502587699262967065529258215569895173844995576007862076842778350382862546305771007553373269714714894268328780431822779077846722965535548199601402487767505928976560993309027632737537826127502150514997832);
      wts[2] = static_cast<Real>(0.3137066458778872873379622019866013132603289990027349376902639450749562719421734969616980762339285560494275746410778086162472468322655616056890624276469758994622503118776562559463287222021520431626467794721603822601295276898652509723185157998353156062419751736972560423953923732838789657919150514997832);
      wts[3] = static_cast<Real>(0.3137066458778872873379622019866013132603289990027349376902639450749562719421734969616980762339285560494275746410778086162472468322655616056890624276469758994622503118776562559463287222021520431626467794721603822601295276898652509723185157998353156062419751736972560423953923732838789657919150514997832);
      wts[4] = static_cast<Real>(0.2223810344533744705443559944262408844301308700512495647259092892936168145704490408536531423771979278421592661012122181231114375798525722419381826674532090577908613289536840402789398648876004385697202157482063253247195590228631570651319965589733545440605952819880671616779621183704306688233150514997832);
      wts[5] = static_cast<Real>(0.2223810344533744705443559944262408844301308700512495647259092892936168145704490408536531423771979278421592661012122181231114375798525722419381826674532090577908613289536840402789398648876004385697202157482063253247195590228631570651319965589733545440605952819880671616779621183704306688233150514997832);
      wts[6] = static_cast<Real>(0.1012285362903762591525313543099621901153940910516849570590036980647401787634707848602827393040450065581543893314132667077154940308923487678731973041136073584690533208824050731976306575729205467961435779467552492328730055025992954089946676810510810729468366466585774650346143712142008566866150514997832);
      wts[7] = static_cast<Real>(0.1012285362903762591525313543099621901153940910516849570590036980647401787634707848602827393040450065581543893314132667077154940308923487678731973041136073584690533208824050731976306575729205467961435779467552492328730055025992954089946676810510810729468366466585774650346143712142008566866150514997832);
    }
    else if ( nQuad_ == 9 ) {
      pts[0] = static_cast<Real>(0);
      pts[1] = static_cast<Real>(-0.8360311073266357942994297880697348765441067181246759961043719796394550068815901188939461970258575402563758103910561868767921700399852813493611963795348388298072683628655858714286307690921827503279179493378017903390282931287792638170061442346288416366768259295268522725491437592698775616386150514997832);
      pts[2] = static_cast<Real>(0.8360311073266357942994297880697348765441067181246759961043719796394550068815901188939461970258575402563758103910561868767921700399852813493611963795348388298072683628655858714286307690921827503279179493378017903390282931287792638170061442346288416366768259295268522725491437592698775616386150514997832);
      pts[3] = static_cast<Real>(-0.9681602395076260898355762029036728700494048004919253295500233118490803743966007530618737492268941116024875911233178159906522811969602509341080006111457157352577320594030742939105200742221799581448832412180479160165668557217628253178605064255816845030589843605433053781978726946425719821479150514997832);
      pts[4] = static_cast<Real>(0.9681602395076260898355762029036728700494048004919253295500233118490803743966007530618737492268941116024875911233178159906522811969602509341080006111457157352577320594030742939105200742221799581448832412180479160165668557217628253178605064255816845030589843605433053781978726946425719821479150514997832);
      pts[5] = static_cast<Real>(-0.3242534234038089290385380146433366085719562607369730888270474768421865795351242491930986016984975672077778257173507373911718045575238432394572865005705333805025491599132630235053630398924931286361909328940173345187813296193687231694926973637651870715469270935223550274475117654585286698075150514997832);
      pts[6] = static_cast<Real>(0.3242534234038089290385380146433366085719562607369730888270474768421865795351242491930986016984975672077778257173507373911718045575238432394572865005705333805025491599132630235053630398924931286361909328940173345187813296193687231694926973637651870715469270935223550274475117654585286698075150514997832);
      pts[7] = static_cast<Real>(-0.6133714327005903973087020393414741847857206049405646928728129422812673464910011985832400139035685845782334895968597685619397117528519746872458346040371559996202334828312987463516926466812888532978280620182027590531371274017229787367921934803381534015176954113597402763904697814697273286917150514997832);
      pts[8] = static_cast<Real>(0.6133714327005903973087020393414741847857206049405646928728129422812673464910011985832400139035685845782334895968597685619397117528519746872458346040371559996202334828312987463516926466812888532978280620182027590531371274017229787367921934803381534015176954113597402763904697814697273286917150514997832);
      wts[0] = static_cast<Real>(0.3302393550012597631645250692869740488788107835726883345930964978584026203073822121441169060216679264298311917359536407155454774502393550012597631645250692869740488788107835726883345930964978584026203073822121441169060216679264298311917359536407155454774502393550012597631645250692869740489);
      wts[1] = static_cast<Real>(0.1806481606948574040584720312429128095143378217320404844983359064713572905449462697645949773031997041476074679602577937226796268460630127231790100804745577374812973964868278705556370432288860477148539230329025541102198218481213990057413494800065234875808239968200871271576666111786816983312150514997832);
      wts[2] = static_cast<Real>(0.1806481606948574040584720312429128095143378217320404844983359064713572905449462697645949773031997041476074679602577937226796268460630127231790100804745577374812973964868278705556370432288860477148539230329025541102198218481213990057413494800065234875808239968200871271576666111786816983312150514997832);
      wts[3] = static_cast<Real>(0.081274388361574411971892158110523650675661720782410750711107676880686686308452062945578554702942576957794073317963038094590048795093955759528141378844750853767333972349567507324558127938133868301667395157245896802611234739695631672003334674766636592975299135275084311484994311087346192507215051499783203);
      wts[4] = static_cast<Real>(0.081274388361574411971892158110523650675661720782410750711107676880686686308452062945578554702942576957794073317963038094590048795093955759528141378844750853767333972349567507324558127938133868301667395157245896802611234739695631672003334674766636592975299135275084311484994311087346192507215051499783203);
      wts[5] = static_cast<Real>(0.3123470770400028400686304065844436655987548612619046455540111655991438973240193165701219218880063538522954773181646973116391818098875271459600370901478405885572589090757645984059641355722376816546561522245422024969266380802745127735793790292136245228820749357799614002097074181144513901973150514997832);
      wts[6] = static_cast<Real>(0.3123470770400028400686304065844436655987548612619046455540111655991438973240193165701219218880063538522954773181646973116391818098875271459600370901478405885572589090757645984059641355722376816546561522245422024969266380802745127735793790292136245228820749357799614002097074181144513901973150514997832);
      wts[7] = static_cast<Real>(0.2606106964029354623187428694186328497718402044372999519399970021196108156688912446476460930950174018273873855356376505133184038238358268707029298682703161767070852826824482373696733967124934731275123758942032745317892944979452416330800688391928576238230768124473665313152599422632809847998150514997832);
      wts[8] = static_cast<Real>(0.2606106964029354623187428694186328497718402044372999519399970021196108156688912446476460930950174018273873855356376505133184038238358268707029298682703161767070852826824482373696733967124934731275123758942032745317892944979452416330800688391928576238230768124473665313152599422632809847998150514997832);
    }
    else if ( nQuad_ == 10 ) {
      pts[0] = static_cast<Real>(-0.1488743389816312108848260011297199846175648594206916957079892535159036173556685213711776297994636912300311608052553388261028901818643765402316761969968090913050737827720371059070942475859422743249837177174247346216914852902942929003193466659082433838094355075996833570230005003837280634351);
      pts[1] = static_cast<Real>(0.1488743389816312108848260011297199846175648594206916957079892535159036173556685213711776297994636912300311608052553388261028901818643765402316761969968090913050737827720371059070942475859422743249837177174247346216914852902942929003193466659082433838094355075996833570230005003837280634351);
      pts[2] = static_cast<Real>(-0.4333953941292471907992659431657841622000718376562464965027015131437669890777035012251027579501177212236829350409989379472742247577232492051267741032822086200952319270933462032011328320387691584063411149801129823141488787443204324766414421576788807708483879452488118549797039287926964254222);
      pts[3] = static_cast<Real>(0.4333953941292471907992659431657841622000718376562464965027015131437669890777035012251027579501177212236829350409989379472742247577232492051267741032822086200952319270933462032011328320387691584063411149801129823141488787443204324766414421576788807708483879452488118549797039287926964254222);
      pts[4] = static_cast<Real>(-0.6794095682990244062343273651148735757692947118348094676648171889525585753950749246150785735704803794998339020473993150608367408425766300907682741718202923543197852846977409718369143712013552962837733153108679126932544954854729341324727211680274268486617121011712030227181051010718804444161);
      pts[5] = static_cast<Real>(0.6794095682990244062343273651148735757692947118348094676648171889525585753950749246150785735704803794998339020473993150608367408425766300907682741718202923543197852846977409718369143712013552962837733153108679126932544954854729341324727211680274268486617121011712030227181051010718804444161);
      pts[6] = static_cast<Real>(-0.8650633666889845107320966884234930485275430149653304525219597318453747551380555613567907289460457706944046310864117651686783001614934535637392729396890950011571349689893051612072435760480900979725923317923795535739290595879776956832427702236942765911483643714816923781701572597289139322313);
      pts[7] = static_cast<Real>(0.8650633666889845107320966884234930485275430149653304525219597318453747551380555613567907289460457706944046310864117651686783001614934535637392729396890950011571349689893051612072435760480900979725923317923795535739290595879776956832427702236942765911483643714816923781701572597289139322313);
      pts[8] = static_cast<Real>(-0.9739065285171717200779640120844520534282699466923821192312120666965952032346361596257235649562685562582330425187742112150221686014344777799205409587259942436704413695764881258799146633143510758737119877875210567067452435368713683033860909388311646653581707125686970668737259229449284383797);
      pts[9] = static_cast<Real>(0.9739065285171717200779640120844520534282699466923821192312120666965952032346361596257235649562685562582330425187742112150221686014344777799205409587259942436704413695764881258799146633143510758737119877875210567067452435368713683033860909388311646653581707125686970668737259229449284383797);
      wts[0] = static_cast<Real>(0.295524224714752870173892994651338329421046717026853601354308029755995938217152329270356595793754216722717164401252558386818490789552005826001936342494186966609562718648884168043231305061535867409083051270663865287483901746874726597515954450775158914556548308329986393605934912382356670244);
      wts[1] = static_cast<Real>(0.295524224714752870173892994651338329421046717026853601354308029755995938217152329270356595793754216722717164401252558386818490789552005826001936342494186966609562718648884168043231305061535867409083051270663865287483901746874726597515954450775158914556548308329986393605934912382356670244);
      wts[2] = static_cast<Real>(0.2692667193099963550912269215694693528597599384608837958005632762421534323191792767642266367092527607555958114503686983086929234693811452415564658846634423711656014432259960141729044528030344411297902977067142537534806284608399276575006911686749842814086288868533208042150419508881916391898);
      wts[3] = static_cast<Real>(0.2692667193099963550912269215694693528597599384608837958005632762421534323191792767642266367092527607555958114503686983086929234693811452415564658846634423711656014432259960141729044528030344411297902977067142537534806284608399276575006911686749842814086288868533208042150419508881916391898);
      wts[4] = static_cast<Real>(0.2190863625159820439955349342281631924587718705226770898809565436351999106529512812426839931772021927865912168728128876347666269080669475688309211843316656677105269915322077536772652826671027878246851010208832173320064273483254756250668415885349420711613410227291565477768928313300688702802);
      wts[5] = static_cast<Real>(0.2190863625159820439955349342281631924587718705226770898809565436351999106529512812426839931772021927865912168728128876347666269080669475688309211843316656677105269915322077536772652826671027878246851010208832173320064273483254756250668415885349420711613410227291565477768928313300688702802);
      wts[6] = static_cast<Real>(0.1494513491505805931457763396576973324025566396694273678354772687532386547266300109459472646347319519140057525610454363382344517067454976014713716011937109528798134828865118770953566439639333773939909201690204649083815618779157522578300343427785361756927642128792412282970150172590842897331);
      wts[7] = static_cast<Real>(0.1494513491505805931457763396576973324025566396694273678354772687532386547266300109459472646347319519140057525610454363382344517067454976014713716011937109528798134828865118770953566439639333773939909201690204649083815618779157522578300343427785361756927642128792412282970150172590842897331);
      wts[8] = static_cast<Real>(0.066671344308688137593568809893331792857864834320158145128694881613412064084087101776785509685058877821090054714520419331487507126254403762139304987316994041634495363706400187011242315504393526242450629832718198718647480566044117862086478449236378557180717569208295026105115288152794421677);
      wts[9] = static_cast<Real>(0.066671344308688137593568809893331792857864834320158145128694881613412064084087101776785509685058877821090054714520419331487507126254403762139304987316994041634495363706400187011242315504393526242450629832718198718647480566044117862086478449236378557180717569208295026105115288152794421677);
    }
    else if ( nQuad_ == 11 ) {
      pts[0] = static_cast<Real>(0);
      pts[1] = static_cast<Real>(-0.2695431559523449723315319854008615246796218624390522816239256318800570662236947357038215922442673013244372186707844201317032610313861598694458740089972098147110859664229110703810548417611253598433629104961304364919694348869220454420381458348829492059599272547136809488213590022808799544432);
      pts[2] = static_cast<Real>(0.2695431559523449723315319854008615246796218624390522816239256318800570662236947357038215922442673013244372186707844201317032610313861598694458740089972098147110859664229110703810548417611253598433629104961304364919694348869220454420381458348829492059599272547136809488213590022808799544432);
      pts[3] = static_cast<Real>(-0.5190961292068118159257256694586095544802271151199284890209226114866959264510728928255987801045490552814812158160909307129173617987548639754427993574234742172395871932055074683104954162884391004726664295542981689952697108207069182656523510389996139790367039311507070658494697286445984831669);
      pts[4] = static_cast<Real>(0.5190961292068118159257256694586095544802271151199284890209226114866959264510728928255987801045490552814812158160909307129173617987548639754427993574234742172395871932055074683104954162884391004726664295542981689952697108207069182656523510389996139790367039311507070658494697286445984831669);
      pts[5] = static_cast<Real>(-0.7301520055740493240934162520311534580496430620261303119783783396870132450585192295954234109712837000619865445021947584266289561346270915477905486641191436834136552472048170425809146165399903795643920039219757937939774490198595636364515457865484524089362582447022507351959888849049554038265);
      pts[6] = static_cast<Real>(0.7301520055740493240934162520311534580496430620261303119783783396870132450585192295954234109712837000619865445021947584266289561346270915477905486641191436834136552472048170425809146165399903795643920039219757937939774490198595636364515457865484524089362582447022507351959888849049554038265);
      pts[7] = static_cast<Real>(-0.8870625997680952990751577693039272666316757512253143849674110555376113138573726674422905951265973741582555646951327940013076695594509798185810180896230599904165467596733228383545894580728599866619872123685840446904096192035118026087613351419345822803475375072199421279490583145056511656277);
      pts[8] = static_cast<Real>(0.8870625997680952990751577693039272666316757512253143849674110555376113138573726674422905951265973741582555646951327940013076695594509798185810180896230599904165467596733228383545894580728599866619872123685840446904096192035118026087613351419345822803475375072199421279490583145056511656277);
      pts[9] = static_cast<Real>(-0.978228658146056992803938001122857390771422408919784415425801065983663799380889988200319398167344769899347932998927873161596618173502598141217454669626015380226646741759135834253706737232732796978240624149905454601562181445875868720775160095398398080194649946120067899312755269024073737418);
      pts[10] = static_cast<Real>(0.978228658146056992803938001122857390771422408919784415425801065983663799380889988200319398167344769899347932998927873161596618173502598141217454669626015380226646741759135834253706737232732796978240624149905454601562181445875868720775160095398398080194649946120067899312755269024073737418);
      wts[0] = static_cast<Real>(0.2729250867779006307144835283363421891560419698947837475976004114532253060391588530116668644807172945701084229222757361285499813638341776869915398053926192454330982469510608038746566885095023623162151300679439207577735716263854791993320131848270376408904547432685960824488963017101545240073);
      wts[1] = static_cast<Real>(0.2628045445102466621806888698905091953727646776031445563800553714855128033395452535524169970170267059591323352348905685930815659959388220486516662277122153406007564563422288846814246265117197586291110715875175614041126916826846274452615523702086591000201133611607179097990732517786141292028);
      wts[2] = static_cast<Real>(0.2628045445102466621806888698905091953727646776031445563800553714855128033395452535524169970170267059591323352348905685930815659959388220486516662277122153406007564563422288846814246265117197586291110715875175614041126916826846274452615523702086591000201133611607179097990732517786141292028);
      wts[3] = static_cast<Real>(0.2331937645919904799185237048431751394317981723169585090273197221219322132188809959540399484382616844165511091916787318112920150795060810133433603801373897814912101090002545933770716225731073421081441218793246900837233861267889573966809233425723055956467735751888062574046043468313611332881);
      wts[4] = static_cast<Real>(0.2331937645919904799185237048431751394317981723169585090273197221219322132188809959540399484382616844165511091916787318112920150795060810133433603801373897814912101090002545933770716225731073421081441218793246900837233861267889573966809233425723055956467735751888062574046043468313611332881);
      wts[5] = static_cast<Real>(0.1862902109277342514260976414316558916912847480402034117815064041738723480087855024693307585179795861311009645389663354315618171782985952444059876613187869462456124841903168186981083148555168528511284378808837151240596308699896596151043118276873358853174012590644772948200410345821126889686);
      wts[6] = static_cast<Real>(0.1862902109277342514260976414316558916912847480402034117815064041738723480087855024693307585179795861311009645389663354315618171782985952444059876613187869462456124841903168186981083148555168528511284378808837151240596308699896596151043118276873358853174012590644772948200410345821126889686);
      wts[7] = static_cast<Real>(0.1255803694649046246346942992239401001976157913954035006639340108179145751324712776346876417140545303978650585450845219359841068839377506073290884347363756769005925474627589346055152226465992230909984623296951248837188959879470592809528962736823949327704509072540524563398665677063470369886);
      wts[8] = static_cast<Real>(0.1255803694649046246346942992239401001976157913954035006639340108179145751324712776346876417140545303978650585450845219359841068839377506073290884347363756769005925474627589346055152226465992230909984623296951248837188959879470592809528962736823949327704509072540524563398665677063470369886);
      wts[9] = static_cast<Real>(0.0556685671161736664827537204425485787285156256968981483483842856741554072807375438836912220723188458102963210282419741638055041804016622427741273933989226320452792795289103667005518691583056421625103412886069481254986095193969566623343095934357856658000335256976480404119666482269782141718);
      wts[10] = static_cast<Real>(0.0556685671161736664827537204425485787285156256968981483483842856741554072807375438836912220723188458102963210282419741638055041804016622427741273933989226320452792795289103667005518691583056421625103412886069481254986095193969566623343095934357856658000335256976480404119666482269782141718);
    }
    else if ( nQuad_ == 12 ) {
      pts[0] = static_cast<Real>(-0.125233408511468915472441369463853129983396916305444273212921754748462056241389688742868298469491359594104598791320510973151599696644634079597205789302813634271497518773646107977862904010858517498034581635360090619153385339857922243809504545097342064247739686883799517760948964137522919201);
      pts[1] = static_cast<Real>(0.125233408511468915472441369463853129983396916305444273212921754748462056241389688742868298469491359594104598791320510973151599696644634079597205789302813634271497518773646107977862904010858517498034581635360090619153385339857922243809504545097342064247739686883799517760948964137522919201);
      pts[2] = static_cast<Real>(-0.3678314989981801937526915366437175612563601413354096213117998795040899295167878738787344285005465772346331263959771452151351521793274393532419916377427538287132038966416227430371828447096318893454788484182261146122752697960937162960050463962319787423676668046033025242558536362617894366679);
      pts[3] = static_cast<Real>(0.3678314989981801937526915366437175612563601413354096213117998795040899295167878738787344285005465772346331263959771452151351521793274393532419916377427538287132038966416227430371828447096318893454788484182261146122752697960937162960050463962319787423676668046033025242558536362617894366679);
      pts[4] = static_cast<Real>(-0.5873179542866174472967024189405342803690985140480524815102708796673406993758952624357107649887482019096015599929288926772310695910886717514249918984370415196579965493152179248683469934224574654227055695910787179434915414363513919167428554596877940491139756923177447689738849120865435563147);
      pts[5] = static_cast<Real>(0.5873179542866174472967024189405342803690985140480524815102708796673406993758952624357107649887482019096015599929288926772310695910886717514249918984370415196579965493152179248683469934224574654227055695910787179434915414363513919167428554596877940491139756923177447689738849120865435563147);
      pts[6] = static_cast<Real>(-0.7699026741943046870368938332128180759849257500189316376644190642491165431084712240164249992234219106176175404542218562070401628526535475949194203515875471151443518462689657014336785786996070706826282210248876021615678923575906254310951538410899341797549230707021382467596975621464477134163);
      pts[7] = static_cast<Real>(0.7699026741943046870368938332128180759849257500189316376644190642491165431084712240164249992234219106176175404542218562070401628526535475949194203515875471151443518462689657014336785786996070706826282210248876021615678923575906254310951538410899341797549230707021382467596975621464477134163);
      pts[8] = static_cast<Real>(-0.9041172563704748566784658661190961925375967092132975465540757606812347957292357904869694278237332678118603828964104223488997198195429960106352490125826829199834735444861420614089910024700968257625822169344644869874616758075784239807438092064065954540171679180850205196702894963912359448494);
      pts[9] = static_cast<Real>(0.9041172563704748566784658661190961925375967092132975465540757606812347957292357904869694278237332678118603828964104223488997198195429960106352490125826829199834735444861420614089910024700968257625822169344644869874616758075784239807438092064065954540171679180850205196702894963912359448494);
      pts[10] = static_cast<Real>(-0.981560634246719250690549090149280822960155199813731510462682121807793244318253982225257267890452235785556492372841273185245457030447077167082769674887528861125655501844826629100412021372015399969612358827884663023371873515839205303744147639383170419389543470920618543180673569225988370568);
      pts[11] = static_cast<Real>(0.981560634246719250690549090149280822960155199813731510462682121807793244318253982225257267890452235785556492372841273185245457030447077167082769674887528861125655501844826629100412021372015399969612358827884663023371873515839205303744147639383170419389543470920618543180673569225988370568);
      wts[0] = static_cast<Real>(0.2491470458134027850005624360429512108304609025696188313953510031162794274572880430311568006180423530648334761178771858330585110736036496880396421037700854294150273722109172825701968430164659192402161982079625520732434085776613788579662540329347837170742904111565650371846972323325015720931);
      wts[1] = static_cast<Real>(0.2491470458134027850005624360429512108304609025696188313953510031162794274572880430311568006180423530648334761178771858330585110736036496880396421037700854294150273722109172825701968430164659192402161982079625520732434085776613788579662540329347837170742904111565650371846972323325015720931);
      wts[2] = static_cast<Real>(0.2334925365383548087608498989248780562594099721997548747305234978214920000794116752806790265085636904667387564397088688338985427884089160966197503884738075353324814517948875038881216279280304248959830878229357729079164423103001879530654707315375809270840669989018891281956753131165193423269);
      wts[3] = static_cast<Real>(0.2334925365383548087608498989248780562594099721997548747305234978214920000794116752806790265085636904667387564397088688338985427884089160966197503884738075353324814517948875038881216279280304248959830878229357729079164423103001879530654707315375809270840669989018891281956753131165193423269);
      wts[4] = static_cast<Real>(0.203167426723065921749064455809798376506518147274590146398594565797645632510472843795144395064605232431160429336863259964961371351902101329079104201895994236856568902452607382802768524457038466812400647581340638998753052152617280593445415722327927963339557545261423500783899286052850767594);
      wts[5] = static_cast<Real>(0.203167426723065921749064455809798376506518147274590146398594565797645632510472843795144395064605232431160429336863259964961371351902101329079104201895994236856568902452607382802768524457038466812400647581340638998753052152617280593445415722327927963339557545261423500783899286052850767594);             wts[6] = static_cast<Real>(0.1600783285433462263346525295433590718720117304908641779098995441579542251732911506816565526370577305270748770968128026272437638608826490446750310024340951121367986902065997927856009804637591399838724493887258633605906166774286382455298444870458396283884610940466728874776625823124924247387);
      wts[7] = static_cast<Real>(0.1600783285433462263346525295433590718720117304908641779098995441579542251732911506816565526370577305270748770968128026272437638608826490446750310024340951121367986902065997927856009804637591399838724493887258633605906166774286382455298444870458396283884610940466728874776625823124924247387);
      wts[8] = static_cast<Real>(0.1069393259953184309602547181939962242145701734703248800051260421028189936274975765405373180963164574135763593331411441611703305169635508448480086523269196005011439044764920482935551535857607910705249218071033795470424895712830930967806467598358517298903536137451828089012822811396588037254);
      wts[9] = static_cast<Real>(0.1069393259953184309602547181939962242145701734703248800051260421028189936274975765405373180963164574135763593331411441611703305169635508448480086523269196005011439044764920482935551535857607910705249218071033795470424895712830930967806467598358517298903536137451828089012822811396588037254);
      wts[10] = static_cast<Real>(0.0471753363865118271946159614850170603170290739948470895605053470038097211520387106708259070754145360966161016755967385796674804082391329967384636510990980857579796788584959896597568705489452579970026951919317931124539907107094212532123682663180160342232703368882666374567833050364187887189);
      wts[11] = static_cast<Real>(0.0471753363865118271946159614850170603170290739948470895605053470038097211520387106708259070754145360966161016755967385796674804082391329967384636510990980857579796788584959896597568705489452579970026951919317931124539907107094212532123682663180160342232703368882666374567833050364187887189);
    }
    else if ( nQuad_ == 13 ) {
      pts[0] = static_cast<Real>(0);
      pts[1] = static_cast<Real>(-0.2304583159551347940655281210979888352115423758835311634692614978371620836885819508128029447646801293238512802493163412495640890375153405001503784765828756320940609583773703635107856626040295675333153435572182437721389578601864974297041861246826819085793735507197123622937060192780190134049);
      pts[2] = static_cast<Real>(0.2304583159551347940655281210979888352115423758835311634692614978371620836885819508128029447646801293238512802493163412495640890375153405001503784765828756320940609583773703635107856626040295675333153435572182437721389578601864974297041861246826819085793735507197123622937060192780190134049);
      pts[3] = static_cast<Real>(-0.4484927510364468528779128521276398678019216674417578789582829474582440920662171436450382085547330100960526355571359685762639767166329516072111020255546785037599607809407667952598452715616858465630977817121137880270573731439851953622128424242071971267594288607339452257394167789291302393212);
      pts[4] = static_cast<Real>(0.4484927510364468528779128521276398678019216674417578789582829474582440920662171436450382085547330100960526355571359685762639767166329516072111020255546785037599607809407667952598452715616858465630977817121137880270573731439851953622128424242071971267594288607339452257394167789291302393212);
      pts[5] = static_cast<Real>(-0.6423493394403402206439846069955156500716973982615768573891424079186435531449935749143883001141915551222467702645471938124047349542653510076380536670929581304956809071476279917564831504481181774928592644247241533885750724246003990645746060102158895028339223102092551500388321655636682424992);
      pts[6] = static_cast<Real>(0.6423493394403402206439846069955156500716973982615768573891424079186435531449935749143883001141915551222467702645471938124047349542653510076380536670929581304956809071476279917564831504481181774928592644247241533885750724246003990645746060102158895028339223102092551500388321655636682424992);
      pts[7] = static_cast<Real>(-0.8015780907333099127942064895828598903056157247905000298973847138592221162266401220300298741853170602523616033760947850754868524870265172116913848151615713263990657479619025395865655761171524921588159617312035481166051434038457294889479503557741074870814870028097866428498670219323004920975);
      pts[8] = static_cast<Real>(0.8015780907333099127942064895828598903056157247905000298973847138592221162266401220300298741853170602523616033760947850754868524870265172116913848151615713263990657479619025395865655761171524921588159617312035481166051434038457294889479503557741074870814870028097866428498670219323004920975);
      pts[9] = static_cast<Real>(-0.917598399222977965206547836500719512390474790111683295895285345659659208589609914283485390340582924451051781021775107230697584955473705597515505582911995071572081209112144345022593992766962120099103362525526209288781594121511958419898880009958109610097900292014649873917718056919907184614);
      pts[10] = static_cast<Real>(0.917598399222977965206547836500719512390474790111683295895285345659659208589609914283485390340582924451051781021775107230697584955473705597515505582911995071572081209112144345022593992766962120099103362525526209288781594121511958419898880009958109610097900292014649873917718056919907184614);
      pts[11] = static_cast<Real>(-0.9841830547185881494728294488071096110649905619258749086940073204285952378756268418605692872614185884013163957673519758317180836009229900608876564494831696937090191044323690614314506240106239812890453400663489623889016813551224775294808781622642601654645797161850389811214950825577959469929);
      pts[12] = static_cast<Real>(0.9841830547185881494728294488071096110649905619258749086940073204285952378756268418605692872614185884013163957673519758317180836009229900608876564494831696937090191044323690614314506240106239812890453400663489623889016813551224775294808781622642601654645797161850389811214950825577959469929);
      wts[0] = static_cast<Real>(0.2325515532308739101945895152688359481566274773067979861186654393447600240807034013827220620427413634206841000047793254586461379668172874966081759288552495345702138908932115725322518529311736104942898149691356484563277770070976864183657390450597243804037010830217623424416631209838003044796);
      wts[1] = static_cast<Real>(0.2262831802628972384120901860397766184347577376155570198649685487715756631204999710418263186663139297211265889107526836426557968078763428902495438151872023237441066027256935875147140517861161869397608640950846733738986712351672898382114993485440591169613748948922640415655895437609919445534);
      wts[2] = static_cast<Real>(0.2262831802628972384120901860397766184347577376155570198649685487715756631204999710418263186663139297211265889107526836426557968078763428902495438151872023237441066027256935875147140517861161869397608640950846733738986712351672898382114993485440591169613748948922640415655895437609919445534);
      wts[3] = static_cast<Real>(0.2078160475368885023125232193060527633865826091995035492191111199165385435650844322605955280466119265759826407051690728416782631246877699599858929066196711949483587279475686942430824453897384184449472747677348480606120504898296573235877308239374369727851300594814230840467881711421900661897);
      wts[4] = static_cast<Real>(0.2078160475368885023125232193060527633865826091995035492191111199165385435650844322605955280466119265759826407051690728416782631246877699599858929066196711949483587279475686942430824453897384184449472747677348480606120504898296573235877308239374369727851300594814230840467881711421900661897);
      wts[5] = static_cast<Real>(0.1781459807619457382800466919960979955128126506610165029867944402891450694987487234947169512223127015069670497200743946097834883241893874241917617416603387078399445321976924996760290699777029401220398372429953593454578073073328073351761308253696219117811497432878451113282968179717189979188);
      wts[6] = static_cast<Real>(0.1781459807619457382800466919960979955128126506610165029867944402891450694987487234947169512223127015069670497200743946097834883241893874241917617416603387078399445321976924996760290699777029401220398372429953593454578073073328073351761308253696219117811497432878451113282968179717189979188);
      wts[7] = static_cast<Real>(0.138873510219787238463601776868871467621862718263298227646355016505772521534889535725181542854605625862409684276125989931583961822392400056140349181450224581984204381964332335793784634023652926657148198489192428511473342464680175442836266082110093820312115079812730984876934297728295818834);
      wts[8] = static_cast<Real>(0.138873510219787238463601776868871467621862718263298227646355016505772521534889535725181542854605625862409684276125989931583961822392400056140349181450224581984204381964332335793784634023652926657148198489192428511473342464680175442836266082110093820312115079812730984876934297728295818834);
      wts[9] = static_cast<Real>(0.0921214998377284479144217759537971209236839998622368390883915473395937297276329262918978157557319728688464206180281115405280755822061330132575025454931122799528778263378580260096595447239712476272992175052866998048878792306641975965562093757627202951860014882629364182187061210206786185036);
      wts[10] = static_cast<Real>(0.0921214998377284479144217759537971209236839998622368390883915473395937297276329262918978157557319728688464206180281115405280755822061330132575025454931122799528778263378580260096595447239712476272992175052866998048878792306641975965562093757627202951860014882629364182187061210206786185036);
      wts[11] = static_cast<Real>(0.0404840047653158795200215922009860600419865457449888681350466075049944605127927104944208124330531617543255657674600847044473453552393229078708618451618261442454009833802490704966043276332314749616597004151381666755063607687770292544492940217462056927723781927519191887428534879952917444979);
      wts[12] = static_cast<Real>(0.0404840047653158795200215922009860600419865457449888681350466075049944605127927104944208124330531617543255657674600847044473453552393229078708618451618261442454009833802490704966043276332314749616597004151381666755063607687770292544492940217462056927723781927519191887428534879952917444979);
    }
    else if ( nQuad_ == 14 ) {
      pts[0] = static_cast<Real>(-0.1080549487073436620662446502198347476119516054742375570408210613080135290117300071301006881766893672374502026424466474638099232632258191427567218197315040975280613727384226506948794430877532150884455563913298190602048364164800243197396659071012506161702814425014635643221773541001328892761);
      pts[1] = static_cast<Real>(0.1080549487073436620662446502198347476119516054742375570408210613080135290117300071301006881766893672374502026424466474638099232632258191427567218197315040975280613727384226506948794430877532150884455563913298190602048364164800243197396659071012506161702814425014635643221773541001328892761);
      pts[2] = static_cast<Real>(-0.3191123689278897604356718241684754668342612035338439565966501872573334405127927831649337054213464131802793151826090394496145640578710017716508863222239624560801212099312854217234880828771645863784793742391213044784251217681147835116435367778962949997448460558214759676525644841351801594858);
      pts[3] = static_cast<Real>(0.3191123689278897604356718241684754668342612035338439565966501872573334405127927831649337054213464131802793151826090394496145640578710017716508863222239624560801212099312854217234880828771645863784793742391213044784251217681147835116435367778962949997448460558214759676525644841351801594858);
      pts[4] = static_cast<Real>(-0.5152486363581540919652907185511886623088852825693060369515047690927849518320556604520720203507728923922907932905090138695274035571340047593918260565305721101163765207320034258082303853204178402034361739066244912248016186415710382355676747454553979637438627635490786064892912451481973721288);
      pts[5] = static_cast<Real>(0.5152486363581540919652907185511886623088852825693060369515047690927849518320556604520720203507728923922907932905090138695274035571340047593918260565305721101163765207320034258082303853204178402034361739066244912248016186415710382355676747454553979637438627635490786064892912451481973721288);
      pts[6] = static_cast<Real>(-0.6872929048116854701480198030193341375384012127471706756192664886281848961831332569473730705052118384106603630216790054729627432715418501010682124688172738908295266288544358991283933860810695937145959049268853887847137691751697848752890551614067877996475717650653147982694804026342351254071);
      pts[7] = static_cast<Real>(0.6872929048116854701480198030193341375384012127471706756192664886281848961831332569473730705052118384106603630216790054729627432715418501010682124688172738908295266288544358991283933860810695937145959049268853887847137691751697848752890551614067877996475717650653147982694804026342351254071);
      pts[8] = static_cast<Real>(-0.8272013150697649931897947426503949610397011014750811815607090542414798308100288735704263901378895453991241406273986535333275661226737816179582645106990793680866931756477801456785985507825114729158304266968496560867214893369794439592826736432286425172143208924251106624044295037127737490111);
      pts[9] = static_cast<Real>(0.8272013150697649931897947426503949610397011014750811815607090542414798308100288735704263901378895453991241406273986535333275661226737816179582645106990793680866931756477801456785985507825114729158304266968496560867214893369794439592826736432286425172143208924251106624044295037127737490111);
      pts[10] = static_cast<Real>(-0.9284348836635735173363911393778742644770392104098376187179624474821310935443598531114139056836575176363551261559882603607008578010786539258018984540044065049415788809817953116114771913082523534585966056536730436866908555508986983297412486132245749388483890945436457404705549484348178721002);
      pts[11] = static_cast<Real>(0.9284348836635735173363911393778742644770392104098376187179624474821310935443598531114139056836575176363551261559882603607008578010786539258018984540044065049415788809817953116114771913082523534585966056536730436866908555508986983297412486132245749388483890945436457404705549484348178721002);
      pts[12] = static_cast<Real>(-0.9862838086968123388415972667040528016760914072392258816440708117777495541324916379106462396651517527602612562941358578689852603067447974494119727032471089820717007295567504818026168797055598944753969294261970695004471812726754299089862565428933676463914802477677291745002965827767360741735);
      pts[13] = static_cast<Real>(0.9862838086968123388415972667040528016760914072392258816440708117777495541324916379106462396651517527602612562941358578689852603067447974494119727032471089820717007295567504818026168797055598944753969294261970695004471812726754299089862565428933676463914802477677291745002965827767360741735);
      wts[0] = static_cast<Real>(0.215263853463157790195876443316260035274997558054128800219776392543618787353994604001024441410819578237256672332436770992948165976464930189035601908050981428041757802691565082287626417365449192946281203662033345376460522564310634412912654698349487266562730897512393716549425155133887783267);
      wts[1] = static_cast<Real>(0.215263853463157790195876443316260035274997558054128800219776392543618787353994604001024441410819578237256672332436770992948165976464930189035601908050981428041757802691565082287626417365449192946281203662033345376460522564310634412912654698349487266562730897512393716549425155133887783267);
      wts[2] = static_cast<Real>(0.2051984637212956039659240656612180557103390613094194517168972902833671448252497203394318399918908957243692694424494287284534856133850644865918702302140316671417873329934748278391381113256848128254396760209050529765354249731237553251469192851898072394707049964721031773292256965337005468577);
      wts[3] = static_cast<Real>(0.2051984637212956039659240656612180557103390613094194517168972902833671448252497203394318399918908957243692694424494287284534856133850644865918702302140316671417873329934748278391381113256848128254396760209050529765354249731237553251469192851898072394707049964721031773292256965337005468577);
      wts[4] = static_cast<Real>(0.1855383974779378137417165901251570362489226029373316590200349250690983502635254444255527311467122229825611215057289188990778964974252160895085525241528364360728640406002723237969714138507534560933312278904499388523844853663939226179218798247606150274514935557012909889503067356410067833406);
      wts[5] = static_cast<Real>(0.1855383974779378137417165901251570362489226029373316590200349250690983502635254444255527311467122229825611215057289188990778964974252160895085525241528364360728640406002723237969714138507534560933312278904499388523844853663939226179218798247606150274514935557012909889503067356410067833406);
      wts[6] = static_cast<Real>(0.1572031671581935345696019386238421566056680373373233749693170438747681763696082985139580933624180762768531519990811885018854374920646576267489242910372646019870010221956474591078423228056106861169077132184669351601383774428385022658899238684439084685022864905124096570215866733146092008329);
      wts[7] = static_cast<Real>(0.1572031671581935345696019386238421566056680373373233749693170438747681763696082985139580933624180762768531519990811885018854374920646576267489242910372646019870010221956474591078423228056106861169077132184669351601383774428385022658899238684439084685022864905124096570215866733146092008329);
      wts[8] = static_cast<Real>(0.1215185706879031846894148090724766259566693456900746722910753925431597438925264923188199062703750071489155506530592569942811574313408868548096421257144546080289185410615420786200564675456293296025406102396367179854059007550049729049892410130191072357341821083329663867464821867539341968434);
      wts[9] = static_cast<Real>(0.1215185706879031846894148090724766259566693456900746722910753925431597438925264923188199062703750071489155506530592569942811574313408868548096421257144546080289185410615420786200564675456293296025406102396367179854059007550049729049892410130191072357341821083329663867464821867539341968434);
      wts[10] = static_cast<Real>(0.0801580871597602098056332770628543095836977853945947652013990654895714744572871698635361908191377559686225015908038847487953091382572604434376755119844740947797387723700536610577178522653954549123135546624971159464576653576521600937489354127710937535198838649279475628473516378736712929573);
      wts[11] = static_cast<Real>(0.0801580871597602098056332770628543095836977853945947652013990654895714744572871698635361908191377559686225015908038847487953091382572604434376755119844740947797387723700536610577178522653954549123135546624971159464576653576521600937489354127710937535198838649279475628473516378736712929573);
      wts[12] = static_cast<Real>(0.0351194603317518630318328761381917806197056092771272765814998901964163228378082705376767969986464636614217324764405511345585478510619843098677334088459571639479324880874445672906474148414770675031860143060108937026176235406760523793904458974659810087587180865408885105556219147609526200925);
      wts[13] = static_cast<Real>(0.0351194603317518630318328761381917806197056092771272765814998901964163228378082705376767969986464636614217324764405511345585478510619843098677334088459571639479324880874445672906474148414770675031860143060108937026176235406760523793904458974659810087587180865408885105556219147609526200925);
    }
    else if ( nQuad_ == 15 ) {
      pts[0] = static_cast<Real>(0);
      pts[1] = static_cast<Real>(-0.2011940939974345223006283033945962078128364544626376796159497246099482390030201876018362580675210590896790225738650942118942792830254885727862468296762689520472323105296106261124651357614417997418035121035408247749648194561179315250580457969565227012849978769073832577847806340363795749473);
      pts[2] = static_cast<Real>(0.2011940939974345223006283033945962078128364544626376796159497246099482390030201876018362580675210590896790225738650942118942792830254885727862468296762689520472323105296106261124651357614417997418035121035408247749648194561179315250580457969565227012849978769073832577847806340363795749473);
      pts[3] = static_cast<Real>(-0.3941513470775633698972073709810454683627527761586982550311653439516089577869614179754971141616597620258935216963564800247584781260358233957151493455553007521886914392606583742154248479579749842360261132809797979651499137490065468199685647693609935982616317942701783226669048753533254183119);
      pts[4] = static_cast<Real>(0.3941513470775633698972073709810454683627527761586982550311653439516089577869614179754971141616597620258935216963564800247584781260358233957151493455553007521886914392606583742154248479579749842360261132809797979651499137490065468199685647693609935982616317942701783226669048753533254183119);
      pts[5] = static_cast<Real>(-0.5709721726085388475372267372539106412383863962827496048532654170541953798697585794834146285698261447791264649702625704035115501912776443761340450804516463781076364549656886694892446365920390093401556752553531482547154572126630162234082965371353094862024333370955907936013838701919590803412);
      pts[6] = static_cast<Real>(0.5709721726085388475372267372539106412383863962827496048532654170541953798697585794834146285698261447791264649702625704035115501912776443761340450804516463781076364549656886694892446365920390093401556752553531482547154572126630162234082965371353094862024333370955907936013838701919590803412);
      pts[7] = static_cast<Real>(-0.724417731360170047416186054613938009630899294584102563551423420704123781677925218996101097603134326269235985493819251120386564200897315435713528175170608440951083020460016262974562085876362569423407165760886935238050225109674832830079599365377790413466864301655149204169505796163086343748);
      pts[8] = static_cast<Real>(0.724417731360170047416186054613938009630899294584102563551423420704123781677925218996101097603134326269235985493819251120386564200897315435713528175170608440951083020460016262974562085876362569423407165760886935238050225109674832830079599365377790413466864301655149204169505796163086343748);
      pts[9] = static_cast<Real>(-0.8482065834104272162006483207742168513662561747369926340957275587606750751741454851976077197508214808509037383571333991774655863067112478024741155233378528783931705752141398941910147200136987021229009687468623820809560831359261245028073597202508315345765272897870964489632790463532026206005);
      pts[10] = static_cast<Real>(0.8482065834104272162006483207742168513662561747369926340957275587606750751741454851976077197508214808509037383571333991774655863067112478024741155233378528783931705752141398941910147200136987021229009687468623820809560831359261245028073597202508315345765272897870964489632790463532026206005);
      pts[11] = static_cast<Real>(-0.9372733924007059043077589477102094712439962735153044579013630763502029737970455279505475861742680865974682404460315684492009513352834390536949245590430527861757465810011883749183601162731625066190523359799844459286625508280580877744877723444752122837802536842521085722280263813016978301407);
      pts[12] = static_cast<Real>(0.9372733924007059043077589477102094712439962735153044579013630763502029737970455279505475861742680865974682404460315684492009513352834390536949245590430527861757465810011883749183601162731625066190523359799844459286625508280580877744877723444752122837802536842521085722280263813016978301407);
      pts[13] = static_cast<Real>(-0.9879925180204854284895657185866125811469728171237614899999975155873884373690194247127220503683191449766751684399007925019395823670692057806992758567920785969340702791275630120497337228079229330198922312006979937161784084500767102113415768221050653691522462833296858362238239685728519647054);
      pts[14] = static_cast<Real>(0.9879925180204854284895657185866125811469728171237614899999975155873884373690194247127220503683191449766751684399007925019395823670692057806992758567920785969340702791275630120497337228079229330198922312006979937161784084500767102113415768221050653691522462833296858362238239685728519647054);
      wts[0] = static_cast<Real>(0.2025782419255612728806201999675193148386621580094773567967041160514353987547460740933934407127880321353514826708299901773095246288719482192675665869139062612256085449558643031836505029978223451416924610397803870997344190817384290577763771236964710158183351656545129738602932076125549319023);
      wts[1] = static_cast<Real>(0.1984314853271115764561183264438393248186925599575419934847379279291247975334342681333149991648178232076602085488930991764791477591610421132086613599988857531192702426415927231327044667892816045834098228729517385124172569721507734356429605231973280268604684028301488322765670316796303356288);
      wts[2] = static_cast<Real>(0.1984314853271115764561183264438393248186925599575419934847379279291247975334342681333149991648178232076602085488930991764791477591610421132086613599988857531192702426415927231327044667892816045834098228729517385124172569721507734356429605231973280268604684028301488322765670316796303356288);
      wts[3] = static_cast<Real>(0.1861610000155622110268005618664228245062260122779284028154957273100132555026991606189497688860993236053997770900138443530672702173881822218913650817524402394454633498455254983854987704558698346743823005532094467225041792102166969649720014347260039240685291809910280840226841347980521837134);
      wts[4] = static_cast<Real>(0.1861610000155622110268005618664228245062260122779284028154957273100132555026991606189497688860993236053997770900138443530672702173881822218913650817524402394454633498455254983854987704558698346743823005532094467225041792102166969649720014347260039240685291809910280840226841347980521837134);
      wts[5] = static_cast<Real>(0.1662692058169939335532008604812088111309001800984129073218651905635535632122785177107051742924155362148446154065718522274181714630170967754925174671836098206991229223858649637789999954737644311199247756973750854153289068732761392073165357013373138619425638694563391801926576061195937788699);
      wts[6] = static_cast<Real>(0.1662692058169939335532008604812088111309001800984129073218651905635535632122785177107051742924155362148446154065718522274181714630170967754925174671836098206991229223858649637789999954737644311199247756973750854153289068732761392073165357013373138619425638694563391801926576061195937788699);
      wts[7] = static_cast<Real>(0.1395706779261543144478047945110283225208502753155112432023911286310884445419078116807682573635713336381490888932766399041110917245491409628233342406440126372784794505759941197273360771302835643088631984018181469133646430155458101237375718557827299435854832981132499961239407405172041096534);
      wts[8] = static_cast<Real>(0.1395706779261543144478047945110283225208502753155112432023911286310884445419078116807682573635713336381490888932766399041110917245491409628233342406440126372784794505759941197273360771302835643088631984018181469133646430155458101237375718557827299435854832981132499961239407405172041096534);
      wts[9] = static_cast<Real>(0.1071592204671719350118695466858693034155437157581019806870223891218779948523157997256858571376086240443980876783750555812718104988405678239974708278615361621323837150901155489565006064488581094283683146572762126337955710255509976788374176850005859153689529410495091691306303339254292612609);
      wts[10] = static_cast<Real>(0.1071592204671719350118695466858693034155437157581019806870223891218779948523157997256858571376086240443980876783750555812718104988405678239974708278615361621323837150901155489565006064488581094283683146572762126337955710255509976788374176850005859153689529410495091691306303339254292612609);
      wts[11] = static_cast<Real>(0.0703660474881081247092674164506673384667080327543307198259072929143870555128742370448404520666939392193554898585950405388046148432772910792956556041537582479070968902376133780797668535183637426522389065978826221602778581837148625124153563760247327246091136423557230784075321996037169475438);
      wts[12] = static_cast<Real>(0.0703660474881081247092674164506673384667080327543307198259072929143870555128742370448404520666939392193554898585950405388046148432772910792956556041537582479070968902376133780797668535183637426522389065978826221602778581837148625124153563760247327246091136423557230784075321996037169475438);
      wts[13] = static_cast<Real>(0.0307532419961172683546283935772044177217481448334340742642282855042371894671171680390387707323994040025169911888594731301931311793307049136572121249488040088053791567453616163473679786846675406619664506995965540924443751786755055481899678620830700956557210823767451729158413495735272185295);
      wts[14] = static_cast<Real>(0.0307532419961172683546283935772044177217481448334340742642282855042371894671171680390387707323994040025169911888594731301931311793307049136572121249488040088053791567453616163473679786846675406619664506995965540924443751786755055481899678620830700956557210823767451729158413495735272185295);
    }
    else if ( nQuad_ == 16 ) {
      pts[0] = static_cast<Real>(-0.0950125098376374401853193354249580631303530556890654566972198172251252982445921329847586929757833520996553912423163124483074773224487565507552825376683317590042639430675226808621968298306398385834094062354452738853673370952242716875153912021891680290435986783119557067235389351844245724875);
      pts[1] = static_cast<Real>(0.0950125098376374401853193354249580631303530556890654566972198172251252982445921329847586929757833520996553912423163124483074773224487565507552825376683317590042639430675226808621968298306398385834094062354452738853673370952242716875153912021891680290435986783119557067235389351844245724875);
      pts[2] = static_cast<Real>(-0.2816035507792589132304605014604961064860694907705998005488347339559251794991307704414402291520401592843373670756676799439586082317318595924277817407374616529972673172532181829993235047128041390568389016122167102956500177825088396891248315076199870306732893196077699300080783278160388268526);
      pts[3] = static_cast<Real>(0.2816035507792589132304605014604961064860694907705998005488347339559251794991307704414402291520401592843373670756676799439586082317318595924277817407374616529972673172532181829993235047128041390568389016122167102956500177825088396891248315076199870306732893196077699300080783278160388268526);
      pts[4] = static_cast<Real>(-0.4580167776572273863424194429835775735400316130355234909011547509477591742902936077354355279359880932508890488802524109819378387263875748374372456802481418656153209542267373920974363173240942222040312612330532653123205501204421111104074762176186316667057022346422007533674383870669928163176);
      pts[5] = static_cast<Real>(0.4580167776572273863424194429835775735400316130355234909011547509477591742902936077354355279359880932508890488802524109819378387263875748374372456802481418656153209542267373920974363173240942222040312612330532653123205501204421111104074762176186316667057022346422007533674383870669928163176);
      pts[6] = static_cast<Real>(-0.6178762444026437484466717640487910189918822177656577941037973555417333177548114244569110304279585031122000569275624151076936925727848010402595876903273247179517798914362511464102876638848562701414071395222427996580524502897074237462419393840518491384428806676527323585381078172060016106792);
      pts[7] = static_cast<Real>(0.6178762444026437484466717640487910189918822177656577941037973555417333177548114244569110304279585031122000569275624151076936925727848010402595876903273247179517798914362511464102876638848562701414071395222427996580524502897074237462419393840518491384428806676527323585381078172060016106792);
      pts[8] = static_cast<Real>(-0.7554044083550030338951011948474422683538136564575030097817571769222968610312716777206220569192494434216539226257757678979776951755606291644397833797722369885243242046906798866349950829217197588925310437168084369511369689261376271132103717538964348180234457737297258035338547704003530793529);
      pts[9] = static_cast<Real>(0.7554044083550030338951011948474422683538136564575030097817571769222968610312716777206220569192494434216539226257757678979776951755606291644397833797722369885243242046906798866349950829217197588925310437168084369511369689261376271132103717538964348180234457737297258035338547704003530793529);
      pts[10] = static_cast<Real>(-0.8656312023878317438804678977123931323873353848475267081035114255677603977124905582571324943647723542038214283313414643013860029908661750240618421060695691357885060000446425683961857266556079460930206564550102153241869468759866739066626836770199244933157211083365506097141892552664327847109);
      pts[11] = static_cast<Real>(0.8656312023878317438804678977123931323873353848475267081035114255677603977124905582571324943647723542038214283313414643013860029908661750240618421060695691357885060000446425683961857266556079460930206564550102153241869468759866739066626836770199244933157211083365506097141892552664327847109);
      pts[12] = static_cast<Real>(-0.9445750230732325760779884155346083450911392725910726009255536520666097889026823042195657287381583189493289311009073188864109526806102494798196007717799111788591676488419490727814170144843220494323478581257881972120927685699837677135359009690477976986581402819250512783872697998966630058366);
      pts[13] = static_cast<Real>(0.9445750230732325760779884155346083450911392725910726009255536520666097889026823042195657287381583189493289311009073188864109526806102494798196007717799111788591676488419490727814170144843220494323478581257881972120927685699837677135359009690477976986581402819250512783872697998966630058366);
      pts[14] = static_cast<Real>(-0.9894009349916499325961541734503326274262740716576451300512239047313241372158253969385364319067981810135134358598978665082530237078791797359303822324413999695095711078087727905307199080635719546126798380959938811380435009735652992230642464639938589347925375828009051127056909219223903533179);
      pts[15] = static_cast<Real>(0.9894009349916499325961541734503326274262740716576451300512239047313241372158253969385364319067981810135134358598978665082530237078791797359303822324413999695095711078087727905307199080635719546126798380959938811380435009735652992230642464639938589347925375828009051127056909219223903533179);
      wts[0] = static_cast<Real>(0.1894506104550684962853967232082831051469089883959029750375132452000228907691330063001339778335339522825338056432096423297083652175404007311690388189431844411595815612767118459616173498175974279783368912090196026294264906147293439818164037940618838412163450874403569646465188699526105009401);
      wts[1] = static_cast<Real>(0.1894506104550684962853967232082831051469089883959029750375132452000228907691330063001339778335339522825338056432096423297083652175404007311690388189431844411595815612767118459616173498175974279783368912090196026294264906147293439818164037940618838412163450874403569646465188699526105009401);
      wts[2] = static_cast<Real>(0.1826034150449235888667636679692199393835562236546492824184951443794304649501111749604004251169852753140450247468164720359323414531767503677231287901939855049593288980367779339835260945463293576871039381184643035967488751413022888111613573519788630660749465009519914881317048638180707164801);
      wts[3] = static_cast<Real>(0.1826034150449235888667636679692199393835562236546492824184951443794304649501111749604004251169852753140450247468164720359323414531767503677231287901939855049593288980367779339835260945463293576871039381184643035967488751413022888111613573519788630660749465009519914881317048638180707164801);
      wts[4] = static_cast<Real>(0.1691565193950025381893120790303599622116394734160282817450829356808036642099305309321553225420782280475616663029174311301070245194017485677345250958562776298891264277426735059304319847991653319152533740752257147529442896440995961657448930494413086332128490066363561659644970262910622840035);
      wts[5] = static_cast<Real>(0.1691565193950025381893120790303599622116394734160282817450829356808036642099305309321553225420782280475616663029174311301070245194017485677345250958562776298891264277426735059304319847991653319152533740752257147529442896440995961657448930494413086332128490066363561659644970262910622840035);
      wts[6] = static_cast<Real>(0.1495959888165767320815017305474785489704910682078364668054219621873604040204179824517786384030592580740666978710981552017441608823947804369332735108707683892775563858478598112791002547854739387534174159632646453366676666558524965031020211316827351475733089362794299590848309953779341872602);
      wts[7] = static_cast<Real>(0.1495959888165767320815017305474785489704910682078364668054219621873604040204179824517786384030592580740666978710981552017441608823947804369332735108707683892775563858478598112791002547854739387534174159632646453366676666558524965031020211316827351475733089362794299590848309953779341872602);
      wts[8] = static_cast<Real>(0.1246289712555338720524762821920164201448868592222026799447505904294109639214653535979691900176556843602150494472276923163298743356166759951219803027530793950381177302731194738185536484170853382049815857557341629265257965586456133775651283810710467884531826649993812111024573155970869490184);
      wts[9] = static_cast<Real>(0.1246289712555338720524762821920164201448868592222026799447505904294109639214653535979691900176556843602150494472276923163298743356166759951219803027530793950381177302731194738185536484170853382049815857557341629265257965586456133775651283810710467884531826649993812111024573155970869490184);
      wts[10] = static_cast<Real>(0.0951585116824927848099251076022462263552635031837126581568222872296317595776819594470249573208049864890125891496209414830603481210533228470521726818393631052313960626642025883937935575862183078092754984560926424378606161380726146645506474202051743093462038386816818873937323729463744264427);
      wts[11] = static_cast<Real>(0.0951585116824927848099251076022462263552635031837126581568222872296317595776819594470249573208049864890125891496209414830603481210533228470521726818393631052313960626642025883937935575862183078092754984560926424378606161380726146645506474202051743093462038386816818873937323729463744264427);
      wts[12] = static_cast<Real>(0.0622535239386478928628438369943776942749865083529068579013035158195357388870438190612117095368517345326677408437010587484095102733930688584375043469278735539030250578328671232854442125221885072444734914927471878528529881785852210014369332953413853555572557051831454875621973358414449964621);
      wts[13] = static_cast<Real>(0.0622535239386478928628438369943776942749865083529068579013035158195357388870438190612117095368517345326677408437010587484095102733930688584375043469278735539030250578328671232854442125221885072444734914927471878528529881785852210014369332953413853555572557051831454875621973358414449964621);
      wts[14] = static_cast<Real>(0.0271524594117540948517805724560181035122673755667607979906103190738041136642161732493257792290308808998974259954086067547083751974232521958283764526154679805418678763257877173475328975259417904071578049294517404669732770687128254946226155762176028585659082598276568361140612201745491935876);
      wts[15] = static_cast<Real>(0.0271524594117540948517805724560181035122673755667607979906103190738041136642161732493257792290308808998974259954086067547083751974232521958283764526154679805418678763257877173475328975259417904071578049294517404669732770687128254946226155762176028585659082598276568361140612201745491935876);
    }
    else if ( nQuad_ == 17 ) {
      pts[0] = static_cast<Real>(0);
      pts[1] = static_cast<Real>(-0.1784841814958478558506774936540655574754193326915256435629518142706975504793005761702889222096350045795804136658695738694619191979106004270228012726401869233061635221498084388495367852285766359307626855910796829121983937561974468156386921610397465033747024112841920833242391965928122710019);
      pts[2] = static_cast<Real>(0.1784841814958478558506774936540655574754193326915256435629518142706975504793005761702889222096350045795804136658695738694619191979106004270228012726401869233061635221498084388495367852285766359307626855910796829121983937561974468156386921610397465033747024112841920833242391965928122710019);
      pts[3] = static_cast<Real>(-0.3512317634538763152971855170953460050405397515756750233191610195477942616467495728975831446035091048810381588477382407912252638664865057412319966011809225709968396403152027573911355548104630952863518856375082639243641691017165305832645101190202334186490287028797692846870706228626308816491);
      pts[4] = static_cast<Real>(0.3512317634538763152971855170953460050405397515756750233191610195477942616467495728975831446035091048810381588477382407912252638664865057412319966011809225709968396403152027573911355548104630952863518856375082639243641691017165305832645101190202334186490287028797692846870706228626308816491);
      pts[5] = static_cast<Real>(-0.5126905370864769678862465686295518745829237224111729059127314989642849249427854802819804123805176014584325052925165192966360139652325847967343193481391404805827724564599814687534893123653481428376081889988243418495226035987975087635446257599915049928041759909862754327315262608198435959332);
      pts[6] = static_cast<Real>(0.5126905370864769678862465686295518745829237224111729059127314989642849249427854802819804123805176014584325052925165192966360139652325847967343193481391404805827724564599814687534893123653481428376081889988243418495226035987975087635446257599915049928041759909862754327315262608198435959332);
      pts[7] = static_cast<Real>(-0.6576711592166907658503022166430023351478058914759732438052316955073242975746975817529911873055101297833759198365079988369392250862712547064647325607093570330252901628820664620720603995070839184219093394018500782102862584336684803597408500457179472106359707247730857472113860258360040733342);
      pts[8] = static_cast<Real>(0.6576711592166907658503022166430023351478058914759732438052316955073242975746975817529911873055101297833759198365079988369392250862712547064647325607093570330252901628820664620720603995070839184219093394018500782102862584336684803597408500457179472106359707247730857472113860258360040733342);
      pts[9] = static_cast<Real>(-0.7815140038968014069252300555204760502239724727405685125133145355125776161595130761733805008669616186539093556523599755816405587726452069844343551330106705582443172919953055322246853811420534803294039661756969840086864690736110181505183617620877407397805711197743018080698887486506560563158);
      pts[10] = static_cast<Real>(0.7815140038968014069252300555204760502239724727405685125133145355125776161595130761733805008669616186539093556523599755816405587726452069844343551330106705582443172919953055322246853811420534803294039661756969840086864690736110181505183617620877407397805711197743018080698887486506560563158);
      pts[11] = static_cast<Real>(-0.8802391537269859021229556944881556926234168179344279003519101593240271305044112115183071938685591895710614181021526073144117263250218799689422297698803582496708490371556589381766868900376270358926962450241091171809069809128556714741015345674751136755372135833147057392935471205895690011357);
      pts[12] = static_cast<Real>(0.8802391537269859021229556944881556926234168179344279003519101593240271305044112115183071938685591895710614181021526073144117263250218799689422297698803582496708490371556589381766868900376270358926962450241091171809069809128556714741015345674751136755372135833147057392935471205895690011357);
      pts[13] = static_cast<Real>(-0.9506755217687677612227169578958030214433850465591087076699692124121153662721985947500369027514683577352656831114736735397342558536859079600136566154101961618491108090235156061787056766646529722765835833734963287469592882914043082555734640610770832418251007307942581964507076020992399886144);
      pts[14] = static_cast<Real>(0.9506755217687677612227169578958030214433850465591087076699692124121153662721985947500369027514683577352656831114736735397342558536859079600136566154101961618491108090235156061787056766646529722765835833734963287469592882914043082555734640610770832418251007307942581964507076020992399886144);
      pts[15] = static_cast<Real>(-0.9905754753144173356754340199406652765077898504595643027839087867423438574706220337113584406497419784359779955952757147938335817826095007304716840652816968608184996218958530105568632398642425148175768665003104870520333564613011532692326789295141276998299471031396925994336345025545551266343);
      pts[16] = static_cast<Real>(0.9905754753144173356754340199406652765077898504595643027839087867423438574706220337113584406497419784359779955952757147938335817826095007304716840652816968608184996218958530105568632398642425148175768665003104870520333564613011532692326789295141276998299471031396925994336345025545551266343);
      wts[0] = static_cast<Real>(0.1794464703562065254582656442618856214487803198976685236676686979555967546062802594045284457525042776008649811893857352435682986331876081111851108866780622936808158737325303170069706877766177174957552596061722460122214923353807537674420503241048324569186636761507104543537545368471074829307);
      wts[1] = static_cast<Real>(0.1765627053669926463252709901131972391509244180007481180431444069132090362072826804523346906840866559301006092111631043456274034660130273980983840504934095470315458660285588518049547114140699023340485341657476841937868857687900376505657106556888601202176730417304723831886390079072130629649);
      wts[2] = static_cast<Real>(0.1765627053669926463252709901131972391509244180007481180431444069132090362072826804523346906840866559301006092111631043456274034660130273980983840504934095470315458660285588518049547114140699023340485341657476841937868857687900376505657106556888601202176730417304723831886390079072130629649);
      wts[3] = static_cast<Real>(0.1680041021564500445099706637883231550211981289650740142699558513403231065000245865612038497389197718236207213486995644444452318049734638929137778300338275722632164587692660117202077357829259551829434673213630654214286185068734272546056171102777956754038016814543267266848603524377507589707);
      wts[4] = static_cast<Real>(0.1680041021564500445099706637883231550211981289650740142699558513403231065000245865612038497389197718236207213486995644444452318049734638929137778300338275722632164587692660117202077357829259551829434673213630654214286185068734272546056171102777956754038016814543267266848603524377507589707);
      wts[5] = static_cast<Real>(0.154045761076810288081431594801958611940483058471017934385264711351446741859676451091617449353450908876424153419065240125716036310829880947811725801178402356930926776121628522534028556989979884655500914919553886780419888060495727120915226164856787273338244789944250590848451274884156660718);
      wts[6] = static_cast<Real>(0.154045761076810288081431594801958611940483058471017934385264711351446741859676451091617449353450908876424153419065240125716036310829880947811725801178402356930926776121628522534028556989979884655500914919553886780419888060495727120915226164856787273338244789944250590848451274884156660718);
      wts[7] = static_cast<Real>(0.1351363684685254732863199817023501973721258532344890203779946106888672429899385440975094483135843229282966867023973980627114962655583899994391835707175743659655559740465951852250510145801277671620980932855460592699403329942015689192694629287216502985314399636780315461266085691716296439668);
      wts[8] = static_cast<Real>(0.1351363684685254732863199817023501973721258532344890203779946106888672429899385440975094483135843229282966867023973980627114962655583899994391835707175743659655559740465951852250510145801277671620980932855460592699403329942015689192694629287216502985314399636780315461266085691716296439668);
      wts[9] = static_cast<Real>(0.1118838471934039710947883856263559267358434242630770500184864824490956796436098356152217045277663097303933856234485005009089118098926769197175416932114933263692040520889707361712734742157806742432060058117182592830428702087289106874985175861316771801379631019461083549422893139993867447325);
      wts[10] = static_cast<Real>(0.1118838471934039710947883856263559267358434242630770500184864824490956796436098356152217045277663097303933856234485005009089118098926769197175416932114933263692040520889707361712734742157806742432060058117182592830428702087289106874985175861316771801379631019461083549422893139993867447325);
      wts[11] = static_cast<Real>(0.0850361483171791808835353701910620738504913892185054757664103607387839927874826477469876042935729999564873422413092975801009051621765385504863648371802661594642985254447697662510671267561256372184607416045118289583482113043069561231013879928175322956122033449176613110877780729058835216161);
      wts[12] = static_cast<Real>(0.0850361483171791808835353701910620738504913892185054757664103607387839927874826477469876042935729999564873422413092975801009051621765385504863648371802661594642985254447697662510671267561256372184607416045118289583482113043069561231013879928175322956122033449176613110877780729058835216161);
      wts[13] = static_cast<Real>(0.055459529373987201129440165358244660512846251953228846993724078710257426003334638373275788744821901102297299483619936822358082783206379864639745886162534575206272618194628448672658150271369557624764374900791878769502590209653421358315740618303892325062804722289534273941957588087316299781);
      wts[14] = static_cast<Real>(0.055459529373987201129440165358244660512846251953228846993724078710257426003334638373275788744821901102297299483619936822358082783206379864639745886162534575206272618194628448672658150271369557624764374900791878769502590209653421358315740618303892325062804722289534273941957588087316299781);
      wts[15] = static_cast<Real>(0.0241483028685479319601100262875653246916973159450252783111851488302183967055104863595852414675449908519473113756040904963477830807558383713007208876834609499285717924393173191172738861013117628311002381876812143174198567792595740020073117811493886032365375159642595860025385521819350292818);
      wts[16] = static_cast<Real>(0.0241483028685479319601100262875653246916973159450252783111851488302183967055104863595852414675449908519473113756040904963477830807558383713007208876834609499285717924393173191172738861013117628311002381876812143174198567792595740020073117811493886032365375159642595860025385521819350292818);
    }
    else if ( nQuad_ == 18 ) {
      pts[0] = static_cast<Real>(-0.0847750130417353012422618529357838117333173869060899200433645176254997906532800016378045507274396124437829275672798731227331774869591555742707869354503347804827247386813860083492943153960827370950056849139593030465987915517910181494323385414322247870220801616210167503447932213563159478785);
      pts[1] = static_cast<Real>(0.0847750130417353012422618529357838117333173869060899200433645176254997906532800016378045507274396124437829275672798731227331774869591555742707869354503347804827247386813860083492943153960827370950056849139593030465987915517910181494323385414322247870220801616210167503447932213563159478785);
      pts[2] = static_cast<Real>(-0.2518862256915055095889728548779112301628617656596404580202710317367844935557242218070775421093862471964631424494432651250600548532785241952804861607909494284233921092736984549611801749674468819063730487487982301410941675444751653415111376578708966480260742797102399929583900718033759327209);
      pts[3] = static_cast<Real>(0.2518862256915055095889728548779112301628617656596404580202710317367844935557242218070775421093862471964631424494432651250600548532785241952804861607909494284233921092736984549611801749674468819063730487487982301410941675444751653415111376578708966480260742797102399929583900718033759327209);
      pts[4] = static_cast<Real>(-0.4117511614628426460359317938330516370789896821200255112811488677903896731004254097743468503936510289261843214273255523679596193584878879367777174383864260484718209265892528661527036305809062189916714650340389469457285122976412592686370359591178240021067563577951998332485195378692379215131);
      pts[5] = static_cast<Real>(0.4117511614628426460359317938330516370789896821200255112811488677903896731004254097743468503936510289261843214273255523679596193584878879367777174383864260484718209265892528661527036305809062189916714650340389469457285122976412592686370359591178240021067563577951998332485195378692379215131);
      pts[6] = static_cast<Real>(-0.5597708310739475346078715485253291369276264857707094166399869440621845282065205196590820483600443686064305949981517700730205326721345021312279007380138111436618919477997252488912228840889276391566530300511555000407709153823662506388077533588735843273742170711587833812410748841733985050272);
      pts[7] = static_cast<Real>(0.5597708310739475346078715485253291369276264857707094166399869440621845282065205196590820483600443686064305949981517700730205326721345021312279007380138111436618919477997252488912228840889276391566530300511555000407709153823662506388077533588735843273742170711587833812410748841733985050272);
      pts[8] = static_cast<Real>(-0.6916870430603532078748910812888483894522705728175077589021626568371973689036292511338108081451326302512715434263282146016602222404375263511358081081355117023215477414027132560924872142002376157090470964638301283208172410282175605496698693378531770444657040491959385718913076247587470628082);
      pts[9] = static_cast<Real>(0.6916870430603532078748910812888483894522705728175077589021626568371973689036292511338108081451326302512715434263282146016602222404375263511358081081355117023215477414027132560924872142002376157090470964638301283208172410282175605496698693378531770444657040491959385718913076247587470628082);
      pts[10] = static_cast<Real>(-0.8037049589725231156824174550145907971032989216119224817504280641953881054289442872282869654424117596467152955000680227836548195338155457937510726733209200198004702508792102856629906949154253696726098507976346863982112440044562546852114733109991186383542181201274590590370795593314392268347);
      pts[11] = static_cast<Real>(0.8037049589725231156824174550145907971032989216119224817504280641953881054289442872282869654424117596467152955000680227836548195338155457937510726733209200198004702508792102856629906949154253696726098507976346863982112440044562546852114733109991186383542181201274590590370795593314392268347);
      pts[12] = static_cast<Real>(-0.8926024664975557392060605911271455154078952713522982141874663149072405824367849358306675870806031563946071223515443126239253900887518555749242643109675905854829563531948787298537904063594779599761695011435151016859951784419272996641462898499377325907533789040905081319348608175241483916026);
      pts[13] = static_cast<Real>(0.8926024664975557392060605911271455154078952713522982141874663149072405824367849358306675870806031563946071223515443126239253900887518555749242643109675905854829563531948787298537904063594779599761695011435151016859951784419272996641462898499377325907533789040905081319348608175241483916026);
      pts[14] = static_cast<Real>(-0.9558239495713977551811958929297763099728441348113064788453876296908980712047732239578823929596339243208559482466056887788906043119002511048689169938982404066468253675430144509701567089745157575736895029193279729376434486706077259413254472259577468248101576870177750661249908727294092126311);
      pts[15] = static_cast<Real>(0.9558239495713977551811958929297763099728441348113064788453876296908980712047732239578823929596339243208559482466056887788906043119002511048689169938982404066468253675430144509701567089745157575736895029193279729376434486706077259413254472259577468248101576870177750661249908727294092126311);
      pts[16] = static_cast<Real>(-0.9915651684209309467300160047061507702525789368454396929196756302398570935917796097517634696126012271523731072179779355873881301740594359250415621032096476004025642882577409246336921223339351585917712567917677623257005985578876226609693419436372609608161063669905409072224762588495765160794);
      pts[17] = static_cast<Real>(0.9915651684209309467300160047061507702525789368454396929196756302398570935917796097517634696126012271523731072179779355873881301740594359250415621032096476004025642882577409246336921223339351585917712567917677623257005985578876226609693419436372609608161063669905409072224762588495765160794);
      wts[0] = static_cast<Real>(0.1691423829631435918406564701349866103341058193703438802698751915490594441915546421709209140397781535882236287858096022651991109570105409679724328782921212245564934063159034512433937773326382340213877306682008490401803026464426090514675100961891997919361138887123759382863686530866127972569);
      wts[1] = static_cast<Real>(0.1691423829631435918406564701349866103341058193703438802698751915490594441915546421709209140397781535882236287858096022651991109570105409679724328782921212245564934063159034512433937773326382340213877306682008490401803026464426090514675100961891997919361138887123759382863686530866127972569);
      wts[2] = static_cast<Real>(0.1642764837458327229860537764659275904123389539973529532444969709116484218452080077553637577470604890284648113509545041993644726379185282871113207576557673424148421704789647486570815505590931928740872581565708587404351827700772678358448394118368041141223571486216522533602397578204727788698);
      wts[3] = static_cast<Real>(0.1642764837458327229860537764659275904123389539973529532444969709116484218452080077553637577470604890284648113509545041993644726379185282871113207576557673424148421704789647486570815505590931928740872581565708587404351827700772678358448394118368041141223571486216522533602397578204727788698);
      wts[4] = static_cast<Real>(0.1546846751262652449254180038363747721932183962673541726666419147277828692306437303123469238522236810610823138635759151314464904508619440285864710786047569608481404523324601264735344481079235728593245977248380570057190791293991887575270796743004370720572961243853465885948926313557194625353);
      wts[5] = static_cast<Real>(0.1546846751262652449254180038363747721932183962673541726666419147277828692306437303123469238522236810610823138635759151314464904508619440285864710786047569608481404523324601264735344481079235728593245977248380570057190791293991887575270796743004370720572961243853465885948926313557194625353);
      wts[6] = static_cast<Real>(0.1406429146706506512047313037519472280955024103309725598834561952604655869277498970306627690269133215655608200217275770685544170251231718644306129004824026118944766630591084763322327613982486061679811396793683948249135515722138610771570619097928166812612498836789391700951269947345168633319);
      wts[7] = static_cast<Real>(0.1406429146706506512047313037519472280955024103309725598834561952604655869277498970306627690269133215655608200217275770685544170251231718644306129004824026118944766630591084763322327613982486061679811396793683948249135515722138610771570619097928166812612498836789391700951269947345168633319);
      wts[8] = static_cast<Real>(0.1225552067114784601845191268002015552281638973334390971672535136540574395994036818381972199250410149007167708455810165486796115018827133040140143404137921288394045516938052752003072949826623487274490764701140571187993024296332307194300911675059927766461093255811877510343876834731962512392);
      wts[9] = static_cast<Real>(0.1225552067114784601845191268002015552281638973334390971672535136540574395994036818381972199250410149007167708455810165486796115018827133040140143404137921288394045516938052752003072949826623487274490764701140571187993024296332307194300911675059927766461093255811877510343876834731962512392);
      wts[10] = static_cast<Real>(0.1009420441062871655628139849248346070628011388876789016103745366556507973804985233524506387113404511235918423705414132943111061599125790124570032420205399215139736906316754374178287629629818649856828420523731488700775027353808841169146345320562519294959618701674957105726006182889016178398);
      wts[11] = static_cast<Real>(0.1009420441062871655628139849248346070628011388876789016103745366556507973804985233524506387113404511235918423705414132943111061599125790124570032420205399215139736906316754374178287629629818649856828420523731488700775027353808841169146345320562519294959618701674957105726006182889016178398);
      wts[12] = static_cast<Real>(0.0764257302548890565291296776166365256053179062083582874495213792344100072917929793417651026783961556386755094377371250624710740108554558830513694718616741579405961948021942164059258052391191883635397652229042344876375847235881442974454263056439849686517694850815001034580970692779127113393);
      wts[13] = static_cast<Real>(0.0764257302548890565291296776166365256053179062083582874495213792344100072917929793417651026783961556386755094377371250624710740108554558830513694718616741579405961948021942164059258052391191883635397652229042344876375847235881442974454263056439849686517694850815001034580970692779127113393);
      wts[14] = static_cast<Real>(0.0497145488949697964533349462026386416808662461289102022463043078655723432958034162950413709493755363901578109123761072246185991085606677862915057965108447312909911728438022161634770405530539475169791168653216398791251592475094524716122801457301171483506316945948812057997564235047218740689);
      wts[15] = static_cast<Real>(0.0497145488949697964533349462026386416808662461289102022463043078655723432958034162950413709493755363901578109123761072246185991085606677862915057965108447312909911728438022161634770405530539475169791168653216398791251592475094524716122801457301171483506316945948812057997564235047218740689);
      wts[16] = static_cast<Real>(0.0216160135264833103133427102664524693876852314755899454620759901413530902373451219032513030698711967035264924116967392053551181478743988660852695341581009207010816978420860521062185588642790444835684731603087600331123347457553616726010767569443955174785105791766212787985301684570081667535);
      wts[17] = static_cast<Real>(0.0216160135264833103133427102664524693876852314755899454620759901413530902373451219032513030698711967035264924116967392053551181478743988660852695341581009207010816978420860521062185588642790444835684731603087600331123347457553616726010767569443955174785105791766212787985301684570081667535);
    }
    else if ( nQuad_ == 19 ) {
      pts[0] = static_cast<Real>(0);
      pts[1] = static_cast<Real>(-0.1603586456402253758680961157407435495048735004708753788746434516063206086803910736567395734994809129318483420183478796475911141450809718607901705094157416287827636723317712230366367245420707236761908155274983931420897352955705565372972610680898139803112996239542230328106130986249914194466);
      pts[2] = static_cast<Real>(0.1603586456402253758680961157407435495048735004708753788746434516063206086803910736567395734994809129318483420183478796475911141450809718607901705094157416287827636723317712230366367245420707236761908155274983931420897352955705565372972610680898139803112996239542230328106130986249914194466);
      pts[3] = static_cast<Real>(-0.3165640999636298319901173288498449178922852191328872451557289867978928858205763979968388920864803510731535081730250430846740428309401136527828721051531490074733436500005309647508972237675340513041782549700501128310120829534588207401491218641251965174007983379995123754411522457867103948772);
      pts[4] = static_cast<Real>(0.3165640999636298319901173288498449178922852191328872451557289867978928858205763979968388920864803510731535081730250430846740428309401136527828721051531490074733436500005309647508972237675340513041782549700501128310120829534588207401491218641251965174007983379995123754411522457867103948772);
      pts[5] = static_cast<Real>(-0.4645707413759609457172671481041023679762857146241365969843088775756841211134254066864495457207319448710918029959714308308930177935413454358856446634805657842775035152802842495561385377870555193779722481176187968697089363598051000265289947360815853090109925771081768757315967784399481847803);
      pts[6] = static_cast<Real>(0.4645707413759609457172671481041023679762857146241365969843088775756841211134254066864495457207319448710918029959714308308930177935413454358856446634805657842775035152802842495561385377870555193779722481176187968697089363598051000265289947360815853090109925771081768757315967784399481847803);
      pts[7] = static_cast<Real>(-0.6005453046616810234696381649462392798683220827322925675865867256683849086968708404247557953314540014353382042090405221838228456526682112284995393891609639483102981113782025164036436092611231897835577754193291700630336379989992738338079975837177331702566819859022567587474004959064248807312);
      pts[8] = static_cast<Real>(0.6005453046616810234696381649462392798683220827322925675865867256683849086968708404247557953314540014353382042090405221838228456526682112284995393891609639483102981113782025164036436092611231897835577754193291700630336379989992738338079975837177331702566819859022567587474004959064248807312);
      pts[9] = static_cast<Real>(-0.7209661773352293786170958608237816296571418329086667749385904713866145591479860528434333826713265745865146368763695893705935426689231443437359816693935769111598553832735946243240592551656966635633193289676856911346143088410502010801975195687785210496353060132822672990510823194865104810051);
      pts[10] = static_cast<Real>(0.7209661773352293786170958608237816296571418329086667749385904713866145591479860528434333826713265745865146368763695893705935426689231443437359816693935769111598553832735946243240592551656966635633193289676856911346143088410502010801975195687785210496353060132822672990510823194865104810051);
      pts[11] = static_cast<Real>(-0.8227146565371428249789224867127139017745384862068300413698638886055003429036858320268662231092605996470225618984594685109936771887421408609253135128638508766857045968943958516156187196628330197774864688134063649677798586588072572520741671269515166379513792151159908452751122210353837037279);
      pts[12] = static_cast<Real>(0.8227146565371428249789224867127139017745384862068300413698638886055003429036858320268662231092605996470225618984594685109936771887421408609253135128638508766857045968943958516156187196628330197774864688134063649677798586588072572520741671269515166379513792151159908452751122210353837037279);
      pts[13] = static_cast<Real>(-0.9031559036148179016426609285323124878093939340573558176950324247616358045901042189575202283692970456300780799018278300312177475152478100919848787125579035155833655812231871735227628634411647215991745485349351959389950818755865635582452618806932993942778802308444798554483289485483294136503);
      pts[14] = static_cast<Real>(0.9031559036148179016426609285323124878093939340573558176950324247616358045901042189575202283692970456300780799018278300312177475152478100919848787125579035155833655812231871735227628634411647215991745485349351959389950818755865635582452618806932993942778802308444798554483289485483294136503);
      pts[15] = static_cast<Real>(-0.9602081521348300308527788406876515266150915032741381059178163344322007499374885132207435614915413177254368192523164742167978833824318991613029627697107716004048389061572001566748259909558453570074951099472348857440923899276141697395726289443107249605043469217918612003537920025001631421361);
      pts[16] = static_cast<Real>(0.9602081521348300308527788406876515266150915032741381059178163344322007499374885132207435614915413177254368192523164742167978833824318991613029627697107716004048389061572001566748259909558453570074951099472348857440923899276141697395726289443107249605043469217918612003537920025001631421361);
      pts[17] = static_cast<Real>(-0.9924068438435844031890176702532604935893164014032107866796794390912319029585228838036749208734188013119794958338236088364145226980238478133549327801874434275998166833278883271648695176361368253792683042178805377659998449085455102400673694779767120278357276901409734267296826856483787011163);
      pts[18] = static_cast<Real>(0.9924068438435844031890176702532604935893164014032107866796794390912319029585228838036749208734188013119794958338236088364145226980238478133549327801874434275998166833278883271648695176361368253792683042178805377659998449085455102400673694779767120278357276901409734267296826856483787011163);
      wts[0] = static_cast<Real>(0.1610544498487836959791636253209167350399025585785169021283231527357710484554980721525407657169290469326322822863184992213743178868498200222270801309797567400348596761477557415796634427690419403563011194249302152575062701292613967331058845014126474128577480085119949784227603045386781841262);
      wts[1] = static_cast<Real>(0.1589688433939543476499564394650472016787801581951260957511749153247409785833850597426104767260064785656726957407308709294681940869471572774193889952003779483998952863244786932834096103688529075704047740756352615495602997460739650278393056497770505071451767267951634011784814089959663984428);
      wts[2] = static_cast<Real>(0.1589688433939543476499564394650472016787801581951260957511749153247409785833850597426104767260064785656726957407308709294681940869471572774193889952003779483998952863244786932834096103688529075704047740756352615495602997460739650278393056497770505071451767267951634011784814089959663984428);
      wts[3] = static_cast<Real>(0.1527660420658596667788554008976629984610082672364286235231015530748110436510632136548098470307870684022812576438514294182842697467316852588339979189564400065158561055368543286606806945813575677225325234593817650217143933943565807393710549158806994880737896661371021634932839064588639773471);
      wts[4] = static_cast<Real>(0.1527660420658596667788554008976629984610082672364286235231015530748110436510632136548098470307870684022812576438514294182842697467316852588339979189564400065158561055368543286606806945813575677225325234593817650217143933943565807393710549158806994880737896661371021634932839064588639773471);
      wts[5] = static_cast<Real>(0.1426067021736066117757461094419029724756683448244738609265716559064582484729032510149400718372501369086306014406580383315195828422829752494295918122366158420984545209886445875554558671077636205184180315501264678352263230297125824594481607430544278129481485570189599544910201063848937021998);
      wts[6] = static_cast<Real>(0.1426067021736066117757461094419029724756683448244738609265716559064582484729032510149400718372501369086306014406580383315195828422829752494295918122366158420984545209886445875554558671077636205184180315501264678352263230297125824594481607430544278129481485570189599544910201063848937021998);
      wts[7] = static_cast<Real>(0.1287539625393362276755157848568771170558395770934630345471043846223036040566316799314330023341520732158916457503562980563768636602583508434307981699008263911842629973311123280171601256818577811518739247152452114541083839100451940773411361442241742604289292646622610041938152943010790361288);
      wts[8] = static_cast<Real>(0.1287539625393362276755157848568771170558395770934630345471043846223036040566316799314330023341520732158916457503562980563768636602583508434307981699008263911842629973311123280171601256818577811518739247152452114541083839100451940773411361442241742604289292646622610041938152943010790361288);
      wts[9] = static_cast<Real>(0.1115666455473339947160239016817659974813318538398937755214611428764589761056464706010042849525923054056261554442105314745763195932849808297160769131997858963255201640444230360365240357831562145828787343616372535224037060083589168167362608964805142848746151312363217280104220309101862162324);
      wts[10] = static_cast<Real>(0.1115666455473339947160239016817659974813318538398937755214611428764589761056464706010042849525923054056261554442105314745763195932849808297160769131997858963255201640444230360365240357831562145828787343616372535224037060083589168167362608964805142848746151312363217280104220309101862162324);
      wts[11] = static_cast<Real>(0.0914900216224499994644620941238396526609116512965987846649893449020622210437972616057477150418889562743146634993042981808904181458448709504359957484813885110802183834295752411339593463140700502937848306309596141966543958396121404946314428704188953782008399937706471108301458402758231527037);
      wts[12] = static_cast<Real>(0.0914900216224499994644620941238396526609116512965987846649893449020622210437972616057477150418889562743146634993042981808904181458448709504359957484813885110802183834295752411339593463140700502937848306309596141966543958396121404946314428704188953782008399937706471108301458402758231527037);
      wts[13] = static_cast<Real>(0.0690445427376412265807082580060130449618480316876131311229004276960349749064390215176630954633349726374642876653899542524801698925666536827737780371699346508028049015854813015646013769733407219337570614953319341361634180376621748813203883305923293567307835594677851391824357149174951095532);
      wts[14] = static_cast<Real>(0.0690445427376412265807082580060130449618480316876131311229004276960349749064390215176630954633349726374642876653899542524801698925666536827737780371699346508028049015854813015646013769733407219337570614953319341361634180376621748813203883305923293567307835594677851391824357149174951095532);
      wts[15] = static_cast<Real>(0.0448142267656996003328381574019942119517542274678576020858545267676429913842794576446690357102559934203280323775860360343000334458319796780552202725286599385464482175556425829180401153232218294624007148445216886976212985328934819110208162425464110028809041117163162427069065186589566497618);
      wts[16] = static_cast<Real>(0.0448142267656996003328381574019942119517542274678576020858545267676429913842794576446690357102559934203280323775860360343000334458319796780552202725286599385464482175556425829180401153232218294624007148445216886976212985328934819110208162425464110028809041117163162427069065186589566497618);
      wts[17] = static_cast<Real>(0.0194617882297264770363120414644384357529066090692866407926804724616014375681055482108520880452674917034745192947532937114169896428264362187916120668360924450291095851299100300403371064818583365857988451546956959577946464366542652257384919563191742022879389849394457667021090268241509572387);
      wts[18] = static_cast<Real>(0.0194617882297264770363120414644384357529066090692866407926804724616014375681055482108520880452674917034745192947532937114169896428264362187916120668360924450291095851299100300403371064818583365857988451546956959577946464366542652257384919563191742022879389849394457667021090268241509572387);
    }
    else if ( nQuad_ == 20 ) {
      pts[0] = static_cast<Real>(-0.0765265211334973337546404093988382110047962668134975008047952443842563420483369782415451141815562156069985056463641327044405378245712628847111887172768133849480713034639232220573314968967976578624002126908309880442644981195204905648677468155050797193996060978666309220354740163118897397462);
      pts[1] = static_cast<Real>(0.0765265211334973337546404093988382110047962668134975008047952443842563420483369782415451141815562156069985056463641327044405378245712628847111887172768133849480713034639232220573314968967976578624002126908309880442644981195204905648677468155050797193996060978666309220354740163118897397462);
      pts[2] = static_cast<Real>(-0.2277858511416450780804961953685746247430889376829274723146357392071713418635558277949521251909687080317737313156043021742990408764128121348527309473103510231012088770889101526899727869312951647279166230222083532237485779698033678575987234663095414012801053067685591624158148344361195752847);
      pts[3] = static_cast<Real>(0.2277858511416450780804961953685746247430889376829274723146357392071713418635558277949521251909687080317737313156043021742990408764128121348527309473103510231012088770889101526899727869312951647279166230222083532237485779698033678575987234663095414012801053067685591624158148344361195752847);
      pts[4] = static_cast<Real>(-0.3737060887154195606725481770249272373957463217056827118279486135156457643730595278958956836345333789447677220885281501530593501068444273954500663893598076876632116269160694032156736748270203663186845460433681971911054257380408445871617371947413964535372457387465138670353227723685242870666);
      pts[5] = static_cast<Real>(0.3737060887154195606725481770249272373957463217056827118279486135156457643730595278958956836345333789447677220885281501530593501068444273954500663893598076876632116269160694032156736748270203663186845460433681971911054257380408445871617371947413964535372457387465138670353227723685242870666);
      pts[6] = static_cast<Real>(-0.5108670019508270980043640509552509984254913292024268334723486198947349703907657281440316830508677791983294306884352623565652540225691147912695617386450606003834133539672964815466909956561514101091590010957544863926131866746678211741307841744076338006709425045419675774556151498035520402212);
      pts[7] = static_cast<Real>(0.5108670019508270980043640509552509984254913292024268334723486198947349703907657281440316830508677791983294306884352623565652540225691147912695617386450606003834133539672964815466909956561514101091590010957544863926131866746678211741307841744076338006709425045419675774556151498035520402212);
      pts[8] = static_cast<Real>(-0.6360536807265150254528366962262859367433891167993684639394466225465412625854301325587031954957613065821171093777259573620410810297092032405446994739443650537331906321586488794157012474756582161960205379517855031717327595950200970542877586428866635712964405838287500018038820406993696960313);
      pts[9] = static_cast<Real>(0.6360536807265150254528366962262859367433891167993684639394466225465412625854301325587031954957613065821171093777259573620410810297092032405446994739443650537331906321586488794157012474756582161960205379517855031717327595950200970542877586428866635712964405838287500018038820406993696960313);
      pts[10] = static_cast<Real>(-0.7463319064601507926143050703556415903107306795691764441395459060685353550381550646811041136206475206123849006516765614963114309728028906789375472400269214183897220068963288979058850209548743013321442485351491957120911186313909609633405415948092225881978601529357254454910055662779167650089);
      pts[11] = static_cast<Real>(0.7463319064601507926143050703556415903107306795691764441395459060685353550381550646811041136206475206123849006516765614963114309728028906789375472400269214183897220068963288979058850209548743013321442485351491957120911186313909609633405415948092225881978601529357254454910055662779167650089);
      pts[12] = static_cast<Real>(-0.8391169718222188233945290617015206853296293650656373732524927255328610939993248099192293405659576492206042203530691409455744267670339561156784556927871651303614757720382422461584374656765616405754841392655432820150389419609038711147956974464600816733249843930304105365594281399220263222406);
      pts[13] = static_cast<Real>(0.8391169718222188233945290617015206853296293650656373732524927255328610939993248099192293405659576492206042203530691409455744267670339561156784556927871651303614757720382422461584374656765616405754841392655432820150389419609038711147956974464600816733249843930304105365594281399220263222406);
      pts[14] = static_cast<Real>(-0.9122344282513259058677524412032981130491847974236917747958822191580708912087190789364447261929213873787603917546460264117368633829388364812137731072764160539290861731887984543836691517730025113125160507543614452998823427889139340036155775261652542681121293143129162250026709325798464559842);
      pts[15] = static_cast<Real>(0.9122344282513259058677524412032981130491847974236917747958822191580708912087190789364447261929213873787603917546460264117368633829388364812137731072764160539290861731887984543836691517730025113125160507543614452998823427889139340036155775261652542681121293143129162250026709325798464559842);
      pts[16] = static_cast<Real>(-0.9639719272779137912676661311972772219120603278061888560635375938920415807843830569800181252559647156313104349159642305288604107945950841916012896973856125442362722204039344929993138383198252917335788877307216356759417337912146196947795957049391561976704802249548893795179890457356617851265);
      pts[17] = static_cast<Real>(0.9639719272779137912676661311972772219120603278061888560635375938920415807843830569800181252559647156313104349159642305288604107945950841916012896973856125442362722204039344929993138383198252917335788877307216356759417337912146196947795957049391561976704802249548893795179890457356617851265);
      pts[18] = static_cast<Real>(-0.9931285991850949247861223884713202782226471309016558961481841312179847176277537808394494024965722092747289403472441901380148603873987776712384136792903672018478363391883474614446467923321455848994412127013518584596328653839755407926529209842664597749680291209275759831792849248136886138353);
      pts[19] = static_cast<Real>(0.9931285991850949247861223884713202782226471309016558961481841312179847176277537808394494024965722092747289403472441901380148603873987776712384136792903672018478363391883474614446467923321455848994412127013518584596328653839755407926529209842664597749680291209275759831792849248136886138353);
      wts[0] = static_cast<Real>(0.1527533871307258506980843319550975934919486451123785972747010498175974531627377815355724878365039059354400184281378782601150279679600783005530039321842277357554019203124915100518499038184507289979208511728338097884847930234840391745787787603960313774754576020521238652300682735542710944263);
      wts[1] = static_cast<Real>(0.1527533871307258506980843319550975934919486451123785972747010498175974531627377815355724878365039059354400184281378782601150279679600783005530039321842277357554019203124915100518499038184507289979208511728338097884847930234840391745787787603960313774754576020521238652300682735542710944263);
      wts[2] = static_cast<Real>(0.1491729864726037467878287370019694366926799040813683164962112178098444225955867806939613260352104810517091385456733800686575677792178649430941723424667532046356472774228709560694803833239714102500755019738911703717170310002053316744710774788331941474319396315320062727688974661711619397638);
      wts[3] = static_cast<Real>(0.1491729864726037467878287370019694366926799040813683164962112178098444225955867806939613260352104810517091385456733800686575677792178649430941723424667532046356472774228709560694803833239714102500755019738911703717170310002053316744710774788331941474319396315320062727688974661711619397638);
      wts[4] = static_cast<Real>(0.1420961093183820513292983250671649330345154133920203033373670829838280874979343676169492242832005826013306857366620140189402437272742182737236253724487588567621276688265922710626976924994669598575042368845029007422245631263596983756760256569446985608014039758824276822804479323371308934112);
      wts[5] = static_cast<Real>(0.1420961093183820513292983250671649330345154133920203033373670829838280874979343676169492242832005826013306857366620140189402437272742182737236253724487588567621276688265922710626976924994669598575042368845029007422245631263596983756760256569446985608014039758824276822804479323371308934112);
      wts[6] = static_cast<Real>(0.131688638449176626898494499748163134916110511146983526996436493708854356429480933143557975183972629245105980054636247016400316830607538704952932577689132201902021181666362879795141713525128804030525410925388026264783643215389540368786522194165102288258575353624172248387510459119776031765);
      wts[7] = static_cast<Real>(0.131688638449176626898494499748163134916110511146983526996436493708854356429480933143557975183972629245105980054636247016400316830607538704952932577689132201902021181666362879795141713525128804030525410925388026264783643215389540368786522194165102288258575353624172248387510459119776031765);
      wts[8] = static_cast<Real>(0.1181945319615184173123773777113822870050412195489687754468899520201747483505115163057286878258190174460626754309231659133886817937422099122542309757118513383856290919785688442004649951483215887378851802439188486330757698272816855138495620519895987774292383111032264708641859679188600095692);
      wts[9] = static_cast<Real>(0.1181945319615184173123773777113822870050412195489687754468899520201747483505115163057286878258190174460626754309231659133886817937422099122542309757118513383856290919785688442004649951483215887378851802439188486330757698272816855138495620519895987774292383111032264708641859679188600095692);
      wts[10] = static_cast<Real>(0.1019301198172404350367501354803498761666916560233925562619716161968523220253943464753493157694798582137585903552548326593234407721971982947468111885795226965581584345480416762264760609122842382150145830070657006817955059316458539298252170005077219224922112385771403739725657450818783267313);
      wts[11] = static_cast<Real>(0.1019301198172404350367501354803498761666916560233925562619716161968523220253943464753493157694798582137585903552548326593234407721971982947468111885795226965581584345480416762264760609122842382150145830070657006817955059316458539298252170005077219224922112385771403739725657450818783267313);
      wts[12] = static_cast<Real>(0.0832767415767047487247581432220462061001778285831632907448820607856930828940794194713751908437908393490961161119327637705991492770889027822910222187699193186160700364391275137979683413090460002469936166941844144330159650771507260085403067307251888321939680464514855705930284166808865996089);
      wts[13] = static_cast<Real>(0.0832767415767047487247581432220462061001778285831632907448820607856930828940794194713751908437908393490961161119327637705991492770889027822910222187699193186160700364391275137979683413090460002469936166941844144330159650771507260085403067307251888321939680464514855705930284166808865996089);
      wts[14] = static_cast<Real>(0.0626720483341090635695065351870416063516010765784363640995843454379748110336656786445637660568322035126032533995920732617572100906357057485305643635379372052174565132880956643096197824783282502822044536246229255150791963516814947223936297508668215819063938781024256991067440283222516531374);
      wts[15] = static_cast<Real>(0.0626720483341090635695065351870416063516010765784363640995843454379748110336656786445637660568322035126032533995920732617572100906357057485305643635379372052174565132880956643096197824783282502822044536246229255150791963516814947223936297508668215819063938781024256991067440283222516531374);
      wts[16] = static_cast<Real>(0.0406014298003869413310399522749321098790906399899515368176068545618322967509873282955389206230443849761898257096750750226100388420589263242455270189562763042394217695623102760213193663915194028524305033283414664156073509698450908818586062170756508088247402639869134213644998336627334563899);
      wts[17] = static_cast<Real>(0.0406014298003869413310399522749321098790906399899515368176068545618322967509873282955389206230443849761898257096750750226100388420589263242455270189562763042394217695623102760213193663915194028524305033283414664156073509698450908818586062170756508088247402639869134213644998336627334563899);
      wts[18] = static_cast<Real>(0.017614007139152118311861962351852816362143105543336732524349326677348419259621847817403105542146097668703716227512570008208322919217356715608110009655621137928066105955538408464981760593482616529445662145250737154216181476956539350020274158495991703186071698688078395432051877120218676412);
      wts[19] = static_cast<Real>(0.017614007139152118311861962351852816362143105543336732524349326677348419259621847817403105542146097668703716227512570008208322919217356715608110009655621137928066105955538408464981760593482616529445662145250737154216181476956539350020274158495991703186071698688078395432051877120218676412);
    }
  }

};

}

#endif
